//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef WLAN_BF_MU_RX_SEQ_SV
`define WLAN_BF_MU_RX_SEQ_SV


class wlan_bf_mu_rx_seq extends wlan_seq_base;

  `uvm_object_utils(wlan_bf_mu_rx_seq)

  PPDU_frame                 ndpa_frame;
  PPDU_frame                 ndp_frame;
  PPDU_frame                 bmfr_poll;
  VHT_NDP_ANNOUNCEMENT_frame ndpa;
  PPDU_frame                 mu_mimo;

  bit [31:0]                 rx_buff_rd_ptr;
  int                        ksr_index;
  int                        frame_num;
  int                        rhd_cnt;
  int                        rpd_cnt;
  bit [3:0]                  TID;
  bit [11:0]                 SSN;
  bit [2:0]                  mu_mimo_userid;

  //--------------------------------------------------------------------
  // variables for NDP
  //--------------------------------------------------------------------
  rand bit  [2:0] Nr;                         // Nr Value To Test
  rand bit  [2:0] Nc[`NB_STA_MAX];            // Nc Value To Test
  rand bit  [1:0] Grouping[`NB_STA_MAX];      // Grouping Value To Test
  rand bit        FeedbackType[`NB_STA_MAX];  // Feedback Type Value To Test
  rand bit        Codebook[`NB_STA_MAX];      // Codebook Value To Test

       bit [15:0] AID;
       int        STAID;
  rand bit [2:0]  num_of_stations;   // number of stations, used for BF Tx test
  rand bit [1:0]  ch_bw;

  //--------------------------------------------------------------------
  // Constraints
  //--------------------------------------------------------------------

`ifdef RW_MUMIMO_RX_EN
  constraint c_Nr {
    Nr inside {1,3};
  }
`else
  constraint c_Nr {
    Nr == 1;
  }
`endif // RW_MUMIMO_RX_EN

  constraint c_FeedbackType {
    foreach (FeedbackType[i]) FeedbackType[i] == 1;
  }

  constraint c_Nc {
    solve Nr before Nc;
`ifdef RW_TXRX_1X1
    foreach (Nc[i]) Nc[i] == 0;
`else
    foreach (Nc[i]) Nc[i] <= 1;
`endif
  }

  constraint c_Grouping {
    foreach (Grouping[i]) Grouping[i] inside {0,1,2};
  }

  constraint c_channel_bw {
    ch_bw == channel_bw_func(VHT);
  }

  constraint c_num_of_stations {
    num_of_stations inside {[2:4]};
  }

  function new (string name = "wlan_bf_mu_rx_seq");
    super.new (name);
  endfunction : new

  function void print_ndp_params(int i=0);
   `uvm_info(get_type_name(), $sformatf("*******************************************************"), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("Reception of VHT NDP Sounding procedure user %0d", i), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Nr                : %2d", (Nr+1)        ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Nc                : %2d", (Nc[i]+1)        ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Grouping          : %2d", Grouping[i]      ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Feedback type     : %s" , (FeedbackType[i]==1'b0)? "SU" : "MU"), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Codebook          : %2d", Codebook[i]   ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Channel BW        : %2d", ch_bw   ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("*******************************************************"), UVM_LOW)
  endfunction: print_ndp_params

  function void set_ndp_params();
    m_mdm_cfg.Nr = Nr;
    m_mdm_cfg.nBeamformingSTA = num_of_stations;
    for (int i=0; i<num_of_stations; i++) begin
      m_mdm_cfg.Nc[i]       = Nc[i];
      m_mdm_cfg.Grouping[i] = Grouping[i];
      m_mdm_cfg.Feedback[i] = FeedbackType[i];
      m_mdm_cfg.Codebook[i] = Codebook[i];
      print_ndp_params(i);
    end
  endfunction : set_ndp_params

  virtual task body();
    super.body();

    // Maximum Nss supported set to 4
    m_regmodel.set_field_value(3'd3, "RXNSSMAX", "RXMODE", "PHYCONFIG");
    m_regmodel.set_field_value(0, "OFDMONLY", "RWNXAGCCNTL", "RIUKARST");

    agcBypass = m_cfg.m_radio_cfg.m_rui_cfg.agcBypass; // set from testcases

    // fork off gain update task
    fork  : GAIN_UPDATE
      begin // thread 1
        if ((`AGC_ON == 1) && (agcBypass == 0)) refresh_RF_data(); // call AGC update task only if necessary
      end
      drive_noise_adc();
    join_none

    //-----------------------------------------
    // determine device and BSS address
    //-----------------------------------------
    create_keystorageRAM();

    create_dev_and_bss_addr();

    // set RX interrupt masks
    m_regmodel.set_field_value(1'b1, "maskrxBuffer1Trigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "masktimerRxTrigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "maskrxBuffer2Trigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "masterTxRxIntEn", "TXRXINTUNMASKREG"); // Enables interrupt generation
    // configure Beamformee register
    m_regmodel.set_field_value(3'b100, "bfrFormatMod", "BFMEECONTROLREG"); //VHT
    m_regmodel.set_field_value(1'h1, "bfmeeEnable", "BFMEECONTROLREG");
    // set AID value for this receiving station
    AID = $urandom();
    m_regmodel.set_field_value(AID, "aid", "BCNCNTRL2REG");

    // put MAC core to active state
    change_mac_core_state(`ACTIVE_STATE);

    ndpa_frame = new("ndpa_frame");
    ndp_frame  = new("ndp_frame");
    ndpa       = new("ndpa");
    bmfr_poll  = new("bmfr_poll");

    // prevent checking of TSF reg
    m_cfg.check_mac_tsf_reg = 0;

    // randomize fields in sequence
    assert (this.randomize());
    set_ndp_params();

    STAID = $urandom_range(0, num_of_stations - 1); // get one station index
    `uvm_info(get_type_name(), $sformatf("STAID used in NDPA: %0d",STAID), UVM_LOW)
    // overwrite register BFMEE with randomized parameters
    m_regmodel.set_field_value(FeedbackType[STAID], "bfmeeMUSupport", "BFMEECONTROLREG");
    m_regmodel.set_field_value(Codebook[STAID], "bfmeeCodebook", "BFMEECONTROLREG");
    m_regmodel.set_field_value(Grouping[STAID], "bfmeeGrouping", "BFMEECONTROLREG");
    m_regmodel.set_field_value(Nr, "bfmeeNr", "BFMEECONTROLREG");
    m_regmodel.set_field_value(Nc[STAID], "bfmeeNc", "BFMEECONTROLREG");

    prepare_sram_for_rx(10000);
    m_regmodel.get_reg_value(rdata, "RXBUF1RDPTRREG");
    rx_buff_rd_ptr = rdata + rhd_header;

    //--------------------------------
    // create NDPA frame for Rx
    //--------------------------------
    // create NDPA MPDU
    assert (ndpa.randomize() with {
      sta_info.size() == num_of_stations;
      sta_info[STAID].nc_index_f == Nc[STAID];
      sta_info[STAID].feedback_type_f == FeedbackType[STAID];
      sta_info[STAID].aid12_f == AID[11:0]; // 12 LSB bits
    });

    assert (ndpa_frame.randomize() with {
      kind == SINGLETON;
      ppdu_format == VHT;
      tx_frame == 0;
    });
    ndpa_frame.preamble_header.ch_bw = ch_bw;

    ndpa_frame.ampdu_frame[0].clear_all_arrays();
    assert (ndpa_frame.ampdu_frame[0].randomize() with {
      ndpa_frame.ampdu_frame[0].ppdu_format == ndpa_frame.ppdu_format;
      ndpa_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
      ndpa_frame.ampdu_frame[0].mpdu_frame_type[0] == VHT_NDP_ANNOUNCEMENT;
      ndpa_frame.ampdu_frame[0].aggregated == 0;
    });
    ndpa_frame.ampdu_frame[0].mpdu_frame[0] = ndpa;
    ndpa_frame.ampdu_frame[0].create_delimiter();
    ndpa_frame.ampdu_frame[0].add_padding();

    ksr_index = get_random_ksr_index(ndpa_frame);
    // set MAC addresses
    ndpa_frame.ampdu_frame[0].set_MAC_addr(
      .RA    (`BROADCAST_ADDR),
      .TA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f),
      .BSSID (bss_addr)
    );

    if (ch_bw == 0 && ndpa_frame.preamble_header.user_header[0].mcs_f == 7'd9) begin
      ndpa_frame.preamble_header.user_header[0].mcs_f = $urandom_range(0,8);
    end

    ndpa_frame.calc_leg_ht_length();

    `uvm_info(get_type_name(),
    $sformatf("NDPA:\n%s", ndpa_frame.sprint()), UVM_LOW)

    //--------------------------------
    // create NDP frame for Rx
    //--------------------------------
    assert (ndp_frame.randomize() with {
      kind == NDP;
      ppdu_format == VHT;
      tx_frame == 0;
    });
    // channel bandwidth needs to be same as in NDPA
    ndp_frame.preamble_header.ch_bw = ndpa_frame.preamble_header.ch_bw;

    `uvm_info(get_type_name(),
    $sformatf("NDP:\n%s", ndp_frame.sprint()), UVM_LOW)

    //--------------------------------
    // create BEAMFORMING REPORT POLL
    // frame for RX
    //--------------------------------
    assert (bmfr_poll.randomize() with {
      kind == SINGLETON;
      ppdu_format == VHT;
      tx_frame == 0;
    });
    bmfr_poll.preamble_header.ch_bw = ndpa_frame.preamble_header.ch_bw;

    bmfr_poll.ampdu_frame[0].clear_all_arrays();
    assert (bmfr_poll.ampdu_frame[0].randomize() with {
      bmfr_poll.ampdu_frame[0].ppdu_format == bmfr_poll.ppdu_format;
      bmfr_poll.ampdu_frame[0].mpdu_frame_type.size() == 1;
      bmfr_poll.ampdu_frame[0].mpdu_frame_type[0] == BEAMFORMING_REPORT_POLL;
      bmfr_poll.ampdu_frame[0].aggregated == 0;
    });

    // set MAC addresses
    bmfr_poll.ampdu_frame[0].set_MAC_addr(
      .RA    (dev_addr),
      .TA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f),
      .BSSID (bss_addr)
    );
    // encrypt BFR
    bmfr_poll.encrypt(.ra(1));
    bmfr_poll.calc_leg_ht_length();

    if (STAID > 0)
      `uvm_info(get_type_name(),$sformatf("BFR POLL frame %s",bmfr_poll.sprint()),UVM_LOW)

    // receive BFR and wait for interrupt Rx trigger
    fork
      // if frames is going to be received trigger will occur,
      // if frames is filtered event will occur
      fork
        wait_mac_rx_trigger();
        rx_frame_filtered();
      join_any

      begin
        // PHY sends frame to MAC core
        drive_phy_adc(ndpa_frame,0,1);
        insert_idle_cycles(11);   // changed to 11, 5us delay is in drie_phy_adc
        drive_phy_adc(ndp_frame);
        // in MU calibration when AID in STA info 1 is not matched
        // and is for some other STA info index then BFM POLL needs
        // to be sent to STA in order to get BFMR
        if (STAID > 0) begin
          insert_idle_cycles($urandom_range(30, 50));
          drive_phy_adc(bmfr_poll);
        end
        else begin
          wait_tx_end();
        end
        wait_tx_end();
      end
    join

    // only if frame was stored in SRAM do read
    if (!stop_pooling) begin
      `uvm_info(get_type_name(),
      "MAC core received frame and stored it to SRAM", UVM_LOW)
      rd_frame_from_sram(rx_buff_rd_ptr);
      // get next RHD pointer value if frame was not filtered
      get_next_rhd_ptr(1, rx_buff_rd_ptr, rx_buff_rd_ptr);
      rd_frame_from_sram(rx_buff_rd_ptr);
      // get next RHD pointer value if frame was not filtered
      get_next_rhd_ptr(1, rx_buff_rd_ptr, rx_buff_rd_ptr);
      if (STAID > 0) begin
        rd_frame_from_sram(rx_buff_rd_ptr);
        // get next RHD pointer value if frame was not filtered
        get_next_rhd_ptr(1, rx_buff_rd_ptr, rx_buff_rd_ptr);
      end
    end

    insert_idle_cycles(20);

    frame_num = $urandom_range(3,5);
    for (int loop=0; loop < frame_num; loop++) begin
      `uvm_info(get_type_name(),$sformatf("Receive Frame number: %0d / %0d",
      loop+1,frame_num),UVM_LOW)

      mu_mimo = new("mu_mimo");
      //--------------------------------
      // create data frame for Rx
      //--------------------------------
      assert (mu_mimo.randomize() with {
        kind == MU_MIMO;
        ppdu_format == VHT;
        tx_frame == 0;
      });

      // index of A-MPDU frame that is being received
      mu_mimo_userid = mu_mimo.preamble_header.mu_mimo_userid;
      ksr_index = get_random_ksr_index(mu_mimo, mu_mimo_userid);
      // set MAC addresses
      mu_mimo.ampdu_frame[mu_mimo_userid].set_MAC_addr(
        .RA    (dev_addr),
        .TA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f),
        .BSSID (bss_addr)
      );
      // get the TID for every frame and update TID table
      TID = mu_mimo.ampdu_frame[mu_mimo_userid].get_MAC_tid_num();
      SSN = mu_mimo.ampdu_frame[mu_mimo_userid].get_MAC_seq_num();
      for (int i = 0; i < m_cfg.TID_table.size(); i++) begin
        if (m_cfg.TID_table[i].TID == TID && m_cfg.TID_table[i].ksr_index == ksr_index) begin
          SSN = m_cfg.TID_table[i].SSN + `WINDOWSIZE + 2;
          mu_mimo.ampdu_frame[mu_mimo_userid].set_MAC_seq_num(SSN);
          break;
        end
      end

      // Write ksr_index into TID_table
      m_cfg.ksr_index = ksr_index;

      mu_mimo.setup_rx_encrypted_data();
      mu_mimo.calc_leg_ht_length();

      `uvm_info(get_type_name(),
      $sformatf("PPDU_frame:\n%s", mu_mimo.sprint()), UVM_LOW)

      // get number of MPDUs in A-MPDU and byte size
      rhd_cnt = mu_mimo.ampdu_frame[mu_mimo_userid].mpdu_frame_type.size()*4*`RHD_LEN;
      rpd_cnt = mu_mimo.ampdu_frame[mu_mimo_userid].size();
      // prepare SRAM for Rx
      prepare_sram_for_rx(rhd_cnt + rpd_cnt);
      m_regmodel.get_reg_value(rdata, "RXBUF1RDPTRREG");
      rx_buff_rd_ptr = rdata + rhd_header;

      // set group membership
      phy_set_mu_group_memb(mu_mimo);
      phy_set_mu_userpos(mu_mimo);

      // receive ACK and wait for interrupt Rx trigger
      fork
        // if frames if going to be received trigger will occur,
        // if frames if filtered event will occur
        fork
          wait_mac_rx_trigger();
          rx_frame_filtered();
        join_any

        begin
          // PHY sends frame to MAC core
          drive_phy_adc(mu_mimo);
          wait_tx_end(); // wait ACK
          drive_noise_adc();
        end
      join

      // only if frame was stored in SRAM do read
      if (!stop_pooling) begin
        `uvm_info(get_type_name(),
        "MAC core received frame and stored it to SRAM", UVM_LOW)
        rd_frame_from_sram(rx_buff_rd_ptr);
      end

      // clear interrupt flag
      m_regmodel.set_field_value(1'b1, "clearrxBuffer1Trigger", "TXRXINTEVENTCLEARREG");
      m_regmodel.set_field_value(1'b1, "cleartimerRxTrigger", "TXRXINTEVENTCLEARREG");
      m_regmodel.set_field_value(1'b1, "clearrxBuffer2Trigger", "TXRXINTEVENTCLEARREG");

      // To give a time for interrupt comparison
      insert_idle_cycles(10);

    end //for

  endtask : body

endclass : wlan_bf_mu_rx_seq


`endif //WLAN_BF_MU_RX_SEQ_SV
