//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef WLAN_SEQ_BASE_SV
`define WLAN_SEQ_BASE_SV


class wlan_seq_base extends uvm_sequence #(uvm_sequence_item);

  `uvm_object_utils(wlan_seq_base)

  wlan_virtual_sequencer     m_vsqr;      // handle to the virtual sequencer
  wlan_config                m_cfg;       // handle to configuration
  mdm_data_model             m_mdm_cfg;   // Handle to matlab model
  KeyStorageRAM              m_ksr;       // handle to key storage RAM
  TOP_register_block         m_regmodel;  // handle to register model

  // for register access
  uvm_status_e               status;
  uvm_reg_data_t             rdata, wdata;

  PPDU_frame                 frame;       // handle to PPDU frame which can be used in MAC-PHY seq

  // addresses of device and BSS generated randomly
  mac_address_t              bss_addr;
  mac_address_t              dev_addr;
  // RHD/RPD header and footer sizes
  bit [7:0]                  rhd_header, rhd_footer;
  bit [7:0]                  rpd_header, rpd_footer;
  // MISC
  bit                        ldpc_cfg_done;
  rand int                   carrier_freq;
  rand bit [7:0]             he_tb_ru_allocation;
  bit                        sram_wrapp;

  //--------------------------------------------------------------------
  // variables needed for running Matlab model
  //--------------------------------------------------------------------
  direction_e                direction = Rx;
  rf_gpio_t                  rf_gpio = `GPIO_WIDTH'd27;
  bit                        agcBypass = 0;

  // events that are used in MAC core tests
  uvm_event_pool             mac_pool;
  uvm_event                  no_resp_trigger;
  uvm_event                  frame_filtered_trigger;

  // flag used in wait for trigger task for stoping pooling process
  bit                        stop_pooling = 0;
  bit                        setgain_timeout = 0;

  bit [31:0]                 bfmer_mem[int];  // replica of BFMER memory

  //-------------------------------------------------
  // list of sequences
  //-------------------------------------------------
  mac_tx_frame_seq              mac_tx;
  mac_rx_frame_seq              mac_rx;
  phy_tx_frame_seq              phy_tx;
  phy_rx_frame_seq              phy_rx;
  phy_rx_frame_err_seq          phy_rx_err;
  modem_stream_gen_samples_seq  adc_gen_seq;
  modem_stream_gen_noise_seq    adc_noise_seq;
  rui_gen_samples_seq           adc_noagc_gen_seq;
  //-------------------------------------------------

  //-------------------------------------------------
  // list of functions and tasks in sequence base
  //-------------------------------------------------
  // function int get_random_ksr_index(ref PPDU_frame frm, input user=0);
  // function bit is_agc_off();
  // function string get_reg_parent(uvm_reg dut_reg);
  // function bit skip_reg(uvm_reg dut_reg);
  // function bit responde_to_frame(ref PPDU_frame frm, input mac_address_t dev_addr);
  // function int get_max_he_length(ref PPDU_frame ppdu);
  // function void set_he_ctrl_ru_allocation(ref PPDU_frame frm, ref he_control_s he_ctrl);
  // task init_mac_core();
  // task create_dev_and_bss_addr();
  // task change_mac_core_state(bit [3:0] state);
  // task wait_mac_core_state_movement(mac_state_e state);
  // task create_keystorageRAM(bit custom = 0, int vlan_cnt = 1, int dev_cnt = 1);
  // task overwrite_keystorageRAM();
  // task search_keystorageRAM(bit [47:0] mac_addr, ref bit search_err);
  // task read_keystorageRAM(bit [9:0] key_index);
  // task set_ac_interrupts(access_category_e ac);
  // task clear_ac_interrupts(access_category_e ac);
  // task set_DMA_for_tx(access_category_e access_cat);
  // task wait_mac_tx_trigger(access_category_e access_cat);
  // task wait_any_mac_tx_trigger(ref access_category_e access_cat);
  // task wait_mac_rx_trigger();
  // task rx_frame_filtered();
  // task create_ppdu_response_frame(ref PPDU_frame frm);
  // task system_reset();
  // task insert_idle_cycles(int t);
  // task wr_frame_to_sram(PPDU_frame            frm,
  // task rd_frame_from_sram(bit [31:0] addr);
  // task prepare_sram_for_rx(int size, bit smaller_rx_segment = 0);
  // task prepare_sram_priority_rx(int size);
  // task rd_tx_status_from_sram(bit [31:0] head, bit skip_athd = 0);
  // task get_next_rhd_ptr(input bit [31:0] head, output bit [31:0] next);
  // task write_word_sram(bit [31:0] addr, bit [31:0] data);
  // task read_word_sram(input bit [31:0] addr, output bit [31:0] data);
  // task write_word_sram_via_ahb(bit [31:0] a, bit [31:0] d);
  // task read_word_sram_via_ahb(bit [31:0] a, output bit [31:0] d);
  // task phy_set_base_config();
  // task riu_set_control_channel();
  // task ldpc_cfg_wr_via_ahb(bit [31:0] a, bit [31:0] d);
  // task ldpc_cfg_rd_via_ahb_via_ahb(bit [31:0] a, bit [31:0] d);
  // task phy_set_ldpc(bit ldpc_en, string file_name = "ldpc_cfg_ram.bin");
  // task phy_set_mu_group_memb(ref PPDU_frame ppdu);
  // task phy_set_mu_userpos(ref PPDU_frame ppdu);
  // task phy_get_mu_userpos(ref PPDU_frame ppdu, ref bit [1:0] userpos);
  // task set_coex_bt_tx(bit val);
  // task set_coex_bt_rx(bit val);
  // task set_coex_bt_event(bit val);
  // task set_coex_bt_pti(bit [3:0] val);
  // task set_coex_bt_channel(bit [6:0] val);
  // task set_coex_bt_bw(bit val);
  // task prepare_lli(
  // task wait_for_lli_irq(bit[15:0] lli_interrupts);
  // task ack_dma_irq(
  // task upstream_dma_transfer(
  // task downstream_dma_transfer(
  // task midstream_dma_transfer(
  // task init_axi_agent();
  // task refresh_RF_data();
  // task drive_phy_adc(ref PPDU_frame frm,
  // task drive_noise_adc();
  // task wait_tx_end();
  // task wait_rx_end();
  // task bfmer_mem_wr(bit [31:0] a, bit [31:0] d);
  // task store_bfmer_report_ahb(ref PPDU_frame frm,
  // task sw_time_on_air_calculation(ref PPDU_frame frm, output [15:0] time_on_air);
  // task overwrite_regmodel();
  //-------------------------------------------------

  //---------------------------------------------
  // constraints
  //---------------------------------------------
  constraint c_carrier_freq {
    carrier_freq inside {2412, 2417, 2422, 2427, 2432, 2437, 2442,
                         2447, 2452, 2457, 2462, 2467, 2472, 2484,
                         5180, 5200, 5220, 5240, 5260, 5280, 5300,
                         5320, 5500, 5520, 5540, 5560, 5580, 5600,
                         5620, 5640, 5560, 5580, 5600, 5620, 5640,
                         5660, 5680, 5700, 5720, 5745, 5765, 5785,
                         5805, 5825};
  }
  //---------------------------------------------

  function new (string name = "wlan_seq_base");
    super.new (name);

    // get event pool and events
    mac_pool = uvm_event_pool::get_global_pool();
    no_resp_trigger = mac_pool.get("no_resp_trigger");
    frame_filtered_trigger = mac_pool.get("frame_filtered_trigger");

    ldpc_cfg_done = 0;
    carrier_freq = 2412;
  endfunction : new

  virtual task pre_body();
  endtask : pre_body

  virtual task body();
    if (!$cast(m_vsqr, m_sequencer))
      `uvm_fatal(get_type_name(), "Virtual sequencer pointer cast failed")

    // Get configuration
    if (!uvm_config_db #(wlan_config)::get(null, "", "wlan_config", m_cfg))
      `uvm_fatal(get_type_name(), "WLAN configuration not set in configuration DB!!!")

    // Initialize handle to matlab model
    if (m_cfg.has_mdm_data_model) begin
      if (!uvm_config_db #(mdm_data_model)::get(null, "", "mdm_data_model", m_mdm_cfg))
        `uvm_fatal(get_type_name(), "Matlab model object missing!")
    end

    // Initialize handle to Key storage RAM
    if (m_cfg.has_keystorageram)
      if (!uvm_config_db #(KeyStorageRAM)::get(null, "", "KeyStorageRAM", m_ksr))
        `uvm_fatal(get_type_name(), "Key storage RAM object missing!")

    // Initialize handle to register model
    if (m_cfg.has_regmodel)
      if (!uvm_config_db #(TOP_register_block)::get(null, "", "TOP_register_block", m_regmodel))
        `uvm_fatal(get_type_name(), "WLAN configuration not set in configuration DB!!!")

    // initial tasks that need to be performed
    // System-wide reset
    system_reset();

    // Initialize PHY registers
    if (m_cfg.modem_init_done == 1'b0) begin
      phy_set_base_config();
`ifdef RW_NX_LDPC_DEC
      phy_set_ldpc();
`endif //RW_NX_LDPC_DEC
    end
`ifndef STANDALONE_PHY
    // initialize MAC core registers
    if (m_cfg.mac_core_init_done == 1'b0) begin
      init_mac_core();
    end
`endif
  endtask : body

  virtual task post_body();
  endtask : post_body

  //==================================================================
  //================= common functions and tasks =====================
  //==================================================================

  //---------------------------------------------------
  // initialize MAC core before any Tx and Rx action
  //---------------------------------------------------
  task init_mac_core();
    uvm_reg_data_t                      mac_ctrl_1;
    MACCORE_reg_block_pkg::RXCNTRL2     rxcntrl2;
    MACPL_reg_block_pkg::RXBUFCONFIGREG rxbufconfigreg;

    `uvm_info(get_type_name(), "MAC CORE initialization start", UVM_LOW)
    m_regmodel.set_reg_value(32'h0000_0153, "RATESREG");
    m_regmodel.set_reg_value(32'h0005_1111, "HTMCSREG");
    m_regmodel.set_field_value(1'b1, "rxFlowCntrlEn", "MACERRRECCNTRLREG");
`ifdef STANDALONE_MAC
    // bypass PHY HW
    m_regmodel.set_field_value(1, "BYPASS", "CTRL");
`endif

    wdata[0]  = 1'b1; // BSS type
    wdata[1]  = 1'b0; // AP
    wdata[2]  = 1'b0; // power management
    wdata[3]  = 1'b0; // contention free period aware
    wdata[7]  = 1'b1; // active clock gating
    wdata[8]  = 1'b0; // disable ACK response
    wdata[9]  = 1'b0; // disable CTS response
    wdata[10] = 1'b0; // disable BAR response
    wdata[11] = 1'b0; // set to 0, if WLAN test is running
    wdata[12] = 1'b0; // MIB table reset
    wdata[13] = 1'b0; // KSR table reset
    wdata[16:14] = 3'd6; // ABGN mode
    wdata[24] = 1'b0; // TSF updated by SW
    wdata[25] = 1'b0; // TSF management disable
    wdata[26] = 1'b0; // RIFS enable
    m_regmodel.set_reg_value(wdata, "MACCNTRL1REG");
    m_regmodel.get_reg_value(mac_ctrl_1, "MACCNTRL1REG");

    `uvm_info(get_type_name(), "Reset MIB table", UVM_LOW)
    wdata[12] = 1'b1; // MIB table reset
    m_regmodel.set_reg_value(wdata, "MACCNTRL1REG");

    insert_idle_cycles(2);

    wdata[12] = 1'b0; // MIB table reset
    m_regmodel.set_reg_value(wdata, "MACCNTRL1REG");

    `uvm_info(get_type_name(), "Reset Key Storage RAM table", UVM_LOW)
    wdata[13] = 1'b1; // KSR table reset
    m_regmodel.set_reg_value(wdata, "MACCNTRL1REG");

    insert_idle_cycles(2);

    wdata[13] = 1'b0; // KSR table reset
    m_regmodel.set_reg_value(wdata, "MACCNTRL1REG");

    `uvm_info(get_type_name(), "Set TSF timer", UVM_LOW)
    wdata = $urandom();
    m_regmodel.set_reg_value(wdata, "TSFLOREG");
    wdata = $urandom();
    m_regmodel.set_reg_value(wdata, "TSFHIREG");
    // set field value tsfUpdatedBySW
    mac_ctrl_1[24] = 1'b1;
    m_regmodel.set_reg_value(mac_ctrl_1, "MACCNTRL1REG");

    // set RX control
    wdata = 32'h7FC7_FFD8;
    m_regmodel.set_reg_value(wdata, "RXCNTRLREG");

    // set RXCNTRL2
    $cast(rxcntrl2, m_regmodel.get_reg_by_name("RXCNTRL2"));
    assert (rxcntrl2.randomize() with {
      dataMHStoredwithPld.value inside {[0:1]};
      mgtMHStoredwithPld.value  inside {[0:1]};
      ctrlMHStoredwithPld.value inside {[0:1]};
      partialUnwrapSize.value   inside {[10:50]};
      dataFrmWrapMode.value     inside {[0:2]};
      mgtFrmWrapMode.value      inside {[0:2]};
      ctrlFrmWrapMode.value     inside {[0:2]};
    });
    rxcntrl2.update(.status(status),.path(UVM_FRONTDOOR));

    // randomize RHD/RPD header and footer size
    $cast(rxbufconfigreg, m_regmodel.get_reg_by_name("RXBUFCONFIGREG"));
    assert (rxbufconfigreg.randomize() with {
      rxBufRPDFooter.value < 16;
      rxBufRPDHeader.value < 16;
      rxBufRHDFooter.value < 16;
      rxBufRHDHeader.value < 16;
    });
    rxbufconfigreg.update(.status(status),.path(UVM_FRONTDOOR));

`ifdef RW_NX_CHBW20
    // Set the MAX Bandwidth to 20MHz
    m_regmodel.set_field_value(2'b00, "maxSupportedBW", "TXBWCNTRLREG");
`elsif RW_NX_CHBW4020
    // Set the MAX Bandwidth to 40MHz
    m_regmodel.set_field_value(2'b01, "maxSupportedBW", "TXBWCNTRLREG");
`elsif RW_NX_CHBW804020
    // Set the MAX Bandwidth to 80MHz
    m_regmodel.set_field_value(2'b10, "maxSupportedBW", "TXBWCNTRLREG");
`endif

    // enable doppler and DCM support in MAC
    m_regmodel.set_field_value(1'b1, "dopplerSupport");
    m_regmodel.set_field_value(1'b1, "dcmSupport");

    `uvm_info(get_type_name(), "MAC CORE initialization done!", UVM_LOW)
    m_cfg.mac_core_init_done = 1'b1;
  endtask : init_mac_core

  //---------------------------------------------------
  // create random unicast address for device and BSS
  // and store them in registers
  //---------------------------------------------------
  task create_dev_and_bss_addr();

    bss_addr[31: 0] = $urandom();
    bss_addr[47:32] = $urandom();
    bss_addr[0]     = 0;

    // set unicast MAC address
    dev_addr[31: 0] = $urandom();
    dev_addr[47:32] = $urandom();
    dev_addr[0]     = 0;

    wdata = dev_addr[47:32];
    m_regmodel.set_reg_value(wdata, "MACADDRHIREG");
    wdata = dev_addr[31:0];
    m_regmodel.set_reg_value(wdata, "MACADDRLOWREG");
    wdata = bss_addr[47:32];
    m_regmodel.set_reg_value(wdata, "BSSIDHIREG");
    wdata = bss_addr[31:0];
    m_regmodel.set_reg_value(wdata, "BSSIDLOWREG");

  endtask : create_dev_and_bss_addr

  //---------------------------------------------------
  // change state of MAC core
  //---------------------------------------------------
  task change_mac_core_state(bit [3:0] state);
    string s;

    m_regmodel.get_field_value(rdata, "currentState", "STATECNTRLREG");
    // convert state to string
    case (state)
      `IDLE_STATE  : s = "IDLE";
      `DOZE_STATE  : s = "DOZE";
      `ACTIVE_STATE: s = "ACTIVE";
      default: s = "UNKNOWN";
    endcase

    if (rdata != `IDLE_STATE && state != `IDLE_STATE) begin
      `uvm_error(get_type_name(), "Changing state from non IDLE to non IDLE is forbidden")
    end
    else if (rdata == `IDLE_STATE) begin
      wdata = state;
      m_regmodel.set_field_value(wdata, "nextState", "STATECNTRLREG");
      `uvm_info(get_type_name(), $sformatf("MAC core state change IDLE -> %s",s), UVM_LOW)
    end
    else begin // when in non IDLE state to IDLE
      wdata = state;
      m_regmodel.set_field_value(wdata, "nextState", "STATECNTRLREG");
      `uvm_info(get_type_name(), $sformatf("MAC core state change %s -> IDLE",s), UVM_LOW)
    end
  endtask : change_mac_core_state

  //---------------------------------------------------
  // check the current state of MAC core
  //---------------------------------------------------
  task wait_mac_core_state_movement(bit [3:0] state);

    do begin
      m_regmodel.get_field_value(rdata, "currentState", "STATECNTRLREG");
    end
    while(rdata[3:0] != state);

  endtask : wait_mac_core_state_movement

  //---------------------------------------------------
  // create and randomize a number of VLAN and device
  // entries inside key storage RAM, after that store
  // content to real memory via register access
  // @custom - if set to 1 task will create KSR with
  //           vlan_cnt number of VLAN stations and dev_cnt
  //           number of devices
  //---------------------------------------------------
  task create_keystorageRAM(bit custom = 0, int vlan_cnt = 1, int dev_cnt = 1);
    bit [9:0] key_index;

    // randomize Key Storage RAM
    if (custom) begin
      assert (m_ksr.randomize() with {
        m_ksr.num_of_vlan == vlan_cnt;
        m_ksr.num_of_dev  == dev_cnt;
      });
    end
    else begin
      assert (m_ksr.randomize());
    end

    `uvm_info(get_type_name(),
      $sformatf("Created KSR with VLAN num = %0d, DEV num = %0d",
      m_ksr.num_of_vlan,m_ksr.num_of_dev), UVM_LOW)

    // program VLAN key storage RAM via register access
    for (int i=0; i<m_ksr.num_of_vlan*`DEFAULT_KEY_CNT; i++) begin
      // encryption
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[31:0];
      m_regmodel.set_reg_value(wdata, "ENCRKEY0REG");
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[63:32];
      m_regmodel.set_reg_value(wdata, "ENCRKEY1REG");
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[95:64];
      m_regmodel.set_reg_value(wdata, "ENCRKEY2REG");
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[127:96];
      m_regmodel.set_reg_value(wdata, "ENCRKEY3REG");

      if (m_ksr.keyRAM[i].ctype_ram_f == 3'b100) begin
        // WPI encryption
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[31:0];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY0REG");
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[63:32];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY1REG");
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[95:64];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY2REG");
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[127:96];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY3REG");
      end
      else if (   m_ksr.keyRAM[i].ctype_ram_f inside {3'b011,3'b101}
               && m_ksr.keyRAM[i].clen_ram_f == 2'b10) begin
        // in case of CCMP/GCMP 256bit encryption
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[159:128];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY0REG");
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[191:160];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY1REG");
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[223:192];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY2REG");
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[255:224];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY3REG");
      end

      wdata = m_ksr.keyRAM[i].mac_addr_ram_f[31:0];
      m_regmodel.set_reg_value(wdata, "ENCRMACADDRLOWREG");
      wdata = m_ksr.keyRAM[i].mac_addr_ram_f[47:32];
      m_regmodel.set_reg_value(wdata, "ENCRMACADDRHIGHREG");
      // write encryption control register
      key_index = i;

      wdata[31]    = 1'b0; // newRead
      wdata[30]    = 1'b1; // newWrite
      wdata[29]    = 1'b0; // newSearch
      wdata[28]    = 1'b0; // searchError
      wdata[25:16] = key_index; // keyIndexRAM
      wdata[13:12] = m_ksr.keyRAM[i].clen_ram_f;
      wdata[10:8]  = m_ksr.keyRAM[i].ctype_ram_f;
      wdata[7:4]   = m_ksr.keyRAM[i].vlan_id_ram_f;
      wdata[3:2]   = m_ksr.keyRAM[i].spp_ram_f;
      wdata[1]     = m_ksr.keyRAM[i].use_def_key_ram_f;
      m_regmodel.set_reg_value(wdata, "ENCRCNTRLREG");

      // wait for write to finish
      do begin
        m_regmodel.get_field_value(rdata, "newWrite", "ENCRCNTRLREG");
      end while (rdata[0] == 1'b1);

    end // for

    // program device key storage RAM via register access
    for (int i=`KEYRAM_DEV_START_ADDR; i<`KEYRAM_DEV_START_ADDR+m_ksr.num_of_dev; i++) begin
      // encryption
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[31:0];
      m_regmodel.set_reg_value(wdata, "ENCRKEY0REG");
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[63:32];
      m_regmodel.set_reg_value(wdata, "ENCRKEY1REG");
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[95:64];
      m_regmodel.set_reg_value(wdata, "ENCRKEY2REG");
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[127:96];
      m_regmodel.set_reg_value(wdata, "ENCRKEY3REG");

      if (m_ksr.keyRAM[i].ctype_ram_f == 3'b100) begin
        // WPI encryption
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[31:0];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY0REG");
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[63:32];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY1REG");
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[95:64];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY2REG");
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[127:96];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY3REG");
      end
      else if (   m_ksr.keyRAM[i].ctype_ram_f inside {3'b011,3'b101}
               && m_ksr.keyRAM[i].clen_ram_f == 2'b10) begin
        // in case of CCMP/GCMP 256bit encryption
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[159:128];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY0REG");
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[191:160];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY1REG");
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[223:192];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY2REG");
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[255:224];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY3REG");
      end

      wdata = m_ksr.keyRAM[i].mac_addr_ram_f[31:0];
      m_regmodel.set_reg_value(wdata, "ENCRMACADDRLOWREG");
      wdata = m_ksr.keyRAM[i].mac_addr_ram_f[47:32];
      m_regmodel.set_reg_value(wdata, "ENCRMACADDRHIGHREG");
      // write encryption control register
      key_index = i;

      wdata[31]    = 1'b0; // newRead
      wdata[30]    = 1'b1; // newWrite
      wdata[29]    = 1'b0; // newSearch
      wdata[28]    = 1'b0; // searchError
      wdata[25:16] = key_index; // keyIndexRAM
      wdata[13:12] = m_ksr.keyRAM[i].clen_ram_f;
      wdata[10:8]  = m_ksr.keyRAM[i].ctype_ram_f;
      wdata[7:4]   = m_ksr.keyRAM[i].vlan_id_ram_f;
      wdata[3:2]   = m_ksr.keyRAM[i].spp_ram_f;
      wdata[1]     = m_ksr.keyRAM[i].use_def_key_ram_f;
      m_regmodel.set_reg_value(wdata, "ENCRCNTRLREG");

      // wait for write to finish
      do begin
        m_regmodel.get_field_value(rdata, "newWrite", "ENCRCNTRLREG");
      end while (rdata[0] == 1'b1);

    end // for
  endtask : create_keystorageRAM

  //---------------------------------------------------
  // task for overwriting, the keyStorageRAM,
  // in case custom constraining
  //---------------------------------------------------
  task overwrite_keystorageRAM();
    bit [9:0] key_index;

    `uvm_info(get_type_name(),
      $sformatf("Created KSR with VLAN num = %0d, DEV num = %0d",
      m_ksr.num_of_vlan,m_ksr.num_of_dev), UVM_LOW)

    // program VLAN key storage RAM via register access
    for (int i=0; i<m_ksr.num_of_vlan*`DEFAULT_KEY_CNT; i++) begin
      // encryption
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[31:0];
      m_regmodel.set_reg_value(wdata, "ENCRKEY0REG");
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[63:32];
      m_regmodel.set_reg_value(wdata, "ENCRKEY1REG");
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[95:64];
      m_regmodel.set_reg_value(wdata, "ENCRKEY2REG");
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[127:96];
      m_regmodel.set_reg_value(wdata, "ENCRKEY3REG");

      if (m_ksr.keyRAM[i].ctype_ram_f == 3'b100) begin
        // WPI encryption
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[31:0];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY0REG");
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[63:32];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY1REG");
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[95:64];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY2REG");
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[127:96];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY3REG");
      end
      else if (   m_ksr.keyRAM[i].ctype_ram_f inside {3'b011,3'b101}
               && m_ksr.keyRAM[i].clen_ram_f == 2'b10) begin
        // in case of CCMP/GCMP 256bit encryption
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[159:128];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY0REG");
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[191:160];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY1REG");
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[223:192];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY2REG");
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[255:224];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY3REG");
      end

      wdata = m_ksr.keyRAM[i].mac_addr_ram_f[31:0];
      m_regmodel.set_reg_value(wdata, "ENCRMACADDRLOWREG");
      wdata = m_ksr.keyRAM[i].mac_addr_ram_f[47:32];
      m_regmodel.set_reg_value(wdata, "ENCRMACADDRHIGHREG");
      // write encryption control register
      key_index = i;

      wdata[31]    = 1'b0; // newRead
      wdata[30]    = 1'b1; // newWrite
      wdata[29]    = 1'b0; // newSearch
      wdata[28]    = 1'b0; // searchError
      wdata[25:16] = key_index; // keyIndexRAM
      wdata[13:12] = m_ksr.keyRAM[i].clen_ram_f;
      wdata[10:8]  = m_ksr.keyRAM[i].ctype_ram_f;
      wdata[7:4]   = m_ksr.keyRAM[i].vlan_id_ram_f;
      wdata[3:2]   = m_ksr.keyRAM[i].spp_ram_f;
      wdata[1]     = m_ksr.keyRAM[i].use_def_key_ram_f;
      m_regmodel.set_reg_value(wdata, "ENCRCNTRLREG");

      // wait for write to finish
      do begin
        m_regmodel.get_field_value(rdata, "newWrite", "ENCRCNTRLREG");
      end while (rdata[0] == 1'b1);

    end // for

    // program device key storage RAM via register access
    for (int i=`KEYRAM_DEV_START_ADDR; i<`KEYRAM_DEV_START_ADDR+m_ksr.num_of_dev; i++) begin
      // encryption
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[31:0];
      m_regmodel.set_reg_value(wdata, "ENCRKEY0REG");
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[63:32];
      m_regmodel.set_reg_value(wdata, "ENCRKEY1REG");
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[95:64];
      m_regmodel.set_reg_value(wdata, "ENCRKEY2REG");
      wdata = m_ksr.keyRAM[i].encr_key_ram_f[127:96];
      m_regmodel.set_reg_value(wdata, "ENCRKEY3REG");

      if (m_ksr.keyRAM[i].ctype_ram_f == 3'b100) begin
        // WPI encryption
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[31:0];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY0REG");
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[63:32];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY1REG");
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[95:64];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY2REG");
        wdata = m_ksr.keyRAM[i].int_wpi_key_ram_f[127:96];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY3REG");
      end
      else if (   m_ksr.keyRAM[i].ctype_ram_f inside {3'b011,3'b101}
               && m_ksr.keyRAM[i].clen_ram_f == 2'b10) begin
        // in case of CCMP/GCMP 256bit encryption
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[159:128];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY0REG");
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[191:160];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY1REG");
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[223:192];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY2REG");
        wdata = m_ksr.keyRAM[i].encr_key_ram_f[255:224];
        m_regmodel.set_reg_value(wdata, "ENCRWPIINTKEY3REG");
      end

      wdata = m_ksr.keyRAM[i].mac_addr_ram_f[31:0];
      m_regmodel.set_reg_value(wdata, "ENCRMACADDRLOWREG");
      wdata = m_ksr.keyRAM[i].mac_addr_ram_f[47:32];
      m_regmodel.set_reg_value(wdata, "ENCRMACADDRHIGHREG");
      // write encryption control register
      key_index = i;

      wdata[31]    = 1'b0; // newRead
      wdata[30]    = 1'b1; // newWrite
      wdata[29]    = 1'b0; // newSearch
      wdata[28]    = 1'b0; // searchError
      wdata[25:16] = key_index; // keyIndexRAM
      wdata[13:12] = m_ksr.keyRAM[i].clen_ram_f;
      wdata[10:8]  = m_ksr.keyRAM[i].ctype_ram_f;
      wdata[7:4]   = m_ksr.keyRAM[i].vlan_id_ram_f;
      wdata[3:2]   = m_ksr.keyRAM[i].spp_ram_f;
      wdata[1]     = m_ksr.keyRAM[i].use_def_key_ram_f;
      m_regmodel.set_reg_value(wdata, "ENCRCNTRLREG");

      // wait for write to finish
      do begin
        m_regmodel.get_field_value(rdata, "newWrite", "ENCRCNTRLREG");
      end while (rdata[0] == 1'b1);

    end // for
  endtask : overwrite_keystorageRAM

  //---------------------------------------------------
  // search for MAC address in KSR
  //---------------------------------------------------
  task search_keystorageRAM(bit [47:0] mac_addr, ref bit search_err);

    wdata = mac_addr[31:0];
    m_regmodel.set_reg_value(wdata, "ENCRMACADDRLOWREG");
    wdata = mac_addr[47:32];
    m_regmodel.set_reg_value(wdata, "ENCRMACADDRHIGHREG");

    // check if write was successful
    m_regmodel.set_field_value(1'b1, "newSearch", "ENCRCNTRLREG");
    // wait for search to finish
    do begin
      m_regmodel.get_field_value(rdata, "newSearch", "ENCRCNTRLREG");
    end while (rdata[0] == 1'b1);

    m_regmodel.get_field_value(rdata, "searchError", "ENCRCNTRLREG");
    search_err = rdata[0];
  endtask : search_keystorageRAM

  //---------------------------------------------------
  // read from KSR by key index
  //---------------------------------------------------
  task read_keystorageRAM(bit [9:0] key_index);

    m_regmodel.set_field_value(key_index, "keyIndexRAM", "ENCRCNTRLREG");

    // check if write was successful
    m_regmodel.set_field_value(1'b1, "newRead", "ENCRCNTRLREG");
    // wait for search to finish
    do begin
      m_regmodel.get_field_value(rdata, "newRead", "ENCRCNTRLREG");
    end while (rdata[0] == 1'b1);

  endtask : read_keystorageRAM

  //---------------------------------------------------
  // when KSR is created get one valid device index
  //---------------------------------------------------
  function int get_random_ksr_index(ref PPDU_frame frm, input user=0);
    int ret;
    cipher_type_e security_type;

    if (   frm.ampdu_frame[user].mpdu_frame_type[0][5:4] == `DATA_MPDU
        || frm.ampdu_frame[user].mpdu_frame_type[0] == ERROR_FRAME
       ) begin

      do begin
        ret = $urandom_range(`KEYRAM_DEV_START_ADDR,
                             `KEYRAM_DEV_START_ADDR+m_ksr.num_of_dev-1);
        security_type = cipher_type_e'(m_ksr.keyRAM[ret].ctype_ram_f);
        // prevent WEP/TKIP in HT/VHT/HE frames and NON_HT_DUP_OFDM
        // prevent WAPI in HE frames
      end while (   (   (security_type inside {WEP,TKIP} && frm.ppdu_format != NON_HT)
                     || (security_type == WAPI           && frm.ppdu_format inside {HE_SU,HE_EXT_SU,HE_MU,HE_TB})
                    )
                 && frm.is_protected(user));
    end
    else if (frm.ampdu_frame[user].mpdu_frame_type[0][5:4] == `MANAGEMENT_MPDU) begin
      // Management frames can be encrypted only using CCMP
      do begin
        ret = $urandom_range(`KEYRAM_DEV_START_ADDR,
                             `KEYRAM_DEV_START_ADDR+m_ksr.num_of_dev-1);
        security_type = cipher_type_e'(m_ksr.keyRAM[ret].ctype_ram_f);
      end while (   security_type != CCMP
                 && frm.is_protected(user));
    end
    else begin
      ret = $urandom_range(`KEYRAM_DEV_START_ADDR,
                           `KEYRAM_DEV_START_ADDR+m_ksr.num_of_dev-1);
    end

    `uvm_info(get_type_name(), $sformatf("KSR index used: %0d",ret), UVM_LOW)
    return ret;
  endfunction : get_random_ksr_index

  //---------------------------------------------------
  // configure Tx interrupt masks for proper access category
  //---------------------------------------------------
  task set_ac_interrupts(access_category_e ac);
    case (ac)
      AC_BK : begin
        // set Tx interrupt masks for AC0
        m_regmodel.set_field_value(1'b1, "maskac0TxTrigger", "TXRXINTUNMASKREG");
        m_regmodel.set_field_value(1'b1, "maskac0ProtTrigger", "TXRXINTUNMASKREG");
        m_regmodel.set_field_value(1'b1, "maskac0TxBufTrigger", "TXRXINTUNMASKREG");
      end
      AC_BE : begin
        // set Tx interrupt masks for AC1
        m_regmodel.set_field_value(1'b1, "maskac1TxTrigger", "TXRXINTUNMASKREG");
        m_regmodel.set_field_value(1'b1, "maskac1ProtTrigger", "TXRXINTUNMASKREG");
        m_regmodel.set_field_value(1'b1, "maskac1TxBufTrigger", "TXRXINTUNMASKREG");
      end
      AC_VI : begin
        // set Tx interrupt masks for AC2
        m_regmodel.set_field_value(1'b1, "maskac2TxTrigger", "TXRXINTUNMASKREG");
        m_regmodel.set_field_value(1'b1, "maskac2ProtTrigger", "TXRXINTUNMASKREG");
        m_regmodel.set_field_value(1'b1, "maskac2TxBufTrigger", "TXRXINTUNMASKREG");
      end
      AC_VO : begin
        // set Tx interrupt masks for AC3
        m_regmodel.set_field_value(1'b1, "maskac3TxTrigger", "TXRXINTUNMASKREG");
        m_regmodel.set_field_value(1'b1, "maskac3ProtTrigger", "TXRXINTUNMASKREG");
        m_regmodel.set_field_value(1'b1, "maskac3TxBufTrigger", "TXRXINTUNMASKREG");
      end
      AC_BEACON: begin
        m_regmodel.set_field_value(1'b1, "maskbcnTxTrigger", "TXRXINTUNMASKREG");
        m_regmodel.set_field_value(1'b1, "maskimpPriTBTT", "GENINTUNMASKREG");
      end
      AC_TB: begin
        // set Tx interrupt masks for trigger based
        m_regmodel.set_field_value(1'b1, "masktbTxTrigger", "TXRXINTUNMASKREG");
        m_regmodel.set_field_value(1'b1, "masktbProtTrigger", "TXRXINTUNMASKREG");
      end
    endcase

    m_cfg.access_category_set = ac;
  endtask : set_ac_interrupts

  //---------------------------------------------------
  // clear Tx interrupt masks for proper access category
  //---------------------------------------------------
  task clear_ac_interrupts(access_category_e ac);
    case (ac)
      AC_BK : begin
        // set Tx interrupt clears for AC0
        m_regmodel.set_field_value(1'b1, "clearac0TxTrigger", "TXRXINTEVENTCLEARREG");
        m_regmodel.set_field_value(1'b1, "clearac0ProtTrigger", "TXRXINTEVENTCLEARREG");
        m_regmodel.set_field_value(1'b1, "clearac0TxBufTrigger", "TXRXINTEVENTCLEARREG");
      end
      AC_BE : begin
        // set Tx interrupt clears for AC1
        m_regmodel.set_field_value(1'b1, "clearac1TxTrigger", "TXRXINTEVENTCLEARREG");
        m_regmodel.set_field_value(1'b1, "clearac1ProtTrigger", "TXRXINTEVENTCLEARREG");
        m_regmodel.set_field_value(1'b1, "clearac1TxBufTrigger", "TXRXINTEVENTCLEARREG");
      end
      AC_VI : begin
        // set Tx interrupt clears for AC2
        m_regmodel.set_field_value(1'b1, "clearac2TxTrigger", "TXRXINTEVENTCLEARREG");
        m_regmodel.set_field_value(1'b1, "clearac2ProtTrigger", "TXRXINTEVENTCLEARREG");
        m_regmodel.set_field_value(1'b1, "clearac2TxBufTrigger", "TXRXINTEVENTCLEARREG");
      end
      AC_VO : begin
        // set Tx interrupt clears for AC3
        m_regmodel.set_field_value(1'b1, "clearac3TxTrigger", "TXRXINTEVENTCLEARREG");
        m_regmodel.set_field_value(1'b1, "clearac3ProtTrigger", "TXRXINTEVENTCLEARREG");
        m_regmodel.set_field_value(1'b1, "clearac3TxBufTrigger", "TXRXINTEVENTCLEARREG");
      end
      AC_BEACON: begin
        m_regmodel.set_field_value(1'b1, "clearbcnTxTrigger", "TXRXINTEVENTCLEARREG");
        m_regmodel.set_field_value(1'b1, "clearimpPriTBTT", "GENINTEVENTCLEARREG");
      end
      AC_TB: begin
        // set Tx interrupt masks for trigger based
        m_regmodel.set_field_value(1'b1, "cleartbTxTrigger", "TXRXINTEVENTCLEARREG");
        m_regmodel.set_field_value(1'b1, "cleartbProtTrigger", "TXRXINTEVENTCLEARREG");
      end
    endcase
  endtask : clear_ac_interrupts

  //---------------------------------------------------
  // configure registers to transmit frame
  //---------------------------------------------------
  task set_DMA_for_tx(access_category_e access_cat);
    string ac;

    // determine which access category is set
    case (access_cat)
      AC_BK    : ac = "ac0";
      AC_BE    : ac = "ac1";
      AC_VI    : ac = "ac2";
      AC_VO    : ac = "ac3";
      AC_BEACON: ac = "bcn";
      AC_TB    : ac = "tb";
    endcase

    // set head pointer for AC
    wdata = m_cfg.m_sram_cfg.thd_head_ptr;
    m_regmodel.set_reg_value(wdata, $sformatf("TX%sHEADPTRREG",ac.toupper()));
    // set new head
    m_regmodel.set_field_value(1'b1, $sformatf("settx%sNewHead",
    (ac=="bcn") ? "Bcn" : ac.toupper()), "DMACNTRLSETREG");
  endtask : set_DMA_for_tx

  //---------------------------------------------------
  // wait for Tx interrupt trigger to occur
  //---------------------------------------------------
  task wait_mac_tx_trigger(access_category_e access_cat);
    string ac;
    // determine which access category is set
    case (access_cat)
      AC_BK    : ac = "ac0";
      AC_BE    : ac = "ac1";
      AC_VI    : ac = "ac2";
      AC_VO    : ac = "ac3";
      AC_BEACON: ac = "bcn";
      AC_TB    : ac = "tb";
    endcase

    stop_pooling = 0;
    begin : TX_TRIGGER
      do begin
        m_regmodel.get_field_value(rdata, $sformatf("set%sTxTrigger",ac), "TXRXINTEVENTSETREG");
      end while (rdata[0] == 1'b0 && !stop_pooling);
      `uvm_info(get_type_name(), $sformatf("TX trigger for %s occured",ac.toupper()),UVM_LOW)
    end
  endtask : wait_mac_tx_trigger

  //---------------------------------------------------
  // wait for any Tx interrupt trigger to occur,
  // returns access category that occured
  //---------------------------------------------------
  task wait_any_mac_tx_trigger(ref access_category_e access_cat);
    string ac;

    stop_pooling = 0;
    begin : ANY_TX_TRIGGER
      do begin
        m_regmodel.get_reg_value(rdata, "TXRXINTEVENTSETREG");
      end while (rdata[11:6] == 6'b000000 && !stop_pooling);

      // determine which access category is set
      case (rdata[11:6])
        6'b000001 : begin access_cat = AC_BK; ac = "AC_BK"; end
        6'b000010 : begin access_cat = AC_BE; ac = "AC_BE"; end
        6'b000100 : begin access_cat = AC_VI; ac = "AC_VI"; end
        6'b001000 : begin access_cat = AC_VO; ac = "AC_VO"; end
        6'b010000 : begin access_cat = AC_BEACON; ac = "AC_BEACON"; end
        6'b100000 : begin access_cat = AC_TB; ac = "AC_TB"; end
        default  : ac = "UNKNOWN";
      endcase
      `uvm_info(get_type_name(), $sformatf("TX trigger for %s occured",ac),UVM_LOW)
    end
  endtask : wait_any_mac_tx_trigger

  //---------------------------------------------------
  // wait for Rx interrupt trigger to occur for buffer
  // 1 or 2
  //---------------------------------------------------
  task wait_mac_rx_trigger();
    stop_pooling = 0;
    begin : RX_TRIGGER
      do begin
        m_regmodel.get_reg_value(rdata, "TXRXINTEVENTSETREG");
      end while (rdata[16] == 1'b0 && rdata[18] == 1'b0 && !stop_pooling);
    end
    `uvm_info(get_type_name(),$sformatf("MAC RX Buffer %0d trigger %s",
    (rdata[16] == 1'b1) ? 1 : (rdata[18] == 1'b1) ? 2 : 0,(stop_pooling)?"stopped":"occured"),UVM_DEBUG)
  endtask : wait_mac_rx_trigger

  //---------------------------------------------------
  // wait for trigger based prot trigger
  //---------------------------------------------------
  task wait_mac_tb_trigger();
    stop_pooling = 0;
    begin : TB_TRIGGER
      do begin
        m_regmodel.get_field_value(rdata, "settbProtTrigger", "TXRXINTEVENTSETREG");
      end while (rdata[0] == 1'b0 && !stop_pooling);
    end
    `uvm_info(get_type_name(),$sformatf("MAC TB prot trigger %s",(stop_pooling)?"stopped":"occured"),UVM_DEBUG)
  endtask : wait_mac_tb_trigger

  //---------------------------------------------------
  // taks will wait for Rx scoreboard to filter received
  // frame and then to stop processes that are pooling
  // status registers
  //---------------------------------------------------
  task rx_frame_filtered();
    frame_filtered_trigger.wait_trigger();
    stop_pooling = 1;
    `uvm_info(get_type_name(),$sformatf("RX frame filtered"),UVM_DEBUG)
  endtask : rx_frame_filtered

  //---------------------------------------------------
  // create response frame for MAC, setting proper preamble
  // header and duration of frame
  //---------------------------------------------------
  task create_ppdu_response_frame(ref PPDU_frame frm);
    bit [2:0]    abgn_mode;
    bit [31:0]   basic_rates;
    bit [31:0]   htmcs;
    bit [31:0]   rdata;
    int          dur;

    // get register values needed for calculations
    rdata = m_regmodel.get_mirrored_reg_value("MACCNTRL1REG");
    abgn_mode = rdata[16:14];
    basic_rates = m_regmodel.get_mirrored_reg_value("RATESREG");
    htmcs = m_regmodel.get_mirrored_reg_value("HTMCSREG");
    // when test sequence needs to responde with CTS preamble
    // header setup needs to stay like in trasmitted frame
    if (frm.ampdu_frame[0].mpdu_frame_type[0] == CTS) begin
      // make Rx vector same as Tx vector when responding to RTS
      frm.preamble_header.copy(m_cfg.m_mac_phy_cfg.pre_hdr);

      // when TX frame sent in HT, response CTS needs to be in HT_MF
      if (frm.preamble_header.format_mod inside {HT_GF,VHT}) begin
`ifdef RW_SUPPORT_AX
        frm.preamble_header.format_mod = HT_MF;
`endif
      end
    end
    else if (frm.ampdu_frame[0].mpdu_frame_type[0] inside {ACK, BLOCK_ACK}) begin
      frm.preamble_header.format_mod = (m_cfg.m_mac_phy_cfg.pre_hdr.ch_bw == 2'b01)
                                       ? NON_HT_DUP_OFDM
                                       : NON_HT;

      if (m_cfg.m_mac_phy_cfg.pre_hdr.format_mod inside {HT_GF,HT_MF,VHT}) begin
        frm.preamble_header.leg_rate = bss_basic_rate_set(
          // legacy rate input
          get_legacy_rate_from_mcs(
            m_cfg.m_mac_phy_cfg.pre_hdr.format_mod,
            m_cfg.m_mac_phy_cfg.pre_hdr.user_header[0].mcs_f),
          // register input
          basic_rates);
      end
      else if (m_cfg.m_mac_phy_cfg.pre_hdr.format_mod inside {HE_SU,HE_EXT_SU,HE_MU,HE_TB}) begin
        frm.preamble_header.leg_rate = bss_basic_rate_set(
          // legacy rate input
          get_legacy_rate_from_mcs(
            m_cfg.m_mac_phy_cfg.pre_hdr.format_mod,
            m_cfg.m_mac_phy_cfg.pre_hdr.user_header_he[0].mcs_f),
          // register input
          basic_rates);
      end
      else begin
        frm.preamble_header.preamble_type = m_cfg.m_mac_phy_cfg.pre_hdr.preamble_type;
        frm.preamble_header.leg_rate = bss_basic_rate_set(m_cfg.m_mac_phy_cfg.pre_hdr.leg_rate,
                                                          basic_rates);
      end
      // set channel bendwith
      if (frm.preamble_header.format_mod != NON_HT) begin
        frm.preamble_header.ch_bw = m_cfg.m_mac_phy_cfg.pre_hdr.ch_bw;
      end
      else begin
        frm.preamble_header.ch_bw = 2'b00;
      end
    end

    // overwrite format modulation
    frm.ppdu_format = frm.preamble_header.format_mod;
    frm.calc_leg_ht_length();

    // calculate duration,
    // dur = TX frame duration - SIFS - Rx frame duration
    dur =   m_cfg.m_mac_phy_cfg.tx_frm_duration
          - get_sifs(abgn_mode)
          - get_ppdu_duration(frm, get_band5G(abgn_mode));
    dur = (dur < 0) ? 0 : dur;
    frm.ampdu_frame[0].set_MAC_duration(dur);

  endtask : create_ppdu_response_frame

  //--------------------------------------------------------------------
  // Generate system-wide reset
  //--------------------------------------------------------------------
  task system_reset();
    rst_random_seq rst_seq;

    rst_seq = rst_random_seq::type_id::create("rst_seq");
    rst_seq.start(m_vsqr.m_rst_sqr);
  endtask : system_reset

  //--------------------------------------------------------------------
  // insert idle cycles (wait time) in [us]
  //--------------------------------------------------------------------
  task insert_idle_cycles(int t);
    rst_idle_seq idle_seq;

    idle_seq = rst_idle_seq::type_id::create("idle_seq");
    assert (idle_seq.randomize() with {idle_seq.cycles == t;});
    idle_seq.start(m_vsqr.m_rst_sqr);
  endtask : insert_idle_cycles

  //--------------------------------------------------------------------
  // write PPDU frame to SRAM
  //--------------------------------------------------------------------
  task wr_frame_to_sram(PPDU_frame            frm,
                        mac_ctrl_info_1_s     mac_ctrl_info_1,
                        mac_ctrl_info_2_s     mac_ctrl_info_2,
                        nav_prot_frm_e        prot,
                        bit [31:0]            addr = 0);

    sram_wr_seq sram_wr_frm;

    `uvm_info(get_type_name(), $sformatf("Write frame to SRAM"), UVM_LOW)
    sram_wr_frm = sram_wr_seq::type_id::create("sram_wr_frm");
    sram_wr_frm.frame              = frm;
    sram_wr_frm.mac_ctrl_info_1    = mac_ctrl_info_1;
    sram_wr_frm.mac_ctrl_info_2    = mac_ctrl_info_2;
    if (frm.kind == NDP) begin
      sram_wr_frm.pt_mac_ctrl_info_2 = 0;
    end
    else begin
      sram_wr_frm.pt_mac_ctrl_info_2.rts_threshold_f     = 12'd200;
      sram_wr_frm.pt_mac_ctrl_info_2.short_retry_limit_f = $urandom_range(10, 15);
      sram_wr_frm.pt_mac_ctrl_info_2.long_retry_limit_f  = $urandom_range(10, 15);
    end
    sram_wr_frm.prot = prot;
    sram_wr_frm.addr = addr;
    // set KSR index that will be used in policy table and compressed
    // policy table
    foreach (frm.ampdu_frame[i]) begin
      sram_wr_frm.pt_ksr_index[i] = m_ksr.get_device_index(frm.ampdu_frame[i].get_MAC_addr(.RA(1)));
    end
    sram_wr_frm.start(m_vsqr.m_sram_sqr);
  endtask : wr_frame_to_sram

  //--------------------------------------------------------------------
  // Read PPDU frame from SRAM
  //--------------------------------------------------------------------
  task rd_frame_from_sram(bit [31:0] addr);
    sram_rd_seq sram_rd_frm;

    `uvm_info(get_type_name(), $sformatf("Read frame from SRAM"), UVM_LOW)
    sram_rd_frm = sram_rd_seq::type_id::create("sram_rd_frm");
    sram_rd_frm.addr = addr;
    sram_rd_frm.start(m_vsqr.m_sram_sqr);
  endtask : rd_frame_from_sram

  //--------------------------------------------------------------------
  // prepare SRAM Rx descriptors for receiving frame
  // @size - size of buffer in bytes
  //--------------------------------------------------------------------
  task prepare_sram_for_rx(int size, bit smaller_rx_segment = 0);
    bit [31:0] start_addr;
    int        total_header_footer;

    start_addr = (smaller_rx_segment) ? `MEM_RX_SEGMENT : `MEM_RX_SEGMENT_START;
    // get footer and header sizes
    rdata = m_regmodel.get_mirrored_reg_value("RXBUFCONFIGREG");
    rhd_header = rdata[7:0]   * 4;
    rhd_footer = rdata[15:8]  * 4;
    rpd_header = rdata[23:16] * 4;
    rpd_footer = rdata[31:24] * 4;
    total_header_footer = rhd_header + rhd_footer + rpd_header + rpd_footer;

    wdata = start_addr;
    m_regmodel.set_reg_value(wdata, "RXBUF1STARTPTRREG");
    m_regmodel.set_reg_value(wdata, "RXBUF1WRPTRREG");
    m_regmodel.set_reg_value(wdata, "RXBUF1RDPTRREG");
    wdata = (start_addr + size + 32'd4000 + total_header_footer) & 32'hFFFF_FFFC;
    m_regmodel.set_reg_value(wdata, "RXBUF1ENDPTRREG");
    // reset memory wrapping flag
    sram_wrapp = 0;
    `uvm_info(get_type_name(),
    $sformatf("Prepare SRAM buffer 1 for reception, start 0x%0h end 0x%0h pointer",start_addr,wdata),UVM_LOW)
  endtask : prepare_sram_for_rx

  //--------------------------------------------------------------------
  // prepare SRAM Rx descriptors for receiving priority frame
  // @size - size of buffer in bytes
  //--------------------------------------------------------------------
  task prepare_sram_priority_rx(int size);
    bit [31:0] start_addr;
    int        total_header_footer;

    start_addr = m_regmodel.get_mirrored_reg_value("RXBUF1ENDPTRREG");
    // get footer and header sizes
    rdata = m_regmodel.get_mirrored_reg_value("RXBUFCONFIGREG");
    rhd_header = rdata[7:0]   * 4;
    rhd_footer = rdata[15:8]  * 4;
    rpd_header = rdata[23:16] * 4;
    rpd_footer = rdata[31:24] * 4;
    total_header_footer = rhd_header + rhd_footer + rpd_header + rpd_footer;

    start_addr = (start_addr + 32'd8) & 32'hFFFF_FFFC;
    wdata = start_addr;
    m_regmodel.set_reg_value(wdata, "RXBUF2STARTPTRREG");
    m_regmodel.set_reg_value(wdata, "RXBUF2WRPTRREG");
    m_regmodel.set_reg_value(wdata, "RXBUF2RDPTRREG");
    wdata = (start_addr + size + total_header_footer) & 32'hFFFF_FFFC;
    m_regmodel.set_reg_value(wdata, "RXBUF2ENDPTRREG");
    // reset memory wrapping flag
    sram_wrapp = 0;
    `uvm_info(get_type_name(),
    $sformatf("Prepare SRAM buffer 2 for reception, start 0x%0h end 0x%0h pointer",start_addr,wdata),UVM_LOW)
  endtask : prepare_sram_priority_rx

  //--------------------------------------------------------------------
  // Read THD status information from SRAM
  //--------------------------------------------------------------------
  task rd_tx_status_from_sram(bit [31:0] head, bit skip_athd = 0);
    sram_rd_tx_status_seq  sram_rd_tx_status;

    `uvm_info(get_type_name(), $sformatf("Read THD status information"), UVM_LOW)
    sram_rd_tx_status = sram_rd_tx_status_seq::type_id::create("sram_rd_tx_status_seq");
    sram_rd_tx_status.addr = head;
    sram_rd_tx_status.skip = skip_athd;
    sram_rd_tx_status.start(m_vsqr.m_sram_sqr);
  endtask : rd_tx_status_from_sram

  //--------------------------------------------------------------------
  // get next pointer to RHD from SRAM
  // @buff - which buffer is used 1 or 2
  // @head - starting pointer of last read
  // @next - next pointer for read
  //--------------------------------------------------------------------
  task get_next_rhd_ptr(input int buff, input bit [31:0] head, output bit [31:0] next);
    bit [31:0] start_ptr, end_ptr;
    bit        payload_zero;
    sram_get_next_rhd_seq sram_get_next_rhd;

    sram_get_next_rhd = sram_get_next_rhd_seq::type_id::create("sram_get_next_rhd");
    sram_get_next_rhd.addr = head;
    sram_get_next_rhd.start(m_vsqr.m_sram_sqr);
    // put to output next RHD pointer got from memory
    next = sram_get_next_rhd.next_rhd_ptr;
    payload_zero = sram_get_next_rhd.payload_zero;
    `uvm_info(get_type_name(), $sformatf("Get next RHD pointer (buffer %0d) = 0x%0h",buff,next), UVM_LOW)

    // get start and end poitners
    start_ptr = m_regmodel.get_mirrored_reg_value($sformatf("RXBUF%0dSTARTPTRREG",buff));
    end_ptr   = m_regmodel.get_mirrored_reg_value($sformatf("RXBUF%0dENDPTRREG",buff));
    // get footer and header sizes
    rdata = m_regmodel.get_mirrored_reg_value("RXBUFCONFIGREG");
    rhd_header = 4 * rdata[7:0];
    rhd_footer = 4 * rdata[15:8];
    rpd_header = 4 * rdata[23:16];
    rpd_footer = 4 * rdata[31:24];
    // set buffer read pointer
    next = (next & 32'hFFFF_FFFC) + 4;
    // check if wrapping will occured if RHD can't fit into remaining memory
    // space (in limits of start-end);
    // put read pointer to start pointer
    if ((next + 4*`RHD_LEN + rhd_header + rhd_footer) > (end_ptr + 4)) begin
      wdata      = next;
      wdata[31]  = sram_wrapp;
      next       = start_ptr;
      sram_wrapp = ~sram_wrapp;
      `uvm_info(get_type_name(), $sformatf("WRAP READ POINTER 1 HEAD(%0h)!!!",head), UVM_LOW)
    end
    // check if wrapping occured during storing of MAC header and payload
    else if (head > next && next inside {[start_ptr:end_ptr]}) begin
      wdata      = next;
      sram_wrapp = ~sram_wrapp;
      wdata[31]  = sram_wrapp;
      `uvm_info(get_type_name(), $sformatf("WRAP READ POINTER 2 HEAD(%0h)!!!",head), UVM_LOW)
    end
    else begin
      wdata      = next;
      wdata[31]  = sram_wrapp;
    end
    // add RHD header
    next += rhd_header;
    if (payload_zero) begin
      next += rhd_footer;
      wdata += rhd_footer;
    end

    m_regmodel.set_reg_value(wdata, $sformatf("RXBUF%0dRDPTRREG",buff));
    // get write pointer value
    m_regmodel.get_reg_value(rdata, $sformatf("RXBUF%0dWRPTRREG",buff));
    `uvm_info(get_type_name(), $sformatf("RXBUF%0dWRPTRREG = %0h",buff,rdata), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("RXBUF%0dRDPTRREG = %0h",buff,wdata), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("Next read pointer = %0h", next), UVM_LOW)
  endtask : get_next_rhd_ptr

  //--------------------------------------------------------------------
  // Write word to SRAM, backdoor
  //--------------------------------------------------------------------
  task write_word_sram(bit [31:0] addr, bit [31:0] data);
    sram_wr_word_seq sram_wr_word;

    `uvm_info(get_type_name(), $sformatf("SRAM - write 0x%0h @ 0x%0h",data,addr), UVM_HIGH)
    sram_wr_word = sram_wr_word_seq::type_id::create("sram_wr_word");
    sram_wr_word.addr = addr;
    sram_wr_word.data = data;
    sram_wr_word.start(m_vsqr.m_sram_sqr);
  endtask : write_word_sram

  //--------------------------------------------------------------------
  // Read word from SRAM, backdoor
  //--------------------------------------------------------------------
  task read_word_sram(input bit [31:0] addr, output bit [31:0] data);
    sram_rd_word_seq sram_rd_word;

    sram_rd_word = sram_rd_word_seq::type_id::create("sram_rd_word");
    sram_rd_word.addr = addr;
    sram_rd_word.data = data;
    sram_rd_word.start(m_vsqr.m_sram_sqr);
    data = sram_rd_word.data;
    `uvm_info(get_type_name(), $sformatf("SRAM - read 0x%0h @ 0x%0h",data,addr), UVM_HIGH)
  endtask : read_word_sram

  //---------------------------------------------
  // write SRAM via AHB, frontdoor
  //---------------------------------------------
  task write_word_sram_via_ahb(bit [31:0] a, bit [31:0] d);
    ahb_master_direct_seq         sram_ahb_access;

    if (a > `SRAMADDREND) begin
      `uvm_error(get_type_name(),$sformatf("SRAM address out of range! End address 0x%h",`SRAMADDREND))
    end

    sram_ahb_access = ahb_master_direct_seq::type_id::create("sram_ahb_access");
    assert (sram_ahb_access.randomize() with {
      addr       == `SRAMADDRSTART + a;
      data[0]    == d;
      access     == WRITE;
      burst_size == 1;
      size       == WORD;
    });
    sram_ahb_access.start(m_vsqr.m_ahb_master_sqr);
    `uvm_info(get_type_name(),$sformatf("SRAM WRITE: @0x%0h - data 0x%0h",a,d), UVM_HIGH)

  endtask : write_word_sram_via_ahb

  //---------------------------------------------
  // read SRAM via AHB, frondoor
  //---------------------------------------------
  task read_word_sram_via_ahb(bit [31:0] a, output bit [31:0] d);
    ahb_master_direct_seq         sram_ahb_access;

    if (a > `SRAMADDREND) begin
      `uvm_error(get_type_name(),$sformatf("SRAM address out of range! End address 0x%h",`SRAMADDREND))
    end

    sram_ahb_access = ahb_master_direct_seq::type_id::create("sram_ahb_access");
    assert (sram_ahb_access.randomize() with {
      addr       == `SRAMADDRSTART + a;
      access     == READ;
      burst_size == 1;
      size       == WORD;
    });
    sram_ahb_access.start(m_vsqr.m_ahb_master_sqr);
    d = sram_ahb_access.read_data[0]; // provide read data
    `uvm_info(get_type_name(),$sformatf("SRAM READ: @0x%0h - data 0x%0h",a,d), UVM_HIGH)

  endtask : read_word_sram_via_ahb

  //--------------------------------------------------------------------
  // Initialize PHY registers
  //--------------------------------------------------------------------
  virtual task phy_set_base_config();
    real invcarrierfreq;

    //enable clock gating
    m_regmodel.set_field_value(1'b1, "sec_mac_pi_tx_clk_gating_en", "MISC_CNTL");
    m_regmodel.set_field_value(1'b1, "sec_mac_crypt_clk_gating_en", "MISC_CNTL");
    m_regmodel.set_field_value(1'b1, "sec_mac_core_tx_clk_gating_en", "MISC_CNTL");
    m_regmodel.set_field_value(1'b1, "sec_mpif_clk_gating_en", "MISC_CNTL");
    m_regmodel.set_field_value(1'b1, "mac_pi_clk_gating_en", "MISC_CNTL");
    m_regmodel.set_field_value(1'b1, "mac_pi_tx_clk_gating_en", "MISC_CNTL");
    m_regmodel.set_field_value(1'b1, "mac_pi_rx_clk_gating_en", "MISC_CNTL");
    m_regmodel.set_field_value(1'b1, "mac_core_clk_gating_en", "MISC_CNTL");
    m_regmodel.set_field_value(1'b1, "mac_crypt_clk_gating_en", "MISC_CNTL");
    m_regmodel.set_field_value(1'b1, "mac_core_tx_clk_gating_en", "MISC_CNTL");
    m_regmodel.set_field_value(1'b1, "mac_core_rx_clk_gating_en", "MISC_CNTL");
    m_regmodel.set_field_value(1'b1, "mac_wt_clk_gating_en", "MISC_CNTL");
    m_regmodel.set_field_value(1'b1, "mpif_clk_gating_en", "MISC_CNTL");
    m_regmodel.set_field_value(1'b1, "CLKEN");

    m_regmodel.set_field_value(4'd2, "TDAdjust20ShortGI");
    m_regmodel.set_field_value(1'b1, "RCCLKFORCE");

    m_regmodel.set_field_value(10'd832, "AutoCorrCompareRatioHigh");
    m_regmodel.set_field_value(10'd256, "AutoCorrCompareRatioLow");
    m_regmodel.set_field_value(8'd25, "TDSYNCOFF20");

`ifdef RW_MUMIMO_RX_EN
    m_regmodel.set_field_value(1'b1, "RXVHTMUMIMOEN");
    m_regmodel.set_field_value(1'b1, "RXHEMUMIMOEN");
`endif

`ifdef RW_NX_VHT_EN
    m_regmodel.set_field_value(1'b1, "RXVHTEN");
    m_regmodel.set_field_value(1'b1, "TXVHTEN");
`else
    m_regmodel.set_field_value(1'b0, "RXVHTEN");
    m_regmodel.set_field_value(1'b0, "TXVHTEN");
`endif

`ifndef RW_BFMEE_EN
    m_regmodel.set_field_value(3'd0, "RXNDPNSTSMAX");
`endif

`ifndef RW_NX_DERIV_CHBW20ONLY
    m_regmodel.set_field_value(4'd4, "TDAdjust40ShortGI");
    m_regmodel.set_field_value(2'd1, "RXCBWMAX");
    m_regmodel.set_field_value(4'd3, "STARTDC");
    m_regmodel.set_field_value(7'd16, "DELAYSYNC");
    m_regmodel.set_field_value(7'd31, "WAITHTSTF");
    m_regmodel.set_field_value(8'd11, "TDSYNCOFF2040");
    m_regmodel.set_field_value(8'd24, "TDSYNCOFF20");
`endif
`ifdef RW_NX_DERIV_CHBW804020ONLY
    m_regmodel.set_field_value(2'd2, "RXCBWMAX");
    m_regmodel.set_field_value(7'd13, "DELAYSYNC");
    m_regmodel.set_field_value(7'd63, "WAITHTSTF");
    m_regmodel.set_field_value(8'd13, "TDSYNCOFF2040");
    m_regmodel.set_field_value(-8'sd22, "TDSYNCOFF2080");
`endif

    m_regmodel.set_field_value(1'b1, "TDFOCPESLOPEEN");
    m_regmodel.set_field_value(1'b1, "FOESTFDPREAMBEN");

    m_regmodel.get_reg_value(rdata, "HDMCONFIG", "PHYCONFIG");

    `uvm_info(get_type_name(), $sformatf(" "), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("PHY RTL Configuration"), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("-----------------"), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf(" "), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - MUMIMOTX  : %0d",rdata[31]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - MUMIMORX  : %0d",rdata[30]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - BFMER     : %0d",rdata[29]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - BFMEE     : %0d",rdata[28]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - LDPCDEC   : %0d",rdata[27]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - LDPCENC   : %0d",rdata[26]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - CHBW      : %0d",rdata[25:24]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - DSSSCCK   : %0d",rdata[23]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - VHT       : %0d",rdata[22]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - HE        : %0d",rdata[21]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - ESS       : %0d",rdata[20]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - RFMODE    : %0d",rdata[19:16]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - NSTS      : %0d",rdata[15:12]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - NSS       : %0d",rdata[11:8]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - NTX       : %0d",rdata[7:4]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf("    - NRX       : %0d",rdata[3:0]), UVM_LOW)
    `uvm_info(get_type_name(), $sformatf(" "), UVM_LOW)

    invcarrierfreq = (2**26)/real'(carrier_freq);
    m_regmodel.set_field_value(int'(invcarrierfreq),"INVCARRIERFREQ");

    // Set MDMCONF
`ifdef RW_NX_DERIV_CHBW20ONLY
  `ifdef RW_TXRX_1X1
    m_regmodel.set_reg_value(32'h0000_0000, "MDMCONF", "PHYCONFIG");
  `else
    m_regmodel.set_reg_value(32'h0000_0110, "MDMCONF", "PHYCONFIG");
  `endif
`elsif RW_NX_DERIV_CHBW4020ONLY
  // 20 and 40 MHz only
  `ifdef RW_TXRX_1X1
    m_regmodel.set_reg_value(32'h0000_0001, "MDMCONF", "PHYCONFIG");
  `else
    m_regmodel.set_reg_value(32'h0000_0111, "MDMCONF", "PHYCONFIG");
  `endif
`elsif RW_NX_DERIV_CHBW804020ONLY
  `ifdef RW_TXRX_1X1
     m_regmodel.set_reg_value(32'h0000_0002, "MDMCONF", "PHYCONFIG");
  `else
     m_regmodel.set_reg_value(32'h0000_0112, "MDMCONF", "PHYCONFIG");
  `endif
`endif

`ifdef RW_TXRX_1X1
    // Set antenna 0 active
    m_regmodel.set_field_value(4'b1, "ACTIVEANT");
`else
    // Set antenna 0 and 1 active
    m_regmodel.set_field_value(4'b11, "ACTIVEANT");
`endif

    // disable AGC cross-up detection
    m_regmodel.set_field_value(12'h200, "CROSSUPTHRQDBM");
`ifdef RW_NX_DERIV_CHBW20ONLY
    // disable AGC HTSTF gain update in 20MHz BW only
//     m_regmodel.set_field_value(1'b0, "HTSTFGAINEN");
`endif
    // disable AGC DC compensation
    m_regmodel.set_field_value(2'd0, "DCDSSSTYPE");
    m_regmodel.set_field_value(2'd0, "DCCENTEREDTYPE");
    m_regmodel.set_field_value(2'd0, "DCAGC20TYPE");
    m_regmodel.set_field_value(2'd0, "DCADCTYPE");

    riu_set_control_channel();

    m_cfg.modem_init_done = 1'b1;
    `uvm_info(get_type_name(),"PHY CONFIGURATION DONE!",UVM_LOW)
  endtask: phy_set_base_config

  //--------------------------------------------------------------------
  // Set random control channel based on frame bandwith
  //--------------------------------------------------------------------
  task riu_set_control_channel();

    `ifdef RW_NX_DERIV_CHBW804020ONLY
      //for 80MHz BW, randomly choose control channels
      randcase
        1: m_regmodel.set_field_value(2'b00, "PSSELECT", "RWNXMACSTATICCONFIG", "RIUKARST"); // Lower channel
        1: m_regmodel.set_field_value(2'b01, "PSSELECT", "RWNXMACSTATICCONFIG", "RIUKARST"); // Lower channel
        1: m_regmodel.set_field_value(2'b10, "PSSELECT", "RWNXMACSTATICCONFIG", "RIUKARST"); // Upper channel
        1: m_regmodel.set_field_value(2'b11, "PSSELECT", "RWNXMACSTATICCONFIG", "RIUKARST"); // Upper channel
      endcase
    `else
      //for 20MHz and 40MHz BW, randomly choose control channels
      randcase
        1: m_regmodel.set_field_value(2'b01, "PSSELECT", "RWNXMACSTATICCONFIG", "RIUKARST"); // Upper channel
        1: m_regmodel.set_field_value(2'b10, "PSSELECT", "RWNXMACSTATICCONFIG", "RIUKARST"); // Lower channel
      endcase
    `endif

  endtask: riu_set_control_channel

  //---------------------------------------------
  // write to LDPC configuration memory via AHB
  //---------------------------------------------
  task ldpc_cfg_wr_via_ahb(bit [31:0] a, bit [31:0] d);
    ahb_master_direct_seq         ldpc_cfg_access;

`ifdef RW_NX_LDPC_DEC
    if (a > `LDPCCFGMEMEND) begin
      `uvm_error(get_type_name(),$sformatf("LDPC CFG memory address out of range! End address 0x%h",`LDPCCFGMEMEND))
    end

    ldpc_cfg_access = ahb_master_direct_seq::type_id::create("ldpc_cfg_access");
    assert (ldpc_cfg_access.randomize() with {
      addr       == `LDPCCFGMEMSTART + a;
      data[0]    == d;
      access     == WRITE;
      burst_size == 1;
      size       == WORD;
    });
    ldpc_cfg_access.start(m_vsqr.m_ahb_master_sqr);
    `uvm_info(get_type_name(),$sformatf("LDPC CFG WRITE: @0x%0h - data 0x%0h",a,d), UVM_DEBUG)
`else
    `uvm_info(get_type_name(), $sformatf("RW_NX_LDPC_DEC not defined task will do nothing"), UVM_LOW)
`endif
  endtask : ldpc_cfg_wr_via_ahb

  //---------------------------------------------
  // write to LDPC configuration memory via AHB
  //---------------------------------------------
  task ldpc_cfg_rd_via_ahb(input bit [31:0] a, output bit [31:0] d);
    ahb_master_direct_seq         ldpc_cfg_access;

`ifdef RW_NX_LDPC_DEC
    if (a > `LDPCCFGMEMEND) begin
      `uvm_error(get_type_name(),$sformatf("LDPC CFG memory address out of range! End address 0x%h",`LDPCCFGMEMEND))
    end

    ldpc_cfg_access = ahb_master_direct_seq::type_id::create("ldpc_cfg_access");
    assert (ldpc_cfg_access.randomize() with {
      addr       == `LDPCCFGMEMSTART + a;
      data[0]    == d;
      access     == READ;
      burst_size == 1;
      size       == WORD;
    });
    ldpc_cfg_access.start(m_vsqr.m_ahb_master_sqr);
    d = ldpc_cfg_access.read_data[0]; // provide read data
    `uvm_info(get_type_name(),$sformatf("LDPC CFG READ: @0x%0h - data 0x%0h",a,d), UVM_DEBUG)
`else
    `uvm_info(get_type_name(), $sformatf("RW_NX_LDPC_DEC not defined task will do nothing"), UVM_LOW)
`endif
  endtask : ldpc_cfg_rd_via_ahb

  //--------------------------------------------------------------------
  // Enable/disable LDPC
  //--------------------------------------------------------------------
  task phy_set_ldpc(string file_name = "ldpc_cfg_ram.bin");
    int         fptr; // file pointer
    int         ret;
    bit [31:0]  data;
    bit [31:0]  addr;
    int         cnt_limit;

   `ifndef RW_NX_LDPC_DEC
     `uvm_error(get_type_name(),
     $sformatf("phy_set_ldpc() Called while RW_NX_LDPC_DEC not defined!"))
   `else
     // load LDPC RAM
     if (ldpc_cfg_done == 0) begin
       fptr = $fopen(file_name, "r");
       if (fptr == 0)
         `uvm_fatal(get_type_name(), $sformatf("Unable to open %s for reading",file_name))

       addr = 0;
       cnt_limit = 4*`RW_NX_DERIV_LDPCCFGRAM_DEPTH;
       // read words from memory and store them in LDPC CFG
       while (!$feof(fptr) && addr < cnt_limit) begin
         ret = $fscanf(fptr, "%b\n", data);
         ldpc_cfg_wr_via_ahb(addr, data);
         addr += 4;
       end//while
       $fclose(fptr);
       ldpc_cfg_done = 1; // no need for multiple loads
     end
   `endif
  endtask: phy_set_ldpc

  //--------------------------------------------------------------------
  // Set GroupID membership for MU-MIMO
  //--------------------------------------------------------------------
  task phy_set_mu_group_memb(ref PPDU_frame ppdu);
    uvm_reg_data_t dec_group;

    if ( ppdu.preamble_header.group_id < 32) begin
      dec_group = 1'b1 << ppdu.preamble_header.group_id;
      m_regmodel.set_reg_value(dec_group, "MUMIMO_GROUPID_TAB0", "PHYCONFIG");
      `uvm_info(get_type_name(), $sformatf("GroupID %0d - wrote 0x%8x to register MUMIMO_GROUPID_TAB0",
      ppdu.preamble_header.group_id, dec_group), UVM_LOW)
    end
    else begin
      dec_group = 1'b1 << (ppdu.preamble_header.group_id - 'd32);
      m_regmodel.set_reg_value(dec_group, "MUMIMO_GROUPID_TAB1", "PHYCONFIG");
      `uvm_info(get_type_name(), $sformatf("GroupID %0d - wrote 0x%8x to register MUMIMO_GROUPID_TAB1",
      ppdu.preamble_header.group_id, dec_group), UVM_LOW)
    end
  endtask: phy_set_mu_group_memb


  //--------------------------------------------------------------------
  // Set User position for MU-MIMO
  //--------------------------------------------------------------------
  task phy_set_mu_userpos(ref PPDU_frame ppdu);
    uvm_reg_data_t dec_group;
    bit [1:0] userpos;

    userpos = ppdu.preamble_header.mu_mimo_userid;

    if ( ppdu.preamble_header.group_id < 16) begin
      dec_group = userpos << (2*ppdu.preamble_header.group_id);
      m_regmodel.set_reg_value(dec_group, "MUMIMO_USERPOSITION_TAB0", "PHYCONFIG");
      `uvm_info(get_type_name(),
      $sformatf("UserPos %0d - wrote 0x%8x to register MUMIMO_USERPOSITION_TAB0",userpos, dec_group), UVM_LOW)
    end
    else if ( ppdu.preamble_header.group_id < 32) begin
      dec_group = userpos << (2*(ppdu.preamble_header.group_id - 'd16));
      m_regmodel.set_reg_value(dec_group, "MUMIMO_USERPOSITION_TAB1", "PHYCONFIG");
      `uvm_info(get_type_name(),
      $sformatf("UserPos %0d - wrote 0x%8x to register MUMIMO_USERPOSITION_TAB1",userpos, dec_group), UVM_LOW)
    end
    else if ( ppdu.preamble_header.group_id < 48) begin
      dec_group = userpos << (2*(ppdu.preamble_header.group_id - 'd32));
      m_regmodel.set_reg_value(dec_group, "MUMIMO_USERPOSITION_TAB2", "PHYCONFIG");
      `uvm_info(get_type_name(),
      $sformatf("UserPos %0d - wrote 0x%8x to register MUMIMO_USERPOSITION_TAB2",userpos, dec_group), UVM_LOW)
    end
    else begin
      dec_group = userpos << (2*(ppdu.preamble_header.group_id - 'd48));
      m_regmodel.set_reg_value(dec_group, "MUMIMO_USERPOSITION_TAB3", "PHYCONFIG");
      `uvm_info(get_type_name(),
      $sformatf("UserPos %0d - wrote 0x%8x to register MUMIMO_USERPOSITION_TAB3",userpos, dec_group), UVM_LOW)
    end
  endtask: phy_set_mu_userpos

  //--------------------------------------------------------------------
  // Get User position for MU-MIMO
  //--------------------------------------------------------------------
  task phy_get_mu_userpos(ref PPDU_frame ppdu, ref bit [1:0] userpos);
    uvm_reg_data_t dec_group;

    if ( ppdu.preamble_header.group_id < 16) begin
      m_regmodel.get_reg_value(dec_group, "MUMIMO_USERPOSITION_TAB0", "PHYCONFIG");
      userpos = dec_group >> (2*ppdu.preamble_header.group_id);
      `uvm_info(get_type_name(),
      $sformatf("UserPos %0d - wrote 0x%8x to register MUMIMO_USERPOSITION_TAB0",userpos, dec_group), UVM_LOW)
    end
    else if ( ppdu.preamble_header.group_id < 32) begin
      m_regmodel.get_reg_value(dec_group, "MUMIMO_USERPOSITION_TAB1", "PHYCONFIG");
      userpos = dec_group >> (2*(ppdu.preamble_header.group_id - 'd16));
      `uvm_info(get_type_name(),
      $sformatf("UserPos %0d - wrote 0x%8x to register MUMIMO_USERPOSITION_TAB1",userpos, dec_group), UVM_LOW)
    end
    else if ( ppdu.preamble_header.group_id < 48) begin
      m_regmodel.get_reg_value(dec_group, "MUMIMO_USERPOSITION_TAB2", "PHYCONFIG");
      userpos = dec_group >> (2*(ppdu.preamble_header.group_id - 'd32));
      `uvm_info(get_type_name(),
      $sformatf("UserPos %0d - wrote 0x%8x to register MUMIMO_USERPOSITION_TAB2",userpos, dec_group), UVM_LOW)
    end
    else begin
      m_regmodel.get_reg_value(dec_group, "MUMIMO_USERPOSITION_TAB3", "PHYCONFIG");
      userpos = dec_group >> (2*(ppdu.preamble_header.group_id - 'd48));
      `uvm_info(get_type_name(),
      $sformatf("UserPos %0d - wrote 0x%8x to register MUMIMO_USERPOSITION_TAB3",userpos, dec_group), UVM_LOW)
    end
  endtask: phy_get_mu_userpos

  //--------------------------------------------------------------------
  // returns 1 if AGC is off/bypassed, either with AGC_ON macro or from
  // test
  //--------------------------------------------------------------------
  function bit is_agc_off();
    if ((`AGC_ON == 1) && (m_cfg.m_radio_cfg.m_rui_cfg.agcBypass == 1'b0)) begin
      return 0;
    end else begin
      return 1;
    end
  endfunction: is_agc_off

  //--------------------------------------------------------------------
  // register tests helper function, return the name of the register's
  // parent
  //--------------------------------------------------------------------
  function string get_reg_parent(uvm_reg dut_reg);
    uvm_reg_block parent;
    parent = dut_reg.get_parent();
    return parent.get_name();
  endfunction: get_reg_parent

  //--------------------------------------------------------------------
  // register tests helper function, returns 1 if dut_reg should be
  // skipped during register tests
  //--------------------------------------------------------------------
  function bit skip_reg(uvm_reg dut_reg);

    // ---------------------------------------------------------------------------------------
    // Part 1 - registers whose implementation depends on certain Verilog macros being defined
    // ---------------------------------------------------------------------------------------

`ifndef RW_WLAN_COEX_EN
    if (get_reg_parent(dut_reg) == "PTA") return 1; // PTA should be skipped if RW_WLAN_COEN_EX
                                                    // is not defined
`endif  // RW_WLAN_COEX_EN

`ifndef RW_EMBEDDED_LA
    if (get_reg_parent(dut_reg) == "LA") return 1; // LA should be skipped if RW_EMBEDDED_LA
                                                   // is not defined
`endif  // RW_EMBEDDED_LA

    if (get_reg_parent(dut_reg) == "HSU")
    begin
      if      (dut_reg.get_name() == "CONTROL")          return 1;
  `ifndef RW_HSU_RSA_EN
      else if (dut_reg.get_name() == "DESTINATION_ADDR") return 1;
      else if (dut_reg.get_name() == "SHA_TAB0")         return 1;
      else if (dut_reg.get_name() == "SHA_TAB1")         return 1;
      else if (dut_reg.get_name() == "SHA_TAB2")         return 1;
      else if (dut_reg.get_name() == "SHA_TAB3")         return 1;
      else if (dut_reg.get_name() == "SHA_TAB4")         return 1;
      else if (dut_reg.get_name() == "SHA_TAB5")         return 1;
      else if (dut_reg.get_name() == "SHA_TAB6")         return 1;
      else if (dut_reg.get_name() == "SHA_TAB7")         return 1;
      else if (dut_reg.get_name() == "SHA_TAB8")         return 1;
      else if (dut_reg.get_name() == "SHA_TAB9")         return 1;
      else if (dut_reg.get_name() == "SHA_TAB10")        return 1;
      else if (dut_reg.get_name() == "SHA_TAB11")        return 1;
      else if (dut_reg.get_name() == "SHA_TAB12")        return 1;
      else if (dut_reg.get_name() == "SHA_TAB13")        return 1;
      else if (dut_reg.get_name() == "SHA_TAB14")        return 1;
      else if (dut_reg.get_name() == "SHA_TAB15")        return 1;
  `endif  // RW_HSU_RSA_EN
    end

`ifndef RW_NX_LDPC_DEC
    if (dut_reg.get_name() == "LDPCCTRL") return 1;
`endif // RW_NX_LDPC_DEC

`ifndef RW_NX_LDPC_DEC
    if (dut_reg.get_name() == "RXCTRL3") return 1;
`endif // RW_NX_LDPC_DEC

`ifndef RW_BFMEE_EN
    if      (dut_reg.get_name() == "SVDCTRL") return 1;
    else if (dut_reg.get_name() == "BFMEECONTROLREG") return 1;
`endif // RW_BFMEE_EN

`ifndef RW_MUMIMO_RX_EN
    if      (dut_reg.get_name() == "MUMIMO_GROUPID_TAB0") return 1;
    else if (dut_reg.get_name() == "MUMIMO_GROUPID_TAB1") return 1;
    else if (dut_reg.get_name() == "MUMIMO_USERPOSITION_TAB0") return 1;
    else if (dut_reg.get_name() == "MUMIMO_USERPOSITION_TAB1") return 1;
    else if (dut_reg.get_name() == "MUMIMO_USERPOSITION_TAB2") return 1;
    else if (dut_reg.get_name() == "MUMIMO_USERPOSITION_TAB3") return 1;
    else if (dut_reg.get_name() == "EQUALCTRL2") return 1;
`endif  // RW_MUMIMO_RX_EN

`ifndef RW_MUMIMO_TX_EN
    if      (dut_reg.get_name() == "SECUSERSTXINTEVENTSETREG") return 1;
    else if (dut_reg.get_name() == "SECUSERSTXINTEVENTCLEARREG") return 1;
    else if (dut_reg.get_name() == "SECUSERSTXINTEVENTUNMASKREG") return 1;
`endif // RW_MUMIMO_TX_EN

`ifndef RW_RADAR_EN
    if      (dut_reg.get_name() == "RADARFIFOSTAT")      return 1;
    else if (dut_reg.get_name() == "RWNXAGCRADAR")       return 1;
    else if (dut_reg.get_name() == "RWNXAGCRADARTIM")    return 1;
    else if (dut_reg.get_name() == "RWNXAGCDCCOMPRADAR") return 1;
`endif  // RW_RADAR_EN

`ifndef RW_NX_AGC_SNR_EN
    if (dut_reg.get_name() == "RWNXAGCNOISECONF") return 1;
`endif  // RW_NX_AGC_SNR_EN

`ifndef RW_WLAN_COEX_EN
    if      (dut_reg.get_name() == "COEXCONTROLREG") return 1;
    else if (dut_reg.get_name() == "COEXPTIREG")     return 1;
    else if (dut_reg.get_name() == "COEXSTATREG")    return 1;
    else if (dut_reg.get_name() == "COEXINTREG")     return 1;
`endif  // RW_WLAN_COEX_EN

`ifndef RW_WAPI_EN
    if      (dut_reg.get_name() == "ENCRWPIINTKEY0REG") return 1;
    else if (dut_reg.get_name() == "ENCRWPIINTKEY1REG") return 1;
    else if (dut_reg.get_name() == "ENCRWPIINTKEY2REG") return 1;
    else if (dut_reg.get_name() == "ENCRWPIINTKEY3REG") return 1;
`endif  // RW_WAPI_EN

`ifndef RW_NX_FIQ_COMP_EN
    if  (dut_reg.get_name() inside { "RWNXFEFIQCTRL",  "RWNXFEFIQCOLL", "RWNXFEFIQEST0", "RWNXFEFIQEST1", "RWNXFEFIQDEL",
                                     "RWNXFEFIQCOEFF0", "RWNXFEFIQCOEFF1", "RWNXFEFIQCOEFF2", "RWNXFEFIQCOEFF3", "RWNXFEFIQCOEFF4",
                                     "RWNXFEFIQCOEFF5", "RWNXFEFIQCOEFF6", "RWNXFEFIQCOEFF7", "RWNXFEFIQCOEFF8", "RWNXFEFIQCOEFF9",
                                     "RWNXFEFIQCOEFF10", "RWNXFEFIQCOEFF11", "RWNXFEFIQCOEFF12", "RWNXFEFIQCOEFF13", "RWNXFEFIQCOEFF14",
                                     "RWNXFEFIQCOEFF15", "RWNXFEFIQCOEFF16", "RWNXFEFIQCOEFF17", "RWNXFEFIQCOEFF18", "RWNXFEFIQCOEFF19",
                                     "RWNXFEFIQCOEFFUPDATE", "RWNXFEFIQCOEFFREQ", "RWNXFEFIQSTART", "RWNXFEFIQDONE"})
      return 1;
`endif  // RW_NX_FIQ_COMP_EN

`ifndef RW_NX_IQ_COMP_EN
    if  (dut_reg.get_name() inside { "RWNXFETXIQCOMP0", "RWNXFETXIQCOMP1", "RWNXFERXIQESTCTRL", "RWNXFERXIQESTLOOP",
                                     "RWNXFERXIQESTCLR", "RWNXFERXIQCOMP0", "RWNXFERXIQCOMP1"}) return 1;
`endif // RW_NX_IQ_COMP_EN

    // ---------------------------------------------------------------------------------------
    // Part 2 - Special cases
    // ---------------------------------------------------------------------------------------

    // MACCORE.MONOTONICCOUNTER1REG - free running counter which counts as soon as reset is deasserted
    // MACCORE.MONOTONICCOUNTER2LOREG, MACCORE.MONOTONICCOUNTER2HIREG - same as above
    // RIUKARST.AGCMEMFIFOSTAT - FIFO is empty after reset, so XLS is wrong
    // RIUKARST.RADARFIFOSTAT - FIFO is empty after reset, so XLS is wrong
    // RIUKARST.RWNXVERSION - depends on Verilog macros, reset value in XLS is wrong
    // RIUKARST.AGCDIGGAIN0STAT - reset value for 40MHz and 80MHz cannot be 0 as in XLS
    // RIUKARST.AGCDIGGAIN1STAT - same as above
    // INTC.IRQ_UNMASK_SET - INTC block RTL implementation and XLS differ a lot
    // INTC.IRQ_UNMASK_CLEAR - INTC block RTL implementation and XLS differ a lot
    // INTC.FIQ_STATUS - INTC block RTL implementation and XLS differ a lot
    // INTC.FIQ_RAWSTATUS - INTC block RTL implementation and XLS differ a lot
    // INTC.FIQ_POLARITY - INTC block RTL implementation and XLS differ a lot
    // INTC.FIQ_UNMASK_SET - INTC block RTL implementation and XLS differ a lot
    // INTC.FIQ_UNMASK_CLEAR - INTC block RTL implementation and XLS differ a lot
    // SYSCTRL.SIGNATURE - RTL uses verilog macro, XLS reset value not updated
    // SYSCTRL.MISC_CNTL - reset value of field fpgab_reset_req difer in XLS and RTL
    // SYSCTRL.TIMER - free running counter which counts as soon as reset is deasserted
    // SYSCTRL.DATE - RTL uses verilog macro, XLS reset value not updated
    // MACCORE.MIBTABLEWRITEREG - cannot be written randomly, needs specific sequence
    // MACCORE.MACCNTRL1REG - same as above
    // MACCORE.MACERRRECCNTRLREG - same as above
    // MACCORE.DEBUGNAVREG - same as above
    // MACCORE.ENCRCNTRLREG - same as above
    // MACCORE.DTIMCFP1REG - same as above
    // DMA.AXI_BURST_LENGTH_LIMIT - different reset values in RTL and XLS
    // DMA.DMA_STATUS (field oft_free) - different reset values in RTL and XLS
    // MACPL.TSFLOREG, MACPL.TSFHIREG - free running counter which counts as soon as reset is deasserted
    // PHYSTATUS.HDMCONFIG - reset values of NSS, NTX, NRX depends on used configuration
    // PHYSTATUS.BFMERSTAT0 - reset value depends on configuration

    if      (dut_reg.get_name() == "MONOTONICCOUNTER1REG") return 1;
    else if (dut_reg.get_name() == "MONOTONICCOUNTER2LOREG") return 1;
    else if (dut_reg.get_name() == "MONOTONICCOUNTER2HIREG") return 1;
    else if (dut_reg.get_name() == "AGCMEMFIFOSTAT") return 1;
    else if (dut_reg.get_name() == "RADARFIFOSTAT") return 1;
    else if (dut_reg.get_name() == "RWNXVERSION") return 1;
    else if (dut_reg.get_name() == "AGCDIGGAIN0STAT") return 1;
    else if (dut_reg.get_name() == "AGCDIGGAIN1STAT") return 1;
    else if (dut_reg.get_name() == "SIGNATURE") return 1;
    else if (dut_reg.get_name() == "DATE") return 1;
    else if (dut_reg.get_name() == "TIMER") return 1;
    else if (dut_reg.get_name() == "MISC_CNTL") return 1;
    else if (dut_reg.get_name() == "MIBTABLEWRITEREG") return 1;
    else if (dut_reg.get_name() == "MACCNTRL1REG") return 1;
    else if (dut_reg.get_name() == "MACERRRECCNTRLREG") return 1;
    else if (dut_reg.get_name() == "STATECNTRLREG") return 1;
    else if (dut_reg.get_name() == "STATUS_CLEAR") return 1;
    else if (dut_reg.get_name() == "SW_CTRL") return 1;
    else if (dut_reg.get_name() == "DEBUGNAVREG") return 1;
    else if (dut_reg.get_name() == "DEBUGBASICNAVREG") return 1;
    else if (dut_reg.get_name() == "ENCRCNTRLREG") return 1;
    else if (dut_reg.get_name() == "DTIM1REG") return 1;
    else if (dut_reg.get_name() == "AXI_BURST_LENGTH_LIMIT") return 1;
    else if (dut_reg.get_name() == "DMA_STATUS") return 1;
    else if (dut_reg.get_name() == "TSFLOREG") return 1;
    else if (dut_reg.get_name() == "TSFHIREG") return 1;
    else if (dut_reg.get_name() == "HDMCONFIG") return 1;
    else if (dut_reg.get_name() == "BFMERSTAT0") return 1;
    else if (dut_reg.get_name() == "BBSERVICEREG") return 1;
    else if (dut_reg.get_name() == "TICK_TIMER_CNT") return 1;
    else if (dut_reg.get_name() == "STATECNTRLREG") return 1;
    else if (dut_reg.get_name() == "SWRESET") return 1;
    else if (dut_reg.get_name() == "CLKEN") return 1;
    else if (dut_reg.get_name() == "ADDCCABUSYSEC20REG") return 1;
    else if (dut_reg.get_name() == "ADDCCABUSYSEC40REG") return 1;
    else if (dut_reg.get_name() == "ADDCCABUSYSEC80REG") return 1;
    else if (dut_reg.get_name() == "CLKGATEFCTRL0") return 1;
    else if (dut_reg.get_name() == "REVISION") return 1;
    else if (get_reg_parent(dut_reg) == "INTC") return 1;
    else if (get_reg_parent(dut_reg) == "MACBYPASS") return 1;
    else if (get_reg_parent(dut_reg) == "CRM") return 1;

    // don't skip register test
    return 0;
  endfunction: skip_reg

  //--------------------------------------------------------------------
  // COEX Bluetooth interface set tasks
  //--------------------------------------------------------------------
  task set_coex_bt_tx(bit val);
    coex_bt_set_seq coex_bt_set;

    coex_bt_set = coex_bt_set_seq::type_id::create("coex_bt_set");
    coex_bt_set.command = BT_SET_TX;
    coex_bt_set.value[`BT_TX] = val;
    coex_bt_set.start(m_vsqr.m_coex_bt_sqr);
    `uvm_info(get_type_name(), $sformatf("Drive COEX BT TX On-going (%0d)",val), UVM_LOW)
  endtask : set_coex_bt_tx

  task set_coex_bt_rx(bit val);
    coex_bt_set_seq coex_bt_set;

    coex_bt_set = coex_bt_set_seq::type_id::create("coex_bt_set");
    coex_bt_set.command = BT_SET_RX;
    coex_bt_set.value[`BT_RX] = val;
    coex_bt_set.start(m_vsqr.m_coex_bt_sqr);
    `uvm_info(get_type_name(), $sformatf("Drive COEX BT RX On-going (%0d)",val), UVM_LOW)
  endtask : set_coex_bt_rx

  task set_coex_bt_event(bit val);
    coex_bt_set_seq coex_bt_set;

    coex_bt_set = coex_bt_set_seq::type_id::create("coex_bt_set");
    coex_bt_set.command = BT_SET_EVENT;
    coex_bt_set.value[`BT_EVENT] = val;
    coex_bt_set.start(m_vsqr.m_coex_bt_sqr);
    `uvm_info(get_type_name(), $sformatf("Drive COEX BT event On-going (%0d)",val), UVM_LOW)
  endtask : set_coex_bt_event

  task set_coex_bt_pti(bit [3:0] val);
    coex_bt_set_seq coex_bt_set;

    coex_bt_set = coex_bt_set_seq::type_id::create("coex_bt_set");
    coex_bt_set.command = BT_SET_PTI;
    coex_bt_set.value[`BT_PTI] = val;
    coex_bt_set.start(m_vsqr.m_coex_bt_sqr);
    `uvm_info(get_type_name(), $sformatf("Drive COEX BT PTI (%0d)",val), UVM_LOW)
  endtask : set_coex_bt_pti

  task set_coex_bt_channel(bit [6:0] val);
    coex_bt_set_seq coex_bt_set;

    coex_bt_set = coex_bt_set_seq::type_id::create("coex_bt_set");
    coex_bt_set.command = BT_SET_CHANNEL;
    coex_bt_set.value[`BT_CHANNEL] = val;
    coex_bt_set.start(m_vsqr.m_coex_bt_sqr);
    `uvm_info(get_type_name(), $sformatf("Drive COEX BT channel (%0d)",val), UVM_LOW)
  endtask : set_coex_bt_channel

  task set_coex_bt_bw(bit val);
    coex_bt_set_seq coex_bt_set;

    coex_bt_set = coex_bt_set_seq::type_id::create("coex_bt_set");
    coex_bt_set.command = BT_SET_BW;
    coex_bt_set.value[`BT_BW] = val;
    coex_bt_set.start(m_vsqr.m_coex_bt_sqr);
    `uvm_info(get_type_name(), $sformatf("Drive COEX BT BW (%0d)",val), UVM_LOW)
  endtask : set_coex_bt_bw

  //--------------------------------------------------------------------
  // AXI DMA tasks
  //--------------------------------------------------------------------

  // ---------------------------------------------------------------------------------------
  // Prepare LLI structure in shared RAM
  // ---------------------------------------------------------------------------------------
  virtual task prepare_lli(
                           bit[14:0] lli_adr,
                           bit[31:0] src_adr,
                           bit[31:0] dst_adr,
                           bit[15:0] num_of_bytes,
                           bit       irq_en     = 1'b0,  // Enable interrupt generation after the end of block handling
                           bit[3:0]  irq_no     = '0,    // IRQ number 0...15
                           bit       counter_en = 1'b0,  // Enable counter increment after the end of block handling
                           bit[3:0]  counter_no = '0,    // Counter number 0...15
                           bit[31:0] nxt_adr    = '0     // Pointer to the next LLI structure
                           );

    `uvm_info(get_type_name(),
    $sformatf("LLI prepared and stored. LLI Adr = %4x. SRC Adr = %4x. DST Adr = %4x Num. of bytes %2d. Next Adr %4x . IRQ En %b, IRQ num. %2d",
    lli_adr, src_adr, dst_adr, num_of_bytes, nxt_adr, irq_en, irq_no), UVM_HIGH)

    write_word_sram(lli_adr + 0, src_adr);   // SRC address
    write_word_sram(lli_adr + 4, dst_adr);   // DST address -> Destination address of a data fragment (for WRITE)
    write_word_sram(lli_adr + 8, { 3'b0, irq_en,irq_no, 3'b0, counter_en, counter_no, num_of_bytes}); // Length + Ctrl
    write_word_sram(lli_adr + 12, nxt_adr);  // Next address field.
  endtask : prepare_lli

  // ---------------------------------------------------------------------------------------
  // Wait until one ore more interrupt flag(s) are set
  // ---------------------------------------------------------------------------------------
  virtual task wait_for_lli_irq(bit[15:0] lli_interrupts);
    uvm_reg_data_t data;
    do begin
      m_regmodel.get_reg_value(data, "INT_RAWSTATUS");
      `uvm_info(get_type_name(), $sformatf("INT_RAWSTATUS = %b.", data), UVM_DEBUG)
    end
    while ((data[15:0] & lli_interrupts) != lli_interrupts);
  endtask : wait_for_lli_irq

  // ---------------------------------------------------------------------------------------
  // Acknowledge one ore more interrupt flag(s)
  // ---------------------------------------------------------------------------------------
  virtual task ack_dma_irq(
                           bit[3:0]  chx_interrupts,
                           bit       err_interrupts,
                           bit[15:0] lli_interrupts
                           );
    m_regmodel.set_reg_value({ chx_interrupts, 3'b0, err_interrupts, lli_interrupts}, "INT_ACK");
  endtask : ack_dma_irq

  // ---------------------------------------------------------------------------------------
  // Data transfer from Shared RAM to AXI RAM
  // ---------------------------------------------------------------------------------------
  virtual task upstream_dma_transfer(
                                     bit[31:0] src_adr,
                                     bit[31:0] dst_adr,
                                     // Number of bytes to transfer
                                     bit[15:0] num_of_bytes,
                                     // LLI location in Shared RAM. By default 0x00
                                     // for Upstream and 0x10 for Downstream
                                     bit[14:0] lli_adr       = 15'h0004,
                                     // DMA channel used for transfer. By default
                                     // CH0/CH1/ for Upstream transfers
                                     bit[2:0]  dma_channel   = 3'h0
                                    );

    string dma_reg_name;

    if(src_adr[1:0] != 0) begin
     `uvm_warning(get_full_name(),  $sformatf("Shared RAM Address is not aligned  = 0x%8x.", src_adr))
    end
    if(dst_adr[1:0] != 0) begin
     `uvm_warning(get_full_name(),  $sformatf("AXI RAM Address is not aligned  = 0x%8x.", dst_adr))
    end
    // Next LLI field does not exist + Counters are not used
    prepare_lli(lli_adr, src_adr, dst_adr, num_of_bytes, 1'b1, dma_channel, '0, '0, '0);

    // Start LLI processing
    dma_reg_name = (dma_channel == 4) ? "CH_LLI_ROOT4" :
                   (dma_channel == 3) ? "CH_LLI_ROOT3" :
                   (dma_channel == 2) ? "CH_LLI_ROOT2" :
                   (dma_channel == 1) ? "CH_LLI_ROOT1" : "CH_LLI_ROOT0";

    m_regmodel.set_reg_value(lli_adr, dma_reg_name);
    // Wait for the end of transfer and clear IRQ status flag
    wait_for_lli_irq(1 << dma_channel);
    ack_dma_irq(0, 0, 1 << dma_channel);

  endtask : upstream_dma_transfer


  // ---------------------------------------------------------------------------------------
  // Data transfer from  AXI RAM to Shared RAM
  // ---------------------------------------------------------------------------------------
  virtual task downstream_dma_transfer(
                                       bit[31:0] src_adr,
                                       bit[31:0] dst_adr,
                                       // Number of bytes to transfer
                                       bit[15:0] num_of_bytes,
                                       // LLI location in Shared RAM. By default
                                       // 0x00 for Upstream and 0x10 for Downstream
                                       bit[14:0] lli_adr       = 15'h0010,
                                       // DMA channel used for transfer. By default
                                       // CH2/CH3 for Downstream/Upstream transfers
                                       bit[2:0]  dma_channel   = 3'h2
                                      );

    string dma_reg_name;

    if(src_adr[1:0] != 0) begin
     `uvm_warning(get_full_name(),  $sformatf("AXI RAM Address is not aligned  = 0x%8x.", src_adr))
    end
    if(dst_adr[1:0] != 0) begin
     `uvm_warning(get_full_name(),  $sformatf("Shared RAM Address is not aligned  = 0x%8x.", dst_adr))
    end
    // Next LLI field does not exist + Counters are not used
    prepare_lli(lli_adr, src_adr, dst_adr, num_of_bytes, 1'b1, dma_channel, '0, '0, '0);

    // Start LLI processing
    dma_reg_name = (dma_channel == 4) ? "CH_LLI_ROOT4" :
                   (dma_channel == 3) ? "CH_LLI_ROOT3" :
                   (dma_channel == 2) ? "CH_LLI_ROOT2" :
                   (dma_channel == 1) ? "CH_LLI_ROOT1" : "CH_LLI_ROOT0";

    m_regmodel.set_reg_value(lli_adr, dma_reg_name);
    // Wait for the end of transfer and clear IRQ status flag
    wait_for_lli_irq(1 << dma_channel);
    ack_dma_irq(0, 0, 1 << dma_channel);

  endtask : downstream_dma_transfer

  // ---------------------------------------------------------------------------------------
  // Data transfer Shared RAM from/to platform
  // ---------------------------------------------------------------------------------------
  virtual task midstream_dma_transfer(
                                       bit[31:0] src_adr,
                                       bit[31:0] dst_adr,
                                       // Number of bytes to transfer
                                       bit[15:0] num_of_bytes,
                                       // LLI location in Shared RAM. By default
                                       // 0x00 for Upstream and 0x10 for Downstream
                                       bit[14:0] lli_adr       = 15'h0020,
                                       // DMA channel used for transfer. By default
                                       // CH0/CH2 for Downstream/Upstream transfers
                                       bit[2:0]  dma_channel   = 3'h4
                                      );

    string dma_reg_name;

    if(src_adr[1:0] != 0) begin
     `uvm_warning(get_full_name(),  $sformatf("AXI RAM Address is not aligned  = 0x%8x.", src_adr))
    end
    if(dst_adr[1:0] != 0) begin
     `uvm_warning(get_full_name(),  $sformatf("Shared RAM Address is not aligned  = 0x%8x.", dst_adr))
    end
    // Next LLI field does not exist + Counters are not used
    prepare_lli(lli_adr, src_adr, dst_adr, num_of_bytes, 1'b1, 4, '0, '0, '0);

    // Start LLI processing
    dma_reg_name = (dma_channel == 4) ? "CH_LLI_ROOT4" :
                   (dma_channel == 3) ? "CH_LLI_ROOT3" :
                   (dma_channel == 2) ? "CH_LLI_ROOT2" :
                   (dma_channel == 1) ? "CH_LLI_ROOT1" : "CH_LLI_ROOT0";

    m_regmodel.set_reg_value(lli_adr, dma_reg_name);
    // Wait for the end of transfer and clear IRQ status flag
    wait_for_lli_irq(1 << dma_channel);
    ack_dma_irq(0, 0, 1 << dma_channel);

  endtask : midstream_dma_transfer

  // ---------------------------------------------------------------------------------------
  // Init. AXI agent
  // ---------------------------------------------------------------------------------------
  virtual task init_axi_agent();
    axi_slave_sequence axi_wr_slv_seq;
    axi_slave_sequence axi_rd_slv_seq;

    axi_wr_slv_seq = axi_slave_sequence::type_id::create("axi_wr_slv_seq");
    axi_rd_slv_seq = axi_slave_sequence::type_id::create("axi_rd_slv_seq");

    fork
      axi_wr_slv_seq.start(m_vsqr.m_axi_vsqr.m_axi_wr_slv_sqr);
      axi_rd_slv_seq.start(m_vsqr.m_axi_vsqr.m_axi_rd_slv_sqr);
    join_none

  endtask : init_axi_agent

  //---------------------------------------------
  // task runs in background, updates ADC samples
  // when RTL sets a new gain value on rf_gpio
  //---------------------------------------------
  virtual task refresh_RF_data();
    forever begin
      fork

        begin : SETUP_GAIN
          wait (m_cfg.m_radio_cfg.m_radio_ctrl_cfg.karst_gain_valid == 1'b1);

          // update noise samples always
          if (m_cfg.m_radio_cfg.m_radio_ctrl_cfg.AGCCount > 0) begin
            // update samples only if data is being sent (not for noise)
            `uvm_info(get_type_name(), $sformatf("Updating RFdata, new rf_gpio value detected (0x%x)",
              m_cfg.m_radio_cfg.m_radio_ctrl_cfg.karst_gain), UVM_HIGH)

            m_mdm_cfg.set_gain(adc_gen_seq.RFdata_re,
                               adc_gen_seq.RFdata_im,
                               m_cfg.m_radio_cfg.m_radio_ctrl_cfg.karst_gain,
                               m_cfg.m_radio_cfg.m_rui_cfg.agcBypass,
                               m_cfg.m_radio_cfg.m_radio_ctrl_cfg.AGCCount);
          end

          wait (m_cfg.m_radio_cfg.m_radio_ctrl_cfg.karst_gain_valid == 1'b0);
        end

        begin : MATLAB_SETGAIN_TIMEOUT
          // DSSS frames take longer TXTIME so timeout needs to be adjusted
          if (   m_mdm_cfg.golden_frame.ppdu_format == NON_HT
              && m_mdm_cfg.golden_frame.preamble_header.leg_rate inside {[0:3]})
            #(20ms); // maximum TXTIME is set for timeout
          else
            #(5.484ms); // maximum TXTIME is set for timeout

          `uvm_warning(get_type_name(),$sformatf("MATLAB setgain timeout occured, refresh RF data stopped!"))
          // signal to all waiting tasks that frame will not be received
          if (adc_gen_seq != null) adc_gen_seq.timeout_occured = 1'b1;
          stop_pooling = 1'b1;
          no_resp_trigger.trigger();
          setgain_timeout = 1'b1;
          #(1us);
          setgain_timeout = 1'b0;
        end

      join_any

      disable fork;
    end//forever
  endtask: refresh_RF_data

  //---------------------------------------------
  // drive PHY inputs
  //---------------------------------------------
  virtual task drive_phy_adc(ref PPDU_frame frm,
                             input bit dont_drive_adc = 0,
                             input bit dont_drive_noise = 0);

    // create sequences
    adc_gen_seq       = modem_stream_gen_samples_seq#(`ADCWIDTH)::type_id::create("adc_gen_seq");
    adc_noagc_gen_seq = rui_gen_samples_seq::type_id::create("adc_noagc_gen_seq");

    // wait 5us
    insert_idle_cycles(5);

    // use the generated PPDU frame to calculate samples for ADC using the matlab model
    rf_gpio = m_cfg.m_radio_cfg.m_radio_ctrl_cfg.karst_gain;
    m_mdm_cfg.execute(direction, frm, rf_gpio, agcBypass);
    // trigger event that new PPDU is executed
    ->m_cfg.new_frame_mdm_exe;
    // skip driving ADC samples
    if (dont_drive_adc) return;

    // get appropriate data depending on if AGC is enabled/disabled
    if ((`AGC_ON == 1) && (agcBypass == 0)) begin
      m_mdm_cfg.get_golden_data(adc_gen_seq.RFdata_re, adc_gen_seq.RFdata_im);
    end else begin
      m_mdm_cfg.get_raw_data(adc_noagc_gen_seq.rxFEout20_re,  adc_noagc_gen_seq.rxFEout20_im,
                             adc_noagc_gen_seq.rxFEout20S_re, adc_noagc_gen_seq.rxFEout20S_im,
                             adc_noagc_gen_seq.rxFEout40_re,  adc_noagc_gen_seq.rxFEout40_im,
                             adc_noagc_gen_seq.rxFEout80_re,  adc_noagc_gen_seq.rxFEout80_im,
                             adc_noagc_gen_seq.bfrSigmadB,    adc_noagc_gen_seq.sigmae2,
                             adc_noagc_gen_seq.snr_dB,        adc_noagc_gen_seq.noise_variance
                            );
      // get RXPARAMTERS
      adc_noagc_gen_seq.rxparams = m_mdm_cfg.get_rxparams();
    end
    // call appropriate sequence depending on if AGC is enabled/disabled
    if ((`AGC_ON == 1) && (agcBypass == 0)) begin
      // stop noise
      if (adc_noise_seq != null) begin
        adc_noise_seq.stop_noise = 1;
      end
      // Send calculated ADC samples using modem_stream_agent
      adc_gen_seq.start(m_vsqr.m_adc_sqr);
    end else begin
      adc_noagc_gen_seq.start(m_vsqr.m_rui_sqr);
    end

    // start generating noise on ADC inputs
    if (!dont_drive_noise)
      drive_noise_adc();

  endtask : drive_phy_adc

  //---------------------------------------------
  // drive
  //---------------------------------------------
  virtual task drive_noise_adc();
    adc_noise_seq = modem_stream_gen_noise_seq#(`ADCWIDTH)::type_id::create("adc_noise_seq");

    // fork off adc_noise_seq to start generating noise on ADC inputs
    fork : START_NOISE_SEQ
      if ((`AGC_ON == 1) && (agcBypass == 0)) begin // thread 1
        // generate noise
        adc_noise_seq.rf_gpio = m_cfg.m_radio_cfg.m_radio_ctrl_cfg.karst_gain;
        adc_noise_seq.start(m_vsqr.m_adc_sqr);
      end
    join_none
  endtask : drive_noise_adc

  //---------------------------------------------
  // wait for TX end signal to assert
  //---------------------------------------------
  task wait_tx_end();
    if (adc_noise_seq != null) begin
      adc_noise_seq.stop_noise = 1;
    end
    fork : WAIT_TX_END
      begin
        wait (m_cfg.m_bus_monitor_cfg.tx_end === 1'b1);
        `uvm_info(get_type_name(),$sformatf("TX END ASSERTED"),UVM_LOW)
      end
      begin
        no_resp_trigger.wait_trigger();
        `uvm_info(get_type_name(),$sformatf("MAC will not responde to this frame"),UVM_LOW)
      end
      begin
        frame_filtered_trigger.wait_trigger();
        `uvm_info(get_type_name(),$sformatf("MAC filtered frame it will not responde"),UVM_LOW)
      end
      begin
        wait (setgain_timeout == 1'b1);
        `uvm_info(get_type_name(),$sformatf("SETGAIN TIMEOUT ASSERTED - no TXEND"),UVM_LOW)
      end
    join_any

    disable fork;
  endtask : wait_tx_end

  //---------------------------------------------
  // wait for TX end signal to assert
  //---------------------------------------------
  task wait_rx_end();
    fork
      wait (setgain_timeout == 1'b1);
      wait (m_cfg.m_bus_monitor_cfg.rx_end === 1'b1);
    join_any

    disable fork;

    if (setgain_timeout)
      `uvm_info(get_type_name(),$sformatf("SETGAIN TIMEOUT ASSERTED - no RXEND"),UVM_LOW)
    else
      `uvm_info(get_type_name(),$sformatf("RX END ASSERTED"),UVM_LOW)
  endtask : wait_rx_end

  //---------------------------------------------
  // write to BFMER memory via AHB
  //---------------------------------------------
  task bfmer_mem_wr(bit [31:0] a, bit [31:0] d);
    ahb_master_direct_seq         bfmer_mem_access;

`ifdef RW_BFMER_EN
    if (a > `BFREPORTMEMEND) begin
      `uvm_error(get_type_name(),$sformatf("BFMER memory address out of range! End address 0x%h",`BFREPORTMEMEND))
    end

    bfmer_mem_access = ahb_master_direct_seq::type_id::create("bfmer_mem_access");
    assert (bfmer_mem_access.randomize() with {
      addr       == `BFREPORTMEMSTART + a;
      data[0]    == d;
      access     == WRITE;
      burst_size == 1;
      size       == WORD;
    });
    bfmer_mem_access.start(m_vsqr.m_ahb_master_sqr);
    `uvm_info(get_type_name(),$sformatf("BFMER MEM WRITE: @0x%0h - data 0x%0h",a,d), UVM_HIGH)
`else
    `uvm_info(get_type_name(), $sformatf("RW_BFMER_EN not defined task will do nothing"), UVM_LOW)
`endif
  endtask : bfmer_mem_wr

  //---------------------------------------------
  // store Beamforming report to BFMER memory via AHB
  //---------------------------------------------
  virtual task store_bfmer_report_ahb(ref PPDU_frame frm,
                                      input int user = 0,
                                      input int id = 0);

    bit [15:0]    bfmer_header;
    bit [7:0]     tx_bfmer_report[];
    int           mod;
    int           byte_cnt;
    int           addr;
    bit [31:0]    data;
    bit [16:0]    ptr_addr;

    // create Beamformer header from Matlab parameters
    bfmer_header = {4'h0,
                    m_mdm_cfg.Feedback[id],
                    m_mdm_cfg.Codebook[id],
                    m_mdm_cfg.Grouping[id],
                    frm.preamble_header.ch_bw,
                    m_mdm_cfg.Nr,
                    m_mdm_cfg.Nc[id] };

    // get Beamforming report from Matlab and store to memory
    mod = ((m_mdm_cfg.txBfrLength[id]+16) % 32); // memory access is 32bits
    mod = (mod) ? (32-mod)/8 : 0;
    // allocate memory for report + 2 bytes for BFMER header + reminder to
    // fill up to 32 bit data
    tx_bfmer_report = new[m_mdm_cfg.txBfrLength[id]/8 + 2 + mod];
    tx_bfmer_report[0] = bfmer_header[7:0];
    tx_bfmer_report[1] = bfmer_header[15:8];

    byte_cnt = 2;
    for (int i=0; i<m_mdm_cfg.txBfrLength[id]; i=i+8) begin
      for (int j=0; j<8; j++) begin
        tx_bfmer_report[byte_cnt][j] = m_mdm_cfg.txBfrCompressedReport[id][i+j];
      end
      byte_cnt++;
    end
    // store pointer to BFM report
    addr = {frm.preamble_header.user_header[user].smm_index_f[7:1], 1'b0} * 2;
    data = bfmer_mem[addr];
    ptr_addr = 16'h0200 + (16'h0200 * user);
    // determine upper or lower 16 bits of word
    if (frm.preamble_header.user_header[user].smm_index_f[0]) begin
      data[31:16] = ptr_addr;
      bfmer_mem[addr] = data;
    end
    else begin
      data[15:0] = ptr_addr;
      bfmer_mem[addr] = data;
    end

    // write pointer
    bfmer_mem_wr(addr, data);
    // store data word to memory
    for (int i=0; i<tx_bfmer_report.size(); i=i+4) begin
      bfmer_mem_wr(ptr_addr, {tx_bfmer_report[i+3],
                              tx_bfmer_report[i+2],
                              tx_bfmer_report[i+1],
                              tx_bfmer_report[i]});
      ptr_addr += 4;
    end

    `uvm_info(get_type_name(),
    $sformatf("BFMER report stored in memory, user=%0d, STAID=%0d",user,id+1),UVM_LOW)
  endtask : store_bfmer_report_ahb

  //---------------------------------------------
  // Use MAC to calculate time on air
  //---------------------------------------------
  task sw_time_on_air_calculation(ref PPDU_frame frm, output [15:0] time_on_air);

    int        user;
    int        nsts_total;
    bit [2:0]  nss;
    bit [6:0]  mcs;

    if (frm.kind == MU_MIMO) begin
      user = frm.user_order[0];
      nsts_total = 0;
      foreach (frm.preamble_header.user_header[i])
        nsts_total += frm.preamble_header.user_header[i].num_sts_f + 1;
    end
    else begin
      user = 0;
      nsts_total = frm.preamble_header.user_header[0].num_sts_f;
    end

    if (frm.preamble_header.format_mod >= HE_SU) begin
      mcs =  frm.preamble_header.user_header_he[user].mcs_f;
    end
    else if (frm.preamble_header.format_mod == VHT) begin
      nss = (frm.preamble_header.stbc > 0) ? (nsts_total+1)/(2*frm.preamble_header.stbc)-1 : nsts_total;
      mcs = {nss[2:0],frm.preamble_header.user_header[user].mcs_f[3:0]};
    end
    else begin
      mcs =  frm.preamble_header.user_header[user].mcs_f;
    end

    // Write the paramaters for TX time calculation
    // to timeOnAirParamReg and timeOnAirParam2Reg
    if (frm.preamble_header.format_mod >= HE_SU)
      wdata[19:0] = frm.preamble_header.user_header_he[0].he_length_f;
    else if (frm.preamble_header.format_mod inside {HT_MF,HT_GF,VHT})
      wdata[19:0] = frm.preamble_header.user_header[0].ht_length_f;
    else
      wdata[19:0] = frm.preamble_header.leg_length;
    //format mode
    if (frm.preamble_header.format_mod inside {NON_HT, NON_HT_DUP_OFDM})
      wdata[23:20] = frm.preamble_header.preamble_type;
    else
      wdata[23:20] = frm.preamble_header.format_mod;

    wdata[25:24]  = frm.preamble_header.gi_type;
    wdata[27:26]  = frm.preamble_header.ch_bw;
    wdata[29:28]  = frm.preamble_header.num_extn_ss;
    wdata[31:30]  = frm.preamble_header.stbc;
    m_regmodel.set_reg_value(wdata, "TIMEONAIRPARAM1REG");

    if (frm.preamble_header.format_mod inside {NON_HT, NON_HT_DUP_OFDM}) begin
      wdata[6:0]  = get_mcs_from_legacy_rate(frm.preamble_header.leg_rate);
      wdata[31:7] = 0;
    end
    else if (frm.preamble_header.format_mod >= HE_SU) begin
      wdata[6:0]   = mcs;
      wdata[7]     = 0;
      wdata[8]     = frm.preamble_header.dcm;
      wdata[10:9]  = frm.preamble_header.he_ltf_type;
      wdata[11]    = frm.preamble_header.sig_b_comp_mode;
      wdata[12]    = frm.preamble_header.dcm_sig_b;
      wdata[15:13] = frm.preamble_header.mcs_sig_b;
      wdata[18:16] = frm.preamble_header.user_header_he[0].pkt_ext_f;
      wdata[19]    = 0;
      wdata[26:20] = frm.preamble_header.user_header_he.size();
      wdata[27]    = 0;
      wdata[30:28] = frm.preamble_header.num_he_ltf;
      wdata[31]    = 0;
    end
    else begin
      wdata[6:0]  = mcs;
      wdata[31:7] = 0;
    end
    m_regmodel.set_reg_value(wdata, "TIMEONAIRPARAM2REG");

    wdata = 0;
    if (frm.preamble_header.format_mod >= HE_SU) begin
      wdata[0] = frm.preamble_header.doppler;
      wdata[1] = frm.preamble_header.midamble_periodicity;
      wdata[6:4] = (frm.preamble_header.format_mod == HE_TB) ? get_ru_type(frm.preamble_header.ru_allocation)             :
                   (frm.preamble_header.format_mod == HE_MU) ? get_ru_type_from_ru_allocation(
                                                                 frm.preamble_header.ru_allocation,
                                                                 frm.preamble_header.user_header_he[user].dut_location_f) :
                                                               get_ru_type_he_su(frm.preamble_header.ch_bw);
    end
    m_regmodel.set_reg_value(wdata, "TIMEONAIRPARAM3REG");

    wdata[31] = 1;
    m_regmodel.set_reg_value(wdata,"TIMEONAIRVALUEREG");
    // wait for calculator to finish
    do begin
      m_regmodel.get_reg_value(rdata, "TIMEONAIRVALUEREG");
    end while (rdata[30] != 1'b1);

    time_on_air = rdata[15:0];

  endtask : sw_time_on_air_calculation

  //----------------------------------------------------------------------
  // function that determines does MAC need to respond to received frame or
  // not. Returns 1 if response is needed else returns 0
  //----------------------------------------------------------------------
  function bit responde_to_frame(ref PPDU_frame frm, input mac_address_t dev_addr);
    mpdu_frame_type_e     frm_type;
    CONTROL_WRAPPER_frame ctrl_wrapp;
    he_control_s          he_ctrl;
    mac_ctrl_1_reg_s      mac_ctrl_1;
    bit                   has_qos_data;
    int                   user;

    mac_ctrl_1 = m_regmodel.get_mirrored_reg_value("MACCNTRL1REG");

    // don't responde to bcast and mcast frames
    if (   dev_addr == `MULTICAST_ADDR
        || dev_addr == `BROADCAST_ADDR) begin
      return 0;
    end

    // handle aggregated frames
    if (frm.kind inside {AGGREGATED, MU_MIMO}) begin
      // find user which is received
      user = (frm.kind == MU_MIMO || frm.preamble_header.ru_with_more_users)
             ? frm.preamble_header.mu_mimo_userid
             : 0;
      // get HE control
      he_ctrl = get_ht_control(frm, user);
      // look if there are QOS_DATA MPDUs
      has_qos_data = 0;
      foreach (frm.ampdu_frame[user].mpdu_frame_type[i]) begin
        if (frm.ampdu_frame[user].mpdu_frame_type[i] == QOS_DATA)
          has_qos_data = 1;
      end

         // normal ACK
      if ((   frm.ampdu_frame[user].get_MAC_addr(.RA(1)) == dev_addr
           && frm.ampdu_frame[user].get_qos_ack_policy() == 2'b00)
          || // HTP ack
          (   frm.ampdu_frame[user].get_MAC_addr(.RA(1)) == dev_addr
           && frm.ampdu_frame[user].get_qos_ack_policy() == 2'b10
           && has_qos_data
           && (he_ctrl.vht_f && he_ctrl.he_f && he_ctrl.ctrl_id_f == 0) //TRS control subfield
           && frm.ppdu_format inside {HE_SU, HE_EXT_SU, HE_MU}))
        return 1;
      else
        return 0;
    end

    // singleton frames
    frm_type = frm.ampdu_frame[0].mpdu_frame_type[0];

    // control wrapper frame carries the real frame type which
    // MAC needs to analyze
    if (frm_type == CONTROL_WRAPPER) begin
      // type cast frame to control wrapper
      ctrl_wrapp = CONTROL_WRAPPER_frame'(frm.ampdu_frame[0].mpdu_frame[0]);
      frm_type = ctrl_wrapp.ctrl_frame_type;
    end

    if (m_cfg.channel_busy && frm_type == RTS)
      return 0;

    case (frm_type)
      QOS_DATA, QOS_DATA_CF_ACK, QOS_DATA_CF_POLL,
      QOS_DATA_CF_ACK_CF_POLL,
      QOS_NULL, QOS_CF_POLL, QOS_CF_ACK_CF_POLL: begin
        // get HE control
        he_ctrl = get_ht_control(frm);
           // normal ACK
        if ((   frm.ampdu_frame[0].get_MAC_addr(.RA(1)) == dev_addr
             && frm.ampdu_frame[0].get_qos_ack_policy() == 2'b00)
            || // HTP ack
            (   frm.ampdu_frame[0].get_MAC_addr(.RA(1)) == dev_addr
             && frm.ampdu_frame[0].get_qos_ack_policy() == 2'b10
             && (he_ctrl.vht_f && he_ctrl.he_f && he_ctrl.ctrl_id_f == 0) //TRS control subfield
             && frm_type inside {QOS_DATA, QOS_NULL}
             && frm.ppdu_format inside {HE_SU, HE_EXT_SU, HE_MU}))
          return 1;
        else
          return 0;
      end

      ACK, CTS, CF_END, CF_END_CF_ACK,
      BLOCK_ACK, ACTION_NOACK, BEACON,
      CF_ACK, CONTROL_WRAPPER, VHT_NDP_ANNOUNCEMENT: begin
        return 0;
      end // no response frames

      BLOCK_ACK_REQUEST: begin
        if (   frm.ampdu_frame[0].get_MAC_addr(.RA(1)) == dev_addr
            && get_bar_variant(frm) inside {BASIC_BAR, COMPRESSED_BAR, EXT_COMPRESSED_BAR})
          return 1;
        else
          return 0;
      end

      RTS: begin
        if (mac_ctrl_1.disable_cts_resp_f)
          return 0;
        else
          return 1;
      end

      default: begin
        if (frm.ampdu_frame[0].get_MAC_addr(.RA(1)) == dev_addr)
          return 1;
        else
          return 0;
      end
    endcase
  endfunction : responde_to_frame

  //----------------------------------------------------------------------
  // overwrite register model with values from file
  //----------------------------------------------------------------------
  task overwrite_regmodel();
    int                 fhandle, fstatus;
    string              fname;
    string              reg_name;
    uvm_reg_data_t      wdata;

    fname = {`STRINGIFY(`SYSPARAM_TXT_DIR),"/",`REGMODEL_DUMP_FILE};
    fhandle = $fopen(fname, "r");
    `uvm_info(get_type_name(),"Overwrite register model",UVM_LOW)
    // if regmodel dump file exist overwrite registers
    if (fhandle) begin
      while (!$feof(fhandle)) begin
        fstatus = $fscanf(fhandle,"%s %h",reg_name,wdata);
        `uvm_info(get_type_name(), $sformatf("Reg %s, value %0h - status %0d",reg_name,wdata,fstatus), UVM_DEBUG)
        m_regmodel.set_reg_value(wdata, reg_name);
      end//while
    end//if
    $fclose(fhandle);
  endtask : overwrite_regmodel

  //----------------------------------------------------------------------
  // overwrite RU allocation inside HE control to proper value, relative to
  // channel bandwith and current primary channel setup
  //----------------------------------------------------------------------
  function void set_he_ctrl_ru_allocation(ref PPDU_frame frm, ref he_control_s he_ctrl, input int user = 0);
    bit [3:0] mdmcfg_bw;
    bit [2:0] mdmcfg_primary;
    int       psdu_len;

    // get modem BW configuration value
    rdata = m_regmodel.get_mirrored_reg_value("MDMCONF");
    mdmcfg_bw = rdata[3:0];
    // get primary channel value
    rdata = m_regmodel.get_mirrored_reg_value("PRIMARYIND");
    mdmcfg_primary = rdata[2:0];

    assert (randomize(he_tb_ru_allocation) with {
      he_tb_ru_allocation[0] == 0;

      // 20MHz
      if (frm.preamble_header.ch_bw == 2'b00) {
        he_tb_ru_allocation[7:1] inside {[0:8],[37:40],[53:54],61};
      }
      // 40MHz
      else if (frm.preamble_header.ch_bw == 2'b01) {
        if (mdmcfg_bw == 2'b00) {
          if (mdmcfg_primary == 1)
            he_tb_ru_allocation[7:1] inside {[9:17],[41:44],[55:56],62};
          else
            he_tb_ru_allocation[7:1] inside {[0:8],[37:40],[53:54],61};
          // RU restrictions for 20MHz operations
          he_tb_ru_allocation[7:1] != 4;  // RU index 5
          he_tb_ru_allocation[7:1] != 13; // RU index 14
        } else {
          he_tb_ru_allocation[7:1] inside {[0:17],[37:44],[53:56],61,62,65};
        }
      }
      // 80MHz
      else if (frm.preamble_header.ch_bw == 2'b10) {
        if (mdmcfg_bw == 2'b00) {
          if (mdmcfg_primary == 0)
            he_tb_ru_allocation[7:1] inside {[0:8],[37:40],[53:54]};
          else if (mdmcfg_primary == 1)
            he_tb_ru_allocation[7:1] inside {[10:12],[14:17],[42:44],56};
          else if (mdmcfg_primary == 2)
            he_tb_ru_allocation[7:1] inside {[19:22],[24:26],[45:47],57};
          else
            he_tb_ru_allocation[7:1] inside {[28:31],[33:36],[49:52],[59:60]};
          // RU restrictions for 20MHz operations
          //                  RU26
          he_tb_ru_allocation[7:1] != 4;  // RU index 5
          he_tb_ru_allocation[7:1] != 9;  // RU index 10
          he_tb_ru_allocation[7:1] != 13; // RU index 14
          he_tb_ru_allocation[7:1] != 18; // RU index 19
          he_tb_ru_allocation[7:1] != 23; // RU index 24
          he_tb_ru_allocation[7:1] != 27; // RU index 28
          he_tb_ru_allocation[7:1] != 32; // RU index 33
          //                  RU52
          he_tb_ru_allocation[7:1] != 41; // RU index 5
          he_tb_ru_allocation[7:1] != 48; // RU index 12
          //                  RU106
          he_tb_ru_allocation[7:1] != 55; // RU index 3
          he_tb_ru_allocation[7:1] != 58; // RU index 6
        } else {
          he_tb_ru_allocation[7:1] inside {[0:67]};
        }
      }
    });
    // set new RU allocation value
    he_ctrl.ctrl_info_f.ru_allocation_f = he_tb_ru_allocation;

    // check if UL symbol number fits PSDU length for response ACK frame
    // carried in HE-TB (TRS)
    while (1) begin
      psdu_len = GenHeTbTrs (.ruLen_i          (get_ru_type(he_ctrl.ctrl_info_f.ru_allocation_f)),
                             .heLtfType_i      (frm.preamble_header.he_ltf_type),
                             .giType_i         (frm.preamble_header.gi_type),
                             .packetExtension_i(frm.preamble_header.user_header_he[0].pkt_ext_f),
                             .mcsIndex_i       (he_ctrl.ctrl_info_f.ul_mcs_f),
                             .dcm_i            (frm.preamble_header.dcm),
                             .numSymb_i        (he_ctrl.ctrl_info_f.ul_data_symbols_f),
                             .funcParamRet_i   (RET_PSDU_LENGTH),
                             .debug_i          (0));
      if (psdu_len < 36) begin
        if (he_ctrl.ctrl_info_f.ul_data_symbols_f == 5'd31) begin
          he_ctrl.ctrl_info_f.ul_mcs_f = (he_ctrl.ctrl_info_f.ul_mcs_f == 1 && frm.preamble_header.dcm == 1)
                                         ? 3 //can't increment to value MCS2
                                         : he_ctrl.ctrl_info_f.ul_mcs_f + 1;
          he_ctrl.ctrl_info_f.ul_data_symbols_f = 0;
        end
        else begin
          he_ctrl.ctrl_info_f.ul_data_symbols_f = he_ctrl.ctrl_info_f.ul_data_symbols_f + 1;
        end
      end
      else begin
        break;
      end
    end//while
    // overwrite HE control in AMPDU
    set_ht_control(frm, he_ctrl, user);

  endfunction : set_he_ctrl_ru_allocation

endclass : wlan_seq_base

`endif //WLAN_SEQ_BASE_SV
