//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef WLAN_TX_DSSS_FRAME_SEQ_SV
`define WLAN_TX_DSSS_FRAME_SEQ_SV


class wlan_tx_dsss_frame_seq extends wlan_seq_base;

  `uvm_object_utils(wlan_tx_dsss_frame_seq)

  PPDU_frame                 data_frame;
  PPDU_frame                 cts_frame;
  PPDU_frame                 ack_frame;
  PPDU_frame                 ba_frame;
  BLOCK_ACK_frame            ba_mpdu;

  mac_ctrl_info_1_s          mac_ctrl_info_1;
  mac_ctrl_info_2_s          mac_ctrl_info_2;

  int                        ksr_index;
  int                        frame_num;
  rand nav_prot_frm_e        prot;
  rand expect_ack_e          expect_ack;
  rand access_category_e     ac;
  real                       invcarrierfreq;

  //---------------------------------------------
  // constraints
  //---------------------------------------------
  constraint c_carrier_freq {
    carrier_freq inside {2412, 2417, 2422, 2427, 2432, 2437, 2442,
                         2447, 2452, 2457, 2462, 2467, 2472, 2484};
  }

  function new (string name = "wlan_tx_dsss_frame_seq");
    super.new (name);
  endfunction : new

  //---------------------------------------------
  // constraints
  //---------------------------------------------
  constraint c_prot_ack {
    prot == NO_PROT;
    ac inside {AC_BE,AC_BK,AC_VI,AC_VO};
    expect_ack == NO_ACK;
  }
  //---------------------------------------------

  virtual task body();
    super.body();

    // setup DUT registers
    m_regmodel.set_field_value(0, "OFDMONLY", "RWNXAGCCNTL", "RIUKARST");

    // Make sure to set 2.4GHz for DSSS
    invcarrierfreq = (2**26)/real'(carrier_freq);
    m_regmodel.set_field_value(int'(invcarrierfreq),"INVCARRIERFREQ");
    // set PSSELECT
    m_regmodel.set_field_value(2'b01, "PSSELECT", "RWNXMACSTATICCONFIG", "RIUKARST");
    // Set the PRIMARYIND register
    m_regmodel.set_reg_value(0, "PRIMARYIND","PHYCONFIG");

    agcBypass = m_cfg.m_radio_cfg.m_rui_cfg.agcBypass; // set from testcase

    //--------------------------------------------
    // determine device and BSS address
    //--------------------------------------------
    create_keystorageRAM();

    create_dev_and_bss_addr();

    // prepare SRAM for Rx, with smaller Rx segment
    prepare_sram_for_rx(1000, 1);
    // don't accept control frames
    m_regmodel.set_field_value(1'b0, "acceptBA",  "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptNotExpectedBA", "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptBAR", "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptRTS", "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptCTS", "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptACK", "RXCNTRLREG");

    // put MAC core to active state
    change_mac_core_state(`ACTIVE_STATE);

    frame_num = $urandom_range(3,5);
    for (int loop=0; loop < frame_num; loop++) begin
      `uvm_info(get_type_name(),$sformatf("Transmit Frame number: %0d / %0d",
      loop+1,frame_num),UVM_LOW)

      data_frame = new("data_frame");
      cts_frame  = new("cts_frame");
      ack_frame  = new("ack_frame");
      ba_frame   = new("ba_frame");
      ba_mpdu    = new("ba_mpdu");

      // randomize fields in sequence
      assert (this.randomize());

      //--------------------------------
      // create data frame for Rx
      //--------------------------------

      assert (data_frame.randomize() with {
        kind == SINGLETON;
        ppdu_format == NON_HT;
        tx_frame == 1;
      });

      // force DSSS frames
      data_frame.preamble_header.leg_rate = $urandom_range(0,3);
      if (data_frame.preamble_header.leg_rate == 0) begin
        data_frame.preamble_header.preamble_type = 1;
      end

      data_frame.ampdu_frame[0].clear_all_arrays();
      data_frame.ampdu_frame[0].is_dsss = 1'b1;
      assert (data_frame.ampdu_frame[0].randomize() with {
        data_frame.ampdu_frame[0].ppdu_format == data_frame.ppdu_format;
        data_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
        data_frame.ampdu_frame[0].mpdu_frame_type[0] inside {DATA,
                                                             DATA_CF_ACK,
                                                             DATA_CF_POLL,
                                                             DATA_CF_ACK_CF_POLL};
        data_frame.ampdu_frame[0].aggregated == 0;
      });

      ksr_index = get_random_ksr_index(data_frame);
      // set MAC addresses
      data_frame.ampdu_frame[0].set_MAC_addr(
        .TA    (dev_addr),
        .RA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f),
        .BSSID (bss_addr)
      );
      // encrypt frame if protection bit is set
      data_frame.encrypt(.ra(1));
      data_frame.calc_leg_ht_length();

      `uvm_info(get_type_name(), $sformatf("Data frame:%s", data_frame.sprint()), UVM_LOW)

      // configure MAC control 1 field for THD
      mac_ctrl_info_1.prot_frm_dur_f   = 16'd0;
      mac_ctrl_info_1.write_ack_f      = 0;
      mac_ctrl_info_1.low_rate_retry_f = 0;
      mac_ctrl_info_1.lstp_prot_f      = 0;
      mac_ctrl_info_1.lstp_f           = 0;
      mac_ctrl_info_1.expect_ack_f     = expect_ack; //normal ACK

      // configure MAC control 2 field for THD
      mac_ctrl_info_2.dont_generate_mh_f = 0;
      mac_ctrl_info_2.dont_encrypt_f     = 0;
      mac_ctrl_info_2.dont_touch_fc_f    = 0;
      mac_ctrl_info_2.dont_touch_dur_f   = 0;
      mac_ctrl_info_2.dont_touch_qos_f   = 0;
      mac_ctrl_info_2.dont_touch_htc_f   = 0;
      mac_ctrl_info_2.dont_touch_tsf_f   = 0;
      mac_ctrl_info_2.dont_touch_dtim_f  = 0;
      mac_ctrl_info_2.dont_touch_fcs_f   = 0;
      mac_ctrl_info_2.under_ba_setup_f   = 0;
      mac_ctrl_info_2.num_blank_delimit_f = 0;
      mac_ctrl_info_2.int_en_f            = 1;

      // determine type of NAV protection frame exchange
      `uvm_info(get_type_name(),$sformatf("NAV protection %s",prot.name()),UVM_LOW)
      // expected ACK
      `uvm_info(get_type_name(),$sformatf("Expected ACK: %s",expect_ack.name()),UVM_LOW)
      // access category
      `uvm_info(get_type_name(),$sformatf("Access category %s",ac.name()),UVM_LOW)

      // Store PPDU frame in SRAM for trasmit
      wr_frame_to_sram(data_frame,
                       mac_ctrl_info_1,
                       mac_ctrl_info_2,
                       prot,
                       32'h80);

      //--------------------------------
      // create CTS frame for Rx
      //--------------------------------
      assert (cts_frame.randomize() with {
        kind == SINGLETON;
        ppdu_format == NON_HT;
        tx_frame == 0;
      });

      cts_frame.ampdu_frame[0].clear_all_arrays();
      assert (cts_frame.ampdu_frame[0].randomize() with {
        cts_frame.ampdu_frame[0].ppdu_format == cts_frame.ppdu_format;
        cts_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
        cts_frame.ampdu_frame[0].mpdu_frame_type[0] == CTS;
        cts_frame.ampdu_frame[0].aggregated == 0;
      });
      cts_frame.calc_leg_ht_length();

      // set MAC addresses
      cts_frame.ampdu_frame[0].set_MAC_addr(
        .RA    (dev_addr)
      );

      //--------------------------------
      // create ACK frame for Rx
      //--------------------------------
      assert (ack_frame.randomize() with {
        kind == SINGLETON;
        ppdu_format == NON_HT;
        tx_frame == 0;
      });

      ack_frame.ampdu_frame[0].clear_all_arrays();
      assert (ack_frame.ampdu_frame[0].randomize() with {
        ack_frame.ampdu_frame[0].ppdu_format == ack_frame.ppdu_format;
        ack_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
        ack_frame.ampdu_frame[0].mpdu_frame_type[0] == ACK;
        ack_frame.ampdu_frame[0].aggregated == 0;
      });
      ack_frame.calc_leg_ht_length();

      // set MAC addresses
      ack_frame.ampdu_frame[0].set_MAC_addr(
        .RA    (dev_addr)
      );

      // set Tx interrupt masks for AC
      set_ac_interrupts(ac);

      set_DMA_for_tx(ac);

      fork
        begin : RST_CTS_DATA_TX
          //------------------------------------------------
          // collect Tx RTS/CTS protection frames
          //------------------------------------------------
          if (prot != NO_PROT) begin
            wait_tx_end();// wait protection frame
            drive_noise_adc();
          end

          //------------------------------------------------
          // send CTS
          //------------------------------------------------
          if (prot inside {RTS_CTS, RTS_CTS_QAP}) begin
            create_ppdu_response_frame(cts_frame);
            drive_phy_adc(cts_frame);
          end

          //------------------------------------------------
          // collect Tx data frame
          //------------------------------------------------
          wait_tx_end();
          drive_noise_adc();
          //------------------------------------------------
          // send ACK
          //------------------------------------------------
          if (expect_ack == NORMAL_ACK) begin
            create_ppdu_response_frame(ack_frame);
            drive_phy_adc(ack_frame);
          end
        end//RST_CTS_DATA_TX

        wait_mac_tx_trigger(ac);

      join
      // read THD status from SRAM
      rd_tx_status_from_sram(
        .head     (m_cfg.m_sram_cfg.thd_head_ptr),
        .skip_athd((expect_ack == NO_ACK) ? 0 : 1) // read A-THD if BAR is sent
      );

      insert_idle_cycles(10);
      // clear interrupt flag
      clear_ac_interrupts(ac);

    end // for loop

  endtask : body


endclass : wlan_tx_dsss_frame_seq

`endif // WLAN_TX_DSSS_FRAME_SEQ_SV
