//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef WLAN_TX_MUMIMO_FRAME_SEQ_SV
`define WLAN_TX_MUMIMO_FRAME_SEQ_SV


class wlan_tx_mumimo_frame_seq extends wlan_seq_base;

  `uvm_object_utils(wlan_tx_mumimo_frame_seq)

  PPDU_frame                 mumimo_frame;
  PPDU_frame                 bfr_frame;
  PPDU_frame                 ndp_frame;
  PPDU_frame                 ndpa_frame;

  mac_ctrl_info_1_s          mac_ctrl_info_1;
  mac_ctrl_info_2_s          mac_ctrl_info_2;

  int                        ksr_index;
  int                        frame_num;
  bit [31:0]                 user1_addr;
  bit [31:0]                 thd_addr_q[$]; // save THD address
  rand nav_prot_frm_e        prot;
  rand expect_ack_e          expect_ack;
  rand access_category_e     ac;
  bit [7:0]                  smm_index[2];

  //--------------------------------------------------------------------
  // variables for NDP
  //--------------------------------------------------------------------
  rand bit  [2:0] Nr;                         // Nr Value To Test
  rand bit  [2:0] Nc[`NB_STA_MAX];            // Nc Value To Test
  rand bit  [1:0] Grouping[`NB_STA_MAX];      // Grouping Value To Test
  rand bit        FeedbackType[`NB_STA_MAX];  // Feedback Type Value To Test
  rand bit        Codebook[`NB_STA_MAX];      // Codebook Value To Test

  rand int        STA_id[2];
  rand bit [2:0]  num_of_stations;   // number of stations, used for BF Tx test
  rand bit [1:0]  ch_bw;


  //--------------------------------------------------------------------
  // Constraints
  //--------------------------------------------------------------------

  constraint c_Nr {
    Nr == 1;
  }
  constraint c_FeedbackType {
    foreach (FeedbackType[i]) FeedbackType[i] inside {0,1};
    foreach (Codebook[i]) Codebook[i] inside  {0,1};
  }

  constraint c_Nc {
    solve Nr before Nc;
    foreach (Nc[i]) Nc[i] == 0;
  }

  constraint c_Grouping {
    foreach (Grouping[i]) Grouping[i] inside {0,1,2};
  }

  constraint c_channel_bw {
    ch_bw == channel_bw_func(VHT);
  }

  constraint c_num_of_stations {
    num_of_stations inside {[2:4]};
  }

  constraint c_staid {
    unique {STA_id};
    STA_id[0] < num_of_stations;
    STA_id[1] < num_of_stations;
  }

  constraint c_prot_ack {
    prot == NO_PROT;
    ac   inside {AC_VI,AC_VO};
    expect_ack == NO_ACK;
  }

  function new (string name = "wlan_tx_mumimo_frame_seq");
    super.new (name);
  endfunction : new

  function void print_ndp_params(int i=0);
   `uvm_info(get_type_name(), $sformatf("*******************************************************"), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("Transmition of VHT NDP Sounding procedure user %0d", i), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Nr                : %2d", (Nr+1)        ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Nc                : %2d", (Nc[i]+1)        ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Grouping          : %2d", Grouping[i]      ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Feedback type     : %s" , (FeedbackType[i]==1'b0)? "SU" : "MU"), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Codebook          : %2d", Codebook[i]   ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("   Channel BW        : %2d", ch_bw   ), UVM_LOW)
   `uvm_info(get_type_name(), $sformatf("*******************************************************"), UVM_LOW)
  endfunction: print_ndp_params

  function void set_ndp_params();
    m_mdm_cfg.Nr = Nr;
    m_mdm_cfg.STAIDList[0] = STA_id[0];
    m_mdm_cfg.STAIDList[1] = STA_id[1];
    m_mdm_cfg.nBeamformingSTA = num_of_stations;

    for (int i=0; i<num_of_stations; i++) begin
      m_mdm_cfg.Nc[i]       = Nc[i];
      m_mdm_cfg.Grouping[i] = Grouping[i];
      m_mdm_cfg.Feedback[i] = FeedbackType[i];
      m_mdm_cfg.Codebook[i] = Codebook[i];
      print_ndp_params(i);
    end
    foreach(STA_id[i])
      `uvm_info(get_type_name(), $sformatf("Station ID: %2d", STA_id[i]+1), UVM_LOW)
  endfunction : set_ndp_params

  virtual task body();
    super.body();

    // setup DUT registers
    m_regmodel.set_field_value(0, "OFDMONLY", "RWNXAGCCNTL", "RIUKARST");

    agcBypass = m_cfg.m_radio_cfg.m_rui_cfg.agcBypass; // set from testcase

    // fork off gain update task
    fork  : GAIN_UPDATE
      if ((`AGC_ON == 1) && (agcBypass == 0))
        refresh_RF_data(); // call AGC update task only if necessary

      drive_noise_adc();
    join_none

    //--------------------------------------------
    // determine device and BSS address
    //--------------------------------------------
    create_keystorageRAM(1,3,20);

    // prevent WAPI encryption type
    foreach(m_ksr.keyRAM[i]) begin
      if (m_ksr.keyRAM[i].ctype_ram_f == WAPI) begin
        m_ksr.keyRAM[i].ctype_ram_f = CCMP;
        m_ksr.keyRAM[i].clen_ram_f = 2'b00;
        m_ksr.keyRAM[i].use_def_key_ram_f = 0;
      end
    end

    // overwrite keyStorageRAM with new constraints
    overwrite_keystorageRAM();

    create_dev_and_bss_addr();

    m_regmodel.set_field_value(1'b1, "maskrxBuffer1Trigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "masktimerRxTrigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "maskrxBuffer2Trigger", "TXRXINTUNMASKREG");
    m_regmodel.set_field_value(1'b1, "masterTxRxIntEn", "TXRXINTUNMASKREG"); // Enables interrupt generation

    // prepare SRAM for Rx, with smaller Rx segment
    prepare_sram_for_rx(5000, 1);
    prepare_sram_priority_rx(1000);
    // don't accept control frames
    m_regmodel.set_field_value(1'b0, "acceptBA", "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptNotExpectedBA", "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptBAR", "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptRTS", "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptCTS", "RXCNTRLREG");
    m_regmodel.set_field_value(1'b0, "acceptACK", "RXCNTRLREG");

    // put MAC core to active state
    change_mac_core_state(`ACTIVE_STATE);

    mumimo_frame = new("mumimo_frame");
    bfr_frame  = new("bfr_frame");
    ndp_frame  = new("ndp_frame");
    ndpa_frame = new("ndpa_frame");

    // randomize fields in sequence
    assert (this.randomize());
    set_ndp_params();

    //--------------------------------
    // create NDPA frame for Tx
    //--------------------------------
    assert (ndpa_frame.randomize() with {
      kind == SINGLETON;
      ppdu_format == VHT;
      tx_frame == 0;
    });
    ndpa_frame.preamble_header.ch_bw = ch_bw;

    ndpa_frame.ampdu_frame[0].clear_all_arrays();
    assert (ndpa_frame.ampdu_frame[0].randomize() with {
      ndpa_frame.ampdu_frame[0].ppdu_format == ndpa_frame.ppdu_format;
      ndpa_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
      ndpa_frame.ampdu_frame[0].mpdu_frame_type[0] == VHT_NDP_ANNOUNCEMENT;
      ndpa_frame.ampdu_frame[0].aggregated == 0;
    });
    if (ndpa_frame.preamble_header.ch_bw == 0 && ndpa_frame.preamble_header.user_header[0].mcs_f == 7'd9) begin
      ndpa_frame.preamble_header.user_header[0].mcs_f = $urandom_range(0,8);
    end
    // set MAC addresses
    ndpa_frame.ampdu_frame[0].set_MAC_addr(
      .RA    (`BROADCAST_ADDR),
      .TA    (dev_addr),
      .BSSID (bss_addr)
    );
    // get KSR index
    ksr_index = get_random_ksr_index(ndpa_frame);
    ndpa_frame.calc_leg_ht_length();

    `uvm_info(get_type_name(),
    $sformatf("NDPA:\n%s", ndpa_frame.sprint()), UVM_LOW)

    // configure MAC control 1 field for THD
    mac_ctrl_info_1.prot_frm_dur_f   = $urandom_range(500, 1000);
    mac_ctrl_info_1.write_ack_f      = 0;
    mac_ctrl_info_1.low_rate_retry_f = 0;
    mac_ctrl_info_1.lstp_prot_f      = 0;
    mac_ctrl_info_1.lstp_f           = 0;
    mac_ctrl_info_1.expect_ack_f     = bit'(NO_ACK);

    // configure MAC control 2 field for THD
    mac_ctrl_info_2.dont_generate_mh_f = 0;
    mac_ctrl_info_2.dont_encrypt_f     = 0;
    mac_ctrl_info_2.dont_touch_fc_f    = 0;
    mac_ctrl_info_2.dont_touch_dur_f   = 1;
    mac_ctrl_info_2.dont_touch_qos_f   = 0;
    mac_ctrl_info_2.dont_touch_htc_f   = 0;
    mac_ctrl_info_2.dont_touch_tsf_f   = 0;
    mac_ctrl_info_2.dont_touch_dtim_f  = 0;
    mac_ctrl_info_2.dont_touch_fcs_f   = 0;
    mac_ctrl_info_2.under_ba_setup_f   = 0;
    mac_ctrl_info_2.num_blank_delimit_f = 0;
    mac_ctrl_info_2.int_en_f            = 1;

    m_cfg.m_sram_cfg.use_cfg_pt_ptr = 0;

    // Store PPDU frame in SRAM for trasmit
    wr_frame_to_sram(ndpa_frame,
                     mac_ctrl_info_1,
                     mac_ctrl_info_2,
                     prot,
                     32'h80);

    // save THD address for future read of status information
    thd_addr_q.push_front(m_cfg.m_sram_cfg.thd_head_ptr);

    // set Tx interrupt masks for AC
    set_ac_interrupts(ac);
    set_DMA_for_tx(ac);

    //--------------------------------
    // create NDP frame for Tx
    //--------------------------------
    assert (ndp_frame.randomize() with {
      kind == NDP;
      tx_frame == 1;
    });
    // channel bandwidth needs to be same as in NDPA
    ndp_frame.preamble_header.ch_bw = ndpa_frame.preamble_header.ch_bw;

    `uvm_info(get_type_name(),
    $sformatf("NDP:\n%s", ndp_frame.sprint()), UVM_LOW)

    // configure MAC control 1 field for THD
    mac_ctrl_info_1.prot_frm_dur_f   = 16'd0;
    mac_ctrl_info_1.write_ack_f      = 0;
    mac_ctrl_info_1.low_rate_retry_f = 0;
    mac_ctrl_info_1.lstp_prot_f      = 0;
    mac_ctrl_info_1.lstp_f           = 0;
    mac_ctrl_info_1.expect_ack_f     = bit'(NORMAL_ACK);

    // configure MAC control 2 field for THD
    mac_ctrl_info_2.dont_generate_mh_f = 0;
    mac_ctrl_info_2.dont_encrypt_f     = 0;
    mac_ctrl_info_2.dont_touch_fc_f    = 0;
    mac_ctrl_info_2.dont_touch_dur_f   = 0;
    mac_ctrl_info_2.dont_touch_qos_f   = 0;
    mac_ctrl_info_2.dont_touch_htc_f   = 0;
    mac_ctrl_info_2.dont_touch_tsf_f   = 0;
    mac_ctrl_info_2.dont_touch_dtim_f  = 0;
    mac_ctrl_info_2.dont_touch_fcs_f   = 0;
    mac_ctrl_info_2.under_ba_setup_f   = 0;
    mac_ctrl_info_2.num_blank_delimit_f = 0;
    mac_ctrl_info_2.int_en_f            = 1;

    // link NDPA and NDP
    write_word_sram(
      m_cfg.m_sram_cfg.thd_head_ptr+`NEXT_ATOMIC_FRAME_OFFSET,
      m_cfg.m_sram_cfg.thd_curr_ptr
    );

    m_cfg.m_sram_cfg.use_cfg_pt_ptr = 1;

    // Store PPDU frame in SRAM for trasmit
    wr_frame_to_sram(ndp_frame,
                     mac_ctrl_info_1,
                     mac_ctrl_info_2,
                     prot,
                     m_cfg.m_sram_cfg.thd_curr_ptr);

    // save THD address for future read of status information
    thd_addr_q.push_front(m_cfg.m_sram_cfg.thd_head_ptr);

    //--------------------------------
    // create BF report frame for Rx
    //--------------------------------
    assert (bfr_frame.randomize() with {
      kind == SINGLETON;
      ppdu_format == VHT;
      tx_frame == 0;
    });

    bfr_frame.ampdu_frame[0].clear_all_arrays();
    assert (bfr_frame.ampdu_frame[0].randomize() with {
      bfr_frame.ampdu_frame[0].ppdu_format == bfr_frame.ppdu_format;
      bfr_frame.ampdu_frame[0].mpdu_frame_type.size() == 1;
      bfr_frame.ampdu_frame[0].mpdu_frame_type[0] == BEAMFORMING_REPORT;
      bfr_frame.ampdu_frame[0].aggregated == 0;
    });
    // encrypt BFR
    bfr_frame.encrypt(.ra(1));
    bfr_frame.calc_leg_ht_length();

    // set MAC addresses
    bfr_frame.ampdu_frame[0].set_MAC_addr(
      .RA    (dev_addr)
    );

    `uvm_info(get_type_name(),
    $sformatf("Beamformig report for MAC:%s", bfr_frame.sprint()), UVM_LOW)

    // Generate beamformed data frame
    assert (mumimo_frame.randomize() with {
      kind == MU_MIMO;
      tx_frame == 1;
      num_of_users == 2;
    });
    mumimo_frame.preamble_header.ch_bw = ch_bw;
    foreach (mumimo_frame.preamble_header.user_header[i]) begin
      if (ch_bw == 0 && mumimo_frame.preamble_header.user_header[i].mcs_f == 7'd9) begin
        mumimo_frame.preamble_header.user_header[i].mcs_f = $urandom_range(0,8);
      end
    end

    fork
      begin : TX_NDPA_NDP_MUMIMO
        //------------------------------------------------
        // collect NDPA frame
        //------------------------------------------------
        wait_tx_end();
        drive_noise_adc();
        insert_idle_cycles(1);

        //------------------------------------------------
        // collect NDP frame
        //------------------------------------------------
        wait_tx_end();
        drive_noise_adc();
        insert_idle_cycles(16);
        //------------------------------------------------
        // send Beamforming report
        //------------------------------------------------
        create_ppdu_response_frame(bfr_frame);
        drive_phy_adc(bfr_frame,0,1);
      end

      wait_mac_tx_trigger(ac);
    join

    // if frames if going to be received trigger will occur,
    // if frames if filtered event will occur
    fork
      wait_mac_rx_trigger();
      rx_frame_filtered();
    join_any
    // store beamforming report to memory
    store_bfmer_report_ahb(mumimo_frame, 0, STA_id[0]);
    store_bfmer_report_ahb(mumimo_frame, 1, STA_id[1]);
    smm_index[0] = mumimo_frame.preamble_header.user_header[0].smm_index_f;
    smm_index[1] = mumimo_frame.preamble_header.user_header[1].smm_index_f;
    insert_idle_cycles(20);
    // read THD status from SRAM
    while (thd_addr_q.size() != 0)
      rd_tx_status_from_sram(thd_addr_q.pop_back());

    // clear interrupt flag
    clear_ac_interrupts(ac);

    insert_idle_cycles(20);

    frame_num = $urandom_range(5,10);
    for (int loop=0; loop < frame_num; loop++) begin
    `uvm_info(get_type_name(),$sformatf("Transmit Frame number: %0d / %0d",
      loop+1,frame_num),UVM_LOW)

      mumimo_frame = new("mumimo_frame");

      assert (mumimo_frame.randomize() with {
        kind == MU_MIMO;
        tx_frame == 1;
        num_of_users == 2;
      });
      mumimo_frame.preamble_header.ch_bw = ch_bw;
      foreach (mumimo_frame.preamble_header.user_header[i]) begin
        if (mumimo_frame.preamble_header.ch_bw == 0 && mumimo_frame.preamble_header.user_header[i].mcs_f == 7'd9) begin
          mumimo_frame.preamble_header.user_header[i].mcs_f = $urandom_range(0,8);
        end
        mumimo_frame.preamble_header.user_header[i].smm_index_f = smm_index[i];
      end
      foreach(mumimo_frame.ampdu_frame[i]) begin
        ksr_index = get_random_ksr_index(mumimo_frame, i);
        // set MAC addresses
        mumimo_frame.ampdu_frame[i].set_MAC_addr(
          .TA    (dev_addr),
          .RA    (m_ksr.keyRAM[ksr_index].mac_addr_ram_f),
          .BSSID (bss_addr)
        );
      end

      // encrypt frame if protection bit is set
      mumimo_frame.encrypt(.ra(1));
      mumimo_frame.calc_leg_ht_length();

      `uvm_info(get_type_name(),
      $sformatf("Data frame:\n%s", mumimo_frame.sprint()), UVM_LOW)

      // configure MAC control 1 field for THD
      mac_ctrl_info_1.prot_frm_dur_f   = 16'd0;
      mac_ctrl_info_1.write_ack_f      = 0;
      mac_ctrl_info_1.low_rate_retry_f = 0;
      mac_ctrl_info_1.lstp_prot_f      = 0;
      mac_ctrl_info_1.lstp_f           = 0;
      mac_ctrl_info_1.expect_ack_f     = expect_ack; //NO ACK

      // configure MAC control 2 field for THD
      mac_ctrl_info_2.dont_generate_mh_f = 0;
      mac_ctrl_info_2.dont_encrypt_f     = 0;
      mac_ctrl_info_2.dont_touch_fc_f    = 0;
      mac_ctrl_info_2.dont_touch_dur_f   = 0;
      mac_ctrl_info_2.dont_touch_qos_f   = 0;
      mac_ctrl_info_2.dont_touch_htc_f   = 0;
      mac_ctrl_info_2.dont_touch_tsf_f   = 0;
      mac_ctrl_info_2.dont_touch_dtim_f  = 0;
      mac_ctrl_info_2.dont_touch_fcs_f   = 0;
      mac_ctrl_info_2.under_ba_setup_f   = 0;
      mac_ctrl_info_2.num_blank_delimit_f = 0;
      mac_ctrl_info_2.int_en_f            = 1;

      m_cfg.m_sram_cfg.use_cfg_pt_ptr = 0;

      // determine type of NAV protection frame exchange
      `uvm_info(get_type_name(),$sformatf("NAV protection %s",prot.name()),UVM_LOW)
      // expected ACK
      `uvm_info(get_type_name(),$sformatf("Expected ACK: %s",expect_ack.name()),UVM_LOW)
      // access category
      `uvm_info(get_type_name(),$sformatf("Access category %s",ac.name()),UVM_LOW)

      // Store PPDU frame in SRAM for trasmit
      wr_frame_to_sram(mumimo_frame,
                       mac_ctrl_info_1,
                       mac_ctrl_info_2,
                       prot,
                       32'h80);

      // set Tx interrupt masks for AC
      set_ac_interrupts(ac);

      set_DMA_for_tx(ac);

      fork
        begin
          //-------------------------------
          // send DATA frame
          //-------------------------------
          wait_tx_end();
          drive_noise_adc();
          //-------------------------------
          // collect BARs
          //-------------------------------
          insert_idle_cycles(1);
          wait_tx_end();
          insert_idle_cycles(1);
          wait_tx_end();
        end
        wait_mac_tx_trigger(ac);
      join

      // read THD status from SRAM for primary user
      rd_tx_status_from_sram(
        .head     (m_cfg.m_sram_cfg.thd_head_ptr),
        .skip_athd((expect_ack == NO_ACK) ? 0 : 1) // read A-THD if BAR is sent
      );
      // read THD status from SRAM for secondary user
      read_word_sram(
        m_cfg.m_sram_cfg.thd_head_ptr+`USER1_AMPDU_THD_PTR, user1_addr);

      rd_tx_status_from_sram(
        .head     (user1_addr),
        .skip_athd((expect_ack == NO_ACK) ? 0 : 1) // read A-THD if BAR is sent
      );

      insert_idle_cycles(20);

      // clear interrupt flag
      clear_ac_interrupts(ac);

      insert_idle_cycles(10);

    end // for loop

  endtask : body

endclass : wlan_tx_mumimo_frame_seq


`endif //WLAN_TX_MUMIMO_FRAME_SEQ_SV
