//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_IMMEDIATE_RESP_SCOREBOARD_SV
`define MAC_IMMEDIATE_RESP_SCOREBOARD_SV

`uvm_analysis_imp_decl(_immediate_resp)
`uvm_analysis_imp_decl(_phy_bf)

class mac_immediate_resp_scoreboard extends mac_scoreboard_base;

  `uvm_component_utils(mac_immediate_resp_scoreboard)

  uvm_analysis_imp_immediate_resp#(mac_phy_seq_item, mac_immediate_resp_scoreboard) mac_phy_export;
  uvm_analysis_imp_phy_bf#(phy_bf_seq_item, mac_immediate_resp_scoreboard) phy_bf_export;
  uvm_tlm_analysis_fifo#(sram_seq_item) sram_fifo;

  // received items queues
  mac_phy_seq_item        mac_phy_q[$], rx_item, tx_item, bfr_poll_item;
  phy_bf_seq_item         bfm_q[$], bfm_report_item;
  sram_seq_item           sram_item;

  uvm_comparer            comparer;
  // list of variables
  bit                     error_cts;
  bit                     error_ack;
  bit                     error_block_ack;
  int                     cts_duration;
  int                     ba_duration;
  int                     ack_duration;
  int                     bfm_duration;
  PPDU_frame              ndpa_frame;
  bit                     ndpa_rcvd; // flag indicating that NDPA is received and
                                     // if NDP follows BFM report is send by MAC
  int                     staid;     // station ID inside NDPA frame
  mpdu_frame_type_e       frm_type;
  CONTROL_WRAPPER_frame   ctrl_wrapp;
  TRIGGER_frame           trig_frm;
  bit [11:0]              WinStart;
  int                     tid_num;
  int                     index_queue[$];
  int                     tid_table_index;
  he_control_s            he_ctrl;
  bit                     has_qos_data; //indication that AMPDU has one or more QOS_DATA MPDUs
  int                     dummy1;
  int                     dummy2;


  function new(string name = "mac_immediate_resp_scoreboard", uvm_component parent = null);
    super.new(name, parent);

    mac_phy_export = new("mac_phy_export", this);
    phy_bf_export  = new("phy_bf_export", this);
    sram_fifo      = new("sram_fifo", this);
    comparer       = new();
    comparer.show_max = 1000; // show number of mismatches
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);
  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase

  //---------------------------------------------------------------------------
  // list of functions for checking
  //---------------------------------------------------------------------------
  extern function void      write_immediate_resp(mac_phy_seq_item i);
  extern function void      write_phy_bf(phy_bf_seq_item i);
  extern task               wait_invalid_MAC_response(int resp_dur);
  extern function void      check_tx_vector(PPDU_preamble_header rxvector,
                                            PPDU_preamble_header txvector, bit is_rts = 0);
  extern function void      check_duration(ref PPDU_frame rx, ref PPDU_frame tx);
  extern function void      check_response_delay();
  extern function void      check_block_ack_content(bit aggregated = 0);
  extern function void      update_TID_table(output int mpdu_cnt, output int seq_num);
  extern function void      calc_response_frame_durations();
  extern function void      check_bfm_report_item_frame(ref PPDU_frame frm);
  extern function void      check_bfm_report_item_frame_he(ref PPDU_frame frm);
  extern function void      check_trigger_resp(TRIGGER_frame trig_frm, PPDU_preamble_header tx_vector, int user=0);
  extern function int       get_bfm_report_duration_he();
  //---------------------------------------------------------------------------

  task run_phase(uvm_phase phase);
    super.run_phase(phase);

    forever begin
      // first receive RX frame, determine if there should be auto reply
      // and performe checking of content inside frames
      wait (mac_phy_q.size() != 0);
      rx_item = mac_phy_q.pop_back();
      error_cts = 0;
      error_ack = 0;
      error_block_ack = 0;

      // determine what is the type of frame and should there be
      // immediate response or not and what kind of respose frame type
      if (rx_item.trans == RX && filter_ppdu_frame(rx_item) == 0) begin

        case (rx_item.frame.kind)
//**********************************************************************
          SINGLETON: begin
//**********************************************************************

            fetch_register_values();
            calc_response_frame_durations();
            ndpa_rcvd = 0;
            has_qos_data = 0;

            frm_type = rx_item.frame.ampdu_frame[0].mpdu_frame_type[0];
            // control wrapper frame carries the real frame type which
            // MAC needs to analyze
            if (frm_type == CONTROL_WRAPPER) begin
              // type cast frame to control wrapper
              ctrl_wrapp = CONTROL_WRAPPER_frame'(rx_item.frame.ampdu_frame[0].mpdu_frame[0]);
              frm_type = ctrl_wrapp.ctrl_frame_type;
            end
            else if (frm_type == TRIGGER) begin
              // type cast frame to Trigger
              trig_frm = new();
              trig_frm.copy(rx_item.frame.ampdu_frame[0].mpdu_frame[0]);
            end

            // determine type of frame and then should there be response
            case (frm_type)

              RTS: begin

                // is MAC device address receiver
                if (   is_MAC_receiver(rx_item.frame)
                    && !mac_ctrl_1.disable_cts_resp_f
                    && !m_cfg.channel_busy) begin

                  // CTS should be received from MAC, else timeout error will
                  // occur after SIFS + CTS duration
                  fork : WAIT_FOR_CTS
                    begin
                      wait (mac_phy_q.size() != 0);
                      tx_item = mac_phy_q.pop_back();
                    end
                    begin
                      #((get_sifs(mac_ctrl_1.abgn_mode_f) + cts_duration) * 1us);
                      error_cts = 1;
                    end
                  join_any

                  disable fork; //WAIT_FOR_CTS

                  if (error_cts) begin
                    `uvm_error(get_type_name(), $sformatf("MAC didn't send CTS response!!!"))
                  end
                  else begin

                    if (tx_item.trans != TX)
                      `uvm_error(get_type_name(),$sformatf("Response frame is not Tx!!!"))
                    // check frame type is CTS
                    if (tx_item.frame.ampdu_frame[0].mpdu_frame_type[0] != CTS)
                      `uvm_error(get_type_name(),
                      $sformatf("Response frame is not CTS (%s)!!!",
                      tx_item.frame.ampdu_frame[0].mpdu_frame_type[0].name()))
                    // check addresses
                    if (get_RA(tx_item.frame) != get_TA(rx_item.frame))
                      `uvm_error(get_type_name(),
                      $sformatf("Response MAC address don't match! RA (%0h) TA(%0h)",
                      get_RA(tx_item.frame),get_TA(rx_item.frame)))

                    check_tx_vector(rx_item.frame.preamble_header,
                                    tx_item.frame.preamble_header, .is_rts(1));
                    check_duration(rx_item.frame, tx_item.frame);
                    check_response_delay();
                  end

                end
                else begin // when MAC is not a receiver
                  wait_invalid_MAC_response(cts_duration);
                end
              end// RTS

              BLOCK_ACK_REQUEST: begin

                // is MAC device address receiver
                if (   is_MAC_receiver(rx_item.frame)
                    && get_bar_variant(rx_item.frame) inside {BASIC_BAR, COMPRESSED_BAR, EXT_COMPRESSED_BAR}
                    && !mac_ctrl_1.disable_bar_resp_f) begin

                  // BA should be received from MAC, else timeout error will
                  // occur after SIFS + BA duration
                  fork : WAIT_FOR_BA
                    begin
                      wait (mac_phy_q.size() != 0);
                      tx_item = mac_phy_q.pop_back();
                    end
                    begin
                      #((get_sifs(mac_ctrl_1.abgn_mode_f) + ba_duration) * 1us);
                      error_block_ack = 1;
                    end
                  join_any

                  disable fork; //WAIT_FOR_BA

                  if (error_block_ack) begin
                    `uvm_error(get_type_name(), $sformatf("MAC didn't send BLOCK_ACK/ACK response!!!"))
                  end
                  else begin

                    if (tx_item.trans != TX)
                      `uvm_error(get_type_name(),$sformatf("Response frame is not Tx!!!"))

                    // check frame type is ACK or BLOCK_ACK
                    if (!tx_item.frame.ampdu_frame[0].mpdu_frame_type[0] inside {ACK, BLOCK_ACK})
                      `uvm_error(get_type_name(),$sformatf("Response frame is not ACK nor BLOCK_ACK!!!"))
                    else if (   tx_item.frame.ampdu_frame[0].mpdu_frame_type[0] == BLOCK_ACK
                             && get_bar_variant(rx_item.frame) == BASIC_BAR)
                      `uvm_error(get_type_name(),$sformatf("Response frame is BLOCK_ACK but it should be ACK!!!"))
                    else if (   tx_item.frame.ampdu_frame[0].mpdu_frame_type[0] == ACK
                             && get_bar_variant(rx_item.frame) == COMPRESSED_BAR)
                      `uvm_error(get_type_name(),$sformatf("Response frame is ACK but it should be BLOCK_ACK!!!"))

                    // check addresses, RA and TA must match in case of BLOCK_ACK
                    if (   get_bar_variant(rx_item.frame) == COMPRESSED_BAR
                        && (   get_RA(tx_item.frame) != get_TA(rx_item.frame)
                            && get_TA(tx_item.frame) != get_RA(rx_item.frame)))
                      `uvm_error(get_type_name(), $sformatf("Response MAC addresses don't match for BLOCK_ACK frame!"))
                    else if (   get_bar_variant(rx_item.frame) == BASIC_BAR
                             && get_RA(tx_item.frame) != get_TA(rx_item.frame))
                      `uvm_error(get_type_name(), $sformatf("Response MAC address don't match for ACK frame!"))

                    if (get_bar_variant(rx_item.frame) == COMPRESSED_BAR) begin
                      check_block_ack_content();
                    end
                    check_tx_vector(rx_item.frame.preamble_header,
                                    tx_item.frame.preamble_header);
                    check_duration(rx_item.frame, tx_item.frame);
                    check_response_delay();
                  end

                end
                else begin // when MAC is not a receiver
                  wait_invalid_MAC_response(ba_duration);
                end
              end// BLOCK_ACK_REQUEST

              QOS_DATA, QOS_DATA_CF_ACK, QOS_DATA_CF_POLL,
              QOS_DATA_CF_ACK_CF_POLL,
              QOS_NULL, QOS_CF_POLL, QOS_CF_ACK_CF_POLL: begin
                // get HE control
                he_ctrl = get_ht_control(rx_item.frame);

                // is MAC device address receiver and ACK policy is 2'b00
                // (normal ACK or implicit Block Ack)
                if ((  is_MAC_receiver(rx_item.frame)
                    && get_block_ack_policy(rx_item.frame) == 2'b00
                    && !mac_ctrl_1.disable_ack_resp_f)
                    ||
                    (  is_MAC_receiver(rx_item.frame)
                    && (he_ctrl.vht_f && he_ctrl.he_f && he_ctrl.ctrl_id_f == 0) //TRS control subfield
                    && get_block_ack_policy(rx_item.frame) == 2'b10
                    && frm_type inside {QOS_DATA, QOS_NULL}
                    && rx_item.frame.ppdu_format inside {HE_SU, HE_EXT_SU, HE_MU}
                    && !mac_ctrl_1.disable_ack_resp_f)) begin

                  // ACK should be received from MAC, else timeout error will
                  // occur after SIFS + ACK duration
                  fork : WAIT_FOR_ACK_QOS
                    begin
                      wait (mac_phy_q.size() != 0);
                      tx_item = mac_phy_q.pop_back();
                    end
                    begin
                      #((get_sifs(mac_ctrl_1.abgn_mode_f) + ack_duration) * 1us);
                      error_ack = 1;
                    end
                  join_any

                  disable fork; //WAIT_FOR_ACK_QOS

                  if (error_ack) begin
                    `uvm_error(get_type_name(), $sformatf("MAC didn't send ACK response 1 !!!"))
                  end
                  else begin

                    if (tx_item.trans != TX)
                      `uvm_error(get_type_name(),$sformatf("Response frame is not Tx!!!"))
                    // check frame type is ACK
                    if (tx_item.frame.ampdu_frame[0].mpdu_frame_type[0] != ACK)
                      `uvm_error(get_type_name(),$sformatf("Response frame is not ACK (%s)!!!",tx_item.frame.ampdu_frame[0].mpdu_frame_type[0].name()))
                    // check addresses
                    if (get_RA(tx_item.frame) != get_TA(rx_item.frame))
                      `uvm_error(get_type_name(), $sformatf("Response MAC address don't match!"))

                    check_tx_vector(rx_item.frame.preamble_header,
                                    tx_item.frame.preamble_header);
                    check_duration(rx_item.frame, tx_item.frame);
                    check_response_delay();
                  end

                end
                else begin // when MAC is not a receiver
                  `uvm_info(get_type_name(), "Wait for invalid MAC response", UVM_LOW)
                  wait_invalid_MAC_response(ack_duration);
                end
              end// QOS subtypes

              // all frames that don't elicit response
              ACK, CTS, CF_END, CF_END_CF_ACK,
              BLOCK_ACK, ACTION_NOACK, BEACON,
              CF_ACK: begin
                wait_invalid_MAC_response(ack_duration);
              end // no response frames

              // if NDPA is receinved set flag because NDP will follow
              VHT_NDP_ANNOUNCEMENT, HE_NDP_ANNOUNCEMENT: begin
                ndpa_rcvd = 1;
                // save NDPA frame for checking
                ndpa_frame = new("ndpa_frame");
                ndpa_frame.copy(rx_item.frame);
                // in BF calibration tests MAC will responde to NDPA+NDP frames
                if (m_cfg.bf_calib_test == 0)
                  wait_invalid_MAC_response(16);
              end

              // If the received frame is Trigger,
              // respond in HE_TB
              TRIGGER: begin
                if (   is_MAC_receiver(rx_item.frame)
                    && !m_cfg.m_mac_phy_cfg.m_phy_cfg.set_cca_to_busy) begin
                  `uvm_info(get_type_name(), $sformatf("The received frame is a Trigger frame"), UVM_LOW)
                  // Predict the Trigger register values
                  predict_trigger_register(trig_frm);

                  // The MAC should be responding using
                  // HE_TB to a received Trigger frame
                  wait (mac_phy_q.size() != 0);
                  tx_item = mac_phy_q.pop_back();

                  if (tx_item.trans != TX)
                    `uvm_error(get_type_name(),$sformatf("Response frame is not Tx!!!"))

                  // check the consistency from the Trigger frame, with
                  // response frame to it
                  check_trigger_resp(trig_frm, tx_item.frame.preamble_header);

                  // these types are handled by SW
                  if (trig_frm.trigger_type inside {{BASIC_TRIGGER,BUFFER_STATUS_REPORT_POLL}}) begin
                    // check content of transmitted MPDU
                    // wait frame to be written in SRAM
                    do begin
                      sram_fifo.get(sram_item);
                    end while (sram_item.cmd != WRITE_TX_FRAME);
                    // overwritte duration field
                    sram_item.frame.ampdu_frame[0].set_MAC_duration(tx_item.frame.ampdu_frame[0].get_MAC_duration());
                    // transmitted frame is encrypted when protection bit is set and
                    // MAC control 2 info don't encrypt is not set, so we need to
                    // decrypt it before comparing it with the frame stored in SRAM
                    tx_item.frame.decrypt(1);
                    `uvm_info(get_type_name(),$sformatf("Trigger based frame: %s",tx_item.frame.sprint()),UVM_DEBUG)

                    // compare frame transmitted by MAC from AC-TB prepared as
                    // immediate response to TRIGGER frame
                    if (sram_item.frame.ampdu_frame[0].compare(tx_item.frame.ampdu_frame[0], comparer) == 0)
                      `uvm_error(get_type_name(), $sformatf("AC_TB frame and MAC-PHY frame don't match!"))
                    else
                      `uvm_info(get_type_name(), $sformatf("AC_TB frame sent by MAC is correct!"), UVM_LOW)
                  end//trigger type

                end
                else if (m_cfg.m_mac_phy_cfg.m_phy_cfg.set_cca_to_busy) begin
                  // Predict the Trigger register values
                  predict_trigger_register(trig_frm);
                  // There should be no response, since the channel is set to busy state
                  wait_invalid_MAC_response(0);
                end
              end//TRIGGER

              // if PS_POLL don't check NAV, response should
              // be generated in form of ACK
              PS_POLL: begin
                // is MAC device address receiver
                if (   is_MAC_receiver(rx_item.frame)
                    && !mac_ctrl_1.disable_ack_resp_f) begin

                  // ACK should be received from MAC, else timeout error will
                  // occur after SIFS + ACK duration
                  fork
                    begin
                      wait (mac_phy_q.size() != 0);
                      tx_item = mac_phy_q.pop_back();
                    end
                    begin
                      #((get_sifs(mac_ctrl_1.abgn_mode_f) + ack_duration) * 1us);
                      error_ack = 1;
                    end
                  join_any

                  disable fork;

                  if (error_ack) begin
                    `uvm_error(get_type_name(), $sformatf("MAC didn't send response to PS_POLL!!!"))
                  end
                  else begin

                    if (tx_item.trans != TX)
                      `uvm_error(get_type_name(),$sformatf("Response frame is not Tx!!!"))
                    // check frame type is ACK
                    if (tx_item.frame.ampdu_frame[0].mpdu_frame_type[0] != ACK)
                      `uvm_error(get_type_name(),$sformatf("Response frame is not ACK (%s)!!!",tx_item.frame.ampdu_frame[0].mpdu_frame_type[0].name()))
                    // check addresses
                    if (get_RA(tx_item.frame) != get_TA(rx_item.frame))
                      `uvm_error(get_type_name(), $sformatf("Response MAC address don't match!"))

                    check_tx_vector(rx_item.frame.preamble_header,
                                    tx_item.frame.preamble_header);
                    check_duration(rx_item.frame, tx_item.frame);
                    check_response_delay();
                  end

                end
                else begin // when MAC is not a receiver
                  wait_invalid_MAC_response(ack_duration);
                end
              end
              // frames that need ACK as response
              default: begin

                // is MAC device address receiver
                if (   is_MAC_receiver(rx_item.frame)
                    && !mac_ctrl_1.disable_ack_resp_f) begin

                  // ACK should be received from MAC, else timeout error will
                  // occur after SIFS + ACK duration
                  fork : WAIT_FOR_ACK
                    begin
                      wait (mac_phy_q.size() != 0);
                      tx_item = mac_phy_q.pop_back();
                    end
                    begin
                      #((get_sifs(mac_ctrl_1.abgn_mode_f) + ack_duration) * 1us + 10us);
                      error_ack = 1;
                    end
                  join_any

                  disable fork; //WAIT_FOR_ACK

                  if (error_ack) begin
                    `uvm_error(get_type_name(), $sformatf("MAC didn't send ACK response 2!!!"))
                  end
                  else begin

                    if (tx_item.trans != TX)
                      `uvm_error(get_type_name(),$sformatf("Response frame is not Tx!!!"))
                    // check frame type is ACK
                    if (tx_item.frame.ampdu_frame[0].mpdu_frame_type[0] != ACK)
                      `uvm_error(get_type_name(),$sformatf("Response frame is not ACK (%s)!!!",tx_item.frame.ampdu_frame[0].mpdu_frame_type[0].name()))
                    // check addresses
                    if (get_RA(tx_item.frame) != get_TA(rx_item.frame))
                      `uvm_error(get_type_name(), $sformatf("Response MAC address don't match!"))

                    check_tx_vector(rx_item.frame.preamble_header,
                                    tx_item.frame.preamble_header);
                    check_duration(rx_item.frame, tx_item.frame);
                    check_response_delay();
                  end

                end
                else begin // when MAC is not a receiver
                  wait_invalid_MAC_response(ack_duration);
                end
              end // data, manegement frames with ACK response

            endcase
          end// SINGLETON

//**********************************************************************
          AGGREGATED: begin
//**********************************************************************
            // look if there are QOS_DATA MPDUs in AMPDU
            has_qos_data = 0;
            foreach (rx_item.frame.ampdu_frame[0].mpdu_frame_type[i]) begin
              if (rx_item.frame.ampdu_frame[0].mpdu_frame_type[i] == QOS_DATA)
                has_qos_data = 1;
            end
            fetch_register_values();
            calc_response_frame_durations();
            ndpa_rcvd = 0;
            he_ctrl = get_ht_control(rx_item.frame);

            // is MAC device address receiver and ACK policy is 2'b00
            // (normal ACK or implicit Block Ack)
            if ((    is_MAC_receiver(rx_item.frame)
                  && get_block_ack_policy(rx_item.frame) == 2'b00
                  && !mac_ctrl_1.disable_ack_resp_f
                )
                || // HTP ack
                (    is_MAC_receiver(rx_item.frame)
                  && get_block_ack_policy(rx_item.frame) == 2'b10
                  && has_qos_data
                  && (he_ctrl.vht_f && he_ctrl.he_f && he_ctrl.ctrl_id_f == 0) //TRS control subfield
                  && !mac_ctrl_1.disable_ack_resp_f
                )
               ) begin

              // ACK should be received from MAC, else timeout error will
              // occur after SIFS + BLOCK_ACK duration
              fork : WAIT_FOR_BLOCK_ACK
                begin
                  wait (mac_phy_q.size() != 0);
                  tx_item = mac_phy_q.pop_back();
                end
                begin
                  #((get_sifs(mac_ctrl_1.abgn_mode_f) + ba_duration) * 1us);
                  error_ack = 1;
                end
              join_any

              disable fork; //WAIT_FOR_BLOCK_ACK

              if (error_ack) begin
                `uvm_error(get_type_name(), $sformatf("MAC didn't send ACK response 3 !!!"))
              end
              else begin

                if (tx_item.trans != TX)
                  `uvm_error(get_type_name(),$sformatf("Response frame is not Tx!!!"))
                // check frame type is ACK
                if (tx_item.frame.ampdu_frame[0].mpdu_frame_type[0] != BLOCK_ACK)
                  `uvm_error(get_type_name(),$sformatf("Response frame is not BLOCK_ACK (%s)!!!",tx_item.frame.ampdu_frame[0].mpdu_frame_type[0].name()))
                // check addresses
                if (   get_RA(tx_item.frame) != get_TA(rx_item.frame)
                    && get_TA(tx_item.frame) != get_RA(rx_item.frame))
                  `uvm_error(get_type_name(), $sformatf("Response MAC address don't match!"))

                check_tx_vector(rx_item.frame.preamble_header,
                                tx_item.frame.preamble_header);
                check_duration(rx_item.frame, tx_item.frame);
                check_response_delay();
                check_block_ack_content(1);
              end

            end
            else begin // when MAC is not a receiver
              update_TID_table(dummy1, dummy2);
              wait_invalid_MAC_response(ba_duration);
            end

          end// AGGREGATED

//**********************************************************************
          NDP: begin
//**********************************************************************
            fetch_register_values();
            bfm_duration = (rx_item.frame.ppdu_format == VHT) ? 500 : 2500;
            has_qos_data = 0;

            if (ndpa_rcvd && is_aid_matching(ndpa_frame,staid)) begin

              // during MU calibration NDPA has multyple STA informations
              // if STA ID is not first beamformer send Beamfroming report
              // poll frame and after that beamforming report is sent
              if (is_mu_calibration(ndpa_frame))  begin
                if (rx_item.frame.ppdu_format != VHT || staid != 0) begin
                  wait (mac_phy_q.size() != 0); //BFR POLL frame
                  bfr_poll_item = mac_phy_q.pop_back();
                  if (filter_ppdu_frame(bfr_poll_item))
                    continue;
                  // calculate duration of BFR HE-TB frame
                  if (rx_item.frame.ppdu_format > VHT)
                    bfm_duration = get_bfm_report_duration_he();
                end
              end

              // BFM report should be received from MAC, else timeout error will
              // occur after SIFS + BFM report duration
              fork : WAIT_FOR_BFM_REPORT
                begin
                  wait (mac_phy_q.size() != 0);
                  tx_item = mac_phy_q.pop_back();
                end
                begin
                  #((get_sifs(mac_ctrl_1.abgn_mode_f) + bfm_duration) * 1us);
                  error_ack = 1;
                end
              join_any

              disable fork; //WAIT_FOR_BFM_REPORT

              if (error_ack) begin
                `uvm_error(get_type_name(), $sformatf("MAC didn't send BFM REPORT response!!!"))
              end
              else begin

                if (tx_item.trans != TX)
                  `uvm_error(get_type_name(),$sformatf("Response frame is not Tx!!!"))
                // check frame type is ACK
                if (tx_item.frame.ampdu_frame[0].mpdu_frame_type[0] != ACTION_NOACK)
                  `uvm_error(get_type_name(),$sformatf("Response frame is not ACTION_NOACK (%s)!!!",
                  tx_item.frame.ampdu_frame[0].mpdu_frame_type[0].name()))

                // check beamforming report frame content
                if (tx_item.frame.ppdu_format == HE_SU) begin
                  check_bfm_report_item_frame_he(tx_item.frame);
                end
                else if (is_mu_calibration(ndpa_frame) && tx_item.frame.ppdu_format == HE_TB) begin
                  check_bfm_report_item_frame_he(tx_item.frame);
                  // type cast frame to Trigger
                  trig_frm = new();
                  trig_frm.copy(bfr_poll_item.frame.ampdu_frame[0].mpdu_frame[0]);
                  // check the consistency from the Trigger frame, with
                  // response frame to it
                  check_trigger_resp(trig_frm, tx_item.frame.preamble_header, staid);
                end
                else begin
                  check_bfm_report_item_frame(tx_item.frame);
                end
              end


            end//ndpa_rcvd
            else begin // when MAC is not a receiver
              wait_invalid_MAC_response(bfm_duration);
            end

          end// NDP
        endcase
      end// if RX
      else begin
        `uvm_info(get_type_name(),$sformatf("Frame filtered, no resp should be triggered"), UVM_LOW);
        wait_invalid_MAC_response(0);
      end
    end// forever

  endtask : run_phase


  function void report_phase(uvm_phase phase);
    super.report_phase(phase);

    if (mac_phy_q.size() > 0) begin
      `uvm_error(get_type_name(),$sformatf("MAC-PHY queue is not empty, items left:"))
      while (mac_phy_q.size() > 0) begin
        rx_item = mac_phy_q.pop_back();
        `uvm_info(get_type_name(), $sformatf("MAC-PHY item: %s", rx_item.sprint()), UVM_LOW)
      end
    end
  endfunction : report_phase

endclass : mac_immediate_resp_scoreboard

  //---------------------------------------------------------------------------
  // implementation of checking functions
  //---------------------------------------------------------------------------

  function void mac_immediate_resp_scoreboard::write_immediate_resp(mac_phy_seq_item i);
    mac_phy_seq_item it;

    it = new();
    it.copy(i);
    mac_phy_q.push_front(it);
    `uvm_info(get_type_name(), $sformatf("MAC-PHY item received %s", it.sprint()), UVM_HIGH)
  endfunction : write_immediate_resp

  function void mac_immediate_resp_scoreboard::write_phy_bf(phy_bf_seq_item i);
    phy_bf_seq_item it;

    it = new i;
    bfm_q.push_front(it);
    `uvm_info(get_type_name(), $sformatf("PHY BF item received %s", it.sprint()), UVM_HIGH)
  endfunction : write_phy_bf

  //---------------------------------------------------------------------------
  // function for checking the consistency from the Trigger frame, with the
  // that are a response to the Trigger frame
  //---------------------------------------------------------------------------

  function void mac_immediate_resp_scoreboard::check_trigger_resp(TRIGGER_frame trig_frm, PPDU_preamble_header tx_vector, int user=0);

    case(trig_frm.trigger_type)
      BASIC_TRIGGER,
      MU_BAR,
      BANDWIDTH_QUERY_REPORT_POLL,
      BUFFER_STATUS_REPORT_POLL,
      BMF_REPORT_POLL:
      begin
        `uvm_info(get_type_name(),$sformatf("%s frame is received",trig_frm.trigger_type.name()),UVM_LOW)
        // The response frame is not HE_TB
        if (tx_vector.format_mod != HE_TB)
          `uvm_error(get_type_name(),$sformatf("Response frame is not in HE_TB!!!"))

        // Check the ch_bw
        if (tx_vector.ch_bw != trig_frm.common_info.bw_f)
          `uvm_error(get_type_name(),$sformatf("The Bandwidth is not correct, for the response frame: expected (%d) received(%d)",
          trig_frm.common_info.bw_f, tx_vector.ch_bw))

        // Check gi_type and he_ltf_type
        if (trig_frm.common_info.gi_and_ltf_type_f == 0) begin
          // For gi_and_ltf_type = 0 => gi_type == 1.6us and he_ltf_type = 1xLTF
          if (    tx_vector.he_ltf_type != 2'b00
              &&  tx_vector.gi_type != 2'b01)
              `uvm_error(get_type_name(),$sformatf("The gi_type (%d) and he_ltf_type (%d) is not correct, in the response frame!!!",
              tx_vector.gi_type,tx_vector.he_ltf_type))
        end
        else if (trig_frm.common_info.gi_and_ltf_type_f == 1) begin
          // For gi_and_ltf_type = 1 => gi_type == 1.6us and he_ltf_type = 2xLTF
          if (    tx_vector.he_ltf_type != 2'b01
              &&  tx_vector.gi_type != 2'b01)
              `uvm_error(get_type_name(),$sformatf("The gi_type (%d) and he_ltf_type (%d) is not correct, in the response frame!!!",
              tx_vector.gi_type,tx_vector.he_ltf_type))
        end
        else if (trig_frm.common_info.gi_and_ltf_type_f == 2) begin
          // For gi_and_ltf_type = 2 => gi_type == 3.2us and he_ltf_type = 4xLTF
          if (    tx_vector.he_ltf_type != 2'b10
              &&  tx_vector.gi_type != 2'b10)
              `uvm_error(get_type_name(),$sformatf("The gi_type (%d) and he_ltf_type (%d) is not correct, in the response frame!!!",
              tx_vector.gi_type,tx_vector.he_ltf_type))
        end
        else begin
          // Not allowed
          `uvm_error(get_type_name(),$sformatf("This value of gi_type and he_ltf_type is reserved, for the response frame!!!"))
        end

        // Check the STBC field
        if (tx_vector.stbc != trig_frm.common_info.stbc_f)
          `uvm_error(get_type_name(),$sformatf("The STBC is not correct, for the response frame, expected (%d) received (%d)",
          trig_frm.common_info.stbc_f, tx_vector.stbc))

        // Check the Pre-FEC Padding Factor
        if (tx_vector.user_header_he[0].pkt_ext_f[1:0] != trig_frm.common_info.pre_fec_padding_f)
          `uvm_error(get_type_name(),$sformatf("The Pre-FEC Padding Factor is not correct, for the response frame, expected (%d) received (%d)",
          trig_frm.common_info.pre_fec_padding_f, tx_vector.user_header_he[0].pkt_ext_f[1:0]))

        // Check the PE Disambiguity
        if (tx_vector.user_header_he[0].pkt_ext_f[2] != trig_frm.common_info.pe_disambiguity_f)
          `uvm_error(get_type_name(),$sformatf("The Pre-FEC Padding Factor is not correct, for the response frame, expected (%d) received (%d)",
          trig_frm.common_info.pe_disambiguity_f, tx_vector.user_header_he[0].pkt_ext_f[2]))

        // Check the Doppler field
        if (tx_vector.doppler != trig_frm.common_info.doppler_f)
          `uvm_error(get_type_name(),$sformatf("The Doppler is not correct, for the response frame, expected (%d) received (%d)",
          trig_frm.common_info.doppler_f, tx_vector.doppler))

        if (trig_frm.common_info.doppler_f == 0) begin
          case (trig_frm.common_info.num_of_he_ltf_symbols_f)
            0: if (tx_vector.num_he_ltf != 0) `uvm_error(get_type_name, $sformatf("Num HE-LTF mismatch expected 0)"))
            1: if (tx_vector.num_he_ltf != 1) `uvm_error(get_type_name, $sformatf("Num HE-LTF mismatch expected 1)"))
            2: if (tx_vector.num_he_ltf != 3) `uvm_error(get_type_name, $sformatf("Num HE-LTF mismatch expected 3)"))
            3: if (tx_vector.num_he_ltf != 5) `uvm_error(get_type_name, $sformatf("Num HE-LTF mismatch expected 5)"))
            4: if (tx_vector.num_he_ltf != 7) `uvm_error(get_type_name, $sformatf("Num HE-LTF mismatch expected 7)"))
            default: `uvm_error(get_type_name(),
                     $sformatf("Unsupported Number of HE-LTF in common info of Trigger frame (%0d)",
                     trig_frm.common_info.num_of_he_ltf_symbols_f))
          endcase
        end
        else begin
          case (trig_frm.common_info.num_of_he_ltf_symbols_f)
            0: if (tx_vector.num_he_ltf != 0) `uvm_error(get_type_name, $sformatf("Num HE-LTF mismatch expected 0)"))
            1: if (tx_vector.num_he_ltf != 1) `uvm_error(get_type_name, $sformatf("Num HE-LTF mismatch expected 1)"))
            2: if (tx_vector.num_he_ltf != 3) `uvm_error(get_type_name, $sformatf("Num HE-LTF mismatch expected 3)"))
            4: if (tx_vector.num_he_ltf != 0) `uvm_error(get_type_name, $sformatf("Num HE-LTF mismatch expected 0)"))
            5: if (tx_vector.num_he_ltf != 1) `uvm_error(get_type_name, $sformatf("Num HE-LTF mismatch expected 1)"))
            6: if (tx_vector.num_he_ltf != 3) `uvm_error(get_type_name, $sformatf("Num HE-LTF mismatch expected 3)"))
            default: `uvm_error(get_type_name(),
                     $sformatf("Unsupported Number of HE-LTF in common info of Trigger frame (%0d)",
                     trig_frm.common_info.num_of_he_ltf_symbols_f))
          endcase
          // check midamble_periodicity
          case (trig_frm.common_info.num_of_he_ltf_symbols_f)
            0,1,2: if (tx_vector.midamble_periodicity != 0) `uvm_error(get_type_name(),$sformatf("Midamble periodicity wrong, should be 10"))
            4,5,6: if (tx_vector.midamble_periodicity != 1) `uvm_error(get_type_name(),$sformatf("Midamble periodicity wrong, should be 20"))
            default: `uvm_error(get_type_name(),
                     $sformatf("Unsupported Number of HE-LTF in common info of Trigger frame (%0d)",
                     trig_frm.common_info.num_of_he_ltf_symbols_f))
          endcase
        end

        // Check Trigger method field
        if (tx_vector.trigger_method != 0)
          `uvm_error(get_type_name(),$sformatf("The trigger method is not correct, for the response frame, expected 0 received (%d)",
          tx_vector.trigger_method))

        // Check BSS color
        if (rx_item.frame.preamble_header.format_mod inside {HE_SU, HE_EXT_SU}) begin
          if (trig_frm.trigger_type == BEAMFORMING_REPORT_POLL && ndpa_rcvd)
            if (tx_vector.bss_color != bfr_poll_item.frame.preamble_header.bss_color)
              `uvm_error(get_type_name(),$sformatf("The BSS color doesn't match to the one in the RX_VECTOR expected (%h) received (%h)",
              bfr_poll_item.frame.preamble_header.bss_color, tx_vector.bss_color))
          else if (tx_vector.bss_color != rx_item.frame.preamble_header.bss_color)
            `uvm_error(get_type_name(),$sformatf("The BSS color doesn't match to the one in the RX_VECTOR expected (%h) received (%h)",
            rx_item.frame.preamble_header.bss_color, tx_vector.bss_color))
        end

        // Check of the leg_length
        if (tx_vector.leg_length != trig_frm.common_info.length_f)
          `uvm_error(get_type_name(),$sformatf("The legacy length is not as in the Trigger frame, expected (%h) received (%h)",
          trig_frm.common_info.length_f, tx_vector.leg_length))

        // Check of the NSS
        if (tx_vector.user_header_he[0].nss_f != trig_frm.user_info[user].ss_allocation_f[5:3])
          `uvm_error(get_type_name(),$sformatf("The N_STS is not set as in ss_allocation field of the Trigger frame, expected (%h) received (%h)",
          trig_frm.user_info[user].ss_allocation_f[5:3], tx_vector.user_header_he[0].nss_f))

        // HE_LTF_MODE
        if (tx_vector.he_ltf_mode != trig_frm.common_info.mumimo_ltf_mode_f)
          `uvm_error(get_type_name(),$sformatf("The HE LTF MODE is not set as in ss_allocation field of the Trigger frame, expected (%h) received (%h)",
          trig_frm.common_info.mumimo_ltf_mode_f, tx_vector.he_ltf_mode))

        // NUM HE LTF
        if (tx_vector.he_ltf_mode != trig_frm.common_info.mumimo_ltf_mode_f)
          `uvm_error(get_type_name(),$sformatf("The NUM HE LTF is not set as in mumimo_ltf_mode_f field of the Trigger frame, expected (%h) received (%h)",
          trig_frm.common_info.mumimo_ltf_mode_f, tx_vector.he_ltf_mode))

        // LDPC EXTRA SYMBOL
        if (tx_vector.ldpc_extra_symbol != trig_frm.common_info.ldpc_extra_symbol_segment_f)
          `uvm_error(get_type_name(),$sformatf("The LDPC EXTRA SYMBOL is not set as in ldpc_extra_symbol_segment_f field of the Trigger frame, expected (%h) received (%h)",
          trig_frm.common_info.ldpc_extra_symbol_segment_f, tx_vector.ldpc_extra_symbol))

        // SPATIAL REUSE
        if (  { tx_vector.spatial_reuse[3],tx_vector.spatial_reuse[2],
                tx_vector.spatial_reuse[1],tx_vector.spatial_reuse[0]}
            !=  trig_frm.common_info.spatial_reuse_f)
          `uvm_error(get_type_name(),$sformatf("The Spatial reuse is not set as in the Trigger frame, expected (%h) received (%h)",
          trig_frm.common_info.spatial_reuse_f,{ tx_vector.spatial_reuse[3],tx_vector.spatial_reuse[2],
          tx_vector.spatial_reuse[1],tx_vector.spatial_reuse[0]}))

        // HE SIGA RESERVED
        if (tx_vector.he_siga_reserved != trig_frm.common_info.he_sig_a_reserved_f)
          `uvm_error(get_type_name(),$sformatf("The HE SIGA RESERVED is not set as in ldpc_extra_symbol_segment_f field of the Trigger frame, expected (%h) received (%h)",
          trig_frm.common_info.he_sig_a_reserved_f, tx_vector.he_siga_reserved))

        // MCS
        if (tx_vector.user_header_he[0].mcs_f != trig_frm.user_info[user].mcs_f)
          `uvm_error(get_type_name(),$sformatf("The MCS is not set as in the Trigger frame, expected (%h) received (%h)",
          trig_frm.user_info[user].mcs_f, tx_vector.user_header_he[0].mcs_f))

        // DCM
        if (tx_vector.dcm != trig_frm.user_info[user].dcm_f)
          `uvm_error(get_type_name(),$sformatf("DCM is not set as in the Trigger frame, expected (%h) received (%h)",
          trig_frm.user_info[user].dcm_f, tx_vector.dcm))

        // STARTING STS NUM
        if (tx_vector.starting_sts_num != trig_frm.user_info[user].ss_allocation_f[2:0])
          `uvm_error(get_type_name(),$sformatf("STARTING STS NUM is not set as in the Trigger frame, expected (%h) received (%h)",
          trig_frm.user_info[user].ss_allocation_f[2:0], tx_vector.starting_sts_num))

        // FEC CODING
        if (tx_vector.user_header_he[0].fec_coding_f != trig_frm.user_info[user].coding_type_f)
          `uvm_error(get_type_name(),$sformatf("The FEC CODING is not set as in the Trigger frame, expected (%h) received (%h)",
          trig_frm.user_info[user].coding_type_f, tx_vector.user_header_he[0].fec_coding_f))

        // RU_ALLOCATION
        if (tx_vector.ru_allocation != trig_frm.user_info[user].ru_allocation_f)
          `uvm_error(get_type_name(),$sformatf("The RU ALLOCATION is not set as in the Trigger frame, expected (%h) received (%h)",
          trig_frm.user_info[user].ru_allocation_f, tx_vector.ru_allocation))

      end
      MU_RTS: begin
        `uvm_info(get_type_name(),$sformatf("MU-RTS is received"), UVM_LOW)

        // Based on RU, determine the channel bandwidth, 9.3.1.22.5 MU-RTS variant
        if (trig_frm.user_info[user].ru_allocation_f[7:1] inside {61,62,63,64}) begin
          // Check the channel bandwidth
          if (tx_vector.ch_bw != 0)
            `uvm_error(get_type_name(),$sformatf("The channel bandwidth is not set to 20MHz in the response frame"))
          // Check the format mod of the response
          if (tx_vector.format_mod != NON_HT)
            `uvm_error(get_type_name(),$sformatf("The CTS response is not in NON_HT"))
        end
        else if (trig_frm.user_info[user].ru_allocation_f[7:1] inside {65,66}) begin
          // Check the channel bandwidth
          if (tx_vector.ch_bw != 1)
            `uvm_error(get_type_name(),$sformatf("The channel bandwidth is not set to 40MHz in the response frame"))
          // Check the format mod of the response
          if (tx_vector.format_mod != NON_HT_DUP_OFDM)
            `uvm_error(get_type_name(),$sformatf("The CTS response is not in NON_HT_DUP_OFDM format"))
        end

        // The legacy rate should be set to 6 Mbps
        if (tx_vector.leg_rate != 4'b1011)
          `uvm_error(get_type_name(),$sformatf("The legacy rate of the CTS response frame is not 6 Mbps"))

        // The power management field should be set to 0
        if (tx_item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.frame_ctrl.power_management_f != 0)
          `uvm_error(get_type_name(),$sformatf("The power management field should be set to 0 !!!"))

        // trigger responding should be set
        if (tx_vector.trigger_responding != 1'b1)
          `uvm_error(get_type_name(), $sformatf("Trigger responding should be set in TXVECTOR!"))

      end
    endcase

  endfunction : check_trigger_resp

  // if there is no response wait for SIFS + BA/ACK duration for
  // the cases when MAC responses unintentionally because that
  // is error behavioural and trigger event to inform sequence
  //---------------------------------------------------------------------------
  task mac_immediate_resp_scoreboard::wait_invalid_MAC_response(int resp_dur);

    #((get_sifs(mac_ctrl_1.abgn_mode_f) + resp_dur) * 1us);
    `uvm_info(get_type_name(),
      $sformatf("MAC core should not respond to frame %s, MAC is receiver (%0d), disable ACK/CTS/BAR response (%0d/%0d/%0d)",
      (rx_item.frame.ampdu_frame[0].aggregated) ? "aggregated" : rx_item.frame.ampdu_frame[0].mpdu_frame_type[0].name(),
      is_MAC_receiver(rx_item.frame),
      mac_ctrl_1.disable_ack_resp_f,
      mac_ctrl_1.disable_cts_resp_f,
      mac_ctrl_1.disable_bar_resp_f), UVM_LOW)

    no_resp_trigger.trigger();
  endtask : wait_invalid_MAC_response

  //---------------------------------------------------------------------------
  // check Tx vector fields of response frame
  //---------------------------------------------------------------------------
  function void mac_immediate_resp_scoreboard::check_tx_vector(PPDU_preamble_header rxvector,
                                                               PPDU_preamble_header txvector,
                                                               bit is_rts = 0);

    int          time_on_air;
    int          rx_rate;
    int          highest_mcs_set;
    bit [1:0]    ack_policy;

    if (rxvector.ch_bw < txvector.ch_bw)
      `uvm_error(get_type_name(), $sformatf("Channel bandwidth of Tx vector is larger then Rx vector"))
    if (rxvector.antenna_set != txvector.antenna_set)
      `uvm_error(get_type_name(), $sformatf("Antenna set not equal"))

    // MCS is only present in HT format mode
    if (txvector.format_mod inside {HT_GF,HT_MF,VHT}) begin
      // Find the highest indexed MCS of the CandidateMCSSet for which the modulation value of
      // each stream is less than or equal to the modulation value of each stream of the MCS of the
      // received frame and for which the coding rate is less than or equal to the coding rate of the MCS
      // from the received frame.
      highest_mcs_set = bss_basic_mcs_set(rxvector.user_header[0].mcs_f,
                                          rxvector.format_mod,
                                          htmcs);
      while (get_coding_rate(highest_mcs_set) > get_coding_rate(txvector.user_header[0].mcs_f)) begin
        highest_mcs_set = bss_basic_mcs_set(highest_mcs_set-1,
                                            rxvector.format_mod,
                                            htmcs);
      end

      if (txvector.user_header[0].mcs_f != highest_mcs_set)
        `uvm_error(get_type_name(), $sformatf("Tx vector MCS not correct (%0d) expct (%0d)",
        txvector.user_header[0].mcs_f, highest_mcs_set))
    end

    // FEC coding is only present in HT format mode
    if (txvector.format_mod inside {HT_GF,HT_MF,VHT}) begin
      if (rxvector.user_header[0].fec_coding_f != txvector.user_header[0].fec_coding_f)
        `uvm_error(get_type_name(), $sformatf("FEC coding not set correctly"))
    end

    // when RTS is sent in HT format, response must be sent in HT_MF (HT_GF is not
    // possible)
    if (rxvector.format_mod inside {HT_GF,HT_MF} && is_rts && txvector.format_mod != HT_MF)
      `uvm_error(get_type_name(), $sformatf("Response for RTS should be in HT, txvector = %s",txvector.format_mod.name()))

    else if (rxvector.format_mod == NON_HT && txvector.format_mod != NON_HT)
      `uvm_error(get_type_name(), $sformatf("Response to NON_HT should be in NON_HT, txvector = %s",txvector.format_mod.name()))

    else if (rxvector.format_mod == NON_HT_DUP_OFDM && txvector.format_mod != NON_HT_DUP_OFDM)
      `uvm_error(get_type_name(), $sformatf("Response to NON_HT_DUP_OFDM should be in NON_HT_DUP_OFDM, txvector = %s",txvector.format_mod.name()))

    else if (   rxvector.format_mod inside {HT_GF,HT_MF,VHT}
             && (txvector.format_mod != NON_HT && txvector.ch_bw == 0) && get_trq_ndp_announcement(rx_item.frame) != 1)

      `uvm_error(get_type_name(), $sformatf("Response to HT should be in NON_HT, txvector = %s",txvector.format_mod.name()))

    else if (   rxvector.format_mod inside {HT_GF,HT_MF,VHT}
             && (txvector.format_mod != NON_HT_DUP_OFDM && txvector.ch_bw == 2'b01) && get_trq_ndp_announcement(rx_item.frame) != 1)

      `uvm_error(get_type_name(), $sformatf("Response to HT should be in NON_HT_DUP_OFDM, txvector = %s",txvector.format_mod.name()))

    // check legacy length when in HT MF
    time_on_air = $ceil(((ComputeTimeOnAirAC ( .length_i     (txvector.user_header[0].ht_length_f),
                                               .preType_i    (txvector.format_mod),
                                               .mcsIndex_i   (txvector.user_header[0].mcs_f),
                                               .shortGI_i    (txvector.gi_type[0]),
                                               .chBW_i       (txvector.ch_bw),
                                               .stbc_i       (txvector.stbc),
                                               .extNSS_i     (txvector.num_extn_ss),
                                               .woPreamble_i (0),
                                               .band5G_i     (get_band5G(mac_ctrl_1.abgn_mode_f)),
                                               .multiUser_i  (0),
                                               .nSTSTotal_i  (txvector.user_header[0].num_sts_f),
                                               .debug_i      (0)
                                                 /*  signal extension  */
                                           ) - ((get_band5G(mac_ctrl_1.abgn_mode_f))?0:6))-20) / 4.0) * 3 - 3;

    if (txvector.format_mod == HT_MF && txvector.leg_length != time_on_air)
      `uvm_error(get_type_name(), $sformatf("Tx vector (HT_MF) legacy length wrong (%0d), expct (%0d)",
      txvector.leg_length,time_on_air))

    // check legacy rate
    rx_rate = (rxvector.format_mod inside {HT_GF,HT_MF,VHT})
              ? get_legacy_rate_from_mcs(rxvector.format_mod, rxvector.user_header[0].mcs_f)
              : (rxvector.format_mod inside {HE_SU,HE_TB,HE_EXT_SU,HE_MU})
              ? get_legacy_rate_from_mcs(rxvector.format_mod, rxvector.user_header_he[0].mcs_f)
              : rxvector.leg_rate;

    if (txvector.format_mod inside {NON_HT,NON_HT_DUP_OFDM}) begin
      if (bss_basic_rate_set(rx_rate, basic_rates) != txvector.leg_rate)
        `uvm_error(get_type_name(), $sformatf("Legacy rate wrong (%0b), expct (%0b)",
        txvector.leg_rate,bss_basic_rate_set(rx_rate, basic_rates)))
    end
    else begin
      if (txvector.leg_rate != 4'b1011)
        `uvm_error(get_type_name(), $sformatf("Legacy rate in HT should be set to 6Mbps (%0b)",
        txvector.leg_rate))
    end

    // check aggregation
    if (txvector.format_mod < VHT && txvector.aggregated)
      `uvm_error(get_type_name(), $sformatf("Response frame can't be aggregated if not VHT or HE"))

    // check short GI
    if (rxvector.gi_type[0] != txvector.gi_type[0] && txvector.format_mod == VHT)
      `uvm_error(get_type_name(),
        $sformatf("Short GI should be same in Rx and Tx vector, (%0b)/(%0b)",
        rxvector.gi_type[0], txvector.gi_type[0]))

    // check if MAC respose to HE-SU
    if (rxvector.format_mod inside {HE_SU, HE_EXT_SU, HE_MU}) begin
      // get HE control
      he_ctrl = get_ht_control(rx_item.frame);
      // get ACK policy
      ack_policy = get_block_ack_policy(rx_item.frame);

      // check did MAC responde in HE-TB when control ID = 0 and ACK policy HTP ACK
      if (he_ctrl.vht_f && he_ctrl.he_f && he_ctrl.ctrl_id_f == 0 && txvector.format_mod != HE_TB)
        `uvm_error(get_type_name(),$sformatf("Responde frame not in HE-TB, respond (%s)",txvector.format_mod.name()))

      if (txvector.format_mod == HE_TB && txvector.txop_duration != calc_txop_duration(rx_item.frame, txvector))
        `uvm_error(get_type_name(),
        $sformatf("TXOP DURATION in responde frame not correct, recv (%0d) - exp (%0d)",
        txvector.txop_duration,calc_txop_duration(rx_item.frame, txvector)))

      // check preamble header parameters
      if (he_ctrl.vht_f && he_ctrl.he_f && he_ctrl.ctrl_id_f == 0 && ack_policy == 2'b10 && txvector.format_mod == HE_TB) begin
        // check RU allocation
        if (he_ctrl.ctrl_info_f.ru_allocation_f != txvector.ru_allocation)
          `uvm_error(get_type_name(),
          $sformatf("RU allocation wrong, HE control (%0h) - Tx vector (%0h)",
          he_ctrl.ctrl_info_f.ru_allocation_f, txvector.ru_allocation))

        // check MCS index
        if (he_ctrl.ctrl_info_f.ul_mcs_f != txvector.user_header_he[0].mcs_f)
          `uvm_error(get_type_name(),
          $sformatf("MCS wrong, HE control (%0h) - Tx vector (%0h)",
          he_ctrl.ctrl_info_f.ul_mcs_f, txvector.user_header_he[0].mcs_f))
      end
    end

    `uvm_info(get_type_name(), $sformatf("Tx vector check OK!"), UVM_LOW)
  endfunction : check_tx_vector

  //---------------------------------------------------------------------------
  // check duration in response frame
  //---------------------------------------------------------------------------
  function void mac_immediate_resp_scoreboard::check_duration(ref PPDU_frame rx, ref PPDU_frame tx);
    int          rx_dur;
    he_control_s he_ctrl;
    HESIGB_s     HESIGB;

    rx_dur = (rx.ampdu_frame[0].mpdu_frame_type[0] == PS_POLL)
             ? 0 // duration in PS_POLL frames is set to 0
             : rx.ampdu_frame[0].get_MAC_duration();
    rx_dur -= get_sifs(mac_ctrl_1.abgn_mode_f); // substract SIFS
    // response scheduling in TRS
    if (rx.preamble_header.format_mod inside {HE_SU, HE_EXT_SU, HE_MU} && tx.preamble_header.format_mod == HE_TB) begin
      // get HE control from received frame
      he_ctrl = get_ht_control(rx);

      rx_dur -= ComputeTimeOnAirAX (
                                    .length_i          (tx.preamble_header.user_header_he[0].he_length_f),
                                    .preType_i         (tx.preamble_header.format_mod),
                                    .mcsIndex_i        (tx.preamble_header.user_header_he[0].mcs_f),
                                    .giType_i          (tx.preamble_header.gi_type),
                                    .ruType_i          (get_ru_type(he_ctrl.ctrl_info_f.ru_allocation_f)),
                                    .heLtfType_i       (tx.preamble_header.he_ltf_type),
                                    .numHeLtf_i        (tx.preamble_header.num_he_ltf),
                                    .dcm_i             (tx.preamble_header.dcm),
                                    .packetExtension_i (tx.preamble_header.user_header_he[0].pkt_ext_f),
                                    .heTbLength_i      (he_ctrl.ctrl_info_f.ul_data_symbols_f),
                                    .triggerMethod_i   (tx.preamble_header.trigger_method),
                                    .doppler_i         (tx.preamble_header.doppler),
                                    .mma_i             (tx.preamble_header.midamble_periodicity),
                                    .stbc_i            (tx.preamble_header.stbc),
                                    .woPreamble_i      (0),
                                    .heSigB_i          (HESIGB),
                                    .debug_i           (0)
                                    );
    end
    else begin
      rx_dur -= get_ppdu_duration(tx, get_band5G(mac_ctrl_1.abgn_mode_f)); // substract duration of response frame
    end
    rx_dur = (rx_dur < 0) ? 0 : rx_dur; // duration can't be negative number

    // check if duration of reponse frame is correct
    if (rx_dur != tx.ampdu_frame[0].get_MAC_duration())
      `uvm_error(get_type_name(),
        $sformatf("Duration of response frame is not correct, (%0d) expct (%0d)",
        tx.ampdu_frame[0].get_MAC_duration(),rx_dur))
    else
      `uvm_info(get_type_name(), $sformatf("Response frame duration OK!"), UVM_LOW)
  endfunction : check_duration

  //---------------------------------------------------------------------------
  // check response delay from Rx end for timing and Tx request
  //---------------------------------------------------------------------------
  function void mac_immediate_resp_scoreboard::check_response_delay();
    real t;
    real tx_delay;
    real sub;

                /*     get clock period in [us]     */
    tx_delay = (1/real'(timings_1.mac_core_clk_freq_f)) * timings_1.tx_chain_delay_in_mac_clk_f;
    // measure time from rxEndFroTiming to Tx request rise
    t = (tx_item.tx_req_rise - rx_item.end_of_frame) / 1000; // [us]
    sub = (get_sifs(mac_ctrl_1.abgn_mode_f)-tx_delay);

    // result must be in tolerance of +-1 [us]
    if (!(sub <= t+1 && sub >= t-1))
      `uvm_error(get_type_name(),
        $sformatf("Response delay not correct: (%0.0f) expct (%0.0f)",t,sub))
    else
      `uvm_info(get_type_name(), $sformatf("Response delay OK!"), UVM_LOW)
  endfunction : check_response_delay

  //---------------------------------------------------------------------------
  // check Block Ack content, sequence number and bitmap
  //---------------------------------------------------------------------------
  function void mac_immediate_resp_scoreboard::check_block_ack_content(bit aggregated = 0);
    BLOCK_ACK_REQUEST_frame bar;
    BLOCK_ACK_frame         ba;
    CONTROL_WRAPPER_frame   ctrl_wrap;
    bit [11:0]              seq_num;
    int                     ack_cnt;
    int                     mpdu_cnt;
    bit [63:0]              bitmap;
    int                     valid_mpdu;

    ba = BLOCK_ACK_frame'(tx_item.frame.ampdu_frame[0].mpdu_frame[0]);
    valid_mpdu = find_first_valid_mpdu(rx_item.frame);

    if (!aggregated) begin
      frm_type = rx_item.frame.ampdu_frame[0].mpdu_frame_type[valid_mpdu];
      if (frm_type == CONTROL_WRAPPER) begin
        ctrl_wrap = CONTROL_WRAPPER_frame'(rx_item.frame.ampdu_frame[0].mpdu_frame[valid_mpdu]);
        frm_type = ctrl_wrap.ctrl_frame_type;
        bar = BLOCK_ACK_REQUEST_frame'(ctrl_wrap.carried_frame);
        // sequence numbers must match in reponse and request frames
        if (bar.starting_seq_num != ba.starting_seq_num && bar.bar_variant == COMPRESSED_BAR)
          `uvm_error(get_type_name(),
            $sformatf("Starting sequence numbers don't match, BAR(%0h)-BA(%0h)",bar.starting_seq_num,ba.starting_seq_num))
      end
      else begin
        bar = BLOCK_ACK_REQUEST_frame'(rx_item.frame.ampdu_frame[0].mpdu_frame[valid_mpdu]);
        // sequence numbers must match in reponse and request frames
        if (bar.starting_seq_num != ba.starting_seq_num && bar.bar_variant == COMPRESSED_BAR)
          `uvm_error(get_type_name(),
            $sformatf("Starting sequence numbers don't match, BAR(%0h)-BA(%0h)",bar.starting_seq_num,ba.starting_seq_num))
      end
    end
    // when response BA is after A-MPDU check bitmap matching
    else begin

      seq_num = rx_item.frame.ampdu_frame[0].get_MAC_seq_num(valid_mpdu);

      // put bitmap bytes to single bit array
      foreach (ba.bitmap[i]) begin
        bitmap[i*8 +: 8] = ba.bitmap[i];
      end
      // count how many ones where in bitmap, that represents number of MPDUs
      // acknowledged
      ack_cnt = $countones(bitmap);
      mpdu_cnt = number_of_valid_mpdu_frames(rx_item.frame, .user(0));

      if (seq_num != -1) begin
         update_TID_table(mpdu_cnt, seq_num);
      end // if seq_num!=-1
      if (ba.starting_seq_num != seq_num && ba.ba_variant == COMP_BA)
        `uvm_error(get_type_name(),
        $sformatf("Sequence number in BLOCK_ACK not matching one in aggregated frame, exp - %0h/recvd - %0h",seq_num,ba.starting_seq_num))
      else
        `uvm_info(get_type_name(),
        $sformatf("Sequence number match in A-MPDU and BLOCK ACK"), UVM_LOW)

      if (mpdu_cnt != ack_cnt)
        `uvm_error(get_type_name(),
        $sformatf("MPDU count in A-MPDU (%0d), acknowledged (%0d) are not equal",mpdu_cnt,ack_cnt))
      else
        `uvm_info(get_type_name(),
        $sformatf("MPDU count in A-MPDU (%0d), acknowledged (%0d)",mpdu_cnt,ack_cnt), UVM_LOW)
    end// aggregated
  endfunction : check_block_ack_content

  //---------------------------------------------------------------------------
  // Function that will update the TID table, after every received AMPDU
  //---------------------------------------------------------------------------
  function void mac_immediate_resp_scoreboard::update_TID_table(output int mpdu_cnt, output int seq_num);

    int                     ack_cnt;
    bit [11:0]              WinStart;
    bit [11:0]              WinEnd;
    int                     bitmap_position;
    int                     SSN;
    int                     prev_SSN;
    bitmap_param_s          new_table;
    bit                     tuple_found = 0;
    int                     temp_index;

    tid_num = rx_item.frame.ampdu_frame[0].get_MAC_tid_num();

    // Search the queue for a TID TA pair
    for (int i = 0; i < m_cfg.TID_table.size(); i++) begin
      if (m_cfg.TID_table[i].TID == tid_num &&
          m_cfg.TID_table[i].ksr_index == m_cfg.ksr_index) begin
        tuple_found = 1;
        tid_table_index = i;
      end
      if ((i == m_cfg.TID_table.size()-1) && tuple_found==1)begin
        // If there is a match change the index_queue
        temp_index = tid_table_index;
        for (int i=0; i< index_queue.size(); i++)begin
          if (index_queue[i]== temp_index) begin
            index_queue.delete(i);
          end
        end
        index_queue.push_back(temp_index);
      end
    end//for

    if (tuple_found == 0) begin

      if (m_cfg.TID_table.size()>= 4) begin
        // delete the oldest bitmap and replace it
        // with a new TID-TA pair
        tid_table_index = index_queue.pop_front();
        m_cfg.TID_table.delete(tid_table_index);
        index_queue.push_back(tid_table_index);
        new_table.TID = tid_num;
        new_table.ksr_index = m_cfg.ksr_index;
        new_table.bitmap = 0;
        m_cfg.TID_table.insert(tid_table_index, new_table);
      end
      else begin
        new_table.ksr_index = m_cfg.ksr_index;
        new_table.TID = tid_num;
        new_table.bitmap = 0;
        m_cfg.TID_table.push_back(new_table);
        tid_table_index = m_cfg.TID_table.size() - 1;
        index_queue.push_back(tid_table_index);
      end

      for (int i = 0; i < rx_item.frame.ampdu_frame[0].mpdu_frame_type.size(); i++) begin
        // If there is no record for the current TID
        if (rx_item.frame.ampdu_frame[0].get_MAC_seq_num(i) != -1) begin
          WinEnd = rx_item.frame.ampdu_frame[0].get_MAC_seq_num(i);
          WinStart = WinEnd - `WINDOWSIZE + 1;
          // Update bitmap with ones in SN place
          m_cfg.TID_table[tid_table_index].bitmap[`WINDOWSIZE - 1] = 1;
          if (i!=rx_item.frame.ampdu_frame[0].mpdu_frame_type.size()-1) begin
            m_cfg.TID_table[tid_table_index].bitmap = m_cfg.TID_table[tid_table_index].bitmap >> 1;
          end
          m_cfg.TID_table[tid_table_index].ksr_index = m_cfg.ksr_index;
          m_cfg.TID_table[tid_table_index].SSN = WinStart;
        end
      end//for
    end
    else begin
       // If there is a record for the current TID and KSR
      WinEnd = m_cfg.TID_table[tid_table_index].SSN + `WINDOWSIZE - 1;
      WinStart = m_cfg.TID_table[tid_table_index].SSN;
      prev_SSN = m_cfg.TID_table[tid_table_index].SSN;

      for (int i = 0; i < rx_item.frame.ampdu_frame[0].mpdu_frame_type.size(); i++) begin
        if (rx_item.frame.ampdu_frame[0].get_MAC_seq_num(i) != -1) begin
          // If a KSR tid_table_index is found in the TID_table
          seq_num = rx_item.frame.ampdu_frame[0].get_MAC_seq_num(i);
          // Full state operation
          if ((WinStart <= seq_num) && (seq_num <= WinEnd)) begin
            // Condition case 1
            prev_SSN = m_cfg.TID_table[tid_table_index].SSN;
            m_cfg.TID_table[tid_table_index].bitmap = m_cfg.TID_table[tid_table_index].bitmap >> seq_num - prev_SSN;
            m_cfg.TID_table[tid_table_index].bitmap[`WINDOWSIZE-1] = 1;
            if (i == rx_item.frame.ampdu_frame[0].mpdu_frame_type.size()-1) begin
              m_cfg.TID_table[tid_table_index].SSN = seq_num - `WINDOWSIZE+1;
            end
          end
          else if ((WinEnd < seq_num) && (seq_num < WinStart + 2048)) begin
            // Condition case 2
            WinStart = seq_num - `WINDOWSIZE +1;
            WinEnd = seq_num;
            m_cfg.TID_table[tid_table_index].bitmap = m_cfg.TID_table[tid_table_index].bitmap >> WinStart - prev_SSN;
            m_cfg.TID_table[tid_table_index].bitmap[`WINDOWSIZE-1] = 1;
            m_cfg.TID_table[tid_table_index].SSN = WinStart;
            prev_SSN = WinStart;
          end
          else begin
            // Condition case 3
            prev_SSN = m_cfg.TID_table[tid_table_index].SSN;
            if (i == rx_item.frame.ampdu_frame[0].mpdu_frame_type.size()-1) begin
              m_cfg.TID_table[tid_table_index].SSN = prev_SSN;
            end
          end
        end
      end//for
    end

    mpdu_cnt = $countones(m_cfg.TID_table[tid_table_index].bitmap);
    // The SSN should be set to the value for the TID_table
    seq_num = m_cfg.TID_table[tid_table_index].SSN;
    tid_table_index = tid_table_index + 1;

    if (tid_table_index == 4) begin
      tid_table_index = 0;
    end

  endfunction : update_TID_table

  //---------------------------------------------------------------------------
  // check content and TxVector of beamforming report frame (ACTION_NOACK)
  //---------------------------------------------------------------------------
  function void mac_immediate_resp_scoreboard::check_bfm_report_item_frame(ref PPDU_frame frm);
    bit [7:0]        category   = 8'd21;
    bit [7:0]        vht_action = 8'd0; // compressed beamforming report
    vht_mimo_ctrl_s  vht_mimo_ctrl;
    int              data_cnt;
    bit [7:0]        data;

    if (frm.ampdu_frame[0].mpdu_frame[0].frame_body[0] != category)
      `uvm_error(get_type_name(),$sformatf("Category not matching 8'd21 (%0d)",
      frm.ampdu_frame[0].mpdu_frame[0].frame_body[0]))

    if (frm.ampdu_frame[0].mpdu_frame[0].frame_body[1] != vht_action)
      `uvm_error(get_type_name(),$sformatf("VHT action not matching 8'd0 (%0d)",
      frm.ampdu_frame[0].mpdu_frame[0].frame_body[1]))

    // store frame body bytes to structure
    vht_mimo_ctrl = {frm.ampdu_frame[0].mpdu_frame[0].frame_body[4],
                     frm.ampdu_frame[0].mpdu_frame[0].frame_body[3],
                     frm.ampdu_frame[0].mpdu_frame[0].frame_body[2]};

    if (vht_mimo_ctrl.sounding_dialog_token_num_f != ndpa_frame.ampdu_frame[0].mpdu_frame[0].frame_body[0][7:2]) begin
      `uvm_info(get_type_name(),$sformatf("VHT MIMO CTRL: %p",vht_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Sounding dialog token mismatch"))
    end
    if (vht_mimo_ctrl.first_feedback_sgmt_f != 1'b1) begin
      `uvm_info(get_type_name(),$sformatf("VHT MIMO CTRL: %p",vht_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("First feedback segment not set to 1"))
    end
    if (vht_mimo_ctrl.rem_feedback_sgmt_f != 0) begin
      `uvm_info(get_type_name(),$sformatf("VHT MIMO CTRL: %p",vht_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Remaining feedback segment not set to 0"))
    end
    if (vht_mimo_ctrl.feedback_type_f != bfmee_ctrl.mu_support_f) begin
      `uvm_info(get_type_name(),$sformatf("VHT MIMO CTRL: %p",vht_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Feedback type wrong, ACTION(%0h) - BFMEECTRLREG(%0h)",
      vht_mimo_ctrl.feedback_type_f, bfmee_ctrl.mu_support_f))
    end
    if (vht_mimo_ctrl.codebook_info_f != bfmee_ctrl.codebook_f) begin
      `uvm_info(get_type_name(),$sformatf("VHT MIMO CTRL: %p",vht_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Codebook info wrong, ACTION(%0h) - BFMEECTRLREG(%0h)",
      vht_mimo_ctrl.codebook_info_f, bfmee_ctrl.codebook_f))
    end
    if (vht_mimo_ctrl.grouping_f != bfmee_ctrl.grouping_f) begin
      `uvm_info(get_type_name(),$sformatf("VHT MIMO CTRL: %p",vht_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Grouping info wrong, ACTION(%0h) - BFMEECTRLREG(%0h)",
      vht_mimo_ctrl.grouping_f, bfmee_ctrl.grouping_f))
    end
    // Nr should be based on the NSS in NDP frame
    if (vht_mimo_ctrl.nr_index_f != rx_item.frame.preamble_header.user_header[0].num_sts_f) begin
      `uvm_info(get_type_name(),$sformatf("VHT MIMO CTRL: %p",vht_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Nr index wrong, ACTION(%0h) - Nsts in NDP(%0h)",
      vht_mimo_ctrl.nr_index_f, rx_item.frame.preamble_header.user_header[0].num_sts_f))
    end
    if (vht_mimo_ctrl.nc_index_f != bfmee_ctrl.nc_f) begin
      `uvm_info(get_type_name(),$sformatf("VHT MIMO CTRL: %p",vht_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Nc index wrong, ACTION(%0h) - BFMEECTRLREG(%0h)",
      vht_mimo_ctrl.nc_index_f, bfmee_ctrl.nc_f))
    end

    // check other register fields, MCS, format mod and short GI
    if (   frm.preamble_header.format_mod == VHT
        && frm.preamble_header.user_header[0].mcs_f != bfmee_ctrl.mcs_f[3:0]) begin
      `uvm_error(get_type_name(),$sformatf("MCS index wrong in VHT, ACTION(%0h) - BFMEECTRLREG(%0h)",
      frm.preamble_header.user_header[0].mcs_f, bfmee_ctrl.mcs_f[3:0]))
    end
    else if (   frm.preamble_header.format_mod inside {HT_MF,HT_GF}
             && frm.preamble_header.user_header[0].mcs_f != bfmee_ctrl.mcs_f) begin
      `uvm_error(get_type_name(),$sformatf("MCS index wrong in HT, ACTION(%0h) - BFMEECTRLREG(%0h)",
      frm.preamble_header.user_header[0].mcs_f, bfmee_ctrl.mcs_f))
    end
    else if (   frm.preamble_header.format_mod == NON_HT
             && frm.preamble_header.leg_rate != from_mcs_idx_to_legacy_rate(bfmee_ctrl.mcs_f)) begin
      `uvm_error(get_type_name(),$sformatf("MCS index wrong in NON HT, ACTION(%0h) - BFMEECTRLREG(%0h)",
      frm.preamble_header.leg_rate, from_mcs_idx_to_legacy_rate(bfmee_ctrl.mcs_f)))
    end

    if (frm.preamble_header.format_mod != bfmee_ctrl.format_mod_f)
      `uvm_error(get_type_name(),$sformatf("Format mod wrong, ACTION(%0h) - BFMEECTRLREG(%0h)",
      frm.preamble_header.format_mod, bfmee_ctrl.format_mod_f))

    if (frm.preamble_header.gi_type[0] != bfmee_ctrl.short_gi_f)
      `uvm_error(get_type_name(),$sformatf("Short GI type wrong, ACTION(%0h) - BFMEECTRLREG(%0h)",
      frm.preamble_header.gi_type[0], bfmee_ctrl.short_gi_f))

    // check beamforming report
    data_cnt = 5;
    if (bfm_q.size() == 0)
      `uvm_error(get_type_name(),$sformatf("Beamforming report not collected on BFM interface!!!"))
    else
      bfm_report_item = bfm_q.pop_back();

    while (bfm_report_item.bfm_data.size() != 0) begin
      data = bfm_report_item.bfm_data.pop_back();
      if (frm.ampdu_frame[0].mpdu_frame[0].frame_body[data_cnt] != data)
        `uvm_error(get_type_name(),$sformatf("BFM report data mismatch, MPDU[%0d]=(%0h) - BFR IF (%0h)",
        data_cnt, frm.ampdu_frame[0].mpdu_frame[0].frame_body[data_cnt], data))
      data_cnt++;
    end

    `uvm_info(get_type_name(), $sformatf("Cheking of BFM report and ACTION frame OK!"), UVM_LOW)
  endfunction : check_bfm_report_item_frame

  //---------------------------------------------------------------------------
  // check content and TxVector of beamforming report frame (ACTION_NOACK) for
  // HE frames
  //---------------------------------------------------------------------------
  function void mac_immediate_resp_scoreboard::check_bfm_report_item_frame_he(ref PPDU_frame frm);
    bit [7:0]        category  = 8'd30;
    bit [7:0]        he_action = 8'd0; // compressed beamforming report
    he_mimo_ctrl_s   he_mimo_ctrl;
    sta_info_he_s    sta_info;
    int              data_cnt;
    bit [7:0]        data;

    if (frm.ampdu_frame[0].mpdu_frame[0].frame_body[0] != category)
      `uvm_error(get_type_name(),$sformatf("Category not matching 8'd21 (%0d)",
      frm.ampdu_frame[0].mpdu_frame[0].frame_body[0]))

    if (frm.ampdu_frame[0].mpdu_frame[0].frame_body[1] != he_action)
      `uvm_error(get_type_name(),$sformatf("HE action not matching 8'd0 (%0d)",
      frm.ampdu_frame[0].mpdu_frame[0].frame_body[1]))

    // store frame body bytes to structure
    he_mimo_ctrl = {frm.ampdu_frame[0].mpdu_frame[0].frame_body[6],
                    frm.ampdu_frame[0].mpdu_frame[0].frame_body[5],
                    frm.ampdu_frame[0].mpdu_frame[0].frame_body[4],
                    frm.ampdu_frame[0].mpdu_frame[0].frame_body[3],
                    frm.ampdu_frame[0].mpdu_frame[0].frame_body[2]};

    if (he_mimo_ctrl.sounding_dialog_token_num_f != ndpa_frame.ampdu_frame[0].mpdu_frame[0].frame_body[0][7:2]) begin
      `uvm_info(get_type_name(),$sformatf("HE MIMO CTRL: %p",he_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Sounding dialog token mismatch"))
    end
    if (he_mimo_ctrl.first_feedback_sgmt_f != 1'b1) begin
      `uvm_info(get_type_name(),$sformatf("HE MIMO CTRL: %p",he_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("First feedback segment not set to 1"))
    end
    if (he_mimo_ctrl.rem_feedback_sgmt_f != 0) begin
      `uvm_info(get_type_name(),$sformatf("HE MIMO CTRL: %p",he_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Remaining feedback segment not set to 0"))
    end
    if (he_mimo_ctrl.feedback_type_f != bfmee_ctrl.mu_support_f) begin
      `uvm_info(get_type_name(),$sformatf("HE MIMO CTRL: %p",he_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Feedback type wrong, ACTION(%0h) - BFMEECTRLREG(%0h)",
      he_mimo_ctrl.feedback_type_f, bfmee_ctrl.mu_support_f))
    end
    if (he_mimo_ctrl.codebook_info_f != bfmee_ctrl.codebook_f) begin
      `uvm_info(get_type_name(),$sformatf("HE MIMO CTRL: %p",he_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Codebook info wrong, ACTION(%0h) - BFMEECTRLREG(%0h)",
      he_mimo_ctrl.codebook_info_f, bfmee_ctrl.codebook_f))
    end
    if (he_mimo_ctrl.grouping_f != bfmee_ctrl.grouping_f) begin
      `uvm_info(get_type_name(),$sformatf("HE MIMO CTRL: %p",he_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Grouping info wrong, ACTION(%0h) - BFMEECTRLREG(%0h)",
      he_mimo_ctrl.grouping_f, bfmee_ctrl.grouping_f))
    end
    // Nr should be based on the NSS in NDP frame
    if (he_mimo_ctrl.nr_index_f != rx_item.frame.preamble_header.user_header_he[0].nss_f) begin
      `uvm_info(get_type_name(),$sformatf("HE MIMO CTRL: %p",he_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Nr index wrong, ACTION(%0h) - Nsts in NDP(%0h)",
      he_mimo_ctrl.nr_index_f, rx_item.frame.preamble_header.user_header_he[0].nss_f))
    end
    if (he_mimo_ctrl.nc_index_f != bfmee_ctrl.nc_f) begin
      `uvm_info(get_type_name(),$sformatf("HE MIMO CTRL: %p",he_mimo_ctrl),UVM_LOW)
      `uvm_error(get_type_name(),$sformatf("Nc index wrong, ACTION(%0h) - BFMEECTRLREG(%0h)",
      he_mimo_ctrl.nc_index_f, bfmee_ctrl.nc_f))
    end
    // check RU start and end index received in NDPA and set in BFR
    sta_info = {ndpa_frame.ampdu_frame[0].mpdu_frame[0].frame_body[4*staid+4],
                ndpa_frame.ampdu_frame[0].mpdu_frame[0].frame_body[4*staid+3],
                ndpa_frame.ampdu_frame[0].mpdu_frame[0].frame_body[4*staid+2],
                ndpa_frame.ampdu_frame[0].mpdu_frame[0].frame_body[4*staid+1]};

    if (he_mimo_ctrl.ru_start_index_f != sta_info.ru_start_index_f)
      `uvm_error(get_type_name(),$sformatf("RU start index wrong! HE MIMO CTRL(%0d)-NDPA (%0d)",
      he_mimo_ctrl.ru_start_index_f, sta_info.ru_start_index_f))

    if (he_mimo_ctrl.ru_end_index_f != sta_info.ru_end_index_f)
      `uvm_error(get_type_name(),$sformatf("RU end index wrong! HE MIMO CTRL(%0d)-NDPA (%0d)",
      he_mimo_ctrl.ru_end_index_f, sta_info.ru_end_index_f))

    // check other register fields, MCS, format mod and short GI
    if (   frm.preamble_header.format_mod == HE_SU
        && frm.preamble_header.user_header_he[0].mcs_f != bfmee_ctrl.mcs_f[3:0]) begin
      `uvm_error(get_type_name(),$sformatf("MCS index wrong in HE, ACTION(%0h) - BFMEECTRLREG(%0h)",
      frm.preamble_header.user_header_he[0].mcs_f, bfmee_ctrl.mcs_f[3:0]))
    end
    else if (   frm.preamble_header.format_mod inside {HT_MF,HT_GF}
             && frm.preamble_header.user_header_he[0].mcs_f != bfmee_ctrl.mcs_f) begin
      `uvm_error(get_type_name(),$sformatf("MCS index wrong in HT, ACTION(%0h) - BFMEECTRLREG(%0h)",
      frm.preamble_header.user_header_he[0].mcs_f, bfmee_ctrl.mcs_f))
    end
    else if (   frm.preamble_header.format_mod == NON_HT
             && frm.preamble_header.leg_rate != from_mcs_idx_to_legacy_rate(bfmee_ctrl.mcs_f)) begin
      `uvm_error(get_type_name(),$sformatf("MCS index wrong in NON HT, ACTION(%0h) - BFMEECTRLREG(%0h)",
      frm.preamble_header.leg_rate, from_mcs_idx_to_legacy_rate(bfmee_ctrl.mcs_f)))
    end

    if (he_mimo_ctrl.feedback_type_f == 1'b1 && frm.preamble_header.format_mod != HE_TB)
      `uvm_error(get_type_name(),$sformatf("Format mod wrong, ACTION(%0h) - Response should be HE-TB",
      frm.preamble_header.format_mod))
    else if (he_mimo_ctrl.feedback_type_f == 1'b0 && frm.preamble_header.format_mod != bfmee_ctrl.format_mod_f)
      `uvm_error(get_type_name(),$sformatf("Format mod wrong, ACTION(%0h) - BFMEECTRLREG(%0h)",
      frm.preamble_header.format_mod, bfmee_ctrl.format_mod_f))

    if (he_mimo_ctrl.feedback_type_f == 1'b0 && frm.preamble_header.gi_type[0] != bfmee_ctrl.short_gi_f)
      `uvm_error(get_type_name(),$sformatf("Short GI mod wrong, ACTION(%0h) - BFMEECTRLREG(%0h)",
      frm.preamble_header.gi_type[0], bfmee_ctrl.short_gi_f))

    // check beamforming report
    data_cnt = 7;
    if (bfm_q.size() == 0)
      `uvm_error(get_type_name(),$sformatf("Beamforming report not collected on BFM interface!!!"))
    else
      bfm_report_item = bfm_q.pop_back();

    while (bfm_report_item.bfm_data.size() != 0) begin
      data = bfm_report_item.bfm_data.pop_back();
      if (frm.ampdu_frame[0].mpdu_frame[0].frame_body[data_cnt] != data)
        `uvm_error(get_type_name(),$sformatf("BFM report data mismatch, MPDU[%0d]=(%0h) - BFR IF (%0h)",
        data_cnt, frm.ampdu_frame[0].mpdu_frame[0].frame_body[data_cnt], data))
      data_cnt++;
    end

    `uvm_info(get_type_name(), $sformatf("Cheking of BFM report and ACTION frame OK!"), UVM_LOW)
  endfunction : check_bfm_report_item_frame_he

  //---------------------------------------------------------------------------
  // calculate frame durations from Rx frame (ACK, BLOCK_ACK, CTS)
  //---------------------------------------------------------------------------
  function void mac_immediate_resp_scoreboard::calc_response_frame_durations();

    int         tolerance;
    int         nsts_total;
    int         mcs;
    bit [2:0]   nss;
    HESIGB_s    HESIGB;

    if (rx_item.frame.preamble_header.format_mod == VHT && rx_item.frame.kind != MU_MIMO) begin
      nsts_total = rx_item.frame.preamble_header.user_header[0].num_sts_f;
      nss = (rx_item.frame.preamble_header.stbc > 0) ? (nsts_total+1)/(2*rx_item.frame.preamble_header.stbc)-1 : nsts_total;
      mcs = {nss[2:0],rx_item.frame.preamble_header.user_header[0].mcs_f[3:0]};
    end
    else begin
      nsts_total = rx_item.frame.preamble_header.user_header[0].num_sts_f;
      mcs = rx_item.frame.preamble_header.user_header[0].mcs_f;
    end

    // get HE control
    he_ctrl = get_ht_control(rx_item.frame);

    if (   rx_item.frame.preamble_header.format_mod >= HE_SU
        //An AP that sends an HE MU PPDU, HE SU PPDU or HE ER SU PPDU that solicits an immediate response
        //carried in an HE TB PPDU shall set the Ack Policy to HTP Ack for each of the QoS Data frames for which
        //it intends to solicit an immediate response (see 10.3.2.13.2 (Acknowledgment procedure for DL MU PPDU
        //in MU format)). If a Management frame that solicits acknowledgment is carried in an HE MU PPDU, then
        //the response is carried in an HE TB PPDU.
        && (   (rx_item.frame.ampdu_frame[0].mpdu_frame_type[0] inside {QOS_DATA,QOS_NULL} && get_block_ack_policy(rx_item.frame) == 2'b10)
            || (rx_item.frame.kind == AGGREGATED && has_qos_data && get_block_ack_policy(rx_item.frame) == 2'b10)
            ||  rx_item.frame.ampdu_frame[0].mpdu_frame_type[0][5:4] == `MANAGEMENT_MPDU
            || (rx_item.frame.ampdu_frame[0].mpdu_frame[0].MAC_header.ht_ctrl.size() > 0)
           )
        && (he_ctrl.vht_f && he_ctrl.he_f && he_ctrl.ctrl_id_f == 0)
       ) begin

      tolerance = 500; // add tolerance

      ack_duration = ComputeTimeOnAirAX ( .length_i          (20),
                                          .preType_i         (HE_TB),
                                          .mcsIndex_i        (he_ctrl.ctrl_info_f.ul_mcs_f),
                                          .giType_i          (rx_item.frame.preamble_header.gi_type),
                                          .ruType_i          (get_ru_type(he_ctrl.ctrl_info_f.ru_allocation_f)),
                                          .heLtfType_i       (rx_item.frame.preamble_header.he_ltf_type),
                                          .numHeLtf_i        (rx_item.frame.preamble_header.num_he_ltf),
                                          .dcm_i             (rx_item.frame.preamble_header.dcm),
                                          .packetExtension_i (rx_item.frame.preamble_header.user_header_he[0].pkt_ext_f),
                                          .heTbLength_i      (0),
                                          .triggerMethod_i   (0),
                                          .doppler_i         (rx_item.frame.preamble_header.doppler),
                                          .mma_i             (rx_item.frame.preamble_header.midamble_periodicity),
                                          .stbc_i            (rx_item.frame.preamble_header.stbc),
                                          .woPreamble_i      (0),
                                          .heSigB_i          (HESIGB),
                                          .debug_i           (0)
                                        ) + tolerance;

      ba_duration  = ComputeTimeOnAirAX ( .length_i     ((get_ba_variant(rx_item.frame) == BASIC_BA) ?
                                                          20 :   // length of ACK
                                                          40  ), // length of compressed BA
                                          .preType_i         (HE_TB),
                                          .mcsIndex_i        (he_ctrl.ctrl_info_f.ul_mcs_f),
                                          .giType_i          (rx_item.frame.preamble_header.gi_type),
                                          .ruType_i          (get_ru_type(he_ctrl.ctrl_info_f.ru_allocation_f)),
                                          .heLtfType_i       (rx_item.frame.preamble_header.he_ltf_type),
                                          .numHeLtf_i        (rx_item.frame.preamble_header.num_he_ltf),
                                          .dcm_i             (rx_item.frame.preamble_header.dcm),
                                          .packetExtension_i (rx_item.frame.preamble_header.user_header_he[0].pkt_ext_f),
                                          .heTbLength_i      (0),
                                          .triggerMethod_i   (0),
                                          .doppler_i         (rx_item.frame.preamble_header.doppler),
                                          .mma_i             (rx_item.frame.preamble_header.midamble_periodicity),
                                          .stbc_i            (rx_item.frame.preamble_header.stbc),
                                          .woPreamble_i      (0),
                                          .heSigB_i          (HESIGB),
                                          .debug_i           (0)
                                        ) + tolerance;
    end
    else begin
      tolerance = 300; // add tolerance

      ack_duration = ComputeTimeOnAirAC (.length_i     (14),   // length of ACK

                                         .preType_i    ((rx_item.frame.preamble_header.format_mod inside {NON_HT, NON_HT_DUP_OFDM}) ?
                                                         rx_item.frame.preamble_header.preamble_type :
                                                         rx_item.frame.preamble_header.format_mod),

                                         .mcsIndex_i   ((rx_item.frame.preamble_header.format_mod inside {NON_HT, NON_HT_DUP_OFDM}) ?
                                                         get_mcs_from_legacy_rate( // if NON_HT*
                                                           bss_basic_rate_set(rx_item.frame.preamble_header.leg_rate, basic_rates) // non HT mode
                                                         ) : // else
                                                         mcs), // HT mode

                                         .shortGI_i    (rx_item.frame.preamble_header.gi_type[0]),
                                         .chBW_i       (rx_item.frame.preamble_header.ch_bw),
                                         .stbc_i       (rx_item.frame.preamble_header.stbc),
                                         .extNSS_i     (rx_item.frame.preamble_header.num_extn_ss),
                                         .woPreamble_i (0),
                                         .band5G_i     (get_band5G(mac_ctrl_1.abgn_mode_f)),
                                         .multiUser_i  (0),
                                         .nSTSTotal_i  (0),
                                         .debug_i      (0)
                                         ) + tolerance;

      ba_duration =  ComputeTimeOnAirAC (.length_i     ((get_ba_variant(rx_item.frame) == BASIC_BA) ?
                                                        14 :   // length of ACK
                                                        32  ), // length of compressed BA

                                         .preType_i    ((rx_item.frame.preamble_header.format_mod inside {NON_HT, NON_HT_DUP_OFDM}) ?
                                                         rx_item.frame.preamble_header.preamble_type :
                                                         rx_item.frame.preamble_header.format_mod),

                                         .mcsIndex_i   ((rx_item.frame.preamble_header.format_mod inside {NON_HT, NON_HT_DUP_OFDM}) ?
                                                         get_mcs_from_legacy_rate( // if NON_HT*
                                                           bss_basic_rate_set(rx_item.frame.preamble_header.leg_rate, basic_rates) // non HT mode
                                                         ) : // else
                                                         mcs), // HT mode

                                         .shortGI_i    (rx_item.frame.preamble_header.gi_type[0]),
                                         .chBW_i       (rx_item.frame.preamble_header.ch_bw),
                                         .stbc_i       (rx_item.frame.preamble_header.stbc),
                                         .extNSS_i     (rx_item.frame.preamble_header.num_extn_ss),
                                         .woPreamble_i (0),
                                         .band5G_i     (get_band5G(mac_ctrl_1.abgn_mode_f)),
                                         .multiUser_i  (0),
                                         .nSTSTotal_i  (0),
                                         .debug_i      (0)
                                         ) + tolerance;

      cts_duration = ComputeTimeOnAirAC (.length_i     (14),   // length of CTS

                                         .preType_i    ((rx_item.frame.preamble_header.format_mod inside {NON_HT, NON_HT_DUP_OFDM}) ?
                                                         rx_item.frame.preamble_header.preamble_type :
                                                         rx_item.frame.preamble_header.format_mod),

                                         .mcsIndex_i   ((rx_item.frame.preamble_header.format_mod inside {NON_HT, NON_HT_DUP_OFDM}) ?
                                                         get_mcs_from_legacy_rate( // if NON_HT*
                                                           bss_basic_rate_set(rx_item.frame.preamble_header.leg_rate, basic_rates) // non HT mode
                                                         ) : // else
                                                         mcs), // HT mode

                                         .shortGI_i    (rx_item.frame.preamble_header.gi_type[0]),
                                         .chBW_i       (rx_item.frame.preamble_header.ch_bw),
                                         .stbc_i       (rx_item.frame.preamble_header.stbc),
                                         .extNSS_i     (rx_item.frame.preamble_header.num_extn_ss),
                                         .woPreamble_i (0),
                                         .band5G_i     (get_band5G(mac_ctrl_1.abgn_mode_f)),
                                         .multiUser_i  (0),
                                         .nSTSTotal_i  (0),
                                         .debug_i      (0)
                                         ) + tolerance;
    end

  endfunction : calc_response_frame_durations

  //-------------------------------------------------------------------------------------
  // calculate duration of response BFR frame in HE
  //-------------------------------------------------------------------------------------
  function int mac_immediate_resp_scoreboard::get_bfm_report_duration_he();
    int bfm_report_size;
    int bfr_he_len;
    int txtime;
    bit [3:0] num_he_ltf;
    bit mma;
    bit [1:0] gi_type;
    bit [1:0] he_ltf_type;
    HESIGB_s HESIGB;
    TRIGGER_frame trigger;
    HE_NDP_ANNOUNCEMENT_frame ndpa;

    trigger = new("trigger");
    trigger.copy(bfr_poll_item.frame.ampdu_frame[0].mpdu_frame[0]);
    ndpa = new("ndpa");
    ndpa.copy(ndpa_frame.ampdu_frame[0].mpdu_frame[0]);

    // Get the right size of the BFM, to set correctly the
    // length in the Trigger frame
    bfm_report_size = he_bfm_report_size_estimator( .Nr            (bfmee_ctrl.nr_f),
                                                    .Nc            (ndpa.sta_info[staid].nc_index_f),
                                                    .ch_bw         (ndpa.ch_bw),
                                                    .grouping      (ndpa.sta_info[staid].feedback_type_ng_f[1]),
                                                    .feedbacktype  (ndpa.sta_info[staid].feedback_type_ng_f[0]),
                                                    .codebook      (ndpa.sta_info[staid].codebook_size_f),
                                                    .ru_start_index(ndpa.sta_info[staid].ru_start_index_f),
                                                    .ru_end_index  (ndpa.sta_info[staid].ru_end_index_f),
                                                    .debug         (0));

    `uvm_info(get_type_name(),$sformatf("Expected beamforming report size = %0d",bfm_report_size),UVM_LOW)
    // calculate size of BFR frame size which will be in HE-TB
    // MAC header + category and action + HE MUMIMO cotrol + BFR size
    // + delimiter + FCS
    bfr_he_len = 28 + 2 + 5 + bfm_report_size + 4 + 4;
    // add padding bytes
    bfr_he_len += (bfr_he_len % 4 != 0) ? (4 - (bfr_he_len % 4)) : 0;
    bfr_he_len += 10; //add more dummy bytes to avoid phy_err flag
    // determine NumHELTF and midamble_periodicity
    if (trigger.common_info.doppler_f)
      {num_he_ltf,mma} = (trigger.common_info.num_of_he_ltf_symbols_f == 0) ? {4'h0,1'b0} :
                         (trigger.common_info.num_of_he_ltf_symbols_f == 1) ? {4'h1,1'b0} :
                         (trigger.common_info.num_of_he_ltf_symbols_f == 2) ? {4'h3,1'b0} :
                         (trigger.common_info.num_of_he_ltf_symbols_f == 4) ? {4'h0,1'b1} :
                         (trigger.common_info.num_of_he_ltf_symbols_f == 5) ? {4'h1,1'b1} :
                         (trigger.common_info.num_of_he_ltf_symbols_f == 6) ? {4'h3,1'b1} : {4'h0,1'b0};
    else
      {num_he_ltf,mma} = (trigger.common_info.num_of_he_ltf_symbols_f == 0) ? {4'h0,1'b0} :
                         (trigger.common_info.num_of_he_ltf_symbols_f == 1) ? {4'h1,1'b0} :
                         (trigger.common_info.num_of_he_ltf_symbols_f == 2) ? {4'h3,1'b0} :
                         (trigger.common_info.num_of_he_ltf_symbols_f == 3) ? {4'h5,1'b0} :
                         (trigger.common_info.num_of_he_ltf_symbols_f == 4) ? {4'h7,1'b0} : {4'h0,1'b0};

    {gi_type, he_ltf_type} = (trigger.common_info.gi_and_ltf_type_f == 0) ? {2'b01, 2'b00} :
                             (trigger.common_info.gi_and_ltf_type_f == 1) ? {2'b01, 2'b01} :
                             (trigger.common_info.gi_and_ltf_type_f == 2) ? {2'b10, 2'b10} : {2'b00 ,2'b00};

    txtime = ComputeTimeOnAirAX (.length_i          (bfr_he_len),
                                 .preType_i         (HE_TB),
                                 .mcsIndex_i        (trigger.user_info[staid].mcs_f),
                                 .giType_i          (gi_type),
                                 .ruType_i          (get_ru_type(trigger.user_info[staid].ru_allocation_f)),
                                 .heLtfType_i       (he_ltf_type),
                                 .numHeLtf_i        (num_he_ltf),
                                 .dcm_i             (trigger.user_info[staid].dcm_f),
                                 .packetExtension_i (trigger.common_info.pre_fec_padding_f),
                                 .heTbLength_i      (0),
                                 .triggerMethod_i   (0),
                                 .doppler_i         (trigger.common_info.doppler_f),
                                 .mma_i             (mma),
                                 .stbc_i            (trigger.common_info.stbc_f),
                                 .woPreamble_i      (0),
                                 .heSigB_i          (HESIGB),
                                 .debug_i           (0));

    txtime = txtime + 100; // add tolerance
    `uvm_info(get_type_name(),$sformatf("Beamforming report TXTIME = %0d",txtime),UVM_LOW)

    return txtime;
  endfunction : get_bfm_report_duration_he

`endif //MAC_IMMEDIATE_RESP_SCOREBOARD_SV
