//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_PTA_SCOREBOARD_SV
`define MAC_PTA_SCOREBOARD_SV

`uvm_analysis_imp_decl(_wlan)
`uvm_analysis_imp_decl(_bt)

class mac_pta_scoreboard extends mac_scoreboard_base;

  `uvm_component_utils(mac_pta_scoreboard)

  uvm_analysis_imp_wlan#(pta_wlan_seq_item, mac_pta_scoreboard)  wlan_export;
  uvm_analysis_imp_bt#(coex_bt_seq_item, mac_pta_scoreboard)     bt_export;

  // received items queues
  pta_wlan_seq_item           wlan_check_it;
  coex_bt_seq_item            bt_check_it;

  // list of variables
  bit               abortrx_when_tx;
  bit               pta_enable;
  bit               basic_priority;
  bit               no_sim_tx;
  bit               no_sim_rx;
  bit               pti_enable;
  bit               chan_enable;
  bit               bt_event_mask;
  bit               wlan_pti_mode;
  bit [6:0]         chan_margin;
  bit               pta_priority;
  bit               chan_overlap;
  bit [3:0]         transfer_abort;
  bit               bt_tx_abort;
  bit               bt_rx_abort;
  bit               wlan_tx_abort;
  bit               wlan_rx_abort;
  bit               received_wlan_packet = 0;
  bit               received_bt_packet = 0;
  bit               first_time_received_bt_packet = 0;
  bit               first_time_received_wlan_packet = 0;
  bit [3:0]         temp_transfer_abort;
  bit [6:0]         wlan_chan_freq;
  bit               wlan_chan_bw;
  bit               wlan_chan_offset;
  bit [3:0]         wlan_pti;
  bit               wlan_tx;
  bit               wlan_rx;
  bit [6:0]         bt_channel;
  bit               bt_bw;
  bit [3:0]         bt_pti;
  bit               bt_rx;
  bit               bt_tx;
  bit               bt_event;

  //======================================
  //    Cover groups
  //======================================
  covergroup pta_cg;

    basic_priority_cp : coverpoint basic_priority {
      bins BT   = {0};
      bins WLAN = {1};
    }

    wlan_tx_abort_cp : coverpoint wlan_check_it.wlan_tx_abort {
      bins wlan_tx_abort_inactive = {0};
      bins wlan_tx_abort_active   = {1};
    }

    wlan_rx_abort_cp : coverpoint wlan_check_it.wlan_rx_abort {
      bins wlan_rx_abort_inactive = {0};
      bins wlan_rx_abort_active   = {1};
    }

    bt_tx_abort_cp : coverpoint bt_check_it.bt_tx_abort {
      bins bt_tx_abort_inactive = {0};
      bins bt_tx_abort_active   = {1};
    }

    bt_rx_abort_cp : coverpoint bt_check_it.bt_rx_abort {
      bins bt_rx_abort_inactive = {0};
      bins bt_rx_abort_active   = {1};
    }

    bt_event_cp : coverpoint bt_check_it.sig_value[`BT_EVENT]{
      bins bt_event_inactive = {0};
      bins bt_event_active   = {1};
    }

    wlan_pti_cp : coverpoint wlan_check_it.sig_value[`WLAN_PTI] {
      option.auto_bin_max = 3;
    }

    bt_pti_cp   : coverpoint bt_check_it.sig_value[`BT_PTI] {
      option.auto_bin_max = 3;
    }

    wlan_chan_freq_cp : coverpoint wlan_check_it.sig_value[`WLAN_CHAN_FREQ] {
      bins         legal_values = {[12:84]};
      illegal_bins out_of_range = {[0:11],[85:127]};
    }

    bt_channel_cp : coverpoint bt_check_it.sig_value[`BT_CHANNEL] {
      bins         legal_values = {[0:78]};
      illegal_bins out_of_range = {[79:127]};
    }

    wlan_tx_cp : coverpoint wlan_check_it.sig_value[`WLAN_TX] {
      bins wlan_tx_active = {1};
    }

    wlan_rx_cp : coverpoint wlan_check_it.sig_value[`WLAN_RX] {
      bins wlan_rx_active = {1};
    }

    bt_tx_cp : coverpoint bt_check_it.sig_value[`BT_TX] {
      bins bt_tx_active = {1};
    }

    bt_rx_cp : coverpoint bt_check_it.sig_value[`BT_RX] {
      bins bt_rx_active = {1};
    }

    basic_pr_pti_cr : cross basic_priority_cp,wlan_pti_cp,bt_pti_cp;

  endgroup


  function new(string name = "mac_pta_scoreboard", uvm_component parent = null);
    super.new(name, parent);

    wlan_export    = new("wlan_export", this);
    bt_export      = new("bt_export", this);
    wlan_check_it = new();
    bt_check_it   = new();
    pta_cg         = new();
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);
  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase

  function void report_phase(uvm_phase phase);
    super.report_phase(phase);

  endfunction : report_phase

  task run_phase(uvm_phase phase);
    super.run_phase(phase);

    wait (first_time_received_wlan_packet==1'b1 && first_time_received_bt_packet==1'b1);

    forever begin
      wait (received_wlan_packet==1'b1 || received_bt_packet==1'b1) begin
        received_wlan_packet = 1'b0;
        received_bt_packet = 1'b0;

        check_pta();

      end
    end

  endtask

  //---------------------------------------------------------------------------
  // list of functions for checking
  //---------------------------------------------------------------------------
  extern function void        write_wlan(pta_wlan_seq_item i);
  extern function void        write_bt(coex_bt_seq_item i);
  extern function bit         calculate_priority(bit basic_priority,
                                                 bit pti_enable,
                                                 bit [3:0] bt_pti,
                                                 bit [3:0] wlan_pti);
  extern function bit         channel_overlap_detection(bit [6:0] wlan_chan_freq,
                                                        bit wlan_chan_bw,
                                                        bit wlan_chan_offset,
                                                        bit [6:0] chan_margin,
                                                        bit [6:0] bt_channel,
                                                        bit bt_bw);
  extern function bit [3:0]   arbitration(bit pta_priority,
                                          bit chan_overlap,
                                          bit wlan_pti_mode,
                                          bit no_sim_tx,
                                          bit no_sim_rx,
                                          bit wlan_tx,
                                          bit wlan_rx,
                                          bit bt_tx,
                                          bit bt_rx,
                                          bit bt_event,
                                          bit chan_enable);
  extern function void        check_pta();
  //---------------------------------------------------------------------------

endclass : mac_pta_scoreboard

function void mac_pta_scoreboard::write_wlan(pta_wlan_seq_item i);

  pta_wlan_seq_item it;

  it = new();
  it.copy(i);
  wlan_check_it = it;
  first_time_received_wlan_packet = 1'b1;

  if ((it.cmd == WLAN_TX_ABORT && it.wlan_tx_abort == 1) || (it.cmd == WLAN_RX_ABORT && it.wlan_rx_abort == 1)) begin
    received_wlan_packet = 1'b1;
  end

endfunction : write_wlan

function void mac_pta_scoreboard::write_bt(coex_bt_seq_item i);

  coex_bt_seq_item  it;

  it = new();
  it.copy(i);
  bt_check_it = it;
  first_time_received_bt_packet = 1'b1;

  if ((it.cmd == BT_TX_ABORT && it.bt_tx_abort == 1) || (it.cmd == BT_RX_ABORT && it.bt_rx_abort == 1)) begin
    received_bt_packet = 1'b1;
  end

endfunction : write_bt

// Function to check the PTA behaviour
function void mac_pta_scoreboard::check_pta();


  bt_channel        = bt_check_it.sig_value[`BT_CHANNEL];
  bt_bw             = bt_check_it.sig_value[`BT_BW];
  bt_pti            = bt_check_it.sig_value[`BT_PTI];
  bt_tx_abort       = bt_check_it.bt_tx_abort;
  bt_rx_abort       = bt_check_it.bt_rx_abort;
  bt_tx             = bt_check_it.sig_value[`BT_TX];
  bt_rx             = bt_check_it.sig_value[`BT_RX];
  bt_event          = bt_check_it.sig_value[`BT_EVENT];

  wlan_chan_freq    = wlan_check_it.sig_value[`WLAN_CHAN_FREQ];
  wlan_chan_bw      = wlan_check_it.sig_value[`WLAN_CHAN_BW];
  wlan_chan_offset  = wlan_check_it.sig_value[`WLAN_CHAN_OFFSET];
  wlan_pti          = wlan_check_it.sig_value[`WLAN_PTI];
  wlan_tx_abort     = wlan_check_it.wlan_tx_abort;
  wlan_rx_abort     = wlan_check_it.wlan_rx_abort;
  wlan_tx           = wlan_check_it.sig_value[`WLAN_TX];
  wlan_rx           = wlan_check_it.sig_value[`WLAN_RX];

  // Get the latest values written in the CONFIG register
  fetch_register_values();
  pta_enable      = pta_config[0];
  basic_priority  = pta_config[1];
  no_sim_tx       = pta_config[2];
  no_sim_rx       = pta_config[3];
  pti_enable      = pta_config[4];
  chan_enable     = pta_config[5];
  bt_event_mask   = pta_config[6];
  wlan_pti_mode   = pta_config[7];
  chan_margin     = pta_config[14:8];
  abortrx_when_tx = pta_config[20];

  pta_cg.sample();

  if (pta_enable==1'b1) begin

    // Mask the bt event, if it is specified in the CONFIG register
    bt_event = bt_event & ~bt_event_mask;
    // Calculate basic priority
    pta_priority = calculate_priority(basic_priority, pti_enable, bt_pti, wlan_pti);

    // Check if there is a channel overlap
    if (chan_enable == 1'b1) begin
      chan_overlap = channel_overlap_detection(wlan_chan_freq, wlan_chan_bw, wlan_chan_offset, chan_margin, bt_channel, bt_bw);
    end

    // Based on the previous calculations check, whether the arbitration is done as it should
    transfer_abort = arbitration(pta_priority, chan_overlap, wlan_pti_mode, no_sim_tx, no_sim_rx, wlan_tx, wlan_rx, bt_tx, bt_rx, bt_event, chan_enable);

    // transfer_abort[0] - bt_tx_abort
    // transfer_abort[1] - bt_rx_abort
    // transfer_abort[2] - wlan_tx_abort
    // transfer_abort[3] - wlan_rx_abort

    // BT TX abort should be set
    if (transfer_abort[0] != bt_tx_abort) begin
      `uvm_error(get_type_name(), $sformatf("Expected value for bt_tx_abort is = %b, actual value is = %b", transfer_abort[0], bt_tx_abort))
    end
    else begin
      `uvm_info(get_type_name(), "The BT TX_ABORT has the right value", UVM_HIGH)
    end

    // BT RX abort should be set
    if (transfer_abort[1] != bt_rx_abort) begin
      `uvm_error(get_type_name(), $sformatf("Expected value for bt_rx_abort is = %b, actual value is = %b", transfer_abort[1], bt_rx_abort))
    end
    else begin
      `uvm_info(get_type_name(), "The BT RX_ABORT has the right value", UVM_HIGH)
    end

    // WLAN TX abort
    if (transfer_abort[2] != wlan_tx_abort) begin
      `uvm_error(get_type_name(), $sformatf("Expected value for wlan_tx_abort is = %b, actual value is = %b", transfer_abort[2], wlan_tx_abort))
    end
    else begin
      `uvm_info(get_type_name(), "The WLAN TX_ABORT has the right value", UVM_HIGH)
    end

    // WLAN RX abort
    if (transfer_abort[3] != wlan_rx_abort) begin
      `uvm_error(get_type_name(), $sformatf("Expected value for wlan_rx_abort is = %b, actual value is = %b", transfer_abort[3], wlan_rx_abort))
    end
    else begin
      `uvm_info(get_type_name(), "The WLAN RX_ABORT has the right value", UVM_HIGH)
    end

  end

endfunction : check_pta

// Calculate priority function is responsible for calculation of a priority, based on input parameters
function bit mac_pta_scoreboard::calculate_priority(bit basic_priority, bit pti_enable, bit [3:0] bt_pti, bit [3:0] wlan_pti);


  if (pti_enable == 1'b0) begin   // the arbitration based on pti is disabled, return
    return basic_priority;        // the priority, depending on basic priority information
  end
  else begin
    if (bt_pti == wlan_pti) begin
      return basic_priority;
    end
    else if (wlan_pti > bt_pti) begin
      return 1; // return 1, which means that WLAN has higher priority
    end
    else if (wlan_pti < bt_pti) begin
      return 0; // return 0, which means tha BT has higher priority
    end
  end

endfunction : calculate_priority

function bit mac_pta_scoreboard::channel_overlap_detection(bit [6:0] wlan_chan_freq, bit wlan_chan_bw, bit wlan_chan_offset, bit [6:0] chan_margin, bit [6:0] bt_channel, bit bt_bw);

  bit [6:0]   chan_freq_dist;
  bit [6:0]   chan_freq_dist_sgn;
  bit [6:0]   chan_freq_dist_min;

  // Get the distance between BT and WLAN
  chan_freq_dist_sgn = wlan_chan_freq - (bt_channel + 2);

  // Get the absolute value
  chan_freq_dist = (chan_freq_dist_sgn[6] == 1'b1) ? (~chan_freq_dist_sgn) + 7'b1 : chan_freq_dist_sgn;
  // Channel margin table
  case ({wlan_chan_bw, bt_bw, wlan_chan_offset, chan_freq_dist_sgn[6]})
    4'b0010, 4'b0000,
    4'b0011, 4'b0001,
    4'b1010, 4'b1001:
      chan_freq_dist_min = {1'b0, chan_margin} + 12;
    4'b0110, 4'b0100,
    4'b0111, 4'b0101,
    4'b1110, 4'b1101:
      chan_freq_dist_min = {1'b0, chan_margin} + 13;
    4'b1000, 4'b1011:
      chan_freq_dist_min = {1'b0, chan_margin} + 32;
    4'b1100, 4'b1111:
      chan_freq_dist_min = {1'b0, chan_margin} + 33;
  endcase

  if (chan_freq_dist < chan_freq_dist_min) begin
    return 1;
  end
  else begin
    return 0;
  end
endfunction : channel_overlap_detection

function bit [3:0] mac_pta_scoreboard::arbitration(bit pta_priority,
                                                   bit chan_overlap,
                                                   bit wlan_pti_mode,
                                                   bit no_sim_tx,
                                                   bit no_sim_rx,
                                                   bit wlan_tx,
                                                   bit wlan_rx,
                                                   bit bt_tx,
                                                   bit bt_rx,
                                                   bit bt_event,
                                                   bit chan_enable);

  bit [3:0]   bt_wlan_transfer_abort = 0;

  // chan_overlap == 1'b1 => There is a channel overlap
  // chan_overlap == 1'b0 => There is no channel overlap
  if (pta_priority == 1'b0) begin
    // BT priority
    case ({wlan_tx, wlan_rx, bt_tx, bt_rx, bt_event})
      5'b00100, 5'b00101:
                begin
                  // BT TX, WLAN IDLE
                  bt_wlan_transfer_abort[2] = chan_overlap & (!pti_enable | wlan_pti_mode) & no_sim_tx; // wlan_tx_abort
                  bt_wlan_transfer_abort[3] = chan_overlap & abortrx_when_tx;             // wlan_rx_abort
                end
      5'b00010, 5'b00011:
                begin
                  // BT RX, WLAN IDLE
                  bt_wlan_transfer_abort[2] = chan_overlap & (!pti_enable | wlan_pti_mode);             // wlan_tx_abort
                  bt_wlan_transfer_abort[3] = chan_overlap & (!pti_enable | wlan_pti_mode) & no_sim_rx; // wlan_rx_abort
                end
      5'b00001: begin
                  // BT EVENT, WLAN IDLE
                  bt_wlan_transfer_abort[2] = (!chan_enable & !pti_enable) & chan_overlap; // wlan_tx_abort
                  bt_wlan_transfer_abort[3] = 0; // wlan_rx_abort
                end
      5'b10100, 5'b10101:
                begin
                  // BT TX, WLAN TX
                  bt_wlan_transfer_abort[2] = chan_overlap & no_sim_tx; // wlan_tx_abort
                  bt_wlan_transfer_abort[3] = chan_overlap & abortrx_when_tx;                        // wlan_rx_abort
                end
      5'b01100, 5'b01101:
                begin
                  // BT TX, WLAN RX
                  bt_wlan_transfer_abort[2] = chan_overlap & no_sim_tx;            // wlan_tx_abort
                  bt_wlan_transfer_abort[3] = chan_overlap & abortrx_when_tx;      // wlan_rx_abort
                end
      5'b10010, 5'b10011:
                // BT RX, WLAN TX
                begin
                  bt_wlan_transfer_abort[2] = chan_overlap;               // wlan_tx_abort
                  bt_wlan_transfer_abort[3] = chan_overlap & no_sim_rx;   // wlan_rx_abort
                end
      5'b01010, 5'b01011:
                begin
                  // BT RX, WLAN RX
                  bt_wlan_transfer_abort[2] = chan_overlap;                        // wlan_tx_abort
                  bt_wlan_transfer_abort[3] = chan_overlap & no_sim_rx;            // wlan_rx_abort
                end
      5'b10001: begin
                  // BT EVENT, WLAN TX
                  bt_wlan_transfer_abort[2] = (!chan_enable & !pti_enable) & chan_overlap; // wlan_tx_abort
                  bt_wlan_transfer_abort[3] = 0;                                           // wlan_rx_abort
                end
      5'b01001: begin
                  // BT EVENT, WLAN RX
                  bt_wlan_transfer_abort[2] = (!pti_enable | wlan_pti_mode) & chan_overlap;                          // wlan_tx_abort
                  bt_wlan_transfer_abort[3] = 0;                                                                     // wlan_rx_abort
                end
      default:  begin
                  bt_wlan_transfer_abort[2] = 0;
                  bt_wlan_transfer_abort[3] = 0;
                end
    endcase
  end
  else begin
    // WLAN priority
    case ({wlan_tx, wlan_rx, bt_tx, bt_rx, bt_event})
      5'b10000: begin
                  // BT IDLE, WLAN TX
                  bt_wlan_transfer_abort[0] = 0; // bt_tx_abort
                  bt_wlan_transfer_abort[1] = 0; // bt_rx_abort
                end
      5'b01000: begin
                  // BT IDLE, WLAN RX
                  bt_wlan_transfer_abort[0] = 0;             // bt_tx_abort
                  bt_wlan_transfer_abort[1] = 0; // bt_rx_abort
                end
      5'b10001: begin
                  // BT EVENT, WLAN TX
                  bt_wlan_transfer_abort[0] = 0; // bt_tx_abort
                  bt_wlan_transfer_abort[1] = 0;             // bt_rx_abort
                end
      5'b01001: begin
                  // BT EVENT, WLAN RX
                  bt_wlan_transfer_abort[0] = 0;             // bt_tx_abort
                  bt_wlan_transfer_abort[1] = 0; // bt_rx_abort
                end
      5'b10100, 5'b10101:
                begin
                  // BT TX, WLAN TX
                  bt_wlan_transfer_abort[0] = chan_overlap & no_sim_tx; // bt_tx_abort
                  bt_wlan_transfer_abort[1] = chan_overlap & abortrx_when_tx;                        // bt_rx_abort
                end
      5'b01100, 5'b01101:
                begin
                  // BT TX, WLAN RX
                  bt_wlan_transfer_abort[0] = chan_overlap; // bt_tx_abort
                  bt_wlan_transfer_abort[1] = chan_overlap & no_sim_rx;            // bt_rx_abort
                end
      5'b10010, 5'b10011:
                begin
                  // BT RX, WLAN TX
                  bt_wlan_transfer_abort[0] = chan_overlap & no_sim_tx;            // bt_tx_abort
                  bt_wlan_transfer_abort[1] = chan_overlap & abortrx_when_tx;     // bt_rx_abort
                end
      5'b01010, 5'b01011:
                begin
                  // BT RX, WLAN RX
                  bt_wlan_transfer_abort[0] = chan_overlap;             // bt_tx_abort
                  bt_wlan_transfer_abort[1] = chan_overlap & no_sim_rx; // bt_rx_abort
                end
      default:  begin
                  bt_wlan_transfer_abort[0] = 0;
                  bt_wlan_transfer_abort[1] = 0;
                end
    endcase
  end

  return bt_wlan_transfer_abort;

endfunction : arbitration

`endif// MAC_PTA_SCOREBOARD_SV
