//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_RX_SCOREBOARD_SV
`define MAC_RX_SCOREBOARD_SV

`uvm_analysis_imp_decl(_mac_rx_sram)
`uvm_analysis_imp_decl(_mac_rx_mac_phy)

class mac_rx_scoreboard extends mac_scoreboard_base;

  `uvm_component_utils(mac_rx_scoreboard)

  uvm_analysis_imp_mac_rx_sram#(sram_seq_item, mac_rx_scoreboard)       sram_export;
  uvm_analysis_imp_mac_rx_mac_phy#(mac_phy_seq_item, mac_rx_scoreboard) mac_phy_export;

  // received items queues
  sram_seq_item           sram_q[$], sram_item;
  mac_phy_seq_item        mac_phy_q[$], mac_phy_item;
  bit                     nav_flag = 0;
  // list of variables
  bit                     aggregated;
  timestamp_t             tsf_in_rhd;
  timestamp_t             tsf_in_frame_timestamp;
  timestamp_t             previous_tsf_value;
  int                     mpdu_cnt;
  int                     valid_mpdu_cnt;
  int                     intra_or_inter_BSS = 0;
  int                     rx_end_delay;

  function new(string name = "mac_rx_scoreboard", uvm_component parent = null);
    super.new(name, parent);

    sram_export    = new("sram_export", this);
    mac_phy_export = new("mac_phy_export", this);
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);
  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase

  task run_phase(uvm_phase phase);
    super.run_phase(phase);

    fork
      // run thread if channel not busy else thread is still running
      nav_counter();
    join_none

    forever begin

      // receive items from both monitors and check them
      wait (mac_phy_q.size() != 0);
      mac_phy_item = mac_phy_q.pop_back();

      fetch_tsf_reg(tsf_in_rhd);

      // For the TSF timer, we have to know the delay between the rx_end
      // and rx_end_for_timing to count this delay when reading the
      // TSF timer register backdoor
      rx_end_delay = ($time - mac_phy_item.end_of_frame)/1000 + 5;

      fork
        check_rx_events();
      join_none

      wait (sram_q.size() != 0);
      sram_item = sram_q.pop_back();

      // decrypt frame if needed before comparison
      if (!rx_ctrl.dont_decrypt_f) begin
        mac_phy_item.frame.decrypt(0);
      end

      mpdu_cnt = 0;
      valid_mpdu_cnt = 0;
      // check Rx vector first, in list only one RHD has valid Rx vector
      // stored inside (valid signal is high)
      foreach (sram_item.rhd_list[rhd_cnt]) begin
        if (sram_item.rhd_list[rhd_cnt].mpdu_statu_info.rx_vector2_valid_f) begin
          if (mac_phy_item.frame.kind != NDP)
            aggregated = mac_phy_item.frame.ampdu_frame[0].aggregated;
          else
            aggregated = 1;

          check_rx_vector(mac_phy_item.frame.preamble_header,
                          sram_item.rhd_list[rhd_cnt]);

          if (!aggregated && m_cfg.check_mac_tsf_reg)
            check_tsf(sram_item.rhd_list[rhd_cnt]);
        end

        // only non NDP frames contain payload so checking can be performed
        if (mac_phy_item.frame.kind != NDP) begin

          // don't compare FCS error frames
          while (mac_phy_item.frame.ampdu_frame[0].mpdu_frame_type[mpdu_cnt] == ERROR_FRAME) begin
            mpdu_cnt++;
            if (mpdu_cnt > mac_phy_item.frame.ampdu_frame[0].mpdu_frame_type.size())
              break; // leave while loop because maximum size is reached
          end// while

          // if we already compared all valid frames exit foreach loop
          if (valid_mpdu_cnt == number_of_valid_mpdu_frames(mac_phy_item.frame, .user(0)))
            break; // leave foreach

          `uvm_info(get_type_name(), $sformatf("Comparing 2 MPDU frames...MPDU[%0d] with RHD[%0d]",mpdu_cnt,rhd_cnt), UVM_LOW)

          check_mpdu_status_info(sram_item.rhd_list[rhd_cnt].mpdu_statu_info,
                                 mac_phy_item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt]);
          // go throught list of RPDs per RHD and check content with received
          // MPDU from MAC-PHY interface
          compare_2_mpdu_frames(mac_phy_item.frame.ampdu_frame[0].mpdu_frame[mpdu_cnt],
                                sram_item.rhd_list[rhd_cnt],
                                sram_item.rpd_list[rhd_cnt]);

          valid_mpdu_cnt++;
          mpdu_cnt++;
        end// not NDP

      end// foreach

    end// forever
  endtask : run_phase

  function void report_phase(uvm_phase phase);
    super.report_phase(phase);

    if (sram_q.size() > 0) begin
      `uvm_error(get_type_name(),$sformatf("SRAM queue is not empty, items left:"))
      while (sram_q.size() > 0) begin
        sram_item = sram_q.pop_back();
        `uvm_info(get_type_name(), $sformatf("SRAM item: %s", sram_item.sprint()), UVM_LOW)
      end
    end
  endfunction : report_phase

  //---------------------------------------------------------------------------
  // list of functions for checking
  //---------------------------------------------------------------------------
  extern function void write_mac_rx_sram(sram_seq_item i);
  extern function void write_mac_rx_mac_phy(mac_phy_seq_item i);
  extern function void check_rx_vector(PPDU_preamble_header ph, RHD rhd);
  extern function void check_tsf(RHD rhd);
  extern function void check_mpdu_status_info(mpdu_statu_info_s mpdu_statu_info,
                                              MPDU_frame        mpdu);
  extern function void compare_2_mpdu_frames(MPDU_frame mpdu, RHD rhd, RPD rpd[]);
  extern task          check_rx_events();
  extern task          nav_counter();
  extern function int  security_header_bytes(ref MPDU_frame mpdu);
  extern function int  security_footer_bytes(ref MPDU_frame mpdu);
  //---------------------------------------------------------------------------

endclass : mac_rx_scoreboard

  //---------------------------------------------------------------------------
  // implementation of checking functions
  //---------------------------------------------------------------------------

  function void mac_rx_scoreboard::write_mac_rx_sram(sram_seq_item i);
    // only when read from SRAM memory is triggered then checking is performed
    if (i.cmd == READ_RX_FRAME) begin
      sram_q.push_front(i);
      `uvm_info(get_type_name(), $sformatf("SRAM item received %s", i.sprint()), UVM_HIGH)
    end
    else begin
      `uvm_info(get_type_name(), $sformatf("MAC RX scorebord discarded %s SRAM item",i.cmd.name()), UVM_DEBUG)
    end
  endfunction : write_mac_rx_sram

  function void mac_rx_scoreboard::write_mac_rx_mac_phy(mac_phy_seq_item i);
    mac_phy_seq_item it;

    it = new();
    it.copy(i);
    fetch_register_values();
    // tx items are not checked
    if (it.trans == TX) begin
      `uvm_info(get_type_name(), $sformatf("MAC RX scorebord discarded TX MAC-PHY item"), UVM_DEBUG)
    end
    // only when reception of PPDU frame is generated then checking is
    // performed and frame is not filtered
    else if (it.trans == RX && filter_ppdu_frame(it) == 0) begin

      `uvm_info(get_type_name(), $sformatf("MAC-PHY item received %s", it.sprint()), UVM_HIGH)
      // skipp NDP for NAV update
      if (it.frame.kind != NDP) begin
        // Check who is the TXOP holder
        txop_holder_addr = get_txop_holder(it.frame);
        // Check whether the frame is an intra-BSS, inter-BSS or neither
        intra_or_inter_BSS = intra_or_inter_BSS_frame(it.frame);

        // update NAV based on the parameters
        updating_two_NAVs(intra_or_inter_BSS, txop_holder_addr, it.frame);
        nav_flag = 1'b1;
      end//NDP

      mac_phy_q.push_front(it);
    end
    else if ((it.trans == RX) && (it.frame.ampdu_frame[0].no_valid_frames == 0)) begin
      txop_holder_addr = get_txop_holder(it.frame);
      // Check whether the frame is an intra-BSS, inter-BSS or neither
      intra_or_inter_BSS = intra_or_inter_BSS_frame(it.frame);

      // update NAV based on the parameters
      updating_two_NAVs(intra_or_inter_BSS, txop_holder_addr, it.frame);
      nav_flag = 1'b1;
      frame_filtered_trigger.trigger();
    end
    else begin
      `uvm_info(get_type_name(), $sformatf("MAC filtered frame %s MAC-PHY item",it.trans.name()), UVM_HIGH)
      frame_filtered_trigger.trigger();
    end
  endfunction : write_mac_rx_mac_phy

  //---------------------------------------------------------------------------
  // function for checking Rx vector received by MAC core
  //---------------------------------------------------------------l------------
  function void mac_rx_scoreboard::check_rx_vector(PPDU_preamble_header ph, RHD rhd);
    common_rxvector_s         common_part;
    common_he_rxvector_s      common_he_part;
    nonht_rxvector_s          nonht_rxv;
    ht_rxvector_s             ht_rxv;
    vht_rxvector_s            vht_rxv;
    he_su_rxvector_s          he_su_rxv;
    he_mu_rxvector_s          he_mu_rxv;
    he_trig_rxvector_ap_s     he_tb_rxv;
    rxvector2_s               rxv2;

    case (ph.format_mod)
      NON_HT, NON_HT_DUP_OFDM: begin
        nonht_rxv = {rhd.received_vector_1b,
                     rhd.received_vector_1a};
        common_part = nonht_rxv.common_part;
      end
      HT_MF, HT_GF: begin
        ht_rxv = {rhd.received_vector_1c,
                  rhd.received_vector_1b,
                  rhd.received_vector_1a};
        common_part = ht_rxv.common_part;
      end
      VHT: begin
        vht_rxv = {rhd.received_vector_1c,
                   rhd.received_vector_1b,
                   rhd.received_vector_1a};
        common_part = vht_rxv.common_part;
      end
      HE_SU, HE_EXT_SU: begin
        he_su_rxv = {rhd.received_vector_1d,
                     rhd.received_vector_1c,
                     rhd.received_vector_1b,
                     rhd.received_vector_1a};

        common_part    = he_su_rxv.common_part;
        common_he_part = he_su_rxv.common_he_part;
      end
      HE_MU: begin
        he_mu_rxv = {rhd.received_vector_1d,
                     rhd.received_vector_1c,
                     rhd.received_vector_1b,
                     rhd.received_vector_1a};

        common_part    = he_mu_rxv.common_part;
        common_he_part = he_mu_rxv.common_he_part;
      end
      HE_TB: begin
        he_tb_rxv = {rhd.received_vector_1d,
                     rhd.received_vector_1c,
                     rhd.received_vector_1b,
                     rhd.received_vector_1a};

        common_part    = he_tb_rxv.common_part;
        common_he_part = he_tb_rxv.common_he_part;
      end
    endcase
    //set RxVector2 bytes
    rxv2 = {rhd.received_vector_2b, rhd.received_vector_2a};

    //-------------------------------------------------------------------------
    // CHECK COMMON PART FOR RXVECTOR
    //-------------------------------------------------------------------------
    // check format mode
    if (ph.format_mod != common_part.format_mode)
      `uvm_error(get_type_name(),
      $sformatf("Format mode wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.format_mod, common_part.format_mode))

    // check channel bandwith
    if (ph.ch_bw != common_part.ch_bw)
      `uvm_error(get_type_name(),
      $sformatf("Channel bendwith wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.ch_bw, common_part.ch_bw))

    // check preamble type
    if (ph.preamble_type != common_part.preamble_type)
      `uvm_error(get_type_name(),
      $sformatf("Preamble type wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.preamble_type, common_part.preamble_type))

    // check antenna set
    if (ph.antenna_set != common_part.antenna_set)
      `uvm_error(get_type_name(),
      $sformatf("Antenna set wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.antenna_set, common_part.antenna_set))

    // check RSSI legacy
    if (ph.rssi_legacy != common_part.rssi_legacy)
      `uvm_error(get_type_name(),
      $sformatf("RSSI legacy wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.rssi_legacy, common_part.rssi_legacy))

    // check legacy length
    if (ph.leg_length != common_part.leg_length)
      `uvm_error(get_type_name(),
      $sformatf("Legacy length wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.leg_length, common_part.leg_length))

    // check regacy rate
    if (ph.leg_rate != common_part.leg_rate)
      `uvm_error(get_type_name(),
      $sformatf("Legacy rate wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.leg_rate, common_part.leg_rate))

    // check RSSI
    if (ph.rssi != common_part.rssi)
      `uvm_error(get_type_name(),
      $sformatf("RSSI wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.rssi, common_part.rssi))
    //-------------------------------------------------------------------------

    if (ph.format_mod inside {NON_HT, NON_HT_DUP_OFDM}) begin

      // check dynamic bandwith
      if (ph.dyn_bw != nonht_rxv.dyn_bw_in_non_ht)
        `uvm_error(get_type_name(),
        $sformatf("Dynamic bendwith wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.dyn_bw, nonht_rxv.dyn_bw_in_non_ht))

      // check L-SIG
      if (ph.l_sig_valid != nonht_rxv.l_sig_valid)
        `uvm_error(get_type_name(),
        $sformatf("L-SIG wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.l_sig_valid, nonht_rxv.l_sig_valid))

      // check channel bendwidth in NON-HT
      if (ph.chbw_in_non_ht != nonht_rxv.ch_bw_in_non_ht)
          `uvm_error(get_type_name(),
          $sformatf("Channel bendwith in NON HT wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.chbw_in_non_ht, nonht_rxv.ch_bw_in_non_ht))

    end//NON-HT

    if (ph.format_mod inside {HT_MF, HT_GF}) begin

      // check sounding
      if (ph.sounding != ht_rxv.sounding)
        `uvm_error(get_type_name(),
        $sformatf("Sounding wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.sounding, ht_rxv.sounding))

      // check smoothing
      if (ph.smoothing != ht_rxv.smoothing)
        `uvm_error(get_type_name(),
        $sformatf("Smoothing wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.smoothing, ht_rxv.smoothing))

      // check GI type
      if (ph.gi_type[0] != ht_rxv.gi_type)
        `uvm_error(get_type_name(),
        $sformatf("Short GI wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.gi_type[0], ht_rxv.gi_type))

      // check aggregated
      if (ph.aggregated != ht_rxv.aggregation)
        `uvm_error(get_type_name(),
        $sformatf("Aggregation wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.aggregated, ht_rxv.aggregation))

      // check STBC
      if (ph.stbc != ht_rxv.stbc)
        `uvm_error(get_type_name(),
        $sformatf("STBC wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.stbc, ht_rxv.stbc))

      // check number of extension spatial streams
      if (ph.num_extn_ss != ht_rxv.num_ext_ss)
        `uvm_error(get_type_name(),
        $sformatf("Extension spetial streams wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.num_extn_ss, ht_rxv.num_ext_ss))

      // check L-SIG
      if (ph.l_sig_valid != ht_rxv.l_sig_valid)
        `uvm_error(get_type_name(),
        $sformatf("L-SIG wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.l_sig_valid, ht_rxv.l_sig_valid))

      // check MCS
      if (ph.user_header[0].mcs_f != ht_rxv.mcs)
        `uvm_error(get_type_name(),
        $sformatf("MCS wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header[0].mcs_f, ht_rxv.mcs))

      // check FEC coding
      if (ph.user_header[0].fec_coding_f != ht_rxv.fec_coding)
        `uvm_error(get_type_name(),
        $sformatf("FEC coding wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header[0].fec_coding_f, ht_rxv.fec_coding))

      // check HT length
      if (ph.user_header[0].ht_length_f != ht_rxv.ht_length)
        `uvm_error(get_type_name(),
        $sformatf("HT length wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header[0].ht_length_f,ht_rxv.ht_length))

    end// HT

    if (ph.format_mod == VHT) begin

      // check sounding
      if (ph.sounding != vht_rxv.sounding)
        `uvm_error(get_type_name(),
        $sformatf("Sounding wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.sounding, vht_rxv.sounding))

      // check beamformed
      if (ph.beamformed != vht_rxv.beamformed)
        `uvm_error(get_type_name(),
        $sformatf("Beamformed wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.beamformed, vht_rxv.beamformed))

      // check GI type
      if (ph.gi_type[0] != vht_rxv.gi_type)
        `uvm_error(get_type_name(),
        $sformatf("Short GI wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.gi_type[0], vht_rxv.gi_type))

      // check STBC
      if (ph.stbc != vht_rxv.stbc)
        `uvm_error(get_type_name(),
        $sformatf("STBC wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.stbc, vht_rxv.stbc))

      // check doze not allowed
      if (ph.doze_not_allowed != vht_rxv.doze_not_allowed)
        `uvm_error(get_type_name(),
        $sformatf("Doze not allowed wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.doze_not_allowed, vht_rxv.doze_not_allowed))

      // check first user
      if (ph.first_user != vht_rxv.first_user)
        `uvm_error(get_type_name(),
        $sformatf("First user wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.first_user, vht_rxv.first_user))

      // check partial AID
      if (ph.partial_aid != vht_rxv.partial_aid)
        `uvm_error(get_type_name(),
        $sformatf("Partial AID wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.partial_aid, vht_rxv.partial_aid))

      // check group ID
      if (ph.group_id != vht_rxv.group_id)
        `uvm_error(get_type_name(),
        $sformatf("Group ID wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.group_id, vht_rxv.group_id))

      // check MCS
      if (ph.user_header[0].mcs_f != vht_rxv.mcs)
        `uvm_error(get_type_name(),
        $sformatf("MCS wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header[0].mcs_f, vht_rxv.mcs))

      // check FEC coding
      if (ph.user_header[0].fec_coding_f != vht_rxv.fec_coding)
        `uvm_error(get_type_name(),
        $sformatf("FEC coding wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header[0].fec_coding_f, vht_rxv.fec_coding))

      // check HT length
      if (ph.user_header[0].ht_length_f != vht_rxv.ht_length)
        `uvm_error(get_type_name(),
        $sformatf("HT length wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header[0].ht_length_f,vht_rxv.ht_length))

    end// VHT

    //-------------------------------------------------------------------------
    // CHECK HE COMMON PART FOR RXVECTOR
    //-------------------------------------------------------------------------
    if (ph.format_mod inside {HE_SU, HE_EXT_SU, HE_MU, HE_TB}) begin

      // check sounding
      if (ph.sounding != common_he_part.sounding)
        `uvm_error(get_type_name(),
        $sformatf("Sounding wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.sounding, common_he_part.sounding))

      // check beamformed
      if (ph.beamformed != common_he_part.beamformed)
        `uvm_error(get_type_name(),
        $sformatf("Beamformed wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.beamformed, common_he_part.beamformed))

      // check GI type
      if (ph.gi_type != common_he_part.gi_type)
        `uvm_error(get_type_name(),
        $sformatf("Short GI wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.gi_type, common_he_part.gi_type))

      // check STBC
      if (ph.stbc != common_he_part.stbc)
        `uvm_error(get_type_name(),
        $sformatf("STBC wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.stbc, common_he_part.stbc))

      // check uplink flag
      if (ph.uplink_flag != common_he_part.uplink_flag)
        `uvm_error(get_type_name(),
        $sformatf("Uplink flag wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.uplink_flag, common_he_part.uplink_flag))

      // check beam change
      if (ph.beam_change != common_he_part.beam_change)
        `uvm_error(get_type_name(),
        $sformatf("Beam change wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.beam_change, common_he_part.beam_change))

      // check DCM
      if (ph.dcm != common_he_part.dcm)
        `uvm_error(get_type_name(),
        $sformatf("DCM wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.dcm, common_he_part.dcm))

      // check HE LTF type
      if (ph.he_ltf_type != common_he_part.he_ltf_type)
        `uvm_error(get_type_name(),
        $sformatf("HE LTF type wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.he_ltf_type, common_he_part.he_ltf_type))

      // check doppler
      if (ph.doppler != common_he_part.doppler)
        `uvm_error(get_type_name(),
        $sformatf("Doppler wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.doppler, common_he_part.doppler))

      // check BSS color
      if (ph.bss_color != common_he_part.bss_color)
        `uvm_error(get_type_name(),
        $sformatf("BSS color wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.bss_color, common_he_part.bss_color))

      // check TXOP duration
      if (ph.txop_duration != common_he_part.txop_duration)
        `uvm_error(get_type_name(),
        $sformatf("TXOP duration wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.txop_duration, common_he_part.txop_duration))

      // check PE duration
      /*if (ph.pe_duration != common_he_part.pe_duration)
        `uvm_error(get_type_name(),
        $sformatf("PE duration wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.pe_duration, common_he_part.pe_duration))*/

      // check spatial reuse
      /*if (ph.spatial_reuse[0] != common_he_part.spatial_reuse)
        `uvm_error(get_type_name(),
        $sformatf("Spetial reuse wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.spatial_reuse[0], common_he_part.spatial_reuse))*/

    end// HE common part of rxVector
    //--------------------------------------------------------------------------

    if (ph.format_mod == HE_SU) begin
      // check spatial reuse
      if (ph.spatial_reuse[0] != he_su_rxv.spatial_reuse)
        `uvm_error(get_type_name(),
        $sformatf("Spetial reuse wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.spatial_reuse[0], he_su_rxv.spatial_reuse))

      // check PE duration
      if (ph.pe_duration != he_su_rxv.pe_duration)
        `uvm_error(get_type_name(),
        $sformatf("PE duration wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.pe_duration, he_su_rxv.pe_duration))

      // check MCS
      if (ph.user_header_he[0].mcs_f != he_su_rxv.mcs)
        `uvm_error(get_type_name(),
        $sformatf("MCS wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header_he[0].mcs_f, he_su_rxv.mcs))

      // check NSS
      if (ph.user_header_he[0].nss_f != he_su_rxv.nss)
        `uvm_error(get_type_name(),
        $sformatf("NSS wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header_he[0].nss_f, he_su_rxv.nss))

      // check FEC coding
      if (ph.user_header_he[0].fec_coding_f != he_su_rxv.fec_coding)
        `uvm_error(get_type_name(),
        $sformatf("FEC coding wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header_he[0].fec_coding_f, he_su_rxv.fec_coding))

      // check HE length
      if (ph.user_header_he[0].he_length_f != he_su_rxv.he_length)
        `uvm_error(get_type_name(),
        $sformatf("HE length wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header_he[0].he_length_f, he_su_rxv.he_length))
    end// HE-SU

    if (ph.format_mod == HE_MU) begin
      // check spatial reuse
      if (ph.spatial_reuse[0] != he_mu_rxv.spatial_reuse)
        `uvm_error(get_type_name(),
        $sformatf("Spetial reuse wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.spatial_reuse[0], he_mu_rxv.spatial_reuse))

      // check PE duration
      if (ph.pe_duration != he_mu_rxv.pe_duration)
        `uvm_error(get_type_name(),
        $sformatf("PE duration wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.pe_duration, he_mu_rxv.pe_duration))

      // check SIG B compression mode
      if (ph.sig_b_comp_mode != he_mu_rxv.sig_b_comp_mode)
        `uvm_error(get_type_name(),
        $sformatf("SIG B compression mode wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.sig_b_comp_mode, he_mu_rxv.sig_b_comp_mode))

      // check DCM SIG B
      if (ph.dcm_sig_b != he_mu_rxv.dcm_sig_b)
        `uvm_error(get_type_name(),
        $sformatf("DCM SIG B wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.dcm_sig_b, he_mu_rxv.dcm_sig_b))

      // check MCS SIG B
      if (ph.mcs_sig_b != he_mu_rxv.mcs_sig_b)
        `uvm_error(get_type_name(),
        $sformatf("DCM SIG B wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.mcs_sig_b, he_mu_rxv.mcs_sig_b))

      // check MCS
      if (ph.user_header_he[0].mcs_f != he_mu_rxv.mcs)
        `uvm_error(get_type_name(),
        $sformatf("MCS wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header_he[0].mcs_f, he_mu_rxv.mcs))

      // check NSS
      if (ph.user_header_he[0].nss_f != he_mu_rxv.nss)
        `uvm_error(get_type_name(),
        $sformatf("NSS wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header_he[0].nss_f, he_mu_rxv.nss))

      // check FEC coding
      if (ph.user_header_he[0].fec_coding_f != he_mu_rxv.fec_coding)
        `uvm_error(get_type_name(),
        $sformatf("FEC coding wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header_he[0].fec_coding_f, he_mu_rxv.fec_coding))

      // check HE length
      if (ph.user_header_he[0].he_length_f != he_mu_rxv.he_length)
        `uvm_error(get_type_name(),
        $sformatf("HE length wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header_he[0].he_length_f, he_mu_rxv.he_length))
    end// HE-MU

    if (ph.format_mod == HE_TB) begin
      // check N user
      if (ph.n_user != he_tb_rxv.n_user)
        `uvm_error(get_type_name(),
        $sformatf("N user wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.n_user, he_tb_rxv.n_user))

      for (int i=1; i<=ph.n_user; i++) begin
        // check MCS
        if (ph.user_header_he[0].mcs_f != he_tb_rxv.user1.mcs)
          `uvm_error(get_type_name(),
          $sformatf("MCS wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header_he[0].mcs_f, he_tb_rxv.user1.mcs))

        // check NSS
        if (ph.user_header_he[0].nss_f != he_tb_rxv.user1.nss)
          `uvm_error(get_type_name(),
          $sformatf("NSS wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header_he[0].nss_f, he_tb_rxv.user1.nss))

        // check FEC coding
        if (ph.user_header_he[0].fec_coding_f != he_tb_rxv.user1.fec_coding)
          `uvm_error(get_type_name(),
          $sformatf("FEC coding wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header_he[0].fec_coding_f, he_tb_rxv.user1.fec_coding))

        // check HE length
        if (ph.user_header_he[0].he_length_f != he_tb_rxv.user1.he_length)
          `uvm_error(get_type_name(),
          $sformatf("HE length wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header_he[0].he_length_f, he_tb_rxv.user1.he_length))

        // check STAID
        if (ph.user_header_he[0].staid_f != he_tb_rxv.user1.staid)
          `uvm_error(get_type_name(),
          $sformatf("HE length wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.user_header_he[0].staid_f, he_tb_rxv.user1.staid))
      end//for loop
    end// HE-TRIG

    //--------------------------------------------------------------------------
    // RxVector2 checking
    //--------------------------------------------------------------------------
    // check RCPI 1
    if (ph.rcpi[0] != rxv2.rcpi1)
      `uvm_error(get_type_name(),
      $sformatf("RCPI 1 wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.rcpi[0], rxv2.rcpi1))

    // check RCPI 2
    if (ph.rcpi[1] != rxv2.rcpi2)
      `uvm_error(get_type_name(),
      $sformatf("RCPI 2 wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.rcpi[1], rxv2.rcpi2))

    // check RCPI 3
    if (ph.rcpi[2] != rxv2.rcpi3)
      `uvm_error(get_type_name(),
      $sformatf("RCPI 3 wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.rcpi[2], rxv2.rcpi3))

    // check RCPI 4
    if (ph.rcpi[3] != rxv2.rcpi4)
      `uvm_error(get_type_name(),
      $sformatf("RCPI 4 wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.rcpi[3], rxv2.rcpi4))

    // check EVM 1
    if (ph.evm[0] != rxv2.evm1)
      `uvm_error(get_type_name(),
      $sformatf("EVM 1 wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.evm[0], rxv2.evm1))

    // check EVM 2
    if (ph.evm[1] != rxv2.evm2)
      `uvm_error(get_type_name(),
      $sformatf("EVM 2 wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.evm[1], rxv2.evm2))

    // check EVM 3
    if (ph.evm[2] != rxv2.evm3)
      `uvm_error(get_type_name(),
      $sformatf("EVM 3 wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.evm[2], rxv2.evm3))

    // check EVM 4
    if (ph.evm[3] != rxv2.evm4)
      `uvm_error(get_type_name(),
      $sformatf("EVM 4 wrong! MPIF - 0x%0h, SRAM - 0x%0h",ph.evm[3], rxv2.evm4))

    `uvm_info(get_type_name(), $sformatf("Received Rx vector is OK!"), UVM_LOW)
  endfunction : check_rx_vector

  //---------------------------------------------------------------------------
  // check timestamp for frame ended on air (made as task because of register
  // backdoor read)
  //---------------------------------------------------------------------------
  function void mac_rx_scoreboard::check_tsf(RHD rhd);
    int                  frmDurationWOPreamb;
    timestamp_t          reg_value, rhd_value;
    timestamp_t          tsf_updated_value;
    timestamp_t          tsf_offset;
    BEACON_frame         beacon;
    PROBE_RESPONSE_frame probe_resp;
    PPDU_frame           ppdu_item;
    HESIGB_s             HESIGB;
    bit                  timestamp_check;

    reg_value = tsf_in_rhd;
    rhd_value = {rhd.tsf_hi, rhd.tsf_lo};

    ppdu_item = mac_phy_item.frame;

    // determine will there be timestamp check
    timestamp_check = (mac_phy_item.frame.ampdu_frame[0].mpdu_frame_type[0] inside {BEACON,PROBE_RESPONSE}) ? 1'b1 : 1'b0;

    // Calculate Frame Duration without preamble
    if (ppdu_item.ppdu_format inside {HE_SU, HE_EXT_SU, HE_TB, HE_MU} && timestamp_check) begin
      frmDurationWOPreamb = $ceil(ComputeTimeOnAirAX  ( .length_i          (ppdu_item.preamble_header.user_header_he[0].he_length_f),
                                                        .preType_i         (ppdu_item.preamble_header.format_mod),
                                                        .mcsIndex_i        (ppdu_item.preamble_header.user_header_he[0].mcs_f),
                                                        .giType_i          (ppdu_item.preamble_header.gi_type),
                                                        .ruType_i          (
                                                          (ppdu_item.ppdu_format inside {HE_SU,HE_EXT_SU}) ?
                                                            get_ru_type_he_su(ppdu_item.preamble_header.ch_bw) :
                                                          (ppdu_item.ppdu_format == HE_MU) ?
                                                            ppdu_item.preamble_header.ru_type :
                                                            // HE_TB
                                                            get_ru_type(ppdu_item.preamble_header.ru_allocation)

                                                        ),//ruType_i
                                                        .heLtfType_i       (ppdu_item.preamble_header.he_ltf_type),
                                                        .numHeLtf_i        (ppdu_item.preamble_header.num_he_ltf),
                                                        .dcm_i             (ppdu_item.preamble_header.dcm),
                                                        .packetExtension_i (ppdu_item.preamble_header.user_header_he[0].pkt_ext_f),
                                                        .heTbLength_i      (0),
                                                        .triggerMethod_i   (0),
                                                        .doppler_i         (ppdu_item.preamble_header.doppler),
                                                        .mma_i             (ppdu_item.preamble_header.midamble_periodicity),
                                                        .stbc_i            (ppdu_item.preamble_header.stbc),
                                                        .woPreamble_i      (1),
                                                        .heSigB_i          (HESIGB),
                                                        .debug_i           (0)
                                                       ));
    end
    else if (timestamp_check) begin
      frmDurationWOPreamb = $ceil(ComputeTimeOnAirAC( .length_i     (ppdu_item.preamble_header.leg_length - ppdu_item.ampdu_frame[0].mpdu_frame[0].MAC_header.size()),
                                                      .preType_i    (ppdu_item.preamble_header.format_mod),
                                                      .mcsIndex_i   ((ppdu_item.preamble_header.format_mod inside {NON_HT, NON_HT_DUP_OFDM}) ?
                                                                      get_mcs_from_legacy_rate(ppdu_item.preamble_header.leg_rate)
                                                                      : ppdu_item.preamble_header.user_header[0].mcs_f),
                                                      .shortGI_i    (ppdu_item.preamble_header.gi_type[0]),
                                                      .chBW_i       (ppdu_item.preamble_header.ch_bw),
                                                      .stbc_i       (ppdu_item.preamble_header.stbc),
                                                      .extNSS_i     (ppdu_item.preamble_header.num_extn_ss),
                                                      .woPreamble_i (1),
                                                      .band5G_i     (get_band5G(mac_ctrl_1.abgn_mode_f)),
                                                      .multiUser_i  (0),
                                                      .nSTSTotal_i  (ppdu_item.preamble_header.user_header[0].num_sts_f),
                                                      .debug_i      (0)
                                                      ));
    end

    if (!(rhd_value <= reg_value+rx_end_delay && rhd_value >= reg_value-rx_end_delay))
      `uvm_error(get_type_name(),
      $sformatf("TSF missmatch! reg value %0h - RHD value %0h",reg_value,rhd_value))
    else
      `uvm_info(get_type_name(), $sformatf("TSF check OK!"), UVM_LOW)

    fetch_register_values();
    reg_value = tsf_in_frame_timestamp;

    // check if BEACON or PROBE_RESPONSE  are received, is timestamp updated
    // correctly in TSF register
    if (   mac_ctrl_1.ap_f == 0 // MAC is STA
        && mac_phy_item.frame.ampdu_frame[0].mpdu_frame_type[0] inside {BEACON,PROBE_RESPONSE}) begin

      case (mac_phy_item.frame.ampdu_frame[0].mpdu_frame_type[0])
        BEACON: begin
          // typecast received MPDU frame to BEACON frame
          beacon = BEACON_frame'(mac_phy_item.frame.ampdu_frame[0].mpdu_frame[0]);
          tsf_offset = (beacon.timestamp + frmDurationWOPreamb) - previous_tsf_value;
          beacon.timestamp = previous_tsf_value + tsf_offset;
          if ((beacon.timestamp != reg_value) && !(reg_value <= beacon.timestamp+6 && reg_value >= beacon.timestamp-6))
            `uvm_error(get_type_name(),
            $sformatf("Timestamp not set as expected from Beacon frame, Timestamp (%0h) expct (%0h)",reg_value, beacon.timestamp))
        end
        PROBE_RESPONSE: begin
          // typecast received MPDU frame to PROBE_RESPONSE frame
          probe_resp = PROBE_RESPONSE_frame'(mac_phy_item.frame.ampdu_frame[0].mpdu_frame[0]);
          tsf_offset = (probe_resp.timestamp + frmDurationWOPreamb) - previous_tsf_value;
          probe_resp.timestamp = previous_tsf_value + tsf_offset;
          if ((probe_resp.timestamp != reg_value) && !(reg_value <= probe_resp.timestamp+6 && reg_value >= probe_resp.timestamp-6))
            `uvm_error(get_type_name(),
            $sformatf("Timestamp not set as expected from Probe Response, Timestamp (%0h) expct (%0h), ",reg_value, probe_resp.timestamp))

        end
      endcase
    end
  endfunction : check_tsf

  //---------------------------------------------------------------------------
  // check MPDU status information in RHD
  //---------------------------------------------------------------------------
  function void mac_rx_scoreboard::check_mpdu_status_info(mpdu_statu_info_s mpdu_statu_info, MPDU_frame mpdu);
    bit multicast_broadcast = 0; // indication that frame has multicast or broadcast address
    bit addr_mismatch = 1;
    bit key_valid = 0;
    int key_index = 0;
    CONTROL_WRAPPER_frame   ctrl_wrap;
    bit [3:0]               carried_frame_subtype;

    fetch_register_values();

    // in case of control wrapper frame carried frame is stored in SRAM
    if (   mpdu.MAC_header.frame_ctrl.type_f == `CONTROL_MPDU
        && mpdu.MAC_header.frame_ctrl.subtype_f == `CONTROL_WRAPPER) begin
      ctrl_wrap = CONTROL_WRAPPER_frame'(mpdu);
      carried_frame_subtype = ctrl_wrap.carried_frame.MAC_header.frame_ctrl.subtype_f;

      // check type and subtype with received MPDU frame
      if (mpdu_statu_info.subtype_f != carried_frame_subtype)
        `uvm_error(get_type_name(),
        $sformatf("mpdu_statu_info.subtype_f missmatch! MDPDU (carried frame) %0h - RHD value %0h",
        carried_frame_subtype, mpdu_statu_info.subtype_f))
    end
    else begin
      // check type and subtype with received MPDU frame
      if (mpdu_statu_info.subtype_f != mpdu.MAC_header.frame_ctrl.subtype_f)
        `uvm_error(get_type_name(),
        $sformatf("mpdu_statu_info.subtype_f missmatch! MDPDU %0h - RHD value %0h",
        mpdu.MAC_header.frame_ctrl.subtype_f, mpdu_statu_info.subtype_f))
    end

    if (mpdu_statu_info.type_f != mpdu.MAC_header.frame_ctrl.type_f)
      `uvm_error(get_type_name(),
      $sformatf("mpdu_statu_info.type_f missmatch! MDPDU %0h - RHD value %0h",
      mpdu.MAC_header.frame_ctrl.type_f, mpdu_statu_info.type_f))

    if (mpdu_statu_info.descr_done_f != 1'b1)
      `uvm_error(get_type_name(), $sformatf("Descriptor done not set!"))

    if (mpdu_statu_info.frm_successful_f != 1'b1)
      `uvm_error(get_type_name(), $sformatf("Frame not successful!"))

    // go throught MPDU addresses and look for broadcast or multicast
    if (mpdu.get_MAC_addr(.RA(1)) inside {`BROADCAST_ADDR,`MULTICAST_ADDR})
      multicast_broadcast = 1'b1;
    else
      multicast_broadcast = 1'b0;

    // look does receiving station address match MAC device address
    // with masking applied
    if (!is_MAC_receiver(mac_phy_item.frame))
      addr_mismatch = 1'b1;
    else
      addr_mismatch = 1'b0;

    if (mpdu_statu_info.ga_frm_f != multicast_broadcast) begin
      if (multicast_broadcast)
        `uvm_error(get_type_name(), $sformatf("Group addressed frame should be set!"))
      else
        `uvm_error(get_type_name(), $sformatf("Group addressed frame should NOT be set!"))
    end

    if (mpdu_statu_info.addr_mismatch_f != addr_mismatch) begin
      if (addr_mismatch)
        `uvm_error(get_type_name(),
        $sformatf("MAC addresses in MPDU don't match with device address!"))
      else
        `uvm_error(get_type_name(),
        $sformatf("MAC address in MPDU has match with device address, but mismatch is set in status!"))
    end

    if (mpdu_statu_info.fcs_err_f != mpdu.fcs_error && rx_ctrl.accept_error_frames_f)
      `uvm_error(get_type_name(),
      $sformatf("There is FCS error but status is not set! Status (%0h)/Frame FCS (%0h)",
      mpdu_statu_info.fcs_err_f, mpdu.fcs_error))

    // check in key storage RAM does it have entry for TA
    if (mpdu.MAC_header.addr.size() >= 2) begin
      for (int i=`KEYRAM_DEV_START_ADDR; i<=`KEYRAM_DEV_END_ADDR; i++) begin
        if (m_ksr.keyRAM[i].mac_addr_ram_f == mpdu.MAC_header.addr[1]) begin
          key_valid = 1'b1;
          key_index = i;
        end
      end

      // check status of key RAM enry
      if (key_valid != mpdu_statu_info.key_sram_valid_idx_f)
        `uvm_error(get_type_name(), $sformatf("KeyRAM valid should be set!"))

      if (key_index != mpdu_statu_info.key_sram_idx_f)
        `uvm_error(get_type_name(),
        $sformatf("KeyRAM index wrong! Entry in KSR %0d - in status %0d",
        mpdu_statu_info.key_sram_idx_f,key_index))
    end

    if (!mpdu.MAC_header.frame_ctrl.protected_frame_f) begin
      if (mpdu_statu_info.decry_type_f != 4'b0000)
        `uvm_error(get_type_name(),
        $sformatf("Frame should not be decrypted since there is no protection!"))
    end
    else begin
      case (mpdu.security_wrap.security_type)
        NULL_KEY: begin
          if (mpdu_statu_info.decry_type_f != 4'b1111)
            `uvm_error(get_type_name(), $sformatf("MAC HW should find NULL KEY!"))
        end
        WEP: begin
          if (mpdu_statu_info.decry_type_f != 4'b0001)
            `uvm_error(get_type_name(), $sformatf("MAC HW should decrypt WEP protection!"))
        end
        TKIP: begin
          if (mpdu_statu_info.decry_type_f != 4'b0010)
            `uvm_error(get_type_name(), $sformatf("MAC HW should decrypt TKIP protection!"))
        end
        CCMP: begin
          if (mpdu_statu_info.decry_type_f != 4'b0011 && mpdu.security_wrap.ccmp_obj.Key.size() == 16)
            `uvm_error(get_type_name(), $sformatf("MAC HW should decrypt CCMP-128 protection!"))
          else if (mpdu_statu_info.decry_type_f != 4'b0100 && mpdu.security_wrap.ccmp_obj.Key.size() == 32)
            `uvm_error(get_type_name(), $sformatf("MAC HW should decrypt CCMP-256 protection!"))
        end
        GCMP: begin
          if (mpdu_statu_info.decry_type_f != 4'b0101 && mpdu.security_wrap.gcmp_obj.Key.size() == 16)
            `uvm_error(get_type_name(), $sformatf("MAC HW should decrypt GCMP-128 protection!"))
          else if (mpdu_statu_info.decry_type_f != 4'b0110 && mpdu.security_wrap.gcmp_obj.Key.size() == 32)
            `uvm_error(get_type_name(), $sformatf("MAC HW should decrypt GCMP-256 protection!"))
        end
        WAPI: begin
          if (mpdu_statu_info.decry_type_f != 4'b0111)
            `uvm_error(get_type_name(), $sformatf("MAC HW should decrypt WAPI protection!"))
        end
        default: `uvm_error(get_type_name(), "Unknown security type decryption!!!")
      endcase

      if (mpdu_statu_info.decry_err_f == 1'b1)
        `uvm_warning(get_type_name(), $sformatf("Decryption failure (MIC failure, ICV err,...)"))
    end

    `uvm_info(get_type_name(), $sformatf("MPDU status information is OK!"), UVM_LOW)
  endfunction : check_mpdu_status_info

  //---------------------------------------------------------------------------
  // compare two frames from MAC-PHY if and one stored in SRAM
  //---------------------------------------------------------------------------
  function void mac_rx_scoreboard::compare_2_mpdu_frames(MPDU_frame mpdu, RHD rhd, RPD rpd[]);
    int         rpd_cnt;
    int         byte_cnt;
    int         addr_cnt;
    bit [47:0]  tmp;
    int         size;
    int         payload_len;
    int         security_header_cnt;
    bit         mh_inside_rhd; // MAC header inside RHD or not
    bit [7:0]   mac_header[];
    CONTROL_WRAPPER_frame ctrl_wrap;

    // check in register is MAC header going to be stored with payload or in RHD
    mh_inside_rhd = (mpdu.MAC_header.frame_ctrl.type_f == `CONTROL_MPDU    && rx_ctrl2.ctrlMHStoredwithPld_f == 1'b0) ? 1'b1 :
                    (mpdu.MAC_header.frame_ctrl.type_f == `MANAGEMENT_MPDU && rx_ctrl2.mgtMHStoredwithPld_f  == 1'b0) ? 1'b1 :
                    (mpdu.MAC_header.frame_ctrl.type_f == `DATA_MPDU       && rx_ctrl2.dataMHStoredwithPld_f == 1'b0) ? 1'b1 :
                                                                                                                        1'b0;

    if (mh_inside_rhd) begin
      mac_header = new[rhd.mac_header.size()](rhd.mac_header);
    end
    else begin
      mac_header = new[rpd[0].payload.size()](rpd[0].payload);
      // if the MAC header is stored, part in RPD[0] part in RPD[1]
      if (rpd.size() > 1) begin
        mac_header = new[mac_header.size()+rpd[1].payload.size()](mac_header);
        for (int i = rpd[0].payload.size(); i < mac_header.size();i++) begin
          mac_header[i] = rpd[1].payload[i-rpd[0].payload.size()];
        end
      end
    end

    // compare size of received frame and one stored in memory,
    size = mpdu.size();
    payload_len = rhd.payload_len;

    if (size != payload_len)
      `uvm_error(get_type_name(),
      $sformatf("MPDU size (%0d) and payload size (%0d) mismatch!",size,payload_len))

    rpd_cnt = 0;
    byte_cnt = 0;
    //----------------------------------------------------------------------------
    // compare MAC header content
    //----------------------------------------------------------------------------
    tmp[15:0] = {mac_header[byte_cnt+1],mac_header[byte_cnt]};
    byte_cnt += 2;
    if (mpdu.MAC_header.frame_ctrl  != tmp[15:0])
      `uvm_error(get_type_name(),
      $sformatf("Frame control wrong: MPDU (%0h) - RHD (%0h))",mpdu.MAC_header.frame_ctrl,tmp[15:0]))

    tmp[15:0] = {mac_header[byte_cnt+1],mac_header[byte_cnt]};
    byte_cnt += 2;
    if (mpdu.MAC_header.duration_id != tmp[15:0])
      `uvm_error(get_type_name(),
      $sformatf("Frame duration wrong: MPDU (%0h) - RHD (%0h))",mpdu.MAC_header.duration_id,tmp[15:0]))

    // check MAC addresses before sequence control
    addr_cnt = 0;
    repeat (`MAC_ADDR_SIZE(mpdu.MAC_header.addr.size())) begin
      tmp = {mac_header[byte_cnt+5],
             mac_header[byte_cnt+4],
             mac_header[byte_cnt+3],
             mac_header[byte_cnt+2],
             mac_header[byte_cnt+1],
             mac_header[byte_cnt] };

      if (mpdu.MAC_header.addr[addr_cnt] != tmp)
        `uvm_error(get_type_name(),
        $sformatf("MAC address[%0d] wrong: MPDU (%0h) - RHD (%0h))",addr_cnt,mpdu.MAC_header.addr[addr_cnt],tmp))

      addr_cnt++;
      byte_cnt += 6;
    end//repeat

    // check sequnce control
    if (mpdu.MAC_header.seq_ctrl.size() == 1) begin
      tmp[15:0] = {mac_header[byte_cnt+1],mac_header[byte_cnt]};
      byte_cnt += 2;
      if (mpdu.MAC_header.seq_ctrl[0] != tmp[15:0])
        `uvm_error(get_type_name(),
        $sformatf("Sequence control wrong: MPDU (%0h) - RHD (%0h))",mpdu.MAC_header.seq_ctrl[0],tmp[15:0]))
    end

    // check 4th MAC address
    if (mpdu.MAC_header.addr.size() == 4) begin
      tmp = {mac_header[byte_cnt+5],
             mac_header[byte_cnt+4],
             mac_header[byte_cnt+3],
             mac_header[byte_cnt+2],
             mac_header[byte_cnt+1],
             mac_header[byte_cnt] };

      if (mpdu.MAC_header.addr[addr_cnt] != tmp)
        `uvm_error(get_type_name(),
        $sformatf("MAC address[%0d] wrong: MPDU (%0h) - RHD (%0h))",addr_cnt,mpdu.MAC_header.addr[addr_cnt],tmp))

      addr_cnt++;
      byte_cnt += 6;
    end

    // check QoS control
    if (mpdu.MAC_header.qos_ctrl.size() == 1) begin
      tmp[15:0] = {mac_header[byte_cnt+1],mac_header[byte_cnt]};
      byte_cnt += 2;
      if (mpdu.MAC_header.qos_ctrl[0] != tmp[15:0])
        `uvm_error(get_type_name(),
        $sformatf("QoS control wrong: MPDU (%0h) - RHD (%0h))",mpdu.MAC_header.qos_ctrl[0],tmp[15:0]))
    end

    // check HT control
    if (mpdu.MAC_header.ht_ctrl.size() == 1) begin
      tmp[31:0] = {mac_header[byte_cnt+3],mac_header[byte_cnt+2],
                   mac_header[byte_cnt+1],mac_header[byte_cnt]};
      byte_cnt += 4;
      if (mpdu.MAC_header.ht_ctrl[0] != tmp[31:0])
        `uvm_error(get_type_name(),
        $sformatf("HT control wrong: MPDU (%0h) - RHD (%0h))",mpdu.MAC_header.ht_ctrl[0],tmp[31:0]))
    end

    //----------------------------------------------------------------------------
    // compare security header bytes which are now linked with MAC
    // header bytes
    //----------------------------------------------------------------------------
    security_header_cnt = 0;
    if (mpdu.is_protected()) begin
      for (int i=0; i < security_header_bytes(mpdu); i++) begin
        if (mpdu.frame_body[i] != mac_header[byte_cnt])
          `uvm_error(get_type_name(),
          $sformatf("Security header byte (%0d) mismatch, MPDU (%0h) - RHD (%0h)!",i,mpdu.frame_body[i],mac_header[byte_cnt]))
        security_header_cnt++;
        byte_cnt++;
      end//for
    end//is_protected
    else if ($cast(ctrl_wrap, mpdu)) begin
      // subtract 10 bytes of MAC header size for frame control, duration,
      // address 1, because they are not stored in control wrapper
      for (int i=0; i < ctrl_wrap.carried_frame.MAC_header.size()-10; i++) begin
        if (mpdu.frame_body[i] != mac_header[byte_cnt])
          `uvm_error(get_type_name(),
          $sformatf("Security header byte (%0d) mismatch, MPDU (%0h) - RHD (%0h)!",i,mpdu.frame_body[i],mac_header[byte_cnt]))
        security_header_cnt++;
        byte_cnt++;
      end//for
    end//CONTROL WRAPPER frame

    //----------------------------------------------------------------------------
    // compare frame body and FCS
    //----------------------------------------------------------------------------
    if (mh_inside_rhd) byte_cnt = 0;

    if (mpdu.frame_body.size() != 0) begin
      for (int i=security_header_cnt; i < mpdu.frame_body.size(); i++) begin
        // when byte count has reached maximum size of RPD payload, reset
        // counter and change to next RPD in chained list
        if ( byte_cnt >= rpd[rpd_cnt].payload.size()) begin
          if (byte_cnt > rpd[rpd_cnt].payload.size()) begin
            byte_cnt = byte_cnt - rpd[rpd_cnt].payload.size();
            rpd_cnt++;
          end
          else begin
            byte_cnt = 0;
            rpd_cnt++;
          end
        end

        if (mpdu.frame_body[i] != rpd[rpd_cnt].payload[byte_cnt])
          `uvm_error(get_type_name(),
          $sformatf("Frame body mismatch: MPDU.frame_body[%0d]-(%0h) - RPD[%0d][%0d]-(%0h))",
          i,mpdu.frame_body[i],rpd_cnt,byte_cnt,rpd[rpd_cnt].payload[byte_cnt]))

        byte_cnt++;
        if (i == mpdu.frame_body.size()-1-security_footer_bytes(mpdu))
          break; // don't compare security footer bytes
      end// foreach
    end// frame_body != 0

    // when payload reminder is on limits of buffer size, rest of content is
    // stored in next RPD
    if (byte_cnt >= rpd[rpd_cnt].payload.size()) begin
      byte_cnt = byte_cnt - rpd[rpd_cnt].payload.size();
      rpd_cnt++;
    end

    tmp[31:0] = {rpd[rpd_cnt].payload[byte_cnt+3],rpd[rpd_cnt].payload[byte_cnt+2],
                 rpd[rpd_cnt].payload[byte_cnt+1],rpd[rpd_cnt].payload[byte_cnt]};

    if (mpdu.security_wrap == null || mpdu.security_wrap.security_type inside {NO_SECURITY,NULL_KEY}) begin
      if (mpdu.FCS != tmp[31:0])
        `uvm_error(get_type_name(),$sformatf("FCS wrong: MPDU (%0h) - RPD (%0h))",mpdu.FCS,tmp[31:0]))
        //`uvm_warning(get_type_name(),$sformatf("FCS wrong: MPDU (%0h) - RPD (%0h))",mpdu.FCS,tmp[31:0]))
    end
    //----------------------------------------------------------------------------

    `uvm_info(get_type_name(), $sformatf("Comparing 2 MPDU frames is OK!"), UVM_LOW)
  endfunction : compare_2_mpdu_frames

  //----------------------------------------------------------------------------
  // Task that imitates the NAV counter inside the MAC
  //----------------------------------------------------------------------------
  task mac_rx_scoreboard::nav_counter();
    // Indicate that the channel is busy,
    // as set by the Virtual CS
    forever begin
      wait(nav_flag  == 1'b1);
      nav_flag = 1'b0;
      m_cfg.channel_busy = 1'b1;
      `uvm_info(get_type_name(), $sformatf("The channel is set to busy state"), UVM_HIGH)

      while (basic_NAV > 0 || intra_NAV > 0) begin
        #1us;

        if (basic_NAV)
          basic_NAV--;

        if (intra_NAV)
          intra_NAV--;

        if (timeout_timer > 0 && start_timeout_cnt) begin
          timeout_timer --;
        end
        else if (start_timeout_cnt && timeout_timer == 0)begin
          basic_NAV = 0;
          intra_NAV = 0;
          start_timeout_cnt = 0;
        end

      end

      // As the counter reached 0,
      // release the channel
      `uvm_info(get_type_name(), $sformatf("The channel is set to idle state"), UVM_HIGH)
      m_cfg.channel_busy = 1'b0;
    end
  endtask : nav_counter

  //----------------------------------------------------------------------------
  // wait for unmasked event to occur in register field, else generate error
  //----------------------------------------------------------------------------
  task mac_rx_scoreboard::check_rx_events();
    int err = 0;

    fetch_register_values();
    // fetch the TSF register value, for offset calculation
    fetch_tsf_reg(previous_tsf_value);

    fork : RX_EVENT_THREADS
      // rxBuffer1Trigger or timerRxTrigger or rxBuffer2Trigger or
      // tbProtTrigger
      if (txrx_int_unmask[18:16] != 3'b000 || txrx_int_unmask[4] != 1'b0) begin
        while (   txrx_int_event[16] != 1'b1
               && txrx_int_event[17] != 1'b1
               && txrx_int_event[18] != 1'b1
               && txrx_int_event[4]  != 1'b1) begin

          txrx_int_event = m_regmodel.get_mirrored_reg_value("TXRXINTEVENTSETREG");
          #1us; // drain some time because of dead lock
        end
      end

      // watchdog timer to prevent infinite loop
      begin
        #(`RX_EVENT_TIMER);
        err = 1;
      end
    join_any

    disable fork;

    if (err) begin
      `uvm_error(get_type_name(),
      $sformatf("RX EVENT DIDN'T OCCUR! mask: 0x%0h event: 0x%0h",txrx_int_unmask,txrx_int_event))
    end
    else begin
      if (txrx_int_event[16])
        `uvm_info(get_type_name(), $sformatf("rxBuffer1Trigger EVENT OCCURED OK!"), UVM_LOW)
      if (txrx_int_event[17])
       `uvm_info(get_type_name(), $sformatf("timerRxTrigger EVENT OCCURED OK!"), UVM_LOW)
      if (txrx_int_event[18])
       `uvm_info(get_type_name(), $sformatf("rxBuffer2Trigger EVENT OCCURED OK!"), UVM_LOW)
      if (txrx_int_event[4])
       `uvm_info(get_type_name(), $sformatf("tbProtTrigger EVENT OCCURED OK!"), UVM_LOW)

      // check did the proper interrupt occur
      if (txrx_int_event[16] && is_priority_rx_frame(mac_phy_item.frame) && !rx_ctrl2.disableRxBuffer2_f)
        `uvm_error(get_type_name(), $sformatf("rxBuffer1Trigger for priority frame!"))
      if (txrx_int_event[18] && !is_priority_rx_frame(mac_phy_item.frame))
        `uvm_error(get_type_name(), $sformatf("rxBuffer2Trigger for non priority frame!"))

    end

    // fetch TSF value when frame is stored in SRAM
    #3us; // delay added to make sure TSF reg is updated
    fetch_tsf_reg(tsf_in_frame_timestamp);
  endtask : check_rx_events

  //----------------------------------------------------------------------------
  // determine how many bytes are in security header
  //----------------------------------------------------------------------------
  function int mac_rx_scoreboard::security_header_bytes(ref MPDU_frame mpdu);
    // if there si no protection added return 0 bytes
    if (mpdu.security_wrap == null) begin
      return 0;
    end
    else begin
      case (mpdu.security_wrap.security_type.name())
        "NULL_KEY": return 0;
        "WEP"     : return 4;
        "TKIP"    : return 8;
        "CCMP"    : return 8;
        "WAPI"    : return 18;
        "GCMP"    : return 8;
        default   : return 0;
      endcase
    end
  endfunction : security_header_bytes

  //----------------------------------------------------------------------------
  // determine how many bytes are in security footer
  //----------------------------------------------------------------------------
  function int mac_rx_scoreboard::security_footer_bytes(ref MPDU_frame mpdu);
    // if there si no protection added return 0 bytes
    if (mpdu.security_wrap == null) begin
      return 0;
    end
    else begin
      case (mpdu.security_wrap.security_type.name())
        "NULL_KEY": return 0;
        "WEP"     : return 4;
        "TKIP"    : return 12;
        "CCMP"    : return mpdu.security_wrap.ccmp_sec.MIC.size(); // 8 or 16 bytes
        "WAPI"    : return 16;
        "GCMP"    : return 16;
        default   : return 0;
      endcase
    end
  endfunction : security_footer_bytes

`endif //MAC_RX_SCOREBOARD_SV
