//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_SCOREBOARD_BASE_SV
`define MAC_SCOREBOARD_BASE_SV

class mac_scoreboard_base extends uvm_scoreboard;

  `uvm_component_utils(mac_scoreboard_base)

  wlan_config              m_cfg;       // handle WLAN env configuration
  KeyStorageRAM            m_ksr;       // handle to key storage RAM
  TOP_register_block       m_regmodel;  // handle to register model

  // events that are used in MAC core tests
  uvm_event_pool           mac_pool;
  uvm_event                no_resp_trigger;
  uvm_event                frame_filtered_trigger;

  // NAV timers
  int                     intra_NAV = 0;
  int                     basic_NAV = 0;
  int                     timeout_timer = 0;
  bit                     start_timeout_cnt = 0;
  int                     cts_time = 0;
  mpdu_frame_type_e       frm_type_base;
  CONTROL_WRAPPER_frame   ctrl_wrapp_base;

  //---------------------------------------------------------------------------
  // list of variables
  //---------------------------------------------------------------------------
  mac_address_t           bss_addr;
  mac_address_t           dev_addr;
  mac_address_t           dev_addr_mask;
  mac_address_t           txop_holder_addr;
  rx_ctrl_reg_s           rx_ctrl;
  mac_ctrl_1_reg_s        mac_ctrl_1;
  bit [31:0]              txrx_int_unmask;
  bit [31:0]              txrx_int_event;
  bit [31:0]              basic_rates;
  bit [31:0]              htmcs;
  bit [31:0]              vhtmcs;
  timings_1_s             timings_1;
  bit [31:0]              tsf_low, tsf_high;
  edca_ac_reg_s           edca_ac0;
  edca_ac_reg_s           edca_ac1;
  edca_ac_reg_s           edca_ac2;
  edca_ac_reg_s           edca_ac3;
  bit [31:0]              edca_has_data;
  bfmee_ctrl_reg_s        bfmee_ctrl;
  bit [31:0]              bcn_ctrl2;
  bit [31:0]              pta_config;
  bit [31:0]              he_config;
  const bit               mac_header_format = 1;
  bit [31:0]              bss_color;
  bit [31:0]              genint_set;
  bit [31:0]              genint_clear;
  bit [31:0]              genint_unmask;
  bit [31:0]              timers_int_unmask;
  bit [31:0]              timers_int_event;
  rxcntrl2_reg_s          rx_ctrl2;
  //---------------------------------------------------------------------------


  function new(string name = "mac_scoreboard_base", uvm_component parent = null);
    super.new(name, parent);

    mac_pool = uvm_event_pool::get_global_pool();
    no_resp_trigger = mac_pool.get("no_resp_trigger");
    frame_filtered_trigger = mac_pool.get("frame_filtered_trigger");
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);

    if (!uvm_config_db #(wlan_config)::get(null, "", "wlan_config", m_cfg))
      `uvm_fatal(get_type_name(), "WLAN configuration not set in configuration DB!!!")

    if (!uvm_config_db #(KeyStorageRAM)::get(null, "", "KeyStorageRAM", m_ksr))
      `uvm_fatal(get_type_name(), "Key storage RAM object missing!")

    if (!uvm_config_db #(TOP_register_block)::get(null, "", "TOP_register_block", m_regmodel))
      `uvm_fatal(get_type_name(), "WLAN configuration not set in configuration DB!!!")

  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase

  virtual task run_phase(uvm_phase phase);
    super.run_phase(phase);
  endtask : run_phase

  virtual function void report_phase(uvm_phase phase);
    super.report_phase(phase);
  endfunction : report_phase

  //---------------------------------------------------------------------------
  // list of functions for checking
  //---------------------------------------------------------------------------
  extern function void            fetch_register_values();
  extern task                     fetch_tsf_reg(ref timestamp_t rdata);
  extern function int             pad_bytes_cnt(int payload_len);
  extern function bit             filter_ppdu_frame(mac_phy_seq_item i);
  extern function int             individual_frame_type_filter(mac_phy_seq_item i);
  extern function void            print_filter_messages(int code, mpdu_frame_type_e t = ASSOCIATION_REQUEST);
  extern function bit             is_MAC_receiver(ref PPDU_frame frm);
  extern function int             intra_or_inter_BSS_frame(ref PPDU_frame frm);
  extern function void            updating_two_NAVs(int intra_or_inter_BSS, mac_address_t txop_addr, ref PPDU_frame frm);
  extern function bit             frame_expecting_resp(PPDU_frame frm, mac_address_t dev_addr);
  extern function mac_address_t   get_RA(ref PPDU_frame frm);
  extern function mac_address_t   get_TA(ref PPDU_frame frm);
  extern function mac_address_t   get_BSSID(ref PPDU_frame frm);
  extern function mac_address_t   get_txop_holder(ref PPDU_frame frm);
  extern function trigger_type_e  get_trigger_type(ref PPDU_frame frm);
  extern function bit             is_aid_matching(ref PPDU_frame frm, ref int staid);
  extern function bit             is_mu_calibration(ref PPDU_frame frm);
  extern function int             number_of_valid_mpdu_frames(ref PPDU_frame frm, input int user);
  extern function int             find_first_valid_mpdu(ref PPDU_frame frm);
  extern function bit             is_rx_trigger_frame(ref PPDU_frame frm);
  extern function void            predict_trigger_register(TRIGGER_frame trig_frm);
  extern function bit [6:0]       calc_txop_duration(ref PPDU_frame frm, ref PPDU_preamble_header txvector, input int user=0);
  //---------------------------------------------------------------------------

endclass : mac_scoreboard_base

  //---------------------------------------------------------------------------
  // fetch register settings
  //---------------------------------------------------------------------------
  function void mac_scoreboard_base::fetch_register_values();

    rx_ctrl              = m_regmodel.get_mirrored_reg_value("RXCNTRLREG");
    rx_ctrl2             = m_regmodel.get_mirrored_reg_value("RXCNTRL2");
    bss_addr[47:32]      = m_regmodel.get_mirrored_reg_value("BSSIDHIREG");
    bss_addr[31:0]       = m_regmodel.get_mirrored_reg_value("BSSIDLOWREG");
    dev_addr[47:32]      = m_regmodel.get_mirrored_reg_value("MACADDRHIREG");
    dev_addr[31:0]       = m_regmodel.get_mirrored_reg_value("MACADDRLOWREG");
    dev_addr_mask[47:32] = m_regmodel.get_mirrored_reg_value("MACADDRHIMASKREG");
    dev_addr_mask[31:0]  = m_regmodel.get_mirrored_reg_value("MACADDRLOWMASKREG");
    txrx_int_unmask      = m_regmodel.get_mirrored_reg_value("TXRXINTUNMASKREG");
    txrx_int_event       = m_regmodel.get_mirrored_reg_value("TXRXINTEVENTSETREG");
    mac_ctrl_1           = m_regmodel.get_mirrored_reg_value("MACCNTRL1REG");
    timings_1            = m_regmodel.get_mirrored_reg_value("TIMINGS1REG");
    basic_rates          = m_regmodel.get_mirrored_reg_value("RATESREG");
    htmcs                = m_regmodel.get_mirrored_reg_value("HTMCSREG");
    vhtmcs               = m_regmodel.get_mirrored_reg_value("VHTMCSREG");
    edca_ac0             = m_regmodel.get_mirrored_reg_value("EDCAAC0REG");
    edca_ac1             = m_regmodel.get_mirrored_reg_value("EDCAAC1REG");
    edca_ac2             = m_regmodel.get_mirrored_reg_value("EDCAAC2REG");
    edca_ac3             = m_regmodel.get_mirrored_reg_value("EDCAAC3REG");
    bfmee_ctrl           = m_regmodel.get_mirrored_reg_value("BFMEECONTROLREG");
    bcn_ctrl2            = m_regmodel.get_mirrored_reg_value("BCNCNTRL2REG");
    pta_config           = m_regmodel.get_mirrored_reg_value("CONFIG");
    he_config            = m_regmodel.get_mirrored_reg_value("HECONFIGREG");
    bss_color            = m_regmodel.get_mirrored_reg_value("BSSCOLORREG");
    genint_set           = m_regmodel.get_mirrored_reg_value("GENINTEVENTSETREG");
    genint_clear         = m_regmodel.get_mirrored_reg_value("GENINTEVENTCLEARREG");
    genint_unmask        = m_regmodel.get_mirrored_reg_value("GENINTUNMASKREG");
    timers_int_unmask    = m_regmodel.get_mirrored_reg_value("TIMERSINTUNMASKREG");
    timers_int_event     = m_regmodel.get_mirrored_reg_value("TIMERSINTEVENTSETREG");

  endfunction : fetch_register_values

  //---------------------------------------------------------------------------
  // fetch TSF register values when MAC-PHY item is received, register read is
  // done via backdoor because TSF register is always changing so read
  // must be done at proper timestamp
  //---------------------------------------------------------------------------
  task mac_scoreboard_base::fetch_tsf_reg(ref timestamp_t rdata);
    uvm_reg      tsf_lo_reg;
    uvm_reg      tsf_hi_reg;
    uvm_status_e status;

    tsf_lo_reg  = m_regmodel.get_reg_by_name("TSFLOREG");
    tsf_lo_reg.read(.status(status), .value(tsf_low), .path(UVM_BACKDOOR));
    tsf_hi_reg  = m_regmodel.get_reg_by_name("TSFHIREG");
    tsf_hi_reg.read(.status(status), .value(tsf_high), .path(UVM_BACKDOOR));

    rdata = {tsf_high,tsf_low};
  endtask : fetch_tsf_reg

  //---------------------------------------------------------------------------
  // calculate number of pad bytes inside payload of RPD,
  // payload is 4 byte aligned
  //---------------------------------------------------------------------------
  function int mac_scoreboard_base::pad_bytes_cnt(int payload_len);
    int skip_cnt;
    int res;

    skip_cnt = payload_len % 4;
    res = 4 - skip_cnt;

    // if result is aligned then there are no pad bytes
    return ((res == 4) ? 0 : res);
  endfunction : pad_bytes_cnt

  //---------------------------------------------------------------------------
  // determine if MAC is receiver, apply also address masking
  //---------------------------------------------------------------------------
  function bit mac_scoreboard_base::is_MAC_receiver(ref PPDU_frame frm);
    MPDU_frame mpdu;
    int        valid_mpdu;

    valid_mpdu = find_first_valid_mpdu(frm);
    mpdu = frm.ampdu_frame[0].mpdu_frame[valid_mpdu];

    if (mpdu != null)
      return ((dev_addr_mask | mpdu.MAC_header.addr[0]) == (dev_addr_mask | dev_addr)) ? 1 : 0;
    else
      return 0;
  endfunction : is_MAC_receiver

  //---------------------------------------------------------------------------
  // get receiver address
  //---------------------------------------------------------------------------
  function mac_address_t mac_scoreboard_base::get_RA(ref PPDU_frame frm);
    int        valid_mpdu;

    valid_mpdu = find_first_valid_mpdu(frm);

    if (frm.ampdu_frame[0].mpdu_frame[valid_mpdu] != null)
      return frm.ampdu_frame[0].mpdu_frame[valid_mpdu].MAC_header.addr[0];
    else
      return 0;
  endfunction : get_RA

  //---------------------------------------------------------------------------
  // get trasmitter address
  //---------------------------------------------------------------------------
  function mac_address_t mac_scoreboard_base::get_TA(ref PPDU_frame frm);
    mac_address_t addr;
    int           valid_mpdu;

    valid_mpdu = find_first_valid_mpdu(frm);

    if (frm.ampdu_frame[0].mpdu_frame[valid_mpdu].MAC_header.addr.size() > 1) begin
      return frm.ampdu_frame[0].mpdu_frame[valid_mpdu].MAC_header.addr[1];
    end
    else begin
      if (   frm.ampdu_frame[0].mpdu_frame_type[valid_mpdu] == CONTROL_WRAPPER
          && frm.ampdu_frame[0].mpdu_frame[valid_mpdu].frame_body.size() > 0) begin

        // get TA address from carried frame
        foreach (frm.ampdu_frame[0].mpdu_frame[valid_mpdu].frame_body[i])
          addr[i*8 +: 8] = frm.ampdu_frame[0].mpdu_frame[valid_mpdu].frame_body[i];

        return addr;
      end
    end

    return 0;
  endfunction : get_TA

  //---------------------------------------------------------------------------
  // get BSSID address
  //---------------------------------------------------------------------------
  function mac_address_t mac_scoreboard_base::get_BSSID(ref PPDU_frame frm);
    int           valid_mpdu;

    valid_mpdu = find_first_valid_mpdu(frm);

    if (frm.ampdu_frame[0].mpdu_frame[valid_mpdu] == null)
      return 0;

    if (frm.ampdu_frame[0].mpdu_frame[valid_mpdu].MAC_header.addr.size() > 2)
      return frm.ampdu_frame[0].mpdu_frame[valid_mpdu].MAC_header.addr[2];
    else
      return 0;
  endfunction : get_BSSID

  //---------------------------------------------------------------------------
  // get TXOP holder
  //---------------------------------------------------------------------------
  function mac_address_t mac_scoreboard_base::get_txop_holder(ref PPDU_frame frm);
    if (get_RA(frm) != 0 && get_TA(frm) != 0 && get_RA(frm) == bss_addr)
      return get_TA(frm);
    else
      return 0;
  endfunction : get_txop_holder

  //---------------------------------------------------------------------------
  // get Trigger type
  //---------------------------------------------------------------------------
  function trigger_type_e mac_scoreboard_base::get_trigger_type(ref PPDU_frame frm);
    TRIGGER_frame trig_frm;
    trig_frm = TRIGGER_frame'(frm.ampdu_frame[0].mpdu_frame[0]);

    return trig_frm.common_info.trigger_type_f;
  endfunction : get_trigger_type

  //---------------------------------------------------------------------------
  // analyse NPDA frame, station info to determine is one of them with
  // AID matching one in BCNCNTRL2REG, also returns staid found
  //---------------------------------------------------------------------------
  function bit mac_scoreboard_base::is_aid_matching(ref PPDU_frame frm, ref int staid);
    VHT_NDP_ANNOUNCEMENT_frame ndpa;
    HE_NDP_ANNOUNCEMENT_frame  he_ndpa;
    // typecast MPDU frame to NDPA
    if (frm.ppdu_format == HE_SU) begin
      he_ndpa = HE_NDP_ANNOUNCEMENT_frame'(frm.ampdu_frame[0].mpdu_frame[0]);
      foreach (he_ndpa.sta_info[i]) begin
        if (he_ndpa.sta_info[i].aid11_f == bcn_ctrl2[26:16]) begin
          `uvm_info(get_type_name(),
          $sformatf("Matching AID found inside NDPA STA info (%0d)",i),UVM_LOW)
          staid = i;
          return 1;
        end
      end
    end
    else begin
      ndpa = VHT_NDP_ANNOUNCEMENT_frame'(frm.ampdu_frame[0].mpdu_frame[0]);
      foreach (ndpa.sta_info[i]) begin
        if (ndpa.sta_info[i].aid12_f == bcn_ctrl2[27:16]) begin
          `uvm_info(get_type_name(),
          $sformatf("Matching AID found inside NDPA STA info (%0d)",i),UVM_LOW)
          staid = i;
          return 1;
        end
      end
    end//else

    return 0;
  endfunction : is_aid_matching

  //---------------------------------------------------------------------------
  // analyse NPDA frame, number of STA infos to determine if calibrations is
  // SU or MU
  //---------------------------------------------------------------------------
  function bit mac_scoreboard_base::is_mu_calibration(ref PPDU_frame frm);
    int                        sta_info_size;
    VHT_NDP_ANNOUNCEMENT_frame ndpa;
    HE_NDP_ANNOUNCEMENT_frame  he_ndpa;

    // typecast MPDU frame to NDPA
    if (frm.ppdu_format == HE_SU) begin
      he_ndpa = HE_NDP_ANNOUNCEMENT_frame'(frm.ampdu_frame[0].mpdu_frame[0]);
      sta_info_size = he_ndpa.sta_info.size();
    end
    else begin
      ndpa = VHT_NDP_ANNOUNCEMENT_frame'(frm.ampdu_frame[0].mpdu_frame[0]);
      sta_info_size = ndpa.sta_info.size();
    end

    return (sta_info_size>1) ? 1'b1 : 1'b0;
  endfunction : is_mu_calibration

  //---------------------------------------------------------------------------
  // determine is received frame TRIGGER frame
  //---------------------------------------------------------------------------
  function bit mac_scoreboard_base::is_rx_trigger_frame(ref PPDU_frame frm);
    if (frm.kind inside {AGGREGATED, NDP, MU_MIMO} || frm.preamble_header.tr == TX)
      return 0;
    else if (frm.ampdu_frame[0].mpdu_frame_type[0] == TRIGGER)
      return 1;
    else
      return 0;
  endfunction : is_rx_trigger_frame

  //---------------------------------------------------------------------------
  // get number of valid MPDU frames (no FSC errors) from A-MPDU frame
  //---------------------------------------------------------------------------
  function int mac_scoreboard_base::number_of_valid_mpdu_frames(ref PPDU_frame frm, input int user);
    int count = 0;

    foreach (frm.ampdu_frame[user].mpdu_frame_type[i]) begin
      if (frm.ampdu_frame[user].mpdu_frame_type[i] != ERROR_FRAME)
        count++;
    end
    return count;
  endfunction : number_of_valid_mpdu_frames

  //---------------------------------------------------------------------------
  // find a fist valid MPDU in AMPDU
  //---------------------------------------------------------------------------
  function int mac_scoreboard_base::find_first_valid_mpdu(ref PPDU_frame frm);
    int ret;

    if (frm.kind == AGGREGATED && frm.ampdu_frame[0].no_valid_frames == 0) begin
      foreach (frm.ampdu_frame[0].mpdu_frame_type[i]) begin
        if (frm.ampdu_frame[0].mpdu_frame_type[i] != ERROR_FRAME) begin
         ret = i;
         break;
        end
      end
    end
    else begin
      ret = 0;
    end

    return ret;
  endfunction : find_first_valid_mpdu

  //---------------------------------------------------------------------------
  // determine does received PPDU frame need to be filtered or not
  // filtering rules are described in MAC-HW user manual (chapter 2.3.4.1)
  // @return: 1 - frame filtered, 0 - frame should be received
  //---------------------------------------------------------------------------
  function bit mac_scoreboard_base::filter_ppdu_frame(mac_phy_seq_item i);
    int ret = 0;
    int valid_mpdu = 0; // index of first valid MPDU in AMPDU frame

    fetch_register_values();

    // special care when NDP frame is received because other filter checks
    // can't be performed bacause there is no Payload data
    if (i.frame.kind == NDP && !rx_ctrl.accept_bfmee_frames_f) begin
      print_filter_messages(50);
      return 1;
    end
    else if (i.frame.kind == NDP) begin
      return 0;
    end
    else if (i.frame.kind != NDP && i.frame.ampdu_frame[0].no_valid_frames == 1) begin
      print_filter_messages(51);
      return 1;
    end
    else begin
      valid_mpdu = find_first_valid_mpdu(i.frame);
    end

    //********************************************************************************
    if (    (   i.frame.ampdu_frame[0].mpdu_frame_type[valid_mpdu] == VHT_NDP_ANNOUNCEMENT
             || i.frame.ampdu_frame[0].mpdu_frame_type[valid_mpdu] == HE_NDP_ANNOUNCEMENT
             || i.frame.ampdu_frame[0].mpdu_frame_type[valid_mpdu] == BEAMFORMING_REPORT_POLL)
            && !rx_ctrl.accept_bfmee_frames_f) begin
      ret = 50;
    end
    //********************************************************************************
    else if (   (i.rx_err || i.phy_err || i.frame.ampdu_frame[0].mpdu_frame_type[valid_mpdu] == ERROR_FRAME)
             && !rx_ctrl.accept_error_frames_f) begin
      ret = 1;
    end
    //********************************************************************************
    else if (get_RA(i.frame) == `BROADCAST_ADDR && !rx_ctrl.accept_broadcast_f) begin
      ret = 2;
    end
    //********************************************************************************
    else if (   get_RA(i.frame) == `BROADCAST_ADDR
             && rx_ctrl.accept_broadcast_f
             && get_BSSID(i.frame) == bss_addr) begin

      ret = (individual_frame_type_filter(i)==0) ? 0 : individual_frame_type_filter(i) + 2;
    end
    //********************************************************************************
    else if (   get_RA(i.frame) == `BROADCAST_ADDR
             && rx_ctrl.accept_broadcast_f
             && get_BSSID(i.frame) != bss_addr
             && rx_ctrl.accept_other_bssid_f) begin

      ret = (individual_frame_type_filter(i)==0) ? 0 : individual_frame_type_filter(i) + 4;
    end
    //********************************************************************************
    else if (   get_RA(i.frame) == `BROADCAST_ADDR
             && rx_ctrl.accept_broadcast_f
             && get_BSSID(i.frame) != bss_addr
             && !rx_ctrl.accept_other_bssid_f) begin

      ret = 7;
    end
    //********************************************************************************
    // check if multicast address is set
    else if ((get_RA(i.frame) & 48'h1) == 48'h1 && !rx_ctrl.accept_multicast_f) begin
      ret = 8;
    end
    //********************************************************************************
    else if (   (get_RA(i.frame) & 48'h1) == 48'h1
             && rx_ctrl.accept_multicast_f
             && get_BSSID(i.frame) == bss_addr) begin

      ret = (individual_frame_type_filter(i)==0) ? 0 : individual_frame_type_filter(i) + 8;
    end
    //********************************************************************************
    else if (   (get_RA(i.frame) & 48'h1) == 48'h1
             && rx_ctrl.accept_multicast_f
             && get_BSSID(i.frame) != bss_addr
             && rx_ctrl.accept_other_bssid_f) begin

      ret = (individual_frame_type_filter(i)==0) ? 0 : individual_frame_type_filter(i) + 10;
    end
    //********************************************************************************
    else if (   (get_RA(i.frame) & 48'h1) == 48'h1
             && rx_ctrl.accept_multicast_f
             && get_BSSID(i.frame) != bss_addr
             && !rx_ctrl.accept_other_bssid_f) begin

      ret = 13;
    end
    //********************************************************************************
    // unicast address
    else if (get_RA(i.frame) == dev_addr && !rx_ctrl.accept_my_unicast_f) begin
      ret = 14;
    end
    //********************************************************************************
    else if (get_RA(i.frame) == dev_addr && rx_ctrl.accept_my_unicast_f) begin

      ret = (individual_frame_type_filter(i)==0) ? 0 : individual_frame_type_filter(i) + 14;
    end
    //********************************************************************************
    // check if unicast address is set
    else if (   get_RA(i.frame) != dev_addr
             && (get_RA(i.frame) & 48'h1) == 48'h0
             && !rx_ctrl.accept_unicast_f) begin

      ret = 17;
    end
    else if (   get_RA(i.frame) != dev_addr
             && (get_RA(i.frame) & 48'h1) == 48'h0
             && get_BSSID(i.frame) == bss_addr
             && rx_ctrl.accept_unicast_f) begin

      ret = (individual_frame_type_filter(i)==0) ? 0 : individual_frame_type_filter(i) + 17;
    end
    else if (   get_RA(i.frame) != dev_addr
             && (get_RA(i.frame) & 48'h1) == 48'h0
             && get_BSSID(i.frame) != bss_addr
             && rx_ctrl.accept_other_bssid_f
             && rx_ctrl.accept_unicast_f) begin

      ret = (individual_frame_type_filter(i)==0) ? 0 : individual_frame_type_filter(i) + 19;
    end
    else if (   get_RA(i.frame) != dev_addr
             && (get_RA(i.frame) & 48'h1) == 48'h0
             && get_BSSID(i.frame) != bss_addr
             && !rx_ctrl.accept_other_bssid_f
             && rx_ctrl.accept_unicast_f) begin

      ret = 22;
    end
    //******************************************************************************
    // check if the reception of Trigger frame enabled (SW & HW)
    else if (   i.frame.ampdu_frame[0].mpdu_frame_type[valid_mpdu] == TRIGGER
             && !he_config[9]
             && !(get_trigger_type(i.frame) inside {BASIC_TRIGGER,
                                                    BUFFER_STATUS_REPORT_POLL})) begin
      ret = 23;
    end
    else if (   i.frame.ampdu_frame[0].mpdu_frame_type[valid_mpdu] == TRIGGER
             && !he_config[8]
             && !(get_trigger_type(i.frame) inside {BMF_REPORT_POLL,
                                                    MU_BAR,
                                                    MU_RTS,
                                                    BANDWIDTH_QUERY_REPORT_POLL})) begin
      ret = 24;
    end

    print_filter_messages(ret, i.frame.ampdu_frame[0].mpdu_frame_type[valid_mpdu]);

    // if there is filter number, return 1 else 0 (frame is not filtered)
    return (ret != 0) ? 1 : 0;
  endfunction : filter_ppdu_frame

  //---------------------------------------------------------------------------
  // helping function for filtering specific type of frame relative to
  // register settings
  // @return: 0 - no filtering
  //          1 - specific frame type filtered
  //          2 - others filter applied
  // (MNG/CTRL/DATA)
  //---------------------------------------------------------------------------
  function int mac_scoreboard_base::individual_frame_type_filter(mac_phy_seq_item i);
    mpdu_frame_type_e     frame_subtype;
    CONTROL_WRAPPER_frame ctrl_wrapp;
    int                   valid_mpdu;

    valid_mpdu = find_first_valid_mpdu(i.frame);
    if (i.frame.ampdu_frame[0].mpdu_frame[valid_mpdu] == null)
      return -1;

    frame_subtype = i.frame.ampdu_frame[0].mpdu_frame_type[valid_mpdu];
    // control wrapper frame carries the real frame type which
    // MAC needs to analyze
    if (frame_subtype == CONTROL_WRAPPER) begin
      // type cast frame to control wrapper
      ctrl_wrapp = CONTROL_WRAPPER_frame'(i.frame.ampdu_frame[0].mpdu_frame[valid_mpdu]);
      frame_subtype = ctrl_wrapp.ctrl_frame_type;
    end

    if (   (   i.frame.ampdu_frame[0].mpdu_frame[valid_mpdu].MAC_header.frame_ctrl.protected_frame_f == 0
            && rx_ctrl.exc_unencrypted_f)
        || (   frame_subtype == BEACON
            && !rx_ctrl.accept_beacon_f
            && !rx_ctrl.accept_all_beacon_f)
        || (frame_subtype == PROBE_RESPONSE    && !rx_ctrl.accept_probe_resp_f)
        || (frame_subtype == PROBE_REQUEST     && !rx_ctrl.accept_probe_req_f)
        || (   frame_subtype == BLOCK_ACK_REQUEST
            && (get_bar_variant(i.frame) inside {MULTI_TID_BAR, GCR_BAR} || !rx_ctrl.accept_bar_f))
        || (   frame_subtype == BLOCK_ACK
            && (!rx_ctrl.accept_ba_f || !rx_ctrl.accept_not_expected_ba_f))
        || (frame_subtype == PS_POLL && !rx_ctrl.accept_ps_poll_f)
        || (frame_subtype == RTS && !rx_ctrl.accept_rts_f)
        || (frame_subtype == CTS && !rx_ctrl.accept_cts_f)
        || (frame_subtype == ACK && !rx_ctrl.accept_ack_f)
        || (frame_subtype inside {CF_END,CF_END_CF_ACK} && !rx_ctrl.accept_cf_end_f)
        || (frame_subtype inside {DATA,
                                  DATA_CF_ACK,
                                  DATA_CF_POLL,
                                  DATA_CF_ACK_CF_POLL} && !rx_ctrl.accept_data_f)
        || (frame_subtype inside {CF_ACK,
                                  CF_POLL,
                                  CF_ACK_CF_POLL} && !rx_ctrl.accept_cf_wo_data_f)
        || (frame_subtype inside {QOS_DATA,
                                  QOS_DATA_CF_ACK,
                                  QOS_DATA_CF_POLL,
                                  QOS_DATA_CF_ACK_CF_POLL} && !rx_ctrl.accept_qdata_f)
        || (frame_subtype inside {QOS_CF_POLL,
                                  QOS_CF_ACK_CF_POLL} && !rx_ctrl.accept_qcf_wo_data_f)
        || (frame_subtype == QOS_NULL && !rx_ctrl.accept_qos_null_f))

      return 1;

    else if (   (   i.frame.ampdu_frame[0].mpdu_frame_type[valid_mpdu][5:4] == `MANAGEMENT_MPDU
                 && !(frame_subtype inside {
                      PROBE_RESPONSE,PROBE_RESPONSE,BEACON}) // not one of these frame subtypes
                 && !rx_ctrl.accept_other_mngm_frames_f)
             || (   i.frame.ampdu_frame[0].mpdu_frame_type[valid_mpdu][5:4] == `CONTROL_MPDU
                    // not one of these frame subtypes
                 && i.frame.ampdu_frame[0].mpdu_frame[valid_mpdu].MAC_header.frame_ctrl.subtype_f inside {4'h0, 4'h1, 4'h3, 4'h6}
                 && !rx_ctrl.accept_other_ctrl_frames_f)
             || (   i.frame.ampdu_frame[0].mpdu_frame_type[valid_mpdu][5:4] == `DATA_MPDU
                    // not one of these frame subtypes
                 && i.frame.ampdu_frame[0].mpdu_frame[valid_mpdu].MAC_header.frame_ctrl.subtype_f == 4'hD
                 && !rx_ctrl.accept_other_data_frames_f))

      return 2;
    else
      return 0; // no filtering
  endfunction : individual_frame_type_filter

  //----------------------------------------------------------------------
  // print filtering rules that were applied
  //----------------------------------------------------------------------
  function void mac_scoreboard_base::print_filter_messages(int code, mpdu_frame_type_e t = ASSOCIATION_REQUEST);
    case (code)
      1 :`uvm_info(get_type_name(),$sformatf("Frame with error discarded"), UVM_LOW)
      2 :`uvm_info(get_type_name(),$sformatf("Broadcast frame not accepted - discarded"), UVM_LOW)
      3 :`uvm_info(get_type_name(),$sformatf("Broadcast frame accepted,BSSID match but individual frame type didn't match settings - discarded"), UVM_LOW)
      4 :`uvm_info(get_type_name(),$sformatf("Broadcast frame accepted,BSSID match but others frame type didn't match settings - discarded"), UVM_LOW)
      5 :`uvm_info(get_type_name(),$sformatf("Broadcast frame accepted,BSSID don't match and others BSS flag enabled but individual frame type didn't match settings - discarded"), UVM_LOW)
      6 :`uvm_info(get_type_name(),$sformatf("Broadcast frame accepted,BSSID don't match and others BSS flag enabled but others frame type didn't match settings - discarded"), UVM_LOW)
      7 :`uvm_info(get_type_name(),$sformatf("Broadcast frame accepted,BSSID don't match and others BSS flag disabled - discarded"), UVM_LOW)
      8 :`uvm_info(get_type_name(),$sformatf("Multicast frame not accepted - discarded"), UVM_LOW)
      9 :`uvm_info(get_type_name(),$sformatf("Multicast frame accepted,BSSID match but individual frame type didn't match settings - discarded"), UVM_LOW)
      10:`uvm_info(get_type_name(),$sformatf("Multicast frame accepted,BSSID match but others frame type didn't match settings - discarded"), UVM_LOW)
      11:`uvm_info(get_type_name(),$sformatf("Multicast frame accepted,BSSID don't match and others BSS flag enabled but individual frame type didn't match settings - discarded"), UVM_LOW)
      12:`uvm_info(get_type_name(),$sformatf("Multicast frame accepted,BSSID don't match and others BSS flag enabled but others frame type didn't match settings - discarded"), UVM_LOW)
      13:`uvm_info(get_type_name(),$sformatf("Multicast frame accepted,BSSID don't match and others BSS flag disabled - discarded"), UVM_LOW)
      14:`uvm_info(get_type_name(),$sformatf("Unicast frame for this device not accepted - discarded"), UVM_LOW)
      15:`uvm_info(get_type_name(),$sformatf("Unicast frame for this device accepted but individual frame type didn't match settings - discarded"), UVM_LOW)
      16:`uvm_info(get_type_name(),$sformatf("Unicast frame for this device accepted but others frame type didn't match settings - discarded"), UVM_LOW)
      17:`uvm_info(get_type_name(),$sformatf("Unicast frame not accepted - discarded"), UVM_LOW)
      18:`uvm_info(get_type_name(),$sformatf("Unicast frame accepted,BSSID match but individual frame type didn't match settings - discarded"), UVM_LOW)
      19:`uvm_info(get_type_name(),$sformatf("Unicast frame accepted,BSSID match but others frame type didn't match settings - discarded"), UVM_LOW)
      20:`uvm_info(get_type_name(),$sformatf("Unicast frame accepted,BSSID don't match and others BSS flag enabled but individual frame type didn't match settings - discarded"), UVM_LOW)
      21:`uvm_info(get_type_name(),$sformatf("Unicast frame accepted,BSSID don't match and others BSS flag enabled but others frame type didn't match settings - discarded"), UVM_LOW)
      22:`uvm_info(get_type_name(),$sformatf("Unicast frame accepted,BSSID don't match and others BSS flag disabled - discarded"), UVM_LOW)
      23:`uvm_info(get_type_name(),$sformatf("SW Trigger frame - discarded"), UVM_LOW)
      24:`uvm_info(get_type_name(),$sformatf("HW Trigger frame - discarded"), UVM_LOW)
      50:`uvm_info(get_type_name(),$sformatf("Beamformed frame not accepted - discarded"), UVM_LOW)
      51:`uvm_info(get_type_name(),$sformatf("NO valid delimiters found - discarded"), UVM_LOW)
      default: `uvm_info(get_type_name(),$sformatf("Frame %s should be received by MAC core",t.name()), UVM_LOW)
    endcase

    if (code != 0) begin
      `uvm_info(get_type_name(),$sformatf("Frame %s discarded",t.name()), UVM_LOW)
    end
    `uvm_info(get_type_name(),$sformatf("RX control register: %p",rx_ctrl), UVM_LOW)
  endfunction : print_filter_messages

  //----------------------------------------------------------------------
  // function for determining whether the frame is intra-BSS or inter-BSS
  //----------------------------------------------------------------------
  function int mac_scoreboard_base::intra_or_inter_BSS_frame(ref PPDU_frame frm);
    bit [3:0]   partial_bss_color;
    int         valid_mpdu;

    // Protection, if the frame is not valid,
    // or wasn't been collected
    if (frm.ampdu_frame[0].no_valid_frames == 1)
      return 0;
    else if (frm.ampdu_frame[0].mpdu_frame[0] == null)
      return 0;
    else
      valid_mpdu = find_first_valid_mpdu(frm);

    // Fetch register values to get BSSID and BSS color
    fetch_register_values();

    // Get partial BSS color 27.16.3 AID assignment
    partial_bss_color = bss_color[3:0] - (bss_addr[47:44]^bss_addr[43:40]);

    // The frame is inter-BSS frame
    if (     (frm.preamble_header.format_mod inside {HE_SU, HE_EXT_SU, HE_TB, HE_MU}
          &&  frm.preamble_header.bss_color != 0
          &&  frm.preamble_header.bss_color != bss_color[5:0])

          || (frm.preamble_header.format_mod == VHT
          &&  frm.preamble_header.partial_aid != bss_addr[47:39]
          &&  frm.preamble_header.group_id != 0)

          || (frm.preamble_header.format_mod == VHT
          &&  frm.preamble_header.partial_aid[8:5] != partial_bss_color
          &&  frm.preamble_header.group_id == 63
          &&  bss_color[8] == 1) // BSS color enable

          || (frm.kind == MU_MIMO
          && frm.preamble_header.format_mod == VHT
          && mac_ctrl_1.ap_f == 1)

          || (frm.preamble_header.format_mod == HE_MU
          &&  frm.preamble_header.uplink_flag == 0
          &&  mac_ctrl_1.ap_f == 1)

          || (get_BSSID(frm) != 0
          &&  get_BSSID(frm) != bss_addr)

          || (get_BSSID(frm) == 0
          &&  get_TA(frm) != 0
          &&  get_RA(frm) != bss_addr
          &&  get_TA(frm) != bss_addr)) begin
      `uvm_info(get_type_name(), $sformatf("The frame is an inter-BSS frame"), UVM_HIGH)
      return 1;
    end
    // The frame is intra-BSS frame
    else if (    (frm.preamble_header.format_mod inside {HE_SU, HE_EXT_SU, HE_TB, HE_MU}
              && (frm.preamble_header.bss_color == 0
              ||  frm.preamble_header.bss_color == bss_color[5:0]))

              || (frm.preamble_header.format_mod == VHT
              &&  frm.preamble_header.partial_aid == bss_addr[47:39]
              &&  frm.preamble_header.group_id == 0)

              || (frm.preamble_header.format_mod == VHT
              &&  frm.preamble_header.partial_aid[8:5] == partial_bss_color
              &&  frm.preamble_header.group_id == 63
              &&  bss_color[8] == 1) // BSS color enable

              || (get_RA(frm) == bss_addr
              ||  get_TA(frm) == bss_addr
              ||  get_BSSID(frm) == bss_addr)

              || (frm.ampdu_frame[0].mpdu_frame_type[valid_mpdu][5:4] == `CONTROL_MPDU
              &&  get_TA(frm) == 0
              &&  get_RA(frm) == txop_holder_addr)) begin
      `uvm_info(get_type_name(), $sformatf("The frame is an intra-BSS frame"), UVM_HIGH)
      return 2;
    end
    // The frame is neither inter nor intra BSS frame
    else begin
      `uvm_info(get_type_name(), $sformatf("The frame is neither inter nor intra BSS frame"), UVM_HIGH)
      return 0;
    end

  endfunction : intra_or_inter_BSS_frame

  //----------------------------------------------------------------------------
  // Decide whether intra NAV or basic NAV should be updated
  //----------------------------------------------------------------------------
  function void mac_scoreboard_base::updating_two_NAVs(int intra_or_inter_BSS, mac_address_t txop_addr, ref PPDU_frame frm);
    //------------------------
    // Decoding:
    // 1  - intra_BSS frame
    // 2  - inter_BSS frame
    // 0  - neither
    //------------------------

    int         nsts_total;
    int         mcs;
    bit [2:0]   nss;
    int         valid_mpdu;

    // Protection, if the frame is not valid,
    // or wasn't been collected
    if (frm.ampdu_frame[0].no_valid_frames == 1)
      return;
    else if (frm.ampdu_frame[0].mpdu_frame[0] == null)
      return;
    else
      valid_mpdu = find_first_valid_mpdu(frm);

    frm_type_base = frm.ampdu_frame[0].mpdu_frame_type[valid_mpdu];
    // control wrapper frame carries the real frame type which
    // MAC needs to analyze
    if (frm_type_base == CONTROL_WRAPPER) begin
      // type cast frame to control wrapper
      ctrl_wrapp_base = CONTROL_WRAPPER_frame'(frm.ampdu_frame[0].mpdu_frame[valid_mpdu]);
      frm_type_base = ctrl_wrapp_base.ctrl_frame_type;
    end

    if (frm.preamble_header.format_mod == VHT && frm.kind != MU_MIMO) begin
      nsts_total = frm.preamble_header.user_header[0].num_sts_f;
      nss = (frm.preamble_header.stbc > 0) ? (nsts_total+1)/(2*frm.preamble_header.stbc)-1 : nsts_total;
      mcs = {nss[2:0],frm.preamble_header.user_header[0].mcs_f[3:0]};
    end
    else begin
      nsts_total = frm.preamble_header.user_header[0].num_sts_f;
      mcs = frm.preamble_header.user_header[0].mcs_f;
    end

    // Update intra NAV
    if (intra_or_inter_BSS == 2) begin
      if (   (get_RA(frm) == txop_addr
          &&  frm.ampdu_frame[0].get_MAC_duration() > intra_NAV
          &&  !is_MAC_receiver(frm))
          ||  !frame_expecting_resp(frm, dev_addr)) begin // TODO: Add Trigger frame
          intra_NAV = frm.ampdu_frame[0].get_MAC_duration();
          `uvm_info(get_type_name(), $sformatf("Updating intra NAV with value = %d", intra_NAV), UVM_HIGH)
      end
    end
    // Updating basic NAV
    else begin
      if(    !is_MAC_receiver(frm)
         &&  frm.ampdu_frame[0].get_MAC_duration() > basic_NAV) begin
         basic_NAV = frm.ampdu_frame[0].get_MAC_duration();
         `uvm_info(get_type_name(), $sformatf("Updating BASIC NAV with value = %d", basic_NAV), UVM_HIGH)
      end
    end

    if (frm.ampdu_frame[0].mpdu_frame_type[valid_mpdu] inside {CF_END, CF_END_CF_ACK}) begin
      basic_NAV = 0;
      intra_NAV = 0;
    end

    // If the NAV was updated from RTS or MU-RTS Trigger frame, NAV should be reset
    // if no RX_START appears for 2xSIFS + CTS_Time + aRXPHYStartDelay + 2xaSlotTime
    if (    frm_type_base == RTS
        &&  !is_MAC_receiver(frm)
        &&  frm.ampdu_frame[0].get_MAC_duration() > basic_NAV
        &&  frm.ampdu_frame[0].get_MAC_duration() > intra_NAV) begin

      // calculate CTS time
      cts_time = ComputeTimeOnAirAC (.length_i     (14),   // length of CTS

                                     .preType_i    ((frm.preamble_header.format_mod inside {NON_HT, NON_HT_DUP_OFDM}) ?
                                                     frm.preamble_header.preamble_type :
                                                     frm.preamble_header.format_mod),

                                     .mcsIndex_i   ((frm.preamble_header.format_mod inside {NON_HT, NON_HT_DUP_OFDM}) ?
                                                     get_mcs_from_legacy_rate( // if NON_HT*
                                                       bss_basic_rate_set(frm.preamble_header.leg_rate, basic_rates) // non HT mode
                                                     ) : // else
                                                     mcs), // HT mode

                                     .shortGI_i    (frm.preamble_header.gi_type[0]),
                                     .chBW_i       (frm.preamble_header.ch_bw),
                                     .stbc_i       (frm.preamble_header.stbc),
                                     .extNSS_i     (frm.preamble_header.num_extn_ss),
                                     .woPreamble_i (0),
                                     .band5G_i     (get_band5G(mac_ctrl_1.abgn_mode_f)),
                                     .multiUser_i  (0),
                                     .nSTSTotal_i  (0),
                                     .debug_i      (0)
                                     );

      timeout_timer = 2*`SIFS_A + cts_time + `RX_PHY_START_DELAY + 2*`ASLOT_TIME;
      start_timeout_cnt = 1;
    end

  endfunction : updating_two_NAVs

  //----------------------------------------------------------------------
  // function that determines does MAC need to respond to received frame or
  // not. Returns 1 if response is needed else returns 0
  //----------------------------------------------------------------------
  function bit mac_scoreboard_base::frame_expecting_resp(PPDU_frame frm, mac_address_t dev_addr);
    mpdu_frame_type_e     frm_type;
    CONTROL_WRAPPER_frame ctrl_wrapp;
    int                   valid_mpdu;

    // Protection, if the frame is not valid,
    // or wasn't been collected
    if (frm.ampdu_frame[0].no_valid_frames == 1)
      return 0;
    else if (frm.ampdu_frame[0].mpdu_frame[0] == null)
      return 0;
    else
      valid_mpdu = find_first_valid_mpdu(frm);

    // don't responde to bcast and mcast frames
    if (   dev_addr == `MULTICAST_ADDR
        || dev_addr == `BROADCAST_ADDR) begin
      return 0;
    end

    frm_type = frm.ampdu_frame[0].mpdu_frame_type[valid_mpdu];
    // control wrapper frame carries the real frame type which
    // MAC needs to analyze
    if (frm_type == CONTROL_WRAPPER) begin
      // type cast frame to control wrapper
      ctrl_wrapp = CONTROL_WRAPPER_frame'(frm.ampdu_frame[0].mpdu_frame[valid_mpdu]);
      frm_type = ctrl_wrapp.ctrl_frame_type;
    end

    case (frm_type)
      QOS_DATA, QOS_DATA_CF_ACK, QOS_DATA_CF_POLL,
      QOS_DATA_CF_ACK_CF_POLL,
      QOS_NULL, QOS_CF_POLL, QOS_CF_ACK_CF_POLL: begin

        if (   frm.ampdu_frame[0].get_MAC_addr(.RA(1)) == dev_addr
            && frm.ampdu_frame[0].mpdu_frame[valid_mpdu].MAC_header.qos_ctrl[0].ack_policy_f == 2'b00)
          return 1;
        else
          return 0;
      end

      ACK, CTS, CF_END, CF_END_CF_ACK,
      BLOCK_ACK, ACTION_NOACK, BEACON,
      CF_ACK, CONTROL_WRAPPER, VHT_NDP_ANNOUNCEMENT: begin
        return 0;
      end // no response frames

      BLOCK_ACK_REQUEST: begin
        if (   frm.ampdu_frame[0].get_MAC_addr(.RA(1)) == dev_addr
            && get_bar_variant(frm) inside {BASIC_BAR, COMPRESSED_BAR, EXT_COMPRESSED_BAR})
          return 1;
        else
          return 0;
      end

      default: begin
        if (frm.ampdu_frame[0].get_MAC_addr(.RA(1)) == dev_addr)
          return 1;
        else
          return 0;
      end
    endcase
  endfunction : frame_expecting_resp

  //----------------------------------------------------------------------
  // function that predicts the fields for the following register:
  //    - RXHETRIGUSERINFOREG
  //    - RXHETRIGCOMMONINFOREG
  //----------------------------------------------------------------------
  function void mac_scoreboard_base::predict_trigger_register(TRIGGER_frame trig_frm);

    bit [2:0]   ru_index;

    if(trig_frm.user_info[0].ru_allocation_f[7:1] inside {[0:36]})
      ru_index = 0;
    else if (trig_frm.user_info[0].ru_allocation_f[7:1] inside {[37:52]})
      ru_index = 1;
    else if (trig_frm.user_info[0].ru_allocation_f[7:1] inside {[53:60]})
      ru_index = 2;
    else if (trig_frm.user_info[0].ru_allocation_f[7:1] inside {[61:64]})
      ru_index = 3;
    else if (trig_frm.user_info[0].ru_allocation_f[7:1] inside {[65:66]})
      ru_index = 4;
    else if (trig_frm.user_info[0].ru_allocation_f[7:1] == 67)
      ru_index = 5;

    // RXHETRIGCOMMONINFOREG
    m_regmodel.predict_field_value(.value(trig_frm.common_info.doppler_f),                    .field_name("ulDoppler"));
    m_regmodel.predict_field_value(.value(trig_frm.common_info.pe_disambiguity_f),            .field_name("ulPEDisambiguity"));
    m_regmodel.predict_field_value(.value(trig_frm.common_info.pre_fec_padding_f),            .field_name("ulPreFecPadding"));
    m_regmodel.predict_field_value(.value(trig_frm.common_info.ldpc_extra_symbol_segment_f),  .field_name("ulLDPCExtrSymb"));
    m_regmodel.predict_field_value(.value(trig_frm.common_info.stbc_f),                       .field_name("ulSTBC"));
    m_regmodel.predict_field_value(.value(trig_frm.common_info.num_of_he_ltf_symbols_f),      .field_name("ulNLTFAndMidamble"));
    m_regmodel.predict_field_value(.value(trig_frm.common_info.mumimo_ltf_mode_f),            .field_name("ulMULTFMode"));
    m_regmodel.predict_field_value(.value(trig_frm.common_info.gi_and_ltf_type_f),            .field_name("ulGILTFType"));
    m_regmodel.predict_field_value(.value(trig_frm.common_info.bw_f),                         .field_name("ulBW"));
    m_regmodel.predict_field_value(.value(trig_frm.common_info.length_f),                     .field_name("ulLength"));
    m_regmodel.predict_field_value(.value(trig_frm.common_info.trigger_type_f),               .field_name("ulTriggerType"));

    // RXHETRIGUSERINFOREG
    if (trigger_type_e'(trig_frm.common_info.trigger_type_f) inside { BASIC_TRIGGER,
                                                                      BMF_REPORT_POLL})
      m_regmodel.predict_field_value(.value(trig_frm.trigger_dependent_user_info[0]),           .field_name("tdUserInfoSubfield")); // wrong value
    m_regmodel.predict_field_value(.value(trig_frm.user_info[0].ss_allocation_f[5:3]),        .field_name("ulNSS"));
    m_regmodel.predict_field_value(.value(trig_frm.user_info[0].dcm_f),                       .field_name("ulDCM"));
    m_regmodel.predict_field_value(.value(trig_frm.user_info[0].mcs_f),                       .field_name("ulMCS"));
    m_regmodel.predict_field_value(.value(trig_frm.user_info[0].coding_type_f),               .field_name("ulFECCoding"));
    m_regmodel.predict_field_value(.value(ru_index),                                          .field_name("ulRUSize"));

  endfunction : predict_trigger_register

  //----------------------------------------------------------------------
  // function for calculating TXOP duration from received frame duration
  //----------------------------------------------------------------------
  function bit [6:0] mac_scoreboard_base::calc_txop_duration(ref PPDU_frame frm, ref PPDU_preamble_header txvector, input int user=0);
    int          rx_frm_dur;
    bit [6:0]    txop_duration;
    he_control_s he_ctrl;
    int          resp_dur;
    HESIGB_s     HESIGB;

    if (frm.ppdu_format inside {HE_SU,HE_EXT_SU,HE_MU}) begin
      rx_frm_dur = frm.ampdu_frame[user].get_MAC_duration();
      // get HE control
      he_ctrl = get_ht_control(frm, user);

      resp_dur = ComputeTimeOnAirAX (
                                    .length_i          (txvector.user_header_he[0].he_length_f),
                                    .preType_i         (txvector.format_mod),
                                    .mcsIndex_i        (he_ctrl.ctrl_info_f.ul_mcs_f),
                                    .giType_i          (txvector.gi_type),
                                    .ruType_i          (get_ru_type(he_ctrl.ctrl_info_f.ru_allocation_f)),
                                    .heLtfType_i       (txvector.he_ltf_type),
                                    .numHeLtf_i        (txvector.num_he_ltf),
                                    .dcm_i             (txvector.dcm),
                                    .packetExtension_i (txvector.user_header_he[0].pkt_ext_f),
                                    .heTbLength_i      (he_ctrl.ctrl_info_f.ul_data_symbols_f),
                                    .triggerMethod_i   (txvector.trigger_method),
                                    .doppler_i         (txvector.doppler),
                                    .mma_i             (txvector.midamble_periodicity),
                                    .stbc_i            (txvector.stbc),
                                    .woPreamble_i      (0),
                                    .heSigB_i          (HESIGB),
                                    .debug_i           (0)
                                    );

      // subtract response frame duration and SIFS
      rx_frm_dur = rx_frm_dur - resp_dur - 16;

      if (frm.preamble_header.txop_duration == 7'd127) begin
        txop_duration = 7'd127; // txop_durationurn UNSPECIFIED
      end
      else begin
        if (rx_frm_dur <= 0) begin
          txop_duration = 0;
        end
        else if (rx_frm_dur < 16'd512) begin
          txop_duration[0] = 1'b0;
          txop_duration[6:1] = $floor(rx_frm_dur/8.0);
        end
        else if (rx_frm_dur < 16'd8448) begin
          txop_duration[0] = 1'b1;
          txop_duration[6:1] = $floor((rx_frm_dur-512)/128.0);
        end
        else begin
          // otherwise set to duration 8448
          txop_duration[0] = 1'b1;
          txop_duration[6:1] = $floor((8448-512)/128.0);
        end
      end
    end

    return txop_duration;
  endfunction : calc_txop_duration

`endif //MAC_SCOREBOARD_BASE_SV
