//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef MAC_TX_SCOREBOARD_SV
`define MAC_TX_SCOREBOARD_SV

`uvm_analysis_imp_decl(_mac_tx_sram)
`uvm_analysis_imp_decl(_mac_tx_mac_phy)

class mac_tx_scoreboard extends mac_scoreboard_base;

  `uvm_component_utils(mac_tx_scoreboard)

  uvm_analysis_imp_mac_tx_sram#(sram_seq_item, mac_tx_scoreboard)       sram_export;
  uvm_analysis_imp_mac_tx_mac_phy#(mac_phy_seq_item, mac_tx_scoreboard) mac_phy_export;

  // received items queues
  sram_seq_item           wr_sram_q[$], status_sram_q[$];
  sram_seq_item           sram_item, sram_status;
  mac_phy_seq_item        mac_phy_q[$], mac_phy_item;
  // collected protection and response frames
  PPDU_frame              rts, cts, ack, ba, bar;
  PPDU_frame              tx_frame;
  uvm_comparer            comparer;
  int                     dur_in_tx_frame;
  int                     rate_ctrl_idx; // save index of PT rate control that was used for Tx frame
  int                     num_rts_retries;
  int                     long_retry_limit;
  timestamp_t             tsf_value;


  function new(string name = "mac_tx_scoreboard", uvm_component parent = null);
    super.new(name, parent);

    sram_export    = new("sram_export", this);
    mac_phy_export = new("mac_phy_export", this);
    comparer       = new();
    comparer.show_max = 1000; // show number of mismatches
  endfunction : new

  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);
  endfunction : build_phase

  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase

  task run_phase(uvm_phase phase);
    int user;

    super.run_phase(phase);

    forever begin
      // receive SRAM items that contains Tx PPDU frame
      wait (wr_sram_q.size() != 0);
      sram_item = wr_sram_q.pop_back();

      // wait for NAV frame protection
      check_nav_frame_protection(num_rts_retries);

      // get TSF register value because some management frames will have
      // timestamp included in frame body
      fetch_tsf_reg(tsf_value);

      // collect transmitted frame
      wait (mac_phy_q.size() != 0);
      mac_phy_item = mac_phy_q.pop_back();
      tx_frame = new();
      tx_frame.copy(mac_phy_item.frame);

      // wait for expected ACK frames
      recv_expected_ack_frame();

      check_frame_duration();

      // set proper duration value in TX frame
      if (!sram_item.mac_ctrl_info_2.dont_touch_dur_f) begin
        foreach (sram_item.frame.ampdu_frame[i]) begin
          sram_item.frame.ampdu_frame[i].set_MAC_duration(dur_in_tx_frame);
        end
      end

      // transmitted frame is encrypted when protection bit is set and
      // MAC control 2 info don't encrypt is not set, so we need to
      // decrypt it before comparing it with the frame stored in SRAM
      if (!sram_item.mac_ctrl_info_2.dont_encrypt_f) begin
        tx_frame.decrypt(1);
      end

      // when management frame BEACON, PROBE_RESPONSE and
      // TIMING_ADVERTISEMENT are sent timestamp field is updated
      // from TSF register
      if (!sram_item.mac_ctrl_info_2.dont_touch_tsf_f) begin
        set_frame_timestamp();
      end

      // in case of MU-MIMO secondary users QoS ACK policy is set to NO ACK
      // by MAC HW so this field needs to be overwritten
      if (sram_item.frame.kind == MU_MIMO) begin
        foreach (sram_item.frame.ampdu_frame[i]) begin
          user = sram_item.frame.user_order[i];
          // skip primary user
          if (i != 0)
            sram_item.frame.ampdu_frame[user].set_qos_ack_policy(2'b11);
        end//foreach
      end

      // in trigger frame test case TX frame is checked in immediate response
      // scoreboard
      if (m_cfg.m_sram_cfg.trigger_based_frame != 1) begin
        if (sram_item.frame.compare(tx_frame, comparer) == 0)
          `uvm_error(get_type_name(), $sformatf("SRAM frame and MAC-PHY frame don't match!"))
        else
          `uvm_info(get_type_name(), $sformatf("SRAM frame and MAC-PHY frame OK!"), UVM_LOW)
      end

      // receive SRAM items that contains Tx status information
      // for MU_MIMO there will be status checking per user
      if (sram_item.frame.kind == NDP) begin
        wait (status_sram_q.size() != 0);
        sram_status = status_sram_q.pop_back();
      end
      else begin
        repeat (sram_item.frame.num_of_users) begin
          wait (status_sram_q.size() != 0);
          sram_status = status_sram_q.pop_back();

          check_thd_status();
        end// repeat
      end//else
    end// forever
  endtask : run_phase

  function void report_phase(uvm_phase phase);
    super.report_phase(phase);

    if (wr_sram_q.size() > 0) begin
      `uvm_error(get_type_name(),$sformatf("SRAM queue is not empty, items left:"))
      while (wr_sram_q.size() > 0) begin
        sram_item = wr_sram_q.pop_back();
        `uvm_info(get_type_name(), $sformatf("SRAM item: %s", sram_item.sprint()), UVM_LOW)
      end
    end
  endfunction : report_phase

  //---------------------------------------------------------------------------
  // list of functions for checking
  //---------------------------------------------------------------------------
  extern function void      write_mac_tx_sram(sram_seq_item i);
  extern function void      write_mac_tx_mac_phy(mac_phy_seq_item i);
  extern task               check_nav_frame_protection(ref int rts_retries);
  extern task               recv_expected_ack_frame();
  extern function void      check_frame_duration();
  extern function bit [3:0] expected_which_descriptor(int cnt=0, int size=0);
  extern function void      check_thd_status();
  extern function void      set_frame_timestamp();
  //---------------------------------------------------------------------------

endclass : mac_tx_scoreboard

  //---------------------------------------------------------------------------
  // implementation of checking functions
  //---------------------------------------------------------------------------

  function void mac_tx_scoreboard::write_mac_tx_sram(sram_seq_item i);
    sram_seq_item it;

    it = new();
    it.copy(i);
    `uvm_info(get_type_name(), $sformatf("SRAM item received %s", it.sprint()), UVM_HIGH)

    // only when read from SRAM memory is triggered then checking is performed
    if (it.cmd == WRITE_TX_FRAME)
      wr_sram_q.push_front(it);
    else if (it.cmd == RD_TX_STATUS)
      status_sram_q.push_front(it);
    else
      `uvm_info(get_type_name(), $sformatf("MAC TX scorebord discarded %s SRAM item",it.cmd.name()), UVM_HIGH)
  endfunction : write_mac_tx_sram

  function void mac_tx_scoreboard::write_mac_tx_mac_phy(mac_phy_seq_item i);
    mac_phy_seq_item it;

    it = new();
    it.copy(i);

    // don't analyse Rx TRIGGER frame bacause checking of it is done in immediate
    // response scoreboard
    if (!is_rx_trigger_frame(it.frame)) begin
      `uvm_info(get_type_name(), $sformatf("MAC-PHY item received %s", it.sprint()), UVM_HIGH)
      mac_phy_q.push_front(it);
    end

  endfunction : write_mac_tx_mac_phy

  //---------------------------------------------------------------------------
  // calculate which decriptor should occur in status information
  // @cnt - MPDU count inside A-MPDU
  // #size - number of MDPUs inside A-MPDU
  //---------------------------------------------------------------------------
  function bit [3:0] mac_tx_scoreboard::expected_which_descriptor(int cnt=0, int size=0);
    expect_ack_e   exp_ack = expect_ack_e'(sram_item.mac_ctrl_info_1.expect_ack_f);
    bit [3:0]      ret;

    case (tx_frame.kind)
      SINGLETON, NDP: begin
        // set under BA setup
        ret[3] = (sram_item.mac_ctrl_info_1.expect_ack_f inside {BA,COMPRESSED_BA})? 1:0;
        ret[2] = 1'b0; //A-MPDU
        ret[1:0] = 2'b00;
      end
      AGGREGATED, MU_MIMO: begin
        // set under BA setup
        ret[3] = (sram_item.mac_ctrl_info_1.expect_ack_f inside {BA,COMPRESSED_BA})? 1:0;
        ret[2] = 1'b1; //A-MPDU
        ret[1:0] = (exp_ack == NO_ACK && cnt == 0) ? 2'b00 : //A-THD
                   (exp_ack == NO_ACK && cnt == 1) ? 2'b01 : //first
                   (exp_ack != NO_ACK && cnt == 0) ? 2'b01 : //first
                   (cnt == size-1) ? 2'b11 : //last
                                     2'b10;  //intermediate
      end
    endcase

    return ret;
  endfunction : expected_which_descriptor

  //---------------------------------------------------------------------------
  // check NAV frame protection, there can be up to 4 policy table Rate
  // Control fields, so there can be up to 4 different constructions of
  // protection frames.
  //---------------------------------------------------------------------------
  task mac_tx_scoreboard::check_nav_frame_protection(ref int rts_retries);
    int retry_limit;
    int retry_rc;
    int num_of_rts_tries;
    int rts_cts_finished = 0;
    int retry_cnt = 0;

    // use set retry limit for counting retry attempts of RTS frame
    retry_limit = sram_item.pt_mac_ctrl_info_2.short_retry_limit_f;
    rts_retries = 0;

    foreach (sram_item.pt_rate_ctrl[i]) begin

      // set number of trial which can be done using this rate control
      retry_rc = sram_item.pt_rate_ctrl[i].num_retry_rc_f;
      rate_ctrl_idx = i; // save index value

      // when total number retries of RTS is over the limit exit loop
      if (retry_cnt > retry_limit) break; //foreach

      // determine which NAV protection is used and wait for MAC-PHY item
      case (sram_item.pt_rate_ctrl[i].nav_prot_frm_f)
        NO_PROT: begin
          `uvm_info(get_type_name(), $sformatf("No NAV protection set in policy table rate control"), UVM_HIGH)
          break;
        end// NO_PROT

        SELF_CTS: begin

          // wait for CTS to be received
          wait (mac_phy_q.size() != 0);
          mac_phy_item = mac_phy_q.pop_back();
          cts = new();
          cts.copy(mac_phy_item.frame);

          // check frame type
          if (cts.ampdu_frame[0].mpdu_frame_type[0] != CTS)
            `uvm_error(get_type_name(),
            $sformatf("NAV protection was Self CTS but MAC didn't send CTS (%s)",
            cts.ampdu_frame[0].mpdu_frame_type[0].name()))

          // check format modulation
          if (cts.preamble_header.format_mod != format_mod_e'(sram_item.pt_rate_ctrl[i].format_mode_prot_f))
            `uvm_error(get_type_name(),
            $sformatf("NAV protection Self CTS frame has wrong format modulation, recvd (%d) - exp (%d)",
            cts.preamble_header.format_mod, sram_item.pt_rate_ctrl[i].format_mode_prot_f))

          // check channel bandwidth
          if (cts.preamble_header.ch_bw != sram_item.pt_rate_ctrl[i].bw_prot_f)
            `uvm_error(get_type_name(),
            $sformatf("NAV protection Self CTS frame has wrong channel bandwidth, recvd (%d) - exp (%d)",
            cts.preamble_header.ch_bw, sram_item.pt_rate_ctrl[i].bw_prot_f))

          // check MCS
          if (   cts.preamble_header.user_header[0].mcs_f != sram_item.pt_rate_ctrl[i].mcs_idx_prot_f
              && cts.preamble_header.format_mod inside {HT_GF,HT_MF,VHT})
            `uvm_error(get_type_name(),
            $sformatf("NAV protection Self CTS frame has wrong MCS index, recvd (%d) - exp (%d)",
            cts.preamble_header.user_header[0].mcs_f, sram_item.pt_rate_ctrl[i].mcs_idx_prot_f))

          `uvm_info(get_type_name(), $sformatf("SELF CTS frame protection OK!"), UVM_LOW)
          break;
        end// SELF_CTS

        RTS_CTS, RTS_CTS_QAP: begin

          num_of_rts_tries = 0;

          while (num_of_rts_tries < retry_rc) begin
            // wait for RTS to be received
            wait (mac_phy_q.size() != 0);
            mac_phy_item = mac_phy_q.pop_back();
            rts = new();
            rts.copy(mac_phy_item.frame);

            // check frame type
            if (rts.ampdu_frame[0].mpdu_frame_type[0] != RTS)
              `uvm_error(get_type_name(),
              $sformatf("NAV protection was RTS_CTS but MAC didn't send CTS (%s)",
              rts.ampdu_frame[0].mpdu_frame_type[0].name()))

            // check format modulation
            if (rts.preamble_header.format_mod != format_mod_e'(sram_item.pt_rate_ctrl[i].format_mode_prot_f))
              `uvm_error(get_type_name(),
              $sformatf("NAV protection RTS_CTS frame has wrong format modulation, recvd (%d) - exp (%d)",
              rts.preamble_header.format_mod, sram_item.pt_rate_ctrl[i].format_mode_prot_f))

            // check channel bandwidth
            if (rts.preamble_header.ch_bw != sram_item.pt_rate_ctrl[i].bw_prot_f)
              `uvm_error(get_type_name(),
              $sformatf("NAV protection RTS_CTS frame has wrong channel bandwidth, recvd (%d) - exp (%d)",
              rts.preamble_header.ch_bw, sram_item.pt_rate_ctrl[i].bw_prot_f))

            // check MCS
            if (   rts.preamble_header.user_header[0].mcs_f != sram_item.pt_rate_ctrl[i].mcs_idx_prot_f
                && rts.preamble_header.format_mod inside {HT_GF,HT_MF,VHT})
              `uvm_error(get_type_name(),
              $sformatf("NAV protection RTS_CTS frame has wrong MCS index, recvd (%d) - exp (%d)",
              rts.preamble_header.user_header[0].mcs_f, sram_item.pt_rate_ctrl[i].mcs_idx_prot_f))

            `uvm_info(get_type_name(), $sformatf("RTS/CTS frame protection OK!"), UVM_LOW)

            // wait for CTS to be received
            wait (mac_phy_q.size() != 0);
            mac_phy_item = mac_phy_q.pop_back();
            cts = new();
            cts.copy(mac_phy_item.frame);

            // check frame type
            if (cts.ampdu_frame[0].mpdu_frame_type[0] == RTS) begin
              num_of_rts_tries++;
              retry_cnt++;
              // return taken item to MAC-PHY queue because RTS frame needs to
              // be checked with proper PT Rate Control settings
              mac_phy_q.push_back(mac_phy_item);
            end
            else if (cts.ampdu_frame[0].mpdu_frame_type[0] == CTS) begin
              rts_cts_finished = 1;
              break; // exit while loop because CTS is received
            end
            else begin
              `uvm_error(get_type_name(),
              $sformatf("NAV protection was RTS_CTS but CTS is not received (%s)",
              cts.ampdu_frame[0].mpdu_frame_type[0].name()))
            end
          end// while

          // when protection has finished exit looping PT Rate Controls
          if (rts_cts_finished) break;

        end// RTS_CTS

        STBC: begin
          `uvm_warning(get_type_name(), $sformatf("STBC protection not supported!!!"))
          break;
        end// STBC
      endcase

    end//foreach
  endtask : check_nav_frame_protection

  //---------------------------------------------------------------------------
  // receive expected ACK frames
  //---------------------------------------------------------------------------
  task mac_tx_scoreboard::recv_expected_ack_frame();
    expect_ack_e exp_ack = expect_ack_e'(sram_item.mac_ctrl_info_1.expect_ack_f);
    int          bar_idx;
    int          user_idx, i;
    PPDU_frame   ack_for_bar;
    int          ack_for_bar_dur;

    fetch_register_values();

    if (exp_ack != NO_ACK) begin
      // wait for expected acknowledgement to be received
      wait (mac_phy_q.size() != 0);
      mac_phy_item = mac_phy_q.pop_back();

      case (exp_ack)
        NORMAL_ACK: begin
          ack = new();
          ack.copy(mac_phy_item.frame);
        end
        BA, COMPRESSED_BA: begin
          ba = new();
          ba.copy(mac_phy_item.frame);
        end
      endcase
      `uvm_info(get_type_name(),$sformatf("Received expected %s frame",exp_ack.name()),UVM_LOW)
    end// exp_ack != NO_ACK

    // when A-MPDU is transmitted and no ACK is set, MAC will send BAR that is
    // attached to A-MPDU
    if (tx_frame.kind == AGGREGATED && exp_ack inside {NO_ACK, NORMAL_ACK}) begin
      // wait for expected BAR to be transmitted
      wait (mac_phy_q.size() != 0);
      mac_phy_item = mac_phy_q.pop_back();

      bar = new();
      bar.copy(mac_phy_item.frame);

      // after BAR is transmitted sequence will respond with ACK
      if (exp_ack == NORMAL_ACK) begin
        wait (mac_phy_q.size() != 0);
        mac_phy_item = mac_phy_q.pop_back();

        ack_for_bar = new("ack_for_bar");
        ack_for_bar.copy(mac_phy_item.frame);
        ack_for_bar_dur = get_ppdu_duration(ack_for_bar, get_band5G(mac_ctrl_1.abgn_mode_f));
        ack_for_bar_dur += get_sifs(mac_ctrl_1.abgn_mode_f);
      end
      else begin
        ack_for_bar_dur = 0;
      end

      // find index of attached BAR frame inside A-MPDU
      bar_idx = sram_item.frame.ampdu_frame[0].mpdu_frame.size()-1;
      // set duration to 0 in attached BAR
      sram_item.frame.ampdu_frame[0].mpdu_frame[bar_idx].set_MAC_duration(ack_for_bar_dur);
      sram_item.frame.ampdu_frame[0].mpdu_frame[bar_idx].recalculate_fcs();
      // check BAR content
      if (bar.ampdu_frame[0].mpdu_frame[0].compare(sram_item.frame.ampdu_frame[0].mpdu_frame[bar_idx], comparer) == 0)
        `uvm_error(get_type_name(), $sformatf("Transmitted BAR and attached BAR don't match!"))
      else
        `uvm_info(get_type_name(),$sformatf("MAC transmitted BAR frame is OK"),UVM_LOW)

    end// tx_frame.kind == AGGREGATED

    // when MU_MIMO is transmitted and no ACK is set, MAC will send N number
    // of BARs, N - number of users
    else if (tx_frame.kind == MU_MIMO) begin
      i = 0;
      // where will be N BARs sent
      repeat (tx_frame.num_of_users) begin
        //set user index
        user_idx = sram_item.frame.user_order[i];

        // wait for expected BAR to be transmitted
        wait (mac_phy_q.size() != 0);
        mac_phy_item = mac_phy_q.pop_back();

        bar = new();
        bar.copy(mac_phy_item.frame);

        // find index of attached BAR frame inside A-MPDU
        bar_idx = sram_item.frame.ampdu_frame[user_idx].mpdu_frame.size()-1;
        // set duration to 0 in attached BAR
        sram_item.frame.ampdu_frame[user_idx].mpdu_frame[bar_idx].set_MAC_duration(0);
        sram_item.frame.ampdu_frame[user_idx].mpdu_frame[bar_idx].recalculate_fcs();
        // check BAR content
        if (bar.ampdu_frame[0].mpdu_frame[0].compare(sram_item.frame.ampdu_frame[user_idx].mpdu_frame[bar_idx], comparer) == 0)
          `uvm_error(get_type_name(), $sformatf("Transmitted BAR and attached BAR don't match!"))
        else
          `uvm_info(get_type_name(),$sformatf("MAC transmitted BAR-user%0d frame is OK",user_idx),UVM_LOW)

        i++;
      end//repeat
    end// tx_frame.kind == MU_MIMO
  endtask : recv_expected_ack_frame

  //---------------------------------------------------------------------------
  // function for checking frame duration in protection frames (RTS, CTS) and
  // duration set in TX frame.
  //---------------------------------------------------------------------------
  function void mac_tx_scoreboard::check_frame_duration();
    nav_prot_frm_e prot = nav_prot_frm_e'(sram_item.pt_rate_ctrl[rate_ctrl_idx].nav_prot_frm_f);
    expect_ack_e   exp_ack = expect_ack_e'(sram_item.mac_ctrl_info_1.expect_ack_f);
    int            tx_frame_dur = 0;
    int            txop_limit = 0;
    int            exp_ack_dur = 0;
    int            dur = 0;

    fetch_register_values();
    // calculate duration of expected ack frame duration
    case (exp_ack)
      NO_ACK:            exp_ack_dur = 0;
      NORMAL_ACK:        exp_ack_dur = get_ppdu_duration(ack, get_band5G(mac_ctrl_1.abgn_mode_f));
      BA, COMPRESSED_BA: exp_ack_dur = get_ppdu_duration(ba, get_band5G(mac_ctrl_1.abgn_mode_f));
    endcase
    tx_frame_dur = get_ppdu_duration(tx_frame, get_band5G(mac_ctrl_1.abgn_mode_f));
    // check duration of protection frames
    case (prot)
      NO_PROT: begin
        dur = 0;
      end

      SELF_CTS: begin
        // if SW has set duration field, HW shouldn't change it
        if (sram_item.mac_ctrl_info_2.dont_touch_dur_f) begin
          dur = sram_item.mac_ctrl_info_1.prot_frm_dur_f;
        end
        else begin
          // calculate CTS frame duration field
          dur =   get_sifs(mac_ctrl_1.abgn_mode_f) + tx_frame_dur
               // only if acknowledge is expected add it to duration
                + ((exp_ack != NO_ACK) ? (get_sifs(mac_ctrl_1.abgn_mode_f) + exp_ack_dur) : 0);
        end

        if (dur != cts.ampdu_frame[0].get_MAC_duration())
          `uvm_error(get_type_name(),
          $sformatf("Duration of Self CTS protection frame wrong (%0d) - exp (%0d)",
          cts.ampdu_frame[0].get_MAC_duration(),dur))
        else
          `uvm_info(get_type_name(), $sformatf("Duration of Self CTS protection frame OK!"), UVM_LOW)

      end//SELF_CTS

      RTS_CTS, RTS_CTS_QAP: begin
        // if SW has set duration field, HW shouldn't change it
        if (sram_item.mac_ctrl_info_2.dont_touch_dur_f) begin
          dur = sram_item.mac_ctrl_info_1.prot_frm_dur_f;
        end
        else begin
          // calculate CTS frame duration field
          dur =   get_sifs(mac_ctrl_1.abgn_mode_f) + get_ppdu_duration(cts, get_band5G(mac_ctrl_1.abgn_mode_f))
                + get_sifs(mac_ctrl_1.abgn_mode_f) + tx_frame_dur
                // only if acknowledge is expected add it to duration
                + ((exp_ack != NO_ACK) ? (get_sifs(mac_ctrl_1.abgn_mode_f) + exp_ack_dur) : 0);
        end

        if (dur != rts.ampdu_frame[0].get_MAC_duration())
          `uvm_error(get_type_name(),
          $sformatf("Duration of RTS in RTS_CTS protection frame wrong (%0d) - exp (%0d)",
          rts.ampdu_frame[0].get_MAC_duration(),dur))
        else
          `uvm_info(get_type_name(), $sformatf("Duration of RTS_CTS/RTS_CTS_QAP protection frame OK!"), UVM_LOW)

      end//RTS_CTS/RTS_CTS_QAP

    endcase// prot

    // set expected duration of TX frame
    dur_in_tx_frame = (exp_ack != NO_ACK)
                      ? get_sifs(mac_ctrl_1.abgn_mode_f) + exp_ack_dur
                      : 0;

  endfunction : check_frame_duration

  //---------------------------------------------------------------------------
  // check THD status information field
  //---------------------------------------------------------------------------
  function void mac_tx_scoreboard::check_thd_status();
    int size = sram_status.status_info_list.size();

    foreach (sram_status.status_info_list[i]) begin
      `uvm_info(get_type_name(),
      $sformatf("status_info_list[%0d]= %p",i,sram_status.status_info_list[i]),  UVM_HIGH)

      if (sram_status.status_info_list[i].descr_done_hw_f != 1'b1)
        `uvm_error(get_type_name(), $sformatf("SRAM status-descr_done_hw_f not set!"))

      if (sram_status.status_info_list[i].descr_done_sw_f != 1'b0)
        `uvm_error(get_type_name(), $sformatf("SRAM status-descr_done_sw_f not reset!"))

      if (sram_status.status_info_list[i].descr_done_hw_f)
        if (sram_status.status_info_list[i].which_descr_sw_f != expected_which_descriptor(i,size))
          `uvm_error(get_type_name(),
          $sformatf("Which descriptor wrong: set (%0b) expct (%0b)",
          sram_status.status_info_list[i].which_descr_sw_f,expected_which_descriptor(i,size)))

      if (sram_status.status_info_list[i].trans_bw_f != tx_frame.preamble_header.ch_bw)
        `uvm_error(get_type_name(),
        $sformatf("SRAM status-transmission bandwidth wrong (%0d)-exp (%0d)!",
        sram_status.status_info_list[i].trans_bw_f, tx_frame.preamble_header.ch_bw))

      if (sram_status.status_info_list[i].frm_successful_f != 1'b1)
        `uvm_error(get_type_name(), $sformatf("SRAM status-frm_successful_f not set!"))

      if (   sram_status.status_info_list[i].ba_frm_rcvd_f != 1'b0
          && sram_item.mac_ctrl_info_1.expect_ack_f inside {BA,COMPRESSED_BA})
        `uvm_error(get_type_name(), $sformatf("SRAM status-ba_frm_rcvd_f set by HW!"))

      if (sram_status.status_info_list[i].retry_limit_reached_f != 0)
        `uvm_error(get_type_name(), $sformatf("SRAM status-retry_limit_reached_f not reset!"))

      //TODO: how to check this fields, with what value?
      //if (sram_status.status_info_list[i].num_mpdu_retries_f != )
      //if (sram_status.status_info_list[i].lifetime_expired_f != )
      //-----------------------------------------------------------

      if (sram_status.status_info_list[i].num_rts_retries_f != num_rts_retries)
        `uvm_error(get_type_name(), $sformatf("SRAM status-num_rts_retries_f wrong (%0d)-exp (%0d)!",
        sram_status.status_info_list[i].num_rts_retries_f, num_rts_retries))
    end

    `uvm_info(get_type_name(), $sformatf("THD status information OK!"), UVM_LOW)
  endfunction : check_thd_status

  //---------------------------------------------------------------------------
  // set TSF value from register to timestamp field in management frame body
  //---------------------------------------------------------------------------
  function void mac_tx_scoreboard::set_frame_timestamp();
    `uvm_info(get_type_name(),$sformatf("set_frame_timestamp: TSF value %0h",tsf_value),UVM_HIGH)

    if (sram_item.frame.kind != NDP) begin
      if (sram_item.frame.ampdu_frame[0].mpdu_frame_type[0] inside {BEACON,PROBE_RESPONSE,TIMING_ADVERTISEMENT}) begin
        for (int i=0; i<8; i++) begin
          sram_item.frame.ampdu_frame[0].mpdu_frame[0].frame_body[i] = tsf_value[i*8 +: 8];
        end
        // 3 LSB bytes of TSF changes from time it gets fetched to time when frame
        // is transmitted so to avoid mismatch LSB byte is taken from trasmitted
        // frame
        sram_item.frame.ampdu_frame[0].mpdu_frame[0].frame_body[0] = tx_frame.ampdu_frame[0].mpdu_frame[0].frame_body[0];
        sram_item.frame.ampdu_frame[0].mpdu_frame[0].frame_body[1] = tx_frame.ampdu_frame[0].mpdu_frame[0].frame_body[1];
        sram_item.frame.ampdu_frame[0].mpdu_frame[0].frame_body[2] = tx_frame.ampdu_frame[0].mpdu_frame[0].frame_body[2];
        // FCS has to be updated after frame body is changed
        sram_item.frame.ampdu_frame[0].mpdu_frame[0].recalculate_fcs();
      end
    end//NDP
  endfunction : set_frame_timestamp

`endif// MAC_TX_SCOREBOARD_SV
