//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef _MODEM_RX_SCOREBOARD_SV_
`define _MODEM_RX_SCOREBOARD_SV_

`uvm_analysis_imp_decl(_modem_rx_mac_phy)

class modem_rx_scoreboard extends modem_scoreboard_base;

  `uvm_component_utils(modem_rx_scoreboard)

  uvm_analysis_imp_modem_rx_mac_phy#(mac_phy_seq_item, modem_rx_scoreboard)  mac_phy_export;
  uvm_tlm_analysis_fifo            #(modem_stream_seq_item#(`DEINTERWIDTH))  deint_fifo;
  uvm_tlm_analysis_fifo            #(modem_stream_seq_item#(`EQUWIDTH))      equ_fifo;
  uvm_tlm_analysis_fifo            #(modem_stream_seq_item#(`EQUWIDTH))      equ1_fifo;
  uvm_tlm_analysis_fifo            #(modem_stream_seq_item#(`LDPCWIDTH))     ldpc_fifo;
  uvm_tlm_analysis_fifo            #(modem_stream_seq_item#(`LDPCDATAWIDTH)) ldpc_data_fifo;
  uvm_tlm_analysis_fifo            #(mem_seq_item)                           fftin_fifo;
  uvm_tlm_analysis_fifo            #(mem_seq_item)                           fftout_fifo;
  uvm_tlm_analysis_fifo            #(modem_stream_seq_item#(`VITERBIWIDTH))  viterbi_in_fifo;

  //--------------------------------------------------------------------
  // Queues of items and single items
  //--------------------------------------------------------------------
  mac_phy_seq_item                        mac_phy_item, mac_phy_queue[$];
  modem_stream_seq_item#(`DEINTERWIDTH)   deint_out_item;
  modem_stream_seq_item#(`EQUWIDTH)       equ_out_item;
  modem_stream_seq_item#(`EQUWIDTH)       equ1_out_item;
  modem_stream_seq_item#(`LDPCWIDTH)      ldpc_out_item;
  modem_stream_seq_item#(`LDPCDATAWIDTH)  ldpc_data_out_item;
  modem_stream_seq_item#(`VITERBIWIDTH)   viterbi_in_item, viterbi_queue[$];
  // Matlab referent data
  soft_bits_array       DeintRefData;
  soft_bits_array       DeintNonCompRefData;
  ldpc_data_array       LDPCOutputData;
  ldpc_array            LDPCRefData;
  dcm_data_array        DCMInRefData;
  dcm_data_array        DCMOutRefData;
  viterbi_array         ViterbiRefData;
  FFT_data_s            FFTRefIn[];
  FFT_data_s            FFTRefOut[];
`ifdef DEBUG_SAMPLES
  // used when debug is enabled
  FFT_data_s            fft_in_rtl;
  FFT_data_s            fft_in_matlab;
  FFT_data_s            fft_out_rtl;
  FFT_data_s            fft_out_matlab;
`endif

  // Rx parameters from Matlab
  rxparameters_s        rx_parameters;
  event                 matlab_data_received;

  // Register fields
  bit [31:0]            primary_channel;

  // Flags that are used to check comparison
  int                   frame_cnt = 0;
  int                   compressed_sb_comparison_done = 0;
  int                   noncompressed_sb_comparison_done = 0;
  bit                   disable_check;
  bit                   skip_sigb_check = 0;
  int                   non_comp_sigb_cnt = 0;
  int                   comp_sigb_cnt = 0;
  bit                   frame_rcv_mac_phy = 0;

  int                   viterbi_in_error_cnt = 0;
  int                   receive_stbc_data = 0;
  // flag indicating frame is HT-MF MCS32 which will be checked only on MPIF boundary
  bit                   ppdu_ht_mf_mcs32 = 0;

  //--------------------------------------------------------------------
  // Constructor
  //--------------------------------------------------------------------
  function new(string name = "modem_rx_scoreboard", uvm_component parent = null);
    super.new(name, parent);
    mac_phy_export  = new("mac_phy_export", this);
    deint_fifo      = new("deint_fifo", this);
    equ_fifo        = new("equ_fifo", this);
    equ1_fifo       = new("equ1_fifo", this);
    ldpc_fifo       = new("ldpc_fifo", this);
    ldpc_data_fifo  = new("ldpc_data_fifo", this);
    fftin_fifo      = new("fftin_fifo", this);
    fftout_fifo     = new("fftout_fifo", this);
    viterbi_in_fifo = new("viterbi_in_fifo", this);
  endfunction : new

  //--------------------------------------------------------------------
  // UVM phasing - build phase
  //--------------------------------------------------------------------
  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);
  endfunction : build_phase

  //--------------------------------------------------------------------
  // UVM phasing - connect phase
  //--------------------------------------------------------------------
  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase

  //--------------------------------------------------------------------
  // UVM phasing - run phase
  //--------------------------------------------------------------------
  task run_phase(uvm_phase phase);
    super.run_phase(phase);

`ifndef RX_INTERNAL_CHECK
    disable_check = 1;
`else
    // when AGC is not bypassed intermediate checkpoints are disabled
    // because they are not bit-exact
    if (m_cfg.insert_modem_rx_error)
      disable_check = 1;
    else if ((`AGC_ON == 0) || (m_cfg.m_radio_cfg.m_rui_cfg.agcBypass == 1))
      disable_check = 0;
    else
      disable_check = 1;
`endif//RX_INTERNAL_CHECK

    fork
      if (m_cfg.check_ppdu_mpif)
        get_input_data();
      // don't compare TimeDomain samples when FreqDomain is tested
      if (m_cfg.has_fft_in_agent && m_cfg.m_fft_in_cfg.is_active == UVM_PASSIVE && !disable_check && !m_modem_cfg.sig_error_injection)
        compare_time_domain_out();
      if (m_cfg.has_fft_out_agent && !disable_check && !m_modem_cfg.sig_error_injection)
        compare_fft_out();
      if (m_cfg.has_deint_agent && !disable_check && !m_modem_cfg.sig_error_injection)
        compare_compressed_softbits();
      if (m_cfg.has_equ_agent && !disable_check && !m_modem_cfg.sig_error_injection)
        compare_noncompressed_softbits();
      if (m_cfg.has_ldpc_agent && !disable_check && !m_modem_cfg.sig_error_injection)
        compare_ldpc_deint_samples();
      if (m_cfg.has_ldpc_data_agent && !disable_check && !m_modem_cfg.sig_error_injection)
        compare_ldpc_out_samples();
      if (m_cfg.has_viterbi_in_agent && !disable_check && !m_modem_cfg.sig_error_injection)
        compare_viterbi_in_data();
      get_matlab_data();
      // When there is a Format violation detected, flush
      // all the FIFO's
      forever begin
        @(posedge m_cfg.m_bus_monitor_cfg.frame_violation);
        frame_rcv_mac_phy = 1;
        flush_fifos();
      end
    join_none;

  endtask : run_phase

  //--------------------------------------------------------------------
  // UVM phasing - report phase
  //--------------------------------------------------------------------
  function void report_phase(uvm_phase phase);
    super.report_phase(phase);
    check_status();
  endfunction : report_phase

  //---------------------------------------------------------------------------
  // list of functions for checking
  //---------------------------------------------------------------------------
  extern function void check_phy_rx_error(PPDU_frame frame);
  extern function int  offset_calculation(int idx, rxparameters_s rx_param, int symb_cnt, int sigb_cnt);
  extern function void write_modem_rx_mac_phy(mac_phy_seq_item i);
  extern function void check_status();
  extern function void flush_fifos();
  extern task          get_input_data();
  extern task          compare_ppdu_frame();
  extern task          compare_time_domain_out();
  extern task          get_matlab_data();
  extern task          compare_viterbi_in_data();
  extern task          compare_fft_out();
  extern task          compare_compressed_softbits();
  extern task          compare_noncompressed_softbits();
  extern task          compare_ldpc_deint_samples();
  extern task          compare_ldpc_out_samples();
  extern task          compare_ldpc_samples_per_symbol(input int idx, input int starting_index);
  extern task          compare_sc_symb_per_ss(input int idx, input int symb_cnt, input rxparameters_s rx_param);
  extern task          compare_noncomp_sc_symb_per_ss(input int idx, input int symb_cnt, input rxparameters_s rx_param);
  extern task          compare_rxtd_status_reg();
  extern task          compare_fdo_samples();
  extern task          compare_tbo_samples();
  //---------------------------------------------------------------------------

endclass : modem_rx_scoreboard

  //====================================================================
  // Helper functions
  //====================================================================
  //--------------------------------------------------------------------
  // Check if rx_err assertion was expected or not...
  //--------------------------------------------------------------------
  function void modem_rx_scoreboard::check_phy_rx_error(PPDU_frame frame);

    if (m_modem_cfg.sig_error_injection) begin
      `uvm_info(get_type_name(),
      $sformatf("Received frame with rx_err flag set, SIG err injected!"),UVM_LOW)
    end
    else if (m_modem_cfg.golden_frame.ppdu_format == HE_MU && m_modem_cfg.hesigbErrorcode != 0) begin
      `uvm_info(get_type_name(),
      $sformatf("Received frame with rx_err flag set, HESIGB err detected!"),UVM_LOW)
      flush_fifos();
    end
    else if (m_modem_cfg.golden_frame.ppdu_format == VHT && m_modem_cfg.vhtsigbErrorcode != 0) begin
      `uvm_info(get_type_name(),
      $sformatf("Received frame with rx_err flag set, VHTSIGB err detected!"),UVM_LOW)
    end
    else if (rx_parameters.format_mod_f == -1) begin
      `uvm_info(get_type_name(),
      $sformatf("Received frame with rx_err flag set, Matlab decoding wrong"),UVM_LOW)
      flush_fifos();
    end
    else begin
      `uvm_error(get_type_name(),$sformatf("Rx_err should not occur without SIG err injection!"))
    end

  endfunction: check_phy_rx_error


 //--------------------------------------------------------------------
  // when a mac_phy_seq_item is available, calls Matlab to get
  // golden data and starts comparison
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::get_input_data();

    forever begin
      wait (mac_phy_queue.size()>0); //wait until there is data in the queue
      mac_phy_item = mac_phy_queue.pop_back();
      if (mac_phy_item.rx_err === 'b0 && !m_modem_cfg.sig_error_injection) begin
        compare_ppdu_frame();
      end else begin
        // PHY asserted MAC-PHY error or RX error line. Check if this was expected
        check_phy_rx_error(mac_phy_item.frame);
        frame_rcv_mac_phy = 1;
      end
    end
  endtask: get_input_data

  //--------------------------------------------------------------------
  // Compares referent frame and MAC-PHY IF frame
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::compare_ppdu_frame();
    uvm_comparer comparer;
    bit [1:0]    userpos;
    bit          dont_compare;

    comparer = new;
    comparer.show_max = 100; // print more than 1 mismatch in PPDU frames
    comparer.check_type = 0;

    // when MU-MIMO Rx only one user will be received, user with matching ID
    // (user position) set in register
    if (m_modem_cfg.fcsOK == -1) begin
      dont_compare = 1;
    end
    else if (m_modem_cfg.golden_frame.kind == MU_MIMO) begin
      userpos = get_mu_userpos(m_modem_cfg.golden_frame);
      // if the register user position is equal with one set in golden frame
      // compare two frames
      if (m_modem_cfg.golden_frame.preamble_header.mu_mimo_userid == userpos)
        dont_compare = 0;
      else
        dont_compare = 1;
    end else begin
      dont_compare = 0;
    end
    // during reception it can occur that some MPDUs are received with
    // errors so we must indicate to compare function this information.
    // NOTE: 'physical' variable is used as indication of fcsOK status
    comparer.physical = (m_modem_cfg.fcsOK == 1) ? 1 : 0;
`ifndef STANDALONE_PHY
    // if frame was encrypted then comparing with golden frame needs to
    // be performed on decrypted frame because inside MPDU there are
    // 2 frame body arrays one encrypted one normal
    mac_phy_item.frame.decrypt(0);
`endif//STANDALONE_PHY

`ifdef RX_INTERNAL_CHECK
    // Compare the FDO samples, after FDDone is generated, only if
    // AGC is bypassed
    if(((`AGC_ON == 0) || (m_cfg.m_radio_cfg.m_rui_cfg.agcBypass == 1)) && m_modem_cfg.fcsOK != -1 && !m_modem_cfg.sig_error_injection) begin
      compare_fdo_samples();
      if (m_modem_cfg.golden_frame.kind != NDP)
        compare_tbo_samples();
    end
`endif

    // compare two PPDU frames only if Matlab didn't provide payload
    if (dont_compare) begin
      `uvm_info(get_type_name(),$sformatf("MU-MIMO frame will not be compared because it's not for this user!"),UVM_LOW)
    end else if (m_modem_cfg.golden_frame.compare(mac_phy_item.frame, comparer) == 0) begin
      `uvm_error(get_type_name(), $sformatf("Received and expected frames differ"))
    end else begin
      `uvm_info(get_type_name(), $sformatf("Received and expected frames are the same"), UVM_LOW)
    end
    frame_rcv_mac_phy = 1;
    flush_fifos();
  endtask: compare_ppdu_frame

  //--------------------------------------------------------------------
  // This task is only used, to get the parameters from Matlab
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::get_matlab_data();
    int    fhandle;

    forever begin
      @(m_cfg.new_frame_mdm_exe);
      flush_fifos();
      // If there is SIG error injection, no data
      // should be fetched from Matlab
      if (m_modem_cfg.sig_error_injection)
        continue;
        
      frame_rcv_mac_phy = 0;
      frame_cnt++;
      // get Matlab rx parameters
      rx_parameters = m_modem_cfg.get_rxparams();
      // determine is PPDU HT-MF MCS32
      if (ppdu_ht_mf_mcs32)
        flush_fifos(); // because there were no checks done, flush fifo's

      ppdu_ht_mf_mcs32 = 0;
      if (m_modem_cfg.golden_frame.ppdu_format == HT_MF)
        ppdu_ht_mf_mcs32 = (m_modem_cfg.golden_frame.preamble_header.user_header[0].mcs_f == 7'd32) ? 1 : 0;

      // For 40 MHz frames, there are twice as much samples
      // on the HE SIGB
      `ifndef RW_NX_CHBW20
        if (rx_parameters.cbw_f == 1)
          rx_parameters.he_sigb_f = rx_parameters.he_sigb_f*2;
      `endif

      DeintRefData = m_modem_cfg.get_soft_bits(.compressed(1));
      DeintNonCompRefData = m_modem_cfg.get_soft_bits(.compressed(0));
      DCMInRefData = m_modem_cfg.get_dcm_input_samples();
      DCMOutRefData = m_modem_cfg.get_dcm_output_samples();
      ViterbiRefData = m_modem_cfg.get_viterbi_samples();
      LDPCRefData = m_modem_cfg.get_ldpc_samples();
      LDPCOutputData = m_modem_cfg.get_ldpc_data_samples();

`ifdef DEBUG_SAMPLES
      fhandle = $fopen("compressed_soft_bit_samples.txt","w");
      $fwrite(fhandle, $sformatf("\tREAL\t\t\t\tINDSC\t\tSYMB\t\tINDSYMB\n"));
      foreach (DeintRefData[i]) begin
        $fwrite(fhandle, $sformatf("%7d\t%7d\t%6d\t%6d\n",
        DeintRefData[i].real_comp_f,
        DeintRefData[i].indsc_f,
        DeintRefData[i].symb_f,
        DeintRefData[i].indsymb_f));
      end
      $fclose(fhandle);

      fhandle = $fopen("dcm_input_samples.txt");
      $fwrite(fhandle, $sformatf("REAL\tSTREAM\tBIT\tSEG\t\tSYMB\n"));
      foreach (DCMInRefData[i]) begin
        $fwrite(fhandle, $sformatf("%5d\t%5d\t%5d\t%5d\t%5d\n",
        DCMInRefData[i].real_f,
        DCMInRefData[i].stream_f,
        DCMInRefData[i].bit_f,
        DCMInRefData[i].seg_f,
        DCMInRefData[i].symb_f));
      end
      $fclose(fhandle);

      fhandle = $fopen("dcm_output_samples.txt");
      $fwrite(fhandle, $sformatf("REAL\tSTREAM\tBIT\tSEG\t\tSYMB\n"));
      foreach (DCMOutRefData[i]) begin
        $fwrite(fhandle, $sformatf("%5d\t%5d\t%5d\t%5d\t%5d\n",
        DCMOutRefData[i].real_f,
        DCMOutRefData[i].stream_f,
        DCMOutRefData[i].bit_f,
        DCMOutRefData[i].seg_f,
        DCMOutRefData[i].symb_f));
      end
      $fclose(fhandle);

      fhandle = $fopen("noncompressed_soft_bit_samples.txt");
      $fwrite(fhandle, $sformatf("\tREAL\t\tINDSC\t\tSYMB\t\tINDSYMB\n"));
      foreach (DeintNonCompRefData[i]) begin
        $fwrite(fhandle, $sformatf("%7d\t%7d\t%6d\t%6d\n",
        DeintNonCompRefData[i].real_f,
        DeintNonCompRefData[i].indsc_f,
        DeintNonCompRefData[i].symb_f,
        DeintNonCompRefData[i].indsymb_f));
      end
      $fclose(fhandle);

      fhandle = $fopen("viterbi_samples.txt","w");
      $fwrite(fhandle, $sformatf("#REAL\tBCC_ES\tINDSYMB\tRXFIELD\tContentChanIdx\n"));
      foreach (ViterbiRefData[i])
      begin
        $fwrite(fhandle, $sformatf("%5d\t%5d\t%5d\t%5d\t%5d\n",
        ViterbiRefData[i].real_f,
        ViterbiRefData[i].bcc_es_f,
        ViterbiRefData[i].indsymb_f,
        ViterbiRefData[i].rxfield_f,
        ViterbiRefData[i].contentchanidx_f));
      end
      $fclose(fhandle);

      fhandle = $fopen("ldpc_samples.txt","w");
      $fwrite(fhandle, $sformatf("#REAL\tBCC_ES\tINDSYMB\tRXFIELD\n"));
      foreach (LDPCRefData[i])
        $fwrite(fhandle, $sformatf("%5d\t%5d\t%5d\t%5d\n",
        LDPCRefData[i].real_f,
        LDPCRefData[i].bcc_es_f,
        LDPCRefData[i].indsymb_f,
        LDPCRefData[i].rxfield_f));
      $fclose(fhandle);

      fhandle = $fopen("ldpc_data_samples.txt","w");
      $fwrite(fhandle, $sformatf("#REAL\tINDSYMB\tRXFIELD\n"));
      foreach (LDPCOutputData[i])
        $fwrite(fhandle, $sformatf("%5d\t%5d\t%5d\n",
        LDPCOutputData[i].real_f,
        LDPCOutputData[i].indsymb_f,
        LDPCOutputData[i].rxfield_f));
      $fclose(fhandle);
`endif

      primary_channel = m_regmodel.get_mirrored_reg_value("PRIMARYIND");

      // check whether the selected user is the longest
      if (m_modem_cfg.golden_frame.kind == MU_MIMO) begin
        if (m_modem_cfg.golden_frame.ppdu_format == VHT) begin
          foreach (m_modem_cfg.golden_frame.preamble_header.user_header[i]) begin
            if (m_modem_cfg.golden_frame.preamble_header.user_header[i].ht_length_f > m_modem_cfg.golden_frame.preamble_header.user_header[m_modem_cfg.golden_frame.preamble_header.mu_mimo_userid].ht_length_f) begin
              `uvm_info(get_type_name(),$sformatf("The received user is not the longest - VHT!!!!!"), UVM_DEBUG)
              break;
            end
          end
        end
        else if (m_modem_cfg.golden_frame.ppdu_format == HE_MU) begin
          foreach (m_modem_cfg.golden_frame.preamble_header.user_header_he[i]) begin
            if (m_modem_cfg.golden_frame.preamble_header.user_header_he[i].he_length_f > m_modem_cfg.golden_frame.preamble_header.user_header_he[m_modem_cfg.golden_frame.preamble_header.mu_mimo_userid].he_length_f) begin
              `uvm_info(get_type_name(),$sformatf("The received user is not the longest - HE_MU!!!!!"), UVM_DEBUG)
              break;
            end
          end
        end
      end

      // trigger event that MDM data is received
      -> matlab_data_received;
    end

  endtask : get_matlab_data

  //--------------------------------------------------------------------
  // compare data from time domain with FFT input referent data
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::compare_time_domain_out();
    mem_seq_item  item;
    FFT_data_s    rtl_sample;
    int           idx;
    int           matlab_idx;
    int           fhandle;
    bit           err_occured;
    bit           secondary;
    int           offset;
    int           hesigb_dec; // decrement symbol counter when HESIGB is present to be aligned with matlab samples
    bit           hesigb_sec_checked; // HESIGB secondary checked

    forever begin
      fftin_fifo.get(item);
      err_occured = 0;
      secondary = 0;
      offset = 0;
      hesigb_sec_checked = 0;
      if (ppdu_ht_mf_mcs32) continue;
      // find first element to see if this is first symbol of frame
      // if yes then get matlab referent data
      if (item.fft_data.first(idx)) begin
        if (item.fft_data[idx].symb_idx_f == 0) begin
          FFTRefIn = m_modem_cfg.get_fft_data_in();
          matlab_idx = 0;
          hesigb_dec = 0;

          if (rx_parameters.format_mod_f == -1) begin
          // If the returned format mod is -1, skip checking
            continue;
          end

          compare_rxtd_status_reg();

`ifdef DEBUG_SAMPLES
          // dump Matlab samples for FFT input
          fhandle = $fopen("fft_in_matlab_samples.txt","w");
          $fwrite(fhandle, $sformatf("#REAL\tIMAG\tINDSYMB\tINDSC\tNSYMB\n"));
          foreach (FFTRefIn[i])
            $fwrite(fhandle, $sformatf("%5d\t%5d\t%5d\t%5d\t%5d\n",
            FFTRefIn[i].re_f,
            FFTRefIn[i].im_f,
            FFTRefIn[i].symb_idx_f,
            FFTRefIn[i].index_f,
            FFTRefIn[i].nsymb_f));
          $fclose(fhandle);
`endif//DEBUG_SAMPLES
        end
        rtl_sample = item.fft_data[idx];
      end
      else begin
        `uvm_error(get_type_name(),"RTL FFT in array empty")
      end//first
      // decrement HESIGB symbol counter
      rtl_sample.symb_idx_f = rtl_sample.symb_idx_f - hesigb_dec;
      // start comparing samples of one symbol,
      // loop while symbol index match
      while (FFTRefIn[matlab_idx].symb_idx_f == rtl_sample.symb_idx_f) begin
        // check if collected array has matching SC index
        if (item.fft_data.exists(FFTRefIn[matlab_idx].index_f)) begin
          idx = FFTRefIn[matlab_idx].index_f;
          rtl_sample = item.fft_data[idx];
          // decrement HESIGB symbol counter
          rtl_sample.symb_idx_f = rtl_sample.symb_idx_f - hesigb_dec;

          // compare real and imag parts
          if (   rtl_sample.re_f !== FFTRefIn[matlab_idx].re_f
              || rtl_sample.im_f !== FFTRefIn[matlab_idx].im_f) begin

            `uvm_error(get_type_name(),
            $sformatf("FFT INPUT samples mismatch RTL (%s) Matlab (%s)",
            FFT2string(rtl_sample), FFT2string(FFTRefIn[matlab_idx])))
            err_occured = 1;
          end
          else begin
            `uvm_info(get_type_name(),
            $sformatf("FFT INPUT samples match RTL (%s) Matlab (%s)",
            FFT2string(rtl_sample), FFT2string(FFTRefIn[matlab_idx])),UVM_DEBUG)
          end

`ifdef DEBUG_SAMPLES
          fft_in_rtl    = rtl_sample;
          fft_in_matlab = FFTRefIn[matlab_idx];
          #1ps; //consume time to signals are visible in waves
`endif//DEBUG_SAMPLES
        end// exists
        //-------------------------------------------------------------
        // SECONDARY CHANNEL
        //-------------------------------------------------------------
        // check if collected array has matching SC index
        if (   item.fft_data_sec.exists(FFTRefIn[matlab_idx].index_f)
            // secondary should be compared on L-LTF, L-SIG symbols and HESIGB
            // when frame is CHBW40
            && (   FFTRefIn[matlab_idx].rxfield_f inside {L_LTF, L_SIG}
                || (FFTRefIn[matlab_idx].rxfield_f == HE_SIGB && rx_parameters.cbw_f == 2'b01))
           ) begin
          secondary = 1;
          // matlab HESIGB are organized: NUM_SIGB * primary + NUM_SIGB * secondary samples
          if (FFTRefIn[matlab_idx].rxfield_f == HE_SIGB && rx_parameters.cbw_f == 2'b01) begin
            offset = (rx_parameters.he_sigb_f*64)/2; // second half are secondary samples
            hesigb_sec_checked = 1;
          end
          else begin
            offset = 64;                             // if not HESIGB
          end

          idx = FFTRefIn[matlab_idx + offset].index_f;
          rtl_sample = item.fft_data_sec[idx];
          // decrement HESIGB symbol counter
          rtl_sample.symb_idx_f = rtl_sample.symb_idx_f - hesigb_dec;

          // compare real and imag parts
          if (   rtl_sample.re_f !== FFTRefIn[matlab_idx + offset].re_f
              || rtl_sample.im_f !== FFTRefIn[matlab_idx + offset].im_f) begin

            `uvm_error(get_type_name(),
            $sformatf("FFT INPUT samples mismatch RTL (SECONDARY) (%s) Matlab (%s)",
            FFT2string(rtl_sample), FFT2string(FFTRefIn[matlab_idx + offset])))
            err_occured = 1;
          end
          else begin
            `uvm_info(get_type_name(),
            $sformatf("FFT INPUT samples match RTL (SECONDARY) (%s) Matlab (%s)",
            FFT2string(rtl_sample), FFT2string(FFTRefIn[matlab_idx + offset])),UVM_DEBUG)
          end
        end// exists

        matlab_idx++;
      end// while

      // on top tb symbol counter will be incremented by 2 on HESIGB but if
      // frame is CHBW20 then there will not be secondary part so symbol
      // counter doesn't have propper value, so it must be decremented for
      // other symbol checking to be aglined with matlab symbol counter
      if (FFTRefIn[matlab_idx-1].rxfield_f == HE_SIGB && rx_parameters.cbw_f == 2'b00) begin
        hesigb_dec++;
      end
      else if (hesigb_sec_checked) begin
        // finished checking of HESIGB symbols, set offset to skip HESIGB
        // secondary samples and no more decrement of symbol counter
        if (FFTRefIn[matlab_idx+offset].rxfield_f == HE_LTF) begin
          offset = (rx_parameters.he_sigb_f*64)/2;
          hesigb_dec = 0;
        end
        else begin
          offset = 0;
          hesigb_dec++;
        end
      end

      if (!err_occured) begin
        `uvm_info(get_type_name(),
        $sformatf("FFT INPUT samples OK (symbol %0d)!",rtl_sample.symb_idx_f),UVM_HIGH)
      end
      if (!err_occured && secondary) begin
        `uvm_info(get_type_name(),
        $sformatf("FFT INPUT samples OK, secondary (symbol %0d)!",rtl_sample.symb_idx_f),UVM_HIGH)
      end
      // in case when secondary samples were checked we need to skipp offset
      // values when indexing
      if (secondary)
        matlab_idx = matlab_idx + offset;
    end// forever
  endtask : compare_time_domain_out

  //--------------------------------------------------------------------
  // compare FFT output samples
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::compare_fft_out();
    mem_seq_item  item;
    FFT_data_s    rtl_sample;
    int           idx;
    int           matlab_idx;
    int           first_idx_symb; // first index of symbol in matlab ref data
    int           fhandle;
    bit           err_occured;
    bit           secondary;
    int           offset;
    int           hesigb_dec; // decrement symbol counter when HESIGB is present to be aligned with matlab samples
    bit           hesigb_sec_checked; // HESIGB secondary checked

    forever begin
      fftout_fifo.get(item);
      err_occured = 0;
      secondary = 0;
      offset = 0;
      hesigb_sec_checked = 0;
      if (ppdu_ht_mf_mcs32) continue;
      // find first element to see if this is first symbol of frame
      // if yes then get matlab referent data
      if (item.fft_data.first(idx)) begin
        if (item.fft_data[idx].symb_idx_f == 0) begin

          FFTRefOut = m_modem_cfg.get_fft_data_out();
          matlab_idx = 0;
          hesigb_dec = 0;

          if (rx_parameters.format_mod_f == -1) begin
          // If the returned format mod is -1, skip checking
            continue;
          end

`ifdef DEBUG_SAMPLES
          // dump Matlab samples for FFT input
          fhandle = $fopen("fft_out_matlab_samples.txt","w");
          $fwrite(fhandle, $sformatf("#REAL\tIMAG\tINDSYMB\tINDSC\tNSYMB\n"));
          foreach (FFTRefOut[i])
            $fwrite(fhandle, $sformatf("%5d\t%5d\t%5d\t%5d\t%5d\n",
            FFTRefOut[i].re_f,
            FFTRefOut[i].im_f,
            FFTRefOut[i].symb_idx_f,
            FFTRefOut[i].index_f,
            FFTRefOut[i].nsymb_f));
          $fclose(fhandle);
`endif//DEBUG_SAMPLES
        end
        // if we got the samples from symbol count checked on previous iteration, we need
        // to reset matlab index to starting value in order to check correct
        else if (item.fft_data[idx].symb_idx_f == rtl_sample.symb_idx_f) begin
          matlab_idx = first_idx_symb;
        end
        rtl_sample = item.fft_data[idx];
        // adjust symbol index for HESIGB
        if (   rx_parameters.format_mod_f == 4'd6
            && rtl_sample.symb_idx_f > 7
            && rtl_sample.symb_idx_f < (7+rx_parameters.he_sigb_f)
           ) begin
           rtl_sample.symb_idx_f = rtl_sample.symb_idx_f - hesigb_dec;
        end
      end
      // secondary samples
      else if (item.fft_data_sec.first(idx)) begin
        secondary = 1;
        rtl_sample = item.fft_data_sec[idx];

        if (   rx_parameters.format_mod_f == 4'd6
            && rtl_sample.symb_idx_f >= 7
            && rtl_sample.symb_idx_f < (7+rx_parameters.he_sigb_f)
           ) begin
          offset = (rx_parameters.he_sigb_f*64)/2;
          // adjust symbol index for HESIGB
          rtl_sample.symb_idx_f = rtl_sample.symb_idx_f + (rx_parameters.he_sigb_f/2) - hesigb_dec;
          hesigb_sec_checked = 1;
        end
        else begin
          offset = 0;
          // adjust symbol index for L-LTF and L-SIG
          rtl_sample.symb_idx_f = rtl_sample.symb_idx_f + 1;
          hesigb_sec_checked = 0;
        end
        matlab_idx = matlab_idx + offset; // add offset
      end
      else begin
        `uvm_error(get_type_name(),"RTL FFT out array empty")
      end//first
      // set value of first index of referent data of symbol
      first_idx_symb = (hesigb_sec_checked) ? (matlab_idx - offset) + 64 : matlab_idx;

      // start comparing samples of one symbol,
      // loop while symbol index match
      while (FFTRefOut[matlab_idx].symb_idx_f == rtl_sample.symb_idx_f) begin
        // check if collected array has matching SC index
        if (   item.fft_data.exists(FFTRefOut[matlab_idx].index_f)
            || item.fft_data_sec.exists(FFTRefOut[matlab_idx].index_f)) begin

          idx = FFTRefOut[matlab_idx].index_f;
          rtl_sample = (secondary) ? item.fft_data_sec[idx] : item.fft_data[idx];
          // adjust symbol index for HESIGB
          if (hesigb_sec_checked)
            rtl_sample.symb_idx_f = rtl_sample.symb_idx_f + (rx_parameters.he_sigb_f/2) - hesigb_dec;
          else if (secondary)
            rtl_sample.symb_idx_f = rtl_sample.symb_idx_f + 1;

          // compare real and imag parts
          if (   rtl_sample.re_f !== FFTRefOut[matlab_idx].re_f
              || rtl_sample.im_f !== FFTRefOut[matlab_idx].im_f) begin

            `uvm_error(get_type_name(),
            $sformatf("FFT OUTPUT (%s) samples mismatch RTL (%s) Matlab (%s)",
            (secondary)?"secondary":"primary", FFT2string(rtl_sample), FFT2string(FFTRefOut[matlab_idx])))
            err_occured = 1;
          end
          else begin
            `uvm_info(get_type_name(),
            $sformatf("FFT OUTPUT (%s) samples match RTL (%s) Matlab (%s)",
            (secondary)?"secondary":"primary", FFT2string(rtl_sample), FFT2string(FFTRefOut[matlab_idx])),UVM_DEBUG)
          end

`ifdef DEBUG_SAMPLES
          fft_out_rtl    = rtl_sample;
          fft_out_matlab = FFTRefOut[matlab_idx];
          #1ps; //consume time to signals are visible in waves
`endif//DEBUG_SAMPLES
        end// exists
        matlab_idx++;
      end// while

      if (hesigb_sec_checked)
        hesigb_dec++;

      if (!err_occured && !secondary) begin
        `uvm_info(get_type_name(),
        $sformatf("FFT OUTPUT samples OK (symbol %0d)!",rtl_sample.symb_idx_f),UVM_HIGH)
      end
      if (!err_occured && secondary) begin
        `uvm_info(get_type_name(),
        $sformatf("FFT OUTPUT samples OK, secondary (symbol %0d)!",rtl_sample.symb_idx_f),UVM_HIGH)
      end
    end// forever
  endtask : compare_fft_out

  //--------------------------------------------------------------------
  // compare LDPC samples in BD deinterleaver
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::compare_ldpc_deint_samples();

    int             n_symb;
    int             nbpsc;
    int             starting_index;
    format_mod_e    format_mod;
    int             data_field;
    int             ldpc_offset;

    forever begin

      @(matlab_data_received);
      // check only LDPC frames
      if (rx_parameters.fec_f == 0) continue;

      n_symb = rx_parameters.nsymb_f;
      nbpsc = rx_parameters.nbpsc0_f;
      format_mod = format_mod_e'(rx_parameters.format_mod_f);

      if (rx_parameters.format_mod_f == -1) begin
        // Skip the comparison, when Matlab returns -1
        continue;
      end

      // encoding of Data fields from Matlab
      case (format_mod)
        NON_HT    : data_field = 18;
        HT_MF,
        HT_GF     : data_field = 19;
        VHT       : data_field = 20;
        HE_SU,
        HE_EXT_SU,
        HE_TB,
        HE_MU     : data_field = 21;
      endcase

      // Get the starting index from the Matlab samples
      // The comparison is done only on the data symbols
      // For every data symbol the comparison is done
      // based on the NBPSC
      for (int offset=0; offset < LDPCRefData.size();) begin
        if (LDPCRefData[offset].rxfield_f == data_field) begin
          fork
            ldpc_fifo.get(ldpc_out_item);
            wait(frame_rcv_mac_phy == 1);
          join_any

          disable fork;
          // in case of HE MUMIMO when our user is not the longest
          if (frame_rcv_mac_phy) break;

          // Do the comparison
          for (int i=0; i<nbpsc; i++) begin
            compare_ldpc_samples_per_symbol(i, offset);
          end
          offset = offset + ldpc_out_item.buffered_data.size()*nbpsc;
        end
        else begin
          offset++;
        end
      end

      `uvm_info(get_type_name(), $sformatf("LDPC samples checking done!!!"), UVM_LOW)

    end // forever

  endtask : compare_ldpc_deint_samples

  //--------------------------------------------------------------------
  // comparison of LDPC samples for every DATA symbol
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::compare_ldpc_samples_per_symbol(input int idx,
                                                            input int starting_index);

    logic signed [`LDPCSBWIDTH:0]     ref_data;
    int                               cnt = 0;
    int                               offset = 0;

    // calculate the offset, based on the starting index of the DATA
    // symbols, from Matlab samples, and other Matlab parameters
    offset = starting_index + idx;

    foreach(ldpc_out_item.buffered_data[i]) begin

      ref_data = signed'(ldpc_out_item.buffered_data[i][`LDPCSBWIDTH*idx +: `LDPCSBWIDTH]);

      if (ref_data !== LDPCRefData[offset+cnt].real_f) begin
        `uvm_error(get_type_name(), $sformatf("LDPC SB%0d mismatch : RTL (%0d) - Matlab (%0d)\n",
                                              idx, ref_data, LDPCRefData[offset+cnt].real_f))
      end
      else begin
        `uvm_info(get_type_name(), $sformatf("LDPC SB%0d match : RTL (%0d) - Matlab (%0d)\n",
                                              idx, ref_data, LDPCRefData[offset+cnt].real_f), UVM_HIGH)
      end

      cnt = cnt + rx_parameters.nbpsc0_f;
    end

  endtask : compare_ldpc_samples_per_symbol

  //--------------------------------------------------------------------
  // comparison of LDPC output samples
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::compare_ldpc_out_samples();

    int                                 n_symb;
    int                                 nbpsc;
    format_mod_e                        format_mod;
    int                                 data_field;
    logic [`LDPCDATAWIDTH-1:0]          ref_data;
    logic [`LDPCDATAWIDTH-1:0]          matlab_ref_data;

    forever begin

      @(matlab_data_received);
      // check only LDPC frames
      if (rx_parameters.fec_f == 0) continue;

      nbpsc = rx_parameters.nbpsc0_f;
      format_mod = format_mod_e'(rx_parameters.format_mod_f);
      n_symb = rx_parameters.nsymb_f;

      if (rx_parameters.format_mod_f == -1) begin
        // Skip the comparison, when Matlab returns -1
        continue;
      end

      // encoding of Data fields from Matlab
      case (format_mod)
        NON_HT      : data_field = 18;
        HT_MF,
        HT_GF       : data_field = 19;
        VHT         : data_field = 20;
        HE_SU,
        HE_EXT_SU,
        HE_TB,
        HE_MU       : data_field = 21;
      endcase

      // Get the starting index from the Matlab samples
      // The comparison is done only on the data symbols
      for (int offset=0; offset<LDPCOutputData.size();) begin
        // comparing only if FCS of OK
        if (LDPCOutputData[offset].rxfield_f == data_field) begin
          fork
            ldpc_data_fifo.get(ldpc_data_out_item);
            wait(frame_rcv_mac_phy == 1);
          join_any

          disable fork;
          // in case of HE MUMIMO when our user is not the longest
          if (frame_rcv_mac_phy) break;

          foreach(ldpc_data_out_item.buffered_data[i]) begin
            ref_data = ldpc_data_out_item.buffered_data[i];

            // Pack the Matlab referent data
            for (int j = 0; j < 8; j++) begin
              matlab_ref_data[j] = LDPCOutputData[offset+j].real_f;
            end

            if (m_modem_cfg.fcsOK == 0) begin
              `uvm_info(get_type_name(),$sformatf("LDPC output not checked - FCS not OK"),UVM_DEBUG)
            end
            else if (ref_data !== matlab_ref_data) begin
              `uvm_error(get_type_name(), $sformatf("LDPC output mismatch : RTL (%0d) - Matlab (%0d)",
                                                    ref_data, matlab_ref_data))
            end
            else begin
             `uvm_info(get_type_name(), $sformatf("LDPC output samples match : RTL (%0d) - Matlab (%0d)",
                                                    ref_data, matlab_ref_data), UVM_DEBUG)
            end
            offset = offset + 8;
          end // foreach buffered data
        end// if data field
        else begin
          offset++;
        end
      end// for LDPCOutputData

      `uvm_info(get_type_name(), $sformatf("Comparison of the LDPC output samples done (%0d bytes)!!!",LDPCOutputData.size()/8), UVM_LOW)

    end//forever matlab data received

  endtask : compare_ldpc_out_samples

  //--------------------------------------------------------------------
  // compare compressed soft bit samples from FreqDomain output
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::compare_compressed_softbits();
    int               n_symb;
    format_mod_e      format_mod;
    int               even_symb_cnt = 0;
    int               carrier_num[int];
    int               sb_index_table[6][] = '{
     /* NBPSC */ /*     SB indexes      */
     /* 1     */ '{0},
     /* 2     */ '{0, 4},
     /* 4     */ '{0, 1, 4, 5},
     /* 6     */ '{0, 1, 2, 4, 5, 6},
     /* 8     */ '{0, 1, 2, 3, 4, 5, 6, 7},
     /* 10    */ '{0, 1, 2, 3, 4, 5, 6, 7, 8, 9}
    };

    forever begin

      @(matlab_data_received);
      if (rx_parameters.format_mod_f == -1 || ppdu_ht_mf_mcs32) begin
        // Skip the comparison, when Matlab returns -1
        compressed_sb_comparison_done++;
        continue;
      end

      format_mod = format_mod_e'(rx_parameters.format_mod_f);

      if (rx_parameters.stbc_f == 1)
        n_symb = rx_parameters.nsymb_f/2 + rx_parameters.nsymb_f%2;
      else
        n_symb = rx_parameters.nsymb_f;

      case (format_mod)
        NON_HT:   n_symb = n_symb+1;
        HT_MF:    n_symb = n_symb+3;
        HT_GF:    n_symb = n_symb+2;
        VHT:      n_symb = n_symb+4;
        HE_SU:    n_symb = n_symb+4;
        HE_MU:    n_symb = n_symb + rx_parameters.he_sigb_f + 4;
      endcase

      for (int symb_cnt=0; symb_cnt < n_symb; symb_cnt++) begin
        fork
          deint_fifo.get(deint_out_item);
          wait(frame_rcv_mac_phy == 1);
        join_any

        disable fork;
        // in case of HE MUMIMO when our user is not the longest
        if (frame_rcv_mac_phy) break;

        // Based on NBPSC, we know which soft bits should
        // be checked
        // Get L-SIG
        if (symb_cnt == 0 && format_mod inside {NON_HT, HT_MF}) begin
          foreach (sb_index_table[0][i])
            compare_sc_symb_per_ss(sb_index_table[0][i], symb_cnt, rx_parameters);
        end
        // Get HT-SIG1 in case of HT-MF
        else if (symb_cnt inside {1} && format_mod == HT_MF) begin
          compare_sc_symb_per_ss(4, symb_cnt, rx_parameters);
        end
        // Get HT-SIG2 in case of HT-MF
        else if (symb_cnt inside {2} && format_mod == HT_MF) begin
          compare_sc_symb_per_ss(4, symb_cnt, rx_parameters);
        end
        // Get HT-SIG in case of HT-GF
        else if (symb_cnt inside {0,1} && format_mod == HT_GF) begin
          compare_sc_symb_per_ss(4, symb_cnt, rx_parameters);
        end
        else if (symb_cnt inside {0,1,3} && format_mod == VHT) begin
          // Compare VHT-SIGB only if the frame is MUMIMO
          if (!(m_modem_cfg.golden_frame.preamble_header.stbc == 1
               && m_modem_cfg.golden_frame.kind != MU_MIMO
               && symb_cnt == 3))
            compare_sc_symb_per_ss(0, symb_cnt, rx_parameters);
        end
        else if (symb_cnt == 2 && format_mod == VHT) begin
          compare_sc_symb_per_ss(4, symb_cnt, rx_parameters);
        end
        else if (symb_cnt inside {0,1,2,3} && format_mod inside {HE_SU, HE_MU}) begin
          compare_sc_symb_per_ss(0, symb_cnt, rx_parameters);
        end
        else if ((symb_cnt >= 4) && (symb_cnt < (rx_parameters.he_sigb_f + 4)) && format_mod == HE_MU) begin
          if (skip_sigb_check == 0) begin
            case(rx_parameters.nbpsc_he_sigb_f)
              0:  begin
                    foreach (sb_index_table[0][i])
                      compare_sc_symb_per_ss(sb_index_table[0][i], symb_cnt, rx_parameters);
                  end
              1:  begin
                    foreach (sb_index_table[1][i])
                      compare_sc_symb_per_ss(sb_index_table[1][i], symb_cnt, rx_parameters);
                  end
              2:  begin
                    foreach (sb_index_table[2][i])
                      compare_sc_symb_per_ss(sb_index_table[2][i], symb_cnt, rx_parameters);
                  end
              3:  begin
                    foreach (sb_index_table[3][i])
                      compare_sc_symb_per_ss(sb_index_table[3][i], symb_cnt, rx_parameters);
                  end
            endcase
          end
          comp_sigb_cnt++;
        end
        else begin
          if (rx_parameters.stbc_f == 1) begin
            // In case of STBC, the indexes has to be collected, from
            // the odd symbols, and attach them to even symbols
            even_symb_cnt++;
            if (even_symb_cnt%2 == 0) begin
              // Update with the previously saved indexes
              foreach (deint_out_item.buffered_data[i]) begin
                deint_out_item.buffered_data[i][`DEINTERWIDTH-1:`DEINTERWIDTH-`CARRIERNBWIDTH] = carrier_num[i];
              end
            end
            else begin
              // Save the indexes and use them for the odd symbols
              foreach (deint_out_item.buffered_data[i]) begin
                // Get the carrier_num
                carrier_num[i] = signed'(deint_out_item.buffered_data[i][`DEINTERWIDTH-1:`DEINTERWIDTH-`CARRIERNBWIDTH]);
              end
            end
          end
          case (rx_parameters.nbpsc0_f)
            1 : begin
                  foreach (sb_index_table[0][i])
                    compare_sc_symb_per_ss(sb_index_table[0][i], symb_cnt, rx_parameters);
                end
            2 : begin
                  foreach (sb_index_table[1][i])
                    compare_sc_symb_per_ss(sb_index_table[1][i], symb_cnt, rx_parameters);
                end
            4 : begin
                  foreach (sb_index_table[2][i])
                    compare_sc_symb_per_ss(sb_index_table[2][i], symb_cnt, rx_parameters);
                end
            6 : begin
                  foreach (sb_index_table[3][i])
                    compare_sc_symb_per_ss(sb_index_table[3][i], symb_cnt, rx_parameters);
                end
            8 : begin
                  foreach (sb_index_table[4][i])
                    compare_sc_symb_per_ss(sb_index_table[4][i], symb_cnt, rx_parameters);
                end
            10: begin
                  foreach (sb_index_table[5][i])
                    compare_sc_symb_per_ss(sb_index_table[5][i], symb_cnt, rx_parameters);
                end
          endcase
        end
      end // for
      `uvm_info(get_type_name(),$sformatf("Comparison of compressed soft bits done!"), UVM_LOW)
      compressed_sb_comparison_done++;
      comp_sigb_cnt = 0;
      even_symb_cnt = 0;
    end // forever
  endtask : compare_compressed_softbits

  //--------------------------------------------------------------------
  // compare noncompressed soft bit samples
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::compare_noncompressed_softbits();
    int               n_symb;
    format_mod_e      format_mod;
    int               sb_index_table[6][] = '{
     /* NBPSC */ /*     SB indexes      */
     /* 1     */ '{0},
     /* 2     */ '{0, 4},
     /* 4     */ '{0, 1, 4, 5},
     /* 6     */ '{0, 1, 2, 4, 5, 6},
     /* 8     */ '{0, 1, 2, 3, 4, 5, 6, 7},
     /* 10    */ '{0, 1, 2, 3, 4, 5, 6, 7, 8, 9}
    };

    forever begin

      @(matlab_data_received);
      if (rx_parameters.format_mod_f == -1 || ppdu_ht_mf_mcs32) begin
        // Skip the comparison, when Matlab returns -1
        noncompressed_sb_comparison_done++;
        continue;
      end
      format_mod = format_mod_e'(rx_parameters.format_mod_f);

      case (format_mod)
        NON_HT:   n_symb = rx_parameters.nsymb_f+1;
        HT_MF:    n_symb = rx_parameters.nsymb_f+3;
        HT_GF:    n_symb = rx_parameters.nsymb_f+2;
        VHT:      n_symb = rx_parameters.nsymb_f+4;
        HE_SU:    n_symb = rx_parameters.nsymb_f+4;
        HE_MU:    n_symb = rx_parameters.nsymb_f + rx_parameters.he_sigb_f + 4;
      endcase

      fork
        for (int symb_cnt=0; symb_cnt < n_symb; symb_cnt++) begin
          fork
            equ_fifo.get(equ_out_item);
            if (symb_cnt >= (n_symb-rx_parameters.nsymb_f) && rx_parameters.stbc_f == 1) begin
              equ1_fifo.get(equ1_out_item);
              receive_stbc_data = 1;
            end
          join
          // Based on NBPSC, we know which soft bits should
          // be checked
          // Get L-SIG
          if (symb_cnt == 0 && format_mod inside {NON_HT,HT_MF}) begin
            foreach (sb_index_table[0][i])
              compare_noncomp_sc_symb_per_ss(sb_index_table[0][i], symb_cnt, rx_parameters);
          end
          // Get HT-SIG1 in case of HT_MF
          else if (symb_cnt inside {1} && format_mod == HT_MF) begin
            compare_noncomp_sc_symb_per_ss(4, symb_cnt, rx_parameters);
          end
          // Get HT-SIG2 in case of HT_MF
          else if (symb_cnt inside {2} && format_mod == HT_MF) begin
            compare_noncomp_sc_symb_per_ss(4, symb_cnt, rx_parameters);
          end
          else if (symb_cnt inside {0,1} && format_mod == HT_GF) begin
            compare_noncomp_sc_symb_per_ss(4, symb_cnt, rx_parameters);
          end
          else if (symb_cnt inside {0,1,3} && format_mod == VHT) begin
            // Compare VHT-SIGB only if the frame is MUMIMO
            if (!(m_modem_cfg.golden_frame.preamble_header.stbc == 1
               && m_modem_cfg.golden_frame.kind != MU_MIMO
               && symb_cnt == 3))
              compare_noncomp_sc_symb_per_ss(0, symb_cnt, rx_parameters);
          end
          else if (symb_cnt == 2 && format_mod == VHT) begin
            compare_noncomp_sc_symb_per_ss(4, symb_cnt, rx_parameters);
          end
          else if (symb_cnt inside {0,1,2,3} && format_mod inside {HE_SU, HE_MU}) begin
            compare_noncomp_sc_symb_per_ss(0, symb_cnt, rx_parameters);
          end
          else if ((symb_cnt >= 4) && (symb_cnt < (rx_parameters.he_sigb_f + 4)) && format_mod == HE_MU) begin
            if (skip_sigb_check == 0) begin
              case(rx_parameters.nbpsc_he_sigb_f)
                0:  begin
                      foreach (sb_index_table[0][i])
                        compare_noncomp_sc_symb_per_ss(sb_index_table[0][i], symb_cnt, rx_parameters);
                    end
                1:  begin
                      foreach (sb_index_table[1][i])
                        compare_noncomp_sc_symb_per_ss(sb_index_table[1][i], symb_cnt, rx_parameters);
                    end
                2:  begin
                      foreach (sb_index_table[2][i])
                        compare_noncomp_sc_symb_per_ss(sb_index_table[2][i], symb_cnt, rx_parameters);
                    end
                3:  begin
                      foreach (sb_index_table[3][i])
                        compare_noncomp_sc_symb_per_ss(sb_index_table[3][i], symb_cnt, rx_parameters);
                    end
              endcase
            end
            non_comp_sigb_cnt++;
          end
          else begin
            case (rx_parameters.nbpsc0_f)
              1 : begin
                    foreach (sb_index_table[0][i])
                      compare_noncomp_sc_symb_per_ss(sb_index_table[0][i], symb_cnt, rx_parameters);
                  end
              2 : begin
                    foreach (sb_index_table[1][i])
                      compare_noncomp_sc_symb_per_ss(sb_index_table[1][i], symb_cnt, rx_parameters);
                  end
              4 : begin
                    foreach (sb_index_table[2][i])
                      compare_noncomp_sc_symb_per_ss(sb_index_table[2][i], symb_cnt, rx_parameters);
                  end
              6 : begin
                    foreach (sb_index_table[3][i])
                      compare_noncomp_sc_symb_per_ss(sb_index_table[3][i], symb_cnt, rx_parameters);
                  end
              8 : begin
                    foreach (sb_index_table[4][i])
                      compare_noncomp_sc_symb_per_ss(sb_index_table[4][i], symb_cnt, rx_parameters);
                  end
              10: begin
                    foreach (sb_index_table[5][i])
                      compare_noncomp_sc_symb_per_ss(sb_index_table[5][i], symb_cnt, rx_parameters);
                  end
            endcase
            if (receive_stbc_data) begin
              // Check from EQU1 interface, the samples on DATA symbols,
              // in case STBC is set in the frame
              symb_cnt++;
              equ_out_item.copy(equ1_out_item);
              case (rx_parameters.nbpsc0_f)
                1 : begin
                      foreach (sb_index_table[0][i])
                        compare_noncomp_sc_symb_per_ss(sb_index_table[0][i], symb_cnt, rx_parameters);
                    end
                2 : begin
                      foreach (sb_index_table[1][i])
                        compare_noncomp_sc_symb_per_ss(sb_index_table[1][i], symb_cnt, rx_parameters);
                    end
                4 : begin
                      foreach (sb_index_table[2][i])
                        compare_noncomp_sc_symb_per_ss(sb_index_table[2][i], symb_cnt, rx_parameters);
                    end
                6 : begin
                      foreach (sb_index_table[3][i])
                        compare_noncomp_sc_symb_per_ss(sb_index_table[3][i], symb_cnt, rx_parameters);
                    end
                8 : begin
                      foreach (sb_index_table[4][i])
                        compare_noncomp_sc_symb_per_ss(sb_index_table[4][i], symb_cnt, rx_parameters);
                    end
                10: begin
                      foreach (sb_index_table[5][i])
                        compare_noncomp_sc_symb_per_ss(sb_index_table[5][i], symb_cnt, rx_parameters);
                    end
              endcase
            end
          end
        end // for
        wait(frame_rcv_mac_phy == 1);
      join_any
      disable fork;
      `uvm_info(get_type_name(),$sformatf("Comparison of non compressed soft bits done!"), UVM_LOW)
      noncompressed_sb_comparison_done++;
      non_comp_sigb_cnt = 0;
      receive_stbc_data = 0;
    end // forever
  endtask : compare_noncompressed_softbits

  //---------------------------------------------------------------------------
  // compare compressed soft bits of symbol per spatial stream
  //---------------------------------------------------------------------------
  task modem_rx_scoreboard::compare_sc_symb_per_ss(input int idx, input int symb_cnt, input rxparameters_s rx_param);

    int                                         offset;
    int                                         cnt;
    logic signed [`SBSTREAMWIDTH:0]             ref_data;
    int                                         tolerance_plus;
    int                                         tolerance_minus;
    int                                         carrier_num;
    format_mod_e                                format_mod;
    int                                         temp_idx;
    int                                         he_mu_temp_idx;
    int                                         match_found = 0;
    int                                         mdm_config;

    logic [`DEINTERWIDTH-`CARRIERNBWIDTH-1:0]   aligned_buff_data[int];

    // Get the configuration of the DUT mdmconfig[3:0]
    // 0 - 20 MHz
    // 1 - 40 MHz
    mdm_config = m_regmodel.get_mirrored_reg_value("MDMCONF");

    foreach (deint_out_item.buffered_data[i]) begin
      // Get the carrier_num
      carrier_num = signed'(deint_out_item.buffered_data[i][`DEINTERWIDTH-1:`DEINTERWIDTH-`CARRIERNBWIDTH]);

      // When the configuration is 20MHz and cbw is 40MHz add offset to the index,
      // on the HE-MU frames, and HE-DATA symbols
      if (symb_cnt >= (rx_parameters.he_sigb_f + 4) && (format_mod_e'(rx_param.format_mod_f) == HE_MU)) begin
        if (mdm_config[3:0] == 0) begin
          if (rx_param.cbw_f == 1) begin
            if (primary_channel == 1)
              carrier_num = carrier_num - 128;
            else
              carrier_num = carrier_num + 128;
          end
          else if (rx_param.cbw_f == 2) begin
            case(primary_channel)
              0 : carrier_num = carrier_num + 384;
              1 : carrier_num = carrier_num + 128;
              2 : carrier_num = carrier_num - 128;
              3 : carrier_num = carrier_num - 384;
            endcase
          end
        end
        else if (mdm_config[3:0] == 1) begin
          if (rx_param.cbw_f == 2) begin
            if (primary_channel == 1)
              carrier_num = carrier_num - 256;
            else
              carrier_num = carrier_num + 256;
          end
        end
      end

      aligned_buff_data[carrier_num] = deint_out_item.buffered_data[i][`DEINTERWIDTH-`CARRIERNBWIDTH-1:0];
    end

    cnt = 0;

    format_mod = format_mod_e'(rx_parameters.format_mod_f);

    if (rx_param.nbpsc_he_sigb_f == 0)
      rx_param.nbpsc_he_sigb_f = 1;
    else
      rx_param.nbpsc_he_sigb_f = rx_param.nbpsc_he_sigb_f*2;

    offset = offset_calculation(idx, rx_param, symb_cnt, comp_sigb_cnt);

    foreach(aligned_buff_data[i])begin
      // Get the item, based on the idx
      ref_data = signed'(aligned_buff_data[i][`SBSTREAMWIDTH*idx +: `SBSTREAMWIDTH]);

      // In case tolerance is defined, it should be
      // taken into account in case of comparison
      tolerance_plus  = DeintRefData[offset+cnt].real_comp_f + `SAMPLE_TOLERANCE;
      tolerance_minus = DeintRefData[offset+cnt].real_comp_f - `SAMPLE_TOLERANCE;

      if (i == DeintRefData[offset+cnt].indsc_f) begin
        match_found = 1;
        if (ref_data !== DeintRefData[offset+cnt].real_comp_f)begin
          // Comparison of the ref_data with the Matlab samples
          if (!(ref_data <= tolerance_plus && ref_data >= tolerance_minus)  || ref_data === 'x) begin
            `uvm_error(get_type_name(), $sformatf("SS%0d_SB%0d mismatch : RTL (%0d) - Matlab (%0d)(tolerance: +/-%0d) on symbol %0d\n",
                                        rx_param.nss_f, idx, ref_data, DeintRefData[offset+cnt].real_comp_f, `SAMPLE_TOLERANCE, symb_cnt))
          end
        end
        else begin
          `uvm_info(get_type_name(), $sformatf("SS%0d_SB%0d match : RTL (%0d) - Matlab (%0d)(tolerance: +/-%0d) on symbol %0d\n",
                                      rx_param.nss_f, idx, ref_data, DeintRefData[offset+cnt].real_comp_f, `SAMPLE_TOLERANCE, symb_cnt),UVM_HIGH)
        end

        if (    symb_cnt == 0 && format_mod inside {NON_HT, HT_MF}
            ||  symb_cnt inside {1,2} && format_mod == HT_MF
            ||  symb_cnt inside {0,1} && format_mod == HT_GF
            ||  symb_cnt inside {0,1,2,3} && format_mod inside {VHT, HE_SU, HE_MU})
          cnt++;
        else if (symb_cnt > 3 && symb_cnt < 4 + rx_param.he_sigb_f && format_mod == HE_MU)
          cnt = cnt + rx_param.nbpsc_he_sigb_f;
        else
          cnt = cnt + rx_param.nbpsc0_f;
      end
      else begin
        cnt = cnt + rx_param.nbpsc0_f;
      end
    end

    if(!match_found)
      `uvm_error(get_type_name(), $sformatf("There was no matching index on compressed SB on the symbol %d", symb_cnt))

  endtask : compare_sc_symb_per_ss

  //---------------------------------------------------------------------------
  // compare noncompressed soft bits of symbol per spatial stream
  //---------------------------------------------------------------------------
  task modem_rx_scoreboard::compare_noncomp_sc_symb_per_ss(input int idx, input int symb_cnt, input rxparameters_s rx_param);

    int                                   offset;
    int                                   cnt;
    logic signed [`DEMAPSBWIDTH-1:0]      ref_data;
    int                                   tolerance_plus;
    int                                   tolerance_minus;
    int                                   index;
    format_mod_e                          format_mod;
    int                                   temp_idx;
    int                                   he_mu_temp_idx;
    int                                   match_found = 0;
    int                                   mdm_config;

    logic [`EQUWIDTH-`CARRIERNBWIDTH-1:0] aligned_buff_data[int];

    // Get the configuration of the DUT mdmconfig[3:0]
    // 0 - 20 MHz
    // 1 - 40 MHz
    mdm_config = m_regmodel.get_mirrored_reg_value("MDMCONF");

    foreach (equ_out_item.buffered_data[i]) begin
      // Get the carrier_num
      index = signed'(equ_out_item.buffered_data[i][`EQUWIDTH-1:`EQUWIDTH-`CARRIERNBWIDTH]);

      // When the configuration is 20MHz and cbw is 40MHz add offset to the index,
      // on the HE-MU frames, and HE-DATA symbols
      if (symb_cnt >= (rx_param.he_sigb_f + 4) && (format_mod_e'(rx_param.format_mod_f) == HE_MU)) begin
        if (mdm_config[3:0] == 0) begin
          if (rx_param.cbw_f == 1) begin
            if (primary_channel == 1)
              index = index - 128;
            else
              index = index + 128;
          end
          else if (rx_param.cbw_f == 2) begin
            case(primary_channel)
              0 : index = index + 384;
              1 : index = index + 128;
              2 : index = index - 128;
              3 : index = index - 384;
            endcase
          end
        end
        else if (mdm_config[3:0] == 1) begin
          if (rx_param.cbw_f == 2) begin
            if (primary_channel == 1)
              index = index - 256;
            else
              index = index + 256;
          end
        end
      end

      aligned_buff_data[index] = equ_out_item.buffered_data[i][`EQUWIDTH-`CARRIERNBWIDTH-1:0];
    end

    cnt = 0;

    format_mod = format_mod_e'(rx_parameters.format_mod_f);

    if (rx_param.nbpsc_he_sigb_f == 0)
      rx_param.nbpsc_he_sigb_f = 1;
    else
      rx_param.nbpsc_he_sigb_f = rx_param.nbpsc_he_sigb_f*2;

    offset = offset_calculation(idx, rx_param, symb_cnt, non_comp_sigb_cnt);

    foreach(aligned_buff_data[i]) begin
      ref_data = signed'(aligned_buff_data[i][`DEMAPSBWIDTH*idx +: `DEMAPSBWIDTH]);
      // In case tolerance is defined, it should be
      // taken into account in case of comparison
      tolerance_plus  = DeintNonCompRefData[offset+cnt].real_f + `SAMPLE_NONCOMP_TOLERANCE;
      tolerance_minus = DeintNonCompRefData[offset+cnt].real_f - `SAMPLE_NONCOMP_TOLERANCE;

      // Comparison of the ref_data with the Matlab samples
      if (i == DeintNonCompRefData[offset+cnt].indsc_f) begin
        match_found = 1;
        if (ref_data !== DeintNonCompRefData[offset+cnt].real_f) begin
          if ((!(ref_data <= tolerance_plus && ref_data >= tolerance_minus)) || ref_data === 'x) begin
            `uvm_error(get_type_name(), $sformatf("Non-compressed SB%0d mismatch on index(%0d) : RTL (%0d) - Matlab (%0d)(tolerance: +/-%0d) on symbol %0d\n",
                                        idx, i, ref_data, DeintNonCompRefData[offset+cnt].real_f, `SAMPLE_NONCOMP_TOLERANCE, symb_cnt))
          end
        end
        else begin
          `uvm_info(get_type_name(), $sformatf("Non-compressed SB%0d on index(%0d) match : RTL (%0d) - Matlab (%0d)(tolerance: +/-%0d) on symbol %0d\n",
                                      idx, i, ref_data, DeintNonCompRefData[offset+cnt].real_f, `SAMPLE_NONCOMP_TOLERANCE, symb_cnt), UVM_HIGH)
        end

        if (    symb_cnt == 0 && format_mod inside {NON_HT,HT_MF}
            ||  symb_cnt inside {1,2} && format_mod == HT_MF
            ||  symb_cnt inside {0,1} && format_mod == HT_GF
            ||  symb_cnt inside {0,1,2,3} && format_mod inside {VHT, HE_SU, HE_MU}) begin
          cnt++;
        end
        else if (symb_cnt > 3 && (symb_cnt < 4 + rx_param.he_sigb_f) && format_mod == HE_MU) begin
          cnt = cnt + rx_param.nbpsc_he_sigb_f;
        end
        else begin
          cnt = cnt + rx_param.nbpsc0_f;
        end
      end
      else begin
        cnt = cnt + rx_param.nbpsc0_f;
      end
    end

    if(!match_found)
      `uvm_error(get_type_name(), $sformatf("There was no matching index on non-compressed SB on the symbol %d", symb_cnt))

  endtask : compare_noncomp_sc_symb_per_ss

  //--------------------------------------------------------------------
  // compare status register values from RxTD
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::compare_rxtd_status_reg();
    bit [31:0]  values1[];
    // get referent status register values

    m_modem_cfg.get_block_field_value(values1,"RXTBE","resultFromTBE","REAL");
    foreach (values1[i])
      `uvm_info(get_type_name(),$sformatf("TBE[%0d]=%0d",i,values1[i]),UVM_LOW)

    // Check corrPhase
    m_modem_cfg.get_block_field_value(values1,"FINETDFO","corrPhase");
    if (values1[0][20:0] !== m_cfg.m_bus_monitor_cfg.status_fine_fo_angle)
      `uvm_error(get_type_name(), $sformatf("StatusFineFOAngle (%0h) mismatch with referent (%0h)",
      values1[0][20:0], m_cfg.m_bus_monitor_cfg.status_fine_fo_angle))
    else
      `uvm_info(get_type_name(), $sformatf("StatusFineFOAngle (%0h) match with referent (%0h)",
      values1[0][20:0], m_cfg.m_bus_monitor_cfg.status_fine_fo_angle), UVM_LOW)

    // Check corrPhase
    m_modem_cfg.get_block_field_value(values1,"COARSETDFO","coarseFOPhase");
    if (values1[0][20:0] !== m_cfg.m_bus_monitor_cfg.status_coarse_fo_angle)
      `uvm_error(get_type_name(), $sformatf("StatusCoarseFOAngle (%0h) mismatch with referent (%0h)",
      values1[0][20:0], m_cfg.m_bus_monitor_cfg.status_coarse_fo_angle))
    else
      `uvm_info(get_type_name(), $sformatf("StatusCoarseFOAngle (%0h) match with referent (%0h)",
      values1[0][20:0], m_cfg.m_bus_monitor_cfg.status_coarse_fo_angle), UVM_LOW)

    // Check totalPower
//     m_modem_cfg.get_block_field_value(values1,"FINETDFO","totalPower");
//     if (values1[0][21:0] !== m_cfg.m_bus_monitor_cfg.totalPower)
//       `uvm_error(get_type_name(), $sformatf("TotalPower (%0h) mismatch with referent (%0h)",
//       values1[0][21:0], m_cfg.m_bus_monitor_cfg.totalPower))
//     else
//       `uvm_info(get_type_name(), $sformatf("TotalPower (%0h) match with referent (%0h)",
//       values1[0][21:0], m_cfg.m_bus_monitor_cfg.totalPower),UVM_LOW)

    // Check noisePower
//     m_modem_cfg.get_block_field_value(values1,"FINETDFO","noisePower");
//     if (values1[0][21:0] !== m_cfg.m_bus_monitor_cfg.noisePower)
//       `uvm_error(get_type_name(), $sformatf("NoisePower (%0h) mismatch with referent (%0h)",
//       values1[0][21:0], m_cfg.m_bus_monitor_cfg.noisePower))
//     else
//       `uvm_info(get_type_name(), $sformatf("NoisePower (%0h) match with referent (%0h)",
//       values1[0][21:0], m_cfg.m_bus_monitor_cfg.noisePower),UVM_LOW)

    // Check RatioNoiseSignal
//     m_modem_cfg.get_block_field_value(values1,"FINETDFO","ratio_noise_signal");
//     if (values1[0][11:0] !== m_cfg.m_bus_monitor_cfg.RatioNoiseSignal)
//       `uvm_error(get_type_name(), $sformatf("RatioNoiseSignal (%0h) mismatch with referent (%0h)",
//       values1[0][11:0], m_cfg.m_bus_monitor_cfg.RatioNoiseSignal))
//     else
//       `uvm_info(get_type_name(), $sformatf("RatioNoiseSignal (%0h) match with referent (%0h)",
//       values1[0][11:0], m_cfg.m_bus_monitor_cfg.RatioNoiseSignal),UVM_LOW)

    // Check RatioShift
//     m_modem_cfg.get_block_field_value(values1,"FINETDFO","nbits_right_shift");
//     if (values1[0][5:0] !== m_cfg.m_bus_monitor_cfg.RatioShift)
//       `uvm_error(get_type_name(), $sformatf("RatioShift (%0h) mismatch with referent (%0h)",
//       values1[0][5:0], m_cfg.m_bus_monitor_cfg.RatioShift))
//     else
//       `uvm_info(get_type_name(), $sformatf("RatioShift (%0h) match with referent (%0h)",
//       values1[0][5:0], m_cfg.m_bus_monitor_cfg.RatioShift),UVM_LOW)

    // Check noisePowerMant
//     m_modem_cfg.get_block_field_value(values1,"FINETDFO","noisePower_Mantis");
//     if (values1[0][9:0] !== m_cfg.m_bus_monitor_cfg.noisePowerMant)
//       `uvm_error(get_type_name(), $sformatf("noisePowerMant (%0h) mismatch with referent (%0h)",
//       values1[0][9:0], m_cfg.m_bus_monitor_cfg.noisePowerMant))
//     else
//       `uvm_info(get_type_name(), $sformatf("noisePowerMant (%0h) match with referent (%0h)",
//       values1[0][9:0], m_cfg.m_bus_monitor_cfg.noisePowerMant),UVM_LOW)

    // Check noisePower_Expo
//     m_modem_cfg.get_block_field_value(values1,"FINETDFO","noisePower_Expo");
//     if (values1[0][4:0] !== m_cfg.m_bus_monitor_cfg.noisePowerExp)
//       `uvm_error(get_type_name(), $sformatf("noisePowerExp (%0h) mismatch with referent (%0h)",
//       values1[0][4:0], m_cfg.m_bus_monitor_cfg.noisePowerExp))
//     else
//       `uvm_info(get_type_name(), $sformatf("noisePowerExp (%0h) match with referent (%0h)",
//       values1[0][4:0], m_cfg.m_bus_monitor_cfg.noisePowerExp),UVM_LOW)

  endtask : compare_rxtd_status_reg

  //--------------------------------------------------------------------
  // Task for checking FDO samples
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::compare_fdo_samples();
    bit [31:0]  values1[];

    if (ppdu_ht_mf_mcs32) return;

    // Check StatusFDOData when not NDP
    if (m_modem_cfg.golden_frame.kind != NDP) begin
      m_modem_cfg.get_block_field_value(values1,"FDOFFSET","FDOData");
      if (values1[0][21:0] !== m_cfg.m_bus_monitor_cfg.StatusFDOData)
        if (m_modem_cfg.golden_frame.kind != MU_MIMO)
          `uvm_error(get_type_name(), $sformatf("StatusFDOData (%0d) mismatch with referent (%0d)",
          signed'(m_cfg.m_bus_monitor_cfg.StatusFDOData), signed'(values1[0][21:0])))
        else
          `uvm_warning(get_type_name(), $sformatf("StatusFDOData (%0d) mismatch with referent (%0d)",
          signed'(m_cfg.m_bus_monitor_cfg.StatusFDOData), signed'(values1[0][21:0])))
      else
        `uvm_info(get_type_name(), $sformatf("StatusFDOData (%0d) match with referent (%0d)",
        signed'(m_cfg.m_bus_monitor_cfg.StatusFDOData), signed'(values1[0][21:0])),UVM_LOW)
    end

    // Check StatusFDOPreamb
    if (!(m_modem_cfg.golden_frame.ppdu_format inside {NON_HT, NON_HT_DUP_OFDM})) begin
      m_modem_cfg.get_block_field_value(values1,"FDOFFSET","FDOPreamb");
      if (values1[0][21:0] !== m_cfg.m_bus_monitor_cfg.StatusFDOPreamb)
        `uvm_error(get_type_name(), $sformatf("StatusFDOPreamb (%0d) mismatch with referent (%0d)",
        signed'(m_cfg.m_bus_monitor_cfg.StatusFDOPreamb), signed'(values1[0][21:0])))
      else
        `uvm_info(get_type_name(), $sformatf("StatusFDOPreamb (%0d) match with referent (%0d)",
        signed'(m_cfg.m_bus_monitor_cfg.StatusFDOPreamb), signed'(values1[0][21:0])),UVM_LOW)
    end

  endtask : compare_fdo_samples

  //--------------------------------------------------------------------
  // Task for checking dPhi and TsNormOut samples
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::compare_tbo_samples();
    bit [31:0]  values1[];
    int         tolerance_plus;
    int         tolerance_minus;

    if (ppdu_ht_mf_mcs32) return;

    // Check StatusdPhi
    m_modem_cfg.get_block_field_value(values1,"TBOFFSET","dphi");
    if (values1[0][24:0] !== m_cfg.m_bus_monitor_cfg.StatusdPhi)
      if (m_modem_cfg.golden_frame.kind != MU_MIMO)
        `uvm_error(get_type_name(), $sformatf("StatusdPhi (%0d) mismatch with referent (%0d)",
        signed'(m_cfg.m_bus_monitor_cfg.StatusdPhi), signed'(values1[0][24:0])))
      else
        `uvm_warning(get_type_name(), $sformatf("StatusdPhi (%0d) mismatch with referent (%0d)",
        signed'(m_cfg.m_bus_monitor_cfg.StatusdPhi), signed'(values1[0][24:0])))
    else
      `uvm_info(get_type_name(), $sformatf("StatusdPhi (%0d) match with referent (%0d)",
      signed'(m_cfg.m_bus_monitor_cfg.StatusdPhi), signed'(values1[0][24:0])),UVM_LOW)

    // Check StatusTsNormOut
    m_modem_cfg.get_block_field_value(values1,"TBOFFSET","TsNormOut");
    if (values1[0][26:0] !== m_cfg.m_bus_monitor_cfg.StatusTsNormOut)
      if (m_modem_cfg.golden_frame.kind != MU_MIMO)
        `uvm_error(get_type_name(), $sformatf("StatusTsNormOut (%0d) mismatch with referent (%0d)",
        m_cfg.m_bus_monitor_cfg.StatusTsNormOut, values1[0][26:0]))
      else
        `uvm_warning(get_type_name(), $sformatf("StatusTsNormOut (%0d) mismatch with referent (%0d)",
        signed'(m_cfg.m_bus_monitor_cfg.StatusTsNormOut), signed'(values1[0][24:0])))
    else
      `uvm_info(get_type_name(), $sformatf("StatusTsNormOut (%0d) match with referent (%0d)",
      m_cfg.m_bus_monitor_cfg.StatusTsNormOut, values1[0][26:0]),UVM_LOW)

  endtask : compare_tbo_samples

  //--------------------------------------------------------------------
  // Function for calculating the offset for the Matlab soft bit samples
  //--------------------------------------------------------------------
  function int modem_rx_scoreboard::offset_calculation(int idx, rxparameters_s rx_param, int symb_cnt, int sigb_cnt);

    format_mod_e                          format_mod;
    int                                   temp_idx;
    int                                   he_mu_temp_idx;
    int                                   offset;
    int                                   base_offset;
    int                                   he_sigb_offset;
    int                                   mdm_config;

    // Get the format mod, to know which fields, are present
    format_mod = format_mod_e'(rx_parameters.format_mod_f);

    // Get the configuration of the DUT mdmconfig[3:0]
    // 0 - 20 MHz
    // 1 - 40 MHz
    mdm_config = m_regmodel.get_mirrored_reg_value("MDMCONF");

    // If the primaryInd is 0, then the channelcontent1 samples are first in order,
    // otherwise, the channelcontent2 are first
    if (mdm_config[3:0] == 1) begin
      if (rx_param.cbw_f == 1) begin
        if (primary_channel == 0) begin
          if (sigb_cnt % 2 == 1) begin
            he_sigb_offset = ((sigb_cnt/2)*(rx_param.nbpsc_he_sigb_f)*52)+(rx_param.he_sigb_f/2)*(rx_param.nbpsc_he_sigb_f)*52;
          end
          else begin
            he_sigb_offset = (sigb_cnt/2)*(rx_param.nbpsc_he_sigb_f)*52;
          end
        end
        else begin
          if (sigb_cnt % 2 == 0) begin
            he_sigb_offset = ((sigb_cnt/2)*(rx_param.nbpsc_he_sigb_f)*52)+(rx_param.he_sigb_f/2)*(rx_param.nbpsc_he_sigb_f)*52;
          end
          else begin
            he_sigb_offset = (sigb_cnt/2)*(rx_param.nbpsc_he_sigb_f)*52;
          end
        end
      end
      else begin
        he_sigb_offset = (symb_cnt - 4)*(rx_param.nbpsc_he_sigb_f)*52;
      end
    end
    else begin
      he_sigb_offset = (symb_cnt - 4)*(rx_param.nbpsc_he_sigb_f)*52;
    end

    if (rx_param.nbpsc0_f inside {1,8,10}) begin
      temp_idx = idx;
    end
    else begin
      if (idx inside {0,1,2})
        temp_idx = idx;
      else if (idx == 4 && rx_param.nbpsc0_f == 2)
        temp_idx = 1;
      else if (idx == 4 && rx_param.nbpsc0_f == 4)
        temp_idx = 2;
      else if (   idx == 4 && rx_param.nbpsc0_f == 6
               || idx == 5 && rx_param.nbpsc0_f == 4)
        temp_idx = 3;
      else if (   idx == 5 && rx_param.nbpsc0_f == 6
               || idx == 6)
        temp_idx = idx - 1;
    end

    if (rx_param.nbpsc_he_sigb_f == 0)
      he_mu_temp_idx = idx;
    else begin
      if (idx inside {0,1,2})
        he_mu_temp_idx = idx;
      else if (idx == 4 && rx_param.nbpsc_he_sigb_f == 2)
        he_mu_temp_idx = 1;
      else if (idx == 4 && rx_param.nbpsc_he_sigb_f == 4)
        he_mu_temp_idx = 2;
      else if (   idx == 4 && rx_param.nbpsc_he_sigb_f == 6
               || idx == 5 && rx_param.nbpsc_he_sigb_f == 4)
        he_mu_temp_idx = 3;
      else if (   idx == 5 && rx_param.nbpsc_he_sigb_f == 6
               || idx == 6)
        he_mu_temp_idx = idx - 1;
    end

    if (rx_param.cbw_f == 1) begin
      base_offset = 300;
    end
    else begin
      base_offset = 244;
    end

    // This offset is used for mapping in the
    // array, that is received from the Matlab

    if (symb_cnt == 0 && format_mod inside {NON_HT,HT_MF})
      offset = symb_cnt * rx_param.nsd_f + temp_idx;
    else if (symb_cnt inside {0,1} && format_mod == HT_GF)
      offset = 48*symb_cnt;
    else if (symb_cnt inside {1,2} && format_mod == HT_MF)
      offset = 96 + (symb_cnt - 1)*48;
    else if (symb_cnt inside {0,1} && format_mod == VHT)
      offset = 48*symb_cnt;
    else if (symb_cnt inside {2,3} && format_mod == VHT)
      offset = 96 + (symb_cnt - 1)*48;
    else if (symb_cnt inside {0,1} && format_mod inside {HE_SU, HE_MU})
      offset = 48*symb_cnt;
    else if (symb_cnt inside {2,3} && format_mod inside {HE_SU, HE_MU})
      offset = 96 + (symb_cnt - 2)*52;
    else if (symb_cnt >= 4 && symb_cnt < (rx_parameters.he_sigb_f + 4) && format_mod == HE_MU)
      offset = 200 + he_sigb_offset + he_mu_temp_idx;
    else if (symb_cnt >= (rx_parameters.he_sigb_f + 4) && format_mod == HE_MU)
      offset = 200 + rx_parameters.he_sigb_f*(rx_param.nbpsc_he_sigb_f)*52 +((symb_cnt - 4 - (rx_parameters.he_sigb_f ))*rx_parameters.nbpsc0_f*rx_parameters.nsd_f) + temp_idx;
    else if (format_mod == NON_HT && symb_cnt > 0)
      offset = 96 + (symb_cnt - 1)*rx_param.nbpsc0_f*rx_param.nsd_f + temp_idx;
    else if (format_mod == HT_MF && symb_cnt > 2)
      offset = 192 + (symb_cnt - 3)*rx_param.nbpsc0_f*rx_param.nsd_f + temp_idx;
    else if (format_mod == HT_GF && symb_cnt > 1)
      offset = 96 + (symb_cnt - 2)*rx_param.nbpsc0_f*rx_param.nsd_f + temp_idx;
    else if (format_mod == VHT && symb_cnt > 3)
      offset = base_offset + (symb_cnt - 4)*rx_param.nbpsc0_f*rx_param.nsd_f + temp_idx;
    else if (symb_cnt > 3 && format_mod == HE_SU)
      offset = 200 + (symb_cnt - 4)*rx_param.nbpsc0_f*rx_param.nsd_f + temp_idx;

    return offset;

  endfunction : offset_calculation

  //--------------------------------------------------------------------
  // Collection of Viterbi In soft bits
  //--------------------------------------------------------------------
  task modem_rx_scoreboard::compare_viterbi_in_data();

    format_mod_e                            format_mod;
    bit [1:0]                               ch_bw;
    bit                                     lsig6m_f;
    bit [11:0]                              leg_length;
    int                                     matlab_idx = 0;
    logic [`VITERBIWIDTH-1:0]               rtl_data;
    logic [`VITERBIWIDTH-1:0]               matlab_rtl_data;
    int                                     hesigb_ch1_offset;
    int                                     hesigb_ch2_offset;
    int                                     content_ch_idx;
    bit                                     stbc;

    forever begin
      @(matlab_data_received);

      format_mod = format_mod_e'(rx_parameters.format_mod_f);
      ch_bw      = rx_parameters.cbw_f;
      lsig6m_f   = rx_parameters.lsig6m_f;
      leg_length = m_modem_cfg.golden_frame.preamble_header.leg_length;
      stbc       = rx_parameters.stbc_f;

      if (rx_parameters.format_mod_f == -1 || ppdu_ht_mf_mcs32) begin
      // If the returned format mod is -1, skip checking
        continue;
      end

      if (format_mod == HE_MU && ch_bw > 0) begin
        // add LSIG, RLSIG, HESIGA samples and align to start of HESIGB
        hesigb_ch1_offset = 0;
        hesigb_ch2_offset = 0;
        content_ch_idx = ViterbiRefData[48 + 48 + 104].contentchanidx_f;

        for (int i=0; i < ViterbiRefData.size(); i++) begin
          if (ViterbiRefData[i].rxfield_f == 11 && ViterbiRefData[i].contentchanidx_f == 1)
            hesigb_ch1_offset++;
          else if (ViterbiRefData[i].rxfield_f == 11 && ViterbiRefData[i].contentchanidx_f == 2)
            hesigb_ch2_offset++;
          else if (ViterbiRefData[i].rxfield_f == 21)
            break;//for loop
        end

        if (content_ch_idx == 2) begin
          hesigb_ch1_offset = 48 + 48 + 104 + hesigb_ch2_offset;
          hesigb_ch2_offset = 48 + 48 + 104;
        end
        else begin
          hesigb_ch2_offset = hesigb_ch1_offset + 48 + 48 + 104;
          hesigb_ch1_offset = 48 + 48 + 104;
        end

        `uvm_info(get_type_name(),
        $sformatf("hesigb_ch1_offset = %0d, hesigb_ch2_offset = %0d",hesigb_ch1_offset,hesigb_ch2_offset),UVM_DEBUG)
      end
      else begin
        hesigb_ch1_offset = 0;
        hesigb_ch2_offset = 0;
        content_ch_idx    = 0;
      end

      for (matlab_idx=0; matlab_idx < ViterbiRefData.size();) begin
        fork
          viterbi_in_fifo.get(viterbi_in_item);
          wait(frame_rcv_mac_phy == 1);
        join_any

        disable fork;
        // in case of HE MUMIMO when our user is not the longest
        if (frame_rcv_mac_phy) break;

        // skip RL-SIG in Legacy if legrate is 6Mbps and legacy length mod 3 is 0
        if(format_mod == NON_HT && lsig6m_f == 1'b1 && leg_length%3 == 0 && ViterbiRefData[matlab_idx].rxfield_f == 6) begin
          matlab_idx = matlab_idx + 48;
        end
        // skip RL-SIG in Legacy if legrate is not 6Mbps
        else if(format_mod == NON_HT && lsig6m_f == 1'b0 && ViterbiRefData[matlab_idx].rxfield_f == 6) begin
          matlab_idx = matlab_idx + 48;
        end
        // skip RL-SIG in HT-MF
        else if(format_mod == HT_MF && ViterbiRefData[matlab_idx].rxfield_f == 6) begin
          matlab_idx = matlab_idx + 48;
        end
        // skip RL-SIG in VHT
        else if(format_mod == VHT && ViterbiRefData[matlab_idx].rxfield_f == 6) begin
          matlab_idx = matlab_idx + 48;
        end
        // skip VHT SIGB
        else if(format_mod == VHT && ViterbiRefData[matlab_idx].rxfield_f == 9 && stbc == 1) begin
          while (ViterbiRefData[matlab_idx].rxfield_f == 9) begin
            matlab_idx++;
          end
          continue;
        end
        else if (format_mod == HE_MU && ch_bw > 0 && ViterbiRefData[matlab_idx].rxfield_f == 11 && content_ch_idx == 2) begin
          matlab_idx = hesigb_ch1_offset;
          content_ch_idx = 0; //reset flag
        end
`ifndef RW_NX_CHBW20
        // set index when checking HESIGB CHBW40
        else if (format_mod == HE_MU && ch_bw > 0 && ViterbiRefData[hesigb_ch2_offset].rxfield_f == 21) begin
          matlab_idx = hesigb_ch2_offset;
          hesigb_ch2_offset = 0; //reset counter
        end
`endif//RW_NX_CHBW20

        foreach(viterbi_in_item.buffered_data[i]) begin
          rtl_data = viterbi_in_item.buffered_data[i];

          // content channel index 1
          if (viterbi_in_item.buffered_data[i][`VITERBICONTCH])
            matlab_rtl_data = {1'b1, // RTL will set MSB to 1 to select content channel 1
                               ViterbiRefData[hesigb_ch2_offset+3].real_f[`VITERBISBWIDTH-1:0],
                               ViterbiRefData[hesigb_ch2_offset+2].real_f[`VITERBISBWIDTH-1:0],
                               ViterbiRefData[hesigb_ch2_offset+1].real_f[`VITERBISBWIDTH-1:0],
                               ViterbiRefData[hesigb_ch2_offset  ].real_f[`VITERBISBWIDTH-1:0]};
          else
            matlab_rtl_data = {ViterbiRefData[matlab_idx+3].real_f[`VITERBISBWIDTH-1:0],
                               ViterbiRefData[matlab_idx+2].real_f[`VITERBISBWIDTH-1:0],
                               ViterbiRefData[matlab_idx+1].real_f[`VITERBISBWIDTH-1:0],
                               ViterbiRefData[matlab_idx  ].real_f[`VITERBISBWIDTH-1:0]};

          if (rtl_data !== matlab_rtl_data) begin
            `uvm_error(get_type_name(), $sformatf("Viterbi input mismatch idx(%0d/%0d) : RTL (%3d %3d %3d %3d) - Matlab (%3d %3d %3d %3d)",matlab_idx/4,ViterbiRefData.size()/4,
                                                  signed'(rtl_data[19:15]),signed'(rtl_data[14:10]),signed'(rtl_data[9:5]),signed'(rtl_data[4:0]),
                                                  signed'(matlab_rtl_data[19:15]),signed'(matlab_rtl_data[14:10]),signed'(matlab_rtl_data[9:5]),signed'(matlab_rtl_data[4:0])));
            viterbi_in_error_cnt++;
          end
          else begin
            `uvm_info(get_type_name(), $sformatf("Viterbi input samples match idx(%0d/%0d) : RTL (%3d %3d %3d %3d)- Matlab (%3d %3d %3d %3d)",matlab_idx/4,ViterbiRefData.size()/4,
                                                  signed'(rtl_data[19:15]),signed'(rtl_data[14:10]),signed'(rtl_data[9:5]),signed'(rtl_data[4:0]),
                                                  signed'(matlab_rtl_data[19:15]),signed'(matlab_rtl_data[14:10]),signed'(matlab_rtl_data[9:5]),signed'(matlab_rtl_data[4:0])), UVM_DEBUG)
          end // if(rtl_data !== matlab_rtl_data)

          if (viterbi_in_item.buffered_data[i][`VITERBICONTCH])
            hesigb_ch2_offset = hesigb_ch2_offset + 4;
          else
            matlab_idx = matlab_idx + 4;

        end //foreach(viterbi_in_item.buffered_data[i])
      end // for(int offset=0; offset < ViterbiRefData.size();) begin
      `uvm_info(get_type_name(), $sformatf("Viterbi input samples (%0d/%0d) checking done!!!",matlab_idx,ViterbiRefData.size()), UVM_LOW);
    end //begin

  endtask

  //--------------------------------------------------------------------
  // collect mac_phy_seq_item from monitor
  //--------------------------------------------------------------------
  function void modem_rx_scoreboard::write_modem_rx_mac_phy(mac_phy_seq_item i);
    mac_phy_seq_item it;
    it = new();
    it.copy(i);

    if (it.trans == RX) begin
      mac_phy_queue.push_front(it);
      `uvm_info(get_type_name(), $sformatf("Received MAC_PHY item :\n%s", it.sprint()), UVM_HIGH)
    end else begin
      `uvm_info(get_type_name(), $sformatf("Received MAC_PHY item is TX, not RX - skipping"), UVM_HIGH)
    end
  endfunction : write_modem_rx_mac_phy

  //--------------------------------------------------------------------
  // this task is used for checking whether the neccessary checks
  // were activated, during simulation
  //--------------------------------------------------------------------
  function void modem_rx_scoreboard::check_status();

    if (m_cfg.has_deint_agent && !disable_check) begin
      if (compressed_sb_comparison_done != frame_cnt)
        `uvm_error(get_type_name(),
        $sformatf("Checking of the Compressed SB was not done!!!, expected check (%0d) actual checks (%0d) ",
        frame_cnt, compressed_sb_comparison_done))
    end

    if (m_cfg.has_equ_agent && !disable_check) begin
      if (noncompressed_sb_comparison_done != frame_cnt)
        `uvm_error(get_type_name(),
        $sformatf("Checking of the Non-Compressed SB was not done!!!, expected check (%0d) actual checks (%0d) ",
        frame_cnt, noncompressed_sb_comparison_done))
    end

    // empty FIFOs
    if (disable_check) begin
      flush_fifos();
    end
  endfunction : check_status

  //--------------------------------------------------------------------
  // this task is used for flushing all the FIFO's
  //--------------------------------------------------------------------
  function void modem_rx_scoreboard::flush_fifos();

    `uvm_info(get_type_name(),$sformatf("All the FIFO's are being emptied"),UVM_LOW)
    deint_fifo.flush();
    equ_fifo.flush();
    equ1_fifo.flush();
    fftin_fifo.flush();
    fftout_fifo.flush();
    viterbi_in_fifo.flush();
    ldpc_fifo.flush();
    ldpc_data_fifo.flush();

  endfunction : flush_fifos
`endif //_MODEM_RX_SCOREBOARD_SV_
