//////////////////////////////////////////////////////////////////////////////
//  Copyright (C) by RivieraWaves.
//  This module is a confidential and proprietary property of RivieraWaves
//  and a possession or use of this module requires written permission
//  from RivieraWaves.
//----------------------------------------------------------------------------
// $Author: $
// Company          : RivieraWaves
//----------------------------------------------------------------------------
// $Revision: $
// $Date: $
// ---------------------------------------------------------------------------
// Dependencies     : None
// Description      :
// Simulation Notes :
// Synthesis Notes  :
// Application Note :
// Simulator        :
// Parameters       :
// Terms & concepts :
// Bugs             :
// Open issues and future enhancements :
// References       :
// Revision History :
// ---------------------------------------------------------------------------
//
//
//////////////////////////////////////////////////////////////////////////////


`ifndef _MODEM_TX_SCOREBOARD_SV_
`define _MODEM_TX_SCOREBOARD_SV_

`uvm_analysis_imp_decl(_dac)
`uvm_analysis_imp_decl(_modem_tx_mac_phy)
`uvm_analysis_imp_decl(_txcore)
`uvm_analysis_imp_decl(_txmap)
`uvm_analysis_imp_decl(_sig)

class modem_tx_scoreboard extends modem_scoreboard_base;

  `uvm_component_utils(modem_tx_scoreboard)

  uvm_analysis_imp_dac#(modem_stream_seq_item#(`DACWIDTH), modem_tx_scoreboard)   dac_export;
  uvm_analysis_imp_modem_tx_mac_phy#(mac_phy_seq_item, modem_tx_scoreboard) mac_phy_export;
  uvm_analysis_imp_txcore#(modem_stream_seq_item#(`TXCOREDATA_WIDTH), modem_tx_scoreboard) txcore_export;
  uvm_analysis_imp_txmap#(modem_stream_seq_item#(`TXMAPDATA_WIDTH), modem_tx_scoreboard) txmap_export;
  uvm_analysis_imp_sig#(bus_monitor_seq_item, modem_tx_scoreboard) sig_export;

  //--------------------------------------------------------------------
  // variables needed for running Matlab model
  //--------------------------------------------------------------------
  direction_e direction = Tx;
  PPDU_frame  ppdu;
  rf_gpio_t   rf_gpio=`GPIO_WIDTH'd27;
  bit         agcBypass = 0;
  bit   [3:0] modemconf = 4'b100;
  bit   [1:0] controlChannel = 2'b10;
  int         num_dac_samples = 0;
  bit         err_occured = 0;
  bit         err_detected = 0;
  int         frame_num;
  bit [31:0]  primary_channel;
  int         num_dac_checks = 0;
  int         num_txcore_checks = 0;
  bit         last_is_dsss; //indicate that last frame was DSSS/CCK

  //--------------------------------------------------------------------
  // Calculated data from Matlab
  //--------------------------------------------------------------------
  DAC_mu_data_t   TxframeOut_re[];
  DAC_mu_data_t   TxframeOut_im[];
  TXCORE_data_s   TxCoreOut[];
  TXMAP_data_s    TxMapOut[];
  // samples for comparing
  TXCORE_data_s   txcore_data_rtl;
  TXCORE_data_s   txcore_data_matlab;
  DAC_data_s      dac_data_rtl1;
  DAC_data_s      dac_data_matlab1;
`ifdef RW_TXRX_2X2
  DAC_data_s      dac_data_rtl2;
  DAC_data_s      dac_data_matlab2;
`endif
  //--------------------------------------------------------------------
  // Queues of items and single items
  //--------------------------------------------------------------------
  mac_phy_seq_item      mac_phy_item, mac_phy_queue[$];
  modem_stream_seq_item#(`DACWIDTH) dac_item, dac_queue[$];
  modem_stream_seq_item#(`TXCOREDATA_WIDTH) txcore_item, txcore_queue[$];
  TXMAP_data_s          txmap_queue[][$];
  bus_monitor_seq_item  sig_item, sig_queue[$];

  //--------------------------------------------------------------------
  // Constructor
  //--------------------------------------------------------------------
  function new(string name = "modem_tx_scoreboard", uvm_component parent = null);
    super.new(name, parent);
    dac_export        = new("dac_export", this);
    mac_phy_export    = new("mac_phy_export", this);
    txcore_export     = new("txcore_export", this);
    txmap_export      = new("txmap_export", this);
    sig_export        = new("sig_export", this);
    frame_num         = 1;
  endfunction : new

  //--------------------------------------------------------------------
  // UVM phasing - build phase
  //--------------------------------------------------------------------
  virtual function void build_phase(uvm_phase phase);
    super.build_phase(phase);
  endfunction : build_phase

  //--------------------------------------------------------------------
  // UVM phasing - connect phase
  //--------------------------------------------------------------------
  virtual function void connect_phase(uvm_phase phase);
    super.connect_phase(phase);
  endfunction : connect_phase

  //--------------------------------------------------------------------
  // UVM phasing - run phase
  //--------------------------------------------------------------------
  task run_phase(uvm_phase phase);
    super.run_phase(phase);
    fork
`ifdef STANDALONE_PHY
      get_referent_data_modem_only();
`else
      get_referent_data();
`endif//STANDALONE_PHY
    join_none;
  endtask : run_phase

 //--------------------------------------------------------------------
  // in report phase clear all leftover items in queue
  //--------------------------------------------------------------------
  function void report_phase(uvm_phase phase);
    super.report_phase(phase);

`ifdef TX_INTERNAL_CHECK
    if (dac_queue.size() > 0 && last_is_dsss == 0) begin
      `uvm_error(get_type_name(),$sformatf("DAC queue is not empty, items left %0d",dac_queue.size()))
      while (dac_queue.size() > 0) begin
        dac_item = dac_queue.pop_back();
        `uvm_info(get_type_name(), $sformatf("DAC item: %s", dac_item.sprint()), UVM_LOW)
      end
    end

    if (txcore_queue.size() > 0) begin
      `uvm_error(get_type_name(),$sformatf("TXCORE FE queue is not empty, items left %0d",txcore_queue.size()))
      while (txcore_queue.size() > 0) begin
        txcore_item = txcore_queue.pop_back();
        `uvm_info(get_type_name(), $sformatf("TXCORE item: %s", txcore_item.sprint()), UVM_LOW)
      end
    end
`endif//TX_INTERNAL_CHECK

`ifdef STANDALONE_PHY
    check_ref_data_status();
`endif//STANDALONE_PHY
  endfunction : report_phase

  //---------------------------------------------------------------------------
  // list of functions for checking
  //---------------------------------------------------------------------------
  extern task          get_referent_data();
  extern task          get_referent_data_modem_only();
  extern function bit  ignore_dsss_sample(int i);
  extern task          compare_dac_data();
  extern task          compare_txcore_data();
  extern function void compare_txmap_samples();
  extern task          compare_sig_values();
  extern function void write_dac(modem_stream_seq_item#(`DACWIDTH) i);
  extern function void write_txcore(modem_stream_seq_item#(`TXCOREDATA_WIDTH) i);
  extern function void write_txmap(modem_stream_seq_item#(`TXMAPDATA_WIDTH) i);
  extern function void write_modem_tx_mac_phy(mac_phy_seq_item i);
  extern function void write_sig(bus_monitor_seq_item i);
  extern function void check_ref_data_status();
  //---------------------------------------------------------------------------
endclass : modem_tx_scoreboard

//====================================================================
// functions and tasks
//====================================================================

  //--------------------------------------------------------------------
  // when a mac_phy_seq_item is available, calls Matlab to get
  // referent data
  //--------------------------------------------------------------------
  task modem_tx_scoreboard::get_referent_data();
    forever begin
      wait (mac_phy_queue.size()>0); //wait until there is data in the queue
      mac_phy_item = mac_phy_queue.pop_back();
      ppdu = mac_phy_item.frame;
      err_occured = 0;
      if (mac_phy_item.phy_err == 0) begin

        agcBypass = m_cfg.m_radio_cfg.m_rui_cfg.agcBypass;
        primary_channel = m_regmodel.get_mirrored_reg_value("PRIMARYIND");
        ppdu.preamble_header.primary_channel = primary_channel;
        m_modem_cfg.station_file_defined = 0;
        m_modem_cfg.execute(direction, ppdu, rf_gpio, agcBypass);
        // get DAC samples
        m_modem_cfg.get_ref_data(TxframeOut_re, TxframeOut_im);
        // if not DSSS-CCK frames
`ifdef TX_INTERNAL_CHECK
        if (ppdu.preamble_header.leg_rate >= 4'd4) begin
          // get referent output data for TXCORE and TXMAPPER
          m_modem_cfg.get_txcore_data(TxCoreOut);
          m_modem_cfg.get_txmapper_data(TxMapOut);
          compare_sig_values();
          //TODO: TXMAPPER checking disabled for now!
          //compare_txmap_samples();
          compare_txcore_data();
        end//if not DSSS-CCK frame
`endif//TX_INTERNAL_CHECK
        compare_dac_data();
        frame_num++;
      end else begin
        // PHY asserted MAC-PHY error line. Check if this was expected
        `uvm_error(get_type_name(), $sformatf("Received frame with phy_err flag set, ignoring this frame for comparison..."))
      end//else
    end//forever
  endtask: get_referent_data

  //--------------------------------------------------------------------
  // Get referent data in case of Modem only TB, PPDU frame is provided in
  // configuration so no waiting for item on MAC-PHY IF
  //--------------------------------------------------------------------
  task modem_tx_scoreboard::get_referent_data_modem_only();
    forever begin
      wait (m_cfg.ppdu_tx_list.size()>0); //wait until there is data in the queue
      ppdu = m_cfg.ppdu_tx_list.pop_back();
      err_occured = 0;
      if (last_is_dsss) dac_queue.delete();

      // first check if there is leftover referen data not checked
      check_ref_data_status();

      agcBypass = m_cfg.m_radio_cfg.m_rui_cfg.agcBypass;
      primary_channel = m_regmodel.get_mirrored_reg_value("PRIMARYIND");
      ppdu.preamble_header.primary_channel = primary_channel;
      m_modem_cfg.execute(direction, ppdu, rf_gpio, agcBypass);
      // get DAC samples
      m_modem_cfg.get_ref_data(TxframeOut_re, TxframeOut_im);
      // if not DSSS-CCK frames
`ifdef TX_INTERNAL_CHECK
      if (ppdu.preamble_header.leg_rate >= 4'd4) begin
        // get referent output data for TXCORE and TXMAPPER
        m_modem_cfg.get_txcore_data(TxCoreOut);
        m_modem_cfg.get_txmapper_data(TxMapOut);
      end//if not DSSS-CCK frame
`endif//TX_INTERNAL_CHECK
      // start checking in parallel
      fork
        // if not DSSS-CCK frames do checking
        if (ppdu.preamble_header.leg_rate >= 4'd4) compare_sig_values();
        //TODO: TXMAPPER checking disabled for now!
        //if (ppdu.preamble_header.leg_rate >= 4'd4) compare_txmap_samples();
        if (ppdu.preamble_header.leg_rate >= 4'd4) compare_txcore_data();
        compare_dac_data();
      join
      frame_num++;
      last_is_dsss = (ppdu.preamble_header.leg_rate < 4'd4) ? 1'b1 : 1'b0;
    end//forver
  endtask: get_referent_data_modem_only

  //--------------------------------------------------------------------
  // Helper function to check if a sample should be compared for DSSS/CCK
  // (for DSSS/CCK, the first and last 200 DAC samples shouldn't be
  // compared)
  //--------------------------------------------------------------------
  function bit modem_tx_scoreboard::ignore_dsss_sample(int i);
    if (   (ppdu.ppdu_format == NON_HT)
        && (ppdu.preamble_header.leg_rate < 'd4)
        && (   (i <= `NUM_OF_SKIPPED_DSSS_SAMPLES)
`ifdef STANDALONE_PHY
            // when modem standalone testing number of DAC samples are not
            // available since checking is done in parallel, so total number
            // of referent samples are used to determine upper checking limit
            || (i >= (TxframeOut_re.size() - `NUM_OF_SKIPPED_DSSS_SAMPLES))))
`else
            || (i >= (num_dac_samples - `NUM_OF_SKIPPED_DSSS_SAMPLES))))
`endif
      return 1;
    else
      return 0;
  endfunction: ignore_dsss_sample

  //--------------------------------------------------------------------
  // check before fetching new referent data from matlab, did we chcked all
  // referent data samples with RTL samples, since there can be situation
  // where RTL wont provide proper amount of data samples
  //--------------------------------------------------------------------
  function void modem_tx_scoreboard::check_ref_data_status();
    if (num_txcore_checks != TxCoreOut.size() && last_is_dsss == 0)
      `uvm_error(get_type_name(),
      $sformatf("TXCORE samples referent data not fully checked! Samples left %0d",TxCoreOut.size()-num_txcore_checks))

    if (num_dac_checks != TxframeOut_re.size())
      `uvm_error(get_type_name(),
      $sformatf("DAC samples referent data not fully checked! Samples left %0d",TxframeOut_re.size()-num_dac_checks))

    //reset counters
    num_dac_checks = 0;
    num_txcore_checks = 0;
  endfunction : check_ref_data_status

  //--------------------------------------------------------------------
  // Compares referent and dac data
  //--------------------------------------------------------------------
  task modem_tx_scoreboard::compare_dac_data();
    int fhandle;          // file handle
    bit signed [`DACWIDTH-1:0] matlab_minus_re, matlab_minus_im;
    bit signed [`DACWIDTH-1:0] matlab_plus_re, matlab_plus_im;

`ifdef DEBUG_SAMPLES
    // dumping TXCORE samples to file for easy debug
    fhandle = $fopen($sformatf("dac_samples_%0d.txt",frame_num),"w");
    $fwrite(fhandle, $sformatf("#\tMatlab\t\tTWO'S COMPLE\t\tRTL\n"));
    $fwrite(fhandle, $sformatf("#REAL\tIMAG\t|\tREAL\tIMAG\t|\tREAL\tIMAG\n"));
`endif//DEBUG_SAMPLES

    `uvm_info(get_type_name(),
      $sformatf("Comparison started - number of referent samples: %0d number of received samples: %0d",
      TxframeOut_re.size(), dac_queue.size()), UVM_FULL)

`ifndef STANDALONE_PHY
    if (((ppdu.ppdu_format != NON_HT) || (ppdu.preamble_header.leg_rate > 'd3)) && (TxframeOut_re.size() != dac_queue.size()))
      `uvm_error(get_type_name(),$sformatf("DAC output array size mistmatch. %0d samples from RTL and %0d samples from Matlab",
       dac_queue.size(),TxframeOut_re.size()))
`endif

    for (int i=0; i<TxframeOut_re.size(); i++) begin
      num_dac_checks++;

      if ((ignore_dsss_sample(i) == 0) || (ppdu.ppdu_format != NON_HT) ||
`ifdef STANDALONE_PHY
         ((ignore_dsss_sample(i) == 1) && (i <= `NUM_OF_SKIPPED_DSSS_SAMPLES))) begin
`else
         ((ignore_dsss_sample(i) == 1) && (i < (num_dac_samples - `NUM_OF_SKIPPED_DSSS_SAMPLES)))) begin
`endif//STANDALONE_PHY
        // make sure there is a DAC sample ready, except for last 200 samples for DSSS
        wait (dac_queue.size()>0);
        dac_item = dac_queue.pop_back();
      end

      // for DSSS/CCK, the first and last 200 samples have to be ignored
      if ( ignore_dsss_sample(i) == 0 || (ppdu.ppdu_format != NON_HT)) begin

        matlab_minus_re = TxframeOut_re[i][0] - `SAMPLE_TOLERANCE;
        matlab_minus_im = TxframeOut_im[i][0] - `SAMPLE_TOLERANCE;
        matlab_plus_re  = TxframeOut_re[i][0] + `SAMPLE_TOLERANCE;
        matlab_plus_im  = TxframeOut_im[i][0] + `SAMPLE_TOLERANCE;

        err_detected = 0;
        if (!(   dac_item.DataI0 <= matlab_plus_re  && dac_item.DataQ0 <= matlab_plus_im
              && dac_item.DataI0 >= matlab_minus_re && dac_item.DataQ0 >= matlab_minus_im)) begin
          `uvm_error(get_type_name(),
          $sformatf("DAC samples mismatch RTL (re:%0d, im:%0d) MATLAB (re:%0d, im:%0d) sample #%0d Tx path 1 (tolerance: +/-%0d).",
          dac_item.DataI0, dac_item.DataQ0, TxframeOut_re[i][0], TxframeOut_im[i][0], i, `SAMPLE_TOLERANCE))
          err_occured = 1;
          err_detected = 1;
        end
        else begin
          `uvm_info(get_type_name(), $sformatf("DAC samples match RTL (re:%0d, im:%0d) MATLAB (re:%0d, im:%0d) sample #%0d Tx path 1.",
            dac_item.DataI0, dac_item.DataQ0, TxframeOut_re[i][0], TxframeOut_im[i][0], i), UVM_FULL)
        end

`ifdef RW_TXRX_2X2
        matlab_minus_re = TxframeOut_re[i][1] - `SAMPLE_TOLERANCE;
        matlab_minus_im = TxframeOut_im[i][1] - `SAMPLE_TOLERANCE;
        matlab_plus_re  = TxframeOut_re[i][1] + `SAMPLE_TOLERANCE;
        matlab_plus_im  = TxframeOut_im[i][1] + `SAMPLE_TOLERANCE;

        if (!(   dac_item.DataI1 <= matlab_plus_re  && dac_item.DataQ1 <= matlab_plus_im
              && dac_item.DataI1 >= matlab_minus_re && dac_item.DataQ1 >= matlab_minus_im)) begin
          `uvm_error(get_type_name(),
          $sformatf("DAC samples mismatch RTL (re:%0d, im:%0d) MATLAB (re:%0d, im:%0d) sample #%0d Tx path 2 (tolerance: +/-%0d).",
          dac_item.DataI1, dac_item.DataQ1, TxframeOut_re[i][1], TxframeOut_im[i][1], i, `SAMPLE_TOLERANCE))
          err_occured = 1;
          err_detected = 1;
        end
        else begin
          `uvm_info(get_type_name(), $sformatf("DAC samples match RTL (re:%0d, im:%0d) MATLAB (re:%0d, im:%0d) sample #%0d Tx path 2.",
            dac_item.DataI1, dac_item.DataQ1, TxframeOut_re[i][1], TxframeOut_im[i][1], i), UVM_FULL)
        end
`endif//RW_TXRX_2X2

`ifdef DEBUG_SAMPLES
      $fwrite(fhandle, $sformatf("%0d\t%0d\t|\t%0d\t%0d\t|\t%0d\t%0d\n",
        TxframeOut_re[i][0],
        TxframeOut_im[i][0],
        two_complement(TxframeOut_re[i][0]),
        two_complement(TxframeOut_im[i][0]),
        dac_item.DataI0,
        dac_item.DataQ0));
      // delay added so data and referent samples can be visible in waveform
      // when debugging is performed
      dac_data_rtl1.re_f = dac_item.DataI0;
      dac_data_rtl1.im_f = dac_item.DataQ0;
      dac_data_matlab1.re_f = TxframeOut_re[i][0];
      dac_data_matlab1.im_f = TxframeOut_im[i][0];
      #1ps;
`endif

      end// if ignore_dsss_sample

    end // for

    num_dac_samples = 0;

    if (dac_queue.size() > 0) begin
      `uvm_info(get_type_name(), $sformatf("DAC samples still in queue, referent data exhausted"), UVM_FULL)

      do begin
        dac_item = dac_queue.pop_back();
        `uvm_info(get_type_name(), $sformatf("Removing DAC item {0x%x,0x%x}",dac_item.DataI0,dac_item.DataQ0 ), UVM_FULL)
      end while (dac_queue.size() != 0 );
    end

    if (!err_occured)
      `uvm_info(get_type_name(),"DAC samples OK!!!",UVM_LOW)

`ifdef DEBUG_SAMPLES
    $fclose(fhandle);
`endif//DEBUG_SAMPLES
  endtask: compare_dac_data

  //--------------------------------------------------------------------
  // Compares referent and txcore data
  //--------------------------------------------------------------------
  task modem_tx_scoreboard::compare_txcore_data();
    int fhandle;          // file handle
    TXCORE_data_s matlab_minus, matlab_plus;

`ifdef DEBUG_SAMPLES
    // dumping TXCORE samples to file for easy debug
    fhandle = $fopen($sformatf("txcore_samples_%0d.txt",frame_num),"w");
    $fwrite(fhandle, $sformatf("#\tMatlab\t\t\t\tRTL\n"));
    $fwrite(fhandle, $sformatf("# REAL \t IMAG \t\t REAL \t IMAG\n"));
`endif//DEBUG_SAMPLES

`ifndef STANDALONE_PHY
    if (TxCoreOut.size() != txcore_queue.size())
      `uvm_error(get_type_name(),$sformatf("TXCORE output array size mistmatch. %0d samples from Matlab and %0d samples from RTL",
       TxCoreOut.size(), txcore_queue.size()))
`endif//STANDALONE_PHY

    for (int i=0; i<TxCoreOut.size(); i++) begin
      num_txcore_checks++;

      wait (txcore_queue.size() > 0);
      txcore_item = txcore_queue.pop_back();
      txcore_data_rtl = txcore_item.stream_data;

      txcore_data_matlab = TxCoreOut[i];

      matlab_minus = updateTXCORE(txcore_data_matlab,-`SAMPLE_TOLERANCE);
      matlab_plus  = updateTXCORE(txcore_data_matlab,`SAMPLE_TOLERANCE);

      err_detected = 0;
      if (!(   txcore_data_rtl.re_f <= matlab_plus.re_f  && txcore_data_rtl.im_f <= matlab_plus.im_f
            && txcore_data_rtl.re_f >= matlab_minus.re_f && txcore_data_rtl.im_f >= matlab_minus.im_f)) begin
        `uvm_error(get_type_name(),$sformatf("TXCORE output mismatch RTL (%s) Matlab (%s) sample: #%0d (tolerance: +/-%0d)",
        TXCORE2string(txcore_data_rtl), TXCORE2string(txcore_data_matlab), i, `SAMPLE_TOLERANCE))
        err_occured = 1;
        err_detected = 1;
      end
`ifdef DEBUG_SAMPLES
      $fwrite(fhandle, $sformatf("%6d\t%6d\t%6d\t%6d\n",
        txcore_data_matlab.re_f,
        txcore_data_matlab.im_f,
        txcore_data_rtl.re_f,
        txcore_data_rtl.im_f));
      // delay added so data and referent samples can be visible in waveform
      // when debugging is performed
      #1ps;
`endif
    end//for

    if (txcore_queue.size() > 0) begin
      `uvm_info(get_type_name(), $sformatf("TXCORE samples still in queue, referent data exhausted"), UVM_FULL)

      do begin
        txcore_item = txcore_queue.pop_back();
        `uvm_info(get_type_name(), $sformatf("Removing TXCORE item {0x%x,0x%x}",txcore_data_matlab.re_f,txcore_data_matlab.im_f ), UVM_FULL)
      end while (txcore_queue.size() != 0 );
    end

    if (!err_occured)
      `uvm_info(get_type_name(),"TXCORE Output samples OK!!!",UVM_LOW)

`ifdef DEBUG_SAMPLES
    $fclose(fhandle);
`endif//DEBUG_SAMPLES
  endtask: compare_txcore_data

  //--------------------------------------------------------------------
  // checking of TX mapper referente samples
  //--------------------------------------------------------------------
  function void modem_tx_scoreboard::compare_txmap_samples();
    int i;
    int fhandle;          // file handle
    TXMAP_data_s  map_data;

`ifdef DEBUG_SAMPLES
    // dumping TXMAPPER samples to file for easy debug
    fhandle = $fopen($sformatf("txmapper_samples_%0d.txt",frame_num),"w");
    $fwrite(fhandle, $sformatf("#\t\tMatlab\t\t\t\tRTL\n"));
    $fwrite(fhandle, $sformatf("#REAL\tIMAG\tINDCA\tINDSYMB\tREAL\tIMAG\tINDCA\tINDSYMB\n"));
`endif//DEBUG_SAMPLES

    // sort first by subcarrier index then by symbol count to be aligned with
    // Matlab reference data
    foreach (txmap_queue[i]) begin
      txmap_queue[i].sort(sample) with (sample.index_f);
      foreach (txmap_queue[i][j])
        `uvm_info(get_type_name(),$sformatf("SORTED[%0d]: %s",i,TXMAP2string(txmap_queue[i][j])),UVM_DEBUG);
    end

    i = 0;
    foreach (txmap_queue[symb]) begin
      while (txmap_queue[symb].size() > 0) begin
        map_data = txmap_queue[symb].pop_front();

        while (i < TxMapOut.size()) begin
          if (   map_data.index_f    == TxMapOut[i].index_f
              && map_data.symb_idx_f == TxMapOut[i].symb_idx_f)
            break; //found match
          else
            i++;
        end

        if (map_data.re_f != TxMapOut[i].re_f || map_data.im_f != TxMapOut[i].im_f) begin
          `uvm_info(get_type_name(),$sformatf("TXMAPPER sample lookup index %0d", i),UVM_LOW)
          `uvm_error(get_type_name(),
            $sformatf("TXMAPPER samples mismatch RTL (%s) Matlab (%s)",
            TXMAP2string(map_data),TXMAP2string(TxMapOut[i])))
          err_occured = 1;
        end

`ifdef DEBUG_SAMPLES
        $fwrite(fhandle, $sformatf("%5d\t%5d\t%5d\t%5d\t%5d\t%5d\t%5d\t%5d\n",
          TxMapOut[i].re_f,
          TxMapOut[i].im_f,
          TxMapOut[i].index_f,
          TxMapOut[i].symb_idx_f,
          map_data.re_f,
          map_data.im_f,
          map_data.index_f,
          map_data.symb_idx_f));
`endif//DEBUG_SAMPLES
      end// while
    end//foreach

    // delete dinamyc arrays for new frame samples
    txmap_queue.delete();

    if (!err_occured)
      `uvm_info(get_type_name(),"TXCORE-MAPPER samples OK!!!",UVM_LOW)

`ifdef DEBUG_SAMPLES
    $fclose(fhandle);
`endif//DEBUG_SAMPLES
  endfunction : compare_txmap_samples

  //--------------------------------------------------------------------
  // compare SIG values with TxVector parameters
  //--------------------------------------------------------------------
  task modem_tx_scoreboard::compare_sig_values();
    wait (sig_queue.size() != 0);
    sig_item = sig_queue.pop_back();

    // compare LSIG
    if (ppdu.ppdu_format != HT_GF) begin
      if (sig_item.l_sig != m_modem_cfg.l_sig) begin
        `uvm_info(get_type_name(),$sformatf("RTL    (%p)",sig_item.l_sig),UVM_LOW)
        `uvm_info(get_type_name(),$sformatf("Matlab (%p)",m_modem_cfg.l_sig),UVM_LOW)
        `uvm_error(get_type_name(),$sformatf("LSIG mismatch!"))
      end
      else begin
        `uvm_info(get_type_name(),$sformatf("LSIG OK!"),UVM_LOW)
      end
    end

    // compare HTSIG
    if (ppdu.ppdu_format inside {HT_GF,HT_MF}) begin
      if (sig_item.ht_sig != m_modem_cfg.ht_sig) begin
        `uvm_info(get_type_name(),$sformatf("RTL    (%p)",sig_item.ht_sig),UVM_LOW)
        `uvm_info(get_type_name(),$sformatf("Matlab (%p)",m_modem_cfg.ht_sig),UVM_LOW)
        `uvm_error(get_type_name(),$sformatf("HTSIG mismatch!"))
      end
      else begin
        `uvm_info(get_type_name(),$sformatf("HTSIG OK!"),UVM_LOW)
      end
    end

    // compare HESIGA
    if (ppdu.ppdu_format >= HE_SU) begin
      case (ppdu.ppdu_format)
        HE_SU,HE_EXT_SU: begin
          if (sig_item.he_siga != m_modem_cfg.he_siga_su) begin
            `uvm_info(get_type_name(),$sformatf("RTL    (%p)",he_siga_su_s'(sig_item.he_siga)),UVM_LOW)
            `uvm_info(get_type_name(),$sformatf("Matlab (%p)",m_modem_cfg.he_siga_su),UVM_LOW)
            `uvm_error(get_type_name(),$sformatf("HESIGA mismatch!"))
          end
          else begin
            `uvm_info(get_type_name(),$sformatf("HESIGA OK!"),UVM_LOW)
          end
        end//HE_SU
        HE_TB: begin
          if (sig_item.he_siga != m_modem_cfg.he_siga_tb) begin
            `uvm_info(get_type_name(),$sformatf("RTL    (%p)",he_siga_tb_s'(sig_item.he_siga)),UVM_LOW)
            `uvm_info(get_type_name(),$sformatf("Matlab (%p)",m_modem_cfg.he_siga_tb),UVM_LOW)
            `uvm_error(get_type_name(),$sformatf("HESIGA mismatch!"))
          end
          else begin
            `uvm_info(get_type_name(),$sformatf("HESIGA OK!"),UVM_LOW)
          end
        end//HE_TB
        HE_MU: begin
          if (sig_item.he_siga != m_modem_cfg.he_siga_mu) begin
            `uvm_info(get_type_name(),$sformatf("RTL    (%p)",he_siga_mu_s'(sig_item.he_siga)),UVM_LOW)
            `uvm_info(get_type_name(),$sformatf("Matlab (%p)",m_modem_cfg.he_siga_mu),UVM_LOW)
            `uvm_error(get_type_name(),$sformatf("HESIGA mismatch!"))
          end
          else begin
            `uvm_info(get_type_name(),$sformatf("HESIGA OK!"),UVM_LOW)
          end
        end//HE_MU
      endcase
    end//if (ppdu.ppdu_format >= HE_SU)
  endtask : compare_sig_values

  //--------------------------------------------------------------------
  // collect modem_stream_seq_item from monitor
  //--------------------------------------------------------------------
  function void modem_tx_scoreboard::write_dac(modem_stream_seq_item#(`DACWIDTH) i);
    modem_stream_seq_item#(`DACWIDTH)  it;
    it = new();
    it.copy(i);

    dac_queue.push_front(it);
    num_dac_samples++;
    `uvm_info(get_type_name(), $sformatf("Received DAC item (dac_queue size: %0d) :\n%s", dac_queue.size(), it.sprint()), UVM_HIGH)
  endfunction : write_dac

  //--------------------------------------------------------------------
  // collect modem_stream_seq_item from TXCORE monitor
  //--------------------------------------------------------------------
  function void modem_tx_scoreboard::write_txcore(modem_stream_seq_item#(`TXCOREDATA_WIDTH) i);
    modem_stream_seq_item#(`TXCOREDATA_WIDTH)  it;
    it = new();
    it.copy(i);

    txcore_queue.push_front(it);
    `uvm_info(get_type_name(),
    $sformatf("Received TXCORE item (txcore_queue size: %0d) :\n%s", txcore_queue.size(), it.sprint()), UVM_HIGH)
  endfunction : write_txcore

  //--------------------------------------------------------------------
  // collect modem_stream_seq_item from TXMAPPER monitor
  //--------------------------------------------------------------------
  function void modem_tx_scoreboard::write_txmap(modem_stream_seq_item#(`TXMAPDATA_WIDTH) i);
    TXMAP_data_s txmap;
    int          size;

    txmap = i.stream_data;

    size = txmap_queue.size();
    if (size == 0)
      txmap_queue = new[1]; //create first symbol entry
    else if (txmap.symb_idx_f != size-1)
      txmap_queue = new[size+1](txmap_queue); //add new queue

    txmap_queue[txmap_queue.size()-1].push_front(txmap);
    `uvm_info(get_type_name(),
    $sformatf("Received TXMAPPER item (txmap_queue size: %0d) :\n%s", txmap_queue.size(), TXMAP2string(txmap)), UVM_HIGH)
  endfunction : write_txmap

  //--------------------------------------------------------------------
  // collect mac_phy_seq_item from monitor
  //--------------------------------------------------------------------
  function void modem_tx_scoreboard::write_modem_tx_mac_phy(mac_phy_seq_item i);

    mac_phy_seq_item it;
    it = new();
    it.copy(i);

    if (it.trans == TX) begin
      mac_phy_queue.push_front(it);
      `uvm_info(get_type_name(), $sformatf("Received MAC_PHY item :\n%s", it.sprint()), UVM_HIGH)
    end else begin
      `uvm_info(get_type_name(), $sformatf("Received MAC_PHY item is RX, not TX - skipping"), UVM_HIGH)
    end
  endfunction : write_modem_tx_mac_phy

  //--------------------------------------------------------------------
  // collect bus_monitor_seq_item from monitor
  //--------------------------------------------------------------------
  function void modem_tx_scoreboard::write_sig(bus_monitor_seq_item i);

    bus_monitor_seq_item it;
    it = new i;

    sig_queue.push_front(it);
    `uvm_info(get_type_name(), $sformatf("Received BUS monitor item, SIG values"), UVM_HIGH)

  endfunction : write_sig

`endif //_MODEM_TX_SCOREBOARD_SV_
