#!/bin/bash
#/////////////////////////////////////////////////////////////////////////////
#/  Copyright (C) by RivieraWaves.
#/  This module is a confidential and proprietary property of RivieraWaves
#/  and a possession or use of this module requires written permission
#/  from RivieraWaves.
#/----------------------------------------------------------------------------
#/ $Author: jvanthou $
#/ Company          : RivieraWaves
#/----------------------------------------------------------------------------
#/ $Revision: 30649 $
#/ $Date: 2017-06-06 11:33:55 +0200 (Tue, 06 Jun 2017) $
#/ ---------------------------------------------------------------------------
#/ Dependencies     : None
#/ Description      : Automatic register extraction script
#/ Application Note :
#/ Terms & concepts :
#/ Bugs             :
#/ Open issues and future enhancements :
#/ References       :
#/ Revision History :
#/ ---------------------------------------------------------------------------
#/////////////////////////////////////////////////////////////////////////////


#--------------------------------------------------------------------#
#--------------------------------------------------------------------#
#--- Display Help                                                 ---#
#--------------------------------------------------------------------#
#--------------------------------------------------------------------#
DisplayHelp()
{
  echo "
    This script creates the RAL register model based on the XLS register definition.

    Options:
      -xls    <xls filename>    : Excel file with the register definition

      -name   <blockname>       : Name of the block

      -rcov   <coverage option> : Sets desired register coverage option. Possible values are:
                                    UVM_NO_COVERAGE    - No register coverage collected
                                    UVM_CVR_REG_BITS   - Individual register bits coverage collected
                                    UVM_CVR_ADDR_MAP   - Individual register and memory addresses
                                    UVM_CVR_FIELD_VALS - Field values coverage
                                    UVM_CVR_ALL        - All coverage models are collected
                                  If not specified, the default value is UVM_CVR_FIELD_VALS

      -bcov   <coverage option> : Sets desired register block coverage option(s). Possible values same
                                  as above. If not specified, the default value is UVM_CVR_ADDR_MAP

      -endian <byte ordering>   : Specifies byte ordering. Possible options are:
                                    UVM_NO_ENDIAN     - Byte ordering not applicable
                                    UVM_LITTLE_ENDIAN - Least-significant bytes first in consecutive addresses
                                    UVM_BIG_ENDIAN    - Most-significant bytes first in consecutive addresses
                                    UVM_LITTLE_FIFO   - Least-significant bytes first at the same address
                                    UVM_BIG_FIFO      - Most-significant bytes first at the same address
                                  If not specified, the default value is UVM_LITTLE_ENDIAN.

      -basea <file>             : Specifies the Verilog file which contains APB base addresses. Default value is
                                  $SOURCESLIB/SIMS_UVM/defines/tbDefines.v

      -h | -help                : Display this message

    Usage :
      gen_reg.sh -xls <xls filename> -name <block name> -rcov <coverage option> -bcov <coverage option> -endian <byte ordering> -basea <file name>

  "
}

#--------------------------------------------------------------------#
#--- PrintErrorWarning <message>                                  ---#
#--------------------------------------------------------------------#
PrintErrorWarning ()
{
  echo "ERROR: $*"
  echo ""
  exit 2
}


NAME=0
XLS=0
RCOV=0
BCOV=0
ENDIAN=0
BASEA=0

while test $# != 0
do

  arg="$1"
  OPTION=`echo $arg | sed 's/^-//'`
  shift
  case ${OPTION}
  in
    name)
      NAME="$1"
      shift
    ;;
    xls)
      XLS="$1"
      shift
    ;;
    rcov)
      RCOV="$1"
      shift
    ;;
    bcov)
      BCOV="$1"
      shift
    ;;
    endian)
      ENDIAN="$1"
      shift
    ;;
    basea)
      BASEA="$1"
      shift
    ;;
    h | help)
      #------------------------------------------------------------------------#
      #--- Display Help                                                     ---#
      #------------------------------------------------------------------------#
      DisplayHelp
      exit 1
    ;;
    *)
      #------------------------------------------------------------------------#
      #--- Unknown Option                                                   ---#
      #------------------------------------------------------------------------#
      echo "Unknown Option ${OPTION}"
      DisplayHelp
      exit 1
    ;;
  esac
done


if [ ! -f $XLS ]
then
  PrintErrorWarning "$XLS filename not found"
fi

if [ $NAME == "0" ]
then
  PrintErrorWarning "The block name has not been set"
fi

XML=`echo $XLS | sed 's/.*\///g' | sed 's/.xls/.xml/'`
echo "Generating register block for $NAME using register definition from $XLS"
$SOURCESLIB/bin/gen_reg/reg_xls2xml.pl $XLS

if [ ! -f $XML ]
then
  PrintErrorWarning "$XML file not found!"
fi

if [ $RCOV == "0" ]
then
  RCOV='UVM_CVR_FIELD_VALS'
fi

if [ $BCOV == "0" ]
then
  BCOV='UVM_CVR_ADDR_MAP'
fi

if [ $ENDIAN == "0" ]
then
  ENDIAN='UVM_LITTLE_ENDIAN'
fi

if [ $BASEA == "0" ]
then
  BASEA=$SOURCESLIB/SIMS_UVM/defines/tbDefines.v
fi

if [ ! -f $BASEA ]
then
   PrintErrorWarning "$BASEA file not found!"
fi

/usr/bin/python $SOURCESLIB/bin/gen_reg/xml2ral.py -x $XML -e $NAME -b $BCOV -r $RCOV -n $ENDIAN -a $BASEA

