#/////////////////////////////////////////////////////////////////////////////
#/  Copyright (C) by RivieraWaves.
#/  This module is a confidential and proprietary property of RivieraWaves
#/  and a possession or use of this module requires written permission
#/  from RivieraWaves.
#/----------------------------------------------------------------------------
#/ $Author          : zpele $
#/ Company          : RivieraWaves
#/----------------------------------------------------------------------------
#/ $Revision: 35482 $
#/ $Date: 2018-10-03 10:49:59 +0200 (Wed, 03 Oct 2018) $
#/ ---------------------------------------------------------------------------
#/ Dependencies     : uvm_ral_snippets.py, ahb_base_addresses.txt
#/ Description      : Script to create a top level register model.
#/                    Type raltop -h for help.
#/ Application Note :
#/ Terms & concepts :
#/ Bugs             :
#/ Open issues and future enhancements :
#/ References       :
#/ Revision History :
#/ ---------------------------------------------------------------------------
#/
#/////////////////////////////////////////////////////////////////////////////

#Parse XML register file and create a verilog file implementing the registers

import os, sys, getopt, re, string
import uvm_ral_snippets
from string import *
from re import *

# Create top level register model
def create_top_regmodel(rcov,bcov,endian,top,offset):

  # Open .sv file and place header
  ral_file=open(top + '_reg_block.sv','w')
  ral_file.write(uvm_ral_snippets.header.substitute(AUTHOR=os.environ['USER'], ENTITY=top))

  # Top level register block definition
  ral_file.write(uvm_ral_snippets.regblock_start.substitute(ENTITY= top))

  # Open temporary file containing all processed blocks and their base addresses
  tmpfile = "ahb_base_addresses.txt"
  base_addr = ""

  reg_block_array=[]
  reg_build_array=[]
  submap_array=[]

  for line in open(tmpfile):
    fields = line.split()
    entity_name = fields[0]
    base_addr   = fields[1]
    reg_block_array.append(uvm_ral_snippets.top_list_sub_blocks.substitute(ENTITY=entity_name))
    reg_build_array.append(uvm_ral_snippets.top_reg_block_create_build.substitute(ENTITY=entity_name))
    submap_array.append(uvm_ral_snippets.top_regs_add_submap.substitute(TOP=top, ENTITY=entity_name, OFFSET=base_addr))

  ral_file.write("\n".join(reg_block_array))
  ral_file.write(uvm_ral_snippets.top_reg_block_new.substitute(ENTITY=top, REGS_COVERAGE=rcov))
  ral_file.write("\n".join(reg_build_array))
  ral_file.write(uvm_ral_snippets.regs_create_map.substitute(ENTITY=top, BASE_REG_MAP_ADDR=offset , REG_MAP_WIDTH_BYTES=4, REG_MAP_ENDIAN=endian))
  ral_file.write("\n".join(submap_array))
  ral_file.write(uvm_ral_snippets.reg_block_end.substitute(ENTITY=top))

# Finally, create a package file for compilation
def create_pkg_file(top):

  # Open .sv file and place header
  ral_file=open('REGMODEL_pkg.sv','w')
  ral_file.write(uvm_ral_snippets.pkg_header.substitute(AUTHOR=os.environ['USER']))

  include_array=[]
  tmpfile = "ahb_base_addresses.txt"

  for line in open(tmpfile):
    fields = line.split()
    entity_name = fields[0]
    include_array.append(uvm_ral_snippets.pkg_include.substitute(ENTITY=entity_name))

  ral_file.write("\n".join(include_array))
  ral_file.write(uvm_ral_snippets.pkg_footer.substitute(TOP=top))

def display_help():
  print "\n    Usage:"
  print "       raltop.py "
  print "       raltop.py -h|--help -r|--rcov <register coverage option> -b|--bcov <register block coverage option> -n|--endian <byte ordering> -t|--top <top entity name> -o|--offset <memory map offset>"
  print "    Creates a UVM register model for the top level entity\n"
  print "    The list of register blocks and their base addresses is read from ahb_base_address.txt."
  print "       Example: python\n"

def main(argv):

  # Get arguments from command line
  short = 'r:b:n:t:o:h'
  long = ('help', 'rcov', 'bcov', 'endian', 'top', 'offset')
  try:
    opts, args = getopt.getopt(sys.argv[1:], short, long)
  except getopt.GetoptError, exc:
     print exc
     display_help()
     return 1

  # Error in arguments
  if args:
    print args
    display_help()
    return 1

  # Default values
  rcov   = 'UVM_CVR_FIELD_VALS'
  bcov   = 'UVM_CVR_ADDR_MAP'
  endian = 'UVM_LITTLE_ENDIAN'
  top    = 'TOP'
  offset = '0'

  for opt,val in opts:
    if opt in ('-h', '--help'):
      display_help()
      return 0
    elif opt in ('-r', '--rcov'):
      rcov = val
    elif opt in ('-b', '--bcov'):
      bcov = val
    elif opt in ('-n', '--endian'):
      endian = val
    elif opt in ('-t', '--top'):
      top = val
    elif opt in ('-o', '--offset'):
      offset = val

  create_top_regmodel(rcov,bcov,endian,top,offset)
  create_pkg_file(top)

# Execute main and exit
if __name__ == '__main__':
  status = main(sys.argv)
  sys.exit(status)
