#/////////////////////////////////////////////////////////////////////////////
#/  Copyright (C) by RivieraWaves.
#/  This module is a confidential and proprietary property of RivieraWaves
#/  and a possession or use of this module requires written permission
#/  from RivieraWaves.
#/----------------------------------------------------------------------------
#/ $Author          : jvanthou $
#/ Company          : RivieraWaves
#/----------------------------------------------------------------------------
#/ $Revision: 34008 $
#/ $Date: 2018-05-18 10:30:14 +0200 (Fri, 18 May 2018) $
#/ ---------------------------------------------------------------------------
#/ Dependencies     : uvm_ral_snippets.py
#/ Description      : Script to create a verilog file describing APB registers
#/                    from an XML description. Type xml2ral -h for help.
#/ Application Note :
#/ Terms & concepts :
#/ Bugs             :
#/ Open issues and future enhancements :
#/ References       :
#/ Revision History :
#/ ---------------------------------------------------------------------------
#/
#/////////////////////////////////////////////////////////////////////////////

#Parse XML register file and create a verilog file implementing the registers

import os, sys, getopt, re, string
import xml.dom.minidom
import uvm_ral_snippets
from string import *
from re import *

# Register class (register info)
class register_class:
  def __init__(self, name=None, add_offset=None, hw_access=None, sw_access=None, reset=None, define=None, verilog_addr=None, shortname=None):
    self.name = name
    self.add_offset = add_offset
    self.hw_access = hw_access
    self.sw_access = sw_access
    self.reset = reset
    self.define = define
    self.verilog_addr = verilog_addr
    self.shortname = shortname


# Field class (register field info)
class field_class:
  def __init__(self, name=None, position=None, width=None, reset=None, typ=None, hw=None, sw=None,
               msb_position=None, msb=None, verilogrange=None, fixed_msb=None, str_binreset=None):
    self.name = name
    self.position = position
    self.width = width
    self.reset = reset
    self.typ = typ
    self.hw = hw
    self.sw = sw
    self.msb_position = msb_position
    self.msb = msb
    self.verilogrange = verilogrange
    self.fixed_msb = fixed_msb
    self.str_binreset = str_binreset


# Integer to Binary string conversion
def dec_2_bin(n10, size, ntype):

  if n10 <0:
      n10 +=  1 << (size+1)

  size = size -1
  lsb = '0'
  if n10 < 0:
    lsb = '1'
    n10 +=  1 << (size+1)

  i = 0
  dl=[]
  s=''
  while n10>=2:
    x = n10 % 2
    n10 = (n10-x) / 2
    dl.append(x)
    i = i + 1
  dl.append(n10)
  for j in range(size-i):
    if lsb == '0':
      dl.append(lsb)

  dl.reverse()
  for i in range(0,len(dl)):
    s=s+str(dl[i])

  # add SystemVerilog binary radix
  s = "'b" + s
  return s


# Integer to Binary string conversion
def hex_2_bin(my_str, size):
  hex2 = dec_2_bin(atoi(my_str,0),size,0)
  return hex2

def translate(name):
    # by default, just uppercase
    result = name.upper()

    # if the entire name is already uppercase, do not do anything
    if result != name:
        # first letter has to be uppercase
        result = name[0].upper()
        # loop on the remaining letters
        for i in range(1, len(name)):
            c = name[i]
            # check if current char is uppercase
            if c.isupper():
                # if it is uppercase:
                #   - if previous char is lower, add _ ; example: CamelKase -> _ before K
                #   - if next char is lower, add _ ; example: DTIMKill -> _ before K
                if name[i-1].islower() \
                   or (i < (len(name)-1) and name[i+1].islower()):
                    result += '_'
                result += c
            # check if current char is digit
            elif c.isdigit():
                # add underscore unless previous char is not digit
                if name[i-1].isdigit() == False:
                    # then add underscore before digit
                    result += '_'
                result += c
            # finally, char is not upper and is not digit, just make it upper
            else:
                result += c.upper()
    return result


# Convert 0x hexadecimal notation to Verilog's 'h notation
def hex2ver(number):
  ver_hex = "'h" + number[2:]
  if (number[:2] != "0x") :
    print "ERROR: hex2ver() number = " + number + " doesn't start with 0x!"
  return ver_hex

# Convert access policy from RivieraWaves definition to UVM pre-defined access policies
# Strip leading and trailing whitespaces from string read from XML, as some fields
# contain a trailing space
# The pre-defined access policies are as follows:
#
#  "RO"    W: no effect, R: no effect
#  "RW"    W: as-is, R: no effect
#  "RC"    W: no effect, R: clears all bits
#  "RS"    W: no effect, R: sets all bits
#  "WRC"   W: as-is, R: clears all bits
#  "WRS"   W: as-is, R: sets all bits
#  "WC"    W: clears all bits, R: no effect
#  "WS"    W: sets all bits, R: no effect
#  "WSRC"  W: sets all bits, R: clears all bits
#  "WCRS"  W: clears all bits, R: sets all bits
#  "W1C"   W: 1/0 clears/no effect on matching bit, R: no effect
#  "W1S"   W: 1/0 sets/no effect on matching bit, R: no effect
#  "W1T"   W: 1/0 toggles/no effect on matching bit, R: no effect
#  "W0C"   W: 1/0 no effect on/clears matching bit, R: no effect
#  "W0S"   W: 1/0 no effect on/sets matching bit, R: no effect
#  "W0T"   W: 1/0 no effect on/toggles matching bit, R: no effect
#  "W1SRC" W: 1/0 sets/no effect on matching bit, R: clears all bits
#  "W1CRS" W: 1/0 clears/no effect on matching bit, R: sets all bits
#  "W0SRC" W: 1/0 no effect on/sets matching bit, R: clears all bits
#  "W0CRS" W: 1/0 no effect on/clears matching bit, R: sets all bits
#  "WO"    W: as-is, R: error
#  "WOC"   W: clears all bits, R: error
#  "WOS"   W: sets all bits, R: error
#  "W1"    W: first one after HARD reset is as-is, other W have no effects, R: no effect
#  "WO1"   W: first one after HARD reset is as-is, other W have no effects, R: error
def get_uvm_access(access, type):

  if (type == "field"):
    if   (access.upper()).strip() == 'R' : result='RO'
    elif (access.upper()).strip() == 'W' : result='WO'
    elif (access.upper()).strip() == 'RW': result='RW'
    elif (access.upper()).strip() == 'C' : result='W1C'
    elif (access.upper()).strip() == 'S' : result='W1S'
    else :
      print 'ERROR - get_uvm_access() got unknown access type: "' + access +'"'
      result = 'None'
  else:
    if   (access.upper()).strip() == 'R' : result='RO'
    elif (access.upper()).strip() == 'W' : result='WO'
    elif (access.upper()).strip() == 'RW': result='RW'
    elif (access.upper()).strip() == 'C' :
      result='RW'
      #print 'WARNING - get_uvm_access() got unsupported UVM register access type: "' + access +'", using RW instead!'
    elif (access.upper()).strip() == 'S' :
      result='RW'
      #print 'WARNING - get_uvm_access() got unsupported UVM register access type: "' + access +'", using RW instead!'
    else :
      print 'ERROR - get_uvm_access() got unknown access type: "' + access +'"'
      result = 'None'


#  print 'DEBUG - get_uvm_access() got access type: ' + access + ' returning: ' + result

  return result

# This function parses the XML file to create the verilog file
def create_registers(doc,entity_name,rcov,bcov,endian):

  # Get max lengths for indentation, and max address for paddr width
  len_max = 0
  addr_max = 0
  addr_len_max = 0
  patch_msb = 0
  for root_node in doc.getElementsByTagName("root"):
    for bank_node in root_node.childNodes:
      # Get register data width from configuration data
      if bank_node.nodeName == "configuration":
        for configuration_node in bank_node.childNodes:
          if configuration_node.nodeName == "width":
            data_high = int(configuration_node.getAttribute('register')) - 1

      else: # bank_node.nodeName != "__configuration__":
        for register_node in bank_node.childNodes:
          if register_node.nodeType == register_node.ELEMENT_NODE:
            # get max constant name length
            register=register_class()
            #register.name = register_node.tagName
            register.name = register_node.getAttribute('name')

            if register.name is not None:
              register.add_offset = register_node.getAttribute('offset')
              #print "Name " + str(register.name)
              #print "Address offset " + str(register.add_offset)
              addr_max = max(addr_max,(atoi(register.add_offset,0)/4))
              register.verilog_addr = register.name + '_ADDR_CT'
              addr_len_max = max(addr_len_max,len(register.verilog_addr))
          for field_node in register_node.childNodes:
            if field_node.nodeType == field_node.ELEMENT_NODE:
              # get max field name length
              field=field_class()
              field.name = str(field_node.getAttribute('name'))
              len_max = max(len_max,len(field.name))

  # Add 1 for nice indentation
  len_max = len_max + 1
  addr_len_max = addr_len_max + 1

  # Open .sv file and place header
  ral_file=open(entity_name+'_reg_block.sv','w')
  ral_file.write(uvm_ral_snippets.header.substitute(AUTHOR=os.environ['USER'], ENTITY=entity_name))

  # Arrays of strings to patch specific parts of the template:

  #macro
  macros_array=[]
  task_array=[]

  registers_array=[]
  regs_in_block_array=[]
  regs_create_build_array=[]
  regs_add_map_array=[]

  current_define = ''
  # Parse XML file and fill patch arrays
  for root_node in doc.getElementsByTagName("root"):
    # Register bank
    for bank_node in root_node.childNodes:
      if bank_node.nodeName != "configuration":
        if (bank_node.nodeType == bank_node.ELEMENT_NODE):
          # get bank register name
          bank_name = bank_node.tagName
          #bank_name = bank_node.getAttribute('name')
          print 'Registers bank : ' + str(bank_name)

          # Register
          for register_node in bank_node.childNodes:
            if register_node.nodeType == register_node.ELEMENT_NODE:
              # get all register info
              register=register_class()
              #register.name = register_node.tagName
              register.name = upper(register_node.getAttribute('name'))

              portout_line = ""
              portin_line = ""
              if register.name is not None:
                build_field_array=[]
                config_field_array=[]
                cov_rd_array=[]
                cov_wr_array=[]
                cov_bin_array=[]

                register.verilog_addr = register.name + '_ADDR_CT'
                register.add_offset = register_node.getAttribute('offset')
                register.hw_access = register_node.getAttribute('hw')
                register.sw_access = register_node.getAttribute('sw')
                register.reset = register_node.getAttribute('reset')
                register.define = register_node.getAttribute('define')

                ral_file.write(uvm_ral_snippets.reg_block.substitute(NAME_OF_REGISTER=register.name))

                register.shortname = re.sub("REG$", '', register.name)
                register.shortname = translate(register.shortname).lower()

                registers_array.append(uvm_ral_snippets.cov_bins.substitute(NAME_OF_REGISTER=register.name,\
                                       REGISTER_OFFSET=hex2ver(register.add_offset)))

                regs_in_block_array.append(uvm_ral_snippets.regs_in_block.substitute(NAME_OF_REGISTER=register.name))

                regs_create_build_array.append(uvm_ral_snippets.regs_create_build.substitute(NAME_OF_REGISTER=register.name,\
                                               UVM_NAME_OF_REGISTER=register.name.upper()))

                regs_add_map_array.append(uvm_ral_snippets.regs_add_map.substitute(ENTITY=entity_name, NAME_OF_REGISTER=register.name,\
                                          OFFSET=hex2ver(register.add_offset), ACCESS=get_uvm_access(register.sw_access, 'register')))

                # Register Field
                has_fields = 0
                reg_is_rw = 0 # used in determining which covergroups should be created
                reg_is_ro = 0 # used in determining which covergroups should be created
                reg_is_wo = 0 # used in determining which covergroups should be created

                for field_node in register_node.childNodes:
                  if field_node.nodeType == field_node.ELEMENT_NODE:
                    # get all register field info
                    field=field_class()
                    #field.name = str(field_node.tagName)
                    field.name = field_node.getAttribute('name')
                    field.width = int(field_node.getAttribute('width'))
                    field.typ = field_node.getAttribute('type')
                    field.hw = field_node.getAttribute('hw')
                    field.sw = field_node.getAttribute('sw')
                    field.position = field_node.getAttribute('position')
                    field.fixed_msb = 0

                    field.msb = int(field.width)-1
                    field.msb_position = field.msb + int(field.position)

                    field.reset = field_node.getAttribute('reset')
                    macros_array.append('   // Macros definition for field ' + str(field.name))

                    if field.reset[0] != "`" :
                      field.str_binreset = dec_2_bin(int(field.reset),field.msb,str(field.typ))
                    else:
                      field.str_binreset = field.reset

                    # Find registers with read-only MSB.
                    if (field.sw[0] == '1') or (field.sw[0] == '0'):
#                      print "Found MSB read register " + field.name
                      # In case of SW written registers with fixed MSB, this MSB is not implemented with a FF.
                      if (field.sw.count('RW') > 0):
                        field.msb = field.msb-1
                      field.fixed_msb = 1

                    ral_file.write(uvm_ral_snippets.reg_field.substitute(FIELD_NAME=field.name))
                    has_fields = 1
                    build_field_array.append(uvm_ral_snippets.build_field.substitute(FIELD_NAME=field.name))
                    field_sw = field.sw[0:2]
                    config_field_array.append(uvm_ral_snippets.config_field.substitute(FIELD_NAME=field.name,\
                             FIELD_SIZE=field.width,FIELD_OFFSET=field.position,FIELD_ACCESS=get_uvm_access(field_sw,'field'),\
                             VOLATILE='1',RESET_VALUE=field.str_binreset, HAS_RESET='1', IS_RAND='1'))

                    if get_uvm_access(field_sw,'field') == "RW":
                      reg_is_rw = 1
                      cov_rd_array.append(uvm_ral_snippets.cov_rd.substitute(FIELD_NAME=field.name,FIELD_LEFT_BOUNDARY=field.msb))
                      cov_wr_array.append(uvm_ral_snippets.cov_wr.substitute(FIELD_NAME=field.name,FIELD_LEFT_BOUNDARY=field.msb))
                    elif get_uvm_access(field_sw,'field') == "RO":
                      reg_is_ro = 1
                      cov_rd_array.append(uvm_ral_snippets.cov_rd.substitute(FIELD_NAME=field.name,FIELD_LEFT_BOUNDARY=field.msb))
                    else:
                      reg_is_wo = 1
                      cov_wr_array.append(uvm_ral_snippets.cov_wr.substitute(FIELD_NAME=field.name,FIELD_LEFT_BOUNDARY=field.msb))

                if has_fields == 0:
                  # the register has no fields, but RAL needs at least one field
                  # so use generic "data" field for the entire register
                  ral_file.write(uvm_ral_snippets.reg_field.substitute(FIELD_NAME='DATA'))
                  build_field_array.append(uvm_ral_snippets.build_field.substitute(FIELD_NAME='DATA'))
                  config_field_array.append(uvm_ral_snippets.config_field.substitute(FIELD_NAME='DATA', FIELD_SIZE=(data_high+1),\
                      FIELD_OFFSET='0',FIELD_ACCESS=get_uvm_access(register.sw_access, 'register'),VOLATILE='1',RESET_VALUE=hex2ver(register.reset),\
                      HAS_RESET='1', IS_RAND='1'))

                  if get_uvm_access(register.sw_access, 'register') == "RW":
                    reg_is_rw = 1
                    cov_rd_array.append(uvm_ral_snippets.cov_rd.substitute(FIELD_NAME='DATA', FIELD_LEFT_BOUNDARY=data_high))
                    cov_wr_array.append(uvm_ral_snippets.cov_wr.substitute(FIELD_NAME='DATA', FIELD_LEFT_BOUNDARY=data_high))
                  elif get_uvm_access(register.sw_access, 'register') == "RO":
                    reg_is_ro = 1
                    cov_rd_array.append(uvm_ral_snippets.cov_rd.substitute(FIELD_NAME='DATA', FIELD_LEFT_BOUNDARY=data_high))
                  else:
                    reg_is_wo = 1
                    cov_wr_array.append(uvm_ral_snippets.cov_wr.substitute(FIELD_NAME='DATA', FIELD_LEFT_BOUNDARY=data_high))

                build_field_array.append("\n")
                config_field_array.append("\n")
                cov_wr_array.append("\n")
                cov_rd_array.append("\n")
                ral_file.write(uvm_ral_snippets.build_func)
                ral_file.write("\n".join(build_field_array))
                ral_file.write("\n".join(config_field_array))
                ral_file.write(uvm_ral_snippets.endfunction)

                if reg_is_rw == 1:
                  ral_file.write(uvm_ral_snippets.cov_wr_func.substitute(ENTITY=entity_name, NAME_OF_REGISTER=register.name))
                  ral_file.write("\n".join(cov_wr_array))
                  ral_file.write(uvm_ral_snippets.endgroup)
                  ral_file.write(uvm_ral_snippets.cov_rd_func.substitute(ENTITY=entity_name, NAME_OF_REGISTER=register.name))
                  ral_file.write("\n".join(cov_rd_array))
                  ral_file.write(uvm_ral_snippets.endgroup)
                elif reg_is_ro == 1:
                  ral_file.write(uvm_ral_snippets.cov_rd_func.substitute(ENTITY=entity_name, NAME_OF_REGISTER=register.name))
                  ral_file.write("\n".join(cov_rd_array))
                  ral_file.write(uvm_ral_snippets.endgroup)
                elif reg_is_wo == 1:
                  ral_file.write(uvm_ral_snippets.cov_wr_func.substitute(ENTITY=entity_name, NAME_OF_REGISTER=register.name))
                  ral_file.write("\n".join(cov_wr_array))
                  ral_file.write(uvm_ral_snippets.endgroup)
                else:
                  print "ERROR - shouldn't have reached this line... "

                if reg_is_rw == 1:
                  ral_file.write(uvm_ral_snippets.sample_new_func_rw.substitute(NAME_OF_REGISTER=register.name, WIDTH_OF_REGISTER=(data_high+1),\
                               HAS_COVERAGE=rcov))
                elif reg_is_ro == 1:
                  ral_file.write(uvm_ral_snippets.sample_new_func_ro.substitute(NAME_OF_REGISTER=register.name, WIDTH_OF_REGISTER=(data_high+1),\
                               HAS_COVERAGE=rcov))
                elif reg_is_wo == 1:
                  ral_file.write(uvm_ral_snippets.sample_new_func_wo.substitute(NAME_OF_REGISTER=register.name, WIDTH_OF_REGISTER=(data_high+1),\
                               HAS_COVERAGE=rcov))



  ral_file.write(uvm_ral_snippets.access_wrap.substitute(ENTITY=entity_name))
  ral_file.write("\n".join(registers_array))
  ral_file.write(uvm_ral_snippets.access_wrap_con.substitute(ENTITY=entity_name))

  ral_file.write(uvm_ral_snippets.regblock_start.substitute(ENTITY=entity_name))
  ral_file.write("\n".join(regs_in_block_array))
  ral_file.write(uvm_ral_snippets.reg_block_new.substitute(ENTITY=entity_name, REGS_COVERAGE=bcov))
  ral_file.write("\n".join(regs_create_build_array))
  ral_file.write(uvm_ral_snippets.regs_create_map.substitute(ENTITY=entity_name, BASE_REG_MAP_ADDR = "0", REG_MAP_WIDTH_BYTES=((data_high+1)/8),\
                 REG_MAP_ENDIAN = endian))

  ral_file.write("\n".join(regs_add_map_array))
  ral_file.write(uvm_ral_snippets.reg_block_end.substitute(ENTITY=entity_name, REGS_COVERAGE=bcov))

# parse tbDefines.v to find the AHB base address of this entity
def get_base_address(entity_name, addrfile):

  # open tbDefines.v and find the line which contains the AHB base address for the processed entity
  # the format of the text is "entity_nameBASEADDRESS", with entity_name in uppercase

  base_address_txt = entity_name.upper() + "BASEADDRESS"
  found = 0
  base_addr = ""

  for line in open(addrfile):
    fields = line.split()
    if len(fields) < 3: continue # we need something like `define XXXBASEADDRESS 32'h01234567
    if (fields[0] == '`define') and (fields[1] == base_address_txt):
      # found the text
      found = 1
      base_addr =fields[2]

  if found == 0:
    print("ERROR: Base address for " + entity_name + " not found!")
 # else:
 #   print("DEBUG: Base address for "  + entity_name + " is " +  base_addr)

  # now open the temporary TXT file we store base addresses to
  tmpfile=open("ahb_base_addresses.txt", "a")
  tmpfile.write(entity_name + " " + base_addr + "\n")

# Create a package file for compilation
def create_pkg_file(entity_name):

  # Open .sv file and place header
  ral_file=open(entity_name + '_reg_block_pkg.sv','w')
  ral_file.write(uvm_ral_snippets.regblock_pkg.substitute(AUTHOR=os.environ['USER'], ENTITY=entity_name))

# Display script help
def display_help():
  print "\n    Usage:\n"
  print "       xml2ral.py -x|--xml <xml_source_file> -e|--entity <verilog_entity_name> -r|--rcov <register coverage option>" +\
                " -b|--bcov <register block coverage option> -n|--endian <byte ordering> -a|--basea <verilog file with base addresses>"
  print "       xml2ral.py -h|--help\n"
  print "    Creates a UVM register model for 'entity' from the 'xml_source_file' XML register description file.\n"
  print "    Example:\n\n\t python -i xml2ral.py -x reg.xml -e phy_registers -r UVM_CVR_FIELD_VALS -b UVM_CVR_ADDR_MAP -e UVM_LITTLE_ENDIAN" +\
        " -a $SOURCESLIB/SIMS_UVM/defines/tbDefines.v\n"


def main(argv):

  # Get arguments from command line
  short = 'x:e:r:b:n:a:h'
  long = ('help', 'xml', 'entity', 'rcov', 'bcov', 'endian', 'basea')
  try:
    opts, args = getopt.getopt(sys.argv[1:], short, long)
  except getopt.GetoptError, exc:
     print exc
     display_help()
     return 1

  # Error in arguments
  if args:
    print args
    display_help()
    return 1

  # Check received arguments
  xmlfile=None
  entity_name=None
  rcov=None
  bcov=None
  endian=None
  addrfile=None

  for opt,val in opts:
    if opt in ('-h', '--help'):
      display_help()
      return 0
    elif opt in ('-x', '--xml'):
      xmlfile = val
    elif opt in ('-e', '--entity'):
      entity_name = val
    elif opt in ('-r', '--rcov'):
      rcov = val
    elif opt in ('-b', '--bcov'):
      bcov = val
    elif opt in ('-n', '--endian'):
      endian = val
    elif opt in ('-a', '--basea'):
      addrfile = val


  if xmlfile is None or entity_name is None or addrfile is None:
    display_help()
    return 1

  if not os.path.isfile(xmlfile):
    print("Specified XML file" + xmlfile + " does not exist! Exiting...")
    return 1

  if not os.path.isfile(addrfile):
    print("Specified Verilog file with base addresses " + addrfile + " does not exist! Exiting...")
    return 1

  # Parse XML registers file
  doc = xml.dom.minidom.parse(xmlfile)
  create_registers(doc,entity_name,rcov,bcov,endian)
  get_base_address(entity_name, addrfile)
  create_pkg_file(entity_name)

# Execute main and exit
if __name__ == '__main__':
  status = main(sys.argv)
  sys.exit(status)
