
#include <stdio.h>
#include <math.h>

#include <svdpi.h>
#include <veriuser.h>


#define SHORT_PRE  96
#define LONG_PRE   192

// OFDM
#define L_STF      8
#define L_LTF      8
#define L_SIG      4

#define OFDM_NON_HT_PRE  (L_STF + L_LTF + L_SIG)

#define HT_STF     4
#define HT_LTF     4
#define HT_SIG     8

#define OFDM_HT_MM_PRE_BASE  (OFDM_NON_HT_PRE + HT_SIG + HT_STF + HT_LTF)

#define HT_GF_STF  8
#define HT_LTF1    8

#define OFDM_HT_GF_PRE_BASE  (HT_GF_STF + HT_LTF1 + HT_SIG)

#define VHT_STF    4
#define VHT_LTF    4
#define VHT_SIG_A  8
#define VHT_SIG_B  4

#define OFDM_VHT_PRE_BASE  (OFDM_NON_HT_PRE + VHT_SIG_A + VHT_STF + VHT_SIG_B + VHT_LTF)

// LEGACY  RATES and NSYM in Mbps
const float LEGACY_NON_OFDM_RATE[4] = {1, 2, 5.5, 11};

// IEEE Std 802.11 - 2012 Table 18-4—Modulation-dependent parameters
const int   LEGACY_OFDM_NDBPS[8] =
{
 // data rate   6    9    12   18   24   36    48    54
 // MCS index   4    5    6    7    8    9     10    11
                24,  36,  48,  72,  96,  144,  192,  216
};

// HT NSYM
#define HT_BW   2
#define HT_MCS  77

// REF IEEE Std 802.11 - 2012 Chptr 20.6 Parameters for HT MCSs
const int HT_NDBPS[HT_BW][HT_MCS] = {
  // BW 20 MHz
  {
      //  MCS Index
      //  0        1         2         3         4         5         6         7
          26,      52,       78,       104,      156,      208,      234,      260,
      //  8        9         10        11        12        13        14        15
          52,      104,      156,      208,      312,      416,      468,      520,
      //  16       17        18        19        20        21        22        23
          78,      156,      234,      312,      468,      624,      702,      780,
      //  24       25        26        27        28        29        30        31
          104,     208,      312,      416,      624,      832,      936,      1040,
      //  32(error)33        34        35        36        37        38        39
          1,       156,      208,      260,      234,      312,      390,      208,
      //  40       41        42        43        44        45        46        47
          260,     260,      312,      364,      364,      416,      312,      390,
      //  48       49        50        51        52        53        54        55
          390,     468,      546,      546,      624,      260,      312,      364,
      //  56       57        58        59        60        61        62        63
          312,     364,      416,      468,      416,      468,      520,      520,
      //  64       65        66        67        68        69        70        71
          572,     390,      468,      546,      468,      546,      624,      702,
      //  72       73        74        75        76
          624,     702,      780,      780,      858
  },
  // BW 40 MHz
  {
      //  MCS Index
      //  0        1         2         3         4         5         6         7
          54,      108,      162,      216,      324,      432,      486,      540,
      //  8        9         10        11        12        13        14        15
          108,     216,      324,      432,      648,      864,      972,      1080,
      //  16       17        18        19        20        21        22        23
          162,     324,      486,      648,      972,      1296,     1458,     1620,
      //  24       25        26        27        28        29        30        31
          216,     432,      648,      864,      1296,     1728,     1944,     2160,
      //  32       33        34        35        36        37        38        39
          24,      324,      432,      540,      486,      648,      810,      432,
      //  40       41        42        43        44        45        46        47
          540,     540,      648,      756,      756,      864,      648,      810,
      //  48       49        50        51        52        53        54        55
          810,     972,      1134,     1134,     1296,     540,      648,      756,
      //  56       57        58        59        60        61        62        63
          648,     756,      864,      972,      864,      972,      1080,     1080,
      //  64       65        66        67        68        69        70        71
          1188,    810,      972,      1134,     972,      1134,     1296,     1458,
      //  72       73        74        75        76
          1296,    1458,     1620,     1620,     1782
  }
};

// VHT NSYM
#define VHT_BW   4
#define VHT_NSS  8
#define VHT_MCS  10

// IEEE P802.11ac D3.0  Chptr 22.5 Parameters for VHT MCSs
const int VHT_NDBPS[VHT_BW][VHT_NSS][VHT_MCS] = {
    // 20MHz
    {
        // NSS 1
        {
            //    MCS Index
            //    0       1        2        3        4        5        6        7        8        9(error)
                  26,     52,      78,      104,     156,     208,     234,     260,     312,     1
        },
        // NSS 2
        {
            //    0       1        2        3        4        5        6        7        8        9(error)
                  52,     104,     156,     208,     312,     416,     468,     520,     624,     1
        },
        // NSS 3
        {
            //    0       1        2        3        4        5        6        7        8        9
                  78,     156,     234,     312,     468,     624,     702,     780,     936,     1040,
        },
        // NSS 4
        {
            //    0       1        2        3        4        5        6        7        8        9(error)
                  104,    208,     312,     416,     624,     832,     936,     1040,    1248,    1
        },
        // NSS 5
        {
            //    0       1        2        3        4        5        6        7        8        9(error)
                  130,    260,     390,     520,     780,     1040,    1170,    1300,    1560,    1
        },
        // NSS 6
        {
            //    0       1        2        3        4        5        6        7        8        9
                  156,    312,     468,     624,     936,     1248,    1404,    1560,    1872,    2080,
        },
        // NSS 7
        {
            //    0       1        2        3        4        5        6        7        8        9(error)
                  182,    364,     546,     728,     1092,    1456,    1638,    1820,    2184,    1
        },
        // NSS 8
        {
            //    0       1        2        3        4        5        6        7        8        9(error)
                  208,    416,     624,     832,     1248,    1664,    1872,    2080,    2496,    1
        }
    },
    // 40MHz
    {
        // NSS 1
        {
            //    0       1        2        3        4        5        6        7        8        9
                  54,     108,     162,     216,     324,     432,     486,     540,     648,     720
        },
        // NSS 2
        {
            //    0       1        2        3        4        5        6        7        8        9
                  108,    216,     324,     432,     648,     864,     972,     1080,    1296,    1440
        },
        // NSS 3
        {
            //    0       1        2        3        4        5        6        7        8        9
                  162,    324,     486,     648,     972,     1296,    1458,    1620,    1944,    2160
        },
        // NSS 4
        {
            //    0       1        2        3        4        5        6        7        8        9
                  216,    432,     648,     864,     1296,    1728,    1944,    2160,    2592,    2880
        },
        // NSS 5
        {
            //    0       1        2        3        4        5        6        7        8        9
                  270,    540,     810,     1080,    1620,    2160,    2430,    2700,    3240,    3600
        },
        // NSS 6
        {
            //    0       1        2        3        4        5        6        7        8        9
                  324,    648,     972,     1296,    1944,    2592,    2916,    3240,    3888,    4320
        },
        // NSS 7
        {
            //    0       1        2        3        4        5        6        7        8        9
                  378,    756,     1134,    1512,    2268,    3024,    3402,    3780,    4536,    5040
        },
        // NSS 8
        {
            //    0       1        2        3        4        5        6        7        8        9
                  432,    864,     1296,    1728,    2592,    3456,    3888,    4320,    5184,    5760
        }
    },
    // 80MHz
    {
        // NSS 1
        {
            //    0       1        2        3        4        5        6        7        8        9
                  117,    234,     351,     468,     702,     936,     1053,    1170,    1404,    1560
        },
        // NSS 2
        {
            //    0       1        2        3        4        5        6        7        8        9
                  234,    468,     702,     936,     1404,    1872,    2106,    2340,    2808,    3120
        },
        // NSS 3
        {
            //    0       1        2        3        4        5        6(error) 7        8        9
                  351,    702,     1053,    1404,    2106,    2808,    1,       3510,    4212,    4680
        },
        // NSS 4
        {
            //    0       1        2        3        4        5        6        7        8        9
                  468,    936,     1404,    1872,    2808,    3744,    4212,    4680,    5616,    6240
        },
        // NSS 5
        {
            //    0       1        2        3        4        5        6        7        8        9
                  585,    1170,    1755,    2340,    3510,    4680,    5265,    5850,    7020,    7800
        },
        // NSS 6
        {
            //    0       1        2        3        4        5        6        7        8        9(error)
                  702,    1404,    2106,    2808,    4212,    5616,    6318,    7020,    8424,    1
        },
        // NSS 7
        {
            //    0       1        2        3        4        5        6(error) 7        8        9
                  819,    1638,    2457,    3276,    4914,    6552,    1,       8190,    9828,    10920
        },
        // NSS 8
        {
            //    0       1        2        3        4        5        6        7        8        9
                  936,    1872,    2808,    3744,    5616,    7488,    8424,    9360,    11232,   12480
        }
    },
    // 160MHz
    {
        // NSS 1
        {
            //    0       1        2        3        4        5        6        7        8        9
                  234,    468,     702,     936,     1404,    1872,    2106,    2340,    2808,    3120
        },
        // NSS 2
        {
            //    0       1        2        3        4        5        6        7        8        9
                  468,    936,     1404,    1872,    2808,    3744,    4212,    4680,    5616,    6240
        },
        // NSS 3
        {
            //    0       1        2        3        4        5        6        7        8        9(error)
                  702,    1404,    2106,    2808,    4212,    5616,    6318,    7020,    8424,    1
        },
        // NSS 4
        {
            //    0       1        2        3        4        5        6        7        8        9
                  936,    1872,    2808,    3744,    5616,    7488,    8424,    9360,    11232,   12480
        },
        // NSS 5
        {
            //    0       1        2        3        4        5        6        7        8        9
                  1170,   2340,    3510,    4680,    7020,    9360,    10530,   11700,   14040,   15600
        },
        // NSS 6
        {
            //    0       1        2        3        4        5        6        7        8        9
                  1404,   2808,    4212,    5616,    8424,    11232,   12636,   14040,   16848,   18720
        },
        // NSS 7
        {
            //    0       1        2        3        4        5        6        7        8        9
                  1638,   3276,    4914,    6552,    9828,    13104,   14742,   16380,   19656,   21840
        },
        // NSS 8
        {
            //    0       1        2        3        4        5        6        7        8        9
                  1872,   3744,    5616,    7488,    11232,   14976,   16848,   18720,   22464,   24960
        }
    }
};

// Function DPI-C to calculate disambiguity
void computeTimeOnAirAC
    (unsigned char *length_i, unsigned char *preType_i, unsigned char *mcsIndex_i, unsigned char *shortGI_i, unsigned char *chBW_i, 
     unsigned char *stbc_i, unsigned char *extNSS_i, unsigned char *woPreamble_i, unsigned char *debug_i, unsigned char *returnDisambiguity_i,
     unsigned char *return_legLength, unsigned char *return_estimatedHtlength, unsigned char *return_disambiguity, unsigned char *return_rxtime)
    
{
//   ---- input
//   length_i
//   preType_i
//   mcsIndex_i
//   shortGI_i
//   chBW_i
//   stbc_i
//   extNSS_i
//   woPreamble_i
//   debug_i
//   returnDisambiguity_i
// 
//   ---- output
//   return_legLength
//   return_estimatedHtlength
//   return_disambiguity
//   return_rxtime
// 

  // Max of 1024 character for debug printing through tb_print
//   char   STR[1024];
  // Total TXTIME
  int    timeOnAir         = 0;
  // Preamble Time
  int    preambleTimeOnAir = 0;
  // LTF Time (coresponding to ((nLTF-1)*4) Since the first LTF1 is already in the define PREAMBLE BASE
  int    ltfTimeOnAir      = 0;
  // PPDU Time
  float  dataTimeOnAir     = 0.0;
  float  dataNSymbol       = 0.0;
  float  tSymbol           = 0.0;

  // IEEE corresponding variables
  int    nSS               = 0;
  int    nSTS              = 0;
  int    nSTSTotal         = 0;
  int    nDBPS             = 0;
  int    nES               = 0;

  // First Step: Get all the IEEE Variables
  // NSS
  if(preType_i[0] == 4){
    // VHT NSS=mcsIndex_i[6:4]
    nSS = ((mcsIndex_i[0] >> 4) & 0x00000007);
    // 0 means 1 nSS
    nSS = nSS + 1;
  }
  else if(preType_i[0] > 1){
    // HT NSS=according to mcsIndex_i[0]
    nSS =   (mcsIndex_i[0] <  8)                   ? 1:
            (mcsIndex_i[0] <  16)                  ? 2:
            (mcsIndex_i[0] <  24)                  ? 3:
            (mcsIndex_i[0] <  32)                  ? 4:
            (mcsIndex_i[0] == 32) && (chBW_i[0] == 1) ? 1://only valid for 40MHz Bandwidth
            (mcsIndex_i[0] == 32) && (chBW_i[0] != 1) ? 0://Not valid
            (mcsIndex_i[0] <  39)                  ? 2:
            (mcsIndex_i[0] <  53)                  ? 3:
            (mcsIndex_i[0] <= 76)                  ? 4:
                                                  0;
  }

//   if(debug_i[0]==1){
//     sprintf(STR,"\nnSS               = %0d (preType_i = %0d, mcsIndex_i[0] = %0d)",nSS,preType_i,mcsIndex_i[0]);
//     io_printf(STR);
//     io_printf("\n");
//     //tb_print(STR);
//   }

  // nSTS
  if(preType_i[0] == 4){
    // VHT IEEE P802.11ac D4.0
    // If STBC == 1 then NSTS = NSS*2 else NSS.
    nSTS  = (stbc_i[0]) ? nSS<<1 : nSS;
  }
  else if(preType_i[0] > 1){
    // HT nSTS IEEE Std 802.11 - 2012 table 20-12 Determining the number of space-time streams
    nSTS =  (nSS == 4) ? nSS : nSS + stbc_i[0];
  }

//   if(debug_i[0]==1){
//     sprintf(STR,"nSTS              = %0d (stbc_i[0] = %0d)",nSTS,stbc_i[0]);
//     io_printf(STR);
//     io_printf("\n");
//     //tb_print(STR);
//   }

  // number of DATA BIT PER SYMBOL
  nDBPS = (preType_i[0] < 2) ?  LEGACY_OFDM_NDBPS[mcsIndex_i[0] - 4]                : // mcsIndex_i[0] from 4 to 11
          (preType_i[0] < 4) ?  HT_NDBPS[chBW_i[0]][mcsIndex_i[0]]                     : // mcsIndex_i[0] from 0 to 76 and chBW_i[0] from 0 to 1
                             VHT_NDBPS[chBW_i[0]][nSS-1][mcsIndex_i[0] & 0x0000000F]; // mcsIndex_i[0][3:0] from 0 to 9 and chBW_i[0] from 0 to 3 and nSS from 1 to 8

//   if(debug_i[0]==1){
//     sprintf(STR,"nDBPS             = %0d (chBW_i[0] = %0d)",nDBPS,chBW_i[0]);
//     io_printf(STR);
//     io_printf("\n");
//     //tb_print(STR);
//   }

  // nES = 6*NES
  // VHT IEEE P802.11ac D4.0
  if(preType_i[0] == 4){
    // 20MHz and  40MHz Bandwidth
    if(chBW_i[0] < 2) {
      nES = (nDBPS <= 2160) ? 6 : (nDBPS <= 4320) ? 12 : 18;
    }
    // 80MHz Bandwidth
    else if(chBW_i[0] == 2) {
      if((nSS == 7) && (nDBPS == 8190)){
        nES = 36;
      }
      else if((nSS == 7) && (nDBPS == 2457)){
        nES = 18;
      }
      else {
        nES = (nDBPS <= 2106) ? 6 : (nDBPS <= 4212) ? 12 : (nDBPS <= 6318) ? 18 : (nDBPS <= 8424) ? 24 : 36;
      }
    }
    // 160MHz Bandwidth
    else if(chBW_i[0] == 3){
      if( ((nSS == 4) && (nDBPS == 9360)) || ((nSS == 7) && (nDBPS == 9828)) ){
        nES = 36;
      }
      else if((nDBPS == 14040) && ((nSS == 5) || (nSS == 6)) ){
        nES = 48;
      }
      else if((nSS == 7) && (nDBPS == 16380)){
        nES = 54;
      }
      else{
        nES = (nDBPS <= 2106)  ? 6  :
              (nDBPS <= 4212)  ? 12 :
              (nDBPS <= 6318)  ? 18 :
              (nDBPS <= 8424)  ? 24 :
              (nDBPS <= 10530) ? 30 :
              (nDBPS <= 12636) ? 36 :
              (nDBPS <= 14742) ? 42 :
              (nDBPS <= 16848) ? 48 :
              (nDBPS <= 18720) ? 54 :
                                 72 ;
      }
    }
  }
  // HT IEEE Std 802.11 - 2012
  else if(preType_i[0] > 1){
    nES = (nDBPS < 1296) ? 6 : 12;
  // LEGACY OFDM
  } else {
    nES = 6;
  }

//   if(debug_i[0]==1){
//     sprintf(STR,"nES               = %0d (nDBPS = %0d)",nES,nDBPS);
//     io_printf(STR);
//     io_printf("\n");
//     //tb_print(STR);
//   }

  // Check if preamble duration is taken in to the calculation
  if(woPreamble_i[0] == 0){
    // preamble time on air
    switch(preType_i[0]){
      case 0:  preambleTimeOnAir = (mcsIndex_i[0] > 3) ? OFDM_NON_HT_PRE : SHORT_PRE; break;
      case 1:  preambleTimeOnAir = (mcsIndex_i[0] > 3) ? OFDM_NON_HT_PRE : LONG_PRE;  break;
      case 2:  preambleTimeOnAir = OFDM_HT_MM_PRE_BASE; break;
      case 3:  preambleTimeOnAir = OFDM_HT_GF_PRE_BASE; break;
      case 4:  preambleTimeOnAir = OFDM_VHT_PRE_BASE;   break;
      default: preambleTimeOnAir = 0;
    }

    // LTF Calculation
    // ltfTimeOnAir = (NLTF - 1) Since the LTF1 is already added in the preambleTimeOnAir
    switch(preType_i[0]){
      // HT MM and HT GF IEEE Std 802.11 - 2012
      case 2:
      case 3:  ltfTimeOnAir  = (nSTS == 3)     ? 3 : nSTS-1;          // Table 20-13—Number of HT-DLTFs required for data space-time streams
               ltfTimeOnAir += (extNSS_i[0] == 3) ? 4 : extNSS_i[0]; break; // Table 20-14—Number of HT-ELTFs required for extension spatial streams

      // VHT IEEE P802.11ac D3.0 Table 22-13 Number of VHT-LTFs required for different numbers of space time streams
      case 4:  ltfTimeOnAir = (nSTS == 3) ? 3   :
                              (nSTS == 5) ? 5   :
                              (nSTS == 7) ? 7   :
                                            nSTS-1; break;
      default: ltfTimeOnAir = 0;
    }

    // Each LTF is 4us
    ltfTimeOnAir = ltfTimeOnAir<<2;
  }

//   if(debug_i[0]==1){
//     sprintf(STR,"preambleTimeOnAir = %0d (woPreamble_i[0] = %0d)",preambleTimeOnAir,woPreamble_i[0]);
//     io_printf(STR);
//     io_printf("\n");
//     //tb_print(STR);
//     sprintf(STR,"ltfTimeOnAir      = %0d (extNSS_i[0] = %0d)",ltfTimeOnAir,extNSS_i[0]);
//     io_printf(STR);
//     io_printf("\n");
//     //tb_print(STR);
//   }

  // Data Time on Air for NON OFDM is Length in bit / data rate
  if (length_i[0] == 0)
  {
    dataTimeOnAir = 0;
  }
  else 
    if((preType_i[0] < 2) && (mcsIndex_i[0] < 4)){
    dataTimeOnAir = ( ((float)(length_i[0]<<3)) / (LEGACY_NON_OFDM_RATE[mcsIndex_i[0]]) );

    // Round up the time
    dataTimeOnAir = ceilf(dataTimeOnAir);

//     if(debug_i[0]==1){
//       sprintf(STR,"dataTimeOnAir     = %0d (length_i = %0d  LEGACY_NON_OFDM_RATE = %0d)",(int)dataTimeOnAir,length_i,(int)LEGACY_NON_OFDM_RATE[mcsIndex_i[0]]);
//       io_printf(STR);
//       io_printf("\n");
//       //tb_print(STR);
//     }
  }
  // Calculate only the number of symbol for Disambiguity
  else if (returnDisambiguity_i[0] != 0)
  {
    // Total of Symbols =            (length_i*8 + 16 + 6*NES)  / NDBPS
    dataNSymbol       = ( ((float)((length_i[0]<<3) + 16 + nES)) / ((float)(nDBPS)) );

    // Round up the dataNSymbol
    dataTimeOnAir = ceilf(dataNSymbol);

    // If STBC See IEEE Std 802.11 - 2012 Table 20-18 then Add one symbol in case of odd number
    if(stbc_i[0] != 0)
    {
      dataTimeOnAir += (float)( ((int)dataTimeOnAir) & 0x00000001); // convert into integer for odd or even detection
    }
  }
  // Data Time on Air for OFDM
  // 1) The full data length in bit is length_i*8 + 16 bits of service field + 6*nES tail bits (IEEE Std 802.11 - 2012 Chapter 18.3.5.x)
  // 2) Convert into Nsymbols according to the MCS (round up the value)
  // 3) If STBC is enabled then add 1 symbol if Nsymbols is odd (STBC send symbols by pairs and twice)
  // 4) Convert Nsymbols into time duration according to Normal GI or Short GI (round up the value)
  // 5) In GF, the TXTIME formula is different than for HT-MM. However, if the packet alignment is included, both become equivalent.
  //   As a consequence, all the duration include the packet alignment duration.
  else {
    // Total of Symbols =            (length_i*8 + 16 + 6*NES)  / NDBPS
    dataNSymbol       = ( ((float)((length_i[0]<<3) + 16 + nES)) / ((float)(nDBPS)) );

    // Round up the dataNSymbol
    dataNSymbol = ceilf(dataNSymbol);

    // If STBC See IEEE Std 802.11 - 2012 Table 20-18 then Add one symbol in case of odd number
    if(stbc_i[0] != 0)
    {
      dataNSymbol += (float)( ((int)dataNSymbol) & 0x00000001); // convert into integer for odd or even detection
    }

//     if(debug_i[0]==1){
//       sprintf(STR,"dataNSymbol       = %0d (length_i = %0d  nDBPS = %0d, stbc_i[0] = %0d)",(int)dataNSymbol,length_i,nDBPS,stbc_i[0]);
//       io_printf(STR);
//       io_printf("\n");
//       //tb_print(STR);
//     }

    // Transform into time
    tSymbol           = (shortGI_i[0]) ? 3.6 : 4.0;

    // Round up the time
    dataTimeOnAir = 4*ceilf(tSymbol*dataNSymbol/4.0);

//     if(debug_i[0]==1){
//       sprintf(STR,"dataTimeOnAir     = %0d (shortGI_i[0] = %0d)",(int)dataTimeOnAir,shortGI_i[0]);
//       io_printf(STR);
//       io_printf("\n");
//       //tb_print(STR);
//     }
  }

  // Calculate the full time on air of the frame or Disambiguity value
  if(returnDisambiguity_i[0] != 0)
  {
    timeOnAir = ((int)(dataTimeOnAir)) % 10;
    if((preType_i[0] == 4) && (shortGI_i[0]) && (timeOnAir == 9))
    {
      timeOnAir = 1;
    }
    else
    {
      timeOnAir = 0;
    }
  }
  else
  {
    timeOnAir = preambleTimeOnAir + ltfTimeOnAir + (int)(dataTimeOnAir);
  }

  // IEEE P802.11ac/D4.0, October 2012 p309
  if ((length_i[0] != 0) && (preType_i[0] == 4))
  {
    return_rxtime[0]       = timeOnAir;

    dataTimeOnAir       = ( ((float)((length_i[0]<<3) + 16 + nES)) / ((float)(nDBPS)) );
    dataTimeOnAir       = ceilf(dataTimeOnAir);
    if(stbc_i[0] != 0)
    {
      dataTimeOnAir += (float)( ((int)dataTimeOnAir) & 0x00000001);
    }

    return_disambiguity[0]  = 0;
    if (((((int)(dataTimeOnAir)) % 10)  == 9) && (shortGI_i[0]) && (preType_i[0] == 4))
    {
      return_disambiguity[0]       = 1;
    }

    return_legLength[0] = (float)(timeOnAir - 20);
    return_legLength[0] = return_legLength[0]/4;
    return_legLength[0] = ceilf(return_legLength[0]);
    return_legLength[0] = return_legLength[0]*3;
    return_legLength[0] = return_legLength[0]-3;

    return_estimatedHtlength[0]   = (return_legLength[0] + 3) / 3;
    return_estimatedHtlength[0]   = ceilf(return_estimatedHtlength[0]);
    return_estimatedHtlength[0]   = return_estimatedHtlength[0] * 4 + 20;
    return_estimatedHtlength[0]   = return_estimatedHtlength[0] - (float)(preambleTimeOnAir) - (float)(ltfTimeOnAir);
    return_estimatedHtlength[0]  /= (shortGI_i[0]) ? 3.6 : 4.0;
    return_estimatedHtlength[0]   = floorf(return_estimatedHtlength[0]);
    return_estimatedHtlength[0]  -= return_disambiguity[0];
    return_estimatedHtlength[0]   = (return_estimatedHtlength[0] * (float)(nDBPS)) - 16 - nES;
    return_estimatedHtlength[0]   = return_estimatedHtlength[0]/8;
    return_estimatedHtlength[0]   = floorf(return_estimatedHtlength[0]);
  }
  else
  {
    return_estimatedHtlength[0]  = 0.0;
    return_legLength[0]          = 12.0;
    return_disambiguity[0]       = 0;
    return_rxtime[0]             = timeOnAir;
  }

//   if(debug_i[0]==1){
//     sprintf(STR,"timeOnAir         = %0d",timeOnAir);
//     io_printf(STR);
//     io_printf("\n");
//     //tb_print(STR);
//   }

}
