#!/bin/bash
set -e

HERE=$(dirname $(readlink -f $0))
cd $HERE

LINUXVERSION=linux-3.19

JOBS=$(( $(grep -c "processor" /proc/cpuinfo) + 1 ))
DEFCONFIG=${DEFCONFIG-diniv6}
if [[ $DEFCONFIG =~ "yocto" ]]
then
    ARM_CROSS_COMPILE=${ARM_CROSS_COMPILE-arm-poky-linux-gnueabi-}
else
    ARM_CROSS_COMPILE=${ARM_CROSS_COMPILE-arm-none-linux-gnueabi-}
fi
X86_CROSS_COMPILE=${X86_CROSS_COMPILE-x86_64-poky-linux-}
LINUXDIR=${LINUXDIR-$LINUXVERSION}
USE_GIT=${USE_GIT-1}

# by default do nothing
target=""
download=0
update=0
prepare=0
module=0
kernel=0

usage="Usage: $(basename $0) <-6|-7> [-j <#jobs>] <-h (help) | -a (all) | [-d (download)] [-u (update)] [-p (prepare)] [-m module] [-k (kernel)] >"

help="
This script will intialize kernel image source to compile rwnx driver

Options:
Target:
-6|-7 : Select the target diniv6 or cevav7.

Action:
-h    : show this help and exit

-d    : Download kernel sources and apply local patches
        If output dir already exist assume source have already been download
        and skip this step.
        Select outdir with environment variable LINUXDIR
        Select creation of git repository with environment variable USE_GIT
-u    : Update: Apply patches.
        Automatically set if new sources files are downloaded.
        When used on an already extracted sources, it will only apply patches
        not already applied.
        (may fail if sources have been modified so that new patches cannot apply)
-p    : Prepare header file for external module compilation
        (This step is enough to compile rwnx driver, but using only this
         will generate link warning when compiling rwnx driver)
-m    : Compile in tree kernel modules
-k    : Compile Kernel image
-a    : Shortcut for -dpmk

Misc:
-j <jobs> : Specify -j option to use for module/kernel compilation
            Default value is # of processor + 1.

Environment Variable:
The following environment variables can be use to modify script behaviour

ARM_CROSS_COMPILE : Prefix to use for diniv6 compilation.
                    Default is $ARM_CROSS_COMPILE

X86_CROSS_COMPILE : Prefix to use for cevav7 compilation.
                    Default is $X86_CROSS_COMPILE

LINUXDIR          : Path (absolute of relative to location of this script) that contains kernel source.
                    (or path where to download kernel source is -d is set)
                    Note: If you're not using the default value, then the same value must be set each time.
                    Default is $HERE/$LINUXDIR.

USE_GIT           : If set to 1 and -d option is set, initialize a git repository on kernels sources.
                    Default is $USE_GIT

DEFCONFIG         : Defconfig to use for diniv6 platform, ignored for v7.
                    Possible values are diniv6 and diniv6_yocto
                    Default is ${DEFCONFIG}
"

function status ()
{
    if [ -t 1 ]
    then
	echo -e "\033[1m$*\033[0m"
    else
	echo -e "$*"
    fi
}

function error ()
{
    if [ -t 1 ]
    then
	echo -e "\033[1;31;40m$*\033[0m"
    else
	echo -e "ERROR: $*"
    fi
}

function set_target() {
    if [ ! $target ]
    then
        echo "No target specified, you must specify -6 or -7 option"
        exit 1
    fi

    if [ $target = "diniv6" ]
    then
        status "Select target diniv6 (config $DEFCONFIG)"
        KMAKE="make ARCH=arm CROSS_COMPILE=$ARM_CROSS_COMPILE"
        OUT="$DEFCONFIG"
        config="${DEFCONFIG}_defconfig"
        kimg=uImage
    else
        status "Select target cevav7"
        KMAKE="make ARCH=x86 CROSS_COMPILE=$X86_CROSS_COMPILE"
        OUT="cevav7"
        config="cevav7_defconfig"
        kimg=bzImage
    fi
}

function do_download() {

    if [ $download -eq 0 ]
    then
        return
    fi

    status "Download and extract kernel sources"
    if [ ! -f $LINUXVERSION.tar.xz ] && [ ! -d $LINUXDIR ] && [ ! -e $LINUXVERSION.tar.xz ] && [ ! -e $LINUXVERSION.tar ]
    then
        wget --no-check-certificate https://www.kernel.org/pub/linux/kernel/v3.x/$LINUXVERSION.tar.xz
    fi

    if [ ! -d $LINUXDIR ] && [ -e $LINUXVERSION.tar.xz ]
    then
        unxz $LINUXVERSION.tar.xz
    fi

    if [ ! -d $LINUXDIR ] && [ -e $LINUXVERSION.tar ]
    then
        local dir=$(dirname $LINUXDIR)
        mkdir -p $dir

        # extract in tmp directory not to overwrite $LINUXVERSION directory
        # if it exist
        mkdir $dir/extract_tmp
        tar xf $LINUXVERSION.tar -C $dir/extract_tmp

        mv $dir/extract_tmp/$LINUXVERSION $LINUXDIR
        rmdir $dir/extract_tmp

        update=2
    fi

    if [ ! -d $LINUXDIR ]
    then
        echo "Error when downloading source"
        exit
    fi
}

function do_update() {
    if [ $update -eq 0 ]
    then
        return
    fi

    declare -a patches
    status "Apply patches to kernel sources"

    cd $LINUXDIR

    for p in $HERE/*patch
    do
	patches+=("$p")
    done

    if [ $update -eq 2 ]
    then
	# new sources
	if [ $USE_GIT -eq 1 ] && ( which git )
        then
	    git init
            touch .scmversion
            git add .
            git commit -q -m "intial tarball for $LINUXVERSION"
            git gc -q
	fi
    else
	# sources already extracted, need to check which patches are already applied
	if [ -d .git ]
	then
	    # source dir controlled with git, simply list patch already applied
	    declare -a patches_applied
	    while read l
	    do
		l=${l#* }
		patches_applied+=("$l")
	    done< <(git log --oneline)

	    for (( i=${#patches_applied[@]}-1 ; i>=0 ; i-- )) ; do
		# limit to 50 character to avoid searching on multiple line
		local p=${patches_applied[$i]}
		if [ ${#p} -gt 50 ]
		then
		    p=${p:0:49}
		    p=${p% *}
		fi

		for (( j=0 ; j<${#patches[@]} ; j++ )) ; do
		    if [ ${patches[$j]} ] && (grep -q -F "$p" ${patches[$j]})
		    then
			patches[$j]=''
			break;
		    fi
		done
	    done
	elif [ -r ".patch_applied" ]
	then
	    # list of patches applied listed in .patch_applied file
	    for (( i=0 ; i<${#patches[@]} ; i++ )) ; do
		local p=$(basename ${patches[$i]})
		if (grep -q $p .patch_applied )
		then
		    patches[$i]=''
		fi
	    done
	else
	    # Find the first patch that would apply without error
	    local skip=-1
	    for (( i=0 ; i<${#patches[@]} ; i++ )) ; do
		local ignored=0 failed=0

		while read l
		do
		    if [[ $l =~ ignored ]]
		    then
			ignored=$((ignored + 1))
		    elif [[ $l =~ hunk.*FAILED ]]
		    then
			failed=$((failed + 1))
		    fi
		done< <(patch -p1 --dry-run -i ${patches[$i]} -t -s -N -p1)

		if [ $ignored -eq 0 -a $failed -eq 0 ]
		then
		    # stop on first patch that would apply.
		    # Assume next patches are not applied
		    break;
		elif [ $ignored -gt 0 ]
		then
		    #if at least one hunk is ignored, assume patch as already been applied
		    echo $(basename ${patches[$i]}) >> .patch_applied
		    patches[$i]=''
		else # ignored=0 failed>0
		    error "Cannot apply ${patches[$i]} and it doesn't seems already applied"
		    error "If patch hasn't been applied then fix conflict."
		    error "To avoid error in next update, or if script didn't detect that patch was already applied,"
		    error "create the file '.patch_applied' with the list of patch already applied"
		    exit
	        fi
	    done
	fi
    fi

    # Actually apply patches
    if [ -d .git ]
    then
	for p in ${patches[@]}
	do
	    if ( ! git am $p)
	    then
		error "Cannot apply $p. Please apply it manaually"
		exit
	    fi
	done
    else
	for p in ${patches[@]}
	do
	    echo Applying: $p
	    if ( patch -p1 -s -i $p )
	    then
		echo $(basename $p) >> .patch_applied
	    else
		error "Cannot apply $p. Please apply it manaually"
		error "Add it to .patch_applied once done"
		exit
	    fi
	done
    fi

    cd $HERE
}

function do_prepare() {
    if [ $prepare -eq 0 ] && [ $module -eq 0 -a $kernel -eq 0 -o -d $LINUXDIR/../$OUT ]
    then
        return
    fi

    status "Config and prepare header file"

    if [ ! -d $LINUXDIR ]
    then
        echo "Error Cannot find $LINUXDIR directory"
        echo "Use same value for LINUXDIR variable used during source download"
        exit
    fi

    if (! $KMAKE -C $LINUXDIR O=$LINUXDIR/../$OUT $config modules_prepare)
    then
	error "compilation target 'modules_prepare' failed"
	exit
    fi
}

function do_module() {
    if [ $module -eq 0 ]
    then
        return
    fi

    # update module version
    if [ -e ".svn" ]
    then
        svnrev=$(svnversion -c | sed 's/.*://')
    elif (git status -uno > /dev/null)
    then
        # maybe git-svn
        idx=0
        while [ "$svnrev" = "" ]
        do
            # loop on all commit to find the first one that match a svn revision
            svnrev=$(git svn find-rev HEAD~$idx)
            if [ $? -ne 0 ]
            then
                svnrev="Unknown Revision"
            elif [ "$svnrev" = "" ]
            then
                idx=$((idx + 1))
            elif [ $idx -gt 0 ] || (git status -uno --porcelain | grep -q '^ M')
            then
                # If this is not the HEAD, or working copy is not clean then we're
                # not at a commited svn revision so add 'M'
                svnrev=$svnrev"M"
            fi
        done
    else
        svnrev="Unknown Revision"
    fi

    if [ -e $LINUXDIR/drivers/platform/Kconfig ]
    then
        sed -i -e "s/\"svn.*\"/\"svn r$svnrev\"/" $LINUXDIR/drivers/platform/Kconfig
    fi

    status "Build modules (svnrev=$svnrev)"
    if (! $KMAKE -C $KERNELDIR INSTALL_MOD_PATH=$KERNELDIR/module_dir modules modules_install)
    then
	error "compilation targets 'modules modules_install' failed"
	exit
    fi
}

function do_kernel() {
    if [ $kernel -eq 0 ]
    then
        return
    fi

    status "Build KERNEL"
    if (! $KMAKE -C $KERNELDIR $kimg)
    then
	error "compilation targets '$kimg' failed"
	exit
    fi
}

## read command line options
while getopts "h67hdupmkaj:" optname
do
    case "$optname" in
        6) target=diniv6;;
        7) target=cevav7;;
        d) download=1;;
	u) update=1;;
        p) prepare=1;;
        m) module=1;;
        k) kernel=1;;
        a) download=1
           prepare=1
           module=1
           kernel=1;;
        j) JOBS=$OPTARG;;
        h) echo $usage
           echo "$help"
           exit;;
        *) echo "Unknown error while processing options (use -h for help)"
           echo $usage
           exit;;
    esac
done
shift $((OPTIND-1))

set_target

# use absolute path
if [ ${LINUXDIR:0:1} != '/' ]
then
    LINUXDIR=$HERE/$LINUXDIR
fi

do_download
do_update
do_prepare
if [ -d $LINUXDIR/../$OUT ]
then
    KERNELDIR=$(cd $LINUXDIR/../$OUT && pwd)
else
    echo "Kernel not configured."
    exit
fi
do_module
do_kernel

echo "Set KERNELDIR variable to $KERNELDIR when compiling rwnx driver"

