/**
 *****************************************************************************************
 * @file genvcd.c
 *
 * @brief Convert binary dump of LA ram into a vcd file.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 *****************************************************************************************
 */

/*
 *****************************************************************************************
 * The WiFi system includes an embedded logic analyzer able to record signals from various
 * sources of the system (SW, MAC HW, PHY, platform). The system can be fully configured
 * by choosing the source (i.e. SW, MAC HW, PHY, platform) and banks of signals within
 * the source that will be recorded by the logic analyzer (aka diagnostic ports).
 * When an assertion check fails, or when manually triggered, the FW uploads to the driver
 * a debug dump containing the logic analyzer memory, as well as other debug information.
 * The driver then creates files within the debugfs folder that are read by the user
 * space script controlling the creation of the user readable debug information.
 * The genvcd executable, called from this script, is used to create a standard VCD file
 * readable by applications such as GtkWave, Eclipse or HW simulation tools from the
 * debug dump. It takes as parameter the path to the debug dump.
 *
 * Depending on the embedded logic analyzer version and the WiFi FPGA board that is used,
 * the executable behaves differently due to the specificities of each version. LA
 * versions older than 5.0 are indeed recording 3 x 32-bit signals coming from three
 * different sources. Each of the three sources can be configured with 2 x 16-bit signal
 * banks or 1 x 32-bit signal bank. Newer LA versions record 7 x 16-bit signals, where
 * each 16-bit set can be taken from any source/bank pair.
 * On Dini board, the choice of sources is not highly flexible: one is the MACPHYIF, the
 * second one is the FPGAA (MAC HW or platform), and the third one is either FPGAB
 * (RIU/MDM) or SW diags.
 *
 * In order to create a meaningful VCD file, the signal names within the VCD have to be
 * created according to the diagnostic port configuration and the timings within the VCD
 * shall be correct. For that, the executable requires:
 *    - The board type, i.e Dini or CevaV7 (located in rwnxplat file)
 *    - The LA version, sampling frequency and sampling mask (located in lamacconf file)
 *    - The diagnostic ports configuration (i.e. which source is recorded by the LA, and
 *      which diagnostic port bank(s) is output from  the source)
 *    - The signal names corresponding to the diagnostic ports that are configured
 *    - The content of the LA memory (located in mactrace file)
 *
 * On Dini board the diagnostic port configuration is retrieved as follows:
 *    - plfdiags: This file contains the information about the sources that are recorded
 *                by the LA (MAC HW or platform on one hand, and PHY or SW on another
 *                hand).
 *    - machwdiags, plfdiags, hdmdiags, fpgabdiags: These files allow retrieving the diag
 *      port banks within the source, for MAC HW, platform, MDM and RIU respectively.
 *
 * On CevaV7 the diagnostic ports configuration is retrieved differently according to the
 * LA version. For versions older than 5.0, the following files are used:
 *    - plfdiags: This file contains the information about the sources that are recorded
 *                by the LA.
 *    - machwdiags, riudiags, hdmdiags, plfdiags: These files allow retrieving the diag
 *      port banks within the source, for MAC HW, RIU, MDM and platform respectively.
 *
 * For newer LA versions, a unique file allows getting the complete diagnostic port
 * configuration as a list of source/bank pairs (file diag_conf.txt).
 *
 * Once the diagnostic port configuration is retrieved, the signal names are obtained by
 * reading the following files.
 * On Dini:
 *    - mac_hwdiag.txt, hdm_hwdiag.txt, fpgaa_hwdiag.txt, fpgab_hwdiag.txt and swdiags:
 *      These files contains the signal names for the MAC HW, MDM, platform, RIU and SW
 *      diags respectively.
 *
 * On CevaV7, LA version < 5.0:
 *    - mac_hwdiag.txt, hdm_hwdiag.txt, riu_hwdiag.txt, nxtop_hwdiag.txt: These files
 *      contains the signal names for the MAC HW/SW, MDM, RIU and platform respectively.
 *
 * On CevaV7, LA versions >= 5.0:
 *    - mac_hwdiag.txt, hdm_he_hwdiag.txt, riu_hwdiag.txt, top_hwdiag.txt: These files
 *      contains the signal names for the MAC HW/SW, MDM, RIU and platform respectively.
 *
 * From these files the executable creates groups of signals for all the diagnostic port
 * banks that are recorded. The groups can be single bits, or group of bits. Each group
 * has a unique id (used internally to the VCD) and a name (that will be displayed by the
 * application reading the VCD). The groups are also defined by their LSB and MSB within
 * the LA memory.
 * When creating the VCD the executable parses the LA memory file and read group values
 * according to their position in the memory.
 *
 *****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include <stdint.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <errno.h>

/*
 * DEFINES
 *****************************************************************************************
 */
#define BANK_MAX 7

#define V6_BOARD 6
#define V7_BOARD 7

#define VERSION(x,y) (((x) << 16) + (y))

#define BIT(x) (1 << x)

/// definition for one group (aka signal)
struct group {
    char name[255];
    char id[15];
    int lsb;
    int msb;
    int width;
    uint32_t msk;
    uint32_t prev;
};

/// definition for a group set (aka bank)
struct groups {
    int cnt;
    struct group group[32]; // Up to 32 groups per 32-bit bank
};


// Path of the top directory
char *top_dir;

/******************************************************************************************
 * UTILS FUNCTION
 ******************************************************************************************/
/**
 * Convert a numeric value into a string of bit
 *
 * @param[out] out       Output buffer
 * @param[in]  val       Numeric value to convert
 * @param[in]  width     Width, in bits, of the value
 * @param[in]  smp_msk   Sampling mask. Bit not in the sampling mask are converted as as 'x'
 *
 * @return 0 if at least one bit has been successfully converted.
 */
static int int2bin(char *out, uint32_t val, int width, uint32_t smp_msk)
{
    uint32_t z;
    int res = -1;

    out[0] = '\0';

    for (z = (1 << (width - 1)); z > 0; z >>= 1)
    {
        if (z & smp_msk)
        {
            strcat(out, ((val & z) == z) ? "1" : "0");
            res = 0;
        }
        else
            strcat(out, "x");
    }

    return res;
}

/**
 * Assign bits in a group set (aka bank) to a group (aka signal)
 *
 * @param[in,out] groups     Group set in which the new group must be created
 * @param[in]     name       Name of the group (What will be displayed by the trace viewer)
 * @param[in]     id_prefix  Prefix to use for this group in the trace.
 * @param[in]     lsb        LSB of the group inside the bank
 * @param[in]     msb        MSB of the group inside the bank
 * @param[in]     val        Initial value for the group
 *
 * @return 0 on success, -1 on error
 */
static int add_group(struct groups *groups, char *name, char *id_prefix, int lsb, int msb, uint32_t val)
{
    struct group *g;

    if (groups->cnt == 32)
        return -1;

    if (msb >= 32)
        return -1;

    g = &groups->group[groups->cnt];

    strcpy(g->name, name);
    sprintf(g->id, "%s%d", id_prefix, groups->cnt);
    g->lsb = lsb;
    g->msb = msb;
    g->width = msb - lsb + 1;
    if (g->width == 32)
        g->msk = 0xFFFFFFFF;
    else
        g->msk = ((1 << g->width) - 1) << g->lsb;
    g->prev = val;

    groups->cnt++;

    return 0;
}

/**
 * Increase the last group (aka signal) created in the group set (aka bank) by 1 bit
 *
 * @param[in,out] groups  Group set in which the last group must be enlarged
 *
 * @return 0 on success, -1 on error
 */
static int enlarge_group(struct groups *groups)
{
    struct group *g;

    if (groups->cnt == 0)
        return -1;

    g = &groups->group[groups->cnt - 1];

    if (g->msb >= 32)
        return -1;

    g->msb++;
    g->width++;
    g->msk |= (1 << g->msb);

    return 0;
}

/**
 * Write list of group defined in a group set in the VCD file
 *
 * @param[in] vcd     Trace file descriptor
 * @param[in] groups  Group set to write
 */
static void declare_groups(FILE *vcd, struct groups *groups)
{
    struct group *g;
    int i;

    for (i = 0, g = &groups->group[0]; i < groups->cnt; i++, g++)
    {
        if (g->width == 1)
        {
            fprintf(vcd, "$var wire 1 %s %s $end\n", g->id, g->name);
        }
        else
        {
            fprintf(vcd, "$var reg %d %s %s [%d:0] $end\n", g->width, g->id, g->name, g->width - 1);
        }
    }
}

/**
 * Write groups initial value
 *
 * @param[in] vcd       Trace file descriptor,
 * @param[in] groups    Group set to write
 * @param[in] sampling  Sampling mask for this group set
 */
static void init_groups(FILE *vcd, struct groups *groups, uint32_t sampling)
{
    struct group *g;
    int i;

    for (i = 0, g = &groups->group[0]; i < groups->cnt; i++, g++)
    {
        uint32_t smp_msk = (sampling & g->msk) >> g->lsb;

        if (g->width == 1)
        {
            if (smp_msk)
                fprintf(vcd, "%d%s\n", g->prev, g->id);
            else
                fprintf(vcd, "x%s\n", g->id);
        }
        else
        {
            char bin[64];
            int2bin(bin, g->prev, g->width, smp_msk);
            fprintf(vcd, "b%s %s\n", bin, g->id);
        }
    }
}

/**
 * Write groups updated value if it changed
 *
 * @param[in] vcd       Trace file descriptor,
 * @param[in] groups    Group set to write
 * @param[in] value     New value for this group set
 * @param[in] sampling  Sampling mask for this group set
 * @param[in] force     Force to dump the value even if it didn't change
 */
static void put_groups(FILE *vcd, struct groups *groups, uint32_t value, uint32_t sampling, int force)
{
    struct group *g;
    int i;

    for (i = 0, g = &groups->group[0]; i < groups->cnt; i++, g++)
    {
        uint32_t curr = (value & g->msk) >> g->lsb;
        uint32_t smp_msk = (sampling & g->msk) >> g->lsb;

        if ((curr != g->prev) || force)
        {
            if (g->width == 1)
            {
                if (smp_msk)
                    fprintf(vcd, "%d%s\n", curr, g->id);
                else if (force)
                    fprintf(vcd, "x%s\n", g->id);
            }
            else
            {
                char bin[64];
                if ((int2bin(bin, curr, g->width, smp_msk) == 0) || force)
                    fprintf(vcd, "b%s %s\n", bin, g->id);
            }
            g->prev = curr;
        }
    }
}

/**
 * Open a file in read mode within Top Directory
 *
 * @param[in]  name  Name of the file
 * @param[out] fd    FILE pointer updated if file is opened
 *
 * @return 0 on success, -1 on error
 */
static int open_file(char *name, FILE **fd)
{
    char filename[256];

    snprintf(filename, 255, "%s/%s", top_dir, name);
    *fd = fopen(filename, "r");

    if (*fd == NULL)
    {
        printf("Cannot open %s: %s\n", name, strerror(errno));
        return -1;
    }

    return 0;
}

/**
 * Read a 32bits register value dumped in ASCII in a file
 *
 * @param[in]  name  Name of the file containing the dump
 * @param[out] val   Pointer to store the register value
 *
 * @return 0 on success, -1 on error
 */
static int read_register_dump(char *name, uint32_t *val)
{
    FILE *fd;
    int res = -1;

    if (open_file(name,  &fd))
        return -1;

    if (fscanf(fd, "%x", val) == 1)
        res = 0;

    fclose(fd);
    return res;
}

/**
 * Read all diags configuration from a file.
 * Starting with LA version 5.0 all diags configuration is dumped in a single file
 * instead a several register dump like previous version
 *
 * configuration format is:
 * <title>
 * - Diag<id0>: <sel0> <port0> (description0...)
 * - Diag<id1>: <sel1> <port1> (description1...)
 * ...
 *
 * @param[in]     name      Name of the configuration file
 * @param[out]    diags     Table to retrieve the configuration.
 *                          A diag value is : selection << 16 || port_id
 * @param[in,out] nb_diags  Contains the size, in number of element, of the diags table.
 *                          It is updated with the actual number of diags updated.
 *
 * @return 0 on success, -1 on error
 */
static int read_diag_conf(char *name, uint32_t *diags, int *nb_diags)
{
    FILE *fd;
    int res = -1;
    char *line = NULL;
    size_t line_len = 0;
    int idx;
    int diag_sel;
    int diag_port;
    int map = 0;
    int cnt = 0;

    if (open_file(name,  &fd))
        return -1;

    // Skip the first line
    if (getline(&line, &line_len, fd) < 0)
        goto end;

    while (1)
    {
        if (getline(&line, &line_len, fd) < 0)
            goto end;

        if ((sscanf(line, "- Diag%d: 0x%x 0x%x (", &idx, &diag_sel, &diag_port) != 3))
            goto end;

        if (idx >= *nb_diags)
            goto end;

        diags[idx] = (diag_sel << 16) + diag_port;
        map |= (1 << idx);
        cnt++;
    }

  end:
    // The configuration must contains successive configurations starting at diag0
    if (cnt && (map == ((1 << cnt) - 1))) {
        res = 0;
        *nb_diags = cnt;
    }

    if (line)
        free(line);
    fclose(fd);
    return res;
}

/**
 * Test if a buffer contains an empty string.
 * Empty string refers to string with only space,tab and or new line.
 *
 * @param[in] line  buffer to test
 *
 * @return 1 if the buffer is an empty string and 1 otherwise
 */
static int is_empty_line(char *line)
{
    char c = *line++;

    while (c != 0)
    {
        // accept \n, space, \t or \r
        if (c != 0x0a && c != 0x20 && c != 0x9 && c != 0x0d)
            return 0;
        c = *line++;
    }

    return 1;
}

/**
 * Read a diag port description (i.e. list of all signals contained in this diag port)
 * and create corresponding group in the group set associated to this diag port.
 *
 * @param[in,out] fd        File descriptor, pointing to the first signal
 *                          description with a diag port
 * @param[in,out] line      Pointer to buffer used to store temporary line.
 *                          May be NULL, but it must be freed by caller is not NULL on exit.
 * @param[in,out] line_len  Pointer to store size on the buffer pointed by line
 * @param[in]     groups    Group set in which group for this diag port must be added
 * @param[in]     pref      Group prefix
 * @param[in,out] lsb       Indicate the LSB of the first group that will be added in the
 *                          group set. Updated with the LSB value to used if new group will
 *                          be added to this group set.
 *
 * @return 0 on success, -1 on error
 */
static int process_diag_port(FILE *fd, char **line, size_t *line_len,
                             struct groups *groups, char *pref, int *lsb)
{
    int bit = *lsb;
    char diag_name[64];

    // Ensure that diag_name will always contains a NULL terminating byte
    memset(diag_name, 0, sizeof(diag_name));;

    while (1)
    {
        if ((getline(line, line_len, fd) < 0) ||
            is_empty_line(*line))
        {
            *lsb = bit;
            return 0;
        }

        strtok(*line, "\n[*");

        if (strcmp(*line, "NULL") == 0)
        {
            // no need to define signal for it
        }
        // Compare with previous diag name to know if it is part of a group
        else if (strcmp(*line, diag_name))
        {
            strncpy(diag_name, *line, sizeof(diag_name) - 1);
            if (add_group(groups, diag_name, pref, bit, bit, 0))
                return -1;
        }
        else if (enlarge_group(groups))
            return -1;

        bit++;
    }

    return -1;
}

/**
 * Find a specific diag port description from a file and add its signal to its associated
 * group set.
 *
 * The description file format is:
 * <DIAG_PORT1_NAME>
 *  <DIAG_PORT1_VALUE (in hexa with 0x prefix)>
 *  <DIAG_PORT1_BIT0_NAME>
 *  <DIAG_PORT1_BIT1_NAME>
 *  ...
 *  <DIAG_PORT1_BITx_NAME>
 *  <empty line>
 *  <DIAG_PORT2_NAME>
 *  ...
 *
 * @param[in]     groups     Group set in which group for this diag port must be added
 * @param[in]     desc       Name of the description file
 * @param[in]     pref       Group prefix
 * @param[in]     diag_port  Id on the diag port
 * @param[in,out] lsb        Indicate the LSB of the first group that will be added in the
 *                           group set. Updated with the LSB value to used if new group will
 *                           be added to this group set.
 *
 * @return 0 on success, -1 on error
 */
static int create_groups_for_bank(struct groups *groups, char *desc, char *pref,
                                          int diag_port, int *lsb)
{
    FILE *fd;
    char *line = NULL;
    size_t line_len = 0;
    char diag_port_name[64];
    int val;
    int res = -1;

    // Ensure that diag_port_name will always contains a NULL terminating byte
    diag_port_name[sizeof(diag_port_name) - 1] = 0;

    if (open_file(desc, &fd))
        return -1;

    if (getline(&line, &line_len, fd) < 0)
        goto end;

    while (1) {
        // skip any empty line
        while (is_empty_line(line)) {
            if (getline(&line, &line_len, fd) < 0)
                goto end;
        }

        // Save diag port name
        strncpy(diag_port_name, line, sizeof(diag_port_name) - 1);

        // read diag_port value
        if ((getline(&line, &line_len, fd) < 0) ||
            (sscanf(line, "0x%x", &val) != 1))
        {
            printf("%s: line is not correctly formatted %s\n", desc, line);
            goto end;
        }

        if (val == diag_port)
        {
            printf(" Diag: %s", diag_port_name);
            if (!process_diag_port(fd, &line, &line_len, groups, pref, lsb))
                res = 0;

            goto end;
        }
        else
        {
            // skip till next empty line
            do {
                if (getline(&line, &line_len, fd) < 0)
                    goto end;
            } while (!is_empty_line(line));
        }
    }

  end:
    if (line)
        free(line);
    fclose(fd);
    return res;
}

/**
 * Create and initialize VCD file
 *
 * @param[out] fd          Updated with trace file descriptor
 * @param[in]  hwdiags     Table of group set
 * @param[in]  sampling    Table of sampling mask for each group set
 * @param[in]  bank_cnt    Number of bank (aka group set). It's the number of
 *                         element in hwdiags adn sampling tables
 * @param[in]  la_version  LA version
 *
 * @return 0 on success, -1 on error
 */
static int create_and_initialize_vcd(FILE **fd, struct groups *hwdiags, uint32_t *sampling,
                                     int bank_cnt, int la_version)
{
    char filename[256];
    int i;

    snprintf(filename, 255, "%s/trace.vcd", top_dir);
    *fd = fopen(filename, "w");
    if (*fd == NULL)
    {
        printf("Cannot create %s: %s\n", filename, strerror(errno));
        return -1;
    }

    /* print file header */
    fprintf(*fd, "$comment\nTOOL: RW LA\n$end\n$date\n$end\n$timescale\n1ps\n$end\n");

    /* declare groups - declare control first */
    declare_groups(*fd, &hwdiags[bank_cnt-1]);
    if (la_version < VERSION(5,0))
    {
        declare_groups(*fd, &hwdiags[2]);
        declare_groups(*fd, &hwdiags[0]);
        declare_groups(*fd, &hwdiags[1]);
    }
    else
    {
        for (i = 0; i < (bank_cnt - 1); i++)
            declare_groups(*fd, &hwdiags[i]);
    }
    fprintf(*fd, "\n$enddefinitions\n");

    /* initialize groups */
    fprintf(*fd, "$dumpvars\n");
    for (i = 0; i < bank_cnt; i++)
        init_groups(*fd, &hwdiags[i], sampling[i]);
    fprintf(*fd, "\n$end\n");

    return 0;
}

/**
 * Read LA configuration from 'lamacconf' file
 *
 * @param[out] sampling  Updated with sampling mask for all LA banks
 * @param[out] version   Update with LA version register value
 *
 * @return 0 on success, -1 on error
 */
static int get_la_conf(uint32_t *sampling, uint32_t *version)
{
    FILE *conf;
    int res = -1;

    if (open_file("lamacconf", &conf))
        return -1;

    // The config contains the sampling mask (3 32bits words)
    if (fread(sampling, 4, 3, conf) != 3)
        goto end;

    // Then the trigger mask and value (6 32bits words).
    // We don't use them so skip them
    fseek(conf, 6 * 4, SEEK_CUR);

    // Then the version (1 32bits word)
    if (fread(version, 4, 1, conf) != 1)
        *version = 0;

    if ((*version & 0xFFFFFF) < VERSION(5,0))
    {
        // Enable everything in the 'control' bank
        sampling[3] = 0xFFFFFFFF;
    }
    else
    {
        int i;

        // Since version 5.0, bank size is 16bits (and no longer 32bits) so dispatch
        // each 32bits mask (for 2 banks) into two 16bits masks.
        for (i = 2; i >= 0; i--)
        {
            sampling[2 * i + 1] = sampling[i] >> 16;
            sampling[2 * i] = sampling[i] & 0xFFFF;
        }

        // sampling mask for bank6 is not dumped, so assume everything is enabled
        sampling[BANK_MAX - 1] = 0xFFFF;

        // And then the 'control' bank
        sampling[BANK_MAX] = 0xFFFF;
    }

    res = 0;

  end:
    fclose(conf);
    return res;
}

/**
 * Get version type from 'rwnxplat' file.
 * Assume V6 platform if this file doesn't exist.
 *
 * @return V6_BOARD or V7_BOARD on success, -1 on error
 */
static int get_board(void)
{
    FILE *platform = NULL;
    char *line = NULL;
    size_t line_len = 0;
    int res = -1;

    if (open_file("rwnxplat", &platform))
    {
        // If no file, this is a v6 platform
        res = V6_BOARD;
    }
    else
    {
        if (getline(&line, &line_len, platform) < 0)
            goto end;

        if (line[1] == '6')
            res = V6_BOARD;
        else if (line[1] == '7')
            res = V7_BOARD;
    }

  end:
    if (line)
        free(line);
    if (platform)
        fclose(platform);
    return res;
}

/**
 * Each LA entry also contains some control bit (e.g. trigger source).
 * They are treated 'normal' signals associated to a specific group set.
 *
 * @param[in,out] control  Group set for the 'control' bank
 * @param[in]     version  LA version
 * @return 0 on success, -1 on error
 */
static int create_control_groups(struct groups *control, int version)
{
    printf("Create Control groups");

    control->cnt = 0;

    if (version < VERSION(4,0))
    {
        printf(" with 1 trigger\n");
        add_group(control, "trigger", "ct", 31,  31,  0);
    } else if (version < VERSION(5,0)) {
        printf(" with 6 triggers\n");
        add_group(control, "ext_trigger", "ct", 28,  31,  0);
        add_group(control, "sw_trigger", "ct", 27,  27,  0);
        add_group(control, "int_trigger", "ct", 26,  26,  0);
    } else {
        printf(" with 6 triggers\n");
        add_group(control, "ext_trigger", "ct", 12,  15,  0);
        add_group(control, "sw_trigger", "ct", 11,  11,  0);
        add_group(control, "int_trigger", "ct", 10,  10,  0);
    }
    return 0;
}

/******************************************************************************************
 * For V6 platform only
 * LA is composed of 3 32bits wide banks.
 ******************************************************************************************/
/// For Dini platform, diag_mux bit in DIAG_CONF register defined if PHY or SW diags were activated
#define DIAG_MUX_BIT BIT(15)
#define DIAG_MUX_PHY 1
#define DIAG_MUX_SW  0

/**
 * Read platform diags configuration to see if PHY diags of Software diags were activaed
 *
 * @return DIAG_MUX_PHY or DIAG_MUX_SW on success, -1 on error
 */
static int get_mux_diag_type(void)
{
    uint32_t diags;

    if (read_register_dump("plfdiags", &diags))
        return -1;

    if (diags & DIAG_MUX_BIT)
    {
        printf("PHY Diag selected\n");
        return DIAG_MUX_PHY;
    }
    else
    {
        printf("SW Diag selected\n");
        return DIAG_MUX_SW;
    }
}


/**
 * Create groups for the software diags.
 * This corresponds to Bank 0
 *
 * @param[in,out] swdiags Group set for the software diags
 * @return 0 on success, -1 on error
 */
static int create_swdiag_groups(struct groups *swdiags)
{
    FILE *fd;
    char *line = NULL;
    size_t line_len = 0;
    int lsb = 0;
    int res = 0;

    printf("Create SW diag groups\n");
    swdiags->cnt = 0;

    if (open_file("swdiags", &fd))
        return -1;

    res = process_diag_port(fd, &line, &line_len, swdiags, "sw", &lsb);
    if (res == 0)
    {
        for (; lsb < 32; lsb++)
        {
            char name[32];
            snprintf(name, sizeof(name), "SWDIAG%d", lsb);

            /* Add to the SW diags groups */
            add_group(swdiags, name, "sw", lsb, lsb, 0);
        }
    }

    if (line)
        free(line);
    fclose(fd);

    return res;
}

/**
 * Create groups for the FPGA B diags.
 * This corresponds to Bank 0 and may contains FPGA B or HDM diags depending
 * of DIAGCONF_CONF1 register in the RF interface.
 *
 * @param[in,out] hwdiags Group set for the FPGA B diags
 * @return 0 on success, -1 on error
 */
static int create_hwdiag_fpgab_groups(struct groups *hwdiags)
{
    uint32_t fpgabdiags;
    int lsb = 0;

    printf("Create HW diag for FPGA B groups");
    hwdiags->cnt = 0;

    // First read the configuration to check if the MAC HW or platform diags are output
    if (read_register_dump("fpgabdiags", &fpgabdiags))
        return -1;

    if ((fpgabdiags >> 24) != 0x14)
    {
        printf("(FPGA B diag port)\n");
        if (create_groups_for_bank(hwdiags, "fpgab_hwdiag.txt", "hwb",
                                           (fpgabdiags >> 24), &lsb))
            return -1;
    }
    else
    {
        uint32_t hdmdiags;
        printf("(HDM diag port)\n");

        if (read_register_dump("hdmdiags", &hdmdiags))
            return -1;

        if (create_groups_for_bank(hwdiags, "hdm_hwdiag.txt", "hwb",
                                           hdmdiags & 0xff, &lsb) ||
            create_groups_for_bank(hwdiags, "hdm_hwdiag.txt", "hwb",
                                           (hdmdiags >> 8) & 0xff, &lsb))
            return -1;
    }

    return 0;
}

/**
 * Create groups for the FPGA A diags.
 * This corresponds to Bank 1 and may contains FPGA A or MAC HW diags depending
 * of DIAG_CONF register in SYSCTRL.
 *
 * @param[in,out] hwdiags Group set for the FPGA A diags
 * @return 0 on success, -1 on error
 */
static int create_hwdiag_fpgaa_groups(struct groups *hwdiags)
{
    uint32_t plfdiags;
    int lsb = 0;

    printf("Create HW diag for FPGA A groups ");
    hwdiags->cnt = 0;

    // First read the configuration to check if the MAC HW or platform diags are output
    if (read_register_dump("plfdiags", &plfdiags))
        return -1;

    if ((plfdiags & 0xff) != 0x0C)
    {
        printf("(FPGA A diag port)\n");
        if (create_groups_for_bank(hwdiags, "fpgaa_hwdiag.txt", "hwa",
                                           (plfdiags & 0xff), &lsb))
            return -1;
    }
    else
    {
        uint32_t machwdiags;
        printf("(MAC HW diag port)\n");

        if (read_register_dump("machwdiags", &machwdiags))
            return -1;

        if (create_groups_for_bank(hwdiags, "mac_hwdiag.txt", "hwa",
                                           machwdiags & 0xff, &lsb) ||
            create_groups_for_bank(hwdiags, "mac_hwdiag.txt", "hwa",
                                           (machwdiags >> 8) & 0xff, &lsb))
            return -1;
    }

    return 0;
}

/**
 * Create groups for the MPIF A diags.
 * This corresponds to Bank 2
 *
 * @param[in,out] mpif Group set for the MPIF diags
 * @return 0 on success, -1 on error
 */
static int create_mpif_groups(struct groups *mpif)
{
    printf("Create MPIF groups\n");
    mpif->cnt = 0;

    add_group(mpif, "mpif_rifsDetected",     "mp", 0,  0,  0);
    add_group(mpif, "mpif_keepRfOn",         "mp", 1,  1,  0);
    add_group(mpif, "mpif_phyErr",           "mp", 2,  2,  0);
    add_group(mpif, "mpif_rxEnd_p",          "mp", 3,  3,  0);
    add_group(mpif, "mpif_rxErr_p",          "mp", 4,  4,  0);
    add_group(mpif, "mpif_rxEndForTiming_p", "mp", 5,  5,  0);
    add_group(mpif, "mpif_ccaSecondary",     "mp", 6,  6,  0);
    add_group(mpif, "mpif_ccaPrimary",       "mp", 7,  7,  0);
    add_group(mpif, "mpif_rxReq",            "mp", 16, 16, 0);
    add_group(mpif, "mpif_txEnd_p",          "mp", 17, 17, 0);
    add_group(mpif, "mpif_txReq",            "mp", 28, 28, 0);
    add_group(mpif, "mpif_rfshutdown",       "mp", 29, 29, 0);
    add_group(mpif, "mpif_phyRdy_u1",        "mp", 30, 30, 0);
    add_group(mpif, "mpif_macDataValid_u1",  "mp", 31, 31, 0);
    add_group(mpif, "mpif_rxData",           "mp", 8,  15, 0);
    add_group(mpif, "mpif_phyRdy",           "mp", 18, 18, 0);
    add_group(mpif, "mpif_macDataValid",     "mp", 19, 19, 0);
    add_group(mpif, "mpif_txData",           "mp", 20, 27, 0);

    return 0;
}

/******************************************************************************************
 * For V7 platform, LA version 4.0
 * LA is composed of 3 32bits banks.
 ******************************************************************************************/
/**
 * Create groups for all banks.
 * Banks configuration is read for several register dump.
 *
 * @param[in,out] hwdiags  Table of group set
 * @return 0 on success, -1 on error
 */
static int create_rwnxdiag_groups(struct groups *hwdiags)
{
    uint32_t plfdiags;
    char pref[] = "nx.";
    int idx;

    hwdiags->cnt = 0;

    // First open the platform configuration file to check which diags are output
    if (read_register_dump("plfdiags", &plfdiags))
        return -1;

    for (idx = 0; idx < 3; idx++)
    {
        int diag = plfdiags & 0xff;
        int port1, port2;
        int lsb = 0;
        plfdiags >>= 8;
        pref[2] = '0' + idx;

        if ((diag == 0x0C) || (diag == 0x12))
        {
            if (diag == 0x12)
            {
                // Software diags port ids
                port1 = 0x1E;
                port2 = 0x1F;
            }
            else
            {
                uint32_t machwdiags;
                if (read_register_dump("machwdiags", &machwdiags))
                    return -1;
                port1 = machwdiags & 0xFF;
                port2 = (machwdiags >> 8) & 0xFF;
            }

            printf("Create HW diag for MAC HW (0x%02x) (0x%02x)\n", port1, port2);
            if (create_groups_for_bank(&hwdiags[idx], "mac_hwdiag.txt", pref, port1, &lsb) ||
                create_groups_for_bank(&hwdiags[idx], "mac_hwdiag.txt", pref, port2, &lsb))
                return -1;
        }
        else if ((diag == 0x14) || (diag == 0x16))
        {
            uint32_t hdmdiags;

            if (read_register_dump("hdmdiags", &hdmdiags))
                 return -1;

            if (diag == 0x14)
            {
                port1 = hdmdiags & 0xFF;
                port2 = (hdmdiags >> 8) & 0xFF;
            }
            else
            {
                port1 = (hdmdiags >> 16) & 0xFF;
                port2 = (hdmdiags >> 24) & 0xFF;
            }

            printf("Create HW diag for HDM (0x%02x) (0x%02x)\n", port1, port2);
            if (create_groups_for_bank(&hwdiags[idx], "hdm_hwdiag.txt", pref, port1, &lsb) ||
                create_groups_for_bank(&hwdiags[idx], "hdm_hwdiag.txt", pref, port2, &lsb))
                return -1;
        }
        else if ((diag == 0x15) || (diag == 0x17))
        {
            uint32_t riudiags;

            if (read_register_dump("riudiags", &riudiags))
                 return -1;

            if (diag == 0x15)
            {
                port1 = riudiags & 0xFF;
                port2 = (riudiags >> 8) & 0xFF;
            }
            else
            {
                port1 = (riudiags >> 16) & 0xFF;
                port2 = (riudiags >> 24) & 0xFF;
            }

            printf("Create HW diag for RIU (0x%02x) (0x%02x)\n", port1, port2);
            if (create_groups_for_bank(&hwdiags[idx], "riu_hwdiag.txt", pref, port1, &lsb) ||
                create_groups_for_bank(&hwdiags[idx], "riu_hwdiag.txt", pref, port2, &lsb))
                return -1;
        }
        else
        {
            printf("Create HW diag for TOP\n");

            //create_groups_for_bank
            if (create_groups_for_bank(&hwdiags[idx], "nxtop_hwdiag.txt", pref, diag, &lsb))
                return -1;
        }
    }

    return 0;
}


/******************************************************************************************
 * For V7 platform, LA version 5.0
 * LA is composed of 7 16bits banks.
 ******************************************************************************************/
/**
 * Create groups for all banks.
 * Configuration is read form a single file.
 * Each bank is configured with a type (TOP, MAC, HDM, RIU) and a port id specific to the type.
 *
 * @param[in,out] hwdiags  Table of group set
 * @return Number of bank on success, -1 on error
 */
static int create_rwnxdiag16_groups(struct groups *hwdiags)
{
    uint32_t diags[BANK_MAX];
    int nb_diags = BANK_MAX;
    char pref[] = "nx.";
    int idx;

    hwdiags->cnt = 0;

    // First open the platform configuration file to check which diags are output
    if (read_diag_conf("diag_conf.txt", diags, &nb_diags))
        return -1;

    for (idx = 0; idx < nb_diags; idx++)
    {
        int diag_sel = diags[idx] >> 16;
        int diag_port = diags[idx] & 0xFFFF;
        int lsb = 0;
        pref[2] = '0' + idx;

        switch (diag_sel)
        {
            case 0:
                if (create_groups_for_bank(&hwdiags[idx], "top_hwdiag.txt", pref, diag_port, &lsb))
                    return -1;
                break;
            case 1:
                if (create_groups_for_bank(&hwdiags[idx], "mac_hwdiag.txt", pref, diag_port, &lsb))
                    return -1;
                break;
            case 2:
                if (create_groups_for_bank(&hwdiags[idx], "hdm_he_hwdiag.txt", pref, diag_port, &lsb))
                    return -1;
                break;
            case 3:
                if (create_groups_for_bank(&hwdiags[idx], "riu_hwdiag.txt", pref, diag_port, &lsb))
                    return -1;
                break;
            default:
                printf("Unknown diag sel %d\n", diag_sel);
                return -1;
        }
    }

    return idx;
}


/**
 * Extract fw trace timestamp from the LA memory dump (v5.0 only)
 *
 * In newer firmware version, before stopping the trace, the counter (32bits) used to
 * timestamp fw trace is dumped on both bank0/1 and bank4/5. (dumping on 2 different
 * couple of bank, allow to ensure we read the correct value).
 * So this function read the last NB_ENTRY and search for the first one that contains
 * the same value on bank0/1 and bank4/5.
 * If such trace is found, the trace is re-read from the beginning to count the number
 * of LA cycles from the first valid trace entry up to here. This allow to update the
 * nb_cycles_orig value so that the timestamp in the vcd file for this trace would be
 * the same as the counter value read.
 *
 * Even if we don't want to use fw trace timestamp (parameter use_fw_ts), this function
 * should still be called so that it allow to skip decoding the trace that contains
 * the counter.
 *
 * If counter is not found this function simply returns 0.
 *
 * @param[in]  trace           File descriptor for the trace
 * @param[in]  la_version      Version of the LA
 * @param[in]  delay           Duration in ps for one LA cycle
 * @param[out] trace_size      Updated with the number of bytes to read. This allow
 *                             to skip decoding trace that contains the counter.
 * @param[out] nb_cycles_orig  Updated with the number of LA cycles to use
 *                             for the first trace entry.
 * @param[in]  use_fw_ts       Whether to update nb_cycles_orig or not.
 *
 * @return 0 on success and -1 on errors.
 */
static int extract_fw_trace_timestamp(FILE *trace, int la_version, double delay,
                                      long *trace_size, uint64_t *nb_cycles_orig,
                                      int use_fw_ts)
{
    #define NB_ENTRY 8
    uint16_t entry_value[8];
    uint32_t fw_trace_timestamp_us = 0;
    uint64_t la_cycles = 0;
    long to_read;
    int first, entry_size;

    if (la_version < VERSION(5,0))
        return 0;

    entry_size = sizeof(entry_value);

    if (fseek(trace, -(NB_ENTRY * entry_size) , SEEK_END) < 0)
        return 0;

    for (int i = 0; i < NB_ENTRY; i++)
    {
        if (fread(entry_value, entry_size, 1, trace) <= 0)
            return 0;

        if (entry_value[7] & 0x8000)
            continue;

        if ((entry_value[0] || entry_value[1]) &&
            (entry_value[0] == entry_value[4]) &&
            (entry_value[1] == entry_value[5]))
        {
            fw_trace_timestamp_us = (entry_value[1] << 16) + entry_value[0];
            printf("Found FW trace timestamp: %"PRIu32"us\n", fw_trace_timestamp_us);

            // Now that we've got the fw_trace_timestamp for this LA entry we need to read
            // the trace from the beginning up to here to find fw_trace_timestamp of the
            // first LA entry
            to_read = ftell(trace);

            // But we don't need to decode the entry that contains the timestamp
            // (and the one before just to be sure)
            *trace_size = to_read - 2 * entry_size;
            break;
        }
    }

    rewind(trace);
    if (!fw_trace_timestamp_us || !use_fw_ts)
        return 0;

    // Count the number of LA cycles from the start of the trace
    first = 1;
    while (to_read)
    {
        if (fread(entry_value, entry_size, 1, trace) <= 0)
            return -1;

        to_read -= entry_size;
        if (entry_value[7] & 0x8000)
        {
            if (!first)
                la_cycles += entry_value[0] + ((entry_value[1] & 0xFF) << 16);
        }
        else
        {
            la_cycles += (entry_value[7] & 0x1FF);
            first = 0;
        }
    }

    // update nb_cycles_orig so that: (nb_cycles_orig + la_cycles) * delay = fw_timestamp
    *nb_cycles_orig = (uint64_t)(((double)fw_trace_timestamp_us * 1000000.0) / delay);
    *nb_cycles_orig -= la_cycles;

    rewind(trace);

    return 0;
}

/******************************************************************************************
 * MAIN
 ******************************************************************************************/
char options[] = "[-u (use int for computation instead of double)] [-t <use fw timestamp as reference>] <top_dir>";

int main(int argc, char **argv)
{
    FILE *vcd = NULL;
    FILE *trace = NULL;
    uint32_t value[BANK_MAX + 1];
    uint32_t sampling[BANK_MAX + 1];
    uint32_t la_version_mac;
    uint64_t delay;
    double delay_d;
    uint64_t nb_cycles_stable = 0, nb_cycles_orig = 0, nb_cycles = 0;
    int freq;
    int i;
    int bank_size = 4;
    int bank_cnt = 4;
    int entry_size;
    int first_entry = 1;
    int res = -1;
    int board;
    int la_version = VERSION(3,0);
    long trace_size;
    int use_uint = 0;
    int use_fw_ts = 0;
    struct groups rwnxdiag[BANK_MAX + 1];
    struct stat st;
    char *prog_name;

    memset(rwnxdiag, 0, sizeof(rwnxdiag));

    prog_name = argv[0];
    argc--;
    argv++;
    while ((argc > 0) && (argv[0][0] == '-'))
    {
        switch (argv[0][1])
        {
            case 'u':
                use_uint = 1;
                break;
            case 't':
                use_fw_ts = 1;
                break;
            case 'h':
                printf("Usage: %s %s\n", prog_name, options);
                return 0;
            default:
                printf("Invalid option %s\n", argv[0]);
                return -1;
        }
        argc--;
        argv++;
    }

    if (argc < 1) {
        printf("Usage: %s %s\n", prog_name, options);
        return -1;
    }

    top_dir = argv[0];
    stat(top_dir, &st);
    if (!S_ISDIR(st.st_mode))
    {
        printf("%s is not a directory\n", top_dir);
        return -1;
    }

    /* Get LA configuration */
    get_la_conf(sampling, &la_version_mac);

    /* In case the sampling frequency information is available, recompute the VCD delay */
    freq = (la_version_mac >> 24);
    if (freq != 0)
    {
        delay_d = 1000000.0/freq;
        delay = (uint64_t)delay_d;
    }
    else
    {
        delay_d = 1000000.0/70.0;
        delay = (uint64_t)delay_d;
    }

    if (la_version_mac)
        la_version = la_version_mac & 0xFFFFFF;

    printf("LA Info.\n Sampling Freq : %i MHz\n Version : %i.%i\n",
           freq, (la_version >> 16), (la_version & 0xFFFF));

    board = get_board();
    if (board == V6_BOARD)
    {
        int mux_diag_type;
        printf("Dini platform\n");

        mux_diag_type = get_mux_diag_type();
        if (mux_diag_type == DIAG_MUX_PHY)
        {
            if (create_hwdiag_fpgab_groups(&rwnxdiag[0]))
                goto end;
        }
        else if (mux_diag_type == DIAG_MUX_SW)
        {
            if (create_swdiag_groups(&rwnxdiag[0]))
                goto end;
        }
        else
            goto end;

        if (create_hwdiag_fpgaa_groups(&rwnxdiag[1]) ||
            create_mpif_groups(&rwnxdiag[2]))
            goto end;
    }
    else if (board == V7_BOARD)
    {
        printf("Ceva v7 platform\n");
        if (la_version < VERSION(5,0))
        {
            if (create_rwnxdiag_groups(rwnxdiag))
                goto end;
        }
        else
        {
            bank_size = 2;
            bank_cnt = create_rwnxdiag16_groups(rwnxdiag) + 1;
            if (bank_cnt <= 0)
                goto end;
        }
    }
    else
    {
        printf("Unsupported platform\n");
        goto end;
    }

    create_control_groups(&rwnxdiag[bank_cnt-1], la_version);

    /* Create VCD file */
    printf("Create VCD file\n");
    if (create_and_initialize_vcd(&vcd, rwnxdiag, sampling, bank_cnt, la_version))
        goto end;

    /* Open LA binary dump */
    if (open_file("mactrace", &trace))
        goto end;

    /* Get trace size */
    fseek(trace, 0, SEEK_END);
    trace_size = ftell(trace);
    rewind(trace);

    /* Check if fw timstamp is present in the trace */
    if (extract_fw_trace_timestamp(trace, la_version, delay_d, &trace_size,
                                   &nb_cycles_orig, use_fw_ts))
    {
        printf("Unexpected error while reading FW trace timestamp\n");
        goto end;
    }

    /* Go through the whole file */
    entry_size = bank_cnt * bank_size;
    while( trace_size >= entry_size)
    {
        for (i = 0; i < bank_cnt; i++)
        {
            if (fread(&value[i], bank_size, 1, trace) != 1)
                goto end;
        }
	trace_size -= entry_size;

        // A trace entry contains the new value for each bank and the number of
        // LA cycles this value was stable. Depending on LA version this number
        // of cycles can be split accross several trace entries.
        if (la_version <= VERSION(2,0))
        {
            if (value[3] & 0x80000000)
            {
                nb_cycles += value[0];
                continue;
            }
            else
            {
                nb_cycles++;
            }
        }
        else if (la_version < VERSION(4,0))
        {
            nb_cycles_stable = value[3] & 0x7FFFFFFF;
            value[3] &= 0x80000000;
        }
        else if (la_version < VERSION(5,0))
        {
            nb_cycles_stable = value[3] & 0x03FFFFFF;
            value[3] &= 0xFC000000;
        }
        else
        {
            uint16_t controls = value[bank_cnt - 1];
            if (controls & 0x8000)
            {
                nb_cycles += (value[0] & 0xFFFF) + ((value[1] & 0x00FF) << 16);
                continue;
            }
            else
            {
                nb_cycles_stable = controls & 0x01FF;
                value[bank_cnt - 1] &= 0xFE00;
            }
        }

        if (first_entry)
            nb_cycles = nb_cycles_orig;

        if (use_uint)
            fprintf(vcd, "#%"PRIu64"\n", (uint64_t)(nb_cycles * delay));
        else
            fprintf(vcd, "#%"PRIu64"\n", (uint64_t)(nb_cycles * delay_d));

        for (i = (bank_cnt - 1); i >= 0; i--)
            put_groups(vcd, &rwnxdiag[i], value[i], sampling[i], first_entry);

        first_entry = 0;
        nb_cycles += nb_cycles_stable;
    }
    res = 0;

  end:
    if (vcd)
        fclose(vcd);
    if (trace)
        fclose(trace);

    return res;
}
