/*
 *  From devmem.c: Simple program to read/write from/to any location in memory.
 *
 *  Copyright (C) 2000, Jan-Derk Bakker (J.D.Bakker@its.tudelft.nl)
 *
 *
 * This software has been developed for the LART computing board
 * (http://www.lart.tudelft.nl/). The development has been sponsored by
 * the Mobile MultiMedia Communications (http://www.mmc.tudelft.nl/)
 * and Ubiquitous Communications (http://www.ubicom.tudelft.nl/)
 * projects.
 *
 * The author can be reached at:
 *
 *  Jan-Derk Bakker
 *  Information and Communication Theory Group
 *  Faculty of Information Technology and Systems
 *  Delft University of Technology
 *  P.O. Box 5031
 *  2600 GA Delft
 *  The Netherlands
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 * RW modification:
 * - change output for read operation (simply print read values)
 * - Search for rwnx device on pci bus
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <dirent.h>
#include <stdint.h>

#define FATAL do { fprintf(stderr, "Error at line %d, file %s (%d) [%s]\n", \
			   __LINE__, __FILE__, errno, strerror(errno)); exit(1); } while(0)

#define PCI_VENDOR_ID_DINIGROUP              0x17DF
#define PCI_DEVICE_ID_DINIGROUP_DNV6_F2PCIE  0x1907

#define PCI_VENDOR_ID_XILINX                 0x10EE
#define PCI_DEVICE_ID_XILINX_CEVA_VIRTEX7    0x7011

int read_bar_address(char *name, int bar, off_t *bar_addr)
{
	FILE *file;
	int i, found = 0;
	char buf[256];
	unsigned long long start, end, flags;

	if ((file = fopen(name, "r")) == NULL)
		return -1;

	// skip bar we don't want
	for (i = 0 ; i < bar ; i++) {
		if (!fgets(buf, sizeof(buf), file))
			goto end;
	}

	if (!fgets(buf, sizeof(buf), file))
		goto end;

	found = sscanf(buf, "%llx %llx %llx", &start, &end, &flags);
	*bar_addr = (start & 0xFF000000);
end:
	fclose(file);
	return (found != 3);
}

int read_id(char *name, int *id)
{
	FILE *file;
	int found = 0;

	if ((file = fopen(name, "r")) == NULL)
		return -1;
	found = fscanf(file, "0x%x", id);
	fclose(file);

	return (found != 1);
}

int find_rwnx_device(off_t *bar_addr)
{
	DIR *devices = NULL;
	struct dirent *current;
	char filename[] = "/sys/bus/pci/devices/xxxxx:xx:xx.xxxxxx/resource ";
	int ret = 1;
	int max_len = sizeof(filename) - 21 - 9 - 1; // 21=/sys/.../devices 9=/resource 1=\0


	if ((devices = opendir("/sys/bus/pci/devices")) == NULL) {
		ret = -1;
		goto end;
	}

	while ((current = readdir(devices)))
	{
		size_t len = strlen(current->d_name);
		int vendor, device;

		if (current->d_name[0] == '.' || len > max_len)
			continue;

		strncpy(&filename[21], current->d_name, len + 1);

		strncpy(&filename[21 + len], "/device\0", 8);
		if (read_id(filename, &device))
			continue;

		strncpy(&filename[21 + len], "/vendor\0", 8);
		if (read_id(filename, &vendor))
			continue;

		strncpy(&filename[21 + len], "/resource\0", 10);
		if (vendor == PCI_VENDOR_ID_DINIGROUP &&
		    device == PCI_DEVICE_ID_DINIGROUP_DNV6_F2PCIE) {
			ret = read_bar_address(filename, 4, bar_addr);
			goto end;

		} else if (vendor == PCI_VENDOR_ID_XILINX &&
			   device == PCI_DEVICE_ID_XILINX_CEVA_VIRTEX7) {
			ret = read_bar_address(filename, 1, bar_addr);
			goto end;
		}
	}

end:
	if (devices)
		closedir(devices);

	return ret;
}


static const char help[] = "\n\
Usage:\t%s [-x] { address } [ type [ data ] ]\n\
 -x      : Don't search for rwnx device\n\
 address : memory address to act upon\n\
 type    : access operation type : [b]yte, [h]alfword, [w]ord\n\
 data    : data to be written\n\n\
If -x is not specified, address is assumed to be a \"rwnx\" system address, and program will automatically\n\
get the bar address for the rwnx device (and return an error if device is not found).\n\
In this case the 8 most significant bits of the address are ignored, replaced by bar address.\n\
";

int main(int argc, char **argv) {
	int fd;
	void *map_base, *virt_addr;
	uint32_t read_result, writeval;
	off_t target, bar;
	int arg_idx = 1;
	int access_type = 'w';
	int search_rwnx = 1;
	unsigned long page_size;
	unsigned long page_mask;

	if(argc < 2) {
		fprintf(stderr, help, argv[0]);
		exit(1);
	}

	if ( strncmp(argv[arg_idx], "-x", 2) == 0 ) {
		arg_idx++;
		search_rwnx = 0;

		if (arg_idx >= argc) {
			fprintf(stderr, help, argv[0]);
			exit(1);
		}
	}

	target = strtoul(argv[arg_idx++], 0, 0);

	if (search_rwnx) {
		if (find_rwnx_device(&bar)) {
			fprintf(stderr, "Cannot find rwnx device\n");
			exit(3);
		}

		if (bar == 0) {
			fprintf(stderr, "No resource allocated for rwnx device\n");
			exit(3);
		}
		target = (target & 0xFFFFFF) | bar;
	}

	if(arg_idx < argc)
		access_type = tolower(argv[arg_idx++][0]);

	if((fd = open("/dev/mem", O_RDWR | O_SYNC)) == -1) FATAL;

	/* Map one page */
	page_size = sysconf(_SC_PAGE_SIZE);
	if ((signed)page_size < 0) FATAL;

	page_mask = page_size - 1;
	map_base = mmap(0, page_size, PROT_READ | PROT_WRITE, MAP_SHARED, fd,
			target & ~page_mask);
	if(map_base == (void *) -1) FATAL;

	virt_addr = map_base + (target & page_mask);
	switch(access_type)
	{
	case 'b':
		read_result = *((volatile uint8_t *) virt_addr);
		break;
	case 'h':
		read_result = *((volatile uint16_t *) virt_addr);
		break;
	case 'w':
		read_result = *((volatile uint32_t *) virt_addr);
		break;
	default:
		fprintf(stderr, "Illegal data type '%c'.\n", access_type);
		exit(2);
	}

	if(arg_idx < argc)
	{
		writeval = (uint32_t)strtoul(argv[arg_idx], 0, 0);
		switch(access_type)
		{
		case 'b':
			*((volatile uint8_t *) virt_addr) = writeval;
			break;
		case 'h':
			*((volatile uint16_t *) virt_addr) = writeval;
			break;
		case 'w':
			*((volatile uint32_t *) virt_addr) = writeval;
			break;
		}
	}
	else
	{
		switch(access_type)
		{
		case 'b':
			printf("%-.2x\n",(uint8_t)(read_result&0xff));
			break;
		case 'h':
			printf("%-.4x\n",(uint16_t)read_result&0xffff);
			break;
		case 'w':
			printf("%-.8x\n",(uint32_t)read_result);
			break;
		}
	}
	if(munmap(map_base, page_size) == -1) FATAL;
	close(fd);
	return 0;
}
