/**
 *****************************************************************************************
 * @file nxreg.c
 *
 * @brief nxreg application: interfacing host for read and write of Platform registers
 *
 * Copyright (C) RivieraWaves 2011-2012
 *
 *****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <assert.h>
#include <string.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
// for vfork
//#define __USE_BSD
#include <unistd.h>

//#include "nxreg.h"


/*
 * DEFINES
 *****************************************************************************************
 */
#define PK_ID                'nxr\x00'
#define PK_ID_HEX            0x6e7872
#define PK_ID_SIZE           4
#define PK_HEADER_SIZE       8
#define MAX_BUFFER_SIZE      4096
#define NX_REG_GUI_PORT      0x6E58 //28248 far zone of available

//Requests
#define REG_WR_REQ           0x00000000
#define REG_RD_REQ           0x00000001
//Responses
#define REG_WR_RSP           0x80000000
#define REG_RD_RSP           0x80000001
#define REG_UNK_RSP          0x8FFFFFFF


//BAR access
#define BAR0_BASE            0xC1C00000
#define BAR1_BASE            0xC1800000

#define BAR1_LOW_ADDR_MASK   0x00000108
#define BAR1_UP_ADDR         0x000000F0
#define BAR1_LOW_ADDR        0x000000F4

struct nxreg_env_tag
{
    struct sockaddr_in sockaddr;
    int listen_fd;

    unsigned char * bar0_base;
    unsigned char * bar1_base;

    uint32_t current_addr_base;
};

/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
///NXREG Context variable, used to store NXREG Context data
struct nxreg_env_tag nxreg_env;

/*
 * FUNCTION DECLARATIONS
 *****************************************************************************************
 */
/*
 *****************************************************************************************
 * @brief Function that checks whether test is True and prints out the error message,
 *        followed by application exit.
 * @param test    Boolean result of a test condition.
 * @param string  Character string to print out before exiting
 *****************************************************************************************
 */
void error_check(int test, char * string);

/*
 *****************************************************************************************
 * @brief Function that formats packet containing response for client with recognizable header
 * @param fd          Client socket handle on which packet will be sent
 * @param param_size  Octet size of payload in packet to send
 * @param msg         Pointer to buffer where the message to send is kept
 *****************************************************************************************
 */
void send_msg(int fd, int param_size, char *msg);

/*
 *****************************************************************************************
 * @brief Function that configures the BAR1 window into the register space, using BAR0
 * @param addr_high  Higher octets of the address base to use through the set window
 *****************************************************************************************
 */
void config_bar1_window(uint32_t addr_high);

/*
 *****************************************************************************************
 * @brief Function which allows reading a register using its full address
 * @param tx_buffer  Buffer for response message to host, with given address and read value
 * @param rx_buffer  Part of command buffer from host - address is extracted from it.
 * @param param_size Command param size - in bytes
 * @return Number of 4octet words which become the parameters in the response to host
 *****************************************************************************************
 */
uint32_t reg_emb_read(uint32_t *tx_buffer, uint32_t *rx_buffer, uint32_t param_size);

/*
 *****************************************************************************************
 * @brief Function which allows writing a register using its full address, and read back
 * @param tx_buffer  Buffer for response message to host, with given address and read back written value
 * @param rx_buffer  Part of command buffer from host - address is extracted from it.
 * @param param_size Command param size - in bytes
 * @return Number of 4octet words which become the parameters in the response to host
 *****************************************************************************************
 */
uint32_t reg_emb_write(uint32_t *tx_buffer, uint32_t *rx_buffer, uint32_t param_size);

/*
 *****************************************************************************************
 * @brief Function that handles interpretation and actions for the received correct
 *        register R/W packets
 * @param fd          Client socket handle on which packet is received and response will be sent
 * @param payl_size   Size of data after header to use for action
 * @param buffer      Pointer value to start of payload part of received packet
 *****************************************************************************************
 */
void reg_rdwr_hdl(int fd, int param_size, char *buffer);

/*
 *****************************************************************************************
 * @brief Function that manages reception and sending responses on the client socket.
 * @param fd    Client socket handle on which reception is done/response packets are sent
 *****************************************************************************************
 */
void client_hdl(int fd);

/*
 *****************************************************************************************
 * @brief Function which allows creating the server socket, binding it to the Address needed
 *        and putting it in listen mode.
 * @param port  TCP/IP socket port number
 *****************************************************************************************
 */
void socket_init(int port);



/*
 * FUNCTION DEFINITIONS
 *****************************************************************************************
 */
void error_check(int test, char * string)
{
    if (!test)
    {
        perror(string);
        exit(-1);
    }
}

void send_msg(int fd, int param_size, char *msg)
{
    //copy ID code at start message buffer
    msg[0] = 'n';
    msg[1] = 'x';
    msg[2] = 'r';
    msg[3] = 0x00;
    //memcpy(msg, PK_ID, PK_ID_SIZE);
    //add length after ID
    *(int*) (msg + PK_ID_SIZE) = htonl(param_size);
    //send filled packet through client socket
    send(fd, msg, param_size + PK_HEADER_SIZE, 0);
}

void config_bar1_window(uint32_t addr_high)
{
    //mask
    //printf ("BAR0 mask 0x%08X\n",nxreg_env.bar0_base + BAR1_LOW_ADDR_MASK);
    *((volatile unsigned int *)(nxreg_env.bar0_base + BAR1_LOW_ADDR_MASK)) = 0xFFF00000;
    //LSB
    //printf ("BAR0 LSB 0x%08X\n",nxreg_env.bar0_base + BAR1_UP_ADDR);
    *((volatile unsigned int *)(nxreg_env.bar0_base + BAR1_UP_ADDR))  = 0x00000000;
    //MSB
    //printf ("BAR0 MSB 0x%08X\n",nxreg_env.bar0_base + BAR1_LOW_ADDR);
    *((volatile unsigned int *)(nxreg_env.bar0_base + BAR1_LOW_ADDR))  = addr_high;
}


uint32_t reg_emb_read(uint32_t *tx_buffer, uint32_t *rx_buffer, uint32_t param_size)
{
    assert(param_size == 4);

    //ensure aligned
    uint32_t address = (ntohl(rx_buffer[0])) & 0xfffffffc;

    //address is based on full register group base
    uint32_t addr_base   = address & 0xFFF00000;
    uint32_t addr_offset = address & 0x000FFFFF;

    //printf ("Read addr offset 0x%08X\n",addr_offset);

    //cfg BAR1 window if necessary
    if(nxreg_env.current_addr_base != addr_base)
    {
        //printf ("Configuring bar window 0x%08X\n",addr_base);
        config_bar1_window(addr_base);
        nxreg_env.current_addr_base = addr_base;
    }
    //printf ("Read addr 0x%08X\n",nxreg_env.bar1_base + addr_offset);
    volatile uint32_t *reg = ((volatile uint32_t *)(nxreg_env.bar1_base + addr_offset));

    // copy address and do the read
    tx_buffer[0] = rx_buffer[0];
    tx_buffer[1] = htonl(*reg);
    printf("  Read value: at[0x%08x] -> 0x%08x\n", address, *reg);

    return 2*4;
}

uint32_t reg_emb_write(uint32_t *tx_buffer, uint32_t *rx_buffer, uint32_t param_size)
{
    assert(param_size == 8);

    uint32_t address = ntohl(rx_buffer[0]) & 0xfffffffc;
    uint32_t value   = ntohl(rx_buffer[1]);

    //address is based on full register group base
    uint32_t addr_base   = address & 0xFFF00000;
    uint32_t addr_offset = address & 0x000FFFFF;

    //cfg BAR1 window if necessary
    if(nxreg_env.current_addr_base != addr_base)
    {
        config_bar1_window(addr_base);
        nxreg_env.current_addr_base = addr_base;
    }

    volatile uint32_t *reg = ((volatile uint32_t *)(nxreg_env.bar1_base + addr_offset));

    *reg = value;

    // copy address and re-read
    tx_buffer[0] = rx_buffer[0];
    tx_buffer[1] = htonl(*reg);
    printf("  Wrote value: at[0x%08x] <- 0x%08x\n", address, value);

    return 2*4;
}

void reg_rdwr_hdl(int fd, int param_size, char *buffer)
{
    //tx buffer - U32 type
    uint32_t tx_buffer[MAX_BUFFER_SIZE / 4];
    //number of octets of payload for tx
    uint32_t tx_size;
    uint32_t *rx_buffer = (uint32_t*) buffer;

    //Parameters are 4 bytes each
    // |  R/W_REQ_CODE   |   ADDR   |   VALUE   |
    switch(ntohl(rx_buffer[0]))
    {
        case REG_WR_REQ:
            printf("W Req received:  ADDR=0x%08X VAL=0x%08X \n", rx_buffer [1], rx_buffer [2]);

            //response buffer
            tx_buffer[2] = htonl(REG_WR_RSP);

            //write
            tx_size = 4 + reg_emb_write(tx_buffer + 3, rx_buffer + 1, param_size - 4);

            break;
        case REG_RD_REQ:
            printf("R Req received:  ADDR=0x%08X \n", ntohl(rx_buffer[1]));

            //response buffer
            tx_buffer[2] = htonl(REG_RD_RSP);

            //read
            tx_size = 4 + reg_emb_read(tx_buffer + 3, rx_buffer + 1, param_size - 4);

            break;
        default:
            printf("Unknown Req received: %d [%x %x %x %x]\n", param_size,
                    rx_buffer [0], rx_buffer [1], rx_buffer [2], rx_buffer [3]);
            //response buffer
            tx_buffer[2] = REG_UNK_RSP;

            tx_size = 4;
            break;
    }

    // send cfm message
    send_msg(fd, tx_size, (char*) tx_buffer);

}

void client_hdl(int fd)
{
    char buffer[MAX_BUFFER_SIZE] __attribute__ ((aligned (4)));

    // Must manage at read:
    // - multiple concatenated messages
    // - last message not fully received

    reset_read:;

    // start receiving msgs at the beginning of the buffer
    char *buffer_ptr = buffer;
    int buffer_size  = sizeof(buffer);

    char *msg_ptr    = buffer;
    int msg_size     = PK_HEADER_SIZE;

    for (;;)
    {
        // Enough room for this msg in the remaining part of the buffer?
        if (buffer + sizeof(buffer) < msg_ptr + msg_size)
        {
            int msg_size_read = buffer_ptr - msg_ptr;

            // no, move the part already received at the beginning
            memmove(buffer, msg_ptr, msg_size_read);
            // and update accordingly pointers
            buffer_ptr  = buffer + msg_size_read;
            buffer_size = sizeof(buffer) - msg_size_read;
            msg_ptr     = buffer;
        }

        // wait some bytes from the client
        int read_size = read(fd, buffer_ptr, buffer_size);

        // check error and EOF (socket closed)
        error_check(0 <= read_size, "ERROR: read socket");
        if (!read_size) break;

        buffer_ptr  += read_size;
        buffer_size -= read_size;

        // in this loop we process each message
        for(;;)
        {
            // if msg header already processed in a previous iteration, skip this
            if (msg_size == PK_HEADER_SIZE)
            {
                int msg_size_read = buffer_ptr - msg_ptr;
                assert (msg_size_read > 0);

                // full header must be received to be processed
                if (msg_size_read < PK_HEADER_SIZE) break;

                // check magic 'nxr' at beginning, if wrong reset
                if (ntohl(*(uint32_t*)msg_ptr) >> 8 != PK_ID_HEX) goto reset_read;

                // retrieve msg size (including header)
                msg_size = ntohl(*(uint32_t*)(msg_ptr+PK_ID_SIZE)) + PK_HEADER_SIZE;
                //printf("Msg size: %d\n", msg_size);
                // msg size must be non null, fit into buffer and be word-aligned
                assert((PK_HEADER_SIZE < msg_size) && (msg_size < sizeof(buffer)) && !(msg_size&3));
            }

            // msg fully received?
            if (buffer_ptr < msg_ptr + msg_size) break;

            // either a R or a W register
            //printf("Msg : [%x:%x:%x:%x:%x:%x:%x:%x]\n", msg_ptr[8], msg_ptr[9], msg_ptr[10], msg_ptr[11],
            //        msg_ptr[12], msg_ptr[13], msg_ptr[14], msg_ptr[15]);
            reg_rdwr_hdl(fd, msg_size - PK_HEADER_SIZE, msg_ptr + PK_HEADER_SIZE);

            // process eventual following message
            msg_ptr += msg_size;
            msg_size = PK_HEADER_SIZE;

            // if no (parital) msg follows, reset read ptr at the buffer start (optim)
            if (msg_ptr == buffer_ptr) goto reset_read;
        }
    }
}

void socket_init(int port)
{
    // create socket instance
    nxreg_env.listen_fd  = socket(PF_INET, SOCK_STREAM, 0);
    error_check(0 <= nxreg_env.listen_fd, "ERROR:Cannot create server socket");

    // allow fast reuse
    int yes = 1;
    setsockopt(nxreg_env.listen_fd, SOL_SOCKET, SO_REUSEADDR, (char *)&yes, sizeof(yes));

    // set socket parameters
    nxreg_env.sockaddr.sin_family      = AF_INET;
    nxreg_env.sockaddr.sin_port        = htons(port);
    nxreg_env.sockaddr.sin_addr.s_addr = 0;

    //bind
    int ret = bind(nxreg_env.listen_fd,
                  (struct sockaddr *)&nxreg_env.sockaddr,
                  sizeof(nxreg_env.sockaddr));
    error_check(0 <= ret,"ERROR:Bind socket");

    // start listening on the socket
    ret = listen(nxreg_env.listen_fd, 0);
    error_check(0 <= ret, "ERROR:Listen on server socket");

    // by default configure the server socket in non blocking
    //fcntl(this->m_serverfd, F_SETFL, O_NONBLOCK);

    // by default configure the client in blocking mode
    //fcntl(this->m_clientfd, F_SETFL, O_NONBLOCK);
}

int main(int argc, char *argv[])
{
    int file_mem;

    //create server socket and wait for client (from nx_reg_gui)
    socket_init(NX_REG_GUI_PORT);

    for(;;)
    {
        printf("*** Waiting for NX REG GUI client...\n");

        // accept Test Agent connection
        socklen_t len = sizeof(nxreg_env.sockaddr);
        int client_fd = accept(nxreg_env.listen_fd, (struct sockaddr *)&nxreg_env.sockaddr, &len);
        assert(client_fd >= 0);

        //fork process
        pid_t pid = vfork();
        error_check(pid >= 0, "ERROR: VFork failed");
        if (pid)
        {
            // parent
            close(client_fd);
        }
        else
        {
            //child

            // set short latency
            int yes = 1;
            setsockopt(client_fd, IPPROTO_TCP, TCP_NODELAY, (char *)&yes, sizeof(yes));

            printf("*** Client connected\n");

            // open mem device
            file_mem = open("/dev/mem",O_SYNC|O_RDWR);
            if(!file_mem)
            {
                printf("ERROR: Can't open /dev/mem !\n");
                exit(-1);
            }

            // map system physical addresses for BAR0 and BAR1
            nxreg_env.bar0_base = mmap(0x0, 0x01000000, PROT_READ|PROT_WRITE, MAP_SHARED,
                                       file_mem,(off_t)BAR0_BASE);

            if(nxreg_env.bar0_base == (void*)-1)
            {
                printf("ERROR:mmap can't map BAR0 physical address !\n");
                exit(-1);
            }
            printf("Mapped BAR0_BASE 0x%08X to 0x%08X \n",BAR0_BASE, nxreg_env.bar0_base);

            nxreg_env.bar1_base = mmap(0x0, 0x01000000, PROT_READ|PROT_WRITE, MAP_SHARED,
                                       file_mem,(off_t)BAR1_BASE);

            if(nxreg_env.bar1_base == (void*)-1)
            {
                printf("ERROR:mmap can't map BAR1 physical address !\n");
                exit(-1);
            }
            printf("Mapped BAR1_BASE 0x%08X to 0x%08X \n",BAR1_BASE, nxreg_env.bar1_base);

            // process input from client
            client_hdl(client_fd);

            printf("*** Client quit\n");
            _exit(0);
        }
    }

    close(nxreg_env.listen_fd);
    close(file_mem);

    return 0;
}
///@}
