/**
 *****************************************************************************************
 * @file cmd.c
 *
 * @brief Localization of commands implementation
 *
 * Copyright (C) RivieraWaves 2011-2012
 *
 *****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include <errno.h>
#include "cmd.h"
#include "rwnx_testmode.h" // must remain synchronized with the rwnx driver

/*
 * FUNCTION DEFINITIONS
 *****************************************************************************************
 */

/*
 *****************************************************************************************
 * @brief Callback for Read commands: prints out the value of the register the user asked
 *        to read.
 * @param msg    message retrieved from the driver
 * @param arg    useless here, just for legacy reason due to the type of the callback
 *****************************************************************************************
 */
static int read_cb(struct nl_msg *msg, void *arg)
{
    struct nlattr *tb[NL80211_ATTR_MAX + 1];
    struct genlmsghdr *gnlh = nlmsg_data(nlmsg_hdr(msg));
    struct nlattr *td[RWNX_TM_ATTR_MAX + 1];

    nla_parse(tb, NL80211_ATTR_MAX, genlmsg_attrdata(gnlh, 0),
              genlmsg_attrlen(gnlh, 0), NULL);

    /* Check if there are data sent back from the LMAC */
    if (!tb[NL80211_ATTR_TESTDATA] || !tb[NL80211_ATTR_WIPHY]) {
        printf("Error: no data returned\n");
        return NL_SKIP;
    }

    /* Retrieve the value of the register */
    nla_parse(td, RWNX_TM_ATTR_MAX, nla_data(tb[NL80211_ATTR_TESTDATA]),
              nla_len(tb[NL80211_ATTR_TESTDATA]), NULL);

    if (!td[RWNX_TM_ATTR_REG_VALUE32]) {
        printf("Error: no valid register value returned\n");
        return NL_SKIP;
    }

    printf("Reg value = 0x%-.8x\n", nla_get_u32(td[RWNX_TM_ATTR_REG_VALUE32]));

    return NL_SKIP;
}

/*
 *****************************************************************************************
 * @brief Function to build the message to set the Debug filters: either the allowed
 *        Modules or the allowed Severity, depending on the 'type' parameter.
 * @param msg    message to be sent to the driver
 * @param argc   usual parameter counter
 * @param argv   usual parameter values
 * @param type   filter type to be set: Module or Severity
 *****************************************************************************************
 */
static int do_setdbgfilter(struct nl_msg *msg, int argc, char **argv, int type)
{
    unsigned int filter;

    if (argc != 1)
        return 3;

    //TODO: check the base
    filter = strtoul(argv[0],(char**)NULL,16)&0xffffffff;

    NLA_PUT_U32(msg, RWNX_TM_ATTR_COMMAND, type);
    NLA_PUT_U32(msg, RWNX_TM_ATTR_REG_FILTER, filter);

    return 0;

nla_put_failure:
    return -ENOBUFS;
}

/*
 *****************************************************************************************
 * @brief Function to build the message to write Register values, but not through the LMAC
 *        message handling but accessing registers through the PCI BAR windows.
 *        This must be used when LMAC FW is stuck.
 *
 * @param msg    message to be sent to the driver
 * @param argc   usual parameter counter
 * @param argv   usual parameter values
 *****************************************************************************************
 */
static int do_debugwrite(struct nl_msg *msg, int argc, char **argv)
{
    unsigned int addr, value;

    if (argc != 2)
        return 3;

    //printf("### do_debugwrite, argc = %d\n", argc);
    addr = strtoul(argv[0],(char**)NULL,16)&0xfffffffc;
    value = strtoul(argv[1],(char**)NULL,16)&0xffffffff;

    NLA_PUT_U32(msg, RWNX_TM_ATTR_COMMAND, RWNX_TM_CMD_APP2DEV_REG_WRITE_DBG);
    NLA_PUT_U32(msg, RWNX_TM_ATTR_REG_OFFSET, addr);
    NLA_PUT_U32(msg, RWNX_TM_ATTR_REG_VALUE32, value);

    return 0;

 nla_put_failure:
    return -ENOBUFS;
}

/*
 *****************************************************************************************
 * @brief Function to build the message to write Register values.
 *
 * @param msg    message to be sent to the driver
 * @param argc   usual parameter counter
 * @param argv   usual parameter values
 *****************************************************************************************
 */
static int do_write(struct nl_msg *msg, int argc, char **argv)
{
    unsigned int addr, value;

    if (argc != 2)
        return 3;

    //printf("### do_write, argc = %d\n", argc);
    addr = strtoul(argv[0],(char**)NULL,16)&0xfffffffc;
    value = strtoul(argv[1],(char**)NULL,16)&0xffffffff;

    NLA_PUT_U32(msg, RWNX_TM_ATTR_COMMAND, RWNX_TM_CMD_APP2DEV_REG_WRITE);
    NLA_PUT_U32(msg, RWNX_TM_ATTR_REG_OFFSET, addr);
    NLA_PUT_U32(msg, RWNX_TM_ATTR_REG_VALUE32, value);

    return 0;

 nla_put_failure:
    return -ENOBUFS;
}

/*
 *****************************************************************************************
 * @brief Function to build the message to read Register values, but not through the LMAC
 *        message handling but accessing registers through the PCI BAR windows.
 *        This must be used when LMAC FW is stuck.
 *
 * @param cb     callback used to print out the register value
 * @param msg    message to be sent to the driver
 * @param argc   usual parameter counter
 * @param argv   usual parameter values
 *****************************************************************************************
 */
static int do_debugread(struct nl_cb *cb, struct nl_msg *msg, int argc, char **argv)
{
    unsigned int addr;

    if (argc != 1)
        return 3;

    addr = strtoul(argv[0],(char**)NULL,16)&0xfffffffc;

    NLA_PUT_U32(msg, RWNX_TM_ATTR_COMMAND, RWNX_TM_CMD_APP2DEV_REG_READ_DBG);
    NLA_PUT_U32(msg, RWNX_TM_ATTR_REG_OFFSET, addr);

    nl_cb_set(cb, NL_CB_VALID, NL_CB_CUSTOM, read_cb, NULL);

    return 0;

nla_put_failure:
    return -ENOBUFS;
}

/*
 *****************************************************************************************
 * @brief Function to build the message to read Register values.
 *
 * @param cb     callback used to print out the register value
 * @param msg    message to be sent to the driver
 * @param argc   usual parameter counter
 * @param argv   usual parameter values
 *****************************************************************************************
 */
static int do_read(struct nl_cb *cb, struct nl_msg *msg, int argc, char **argv)
{
    unsigned int addr;

    if (argc != 1)
        return 3;

    addr = strtoul(argv[0],(char**)NULL,16)&0xfffffffc;

    //printf("### do_read, argc = %d\n", argc);
    //printf("### do_read, argv[0] = %s\n", argv[0]);
    //printf("### do_read, addr = %-.8x\n", addr);

    NLA_PUT_U32(msg, RWNX_TM_ATTR_COMMAND, RWNX_TM_CMD_APP2DEV_REG_READ);
    NLA_PUT_U32(msg, RWNX_TM_ATTR_REG_OFFSET, addr);

    nl_cb_set(cb, NL_CB_VALID, NL_CB_CUSTOM, read_cb, NULL);

    return 0;

nla_put_failure:
    return -ENOBUFS;
}

/*
 *****************************************************************************************
 * @brief Function to parse the kind of commands passed by the user
 *
 * @param cb     callback to be given to lower functions if needed
 * @param msg    message to be sent to the driver
 * @param argc   usual parameter counter
 * @param argv   usual parameter values
 *****************************************************************************************
 */
int do_commands(struct nl_cb *cb, struct nl_msg *msg, int argc, char **argv)
{

    if (argc <= 0)
        return 3;

    if (strcmp(*argv, "rr") == 0)
    {
        /* Read Register command */
        return do_read(cb, msg, argc - 1, argv + 1);
    }
    else if (strcmp(*argv, "wr") == 0)
    {
        /* Write Register command */
        return do_write(msg, argc - 1, argv + 1);
    }
    else if (strcmp(*argv, "smf") == 0)
    {
        if (strcmp(*(argv+1), "help") == 0)
        {
            printf("### Modules bit values are:\n");
            printf("0...DBG_MOD_IDX_KE\n");
            printf("1...DBG_MOD_IDX_DBG\n");
            printf("2...DBG_MOD_IDX_IPC\n");
            printf("3...DBG_MOD_IDX_DMA\n");
            printf("4...DBG_MOD_IDX_MM\n");
            printf("5...DBG_MOD_IDX_TX\n");
            printf("6...DBG_MOD_IDX_RX\n");
            printf("7...DBG_MOD_IDX_PHY\n");
            printf("8...DBG_MOD_IDX_MAX\n\n");

            return HELP;
        }
        /* Set Debug Module filter command */
        return do_setdbgfilter(msg, argc - 1, argv + 1, RWNX_TM_CMD_APP2DEV_SET_DBGMODFILTER);
    }
    else if (strcmp(*argv, "ssf") == 0)
    {
        if (strcmp(*(argv+1), "help") == 0)
        {
            printf("### Severity values are:\n");
            printf("0x00...DBG_SEV_IDX_NONE\n");
            printf("0x01...DBG_SEV_IDX_CRT and above\n");
            printf("0x02...DBG_SEV_IDX_ERR and above\n");
            printf("0x03...DBG_SEV_IDX_WRN and above\n");
            printf("0x04...DBG_SEV_IDX_INF and above\n");
            printf("0x05...DBG_SEV_IDX_VRB and above\n\n");

            return HELP;
        }

        /* Set Debug Severity filter command */
        return do_setdbgfilter(msg, argc - 1, argv + 1, RWNX_TM_CMD_APP2DEV_SET_DBGSEVFILTER);
    }
    else if (strcmp(*argv, "drr") == 0)
    {
        /* Debug Read Register command (not sent to the LMAC) */
        return do_debugread(cb, msg, argc - 1, argv + 1);
    }
    else if (strcmp(*argv, "dwr") == 0)
    {
        /* Debug Write Register command (not sent to the LMAC) */
        return do_debugwrite(msg, argc - 1, argv + 1);
    }


    return 1;
}
