/**
 *****************************************************************************************
 * @file rw_comm.c
 *
 * @brief Main file for rw application, sending DBG commands to LMAC layer
 *
 * Copyright (C) RivieraWaves 2011-2012
 *
 *****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include <errno.h>
#include <net/if.h>

#include "cmd.h"

/*
 * DEFINES
 *****************************************************************************************
 */
struct nl80211_state {
    struct nl_sock *nl_sock;
    struct nl_cache *nl_cache;
    struct genl_family *nl80211;
};

/*
 * FUNCTION DEFINITIONS
 *****************************************************************************************
 */
static int nl80211_init(struct nl80211_state *state)
{
    int err;

    state->nl_sock = nl_socket_alloc();
    if (!state->nl_sock) {
        fprintf(stderr, "Error: Failed to allocate netlink socket.\n");
        return -ENOMEM;
    }

    if (genl_connect(state->nl_sock)) {
        fprintf(stderr, "Error: Failed to connect to generic netlink.\n");
        err = -ENOLINK;
        goto out_handle_destroy;
    }

    if (genl_ctrl_alloc_cache(state->nl_sock, &state->nl_cache)) {
        fprintf(stderr, "Error: Failed to allocate generic netlink cache.\n");
        err = -ENOMEM;
        goto out_handle_destroy;
    }

    state->nl80211 = genl_ctrl_search_by_name(state->nl_cache, "nl80211");
    if (!state->nl80211) {
        fprintf(stderr, "Error: nl80211 not found.\n");
        err = -ENOENT;
        goto out_cache_free;
    }

    /*
     * Enable peek mode so drivers can send large amounts
     * of data in blobs without problems. */
    nl_socket_enable_msg_peek(state->nl_sock);

    return 0;

out_cache_free:
    nl_cache_free(state->nl_cache);
out_handle_destroy:
    nl_socket_free(state->nl_sock);
    return err;
}

static void nl80211_cleanup(struct nl80211_state *state)
{
    genl_family_put(state->nl80211);
    nl_cache_free(state->nl_cache);
    nl_socket_free(state->nl_sock);
}

static int error_handler(struct sockaddr_nl *nla, struct nlmsgerr *err,
                        void *arg)
{
    int *ret = arg;
    *ret = err->error;
    printf("### error_handler, error = %d\n", err->error);
    return NL_STOP;
}

static int finish_handler(struct nl_msg *msg, void *arg)
{
    int *ret = arg;
    *ret = 0;
    //printf("### finish_handler\n");
    return NL_SKIP;
}

static int ack_handler(struct nl_msg *msg, void *arg)
{
    int *ret = arg;
    *ret = 0;
    //printf("### ack_handler\n");
    return NL_STOP;
}

/*
 *****************************************************************************************
 * @brief Real handling of the commands, takes care of all the netlink stuff and calls
 *        the command parsing function.
 *
 * @param state  netlink state
 * @param argc   usual parameter counter
 * @param argv   usual parameter values
 *****************************************************************************************
 */
static int handle(struct nl80211_state *state, int argc, char **argv)
{
    struct nl_cb *cb;
    struct nl_msg *msg;
    struct nlattr *nest;
    int devidx = 0;
    int err;

    if (!argc) {
        printf("no dev/phy given\n");
        return 1;
    }

    msg = nlmsg_alloc();
    if (!msg) {
        fprintf(stderr, "failed to allocate netlink message\n");
        return 2;
    }

    cb = nl_cb_alloc(NL_CB_DEFAULT);
    if (!cb) {
        fprintf(stderr, "failed to allocate netlink callbacks\n");
        err = 2;
        goto out_free_msg;
    }

    genlmsg_put(msg, 0, 0, genl_family_get_id(state->nl80211), 0,
                0, NL80211_CMD_TESTMODE, 0);

    devidx = if_nametoindex(*argv);
    if (devidx) {
        //printf("Device found: %d\n", devidx);
        NLA_PUT_U32(msg, NL80211_ATTR_IFINDEX, devidx);
    }
    else
    {
        printf("No device found!\n");
        goto out;
    }

    argc--;
    argv++;

    nest = nla_nest_start(msg, NL80211_ATTR_TESTDATA);
    if (!nest)
        return 4;

    err = do_commands(cb, msg, argc, argv);

    nla_nest_end(msg, nest);
    if (err)
        goto out;

    err = nl_send_auto_complete(state->nl_sock, msg);
    if (err < 0)
        goto out;

    err = 1;

    nl_cb_err(cb, NL_CB_CUSTOM, error_handler, &err);
    nl_cb_set(cb, NL_CB_FINISH, NL_CB_CUSTOM, finish_handler, &err);
    nl_cb_set(cb, NL_CB_ACK, NL_CB_CUSTOM, ack_handler, &err);

    while (err > 0){
        nl_recvmsgs(state->nl_sock, cb);
        }
out:
    nl_cb_put(cb);
out_free_msg:
    nlmsg_free(msg);
    //printf("err value when leaving: 0x%x\n", err);
    return err;
nla_put_failure:
    fprintf(stderr, "building message failed\n");
    return 2;
}

/*
 *****************************************************************************************
 * @brief Main entry point of the application.
 *
 * @param argc   usual parameter counter
 * @param argv   usual parameter values
 *****************************************************************************************
 */
int main(int argc, char **argv)
{
    struct nl80211_state state;
    int err;

    /* strip off self */
    argc--;
    argv++;

    err = nl80211_init(&state);
    if (err)
        return 1;

    err = handle(&state, argc, argv);

    if ((err > 0) && (err!=HELP))
        printf("error! 0x%x\n", err);
    else if (err < 0)
        printf("error: %d (%s)\n", err, strerror(-err));

    nl80211_cleanup(&state);

    return err;
}
