########################################################################################
#
# scutils.py
#
# Utility library for all SConscripts
#
# Copyright (C) RivieraWaves 2011-2019
#
# $Id$
#
#########################################################################################

import sys
import os
import os.path
import shutil
from os.path import join
import datetime
import subprocess
import platform
import struct

#-----------------------------------------------------------
# Scons include + utility function
#-----------------------------------------------------------

from SCons.Script import *

def ext_path(scons_path):
    """Convert an absolute SCons path (with eventual #)
       to an normal path for external tools"""

    if scons_path[0] == '#':
        if len(scons_path) > 1 and scons_path[1] == '/':
            scons_path = scons_path[2:]
        else:
            scons_path = scons_path[1:]

        scons_path = join(Dir('#').abspath, scons_path)
    return os.path.normpath(scons_path)

def diff_path(path_base, path_ext):
    """Get the relative path from path_ext to path_base.
       work only if ext is included in base"""

    diff = ''
    path_base   = os.path.abspath(ext_path(path_base))
    path_ext    = os.path.abspath(ext_path(path_ext))

    while path_base != path_ext:
        path_ext, path_last = os.path.split(path_ext)
        if not path_last: raise #not included!
        diff     = join (path_last, diff)

    return os.path.normpath(diff)

#def cyg2win(path):
#    if path[:10] == '/cygdrive/':
#        path = path[10] + ':' + path[11:]
#    elif path[0] == '/':
#        print "ERROR ", path
#    else:
#        print "relative ", path
#    return path.replace('/','\\')


#-----------------------------------------------------------
# Local includes
#-----------------------------------------------------------
def rvds_find():
    rvds = 'rvds41'
    try:
        rvds_path = os.environ['ARMCC41BIN']
    except KeyError:
        rvds = 'rvds31'
        try:
            rvds_path = os.environ['RVCT31BIN']
        except KeyError:
            print "Neither RVDS 3.1 nor 4.1 is installed on this computer"
            raise
    # First search in the SCons path and then the OS path:
    return rvds

def env_create(arch):
    assert(arch in ('aps3', 'arm', 'tl4xx', 'cortex', 'ceva-x', 'risc-v'))

    def_optim = '2'

    # Get build tool from command line or default
    if (arch == 'aps3'):
        defaulttool = ARGUMENTS.get('BT','aps-gcc')
        buildtool   = ARGUMENTS.get('BT_APS', defaulttool).lower()
        def_optim   = 's'
    elif (arch == 'arm'):
        defaulttool = ARGUMENTS.get('BT','armgcc_4_8')
        buildtool   = ARGUMENTS.get('BT_ARM', defaulttool).lower()
    elif (arch == 'cortex'):
        defaulttool = ARGUMENTS.get('BT','armgcc_4_8')
        buildtool   = ARGUMENTS.get('BT_ARM', defaulttool).lower()
    elif (arch == 'tl4xx'):
        defaulttool = ARGUMENTS.get('BT','tl4')
        buildtool   = ARGUMENTS.get('BT_TL4', defaulttool).lower()
    elif (arch == 'ceva-x'):
        defaulttool = ARGUMENTS.get('BT','ceva-x-cc')
        buildtool   = ARGUMENTS.get('BT_X1', defaulttool).lower()
    elif (arch == 'risc-v'):
        defaulttool = ARGUMENTS.get('BT','riscv32')
        buildtool   = ARGUMENTS.get('BT_RISCV', defaulttool).lower()
        def_optim   = 's'
    else:
        assert(0)

    # sanity checks
    assert(buildtool in ('rvds', 'gnuarm', 'armgcc_4_8', 'aps-gcc', 'tl4', 'ceva-x-cc', 'riscv32'))

    if (arch == 'aps3'):
        cpu = 'aps'
    elif (arch == 'arm'):
        # use arm7 by default
        cpu  = ARGUMENTS.get('CPU', 'arm7')
        assert(cpu in ('arm7', 'arm9'))
        if cpu.lower() == 'arm7' or cpu == '':
            cpu = 'arm7tdmi-s'
        elif env['CPU'].lower() == 'arm9':
            cpu = 'arm926ej-s'
    elif (arch == 'cortex'):
        # use cortex-m0 by default
        cpu  = ARGUMENTS.get('CPU', 'cortex-m0')
        assert(cpu in ('cortex-m0', 'cortex-m4'))
    elif (arch == 'tl4xx'):
        cpu  = ARGUMENTS.get('CPU', 'tl410')
        assert(cpu in ('tl410', 'tl411', 'tl420', 'tl421'))
    elif (arch == 'ceva-x'):
        cpu  = 'cevax1'
    elif (arch == 'risc-v'):
        cpu = ARGUMENTS.get('CPU', 'zeroriscy')
        assert(cpu in ('ri5cy', 'zeroriscy'))

    # create environment
    if (buildtool == 'aps-gcc'):
        env = Environment(tools = ['aps-gcc'])
    elif (buildtool == 'rvds'):
        tool = scutils.rvds_find()
        env = Environment(tools = [tool])
    elif (buildtool == 'gnuarm'):
        env = Environment(tools = ['gnuarm'])
    elif (buildtool == 'armgcc_4_8'):
        env = Environment(tools = ['armgcc_4_8'], CPU = cpu)
    elif (buildtool == 'tl4'):
        env = Environment(tools = ['tl4'])
    elif (buildtool == 'ceva-x-cc'):
        env = Environment(tools = ['ceva-x-cc'])
    elif (buildtool == 'riscv32'):
        env = Environment(tools = ['riscv32'], CPU = cpu)

    env.AppendENVPath('PATH', os.environ['PATH'])

    # save arch and buildtool in environment
    env['ARCH']      = arch
    env['CPU']       = cpu
    env['BUILDTOOL'] = buildtool

    # optimization level: default depends on ARCH
    env['OPTIM']     = ARGUMENTS.get('O', def_optim)

    return env

def env_init():
    #-----------------------------------------------------------
    # Default initialization for environments
    #-----------------------------------------------------------

    ################# General Configuration ####################

    # Used architecture (available values: aps3, tl4xx, arm, ceva-x, risc-v), here is mentioned the processor itself
    env = env_create( ARGUMENTS.get('ARCH','risc-v').lower() )
    # Product that is built (lmac, fmac or fhost)
    env['PRODUCT']  = ARGUMENTS.get('PRODUCT', 'lmac').lower()
    # static analysis tool (all, cppcheck, flawfinder, off)
    env['ANALYSIS'] =  ARGUMENTS.get('ANALYSIS', 'off')
    # cppcheck check all configuration (on, off)
    env['CPPCHECK_ALL_CFG'] = ARGUMENTS.get('CPPCHECK_ALL_CFG', 'off')
    # Used platform (available values: virtex6, virtex7, tlm)
    env['PLATFORM'] = ARGUMENTS.get('PLATFORM', 'virtex7').lower()
    # verbose level: 0 by default (0 or 1 supported)
    env['VERBOSE']  = int(ARGUMENTS.get('V', 0))
    # save temporary files: 0 by default (0 or 1 supported)
    env['SAVETMP']  = int(ARGUMENTS.get('SAVETMP', 0))
    # only for embedded compilation: profiling enabled or not
    env['PROF']     = ARGUMENTS.get('PROF', 'on').lower()
    # only for embedded compilation: system statistics enabled or not
    env['STATS']    = ARGUMENTS.get('STATS', 'on').lower()
    # only for RVDS: remarks enabled (0 or 1 supported)
    env['REM']      = int(ARGUMENTS.get('REM', 0))
    # maximum number of virtual interfaces supported (must be a power of 2, min 1, max 4)
    env['VIF']      = ARGUMENTS.get('VIF', '4')
    # maximum number of peer devices supported
    env['STA']      = ARGUMENTS.get('STA', '10')
    # Debug level (0: no debug info, 1: limited debug info, 2: full debug info)
    env['DBG']      = ARGUMENTS.get('DBG', '2')
    # Debug dump (on, off)
    env['DBGDUMP']  = ARGUMENTS.get('DBGDUMP', 'on').lower()
    # Debug key RAM dump (on, off)
    env['DBGDUMPKEY']  = ARGUMENTS.get('DBGDUMPKEY', 'on').lower()
    # Stack Overflow check (on, off)
    env['STACK_CHECK'] = ARGUMENTS.get('STACK_CHECK', 'off').lower()
    # Trace Buffer
    env['TRACE']    = ARGUMENTS.get('TRACE', 'on').lower()
    # MAC version (v10, v21)
    env['MAC']      = ARGUMENTS.get('MAC', 'v10')
    # Modem version (v10, v11, v20, v21, v22 or v30)
    env['MDM']      = ARGUMENTS.get('MDM', 'v22')
    # IPC version (v10 or v11)
    env['IPC']      = ARGUMENTS.get('IPC', 'v11')
    # Type of PHY (trident, karst, elma), for karst use MDM v20 or higher
    env['PHY']      = ARGUMENTS.get('PHY', 'karst').lower()
    # Number of descriptors in BK TX queue (must be a power of 2, min 4, max 64)
    env['TXDESC0']  = ARGUMENTS.get('TXDESC0', '8').lower()
    # Number of descriptors in BE TX queue (must be a power of 2, min 4, max 64)
    env['TXDESC1']  = ARGUMENTS.get('TXDESC1', '64').lower()
    # Number of descriptors in VI TX queue (must be a power of 2, min 4, max 64)
    env['TXDESC2']  = ARGUMENTS.get('TXDESC2', '64').lower()
    # Number of descriptors in VO TX queue (must be a power of 2, min 4, max 64)
    env['TXDESC3']  = ARGUMENTS.get('TXDESC3', '32').lower()
    # Number of descriptors in BCN TX queue (must be a power of 2, min 4, max 64)
    env['TXDESC4']  = ARGUMENTS.get('TXDESC4', '8').lower()
    # POWERSAVE Support (on, off)
    env['PS']       = ARGUMENTS.get('PS', 'on').lower()
    # UAPSD Support (on, off)
    env['UAPSD']    = ARGUMENTS.get('UAPSD', 'on').lower()
    # DPSM Support (on, off)
    env['DPSM']     = ARGUMENTS.get('DPSM', 'on').lower()
    # Radar detection Support (on, off)
    env['RADAR']    = ARGUMENTS.get('RADAR', 'off').lower()
    # Unsupported HT Frame Logging Support (on, off)
    env['UF']       = ARGUMENTS.get('UF', 'on').lower()
    # Monitor + Data interface Support (on, off)
    env['MON_DATA'] = ARGUMENTS.get('MON_DATA', 'off').lower()
    # AMSDU Support (on, off)
    env['AMSDU']    = ARGUMENTS.get('AMSDU', 'on').lower()
    # Recovery mechanism (on, off)
    env['REC']      = ARGUMENTS.get('REC', 'on').lower()
    # Beaconing modes supported or not
    env['BCN']      = ARGUMENTS.get('BCN', 'on').lower()
    # A-MPDU transmission supported or not
    env['AGG']      = ARGUMENTS.get('AGG', 'on').lower()
    # Maximum A-MSDU size supported in reception (4K, 8K or 12K)
    env['AMSDURX']  = ARGUMENTS.get('AMSDURX', '12K').lower()
    # Minimum spacing between two MPDUs inside an A-MPDU that we transmit (4us-32us)
    env['SPC']      = ARGUMENTS.get('SPC', '16').lower()
    # WAPI Support (on, off)
    env['WAPI']     = ARGUMENTS.get('WAPI', 'off').lower()
    # BFMEE support (on, off)
    env['BFMEE']    = ARGUMENTS.get('BFMEE', 'on').lower()
    # A-MPDU length adaptation for BW support (on, off)
    env['BWLEN']    = ARGUMENTS.get('BWLEN', 'on').lower()
    # Compilation for a HW supporting key RAM configuration (on, off)
    env['KEYCFG']   = ARGUMENTS.get('KEYCFG', 'on').lower()
    # TDLS support (on, off)
    env['TDLS']     = ARGUMENTS.get('TDLS', 'on')

    #================ Beamformer Configuration =================
    # BFMER support (on, off)
    env['BFMER']    = ARGUMENTS.get('BFMER', 'on').lower()

    #==================== Mesh Configuration ===================
    # Number of supported Mesh Point Interfaces
    env['MESH_VIF']   = ARGUMENTS.get('MESH_VIF', '0')
    # Number of supported Mesh Links (shared between the Mesh VIFs)
    env['MESH_LINK']  = ARGUMENTS.get('MESH_LINK', '2')
    # Number of supported Mesh Paths
    env['MESH_PATH']  = ARGUMENTS.get('MESH_PATH', '5')
    # Number of supported Mesh Proxy Information
    env['MESH_PROXY'] = ARGUMENTS.get('MESH_PROXY', '10')

    #==================== P2P Configuration ====================
    # Number of P2P links that can be created in parallel (0 (P2P Disabled), 1, 2, ...)
    env['P2P']      = ARGUMENTS.get('P2P', '2').lower()
    # P2P GO role support (on, off)
    env['P2P_GO']   = ARGUMENTS.get('P2P_GO', 'on').lower()

    #=========== MU-MIMO Transmitter Configuration =============
    # Maximum number of users supported (1-4) - A value greater than 1 enables MU-MIMO support
    # Valid only if BFMER is set to on
    env['MUCNT']    = ARGUMENTS.get('MUCNT', '1').lower()

    #=========== MU-MIMO receiver Configuration =============
    # MU-MIMO RX support  (on, off) - Valid only if BFMEE is set to on
    env['MURX']     = ARGUMENTS.get('MURX', 'on').lower()

    #=========== HSU Configuration =============
    # HSU support (0=Don't use HSU, software only, 1=use HSU if available, software otherwise, 2=Only use HSU)
    env['HSU']      = ARGUMENTS.get('HSU', '1')

    ################ LMAC-only Configuration ###################

    # Connection Monitoring Support (on, off)
    env['CMON']     = ARGUMENTS.get('CMON', 'on').lower()
    # Multi-role (AP+STA, STA+STA) Support (on, off)
    env['MROLE']    = ARGUMENTS.get('MROLE', 'on').lower()
    # Scanning in LMAC Support (on, off)
    env['HWSCAN']   = ARGUMENTS.get('HWSCAN', 'on').lower()
    # Autonomous beacon transmission in LMAC Support (on, off)
    env['AUTOBCN']  = ARGUMENTS.get('AUTOBCN', 'on').lower()

    ################ FMAC-only Configuration ###################

    # VHT support in FullMAC (on, off)
    env['VHT']      = ARGUMENTS.get('VHT', 'on')
    # HE support in FullMAC (on, off)
    env['HE']       = ARGUMENTS.get('HE', 'on')
    # Maximum number of RX block ack agreements we can handle in parallel (0 to disable BA in RX)
    env['BARX']     = ARGUMENTS.get('BARX', '5').lower()
    # Number of buffers per reordering instance (minimum 4)
    env['REORDBUF'] = ARGUMENTS.get('REORDBUF', '64').lower()
    # Maximum number of TX block ack agreements we can handle in parallel (valid only if AGG is on)
    env['BATX']     = ARGUMENTS.get('BATX', '5').lower()
    # MFP support (on, off)
    env['MFP']      = ARGUMENTS.get('MFP', 'off')
    # Antenna Diversity support (on, off)
    env['ANT_DIV']  = ARGUMENTS.get('ANT_DIV', 'on')

    ################ FHOST/RTOS Configuration ###################
    def_rtos = 'none'
    def_nets = 'none'
    if env['PRODUCT'] == 'fhost':
        def_rtos = 'freertos'
        def_nets = 'lwip'

    # Used RTOS if any (available values: none, freertos)
    env['RTOS']          = ARGUMENTS.get('RTOS', def_rtos).lower()
    # Used Network Stack if any (available values: none, lwip)
    env['NETS']          = ARGUMENTS.get('NETS', def_nets).lower()
    # Main application for FHOST firmware (available value: ipc, example, none)
    env['FHOST_APP']     = ARGUMENTS.get('FHOST_APP', 'ipc').lower()
    # Traffic Generator for WiFi certification (on, off)
    env['TG']            = ARGUMENTS.get('TG', 'on').lower()
    # Monitor mode for packets sniffing (on, off)
    env['FHOST_MONITOR'] = ARGUMENTS.get('FHOST_MONITOR', 'on').lower()
    # Smartconfig (on, off)
    env['SMARTCONFIG']   = ARGUMENTS.get('SMARTCONFIG', 'on').lower()
    # Smartconf mode (available value: example)
    env['SMART_MODE'] = ARGUMENTS.get('SMART_MODE', 'example').lower()
    # External crypto library (available value: none, mbedtls)
    env['CRYPTO'] = ARGUMENTS.get('CRYPTO', 'none').lower()
    #-----------------------------------------------------------
    # General Configuration
    #-----------------------------------------------------------
    # Whether firmware should be built as a library or an executable
    env['LIBRARY'] = ARGUMENTS.get('LIB', 'off').lower()

    # Whether size of CPU memory must be updated in the linker script
    # Only used for toolchain using 'ld' linker (e.g. CPU_MEM=1024K)
    env['CPU_MEM'] = ARGUMENTS.get('CPU_MEM', '')

    if 'I' in ARGUMENTS:
        env['INST'] = ARGUMENTS['I']
    elif 'NX_INSTALL' in os.environ:
        env['INST'] = os.environ['NX_INSTALL']
    else:
        env['INST'] = ''

    runcov = int(ARGUMENTS.get('RUNCOV', 0))

    if runcov == 0:
        runcov = int(ARGUMENTS.get('COVRUN', 0))

    if runcov:
        env['COV'] = runcov
        env['RUN'] = runcov

    # Max command line length for the linker. 0 means disabled.
    # Work with gcc 4.2.x or above
    # Work with python-win32
    # Therefore, will fail with gcc-3 and python-cygwin
    if sys.platform == 'win32':
        env['MAXLINELENGTH'] = 8000
    elif sys.platform == 'cygwin':
        env['MAXLINELENGTH'] = 0
    else:
        env['MAXLINELENGTH'] = 8000

    # speed up?
    env.Decider('MD5')  #MD5-timestamp, MD5, timestamp-match
    #env.SourceCode('.', None)

    # set output strings
    if env['VERBOSE'] == 0:
        outstr_set(env, env['BUILDTOOL'])

    #if env['COV'] == 2:
    #    winbase = os.path.abspath('../..').replace('\\','\\\\\\\\')
    #    env['SED_SUBST_OBJ'] = "'/" + winbase + "/ s+\\\\\\\\+/+g'"
    #    env['SED_PATH'] = SCons.Util.WhereIs("sed.exe")

    return env

#-----------------------------------------------------------
# output strings
#-----------------------------------------------------------

OUTSTR = '%16s ${TARGET.file}'

OUTSTR_REG = '[py REG]'
OUTSTR_MIB = '[py MIB]'
OUTSTR_TTR = '[py TTR]'
OUTSTR_R_ELF = '[rvds ELF]'
OUTSTR_G_ELF = '[gnuarm ELF]'
OUTSTR_AGC_ELF = '[armgcc_4_8 ELF]'
OUTSTR_A_ELF = '[aps-gcc ELF]'
OUTSTR_A_HEX = '[aps-gcc HEX]'
OUTSTR_V_ELF = '[riscv_gcc ELF]'
OUTSTR_V_HEX = '[riscv_gcc HEX]'
OUTSTR_T_OUT = '[tl4 COFF]'
OUTSTR_T_SORT = '[tl4 SRT]'
OUTSTR_T_HEX = '[tl4 HEX]'
OUTSTR_X_OUT = '[ceva-x-cc COFF]'
OUTSTR_X_SORT = '[ceva-x-cc SRT]'
OUTSTR_X_HEX = '[ceva-x-cc HEX]'
OUTSTR_FLT = '[elf2bflt]'
OUTSTR_SEDX = '[sed exe]'
OUTSTR_SEDO = '[sed obj]'
OUTSTR_LC_I = '[lcov init]'
OUTSTR_LC_P = '[lcov proc]'
OUTSTR_LC_G = '[locv gen]'
OUTSTR_DICT = '[py DICT]'
OUTSTR_MAP = '[py MAP]'

def outstr_set(env, buildtool):

    cc_str  = OUTSTR % ('['+buildtool+'  CC]')
    ld_str  = OUTSTR % ('['+buildtool+'  LD]')
    as_str  = OUTSTR % ('['+buildtool+'  AS]')
    ar_str  = OUTSTR % ('['+buildtool+'  AR]')

    env.Replace(CCCOMSTR   = cc_str)
    env.Replace(LINKCOMSTR = ld_str)
    env.Replace(ASCOMSTR   = as_str)
    env.Replace(ASPPCOMSTR = as_str)
    env.Replace(ARCOMSTR = ar_str)


def outstr_gen(s, target, source, env):
    for t in target:
        print '%16s ' % (env['OUTSTR']) + t.name


def outstr_utest(s, target, source, env):
    for t in target:
        print '%16s ' % ('[' + source[0].name + ']') + t.name

def outstr_inst(s, target, source, env):
    for t in target:
        print '%16s ' % ('[install]') + t.abspath

def outstr_date(s, target, source, env):
    for t in target:
        print '%16s ' % ('[py date]') + t.name

def outstr_none(s, target, source, env):
    pass

sys.path.append(ext_path('#/../tools/reg_xls2h'))
import reg_xls2h
import re

def build_reg(target, source, env):
    xlsname = os.path.normpath(str(source[0]))
    hname = os.path.normpath(str(target[0]))
    match = re.match("_(.*)", os.path.basename(hname))
    if match != None:
        hname = os.path.join(os.path.dirname(hname), match.group(1))

    regaddr = env.get('REGADDR')
    sheetname = env.get('SHEETNAME')
    regaccess = env.get('REGACCESS')
    regprefix = env.get('REGPREFIX')

    #print("build reg: %s %s %s"%(xlsname, hname, regaddr))

    # parse the XLS name
    regs = reg_xls2h.reg_xls2h(xlsname, sheetname, regaccess, regprefix, verbose=False)
    regs.read_xls()

    regs.gen_header("", regaddr, hname, env.get("FORSIMU") == "True",
                    env.get('REGFORMAT') == "long")

def parse_register_mapping(reg_cfg_dir, env):
    """ Parse regmap.txt files in specific order, and save defined addresses in env """

    file_list = [ os.path.join(reg_cfg_dir, 'regmap.txt') ,
                  os.path.join(reg_cfg_dir, env['PHY'], 'regmap.txt') ,
                  os.path.join(reg_cfg_dir, env['ARCH'], 'regmap.txt') ,
                  os.path.join(reg_cfg_dir, env['PLATFORM'], 'regmap.txt'),
                  os.path.join(reg_cfg_dir, env['PLATFORM'], env['PLFVER'], 'regmap.txt') ]

    for f in file_list:
        if os.path.isfile(f):
            with open(f) as regmap:
                for line in regmap:
                    if not re.match("^\s*(#|$)", line):
                        res = line.split()
                        env[res[0]] = res[1]

def build_regs(env, reg_cfg_dir, reg_build_dir, reg_info_dir, reg_import_dir):
    """Setup the build rules for the firmware or simulation registers from the list"""
    forsimu = False

    if not os.path.isdir(reg_info_dir):
        os.makedirs(reg_info_dir)

    parse_register_mapping(reg_cfg_dir, env)

    elt_list = [ [os.path.join(reg_cfg_dir, env['PLATFORM'], 'reglist.txt'),
                  os.path.join(reg_build_dir, env['PLATFORM'], env['ARCH'])],
                 [os.path.join(reg_cfg_dir, env['ARCH'], 'reglist.txt'),
                  os.path.join(reg_build_dir, env['PLATFORM'], env['ARCH'])],
                 [os.path.join(reg_cfg_dir, env['PHY'], 'reglist.txt'),
                  os.path.join(reg_build_dir, env['PLATFORM'], env['ARCH'], env['PHY'])] ]

    reg_info = open(os.path.join(reg_info_dir, 'regmap_info.txt'), 'w')

    for elt in elt_list:
        reglist = elt[0]
        reg_build_dir = elt[1]

        if not os.path.isfile(reglist):
            continue

        if not os.path.isdir(reg_build_dir):
            os.makedirs(reg_build_dir)

        for line in file_list_read(env, reglist, reg_import_dir):

            # check that the format is met : (sheetname) 0xXXXXXX (reg_access) (reg_prefix) (short|long)
            match = re.match("(.*)\s+(0x[0-9a-fA-F]+)\s+(.*)\s+(.*)\s+(short|long)\s*(\s+.*)?$", line)
            if match == None:
                print("file '%s' line '%s' does not match format" % (reglist, line))
                sys.exit()

            # build the filename
            xlsname = ext_path(match.group(1)).strip()
            basename, ext = os.path.splitext(os.path.basename(xlsname))

            reg_info.write(match.group(2) + "\t" + xlsname  + "\n")

            # check that the input file exists
            if not os.path.isfile(xlsname):
                # If excel sheet does not exist, command is not added in the list
                # This is to avoid scons error when file is included under disabled
                # preprocessor flag
                continue

            # different cases for RF, modem, and everything else
            sheetname="Registers"
            hname = os.path.join(reg_build_dir, "reg_"+basename.lower())
            _hname = os.path.join(reg_build_dir, "_reg_"+basename.lower())

            if match.lastindex == 6:
                sheetname = match.group(6).strip()
                if sheetname != 'LA':
                    hname += '_' + sheetname.lower()
                    _hname += '_' + sheetname.lower()
            hname += ".h"
            _hname += ".h"
            regaddr = match.group(2)
            regaccess = match.group(3)
            regprefix = match.group(4)
            regformat = match.group(5)

            # Take into account scripts modif also!
            Depends(hname, "#/../tools/reg_xls2h/reg_xls2h.py")
            Depends(hname, "#/../tools/reg_xls2h/reg_xls2h_h.py")
            Depends(hname, "#/../tools/reg_xls2h/reg_xls2h_xls.py")
            Depends(_hname, "#/../tools/reg_xls2h/reg_xls2h.py")
            Depends(_hname, "#/../tools/reg_xls2h/reg_xls2h_h.py")
            Depends(_hname, "#/../tools/reg_xls2h/reg_xls2h_xls.py")

            # command to run
            env.Command(hname, xlsname, build_reg, REGADDR=regaddr, FORSIMU=str(forsimu),
                        REGACCESS=regaccess, REGPREFIX=regprefix, REGFORMAT=regformat,
                        SHEETNAME=sheetname, PRINT_CMD_LINE_FUNC=outstr_gen,
                        OUTSTR=OUTSTR_REG)

            env.Command(_hname, xlsname, build_reg, REGADDR=regaddr, FORSIMU=str(forsimu),
                        REGACCESS=regaccess, REGPREFIX=regprefix, REGFORMAT=regformat,
                        SHEETNAME=sheetname, PRINT_CMD_LINE_FUNC=outstr_gen,
                        OUTSTR=OUTSTR_REG)

    reg_info.close()


def file_list_read(env, filelist_name, append_path = None, must_exist = True):
    """Read a list from a file"""
    filelist = []

    try:
        fin = open(filelist_name, 'rU')
    except:
        if must_exist:
            print("Cannot open list file: " + filelist_name)
            sys.exit()
        else:
            return filelist

    comment_re = re.compile(r'^\s*(#|$)')
    include_re = re.compile(r'^\s*include (.*)')

    if append_path:
        directory = append_path
    else:
        directory = os.path.dirname(filelist_name)

    for line in fin:
        line =  env.subst(line.strip())

        if not comment_re.match(line):
            include = include_re.match(line)
            if include:
                include_file = os.path.join(directory, include.group(1))
                filelist += file_list_read(env, include_file)
            else:
                abs_filename =  os.path.abspath(os.path.join(directory, line))
                filelist.append(abs_filename)

    fin.close()

    return filelist

def action_sed_obj(target, source, env):

    gcno = str(target[0]).replace('.obj', '.gcno')

    if os.access(gcno, os.W_OK):
        os.spawnl(os.P_WAIT, env['SED_PATH'], "sed", "-b", "-i", "-e", env['SED_SUBST_OBJ'], gcno)

    return 0

def build_object(env, src_dir, defines = {}):
    """ Register compilation for each file found in the sourcelist.txt file present
    in 'src_dir' directory. The path of the object file is the path of the source
    file in which dirname of 'src_dir' is replaced by 'env['OBJ_DIR]'.
    If trace is enabled, it also register the dictionary creation for this file
    """
    objlist = []

    # replace dir up to parent directory to avoid duplicated 'VariantDir'
    top_dir = os.path.dirname(src_dir)
    obj_dir = env['OBJ_DIR']

    for src_file in file_list_read(env, os.path.join(src_dir, 'sourcelist.txt')):
        # Replace source directory by build directory so that object and dictionary will
        # be generated in build directory
        build_file = src_file.replace(top_dir, obj_dir, 1);

        # and alias directories so that the build_file can be found
        env.VariantDir(os.path.dirname(build_file), os.path.dirname(src_file), duplicate=0)

        # If trace is activated, create dictionary part for this file
        traceflags = ""
        if env["TRACE"] == "on":
            env['FILE_ID'] = env.get('FILE_ID', 0) + 1
            if env['FILE_ID'] == 1:
                env['DICT_FILES'] = []

            dict_file = re.sub("\.[^.]+", ".dict", build_file)
            env['DICT_FILES'] += env.Command(dict_file, build_file, build_file_dictionary, PRINT_CMD_LINE_FUNC=outstr_none,
                                             OUTSTR=OUTSTR_DICT, FILE_ID=env["FILE_ID"])
            traceflags = "-DTRACE_FILE_ID=" + `env["FILE_ID"]`

        # add the object to the list
        objlist.append(env.Object(build_file, TRACEFLAGS=traceflags, **defines))

        # For analysis tools, keep list of source files
        if (src_file.rsplit('.', 1)[1] == 'c'):
            env['SRC_LIST'].append(src_file)

    env['SRC_DIR'].append(src_dir)

    return objlist

def do_build_map(target, source, env):
    """ For TL4/X1 target this function update map file with the list of object.
    For other target using ld linker if requested (via CPU_MEM parameter) this
    function will update the CPU size in the linker script otherwise it simply copy it """
    target_filename = target[0].rstr()
    source_filename = source[0].rstr()

    if (env['ARCH'] == 'tl4xx' or env['ARCH'] == 'ceva-x'):
        # For CEVA linker, need to add the list on object in the map.txt file
        map_src_file = open(source_filename, 'r')
        map_dst_file = open(target_filename, 'w')
        obj_path_list = env['OBJECTS']

        # crt0 must be first, or rather our own version of crt0 contents
        # crtn must be last
        for l in map_src_file.readlines():
            map_dst_file.write(l)

            # object files
            if l.find('objects:') != -1:
                objects_list=[]
                for o_path in obj_path_list:
                    path = '    ' + o_path[0].rstr()
                    vers = path.find('build_version.c')
                    if vers != -1:
                        path_l = list(path)
                        path_l[vers + 14] = 'o'
                        path = "".join(path_l)
                    crt0 = path.find('crt0.o')
                    if crt0 != -1:
                        objects_list.insert(0, path)
                        continue
                    crtn = path.find('crtn.o')
                    if crtn != -1:
                        crtn_o = path
                        continue
                    objects_list.append(path)
                objects_list.append(crtn_o)
                map_dst_file.write("\n".join(objects_list))

        map_src_file.close()
        map_dst_file.close()
    elif env['CPU_MEM']:
        # If
        map_src_file = open(source_filename, 'r')
        map_dst_file = open(target_filename, 'w')
        in_memory_section = 0

        for l in map_src_file.readlines():
            if in_memory_section == 0:
                if re.match('^\s*MEMORY', l):
                    in_memory_section = 1
            elif in_memory_section == 1:
                if re.match('\}', l):
                    in_memory_section = 2;
                elif re.match('.*(ram|cpu|program|data)(_memory)?.*(LENGTH|len|l)\s*=', l):
                    l = re.sub('((LENGTH|len|l)\s*=\s*).*$', r'\1', l).strip('\n') + env['CPU_MEM'] + "\n"

            map_dst_file.write(l)

        map_src_file.close()
        map_dst_file.close()
    else:
        shutil.copyfile(source_filename, target_filename)

    target[0].built()

def build_map(target, source, env, objects):
    map_file =  env.Command(target, source, do_build_map, PRINT_CMD_LINE_FUNC=outstr_gen,
                            OUTSTR=OUTSTR_MAP, OBJECTS=objects);
    # Always rebuilt the map file as its contents may change because of compilation parameter
    env.AlwaysBuild(map_file)
    return map_file


def win2unix(target, source, env):
    text = open(str(source[0]), "U").read()
    text = text.replace("\r\n", "\n")
    open(str(target[0]), "wb").write(text)

def update_coff_hex(target, source, env):
    """ Update COFF Hex according to the endianness and the base addresses of code and data memories.
    The driver loads code and data memory using little endian. """

    #Open target hex file
    target_hex = open(str(target[0]), "wb")
    #read sort hex file
    sort_hex = open(str(source[0]), "r")

    #Store all the lines in a list
    lines = [line for line in sort_hex]
    line_idx = 0

    #Loop over every group of 4 lines
    while line_idx < len(lines):
        #Use a buffer of 4 lines
        line_buff = lines[line_idx:line_idx+4]
        #Set memory type to CODE_MEM or DATA_MEM
        mem_type = 'CODE_MEM' if line_buff[0][0] == 'C' else 'DATA_MEM'
        #Initialize memory segment, address and data lists
        mem_seg = [0] * 4
        addr = [0] * 4
        data = [0] * 4

        #Read bytes
        for byte_idx in range(4):
            if byte_idx < len(line_buff):
                (mem_seg[byte_idx], addr[byte_idx], data[byte_idx]) = struct.unpack("cx8sx2s", line_buff[byte_idx].strip())
                #Add code memory base address
                addr[byte_idx] = int(addr[byte_idx], 16) + env[mem_type]['base_add']

            #If current memory section is over or addresses are not consecutive
            if (byte_idx > 0 and
                (mem_seg[byte_idx] != mem_seg[byte_idx - 1] or addr[byte_idx] != addr[byte_idx - 1] + 1)):
                mem_seg[byte_idx] = mem_seg[byte_idx - 1]
                addr[byte_idx] = addr[byte_idx - 1] + 1
                #Set byte to 0
                data[byte_idx] = "00"
                #Shift back line buffer
                line_idx -= 1

        #If Big Endian
        if (env[mem_type]['endianness'] == 'big'):
            #Invert data list order
            data = data[::-1]

        #Write in target file
        for byte_idx in range(4):
            target_hex.write("C:%08x %s\n" % (addr[byte_idx], data[byte_idx]))

        #Shift line buffer in the input file
        line_idx += 4

    #Close target hex file
    target_hex.close()
    #Close sort hex file
    sort_hex.close()

def rvds_elf2bin(env, target, source):
    return env.Command(target+'.bin', source, 'fromelf -c --bin --output $TARGET $SOURCE',
                       PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_R_ELF)

def aps_elf2ihex(env, target, source):
    return  env.Command(target+'.ihex', source, 'aps-objcopy --output-target=ihex $SOURCE $TARGET',
                        PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_A_HEX)

def gnuarm_elf2bin(env, target, source):
    return env.Command(target+'.bin', source, 'arm-elf-objcopy --output-target=binary $SOURCE $TARGET',
                       PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_G_ELF)

def armgcc_4_8_elf2bin(env, target, source):
    return env.Command(target+'.bin', source, 'arm-none-eabi-objcopy --output-target=binary $SOURCE $TARGET',
                       PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_AGC_ELF)

def riscv_elf2ihex(env, target, source):
    return env.Command(target+'.ihex', source, 'riscv32-unknown-elf-objcopy --output-target=ihex $SOURCE $TARGET',
                       PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_V_HEX)

def tl4_coff2hex(env, target, source):
    # with V15 tools, coffutil dump the output of fw0.out on stdout
    if sys.platform == 'win32':
        out = env.Command(target+'0.out', source, 'coffutil -quiet -c $SOURCE > nul',
                          PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_T_OUT)
    else:
        out = env.Command(target+'0.out', source, 'coffutil -quiet -c $SOURCE > /dev/null',
                          PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_T_OUT)
    out = env.Command(target+'.sort', out, 'sorthex $SOURCE $TARGET',
                       PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_T_SORT)
    return env.Command(target+'.hex', out, win2unix,
                       PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_T_HEX)

def ceva_x_coff2ihex(env, target, source):
    """ Convert .a int o ihex files: First extract raw data of code and data section usign coffutil and sorthex.
    Then convert those hex files to ihex format using intelhex executable. Finally merge ihex files generated (one
    for code and one for data) """
    if sys.platform == 'win32':
        out = env.Command(target+'0.out', source, 'coffutil -quiet -c $SOURCE > nul',
                          PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_X_OUT)
    else:
        out = env.Command(target+'0.out', source, 'coffutil -quiet -c $SOURCE > /dev/null',
                          PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_X_OUT)

    out = env.Command(target+'0.sort', out, 'sorthex $SOURCE $TARGET',
                      PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_X_SORT)

    out = env.Command(target + '.sort', out, update_coff_hex ,
                       PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_X_SORT)

    #Execute intelhex to generate iHex files for code and data memory
    return env.Command(target+'.ihex', out, 'intelhex -quiet $SOURCE',
                       PRINT_CMD_LINE_FUNC=outstr_gen, OUTSTR=OUTSTR_X_HEX)


def gcov_cygwin(env, target):

    winconf = os.path.abspath('..').replace('\\','/')
    cygconf = '/cygdrive/' + winconf[0].lower() + winconf[2:]
    winbuild = os.path.abspath(ext_path(env['OBJ_DIR'])) + '\\'

    # awfull bug in scons/python/cygwin: if '*' in the argument, '\' must be duplicated!
    #winbuild = winbuild.replace('\\', '\\\\\\\\')

    sed_subst = "'s+" + cygconf + "/" + winbuild.replace('\\', '\\\\\\\\') + "\\(.*\\)\\.gcda" \
                "+" + winbuild.replace('\\', '/') + "\\1\\.gcda\\x00" + cygconf + "+g'"

    action_cmd = "sed -b -i " + sed_subst + " " + str(target[0])
    action_str = '%16s $TARGET.name' % (OUTSTR_SEDX)
    env.AddPostAction(target, Action(action_cmd, action_str))




#-----------------------------------------------------------
# build string
#-----------------------------------------------------------

build_string_file = '#../src/build/build_version.c'

def get_id_string():
    """ Return info line with logname + date + time
    """
    if platform.system() == "Windows":
        user = 'username'
    else:
        user = 'LOGNAME'
    return os.environ.get(user,'-') + \
        datetime.datetime.today().strftime(' %b %d %Y %H:%M:%S')


def get_svn_string(path):
    """ Return info line with svn revision number
    """
    if SCons.Util.WhereIs('svnversion') == None:
        return ''

    get_svn_ver = subprocess.Popen(["svnversion", "-c", path], stderr=subprocess.PIPE, stdout=subprocess.PIPE)
    if get_svn_ver.stderr.read().strip('\n\t\r') == '':
        svn_ver = ' - svn' + re.sub(r'.*:', '', get_svn_ver.stdout.read().strip('\n\t\r')).split(" ", 2)[0]
        if svn_ver == " - svnUnversioned" :
            try:
                # not a svn working copy, try git-svn
                svn_ver = get_git_svn_string(path)
            except OSError:
                svn_ver = ''
    else:
        svn_ver = ''

    return svn_ver

def get_git_svn_string(path):
    """ Return info line with svn revision number using git-svn
        (in case it isn't available with svn)
    """
    idx = 0
    svn_ver = ''

    while 1 :
        get_svn_ver = subprocess.Popen(["git", "-C", path, "svn", "find-rev", "HEAD~" + `idx`], stderr=subprocess.PIPE, stdout=subprocess.PIPE)
        if get_svn_ver.stderr.read().strip('\n\t\r') == '':
            revision = get_svn_ver.stdout.read().strip('\n\t\r')
            if revision == "":
                idx = idx + 1
            else:
                svn_ver = ' - svn' + revision
                if idx > 0 :
                    svn_ver = svn_ver + "M"
                    break
                else :
                    # check if working copy is clean
                    p1 = subprocess.Popen(["git", "-C", path, "status", "-uno", "--porcelain"], stdout=subprocess.PIPE)
                    p2 = subprocess.Popen(["grep", "-q", "^ M"], stdin=p1.stdout)
                    p2.communicate()
                    if p2.returncode == 0:
                        svn_ver = svn_ver + "M"
                    else:
                        svn_ver = svn_ver + " "
                    break;
        else:
            break

    # append git info
    if svn_ver != '':
        try:
            sha1 = subprocess.check_output(["git", "-C", path, "rev-parse", "--short", "HEAD"], stderr=subprocess.STDOUT).strip('\n\t\r')
            p1 = subprocess.Popen(["git", "-C", path, "status", "-uno", "--porcelain"], stdout=subprocess.PIPE)
            p2 = subprocess.Popen(["grep", "-q", "^ M"], stdin=p1.stdout)
            p2.communicate()
            if p2.returncode == 0:
                sha1 = sha1 + "M"
            else:
                sha1 = sha1 + " "
            try:
                branch = subprocess.check_output(["git", "-C", path, "symbolic-ref", "--short", "HEAD"], stderr=subprocess.STDOUT).strip('\n\t\r')
            except subprocess.CalledProcessError:
                branch = "detached"

            svn_ver = svn_ver + " (" + sha1 + "/" + branch + ")"
        except subprocess.CalledProcessError:
            pass

    return svn_ver

def build_version_cmd(target, source, env):
    """ SCons command generating the build_version.c file. This file must be generated;
        it cannot be committed because of its frequent modifications
    """

    path = os.path.join(os.path.dirname(target[0].path), '../../../../../..')
    f = open(target[0].path, 'w')
    f.write(  '// automatically generated by SCons before each link\n'
            + '#include "version.h"\n'
            + 'const char nx_version_str[] = NX_VERSION_STR;\n'
            + 'const char nx_build_date[] = "' + get_id_string() + get_svn_string(path) + '";\n')

    if env['EXTRA_COMPONENT']:
        versions = {}
        align = 0

        for p in env['EXTRA_COMPONENT']:
            name,path = p.split('=')
            versions[name] = get_svn_string(path)
            if len(name) > align:
                align = len(name)

        f.write('const char nx_build_fhost[] = "extra component version:\\n');
        for p in versions:
            f.write('\\t'+' ' * (align - len(p)) + p + versions[p] + "\\n");
        f.write('";\n')

    f.close()


def build_version_add(nodes, env, folder):
    build_string_file = os.path.join(folder, 'build_version.c')

    """ Register the command generating build_version.c and add it to the nodes
    """
    nodes.append(env.Command(build_string_file, nodes, build_version_cmd,
                             PRINT_CMD_LINE_FUNC = outstr_date))

    if env["TRACE"] == "on":
        file_dict = re.sub("\.[^.]+", ".dict", build_string_file)
        env['DICT_FILES'] += env.Command(file_dict, build_string_file, build_version_dictionary,
                                         PRINT_CMD_LINE_FUNC=outstr_none, OUTSTR=OUTSTR_DICT)


#-----------------------------------------------------------
# Trace dictionary
#-----------------------------------------------------------
def build_version_dictionary(target, source, env):
    """ Extract firmware version, to include it in the final dictionary """
    dictionary  = open(target[0].rstr(), 'w')
    version_file = open(source[0].rstr(), 'r')
    version_re   = re.compile(r'.*nx_build_date\[\] = "(.*)"')
    for line in version_file:
        v = version_re.match(line)
        if v:
            version = v.group(1)

    dictionary.write("# Generated for firmware:\n# " + version + "\n");
    version_file.close
    dictionary.close
    target[0].built();


def build_file_dictionary(target, source, env):
    """Extract all calls to macro TRACE(_xxx) in the source file and print the trace string
       with the corresponding ID (= (file_id << 16) + line_number) in a temporary .dict file """
    trace_start_re = re.compile(r'.*(?:\bTRACE(?:_BUF)?(?:_(\w+))?)\s*\(\s*(.*)$')
    trace_params_re = re.compile(r'((?:\s*\w+\s*,){0,2})\s*"\s*(.*)\s*"\s*(\)|,).*')
    dictionary = open(target[0].rstr(), 'w')
    source_file = open(source[0].rstr(), 'r')
    line_idx = 1
    count = 0
    params = ""

    # Add entry to convert file_id to file name
    filename = os.path.relpath(source[0].rstr(), ext_path('#/../..'))
    if sys.platform == 'win32':
        filename = string.replace(filename, "\\", "/")
    dictionary.write("\t" + `env['FILE_ID']` + ' => "' + filename + '",\n')

    # Extract all calls to TRACE macro. Reassemble (if needed) parameters
    # on the same line before parsing
    for line in source_file:
        if count == 0:
            t = trace_start_re.match(line)
            if t:
                trace_compo = t.group(1)
                params = t.group(2)
                count = 1
                for c in params.rstrip():
                    if c == "(":
                        count += 1
                    elif c == ")":
                        count -= 1
                        if count == 0:
                            break
        else:
            for c in line.rstrip():
                params = params + c
                if c == "(":
                    count +=  1
                elif c == ")":
                    count -=  1
                    if count == 0:
                        break

        if count == 0 and params != "":
            t = trace_params_re.match(params)
            if t:
                if t.group(1) :
                    tmp = t.group(1).split(',')
                    if len(tmp) == 3:
                        trace_level = "[" + tmp[0].strip() + "][" + tmp[1].strip() + "] "
                    elif trace_compo:
                        trace_level = "[" + trace_compo + "][" + tmp[0].strip() + "] "
                    else:
                        trace_level = "[" + tmp[0].strip() + "] "
                elif trace_compo:
                    trace_level = "[" + trace_compo + "] "
                else:
                    trace_level = ""

                escape = False
                trace_str = ""
                # Need some more post processing of the format string:
                # - need to remove all "[:space:]*" due to format string on multiple lines
                #   but not \"[:space:]*"
                #   => this is done with re.sub
                # - If un-escaped " is found in the string it means that trace_params_re also match part of
                #   the parameters that where using ", so format string must end here
                # - If un-escaped $ or @ are used in the format string need to escape then in the dictionary
                for c in re.sub('([^\\\])"\s*"', '\\1', t.group(2)):
                    if c == '"' and not escape:
                        break
                    elif (c == '$' or c == '@') and not escape:
                        trace_str += '\\'

                    if c == '\\' and not escape:
                        escape = True
                    else :
                        escape = False

                    trace_str += c

                trace_id = (env['FILE_ID'] << 16) + line_idx
                dictionary.write("\t" + `trace_id` + ' => "' + trace_level + trace_str + '",\n')
            params = ""

        line_idx += 1

    source_file.close
    dictionary.close
    target[0].built();

def xstr(s):
    """ Convert None to empty string """
    return '' if s is None else s

def build_dictionary_remove_comment_space(s):
    """ Remove leading and trailing spaces of a string"""
    res = s
    res = re.sub('^\s*\*?\s*', '', res)
    res = re.sub('\s*$', '', res)
    return res

def build_dictionary_parse_enum(e, c, enum_list, enum_val_list):
    """Extract value for enum definition provided by parameter e.
    For the <x>th value of the enum the corresponding comment is available in the list
    parameter c[x]. Parameter enum_val_list contains all previously defined value.
    If successfully parsed the enum is added in the dict parameter enum_list and
    all its value are added to the dict parameter enum_val_list.
    """
    enum_re = re.compile(r'(\w+){(.*[^,]),?}');
    enum_val_re = re.compile(r'(\w+)(?:=(\w+))?')
    match = enum_re.match(e)
    if match:
        enum_name = match.group(1).lower()
        enum_list[enum_name] = []
        enum_vals  = re.split(',', match.group(2))
        cur_val = -1
        index = 0
        for item in enum_vals:
            match = enum_val_re.match(item)
            if match:
                name = match.group(1)
                val = match.group(2)
                if val:
                    try:
                        cur_val = int(val)
                    except ValueError:
                        try:
                            cur_val = enum_val_list[val]
                        except  KeyError:
                            print "Unknown enum value: " + val
                            cur_val = 0
                else:
                    cur_val += 1
                enum_val_list[name] = cur_val
                t = cur_val, name, c[index]
                enum_list[enum_name].append(t)
            else:
                print "Unexpected enum value: " + item
            index += 1;
    else:
        print "Unexpected enum: " + e

def build_dictionary_parse_component(filename):
    """ Parse all enums present in the provided file, to extract the list of components and all
    defined filters. It will search for component list in an enum named trace_compo with value prefixed with
    TRACE_COMPO_, and filters in enums named trace_level_<compo> with value prefixed with TRACE_LVL_.
    Returns a string containing perl hash to be included in dictionary.
    """
    myfile = open(filename, 'r')
    enum_start_re = re.compile(r'^\s*enum(.*)')
    enum_stop_re = re.compile(r'(.*})')
    comment_line_re = re.compile(r'(.*)(?:///?\s*(.*)\s*$|/\*\*?\s*(.*)\s*\*/(.*))')
    comment_start_re = re.compile(r'(.*)/\*\*?\s*(.*)')
    comment_end_re = re.compile(r'(.*)\*/(.*)')

    enum_list = {}
    enum_val_list = {}

    enum = ""
    started = False
    in_comment = False
    comment_val = ""
    comments = []

    for line in myfile:

        if not started:
            start = enum_start_re.match(line)
            if start:
                started = True
                comments = []
                enum = ""
                line = start.group(1)

        if started:
            while line != "":
                if in_comment:
                    comment = comment_end_re.match(line)
                    if comment:
                        line = xstr(comment.group(2))
                        comment_val += build_dictionary_remove_comment_space(xstr(comment.group(1)))
                        in_comment = False
                    else:
                        comment_val += build_dictionary_remove_comment_space(line)
                        line = ""
                else:
                    comment = comment_line_re.match(line)
                    comment_start = comment_start_re.match(line)
                    end = enum_stop_re.match(line)
                    if comment:
                        comment_val += build_dictionary_remove_comment_space(xstr(comment.group(2)) + xstr(comment.group(3)))
                        line = xstr(comment.group(1)) + xstr(comment.group(4))
                    elif comment_start:
                        in_comment = True;
                        comment_val += build_dictionary_remove_comment_space(xstr(comment_start.group(2)))
                        line = xstr(comment_start.group(1))
                    else:
                        if end:
                            started = False
                            line = end.group(1)
                        line = re.sub('\s*', '', line)
                        if line != '':
                            enum += line
                            if re.match('.*[,}]', line):
                                comments.append(comment_val)
                                comment_val = ""
                            line = ""
                        if not started :
                            build_dictionary_parse_enum(enum, comments, enum_list, enum_val_list)

    res=""
    try:
        compo_list = enum_list['trace_compo']
        res = "push @EXPORT, qw(%trace_compo);\n"
        res += "our %trace_compo = (\n"
        indent = 2
        for compo in compo_list:
            name = re.sub('TRACE_COMPO_', '', compo[1]).lower()
            tmp = ""
            if name == 'max':
                continue
            else:
                tmp = "\"" + name.upper() + "\""
            tmp += " => { "
            res += " " * indent
            res += tmp
            indent += len(tmp)
            res += "id => " + `compo[0]` + ",\n"
            try:
                filter_list = enum_list['trace_level_' + name]
                tmp1 = "filters => {"
                res += " " * indent + tmp1 + "\n"
                indent += len(tmp1)
                for filt in filter_list:
                    tmp2 = re.sub('TRACE_LVL_%s_' % (name.upper()), '' , filt[1]).upper() + " => {"
                    res += " " * indent + tmp2
                    indent += len(tmp2)
                    res += "bit  => " + `filt[0]` + ",\n"
                    res += " " * indent + "desc => \"" + filt[2] + "\",\n"
                    indent -= len(tmp2)
                    res += " " * indent + "},\n"
                indent -= len(tmp1)
                res += " " * indent + "},\n"
            except KeyError:
                # no specific filter for this component
                pass
            indent -= len(tmp)
            res += " " * indent + "},\n"
        res += ");\n\n"
    except KeyError:
        print "Cannot find enum trace_compo"

    return res

def build_dictionary(target, source, env):
    """ Merge all temporary .dict files into a single .pm file and parse file
        trace_compo.h to get the list of components and filters.
        (may be a better idea to write a xml file but since decoder will be a
         perl script, for simplicity generate a perl module) """
    dictionary = open(target[0].rstr(), 'w')

    # First see if we can include task ID in the dictionary
    task=''
    task_update = os.path.join(env['TOOLDIR'], "trace/task_update.pl")
    if os.path.isfile(task_update):
        try:
            subprocess.check_call(["perl", task_update, "--no-package"], stderr=subprocess.PIPE, stdout=subprocess.PIPE)
            task=os.path.join(env['TOOLDIR'], "trace/task.pm")
            if not os.path.isfile(task):
                task=''
        except OSError:
            print "Install perl to get full trace functionality"
            pass
        except subprocess.CalledProcessError:
            print "Error reported by : " + task_update
            pass

    # extract dictionary that contains version
    for f in source:
        if f.rstr().find("build_version.dict") != -1:
            file_dict = open(f.rstr(), 'r')
            dictionary.write(file_dict.read())
            file_dict.close()
            source.remove(f)
            break

    package = re.sub(r"\.pm", "", os.path.basename(target[0].rstr()))
    dictionary.write("package " + package + ";\n\n")

    dictionary.write("use strict;\n")
    dictionary.write("use Exporter;\n")
    dictionary.write("use vars qw(@ISA @EXPORT);\n")
    dictionary.write("@ISA = qw(Exporter);\n\n")

    # Trace components
    trace_compo=""
    for p in env['CPPPATH']:
        if os.path.isfile(os.path.join(p, "trace_compo.h")):
            dictionary.write(build_dictionary_parse_component(os.path.join(p, "trace_compo.h")))
            break

    # Trace points
    dictionary.write("push @EXPORT, qw(%fw_dict);\n")
    dictionary.write("our %fw_dict = (\n")
    for f in source:
        file_dict = open(f.rstr(), 'r')
        dictionary.write(file_dict.read())
        file_dict.close()
    dictionary.write(");\n\n");

    # Task/Messages ids
    if not task=='':
        file_task = open(task, 'r');
        dictionary.write(file_task.read());
        file_task.close();

    dictionary.write("\n1;\n");
    dictionary.close();

def build_dict(target, env):
    Depends(target, "#/../tools/trace/task_update.pl")
    return env.Command(target, env['DICT_FILES'], build_dictionary, PRINT_CMD_LINE_FUNC=outstr_gen,
                       OUTSTR=OUTSTR_DICT, TOOLDIR=ext_path('#/../tools/'))

def static_analysis(tool_name, build_dir, env):
    if tool_name == 'cppcheck':
        sources = []
        includes = []
        for src in env['SRC_LIST']:
            pos = src.find('$')
            while pos != -1:
                stop = src.find('\\', pos)
                if stop == -1:
                    stop = src.find('/', pos)
                if stop == -1:
                    stop = src.find('}', pos)
                    if stop > -1: stop += 1
                if stop == -1:
                    var = src[pos:]
                else:
                    var = src[pos:stop]

                key = var[1:].replace('{', '').replace('}', '')
                src = src.replace(var, env.get(key, ""))
                pos = src.find('$', pos)
            sources.append(src)

        for inc in env['CPPPATH']:
            pos = inc.find('$')
            while pos != -1:
                stop = inc.find('\\', pos)
                if stop == -1:
                    stop = inc.find('/', pos)
                if stop == -1:
                    stop = inc.find('}', pos)
                    if stop > -1: stop += 1
                if stop == -1:
                    var = inc[pos:]
                else:
                    var = inc[pos:stop]

                key = var[1:].replace('{', '').replace('}', '')
                inc = inc.replace(var, env.get(key, ""))
                pos = inc.find('$', pos)
            includes.append(inc)
        text = True
        if sys.platform == "win32" or sys.platform == 'cygwin':
            cmd = ext_path('#/../tools/analysis_tools/Cppcheck/cppcheck.exe')
            html = True
        elif sys.platform == "linux2":
            cmd = ext_path('#/../tools/analysis_tools/Cppcheck/cppcheck')
            html = False
        else:
            print "System platform %s not supported"%(sys.platform)
            return
        options = ['--template=gcc', '--enable=all', '--std=posix', '--std=c99', '--language=c', '--suppress=missingIncludeSystem', '-D__GNUC__']
        if env['CPPCHECK_ALL_CFG'] == 'on':
            options.append('--force')
        else:
            options += ['-D%s'%(e) for e in env['CPPDEFINES']]
        options += ['-I%s'%(e) for e in includes]
        txt_file = os.path.join(build_dir, 'analysis', 'cppcheck', 'result.txt')

        if text:
            if not os.path.exists(os.path.dirname(txt_file)):
                os.makedirs(os.path.dirname(txt_file))
            args = [cmd] + options + sources
            f_err = open(txt_file,"wb")
            p = subprocess.Popen(
                args,
                stderr=f_err,
                universal_newlines = True)
            p.communicate()
            f_err.close()
            print "%s %s %s"%(cmd, options, sources)

        if html:
            import pkg_resources

            options.append('--xml')
            options.append('--xml-version=2')
            xml_file = os.path.join(build_dir, 'analysis', 'cppcheck', 'result.xml')
            html_report_tool = os.path.join(os.path.dirname(cmd), 'htmlreport', 'cppcheck-htmlreport.py')
            html_report_path  = os.path.join(os.path.dirname(xml_file), 'htmlreport')

            req = pkg_resources.Requirement.parse("Pygments")
            if (not any(dist in req for dist in pkg_resources.working_set)):
                import pip

                path = os.path.join(os.path.dirname(html_report_tool), 'Pygments-2.2.0-py2.py3-none-any.whl')
                if (int(pip.__version__[0]) >= 10):
                    from pip._internal import main as pip_main
                    pip_main(['install', path])
                else:
                    pip.main(['install', path])

            if not os.path.exists(html_report_path):
                os.makedirs(html_report_path)
            args = [cmd] + options + sources

            f_err = open(xml_file,"wb")
            p = subprocess.Popen(
                                 args,
                                 stderr=f_err,
                                 universal_newlines = True)
            p.communicate()
            f_err.close()

            args = ["python", html_report_tool, "--file=" + xml_file, "--report-dir=" + html_report_path, "--source-dir="+env['SRC_DIR'][0]] + env['SRC_DIR'][1:] + includes
            p = subprocess.Popen(args)
            p.communicate()
            print ' '.join(args)
        fd = open(txt_file, 'r')
        print fd.read()
        fd.close()
        print "%s %s %s"%(cmd, options, sources)
    elif tool_name == 'flawfinder':
        sources = ' '.join(env['SRC_DIR'])
        cmd = ext_path('#/../tools/analysis_tools/flawfinder-1.31/flawfinder')
        options = "-H -C -c %s"%(sources)
        html_file = os.path.join(build_dir, 'analysis', 'flawfinder', 'result.html')

        if not os.path.exists(os.path.dirname(html_file)):
            os.makedirs(os.path.dirname(html_file))

        args = "python %s %s"%(cmd, options)
        f_out = open(html_file,"wb")
        p = subprocess.Popen(
                             args,
                             env=os.environ.copy(),
                             stdout=f_out,
                             shell=True,
                             universal_newlines=True)
        p.communicate()
        f_out.close()
