/**
 ****************************************************************************************
 *
 * @file bfr.c
 *
 * @brief Beamformer Module Implementation.
 *
 * Copyright (C) RivieraWaves 2016-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup BFR
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */

#include "bfr.h"

#if (RW_BFMER_EN)

// Station Definition
#include "sta_mgmt.h"
// VIF Definition
#include "vif_mgmt.h"
// Status Codes Definition
#include "co_status.h"
// General Purpose DMA Definition
#include "hal_dma.h"
// For use of rxl_mpdu_free function
#include "rxl_hwdesc.h"
// For TPC related definitions
#include "tpc.h"
#if NX_MFP
// For MFP related definitions
#include "mfp.h"
// For TXU related definitions
#include "txu_cntrl.h"
#endif
// For TD related definitions
#include "td.h"
#include "ke_timer.h"

/*
 * DEFINES
 ****************************************************************************************
 */

/**
 * Number of bits of a PHY element in matrix for a SU feedback type.
 * We are considering here the worst case for received report length, hence codebook = 1
 */
#define BFR_NB_BIT_PHY_SU               (4)
/**
 * Number of bits of a PHY element in matrix for a MU feedback type.
 * We are considering here the worst case for received report length, hence codebook = 1
 */
#define BFR_NB_BIT_PHY_MU               (7)
/**
 * Number of bits of a PSY element in matrix for a SU feedback type.
 * We are considering here the worst case for received report length, hence codebook = 1
 */
#define BFR_NB_BIT_PSY_SU               (6)
/**
 * Number of bits of a PSY element in matrix for a MU feedback type.
 * We are considering here the worst case for received report length, hence codebook = 1
 */
#define BFR_NB_BIT_PSY_MU               (9)

/// Duration of NDPA frame sent for SU (in us)
#define BFR_NDPA_SU_DURATION            (56)
/// Duration of NDP frame (in us)
#define BFR_NDP_DURATION                (47)
/// Duration of BRP frame sent for MU (in us)
#define BFR_BRP_DURATION                (52)
/// Duration of 20 bytes for a Beanforming Compressed Report Frame (in us)
#define BFR_BCR_DURATION                (8)
/// SIFS Duration (in us)
#define BFR_SIFS_DURATION               (14)

/*
 * MACROS
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @brief Get an estimation of the TX duration in us for a Beamforming Compressed Report frame
 * based on its length.
 * @param[in] length  Report length.
 ****************************************************************************************
 */
#define BFR_GET_REPORT_DURATION(length)                                         \
             (((length + 20) / 20) * BFR_BCR_DURATION)                          \

/**
 ****************************************************************************************
 * @brief Macro to be used in order to update AID12 value in a NDPA frame sent during a
 * SU calibration. Warning: Clear all other subfields of the STA Info field.
 * @param[in] sta_info      Pointer to the STA Info field value to be updated
 * @param[in] aid           Full AID
 ****************************************************************************************
 */
#define BFR_STA_INFO_SET_AID12(sta_info, aid)                                  \
            (sta_info = (aid & BFR_NDPA_AID12_MASK))                           \

/**
 ****************************************************************************************
 * @brief Macro to be used in order to update Feedback Type value in a NDPA frame sent
 * during a SU calibration.
 * @param[in] sta_info      Pointer to the STA Info field value to be updated
 * @param[in] fback_type    Feedback Type value
 ****************************************************************************************
 */
#define BFR_STA_INFO_SET_FBACK_TYPE(sta_info, fback_type)                      \
            (sta_info |= (fback_type << BFR_NPDA_FBACK_POS))                   \

/**
 ****************************************************************************************
 * @brief Macro to be used in order to update NC Index value in a NDPA frame sent
 * during a SU calibration.
 * @param[in] sta_info      Pointer to the STA Info field value to be updated
 * @param[in] nc_index      NC Index
 ****************************************************************************************
 */
#define BFR_STA_INFO_SET_NC_INDEX(sta_info, nc_index)                          \
            (sta_info |= (nc_index << BFR_NPDA_NC_IDX_POS))                    \

/**
 ****************************************************************************************
 * @brief Macro to be used in order to extract subfield value in VHT MIMO Control field.
 * @param[in] vht_mimo_ctrl  VHT MIMO Control field value
 * @param[in] field          Field to be read.
 *                           Possible values are:
 *                              NC -> Nc Index,
 *                              NR -> Nr Index,
 *                              CHAN_WIDTH -> Channel Width,
 *                              GROUPING -> Grouping,
 *                              CODEBOOK -> Codebook Information,
 *                              FEEDBACK -> Feedback Type,
 *                              REM_SEGM -> Remaining Feedback Segments
 *                              FIRST_SEGM -> First Feedback Segment
 *                              TOKEN -> Sounding Dialog Token Number
 ****************************************************************************************
 */
#define BFR_VHT_MIMO_GET_VALUE(vht_mimo_ctrl, field)                           \
            ((vht_mimo_ctrl >> BFR_MIMO_CTRL_ ## field ## _OFFSET) &           \
                               BFR_MIMO_CTRL_ ## field ## _MASK)               \

/**
 ****************************************************************************************
 * @brief Macro indicating if provided status bit is set in a status bitfield.
 * @param[in] status    Status bitfield value
 * @param[in] bit_pos   Position index of the bit in the bit field
 ****************************************************************************************
 */
#define BFR_IS_STATUS_BIT_SET(status, bit_pos)                                 \
            (status & (1 << bit_pos))                                          \

/**
 ****************************************************************************************
 * @brief Set a status bit to 1 in a given status bitfield.
 * @param[in] status    Status bitfield value
 * @param[in] bit_pos   Position index of the bit in the bit field
 ****************************************************************************************
 */
#define BFR_SET_STATUS_BIT(status, bit_pos)                                    \
            (status |= (1 << bit_pos))                                         \

/**
 ****************************************************************************************
 * @brief Set a status bit to 0 in a given status bitfield.
 * @param[in] status    Status bitfield value
 * @param[in] bit_pos   Position index of the bit in the bit field
 ****************************************************************************************
 */
#define BFR_RESET_STATUS_BIT(status, bit_pos)                                  \
            (status &= ~(1 << bit_pos))                                        \

/**
 ****************************************************************************************
 * @brief Compute the Number of Angles in function of Nc and Nr values
 * @param[in] nc    Nc Value
 * @param[in] nr    Nr Value
 * @return Computed Na value
 *              Na = Nc * (2 * Nr - Nc - 1)
 ****************************************************************************************
 */
#define BFR_GET_NA(nc, nr)                                                     \
            (nc * ((2 * nr) - nc - 1))                                         \

/*
 * "PRIVATE" VARIABLES
 ****************************************************************************************
 */

/// Nodes that can be used to describe state of BFR Memory
struct bfr_mem_node bfr_nodes[BFR_MEM_NODE_NB];
/// BFR Information for each STA
struct bfr_info_sta bfr_sta_infos[NX_REMOTE_STA_MAX];
/// SMM Pointers
uint16_t bfr_smm_ptrs[BFR_MEM_REP_NB_MAX];
/// BFR Environment
struct bfr_env_tag bfr_env;

/**
 * Array providing Ns value in function of Channel Width and Grouping value.
 *  * As this array is used in order to compute the expected report length in worst
 * case, it is assumed that beamformee do not use grouping.
 * Ns = Number of carriers in Compressed Beamforming Feedback.
 * Refer to Table 8-53g in IEEE Std 802.11ac-2013 specification for details.
 */
const uint16_t bfr_ns[] = {
    [PHY_CHNL_BW_20] = 52,
    [PHY_CHNL_BW_40] = 108,
    [PHY_CHNL_BW_80] = 234,
    [PHY_CHNL_BW_160] = 468,
    [PHY_CHNL_BW_80P80] = 468,
};

/**
 * Array providing Ns' value in function of Channel Width.
 * As this array is used in order to compute the expected report length in worst
 * case, it is assumed that beamformee do not use grouping.
 * Ns' = Number of carriers in MU Exclusive Beamforming Feedback.
 * Refer to Table 8-53g in IEEE Std 802.11ac-2013 specification for details.
 */
const uint8_t bfr_nsp[] = {
    [PHY_CHNL_BW_20] = 30,
    [PHY_CHNL_BW_40] = 58,
    [PHY_CHNL_BW_80] = 122,
    [PHY_CHNL_BW_160] = 244,
    [PHY_CHNL_BW_80P80] = 244,
};

/*
 * "PRIVATE" FUNCTION DECLARATIONS
 ****************************************************************************************
 */

static void bfr_start_next_calibration(void);
static void bfr_calibrate(void);

/*
 * "PRIVATE" FUNCTION DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @param[in] vif Vif pointer
 * @return Bandwith to use to send BFR control frame for this vif.
 ****************************************************************************************
 */
__INLINE uint8_t bfr_get_bw(struct vif_info_tag *vif)
{
    uint8_t bw;
    #if (!NX_CHNL_CTXT)
    struct phy_channel_info phy_info;
    phy_get_channel(&phy_info, PHY_PRIM);
    bw = (uint8_t) PHY_INFO_BW(phy_info);
    #else
    ASSERT_ERR(vif->chan_ctxt);
    bw = vif->chan_ctxt->channel.type;
    #endif
    ASSERT_ERR(bw < PHY_CHNL_BW_OTHER);
    return bw;
}

/**
 ****************************************************************************************
 * @brief
 * @param[in] env       Should always be a NULL pointer
 ****************************************************************************************
 */
static void bfr_calib_tmr_exp_cb(void *env)
{
    // Start calibration
    bfr_calibrate();

    // Restart the timer
    mm_timer_set(&bfr_env.calib_tmr, ke_time() + bfr_env.calib_tmr_dur);
}

/**
 ****************************************************************************************
 * @brief Handle end of calibration for a given STA
 * If the calibration was a SU calibration, the bfr module goes back to the IDLE state.
 * If it was a MU calibration and the uploaded segment is the last segment of the last
 * expected report, the bfr module goes back to the IDLE state.
 * Finally the RX software descriptor is free.
 * @param[in] bfr           STA for which calibration is over
 * @param[in] updated       Indicate if report is host memory has been updated and must be
 *                          downloaded in PHY memory upon next transmission request
 ****************************************************************************************
 */
static void bfr_end_calibration_for_sta(struct bfr_info_sta *bfr, bool updated)
{
    // Update STA state
    BFR_RESET_STATUS_BIT(bfr->status, BFR_STA_CALIBRATE);

    if (updated)
    {
        // Keep in mind that report has been updated, it will be downloaded upon next transmission.
        BFR_SET_STATUS_BIT(bfr->status, BFR_STA_REPORT_UPDATED);
        // Report in host memory is valid
        BFR_SET_STATUS_BIT(bfr->status, BFR_STA_HAS_VALID_REPORT);

        // Calibration is successful, store calibration time
        bfr->last_calib_time = hal_machw_time();
        // Update the index of the report currently used
        bfr->host_report_idx ^= 1;
    }

    // If SU calibration, go back to IDLE state
    if (bfr_env.status == BFR_STATUS_SU_CALIB)
    {
        // For profiling
        PROF_BFR_SU_CALIB_CLR();

        bfr_env.status = BFR_STATUS_IDLE;
    }
    else // MU Calibration
    {
        // Decrease number of expected reports
        bfr_env.nb_reports--;

        // If all reports have been received, stop MU Calibration
        if (bfr_env.nb_reports == 0)
        {
            // For profiling
            PROF_BFR_MU_CALIB_CLR();

            bfr_env.status = BFR_STATUS_IDLE;
        }
    }

    // If status is IDLE, can try to start a new calibration
    if (bfr_env.status == BFR_STATUS_IDLE)
    {
        bfr_start_next_calibration();
    }
}

/**
 ****************************************************************************************
 * @brief Handle confirmation of either the NDPA packet or the NDP frame or one of the
 * BFR Poll frames.
 * -> Case of a SU calibration: in case of unsuccessful reception of beamforming report
 * response, the NDP is marked as done with retry limit reached.
 * @param[in] env     Has been configured so that value is the address of the struct
 *                    bfr_info_sta for which frame has been sent.
 * @param[in] status  Transmission status provided by the HW.
 ****************************************************************************************
 */
static void bfr_tx_frame_cfm(void *env, uint32_t status)
{
    // Cast the environment
    struct bfr_info_sta *bfr = (struct bfr_info_sta *)env;

    // Verify that the connection still exists
    if (bfr->sta_idx != INVALID_STA_IDX)
    {
        /*
         * Extract from RW-WLAN-nX-MAC-HW-UM v3.00, section 2.6.2.2.1.1
         * "In case of unsuccessful reception of beamforming report response, the NDP is
         * marked as done with retry limit reached".
         * Same behavior implemented for Beamforming Report Poll frame.
         */
        if (!(status & DESC_DONE_TX_BIT) || (status & RETRY_LIMIT_REACHED_BIT))
        {
            // Abort calibration for this STA
            bfr_end_calibration_for_sta(bfr, false);
            // A new calibration will be restarted later
        }
        // else continue until report indication.
    }
}

/**
 ****************************************************************************************
 * @brief Handle upload confirmation of a beamforming report segment in the host memory.
 * This function is intended to be used only once the last descriptor has been uploaded,
 * If the uploaded segment is the last expected segment, the calibration for the STA can
 * be considered as done.
 * @param[in] env       Pointer to the BFR information associated with a given STA
 *                      (struct bfr_info_sta).
 ****************************************************************************************
 */
static void bfr_dma_upload_cfm(void *env)
{
    // Get BFR information
    struct bfr_info_sta *bfr = (struct bfr_info_sta *)env;

    // For profiling
    PROF_BFR_UPLOAD_CLR();

    // Verify that the connection still exists
    if (bfr->sta_idx != INVALID_STA_IDX)
    {
        // Decrease expected number of confirmations for this STA
        bfr->nb_segments--;

        // If all segments have been uploaded, end calibration procedure for this STA.
        if (bfr->nb_segments == 0)
        {
            // Calibration is done for this STA
            bfr_end_calibration_for_sta(bfr, true);
        }
    }
}

/**
 ****************************************************************************************
 * @brief Check if provided frame is a VHT Compressed Beamforming frame.
 * @param[in] frame     Pointer to the first byte of the frame.
 * @return false for frame format is not correct else true.
 ****************************************************************************************
 */
static bool bfr_is_vht_bfr(uint8_t *frame)
{
    // Return if frame is VHT Compressend Beamforming frame
    bool is_vht_bfr = false;

    do
    {
        // Frame MAC Header
        struct mac_hdr *mac_hdr = (struct mac_hdr *)frame;

        // Check if frame is a Management Action No Ack frame
        if (!((mac_hdr->fctl & MAC_FCTRL_TYPESUBTYPE_MASK) == MAC_FCTRL_ACTION_NO_ACK))
        {
            break;
        }

        // Check if frame is a VHT Action frame
        if (co_read8p(CPU2HW(frame) + BFR_VHT_BFR_CATEGORY_OFFSET) != MAC_VHT_ACTION_CATEGORY)
        {
            break;
        }

        // Check if frame is a VHT Compressed Beamforming frame
        if (co_read8p(CPU2HW(frame) + BFR_VHT_BFR_VHT_ACTION_OFFSET) != MAC_COMP_BEAMFORMING_VHT_ACTION)
        {
            break;
        }

        is_vht_bfr = true;
    } while (0);

    return (is_vht_bfr);
}

/**
 ****************************************************************************************
 * @brief Check if a frame can be sent using beamforming.
 * Following conditions have to be fulfilled:
 *      - The frame is a VHT frame
 *      - The MPDU is a singleton, or is the first MPDU of an AMPDU
 * @param[in] txdesc     TX Descriptor containing information about the frame to transmit
 * @param[in] bfr        BFR info for the STA.
 * @return True if the frame can be sent using beamforming, else False.
 ****************************************************************************************
 */
static bool bfr_is_frame_bf_compliant(struct txdesc *txdesc, struct bfr_info_sta *bfr)
{
    // Indicate if frame can be transmitted using beamforming
    bool bf_compliant = false;

    do
    {
        #if (RW_MUMIMO_TX_EN)
        // Check if the MPDU is inside a MU-MIMO PPDU, in that case we have to beamform
        if (is_in_mumimo_ppdu(txdesc))
        {
            bf_compliant = true;
            break;
        }
        else
        #endif //(RW_MUMIMO_TX_EN)
        {
            #if (NX_AMPDU_TX)
            if (is_mpdu_agg(txdesc))
            {
                if (is_mpdu_first(txdesc))
                {
                    // Number of spatial streams for the transmission
                    uint32_t tx_nss;

                    // Check if frame is a VHT frame
                    if (((txdesc->umac.phy_flags & FORMAT_MOD_TX_RCX_MASK) >> FORMAT_MOD_TX_RCX_OFT)
                                                                                    != FORMATMOD_VHT)
                    {
                        break;
                    }

                    tx_nss = ((((txdesc->umac.phy_flags & MCS_INDEX_TX_RCX_MASK) >> MCS_INDEX_TX_RCX_OFT)
                                        & VHT_NSS_MASK) >> VHT_NSS_OFT);

                    // Do not beamform if number of used spatial streams is the number of antennas
                    if (tx_nss == phy_get_ntx())
                    {
                        break;
                    }

                    // Check that number of spatial streams is compliant with the received report
                    if ((uint8_t)tx_nss > bfr->last_rx_nc)
                    {
                        break;
                    }
                }
                else
                {
                    break;
                }
            }
            else
            #endif //(NX_AMPDU_TX)
            {
                // Check if the frame can be beamformed as a unicast
                if (!bfr_is_bfmed_sglt_allowed(txdesc))
                {
                    break;
                }
            }
        }

        // Transmission can be done using beamforming
        bf_compliant = true;
    } while (0);

    return (bf_compliant);
}

/**
 ****************************************************************************************
 * @brief Increase the nb_frames value in a given bfr_mem_node structure.
 * It allows maintaining the number of downloaded nodes that are currently being used.
 * If no node can be reused, any download will be refused.
 * @param[in] node      Node to be updated
 ****************************************************************************************
 */
static void bfr_increase_nb_frames(struct bfr_mem_node *node)
{
    if (node->nb_frames == 0)
    {
        bfr_env.nb_used_nodes++;
    }

    node->nb_frames++;
}

/**
 ****************************************************************************************
 * @brief This function is in charge of removing a memory node from the list of used
 * memory nodes.
 * @param[in] node      Node to be extracted
 ****************************************************************************************
 */
static void bfr_remove_node(struct bfr_mem_node *node)
{
    // Extract node
    co_list_extract(&bfr_env.nodes, &node->hdr);

    // Check if node is linked with a STA
    if (node->sta_idx != INVALID_STA_IDX)
    {
        // Get BFR information
        struct bfr_info_sta *bfr = &bfr_sta_infos[node->sta_idx];

        // Sanity check
        ASSERT_ERR(bfr->node == node);

        // Unlink the STA and the node
        bfr->node = NULL;
    }

    ASSERT_WARN(node->smm_index < BFR_MEM_REP_NB_MAX);

    // Free the SMM index that was used for this node
    bfr_smm_ptrs[node->smm_index] = 0;
    node->smm_index = BFR_INVALID_SMM_INDEX;
    node->sta_idx = INVALID_STA_IDX;

    // Push it in list of free nodes
    co_list_push_back(&bfr_env.free_nodes, &node->hdr);
}

/**
 ****************************************************************************************
 * @brief Get the number of nodes that must be replaced in the list of used node starting
 * from a given a node in  order to be able to allocate the required length in the BFR
 * memory.
 * @param[in] needed_length         Length to be found in the BFR memory
 * @param[in,out] start_node        First node to consider. If a non-free node is found
 * before we found the needed length, the pointer value is modified with the address of
 * the node following the non-free node in order to make the node research faster.
 * @param[in,out] nb_nodes          Pointer to the variable indicating the number of nodes
 * to be replaced.
 * @return True if the requested length has been found, else false.
 ****************************************************************************************
 */
static bool bfr_check_next_nodes(uint16_t needed_length, struct bfr_mem_node **start_node,
                                 uint8_t *nb_nodes)
{
    // Remaining length to be found
    uint16_t rem_len = needed_length;
    // Get node after the indicated nodes
    struct bfr_mem_node *cnode = (*start_node);

    // Initialize number of considered nodes
    *nb_nodes = 0;

    // Loop until no more nodes are available or all needed length has been found
    while (cnode && rem_len)
    {
        // Get next node
        struct bfr_mem_node *next_node = (struct bfr_mem_node *)cnode->hdr.next;

        // Increase number of considered nodes
        (*nb_nodes)++;

        // Check if node can be overwritten
        if (cnode->nb_frames)
        {
            // Stop to search
            *start_node = cnode;
            break;
        }

        // Update remaining length to find
        rem_len -= co_min(rem_len, cnode->length);

        if (!rem_len)
        {
            break;
        }

        // Check if we reached the last node
        if (next_node)
        {
            // We can use the free memory between the two nodes
            rem_len -= co_min(rem_len, next_node->start_addr - (cnode->start_addr + cnode->length));
        }
        else
        {
            // We can use the memory between the last used node and end of the memory
            rem_len -= co_min(rem_len, bfr_env.bfr_mem_eaddr - (cnode->start_addr + cnode->length));
        }

        // Get next node
        cnode = next_node;
    }

    // Return if the required length has been found or not
    return (rem_len == 0);
}

/**
 ****************************************************************************************
 * @brief Parse the BFR memory content in order to find a position at which a beamforming
 * report with a given length can be downloaded.
 * @param[in] length    Length of the report to be downloaded.
 * @return A pointer to the found/created memory node, or NULL if download cannot be done
 * due to a lack of memory.
 ****************************************************************************************
 */
static struct bfr_mem_node *bfr_get_node(uint16_t length)
{
    // Length of report blocks shall be alignment on words
    uint16_t pad_length = (length & 0xFFFC) + 4;

    // Get first used node
    struct bfr_mem_node *cnode = (struct bfr_mem_node *)co_list_pick(&bfr_env.nodes);
    // New node that can be used
    struct bfr_mem_node *nnode = NULL;
    // Keep previous node
    struct bfr_mem_node *pnode = NULL;

    // Best previous node for insertion
    struct bfr_mem_node *bnode = NULL;
    // Start address for new node
    uint16_t best_addr = 0;
    // Number of report overwritten for the best found case
    uint8_t nb_nodes = 0xFF;

    if (!cnode)
    {
        // Whole memory is free, we can put the report at the beginning of the BFR Memory
        best_addr = BFR_MEM_START_ADDR;
        nb_nodes = 0;
    }
    else
    {
        // Start Address
        uint16_t start_addr = BFR_MEM_START_ADDR;

        // Go through the memory and find the best place for the new report
        while (cnode)
        {
            // Compute available length before the used node
            uint16_t ava_length = cnode->start_addr - start_addr;

            // Check if enough room can be found
            if (ava_length >= pad_length)
            {
                // Can directly use this place
                nb_nodes = 0;
                bnode = pnode;
                best_addr = start_addr;
                break;
            }
            else
            {
                do
                {
                    // Number of node to be removed from cnode
                    uint8_t new_nb_nodes = 0xFF;

                    // Check if a good position replacing a report has already been found
                    if (nb_nodes == 1)
                    {
                        break;
                    }

                    // Check number of nodes to be removed if we insert the report here
                    if (!bfr_check_next_nodes(pad_length - ava_length, &cnode, &new_nb_nodes))
                    {
                        break;
                    }

                    if (new_nb_nodes < nb_nodes)
                    {
                        // Keep this position in mind
                        nb_nodes = new_nb_nodes;
                        best_addr = start_addr;
                        bnode = pnode;
                    }

                } while (0);
            }

            // Update start_addr with end of current block
            start_addr = cnode->start_addr + cnode->length;

            // Get next node
            pnode = cnode;
            cnode = (struct bfr_mem_node *)cnode->hdr.next;
        }

        // If no free place has been found check end of memory, after last node
        if (best_addr == 0)
        {
            // Compute available length between end of last node and end of memory
            uint16_t ava_length = bfr_env.bfr_mem_eaddr - start_addr;

            // Check if enough room can be found
            if (ava_length >= pad_length)
            {
                // Can directly use this place
                nb_nodes = 0;
                bnode = pnode;
                best_addr = start_addr;
            }
        }
    }

    // Check if a position has been found
    if (best_addr != 0)
    {
        // Remove nodes
        while (nb_nodes)
        {
            // Get the node to extract
            cnode = (bnode)
                    ? (struct bfr_mem_node *)bnode->hdr.next
                    : (struct bfr_mem_node *)co_list_pick(&bfr_env.nodes);

            // Extract the node
            bfr_remove_node(cnode);

            nb_nodes--;
        }

        // Get a new memory node if no node was available before
        nnode =  (struct bfr_mem_node *)co_list_pop_front(&bfr_env.free_nodes);

        // Sanity check
        ASSERT_ERR(nnode);

        // And start to fill the information
        nnode->start_addr = best_addr;
        nnode->length = pad_length;
        nnode->nb_frames = 0;

        // Insert in list of used nodes
        co_list_insert_after(&bfr_env.nodes, (bnode) ? &bnode->hdr : NULL, &nnode->hdr);
    }

    // Return new node
    return (nnode);
}

/**
 ****************************************************************************************
 * @brief Look for an used SMM index to be used for a memory node.
 * @param[in] node      Memory node waiting for a SMM Index
 * @return BFR_INVALID_SMM_INDEX if no free SMM index is available, else the SMM Index to
 * be used.
 ****************************************************************************************
 */
static uint8_t bfr_get_smm_index(struct bfr_mem_node *node)
{
    uint8_t smm_index = BFR_INVALID_SMM_INDEX;

    // Go through the list of smm pointer in order to find an available SMM index
    for (int i = 0; i < BFR_MEM_REP_NB_MAX; i++)
    {
        uint16_t *smm_ptr = &bfr_smm_ptrs[i];

        if (*smm_ptr == 0)
        {
            // NOTE : BFR Memory is write-only and accesses are 32-bit only
            uint32_t a_bfr;
            uint32_t index_val;

            *smm_ptr = node->start_addr;
            smm_index = i;

            // Get report pointer in BFR Memory
            a_bfr = BFR_MEM_BASE_ADDR + ((smm_index >> 1) * 4);

            if (smm_index & 1)
            {
                uint16_t *adj_smm_ptr = &bfr_smm_ptrs[i - 1];

                index_val = (uint32_t)(*adj_smm_ptr) | ((uint32_t)(*smm_ptr) << 16);
            }
            else
            {
                uint16_t *adj_smm_ptr = &bfr_smm_ptrs[i + 1];

                index_val = (uint32_t)(*smm_ptr) | ((uint32_t)(*adj_smm_ptr) << 16);
            }

            // Set new value
            REG_PL_WR(a_bfr, index_val);

            break;
        }
    }

    return (smm_index);
}

/**
 ****************************************************************************************
 * @brief Compute the maximal report length that could be received from a given Beamformee
 * based on the connection parameters.
 * In order to consider the worst case, it is considered that no carriers are grouped
 * together and that codebook is used.
 * @param[in] bfr           BFR Information for a given STA
 * @param[in] mu            Indicate if report will be received as part of a MU calibration.
 *                          In that the length of the MU Exclusive Beamforming Report will
 *                          be computed.
 * @return Computed length of the report.
 ****************************************************************************************
 */
static uint16_t bfr_get_report_max_length(struct bfr_info_sta *bfr, bool mu)
{
    // Length in bits
    uint32_t length;
    // Number of rows, number of columns, number of bits for phy and psy values, number of angles
    uint8_t nr, nc, b_phy, b_psy, na;
    // Number of carriers in Compressed Beamforming Feedback
    uint16_t ns;
    // Retrieve bandwitdh
    uint8_t bw = bfr_get_bw(&vif_info_tab[bfr->vif_idx]);

    if (mu)
    {
        nc = bfr->bfee_nss + 1;
        b_phy = BFR_NB_BIT_PHY_MU;
        b_psy = BFR_NB_BIT_PSY_MU;
    }
    else
    {
        nc = 1;
        b_phy = BFR_NB_BIT_PHY_SU;
        b_psy = BFR_NB_BIT_PSY_SU;
    }

    nr = bfr_env.bfer_nss + 1;
    na = BFR_GET_NA(nc, nr);
    ns = bfr_ns[bw];

    // Deduce length of Compressend Beamforming Feedback
    length = (8 * nc) + (ns * na * ((b_phy + b_psy) >> 1));

    // If MU calibration, add length of MU Exclusive Beamforming Report
    if (mu)
    {
        // Number of carriers in MU Exclusive Beamforming Report
        uint8_t nsp = bfr_nsp[bw];

        length += 4 * nc * nsp;
    }

    // Return length in bytes
    return (uint16_t)(length / 8);
}

/**
 ****************************************************************************************
 * @brief Allocate a frame descriptor and build a Null Data Packet Announcement Frame to
 * be used during SU calibration with a given peer station.
 * See 802.11ac-2013, section 8.3.1.20 for more details.
 * It has been checked before that:
 *      - Peer STA supports VHT and is SU Beamformee capable
 *      - Current channel for the STA is on 5GHz band (use of VHT)
 * @param[in] bfr          BFR Information for a given STA
 * @param[out] nb_devs     Pointer on number of calibration
 *
 * @return A pointer to the TX Header Descriptor structure allocated for the NDPA. This
 * descriptor shall be linked with the TX Header Descriptor allocated for the NDP frame.
 ****************************************************************************************
 */
static struct txdesc *bfr_prep_ndpa(struct bfr_info_sta *bfr, uint8_t *nb_devs)
{
    // HW TX Header Descriptor
    struct tx_hd *thd = NULL;
    // Next BFR device information
    struct bfr_info_sta *next_bfr;
    // Get VIF Information
    struct vif_info_tag *vif;
    // Allocated frame
    struct txl_frame_desc_tag *frame = NULL;
    struct mac_hdr_ctrl *mac_hdr;
    // Policy table
    struct tx_policy_tbl *pol_tbl;
    uint32_t paid_gid;


    // Check if MU-capable devices can be grouped together on the VIF
    do
    {
        // Initialize number of calibrations
        *nb_devs = 1;

        // Clear STA Info field, except the AID. Feedback type is then SU.
        bfr->sta_info &= BFR_NDPA_AID12_MASK;

        vif = &vif_info_tab[bfr->vif_idx];

        // Extract bandwidth for the calibration
        bfr_env.calib_bw = bfr_get_bw(vif);

        if (!BFR_IS_STATUS_BIT_SET(bfr->status, BFR_STA_MU_CAPABLE))
        {
            break;
        }

        // Update STA Info field. Feedback type is MU.
        BFR_STA_INFO_SET_FBACK_TYPE(bfr->sta_info, BFR_MU_TYPE);
        BFR_STA_INFO_SET_NC_INDEX(bfr->sta_info, 0);

        // If peer device is not STA, only one STA info can be inserted in the NDPA
        if (vif->type == VIF_STA)
        {
            break;
        }

        next_bfr = (struct bfr_info_sta *)bfr->hdr.next;

        /*
         * If next devices are also MU capable and are on the same VIF, we can group them together
         * for the calibration.
         */
        while (next_bfr)
        {
            #if (NX_CHNL_CTXT)
            // Get next VIF information
            struct vif_info_tag *next_vif;
            #endif //(NX_CHNL_CTXT)

            // Check if peer STA is MU capable
            if ((bfr->vif_idx != next_bfr->vif_idx) ||
                !BFR_IS_STATUS_BIT_SET(next_bfr->status, BFR_STA_MU_CAPABLE))
            {
                break;
            }

            #if (NX_CHNL_CTXT)
            next_vif = &vif_info_tab[next_bfr->vif_idx];

            // Verify that we are on the same channel context
            if (next_vif->chan_ctxt != vif->chan_ctxt)
            {
                break;
            }
            #endif //(NX_CHNL_CTXT)

            // Increase number of calibration
            (*nb_devs)++;

            // Clear STA Info field, except the AID.
            next_bfr->sta_info &= BFR_NDPA_AID12_MASK;

            // Update STA Info field. Feedback type is MU.
            BFR_STA_INFO_SET_FBACK_TYPE(next_bfr->sta_info, BFR_MU_TYPE);
            BFR_STA_INFO_SET_NC_INDEX(next_bfr->sta_info, 0);

            next_bfr = (struct bfr_info_sta *)next_bfr->hdr.next;
        }
    } while (0);

    next_bfr = bfr;

    // If we calibrate only 1 device, we switch to SU feedback
    if (*nb_devs == 1)
    {
        // Clear STA Info field, except the AID. Feedback type is then SU.
        bfr->sta_info &= BFR_NDPA_AID12_MASK;
    }

    // Allocate a frame descriptor from the TX path
    frame = txl_frame_get(TX_DEFAULT_NDPA_BRP,
                          BFR_NDPA_SU_LEN + ((*nb_devs - 1) * BFR_NDPA_STA_INFO_LEN));

    // Verify that enough memory has been found for the frame
    if (frame == NULL)
        return NULL;

    // Get the buffer pointer
    mac_hdr = txl_buffer_payload_get(&frame->txdesc);

    // Fill-in the MAC Header
    mac_hdr->fctl = MAC_FCTRL_VHT_NDPA;
    // Duration includes NDPA duration, NDP duration, SIFS and report durations (worst case)
    mac_hdr->durid = BFR_NDPA_SU_DURATION + BFR_SIFS_DURATION + BFR_NDP_DURATION;

    // Add expected report durations
    for (int i = 0; i < *nb_devs; i++)
    {
        mac_hdr->durid += (BFR_SIFS_DURATION
                    + BFR_GET_REPORT_DURATION(bfr_get_report_max_length(next_bfr, (*nb_devs > 1))));

        next_bfr = (struct bfr_info_sta *)next_bfr->hdr.next;
    }

    next_bfr = bfr;

    /*
     * If the VHT NDP Announcement frame includes more than one STA Info field, the RA shall be
     * set to the broadcast address, else the RA shall be set to the MAC address of the VHT
     * beamformee.
     */
    if (*nb_devs == 1)
    {
        struct sta_info_tag *sta = &sta_info_tab[bfr->sta_idx];
        mac_hdr->addr1 = *sta_mgmt_get_peer_addr(bfr->sta_idx);
        paid_gid = sta->paid_gid;
    }
    else
    {
        mac_hdr->addr1 = mac_addr_bcst;
        paid_gid = 63 << GID_TX_OFT;
    }
    mac_hdr->addr2 = *vif_mgmt_get_addr(bfr->vif_idx);

    // Fill-in the Frame Content
    for (int i = 0; i < *nb_devs; i++)
    {
        // Set STA Info
        co_write16p(CPU2HW(mac_hdr) + BFR_NDPA_STA_INFO_OFFSET + (i * BFR_NDPA_STA_INFO_LENGTH),
                    next_bfr->sta_info);

        // Store the token
        next_bfr->last_token = bfr_env.token;

        next_bfr = (struct bfr_info_sta *)next_bfr->hdr.next;
    }

    // Set token -- !!! Bit 0 and Bit 1 are reserved !!!
    co_write8p(CPU2HW(mac_hdr) + BFR_NDPA_DIALOG_TOKEN_OFFSET,
               bfr_env.token << BFR_NPDA_TOKEN_POS);

    // Update the token for the next calibration
    bfr_env.token = (bfr_env.token + 1) & BFR_NDPA_TOKEN_MAX;

    /*
     * Configure HW Descriptor
     * The expectedAck field must be set to 0 - No acknowledgement.
     * protFrmDur must be provided and dontTouchDur bit shall be set.
     */
    thd = &frame->txdesc.lmac.hw_desc->thd;
    thd->macctrlinfo1 |= (mac_hdr->durid << PROT_FRM_DURATION_OFT);
    thd->macctrlinfo1 &= ~EXPECTED_ACK_MSK;
    thd->macctrlinfo2 |= DONT_TOUCH_DUR;

    thd->nextfrmexseq_ptr = 0;
    thd->nextmpdudesc_ptr = 0;
    thd->macctrlinfo2 &= ~(WHICHDESC_MSK | UNDER_BA_SETUP_BIT);
    thd->phyctrlinfo = paid_gid;
    thd->statinfo = 0;

    // Get policy table
    pol_tbl = (struct tx_policy_tbl *)HW2CPU(thd->policyentryaddr);

    // Set BW
    pol_tbl->ratecntrlinfo[0] &= ~BW_TX_RCX_MASK;
    pol_tbl->ratecntrlinfo[0] |= ((uint32_t)bfr_env.calib_bw << BW_TX_RCX_OFT);

    #if (NX_CHNL_CTXT || NX_P2P)
    // Set VIF and STA indexes
    frame->txdesc.host.vif_idx = bfr->vif_idx;
    frame->txdesc.host.staid   = bfr->sta_idx;
    #endif //(NX_CHNL_CTXT || NX_P2P)

    return (&(frame->txdesc));
}

/**
 ****************************************************************************************
 * @brief Allocate a frame descriptor and build a Null Data Packet Frame to be used during
 * SU calibration with a given peer station.
 * It has been checked before that:
 *      - Peer STA supports VHT and is SU Beamformee capable
 *      - Current channel for the STA is on 5GHz band (use of VHT)
 * @param[in] bfr           BFR Information for a given STA
 * @param[in] nb_devs       Number of devices to which the NDP is sent.
 * @return A pointer to the TX Descriptor structure allocated for the NDP.
 ****************************************************************************************
 */
static struct txdesc *bfr_prep_ndp(struct bfr_info_sta *bfr, uint8_t nb_devs)
{
    // Allocate a frame descriptor from the TX path
    struct txl_frame_desc_tag *frame = txl_frame_get(TX_DEFAULT_NDP, 0);
    // HW TX Header Descriptor
    struct tx_hd *thd;
    // Policy table
    struct tx_policy_tbl *pol_tbl;

    if (frame == NULL)
        return NULL;

    // Fill-in the confirmation structure
    frame->cfm.cfm_func = bfr_tx_frame_cfm;
    frame->cfm.env = bfr;

    /*
     * Configure HW Descriptor
     * The expectedAck field must be set to 1 - Normal ACK.
     * dontTouchDur bit shall be reset.
     */
    thd = &frame->txdesc.lmac.hw_desc->thd;
    thd->macctrlinfo1 &= ~EXPECTED_ACK_MSK;
    thd->macctrlinfo1 |= EXPECTED_ACK_NORMAL_ACK;
    thd->macctrlinfo2 &= ~DONT_TOUCH_DUR;
    thd->phyctrlinfo &= ~(GID_TX_MASK | PAID_TX_MASK);
    if (nb_devs > 1)
    {
        thd->phyctrlinfo = 63 << GID_TX_OFT;
    }
    else
    {
        struct sta_info_tag *sta = &sta_info_tab[bfr->sta_idx];
        thd->phyctrlinfo = sta->paid_gid;
    }

    thd->nextfrmexseq_ptr = 0;
    thd->nextmpdudesc_ptr = 0;
    thd->macctrlinfo2 &= ~(WHICHDESC_MSK | UNDER_BA_SETUP_BIT);
    thd->statinfo = 0;

    // Get policy table
    pol_tbl = (struct tx_policy_tbl *)HW2CPU(thd->policyentryaddr);

    // Set BW
    pol_tbl->ratecntrlinfo[0] &= ~BW_TX_RCX_MASK;
    pol_tbl->ratecntrlinfo[0] |= ((uint32_t)bfr_env.calib_bw << BW_TX_RCX_OFT);

    #if (NX_CHNL_CTXT || NX_P2P)
    // Set VIF and STA indexes
    frame->txdesc.host.vif_idx = bfr->vif_idx;
    frame->txdesc.host.staid   = bfr->sta_idx;
    #endif //(NX_CHNL_CTXT || NX_P2P)

    return (&(frame->txdesc));
}

/**
 ****************************************************************************************
 * @brief Allocate a frame descriptor and build a Beamforming Report Poll Frame to be used during
 * MU calibration with a given peer station.
 * It has been checked before that:
 *      - Peer STA supports VHT and is MU Beamformee capable
 *      - Current channel for the STA is on 5GHz band (use of VHT)
 * @param[in] bfr           BFR Information for a given STA
 * @return A pointer to the TX Descriptor structure allocated for the BFR Poll.
 ****************************************************************************************
 */
static struct txdesc *bfr_prep_bfr_poll(struct bfr_info_sta *bfr)
{
    // Allocate a frame descriptor from the TX path
    struct txl_frame_desc_tag *frame = txl_frame_get(TX_DEFAULT_NDPA_BRP, BFR_BRP_LEN);
    // HW TX Header Descriptor
    struct tx_hd *thd;
    struct mac_hdr_ctrl *mac_hdr;
    // Policy table
    struct tx_policy_tbl *pol_tbl;
    struct sta_info_tag *sta = &sta_info_tab[bfr->sta_idx];

    // Verify that enough memory has been found for the frame
    if (frame == NULL)
    {
        return NULL;
    }

    // Get the buffer pointer
    mac_hdr = txl_buffer_payload_get(&frame->txdesc);

    // Fill-in the MAC Header
    mac_hdr->fctl = MAC_FCTRL_BFM_REPORT_POLL;
    mac_hdr->durid = 0;
    mac_hdr->addr1 = *sta_mgmt_get_peer_addr(bfr->sta_idx);
    mac_hdr->addr2 = *vif_mgmt_get_addr(bfr->vif_idx);

    // Fill-in the Frame Content - Request for all feedback segments
    co_write8p(CPU2HW(mac_hdr) + BFR_BRP_RETX_BITMAP_OFFSET, 0xFF);

    /*
     * Configure HW Descriptor
     * The expectedAck field must be set to 1 - Normal ACK.
     * dontTouchDur bit shall be reset.
     */
    thd = &frame->txdesc.lmac.hw_desc->thd;
    thd->macctrlinfo1 &= ~EXPECTED_ACK_MSK;
    thd->macctrlinfo1 |= EXPECTED_ACK_NORMAL_ACK;
    thd->macctrlinfo2 &= ~DONT_TOUCH_DUR;

    thd->nextfrmexseq_ptr = 0;
    thd->nextmpdudesc_ptr = 0;
    thd->macctrlinfo2 &= ~(WHICHDESC_MSK | UNDER_BA_SETUP_BIT);
    thd->statinfo = 0;
    thd->phyctrlinfo = sta->paid_gid;

    // Get policy table
    pol_tbl = (struct tx_policy_tbl *)HW2CPU(thd->policyentryaddr);

    // Set BW
    pol_tbl->ratecntrlinfo[0] &= ~BW_TX_RCX_MASK;
    pol_tbl->ratecntrlinfo[0] |= ((uint32_t)bfr_env.calib_bw << BW_TX_RCX_OFT);

    // Fill-in the confirmation structure
    frame->cfm.cfm_func = bfr_tx_frame_cfm;
    frame->cfm.env = bfr;

    #if (NX_CHNL_CTXT || NX_P2P)
    // Set VIF and STA indexes
    frame->txdesc.host.vif_idx = bfr->vif_idx;
    frame->txdesc.host.staid   = bfr->sta_idx;
    #endif //(NX_CHNL_CTXT || NX_P2P)

    return (&(frame->txdesc));
}

/**
 ****************************************************************************************
 * @brief Handle upload of a received VHT Beamforming Report in the host memory.
 * If report length exceeds maximum MPDU length, the report shall be split in several segments
 * by the beamformee:
 *      - Up to 8 segments.
 *      - All the segments shall be sent in a single A-MPDU in an ordered way.
 *      - Remaining Feedback Segments and First Feedback Segment subfields of VHT MIMO
 * Control field allows identification of segments.
 * When received in SW a MPDU is described and split over several buffers (and several
 * Payload Descriptors).
 * Hence two levels of fragmentation are handled.
 *
 * bfr_dma_upload_cfm callback will be called after end of segment upload.
 * Once all segments have been uploaded, calibration can be considered as over for the STA.
 *
 * @param[in] bfr           Beamformer information for a given STA.
 * @param[in] rxdesc        Software descriptor containing information about the received
 *                          MPDU.
 * @param[in] frame         Pointer to the received VHT Compressed Beamforming report frame
 *
 * @return A boolean value indicating if the operation has been properly finished and if
 *         the provided software descriptor can be freed.
 *         If true, bfr module will be in charge of freeing the descriptor.
 *         Else it is rxl_cntrl module responsability.
 ****************************************************************************************
 */
static uint8_t bfr_upload_report(struct bfr_info_sta *bfr, struct rxdesc *rxdesc,
                                 uint8_t *frame)
{
    // Returned status, consider the upload is valid
    uint8_t status = BFR_RX_STATUS_VALID;

    // Get MAC HW descriptor
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);

    // Index of the currently unused host report, where the upload is done
    uint16_t report_idx = bfr->host_report_idx ^ 1;
    // Address in host memory for report upload
    uint32_t host_addr = bfr->host_report_addr[report_idx];
    // Keep VHT MIMO Control value, value in frame will be overwritten in order to insert BFR Header
    uint32_t vht_mimo_ctrl = co_read32p(CPU2HW(frame) + BFR_VHT_BFR_VHT_MIMO_CTRL_OFFSET);
    /*
     * Payload offset (for first payload descriptor, no need to copy the MAC Header and needless fields
     * before the report.
     */
    uint16_t payload_offset = BFR_VHT_REPORT_OFFSET;
    // Total length of the received MPDU
    uint16_t upload_len = dma_hdrdesc->hd.frmlen;

    // If report is not a 80MHz report or is not for 2 spatial streams, reject it
    if ((BFR_VHT_MIMO_GET_VALUE(vht_mimo_ctrl, CHAN_WIDTH) != bfr_env.calib_bw)
           /*|| (BFR_VHT_MIMO_GET_VALUE(vht_mimo_ctrl, NC) != 1)*/)
    {
        return (BFR_RX_STATUS_ERROR);
    }

    // For profiling
    PROF_BFR_UPLOAD_SET();

    if (!BFR_VHT_MIMO_GET_VALUE(vht_mimo_ctrl, FIRST_SEGM))
    {
        // Do not overwrite previously uploaded segments
        host_addr += bfr->host_report_len[report_idx];
    }
    else
    {
        // Reset report length
        bfr->host_report_len[report_idx] = 0;
        // And store number of reports to be confirmed
        bfr->nb_segments = 1 + BFR_VHT_MIMO_GET_VALUE(vht_mimo_ctrl, REM_SEGM);

        // Move back payload offset for the BFR Header whose length is 2 bytes
        payload_offset -= 2;

        // Set the BFR Header, match with the first 16 bits of the received VHT MIMO Control value
        co_write16p(CPU2HW(frame) + BFR_VHT_BFR_VHT_MIMO_CTRL_OFFSET + 1,
                    (uint16_t)vht_mimo_ctrl);

        // Keep received Nc
        bfr->last_rx_nc = BFR_VHT_MIMO_GET_VALUE(vht_mimo_ctrl, NC);
    }

    upload_len -= payload_offset;

    bfr->host_report_len[report_idx] += upload_len;
    if (bfr->host_report_len[report_idx] > bfr->host_max_report_len)
    {
        // Finally end calibration for the STA
        bfr_end_calibration_for_sta(bfr, false);

        // Return an error status
        return BFR_RX_STATUS_ERROR;
    }

    // Program upload of report
    rxl_mpdu_partial_transfer(rxdesc, upload_len, host_addr, payload_offset,
                              bfr_dma_upload_cfm, bfr);

    return (status);
}

/**
 ****************************************************************************************
 * @brief Handle download of a beamforming report from the host memory to the PHY memory
 * so that it can be used for the following frame transmissions.
 * It first checks if a report used by the provided STA is already present in the PHY memory
 * and if it can be overwritten.
 * If not, it searches for a new memory node that could contain the report.
 * @param[in] bfr             Beamformer information for a given STA.
 * @param[in] dma_desc_bfr    DMA descriptor to be fulfilled in case download of a report
 *                            from the host memory to the phy memory is required.
 * @return SMM index of the slot in which report has been downloaded and  @ref
 *         BFR_INVALID_SMM_INDEX if no slot has been found.
 ****************************************************************************************
 */
static uint8_t bfr_download_report(struct bfr_info_sta *bfr, struct dma_desc *dma_desc_bfr)
{
    // Look for available memory in the BFR memory
    struct bfr_mem_node *node = NULL;
    // Pointer to the node currently used by the STA
    struct bfr_mem_node *sta_node = bfr->node;
    // Index of the host report that is downloaded
    uint16_t report_idx = bfr->host_report_idx;

    // For profiling
    PROF_BFR_DOWNLOAD_SET();

    // Check if we can quickly reuse the node currently used by the STA or if node is now useless
    if (sta_node)
    {
        do
        {
            if (sta_node->nb_frames)
            {
                // Unlink the node, will be removed once all confirmation are received
                sta_node->sta_idx = INVALID_STA_IDX;
                break;
            }

            // Check if report length has been modified, apply padding for host report length
            if (((bfr->host_report_len[report_idx] & 0xFFFC) + 4) != sta_node->length)
            {
                // Remove the node
                bfr_remove_node(sta_node);
                break;
            }

            // Can directly reuse this node
            node = sta_node;
        } while (0);
    }

    // Check if a new node has to be used and if a new one can be found
    if (!node && (bfr_env.nb_used_nodes < BFR_MEM_NODE_NB))
    {
        node = bfr_get_node(bfr->host_report_len[report_idx]);

        // Check if a free node has been found
        if (node)
        {
            // Link the node with the STA
            bfr->node = node;
            node->sta_idx = bfr->sta_idx;

            // Get and store the available smm_index
            node->smm_index = bfr_get_smm_index(node);

            // Sanity check
            ASSERT_ERR(node->smm_index != BFR_INVALID_SMM_INDEX);
        }
    }

    // Check if free memory has been found for the download
    if (node)
    {
        // Configure the DMA descriptor for the report
        dma_desc_bfr->src = bfr->host_report_addr[report_idx];
        dma_desc_bfr->dest = BFR_MEM_BASE_ADDR + node->start_addr;
        dma_desc_bfr->length = bfr->host_report_len[report_idx];
        dma_desc_bfr->ctrl = 0;
    }

    // For profiling
    PROF_BFR_DOWNLOAD_CLR();

    return ((node) ? node->smm_index : BFR_INVALID_SMM_INDEX);
}

/**
 ****************************************************************************************
 * @brief Starts the next pending SU calibration.
 * The beamformer shall send a Null Data Packet Announcement (NDPA) frame followed by a
 * Null Data Packet (NDP) after SIFS. This frame has a known fixed format. By analysing the
 * NDP, the beamformee calculates a feedback matrix that is sent to the beamformer after SIFS.
 ****************************************************************************************
 */
static void bfr_start_next_calibration(void)
{
    // Device to calibrate with
    struct bfr_info_sta *bfr;
    // Number of devices for which calibration is started (if > 1, MU Calibration is used)
    uint8_t nb_devs;
    // Number of packets transmitted during that calibration procedure
    int i, nb_pkts = 0;
    // Flag indicating if all BFR poll frames have been correctly prepared
    bool bfr_poll_ok;
    // TX Descriptor
    struct txdesc *txdesc;
    // Table of TX descriptor pointers used for the calibration (NDPA+NDP+1 per MU dev)
    struct txdesc *calib_txdesc[2 + RW_USER_MAX];

    do
    {
        // Check that BFR module is not handling another calibration
        if (bfr_env.status != BFR_STATUS_IDLE)
            break;

        // Get first calibration request
        bfr = (struct bfr_info_sta *)co_list_pick(&bfr_env.calib_sched);
        if (!bfr)
            break;

        // Sanity check
        ASSERT_ERR(bfr->sta_idx != INVALID_STA_IDX);

        // Prepare the NDPA frame, will decide if MU or SU calibration has to be performed
        txdesc = bfr_prep_ndpa(bfr, &nb_devs);
        if (!txdesc)
            break;

        // Store the pointer to the TX descriptor in the table
        calib_txdesc[nb_pkts++] = txdesc;

        // Prepare the NDP frame
        txdesc = bfr_prep_ndp(bfr, nb_devs);
        if (!txdesc)
            break;

        // Store the pointer to the TX descriptor in the table
        calib_txdesc[nb_pkts++] = txdesc;

        // Prepare the BFR poll frames
        bfr_poll_ok = true;
        for (i = 1; i < nb_devs; i++)
        {
            // No BFR poll sent to the first STA so go directly to next one
            bfr = (struct bfr_info_sta *)co_list_next(&bfr->hdr);
            ASSERT_ERR(bfr != NULL);

            // Prepare the BFR poll
            txdesc = bfr_prep_bfr_poll(bfr);
            if (!txdesc)
            {
                bfr_poll_ok = false;
                break;
            }

            // Store the pointer to the TX descriptor in the table
            calib_txdesc[nb_pkts++] = txdesc;
        }

        // Check if all necessary BFR poll were prepared
        if (!bfr_poll_ok)
            break;

        // Check whether we are starting a MU calibration or not
        if (nb_devs > 1)
        {
            // For profiling
            PROF_BFR_MU_CALIB_SET();

            // Set number of expected reports
            bfr_env.nb_reports = nb_devs;

            // Update BFR Environment status
            bfr_env.status = BFR_STATUS_MU_CALIB;
        }
        else
        {
            // For profiling
            PROF_BFR_SU_CALIB_SET();

            // Update BFR Environment status
            bfr_env.status = BFR_STATUS_SU_CALIB;
        }

        for (i = 0; i < nb_devs; i++)
        {
            // Get device information from the lists of pending calibrations
            bfr = (struct bfr_info_sta *)co_list_pop_front(&bfr_env.calib_sched);
            co_list_push_back(&bfr_env.bfee_sta_list, &bfr->hdr);

            // Keep in mind we are calibrating for this STA
            BFR_SET_STATUS_BIT(bfr->status, BFR_STA_CALIBRATE);
        }

        /*
         * Frames are pushed to the MACHW only after a DMA transfer interrupt. This critical
         * section allows to be sure to have enough time for chaining NDPA, NDP and BFR
         * poll packets without any HW fetching.
         */
        GLOBAL_INT_DISABLE();
        for (i = 0; i < nb_pkts; i++)
        {
            txl_cntrl_push_int(calib_txdesc[i], AC_VO);
        }
        GLOBAL_INT_RESTORE();

        // Calibration is successfully started, return
        return;
    } while (0);

    // If we fall into this part of code it means that some of the TX packet
    // preparation has failed
    // Free allocated frames
    for (i = 0; i < nb_pkts; i++)
    {
        // Push the NDPA frame for TX
        txl_frame_release(calib_txdesc[i], false);
    }
}

/**
 ****************************************************************************************
 * @brief Go through all connected beamformee-capable STAs and check if start of a new
 * calibration is required for this connection.
 ****************************************************************************************
 */
static void bfr_calibrate(void)
{
    // Get current time
    uint32_t current_time = ke_time();
    // Pointer to the first SU-Beamformee device inserted
    struct bfr_info_sta *su_bfr = NULL;
    // Get first STA to calibrate
    struct bfr_info_sta *bfr = (struct bfr_info_sta *)co_list_pick(&bfr_env.bfee_sta_list);

    // Go through all BFR-oriented STA information not yet scheduled for calibration
    while (bfr)
    {
        // Get next STA
        struct bfr_info_sta *next_bfr = (struct bfr_info_sta *)bfr->hdr.next;

        do
        {
            // Check if calibration not already in progress for this device
            if (BFR_IS_STATUS_BIT_SET(bfr->status, BFR_STA_CALIBRATE))
            {
                break;
            }

            /*
             * If traffic has been detected or if last
             * calibration has been performed more than BFR_CALIB_MAX_DELAY us ago, start a
             * calibration with the STA.
             */
            if (!td_sta_has_tx_traffic(bfr->sta_idx))
            {
                if (ke_time_abs_cmp(current_time, bfr->last_calib_time + BFR_CALIB_MAX_DELAY))
                {
                    break;
                }
            }

            #if (NX_UMAC_PRESENT)
            // Don't calibrate stations that are sleeping
            if (sta_mgmt_is_in_ps(bfr->sta_idx))
            {
                break;
            }
            #endif //(NX_UMAC_PRESENT)

            // Remove from list of not scheduled beamformee-capable STAs
            co_list_extract(&bfr_env.bfee_sta_list, &bfr->hdr);

            /*
             * Insert the device in the list of pending calibrations. We group device that are MU
             * Beamformee capable together. These devices are inserted before the list of device
             * which are SU-Beamformee only.
             */
            if (!BFR_IS_STATUS_BIT_SET(bfr->status, BFR_STA_MU_CAPABLE))
            {
                if (!su_bfr)
                {
                    su_bfr = bfr;
                }

                // Add the device to the list of devices to calibrate
                co_list_push_back(&bfr_env.calib_sched, &bfr->hdr);
            }
            else
            {
                // Insert before SU-Beamformee only device
                co_list_insert_before(&bfr_env.calib_sched, (struct co_list_hdr *)su_bfr,
                                      &bfr->hdr);
            }
        } while (0);

        // Get next STA
        bfr = next_bfr;
    }

    // Start next calibration if possible
    bfr_start_next_calibration();
}

/**
 ****************************************************************************************
 * @brief Reset information stored in struct bfr_info_sta element allocated for a STA
 * @param[in] bfr         BFR Information for a given STA
 ****************************************************************************************
 */
static void bfr_init_sta_info(struct bfr_info_sta *bfr)
{
    // Fully reset BFR information for the STA
    memset(bfr, 0, sizeof(struct bfr_info_sta));

    // Reset STA index
    bfr->sta_idx = INVALID_STA_IDX;
}

/**
 ****************************************************************************************
 * @brief Initialize index table in BFR Report Memory
 ****************************************************************************************
 */
static void bfr_init_index_table(void)
{
    for (int i = 0; i < (BFR_MEM_REP_NB_MAX / 2); i++)
    {
        uint32_t *bfr_ptr = HW2CPU(BFR_MEM_BASE_ADDR + (i * 4));

        *bfr_ptr = 0;
    }
}

/*
 * PUBLIC FUNCTION DEFINITIONS
 ****************************************************************************************
 */

void bfr_init(void)
{
    struct bfr_mem_node *node;

    // Fully reset the BFR environment variable
    memset(&bfr_env, 0, sizeof(bfr_env));

    // Verify we can be beamformer
    bfr_env.bfr_enabled = hal_machw_bfmer_support();

    if (bfr_is_enabled())
    {
        // Read memory size in a number of bytes
        bfr_env.bfr_mem_eaddr = (phy_get_bfr_mem_size() << 10);

        // Fully reset content of the memory nodes
        memset(&bfr_nodes[0], 0, BFR_MEM_NODE_NB * sizeof(struct bfr_mem_node));

        // Initialize list of available memory nodes
        for (int i = 0; i < BFR_MEM_NODE_NB; i++)
        {
            // Get node
            node = &bfr_nodes[i];

            // Set the node index
            node->idx = i;
            // Insert the node
            co_list_push_back(&bfr_env.free_nodes, &node->hdr);
        }

        // Initialize BFR information stored for each STA
        for (int i = 0; i < NX_REMOTE_STA_MAX; i++)
        {
            bfr_init_sta_info(&bfr_sta_infos[i]);
        }

        // Initialize SMM Pointers values
        for (int i = 0; i < BFR_MEM_REP_NB_MAX; i++)
        {
            bfr_smm_ptrs[i] = 0;
        }

        // Initialize callback for calibration timer
        bfr_env.calib_tmr.cb = bfr_calib_tmr_exp_cb;
        // Set calibration timer duration
        bfr_env.calib_tmr_dur = BFR_CALIB_TMR_DFLT_DUR;

        // Initialize Index Table in BFR memory
        bfr_init_index_table();

        // Get our number of spatial stream
        bfr_env.bfer_nss = phy_get_nss();

        PROF_SMM_IDX_SET(0xF);
    }
    else
    {
        dbg(D_CRT "Disable Beamformer as MAC and/or PHY does not support it\n");
    }
}

void bfr_add_sta_ind(uint8_t sta_idx, bool mu_capable, uint16_t aid, uint32_t host_addr,
                     uint16_t host_length, uint8_t nss)
{
    // Get VIF Info
    struct vif_info_tag *vif = &vif_info_tab[sta_mgmt_get_vif_idx(sta_idx)];
    uint8_t band = vif_mgmt_get_band(vif);

    // Check if channel used for the connection is a 5GHz channel
    if (band == PHY_BAND_5G)
    {
        // Get BFR information for this STA
        struct bfr_info_sta *bfr = &bfr_sta_infos[sta_idx];

        // Set BFR parameters
        bfr->sta_idx = sta_idx;
        bfr->host_max_report_len = host_length/2;
        bfr->host_report_addr[0] = host_addr;
        bfr->host_report_addr[1] = host_addr + bfr->host_max_report_len;
        bfr->host_report_idx = 0;
        bfr->bfee_nss = nss;
        bfr->vif_idx = sta_mgmt_get_vif_idx(sta_idx);
        bfr->last_calib_time = hal_machw_time();

        if (vif->type == VIF_AP)
        {
            BFR_STA_INFO_SET_AID12(bfr->sta_info, aid);
        }
        else
        {
            // Force AID to 0
            BFR_STA_INFO_SET_AID12(bfr->sta_info, 0);
        }

        if (mu_capable)
        {
            BFR_SET_STATUS_BIT(bfr->status, BFR_STA_MU_CAPABLE);
        }

        // If first STA, start calibration timer
        if (co_list_is_empty(&bfr_env.bfee_sta_list) && co_list_is_empty(&bfr_env.calib_sched))
        {
            mm_timer_set(&bfr_env.calib_tmr, hal_machw_time() + bfr_env.calib_tmr_dur);
        }

        // Add STA in list of beamformee capable STAs
        co_list_push_back(&bfr_env.bfee_sta_list, &bfr->hdr);
    }
}

void bfr_del_sta_ind(uint8_t sta_idx)
{
    // Get BFR information for this STA
    struct bfr_info_sta *bfr = &bfr_sta_infos[sta_idx];

    do
    {
        if (bfr->sta_idx == INVALID_STA_IDX)
        {
            break;
        }

        // Remove the STA from its list (either bfee_sta_list or calib_sched)
        co_list_extract(&bfr_env.bfee_sta_list, &bfr->hdr);
        co_list_extract(&bfr_env.calib_sched, &bfr->hdr);

        // Check if we were calibrating
        if (BFR_IS_STATUS_BIT_SET(bfr->status, BFR_STA_CALIBRATE))
        {
            bfr_end_calibration_for_sta(bfr, false);
        }

        // Stop the calibration timer if no more STA to interact with
        if (co_list_is_empty(&bfr_env.bfee_sta_list) && co_list_is_empty(&bfr_env.calib_sched))
        {
            mm_timer_clear(&bfr_env.calib_tmr);
        }

        if (bfr->node)
        {
            if (bfr->node->nb_frames)
            {
                bfr->node->sta_idx = INVALID_STA_IDX;
            }
            else
            {
                // Remove the node
                bfr_remove_node(bfr->node);
            }
        }

        // Initialize beamforming structure
        bfr_init_sta_info(bfr);
    } while (0);
}

uint8_t bfr_rx_frame_ind(uint8_t sta_idx, struct rxdesc *rxdesc, uint8_t *frame)
{
    // Returned status, consider frame is not a VHT Compressed Beamforming frame by default
    uint8_t status = BFR_RX_STATUS_NOT_VALID;

    do
    {
        // Get BFR information for this STA
        struct bfr_info_sta *bfr = &bfr_sta_infos[sta_idx];
        // VHT Mimo Control
        uint32_t vht_mimo_ctrl;

        // Check if STA is beamformee capable
        if (bfr->sta_idx == INVALID_STA_IDX)
        {
            break;
        }

        // Check if calibration is in progress from this STA
        if (!BFR_IS_STATUS_BIT_SET(bfr->status, BFR_STA_CALIBRATE))
        {
            break;
        }

        // Check if frame is a VHT Compressed Beamforming Report
        if (!bfr_is_vht_bfr(frame))
        {
            break;
        }

        // For profiling
        PROF_BFR_RX_BFR_SET();

        vht_mimo_ctrl = co_read32p(CPU2HW(frame) + BFR_VHT_BFR_VHT_MIMO_CTRL_OFFSET);

        // Check if received dialog token is the expected one
        if (bfr->last_token != BFR_VHT_MIMO_GET_VALUE(vht_mimo_ctrl, TOKEN))
        {
            // End calibration
            bfr_end_calibration_for_sta(bfr, false);
            break;
        }

        // BFR module is now responsible for SW Descriptor
        status = bfr_upload_report(bfr, rxdesc, frame);

        if (status == BFR_RX_STATUS_ERROR)
        {
            // End calibration
            bfr_end_calibration_for_sta(bfr, false);
        }

        // For profiling
        PROF_BFR_RX_BFR_CLR();
    } while (0);

    return (status);
}

bool bfr_tx_frame_ind(struct txdesc *txdesc, struct dma_desc *dma_desc_bfr)
{
    // Get BFR information for this STA
    struct bfr_info_sta *bfr;
    // Indicate if a report have to be downloaded from host memory
    bool download = false;
    // SMM Index
    uint8_t smm_index = BFR_INVALID_SMM_INDEX;

    do
    {
        if (txdesc->host.staid == INVALID_STA_IDX)
        {
            break;
        }

        bfr = &bfr_sta_infos[txdesc->host.staid];

        // Check if STA is beamformee capable
        if (bfr->sta_idx == INVALID_STA_IDX)
        {
            break;
        }

        // Check if frame can be beamformed
        if (!bfr_is_frame_bf_compliant(txdesc, bfr))
        {
            break;
        }

        // For profiling
        PROF_BFR_TX_IND_SET();

        // Check if STA report is already present in phy memory
        if (bfr->node)
        {
            /*
             * Check if a new report has been received.
             * In that case, do not use current report but try to download a new one
             */
            if (!BFR_IS_STATUS_BIT_SET(bfr->status, BFR_STA_REPORT_UPDATED))
            {
                struct bfr_mem_node *node = bfr->node;

                smm_index = node->smm_index;

                // Increase the number of packets handling
                bfr_increase_nb_frames(bfr->node);

                // Link the BFR Node with the TX Descriptor so that we can decrease the nb_frames value upon confirmation
                txdesc->lmac.bfr_node = bfr->node;
                break;
            }
        }

        // Check if report can be downloaded from host memory
        if (!BFR_IS_STATUS_BIT_SET(bfr->status, BFR_STA_HAS_VALID_REPORT))
        {
            break;
        }

        // Try to download the report from host memory
        smm_index = bfr_download_report(bfr, dma_desc_bfr);

        if (smm_index != BFR_INVALID_SMM_INDEX)
        {
            // If a new report has been downloaded, clear the update bit
            BFR_RESET_STATUS_BIT(bfr->status, BFR_STA_REPORT_UPDATED);

            download = true;

            // Increase the number of packets handling
            bfr_increase_nb_frames(bfr->node);

            // Link the BFR Node with the TX Descriptor so that we can decrease the nb_frames value upon confirmation
            txdesc->lmac.bfr_node = bfr->node;
        }
    } while (0);

    // For profiling
    PROF_BFR_TX_IND_CLR();
    PROF_SMM_IDX_SET(smm_index);

    return (download);
}

void bfr_tx_cfm(struct txdesc *txdesc)
{
    do
    {
        // Get node used for the transmission
        struct bfr_mem_node *node = txdesc->lmac.bfr_node;

        if (!node)
        {
            // Beamforming has not been used for this transmission
            break;
        }

        // For profiling
        PROF_BFR_TX_CFM_SET();

        // Critical section because nb_frames and nb_used_nodes are incremented
        // in bfr_tx_frame_ind() which is called under interrupt
        GLOBAL_INT_DISABLE();

        // Sanity check
        ASSERT_WARN(node->nb_frames);

        // Decrease number of frames to be handled
        node->nb_frames--;

        if (!node->nb_frames)
        {
            bfr_env.nb_used_nodes--;
        }

        /*
         * If node is no more linked with a STA and no more frame has to be sent with this report,
         * free the part of memory
         */
        if ((node->sta_idx == INVALID_STA_IDX) && (!node->nb_frames))
        {
            bfr_remove_node(node);
        }

        GLOBAL_INT_RESTORE();

        // For profiling
        PROF_BFR_TX_CFM_CLR();
    } while (0);
}

#if (NX_UMAC_PRESENT)
uint8_t bfr_get_last_nc(uint8_t sta_idx)
{
    // Get BFR information for this STA
    struct bfr_info_sta *bfr = &bfr_sta_infos[sta_idx];
    // NC
    uint8_t nc = BFR_INVALID_NC;

    do
    {
        // Check if STA is beamformee capable
        if (bfr->sta_idx == INVALID_STA_IDX)
        {
            break;
        }

        nc = bfr->last_rx_nc;
    } while (0);

    return (nc);
}
#endif //(NX_UMAC_PRESENT)

#if RW_MUMIMO_TX_EN
/**
 ****************************************************************************************
 * @brief Handle confirmation of the group update frame transmitted to the peer STA
 *
 * @param[in] env     Has been configured so that value is the address of the parameter
 *                    pointer of the group update request sent by the host.
 * @param[in] status  Transmission status provided by the HW.
 ****************************************************************************************
 */
static void bfr_group_update_cfm(void *env, uint32_t status)
{
    // Cast the environment
    struct mm_mu_group_update_req const *param = env;
    struct ke_msg *msg = ke_param2msg(param);

    // Send the confirmation to the upper layers
    ke_msg_send_basic(MM_MU_GROUP_UPDATE_CFM, msg->src_id, msg->dest_id);

    // Free the message structure
    ke_msg_free(msg);
}

void bfr_group_update_req(struct mm_mu_group_update_req const *param)
{
    uint8_t sta_idx = param->sta_idx;
    struct txl_frame_desc_tag *frame;
    struct mac_hdr *buf;
    struct tx_hd *thd;
    uint32_t length = 0;
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];
    uint32_t payload;
    int txtype;
    int i;

    // Allocate a frame descriptor from the TX path
    txtype = vif_mgmt_get_txtype(vif);
    frame = txl_frame_get(txtype, NX_TXFRAME_LEN);
    if (frame == NULL)
    {
        bfr_group_update_cfm((void *)param, 0);
        return;
    }

    // update Tx power in policy table
    tpc_update_frame_tx_power(vif, frame);

    // Get the buffer pointer
    buf = txl_buffer_payload_get(&frame->txdesc);

    // Fill-in the frame
    buf->fctl  = MAC_FCTRL_ACTION;
    // TODO[LT] - Set duration
    buf->durid = 0;
    // Set RA
    MAC_ADDR_CPY(&buf->addr1, sta_mgmt_get_peer_addr(sta_idx));
    // Set TA
    MAC_ADDR_CPY(&buf->addr2, vif_mgmt_get_addr(sta->inst_nbr));
    // Set BSSID
    MAC_ADDR_CPY(&buf->addr3, vif_mgmt_get_addr(sta->inst_nbr));
    buf->seq = txl_get_seq_ctrl();

    // Fill-in the confirmation structure
    frame->cfm.cfm_func = bfr_group_update_cfm;
    frame->cfm.env = (void *)param;

    #if (NX_CHNL_CTXT || NX_P2P)
    // Set VIF and STA indexes
    frame->txdesc.host.vif_idx = sta->inst_nbr;
    frame->txdesc.host.staid   = sta_idx;
    #endif //(NX_CHNL_CTXT || NX_P2P)

    length = MAC_SHORT_MAC_HDR_LEN;

    #if NX_MFP
    frame->txdesc.umac.head_len = 0;
    frame->txdesc.umac.tail_len = 0;
    if (MFP_UNICAST_PROT == mfp_protect_mgmt_frame(&frame->txdesc, buf->fctl,
                                                   MAC_VHT_ACTION_CATEGORY))
    {
        txu_cntrl_protect_mgmt_frame(&frame->txdesc, buf, MAC_SHORT_MAC_HDR_LEN);
        length += frame->txdesc.umac.head_len;
    }
    #endif

    payload = CPU2HW(buf) + length;
    length += MAC_GROUP_ID_MGT_PAYL_LENGTH;

    co_write8p(payload, MAC_VHT_ACTION_CATEGORY);
    co_write8p(payload + MAC_ACTION_ACTION_OFT, MAC_GROUP_ID_MGMT_VHT_ACTION);

    // Erase the membership status array
    for (i = 0; i < 8; i++)
    {
        co_write8p(payload + MAC_GROUP_ID_MGT_MEMBERSHIP_OFT + i, 0);
    }

    // Erase the user position array
    for (i = 0; i < 16; i++)
    {
        co_write8p(payload + MAC_GROUP_ID_MGT_USER_POS_OFT + i, 0);
    }

    // Set the correct bits in the membership status array and user position
    for (i = 0; i < param->group_cnt; i++)
    {
        int group_id = param->groups[i].group_id;
        uint8_t user_pos = param->groups[i].user_pos;
        int group_byte_idx = group_id / 8;
        int group_bit = group_id % 8;
        int user_byte_idx = group_id / 4;
        int user_bit = (group_id % 4) * 2;
        uint32_t membership_addr = payload + MAC_GROUP_ID_MGT_MEMBERSHIP_OFT + group_byte_idx;
        uint32_t userpos_addr = payload + MAC_GROUP_ID_MGT_USER_POS_OFT + user_byte_idx;

        co_write8p(membership_addr, co_read8p(membership_addr) | (1 << group_bit));
        co_write8p(userpos_addr, co_read8p(userpos_addr) | (user_pos << user_bit));
    }

    #if NX_MFP
    length += frame->txdesc.umac.tail_len;
    #endif
    thd             = &frame->txdesc.lmac.hw_desc->thd;
    thd->dataendptr = (uint32_t)thd->datastartptr + length - 1;
    thd->frmlen     = length + MAC_FCS_LEN;

    // Push the frame for TX
    txl_frame_push(frame, AC_VO);
}
#endif //RW_MUMIMO_TX_EN

#endif //(RW_BFMER_EN)

/// @}
