/**
 ****************************************************************************************
 *
 * @file chan.c
 *
 * @brief MAC Management channel management implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup CHAN
 *
 * The CHAN module is used to switch the current operating channel at runtime.
 * This is not related to the CSA (Channel switch Annoucement) procedure.
 * This is needed during a SCAN, a Remain on Channel (ROC) or when multiple vifs are
 * active on different operating channels.
 *
 * @{
 ****************************************************************************************
 */

#include "chan.h"
#include "mac_frame.h"
#include "mm.h"
#include "mm_timer.h"
#include "mm_bcn.h"
#include "co_endian.h"
#include "sta_mgmt.h"
#include "vif_mgmt.h"
#include "phy.h"
#include "dbg.h"
#include "rd.h"
#include "ps.h"
#include "txl_cntrl.h"
#include "txl_frame.h"
#include "rxl_hwdesc.h"
#include "hal_machw.h"
#include "td.h"
#include "scan.h"
#include "tpc.h"
#if (NX_UMAC_PRESENT)
#include "me.h"
#endif //(NX_UMAC_PRESENT)
#if (NX_MDM_VER >= 20)
#include "reg_riu.h"
#endif //(NX_MDM_VER >= 20)
#include "tdls.h"
#if (NX_P2P)
#include "p2p.h"
#endif //(NX_P2P)
#if NX_MAC_HE
#include "txl_he.h"
#endif


#if (NX_CHNL_CTXT)

/*
 * DEFINES
 ****************************************************************************************
 */
/// Maximum number of scheduled switch
#define CHAN_SWITCH_CNT  4

/// Minimal presence duration on a channel (in us)
#define CHAN_MIN_PRES_DUR_US (5000)

/// Delay before switching on a Scan or a ROC channel, indicate minimal duration between
/// two successive scan/roc operations
#define CHAN_CONN_LESS_DELAY   (30000)

/// Flags indicating that a scan ROC is pending
#define CHAN_ROC_SCAN_PENDING_MASK  (CO_BIT(CHAN_ENV_ROC_BIT) | CO_BIT(CHAN_ENV_SCAN_BIT))

/// Drift threshold of STA/P2P-CLI TBTT to re-schedule NOA, in us
#define CHAN_P2P_NOA_RESCHEDULE_THRESHOLD 2000

/// Drift threshold of STA/P2P-CLI TBTT to re-sync P2P GO TBTT with NOA, in us
#define CHAN_P2P_NOA_RESYNC_THRESHOLD 200

/// Internal structure to represent window
struct chan_window
{
    /// Start of window (timestamp in us)
    uint32_t start;
    /// End of window (timestamp in us)
    uint32_t end;
};

/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */

/// MM Channel module context variable
struct chan_env_tag chan_env;

/// Pool of channel contexts
struct chan_ctxt_tag chan_ctxt_pool[CHAN_CHAN_CTXT_CNT];

/// Pool of channel switch descriptor
struct chan_switch_tag chan_switch_pool[CHAN_SWITCH_CNT];

/*
 * PRIVATE FUNCTIONS DECLARATION
 ****************************************************************************************
 */

static void chan_switch_start(struct chan_ctxt_tag *ctxt);
static void chan_switch_channel(void);

/*
 * PRIVATE FUNCTIONS DEFINITION
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Configure timer for next channel switch.
 *
 * @param[in] ch_switch Channel switch information
 ****************************************************************************************
 */
__INLINE void chan_switch_set_timer(struct chan_switch_tag *ch_switch)
{
    chan_env.tmr_switch.env = ch_switch;
    if (ch_switch)
    {
        if (hal_machw_time_past(ch_switch->time - 20))
            chan_env.tmr_switch.cb(ch_switch);
        else
            mm_timer_set(&chan_env.tmr_switch, ch_switch->time);
    }
}

/**
 ****************************************************************************************
 * @brief Cancel timer for channel switch.
 ****************************************************************************************
 */
__INLINE void chan_switch_reset_timer(void)
{
    mm_timer_clear(&chan_env.tmr_switch);
    chan_env.tmr_switch.env = NULL;
}

/**
 ****************************************************************************************
 * @brief Extract first element of scheduled channel switch list.
 *
 * @return Pointer to next scheduled channel switch information (can be NULL)
 ****************************************************************************************
 */
__INLINE struct chan_switch_tag *chan_switch_pop(void)
{
    return (struct chan_switch_tag *)co_list_pop_front(&chan_env.list_switch);
}

/**
 ****************************************************************************************
 * @brief Get first element of scheduled channel switch list.
 *
 * @return Pointer to next scheduled channel switch information (can be NULL)
 ****************************************************************************************
 */
__INLINE struct chan_switch_tag *chan_switch_pick(void)
{
    return (struct chan_switch_tag *)co_list_pick(&chan_env.list_switch);
}

/**
 ****************************************************************************************
 * @brief Get last element of scheduled channel switch list.
 *
 * @return Pointer to last scheduled channel switch information (can be NULL)
 ****************************************************************************************
 */
__INLINE struct chan_switch_tag *chan_switch_last(void)
{
    return (struct chan_switch_tag *)co_list_pick_last(&chan_env.list_switch);
}

/**
 ****************************************************************************************
 * @brief Insert on element at the end of the scheduled channel switch list.
 *
 * @param[in] ch_switch Channel switch information
 ****************************************************************************************
 */
__INLINE void chan_switch_push_back(struct chan_switch_tag *ch_switch)
{
    co_list_push_back(&chan_env.list_switch, &ch_switch->list_hdr);
}

/**
 ****************************************************************************************
 * @brief Check is scheduled channel switch list is empty.
 *
 * @return true is no channel switch is scheduled and false otherwise.
 ****************************************************************************************
 */
__INLINE bool chan_switch_no_schedule(void)
{
    return co_list_is_empty(&chan_env.list_switch);
}

/**
 ****************************************************************************************
 * @brief Allocate a channel switch information element.
 *
 * @return Pointer to allocated channel switch information element
 ****************************************************************************************
 */
__INLINE struct chan_switch_tag *chan_switch_alloc(void)
{
    struct chan_switch_tag *ch_switch;
    ch_switch = (struct chan_switch_tag *)co_list_pop_front(&chan_env.list_free_switch);
    ASSERT_ERR(ch_switch != NULL);
    return ch_switch;
}

/**
 ****************************************************************************************
 * @brief Free a channel switch information element.
 *
 * @param[in] ch_switch Channel switch information to free
 ****************************************************************************************
 */
__INLINE void chan_switch_free(struct chan_switch_tag *ch_switch)
{
    co_list_push_back(&chan_env.list_free_switch, &ch_switch->list_hdr);
}

/**
 ****************************************************************************************
 * @brief Get first element of tbtt list.
 *
 * @return Pointer to next TBTT (can be NULL).
 ****************************************************************************************
 */
__INLINE struct chan_tbtt_tag *chan_tbtt_pick(void)
{
    return (struct chan_tbtt_tag *)co_list_pick(&chan_env.list_tbtt);
}

/**
 ****************************************************************************************
 * @brief Get next element of tbtt list.
 *
 * @param[in] tbtt TBTT.
 * @return Pointer to next TBTT (can be NULL).
 ****************************************************************************************
 */
__INLINE struct chan_tbtt_tag *chan_tbtt_next(struct chan_tbtt_tag *tbtt)
{
    return (struct chan_tbtt_tag *)tbtt->list_hdr.next;
}

/**
 ****************************************************************************************
 * @brief Remove TBTT form the list.
 *
 * @param[in] tbtt TBTT element to remove from the list.
 ****************************************************************************************
 */
__INLINE void chan_tbtt_remove(struct chan_tbtt_tag * tbtt)
{
    co_list_extract(&chan_env.list_tbtt, &tbtt->list_hdr);
    tbtt->status &= ~CHAN_TBTT_PROG;
}

/**
 ****************************************************************************************
 * @brief Insert TBTT before another TBTT.
 *
 * @param[in] tbtt      TBTT to add in the list.
 * @param[in] tbtt_next TBTT before which new element must be added.
 ****************************************************************************************
 */
__INLINE void chan_tbtt_insert_before(struct chan_tbtt_tag *tbtt,
                                      struct chan_tbtt_tag *tbtt_next)
{
    co_list_insert_before(&chan_env.list_tbtt, &tbtt_next->list_hdr, &tbtt->list_hdr);
    tbtt->status |= CHAN_TBTT_PROG;
}

/**
 ****************************************************************************************
 * @brief Get TBTT interval for a vif.
 *
 * @param[in] vif Vif structure
 * @return TBTT interval in us.
 ****************************************************************************************
 */
static uint32_t chan_vif_tbtt_intv(struct vif_info_tag *vif)
{
    if (vif->type == VIF_STA)
    {
        return sta_info_tab[vif->u.sta.ap_id].bcn_int;
    }
    #if NX_BCN_AUTONOMOUS_TX
    else if (vif->type == VIF_AP)
    {
        return vif->u.ap.bcn_int << 10;
    }
    #endif

    TRACE_CHAN(ERR, "TODO: chan_vif_tbtt_intv");
    return 100 * 1024;
}

/**
 ****************************************************************************************
 * @brief Get TBTT duration for a vif.
 *
 * For now fixed to @ref VIF_MGMT_BCN_TO_DUR.
 *
 * @param[in] vif Vif structure
 * @return TBTT duration in us.
 ****************************************************************************************
 */
static uint32_t chan_vif_tbtt_dur(struct vif_info_tag *vif)
{
    return VIF_MGMT_BCN_TO_DUR;
}

/**
 ****************************************************************************************
 * @brief Update the status of a provided channel context.
 *
 * @param[in] ctxt     Channel Context entry
 * @param[in] status   New status of the provided channel context
 ****************************************************************************************
 */
static void chan_upd_ctxt_status(struct chan_ctxt_tag *ctxt, uint8_t status)
{
    ctxt->status = status;
    TRACE_CHAN(STATUS, "{CTXT-%d} Update status %d", ctxt->idx, status);
}

/**
 ****************************************************************************************
 * @brief Get pointer to first active traffic channel context.
 *
 * @return pointer to first active traffic channel context (can be NULL).
 ****************************************************************************************
 */
static struct chan_ctxt_tag *chan_get_first_traf_ctxt(void)
{
    int idx;

    for (idx = 0; idx < CHAN_TRAF_CTXT_CNT; idx++)
    {
        if (chan_ctxt_pool[idx].idx != CHAN_CTXT_UNUSED)
            return &chan_ctxt_pool[idx];
    }

    return NULL;
}

/**
 ****************************************************************************************
 * @brief Get pointer to another traffic channel context.
 *
 * Return pointer to the next active channel context associated to a vif, starting for
 * channel context passed in parameter.
 *
 * @param[in] ctxt Channel context
 * @return pointer to another traffic channel context (can be NULL).
 ****************************************************************************************
 */
static struct chan_ctxt_tag *chan_get_next_traf_ctxt(struct chan_ctxt_tag *ctxt)
{
    int nb = CHAN_TRAF_CTXT_CNT - 1;
    int idx = ctxt->idx + 1;

    while (nb > 0)
    {
        if (idx == CHAN_TRAF_CTXT_CNT)
            idx = 0;

        if (chan_ctxt_pool[idx].idx != CHAN_CTXT_UNUSED)
            return &chan_ctxt_pool[idx];

        nb--;
        idx++;
    }

    return NULL;
}

/**
 ****************************************************************************************
 * @brief Send CHANNEL_PRE_SWITCH_IND message to the upper layers.
 *
 * Host should stop pushing packets for transmission after reception of this message.
 *
 * @param[in] old_ctxt  Channel context we are about to leave
 ****************************************************************************************
 */
static void chan_send_pre_switch_ind(struct chan_ctxt_tag *old_ctxt)
{
    struct mm_channel_pre_switch_ind *ind;

    if (old_ctxt->idx == CHAN_SCAN_CTXT_IDX)
        return;

    ind = KE_MSG_ALLOC(MM_CHANNEL_PRE_SWITCH_IND, TASK_API, TASK_MM,
                       mm_channel_pre_switch_ind);

    ind->chan_index = old_ctxt->idx;

    ke_msg_send(ind);
}

/**
 ****************************************************************************************
 * @brief Send CHANNEL_SWITCH_IND message to the upper layers.
 *
 * Host can start pushing packets for transmission after reception of this message.
 *
 * @param[in] new_ctxt  Channel context on which we have switched
 ****************************************************************************************
 */
static void chan_send_switch_ind(struct chan_ctxt_tag *new_ctxt)
{
    struct mm_channel_switch_ind *ind;

    if (new_ctxt->idx == CHAN_SCAN_CTXT_IDX)
        return;

    ind = KE_MSG_ALLOC(MM_CHANNEL_SWITCH_IND, TASK_API,
                       TASK_MM, mm_channel_switch_ind);

    ind->chan_index = new_ctxt->idx;
    // If ROC triggered internally, do not notify driver
    #if NX_UMAC_PRESENT && NX_TDLS
    ind->roc = ((new_ctxt->idx == CHAN_ROC_CTXT_IDX) && (new_ctxt->taskid != TASK_MM) && (new_ctxt->taskid != TASK_TDLS));
    ind->roc_tdls = new_ctxt->roc_tdls;
    #else
    ind->roc = ((new_ctxt->idx == CHAN_ROC_CTXT_IDX) && (new_ctxt->taskid != TASK_MM));
    ind->roc_tdls = 0;
    #endif
    ind->vif_index = new_ctxt->vif_index;

    ke_msg_send(ind);
}

#if (NX_HW_SCAN)
/**
 ****************************************************************************************
 * @brief Send a channel survey to the host by using the MM_CHANNEL_SURVEY_IND message.
 *
 ****************************************************************************************
 */
static void chan_send_survey_ind(void)
{
    struct mac_chan_def const *scan_chan = scan_get_chan();
    struct mm_channel_survey_ind *ind = KE_MSG_ALLOC(MM_CHANNEL_SURVEY_IND,
                                                     TASK_API, TASK_MM,
                                                     mm_channel_survey_ind);
    #if (NX_MDM_VER >= 20)
    uint8_t read_counter = 10;
    #endif //(NX_MDM_VER >= 20)

    ind->freq = scan_chan->freq;
    ind->chan_time_ms = ((scan_chan->flags & CHAN_NO_IR) ? SCAN_PASSIVE_DURATION : SCAN_ACTIVE_DURATION) / 1000;
    // Time values read in register are in units of us (-> /1000)
    ind->chan_time_busy_ms = nxmac_edca_cca_busy_get() / 1000;
    ind->noise_dbm = 0;

    #if (NX_MDM_VER >= 20)
    // If read noise value is 0, value is not valid because of re-sync between AGC clock
    // and AHB clock. Register value is updated once AGC is in listen state.
    while (read_counter--)
    {
        // During scan, config used is 1x1, noise has to be read to antenna 0
        int8_t reg_noise = (int8_t)riu_inbdpow20pnoisedbm0_getf();

        if (reg_noise != 0)
        {
            ind->noise_dbm = reg_noise;
            break;
        }
    }
    #endif //(NX_MDM_VER >= 20)

    ke_msg_send(ind);
}
#endif //(NX_HW_SCAN)

/**
 ****************************************************************************************
 * @brief Send MM_REMAIN_ON_CHANNEL_EXP_IND message to the upper layers.
 *
 * It informs the host that the remain on channel has expired.
 *
 * @param[in] ctxt Pointer to the RoC Channel Context
 ****************************************************************************************
 */
static void chan_send_roc_exp_ind(struct chan_ctxt_tag *ctxt)
{
    struct mm_remain_on_channel_exp_ind *ind = KE_MSG_ALLOC(MM_REMAIN_ON_CHANNEL_EXP_IND,
                                                            ctxt->taskid, TASK_MM,
                                                            mm_remain_on_channel_exp_ind);

    ind->chan_ctxt_index = ctxt->idx;
    ind->vif_index       = ctxt->vif_index;

    ke_msg_send(ind);
}

#if (NX_P2P_GO)
/**
 ****************************************************************************************
 * @brief Test if a vif is a P2P GO
 *
 * @param[in]  vif VIF Entry
 * @return true is vif is a P2P GO interface and false otherwise
 ****************************************************************************************
 */
__INLINE bool chan_is_p2p_go(struct vif_info_tag *vif)
{
    return ((vif->p2p) &&
            (p2p_info_tab[vif->p2p_index].role == P2P_ROLE_GO));
}

/**
 ****************************************************************************************
 * @brief Return the minimal window that contains both window
 *
 * Given two recursive window (same interval), this functions return the width on the
 * minimal window that would contains both windows.
 *
 * @param[in]  w1       First window
 * @param[in]  w2       Second window
 * @param[out] w_res    Updated with the result window (can be NULL)
 * @param[in]  interval Window interval in us
 *
 * @return Width, in us, of the result window.
 ****************************************************************************************
 */
static uint32_t chan_window_merge_duration(struct chan_window *w1,
                                           struct chan_window *w2,
                                           struct chan_window *w_res,
                                           uint32_t interval)
{
    struct chan_window *w;
    uint32_t duration;
    uint32_t delta;

    // Need to align window timestamp in the same interval
    if (hal_machw_time_cmp(w1->start, w2->start))
    {
        w = w1;
        delta = w2->start - w1->start;
    }
    else
    {
        w = w2;
        delta = w1->start - w2->start;
    }

    // Use loop instead of % as in most cases it will only take a few iterations
    // which is faster than %
    while (delta > interval) {
        delta -= interval;
        w->start += interval;
        w->end += interval;
    }
    if (delta > (interval / 2))
    {
        w->start += interval;
        w->end += interval;
    }

    if (hal_machw_time_cmp(w1->end, w2->end))
    {
        duration = w2->end;
        if (w_res)
            w_res->end = w2->end;
    }
    else
    {
        duration = w1->end;
        if (w_res)
            w_res->end = w1->end;
    }

    if (hal_machw_time_cmp(w1->start, w2->start))
    {
        duration -= w1->start;
        if (w_res)
            w_res->start = w1->start;
    }
    else
    {
        duration -= w2->start;
        if (w_res)
            w_res->start = w2->start;
    }

    return duration;
}

/**
 ****************************************************************************************
 * @brief Stop concurrent NOA for a P2P GO interface.
 *
 * @param[in] p2pgo_vif VIF entry for the P2P GO interface
 ****************************************************************************************
 */
static void chan_stop_p2pgo_noa(struct vif_info_tag *p2pgo_vif)
{
    struct p2p_info_tag *p2p = &p2p_info_tab[p2pgo_vif->p2p_index];
    struct vif_info_tag *vif;
    int i;

    for (i = 0; i < P2P_NOA_NB_MAX; i ++)
    {
        if ((p2p->noa[i].noa_type == P2P_NOA_TYPE_CONCURRENT) &&
            (p2p->noa[i].noa_status != P2P_NOA_TIMER_NOT_STARTED))
            p2p_go_noa_stop(p2pgo_vif, i, false);
    }

    vif = vif_mgmt_first_used();
    while (vif)
    {
        if (vif->tbtt_switch.p2p_noa_vif_index == p2pgo_vif->index)
        {
            vif->tbtt_switch.p2p_noa_vif_index = INVALID_VIF_IDX;
        }
        vif = vif_mgmt_next(vif);
    }
}

/**
 ****************************************************************************************
 * @brief Start concurrent NOA for a P2P GO interface
 *
 * When a P2P GO is created and there are several active channel, it is necessary to start
 * NOA (Notice of Absence) on the P2P GO interface to spend time on the other active
 * channel. Indeed the P2P GO interface only needs to be present when not inside an
 * absence period.
 *
 * This function will:
 * - Start a periodic NOA for each interface on the other channel. For now the duration
 *   on this NOA is selected so that the P2P GO interface is absent half the time
 *   (To be updated to adapt repartition based on traffic)
 * - If two interfaces share the same beacon interval and their beacon are close enough
 *   then only one NOA will be started for both interfaces
 *
 * @param[in] p2pgo_vif VIF entry for the P2P GO interface
 ****************************************************************************************
 */
static void chan_start_p2pgo_noa(struct vif_info_tag *p2pgo_vif)
{
    struct vif_info_tag *vif;
    struct chan_noa {
        uint32_t intv;
        uint32_t dur;
        struct chan_window win;
        uint32_t vifs;
    } noa[P2P_NOA_NB_MAX];
    int noa_idx = 0;
    uint32_t p2p_intv;
    #if NX_BEACONING
    bool ap_tbtt_moved = false;
    int32_t ap_tbtt_move_min = 0, ap_tbtt_move_max = 0;
    #endif
    struct chan_ctxt_tag *oth_ctxt = NULL;
    uint32_t nb_vif_oth_ctxt = 0;

    if (chan_env.nb_sched_ctxt < 2)
        return;

    ASSERT_ERR((p2pgo_vif->chan_ctxt->p2pgo_vif_index == INVALID_VIF_IDX) ||
               (p2pgo_vif->chan_ctxt->p2pgo_vif_index == p2pgo_vif->index));
    chan_stop_p2pgo_noa(p2pgo_vif);
    p2pgo_vif->chan_ctxt->p2pgo_vif_index = p2pgo_vif->index;

    p2p_intv = chan_vif_tbtt_intv(p2pgo_vif);
    oth_ctxt = chan_get_next_traf_ctxt(p2pgo_vif->chan_ctxt);
    ASSERT_ERR(oth_ctxt);
    nb_vif_oth_ctxt = oth_ctxt->nb_linked_vif;

    vif = vif_mgmt_first_used();
    while (vif)
    {
        if (vif->chan_ctxt && (vif->chan_ctxt != p2pgo_vif->chan_ctxt))
        {
            uint32_t dur, intv;
            bool merged = false;

            intv = chan_vif_tbtt_intv(vif);
            if (intv >= (2 * p2p_intv))
            {
                intv = intv / (intv / p2p_intv);
            }
            dur = intv / 2; // For 50-50 repartition
            dur = dur / nb_vif_oth_ctxt;
            if (dur < (CHAN_SWITCH_DELAY + chan_vif_tbtt_dur(vif)))
                dur = CHAN_SWITCH_DELAY + chan_vif_tbtt_dur(vif);

            // Merge with previous NOA if possible
            for (int i = 0 ; i < noa_idx; i ++)
            {
                struct chan_window win1, win2;
                uint32_t win_dur, dur_merge;

                // can only merge NOA with same intv
                if (noa[i].intv != intv)
                    continue;

                win1.start = vif->tbtt_timer.time;
                win1.end = win1.start + chan_vif_tbtt_dur(vif);
                win2 = noa[i].win;
                win_dur = chan_window_merge_duration(&win1, &win2, &win1, intv);

                // Merging NOA is like removing one vif (from sharing time point of view)
                ASSERT_ERR(nb_vif_oth_ctxt > 1);
                dur_merge = (dur * nb_vif_oth_ctxt) / (nb_vif_oth_ctxt - 1);
                // TBTTs cannot be to far apart as the NOA must include one switch
                // The second switch is taken out of P2P channel time.
                dur_merge -= CHAN_SWITCH_DELAY;
                if (dur_merge < noa[i].dur)
                    dur_merge = noa[i].dur;

                if (win_dur < dur_merge)
                {
                    merged = true;
                    noa[i].dur = dur_merge + CHAN_SWITCH_DELAY;
                    noa[i].win = win1;
                    noa[i].vifs |= CO_BIT(vif->index);
                    break;
                }
            }

            if (!merged)
            {
                ASSERT_ERR(noa_idx < P2P_NOA_NB_MAX);

                noa[noa_idx].intv = intv;
                noa[noa_idx].dur = dur;
                noa[noa_idx].win.start = vif->tbtt_timer.time;
                noa[noa_idx].win.end = noa[noa_idx].win.start + chan_vif_tbtt_dur(vif);
                ASSERT_ERR(vif->index < 32);
                noa[noa_idx].vifs = CO_BIT(vif->index);
                noa_idx++;
            }
        }
        vif = vif_mgmt_next(vif);
    }

    for (int i = 0 ; i < noa_idx; i++)
    {
        struct chan_window win;
        uint32_t start, dur;
        uint32_t start_offset = 2000; // Start 2ms before TBTT to handle drift
        uint8_t noa_idx;

        #if NX_BEACONING
        // If p2pgo and "controlling" vifs have a multiple beacon period, updating AP TBTT
        // will allow to have p2p TBTT in a presence period.
        if ((noa[i].intv % p2p_intv) == 0)
        {
            uint32_t intv = co_min(noa[i].intv, p2p_intv);
            uint32_t ap_next_tbtt;
            int32_t  time_after_tbtt, tbtt_move;
            int32_t tbtt_move_min, tbtt_move_max;

            // At this point beacon interval may not be programmed yet
            ap_next_tbtt = hal_machw_time() + (nxmac_next_tbtt_get() << 5);

            // compute time after end on last TBTT in NOA and TBTT of P2P-GO
            win.start = ap_next_tbtt;
            win.end = win.start + chan_vif_tbtt_dur(p2pgo_vif);
            time_after_tbtt = chan_window_merge_duration(&noa[i].win, &win, NULL, intv);
            if (hal_machw_time_cmp(noa[i].win.start , win.start))
                time_after_tbtt = win.start - noa[i].win.end;
            else
                time_after_tbtt = intv - time_after_tbtt;

            // NOA duration from the end of the sta TBTT
            tbtt_move = (noa[i].dur - (noa[i].win.end - noa[i].win.start) - start_offset);
            if (tbtt_move < (int32_t)CHAN_SWITCH_DELAY)
                tbtt_move = (int32_t)CHAN_SWITCH_DELAY;

            tbtt_move_min = tbtt_move - time_after_tbtt;
            tbtt_move_max = tbtt_move_min + (noa[i].intv - noa[i].dur
                                             - CHAN_SWITCH_DELAY - chan_vif_tbtt_dur(p2pgo_vif));


            if (ap_tbtt_moved)
            {
                // See if there is a value that would fit all NOA.
                if (tbtt_move_min > ap_tbtt_move_max)
                {
                    tbtt_move_min -= p2p_intv;
                    tbtt_move_max -= p2p_intv;
                }
                else if (ap_tbtt_move_min > tbtt_move_max)
                {
                    tbtt_move_max += p2p_intv;
                    tbtt_move_min += p2p_intv;
                }

                if ((ap_tbtt_move_max >= tbtt_move_min) &&
                    (tbtt_move_max >= ap_tbtt_move_min))
                {
                    if (tbtt_move_min > ap_tbtt_move_min)
                        ap_tbtt_move_min = tbtt_move_min;
                    if (tbtt_move_max < ap_tbtt_move_max)
                        ap_tbtt_move_max = tbtt_move_max;
                }
            }
            else
            {
                ap_tbtt_move_min = tbtt_move_min;
                ap_tbtt_move_max = tbtt_move_max;
                ap_tbtt_moved = true;
            }
        }
        #endif //NX_BEACONING

        // Need to include switching time in the NOA duration
        start_offset += CHAN_SWITCH_DELAY;
        start = noa[i].win.start - start_offset;
        dur = noa[i].dur + CHAN_SWITCH_DELAY;

        // Need to re-adjust start date to start on next iteration
        // Would be more compact using div/modulo but not faster as in most cases
        // while loop will have 0 iterations ()
        if (hal_machw_time_past(start))
        {
            start += noa[i].intv;
            while (hal_machw_time_past(start))
                start += noa[i].intv;
        }
        else if (!hal_machw_time_past(start - noa[i].intv))
        {
            start -= noa[i].intv;
            while (!hal_machw_time_past(start - noa[i].intv))
                start -= noa[i].intv;
        }

        noa_idx = p2p_go_noa_start(p2pgo_vif, true, false, P2P_NOA_CONTINUOUS_COUNTER,
                                   noa[i].intv, dur, start);

        noa[i].win.start = start;
        noa[i].win.end = start + dur;
        for (int j = 0; j < NX_VIRT_DEV_MAX; j++)
        {
            if (!(noa[i].vifs & CO_BIT(j)))
                continue;
            vif = &vif_info_tab[j];

            win.start = vif->tbtt_timer.time;
            win.end = win.start + chan_vif_tbtt_dur(vif);
            chan_window_merge_duration(&noa[i].win, &win, NULL, noa[i].intv);

            ASSERT_ERR(vif->tbtt_switch.p2p_noa_vif_index == INVALID_VIF_IDX);
            vif->tbtt_switch.p2p_noa_vif_index = p2pgo_vif->index;
            vif->tbtt_switch.p2p_noa_index = noa_idx;
            vif->tbtt_switch.p2p_noa_tbtt_to_end = noa[i].win.end - win.start;
            vif->tbtt_switch.p2p_noa_drift = 0;
        }

        TRACE_CHAN(NOA, "{VIF-%d}{CTXT-%d} Start periodic NOA for TBTT on vifs %x"
                   ": duration=%ldus interval=%ldus start %t",
                   p2pgo_vif->index, p2pgo_vif->chan_ctxt->idx, noa[i].vifs, TR_32(dur),
                   TR_32(noa[i].intv), TR_32(start));
    }

    #if NX_BEACONING
    if (ap_tbtt_moved)
    {
        int32_t tbtt_move = ap_tbtt_move_min + 5000;
        if (tbtt_move > ap_tbtt_move_max)
            tbtt_move = ap_tbtt_move_max;

        if (tbtt_move > ((int32_t)p2p_intv / 2))
            tbtt_move -= p2p_intv;
        else if (tbtt_move < (-1 * ((int32_t)p2p_intv / 2)))
            tbtt_move += p2p_intv;

        p2pgo_vif->tbtt_switch.p2p_noa_tsf_update = mm_ap_tbtt_move(tbtt_move);
        p2pgo_vif->tbtt_switch.status |= CHAN_TBTT_P2P_NOA_TSF_UPDATE;
    }
    #endif //NX_BEACONING
}

/**
 ****************************************************************************************
 * @brief Return P2P GO interface.
 *
 * @param[out] p2pgo_vif Updated with pointer to vif structure for P2P GO interface.
 * @return true if a P2P GO interface exist and false otherwise
 ****************************************************************************************
 */
__INLINE bool chan_get_p2pgo(struct vif_info_tag **p2pgo_vif)
{
    for (int ctxt_idx = 0; ctxt_idx < CHAN_TRAF_CTXT_CNT; ctxt_idx++)
    {
        struct chan_ctxt_tag *ctxt = &chan_ctxt_pool[ctxt_idx];

        if ((ctxt->idx != CHAN_CTXT_UNUSED) &&
            (ctxt->p2pgo_vif_index != INVALID_VIF_IDX))
        {
            *p2pgo_vif = &vif_info_tab[ctxt->p2pgo_vif_index];
            return true;
        }
    }
    return false;
}

/**
 ****************************************************************************************
 * @brief Start a non-periodic NOA on a P2P GO interface.
 *
 * When a TBTT must be skipped for a P2P GO interface, a non periodic NOA must be started
 * to inform client. Indeed P2P GO must be present for TBTT even during periodic NOA.
 *
 * @param[in] vif      Vif structure for P2P GO interface.
 * @param[in] start    Timestamp, in us, of the skipped TBTT.
 * @param[in] tbtt     TBTT (of another vif) that induces this NOA. (may be NULL)
 * @return true if NOA has been started and false otherwise.
 ****************************************************************************************
 */
static bool chan_start_p2pgo_single_noa(struct vif_info_tag *vif, uint32_t start,
                                        struct chan_tbtt_tag *tbtt)
{
    struct vif_info_tag *p2pgo_vif = vif;
    uint32_t duration;

    if (!p2pgo_vif && !chan_get_p2pgo(&p2pgo_vif))
        return false;

    if (tbtt && (tbtt->p2p_noa_vif_index != INVALID_VIF_IDX))
    {
        // If a TBTT induce this NOA, then we already have a concurrent NOA
        // that should cover this TBTT. But because of drift or CHAN_SWITCH_DELAY
        // the periodic NOA may end a little before the requested single NOA.
        // If this is the case, a channel switch will be scheduled during the short
        // presence period. To avoid this start single NOA earlier so that both NOA
        // overlap
        struct vif_info_tag *vif_noa =  &vif_info_tab[tbtt->vif_index];
        uint32_t cc_noa_end;
        uint32_t cc_noa_intv = chan_vif_tbtt_intv(vif_noa);
        uint32_t noa_end = tbtt->time + chan_vif_tbtt_dur(vif_noa) + CHAN_SWITCH_DELAY;

        cc_noa_end = p2p_go_get_next_noa_end_date(vif, tbtt->p2p_noa_index);
        while (hal_machw_time_cmp(cc_noa_end + (cc_noa_intv / 2), start))
        {
            cc_noa_end += cc_noa_intv;
        }
        if (hal_machw_time_cmp(cc_noa_end, start))
        {
            start = cc_noa_end - 1;
        }

        if (hal_machw_time_cmp(noa_end + 5000, start))
            duration = 5000;
        else
            duration = noa_end - start;
    }
    else
    {
        duration = chan_vif_tbtt_dur(vif) + CHAN_SWITCH_DELAY;
    }

    return (p2p_go_noa_start(p2pgo_vif, true, false, 1,
                             0, duration, start) != P2P_INVALID_IDX);
}
#endif //(NX_P2P_GO)

/**
 ****************************************************************************************
 * @brief Check if TBTT constraint for two vifs overlap.
 *
 * @param[in] tbtt1 TBTT information of first vif
 * @param[in] tbtt2 TBTT information of second vif
 * @return true if TBTTs overlap and false otherwise.
 ****************************************************************************************
 */
static bool chan_tbtt_overlap(struct chan_tbtt_tag *tbtt1, struct chan_tbtt_tag *tbtt2)
{
    struct vif_info_tag *vif1 = &vif_info_tab[tbtt1->vif_index];
    struct vif_info_tag *vif2 = &vif_info_tab[tbtt2->vif_index];
    uint32_t start1, end1, start2, end2;

    if (vif1->chan_ctxt == vif2->chan_ctxt)
        return false;

    start1 = tbtt1->time;
    end1 = start1 + chan_vif_tbtt_dur(vif1) + CHAN_SWITCH_DELAY;
    start2 = tbtt2->time;
    end2 = start2 + chan_vif_tbtt_dur(vif2) + CHAN_SWITCH_DELAY;

    if ((hal_machw_time_cmp(start1, start2) && hal_machw_time_cmp(start2, end1)) ||
        (hal_machw_time_cmp(start2, start1) && hal_machw_time_cmp(start1, end2)) ||
        (start2 == start1)) {
        return true;
    }

    return false;
}

/**
 ****************************************************************************************
 * @brief Check if TBTT must be skipped because of scheduled switch or fix channel.
 *
 * Check if the TBTT scheduled at @p tbtt_time on channel context @p ctxt have to be
 * skipped. There are two conditions that may force to skip the TBTT:
 * 1. If there are already scheduled channel switchs:
 *    (to avoid useless complexity, only the last scheduled switch is taken into account)
 *
 *   - If the target channel of this switch is the same as the channel passed in parameter
 *     the function checks that the channel will be active at @p tbbt_time (i.e. switch
 *     starts at least @ref CHAN_SWITCH_DELAY us before @p tbtt_time).
 *     This assumes that the switch cannot be scheduled early because of other constraint.
 *
 *   - If the target channel is not the same, then the function checks if it's possible to
 *     switch to the new channel after the scheduled switch and be present at @p tbtt_time
 *    (i.e. if @p tbtt_time happens @ref CHAN_SWITCH_DELAY us after the end of the switch)
 *
 * Note: This case can only happen when a new vif is started. Indeed, channel switches
 * are scheduled at each TBTT until next TBTT, so when this function is called (at TBTT)
 * the list will always be empty unless this is the first TBTT of a vif.
 *
 * 2. Because of fix channel context:
 *    If a fix channel context is currently set, the function checks if the fix channel is
 *    the same as the target or if "fix period" ends at least @ref CHAN_SWITCH_DELAY
 *    before @p tbtt_time.
 *
 * @param[in] tbtt_time Date of the TBTT (absolute timestamp in us)
 * @param[in] ctxt      Channel context of the TBTT
 * @return true if it won't be possible to have channel @p ctxt active at @p tbtt_time,
 * false otherwise.
 ****************************************************************************************
 */
__INLINE bool chan_tbtt_skip_fix_ctxt(uint32_t tbtt_time, struct chan_ctxt_tag *ctxt)
{
    if (!chan_switch_no_schedule())
    {
        struct chan_switch_tag *ch_switch = chan_switch_last();
        uint32_t date = ch_switch->time + CHAN_SWITCH_DELAY;

        if (ch_switch->ctxt != ctxt)
        {
            date += CHAN_SWITCH_DELAY;
            if (ch_switch->tbtt)
            {
                date += chan_vif_tbtt_dur(&vif_info_tab[ch_switch->tbtt->vif_index]);
            }
        }
        return hal_machw_time_cmp(tbtt_time, date);
    }
    else if (chan_env.fix_ctxt &&
             (chan_env.fix_ctxt != ctxt))
    {
        return hal_machw_time_cmp(tbtt_time, chan_env.fix_until + CHAN_SWITCH_DELAY);
    }

    return false;
}

/**
 ****************************************************************************************
 * @brief Add next TBTT for a vif.
 *
 * This will add the next TBTT element for a vif in chan_env.list_tbtt. This list is only
 * used by the module to scheduled channel switch.
 * If it's not possible to add this TBTT, because of scheduled switchs
 * (@ref chan_tbtt_skip_fix_ctxt) or TBTT overlap (@ref chan_tbtt_overlap), the function
 * is called recursively with parameter @p tbtt_time incremented by one beacon interval
 * and parameter @p skip_cnt incremented by one.
 *
 * @param[in] vif       VIF structure.
 * @param[in] tbtt_time Date of the next TBTT (absolute timestamp in us)
 * @param[in] skip_cnt  Number of TBTT skipped (only for debug).
 ****************************************************************************************
 */
static void chan_add_next_tbtt(struct vif_info_tag *vif, uint32_t tbtt_time,
                               uint32_t skip_cnt)
{
    struct chan_tbtt_tag *elem = NULL, *tbtt = &vif->tbtt_switch;
    bool skip_it = false;
    uint32_t end_tbtt;

    if ((skip_cnt == 0) && (tbtt->status & CHAN_TBTT_PROG))
        chan_tbtt_remove(tbtt);

    if (chan_tbtt_skip_fix_ctxt(tbtt_time, vif->chan_ctxt))
    {
        skip_it = true;
    }
    else
    {
        tbtt->time = tbtt_time;
        // -1 because hal_machw_time_cmp use < and not <=
        end_tbtt = tbtt->time + chan_vif_tbtt_dur(vif) + CHAN_SWITCH_DELAY - 1;

        elem = chan_tbtt_pick();
        while (elem)
        {
            if (hal_machw_time_cmp(end_tbtt, elem->time))
                break;

            if (chan_tbtt_overlap(tbtt, elem))
            {
                skip_it = true;
                break;
            }
            else if (hal_machw_time_cmp(tbtt_time, elem->time))
            {
                break;
            }

            elem = chan_tbtt_next(elem);
        }
    }

    if (skip_it)
    {
        #if (NX_P2P_GO)
        if (chan_is_p2p_go(vif))
        {
            chan_start_p2pgo_single_noa(vif, tbtt_time, elem);
        }
        #endif //(NX_P2P_GO)
        return chan_add_next_tbtt(vif, tbtt_time + chan_vif_tbtt_intv(vif), skip_cnt + 1);
    }

    chan_tbtt_insert_before(tbtt, elem);
    TRACE_CHAN(TBTT, "{VIF-%d} schedule next TBTT %t (skip=%d)", vif->index,
               TR_32(tbtt_time), skip_cnt);
}

/**
 ****************************************************************************************
 * @brief Share time between two channel context.
 *
 * For now do a 50/50 share between channels. To be updated
 *
 * @param[in] dur_to_split Time, in us, to share between the channels
 *                         (not including dur_spent).
 * @param[in] dur_spent    Time, in us, already spent on current channel.
 * @param[in] now          Current time (absolute timestamp in us)
 * @return date of the switch.
 ****************************************************************************************
 */
static uint32_t chan_share_medium(int32_t dur_to_split, uint32_t dur_spent, uint32_t now)
{
    uint32_t split_dur = (dur_to_split + dur_spent) / 2;
    if (split_dur < dur_spent)
        split_dur = 0;
    else
        split_dur -= dur_spent;
    return now + split_dur;
}

/**
 ****************************************************************************************
 * @brief Schedule a channel switch.
 *
 * @param[in] date Date of the switch (absolute timestamp in us)
 * @param[in] ctxt Channel context to switch to
 * @param[in] tbtt TBTT information that cause this switch (may be NULL)
 ****************************************************************************************
 */
static void chan_schedule_one_switch(uint32_t date, struct chan_ctxt_tag *ctxt,
                                     struct chan_tbtt_tag *tbtt)
{
    struct chan_switch_tag *ch_switch = chan_switch_alloc();
    bool set_timer = chan_switch_no_schedule();
    TRACE_CHAN(SWITCH, "{CTXT-%d} schedule switch at %t",
               ctxt ? ctxt->idx : 255, TR_32(date));
    ch_switch->time = date;
    ch_switch->ctxt = ctxt;
    ch_switch->tbtt = tbtt;

    chan_switch_push_back(ch_switch);
    if (set_timer)
        chan_switch_set_timer(ch_switch);
}

#if NX_P2P_GO
/**
 ****************************************************************************************
 * @brief Schedule a switch (and return) to other context when TBTT ends on a channel
 *        with a P2P GO interface.
 *
 * Depending of P2P GO presence and its next scheduled NOA, the function checks if it
 * is possible to switch to the other active channel :
 * - If the P2P GO interface is currently present, a switch can only be scheduled if the
 *   next p2p absence happens before the next TBTT (and is long enough).
 * - If the P2P GO interface is currently absent, switch immediately if absence period
 *   will last enough, or schedule a switch if another absence period happens before the
 *   the next TBTT (Both conditions can be true leading to schedule 4 switches in total)
 *
 * @note Because @ref p2p_go_get_next_NOA_date only provides date for next NOA, the
 * function assumes that only one NOA can happen until next TBTT. This should be true for
 * NOA generated internally to support multiple channels, but not for NOA generated by
 * host.
 *
 * @param[in]     p2p_vif_index Vif index of the P2P GO interface
 * @param[in]     now           Current date
 * @param[out]    switch_date   Updated with the switch date
 * @param[in,out] return_date   Contain the latest date to return to P2P GO channel
 *                              (including CHAN_SWITCH_DELAY), and may be updated if
 *                              return must happen earlier.
 * @return true if a switch is possible (and in this case @p switch_date and
 * @p return_date indicates the date) and false otherwise.
 ****************************************************************************************
 */
static bool chan_schedule_extra_switch_from_p2pgo_ctxt(int p2p_vif_index,
                                                       uint32_t now,
                                                       uint32_t *switch_date,
                                                       uint32_t *return_date)
{
    struct vif_info_tag *p2p_vif = &vif_info_tab[p2p_vif_index];
    uint32_t p2p_next_present, p2p_next_absent;
    bool p2p_present =  p2p_go_get_next_NOA_date(p2p_vif,
                                                 &p2p_next_present,
                                                 &p2p_next_absent);
    bool switch_scheduled = false;
    uint32_t tbtt_return_date = *return_date;

    if (p2p_present)
    {
        *switch_date = p2p_next_absent;
        if (hal_machw_time_cmp((p2p_next_present - CHAN_SWITCH_DELAY), *return_date))
            *return_date = p2p_next_present - CHAN_SWITCH_DELAY;
        if (hal_machw_time_cmp(*return_date,
                               *switch_date + CHAN_SWITCH_DELAY + CHAN_MIN_PRES_DUR_US))
        {
            return false;
        }
    }
    else
    {
        if (hal_machw_time_cmp(now + 2 * CHAN_SWITCH_DELAY + CHAN_MIN_PRES_DUR_US,
                               p2p_next_present))
        {
            *switch_date = now;
            if (hal_machw_time_cmp((p2p_next_present - CHAN_SWITCH_DELAY), *return_date))
                *return_date = p2p_next_present - CHAN_SWITCH_DELAY;

            switch_scheduled = true;
        }

        if (hal_machw_time_cmp(p2p_next_absent + CHAN_SWITCH_DELAY + CHAN_MIN_PRES_DUR_US,
                               tbtt_return_date))
        {
            if(switch_scheduled)
            {
                chan_schedule_one_switch(now, chan_get_next_traf_ctxt(p2p_vif->chan_ctxt),
                                         NULL);
                chan_schedule_one_switch(*return_date, p2p_vif->chan_ctxt, NULL);
                *return_date = tbtt_return_date;
            }

            // Assume that p2p_next_next_presence date is after return date (may not
            // always be the case if non concurrent NOA are scheduled)
            *switch_date = p2p_next_absent;
        }
        else if (!switch_scheduled)
        {
            return false;
        }
    }
    return true;
}


/**
 ****************************************************************************************
 * @brief Schedule a switch (and return) to a context with P2P GO interface when TBTT
 *        ends on a channel without a P2P GO interface.
 *
 * Checks if a presence period will happen before the next TBTT and if so schedule switch
 * to respect it. The NOA started by @ref chan_start_p2pgo_noa should ensure that the
 * P2P GO interface would be absent for this TBTT and the next one.
 *
 * @note Just like @ref chan_schedule_extra_switch_from_p2pgo_ctxt the function assumes
 * that only one presence period can happen until next TBTT.
 *
 * @param[in]     p2p_vif_index Vif index of the P2P GO interface
 * @param[out]    switch_date   Updated with the switch date
 * @param[in,out] return_date   Contain the latest date to return to P2P GO channel
 *                              (including CHAN_SWITCH_DELAY), and may be updated if
 *                              return must happen earlier.
 * @return true if a switch is possible (and in this case @p switch_date and
 * @p return_date indicates the date) and false otherwise.
 ****************************************************************************************
 */
static bool chan_schedule_extra_switch_to_p2pgo_ctxt(int p2p_vif_index,
                                                     uint32_t *switch_date,
                                                     uint32_t *return_date)
{
    struct vif_info_tag *p2p_vif = &vif_info_tab[p2p_vif_index];
    uint32_t p2p_next_present, p2p_next_absent;
    bool p2p_present =  p2p_go_get_next_NOA_date(p2p_vif,
                                                 &p2p_next_present,
                                                 &p2p_next_absent);
    // sanity check
    if (p2p_present)
    {
        TRACE_CHAN(ERR, "P2P GO present while processing TBTT on another ctxt ..."
                   "next absent %t", TR_32(p2p_next_absent));
    }

    if (hal_machw_time_cmp(*return_date, p2p_next_present - CHAN_SWITCH_DELAY))
        // No P2P go presence before next TBTT, don't schedule any switch
        return false;
    *switch_date = p2p_next_present - CHAN_SWITCH_DELAY;

    // sanity check
    if (hal_machw_time_cmp(*return_date, p2p_next_absent))
    {
        TRACE_CHAN(ERR, "P2P GO absence %t happens after next TBTT %t",
                   TR_32(p2p_next_absent), TR_32(*return_date));
        // TODO rescheduled NOA ?
    }
    else
    {
        *return_date = p2p_next_absent;
    }

    return true;
}

/**
 ****************************************************************************************
 * @brief Schedule a switch when TBTT window ends, from a context with a P2P GO
 *        interface to another context.
 *
 * The NOA started by @ref chan_start_p2pgo_noa (should) ensure that the P2P GO interface
 * will be in an absence period for the TBTT on the next context.
 * Still a presence period can happen before next TBTT, and in this case 2 extra switches
 * will be scheduled.
 *
 * @param[in]     p2p_vif_index Vif index of the P2P GO interface
 * @param[in]     next_ctxt     Pointer to the otehr context
 * @param[in]     now           Current date
 * @param[in,out] switch_date   Contain the latest date to switch from P2P GO channel
 *                              (including CHAN_SWITCH_DELAY), and may be updated if
 *                              switch can happen earlier.
 ****************************************************************************************
 */
static void chan_schedule_switch_from_p2pgo_ctxt(int p2p_vif_index,
                                                 struct chan_ctxt_tag *next_ctxt,
                                                 uint32_t now,
                                                 uint32_t *switch_date)
{
    struct vif_info_tag *p2p_vif = &vif_info_tab[p2p_vif_index];
    uint32_t p2p_next_present, p2p_next_absent;
    bool p2p_present =  p2p_go_get_next_NOA_date(p2p_vif,
                                                 &p2p_next_present,
                                                 &p2p_next_absent);

    if (p2p_present)
    {
        if (hal_machw_time_cmp(*switch_date, p2p_next_absent))
        {
            TRACE_CHAN(ERR, "P2P GO next absence [%t..%t] happens after next TBTT %t",
                       TR_32(p2p_next_absent), TR_32(p2p_next_present),
                       TR_32(*switch_date));
            // TODO rescheduled NOA ?
            return;
        }
        else if (hal_machw_time_cmp(p2p_next_present, *switch_date))
        {
            if ((p2p_next_present - p2p_next_absent) > CHAN_SWITCH_DELAY + CHAN_MIN_PRES_DUR_US)
            {
                chan_schedule_one_switch(p2p_next_absent, next_ctxt, NULL);
                chan_schedule_one_switch(p2p_next_present - CHAN_SWITCH_DELAY,
                                         p2p_vif->chan_ctxt, NULL);
            }
        }
        else
            *switch_date = p2p_next_absent;
    }
    else
    {
        if (hal_machw_time_cmp(*switch_date, p2p_next_present))
            *switch_date = now;
        else
        {
            if ((p2p_next_present - now) > CHAN_SWITCH_DELAY + CHAN_MIN_PRES_DUR_US)
            {
                chan_schedule_one_switch(now, next_ctxt, NULL);
                chan_schedule_one_switch(p2p_next_present - CHAN_SWITCH_DELAY,
                                         p2p_vif->chan_ctxt, NULL);
            }

            if (hal_machw_time_cmp(*switch_date, p2p_next_absent))
            {
                TRACE_CHAN(ERR, "P2P GO presence [%t..%t] will overlap with next TBTT %t",
                           TR_32(p2p_next_present), TR_32(p2p_next_absent),
                           TR_32(*switch_date));
                // TODO rescheduled NOA ?
                return;
            }
            *switch_date = p2p_next_absent;
        }
    }
}

/**
 ****************************************************************************************
 * @brief Schedule a switch from a context without P2P GO interface when TBTT
 *        ends to a context with P2P GO interface .
 *
 * Schedule switch to match next presence period or the next TBTT (whichever happens
 * first). Still a absence period can happen before TBTT and in this case 2 extra switches
 * will be scheduled.
 *
 * @param[in]     p2p_vif_index Vif index of the P2P GO interface
 * @param[in]     ctxt          Context of the current ending TBTT window
 * @param[in]     now           Current date
 * @param[in,out] switch_date   Contain the latest date to switch to P2P GO channel
 *                              (including CHAN_SWITCH_DELAY), and may be updated if
 *                              switch must happen earlier.
 ****************************************************************************************
 */
static void chan_schedule_switch_to_p2pgo_ctxt(int p2p_vif_index,
                                               struct chan_ctxt_tag *ctxt,
                                               uint32_t now, uint32_t *switch_date)
{
    struct vif_info_tag *p2p_vif = &vif_info_tab[p2p_vif_index];
    uint32_t p2p_next_present, p2p_next_absent;
    bool p2p_present =  p2p_go_get_next_NOA_date(p2p_vif,
                                                 &p2p_next_present,
                                                 &p2p_next_absent);
    uint32_t tbtt_switch_date = *switch_date;

    p2p_next_present -= CHAN_SWITCH_DELAY;
    if (p2p_present)
    {
        TRACE_CHAN(ERR, "P2P GO present while processing TBTT on another ctxt ..."
                   "next absent %t", TR_32(p2p_next_absent));

        *switch_date = now;
    }
    else if (hal_machw_time_cmp(p2p_next_present, *switch_date))
    {
        *switch_date = p2p_next_present;
    }

    if (hal_machw_time_cmp(p2p_next_absent + CHAN_SWITCH_DELAY + CHAN_MIN_PRES_DUR_US,
                           tbtt_switch_date))
    {
        chan_schedule_one_switch(*switch_date, p2p_vif->chan_ctxt, NULL);
        chan_schedule_one_switch(p2p_next_absent, ctxt, NULL);

        if (p2p_present && hal_machw_time_cmp(p2p_next_present, tbtt_switch_date))
            *switch_date = p2p_next_present;
        else
            *switch_date = tbtt_switch_date;
    }

}

#endif //NX_P2P_GO

/**
 ****************************************************************************************
 * @brief Schedule all channel switches from end of current TBTT until next TBTT.
 *
 * This function is called at each end of TBTT and schedules switches until next TBTT.
 * (When this function is called there are necessarily two active channels contexts)
 *
 * There are two main cases:
 * 1. The next TBTT is on the same channel
 *    In this case check if it's possible to stay at least @ref CHAN_MIN_PRES_DUR_US
 *    on the other active channel and come back before next TBTT.
 *    If it's not possible no switch are scheduled.
 *
 * 2. The Next TBTT is on another channel.
 *    Tests done on @ref chan_add_next_tbtt ensure that we can switch to the other
 *    channel before its TBTT starts.
 *    The function simply schedules the switch to share time between both channels.
 *
 * Note: In normal case, when this function is called current channel is the channel of
 * the TBTT. But it may not always be the case:
 * The switch took more that expected (@ref CHAN_SWITCH_DELAY). In this case act like
 * we are on the TBTT channel.
 *
 * @param[in] vif        Vif of the current TBTT.
 * @param[in] now        Date of end of current TBTT.
 * @param[in] tbtt_dur   Duration spent on channel for the TBTT, in us.
 ****************************************************************************************
 */
static void chan_schedule_next_switch(struct vif_info_tag *vif, uint32_t now,
                                      uint32_t tbtt_dur)
{
    struct chan_ctxt_tag *ctxt = vif->chan_ctxt;
    struct chan_tbtt_tag *tbtt_next = chan_tbtt_pick();
    struct vif_info_tag *vif_next = &vif_info_tab[tbtt_next->vif_index];
    int32_t dur_to_split = tbtt_next->time - now - CHAN_SWITCH_DELAY;

    // sanity check
    if (chan_env.current_ctxt != ctxt)
    {
        TRACE_CHAN(ERR, "Schedule switch: not on expected channel."
                   " current_ctxt=%d expected_channel=%d status=%d",
                   chan_env.current_ctxt ? chan_env.current_ctxt->idx: -1,
                   ctxt->idx, ctxt->status);
    }

    if (ctxt == vif_next->chan_ctxt)
    {
        // Next TBTT is on the same ctxt, check if we can spent at at least
        // CHAN_MIN_PRES_DUR_US on the other channel before it.
        // This means two channel switches so take it into account when sharing medium
        uint32_t switch_date = now, return_date;

        dur_to_split -= CHAN_SWITCH_DELAY;
        if (dur_to_split < CHAN_MIN_PRES_DUR_US)
            return;

        return_date = tbtt_next->time - CHAN_SWITCH_DELAY;

        #if (NX_P2P_GO)
        if (ctxt->p2pgo_vif_index != INVALID_VIF_IDX)
        {
            if (!chan_schedule_extra_switch_from_p2pgo_ctxt(ctxt->p2pgo_vif_index,
                                                            now, &switch_date,
                                                            &return_date))
                return ;
        }
        else if (vif->tbtt_switch.p2p_noa_vif_index != INVALID_VIF_IDX)
        {
            if (!chan_schedule_extra_switch_to_p2pgo_ctxt(vif->tbtt_switch.p2p_noa_vif_index,
                                                          &switch_date,
                                                          &return_date))
                return ;
        }
        else
        #endif // (NX_P2P_GO)
        {
            switch_date = chan_share_medium(dur_to_split, tbtt_dur, now);
        }

        chan_schedule_one_switch(switch_date, chan_get_next_traf_ctxt(ctxt), NULL);
        chan_schedule_one_switch(return_date, ctxt, tbtt_next);
    }
    else
    {
        // next TBTT on the other channel, prepare just one switch
        uint32_t switch_date = tbtt_next->time - CHAN_SWITCH_DELAY;

        #if (NX_P2P_GO)
        if (ctxt->p2pgo_vif_index != INVALID_VIF_IDX)
        {
            chan_schedule_switch_from_p2pgo_ctxt(ctxt->p2pgo_vif_index,
                                                 vif_next->chan_ctxt,
                                                 now, &switch_date);
        }
        else if (vif->tbtt_switch.p2p_noa_vif_index != INVALID_VIF_IDX)
        {
            chan_schedule_switch_to_p2pgo_ctxt(vif->tbtt_switch.p2p_noa_vif_index,
                                               ctxt, now, &switch_date);
        }
        else
        #endif // (NX_P2P_GO)
        {
            switch_date = chan_share_medium(dur_to_split, tbtt_dur, now);
        }

        chan_schedule_one_switch(switch_date, vif_next->chan_ctxt, tbtt_next);
    }
}

/**
 ****************************************************************************************
 * @brief Cancel all scheduled channel switches
 ****************************************************************************************
 */
static void chan_cancel_all_switch(void)
{
    struct chan_switch_tag *ch_switch;

    ch_switch = chan_switch_pop();
    if (ch_switch)
    {
        chan_switch_reset_timer();
        while (ch_switch)
        {
            chan_switch_free(ch_switch);
            ch_switch = chan_switch_pop();
        }
    }
}

/**
 ****************************************************************************************
 * @brief Fix a channel context for a specified period.
 *
 * This function is called when a channel switch must be forced for a given time,
 * for a SCAN or a ROC procedure.
 * All scheduled switches are removed (even if they are scheduled after the end of the
 * fixed period) and all TBTT scheduled before the end of the fixed period are delayed.
 * Once fixed period will be over, @ref chan_fix_ctxt_end will select the next channel to
 * switch to and scheduling of channel switch will re-start on the following TBTT.
 *
 * If this function is called during TBTT window, the switch is delayed until all TBTT
 * are closed. The delay is done by setting CHAN_ENV_FIX_CTXT_PENDING_BIT status and when
 * TBTT ends (@ref chan_bcn_to_evt) this function is called back.
 * Note that in this case (switch delayed) fix_until is still updated, so that
 * @ref chan_tbtt_skip_fix_ctxt will prevent TBTT that would be delayed when this function
 * is called back
 *
 * @param[in] ctxt  Channel context to force.
 ****************************************************************************************
 */
static void chan_fix_ctxt_until(struct chan_ctxt_tag *ctxt)
{
    struct chan_tbtt_tag *tbtt;
    uint32_t time = hal_machw_time() + ctxt->duration_us;

    if (ctxt != chan_env.current_ctxt)
        time += CHAN_SWITCH_DELAY;

    chan_env.fix_ctxt = ctxt;
    chan_env.fix_until = time;

    if (chan_env.nb_active_tbtt)
    {
        TRACE_CHAN(SWITCH, "Delay fixed channel while in TBTT window");
        chan_env.status |= CO_BIT(CHAN_ENV_FIX_CTXT_PENDING_BIT);
        return;
    }
    TRACE_CHAN(SWITCH, "Fix {CTXT-%d} until %t", ctxt->idx, TR_32(time));

    chan_env.status &= ~CO_BIT(CHAN_ENV_FIX_CTXT_PENDING_BIT);
    chan_cancel_all_switch();
    chan_schedule_one_switch(time, NULL, NULL);

    // Need to delay all TBTT already scheduled before the end of the fixed period
    tbtt = chan_tbtt_pick();
    if (tbtt)
    {
        while (tbtt && hal_machw_time_cmp(tbtt->time, time + CHAN_SWITCH_DELAY))
        {
            struct vif_info_tag *vif = &vif_info_tab[tbtt->vif_index];
            chan_add_next_tbtt(vif, tbtt->time + chan_vif_tbtt_intv(vif), 0);
            tbtt = chan_tbtt_pick();
        }
    }

    if (!chan_env.switch_ctxt)
    {
        chan_switch_start(ctxt);
    }
    else
    {
        ASSERT_ERR(chan_env.status & CHAN_ROC_SCAN_PENDING_MASK);
    }
}

/**
 ****************************************************************************************
 * @brief Callback called upon Connection Less Delay timer expiration.
 *        If both Scan and RoC are scheduled, start RoC.
 *
 * @param[in] env Should be a NULL pointer, do not use it
 ****************************************************************************************
 */
static void chan_conn_less_delay_evt(void *env)
{
    struct chan_ctxt_tag *ctxt = NULL;
    TRACE_CHAN(CON_LESS, "{CON-LESS} delay timer expired. status = %x", chan_env.status);

    // In waiting, start RoC in priority
    if (chan_env.status & CO_BIT(CHAN_ENV_ROC_WAIT_BIT))
    {
        // Sanity check, no RoC should be in progress here
        ASSERT_ERR((chan_env.status & CO_BIT(CHAN_ENV_ROC_BIT)) == 0);

        // Update status
        chan_env.status &= ~CO_BIT(CHAN_ENV_ROC_WAIT_BIT);
        chan_env.status |= CO_BIT(CHAN_ENV_ROC_BIT);
        ctxt = &chan_ctxt_pool[CHAN_ROC_CTXT_IDX];
    }
    // else start Scan
    else if (chan_env.status & CO_BIT(CHAN_ENV_SCAN_WAIT_BIT))
    {
        // Sanity check, no Scan should be in progress here
        ASSERT_ERR((chan_env.status & CO_BIT(CHAN_ENV_SCAN_BIT)) == 0);

        // Update status
        chan_env.status &= ~CO_BIT(CHAN_ENV_SCAN_WAIT_BIT);
        chan_env.status |= CO_BIT(CHAN_ENV_SCAN_BIT);
        ctxt = &chan_ctxt_pool[CHAN_SCAN_CTXT_IDX];
    }
    else
    {
        ASSERT_WARN("No SCAN/ROC pending");
        return;
    }

    chan_fix_ctxt_until(ctxt);
}

/**
 ****************************************************************************************
 * @brief Program the timer used for delaying Scan/RoC operations.
 *        chan_conn_less_delay_evt is called upon timer expiration
 ****************************************************************************************
 */
static void chan_conn_less_delay_prog(void)
{
    TRACE_CHAN(CON_LESS, "{CON-LESS} Start delay timer. status = %x", chan_env.status);

    if (chan_env.status & CO_BIT(CHAN_ENV_DELAY_PROG_BIT))
        return;

    chan_env.status |= CO_BIT(CHAN_ENV_DELAY_PROG_BIT);

    if (chan_env.nb_sched_ctxt)
    {
        mm_timer_set(&chan_env.tmr_conn_less, ke_time() + CHAN_CONN_LESS_DELAY);
    }
    else
    {
        chan_conn_less_delay_evt(NULL);
    }
}

/**
 ****************************************************************************************
 * @brief Handle ends of connection-less channel context.
 *
 * When time allocated for a connection-less channel context is over some specific
 * action need to be done:
 * - For SCAN: need to send scan survey
 * - For ROC : need to indication that ROC is over
 *
 * @param[in] ctxt  Connection-less Channel context.
 ****************************************************************************************
 */
static void chan_conn_less_ctxt_end(struct chan_ctxt_tag *ctxt)
{
    chan_env.current_ctxt = NULL;

    #if (NX_POWERSAVE)
    GLOBAL_INT_DISABLE();
    ps_env.prevent_sleep &= ~PS_SCAN_ONGOING;
    GLOBAL_INT_RESTORE();
    nxmac_pwr_mgt_setf(chan_env.pm);
    #endif //(NX_POWERSAVE)

    if (ctxt->idx == CHAN_SCAN_CTXT_IDX)
    {
        chan_env.status &= ~CO_BIT(CHAN_ENV_SCAN_BIT);
        #if (NX_HW_SCAN)
        chan_send_survey_ind();
        #endif //(NX_HW_SCAN)
        ke_msg_send_basic(MM_SCAN_CHANNEL_END_IND, TASK_SCAN, TASK_NONE);
    }
    else if (ctxt->idx == CHAN_ROC_CTXT_IDX)
    {
        chan_env.status &= ~CO_BIT(CHAN_ENV_ROC_BIT);

        if (ctxt->taskid != TASK_MM)
        {
            #if NX_UMAC_PRESENT && NX_TDLS
            if (ctxt->taskid == TASK_TDLS)
            {
                tdls_send_chan_switch_base_ind(ctxt);
            }
            else
            #endif //NX_UMAC_PRESENT && NX_TDLS
            {
                chan_send_roc_exp_ind(ctxt);
            }
        }
        else
        {
            chan_env.status &= ~CO_BIT(CHAN_ENV_BCN_DETECT_BIT);
        }
    }

    ctxt->idx = CHAN_CTXT_UNUSED;
    chan_env.status &= ~CO_BIT(CHAN_ENV_DELAY_PROG_BIT);

    if (chan_env.status & (CO_BIT(CHAN_ENV_ROC_WAIT_BIT) | CO_BIT(CHAN_ENV_SCAN_WAIT_BIT)))
    {
        chan_conn_less_delay_prog();
    }

    if (!chan_env.switch_ctxt)
    {
        mm_force_idle_req();
        mm_back_to_host_idle();
    }
}

/**
 ****************************************************************************************
 * @brief Return the new channel context to switch to after a fixed channel context.
 *
 * When a fixed channel is configured all previously configured switches are removed
 * (@ref chan_fix_ctxt_until). This function returns the new channel to switch to when
 * the fixed period ends. If there are more than one active channel context, it selects
 * the channel of the next TBTT, otherwise it selects the active channel (if any).
 *
 * @return Channel context to switch to. (can be NULL)
 ****************************************************************************************
 */
static struct chan_ctxt_tag *chan_fix_ctxt_end(void)
{
    chan_env.fix_ctxt = NULL;

    if (chan_env.nb_sched_ctxt == 0)
    {
        return NULL;
    }
    else if (chan_env.nb_sched_ctxt == 1)
    {
        return chan_get_first_traf_ctxt();
    }
    else
    {
        struct chan_tbtt_tag *tbtt = chan_tbtt_pick();
        if (!tbtt)
        {
            return chan_get_first_traf_ctxt();
        }
        else
        {
            return vif_info_tab[tbtt->vif_index].chan_ctxt;
        }
    }
}

/**
 ****************************************************************************************
 * @brief Timer callback for channel switch.
 *
 * When this callback is called it's time to:
 * - Select the next channel if it's the end on a fixed period.
 * - Start switch to the next channel (if any).
 * - Process end of current channel (for connection-lestt contexts only)
 * - Program timer for next switch (if any).
 *
 * @param[in] env Channel context to switch to
 ****************************************************************************************
 */
static void chan_tmr_switch_cb(void *env)
{
    struct chan_switch_tag *ch_switch = chan_switch_pop();
    ASSERT_ERR((void *)ch_switch == env);
    ASSERT_ERR(chan_env.current_ctxt);

    chan_upd_ctxt_status(chan_env.current_ctxt, CHAN_NOT_PROG);

    if (chan_env.fix_ctxt)
        ch_switch->ctxt = chan_fix_ctxt_end();

    if (ch_switch->ctxt)
        chan_switch_start(ch_switch->ctxt);

    chan_switch_free(ch_switch);

    if (chan_env.current_ctxt->idx >= CHAN_TRAF_CTXT_CNT)
        chan_conn_less_ctxt_end(chan_env.current_ctxt);

    chan_switch_set_timer(chan_switch_pick());
}

/**
 ****************************************************************************************
 * @brief Function called once SW and HW are ready for a channel switch.
 *
 * It checks if ROC or SCAN are pending and if so change the target ctxt to this.
 ****************************************************************************************
 */
static void chan_pre_switch_channel(void)
{
    struct chan_ctxt_tag *new_ctxt = NULL;

    if (chan_env.status & CO_BIT(CHAN_ENV_SCAN_BIT))
    {
        // If a Scan Request is pending, jump on Scan channel
        new_ctxt = &chan_ctxt_pool[CHAN_SCAN_CTXT_IDX];
    }
    else if (chan_env.status & CO_BIT(CHAN_ENV_ROC_BIT))
    {
        // If a RoC Request is pending, jump on RoC Channel
        new_ctxt = &chan_ctxt_pool[CHAN_ROC_CTXT_IDX];
    }

    if (new_ctxt && (new_ctxt != chan_env.switch_ctxt))
    {
       if (chan_env.switch_ctxt)
       {
           chan_upd_ctxt_status(chan_env.switch_ctxt, CHAN_NOT_PROG);
       }
       chan_env.switch_ctxt = new_ctxt;
    }

    if (chan_env.switch_ctxt == NULL)
    {
        TRACE_CHAN(ERR, "No ctxt scheduled in chan_pre_switch_channel");
        return;
    }

    // Finally go on new channel
    chan_switch_channel();
}

/**
 ****************************************************************************************
 * @brief Function called during the Channel Switch procedure each time a NULL packet used
 *        for Absence indication is confirmed.
 *        The cfm_cnt counter value in the chan environment is decremented. When its value
 *        comes to 0, we can switch on a new channel.
 *
 * @param[in] dummy  Pointer to env passed when pushing the frame. (NULL in this case)
 * @param[in] status Status of the transmission of NULL packet.
 *
 ****************************************************************************************
 */
static void chan_tx_cfm(void *dummy, uint32_t status)
{
    TRACE_CHAN(NOA, "absence notification confirmed status=%lx (remains %d)",
               TR_32(status), chan_env.cfm_cnt - 1);

    // Sanity check - We shall be waiting for at least 1 TX confirmation
    ASSERT_ERR(chan_env.cfm_cnt);
    chan_env.cfm_cnt--;

    // Check if all confirmations have been received
    if (chan_env.cfm_cnt == 0)
    {
        // Request to go to IDLE again, so that we can perform the channel switch
        mm_force_idle_req();
        chan_pre_switch_channel();
    }
}

/**
 ****************************************************************************************
 * @brief For each VIF linked with the channel context we are switching on, we send a NULL
 *        frame with the PM bit set to 0 in order to indicate our presence.
 ****************************************************************************************
 */
static void chan_notify_presence(void)
{
    struct vif_info_tag *vif = vif_mgmt_first_used();
    struct chan_ctxt_tag *cur_ctxt = chan_env.current_ctxt;

    #if NX_MAC_HE
    // Restart the HE TB operation
    txl_he_tb_enable();
    #endif

    #if (NX_POWERSAVE)
    if ((ps_env.ps_on) && !(ps_env.prevent_sleep & PS_PSM_PAUSED))
    {
        // STA VIFs are not active, so no need to indicate our presence
        return;
    }
    #endif //(NX_POWERSAVE)

    // Reset the PwrMgt bit in all frames
    nxmac_pwr_mgt_setf(0);

    // Go through all the active VIFs
    while (vif != NULL)
    {
        if (vif->chan_ctxt == cur_ctxt)
        {
            if ((vif->type == VIF_STA) && vif->active)
            {
                #if (NX_P2P)
                bool send = true;

                // If VIF is for P2P, check that GO is present
                if (vif->p2p)
                {
                    send = p2p_info_tab[vif->p2p_index].is_go_present;
                }

                if (send)
                #endif //(NX_P2P)
                {
                    // Send a NULL frame to indicate the AP we are back on the channel
                    txl_frame_send_null_frame(vif->u.sta.ap_id, NULL, NULL);
                    TRACE_CHAN(NOA, "{VIF-%d} presence notification sent", vif->index);
                }
            }
        }

        vif = vif_mgmt_next(vif);
    }
}

/**
 ****************************************************************************************
 * @brief For each VIF linked with the channel context we are leaving, we send a NULL
 *        frame with the PM bit set to 1 in order to indicate our absence.
 *
 * @return Number of NULL frame pushed.
 ****************************************************************************************
 */
static int chan_notify_absence(void)
{
    int cfm_cnt = 0;
    struct vif_info_tag *vif = vif_mgmt_first_used();
    struct chan_ctxt_tag *cur_ctxt;
    uint8_t prev_status;

    #if NX_MAC_HE
    // Stop the HE TB operation
    txl_he_tb_disable();
    #endif

    if (chan_env.current_ctxt == NULL)
    {
        return 0;
    }

    cur_ctxt = chan_env.current_ctxt;

    #if (NX_POWERSAVE)
    if ((ps_env.ps_on) && !(ps_env.prevent_sleep & PS_PSM_PAUSED))
    {
        // STA VIFs are already in PS-mode, so no need to indicate our absence
        return 0;
    }
    #endif //(NX_POWERSAVE)


    // Set the PwrMgt bit in all frames
    nxmac_pwr_mgt_setf(1);

    // Go through all the active VIFs
    while (vif != NULL)
    {
        if (vif->chan_ctxt == cur_ctxt)
        {
            if ((vif->type == VIF_STA) && vif->active &&
                (vif->u.sta.ap_id != INVALID_STA_IDX))
            {
                #if (NX_P2P)
                bool send = true;

                // If VIF is for P2P, check that GO is present
                if (vif->p2p)
                {
                    send = p2p_info_tab[vif->p2p_index].is_go_present;
                }

                if (send)
                #endif //(NX_P2P)
                {
                    // Send a NULL frame to indicate to the AP that we are leaving the
                    // channel and update status to allow frame transmission.
                    prev_status = cur_ctxt->status;
                    cur_ctxt->status = CHAN_SENDING_NOA;
                    if (txl_frame_send_null_frame(vif->u.sta.ap_id, chan_tx_cfm, NULL) == CO_OK)
                    {
                        cfm_cnt++;
                    }
                    TRACE_CHAN(NOA, "{VIF-%d} absence notification sent", vif->index);
                    cur_ctxt->status = prev_status;
                }
            }
        }

        vif = vif_mgmt_next(vif);
    }

    // Save the number of TX confirmation expected
    chan_env.cfm_cnt = cfm_cnt;

    if (cfm_cnt)
    {
        chan_upd_ctxt_status(chan_env.switch_ctxt, CHAN_WAIT_NOA_CFM);
        // NULL frames programmed, we need to reactivate the HW
        mm_active();
    }

    return (cfm_cnt);
}

/**
 ****************************************************************************************
 * @brief This callback is called once HW has gone to the IDLE state during a channel
 *        switch procedure.
 ****************************************************************************************
 */
static void chan_goto_idle_cb(void)
{
    TRACE_CHAN(SWITCH, "HW entered idle state");

    // Flush all the TX and RX queues
    mm_force_idle_req();

    // Check if we need to notify the absence of the local VIFs to the peers
    if (chan_notify_absence())
        return;

    // Switch on a new channel
    chan_pre_switch_channel();
}

/**
 *****************************************************************************************
 * @brief Send a MM_FORCE_IDLE_REQ message to TASK_MM in order to require the HW to
 *        enter in IDLE mode
 ****************************************************************************************
 */
static void chan_send_force_idle(void)
{
    struct mm_force_idle_req *req = KE_MSG_ALLOC(MM_FORCE_IDLE_REQ,
                                                 TASK_MM, TASK_NONE,
                                                 mm_force_idle_req);

    req->cb = chan_goto_idle_cb;

    ke_msg_send(req);
}

/**
 ****************************************************************************************
 * @brief Switch from chan_env.current_ctxt to chan_env.switch_ctxt.
 *        HW is supposed to be in IDLE mode when entering in this function.
 *        current_ctxt value is replaced by chan_switch value and chan_switch is set to
 *        NULL.
 ****************************************************************************************
 */
static void chan_switch_channel(void)
{
    struct chan_ctxt_tag *ctxt = chan_env.switch_ctxt;
    struct mac_chan_op *chan = &ctxt->channel;

    PROF_CHAN_CTXT_SWITCH_CLR();
    PROF_MM_SET_CHANNEL_SET();

    TRACE_CHAN(INF, "{CTXT-%d} switch to channel freq=%dMHz bw=%dMHz", ctxt->idx,
               chan->prim20_freq, (1 <<  chan->type) * 20);

    // Program the RF with the new channel
    phy_set_channel(chan, PHY_PRIM);
    tpc_update_tx_power(chan->tx_power);

    PROF_MM_SET_CHANNEL_CLR();
    PROF_CHAN_CTXT_IDX_SET(ctxt->idx);

    // Set the basic rates in the HW according to the band
    nxmac_rates_set(mm_env.basic_rates[chan->band]);

    // Indicate to the upper layers that a channel switch has occurred
    chan_send_switch_ind(ctxt);

    if (chan_env.current_ctxt)
        chan_upd_ctxt_status(chan_env.current_ctxt, CHAN_NOT_PROG);

    // Store the current channel to the environment
    chan_env.current_ctxt = ctxt;
    // Channel switch is now complete
    chan_env.switch_ctxt = NULL;

    chan_upd_ctxt_status(ctxt, CHAN_WAITING_END);

    // Check if we switch to an operating channel (Traffic or RoC), or a scanning one
    if (ctxt->idx != CHAN_SCAN_CTXT_IDX)
    {
        if (ctxt->idx != CHAN_ROC_CTXT_IDX)
        {
            // Notify to the peer devices that we are back
            chan_notify_presence();
        }

        if (ctxt->idx < CHAN_TRAF_CTXT_CNT)
        {
            struct vif_info_tag *vif = vif_mgmt_first_used();

            while (vif)
            {
                if (vif->chan_ctxt == ctxt)
                {
                    #if (NX_TD)
                    // Keep in mind that we have been present on the traffic channel
                    td_env_tab[vif->index].has_active_chan = true;
                    #endif //(NX_TD)

                    vif_mgmt_send_postponed_frame(vif);
                }

                vif = (struct vif_info_tag *)co_list_next(&vif->list_hdr);
            }

            #if (!NX_UMAC_PRESENT)
            if (ctxt->taskid != TASK_NONE)
            {
                // Inform the host as required
                ke_msg_send_basic(MM_CHAN_CTXT_SCHED_CFM, ctxt->taskid, TASK_MM);
                ctxt->taskid = TASK_NONE;
            }
            #endif //(!NX_UMAC_PRESENT)
        }
    }
    else
    {
        // Clear CCA busy register (see MM_CHANNEL_SURVEY_IND use)
        nxmac_edca_cca_busy_set(0);

        // Confirm the channel switch to the scanning module
        ke_msg_send_basic(MM_SCAN_CHANNEL_START_IND, TASK_SCAN, TASK_NONE);
    }

    #if (NX_POWERSAVE)
    if (ctxt->idx >= CHAN_SCAN_CTXT_IDX)
    {
        // Disable PS while we are scanning
        GLOBAL_INT_DISABLE();
        ps_env.prevent_sleep |= PS_SCAN_ONGOING;
        GLOBAL_INT_RESTORE();

        // No PM bit set in ProbeReq
        chan_env.pm = nxmac_pwr_mgt_getf();
        nxmac_pwr_mgt_setf(0);
    }
    #endif //(NX_POWERSAVE)

    // Go to active state
    mm_active();
}

/**
 ****************************************************************************************
 * @brief Initiate channel switch procedure in order to jump to indicated channel.
 *        If we already are present on provided channel or if a switch procedure is
 *        already in progress, nothing is done.
 *        Else requests the HW to go in IDLE state.
 *
 * @param[in] ctxt   Channel context on which switch is required
 ****************************************************************************************
 */
static void chan_switch_start(struct chan_ctxt_tag *ctxt)
{
    TRACE_CHAN(SWITCH, "{CTXT-%d} Start switch", ctxt->idx);

    if (chan_env.switch_ctxt)
    {
        TRACE_CHAN(SWITCH, "Switch to {CTXT-%d} replaced by switch to {CTXT-%d}",
                   chan_env.switch_ctxt->idx, ctxt->idx);
        chan_env.switch_ctxt = ctxt;
        return;
    }

    if (chan_env.current_ctxt == ctxt)
    {
        TRACE_CHAN(SWITCH, "skip: already current channel");
        chan_upd_ctxt_status(chan_env.current_ctxt, CHAN_WAITING_END);
        return;
    }

    PROF_CHAN_CTXT_WAIT_END_CLR();

    if (chan_env.current_ctxt)
    {
        // Indicate to the upper layers that a channel switch will occur soon
        chan_send_pre_switch_ind(chan_env.current_ctxt);
    }

    #if NX_BCN_AUTONOMOUS_TX
    if (mm_bcn_transmitting())
    {
        TRACE_CHAN(ERR, "Channel switch while beacon are being transmitted");
        GLOBAL_INT_DISABLE();
        txl_cntrl_halt_ac(AC_BCN);
        txl_cntrl_flush_ac(AC_BCN, DESC_DONE_SW_TX_BIT);
        GLOBAL_INT_RESTORE();
    }
    #endif //NX_BCN_AUTONOMOUS_TX

    chan_env.switch_ctxt = ctxt;
    chan_upd_ctxt_status(ctxt, CHAN_GOTO_IDLE);
    chan_send_force_idle();
}

#if (NX_UMAC_PRESENT)
/**
 ****************************************************************************************
 * @brief Go through the list of channel contexts in order to check if a channel context,
 * using compatible parameters than those provided by the host, is currently used.
 *
 * By "compatible", it means that they have the same primary channel and one
 * configuration include the other one.
 * If the requested configuration is wider than the current configuration, then the
 * latter one is updated.
 * Note: The check assume valid 802.11 channel configuration (e.g. given a primary
 * channel there is only one valid 80MHz channel defined)
 *
 * @param[in]     chan_req  Parameters of the channel to be added
 * @param[in,out] idx       Pointer to the found channel context index
 *
 * @return true if a compatible channel context has been found, else false
 ****************************************************************************************
 */
static bool chan_check_chan(struct mac_chan_op const *chan_req, uint8_t *idx)
{
    bool found = false;

    for (int ctxt_idx = 0; ctxt_idx < CHAN_TRAF_CTXT_CNT; ctxt_idx++)
    {
        struct chan_ctxt_tag *ctxt = &chan_ctxt_pool[ctxt_idx];
        struct mac_chan_op *chan;

        // Only consider added channel contexts
        if (ctxt->idx == CHAN_CTXT_UNUSED)
            continue;

        chan = &ctxt->channel;

        if ((chan->band == chan_req->band) &&
            (chan->prim20_freq == chan_req->prim20_freq))
        {
            if ((chan->type == chan_req->type) &&
                ((chan->center1_freq == chan_req->center1_freq) &&
                 (chan->center2_freq == chan_req->center2_freq)))
            {
                // Exact same configuration
                *idx = ctxt_idx;
                found = true;
                break;
            }
            else if ((chan->type < chan_req->type) &&
                     (chan->type != PHY_CHNL_BW_160))
            {
                // Requested channel configuration "include" current configuration.
                // Update current configuration.
                chan->type = chan_req->type;
                chan->center1_freq = chan_req->center1_freq;
                chan->center2_freq = chan_req->center2_freq;

                TRACE_CHAN(CREATE, "{CTXT-%d} Update channel config freq=%dMHz bw=%dMHz",
                           ctxt->idx, chan->prim20_freq, (1 <<  chan->type) * 20);

                // Need to re-configure now if there is only one channel context active
                if ((chan_env.current_ctxt == ctxt) &&
                    (chan_env.nb_sched_ctxt == 1))
                {
                    phy_set_channel(chan, PHY_PRIM);
                }

                *idx = ctxt_idx;
                found = true;
                break;
            }
            else if ((chan->type > chan_req->type) &&
                     (chan_req->type != PHY_CHNL_BW_160))
            {
                // Requested channel configuration is included in current configuration
                *idx = ctxt_idx;
                found = true;
                break;
            }
        }
    }

    return (found);
}
#endif //(NX_UMAC_PRESENT)


#if (NX_P2P_GO && NX_BEACONING)
/**
 ****************************************************************************************
 * @brief Move P2P GO TBTT and NOA to resyncrhonize with other AP's TBTT
 *
 * When P2P GO interface is active with multiple channel context NOA are started to allow
 * context switch to respect TBTT on other (STA or P2P CLI) interfaces
 * @ref chan_start_p2pgo_noa.
 * When a drift is detected between the P2P GO TBTT and other AP's TBTT, this function is
 * called to move P2P GO TBTT and next occurence of NOA.
 * The goal is to keep other AP's TBTT inside the programmed NOA.
 *
 * @param[in] vif VIF structure of the P2P GO interface
 ****************************************************************************************
 */
static void chan_p2p_noa_resync(struct vif_info_tag *vif)
{
    vif->tbtt_switch.p2p_noa_tsf_update = mm_ap_tbtt_move(vif->tbtt_switch.p2p_noa_tsf_update);
    p2p_go_noa_concurrent_move(vif, vif->tbtt_switch.p2p_noa_tsf_update);
    vif->tbtt_switch.status &= ~(CHAN_TBTT_P2P_NOA_RESYNC);
    vif->tbtt_switch.status |= CHAN_TBTT_P2P_NOA_TSF_UPDATE;
}

/**
 ****************************************************************************************
 * @brief Post process after a TBTT move on a P2P GO interface.
 *
 * When TBTT of a P2P GO interface is moved this function is called to update the
 * mm_ap_pre_tbtt timer and the TBTT date in vif->tbtt_switch structure.
 * This is called when the TBTT window of the P2P GO ends.
 *
 * In some very unlikely cases, it may also reschedule context switches:
 * - This TBTT has been skipped by the CHAN module
 * - The last scheduled switch is based on previous TBTT date.
 *
 * @param[in] vif VIF structure of the P2P GO interface
 ****************************************************************************************
 */
static void chan_p2p_noa_tsf_updated(struct vif_info_tag *vif)
{
    struct chan_tbtt_tag *tbtt = &vif->tbtt_switch;

    mm_timer_set(&vif->tbtt_timer, vif->tbtt_timer.time + tbtt->p2p_noa_tsf_update);

    tbtt->status &= ~(CHAN_TBTT_P2P_NOA_TSF_UPDATE);
    if (tbtt->status & CHAN_TBTT_PROG)
    {
        struct chan_switch_tag *ch_switch = chan_switch_last();
        if (ch_switch && (ch_switch->tbtt == tbtt))
        {
            chan_cancel_all_switch();
            chan_add_next_tbtt(vif, vif->tbtt_timer.time, 0);
            chan_schedule_next_switch(vif, hal_machw_time(), 0);
        }
        else
        {
            chan_add_next_tbtt(vif, vif->tbtt_timer.time, 0);
        }
    }
}

/**
 ****************************************************************************************
 * @brief Check if drift in TBTT implies a resync (or restart) of NOA
 *
 * This is called when TBTT window starts on a VIF that induce a NOA on a P2P GO vif.
 * It compares the time between this TBTT and the end of the induced NOA, with the value
 * when the NOA has been created (@ref chan_start_p2pgo_noa).
 * - If the delta, in absolute, is greater than @ref CHAN_P2P_NOA_RESCHEDULE_THRESHOLD
 *   then it's too late and NOA will be restarted (i.e. @ref chan_start_p2pgo_noa will
 *   be called)
 * - If the delta, in absolute, is greater than @ref CHAN_P2P_NOA_RESCHEDULE_THRESHOLD
 *   then P2P GO TBTT and NOA will be resynced (@ref chan_p2p_noa_resync).
 *   If there are several NOA, then delta is averaged among all NOA.
 *
 * @param[in] vif        VIF that induce a NOA of the P2P GO interface
 * @param[in] tbtt_time  Date, in us, of the latest TBTT
 ****************************************************************************************
 */
static void chan_p2p_noa_resync_check(struct vif_info_tag *vif, uint32_t tbtt_time)
{
    struct chan_tbtt_tag *tbtt = &vif->tbtt_switch;
    struct vif_info_tag *p2pgo_vif;
    uint32_t tbtt_to_end;

    p2pgo_vif = &vif_info_tab[tbtt->p2p_noa_vif_index];

    if (p2pgo_vif->tbtt_switch.status & (CHAN_TBTT_P2P_NOA_TSF_UPDATE |
                                         CHAN_TBTT_P2P_NOA_RESYNC))
        return;

    tbtt_to_end = (p2p_go_get_next_noa_end_date(p2pgo_vif, tbtt->p2p_noa_index) -
                   tbtt_time);
    tbtt->p2p_noa_drift = tbtt->p2p_noa_tbtt_to_end - tbtt_to_end;

    if (co_abs(tbtt->p2p_noa_drift) > CHAN_P2P_NOA_RESCHEDULE_THRESHOLD)
    {
        // For some reason we were not able to resync NOA, so simply reschedule them
        // on next TBTT of P2P GO interface
        p2pgo_vif->tbtt_switch.status &= ~(CHAN_TBTT_P2P_NOA_RESYNC);
        p2pgo_vif->tbtt_switch.status |= CHAN_TBTT_SCHEDULE_P2P_NOA;
    }
    else if (co_abs(tbtt->p2p_noa_drift) > CHAN_P2P_NOA_RESYNC_THRESHOLD)
    {
        int16_t min = 0, max = 0;
        struct vif_info_tag *_vif;
        _vif = vif_mgmt_first_used();
        while (_vif)
        {
            if (_vif->tbtt_switch.p2p_noa_vif_index == p2pgo_vif->index)
            {
                if (_vif->tbtt_switch.p2p_noa_drift < min)
                    min = _vif->tbtt_switch.p2p_noa_drift;
                if (_vif->tbtt_switch.p2p_noa_drift > max)
                    max = _vif->tbtt_switch.p2p_noa_drift;
            }
            _vif = vif_mgmt_next(_vif);
        }

        if ((min >= 0 && max >= 0) ||
            (min <= 0 && max <= 0))
        {
            p2pgo_vif->tbtt_switch.status |= CHAN_TBTT_P2P_NOA_RESYNC;
            p2pgo_vif->tbtt_switch.p2p_noa_tsf_update = (min + max) / 2;
        }
    }
}

#endif //(NX_P2P_GO && NX_BEACONING)
/**
 ****************************************************************************************
 * @brief Reset TBTT list in channel module
 *
 ****************************************************************************************
 */
static void chan_reset_tbtt_list(void)
{
    int i;
    co_list_init(&chan_env.list_tbtt);

    for (i = 0 ; i < NX_VIRT_DEV_MAX; i++) {
        struct vif_info_tag *vif = &vif_info_tab[i];
        vif->tbtt_switch.status &= ~(CHAN_TBTT_PROG);
    }
}

/**
 ****************************************************************************************
 * @brief Initialize all parameters stored in a channel context entry.
 *
 * @param[in] ctxt  Channel context entry to be initialazed
 ****************************************************************************************
 */
static void chan_ctxt_init(struct chan_ctxt_tag *ctxt)
{
    memset(ctxt, 0, sizeof(struct chan_ctxt_tag));

    ctxt->taskid = TASK_NONE;
    ctxt->idx    = CHAN_CTXT_UNUSED;
    #if (NX_P2P)
    ctxt->p2pgo_vif_index = INVALID_VIF_IDX;
    #endif //(NX_P2P)
}

/*
 * PUBLIC FUNCTIONS DEFINITION
 ****************************************************************************************
 */
void chan_init(void)
{
    memset(&chan_env, 0, sizeof(chan_env));

    // Initialize the free channel context list
    for (int i = 0; i < CHAN_CHAN_CTXT_CNT; i++)
    {
        struct chan_ctxt_tag *ctxt = &chan_ctxt_pool[i];

        chan_ctxt_init(ctxt);

        if (i < NX_CHAN_CTXT_CNT)
        {
            // Add it to the free channel context list
            co_list_push_back(&chan_env.list_free_ctxt, &ctxt->list_hdr);
        }
        else if (i == CHAN_SCAN_CTXT_IDX)
        {
            ctxt->channel.center2_freq = 0;
            ctxt->channel.type = PHY_CHNL_BW_20;
        }
    }

    co_list_init(&chan_env.list_free_switch);
    co_list_init(&chan_env.list_switch);
    for (int i = 0; i < CHAN_SWITCH_CNT; i++)
    {
        chan_switch_free(&chan_switch_pool[i]);
    }

    // Set Connection Less Delay Time callback
    chan_env.tmr_conn_less.cb = chan_conn_less_delay_evt;
    // Set Channel Switch Timer callback
    chan_env.tmr_switch.cb = chan_tmr_switch_cb;
    chan_env.tmr_switch.env = NULL;
}

#if (NX_HW_SCAN)
void chan_scan_req(uint8_t band, uint16_t freq, int8_t pwr, uint32_t duration_us,
                   uint8_t flags, uint8_t vif_index)
{
    struct chan_ctxt_tag *ctxt = &chan_ctxt_pool[CHAN_SCAN_CTXT_IDX];

    TRACE_CHAN(CON_LESS, "{VIF-%d} Start scan freq=%dMHz duration=%ldus pwr=%ddBm",
               vif_index, freq, TR_32(duration_us), pwr);

    ASSERT_ERR(ctxt->idx == CHAN_CTXT_UNUSED);

    ctxt->idx                  = CHAN_SCAN_CTXT_IDX;
    ctxt->taskid               = TASK_NONE;
    ctxt->channel.band         = band;
    ctxt->channel.center1_freq = freq;
    ctxt->channel.prim20_freq  = freq;
    ctxt->channel.tx_power     = pwr;
    ctxt->channel.flags        = flags;
    ctxt->vif_index            = vif_index;
    ctxt->status               = CHAN_NOT_PROG;
    ctxt->duration_us          = duration_us;

    chan_env.status |= CO_BIT(CHAN_ENV_SCAN_WAIT_BIT);

    // Check if we can start the Connection Less Delay Timer
    if (!(chan_env.status & CO_BIT(CHAN_ENV_DELAY_PROG_BIT)))
    {
        chan_conn_less_delay_prog();
    }
}
#endif //(NX_HW_SCAN)

uint8_t chan_roc_req(struct mm_remain_on_channel_req const *req, ke_task_id_t taskid)
{
    uint8_t status = CO_FAIL;
    struct chan_ctxt_tag *ctxt = &chan_ctxt_pool[CHAN_ROC_CTXT_IDX];

    TRACE_CHAN(CON_LESS, "ROC op=%d freq=%d MHz duration=%dms pwr=%ddBm",
               req->op_code, req->chan.prim20_freq, req->duration_ms, req->chan.tx_power);

    switch (req->op_code)
    {
        case (MM_ROC_OP_START):
        {
            // Check if a remain on channel is in progress
            if (ctxt->idx != CHAN_CTXT_UNUSED)
            {
                break;
            }

            ctxt->idx = CHAN_ROC_CTXT_IDX;
            ctxt->channel = req->chan;
            ctxt->taskid = taskid;
            ctxt->status = CHAN_NOT_PROG;
            ctxt->duration_us = (req->duration_ms * 1000);
            ctxt->vif_index = req->vif_index;

            // If internal RoC (use for beacon detection), do not delay
            #if NX_UMAC_PRESENT && NX_TDLS
            if ((taskid == TASK_MM) || (taskid == TASK_TDLS))
            #else
            if (taskid == TASK_MM)
            #endif
            {
                chan_env.status |= CO_BIT(CHAN_ENV_ROC_BIT);
                #if NX_UMAC_PRESENT && NX_TDLS
                if (taskid == TASK_TDLS)
                {
                    ctxt->roc_tdls = true;
                }
                #endif

                chan_fix_ctxt_until(ctxt);
            }
            else
            {
                chan_env.status |= CO_BIT(CHAN_ENV_ROC_WAIT_BIT);
                chan_conn_less_delay_prog();
            }

            status = CO_OK;
        } break;

        case (MM_ROC_OP_CANCEL):
        {
            // Check if a remain on channel context is used
            if (ctxt->idx == CHAN_CTXT_UNUSED)
            {
                status = CO_OK;
                break;
            }

            #if NX_UMAC_PRESENT && NX_TDLS
            if (taskid == TASK_TDLS)
            {
                ctxt->roc_tdls = false;
            }
            #endif

            // React depending on the current context state
            switch (ctxt->status)
            {
                case (CHAN_NOT_PROG):
                {
                    chan_env.status &= ~CO_BIT(CHAN_ENV_ROC_WAIT_BIT);

                    #if NX_UMAC_PRESENT && NX_TDLS
                    if (taskid == TASK_TDLS)
                    {
                        tdls_send_chan_switch_base_ind(ctxt);
                    }
                    else
                    #endif
                    {
                        chan_send_roc_exp_ind(ctxt);
                    }
                } break;

                case (CHAN_GOTO_IDLE):
                case (CHAN_WAIT_NOA_CFM):
                {
                    chan_env.status &= ~CO_BIT(CHAN_ENV_ROC_BIT);
                    chan_env.switch_ctxt = NULL;
                } break;

                case (CHAN_WAITING_END):
                {
                    chan_switch_reset_timer();
                    chan_tmr_switch_cb(chan_switch_pick());
                } break;

                default:
                {
                } break;
            }

            // Reset the index
            ctxt->idx = CHAN_CTXT_UNUSED;

            // Check if we have to stop the delay timer
            if ((chan_env.status & CO_BIT(CHAN_ENV_DELAY_PROG_BIT)) &&
                ((chan_env.status & CO_BIT(CHAN_ENV_SCAN_WAIT_BIT)) == 0))
            {
                chan_env.status &= ~CO_BIT(CHAN_ENV_DELAY_PROG_BIT);
                mm_timer_clear(&chan_env.tmr_conn_less);
            }

            status = CO_OK;
        } break;

        default:
            break;
    }

    return (status);
}

uint8_t chan_ctxt_add(struct mac_chan_op const *chan_req, uint8_t *idx)
{
    struct chan_ctxt_tag *ctxt;

    #if (NX_UMAC_PRESENT)
    // Check if a channel with the provided parameters as already been added
    if (chan_check_chan(chan_req, idx))
    {
        // Channel already exists, allocation is successful
        return CO_OK;
    }
    #endif //(NX_UMAC_PRESENT)

    // Allocate a channel context from the list
    ctxt = (struct chan_ctxt_tag *)co_list_pop_front(&chan_env.list_free_ctxt);
    if (ctxt == NULL)
    {
        TRACE_CHAN(ERR, "No more channel context freq=%dMHz bw=%dMHz",
                   chan_req->prim20_freq, (1 << chan_req->type) * 20 );
        return CO_FAIL;
    }

    // Initialize the operating channel context structure
    *idx = ctxt->idx = CO_GET_INDEX(ctxt, chan_ctxt_pool);
    ctxt->channel = *chan_req;

    TRACE_CHAN(CREATE, "{CTXT-%d} Create channel context: freq=%dMHz bw=%dMHz pwr=%ddBm",
               *idx, chan_req->prim20_freq, (1 << chan_req->type) * 20,  chan_req->tx_power);

    return CO_OK;
}

void chan_ctxt_del(uint8_t chan_idx)
{
    struct chan_ctxt_tag *ctxt = &chan_ctxt_pool[chan_idx];

    TRACE_CHAN(CREATE, "{CTXT-%d} Delete channel context",  chan_idx);

    // Sanity checks - An unused channel should not be freed
    ASSERT_ERR(ctxt->idx != CHAN_CTXT_UNUSED);
    // Sanity checks - No more VIFs should be linked with the channel
    ASSERT_ERR(ctxt->nb_linked_vif == 0);

    // Push back the channel context in the free list
    co_list_push_back(&chan_env.list_free_ctxt, &ctxt->list_hdr);

    // Reset channel context information
    chan_ctxt_init(ctxt);
}

void chan_ctxt_link(uint8_t vif_idx, uint8_t ctxt_idx)
{
    struct chan_ctxt_tag *ctxt = &chan_ctxt_pool[ctxt_idx];
    struct vif_info_tag *vif   = &vif_info_tab[vif_idx];

    // Sanity checks
    ASSERT_ERR(vif->chan_ctxt == NULL);
    ASSERT_ERR(ctxt->idx != CHAN_CTXT_UNUSED);

    TRACE_CHAN(CREATE, "{CTXT-%d} link to {VIF-%d}: status=%d nb_vif=%d",
               ctxt_idx, vif_idx, ctxt->status, ctxt->nb_linked_vif);

    // Channel context will now be linked to VIF
    vif->chan_ctxt = ctxt;
    vif->tbtt_switch.status = 0;

    // Increase number of VIFs linked with the channel context
    ctxt->nb_linked_vif++;

    // Schedule the channel if first link
    if (ctxt->nb_linked_vif == 1)
    {
        // Update channel status
        ctxt->status = CHAN_NOT_PROG;

        // Increase number of scheduled contexts
        chan_env.nb_sched_ctxt++;
        ASSERT_ERR(chan_env.nb_sched_ctxt <= 2);

        if (chan_env.nb_sched_ctxt == 2)
        {
            chan_reset_tbtt_list();
        }

        // Insert the context in the list of scheduled contexts
        co_list_push_back(&chan_env.list_sched_ctxt, &ctxt->list_hdr);

        // Switch to this new channel as soon as posible.
        // Mandatory for STA in order to finish connection.
        if (chan_env.status & CHAN_ROC_SCAN_PENDING_MASK)
        {
            struct chan_switch_tag *ch_switch = chan_switch_pick();
            ASSERT_ERR(ch_switch);
            ch_switch->ctxt = ctxt;
        }
        else if (chan_env.switch_ctxt)
        {
            chan_env.switch_ctxt = ctxt;
        }
        else
        {
            chan_switch_start(ctxt);
        }
    }

    #if (NX_P2P_GO)
    vif->tbtt_switch.p2p_noa_vif_index = INVALID_VIF_IDX;

    if (chan_is_p2p_go(vif))
    {
        vif->tbtt_switch.status |= CHAN_TBTT_SCHEDULE_P2P_NOA;
    }
    else
    {
        struct vif_info_tag *p2pgo_vif;
        if (chan_get_p2pgo(&p2pgo_vif) &&
            (p2pgo_vif->chan_ctxt != ctxt))
        {
            // Cannot start NOA here as TBTT date is probably not known yet for this vif
            vif->tbtt_switch.status |= CHAN_TBTT_SCHEDULE_P2P_NOA;
        }

    }

    #endif //(NX_P2P_GO)

    chan_update_tx_power(ctxt);

    #if NX_MON_DATA
    if (vif_mgmt_env.monitor_vif != vif_idx && vif_mgmt_env.monitor_vif != INVALID_STA_IDX)
    {
        chan_ctxt_link_monitor(ctxt_idx);
    }
    #endif
}

void chan_ctxt_unlink(uint8_t vif_idx)
{
    struct vif_info_tag *vif = &vif_info_tab[vif_idx];
    struct chan_ctxt_tag *ctxt = vif->chan_ctxt;

    // Sanity checks
    ASSERT_ERR(ctxt != NULL);
    TRACE_CHAN(CREATE, "{CTXT-%d} unlink from {VIF-%d}: status=%d nb_vif=%d",
               ctxt->idx, vif_idx, ctxt->status, ctxt->nb_linked_vif);

    // Remove TBTT Switch element and reset status
    chan_tbtt_remove(&vif->tbtt_switch);
    if (vif->tbtt_switch.status & CHAN_TBTT_WAIT_TO)
        chan_env.nb_active_tbtt--;
    vif->tbtt_switch.status = 0;

    // Channel context will now be linked to VIF
    vif->chan_ctxt = NULL;
    // Decrease number of VIFs linked with the channel context
    ctxt->nb_linked_vif--;

    if (ctxt->status != CHAN_NOT_SCHEDULED)
    {
        #if (NX_P2P_GO)
        struct vif_info_tag *p2pgo_vif;

        if (ctxt->p2pgo_vif_index == vif->index)
        {
            chan_stop_p2pgo_noa(vif);
            ctxt->p2pgo_vif_index = INVALID_VIF_IDX;
        }
        else if (chan_get_p2pgo(&p2pgo_vif) &&
                 (p2pgo_vif->chan_ctxt != ctxt))
        {
            if (ctxt->nb_linked_vif)
                chan_start_p2pgo_noa(p2pgo_vif);
            else
                chan_stop_p2pgo_noa(p2pgo_vif);
        }
        #endif //(NX_P2P_GO)

        // If no more VIFs are linked with the channel, unschedule it
        if (!ctxt->nb_linked_vif)
        {
            co_list_extract(&chan_env.list_sched_ctxt, &ctxt->list_hdr);
            chan_upd_ctxt_status(ctxt, CHAN_NOT_SCHEDULED);

            // Decrease number of scheduled channels
            chan_env.nb_sched_ctxt--;
            ASSERT_ERR(chan_env.nb_sched_ctxt <= 1);

            if (chan_env.nb_sched_ctxt)
            {
                chan_reset_tbtt_list();
                if (!chan_env.fix_ctxt)
                    chan_cancel_all_switch();
            }

            if (chan_env.current_ctxt == ctxt)
            {
                chan_env.current_ctxt = NULL;

                if (chan_env.nb_sched_ctxt)
                {
                    chan_switch_start(chan_get_next_traf_ctxt(ctxt));
                }
            }
            else if (chan_env.switch_ctxt == ctxt)
            {
                chan_env.switch_ctxt = chan_get_next_traf_ctxt(ctxt);
            }
        }
    }

    #if (NX_UMAC_PRESENT)
    if (ctxt->nb_linked_vif == 0)
    {
        chan_ctxt_del(ctxt->idx);
    }
    #endif //(NX_UMAC_PRESENT)

    chan_update_tx_power(ctxt);
}

void chan_ctxt_update(struct mm_chan_ctxt_update_req const *upd_req)
{
    struct chan_ctxt_tag *ctxt = &chan_ctxt_pool[upd_req->chan_index];
    struct mac_chan_op const *chan = &upd_req->chan;

    TRACE_CHAN(CREATE, "{CTXT-%d} update config: freq=%dMHz bw=%dMHz pwr=%ddBm",
               upd_req->chan_index, chan->prim20_freq, (1 << chan->type) * 20,
               chan->tx_power);

    // Update the channel context
    ctxt->channel = upd_req->chan;

    if (chan_env.current_ctxt == ctxt)
    {
        // Handle the packets already in the RX queue to ensure that the channel info
        // indicated to the upper MAC is correct. This has to be done with interrupts
        // disabled, as the normal handling of the packets is done under interrupt.
        GLOBAL_INT_DISABLE();
        rxl_mpdu_isr();
        rxl_cntrl_evt(0);
        GLOBAL_INT_RESTORE();

        PROF_MM_SET_CHANNEL_SET();

        // Program the RF with the new channel
        phy_set_channel(chan, PHY_PRIM);
        tpc_update_tx_power(ctxt->channel.tx_power);

        PROF_MM_SET_CHANNEL_CLR();
    }
}

#if (!NX_UMAC_PRESENT)
void chan_ctxt_sched(struct mm_chan_ctxt_sched_req const *sched_req, ke_task_id_t taskid)
{
    struct chan_ctxt_tag *ctxt;

    TRACE_CHAN(SWITCH, "{CTXT-%d} schedule requested", sched_req->chan_index);

    ctxt = &chan_ctxt_pool[sched_req->chan_index];

    // Sanity check
    ASSERT_ERR(ctxt->status != CHAN_NOT_SCHEDULED);

    // Check if we already are on the channel
    if (ctxt->status >= CHAN_WAITING_END)
    {
        // Send the confirmation immediately
        ke_msg_send_basic(MM_CHAN_CTXT_SCHED_CFM, taskid, TASK_MM);
    }
    else
    {
        // Store the task id, CHAN_CTXT_SCHED_CFM will be sent next time we jump on the channel
        ctxt->taskid = taskid;
    }
}
#endif //(!NX_UMAC_PRESENT)

#if (NX_UMAC_PRESENT)
void chan_ctxt_link_monitor(uint8_t ctxt_idx)
{
    struct vif_info_tag *vif = &vif_info_tab[vif_mgmt_env.monitor_vif];

    // Check if monitor channel is already linked
    if (vif->chan_ctxt)
    {
        // Unlink monitor channel
        chan_ctxt_unlink(vif_mgmt_env.monitor_vif);
    }
    else
    {
        struct me_set_active_req *idle =  KE_MSG_ALLOC(ME_SET_ACTIVE_REQ, TASK_ME,
                                                       TASK_ME, me_set_active_req);

        idle->active = true;
        idle->vif_idx = vif->index;
        ke_msg_send(idle);
    }

    // Link monitor channel
    chan_ctxt_link(vif_mgmt_env.monitor_vif, ctxt_idx);
}
#endif //(NX_UMAC_PRESENT)

int chan_tbtt_start(struct vif_info_tag *vif, uint32_t tbtt_time, uint32_t next_tbtt_time)
{
    struct chan_tbtt_tag *tbtt = &vif->tbtt_switch;
    struct chan_tbtt_tag *tbtt_scheduled = chan_tbtt_pick();
    bool skip = (!tbtt_scheduled ||
                 tbtt_scheduled->time != tbtt_time);
    bool on_channel = (vif->chan_ctxt == chan_env.current_ctxt);


    while (skip && tbtt_scheduled &&
           hal_machw_time_cmp(tbtt_scheduled->time, tbtt_time))
    {
        chan_tbtt_remove(tbtt_scheduled);
        TRACE_CHAN(ERR, "Drop next scheduled {VIF-%d} %t in the past",
                   tbtt_scheduled->vif_index, TR_32(tbtt_scheduled->time));
        tbtt_scheduled = chan_tbtt_pick();
        skip = (!tbtt_scheduled ||
                 tbtt_scheduled->time != tbtt_time);
    }

    TRACE_CHAN(TBTT, "{VIF-%d}{CTXT-%d} TBTT start skip_it=%d, current_ctxt=%d, tbtt_time=%t",
               vif->index, vif->chan_ctxt->idx, skip,
               chan_env.current_ctxt ? chan_env.current_ctxt->idx : -1, TR_32(tbtt_time));

    if (chan_env.nb_sched_ctxt < 2)
    {
        tbtt->status |= CHAN_TBTT_WAIT_TO;
        chan_env.nb_active_tbtt++;
        return !on_channel;
    }


    #if (NX_P2P_GO && NX_BEACONING)
    if (tbtt->p2p_noa_vif_index != INVALID_VIF_IDX)
    {
        chan_p2p_noa_resync_check(vif, tbtt_time);
    }
    else if (chan_is_p2p_go(vif))
    {
        if (tbtt->status & CHAN_TBTT_SCHEDULE_P2P_NOA)
        {
            chan_start_p2pgo_noa(vif);
            tbtt->status &= ~(CHAN_TBTT_SCHEDULE_P2P_NOA);
        }
        else if (tbtt->status & CHAN_TBTT_P2P_NOA_RESYNC)
        {
            chan_p2p_noa_resync(vif);
        }
    }
    #endif //(NX_P2P_GO)

    if (skip)
    {
        tbtt->status |= CHAN_TBTT_SKIPPED;
        #if (NX_P2P_GO)
        if (chan_is_p2p_go(vif))
        {
            if ((tbtt->time != next_tbtt_time) &&
                hal_machw_time_cmp(tbtt->time, next_tbtt_time + 50) &&
                hal_machw_time_cmp(next_tbtt_time, tbtt->time + 50))
                tbtt->time = next_tbtt_time;
        }
        #endif //(NX_P2P_GO)
    }
    else
    {
        #if (NX_P2P_GO)
        if (chan_is_p2p_go(vif))
        {
            // Need to add it now, to set single NOA in beacon if next TBTT
            // need to be skip. Moreoever, contrary to STA/P2P cli vifs tbbt time
            // won't change when chan_bcn_to_evt is called so it's ok to do it now.
            chan_add_next_tbtt(vif, next_tbtt_time, 0);
            tbtt->status |= CHAN_TBTT_ALREADY_ADDED;
        }
        else
        #endif //(NX_P2P_GO)
        {
            chan_tbtt_remove(tbtt);
        }
        tbtt->status |= CHAN_TBTT_WAIT_TO;
        chan_env.nb_active_tbtt++;
    }

    return !on_channel;
}

void chan_tbtt_updated(struct vif_info_tag *vif)
{
    struct chan_tbtt_tag *tbtt = &vif->tbtt_switch;
    struct chan_switch_tag *ch_switch;

    if (chan_env.nb_sched_ctxt < 2)
        return;

    #if (NX_P2P_GO)
    if (tbtt->status & CHAN_TBTT_SCHEDULE_P2P_NOA)
    {
        struct vif_info_tag *p2pgo_vif;
        if (chan_get_p2pgo(&p2pgo_vif))
            chan_start_p2pgo_noa(p2pgo_vif);
        tbtt->status &= ~(CHAN_TBTT_SCHEDULE_P2P_NOA);
    }
    #endif //(NX_P2P_GO)

    if (tbtt->status & CHAN_TBTT_WAIT_TO)
    {
        // This will be taken into account in chan_bcn_to_evt()
        return;
    }
    else if (tbtt->status & CHAN_TBTT_SKIPPED)
    {
        // Even if TBTT is skipped if beacon is received need to update its next TBTT date
        // Remove SKIPPED status not to re-add the TBTT in chan_bcn_to_evt.
        tbtt->status &= ~(CHAN_TBTT_SKIPPED);
    }

    TRACE_CHAN(TBTT, "{VIF-%d} Beacon received outside of TBTT window", vif->index);

    ch_switch = chan_switch_last();
    if (ch_switch && (ch_switch->tbtt == tbtt))
    {
        // Channel switches have been scheduled based on previous TBTT, need to update
        if ((hal_machw_time_cmp(tbtt->time, vif->tbtt_timer.time) &&
             hal_machw_time_cmp(vif->tbtt_timer.time, tbtt->time + CHAN_SWITCH_DELAY)) ||
            (hal_machw_time_cmp(vif->tbtt_timer.time, tbtt->time) &&
             hal_machw_time_cmp(tbtt->time, vif->tbtt_timer.time + CHAN_SWITCH_DELAY)))
        {
            // Delta is small, simply update switch timer
            tbtt->time = vif->tbtt_timer.time;
            if (hal_machw_time_cmp(tbtt->time - CHAN_SWITCH_DELAY, ch_switch->time))
            {
                ch_switch->time = tbtt->time - CHAN_SWITCH_DELAY;
                if (ch_switch == chan_env.tmr_switch.env)
                {
                    chan_switch_reset_timer();
                    chan_switch_set_timer(ch_switch);
                }
            }
        }
        else
        {
            // Delta is significant, need to re-schedule switches
            chan_cancel_all_switch();
            chan_add_next_tbtt(vif, vif->tbtt_timer.time, 0);
            chan_schedule_next_switch(vif, hal_machw_time(), 0);
        }
    }
    else
    {
        // This TBTT hasn't been used to schedule channel switches, simply update it
        chan_add_next_tbtt(vif, vif->tbtt_timer.time, 0);
    }
}

void chan_bcn_to_evt(struct vif_info_tag *vif)
{
    struct chan_tbtt_tag *tbtt = &vif->tbtt_switch;
    bool sched_switch = true;
    uint32_t next_tbtt_time, prev_tbtt_time;

    TRACE_CHAN(TBTT, "{VIF-%d}{CTXT-%d} BCN TimeOut status = %d beacon_lost = %d",
               tbtt->vif_index, vif->chan_ctxt->idx, tbtt->status,
               #if NX_CONNECTION_MONITOR
               vif->u.sta.beacon_loss_cnt
               #else
               -1
               #endif
               );

    if (chan_env.nb_sched_ctxt < 2)
    {
        if ((tbtt->status & CHAN_TBTT_WAIT_TO))
        {
            tbtt->status &= ~(CHAN_TBTT_WAIT_TO);
            chan_env.nb_active_tbtt--;
            if (!chan_env.nb_active_tbtt &&
                (chan_env.status & CO_BIT(CHAN_ENV_FIX_CTXT_PENDING_BIT)))
                chan_fix_ctxt_until(chan_env.fix_ctxt);
        }
        return;
    }

    #if (NX_P2P_GO && NX_BEACONING)
    if ((tbtt->status & CHAN_TBTT_P2P_NOA_TSF_UPDATE))
        chan_p2p_noa_tsf_updated(vif);
    #endif //(NX_P2P_GO && NX_BEACONING)

    next_tbtt_time = vif->tbtt_timer.time;
    if (tbtt->status & CHAN_TBTT_SKIPPED)
    {
        // This TBTT has been skipped by CHAN module, and no beacon has been received
        // (otherwise chan_tbtt_updated would have reset the SKIPPED status).
        // In some cases TBTT may not be present in the list, so re-add it if needed.
        if (!(tbtt->status & CHAN_TBTT_PROG))
            chan_add_next_tbtt(vif, next_tbtt_time, 0);
        tbtt->status &= ~(CHAN_TBTT_SKIPPED);
        return;
    }
    else if (!(tbtt->status & CHAN_TBTT_WAIT_TO))
    {
        return;
    }

    tbtt->status &= ~(CHAN_TBTT_WAIT_TO);

    // Scheduled fixed context if any pending (Do it before adding TBTT)
    chan_env.nb_active_tbtt--;
    if (!chan_env.nb_active_tbtt &&
        (chan_env.status & CO_BIT(CHAN_ENV_FIX_CTXT_PENDING_BIT)))
    {
        chan_fix_ctxt_until(chan_env.fix_ctxt);
        sched_switch = false;
    }
    else if (chan_env.nb_active_tbtt)
    {
        sched_switch = false;
    }

    // Add next TBTT unless it has already been added
    prev_tbtt_time = tbtt->time;
    if (tbtt->status & CHAN_TBTT_ALREADY_ADDED)
    {
        tbtt->status &= ~(CHAN_TBTT_ALREADY_ADDED);
        prev_tbtt_time -= chan_vif_tbtt_intv(vif);
    }
    else
    {
        chan_add_next_tbtt(vif, next_tbtt_time, 0);
    }

    // And schedule switches unless a Fixed context has been configured.
    if (sched_switch)
    {
        uint32_t now = hal_machw_time();
        uint32_t tbtt_dur = now - prev_tbtt_time;
        chan_schedule_next_switch(vif, now, tbtt_dur);
    }
}

void chan_bcn_detect_start(struct vif_info_tag *vif)
{
    struct chan_ctxt_tag *ctxt = vif->chan_ctxt;

    // Sanity check
    ASSERT_ERR(ctxt);

    TRACE_CHAN(TBTT, "{CTXT-%d} Start Beacon detection procedure for {VIF-%d}",
               ctxt->idx, vif->index);

    // TODO [LT] - Beacon Detection status should be kept by VIF
    // TODO [LT] - Once beacon is received we could use Cancel Remain on Channel
    if (!(chan_env.status & CO_BIT(CHAN_ENV_BCN_DETECT_BIT)) &&
        (chan_env.nb_sched_ctxt > 1))
    {
        struct sta_info_tag *sta = &sta_info_tab[vif->u.sta.ap_id];

        // Start a remain on channel procedure in order to received all the beacons
        struct mm_remain_on_channel_req *req = KE_MSG_ALLOC(MM_REMAIN_ON_CHANNEL_REQ,
                                                            TASK_MM, TASK_MM,
                                                            mm_remain_on_channel_req);

        // Fill request parameters
        req->op_code   = MM_ROC_OP_START;
        req->vif_index = vif->index;
        req->chan      = ctxt->channel;
        req->duration_ms  = (sta->bcn_int - 5000) / 1000;

        ke_msg_send(req);

        // Set Beacon Detection bit in environment status
        chan_env.status |= CO_BIT(CHAN_ENV_BCN_DETECT_BIT);
    }
}

#if (NX_P2P)
void chan_p2p_absence_update(struct chan_ctxt_tag *ctxt, bool absence)
{
    // TODO
}
#endif //(NX_P2P)

bool chan_is_on_channel(struct vif_info_tag *vif)
{
    bool is_on_channel = false;

    if (chan_env.current_ctxt)
    {
        if (chan_env.current_ctxt->idx < CHAN_SCAN_CTXT_IDX)
        {
            is_on_channel = (vif->chan_ctxt == chan_env.current_ctxt);
        }
        else
        {
            is_on_channel = (chan_env.current_ctxt->vif_index == vif->index);
        }
    }

    return (is_on_channel);
}

bool chan_is_tx_allowed(struct vif_info_tag *vif)
{
    if ((!chan_is_on_channel(vif)) ||
        (chan_env.switch_ctxt &&
         (chan_env.current_ctxt->status != CHAN_SENDING_NOA)))
    {
        return false;
    }

    return true;
}

bool chan_is_on_operational_channel(struct vif_info_tag *vif)
{
    bool is_on_channel = false;

    if (chan_env.current_ctxt)
    {
        if (chan_env.current_ctxt->idx < CHAN_SCAN_CTXT_IDX)
        {
            is_on_channel = (vif->chan_ctxt == chan_env.current_ctxt);
        }
    }

    return (is_on_channel);
}

void chan_update_tx_power(struct chan_ctxt_tag *ctxt)
{
    int8_t i, min_pwr = VIF_UNDEF_POWER;

    if (ctxt->nb_linked_vif)
    {
        for (i = 0 ; i < NX_VIRT_DEV_MAX; i++) {
            struct vif_info_tag *vif = &vif_info_tab[i];

            if (vif->chan_ctxt == ctxt)
            {
                #if NX_UMAC_PRESENT
                if (vif->user_tx_power < min_pwr)
                    min_pwr = vif->user_tx_power;
                #endif
                if (vif->tx_power < min_pwr)
                    min_pwr = vif->tx_power;

                break;
            }
        }
    }

    if (min_pwr != VIF_UNDEF_POWER)
        ctxt->channel.tx_power = min_pwr;
    else
        // Should only be true when no vif is linked to the context
        ctxt->channel.tx_power = MM_DEFAULT_TX_POWER;

    TRACE_LMAC(TPC, "{CTXT-%d} updated channel context tx_power: %ddBm",
               ctxt->idx, ctxt->channel.tx_power);

    // apply update immediately if this is the active channel
    if (chan_env.current_ctxt == ctxt)
        tpc_update_tx_power(ctxt->channel.tx_power);
}

#endif //(NX_CHNL_CTXT)

/// @} end of group
