/**
 ****************************************************************************************
 *
 * @file hal_dma.c
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 * @brief General purpose DMA functions
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup GPDMA
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
// for mem* functions
#include <string.h>

// for assertions
#include "dbg_assert.h"

#include "ke_event.h"

#include "hal_dma.h"

#include "dbg.h"

#if NX_GP_DMA
/// Array of conversion between the DMA transfer type and the DMA channel that is used
const uint8_t dma2chan[DMA_MAX] =
{
    [DMA_DL] = IPC_DMA_CHANNEL_CTRL_TX,
    [DMA_UL] = IPC_DMA_CHANNEL_CTRL_RX,
};

/// Array of conversion between the DMA transfer type and the DMA LLI that is used
const uint8_t dma2lli[DMA_MAX] =
{
    [DMA_DL] = IPC_DMA_LLI_GP_DL,
    [DMA_UL] = IPC_DMA_LLI_GP_UL,
};

/// Configure the bridge DMA control field
#define DMA_LLICTRL(index) (IPC_DMA_LLI_COUNTER_EN|(dma2lli[index]<<IPC_DMA_LLI_COUNTER_POS) \
                           | IPC_DMA_LLI_IRQ_EN|(dma2lli[index] << IPC_DMA_LLI_IRQ_POS))


/*
 * GLOBAL VARIABLES DEFINITIONS
 ****************************************************************************************
 */
struct hal_dma_env_tag hal_dma_env;

#if (HAL_DMA_POOL)
/// Pool of General Purpose DMA descriptors
struct hal_dma_desc_tag hal_dma_desc_pool[HAL_DMA_DESC_POOL_SIZE];
#endif //(HAL_DMA_POOL)

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
#if (HAL_DMA_POOL)
/**
 ****************************************************************************************
 * @brief Check whether the GP DMA descriptor passed has parameter is coming from the pool
 * of descriptors or not.
 *
 * @param[in] desc     Descriptor that is tested
 *
 * @return true if the descriptor is coming from the pool, false otherwise
 ****************************************************************************************
 */
__INLINE bool hal_dma_is_in_pool(struct hal_dma_desc_tag *desc)
{
    return((desc >= &hal_dma_desc_pool[0]) &&
           (desc <= &hal_dma_desc_pool[HAL_DMA_DESC_POOL_SIZE - 1]));
}
#endif //(HAL_DMA_POOL)

void hal_dma_init(void)
{
    int i;

    // Initialize DMA lists
    for (i = 0; i < DMA_MAX; i++)
    {
        co_list_init(&hal_dma_env.prog[i]);
        hal_dma_env.lli_cnt[i] = dma_lli_counter_get(dma2lli[i]);
    }

    #if (HAL_DMA_POOL)
    // Fully reset content of the GP DMA descriptors
    memset(&hal_dma_desc_pool[0], 0, HAL_DMA_DESC_POOL_SIZE * sizeof(struct hal_dma_desc_tag));

    // Initialize pool of GP DMA descriptors
    co_list_init(&hal_dma_env.free_gp_dma_descs);

    for (i = 0; i < HAL_DMA_DESC_POOL_SIZE; i++)
    {
        // Insert the descriptor in the list of free descriptors
        co_list_push_back(&hal_dma_env.free_gp_dma_descs, &hal_dma_desc_pool[i].hdr);
    }
    #endif //(HAL_DMA_POOL)
}


void hal_dma_push(struct hal_dma_desc_tag *desc, int type)
{
    struct dma_desc *dma_desc = desc->dma_desc;
    struct co_list *list = &hal_dma_env.prog[type];

    // Check if a callback function is programmed
    if ((desc->cb != NULL)
        #if (HAL_DMA_POOL)
        || hal_dma_is_in_pool(desc)
        #endif //(HAL_DMA_POOL)
       )
    {
        // Enable interrupt and LLI counter increment
        dma_desc->ctrl = DMA_LLICTRL(type);

        // Push the descriptor to the list - Done in critical section as this function
        // is sometimes called from interrupt
        GLOBAL_INT_DISABLE();
        co_list_push_back(list, &desc->hdr);
        GLOBAL_INT_RESTORE();
    }
    else
    {
        // Disable interrupt and LLI counter increment
        dma_desc->ctrl = 0;
    }

    // Push the DMA descriptor to the DMA engine
    dma_push(dma_desc, dma_desc, dma2chan[type]);
}

void hal_dma_evt(int dma_queue)
{
    int lli = dma2lli[dma_queue];

    // Trigger the kernel event
    if (dma_queue == DMA_DL)
    {
        ke_evt_clear(KE_EVT_GP_DMA_DL_BIT);
    }
    else
    {
        ke_evt_clear(KE_EVT_GP_DMA_UL_BIT);
    }

    // Go through the list of DMA programmed and flush the ones that are completed
    while (hal_dma_env.lli_cnt[dma_queue] != dma_lli_counter_get(lli))
    {
        struct hal_dma_desc_tag *desc;

        // Done in critical section as the list might be accessed from interrupt
        GLOBAL_INT_DISABLE();
        desc = (struct hal_dma_desc_tag *) co_list_pop_front(&hal_dma_env.prog[dma_queue]);
        GLOBAL_INT_RESTORE();

        // Sanity check - We shall have a descriptor available
        ASSERT_ERR(desc != NULL);

        // Acknowledge the DMA interrupt
        dma_int_ack_clear(CO_BIT(lli + DMA_LLI_IRQ_LSB));

        // Call the callback function if any
        if (desc->cb != NULL)
        {
            desc->cb(desc->env, dma_queue);
        }

        #if (HAL_DMA_POOL)
        // Check if descriptor has been taken in the pool of descriptors
        if (hal_dma_is_in_pool(desc))
        {
            co_list_push_back(&hal_dma_env.free_gp_dma_descs, &desc->hdr);
        }
        #endif //(HAL_DMA_POOL)

        // Increase the LLI counter
        hal_dma_env.lli_cnt[dma_queue]++;
    }

    // Reenable DMA LLI IRQ
    dma_lli_enable(lli);
}

void hal_dma_dl_irq(void)
{
    // Disable DMA IRQ for this LLI
    dma_lli_disable(IPC_DMA_LLI_GP_DL);

    // Acknowledge the DMA interrupt
    dma_int_ack_clear(CO_BIT(IPC_DMA_LLI_GP_DL + DMA_LLI_IRQ_LSB));

    // Trigger the associated kernel event
    ke_evt_set(KE_EVT_GP_DMA_DL_BIT);
}

void hal_dma_ul_irq(void)
{
    // Disable DMA IRQ for this LLI
    dma_lli_disable(IPC_DMA_LLI_GP_UL);

    // Acknowledge the DMA interrupt
    dma_int_ack_clear(CO_BIT(IPC_DMA_LLI_GP_UL + DMA_LLI_IRQ_LSB));

    // Trigger the associated kernel event
    ke_evt_set(KE_EVT_GP_DMA_UL_BIT);
}

#if (HAL_DMA_POOL)
struct hal_dma_desc_tag *hal_dma_get_desc(void)
{
    return ((struct hal_dma_desc_tag *)co_list_pop_front(&hal_dma_env.free_gp_dma_descs));
}

void hal_dma_release_desc(struct hal_dma_desc_tag *gp_dma_desc)
{
    co_list_push_back(&hal_dma_env.free_gp_dma_descs, &gp_dma_desc->hdr);
}
#endif //(HAL_DMA_POOL)

#endif

/// @}
