/**
 ****************************************************************************************
 *
 * @file hal_machw.c
 *
 * @brief Implementation of the Initialization of Interrupt control registers and handling
 * of the Interrupts
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 *****************************************************************************************
 * @addtogroup MACHW
 * @{
 *****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "dbg_assert.h"

#include "co_endian.h"
#include "mm.h"
#include "ke_event.h"

#include "hal_machw.h"

#include "rxl_cntrl.h"
#include "rxl_hwdesc.h"

#include "reg_mac_core.h"
#include "reg_mac_pl.h"

#include "ps.h"

#include "dbg.h"

#include "macif.h"
#include "crm.h"

/// Error interrupts
#define HW_ERROR_IRQ (NXMAC_RX_FIFO_OVER_FLOW_BIT | NXMAC_MAC_PHYIF_OVERFLOW_BIT |      \
                      NXMAC_PT_ERROR_BIT | NXMAC_HW_ERR_BIT | NXMAC_PHY_ERR_BIT |       \
                      NXMAC_AC_0_TX_DMA_DEAD_BIT | NXMAC_AC_1_TX_DMA_DEAD_BIT |         \
                      NXMAC_AC_2_TX_DMA_DEAD_BIT | NXMAC_AC_3_TX_DMA_DEAD_BIT |         \
                      NXMAC_BCN_TX_DMA_DEAD_BIT | NXMAC_MAC_PHYIF_UNDER_RUN_BIT |       \
                      NXMAC_RX_HEADER_DMA_DEAD_BIT | NXMAC_RX_PAYLOAD_DMA_DEAD_BIT)

/// HE TX timeout bit
#if NX_MAC_HE
#define HE_TIMER_BIT HAL_HE_TB_TIMER_BIT
#else
#define HE_TIMER_BIT 0
#endif

/// Timeout error interrupts
#define TIMEOUT_IRQ  (HAL_AC0_TIMER_BIT | HAL_AC1_TIMER_BIT | HAL_AC2_TIMER_BIT |       \
                      HAL_AC3_TIMER_BIT | HAL_BCN_TIMER_BIT | HAL_IDLE_TIMER_BIT |      \
                      HE_TIMER_BIT)

/// TX interrupt bits
#if NX_AMSDU_TX
#define TX_IRQ (NXMAC_AC_0_TX_TRIGGER_BIT | NXMAC_AC_1_TX_TRIGGER_BIT |                 \
                NXMAC_AC_2_TX_TRIGGER_BIT | NXMAC_AC_3_TX_TRIGGER_BIT |                 \
                NXMAC_BCN_TX_TRIGGER_BIT | NXMAC_AC_0_TX_BUF_TRIGGER_BIT |              \
                NXMAC_AC_1_TX_BUF_TRIGGER_BIT | NXMAC_AC_2_TX_BUF_TRIGGER_BIT |         \
                NXMAC_AC_3_TX_BUF_TRIGGER_BIT | NXMAC_BCN_TX_BUF_TRIGGER_BIT | MU_MIMO_MASTER_TX_IRQ)
#else
#define TX_IRQ (NXMAC_AC_0_TX_TRIGGER_BIT | NXMAC_AC_1_TX_TRIGGER_BIT |                 \
                NXMAC_AC_2_TX_TRIGGER_BIT | NXMAC_AC_3_TX_TRIGGER_BIT |                 \
                NXMAC_BCN_TX_TRIGGER_BIT | MU_MIMO_MASTER_TX_IRQ)
#endif

#if NX_RX_RING
/// Fake definition for compilation
#define NXMAC_ENCR_RX_FIFO_RESET_BIT 0
/// RX interrupt bits
#define RX_IRQ (NXMAC_RX_BUFFER_1_TRIGGER_BIT)
#else
/// RX interrupt bits
#define RX_IRQ (NXMAC_COUNTER_RX_TRIGGER_BIT | NXMAC_TIMER_RX_TRIGGER_BIT)
#endif

/// Bandwidth drop interrupt bits
#if (NX_BW_LEN_ADAPT)
#define BW_DROP_IRQ (NXMAC_AC_0BW_DROP_TRIGGER_BIT | NXMAC_AC_1BW_DROP_TRIGGER_BIT |    \
                     NXMAC_AC_2BW_DROP_TRIGGER_BIT | NXMAC_AC_3BW_DROP_TRIGGER_BIT)
#else
#define BW_DROP_IRQ  0
#endif

/// FSMs and FIFOs that have to be reset by the error recovery mechanism
#define MAC_HW_RESET (NXMAC_HW_FSM_RESET_BIT | NXMAC_RX_FIFO_RESET_BIT |                \
                      NXMAC_TX_FIFO_RESET_BIT | NXMAC_MAC_PHYIFFIFO_RESET_BIT |         \
                      NXMAC_ENCR_RX_FIFO_RESET_BIT)

/// Timeout duration when a IDLE state is requested to the HW
#define IDLE_REQ_TIMEOUT  50000

/**
 ****************************************************************************************
 * @brief Macro used to apply a ratio on a MAC timing parameter when switching from one
 * MAC core clock value to another one. The macro first reads the parameter in the MAC
 * HW register, apply the ratio and then write it back to the register.
 * @param[in] param  Name of the parameter
 * @param[in] old    Old clock value
 * @param[in] new    New clock value
 ****************************************************************************************
 */
#define TIMING_UPDATE(param, old, new)                                         \
                     (nxmac_ ## param ## _in_mac_clk_setf((uint32_t)           \
                                        nxmac_ ## param ## _in_mac_clk_getf()  \
                                          * (uint32_t)(new) / (uint32_t)(old)))

#if NX_MULTI_ROLE
/// Table of conversion between a RX vector rate to a MAC HW rate
const uint8_t rxv2macrate[] = {
    0,                          /* 0 */
    1,                          /* 1 */
    2,                          /* 2 */
    3,                          /* 3 */
    -1,                         /* 4 */
    -1,                         /* 5 */
    -1,                         /* 6 */
    -1,                         /* 7 */
    10,                         /* 8 */
    8,                          /* 9 */
    6,                          /* 10 */
    4,                          /* 11 */
    11,                         /* 12 */
    9,                          /* 13 */
    7,                          /* 14 */
    5                           /* 15 */
};
#endif

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief IDLE Interrupt Handler

 * This function handles the idle interrupt raised when the hardware changes to idle
 * state. The PS module sets the HW to idle and waits for this interrupt to change the HW
 * state to DOZE. The MAC management module uses this interrupt to program the HW to next
 * relevant state.
 ****************************************************************************************
 */
static void hal_machw_idle_irq_handler(void)
{
    // Sanity check: Ensure that the HW is effectively in IDLE state
    ASSERT_REC(nxmac_current_state_getf() == HW_IDLE);

    // For profiling
    PROF_MM_HW_IDLE_CLR();

    // Disable the go to IDLE timer interrupt
    nxmac_timers_int_un_mask_set(nxmac_timers_int_un_mask_get() & ~HAL_IDLE_TIMER_BIT);

    #if NX_POWERSAVE
    // We can sleep again
    ps_env.prevent_sleep &= ~PS_IDLE_REQ_PENDING;
    #endif

    // Trigger the IDLE state event
    ke_evt_set(KE_EVT_HW_IDLE_BIT);
}

/**
 ****************************************************************************************
 * @brief Updates the MAC HW timing registers according to the frequency actually used.
 *
 * @param[in] newfreq Frequency of the MAC HW core, in MHz
 ****************************************************************************************
 */
static void hal_machw_setfreq(uint8_t newfreq)
{
    uint8_t oldfreq;

    oldfreq = nxmac_mac_core_clk_freq_getf();

    // TIMINGS_1
    nxmac_mac_core_clk_freq_setf(newfreq);
    TIMING_UPDATE(tx_rf_delay, oldfreq, newfreq);
    TIMING_UPDATE(tx_chain_delay, oldfreq, newfreq);
    #if NX_MAC_HE
    // Temporary - Will have to be removed once MAC HW has the correct default value
    nxmac_tx_chain_delay_in_mac_clk_setf(0x112);
    #endif

    // TIMINGS_2
    TIMING_UPDATE(slot_time, oldfreq, newfreq);

    // TIMINGS_3
    TIMING_UPDATE(rx_rf_delay, oldfreq, newfreq);
    #if !NX_MAC_HE
    TIMING_UPDATE(tx_delay_rf_on, oldfreq, newfreq);
    #endif
    TIMING_UPDATE(mac_proc_delay, oldfreq, newfreq);

    // TIMINGS_4
    if (newfreq<30)
        // WTClk running at 3x macCoreClk freq
        nxmac_wt_2_crypt_clk_ratio_setf(3);
    else if (newfreq<60)
        // WTClk running at 2x macCoreClk freq
        nxmac_wt_2_crypt_clk_ratio_setf(2);
    else
        // WTClk running at macCoreClk freq
        nxmac_wt_2_crypt_clk_ratio_setf(1);

    // TIMINGS_5
    TIMING_UPDATE(sifs_b, oldfreq, newfreq);

    // TIMINGS_6
    TIMING_UPDATE(sifs_a, oldfreq, newfreq);

    // TIMINGS_7
    // Nothing to do

    // TIMINGS_8
    // Nothing to do

    // TIMINGS_9
    TIMING_UPDATE(rifs_to, oldfreq, newfreq);
    #if !NX_MAC_HE
    TIMING_UPDATE(rifs, oldfreq, newfreq);
    #endif
    TIMING_UPDATE(tx_dma_proc_dly, oldfreq, newfreq);
}

void hal_machw_idle_req(void)
{
    uint32_t curr_time;

    // Sanity check: HW state is not supposed to be IDLE to handle this request
    ASSERT_REC(nxmac_current_state_getf() != HW_IDLE);

    // For profiling
    PROF_MM_HW_IDLE_SET();

    // Enable timer irq
    GLOBAL_INT_DISABLE();
    curr_time = hal_machw_time();
    nxmac_abs_timer_set(HAL_IDLE_TIMER, curr_time + IDLE_REQ_TIMEOUT);

    nxmac_timers_int_event_clear(HAL_IDLE_TIMER_BIT);
    nxmac_timers_int_un_mask_set(nxmac_timers_int_un_mask_get() | HAL_IDLE_TIMER_BIT);

    // Request to MAC HW to switch to IDLE state
    nxmac_next_state_setf(HW_IDLE);

    #if NX_POWERSAVE
    // Prevent from sleeping during the switch to IDLE
    ps_env.prevent_sleep |= PS_IDLE_REQ_PENDING;
    #endif

    GLOBAL_INT_RESTORE();
}


void hal_machw_stop(void)
{
    // Reset all the MAC HW state machines and registers
    nxmac_soft_reset_setf(1);
    while (nxmac_soft_reset_getf());
}

void hal_machw_init(void)
{
    #if NX_MAC_HE
    int8_t min, max;
    #endif

    // Soft Reset
    nxmac_soft_reset_setf(1);
    while (nxmac_soft_reset_getf());

    // Update the timing registers of the MAC according to the MAC core clock
    hal_machw_setfreq(crm_get_mac_freq());

    // enable MAC HW general interrupt events
    nxmac_gen_int_enable_set(NXMAC_MASTER_GEN_INT_EN_BIT | NXMAC_IDLE_INTERRUPT_BIT |
                             NXMAC_ABS_GEN_TIMERS_BIT | HW_ERROR_IRQ);

    #if !NX_MAC_HE
    // On old MAC HW version this error could happen from time to time, but does not
    // require any special handling from SW, so we disable it
    nxmac_enable_mac_phyif_overflow_setf(0);
    #endif

    #if NX_MDM_VER >= 20
    // delegate MPIF interface flow control to PHY
    nxmac_rate_controller_mpif_setf(0);
    #else
    // On old versions of the Modem enable the rxEndForTiming Error Recovery
    nxmac_rx_end_for_timing_err_rec_setf(1);
    #endif

    #if NX_KEY_RAM_CONFIG
    ASSERT_ERR(MM_STA_TO_KEY(NX_REMOTE_STA_MAX - 1) <= nxmac_sta_key_max_index_getf());
    nxmac_encr_ram_config_pack(NX_VIRT_DEV_MAX, MM_STA_TO_KEY(NX_REMOTE_STA_MAX - 1),
                               MM_STA_TO_KEY(0));
    #endif

    // enable MAC HW TX and RX interrupt events
    nxmac_tx_rx_int_enable_set(TX_IRQ | BW_DROP_IRQ | RX_IRQ | NXMAC_MASTER_TX_RX_INT_EN_BIT);

    #if RW_MUMIMO_TX_EN
    nxmac_sec_users_tx_int_event_un_mask_set(TX_SEC_IRQ_BITS | NXMAC_MASTER_SEC_USERS_TX_INT_EN_BIT);
    #endif

    // At initialization we are not supposed to reply to any frame (in case monitor mode
    // would be used), so disable the auto-reply capabilities of the HW
    nxmac_mac_cntrl_1_set(nxmac_mac_cntrl_1_get() | NXMAC_DISABLE_ACK_RESP_BIT |
                                                    NXMAC_DISABLE_CTS_RESP_BIT |
                                                    NXMAC_DISABLE_BA_RESP_BIT |
                                                    NXMAC_ACTIVE_CLK_GATING_BIT |
                                                    NXMAC_ENABLE_LP_CLK_SWITCH_BIT |
                                                    NXMAC_RX_RIFS_EN_BIT);
    #if NX_MAC_HE
    // RIFS not supported in HE
    nxmac_rx_rifs_en_setf(0);
    #endif

    // enable the RX flow control
    nxmac_rx_flow_cntrl_en_setf(1);

    // enable reception of all frames (i.e. monitor mode)
    nxmac_rx_cntrl_set(MM_RX_FILTER_MONITOR);

    // put default values of the beacon timings
    nxmac_bcn_cntrl_1_pack(255, 1, HAL_MACHW_BCN_TX_DELAY_US / 128, 100);

    // limit the maximum RX size to the maximum A-MSDU size we support
    nxmac_max_rx_length_set(RWNX_MAX_AMSDU_RX);

    // configure the EDCA control register
    nxmac_edca_cntrl_pack(0, 0, 0, 0);

    // Default power level
    #if NX_MAC_HE
    phy_get_rf_gain_capab(&max, &min);
    nxmac_max_power_level_pack(min, max, max);
    #else
    nxmac_max_power_level_pack(0x20, 0x20);
    #endif

    // reset MIB Table
    nxmac_mib_table_reset_setf(1);

    // reset Key storage RAM
    nxmac_key_sto_ram_reset_setf(1);

    // Initialize diagnostic ports
    nxmac_debug_port_sel_pack(0x1C, 0x25);

    // Enable the dynamic bandwidth feature
    nxmac_dyn_bw_en_setf(1);

    // Set the number of TX chains the HW shall use for immediate responses
    nxmac_max_phy_ntx_setf(phy_get_ntx() + 1);

    #if NX_MULTI_ROLE
    // If Multi-role is enabled, the SW manages by itself the STA TSF. It it
    // therefore necessary to disable the corresponding HW feature.
    nxmac_tsf_mgt_disable_setf(1);
    #endif

    #if RW_BFMEE_EN
    // Check if beamformee is supported by the MAC HW and PHY
    if (hal_machw_bfmee_support())
    {
        // Initialize beamformee registers
        nxmac_bfmee_nc_setf(phy_get_nss());
        nxmac_bfmee_nr_setf(3);
        nxmac_bfmee_codebook_setf(1);
        nxmac_bfmee_enable_setf(1);
        #if NX_MAC_HE
        if (hal_machw_he_support())
        {
            nxmac_disable_svd_rx_pause_setf(1);
            nxmac_bfr_format_mod_setf(FORMATMOD_HT_MF);
        }
        #endif
        #if (RW_MUMIMO_RX_EN)
        if (hal_machw_mu_mimo_rx_support())
            nxmac_bfmee_mu_support_setf(1);
        #endif
    }
    #endif
}

void hal_machw_disable_int(void)
{
    nxmac_enable_master_gen_int_en_setf(0);
    nxmac_enable_master_tx_rx_int_en_setf(0);
}

#if NX_DEBUG_DUMP
void hal_machw_get_diag_state(void)
{
    int i;
    uint8_t diag;
    struct dbg_debug_info_tag *dbg_info = &(debug_info.dbg_info);

    // Get RX status pointers
    #if NX_RX_RING
    dbg_info->rhd_hw_ptr = 0;
    dbg_info->rbd_hw_ptr = 0;
    #else
    dbg_info->rhd_hw_ptr = nxmac_debug_rx_hdr_c_ptr_get();
    dbg_info->rbd_hw_ptr = nxmac_debug_rx_pay_c_ptr_get();
    #endif

    // Get diagnostic port state for the MAC
    diag = nxmac_debug_port_sel_1_getf();
    for (i = 0; i < DBG_DIAGS_MAC_MAX; i++)
    {
        // Go through the different banks and copies the diag values
        nxmac_debug_port_sel_1_setf(i);
        dbg_info->diags_mac[i] = nxmac_debug_port_value_get() & 0xFFFF;
    }
    nxmac_debug_port_sel_1_setf(diag);

    // Get diagnostic port state for the PHY
    phy_get_diag_state(dbg_info);
}
#endif

void hal_machw_reset(void)
{
    // Disable clock gating during SW reset
    nxmac_active_clk_gating_setf(0);

    // Request to MAC HW to switch to IDLE state, in order to ensure that it will be
    // IDLE after reset of state machines and FIFOs
    nxmac_next_state_setf(HW_IDLE);

    // Reset state machines and FIFOs
    nxmac_mac_err_rec_cntrl_set(MAC_HW_RESET);
    //while (nxmac_mac_err_rec_cntrl_get() & MAC_HW_RESET);
    while (nxmac_current_state_getf() != HW_IDLE);

    #if NX_POWERSAVE
    // We are in IDLE so no need to wait anymore for the IDLE interrupt
    ps_env.prevent_sleep &= ~PS_IDLE_REQ_PENDING;
    #endif

    // Reenable the RX flow control
    nxmac_rx_flow_cntrl_en_setf(1);

    // Disable the timeout error interrupts
    nxmac_timers_int_un_mask_set(nxmac_timers_int_un_mask_get() & ~TIMEOUT_IRQ);

    // Acknowledge possibly pending interrupts
    nxmac_tx_rx_int_ack_clear(0xFFFFFFFF);
    nxmac_gen_int_ack_clear(NXMAC_IDLE_INTERRUPT_BIT | HW_ERROR_IRQ);

    nxmac_enable_master_gen_int_en_setf(1);
    nxmac_enable_master_tx_rx_int_en_setf(1);

    // Re-enable clock gating during SW reset
    nxmac_active_clk_gating_setf(1);
}

uint8_t hal_machw_search_addr(struct mac_addr *addr)
{
    uint8_t sta_idx = INVALID_STA_IDX;
    uint32_t enc_cntrl;

    // Copy the MAC addr
    nxmac_encr_mac_addr_low_set(addr->array[0] | (((uint32_t)addr->array[1]) << 16));
    nxmac_encr_mac_addr_high_set(addr->array[2]);

    // Write the control field
    nxmac_encr_cntrl_set(NXMAC_NEW_SEARCH_BIT);

    // Poll for the completion of the search
    do
    {
        enc_cntrl = nxmac_encr_cntrl_get();
    } while(enc_cntrl & NXMAC_NEW_SEARCH_BIT);

    // Check if the search was successful or not
    if (!(enc_cntrl & NXMAC_SEARCH_ERROR_BIT))
        // Compute the SW STA index from the HW index
        sta_idx = ((enc_cntrl & NXMAC_KEY_INDEX_RAM_MASK) >> NXMAC_KEY_INDEX_RAM_LSB)
                    - MM_SEC_DEFAULT_KEY_COUNT;

    return (sta_idx);
}

void hal_machw_monitor_mode(void)
{
    // Disable the TBTT interrupts
    nxmac_enable_imp_pri_tbtt_setf(0);
    nxmac_enable_imp_sec_tbtt_setf(0);

    // Disable the auto-reply capabilities of the HW
    nxmac_mac_cntrl_1_set(nxmac_mac_cntrl_1_get() | NXMAC_DISABLE_ACK_RESP_BIT |
                                                    NXMAC_DISABLE_CTS_RESP_BIT |
                                                    NXMAC_DISABLE_BA_RESP_BIT);

    // Enable reception of all frames (i.e. monitor mode)
    mm_rx_filter_umac_set(MM_RX_FILTER_MONITOR);

    // Configure RX path for monitor mode
    rxl_hwdesc_monitor(true);

    // set default mode of operation
    nxmac_abgn_mode_setf(MODE_802_11N_5);

    // reset Key storage RAM
    nxmac_key_sto_ram_reset_setf(1);
}

int hal_machw_tsf_move(int32_t offset)
{
    int err = 0;
    uint32_t abs_offset = co_abs(offset);

    GLOBAL_INT_DISABLE();
    uint32_t macctrl1 = nxmac_mac_cntrl_1_get() | NXMAC_TSF_UPDATED_BY_SW_BIT;
    uint32_t tsf_lo = nxmac_tsf_lo_get();

    // Check if the offset could cause a wrap of the tsf_lo
    if (((offset > 0) && (tsf_lo < abs_offset)) ||
        ((offset < 0) && ((0xFFFFFFFF - tsf_lo) < (abs_offset + 32))))
    {
        // Not a good time to move the TSF
        err = -1;
    }
    else
    {
        // Read again the TSF LO to get the latest value
        tsf_lo = nxmac_tsf_lo_get();
        tsf_lo -= offset;
        // Do the operation thrice to ensure it is taken into account by the MAC HW
        nxmac_tsf_lo_set(tsf_lo);
        nxmac_mac_cntrl_1_set(macctrl1);
        nxmac_tsf_lo_set(tsf_lo);
        nxmac_mac_cntrl_1_set(macctrl1);
        nxmac_tsf_lo_set(tsf_lo);
        nxmac_mac_cntrl_1_set(macctrl1);
    }
    GLOBAL_INT_RESTORE();

    return err;
}

bool hal_machw_sleep_check(void)
{
    uint32_t timer_msk = nxmac_timers_int_un_mask_get();

    #if !NX_MULTI_ROLE
    uint32_t tbtt = hal_machw_time() + ((uint32_t)nxmac_next_tbtt_get() << 5);

    if (hal_machw_time_past(tbtt - 2000))
        return false;
    #endif

    // Go through all the enabled timer to check if one will expire too soon to sleep
    for (int i = 0; i < HAL_TIMER_MAX; i++)
    {
        uint32_t timer_bit = CO_BIT(i);

        if ((timer_msk & timer_bit) && (hal_machw_time_past(nxmac_abs_timer_get(i) - 2000)))
        {
            ASSERT_ERR(!hal_machw_time_past(nxmac_abs_timer_get(i) + 5000));
            return false;
        }
    }

    return true;
}


/**
 ****************************************************************************************
 * @brief Absolute timer interrupt Handler

 * This function handles the HW timer interrupts. It checks the source of the interrupt
 * and execute the required functions.
 ****************************************************************************************
 */
static void hal_machw_abs_timer_handler(void)
{
    uint32_t timer_pending = nxmac_timers_int_event_get();

    // Acknowledge the timer interrupt
    nxmac_timers_int_event_clear(timer_pending);

    if (timer_pending & HAL_KE_TIMER_BIT)
        // Set the kernel timer event
        ke_evt_set(KE_EVT_KE_TIMER_BIT);

    #if !NX_FULLY_HOSTED
    if (timer_pending & HAL_RX_TIMER_BIT)
        rxl_timeout_int_handler();
    #endif

    #if NX_MM_TIMER
    if (timer_pending & HAL_MM_TIMER_BIT)
        // Trigger the primary TBTT event.
        ke_evt_set(KE_EVT_MM_TIMER_BIT);
    #endif

    // Check for timeout errors
    ASSERT_REC(!(timer_pending & HAL_AC0_TIMER_BIT));
    ASSERT_REC(!(timer_pending & HAL_AC1_TIMER_BIT));
    ASSERT_REC(!(timer_pending & HAL_AC2_TIMER_BIT));
    ASSERT_REC(!(timer_pending & HAL_AC3_TIMER_BIT));
    ASSERT_REC(!(timer_pending & HAL_BCN_TIMER_BIT));
    ASSERT_REC(!(timer_pending & HAL_IDLE_TIMER_BIT));
    #if NX_MAC_HE
    ASSERT_REC(!(timer_pending & HAL_HE_TB_TIMER_BIT));
    #endif
}

void hal_machw_gen_handler(void)
{
    uint32_t genirq_pending = nxmac_gen_int_status_get() & nxmac_gen_int_enable_get();

    // clear all the interrupts
    nxmac_gen_int_ack_clear(genirq_pending);

    #if NX_BEACONING || ((NX_POWERSAVE || NX_CONNECTION_MONITOR || NX_REORD || NX_UMAC_PRESENT) & !NX_MULTI_ROLE)
    // Check for primary TBTT interrupts
    if (genirq_pending & (NXMAC_IMP_PRI_TBTT_BIT | NXMAC_IMP_PRI_DTIM_BIT))
    {
        // For profiling
        PROF_BCN_PRIM_TBTT_IRQ_SET();

        #if !NX_BCN_AUTONOMOUS_TX
        // Trigger the interrupt to the host
        macif_prim_tbtt_ind();
        #endif

        // Trigger the primary TBTT event.
        ke_evt_set(KE_EVT_PRIMARY_TBTT_BIT);

        // For profiling
        PROF_BCN_PRIM_TBTT_IRQ_CLR();
    }
    #endif

    #if NX_BEACONING
    // Check for secondary TBTT interrupts
    if (genirq_pending & (NXMAC_IMP_SEC_TBTT_BIT | NXMAC_IMP_SEC_DTIM_BIT))
    {
        // For profiling
        PROF_BCN_SEC_TBTT_IRQ_SET();

        #if !NX_BCN_AUTONOMOUS_TX
        // Trigger the interrupt to the host
        macif_sec_tbtt_ind();
        #endif

        // Trigger the secondary TBTT event.
        ke_evt_set(KE_EVT_SECONDARY_TBTT_BIT);

        // For profiling
        PROF_BCN_SEC_TBTT_IRQ_CLR();
    }
    #endif

    // Check for IDLE interrupt Handler
    if (genirq_pending & NXMAC_IDLE_INTERRUPT_BIT)
    {
        // Call Idle interrupt handler
        hal_machw_idle_irq_handler();
    }

    // Check for TIMER interrupt Handler
    if (genirq_pending & NXMAC_ABS_GEN_TIMERS_BIT)
    {
        // Handle the timer interrupt in a specific handler
        hal_machw_abs_timer_handler();
    }

    // Check for HW errors
    ASSERT_REC(!(genirq_pending & NXMAC_PHY_ERR_BIT));
    ASSERT_REC(!(genirq_pending & NXMAC_MAC_PHYIF_UNDER_RUN_BIT));
    ASSERT_REC(!(genirq_pending & NXMAC_MAC_PHYIF_OVERFLOW_BIT));
    ASSERT_REC(!(genirq_pending & NXMAC_RX_FIFO_OVER_FLOW_BIT));
    ASSERT_REC(!(genirq_pending & NXMAC_PT_ERROR_BIT));
    ASSERT_REC(!(genirq_pending & NXMAC_AC_0_TX_DMA_DEAD_BIT));
    ASSERT_REC(!(genirq_pending & NXMAC_AC_1_TX_DMA_DEAD_BIT));
    ASSERT_REC(!(genirq_pending & NXMAC_AC_2_TX_DMA_DEAD_BIT));
    ASSERT_REC(!(genirq_pending & NXMAC_AC_3_TX_DMA_DEAD_BIT));
    ASSERT_REC(!(genirq_pending & NXMAC_BCN_TX_DMA_DEAD_BIT));
    ASSERT_REC(!(genirq_pending & NXMAC_RX_HEADER_DMA_DEAD_BIT));
    ASSERT_REC(!(genirq_pending & NXMAC_RX_PAYLOAD_DMA_DEAD_BIT));
    ASSERT_REC(!(genirq_pending & NXMAC_HW_ERR_BIT));
}

/// @}  // end of group HAL_MACHW
