/**
 ****************************************************************************************
 *
 * @file mm.c
 *
 * @brief MAC Management module implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup MM
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "mac_frame.h"
#include "mm.h"
#include "mm_timer.h"
#include "mm_bcn.h"
#include "co_endian.h"
#include "sta_mgmt.h"
#include "vif_mgmt.h"
#include "phy.h"
#include "dbg.h"
#include "rd.h"
#include "ps.h"
#include "txl_cntrl.h"
#include "txl_frame.h"
#include "rxl_cntrl.h"
#include "rxl_hwdesc.h"
#include "hal_machw.h"
#include "scan.h"
#include "chan.h"
#include "hal_dma.h"
#if (NX_P2P)
#include "p2p.h"
#endif //(NX_P2P)
#if (NX_TD)
#include "td.h"
#endif //(NX_TD)
#if (RW_BFMER_EN)
#include "bfr.h"
#endif //(RW_BFMER_EN)
#if (RW_UMESH_EN)
#include "mesh.h"
#endif //(RW_UMESH_EN)
#if (NX_UMAC_PRESENT && NX_BEACONING)
#include "apm.h"
#endif // NX_BEACONING

#include "reg_mac_core.h"
#include "reg_mac_pl.h"

// For IE access functions
#include "mac_ie.h"


/// Event mask of the TBTT kernel events
#if NX_BEACONING
#define MM_TBTT_EVT_MASK (KE_EVT_PRIMARY_TBTT_BIT | KE_EVT_SECONDARY_TBTT_BIT)
#else
#define MM_TBTT_EVT_MASK KE_EVT_PRIMARY_TBTT_BIT
#endif

/// Beacon loss threshold above which we consider the connection as lost
#define MM_BEACON_LOSS_THD 10

/// Periodicity of keep-alive NULL frame transmission
#define MM_KEEP_ALIVE_PERIOD (30 * 1000000)   ///< 30s

/// Mask of the TBTT interrupts
#define MM_TBTT_IRQ_MASK (NXMAC_IMP_PRI_DTIM_BIT | NXMAC_IMP_PRI_TBTT_BIT)

/// Mask used for the MAC address compatibility checking
#define MM_MAC_ADDR_MSK  ((NX_VIRT_DEV_MAX - 1) << 8)

/// Macro returning the maximum duration of A-MPDUs according to the TXOP limit
#define TXOP(limit)  (((limit)==0) || ((limit) > MM_DEFAULT_MAX_AMPDU_DURATION))?       \
                                  MM_DEFAULT_MAX_AMPDU_DURATION:(limit);

#if NX_MULTI_ROLE
/// Wake up delay before TBTT is occurring
#define TBTT_DELAY   400  ///< 400us
#endif

/**
 ****************************************************************************************
 * Write parameters to the key RAM.
 * @param[in] key_idx Index of the key RAM entry to be written
 * @param[in] ctype Encryption type
 * @param[in] vif_idx Index of the VIF the key is belonging to
 * @param[in] spp SPP parameter
 * @param[in] usedefkey Flag indicating whether the default key shall be used for this
 *                      entry
 * @param[in] clen Flag indicating the key length
 ****************************************************************************************
 */
#if !NX_MAC_HE
// newRead, newWrite, newSearch, SearchError, keyIdx, cType, vlanIdx, spp, usedefkey, cLen
#define KEY_RAM_WR(key_idx, ctype, vif_idx, spp, usedefkey, clen)                        \
{                                                                                        \
    nxmac_encr_cntrl_pack(0, 1, 0, 0, key_idx, ctype, vif_idx, spp, usedefkey, clen);    \
    while(nxmac_new_write_getf());                                                       \
}
#else
// newRead, newWrite, newSearch, SearchError, keyIdx, cLen, cType, vlanIdx, spp, usedefkey
#define KEY_RAM_WR(key_idx, ctype, vif_idx, spp, usedefkey, clen)                        \
{                                                                                        \
    nxmac_encr_cntrl_pack(0, 1, 0, 0, key_idx, clen, ctype, vif_idx, spp, usedefkey);    \
    while(nxmac_new_write_getf());                                                       \
}
#endif


/// Margin taken when checking if the computed TBTT is not in the past
#if NX_CHNL_CTXT
#define MM_TBTT_COMPUTE_MARGIN (CHAN_SWITCH_DELAY + 300)
#else
#define MM_TBTT_COMPUTE_MARGIN 300
#endif

/// Task identifier used for transmission of indication from MM to UMAC
#if NX_UMAC_PRESENT
#define TASK_IND  TASK_SM
#else
#define TASK_IND  TASK_API
#endif

/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
///  Global data for maintaining BSS and STA information

/**  LMAC MM Context variable, used to store MM Context data
 */
struct mm_env_tag mm_env;


/**
 ****************************************************************************************
 * @brief Compute the cType and cLen to be set to the key RAM depending on the Cipher
 * and the key length.
 *
 * @param[in]  cipher_suite  Cipher suite to be used (@ref mac_cipher_suite)
 * @param[in]  key           Pointer to the key structure
 * @param[out] ctype         cType parameter to be set to the key RAM
 * @param[out] clen          cLen parameter to be set to the key RAM
 ****************************************************************************************
 */
static void mm_key_ram_param_get(uint8_t cipher_suite, struct mac_sec_key const *key,
                                 uint8_t *ctype, uint8_t *clen)
{
    *clen = 1;
    *ctype = MM_SEC_CTYPE_NULL;

    // Check which encryption type has to be used
    switch(cipher_suite)
    {
        case MAC_CIPHER_WEP40:
            *clen = 0;
        case MAC_CIPHER_WEP104:
            *ctype = MM_SEC_CTYPE_WEP;
            break;
        case MAC_CIPHER_TKIP:
            *ctype = MM_SEC_CTYPE_TKIP;
            break;
        case MAC_CIPHER_CCMP:
            #if NX_MAC_HE
            switch (key->length)
            {
                case 16:
                    *clen = 0;
                    break;
                case 32:
                    *clen = 2;
                    break;
                default:
                    break;
            }
            #endif
            *ctype = MM_SEC_CTYPE_CCMP;
            break;
        #if RW_WAPI_EN
        case MAC_CIPHER_WPI_SMS4:
            *ctype = MM_SEC_CTYPE_WPI_SMS4;
            break;
        #endif
        default:
            ASSERT_ERR(0);
            break;
    }


}

#if NX_BEACONING
 /**
  ****************************************************************************************
  * @brief Callback function called when the TBTT move can be performed.
  * The function will send an internal message to the MM in order to handle the TBTT move
  * in IDLE state.
  *
  * @param[in] offset     Offset to be applied to the TBTT
  ****************************************************************************************
  */
static void mm_ap_tbtt_move_cb(void *offset)
{
    // Allocate TBTT move message
    struct mm_tbtt_move_req *req = KE_MSG_ALLOC(MM_TBTT_MOVE_REQ, TASK_MM, TASK_NONE, mm_tbtt_move_req);

    // Set the offset
    req->offset = (int32_t)offset;

    // Send the message
    ke_msg_send(req);
}
#endif

void mm_env_max_ampdu_duration_set(void)
{
    // Initialize the TXOP values
    mm_env.ampdu_max_dur[AC_BK] = TXOP(nxmac_tx_op_limit_0_getf());
    mm_env.ampdu_max_dur[AC_BE] = TXOP(nxmac_tx_op_limit_1_getf());
    mm_env.ampdu_max_dur[AC_VI] = TXOP(nxmac_tx_op_limit_2_getf());
    mm_env.ampdu_max_dur[AC_VO] = TXOP(nxmac_tx_op_limit_3_getf());

    #if NX_BEACONING
    // For BCN queue, put same parameter as VO
    mm_env.ampdu_max_dur[AC_BCN] = mm_env.ampdu_max_dur[AC_VO];
    #endif
}

void mm_env_init(void)
{
    // Reset the complete environment
    memset(&mm_env, 0, sizeof(mm_env));

    mm_env.prev_mm_state = MM_IDLE;
    mm_env.prev_hw_state = HW_IDLE;
    #if NX_UMAC_PRESENT
    mm_env.host_idle = 1;
    #endif
    mm_env.rx_filter_lmac_enable = 0;
    mm_rx_filter_umac_set(MM_RX_FILTER_MONITOR);

    // Initialize the TXOP values
    mm_env_max_ampdu_duration_set();
    #if NX_BEACONING
    mm_env.tbtt_move_tmr.cb = mm_ap_tbtt_move_cb;
    #endif
}

/**
 * Initialize all MM related context and data etc....
 */
void mm_init(void)
{
    // TODO reset PHY

    // Initialize the MAC HW
    hal_machw_init();

    // Init All LMAC MM env data
    mm_env_init();

    // Initialize the VIF table
    vif_mgmt_init();

    // Initialize the peer station tables
    sta_mgmt_init();

    #if (NX_TD)
    // Initialize the TD module
    td_init();
    #endif //(NX_TD)

    #if NX_POWERSAVE
    // Initialize the PS module
    ps_init();
    #endif

    #if (NX_P2P)
    // Initialize P2P module
    p2p_init();
    #endif //(NX_P2P)

    // Initialize Tx
    txl_cntrl_init();

    // Initialize Rx
    rxl_init();

    #if NX_RADAR_DETECT
    // Initialize radar
    rd_init();
    #endif

    #if NX_MM_TIMER
    mm_timer_init();
    #endif

    #if NX_HW_SCAN
    scan_init();
    #endif

    #if NX_CHNL_CTXT
    chan_init();
    #endif

    #if NX_GP_DMA
    hal_dma_init();
    #endif

    #if NX_BCN_AUTONOMOUS_TX
    // Init MM beacon module
    mm_bcn_init();
    #endif

    #if (RW_BFMER_EN)
    // Initialize Beamformer module
    bfr_init();
    #endif //(RW_BFMER_EN)
}

#if NX_MULTI_ROLE
void mm_tbtt_compute(struct bcn_frame *bcn, uint16_t len, struct rx_hd *rhd,
                     struct vif_info_tag *vif,
                     struct sta_info_tag *sta, uint32_t tim)
{
    // Get peer AP information
    uint64_t tsf_start_local = ((uint64_t)rhd->tsflo) | (((uint64_t)rhd->tsfhi) << 32);
    uint64_t tsf_start_peer = bcn->tsf;
    int64_t tsf_offset;
    uint32_t next_tbtt;
    uint32_t duration_to_timestamp;
    uint32_t duration_of_frame;
    uint64_t next_tbtt_tsf;
    uint64_t tbtt_tsf;
    uint32_t bcn_int = (bcn->bcnint << 10);
    uint16_t interval = 1;
    uint32_t drift = 0;

    #if NX_POWERSAVE
    if ((vif->type == VIF_STA) && (vif->u.sta.listen_interval))
    {
        interval = vif->u.sta.listen_interval;
    }
    else
    #endif
    if (tim)
    {
        // Get DTIM information
        interval = co_read8p(tim + MAC_TIM_CNT_OFT);
        if (interval == 0)
            interval = co_read8p(tim + MAC_TIM_PERIOD_OFT);
    }

    #if (NX_POWERSAVE)
    drift = sta->drift * interval;
    #endif //(NX_POWERSAVE)

    // Compute the local time at first bit of timestamp
    duration_of_frame = hal_machw_rx_bcn_duration(rhd, len);
    duration_to_timestamp = hal_machw_rx_bcn_duration(rhd, MAC_BEACON_TIMESTAMP_OFT);
    tsf_start_local -= (duration_of_frame - duration_to_timestamp);

    // Compute the TSF offset between the peer and the local counters
    tsf_offset = tsf_start_peer - tsf_start_local;

    #if (NX_P2P)
    if (vif->type == VIF_STA)
    {
        vif->u.sta.ctw_add_dur = drift + TBTT_DELAY;
        vif->u.sta.last_tsf_offset = tsf_offset;
    }
    #endif //(NX_P2P)

    // Compute what was the current TBTT time
    tbtt_tsf = (tsf_start_peer / bcn_int) * bcn_int;

    // Check if the beacon was not sent too early
    if (tbtt_tsf > (tsf_start_peer - duration_to_timestamp))
        tbtt_tsf = tsf_start_peer - duration_to_timestamp;

    // Compute the next TBTT time at peer time
    next_tbtt_tsf = tbtt_tsf + interval * bcn_int;

    // Compute the next TBTT at local time based on the TSF offset
    next_tbtt = (uint32_t)(next_tbtt_tsf - tsf_offset) - TBTT_DELAY - drift;

    // Check if next TBTT is in the past
    if (hal_machw_time_past(next_tbtt - MM_TBTT_COMPUTE_MARGIN))
    {
        next_tbtt += bcn_int;
    }

    // Convert TSF time to local time
    next_tbtt += (ke_time() - nxmac_tsf_lo_get());

    #if (RW_UMESH_EN)
    if (vif->type == VIF_MESH_POINT)
    {
        // Forward extracted TBTT information to the Mesh module
        sta->bcn_int = bcn_int;
        mesh_update_tbtt_info(vif, sta, next_tbtt);
    }
    else
    #endif //(RW_UMESH_EN)
    {
        // Program the next TBTT
        if (next_tbtt != vif->tbtt_timer.time)
        {
            mm_timer_set(&vif->tbtt_timer, next_tbtt);
            #if NX_CHNL_CTXT
            chan_tbtt_updated(vif);
            #endif
        }
    }
}
#endif

#if NX_CONNECTION_MONITOR
/**
 ****************************************************************************************
 * @brief Callback function indicating the completion of the NULL frame transmission used
 * to probe the AP in case we don't receive any more beacons
 *
 * @param[in] env     Pointer to the VIF entry
 * @param[in] status  Status of the transmission
 ****************************************************************************************
 */
static void mm_ap_probe_cfm(void *env, uint32_t status)
{
    struct vif_info_tag *vif = (struct vif_info_tag *) env;

    // Check if NULL frame was acknowledged
    if (status & FRAME_SUCCESSFUL_TX_BIT)
    {
        // Frame was acknowledged, reset the beacon loss counter
        vif->u.sta.beacon_loss_cnt = 0;
    }
    else
    {
        // Otherwise, we consider that the connection is lost
        mm_send_connection_loss_ind(vif);
    }
}

/**
 ****************************************************************************************
 * @brief Function computing the CRC of the received beacon
 * The function skips the information elements handled in the LMAC (such as the TIM). It
 * also returns the address of the TIM in the beacon buffer.
 *
 * @param[in] bcn     Pointer to the BCN frame
 * @param[in] len     Length of the BCN frame
 * @param[out] tim    Variable to which the TIM address will be written
 *
 * @return The computed CRC
 ****************************************************************************************
 */
static uint32_t mm_compute_beacon_crc(struct bcn_frame *bcn, uint16_t len, uint32_t *tim)
{
    uint32_t crc;
    uint32_t bcn_addr = CPU2HW(bcn);
    uint32_t addr = bcn_addr + MAC_BEACON_VARIABLE_PART_OFT;

    // Only beacon interval and capability information are of interest in the constant part
    crc = co_crc32(bcn_addr + MAC_BEACON_INTERVAL_OFT, 4, 0);

    // Remove the length of the constant part + MAC header
    len -= MAC_BEACON_VARIABLE_PART_OFT;

    // By default we consider that we won't find the TIM IE
    *tim = 0;

    // Now compute the CRC on the variable part IEs, skipping the ones we handle internally
    while (len >= MAC_INFOELT_INFO_OFT)
    {
        uint8_t ie_id = co_read8p(addr++);
        uint8_t ie_len = co_read8p(addr++);

        // Ensure IE is complete
        if ((ie_len + MAC_INFOELT_INFO_OFT) > len)
            break;

        // Check if we compute the CRC on this IE or not
        switch (ie_id)
        {
            case MAC_ELTID_TIM:
                // Skip this element, so we do nothing
                *tim = addr - MAC_INFOELT_INFO_OFT;
                break;
            default:
                // Compute the CRC on this element
                crc = co_crc32(addr, ie_len, crc);
                break;
        }

        // Decrease the length and increase the pointer
        len -= ie_len + MAC_INFOELT_INFO_OFT;
        addr += ie_len;
    }

    return (crc);
}
#endif

#if NX_CONNECTION_MONITOR || NX_MULTI_ROLE
bool mm_check_beacon(struct rx_hd *rhd, struct vif_info_tag *vif,
                     struct sta_info_tag *sta, uint32_t *tim)
{
    struct rx_pbd *pbd = HW2CPU(rhd->first_pbd_ptr);
    struct bcn_frame *bcn = HW2CPU(pbd->datastartptr);
    uint16_t len = rhd->frmlen;

    #if NX_CONNECTION_MONITOR
    int8_t rssi;
    int8_t rx_rssi[2];
    uint32_t crc_prev = vif->u.sta.mon_last_crc;

    // Reset the beacon loss count
    vif->u.sta.beacon_loss_cnt = 0;

    #if NX_UMAC_PRESENT
    if (vif->u.sta.csa_occured)
    {
        mm_send_csa_traffic_ind(vif->index, true);
        vif->u.sta.csa_occured = false;
    }
    #endif

    #if (NX_P2P_GO)
    // Remind that at least one beacon has been received
    vif->u.sta.bcn_rcved = true;
    #endif //(NX_P2P_GO)

    // Check if we need to send a keep-alive frame
    if ((ke_time_past(vif->u.sta.mon_last_tx + MM_KEEP_ALIVE_PERIOD)) &&
            (txl_frame_send_null_frame(vif->u.sta.ap_id, NULL, NULL) == CO_OK))
    {
        // Update the keep-alive time
        vif->u.sta.mon_last_tx = ke_time();
    }

    // Retrieve the RSSI values from the RX vector
    rssi = hal_desc_get_rssi(&rhd->rx_vec_1, rx_rssi);

    #if (NX_ANT_DIV)
    // Update RSSI
    mm_ant_div_update_rssi(rx_rssi);
    #endif // (NX_ANT_DIV)

    // Check if RSSI is below or above the threshold
    mm_check_rssi(vif, rssi);

    // Compute the beacon CRC to check if some fields have changed
    vif->u.sta.mon_last_crc = mm_compute_beacon_crc(bcn, len, tim);
    #else // NX_CONNECTION_MONITOR
    *tim = mac_ie_tim_find(pbd->datastartptr + MAC_BEACON_VARIABLE_PART_OFT,
                           len - MAC_BEACON_VARIABLE_PART_OFT);
    #endif // NX_CONNECTION_MONITOR

    #if NX_MULTI_ROLE
    // Compute the time of the next TBTT
    mm_tbtt_compute(bcn, len, rhd, vif, sta, *tim);
    #endif

    #if NX_CONNECTION_MONITOR
    return (crc_prev != vif->u.sta.mon_last_crc);
    #else
    return (true);
    #endif
}
#endif

void mm_reset(void)
{
    // Check what was the state of the MM when the error occurred and behave accordingly
    switch (ke_state_get(TASK_MM))
    {
        case MM_ACTIVE:
            // MM was active or doze, so put back the HW in active state
            mm_active();
            break;

        default:
            // MM was IDLE, or going to IDLE, so set its state to IDLE
            ke_state_set(TASK_MM, MM_IDLE);
            break;
    }
}

void mm_active(void)
{
    // Put the HW in active state
    nxmac_next_state_setf(HW_ACTIVE);
    ke_state_set(TASK_MM, MM_ACTIVE);
}

#if NX_POWERSAVE || NX_CONNECTION_MONITOR || NX_MULTI_ROLE
void mm_sta_tbtt(void *env)
{
    // Get the VIF entry from the env pointer
    struct vif_info_tag *vif = (struct vif_info_tag *)env;
    #if (NX_MULTI_ROLE || NX_CHNL_CTXT)
    // TBTT Time
    uint32_t tbtt_time, next_tbtt_time;
    #endif //(NX_MULTI_ROLE || NX_CHNL_CTXT)

    PROF_STA_TBTT_SET();
    PROF_TBTT_IDX_SET(vif->chan_ctxt->idx);

    do
    {
        if (!vif->active)
            // STA is not associated, exit immediately
            break;

        #if NX_UMAC_PRESENT
        if (vif->u.sta.csa_count)
        {
            vif->u.sta.csa_count--;
            if (vif->u.sta.csa_count <= 1)
            {
                vif_mgmt_switch_channel(vif);
                break;
            }
            else if (vif->u.sta.csa_count == 2)
            {
                mm_send_csa_traffic_ind(vif->index, false);
            }
        }
        #endif

        #if (NX_MULTI_ROLE || NX_CHNL_CTXT)
        tbtt_time = vif->tbtt_timer.time;
        next_tbtt_time = tbtt_time + sta_info_tab[vif->u.sta.ap_id].bcn_int;
        #endif //(NX_MULTI_ROLE || NX_CHNL_CTXT)

        #if NX_MULTI_ROLE
        // Program next TBTT based on previous TBTT
        mm_timer_set(&vif->tbtt_timer, next_tbtt_time);
        #endif

        #if (NX_P2P || NX_CHNL_CTXT)
        vif_mgmt_bcn_to_prog(vif);
        #endif //(NX_P2P || NX_CHNL_CTXT)

        #if (NX_P2P)
        p2p_tbtt_handle(vif, tbtt_time + vif->u.sta.ctw_add_dur);
        #endif //(NX_P2P)

        #if (NX_CHNL_CTXT)
        if (chan_tbtt_start(vif, tbtt_time, next_tbtt_time))
            break;
        #endif //(NX_CHNL_CTXT)

        #if NX_POWERSAVE
        vif->prevent_sleep |= PS_VIF_WAITING_BCN;
        #endif

        #if NX_CONNECTION_MONITOR
        // Increase the beacon loss count (it will be reset upon beacon reception)
        vif->u.sta.beacon_loss_cnt++;

        // Check if we reached the beacon loss threshold
        if (vif->u.sta.beacon_loss_cnt > MM_BEACON_LOSS_THD)
        {
            // No beacons received for a long time, send a NULL frame to the AP
            txl_frame_send_null_frame(vif->u.sta.ap_id, mm_ap_probe_cfm, vif);
        }
        #if (NX_CHNL_CTXT)
        else if (vif->u.sta.beacon_loss_cnt > (MM_BEACON_LOSS_THD - 1))
        {
            // Try to spend more time on channel in order to catch a beacon
            chan_bcn_detect_start(vif);
        }
        #endif //(NX_CHNL_CTXT)
        #endif

        #if (NX_ANT_DIV)
        // Check if an antenna switch is needed
        mm_ant_div_check(vif->u.sta.beacon_loss_cnt);
        #endif //(NX_ANT_DIV)

    } while(0);
    PROF_STA_TBTT_CLR();
}
#endif

#if NX_BEACONING
/**
 ****************************************************************************************
 * @brief Handler for KE_EVT_PRIMARY_TBTT and KE_EVT_SECONDARY_TBTT events for
 *        beaconning interfaces.
 *
 * It prepares beacon transmission for all beaconning interfaces.
 *
 * @param[in] evt Event being processed
 ****************************************************************************************
 */
static void mm_ap_tbtt(uint32_t evt)
{
    struct vif_info_tag *vif = (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

    PROF_AP_TBTT_SET();

    // Protect from interrupt while flushing the queue
    GLOBAL_INT_DISABLE();

    // Halt the beacon queue
    txl_cntrl_halt_ac(AC_BCN);

    // Flush the beacon queue in case some old packets are still blocked
    txl_cntrl_flush_ac(AC_BCN, DESC_DONE_SW_TX_BIT);

    // Reenable the interrupts
    GLOBAL_INT_RESTORE();

    while (vif != NULL)
    {
        switch (vif->type)
        {
            case (VIF_AP):
            #if (RW_MESH_EN)
            case (VIF_MESH_POINT):
            #endif //(RW_MESH_EN)
            {
                #if (NX_CHNL_CTXT || NX_P2P_GO)
                uint32_t beacon_int;
                uint32_t next_tbtt;
                #endif //(NX_CHNL_CTXT || NX_P2P_GO)

                #if (NX_UMAC_PRESENT)
                vif->u.ap.bcn_tbtt_cnt--;

                // Check if VIF has to send a beacon on this TBTT
                if (vif->u.ap.bcn_tbtt_cnt)
                {
                    break;
                }

                // Reset the TBTT counter
                vif->u.ap.bcn_tbtt_cnt = vif->u.ap.bcn_tbtt_ratio;
                #endif //(NX_UMAC_PRESENT)

                #if (NX_CHNL_CTXT || NX_P2P_GO)
                // Program the Beacon Timeout timer
                vif_mgmt_bcn_to_prog(vif);
                #endif //(NX_P2P || NX_CHNL_CTXT)

                #if (NX_P2P_GO)
                p2p_tbtt_handle(vif, vif->tbtt_timer.time + HAL_MACHW_BCN_TX_DELAY_US +
                                MM_PRE_AP_TBTT_DELAY_US);
                #endif //(NX_P2P_GO)

                #if (NX_CHNL_CTXT || NX_P2P_GO)
                beacon_int = (uint32_t)vif->u.ap.bcn_int << 10;
                next_tbtt = nxmac_next_tbtt_get() << 5;
                if (next_tbtt < (HAL_MACHW_BCN_TX_DELAY_US + MM_PRE_AP_TBTT_DELAY_US))
                    next_tbtt += beacon_int;
                next_tbtt += ke_time();
                next_tbtt -= (HAL_MACHW_BCN_TX_DELAY_US + MM_PRE_AP_TBTT_DELAY_US);
                #endif //(NX_CHNL_CTXT || NX_P2P_GO)

                #if (NX_CHNL_CTXT)
                if (vif->chan_ctxt != NULL)
                {
                    chan_tbtt_start(vif, vif->tbtt_timer.time, next_tbtt);
                }
                #endif //(NX_CHNL_CTXT)

                #if (NX_P2P_GO)
                if (vif->p2p)
                {
                    /*
                     * When HW is in doze mode, it does not trigger the TBTT interrupt anymore. Hence we use a
                     * timer in order to wake up slightly before AP_TBTT occurs
                     */
                    mm_timer_set(&vif->tbtt_timer, next_tbtt);
                }
                #endif //(NX_P2P_GO)
            } break;

            default:
            {
            } break;
        }

        // Go to next entry
        vif = vif_mgmt_next(vif);
    }

    #if NX_BCN_AUTONOMOUS_TX
    // Transmit the beacon(s)
    mm_bcn_transmit();
    #endif

    PROF_AP_TBTT_CLR();
}
#endif

#if (NX_P2P_GO && NX_POWERSAVE)
void mm_ap_pre_tbtt(void *env)
{
    // Get the VIF on which the TBTT will occur soon
    struct vif_info_tag *vif = (struct vif_info_tag *)env;

    // Inform the P2P module about the coming TBTT
    p2p_go_pre_tbtt(vif);
}
#endif //(NX_P2P_GO && NX_POWERSAVE)

#if NX_BEACONING && ((NX_POWERSAVE || NX_CONNECTION_MONITOR || NX_UMAC_PRESENT) && !NX_MULTI_ROLE)
void mm_tbtt_evt(int dummy)
{
    uint32_t evt = ke_evt_get() & MM_TBTT_EVT_MASK;
    #if NX_POWERSAVE || NX_CONNECTION_MONITOR || NX_UMAC_PRESENT
    // Get the current VIF entry. For the moment we support only one VIF in STA
    // mode, so the first of the list is this one
    struct vif_info_tag *vif =
                        (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);
    #endif

    PROF_HW_TBTT_EVT_SET();

    // Sanity check - Primary and secondary TBTT events should not be active
    // at the same time
    ASSERT_ERR(evt != MM_TBTT_EVT_MASK);

    // Clear the event
    ke_evt_clear(evt);

    #if NX_POWERSAVE || NX_CONNECTION_MONITOR
    // Check if VIF is of STA type
    if (vif->type == VIF_STA)
    {
        // Call the specific STA TBTT handler
        mm_sta_tbtt(vif);
    }
    #if NX_BEACONING
    else
    #endif
    #endif
    {
        #if NX_BEACONING
        // Check if beaconing is enabled
        if (mm_env.beaconing)
        {
            mm_ap_tbtt(evt);
        }
        #endif
    }

    PROF_HW_TBTT_EVT_CLR();
}
#elif ((NX_POWERSAVE || NX_CONNECTION_MONITOR || NX_UMAC_PRESENT) && !NX_MULTI_ROLE)
void mm_tbtt_evt(int dummy)
{
    uint32_t evt = ke_evt_get() & MM_TBTT_EVT_MASK;
    // Get the current VIF entry. For the moment we support only one VIF in STA
    // mode, so the first of the list is this one
    struct vif_info_tag *vif =
                        (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

    PROF_HW_TBTT_EVT_SET();

    // Sanity check - Primary and secondary TBTT events should not be active
    // at the same time
    ASSERT_ERR(evt != MM_TBTT_EVT_MASK);

    // Clear the event
    ke_evt_clear(evt);

    // Call the specific STA TBTT handler
    mm_sta_tbtt(vif);

    PROF_HW_TBTT_EVT_CLR();
}
#elif NX_BEACONING
void mm_tbtt_evt(int dummy)
{
    uint32_t evt = ke_evt_get() & MM_TBTT_EVT_MASK;

    PROF_HW_TBTT_EVT_SET();

    // Sanity check - Primary and secondary TBTT events should not be active
    // at the same time
    ASSERT_ERR(evt != MM_TBTT_EVT_MASK);

    // Clear the event
    ke_evt_clear(evt);

    // Call the AP specific TBTT handler
    mm_ap_tbtt(evt);

    PROF_HW_TBTT_EVT_CLR();
}
#endif

uint8_t mm_sec_machwaddr_wr(uint8_t sta_idx, uint8_t inst_nbr)
{
    uint8_t hw_sta_idx;
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    uint32_t enc_cntrl;

    // Compute the HW STA index
    hw_sta_idx = MM_SEC_DEFAULT_KEY_COUNT + sta_idx;

    // Copy the MAC addr
    nxmac_encr_mac_addr_low_set(sta->mac_addr.array[0] | (((uint32_t)sta->mac_addr.array[1]) << 16));
    nxmac_encr_mac_addr_high_set(sta->mac_addr.array[2]);

    // Reset the key data
    nxmac_encr_key_0_set(0);
    nxmac_encr_key_1_set(0);
    nxmac_encr_key_2_set(0);
    nxmac_encr_key_3_set(0);

    // For AP vif, key ram is not cleared when STA is removed, ensure that there is only
    // one entry for this MAC, by deleting previous one if present
    nxmac_encr_cntrl_set(NXMAC_NEW_SEARCH_BIT);
    do
    {
        enc_cntrl = nxmac_encr_cntrl_get();
    } while(enc_cntrl & NXMAC_NEW_SEARCH_BIT);

    if (!(enc_cntrl & NXMAC_SEARCH_ERROR_BIT))
    {
        uint8_t hw_sta_idx_prev = nxmac_key_index_ram_getf();

        if (hw_sta_idx_prev != hw_sta_idx)
        {
            nxmac_encr_mac_addr_low_set(0xFFFFFFFF);
            nxmac_encr_mac_addr_high_set(0xFFFFFFFF);
            KEY_RAM_WR(hw_sta_idx_prev, 0, 0, 0, 0, 0);
            nxmac_encr_mac_addr_low_set(sta->mac_addr.array[0] | (((uint32_t)sta->mac_addr.array[1]) << 16));
            nxmac_encr_mac_addr_high_set(sta->mac_addr.array[2]);
        }
    }

    // Write control field
    // keyIdx, cType, vlanIdx, spp, usedefkey, cLen
    KEY_RAM_WR(hw_sta_idx, 0, inst_nbr, 0, 1, 0);

    return (hw_sta_idx);
}

uint8_t mm_sec_machwkey_wr(struct mm_key_add_req const *param)
{
    uint8_t clen = 1;
    uint8_t ctype = MM_SEC_CTYPE_NULL;
    uint8_t key_idx_hw;
    uint8_t sta_idx = param->sta_idx;
    struct mac_sec_key const *key = &param->key;
    uint8_t vlan_idx = param->inst_nbr;

    // Get index to be written in the HW table
    if (sta_idx == INVALID_STA_IDX)
    {
        #if NX_MFP
        // For now Keys for MFP are not installed in hw
        if (param->cipher_suite == MAC_CIPHER_BIP_CMAC_128)
        {
            key_idx_hw = MM_VIF_TO_MFP_KEY(param->key_idx, param->inst_nbr);
            vif_mgmt_add_key(param, key_idx_hw);
            return key_idx_hw;
        }
        #endif

        // Default key index
        key_idx_hw = MM_VIF_TO_KEY(param->key_idx, param->inst_nbr);

        // Put an invalid MAC addr
        nxmac_encr_mac_addr_low_set(0xFFFFFFFF);
        nxmac_encr_mac_addr_high_set(0xFFFFFFFF);

        #if (NX_UMAC_PRESENT)
        // Set the key parameters to the VIF
        vif_mgmt_add_key(param, key_idx_hw);
        #endif
    }
    else
    {
        #if (RW_MESH_EN)
        // Get VIF Information
        struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];
        #endif //(RW_MESH_EN)
        struct sta_info_tag *sta = &sta_info_tab[sta_idx];

        // Sanity check
        ASSERT_ERR(sta_idx < STA_MAX);

        // Pairwise key index
        key_idx_hw = MM_STA_TO_KEY(sta_idx);

        #if (RW_MESH_EN)
        if (vif->type == VIF_MESH_POINT)
        {
            vlan_idx = sta->mlink_idx + NX_VIRT_DEV_MAX;

            if (!param->pairwise)
            {
                if (param->cipher_suite == MAC_CIPHER_CCMP)
                {
                    // Peer Mesh Group Key Index
                    key_idx_hw = MM_MLINK_TO_KEY(param->key_idx + 1, sta->mlink_idx);

                    // Put an invalid MAC addr
                    nxmac_encr_mac_addr_low_set(0xFFFFFFFF);
                    nxmac_encr_mac_addr_high_set(0xFFFFFFFF);
                }
                #if (NX_UMAC_PRESENT)
                else
                {
                    key_idx_hw = MM_STA_TO_MESH_MFP_KEY(param->key_idx, sta_idx);

                    sta_mgmt_add_key(param, key_idx_hw);

                    return (key_idx_hw);
                }
                #endif //(NX_UMAC_PRESENT)
            }
        }

        if (param->pairwise)
        #endif //(RW_MESH_EN)
        {
            #if (NX_UMAC_PRESENT)
            // Set the key parameters to the STA
            sta_mgmt_add_key(param, key_idx_hw);
            #endif

            // Copy MAC addr
            nxmac_encr_mac_addr_low_set(sta->mac_addr.array[0] | (((uint32_t)sta->mac_addr.array[1]) << 16));
            nxmac_encr_mac_addr_high_set(sta->mac_addr.array[2]);
        }
    }

    // Get cType and cLen from Cipher Suite
    mm_key_ram_param_get(param->cipher_suite, key, &ctype, &clen);

    // Copy key data
    nxmac_encr_key_0_set(key->array[0]);
    nxmac_encr_key_1_set(key->array[1]);
    nxmac_encr_key_2_set(key->array[2]);
    nxmac_encr_key_3_set(key->array[3]);

    #if RW_WAPI_EN
    if (ctype == MM_SEC_CTYPE_WPI_SMS4)
    {
        nxmac_encr_wpi_int_key_0_set(key->array[4]);
        nxmac_encr_wpi_int_key_1_set(key->array[5]);
        nxmac_encr_wpi_int_key_2_set(key->array[6]);
        nxmac_encr_wpi_int_key_3_set(key->array[7]);
    }
    #endif

    // Write control field
    // keyIdx, cType, vlanIdx, spp, usedefkey, cLen
    KEY_RAM_WR(key_idx_hw, ctype, vlan_idx, param->spp, 0, clen);

    return (key_idx_hw);
}

void mm_sec_machwkey_del(uint8_t hw_key_idx)
{

    #if NX_MFP
    if (hw_key_idx >= MM_SEC_MAX_KEY_NBR)
    {
        vif_mgmt_del_key(&vif_info_tab[MM_MFP_KEY_TO_VIF(hw_key_idx)],
                         MM_MFP_KEY_TO_KEYID(hw_key_idx));
        return;
    }
    #endif

    // Get index to be written in the HW table
    if (hw_key_idx >= MM_SEC_DEFAULT_KEY_COUNT)
    {
        // Pairwise key index
        uint8_t sta_idx = MM_KEY_TO_STA(hw_key_idx);
        struct sta_info_tag *sta = &sta_info_tab[sta_idx];
        struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];

        #if (NX_UMAC_PRESENT)
        sta_mgmt_del_key(sta);
        #endif

        // For AP VIF don't clear key ram so that AP is able to detect old STA still
        // using the key. (If key is cleared MACHW will simply ignore frames encrypted
        // with unknown key)
        if (vif->type == VIF_AP)
            return;

        // Copy MAC addr
        nxmac_encr_mac_addr_low_set(sta->mac_addr.array[0] | (((uint32_t)sta->mac_addr.array[1]) << 16));
        nxmac_encr_mac_addr_high_set(sta->mac_addr.array[2]);
    }
    else
    {
        // Put an invalid MAC addr
        nxmac_encr_mac_addr_low_set(0xFFFFFFFF);
        nxmac_encr_mac_addr_high_set(0xFFFFFFFF);

        #if (RW_MESH_EN)
        if (hw_key_idx < MM_SEC_DEFAULT_VIF_KEY_COUNT)
        #endif //(RW_MESH_EN)
        {
            #if (NX_UMAC_PRESENT)
            vif_mgmt_del_key(&vif_info_tab[MM_KEY_TO_VIF(hw_key_idx)],
                             MM_KEY_TO_KEYID(hw_key_idx));
            #endif
        }
    }

    // Reset the key data
    nxmac_encr_key_0_set(0);
    nxmac_encr_key_1_set(0);
    nxmac_encr_key_2_set(0);
    nxmac_encr_key_3_set(0);

    // Write control field
    // keyIdx, cType, vlanIdx, spp, usedefkey, cLen
    KEY_RAM_WR(hw_key_idx, 0, 0, 0, 0, 0);
}

void mm_sec_machwaddr_del(uint8_t sta_idx)
{
    uint8_t hw_sta_idx;

    // Compute the HW STA index
    hw_sta_idx = MM_SEC_DEFAULT_KEY_COUNT + sta_idx;

    // Put an invalid MAC addr
    nxmac_encr_mac_addr_low_set(0xFFFFFFFF);
    nxmac_encr_mac_addr_high_set(0xFFFFFFFF);

    // Reset the key data
    nxmac_encr_key_0_set(0);
    nxmac_encr_key_1_set(0);
    nxmac_encr_key_2_set(0);
    nxmac_encr_key_3_set(0);

    // Write control field
    // keyIdx, cType, vlanIdx, spp, usedefkey, cLen
    KEY_RAM_WR(hw_sta_idx, 0, 0, 0, 0, 0);
}

void mm_hw_idle_evt(int dummy)
{
    // Clear the event
    ke_evt_clear(KE_EVT_HW_IDLE_BIT);

    // Put the MM task in IDLE state
    ke_state_set(TASK_MM, MM_IDLE);
}

#if NX_MULTI_ROLE
void mm_hw_info_set(struct mac_addr const *mac_addr)
{
    // By default we are configured as STA
    nxmac_ap_setf(0);
    nxmac_bss_type_setf(1);
    nxmac_mac_addr_hi_mask_set(MM_MAC_ADDR_MSK);

    // Reset TSF
    nxmac_tsf_lo_set(0);
    nxmac_tsf_hi_set(0);

    // Set the MAC address of the interface to the MAC HW
    nxmac_mac_addr_low_set(mac_addr->array[0] | (((uint32_t)mac_addr->array[1]) << 16));
    nxmac_mac_addr_hi_set(mac_addr->array[2]);

    // Now that we have a MAC Address we are supposed to reply
    nxmac_mac_cntrl_1_set(nxmac_mac_cntrl_1_get() & ~(NXMAC_DISABLE_ACK_RESP_BIT
                            | NXMAC_DISABLE_CTS_RESP_BIT | NXMAC_DISABLE_BA_RESP_BIT));

    // Enable reception of useful frames only
    mm_rx_filter_umac_set(MM_RX_FILTER_ACTIVE);

    // Configure RX path for active mode
    rxl_hwdesc_monitor(false);
}

void mm_hw_ap_info_set(void)
{
    // Enable the AP mode. This will trigger the beacon transmission at TBTT.
    nxmac_ap_setf(1);

    #if NX_UMAC_PRESENT
    // Configure the RX filter to receive PS-poll and beacons from other networks
    mm_rx_filter_umac_set(MM_RX_FILTER_ACTIVE | (NXMAC_ACCEPT_PS_POLL_BIT
                                               | NXMAC_ACCEPT_ALL_BEACON_BIT));
    #endif

    // Configure RX path for active mode
    rxl_hwdesc_monitor(false);

    // Enable TBTT HW interrupt
    nxmac_gen_int_ack_clear(MM_TBTT_IRQ_MASK);
    nxmac_gen_int_enable_set(nxmac_gen_int_enable_get() | MM_TBTT_IRQ_MASK);
}

void mm_hw_ap_info_reset(void)
{
    // Disable the AP mode
    nxmac_ap_setf(0);

    #if NX_UMAC_PRESENT
    // Configure the RX filter to discard PS-poll and beacons from other networks
    mm_rx_filter_umac_set(MM_RX_FILTER_ACTIVE);
    #endif

    // Configure RX path for active mode
    rxl_hwdesc_monitor(false);

    // Disable TBTT HW interrupt
    nxmac_gen_int_ack_clear(MM_TBTT_IRQ_MASK);
    nxmac_gen_int_enable_set(nxmac_gen_int_enable_get() & ~MM_TBTT_IRQ_MASK);
}
#else
void mm_hw_interface_info_set(uint8_t type, struct mac_addr const *mac_addr)
{
    // Configure the modes according to the interface type
    switch (type)
    {
        case VIF_AP:
            nxmac_ap_setf(1);
            nxmac_bss_type_setf(1);
            nxmac_mac_addr_hi_mask_set(MM_MAC_ADDR_MSK);
            nxmac_gen_int_ack_clear(MM_TBTT_IRQ_MASK);
            nxmac_gen_int_enable_set(nxmac_gen_int_enable_get() | MM_TBTT_IRQ_MASK);
            break;
        case VIF_IBSS:
            nxmac_ap_setf(0);
            nxmac_bss_type_setf(0);
            nxmac_mac_addr_hi_mask_set(0);
            nxmac_gen_int_enable_set(nxmac_gen_int_enable_get() | MM_TBTT_IRQ_MASK);
            break;
        case VIF_MONITOR:
        case VIF_STA:
            nxmac_ap_setf(0);
            nxmac_bss_type_setf(1);
            nxmac_mac_addr_hi_mask_set(0);
            #if NX_POWERSAVE || NX_CONNECTION_MONITOR || NX_UMAC_PRESENT
            nxmac_gen_int_ack_clear(MM_TBTT_IRQ_MASK);
            nxmac_gen_int_enable_set(nxmac_gen_int_enable_get() | MM_TBTT_IRQ_MASK);
            #else
            nxmac_gen_int_enable_set(nxmac_gen_int_enable_get() & ~MM_TBTT_IRQ_MASK);
            #endif
            break;
        default:
            ASSERT_ERR(0);
            break;
    }

    // Reset TSF
    nxmac_tsf_lo_set(0);
    nxmac_tsf_hi_set(0);

    // Set the MAC address of the interface to the MAC HW
    nxmac_mac_addr_low_set(mac_addr->array[0] | (((uint32_t)mac_addr->array[1]) << 16));
    nxmac_mac_addr_hi_set(mac_addr->array[2]);

    // Now that we have a MAC Address we are supposed to reply
    nxmac_mac_cntrl_1_set(nxmac_mac_cntrl_1_get() & ~(NXMAC_DISABLE_ACK_RESP_BIT
                            | NXMAC_DISABLE_CTS_RESP_BIT | NXMAC_DISABLE_BA_RESP_BIT));

    // Enable reception of useful frames only
    mm_rx_filter_umac_set(MM_RX_FILTER_ACTIVE);

    // Configure RX path for active mode
    rxl_hwdesc_monitor(false);
}
#endif

#if (NX_AMPDU_TX)
bool mm_ba_agmt_tx_exists(uint8_t sta_idx, uint8_t tid)
{
    #if (NX_UMAC_PRESENT)
    return (sta_mgmt_get_tx_bam_idx(sta_idx, tid) != BAM_INVALID_TASK_IDX);
    #else
    // 0 buffer size -> no BA agreement has been set here
    return (sta_mgmt_get_tx_buff_size(sta_idx, tid));
    #endif //(NX_UMAC_PRESENT)
}
#endif //(NX_AMPDU_TX)

#if (NX_REORD || NX_UMAC_PRESENT)
bool mm_ba_agmt_rx_exists(uint8_t sta_idx, uint8_t tid)
{
    #if (NX_UMAC_PRESENT)
    return (sta_mgmt_get_rx_bam_idx(sta_idx, tid) != BAM_INVALID_TASK_IDX);
    #else
    // If the pointer to the reordering structure is NULL, does not exists
    return (sta_info_tab[sta_idx].ba_agmts_rx[tid] != NULL);
    #endif //(NX_UMAC_PRESENT)
}
#endif //(NX_REORD)

#if NX_BEACONING
int32_t mm_ap_tbtt_move(int32_t offset)
{
    uint32_t time_to_next;
    uint32_t tbtt_next;
    uint32_t beacon_int = nxmac_beacon_int_getf() * TU_DURATION;
    uint32_t prog_delay = beacon_int / 4;
    uint32_t abs_offset = co_abs(offset);
    uint32_t comp_margin = 100;  // Margin used to handle the inaccuracy in the delays

    // Check if the TBTT move is possible and required
    if (mm_env.tbtt_move_ongoing || (offset == 0))
        return 0;

    mm_env.tbtt_move_ongoing = true;

    // Correct the offset, that should not be greater than half a beacon period
    if ((offset > 0) && (abs_offset > (beacon_int/2)))
        offset = (int32_t)(beacon_int/2);
    else if ((offset < 0) && (abs_offset > (beacon_int/2)))
        offset = -(int32_t)(beacon_int/2);

    GLOBAL_INT_DISABLE();
    time_to_next = nxmac_next_tbtt_get() * 32;
    tbtt_next = hal_machw_time() + time_to_next;
    // Two cases are possible, depending if the offset is positive or negative
    if (offset > 0)
    {
        // Offset is positive so we will delay the TBTT by <abs_offset us>. In order to
        // avoid getting a fake TBTT event from the HW <abs_offset us> after the previous
        // TBTT, we need to modify the TSF at least <abs_offset us> after the previous TBTT,
        // but not too close to the next one to ensure that the delay is done in time
        uint32_t time_from_prev = beacon_int - time_to_next;

        // If we don't have enough margin till the next TBTT, it is too late to apply the
        // delay on it. We therefore program a timer to perform the delay one-fourth
        // beacon interval before the TBTT after
        if (time_to_next < prog_delay)
        {
            mm_env.tbtt_move_tmr.env = (void *)offset;
            mm_timer_set(&mm_env.tbtt_move_tmr, tbtt_next + beacon_int - prog_delay);
        }
        // If we are less than abs_offset after the previous TBTT, we program a timer
        // to perform the delay one-fourth beacon interval before the next TBTT
        else if (time_from_prev < (abs_offset + comp_margin))
        {
            mm_env.tbtt_move_tmr.env = (void *)offset;
            mm_timer_set(&mm_env.tbtt_move_tmr, tbtt_next - prog_delay);
        }
        // Otherwise we are in the window where we can request the delay immediately
        else
        {
            mm_ap_tbtt_move_cb((void *)offset);
        }
    }
    else
    {
        // Offset is negative so we will advance the TBTT by x us. In order to avoid
        // missing a TBTT event from the HW, we need to modify the TSF at least x us
        // before the next TBTT
        // If we are early enough in the beacon period, we request the delay to be done
        // immediately
        if (time_to_next > (abs_offset + prog_delay))
        {
            mm_ap_tbtt_move_cb((void *)offset);
        }
        // Otherwise it might be too late to apply the delay immediately, so we program
        // a timer to perform it just after the next TBTT
        else
        {
            mm_env.tbtt_move_tmr.env = (void *)offset;
            mm_timer_set(&mm_env.tbtt_move_tmr, tbtt_next + comp_margin);
        }
    }
    GLOBAL_INT_RESTORE();

    return offset;
}
#endif

void mm_back_to_host_idle(void)
{
    // Sanity check - This function shall be called while HW is IDLE
    ASSERT_ERR(ke_state_get(TASK_MM) == MM_HOST_BYPASSED);

    // Change the state of the MM
    if (mm_env.host_idle == 0)
        mm_active();
    else
        ke_state_set(TASK_MM, MM_IDLE);
}

void mm_force_idle_req(void)
{
    // Sanity check - The force IDLE procedure shall not be performed while the NO IDLE
    // state is present for a beacon transmission. If this assertion is triggered it
    // probably means that the scheduling of a channel switch is done at a bad time.
    ASSERT_ERR(ke_state_get(TASK_MM) != MM_NO_IDLE);

    // Disable interrupts
    GLOBAL_INT_DISABLE();

    // Reset the MAC HW (this will reset the PHY too)
    hal_machw_reset();

    // Reset the RX path
    rxl_reset();

    // Reset the TX path
    txl_reset();

    // Reset the MM state
    ke_state_set(TASK_MM, MM_HOST_BYPASSED);
    mm_env.prev_mm_state = MM_IDLE;
    mm_env.prev_hw_state = HW_IDLE;

    // Restore the interrupts
    GLOBAL_INT_RESTORE();
}

void mm_no_idle_start(void)
{
    ASSERT_ERR((ke_state_get(TASK_MM) != MM_HOST_BYPASSED) &&
               (ke_state_get(TASK_MM) != MM_IDLE));

    if (ke_state_get(TASK_MM) == MM_GOING_TO_IDLE)
    {
        nxmac_next_state_setf(HW_ACTIVE);
        GLOBAL_INT_DISABLE();
        ke_evt_clear(KE_EVT_HW_IDLE_BIT);
        if (nxmac_status_idle_interrupt_getf())
        {
            nxmac_gen_int_ack_clear(NXMAC_IDLE_INTERRUPT_BIT);
        }
        GLOBAL_INT_RESTORE();
    }
    ke_state_set(TASK_MM, MM_NO_IDLE);
}

void mm_no_idle_stop(void)
{
    if (ke_state_get(TASK_MM) == MM_NO_IDLE)
        ke_state_set(TASK_MM, MM_ACTIVE);
}

uint8_t mm_sta_add(struct mm_sta_add_req const *param, uint8_t  *sta_idx,
                   uint8_t *hw_sta_idx)
{
    uint8_t status;

    // Register the new station
    status = sta_mgmt_register(param, sta_idx);

    // Check if the registration was successful
    if (status == CO_OK)
    {
        struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];
        struct sta_info_tag *sta = &sta_info_tab[*sta_idx];

        // Compute the HW STA index
        *hw_sta_idx = mm_sec_machwaddr_wr(*sta_idx, param->inst_nbr);
        if (sta->linked_sta)
        {
            mm_sec_machwaddr_wr(sta->linked_sta->staid, param->inst_nbr);
        }

        // Check if the VIF is of STA type
        #if NX_TDLS
        if ((vif->type == VIF_STA) && (!param->tdls_sta))
        #else
        if (vif->type == VIF_STA)
        #endif
        {
            // We save the AP index for later use
            vif->u.sta.ap_id = *sta_idx;
            vif->u.sta.bssid_index = param->bssid_index;
            vif->u.sta.max_bssid_ind = param->max_bssid_ind;
        }
        #if (RW_UMESH_EN)
        else if (vif->type == VIF_MESH_POINT)
        {
            // Inform the Mesh Module that the STA is now registered
            mesh_add_sta_cfm(param->inst_nbr, *sta_idx);
        }
        #endif //(RW_UMESH_EN)
    }

    return (status);
}

void mm_sta_del(uint8_t sta_idx)
{
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];

    // Check if the VIF is of STA type
    if ((vif->type == VIF_STA)
        #if NX_TDLS
        && !sta->is_tdls
        #endif
       )
    {
        // We save the AP index for later use
        vif->u.sta.ap_id = INVALID_STA_IDX;
    }
    #if (NX_UMAC_PRESENT || NX_P2P_GO)
    else
    {
        #if (RW_UMESH_EN)
        if (vif->type == VIF_MESH_POINT)
        {
            // Inform the Mesh Module that the STA has been unregistered
            mesh_del_sta_cfm(sta->mlink_idx);
        }
        #endif //(RW_UMESH_EN)

        // Check if the station is in PS or not
        if (sta->ps_state == PS_MODE_ON)
        {
            // Update the number of PS stations
            vif->u.ap.ps_sta_cnt--;

            #if (NX_UMAC_PRESENT)
            if (!vif->u.ap.ps_sta_cnt)
            {
                PROF_PS_BCMC_STATE_SET();
                PROF_PS_STATE_VAL_SET(PS_MODE_OFF);
                mm_ps_change_ind(VIF_TO_BCMC_IDX(vif->index), PS_MODE_OFF);
                #if NX_BEACONING
                apm_tx_int_ps_clear(vif, VIF_TO_BCMC_IDX(vif->index));
                #endif // NX_BEACONING
                PROF_PS_BCMC_STATE_CLR();
            }
            #endif //(NX_UMAC_PRESENT)
        }
    }
    #endif //(NX_UMAC_PRESENT || NX_P2P_GO)

    // Delete the station MAC address from the key storage, except for AP vif so that
    // AP can detect STA still using old key.
    if (vif->type != VIF_AP)
    {
        mm_sec_machwaddr_del(sta_idx);
        if (sta->linked_sta)
        {
            mm_sec_machwaddr_del(sta->linked_sta->staid);
        }
    }

    // Unregister the station
    sta_mgmt_unregister(sta_idx);
}

#if (NX_CONNECTION_MONITOR)
void mm_send_connection_loss_ind(struct vif_info_tag *vif)
{
    // Otherwise, we consider that the connection is lost
    struct mm_connection_loss_ind *ind =
       KE_MSG_ALLOC(MM_CONNECTION_LOSS_IND, TASK_IND, TASK_MM, mm_connection_loss_ind);

    // Fill-in the indication message parameters
    ind->inst_nbr = vif->index;

    // Send the indication to the upper layers
    ke_msg_send(ind);
}
#endif //(NX_CONNECTION_MONITOR)

void mm_check_rssi(struct vif_info_tag *vif, int8_t rssi)
{
    int8_t rssi_old = vif->u.sta.rssi;
    int8_t rssi_thold = vif->u.sta.rssi_thold;
    int8_t rssi_hyst = vif->u.sta.rssi_hyst;
    bool rssi_status = vif->u.sta.rssi_status;

    // Update current RSSI
    vif->u.sta.rssi = rssi;

    // Check if threshold is set
    if (rssi_thold == 0)
        return;

    // Check RSSI
    if ((rssi_status == 0) && (rssi < rssi_old) && (rssi < (rssi_thold - rssi_hyst)))
    {
        rssi_status = 1;
    }
    else if ((rssi_status == 1) && (rssi > rssi_old) && (rssi > (rssi_thold + rssi_hyst)))
    {
        rssi_status = 0;
    }

    if (rssi_status != vif->u.sta.rssi_status)
    {
        // If RSSI status is changed send MM_RSSI_STATUS_IND message
        struct mm_rssi_status_ind *ind =
           KE_MSG_ALLOC(MM_RSSI_STATUS_IND, TASK_API, TASK_MM, mm_rssi_status_ind);

        // Fill-in the indication message parameters
        ind->vif_index = vif->index;
        ind->rssi_status = rssi_status;
        ind->rssi = rssi;

        // Send the indication to the upper layers
        ke_msg_send(ind);
    }

    // Update current RSSI status
    vif->u.sta.rssi_status = rssi_status;
}

#if NX_UMAC_PRESENT
void mm_send_pktloss_ind(struct vif_info_tag *vif, uint8_t sta_idx, uint32_t num_pkts)
{

    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    struct mm_pktloss_ind *ind =
       KE_MSG_ALLOC(MM_PKTLOSS_IND, TASK_API, TASK_MM, mm_pktloss_ind);

    // Fill-in the indication message parameters
    ind->vif_index = vif->index;
    MAC_ADDR_CPY(&ind->mac_addr, &sta->mac_addr);
    ind->num_packets = num_pkts;

    // Send the indication to the upper layers
    ke_msg_send(ind);
}

void mm_send_csa_traffic_ind(uint8_t vif_index, bool enable)
{
    struct mm_csa_traffic_ind *ind = KE_MSG_ALLOC(MM_CSA_TRAFFIC_IND, TASK_API, TASK_MM,
                                                  mm_csa_traffic_ind);
    ind->vif_index = vif_index;
    ind->enable = enable;
    ke_msg_send(ind);
}
#endif

#if (NX_ANT_DIV)
void mm_ant_div_init(bool enable)
{
    // Set enable flag
    mm_env.ant_div.enable = enable;
    // Set switch enable flag
    mm_env.ant_div.sw_enable = enable;
    // Reset number of active VIFs
    mm_env.ant_div.nvif_active = 0;
    // Reset RSSI values (current and initial) for both antennas
    mm_ant_div_reset_rssi(0);
    mm_ant_div_reset_rssi(1);
}

void mm_ant_div_switch_and_stop(void)
{
    // Check if antenna diversity algorithm is enabled and
    // no other connection is active
    if (mm_env.ant_div.enable && (mm_env.ant_div.nvif_active == 0))
    {
        // Switch antenna paths before running a scan
        mm_ant_div_switch_antenna();
    }
    // Disable switch
    mm_env.ant_div.sw_enable = 0;
}

void mm_ant_div_update(void)
{
    // Disable antenna switch if more than 1 connection is already active
    if (!mm_env.ant_div.enable || (mm_env.ant_div.nvif_active > 1))
    {
        mm_env.ant_div.sw_enable = 0;
        if (phy_get_nrx() == 0)
        {
            // Reset RSSI values (current and initial) for both antennas
            mm_ant_div_reset_rssi(0);
            mm_ant_div_reset_rssi(1);
            // Reset update counter
            mm_env.ant_div.update_cnt = 0;
            // Keep in mind we need to update the initial RSSI for the current antenna
            mm_env.ant_div.init_rssi_upd_req = 1;
        }
    }
    else
    {
        mm_env.ant_div.sw_enable = 1;
    }
}

void mm_ant_div_restore(bool active)
{
    // Update number of active NVIFs
    if (active)
    {
        mm_env.ant_div.nvif_active += 1;
    }
    else
    {
        mm_env.ant_div.nvif_active -= 1;
    }
    // Update antenna switch flag
    mm_ant_div_update();
}

void mm_ant_div_switch_antenna(void)
{
    // Check if antenna switch is enabled
    if (!mm_env.ant_div.sw_enable)
    {
        return;
    }
    dbg(D_INF D_MM "Antenna diversity: switch antenna %d --> %d\n",
        mm_env.ant_div.cur_ant, !mm_env.ant_div.cur_ant);
    // Switch antenna
    mm_env.ant_div.cur_ant = phy_switch_antenna_paths();
    if (phy_get_nrx() == 0)
    {
        // Reset update counter
        mm_env.ant_div.update_cnt = 0;
        // Keep in mind we need to update the initial RSSI for the current antenna
        mm_env.ant_div.init_rssi_upd_req = 1;
    }
}

void mm_ant_div_reset_rssi(uint8_t ant)
{
    // Reset current RSSI
    mm_env.ant_div.rssi[ant] = ANT_DIV_RSSI_RESET;
    // Reset initial RSSI
    mm_env.ant_div.init_rssi[ant] = ANT_DIV_RSSI_RESET;
}

void mm_ant_div_update_rssi(int8_t *rx_rssi)
{
    ASSERT_ERR(rx_rssi != NULL);
    // Pointer to the antenna diversity structure
    struct antenna_diversity *ant_div = &mm_env.ant_div;
    // Update RSSI values
    if (phy_get_nrx() == 0)
    {
        // Update current RSSI value
        ant_div->rssi[ant_div->cur_ant] = rx_rssi[0];
        // Update initial RSSI value if requested
        if (ant_div->init_rssi_upd_req)
        {
            ant_div->init_rssi[ant_div->cur_ant] = rx_rssi[0];
            ant_div->init_rssi_upd_req = 0;
        }
    }
    else
    {
        // Update current RSSI values
        ant_div->rssi[ant_div->cur_ant] = rx_rssi[0];
        ant_div->rssi[!ant_div->cur_ant] = rx_rssi[1];
    }
}

void mm_ant_div_check(uint8_t beacon_loss_cnt)
{
    // Check if antenna switch is enabled
    if (!mm_env.ant_div.sw_enable)
    {
        return;
    }
    // Pointer to the antenna diversity structure
    struct antenna_diversity *ant_div = &mm_env.ant_div;
    // Check if antenna switch is needed in case of 1x1 station
    if (phy_get_nrx() == 0)
    {
        // Boolean used to request a switch of antenna
        bool switch_antenna = 0;
        // Distance from current antenna RSSI and the golden range
        uint8_t gr_dist_cur = 0;
        // Distance from the other antenna RSSI and the golden range
        uint8_t gr_dist_other = 0;
        // Current RSSI
        int8_t cur_rssi = ant_div->rssi[ant_div->cur_ant];
        // Current initial RSSI
        int8_t cur_init_rssi = ant_div->init_rssi[ant_div->cur_ant];
        // Other antenna RSSI
        int8_t other_rssi = ant_div->rssi[!ant_div->cur_ant];

        // Update counter of beacon periods without any antenna switch
        ant_div->update_cnt++;

        // Get distance from golden range for the current antenna RSSI
        gr_dist_cur = mm_get_golden_range_distance(cur_rssi);
        // Get distance from golden range for the other antenna RSSI
        gr_dist_other = mm_get_golden_range_distance(other_rssi);
        // Check if antenna diversity beacon loss threshold is exceeded
        if (beacon_loss_cnt > ANT_DIV_BEACON_LOSS_THD)
        {
            // Reset current antenna RSSI values
            mm_ant_div_reset_rssi(ant_div->cur_ant);
            // Keep in mind that the antenna should be switched
            switch_antenna = 1;
        }
        // Switch antenna if distance from golden range of the other RSSI is less than
        // distance from golden range of the current RSSI
        else if (gr_dist_other < gr_dist_cur)
        {
            switch_antenna = 1;
        }
        // Switch antenna if current antenna RSSI is changed of more than ANT_DIV_RSSI_STEP_CHANGE
        // compared to its initial RSSI
        else if ((cur_rssi < ANT_DIV_RSSI_GOLDEN_MIN) &&
                 (cur_rssi < (cur_init_rssi - ANT_DIV_RSSI_STEP_CHANGE)))
        {
            switch_antenna = 1;
        }
        else if ((cur_rssi > ANT_DIV_RSSI_GOLDEN_MAX) &&
                 (cur_rssi > (cur_init_rssi + ANT_DIV_RSSI_STEP_CHANGE)))
        {
            switch_antenna = 1;
        }
        // Check if current RSSI is ouside the golden range and the current antenna has not
        // been switched for at least ANT_DIV_UPDATE_CNT_THD beacon periods
        else if (((cur_rssi < ANT_DIV_RSSI_GOLDEN_MIN) ||
                 (cur_rssi > ANT_DIV_RSSI_GOLDEN_MAX)) &&
                 (ant_div->update_cnt > ANT_DIV_UPDATE_CNT_THD))
        {
            // Keep in mind that the antenna should be switched
            switch_antenna = 1;
        }
        // Switch antenna if requested
        if (switch_antenna)
        {
            dbg(D_INF D_MM "Antenna diversity: "
                "Ant%d=%d(%d) Ant%d=%d(%d) / Gold_range=[%d:%d] / Upd_cnt=%d\n",
                ant_div->cur_ant, cur_rssi, cur_init_rssi,
                !ant_div->cur_ant, other_rssi, ant_div->init_rssi[!ant_div->cur_ant],
                ANT_DIV_RSSI_GOLDEN_MIN, ANT_DIV_RSSI_GOLDEN_MAX, mm_env.ant_div.update_cnt);
            // Request an antenna switch
            ke_msg_send_basic(MM_SWITCH_ANTENNA_REQ, TASK_MM, TASK_NONE);
        }
    }
    // Check if antenna switch is needed in case of 2x2 station
    else
    {
        // Check RSSI on both antennas and select the best one as primary antenna
        int8_t rssi_0 = ant_div->rssi[ant_div->cur_ant];
        int8_t rssi_1 = ant_div->rssi[!ant_div->cur_ant];
        if (rssi_1 > (rssi_0 + ANT_DIV_RSSI_THD))
        {
            dbg(D_INF D_MM "Antenna diversity: Ant%d=%d Ant%d=%d\n",
                ant_div->cur_ant, rssi_0, !ant_div->cur_ant, rssi_1);
            // Switch antenna
            mm_ant_div_switch_antenna();
        }
    }
}
#endif //(NX_ANT_DIV)

/// @} end of group
