/**
 ****************************************************************************************
 *
 * @file mm_bcn.c
 *
 * @brief MAC Beacon management module implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup MM_BCN
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "mac_frame.h"
#include "mm.h"
#include "mm_timer.h"
#include "co_endian.h"
#include "sta_mgmt.h"
#include "vif_mgmt.h"
#include "phy.h"
#include "dbg.h"
#include "rd.h"
#include "ps.h"
#include "txl_cntrl.h"
#include "txl_frame.h"
#include "rxl_cntrl.h"
#include "hal_machw.h"
#include "scan.h"
#include "chan.h"
#include "mm_bcn.h"
#include "hal_dma.h"
#include "macif.h"
#include "tpc.h"

#include "reg_mac_core.h"
#include "reg_mac_pl.h"
#if (NX_UMAC_PRESENT)
#include "me_utils.h"
#endif

#if NX_BCN_AUTONOMOUS_TX

/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
///  Global data for maintaining beacon information

/**  LMAC MM BCN Context variable, used to store MM BCN Context data
 */
struct mm_bcn_env_tag mm_bcn_env;

/**
 ****************************************************************************************
 * @brief Update TIM IE in the beacon
 *
 * Proceeds with the update of TIM IE in the beacon, following a call to @ref mm_tim_update.
 * Message @ref MM_TIM_UPDATE_CFM is send to requester to confirm update.
 *
 * @param[in] param Traffic indication to update
 ****************************************************************************************
 */
static void mm_tim_update_proceed(struct mm_tim_update_req const *param)
{
    struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];

    // No beacon transmission ongoing, proceed immediately to the update
    if (param->aid == 0)
    {
        // Update BC/MC status bit
        if (param->tx_avail)
            vif->u.ap.bc_mc_status = VIF_AP_BCMC_BUFFERED;
        else
            vif->u.ap.bc_mc_status = 0;
    }
    else
    {
        do
        {
            struct tx_pbd *pbd_tim = &txl_tim_desc[param->inst_nbr][0];
            struct tx_pbd *pbd_bmp = &txl_tim_desc[param->inst_nbr][1];
            uint32_t tim_ie = CPU2HW(&txl_tim_ie_pool[param->inst_nbr][0]);
            uint32_t tim_bmp = CPU2HW(&txl_tim_bitmap_pool[param->inst_nbr][0]);

            // Compute the byte and bit numbers for this AID
            uint8_t n = param->aid / 8;
            uint8_t mask = CO_BIT(param->aid % 8);
            uint8_t val = co_read8p(tim_bmp + n);

            // Check if we have to set or reset a bit
            if (param->tx_avail)
            {
                // Check if the bit is already set
                if (val & mask)
                    break;

                TRACE_AP(PS, "{VIF-%d} Traffic buffered for STA aid=%d",
                         param->inst_nbr, param->aid);

                // Set the bit in the bitmap
                co_write8p(tim_bmp + n, val | mask);

                // One more bit is set
                vif->u.ap.tim_bitmap_set++;

                // Check if Lower limit of the partial virtual bitmap needs to be updated
                if (n < vif->u.ap.tim_n1)
                {
                    // Lower limit shall be an even number
                    vif->u.ap.tim_n1 = n & 0xFE;

                    // Update the start pointer in the TBD
                    pbd_bmp->datastartptr = tim_bmp + vif->u.ap.tim_n1;
                }

                // Check if Upper limit of the partial virtual bitmap needs to be updated
                if (n > vif->u.ap.tim_n2)
                {
                    // Update Upper limit
                    vif->u.ap.tim_n2 = n;

                    // Update the end pointer in the TBD
                    pbd_bmp->dataendptr = tim_bmp + vif->u.ap.tim_n2;
                }

                // Update the TIM length
                vif->u.ap.tim_len = vif->u.ap.tim_n2 - vif->u.ap.tim_n1 + 6;

                // Update the TIM PBD
                co_write8p(tim_ie + MAC_TIM_LEN_OFT, vif->u.ap.tim_len - 2);
                co_write8p(tim_ie + MAC_TIM_BMPC_OFT, vif->u.ap.tim_n1);
                pbd_tim->dataendptr = tim_ie + MAC_TIM_BMPC_OFT;
                pbd_tim->next = CPU2HW(pbd_bmp);
            }
            else
            {
                // Check if the bit is already reset
                if (!(val & mask))
                    break;

                TRACE_AP(PS, "{VIF-%d} No more traffic buffered for STA aid=%d",
                         param->inst_nbr, param->aid);

                // Reset the bit in the bitmap
                co_write8p(tim_bmp + n, val & ~mask);

                // One more bit is reset
                vif->u.ap.tim_bitmap_set--;

                // Check if the bitmap has still some bits set
                if (vif->u.ap.tim_bitmap_set)
                {
                    // Check if we need to update the Lower Limit
                    if ((n & 0xFE) == vif->u.ap.tim_n1)
                    {
                        // Search for the new Lower Limit
                        while ((vif->u.ap.tim_n1 != MAC_TIM_SIZE) &&
                               (co_read8p(tim_bmp + vif->u.ap.tim_n1) == 0))
                        {
                            vif->u.ap.tim_n1++;
                        }

                        // Lower Limit shall be an even number
                        vif->u.ap.tim_n1 &= 0xFE;

                        // Update the start pointer in the TBD
                        pbd_bmp->datastartptr = tim_bmp + vif->u.ap.tim_n1;
                    }

                    // Check if we need to update the Upper Limit
                    if (n == vif->u.ap.tim_n2)
                    {
                        // Search for the new Upper Limit
                        while ((vif->u.ap.tim_n2 != 0) &&
                               (co_read8p(tim_bmp + vif->u.ap.tim_n2) == 0))
                        {
                            vif->u.ap.tim_n2--;
                        }

                        // Update the end pointer in the TBD
                        pbd_bmp->dataendptr = tim_bmp + vif->u.ap.tim_n2;
                    }

                    // Update the TIM length
                    vif->u.ap.tim_len = vif->u.ap.tim_n2 - vif->u.ap.tim_n1 + 6;

                    // Update the TIM PBD
                    co_write8p(tim_ie + MAC_TIM_LEN_OFT, vif->u.ap.tim_len - 2);
                    co_write8p(tim_ie + MAC_TIM_BMPC_OFT, vif->u.ap.tim_n1);
                }
                else
                {
                    // Update the TIM
                    vif->u.ap.tim_len = MAC_TIM_BMP_OFT + 1;
                    vif->u.ap.tim_n1 = -1;
                    vif->u.ap.tim_n2 = 0;
                    co_write8p(tim_ie + MAC_TIM_LEN_OFT, vif->u.ap.tim_len - 2);
                    co_write8p(tim_ie + MAC_TIM_BMPC_OFT, 0);
                    // Update the TIM PBD
                    pbd_tim->dataendptr = tim_ie + MAC_TIM_BMP_OFT;
                    pbd_tim->next = CPU2HW(&txl_bcn_end_desc[param->inst_nbr]);
                    // Update the end pointer in the TBD
                    pbd_bmp->dataendptr = tim_bmp + vif->u.ap.tim_n2;
                }
            }
        } while(0);
    }

    // Confirm the TIM update to the host
    ke_msg_send_basic(MM_TIM_UPDATE_CFM, ke_param2msg(param)->src_id, TASK_MM);

    // Free the TIM parameters
    ke_msg_free(ke_param2msg(param));
}

/**
 ****************************************************************************************
 * @brief Update tx descriptor for the beacon
 *
 * Updates TX descriptor for beacon transmission, following a beacon update.
 * (@ref mm_bcn_change)
 *
 * @param[in] vif       VIF entry for which the beacon must be updated
 * @param[in] param     Parameters of the new beacon
 ****************************************************************************************
 */
static void mm_bcn_desc_prep(struct vif_info_tag *vif,
                             struct mm_bcn_change_req const *param)
{
    struct txl_frame_desc_tag *frame = &vif->u.ap.bcn_desc;
    struct tx_hd *thd = &frame->txdesc.lmac.hw_desc->thd;
    struct tx_pbd *pbd_bcn = &txl_bcn_end_desc[vif->index];
    uint32_t tim_bcn = thd->datastartptr + param->tim_oft;
    uint32_t tim_ie = CPU2HW(&txl_tim_ie_pool[vif->index][0]);
    struct tx_policy_tbl *pol;
    uint8_t band;
    uint32_t bcn_len = param->bcn_len - param->tim_len;

    // Store the current beacon length for later update of the THD
    vif->u.ap.bcn_len = bcn_len;

    #if (NX_UMAC_PRESENT)
    // Verify the protection and bandwidth status
    me_beacon_check(vif->index, param->bcn_len, thd->datastartptr);
    #endif

    // Fill-up the TX header descriptor
    thd->dataendptr = thd->datastartptr + param->tim_oft - 1;
    pbd_bcn->datastartptr = thd->dataendptr + param->tim_len + 1;
    pbd_bcn->dataendptr = pbd_bcn->datastartptr + bcn_len - param->tim_oft - 1;
    pbd_bcn->bufctrlinfo = 0;

    // Get band
    band = vif_mgmt_get_band(vif);

    #if (NX_P2P_GO)
    // If the interface is a P2P interface we cannot use default 2.4GhHz policy table (11b rate forbidden)
    if (vif->p2p)
    {
        #if (NX_UMAC_PRESENT)
        pol = &txl_buffer_control_5G.policy_tbl;
        #else
        pol = &txl_frame_pol_5G;
        #endif //(NX_UMAC_PRESENT)
    }
    else
    #endif //(NX_P2P_GO)
    {
        #if (NX_UMAC_PRESENT)
        pol = (band == PHY_BAND_2G4) ? &txl_buffer_control_24G.policy_tbl : &txl_buffer_control_5G.policy_tbl;
        #else
        pol = (band == PHY_BAND_2G4) ? &txl_frame_pol_24G : &txl_frame_pol_5G;
        #endif //(NX_UMAC_PRESENT)
    }

    // Set TX power
    pol->powercntrlinfo[0] = TX_PWR_LEVEL_SET(nxmac_ofdm_max_pwr_level_getf());
    thd->policyentryaddr = CPU2HW(pol);
    thd->phyctrlinfo = 0;
    thd->macctrlinfo2 = 0;
    thd->first_pbd_ptr = CPU2HW(&txl_tim_desc[vif->index][0]);

    // The beacon is now configured for this VIF
    vif->u.ap.bcn_configured = true;
    co_write8p(tim_ie + MAC_TIM_PERIOD_OFT, co_read8p(tim_bcn + MAC_TIM_PERIOD_OFT));
}

/**
 ****************************************************************************************
 * @brief Update CSA counter
 *
 * Initializes CSA counter, following a beacon update (@ref mm_bcn_change).
 *
 * @param[in] vif       VIF entry for which the beacon must be updated
 * @param[in] param     Parameters of the new beacon
 ****************************************************************************************
 */
static void mm_bcn_csa_init(struct vif_info_tag *vif,
                            struct mm_bcn_change_req const *param)
{
    struct txl_frame_desc_tag *frame = &vif->u.ap.bcn_desc;
    struct tx_hd *thd = &frame->txdesc.lmac.hw_desc->thd;
    uint8_t i;

    // reset value
    vif->u.ap.csa_count = 0;
    for (i = 0; i < BCN_MAX_CSA_CPT; i++)
    {
        vif->u.ap.csa_oft[i] = param->csa_oft[i];
    }

    if (param->csa_oft[0] > 0)
    {
        vif->u.ap.csa_count = co_read8p(thd->datastartptr + param->csa_oft[0]);
        vif->u.ap.csa_count++;
    }
}

/**
 ****************************************************************************************
 * @brief Send CSA counter value to host.
 *
 * Sends new value of CSA counter to host after beacon transmission. This allow the host
 * to keep CSA counter up to date if needed in other mgmt frame (e.g. probe response)
 *
 * @param[in] vif_index Index of the VIF entry
 * @param[in] csa_count CSA counter
 ****************************************************************************************
 */
static void mm_bcn_send_csa_counter_ind(uint8_t vif_index, uint8_t csa_count)
{
    // Allocate the indication structure
    struct mm_csa_counter_ind *ind = KE_MSG_ALLOC(MM_CSA_COUNTER_IND, TASK_API,
                                                  TASK_MM, mm_csa_counter_ind);

    ind->vif_index = vif_index;
    ind->csa_count = csa_count;

    // Send the message
    ke_msg_send(ind);
}

/**
 ****************************************************************************************
 * @brief Update beacon field before transmission.
 *
 * This function updates "dynamic" fields of the beacon (like TIM, CSA counter) before
 * transmission
 *
 * @param[in] vif       VIF entry thal will send the beacon
 *
 * @return value of the group traffic bit in the TIM IE (i.e. bit0 of Bitmap Control)
 ****************************************************************************************
 */
static uint8_t mm_bcn_build(struct vif_info_tag *vif)
{
    struct txl_frame_desc_tag *frame = &vif->u.ap.bcn_desc;
    struct tx_hd *thd = &frame->txdesc.lmac.hw_desc->thd;
    uint32_t tim_ie = CPU2HW(&txl_tim_ie_pool[vif->index][0]);
    uint8_t bmpc = co_read8p(tim_ie + MAC_TIM_BMPC_OFT);

    // Update the THD length
    thd->frmlen = vif->u.ap.bcn_len + vif->u.ap.tim_len + MAC_FCS_LEN;

    #if (NX_P2P_GO)
    if (vif->p2p)
    {
        // Length of NOA Information Element
        uint8_t noa_len = p2p_go_bcn_get_noa_len(vif->p2p_index);

        if (noa_len)
        {
            // Get TX payload buffer descriptor
            struct tx_pbd *noa_pbd = &txl_p2p_noa_desc[vif->index];

            // Add length of P2P NOA Information Element if present
            thd->frmlen += noa_len;

            // Update NOA Payload descriptor Data End pointer
            noa_pbd->dataendptr = noa_pbd->datastartptr + noa_len - 1;
        }
    }
    #endif //(NX_P2P_GO)

    #if (RW_UMESH_EN)
    if (vif->type == VIF_MESH_POINT)
    {
        uint8_t mesh_ies_len = mesh_get_vendor_ies_len(vif->mvif_idx);

        // Update dynamic fields (Number of Peerings, ...)
        mesh_update_beacon(vif);

        // Check if IEs have to be added at the end of the beacon
        if (mesh_ies_len)
        {
            // Get TX payload buffer descriptor
            struct tx_pbd *add_ies_pbd = &txl_mesh_add_ies_desc[vif->mvif_idx];

            // Update frame length
            thd->frmlen += mesh_ies_len;

            // Update NOA Payload descriptor Data End pointer
            add_ies_pbd->dataendptr = add_ies_pbd->datastartptr + mesh_ies_len - 1;

            // Add the additional element at beacon end
            txl_bcn_end_desc[vif->index].next = CPU2HW(add_ies_pbd);
        }
        else
        {
            txl_bcn_end_desc[vif->index].next = CPU2HW(NULL);
        }
    }
    #endif //(RW_UMESH_EN)

    // Update the sequence number in the beacon
    co_write16(HW2CPU(thd->datastartptr + MAC_HEAD_CTRL_OFT), txl_get_seq_ctrl());
    // Update the DTIM count and period
    co_write8p(tim_ie + MAC_TIM_CNT_OFT, vif->u.ap.dtim_count);

    // Update DTIM count for next beacon
    if (vif->u.ap.dtim_count == 0)
    {
        if (vif->u.ap.bc_mc_status || macif_tx_q_has_data(AC_BCN))
            bmpc |= MAC_TIM_BCMC_PRESENT;
        else
            bmpc &= ~MAC_TIM_BCMC_PRESENT;
        vif->u.ap.dtim_count = co_read8p(tim_ie + MAC_TIM_PERIOD_OFT);
    }
    else if (vif->u.ap.bc_mc_status & VIF_AP_BCMC_MOREDATA)
    {
        // TIM IE may have been updated since previous beacon, so need to reset
        bmpc |= MAC_TIM_BCMC_PRESENT;
    }
    else
    {
         bmpc &= ~MAC_TIM_BCMC_PRESENT;
    }
    co_write8p(tim_ie + MAC_TIM_BMPC_OFT, bmpc);
    vif->u.ap.dtim_count--;

    // Update CSA counter in the beacon
    if (vif->u.ap.csa_count)
    {
        uint8_t i;
        vif->u.ap.csa_count --;
        for (i = 0; i < BCN_MAX_CSA_CPT; i++)
        {
            if (vif->u.ap.csa_oft[i] == 0)
                break;
            co_write8p((uint32_t)HW2CPU(thd->datastartptr) + vif->u.ap.csa_oft[i],
                       vif->u.ap.csa_count);
        }
        if (vif->u.ap.csa_count)
        {
            mm_bcn_send_csa_counter_ind(vif->index, vif->u.ap.csa_count);
        }
        #if NX_UMAC_PRESENT
        // keep csa_count to 1 until beacon is successfully transmistted
        if (vif->u.ap.csa_count == 0)
            vif->u.ap.csa_count = 1;
        #endif
    }

    // update Tx power in policy table
    tpc_update_frame_tx_power(vif, frame);

    return(bmpc & MAC_TIM_BCMC_PRESENT);
}

/**
 ****************************************************************************************
 * @brief Callback associated to DMA transfer of beacon from host memory
 *
 * This function will re-initialize beacon infomration once a new beacon has been
 * downloaded from host memory.
 *
 * @param[in] env       VIF entry
 * @param[in] dma_queue DMA queue used for transfer
 ***************************************************************************************
 */
static void mm_bcn_updated(void *env, int dma_queue)
{
    struct vif_info_tag *vif = (struct vif_info_tag *) env;
    struct ke_msg *msg = ke_param2msg(mm_bcn_env.param);

    // Confirm the beacon update to the host
    ke_msg_send_basic(MM_BCN_CHANGE_CFM, msg->src_id, TASK_MM);

    // Update the frame descriptor
    mm_bcn_desc_prep(vif, mm_bcn_env.param);

    // Init CSA counter
    mm_bcn_csa_init(vif, mm_bcn_env.param);

    // Update status
    mm_bcn_env.update_ongoing = false;
    // Check if the beacons need to be sent immediately
    if (mm_bcn_env.tx_pending)
        mm_bcn_transmit();

    // No more beacon data in memory
    mm_bcn_env.param = NULL;

    // Free the beacon parameters
    ke_msg_free(msg);
}

/**
 ****************************************************************************************
 * @brief Initiates download of Beacon frame.
 *
 * Following a call to @ref mm_bcn_change, this fucntion configures the DMA transfer to
 * download beacon from host memory. @ref mm_bcn_updated is called once the transfer is
 * complete.
 *
 * @param[in] param Parameters of the new beacon
 ****************************************************************************************
 */
static void mm_bcn_update(struct mm_bcn_change_req const *param)
{
    struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];

    // Update status
    mm_bcn_env.update_pending = false;
    mm_bcn_env.update_ongoing = true;

    #if (NX_UMAC_PRESENT && RW_MESH_EN)
    // If the beacon is provided for a Mesh Point VIF, it does not come from the host
    // memory. It has been filled by UMAC's Mesh Module and hence does not require
    // any DMA transfer.
    if (vif->type == VIF_MESH_POINT)
    {
        mm_bcn_updated(vif, DMA_DL);
    }
    else
    #endif //(NX_UMAC_PRESENT && RW_MESH_EN)
    {
        struct txl_buffer_tag *buffer = (struct txl_buffer_tag *)&txl_bcn_pool[param->inst_nbr][0];
        #if ! NX_FULLY_HOSTED
        struct hal_dma_desc_tag *dma = &mm_bcn_env.dma;

        // Fill in the DMA descriptor
        dma->env = vif;
        dma->dma_desc->src = param->bcn_ptr;
        dma->dma_desc->dest = CPU2HW(buffer->payload);
        dma->dma_desc->length = param->bcn_len;

        // Program the download
        hal_dma_push(dma, DMA_DL);
        #else

        // Cannot use DMA for CPU mem to Shared memory transfer
        memcpy(buffer->payload, (void *)param->bcn_ptr, param->bcn_len);
        mm_bcn_updated(vif, DMA_DL);
        #endif // NX_FULLY_HOSTED
    }
}

/**
 ****************************************************************************************
 * @brief Callback associated to transmission of beacon.
 *
 * Once beacon has been transmitted, this function will take care to:
 * - process pending updates (like tim or complete beacon update)
 * - and all post beacon actions (like channel switch if CSA is over)
 *
 * @param[in] env    Vif entry that just send the beacon
 * @param[in] status Status of the transmission
 ****************************************************************************************
 */
static void mm_bcn_transmitted(void *env, uint32_t status)
{
    #if (NX_P2P_GO) || (NX_UMAC_PRESENT)
    // Attached env is the VIF Entry on which beacon has been sent, get VIF index
    struct vif_info_tag *vif = (struct vif_info_tag *)env;
    #endif //(NX_P2P_GO) || (NX_UMAC_PRESENT)

    // Sanity check - We shall be still waiting for at least one confirmation
    ASSERT_ERR(mm_bcn_env.tx_cfm);

    // One confirmation less to be expected
    mm_bcn_env.tx_cfm--;

    // Check if all confirmations have been received
    if (mm_bcn_env.tx_cfm == 0)
    {
        // The beacon(s) were transmitted, ask the MM to allow the IDLE state again
        mm_no_idle_stop();

        // Check if we need to update a beacon
        if (mm_bcn_env.update_pending)
            mm_bcn_update(mm_bcn_env.param);

        // Check if we need to update the TIM
        while (!co_list_is_empty(&mm_bcn_env.tim_list))
        {
            mm_tim_update_proceed(ke_msg2param((struct ke_msg *)
                                                co_list_pop_front(&mm_bcn_env.tim_list)));
        }

        #if (NX_P2P_GO)
        // Check if we have a pending NOA update
        if (mm_bcn_env.p2p_noa_req[vif->index] != P2P_BCN_UPD_OP_NONE)
        {
            mm_bcn_update_p2p_noa(vif->index, mm_bcn_env.p2p_noa_req[vif->index]);
        }
        #endif //(NX_P2P_GO)

        #if NX_UMAC_PRESENT
        if (vif->u.ap.csa_count == 1)
        {
            vif_mgmt_switch_channel(vif);
        }
        #endif // NX_UMAC_PRESENT

        #if (RW_UMESH_EN)
        mesh_ps_beacon_cfm_handle(vif);
        #endif //(RW_UMESH_EN)
    }

}

#if (NX_P2P_GO)
/**
 ****************************************************************************************
 * @brief Initializes Buffer payload descriptor for NOA IE, so that it could be added in
 *        beacon if needed.
 *
 * @param[in] vif_index VIF index
 ****************************************************************************************
 */
static void mm_bcn_init_p2p_noa(uint8_t vif_index)
{
    // Get TX payload buffer descriptor
    struct tx_pbd *pbd = &txl_p2p_noa_desc[vif_index];
    // Get payload address
    uint32_t p2p_noa_ie = CPU2HW(&txl_p2p_noa_ie_pool[vif_index][0]);

    // Prepare the descriptor
    pbd->upatterntx   = TX_PAYLOAD_DESC_PATTERN;
    pbd->datastartptr = p2p_noa_ie;
    pbd->next         = (uint32_t)NULL;
    pbd->bufctrlinfo  = 0;
}
#endif //(NX_P2P_GO)

#if (RW_UMESH_EN)
/**
 ****************************************************************************************
 * @brief Initializes Buffer payload descriptor for MESH IE, so that it could be added in
 *        beacon if needed.
 ****************************************************************************************
 */
static void mm_bcn_init_mesh_add_ies_desc(void)
{
    for (int i = 0; i < RW_MESH_VIF_NB; i++)
    {
        // Get TX payload buffer descriptor
        struct tx_pbd *pbd = &txl_mesh_add_ies_desc[i];
        // Get payload address
        uint32_t mesh_add_ie = CPU2HW(&txl_mesh_add_ies[i].buf[0]);

        // Prepare the descriptor
        pbd->upatterntx   = TX_PAYLOAD_DESC_PATTERN;
        pbd->datastartptr = mesh_add_ie;
        pbd->next         = (uint32_t)NULL;
        pbd->bufctrlinfo  = 0;
    }
}
#endif //(RW_UMESH_EN)

/**
 ****************************************************************************************
 * @brief Initializes buffer of TIM IE that will be included in beacon (and related
 * variables in the vif structure)
 *
 * @param[in] vif       VIF entry
 ****************************************************************************************
 */
static void mm_bcn_init_tim(struct vif_info_tag *vif)
{
    uint8_t inst_nbr = vif->index;
    struct tx_pbd *pbd = &txl_tim_desc[inst_nbr][0];
    uint32_t tim_ie = CPU2HW(&txl_tim_ie_pool[inst_nbr][0]);
    uint32_t tim_bmp = CPU2HW(&txl_tim_bitmap_pool[inst_nbr][0]);

    // Initialize the DTIM count
    vif->u.ap.dtim_count = 0;
    vif->u.ap.tim_len = MAC_TIM_BMP_OFT + 1;
    vif->u.ap.tim_bitmap_set = 0;
    vif->u.ap.tim_n1 = -1;
    vif->u.ap.tim_n2 = 0;
    vif->u.ap.bc_mc_status = 0;

    // First part of the TIM
    pbd->upatterntx = TX_PAYLOAD_DESC_PATTERN;
    pbd->datastartptr = tim_ie + MAC_TIM_ID_OFT;
    pbd->dataendptr = tim_ie + MAC_TIM_BMP_OFT;
    pbd->next = CPU2HW(&txl_bcn_end_desc[inst_nbr]);
    pbd->bufctrlinfo = 0;
    co_write8p(tim_ie + MAC_TIM_ID_OFT, MAC_ELTID_TIM);
    co_write8p(tim_ie + MAC_TIM_LEN_OFT, 4);
    co_write8p(tim_ie + MAC_TIM_CNT_OFT, vif->u.ap.dtim_count);
    co_write8p(tim_ie + MAC_TIM_PERIOD_OFT, 1); // Will be updated when receiving the beacon from host
    co_write8p(tim_ie + MAC_TIM_BMPC_OFT, 0);
    co_write8p(tim_ie + MAC_TIM_BMP_OFT, 0);

    // Reset the TIM virtual bitmap
    pbd = &txl_tim_desc[inst_nbr][1];
    pbd->upatterntx = TX_PAYLOAD_DESC_PATTERN;
    pbd->dataendptr = tim_bmp + vif->u.ap.tim_n2;
    pbd->next = CPU2HW(&txl_bcn_end_desc[inst_nbr]);
    memset(txl_tim_bitmap_pool[inst_nbr], 0, sizeof(txl_tim_bitmap_pool[inst_nbr]));

    // Initialize post-TIM TX PBD
    pbd = &txl_bcn_end_desc[inst_nbr];
    pbd->upatterntx = TX_PAYLOAD_DESC_PATTERN;
    pbd->next = 0;
    pbd->bufctrlinfo = 0;
}

/*
 * PUBLIC FUNCTION DEFINITIONS
 ******************************************************************************
 */
void mm_bcn_init(void)
{
    #if (NX_P2P_GO)
    uint8_t counter;
    #endif //(NX_P2P_GO)

    // Reset the complete environment
    memset(&mm_bcn_env, 0, sizeof(mm_bcn_env));

    mm_bcn_env.dma.dma_desc = &bcn_dwnld_desc;
    mm_bcn_env.dma.cb = mm_bcn_updated;
    co_list_init(&mm_bcn_env.tim_list);

    #if (NX_P2P_GO)
    for (counter = 0; counter < NX_VIRT_DEV_MAX; counter++)
    {
        // Initialize NOA Beacon payload
        p2p_go_bcn_init_noa_pyld(CPU2HW(&txl_p2p_noa_ie_pool[counter][0]));
    }
    #endif //(NX_P2P_GO)

    #if (RW_UMESH_EN)
    mm_bcn_init_mesh_add_ies_desc();
    #endif //(RW_UMESH_EN)
}

void mm_bcn_init_vif(struct vif_info_tag *vif)
{
    struct txl_frame_desc_tag *frame = &vif->u.ap.bcn_desc;
    struct txl_buffer_tag *buffer = (struct txl_buffer_tag *)&txl_bcn_pool[vif->index][0];
    struct tx_hw_desc *hwdesc = &txl_bcn_hwdesc_pool[vif->index];
    #if NX_UMAC_PRESENT
    struct txl_buffer_control *bufctrl = &txl_bcn_buf_ctrl[vif->index];
    #else
    struct txl_buffer_control *bufctrl = &buffer->buffer_control;
    #endif
    struct tx_hd *thd;

    // Initialize the frame descriptor
    txl_frame_init_desc(frame, buffer, hwdesc, bufctrl);

    // Initialize the TIM buffer
    mm_bcn_init_tim(vif);

    #if (NX_P2P_GO)
    if (vif->p2p)
    {
        mm_bcn_init_p2p_noa(vif->index);
    }
    #endif //(NX_P2P_GO)

    // Initialize some static fields of the THD
    thd = &frame->txdesc.lmac.hw_desc->thd;
    thd->phyctrlinfo = 0;
    thd->macctrlinfo2 = 0;
    thd->first_pbd_ptr = 0;

    // Initialize callback function
    frame->cfm.cfm_func = mm_bcn_transmitted;
    frame->cfm.env = vif;
}

void mm_bcn_change(struct mm_bcn_change_req const *param)
{
    // Save the message for later handling
    mm_bcn_env.param = param;

    // Check if the beacon transmission is already ongoing
    if (mm_bcn_env.tx_cfm)
    {
        mm_bcn_env.update_pending = true;
    }
    else
    {
        // No beacon transmission ongoing, proceed immediately to the update
        mm_bcn_update(param);
    }
}

void mm_tim_update(struct mm_tim_update_req const *param)
{
    // Check if the beacon transmission is already ongoing
    if (mm_bcn_env.tx_cfm)
    {
        // Push the request into the list for later handling
        co_list_push_back(&mm_bcn_env.tim_list, &ke_param2msg(param)->hdr);
    }
    else
    {
        // Proceed immediately to the request
        mm_tim_update_proceed(param);
    }
}

void mm_bcn_transmit(void)
{
    struct vif_info_tag *vif = vif_mgmt_first_used();
    uint32_t next_tbtt;
    bool first_vif = true;
    bool tbtt_missed = false;
    uint8_t bc_mc = 0;

    // Sanity check - All previously chained beacons shall be transmitted
    ASSERT_ERR(!mm_bcn_env.tx_cfm);

    // Check if a beacon update is ongoing
    if (mm_bcn_env.update_ongoing)
    {
        mm_bcn_env.tx_pending = true;
        return;
    }

    // No more beacon transmission pending
    mm_bcn_env.tx_pending = false;

    next_tbtt = nxmac_next_tbtt_get() << 5;

    // Go through the VIFs and send the beacons of the AP ones
    while (vif != NULL)
    {
        // Check if VIF is an AP one
        if (((vif->type == VIF_AP)
            #if (RW_MESH_EN)
                || (vif->type == VIF_MESH_POINT)
            #endif //(RW_MESH_EN)
                ) && vif->u.ap.bcn_configured && vif->active
            #if (NX_UMAC_PRESENT)
                  && (vif->u.ap.bcn_tbtt_cnt == vif->u.ap.bcn_tbtt_ratio)
            #endif //(NX_UMAC_PRESENT)
                  )
        {
            // Update the frame
            if (mm_bcn_build(vif))
            {
                bc_mc++;
            }

            if (first_vif &&
                (next_tbtt > (HAL_MACHW_BCN_TX_DELAY_US + MM_PRE_AP_TBTT_DELAY_US) ||
                 next_tbtt < 50))
            {
                #if NX_CHNL_CTXT
                if (chan_is_on_operational_channel(vif))
                #endif
                {
                    TRACE_CHAN(ERR, "{VIF-%d} Don't push beacon as TBTT is already passed",
                               vif->index);
                }
                tbtt_missed = true;
            }
            first_vif = false;

            if (tbtt_missed)
            {
                bc_mc = 0;
                vif = vif_mgmt_next(vif);
                continue;
            }

            #if NX_CHNL_CTXT
            if (chan_is_on_operational_channel(vif))
            #endif
            {
                if (bc_mc)
                {
                    #if NX_UMAC_PRESENT
                    mm_traffic_req_ind(VIF_TO_BCMC_IDX(vif->index), 0, false);
                    #else
                    vif->u.ap.bc_mc_nb = macif_tx_q_len(AC_BCN, vif->index);
                    #endif
                }

                #if NX_CHNL_CTXT
                // Set VIF and STA indexes
                vif->u.ap.bcn_desc.txdesc.host.vif_idx = vif->index;
                vif->u.ap.bcn_desc.txdesc.host.staid   = 0xFF;

                // Push the TX frame descriptor
                if (txl_frame_push(&vif->u.ap.bcn_desc, AC_BCN))
                {
                    mm_bcn_env.tx_cfm++;
                }
                #else
                txl_frame_push(&vif->u.ap.bcn_desc, AC_BCN);
                mm_bcn_env.tx_cfm++;
                #endif //(NX_CHNL_CTXT)

                if (mm_bcn_env.tx_cfm == 1)
                {
                    // We have programmed one beacon for transmission, so ask the MM to
                    // disallow the IDLE state
                    mm_no_idle_start();
                }

                #if NX_UMAC_PRESENT
                if (bc_mc) {
                    struct sta_info_tag *sta_bcmc = &sta_info_tab[VIF_TO_BCMC_IDX(vif->index)];
                    if (sta_bcmc->traffic_avail & PS_TRAFFIC_INT)
                    {
                        sta_bcmc->ps_service_period = (PS_SERVICE_PERIOD | BCN_SERVICE_PERIOD);
                        sta_mgmt_send_postponed_frame(vif, sta_bcmc, 0);
                        sta_bcmc->ps_service_period = NO_SERVICE_PERIOD;
                    }
                    bc_mc = 0;
                }
                #endif // NX_UMAC_PRESENT
            }
            #if NX_CHNL_CTXT
            else
            {
                bc_mc = 0;
            }
            #endif
        }

        // Go to next VIF
        vif = vif_mgmt_next(vif);
    }

    #if !NX_UMAC_PRESENT
    // Once all beacons have been programmed, go on with the BC/MC traffic
    if (bc_mc)
        macif_tx_evt(AC_BCN);
    #endif
}

#if (NX_P2P_GO)
void mm_bcn_update_p2p_noa(uint8_t vif_index, uint8_t operation)
{
    // Check if the beacon transmission is already ongoing
    if (mm_bcn_env.tx_cfm)
    {
        // Save the message for later handling
        mm_bcn_env.p2p_noa_req[vif_index] = operation;
    }
    else
    {
        // Get VIF Info entry
        struct vif_info_tag *vif = &vif_info_tab[vif_index];

        // React accordingly with the operation code
        switch (operation)
        {
            case (P2P_BCN_UPD_OP_NOA_ADD):
            {
                txl_bcn_end_desc[vif_index].next = CPU2HW(&txl_p2p_noa_desc[vif_index]);
            } break;

            case (P2P_BCN_UPD_OP_NOA_RMV):
            {
                txl_bcn_end_desc[vif_index].next = (uint32_t)NULL;
            } break;

            case (P2P_BCN_UPD_OP_NOA_UPD):
            {
                // Let the P2P module fulfill the payload part
                p2p_go_bcn_upd_noa_pyld(vif->p2p_index, CPU2HW(&txl_p2p_noa_ie_pool[vif_index][0]));
            } break;

            default:
            {
                ASSERT_ERR(0);
            }
        }

        // No P2P NOA update is pending
        mm_bcn_env.p2p_noa_req[vif_index] = P2P_BCN_UPD_OP_NONE;

        // Inform the P2P module about operation completion
        p2p_go_bcn_op_done(vif->p2p_index, operation);
    }
}
#endif //(NX_P2P_GO)

#if (NX_UMAC_PRESENT && RW_MESH_EN)
struct txl_buffer_tag *mm_bcn_get_buffer(uint8_t vif_index)
{
    return (struct txl_buffer_tag *)&txl_bcn_pool[vif_index][0];
}
#endif //(NX_UMAC_PRESENT && RW_MESH_EN)

#endif //(NX_BCN_AUTONOMOUS_TX)

/// @} end of group
