/**
 ****************************************************************************************
 *
 * @file mm_task.c
 *
 * @brief MAC Management task.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 *****************************************************************************************
 * @addtogroup MM_TASK
 * @{
 *****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "co_endian.h"

#include "mac_frame.h"
#include "mm.h"
#include "ps.h"
#if NX_UMAC_PRESENT
#include "me.h"
#include "rxu_cntrl.h"
#endif
#include "mm_task.h"
#include "mm_bcn.h"
#include "phy.h"

#include "reg_mac_pl.h"
#include "reg_mac_core.h"
#include "hal_machw.h"
#include "dbg.h"

#include "rxl_hwdesc.h"
#include "rxl_cntrl.h"
#include "txl_cntrl.h"

#include "sta_mgmt.h"
#include "vif_mgmt.h"

#include "version.h"

#include "ke_timer.h"

#include "chan.h"
#if (RW_BFMER_EN)
#include "bfr.h"
#endif //(RW_BFMER_EN)

#include "tpc.h"

/*
 * MESSAGE HANDLERS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief MM module @ref MM_VERSION_REQ message handler.
 *
 * Sent by host.\n
 * It reads the FW and HW versions and reports them to the requester in a @ref
 * MM_VERSION_CFM message.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_version_req_handler(ke_msg_id_t const msgid,
                       void const *param,
                       ke_task_id_t const dest_id,
                       ke_task_id_t const src_id)
{
    // Allocate the response message
    struct mm_version_cfm *msg = KE_MSG_ALLOC(MM_VERSION_CFM, src_id, dest_id,
                                              mm_version_cfm);

    // Fill in the parameters
    msg->version_lmac = NX_VERSION;
    msg->version_machw_1 = nxmac_version_1_get();
    msg->version_machw_2 = nxmac_version_2_get();
    msg->max_sta_nb = NX_REMOTE_STA_MAX;
    msg->max_vif_nb = NX_VIRT_DEV_MAX;
    phy_get_version(&msg->version_phy_1, &msg->version_phy_2);

    msg->features = (0
    #if (NX_BEACONING)
            | CO_BIT(MM_FEAT_BCN_BIT)
    #endif //(NX_BEACONING)
    #if (NX_BCN_AUTONOMOUS_TX)
            | CO_BIT(MM_FEAT_AUTOBCN_BIT)
    #endif //(NX_BCN_AUTONOMOUS_TX)
    #if (NX_HW_SCAN)
            | CO_BIT(MM_FEAT_HWSCAN_BIT)
    #endif //(NX_HW_SCAN)
    #if (NX_CONNECTION_MONITOR)
            | CO_BIT(MM_FEAT_CMON_BIT)
    #endif //(NX_CONNECTION_MONITOR)
    #if (NX_MULTI_ROLE)
            | CO_BIT(MM_FEAT_MROLE_BIT)
    #endif //(NX_MULTI_ROLE)
    #if (NX_RADAR_DETECT)
            | CO_BIT(MM_FEAT_RADAR_BIT)
    #endif //(NX_RADAR_DETECT)
    #if (NX_POWERSAVE)
            | CO_BIT(MM_FEAT_PS_BIT)
    #endif //(NX_POWERSAVE)
    #if (NX_UAPSD)
            | CO_BIT(MM_FEAT_UAPSD_BIT)
    #endif //(NX_UAPSD)
    #if (NX_DPSM)
            | CO_BIT(MM_FEAT_DPSM_BIT)
    #endif //(NX_DPSM)
    #if (NX_AMPDU_TX)
            | CO_BIT(MM_FEAT_AMPDU_BIT)
    #endif //(NX_AMPDU_TX)
    #if (NX_AMSDU_TX)
            | CO_BIT(MM_FEAT_AMSDU_BIT)
    #endif //(NX_AMSDU_TX)
    #if (NX_CHNL_CTXT)
            | CO_BIT(MM_FEAT_CHNL_CTXT_BIT)
    #endif //(NX_CHNL_CTXT)
    #if (NX_P2P)
            | CO_BIT(MM_FEAT_P2P_BIT)
    #endif //(NX_P2P)
    #if (NX_P2P_GO)
            | CO_BIT(MM_FEAT_P2P_GO_BIT)
    #endif //(NX_P2P_GO)
    #if (NX_UMAC_PRESENT)
            | CO_BIT(MM_FEAT_UMAC_BIT)
    #endif //(NX_UMAC_PRESENT)
    #if (NX_REORD)
            | CO_BIT(MM_FEAT_REORD_BIT)
    #endif //(NX_REORD)
    #if (NX_MFP)
            | CO_BIT(MM_FEAT_MFP_BIT)
    #endif //(NX_MFP)
    #if (RW_MESH_EN)
            | CO_BIT(MM_FEAT_MESH_BIT)
    #endif //(RW_MESH_EN)
    #if (NX_TDLS)
            | CO_BIT(MM_FEAT_TDLS_BIT)
    #endif //(NX_TDLS)
    #if (NX_ANT_DIV)
            | CO_BIT(MM_FEAT_ANT_DIV_BIT)
    #endif //(NX_ANT_DIV)
        );

    #if (RW_BFMEE_EN)
    if (hal_machw_bfmee_support())
        msg->features |= CO_BIT(MM_FEAT_BFMEE_BIT);
    #endif //(RW_BFMEE_EN)

    #if (RW_BFMER_EN)
    msg->features |= CO_BIT(MM_FEAT_BFMER_BIT);
    #endif //(RW_BFMER_EN)

    #if (RW_MUMIMO_RX_EN)
    if (hal_machw_mu_mimo_rx_support())
        msg->features |= CO_BIT(MM_FEAT_MU_MIMO_RX_BIT);
    #endif //(RW_MUMIMO_RX_EN)

    #if (RW_MUMIMO_TX_EN)
    msg->features |= CO_BIT(MM_FEAT_MU_MIMO_TX_BIT);
    #endif //(RW_MUMIMO_TX_EN)

    #if (RW_WAPI_EN)
    if (nxmac_wapi_getf())
        msg->features |= CO_BIT(MM_FEAT_WAPI_BIT);
    #endif //(RW_WAPI_EN)

    #if (!NX_UMAC_PRESENT) || (NX_VHT)
    if (phy_vht_supported())
        msg->features |= CO_BIT(MM_FEAT_VHT_BIT);
    #endif

    #if (!NX_UMAC_PRESENT) || (NX_HE)
    if (hal_machw_he_support())
        msg->features |= CO_BIT(MM_FEAT_HE_BIT);
    #endif

    #if (NX_UF_EN)
    if (phy_uf_supported())
        msg->features |=  CO_BIT(MM_FEAT_UF_BIT);
    #endif

    #if (NX_MON_DATA)
    msg->features |=  CO_BIT(MM_FEAT_MON_DATA_BIT);
    #endif

    #if (RWNX_MAX_AMSDU_RX > 8192)
    msg->features |=  2 << MM_AMSDU_MAX_SIZE_BIT0;
    #elif (RWNX_MAX_AMSDU_RX > 4096)
    msg->features |=  1 << MM_AMSDU_MAX_SIZE_BIT0;
    #endif

    // Send the response
    ke_msg_send(msg);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_RESET_REQ message handler.
 *
 * Sent by host.\n
 * It
 * - Resets all data structure and variables
 * - Programs MAC HW register for Reset
 * - Initializes all other contexts and modules.
 *
 * After RESET procedure is completed it sends the @ref MM_RESET_CFM message.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_reset_req_handler(ke_msg_id_t const msgid,
                     void const *param,
                     ke_task_id_t const dest_id,
                     ke_task_id_t const src_id)
{
    // For safety do all the reset procedure with interrupts disabled
    GLOBAL_INT_DISABLE();

    // Stop the MAC HW operation
    hal_machw_stop();

    // Stop the PHY operation
    phy_stop();

    // Reset the KE timer module
    ke_timer_reset();

    #if (NX_UMAC_PRESENT)
    // Re-initialize the UMAC
    me_init();
    #endif //(NX_UMAC_PRESENT)

    // Re-Initialize the SW
    mm_init();

    #if NX_SYS_STAT
    // Reset the system statistic logging
    dbg_sys_stat_reset();
    #endif

    // Reenable interrupts
    GLOBAL_INT_RESTORE();

    // Send the confirmation
    ke_msg_send_basic(MM_RESET_CFM, src_id, dest_id);

    // Go to the IDLE state
    ke_state_set(TASK_MM, MM_IDLE);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_START_REQ message handler.
 *
 * Sent by host.\n
 * It starts the PHY and MACHW.
 *
 * Message @ref MM_START_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_start_req_handler(ke_msg_id_t const msgid,
                     struct mm_start_req const *param,
                     ke_task_id_t const dest_id,
                     ke_task_id_t const src_id)
{
    struct mac_chan_op chan;
    // Sanity check: This message can be handled in IDLE state only
    ASSERT_ERR(ke_state_get(dest_id) == MM_IDLE);

    // Initialize the PHY
    phy_init(&param->phy_cfg);

    // Start the PHY on the default channel
    chan.band = PHY_BAND_2G4;
    chan.type = PHY_CHNL_BW_20;
    chan.prim20_freq = 2484;
    chan.center1_freq = 2484;
    chan.center2_freq = 0;
    chan.flags = 0;
    phy_set_channel(&chan, PHY_PRIM);
    tpc_update_tx_power(MM_DEFAULT_TX_POWER);

    #if NX_UAPSD
    // Set the U-APSD timeout
    ps_env.uapsd_timeout = param->uapsd_timeout * MILLI2MICRO;
    #endif

    #if NX_POWERSAVE
    /// Save local LP clock accuracy (in ppm)
    mm_env.lp_clk_accuracy = param->lp_clk_accuracy;
    #endif

    // Send the confirmation
    ke_msg_send_basic(MM_START_CFM, src_id, dest_id);

    mm_active();
    #if (NX_UMAC_PRESENT)
    // Request to MAC HW to switch to IDLE state
    hal_machw_idle_req();

    // Adjust the MM state accordingly
    ke_state_set(dest_id, MM_GOING_TO_IDLE);
    #endif

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_CHANNEL_REQ message handler.
 *
 * Send by host.\n
 * It configures the channel used at PHY level.\n
 * If Channel context is used (@ref NX_CHNL_CTXT), this function can only be used on
 * secondary channel.
 *
 * Message @ref MM_SET_CHANNEL_CFM is sent when completed.
 *
 * @note: Message @ref MM_SET_CHANNEL_REQ is "delayed" by function @ref mm_hw_config_handler
 * until HW is in IDLE state.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
#if (!NX_CHNL_CTXT)
static int
mm_set_channel_req_handler(ke_msg_id_t const msgid,
                           struct mm_set_channel_req const *param,
                           ke_task_id_t const dest_id,
                           ke_task_id_t const src_id)
{
     struct mm_set_channel_cfm *cfm;
     struct mac_chan_op const *chan = &param->chan;

    // Once in IDLE, handle the packets already in the RX queue to ensure that the
    // channel information indicated to the upper MAC is correct. This has to be done with
    // interrupts disabled, as the normal handling of the packets is done under interrupt
    GLOBAL_INT_DISABLE();
    rxl_mpdu_isr();
    rxl_cntrl_evt(0);
    GLOBAL_INT_RESTORE();

    // Allocate confirmation message
    cfm = KE_MSG_ALLOC(MM_SET_CHANNEL_CFM, src_id, dest_id, mm_set_channel_cfm);

    // For profiling
    PROF_MM_SET_CHANNEL_SET();

    // Now we can move the PHY to the requested channel
    phy_set_channel(chan, param->index);

    if (param->index == PHY_PRIM)
    {
        TRACE_CHAN(INF, "switch to channel freq=%dMHz bw=%dMHz pwr=%ddBm",
                   chan->prim20_freq, (1 <<  chan->type) * 20, chan->tx_power);
        tpc_update_tx_power(chan->tx_power);
        cfm->power = chan->tx_power;
        phy_get_rf_gain_idx(&cfm->power, &cfm->radio_idx);
    }

    // For profiling
    PROF_MM_SET_CHANNEL_CLR();

    // Send the confirmation
    ke_msg_send(cfm);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

#else
static int
mm_set_channel_req_handler(ke_msg_id_t const msgid,
                           struct mm_set_channel_req const *param,
                           ke_task_id_t const dest_id,
                           ke_task_id_t const src_id)
{
    struct mm_set_channel_cfm *cfm;

    // Allocate confirmation message
    cfm = KE_MSG_ALLOC(MM_SET_CHANNEL_CFM, src_id, dest_id, mm_set_channel_cfm);

    // For profiling
    PROF_MM_SET_CHANNEL_SET();

    if (param->index != PHY_PRIM)
        phy_set_channel(&param->chan, param->index);

    // For profiling
    PROF_MM_SET_CHANNEL_CLR();

     // Send the confirmation
    ke_msg_send(cfm);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}
#endif //(!NX_CHNL_CTXT)

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_DTIM_REQ message handler.
 *
 * Sent by host.\n
 * It configures the dtim period in the dedicated HW registers.
 *
 * Message @ref MM_SET_DTIM_CFM is sent when completed.
 *
 * @note: Message @ref MM_SET_DTIM_REQ is "delayed" by function @ref mm_hw_config_handler
 * until HW is in IDLE state.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_dtim_req_handler(ke_msg_id_t const msgid,
                        struct mm_set_dtim_req const *param,
                        ke_task_id_t const dest_id,
                        ke_task_id_t const src_id)
{
    // Set the DTIM period in the HW
    nxmac_dtim_period_setf(param->dtim_period);
    nxmac_dtim_updated_by_sw_setf(1);

    // Send the confirmation
    ke_msg_send_basic(MM_SET_DTIM_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_BEACON_INT_REQ message handler.
 *
 * Sent by UMAC.\n
 * It configures the beacon period in the dedicated HW registers.
 *
 * Message @ref MM_SET_BEACON_INT_CFM is sent when completed.
 *
 * @note: Message @ref MM_SET_BEACON_INT_REQ is "delayed" by function @ref mm_hw_config_handler
 * until HW is in IDLE state.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_beacon_int_req_handler(ke_msg_id_t const msgid,
                              struct mm_set_beacon_int_req const *param,
                              ke_task_id_t const dest_id,
                              ke_task_id_t const src_id)
{
    struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];

    // Check if VIF is a AP or STA one
    if (vif->type == VIF_STA)
    {
        #if (NX_MULTI_ROLE)
        #if (!NX_UMAC_PRESENT)
        if (vif->u.sta.ap_id == INVALID_STA_IDX)
        {
            vif->u.sta.ap_bcn_intv = param->beacon_int;
        }
        else
        #endif //(!NX_UMAC_PRESENT)
        {
            // Get peer AP information
            struct sta_info_tag *sta = &sta_info_tab[vif->u.sta.ap_id];

            // For STA, the beacon interval is only used by the FW
            sta->bcn_int = param->beacon_int * TU_DURATION;
        }
        #else //(NX_MULTI_ROLE)
        nxmac_beacon_int_setf(param->beacon_int * TU_DURATION);
        #endif //(NX_MULTI_ROLE)
    }
    else
    {
        // For AP or MP, set the beacon interval in the HW
        vif_mgmt_set_ap_bcn_int(vif, param->beacon_int);
    }

    // Send the confirmation
    ke_msg_send_basic(MM_SET_BEACON_INT_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_BASIC_RATES_REQ message handler.
 *
 * Sent by UAMC.\n
 * It configures the basic rates allowed in the dedicated HW registers.
 *
 * Message @ref MM_SET_BASIC_RATES_CFM is sent when completed.
 *
 * @note: Message @ref MM_SET_BASIC_RATES_REQ is "delayed" by function @ref mm_hw_config_handler
 * until HW is in IDLE state.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_basic_rates_req_handler(ke_msg_id_t const msgid,
                               struct mm_set_basic_rates_req const *param,
                               ke_task_id_t const dest_id,
                               ke_task_id_t const src_id)
{
    #if NX_CHNL_CTXT
    struct chan_ctxt_tag *ctxt = chan_env.current_ctxt;

    // Get the new ones
    mm_env.basic_rates[param->band] = param->rates;

    // Check if we are currently on the band
    if (ctxt && (ctxt->channel.band == param->band))
        // Set the basic rates in HW
        nxmac_rates_set(mm_env.basic_rates[param->band]);

    #else
    // Set the basic rates in the HW
    nxmac_rates_set(param->rates);
    #endif

    // Send the confirmation
    ke_msg_send_basic(MM_SET_BASIC_RATES_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_FILTER_REQ message handler.
 *
 * Sent by host.\n
 * It configures the RX filter in the dedicated HW registers.
 *
 * Message @ref MM_SET_FILTER_CFM is sent when completed.
 *
 * @note: Message @ref MM_SET_FILTER_REQ is "delayed" by function @ref mm_hw_config_handler
 * until HW is in IDLE state.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_filter_req_handler(ke_msg_id_t const msgid,
                          struct mm_set_filter_req const *param,
                          ke_task_id_t const dest_id,
                          ke_task_id_t const src_id)
{
    // Set the UMAC RX filter
    mm_rx_filter_umac_set(param->filter);

    // Send the confirmation
    ke_msg_send_basic(MM_SET_FILTER_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_BSSID_REQ message handler.
 *
 * Sent by UMAC.\n
 * It configures the BSSID in the dedicated HW registers.
 *
 * Message @ref MM_SET_BSSID_CFM is sent when completed.
 *
 * @note: Message @ref MM_SET_BSSID_REQ is "delayed" by function @ref mm_hw_config_handler
 * until HW is in IDLE state.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_bssid_req_handler(ke_msg_id_t const msgid,
                         struct mm_set_bssid_req const *param,
                         ke_task_id_t const dest_id,
                         ke_task_id_t const src_id)
{
    #if (NX_MULTI_ROLE || NX_TDLS)
    struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];

    // Copy the BSSID in the VIF structure
    memcpy(&vif->bssid, &param->bssid, sizeof(param->bssid));

    #endif /* (NX_MULTI_ROLE || NX_TDLS ) */

    #if (NX_MULTI_ROLE)
    // Check if there is currently only one VIF created
    if (vif_mgmt_used_cnt() == 1)
    {
        // Only one VIF created, so put the BSSID in the HW
        // write lower 4 bytes of BSSID
        nxmac_bss_id_low_setf(param->bssid.array[0] | (((uint32_t)param->bssid.array[1]) << 16));

        // write higher 2 bytes of BSSID
        nxmac_bss_id_high_setf(param->bssid.array[2]);
    }
    #else
    // write lower 4 bytes of BSSID
    nxmac_bss_id_low_setf(param->bssid.array[0] | (((uint32_t)param->bssid.array[1]) << 16));

    // write higher 2 bytes of BSSID
    nxmac_bss_id_high_setf(param->bssid.array[2]);
    #endif /* (NX_MULTI_ROLE) */

    // Send the confirmation
    ke_msg_send_basic(MM_SET_BSSID_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_EDCA_REQ message handler.
 *
 * Sent by UMAC.\n
 * It configures the EDCA parameters in the dedicated HW registers.
 *
 * Message @ref MM_SET_EDCA_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_edca_req_handler(ke_msg_id_t const msgid,
                         struct mm_set_edca_req const *param,
                         ke_task_id_t const dest_id,
                         ke_task_id_t const src_id)
{
    struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];

    // Store the parameters in the VIF
    vif->txq_params[param->hw_queue] = param->ac_param;

    // If VIF is active, change the parameters immediately
    if (vif->active)
    {
        // Put the EDCA parameters in the correct register according to the queue index
        switch (param->hw_queue)
        {
            case AC_BK:
                nxmac_edca_ac_0_set(param->ac_param);
                break;
            case AC_BE:
                nxmac_edca_ac_1_set(param->ac_param);
                break;
            case AC_VI:
                nxmac_edca_ac_2_set(param->ac_param);
                break;
            default:
                nxmac_edca_ac_3_set(param->ac_param);
                break;
        }

        // Store A-MPDU maximum duration parameters in environment for faster access by TX path
        mm_env_max_ampdu_duration_set();
    }

    #if NX_UAPSD
    // Store the information on UAPSD for the queue and interface index
    if (vif->type == VIF_STA)
    {
        ps_uapsd_set(vif, param->hw_queue, param->uapsd);
    }
    #endif

    // Send the confirmation
    ke_msg_send_basic(MM_SET_EDCA_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_MU_EDCA_REQ message handler.
 *
 * Sent by UMAC.\n
 * It configures the MU EDCA parameters to the TX HE module.
 *
 * Message @ref MM_SET_MU_EDCA_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_mu_edca_req_handler(ke_msg_id_t const msgid,
                           struct mm_set_mu_edca_req const *param,
                           ke_task_id_t const dest_id,
                           ke_task_id_t const src_id)
{
    #if NX_MAC_HE
    // Call the TX HE module
    txl_he_mu_edca_param_set(param);
    #endif

    // Send the confirmation
    ke_msg_send_basic(MM_SET_MU_EDCA_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_UORA_REQ message handler.
 *
 * Sent by UMAC.\n
 * It configures the UORA parameters to the TX HE module.
 *
 * Message @ref MM_SET_UORA_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_uora_req_handler(ke_msg_id_t const msgid,
                        struct mm_set_uora_req const *param,
                        ke_task_id_t const dest_id,
                        ke_task_id_t const src_id)
{
    #if NX_MAC_HE
    // Call the TX HE module
    txl_he_uora_param_set(param);
    #endif

    // Send the confirmation
    ke_msg_send_basic(MM_SET_UORA_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}
/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_BSS_COLOR_REQ message handler.
 *
 * Sent by UMAC.\n
 * It configures the BSS color register of the MAC HW.
 *
 * Message @ref MM_SET_BSS_COLOR_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_bss_color_req_handler(ke_msg_id_t const msgid,
                             struct mm_set_bss_color_req const *param,
                             ke_task_id_t const dest_id,
                             ke_task_id_t const src_id)
{
    #if NX_MAC_HE
    // Set the BSS color register
    nxmac_bss_color_set(param->bss_color);
    #endif

    // Send the confirmation
    ke_msg_send_basic(MM_SET_BSS_COLOR_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_TXOP_RTS_THRES_REQ message handler.
 *
 * Sent by UMAC.\n
 * It configures the TXOP duration RTS threshold to the indicated VIF.
 *
 * Message @ref MM_SET_TXOP_RTS_THRES_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_txop_rts_thres_req_handler(ke_msg_id_t const msgid,
                                  struct mm_set_txop_rts_thres_req const *param,
                                  ke_task_id_t const dest_id,
                                  ke_task_id_t const src_id)
{
    #if NX_MAC_HE
    struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];

    vif->txop_dur_rts_thres = param->txop_dur_rts_thres;
    #endif

    // Send the confirmation
    ke_msg_send_basic(MM_SET_TXOP_RTS_THRES_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_SLOTTIME_REQ message handler.
 *
 * Sent by Host.\n
 * It configures the slot time in the dedicated HW registers.
 *
 * Message @ref MM_SET_SLOTTIME_CFM is sent when completed.
 *
 * @note: Message @ref MM_SET_SLOTTIME_REQ is "delayed" by function @ref mm_hw_config_handler
 * until HW is in IDLE state.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_slottime_req_handler(ke_msg_id_t const msgid,
                            struct mm_set_slottime_req const *param,
                            ke_task_id_t const dest_id,
                            ke_task_id_t const src_id)
{
    uint16_t mac_core_clk;

    // Get MAC core clock value
    mac_core_clk = nxmac_mac_core_clk_freq_getf();

    // Set slot time
    nxmac_timings_2_pack(mac_core_clk * param->slottime, param->slottime);

    // Send the confirmation
    ke_msg_send_basic(MM_SET_SLOTTIME_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_MODE_REQ message handler.
 *
 * Sent by Host.\n
 * It configures the 802.11 mode in the dedicated HW registers.
 *
 * Message @ref MM_SET_MODE_CFM is sent when completed.
 *
 * @note: Message @ref MM_SET_MODE_REQ is "delayed" by function @ref mm_hw_config_handler
 * until HW is in IDLE state.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_mode_req_handler(ke_msg_id_t const msgid,
                        struct mm_set_mode_req const *param,
                        ke_task_id_t const dest_id,
                        ke_task_id_t const src_id)
{
    // Set slot time
    nxmac_abgn_mode_setf(param->abgnmode);

    // Send the confirmation
    ke_msg_send_basic(MM_SET_MODE_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_VIF_STATE_REQ message handler.
 *
 * Sent by UMAC or MESH module.\n
 * It changes state of the the specified interface. When STA interface become active it
 * also start beacon/connection monitoring.
 *
 * Message @ref MM_SET_VIF_STATE_CFM is sent when completed.
 *
 * @note: Message @ref MM_SET_VIF_STATE_REQ is "delayed" by function @ref mm_hw_config_handler
 * until HW is in IDLE state.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_vif_state_req_handler(ke_msg_id_t const msgid,
                             struct mm_set_vif_state_req const *param,
                             ke_task_id_t const dest_id,
                             ke_task_id_t const src_id)
{
    struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];

    // Save the state of the VIF
    vif->active = param->active;

    // Check if the VIF is of STA type
    if (vif->type == VIF_STA)
    {
        if (param->active)
        {
            // Get peer AP information
            struct sta_info_tag *sta = &sta_info_tab[vif->u.sta.ap_id];
            #if NX_POWERSAVE
            uint32_t drift;
            #endif
            #if NX_MULTI_ROLE
            // Compute default value of next TBTT based on current local time
            uint32_t next_tbtt = ke_time() + sta->bcn_int;

            // Program the next TBTT
            mm_timer_set(&vif->tbtt_timer, next_tbtt);
            #endif

            // We save the AP index for later use
            sta->aid = param->aid;

            // Set the BSSID mask according to the registered VIFs
            vif_mgmt_set_bssid_mask();

            #if (RW_BFMEE_EN)
            /*
             * If we are STA and MU Beamformee Capable we have to keep the AID in a register so that when we receive
             * a NDPA with several STA Info HW can detect the STA Info that is dedicated to us
             * Note: We currently support to be only one time MU Beamformee as STA.
             */
            nxmac_aid_setf(param->aid);
            #endif //(RW_BFMEE_EN)

            #if NX_MAC_HE
            if (hal_machw_he_support())
            {
                phy_set_aid(param->aid);

                // Enable the HE TB operation
                txl_he_tb_enable();
            }
            #endif

            #if NX_POWERSAVE
            vif->u.sta.listen_interval = 0;
            vif->u.sta.dont_wait_bcmc = false;
            // Compute the maximum drift we could suffer due to the LP clock inaccuracy
            drift = ((uint32_t)(mm_env.lp_clk_accuracy + MM_AP_CLK_ACCURACY) *
                     (uint32_t)sta->bcn_int) / 1000000;
            sta->drift = (uint16_t)drift;
            #if NX_UAPSD
            vif->u.sta.uapsd_last_rxtx = ke_time();
            #endif
            vif->prevent_sleep |= PS_VIF_WAITING_BCN;
            #endif

            #if NX_CONNECTION_MONITOR
            // Reset the beacon loss count and keep-alive frame time
            vif->u.sta.beacon_loss_cnt = 0;
            vif->u.sta.mon_last_crc = 0;
            vif->u.sta.mon_last_tx = ke_time();
            #endif

            #if (NX_CHNL_CTXT)
            // Try to spend more time on channel in order to catch a beacon
            chan_bcn_detect_start(vif);
            #endif //(NX_CHNL_CTXT)
        }
        else
        {
            #if NX_MULTI_ROLE
            // Clear the TBTT timer
            mm_timer_clear(&vif->tbtt_timer);
            #endif
            #if NX_MAC_HE
            // Enable the HE TB operation
            txl_he_tb_disable();
            #endif
        }
    }

    #if (NX_P2P)
    p2p_set_vif_state(vif, param->active);
    #endif //(NX_P2P)

    #if (NX_ANT_DIV)
    mm_ant_div_restore(vif->active);
    #endif //(NX_ANT_DIV)

    // Put the EDCA parameters in the registers
    if (param->active)
    {
        nxmac_edca_ac_0_set(vif->txq_params[AC_BK]);
        nxmac_edca_ac_1_set(vif->txq_params[AC_BE]);
        nxmac_edca_ac_2_set(vif->txq_params[AC_VI]);
        nxmac_edca_ac_3_set(vif->txq_params[AC_VO]);

        // Store A-MPDU maximum duration parameters in environment for faster access by TX path
        mm_env_max_ampdu_duration_set();
    }

    // Send the confirmation
    ke_msg_send_basic(MM_SET_VIF_STATE_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_POWER_REQ message handler.
 *
 * Sent by host.\n
 * It configures the tx power to use on a vif. If power asked is outside of what is allowed
 * by regulatory set power to maxumum allowed value.
 *
 * Message @ref MM_SET_POWER_CFM is sent when completed with the actual power configured.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_power_req_handler(ke_msg_id_t const msgid,
                         struct mm_set_power_req const *param,
                         ke_task_id_t const dest_id,
                         ke_task_id_t const src_id)
{
    struct mm_set_power_cfm *cfm;
    struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];

    // Allocate confirmation message
    cfm = KE_MSG_ALLOC(MM_SET_POWER_CFM, src_id, dest_id, mm_set_power_cfm);

    #if NX_UMAC_PRESENT
    vif->user_tx_power = param->power;
    tpc_update_vif_tx_power(vif);
    #else
    tpc_set_vif_tx_power(vif, param->power);
    TRACE_LMAC(TPC, "{VIF-%d} host request TX power %ddBm set to %ddBm",
               vif->index, param->power, vif->tx_power);
    #endif /* NX_UMAC_PRESENT */
    tpc_get_vif_tx_power(vif, &cfm->power, &cfm->radio_idx);

    // Send the confirmation
    ke_msg_send(cfm);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_DBG_TRIGGER_REQ message handler.
 *
 * Sent by Host.\n
 * It forces firmware to upload debug dump.
 *
 * @note: Message @ref MM_DBG_TRIGGER_REQ is "delayed" by function @ref mm_hw_config_handler
 * until HW is in IDLE state.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_dbg_trigger_req_handler(ke_msg_id_t const msgid,
                           struct mm_dbg_trigger_req const *param,
                           ke_task_id_t const dest_id,
                           ke_task_id_t const src_id)
{
    // Call the platform function to force the trigger
    dbg_force_trigger(param->error);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_ADD_IF_REQ message handler.
 *
 * Sent by host.\n
 * It creates a new interface.
 *
 * Message @ref MM_ADD_IF_CFM is sent when completed.
 *
 * @note: Message @ref MM_ADD_IF_REQ is "delayed" by function @ref mm_hw_config_handler
 * until HW is in IDLE state.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_add_if_req_handler(ke_msg_id_t const msgid,
                      struct mm_add_if_req const *param,
                      ke_task_id_t const dest_id,
                      ke_task_id_t const src_id)
{
    struct mm_add_if_cfm *cfm;

    // Allocate confirmation message
    cfm = KE_MSG_ALLOC(MM_ADD_IF_CFM, src_id, dest_id, mm_add_if_cfm);

    // Register the VIF
    cfm->status = vif_mgmt_register(&param->addr, param->type, param->p2p, &cfm->inst_nbr);

    // Send the confirmation
    ke_msg_send(cfm);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_REMOVE_IF_REQ message handler.
 *
 * Sent by host.\n
 * It deletes a existing interface.
 *
 * Message @ref MM_REMOVE_IF_CFM is sent when completed.
 *
 * @note: Message @ref MM_REMOVE_IF_REQ is "delayed" by function @ref mm_hw_config_handler
 * until HW is in IDLE state.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_remove_if_req_handler(ke_msg_id_t const msgid,
                         struct mm_remove_if_req const *param,
                         ke_task_id_t const dest_id,
                         ke_task_id_t const src_id)
{
    // Check if VIF index is valid
    if (param->inst_nbr < NX_VIRT_DEV_MAX)
        // Unregister the VIF
        vif_mgmt_unregister(param->inst_nbr);

    // Check if we have removed the last VIF
    if (co_list_is_empty(&vif_mgmt_env.used_list))
        // No more active VIFs, go back to monitor mode
        hal_machw_monitor_mode();

    // Send the confirmation
    ke_msg_send_basic(MM_REMOVE_IF_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_IDLE_REQ message handler.
 *
 * Sent by Host or ME task.\n
 * Request MM task to set MACHW in idle state.
 *
 * Message @ref MM_SET_IDLE_CFM is sent when IDLE set is reached.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_idle_req_handler(ke_msg_id_t const msgid,
                        struct mm_set_idle_req const *param,
                        ke_task_id_t const dest_id,
                        ke_task_id_t const src_id)
{
    if ((ke_state_get(dest_id) == MM_HOST_BYPASSED) ||
        (ke_state_get(dest_id) == MM_NO_IDLE))
    {
        // UMAC requests for IDLE or active are currently bypassed.
        return (KE_MSG_SAVED);
    }

    // Save the state that is requested by the host
    mm_env.host_idle = param->hw_idle;

    // Check if HW has to be put in IDLE or ACTIVE state
    if (param->hw_idle)
    {
        // Check if we are in IDLE state
        switch (ke_state_get(dest_id))
        {
            case MM_IDLE:
                // Sanity check: As MM state is IDLE, HW state is also supposed to be IDLE
                ASSERT_ERR(nxmac_current_state_getf() == HW_IDLE);

                mm_env.prev_mm_state = MM_IDLE;
                mm_env.prev_hw_state = HW_IDLE;
                break;

            case MM_GOING_TO_IDLE:
                // MAC is currently going to IDLE, so simply save the message. It will be
                // rescheduled once the MAC has switched to IDLE.
                return (KE_MSG_SAVED);

            default:
                // Request to MAC HW to switch to IDLE state
                hal_machw_idle_req();

                // Adjust the MM state accordingly
                ke_state_set(dest_id, MM_GOING_TO_IDLE);

                // Save the message
                return (KE_MSG_SAVED);
        }
    }
    else
    {
        if (ke_state_get(dest_id) == MM_GOING_TO_IDLE)
        {
            // We are currently going to IDLE, wait for this to be complete before
            // accepting a new request from host
            return (KE_MSG_SAVED);
        }

        mm_active();
    }

    // Send the confirmation
    ke_msg_send_basic(MM_SET_IDLE_CFM, src_id, dest_id);

    // Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_FORCE_IDLE_REQ message handler.
 *
 * Sent by P2P or CHAN modules.\n
 * It requests the  MM task to set MACHW in idle state and to execute callback provided
 * in message parameters once IDLE state is reached.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_force_idle_req_handler(ke_msg_id_t const msgid,
                          struct mm_force_idle_req const *param,
                          ke_task_id_t const dest_id,
                          ke_task_id_t const src_id)
{
    // Sanity check - The force IDLE procedure shall not be performed while the NO IDLE
    // state is present for a beacon transmission. If this assertion is triggered it
    // probably means that the scheduling of a channel switch is done at a bad time.
    ASSERT_ERR(ke_state_get(dest_id) != MM_NO_IDLE);

    // Check if we are in IDLE state
    switch (ke_state_get(dest_id))
    {
        case MM_IDLE:
            // Sanity check: As MM state is IDLE, HW state is also supposed to be IDLE
            ASSERT_ERR(nxmac_current_state_getf() == HW_IDLE);
            break;

        case MM_GOING_TO_IDLE:
            // MAC is currently going to IDLE, so simply save the message. It will be
            // rescheduled once the MAC has switched to IDLE.
            return (KE_MSG_SAVED);

        default:
            // Request to MAC HW to switch to IDLE state
            hal_machw_idle_req();

            // Adjust the MM state accordingly
            ke_state_set(dest_id, MM_GOING_TO_IDLE);

            // Save the message
            return (KE_MSG_SAVED);
    }

    // IDLE requests from host are now bypassed
    ke_state_set(dest_id, MM_HOST_BYPASSED);

    // Call the callback function
    param->cb();

    // Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_STA_ADD_REQ message handler.
 *
 * Sent from UMAC or MESH modules.\n
 * It allocates and initializes a STA entry from the @p free_sta_list in @ref sta_info_env.
 *
 * Message @ref MM_STA_ADD_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_sta_add_req_handler(ke_msg_id_t const msgid,
                       struct mm_sta_add_req const *param,
                       ke_task_id_t const dest_id,
                       ke_task_id_t const src_id)
{
    // Allocate the response structure
    struct mm_sta_add_cfm *rsp = KE_MSG_ALLOC(MM_STA_ADD_CFM, src_id, dest_id, mm_sta_add_cfm);

    // Register the new station
    rsp->status = mm_sta_add(param, &rsp->sta_idx, &rsp->hw_sta_idx);

    // Send the confirmation
    ke_msg_send(rsp);

    return (KE_MSG_CONSUMED);
}


/**
 ****************************************************************************************
 * @brief MM module @ref MM_STA_DEL_REQ message handler.
 *
 * Sent from UMAC or MESH modules.\n
 * It removes a STA entry.\n
 *
 * Message @ref MM_STA_DEL_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_sta_del_req_handler(ke_msg_id_t const msgid,
                       struct mm_sta_del_req const *param,
                       ke_task_id_t const dest_id,
                       ke_task_id_t const src_id)
{
    // Delete the station
    mm_sta_del(param->sta_idx);

    // Send the confirmation to the upper MAC
    ke_msg_send_basic(MM_STA_DEL_CFM, src_id, dest_id);

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_KEY_ADD_REQ message handler.
 *
 * Sent by Host.\n
 * It adds a key used for encryption in the hardware.
 *
 * Message @ref MM_KEY_ADD_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_key_add_req_handler(ke_msg_id_t const msgid,
                       struct mm_key_add_req const *param,
                       ke_task_id_t const dest_id,
                       ke_task_id_t const src_id)
{
    uint8_t hw_key_idx;
    struct mm_key_add_cfm *keyadd_cfm_param =
        KE_MSG_ALLOC(MM_KEY_ADD_CFM, src_id, dest_id, mm_key_add_cfm);

    // Assert on invalid Key Index.
    #if NX_MFP
    ASSERT_ERR(param->key_idx < MAC_DEFAULT_MFP_KEY_COUNT);
    #else
    ASSERT_ERR(param->key_idx < MAC_DEFAULT_KEY_COUNT);
    #endif

    // Assert on invalid Key Length.
    ASSERT_ERR(param->key.length <= MAC_SEC_KEY_LEN);

    // Assert on invalid Key Length.
    ASSERT_ERR(param->cipher_suite <= MAC_CIPHER_BIP_CMAC_128);

    // Copy the key in the HW
    hw_key_idx = mm_sec_machwkey_wr(param);

    // KEY_ADD_CFM return Parameter.
    keyadd_cfm_param->hw_key_idx = hw_key_idx;
    keyadd_cfm_param->status = CO_OK;

    // Post the MM_KEY_ADD_CFM message in QUEUE
    ke_msg_send(keyadd_cfm_param);

    // Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_KEY_DEL_REQ message handler.
 *
 * Sent by Host.\n
 * It removes an encryption key from the hardware.
 *
 * Message @ref MM_KEY_DEL_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_key_del_req_handler(ke_msg_id_t const msgid,
                       struct mm_key_del_req const *param,
                       ke_task_id_t const dest_id,
                       ke_task_id_t const src_id)
{
    // Assert on invalid Key Index.
    #if NX_MFP
    ASSERT_ERR(param->hw_key_idx <= MM_SEC_MAX_MFP_KEY_NBR);
    #else
    ASSERT_ERR(param->hw_key_idx <= MM_SEC_MAX_KEY_NBR);
    #endif

    // Disable the key in the HW
    mm_sec_machwkey_del(param->hw_key_idx);

    // Post the MM_KEY_DEL_CFM message in QUEUE
    ke_msg_send_basic(MM_KEY_DEL_CFM, src_id, dest_id);

    // Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_BA_ADD_REQ message handler.
 *
 * Sent by UMAC.\n
 * It registers the received BA agreement details (tid, ssn, buffer size) for the
 * appropriate STA.
 *
 * Message @ref MM_BA_ADD_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_ba_add_req_handler(ke_msg_id_t const msgid,
                      struct mm_ba_add_req const *param,
                      ke_task_id_t const dest_id,
                      ke_task_id_t const src_id)
{
    uint8_t status = BA_AGMT_ESTABLISHED;

    do
    {
        // Check BA type
        if (param->type == BA_AGMT_TX)
        {
            #if (NX_AMPDU_TX)
            // Buffer Size
            uint8_t bufsz;

            #if (!NX_UMAC_PRESENT)
            // Check if a BA agreement already exists
            if (mm_ba_agmt_tx_exists(param->sta_idx, param->tid))
            {
                status = BA_AGMT_ALREADY_EXISTS;
                break;
            }
            #endif //(!NX_UMAC_PRESENT)

            // Limit the maximum number of MPDUs in the A-MPDU to the minimum of
            // the half of the BA window and the half of the number of TX descriptors
            // per queue
            if (param->bufsz < nx_txdesc_cnt[mac_tid2ac[param->tid]])
            {
                bufsz = param->bufsz / 2;
            }
            else
            {
                bufsz = nx_txdesc_cnt[mac_tid2ac[param->tid]] / 2;
            }

            sta_mgmt_set_tx_buff_size(param->sta_idx, param->tid, bufsz);
            sta_mgmt_set_tx_ssn(param->sta_idx, param->tid, param->ssn);
            #else
            status = BA_AGMT_NOT_SUPPORTED;
            #endif //(NX_AMPDU_TX)
        }
        else
        {
            #if (NX_REORD)
            // Create the reordering instance
            if (!rxu_cntrl_reord_create(&sta_info_tab[param->sta_idx], param->tid,
                                        param->ssn))
            {
                status = BA_AGMT_NO_MORE_BA_AGMT;
                break;
            }
            #endif //(NX_REORD)

            // Reset the MAC HW BlockAck bitmaps
            // Currently it is not possible to reset per STA/TID, so reset all of them
            nxmac_ba_ps_bitmap_reset_setf(1);
        }
    } while (0);

    // Allocate kernel message space for the confirmation to send
    struct mm_ba_add_cfm *cfm = KE_MSG_ALLOC(MM_BA_ADD_CFM, src_id, dest_id, mm_ba_add_cfm);

    cfm->sta_idx = param->sta_idx;
    cfm->tid     = param->tid;
    cfm->status  = status;

    // Post the MM_BA_ADD_CFM message in QUEUE
    ke_msg_send(cfm);

    // Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}


#if (NX_AMPDU_TX || NX_REORD)
/**
 ****************************************************************************************
 * @brief MM module @ref MM_BA_DEL_REQ message handler.
 *
 * Sent by UMAC.\n
 * It unregisters the BA agreement information from the STA info table.
 *
 * Message @ref MM_BA_DEL_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_ba_del_req_handler(ke_msg_id_t const msgid,
                     struct mm_ba_del_req const *param,
                     ke_task_id_t const dest_id,
                     ke_task_id_t const src_id)
{
    uint8_t status = BA_AGMT_DELETED;

    do
    {
        // Check BA type
        if (param->type == BA_AGMT_TX)
        {
            #if (NX_AMPDU_TX)
            // Check if the specified BA agreement already exists
            if (!mm_ba_agmt_tx_exists(param->sta_idx, param->tid))
            {
                status = BA_AGMT_DOESNT_EXIST;
                break;
            }

            // Unregister BA agreement information from (STA, TID) -> invalidate bufsz
            sta_mgmt_set_tx_buff_size(param->sta_idx, param->tid, 0);
            sta_mgmt_set_tx_ssn(param->sta_idx, param->tid, 0);
            #else
            status = BA_AGMT_NOT_SUPPORTED;
            #endif //(NX_AMPDU_TX)
        }
        else
        {
            #if (NX_REORD)
            // Check if the specified BA agreement already exists
            if (!mm_ba_agmt_rx_exists(param->sta_idx, param->tid))
            {
                status = BA_AGMT_DOESNT_EXIST;
                break;
            }

            // Delete the reordering instance
            rxu_cntrl_reord_delete(&sta_info_tab[param->sta_idx], param->tid);

            #else
            status = BA_AGMT_NOT_SUPPORTED;
            #endif //(NX_REORD)
        }
    } while (0);

    // Allocate kernel message space for the confirmation to send
    struct mm_ba_del_cfm *cfm = KE_MSG_ALLOC(MM_BA_DEL_CFM, src_id, dest_id, mm_ba_del_cfm);

    cfm->sta_idx = param->sta_idx;
    cfm->tid     = param->tid;
    cfm->status  = status;

    // Post the MM_BA_DEL_CFM message in QUEUE
    ke_msg_send(cfm);

    // Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}
#endif //(NX_AMPDU_TX || NX_REORD)

#if NX_BEACONING
/**
 ****************************************************************************************
 * @brief MM module @ref MM_TBTT_MOVE_REQ message handler.
 *
 * It calls the MAC HW API that moves the TSF.
 *
 * @note: Message @ref MM_TBTT_MOVE_REQ is "delayed" by function @ref mm_hw_config_handler
 * until HW is in IDLE state.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_tbtt_move_req_handler(ke_msg_id_t const msgid,
                         struct mm_tbtt_move_req const *param,
                         ke_task_id_t const dest_id,
                         ke_task_id_t const src_id)
{
    // No more TBTT move ongoing after that
    mm_env.tbtt_move_ongoing = false;

    // Update the TSF
    if (hal_machw_tsf_move(param->offset))
    {
        // Not a good time for the move, reprogram it
        mm_ap_tbtt_move(param->offset);
    }

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}
#endif // NX_BEACONING

#if (NX_CHNL_CTXT)
#if (!NX_UMAC_PRESENT)
/**
 ****************************************************************************************
 * @brief MM module @ref MM_CHAN_CTXT_ADD_REQ message handler.
 *
 * Sent by Host.\n
 * Creates a new channel context.
 *
 * Message @ref MM_CHAN_CTXT_ADD_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_chan_ctxt_add_req_handler(ke_msg_id_t const msgid,
                             struct mm_chan_ctxt_add_req const *param,
                             ke_task_id_t const dest_id,
                             ke_task_id_t const src_id)
{
    // Allocate confirmation message
    struct mm_chan_ctxt_add_cfm *cfm = KE_MSG_ALLOC(MM_CHAN_CTXT_ADD_CFM, src_id, dest_id, mm_chan_ctxt_add_cfm);

    // Register the VIF
    cfm->status = chan_ctxt_add(&(param->chan), &cfm->index);

    // Send the confirmation
    ke_msg_send(cfm);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_CHAN_CTXT_DEL_REQ message handler.
 *
 * Sent by Host.\n
 * Deletes a existing channel context.
 *
 * Message @ref MM_CHAN_CTXT_DEL_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_chan_ctxt_del_req_handler(ke_msg_id_t const msgid,
                             struct mm_chan_ctxt_del_req const *param,
                             ke_task_id_t const dest_id,
                             ke_task_id_t const src_id)
{
    // Delete the channel context
    chan_ctxt_del(param->index);

    // Send the confirmation
    ke_msg_send_basic(MM_CHAN_CTXT_DEL_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_CHAN_CTXT_LINK_REQ message handler.
 *
 * Sent by Host.\n
 * It links a vif to a channel context (previously created with @ref MM_CHAN_CTXT_ADD_REQ)
 *
 * Message @ref MM_CHAN_CTXT_LINK_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_chan_ctxt_link_req_handler(ke_msg_id_t const msgid,
                              struct mm_chan_ctxt_link_req const *param,
                              ke_task_id_t const dest_id,
                              ke_task_id_t const src_id)
{
    if (param->chan_switch)
    {
        // In case of channel switch unlink current ctxt first
        chan_ctxt_unlink(param->vif_index);
    }

    // Link the channel context to the VIF
    chan_ctxt_link(param->vif_index, param->chan_index);

    // Send the confirmation
    ke_msg_send_basic(MM_CHAN_CTXT_LINK_CFM, src_id, dest_id);

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_CHAN_CTXT_SCHED_REQ message handler.
 *
 * Sent by Host.\n
 * It requests fw to switch to the specified channel context.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_chan_ctxt_sched_req_handler(ke_msg_id_t const msgid,
                               struct mm_chan_ctxt_sched_req const *param,
                               ke_task_id_t const dest_id,
                               ke_task_id_t const src_id)
{
    // Schedule the channel context
    chan_ctxt_sched(param, src_id);

    //  Return the status.
    return (KE_MSG_CONSUMED);
}
#endif //(!NX_UMAC_PRESENT)

/**
 ****************************************************************************************
 * @brief MM module @ref MM_CHAN_CTXT_UNLINK_REQ message handler.
 *
 * Sent by Host.\n
 * It unlinks a vif from a channel context.
 *
 * Message @ref MM_CHAN_CTXT_UNLINK_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_chan_ctxt_unlink_req_handler(ke_msg_id_t const msgid,
                                struct mm_chan_ctxt_unlink_req const *param,
                                ke_task_id_t const dest_id,
                                ke_task_id_t const src_id)
{
    // Link the channel context to the VIF
    chan_ctxt_unlink(param->vif_index);

    // Send the confirmation
    ke_msg_send_basic(MM_CHAN_CTXT_UNLINK_CFM, src_id, dest_id);

    //  Return the status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_CHAN_CTXT_UPDATE_REQ message handler.
 *
 * Sent by UMAC.\n
 * It updates the channel configuration contained in a channel context.
 *
 * Message @ref MM_CHAN_CTXT_UPDATE_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_chan_ctxt_update_req_handler(ke_msg_id_t const msgid,
                                struct mm_chan_ctxt_update_req const *param,
                                ke_task_id_t const dest_id,
                                ke_task_id_t const src_id)
{
    // Link the channel context to the VIF
    chan_ctxt_update(param);

    // Send the confirmation
    ke_msg_send_basic(MM_CHAN_CTXT_UPDATE_CFM, src_id, dest_id);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_REMAIN_ON_CHANNEL_REQ message handler.
 *
 * Sent by UMAC, CHAN and TDLS modules.\n
 * It starts a "Remain On Channel" procedure.
 *
 * Message @ref MM_REMAIN_ON_CHANNEL_CFM is sent when completed (except if the requester
 * task is MM or TDLS).
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_remain_on_channel_req_handler(ke_msg_id_t const msgid,
                                 struct mm_remain_on_channel_req const *param,
                                 ke_task_id_t const dest_id,
                                 ke_task_id_t const src_id)
{
    // Start RoC
    uint8_t status = chan_roc_req(param, src_id);

    #if (NX_TDLS)
    if ((src_id != TASK_MM) && (src_id != TASK_TDLS))
    #else
    if (src_id != TASK_MM)
    #endif
    {
        // Send back the confirmation
        struct mm_remain_on_channel_cfm *cfm = KE_MSG_ALLOC(MM_REMAIN_ON_CHANNEL_CFM,
                                                            src_id, dest_id,
                                                            mm_remain_on_channel_cfm);

        cfm->op_code         = param->op_code;
        cfm->status          = status;
        cfm->chan_ctxt_index = CHAN_ROC_CTXT_IDX;

        ke_msg_send(cfm);
    }

    return (KE_MSG_CONSUMED);
}
#endif //(NX_CHNL_CTXT)

#if NX_BCN_AUTONOMOUS_TX
/**
 ****************************************************************************************
 * @brief MM module @ref MM_BCN_CHANGE_REQ message handler.
 *
 * Sent by UMAC or MESH modules.\n
 * It updates (or set) content of the beacon frame for one interface
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_bcn_change_req_handler(ke_msg_id_t const msgid,
                          struct mm_bcn_change_req const *param,
                          ke_task_id_t const dest_id,
                          ke_task_id_t const src_id)
{
    // Schedule the channel context
    mm_bcn_change(param);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_NO_FREE);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_TIM_UPDATE_REQ message handler.
 *
 * Sent by UMAC.\n
 * It provides information to update the TIM IE in the beacon frame.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_tim_update_req_handler(ke_msg_id_t const msgid,
                          struct mm_tim_update_req const *param,
                          ke_task_id_t const dest_id,
                          ke_task_id_t const src_id)
{
    // Schedule the channel context
    mm_tim_update(param);

    //  Return the KE_MSG_CONSUMED status.
    return (KE_MSG_NO_FREE);
}
#endif

#if (NX_ANT_DIV)
/**
 ****************************************************************************************
 * @brief MM module @ref MM_ANT_DIV_INIT_REQ message handler.
 *
 * Sent by UMAC.\n
 * It initializes the antenna diversity algorithm.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_ant_div_init_req_handler(ke_msg_id_t const msgid,
                            struct mm_ant_div_init_req const *param,
                            ke_task_id_t const dest_id,
                            ke_task_id_t const src_id)
{
    // Initialize the parameters of antenna diversity
    mm_ant_div_init(param->enable);

    // Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_ANT_DIV_STOP_REQ message handler.
 *
 * Sent by UMAC.\n
 * It stops the antenna diversity algorithm.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_ant_div_stop_req_handler(ke_msg_id_t const msgid,
                            void const *param,
                            ke_task_id_t const dest_id,
                            ke_task_id_t const src_id)
{
    // Switch antenna and stop the antenna diversity algorithm
    mm_ant_div_switch_and_stop();

    // Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}


/**
 ****************************************************************************************
 * @brief MM module @ref MM_ANT_DIV_UPDATE_REQ message handler.
 *
 * Sent by UMAC.\n
 * It updates the antenna diversity parameters.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_ant_div_update_req_handler(ke_msg_id_t const msgid,
                              void const *param,
                              ke_task_id_t const dest_id,
                              ke_task_id_t const src_id)
{
    // Update the parameters of the antenna diversity
    mm_ant_div_update();

    // Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SWITCH_ANTENNA_REQ message handler.
 *
 * Sent by UMAC or MM task.\n
 * It switches the antenna connected to path_0.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_switch_antenna_req_handler(ke_msg_id_t const msgid,
                              void const *param,
                              ke_task_id_t const dest_id,
                              ke_task_id_t const src_id)
{
    // Switch antenna
    mm_ant_div_switch_antenna();

    // Return the KE_MSG_CONSUMED status.
    return (KE_MSG_CONSUMED);
}
#endif

/**
 ****************************************************************************************
 * @brief Postpones message processing until HW reach IDLE state.
 *
 * It is used as handler for the following messages:
 * @ref MM_SET_CHANNEL_REQ, @ref MM_SET_FILTER_REQ, @ref MM_ADD_IF_REQ,
 * @ref MM_REMOVE_IF_REQ, @ref MM_SET_BASIC_RATES_REQ, @ref MM_SET_BEACON_INT_REQ,
 * @ref MM_SET_DTIM_REQ, @ref MM_SET_BSSID_REQ, @ref MM_SET_EDCA_REQ,
 * @ref MM_SET_SLOTTIME_REQ, @ref MM_SET_MODE_REQ, @ref MM_SET_VIF_STATE_REQ,
 * @ref MM_BA_ADD_REQ, @ref MM_CHAN_CTXT_UPDATE_REQ, @ref MM_DBG_TRIGGER_REQ,
 * @ref MM_SWITCH_ANTENNA_REQ,
 *
 * When the message is received,
 * - If HW is already in IDLE state then the corresponding handler function is executed
 *   immediately.
 * - Otherwise, it will first move HW to IDLE state and call handler function when IDLE
 *   state is reached.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_hw_config_handler(ke_msg_id_t const msgid,
                     void const *param,
                     ke_task_id_t const dest_id,
                     ke_task_id_t const src_id)
{
    int status = KE_MSG_SAVED;

    // Check if we are in IDLE state
    switch (ke_state_get(dest_id))
    {
        case MM_IDLE:
            // Sanity check: As MM state is IDLE, HW state is also supposed to be IDLE
            ASSERT_ERR(nxmac_current_state_getf() == HW_IDLE);

            // Now that we are in IDLE state, handle the configuration request
            switch (msgid)
            {
                case MM_SET_CHANNEL_REQ:
                    status = mm_set_channel_req_handler(msgid, param, dest_id, src_id);
                    break;
                case MM_SET_FILTER_REQ:
                    status = mm_set_filter_req_handler(msgid, param, dest_id, src_id);
                    break;
                case MM_ADD_IF_REQ:
                    status = mm_add_if_req_handler(msgid, param, dest_id, src_id);
                    break;
                case MM_REMOVE_IF_REQ:
                    status = mm_remove_if_req_handler(msgid, param, dest_id, src_id);
                    break;
                case MM_SET_BASIC_RATES_REQ:
                    status = mm_set_basic_rates_req_handler(msgid, param, dest_id, src_id);
                    break;
                case MM_SET_BEACON_INT_REQ:
                    status = mm_set_beacon_int_req_handler(msgid, param, dest_id, src_id);
                    break;
                case MM_SET_DTIM_REQ:
                    status = mm_set_dtim_req_handler(msgid, param, dest_id, src_id);
                    break;
                case MM_SET_BSSID_REQ:
                    status = mm_set_bssid_req_handler(msgid, param, dest_id, src_id);
                    break;
                case MM_SET_SLOTTIME_REQ:
                    status = mm_set_slottime_req_handler(msgid, param, dest_id, src_id);
                    break;
                case MM_SET_MODE_REQ:
                    status = mm_set_mode_req_handler(msgid, param, dest_id, src_id);
                    break;
                case MM_SET_VIF_STATE_REQ:
                    status = mm_set_vif_state_req_handler(msgid, param, dest_id, src_id);
                    break;
                case MM_BA_ADD_REQ:
                    status = mm_ba_add_req_handler(msgid, param, dest_id, src_id);
                    break;
                #if (NX_CHNL_CTXT)
                case MM_CHAN_CTXT_UPDATE_REQ:
                    status = mm_chan_ctxt_update_req_handler(msgid, param, dest_id, src_id);
                    break;
                #endif //(NX_CHNL_CTXT)
                case MM_DBG_TRIGGER_REQ:
                    status = mm_dbg_trigger_req_handler(msgid, param, dest_id, src_id);
                    break;
                #if (NX_ANT_DIV)
                case MM_SWITCH_ANTENNA_REQ:
                    status = mm_switch_antenna_req_handler(msgid, param, dest_id, src_id);
                    break;
                #endif //(NX_ANT_DIV)
                #if NX_BEACONING
                case MM_TBTT_MOVE_REQ:
                    status = mm_tbtt_move_req_handler(msgid, param, dest_id, src_id);
                    break;
                #endif //NX_BEACONING

                default:
                    ASSERT_ERR(0);
                    break;
            }

            // Restore the HW state
            nxmac_next_state_setf(mm_env.prev_hw_state);

            // As well as the MM state
            ke_state_set(dest_id, mm_env.prev_mm_state);
            break;

        case MM_GOING_TO_IDLE:
        case MM_HOST_BYPASSED:
        case MM_NO_IDLE:
            // MAC is currently going to IDLE, or IDLE is disallowed, so simply save the
            // message. It will be rescheduled once the MAC has switched to IDLE or once
            // IDLE is allowed again.
            break;

        default:
            // Store the current HW and MM states for later restoring
            mm_env.prev_hw_state = nxmac_current_state_getf();
            mm_env.prev_mm_state = ke_state_get(dest_id);

            // Request to MAC HW to switch to IDLE state
            hal_machw_idle_req();

            // Adjust the MM state accordingly
            ke_state_set(dest_id, MM_GOING_TO_IDLE);
            break;
    }

    return (status);
}

#if NX_POWERSAVE
/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_PS_MODE_REQ message handler.
 *
 * Sent by UMAC.\n
 * It indicates LMAC whether PowerSave should be enabled or disabled.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_ps_mode_req_handler(ke_msg_id_t const msgid,
                           struct mm_set_ps_mode_req const *param,
                           ke_task_id_t const dest_id,
                           ke_task_id_t const src_id)
{
    // Ask the PS module to change the Power-save mode
    ps_set_mode(param->new_state, src_id);

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_PS_OPTIONS_REQ message handler.
 *
 * Sent by UMAC or SM module.\n
 * It configures the power save mode for STATION interfaces.
 *
 * Message @ref MM_SET_PS_OPTIONS_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_ps_options_req_handler(ke_msg_id_t const msgid,
                              struct mm_set_ps_options_req const *param,
                              ke_task_id_t const dest_id,
                              ke_task_id_t const src_id)
{
    struct vif_info_tag *vif = &vif_info_tab[param->vif_index];

    // Sanity check - These parameters apply only for a STA interface
    ASSERT_ERR(vif->type == VIF_STA);

    // Set the parameters
    vif->u.sta.listen_interval = param->listen_interval;
    vif->u.sta.dont_wait_bcmc = param->dont_listen_bc_mc;

    // Send the confirmation
    ke_msg_send_basic(MM_SET_PS_OPTIONS_CFM, src_id, dest_id);

    return (KE_MSG_CONSUMED);
}
#endif

#if (NX_P2P_GO)
/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_P2P_NOA_REQ message handler.
 *
 * Sent by Host.\n
 * It configures Notice of Absence on a P2P GO interface.
 *
 * Message @ref MM_SET_P2P_NOA_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int mm_set_p2p_noa_req_handler(ke_msg_id_t const msgid,
                                      struct mm_set_p2p_noa_req const *param,
                                      ke_task_id_t const dest_id,
                                      ke_task_id_t const src_id)
{
    // Allocate confirmation message
    struct mm_set_p2p_noa_cfm *cfm = KE_MSG_ALLOC(MM_SET_P2P_NOA_CFM, TASK_API,
                                                  TASK_MM, mm_set_p2p_noa_cfm);
    struct vif_info_tag *vif;

    cfm->status = CO_FAIL;

    if ((param->vif_index <= NX_VIRT_DEV_MAX) &&
        (vif = &vif_info_tab[param->vif_index]) && vif->p2p)
    {
        if (param->count)
        {
            // Compute next TBTT instant in order to deduce the NOA start time
            uint32_t beacon_int = (uint32_t)vif->u.ap.bcn_int << 10;
            uint32_t next_tbtt  = ((nxmac_tsf_lo_get() / beacon_int) + 1) * beacon_int;
            uint8_t noa_idx;

            // Next TBTT time is currently based on TSF value, as TSF counter and
            // Monoatomic counter might have different values, we have to convert this
            // value in local time in order to use internal timers.
            // The NOA start time inserted in the beacon will be a TSF value.
            next_tbtt += (ke_time() - nxmac_tsf_lo_get());

            // Start NOA procedure and check if NOA has been successfully added
            noa_idx = p2p_go_noa_start(vif, false, param->dyn_noa, param->count,
                                       param->interval_us, param->duration_us,
                                       next_tbtt + param->start_offset);
            if (noa_idx != P2P_INVALID_IDX)
            {
                cfm->status = CO_OK;
            }
        }
        else
        {
            cfm->status = p2p_go_noa_stop(vif, param->noa_inst_nb, true);
        }
    }

    // Send the message
    ke_msg_send(cfm);

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief MM module @ref MM_SET_P2P_OPPPS_REQ message handler.
 *
 * Sent by host.\n
 * It configures P2P Opportunistic Power Save on a P2P GO interface.
 *
 * Message @ref MM_SET_P2P_OPPPS_CFM is sent when completed.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int mm_set_p2p_oppps_req_handler(ke_msg_id_t const msgid,
                                        struct mm_set_p2p_oppps_req const *param,
                                        ke_task_id_t const dest_id,
                                        ke_task_id_t const src_id)
{
    // Allocate confirmation message
    struct mm_set_p2p_oppps_cfm *cfm = KE_MSG_ALLOC(MM_SET_P2P_OPPPS_CFM, TASK_API,
                                                    TASK_MM, mm_set_p2p_oppps_cfm);
    struct vif_info_tag *vif;

    cfm->status = CO_FAIL;

    if ((param->vif_index <= NX_VIRT_DEV_MAX) &&
        (vif = &vif_info_tab[param->vif_index]) && vif->p2p)
    {
        cfm->status = CO_OK;

        if (param->ctwindow)
        {
            // Start OPPPS procedure
            p2p_go_oppps_start(vif, param->ctwindow);
        }
        else
        {
            p2p_go_oppps_stop(vif);
        }
    }

    // Send the message
    ke_msg_send(cfm);

    return (KE_MSG_CONSUMED);
}
#endif //(NX_P2P_GO)

#if (RW_BFMER_EN)
/**
 ****************************************************************************************
 * @brief MM module @ref MM_BFMER_ENABLE_REQ message handler.
 *
 * Sent by Host.\n
 * It enables beamformed transmission with a specifc STA.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int mm_bfmer_enable_req_handler(ke_msg_id_t const msgid,
                                       struct mm_bfmer_enable_req const *param,
                                       ke_task_id_t const dest_id,
                                       ke_task_id_t const src_id)
{
    // Check that connection exists
    if (bfr_is_enabled() && (sta_info_tab[param->sta_idx].staid != INVALID_STA_IDX))
    {
        bfr_add_sta_ind(param->sta_idx, param->vht_mu_bfmee, param->aid, param->host_bfr_addr,
                        param->host_bfr_size, param->rx_nss);
    }

    return (KE_MSG_CONSUMED);
}
#endif //(RW_BFMER_EN)

#if (RW_MUMIMO_TX_EN)
/**
 ****************************************************************************************
 * @brief MM module @ref MM_MU_GROUP_UPDATE_REQ message handler.
 *
 * Sent by Host.\n
 * It updates the list of MU groups.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int mm_mu_group_update_req_handler(ke_msg_id_t const msgid,
                                          struct mm_mu_group_update_req const *param,
                                          ke_task_id_t const dest_id,
                                          ke_task_id_t const src_id)
{
    // Update the peer
    bfr_group_update_req(param);

    // Freeing done when the request is confirmed to the host
    return (KE_MSG_NO_FREE);
}
#endif //(RW_MUMIMO_TX_EN)

/**
 ****************************************************************************************
 * @brief MM module @ref MM_CFG_RSSI_REQ message handler.
 *
 * Sent by Host.\n
 * It configures connection quality monitor RSSI threshold.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int mm_cfg_rssi_req_handler(ke_msg_id_t const msgid,
                                   struct mm_cfg_rssi_req const *param,
                                   ke_task_id_t const dest_id,
                                   ke_task_id_t const src_id)
{
    struct vif_info_tag *vif = &vif_info_tab[param->vif_index];

    // Sanity check - These parameters apply only for a STA interface
    ASSERT_ERR(vif->type == VIF_STA);

    // Set the parameters
    vif->u.sta.rssi_thold = param->rssi_thold;
    vif->u.sta.rssi_hyst = param->rssi_hyst;
    vif->u.sta.rssi_status = 0;

    return (KE_MSG_CONSUMED);
}

/*
 * TASK DESCRIPTOR DEFINITIONS
 ****************************************************************************************
 */
/// Message handlers in state DEFAULT.
const struct ke_msg_handler mm_default_state[] =
{
    // From UMAC
    {MM_START_REQ, (ke_msg_func_t)mm_start_req_handler},
    // From UMAC
    {MM_VERSION_REQ, (ke_msg_func_t)mm_version_req_handler},
    // From UMAC
    {MM_ADD_IF_REQ, (ke_msg_func_t)mm_hw_config_handler},
    // From UMAC
    {MM_REMOVE_IF_REQ, (ke_msg_func_t)mm_hw_config_handler},
    // From UMAC
    {MM_RESET_REQ, (ke_msg_func_t)mm_reset_req_handler},
    // From UMAC
    {MM_SET_CHANNEL_REQ, (ke_msg_func_t)mm_hw_config_handler},
    // From UMAC
    {MM_SET_BASIC_RATES_REQ, (ke_msg_func_t)mm_hw_config_handler},
    // From UMAC
    {MM_SET_BEACON_INT_REQ, (ke_msg_func_t)mm_hw_config_handler},
    // From UMAC
    {MM_SET_DTIM_REQ, (ke_msg_func_t)mm_hw_config_handler},
    // From UMAC
    {MM_SET_FILTER_REQ, (ke_msg_func_t)mm_hw_config_handler},
    // From UMAC
    {MM_SET_BSSID_REQ, (ke_msg_func_t)mm_hw_config_handler},
    // From UMAC
    {MM_SET_EDCA_REQ, (ke_msg_func_t)mm_set_edca_req_handler},
    // From UMAC
    {MM_SET_MU_EDCA_REQ, (ke_msg_func_t)mm_set_mu_edca_req_handler},
    // From UMAC
    {MM_SET_UORA_REQ, (ke_msg_func_t)mm_set_uora_req_handler},
    // From UMAC
    {MM_SET_BSS_COLOR_REQ, (ke_msg_func_t)mm_set_bss_color_req_handler},
    // From UMAC
    {MM_SET_TXOP_RTS_THRES_REQ, (ke_msg_func_t)mm_set_txop_rts_thres_req_handler},
    // From UMAC
    {MM_SET_SLOTTIME_REQ, (ke_msg_func_t)mm_hw_config_handler},
    // From UMAC
    {MM_SET_MODE_REQ, (ke_msg_func_t)mm_hw_config_handler},
    // From UMAC
    {MM_SET_VIF_STATE_REQ, (ke_msg_func_t)mm_hw_config_handler},
    // From UMAC
    {MM_DBG_TRIGGER_REQ, (ke_msg_func_t)mm_hw_config_handler},
    // From UMAC
    {MM_SET_IDLE_REQ, (ke_msg_func_t)mm_set_idle_req_handler},
    // From internal
    {MM_FORCE_IDLE_REQ, (ke_msg_func_t)mm_force_idle_req_handler},
    // From UMAC
    {MM_SET_POWER_REQ, (ke_msg_func_t)mm_set_power_req_handler},
    // From UMAC
    {MM_KEY_ADD_REQ, (ke_msg_func_t)mm_key_add_req_handler},
    // From UMAC
    {MM_KEY_DEL_REQ, (ke_msg_func_t)mm_key_del_req_handler},
    // From UMAC
    {MM_STA_ADD_REQ, (ke_msg_func_t)mm_sta_add_req_handler},
    // From UMAC
    {MM_STA_DEL_REQ, (ke_msg_func_t)mm_sta_del_req_handler},

    // From UMAC
    {MM_BA_ADD_REQ,  (ke_msg_func_t)mm_hw_config_handler},
    #if (NX_AMPDU_TX || NX_REORD)
    // From UMAC
    {MM_BA_DEL_REQ,  (ke_msg_func_t)mm_ba_del_req_handler},
    #endif

    #if (NX_CHNL_CTXT)
    #if (!NX_UMAC_PRESENT)
    // From UMAC
    {MM_CHAN_CTXT_ADD_REQ, (ke_msg_func_t)mm_chan_ctxt_add_req_handler},
    // From UMAC
    {MM_CHAN_CTXT_DEL_REQ,  (ke_msg_func_t)mm_chan_ctxt_del_req_handler},
    // From UMAC
    {MM_CHAN_CTXT_LINK_REQ, (ke_msg_func_t)mm_chan_ctxt_link_req_handler},
    // From UMAC
    {MM_CHAN_CTXT_SCHED_REQ,  (ke_msg_func_t)mm_chan_ctxt_sched_req_handler},
    #endif //(!NX_UMAC_PRESENT)
    // From UMAC
    {MM_CHAN_CTXT_UNLINK_REQ,  (ke_msg_func_t)mm_chan_ctxt_unlink_req_handler},
    // From UMAC
    {MM_CHAN_CTXT_UPDATE_REQ,  (ke_msg_func_t)mm_hw_config_handler},
    // From UMAC
    {MM_REMAIN_ON_CHANNEL_REQ,  (ke_msg_func_t)mm_remain_on_channel_req_handler},
    #endif //(NX_CHNL_CTXT)

    #if NX_BCN_AUTONOMOUS_TX
    // From UMAC
    {MM_BCN_CHANGE_REQ,  (ke_msg_func_t)mm_bcn_change_req_handler},
    // From UMAC
    {MM_TIM_UPDATE_REQ,  (ke_msg_func_t)mm_tim_update_req_handler},
    #endif

    #if NX_POWERSAVE
    // From UMAC
    {MM_SET_PS_MODE_REQ, (ke_msg_func_t)mm_set_ps_mode_req_handler},
    // From UMAC
    {MM_SET_PS_OPTIONS_REQ, (ke_msg_func_t)mm_set_ps_options_req_handler},
    #endif

    #if (NX_P2P_GO)
    // From UMAC
    {MM_SET_P2P_NOA_REQ,   (ke_msg_func_t)mm_set_p2p_noa_req_handler},
    // From UMAC
    {MM_SET_P2P_OPPPS_REQ, (ke_msg_func_t)mm_set_p2p_oppps_req_handler},
    #endif //(NX_P2P_GO)

    #if (RW_BFMER_EN)
    // From UMAC
    {MM_BFMER_ENABLE_REQ, (ke_msg_func_t)mm_bfmer_enable_req_handler},
    #endif //(RW_BFMER_EN)
    #if (RW_MUMIMO_TX_EN)
    // From UMAC
    {MM_MU_GROUP_UPDATE_REQ, (ke_msg_func_t)mm_mu_group_update_req_handler},
    #endif //(RW_MUMIMO_TX_EN)

    {MM_CFG_RSSI_REQ, (ke_msg_func_t)mm_cfg_rssi_req_handler},

    #if (NX_ANT_DIV)
    // From UMAC
    {MM_ANT_DIV_INIT_REQ, (ke_msg_func_t)mm_ant_div_init_req_handler},
    // From UMAC
    {MM_ANT_DIV_STOP_REQ, (ke_msg_func_t)mm_ant_div_stop_req_handler},
    // From UMAC
    {MM_ANT_DIV_UPDATE_REQ, (ke_msg_func_t)mm_ant_div_update_req_handler},
    // From UMAC or internal
    {MM_SWITCH_ANTENNA_REQ, (ke_msg_func_t)mm_hw_config_handler},
    #endif //(NX_ANT_DIV)
    #if NX_BEACONING
    {MM_TBTT_MOVE_REQ, (ke_msg_func_t)mm_hw_config_handler},
    #endif //(NX_BEACONING)
};

/// Specifies the message handler structure for every input state of STA State Machine
const struct ke_state_handler mm_state_handler[MM_STATE_MAX] =
{
    /// IDLE State message handlers.
    [MM_IDLE] = KE_STATE_HANDLER_NONE,
    /// JOIN State message handlers.
    [MM_ACTIVE] = KE_STATE_HANDLER_NONE,
};

/// Specifies the message handlers that are common to all states.
const struct ke_state_handler mm_default_handler =
    KE_STATE_HANDLER(mm_default_state);

/// Defines the placeholder for the states of all the task instances.
ke_state_t mm_state[MM_IDX_MAX];

/// @} end of group
