/**
 ****************************************************************************************
 * @file p2p.c
 *
 * @brief Wi-Fi Peer-to-Peer (P2P) Management
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup P2P
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */

#include "p2p.h"

#if (NX_P2P)

#include "mac_ie.h"
#include "ps.h"
#include "mm.h"
#include "mm_bcn.h"
#include "vif_mgmt.h"

#if (NX_P2P_GO)
#include "td.h"
#endif //(NX_P2P_GO)

/*
 * GLOBAL VARIABLES DECLARATION
 ****************************************************************************************
 */

#if (NX_P2P_GO)
struct p2p_env_tag p2p_env;
#endif //(NX_P2P_GO)

struct p2p_info_tag p2p_info_tab[NX_P2P_VIF_MAX];

/*
 * PRIVATE FUNCTIONS DECLARATION
 ****************************************************************************************
 */

__INLINE void p2p_go_noa_get_intv_cnt(struct p2p_noa_info_tag *noa);
static void p2p_cli_noa_cancel(struct p2p_noa_info_tag *noa);

/*
 * PRIVATE FUNCTIONS
 ****************************************************************************************
 */

/**
****************************************************************************************
 * @brief Convert local (P2P CLI) time to peer (P2P GO) time
 *
 * @param[in] vif         Vif pointer for the P2P CLI interface
 * @param[in] local_time  Local time to convert
 * @return Local time converted in Peer time
 ****************************************************************************************
 */
static inline uint32_t p2p_local_2_peer_time(struct vif_info_tag *vif, uint32_t local_time)
{
    ASSERT_ERR(vif->type == VIF_STA);
    return local_time + vif->u.sta.last_tsf_offset;
}

/**
****************************************************************************************
 * @brief Convert peer (P2P GO) time to local (P2P CLI) time
 *
 * @param[in] vif        Vif pointer for the P2P CLI interface
 * @param[in] peer_time  Peer time to convert
 * @return Peer time converted in Local time
 ****************************************************************************************
 */
static inline uint32_t p2p_peer_2_local_time(struct vif_info_tag *vif, uint32_t peer_time)
{
    ASSERT_ERR(vif->type == VIF_STA);
    return peer_time - vif->u.sta.last_tsf_offset;
}

/**
 ****************************************************************************************
 * @brief Look for specific attribute inside a provided information element (IE).
 *  For example, can be used to find Notice of Absence attribute inside P2P IE.
 *
 * @param[in]  ie_addr       IE Start Address
 * @param[in]  ie_total_len  IE Total Length
 * @param[in]  att_id        Attribute ID to look for
 *
 * @return  Start address of the attribute if found, else 0.
 ****************************************************************************************
 */
static uint32_t p2p_att_find(uint32_t ie_addr, uint16_t ie_total_len, uint8_t att_id)
{
    uint32_t ie_end = ie_addr + ie_total_len;

    // Loop as long as we do not go beyond the frame size
    while (ie_addr < ie_end)
    {
        // Check if the current Attrbute ID is the one looked for
        if (att_id == co_read8p(ie_addr))
        {
            // The Attribute ID matches, return the address of the element
            return (ie_addr);
        }

        // Move on to the next attribute
        ie_addr += co_read16p(ie_addr + P2P_ATT_LEN_OFT) + P2P_ATT_BODY_OFT;
    }

    return (0);
}

/**
 ****************************************************************************************
 * @brief Callback called once system entered in IDLE state before peer device absence.
 ****************************************************************************************
 */
static void p2p_goto_idle_cb(void)
{
    TRACE_P2P(PS, "HW is now in IDLE, can flush TX/RX queues");

    // Flush all the TX and RX queues
    mm_force_idle_req();

    // Go back to ACTIVE state
    mm_active();
}

/**
 ****************************************************************************************
 * @brief Check if at least one NOA absence is currently in progress.
 * Also return the lowest NOA count.
 *
 * @param[in]     p2p               P2P Entry
 * @param[out] low_noa_count     Lower found NOA count value
 *
 * @return true if at least NOA currently in its absence period is found, and false
 * otherwise.
 ****************************************************************************************
 */
static bool p2p_noa_wait_end_abs(struct p2p_info_tag *p2p, uint8_t *low_noa_count)
{
    uint8_t noa_idx;
    bool wait_end_abs = false;
    *low_noa_count = 255;

    for (noa_idx = 0; noa_idx < P2P_NOA_NB_MAX; noa_idx++)
    {
        struct p2p_noa_info_tag *noa = &p2p->noa[noa_idx];
        if ((noa->noa_status == P2P_NOA_TIMER_WAIT_END_ABS)
            #if (NX_P2P_GO)
            && !(p2p->noa_paused && noa->dyn_noa && (noa->noa_type == P2P_NOA_TYPE_NORMAL))
            #endif //(NX_P2P_GO)
            )
        {
            wait_end_abs = true;
            if (noa->noa_init_counter < *low_noa_count)
            {
                *low_noa_count = noa->noa_init_counter;
            }
        }
    }

    return (wait_end_abs);
}

/**
 ****************************************************************************************
 * @brief Update P2P GO next absence and presence date.
 *
 * Using NOA configured, this function updates the date of the next absence and presence
 * of the p2p GO interface.
 * This function is called when NOA is started/stopped or when NOA status is updated.
 *
 * @param[in]  p2p  P2P Entry
 ****************************************************************************************
 */
static void p2p_update_noa_presence_date(struct p2p_info_tag *p2p)
{
    bool first_noa = true;
    bool no_next_absent = true;
    uint32_t in_progress = 0;
    uint32_t next_present = 0;
    uint32_t next_absent = 0;
    struct p2p_noa_info_tag *noa;
    uint32_t absent, present;

    // First pass only active NOA with counter == 1
    for (noa = &p2p->noa[0]; noa < &p2p->noa[P2P_NOA_NB_MAX]; noa++)
    {
        if ((noa->noa_status != P2P_NOA_TIMER_WAIT_END_ABS) || (noa->noa_counter > 1)
            #if (NX_P2P_GO)
            || (p2p->noa_paused && noa->dyn_noa && (noa->noa_type == P2P_NOA_TYPE_NORMAL))
            #endif //(NX_P2P_GO)
            )
            continue;

        present = noa->noa_timer.time;

        if (first_noa)
        {
            next_present = present;
            first_noa = false;
        }
        else if (hal_machw_time_cmp(next_present, present))
        {
            next_present = present;
        }
        in_progress++;
    }

    // second pass only active periodic NOA
    for (noa = &p2p->noa[0]; noa < &p2p->noa[P2P_NOA_NB_MAX]; noa++)
    {
        if ((noa->noa_status != P2P_NOA_TIMER_WAIT_END_ABS) || (noa->noa_counter <= 1)
            #if (NX_P2P_GO)
            || (p2p->noa_paused && noa->dyn_noa && (noa->noa_type == P2P_NOA_TYPE_NORMAL))
            #endif //(NX_P2P_GO)
            )
            continue;

        present = noa->noa_timer.time;
        absent = present + noa->noa_intv_us - noa->noa_dur_us;

        if (first_noa)
        {
            next_present = present;
            next_absent = absent;
            no_next_absent = false;
            first_noa = false;
        }
        else
        {
            int counter = noa->noa_counter;
            while (hal_machw_time_cmp(absent, next_present) && --counter)
            {
                absent += noa->noa_intv_us;
                present += noa->noa_intv_us;
            }
            if (!counter)
                continue;

            if (hal_machw_time_cmp(next_present, present))
                next_present = present;

            if ((counter > 1) &&
                (no_next_absent || hal_machw_time_cmp(absent, next_absent)))
            {
                next_absent = absent;
                no_next_absent = false;
            }
        }
        in_progress++;
    }

    // last pass non active NOA
    for (noa = &p2p->noa[0]; noa < &p2p->noa[P2P_NOA_NB_MAX]; noa++)
    {
        if (noa->noa_status != P2P_NOA_TIMER_WAIT_NEXT_ABS)
            continue;

        absent = noa->noa_timer.time;
        present = absent + noa->noa_dur_us;

        if (first_noa)
        {
            next_present = present;
            next_absent = absent;
            no_next_absent = false;
            first_noa = false;
        }
        else if (in_progress)
        {
            // Merge active NOA and non active NOA
            int counter = noa->noa_counter;
            while (hal_machw_time_cmp(present, next_present) && counter--)
            {
                absent += noa->noa_intv_us;
                present += noa->noa_intv_us;
            }
            if (counter < 0)
                continue;

            if (no_next_absent)
            {
                if (hal_machw_time_cmp(next_present, absent))
                {
                    next_absent = absent;
                    no_next_absent = false;
                }
                else
                {
                    next_present = present;
                    if (counter > 0)
                    {
                        next_absent = absent + noa->noa_intv_us;
                        no_next_absent = false;
                    }
                }
            }
            else if (hal_machw_time_cmp(next_absent, absent))
            {
                // nothing to update
            }
            else if (hal_machw_time_cmp(next_present, absent))
            {
                next_absent = absent;
            }
            else if (hal_machw_time_cmp(present, next_absent))
            {
                next_present = present;
                absent += noa->noa_intv_us;
                if ((counter > 1) &&
                    hal_machw_time_cmp(absent, next_absent))
                    next_absent = absent;
            }
            else
            {
                // This NOA overlap current [next_present .. next_absent] interval
                // Would need a new pass to find correct next_present/next_absent
                ASSERT_WARN(0);
                next_present = present;
                next_absent = absent + noa->noa_intv_us;
            }
        }
        else
        {
            // Merge only non active NOA.
            if (hal_machw_time_cmp(present, next_absent))
            {
                // This NOA ends before current [next_absent .. next_present] interval
                next_present = present;
                next_absent = absent;
            }
            else if (hal_machw_time_cmp(next_present, absent))
            {
                // nothing to update, as this NOA starts after curretn interval
            }
            else
            {
                // NOA overlap with current [next_absent .. next_present] interval
                if (hal_machw_time_cmp(absent, next_absent))
                    next_absent = absent;
                if (hal_machw_time_cmp(next_present, present))
                    next_present = present;
            }
        }
    }

    p2p->next_present = next_present;
    if (no_next_absent)
        p2p->next_absent = next_present + 10;
    else
        p2p->next_absent = next_absent;
}

/**
 ****************************************************************************************
 * @brief Update Power Save status of the GO.
 *
 * Update the Power Save status of the GO in the P2P connection.
 * This function can be called on the client or the GO interface.
 *
 * @param[in] p2p  P2P Entry for which power state must be updated.
 ****************************************************************************************
 */
static void p2p_update_go_ps_state(struct p2p_info_tag *p2p)
{
    bool is_go_present = true;
    bool tbtt_in_noa = false;
    uint8_t low_noa_count;

    if (p2p_noa_wait_end_abs(p2p, &low_noa_count))
    {
        // At least one NOA is in progress,
        // The order of precedence for determining P2P Group Owner power save state shall
        // be as follow:
        //   Highest 1 - Absence due to a non-periodic Notice of Absence (Count = 1)
        //           2 - Presence from TBTT until the end of Beacon frame transmission
        //           3 - Presence during the CTWindow
        //    Lowest 4 - Absence for a periodic Notice of Absence (Count > 1)
        if ((low_noa_count == 1) ||                                // non-periodic NoA or
            (!p2p->is_waiting_bcn &&                               // ( not in TBTT and
             (p2p->oppps_status != P2P_OPPPS_TIMER_WAIT_END_CTW))) //   not in CTwindow )
        {
            is_go_present = false;
        }
        else if (p2p->is_waiting_bcn)
        {
            tbtt_in_noa = true;
        }
    }
    else
    {
        // No NoA is progress, still GO may be absent if oppPS is enabled
        do
        {
            // Check if Opportunistic PS is supported
            if (p2p->oppps_ctw == 0)
            {
                // Go is present
                break;
            }

            // Check if the CTWindow is opened
            if (p2p->oppps_status == P2P_OPPPS_TIMER_WAIT_END_CTW)
            {
                // Go is present
                break;
            }

            #if (NX_P2P_GO)
            if (p2p->role == P2P_ROLE_GO)
            {
                // As GO we now have to check if all peer P2P client are in sleep mode.
                // In that case, we can enter in sleep mode until the next TBTT.
                struct vif_info_tag *vif = &vif_info_tab[p2p->vif_index];

                if (vif->u.ap.ps_sta_cnt != vif->p2p_link_nb)
                {
                     break;
                }
            }
            else
            #endif //(NX_P2P_GO)
            {
                // As CLI, consider the GO as absent only if PS mode is ON and not PAUSED
                #if (NX_POWERSAVE)
                if (!ps_env.ps_on || (ps_env.prevent_sleep & PS_PSM_PAUSED))
                {
                    break;
                }
                #endif //(NX_POWERSAVE)
            }

            is_go_present = false;
        } while (0);
    }

    // Check if presence status has been modified
    if (is_go_present != p2p->is_go_present)
    {
        #if (NX_CHNL_CTXT)
        struct vif_info_tag *vif = &vif_info_tab[p2p->vif_index];
        #endif //(NX_CHNL_CTXT)

        // We have to inform the driver in order to avoid sending of data during absence
        struct mm_p2p_vif_ps_change_ind *ind = KE_MSG_ALLOC(MM_P2P_VIF_PS_CHANGE_IND,
                                                              TASK_API, TASK_MM,
                                                              mm_p2p_vif_ps_change_ind);

        ind->vif_index = p2p->vif_index;
        ind->ps_state  = (is_go_present) ? PS_MODE_OFF : PS_MODE_ON;

        TRACE_P2P(NOA_DET, "{VIF-%d} GO is present = %d", p2p->vif_index, is_go_present);

        // Send the indication
        ke_msg_send(ind);

        // Store new status
        p2p->is_go_present = is_go_present;

        #if (NX_CHNL_CTXT)
        // Notify chan module about the P2P state update
        chan_p2p_absence_update(vif->chan_ctxt, !is_go_present);
        #endif //(NX_CHNL_CTXT)

        #if (NX_UAPSD)
        // Notify the PS module about the P2P state update
        ps_p2p_absence_update(vif, !is_go_present);
        #endif //(NX_UAPSD)

        if (!is_go_present)
        {
            PROF_P2P_ABSENCE_SET();

            // Update prevent_sleep state
            #if (NX_P2P_GO)
            if (vif->type == VIF_AP)
            {
                #if (NX_POWERSAVE)
                // Allow to enter in doze mode
                vif->prevent_sleep &= ~PS_VIF_P2P_GO_PRESENT;
                #endif //(NX_POWERSAVE)
                if (mm_bcn_transmitting())
                {
                    GLOBAL_INT_DISABLE();
                    txl_cntrl_halt_ac(AC_BCN);
                    txl_cntrl_flush_ac(AC_BCN, DESC_DONE_SW_TX_BIT);
                    GLOBAL_INT_RESTORE();
                }
            }
            #endif //(NX_P2P_GO)

            // Require to enter to idle state in order to flush current TX packets
            struct mm_force_idle_req *req = KE_MSG_ALLOC(MM_FORCE_IDLE_REQ,
                                                         TASK_MM, TASK_NONE,
                                                         mm_force_idle_req);

            // Set the request parameters
            req->cb = p2p_goto_idle_cb;

            // Send the request
            ke_msg_send(req);
        }
        else
        {
            PROF_P2P_ABSENCE_CLR();

            // Update prevent_sleep state
            #if (NX_P2P_GO && NX_POWERSAVE)
            if (vif->type == VIF_AP)
            {
                // Disallow to enter in doze mode
                vif->prevent_sleep |= PS_VIF_P2P_GO_PRESENT;
            }
            #endif //(NX_P2P_GO && NX_POWERSAVE)
        }
    }

    if ((p2p->role == P2P_ROLE_CLIENT) && is_go_present && !tbtt_in_noa)
    {
        // GO is present, send postponed traffic unless it is present only to send its beacon.
        vif_mgmt_send_postponed_frame(&vif_info_tab[p2p->vif_index]);
    }
}

/**
 ****************************************************************************************
 * @brief Program the NOA Timer for a given P2P entry.
 *        Call @ref p2p_noa_timer_end function once timer expires.
 *
 * @param[in]  noa          P2P NOA Entry for which the timer is programmed
 * @param[in]  time         NOA Timer Expiration Time
 ****************************************************************************************
 */
static void p2p_noa_timer_prog(struct p2p_noa_info_tag *noa, uint32_t time)
{
    TRACE_P2P(NOA_DET, "{VIF-%d} Set NOA-%d timer %t",
              p2p_info_tab[noa->p2p_index].vif_index, noa->noa_inst, TR_32(time));

    mm_timer_set(&noa->noa_timer, time);
}

/**
 ****************************************************************************************
 * @brief Function called upon NOA timer expiration. Handle the timer state machine
 *        update and computation of next timer expiration time.
 *
 * @param[in]  env    P2P Entry for which timer was programmed.
 ****************************************************************************************
 */
static void p2p_noa_timer_end(void *env)
{
    struct p2p_noa_info_tag *noa = (struct p2p_noa_info_tag *)env;
    struct p2p_info_tag *p2p = &p2p_info_tab[noa->p2p_index];
    struct vif_info_tag *vif = &vif_info_tab[p2p->vif_index];
    uint8_t next_noa_status = noa->noa_status;
    uint32_t next_time;

    TRACE_P2P(NOA_DET, "{VIF-%d} NOA-%d timer expire status=%d",
              p2p->vif_index, noa->noa_inst, noa->noa_status);

    #if NX_P2P_GO
    if (p2p->role == P2P_ROLE_GO)
    {
        // Directly use timer time
        next_time = noa->noa_timer.time;
    }
    else
    #endif /* NX_P2P_GO */
    {
        // Get time based on peer TSF value.
        next_time = p2p_peer_2_local_time(vif, noa->peer_next_noa_time);
    }

    // Loop in order to avoid programming in the past
    do
    {
        // React based on the current NOA status
        switch (next_noa_status)
        {
            case (P2P_NOA_TIMER_WAIT_NEXT_ABS):
            {
                PROF_P2P_NOA_ABS_SET(noa->noa_inst);

                next_noa_status = P2P_NOA_TIMER_WAIT_END_ABS;
                next_time      += (noa->noa_dur_us);
            } break;

            case (P2P_NOA_TIMER_WAIT_END_ABS):
            {
                PROF_P2P_NOA_ABS_CLR(noa->noa_inst);

                // Decrement number of remaining absences if not continuous
                if (noa->noa_counter != P2P_NOA_CONTINUOUS_COUNTER)
                {
                    noa->noa_counter--;
                }

                // Check if another absence is scheduled
                if (noa->noa_counter)
                {
                    next_noa_status = P2P_NOA_TIMER_WAIT_NEXT_ABS;
                    next_time      += (noa->noa_intv_us - noa->noa_dur_us);

                    #if (NX_P2P_GO)
                    // If GO, update start time
                    if ((p2p->role == P2P_ROLE_GO) && p2p->is_noa_bcn)
                    {
                        // Next start time
                        noa->noa_start_time += noa->noa_intv_us;
                        // Decrement start time update counter
                        noa->noa_time_upd_cnt--;

                        if (!noa->noa_time_upd_cnt)
                        {
                            // Update NOA Init Counter
                            noa->noa_init_counter = noa->noa_counter;

                            // Recompute start time update counter
                            p2p_go_noa_get_intv_cnt(noa);

                            // Update the beacon
                            mm_bcn_update_p2p_noa(vif->index, P2P_BCN_UPD_OP_NOA_UPD);
                        }
                    }
                    #endif //(NX_P2P_GO)
                }
                else
                {
                    next_noa_status = P2P_NOA_TIMER_NOT_STARTED;

                    #if (NX_P2P_GO)
                    // Stop the NOA
                    if (p2p->role == P2P_ROLE_GO)
                    {
                        p2p_go_noa_stop(vif, noa->noa_inst, false);
                    }
                    else
                    #endif //(NX_P2P_GO)
                    {
                        p2p_cli_noa_cancel(noa);
                    }
                }
            } break;

            default:
            {
                // Should not happen
                ASSERT_ERR(0);
            }
        }
    } while (hal_machw_time_cmp(next_time, ke_time() + P2P_NOA_TIMER_MARGIN) &&
             (next_noa_status != P2P_NOA_TIMER_NOT_STARTED));

    // Reprogram NOA timer if needed
    if (next_noa_status != P2P_NOA_TIMER_NOT_STARTED)
    {
        if (p2p->role == P2P_ROLE_CLIENT)
        {
            // Keep the next time based on peer TSF
            noa->peer_next_noa_time = p2p_local_2_peer_time(vif, next_time);
        }

        p2p_noa_timer_prog(noa, next_time);

        // Update the NOA status
        noa->noa_status = next_noa_status;
    }

    p2p_update_noa_presence_date(p2p);

    // Update GO presence status
    p2p_update_go_ps_state(p2p);

    #if (NX_P2P_GO && NX_POWERSAVE)
    /*
     * When doze mode is used while we are AP, TBTT interrupts are no more generated.
     * Hence to be sure to not be in Doze mode at this time, set P2P_GO_PRESENT bit at the
     * end of the NOA absence (occurs few us before the interrupt)
     */
    if ((vif->type == VIF_AP) &&
        (next_noa_status == P2P_NOA_TIMER_WAIT_NEXT_ABS))
    {
        vif->prevent_sleep |= PS_VIF_P2P_GO_PRESENT;
    }
    #endif //(NX_P2P_GO && NX_POWERSAVE)
}

/**
 ****************************************************************************************
 * @brief Callback for Opportunistic Power Save timer.
 *
 * Called when oppPS timer expire, for for P2P client interface.
 * The timer is started after reception of the beacon, if the GO is advertising oppPS.
 * The duration of the timer, aka the CTWindow, is given by the GO. At the end on the
 * CTWindow the GO may enter power save mode, so it is necessary to update it status.
 *
 * @param[in] env   P2P Entry for which timer was programmed.
 *
 ****************************************************************************************
 */
static void p2p_oppps_timer_end(void *env)
{
    struct p2p_info_tag *p2p = (struct p2p_info_tag *)env;

    TRACE_P2P(PS, "{VIF-%d} oppPS timer end status=%d", p2p->vif_index, p2p->oppps_status);

    if (p2p->oppps_status == P2P_OPPPS_TIMER_WAIT_END_CTW)
    {
        PROF_P2P_CTW_CLR();
        p2p->oppps_status = P2P_OPPPS_TIMER_NOT_STARTED;
        p2p_update_go_ps_state(p2p);
    }
    else if (p2p->oppps_status == P2P_OPPPS_TIMER_WAIT_START_CTW)
    {
        PROF_P2P_CTW_SET();
        p2p->oppps_status = P2P_OPPPS_TIMER_WAIT_END_CTW;
        mm_timer_set(&p2p->oppps_timer, p2p->oppps_ctw_end);
        p2p_update_go_ps_state(p2p);
    }
}

#if (NX_P2P_GO)
/**
 ****************************************************************************************
 * @brief Send a MM_P2P_NOA_UPD_IND to the host so that it can maintain a status about
 * the NoA scheme currently applied locally.
 *
 * @param[in] p2p               P2P Entry
 * @param[in] noa_instance      NoA Instance Updated
 ****************************************************************************************
 */
static void p2p_go_send_noa_upd_ind(struct p2p_info_tag *p2p, uint8_t noa_instance)
{
    struct p2p_noa_info_tag *noa = &p2p->noa[noa_instance];
    struct mm_p2p_noa_upd_ind *ind = KE_MSG_ALLOC(MM_P2P_NOA_UPD_IND, TASK_API,
                                                    TASK_MM, mm_p2p_noa_upd_ind);

    ind->vif_index = p2p->vif_index;
    ind->noa_inst_nb = noa_instance;
    ind->count = (noa->noa_status != P2P_NOA_TIMER_NOT_STARTED) ? noa->noa_init_counter : 0;

    if (ind->count)
    {
        ind->noa_type = noa->noa_type;
        ind->duration_us = noa->noa_dur_us;
        ind->interval_us = noa->noa_intv_us;
        ind->start_time = noa->noa_start_time;
    }

    // Send the message
    ke_msg_send(ind);
}

/**
 ****************************************************************************************
 * @brief Get the number of NOA timer currently configured
 *
 * @param[in] p2p P2P Entry (necessarily a GO).
 *
 * @return The number of NOA configured for the P2P entry
 ****************************************************************************************
 */
static uint8_t p2p_go_get_noa_inst(struct p2p_info_tag *p2p)
{
    uint8_t noa_idx;

    for (noa_idx = 0; noa_idx < P2P_NOA_NB_MAX; noa_idx++)
    {
        struct p2p_noa_info_tag *noa = &p2p->noa[noa_idx];

        // Check if the NOA timer is active
        if (noa->noa_status == P2P_NOA_TIMER_NOT_STARTED)
        {
            break;
        }
    }

    return (noa_idx);
}

/**
 ****************************************************************************************
 * @brief Compute the number of NOA absence periods before updating Start Time field
 *        in NOA attribute.
 *
 * Do nothing for non-recursive NOA
 *
 * @param[in,out] noa P2P NOA Entry, update the value of noa_time_upd_cnt
 ****************************************************************************************
 */
__INLINE void p2p_go_noa_get_intv_cnt(struct p2p_noa_info_tag *noa)
{
    if (noa->noa_intv_us == 0)
        return;

    // "The P2P Group Owner shall update the Start Time field [...] every 2^31us"
    // Compute number of absences before next start time update.
    // Update the start time one absence before.
    noa->noa_time_upd_cnt = ((1 << 31) / noa->noa_intv_us) - 1;
}
#endif //(NX_P2P_GO)

/**
 ****************************************************************************************
 * @brief A frame containing a Notice of Absence attribute can be received after the
 * indicated start time.
 * Hence we have to compute how many absences have been missed since start time in order
 * to properly program the next absence time.
 *
 * @param[in]  noa         NOA Entry
 * @param[in]  vif_index   Vif index
 * @param[in]  start_time  Start Time value read in the NOA attribute (4 lower bytes of TSF)
 * @param[in]  tsf_peer    TSF value provided by the peer device
 *
 * @return Peer device next absence time
 ****************************************************************************************
 */
static uint32_t p2p_cli_noa_get_abs_time(struct p2p_noa_info_tag *noa, uint8_t vif_index,
                                         uint32_t start_time, uint32_t tsf_peer)
{
    struct vif_info_tag *vif = &vif_info_tab[vif_index];
    uint32_t absence_time;
    uint32_t missed_intv;

    if (tsf_peer >= start_time)
    {
        // If only one absence is scheduled it is too late.
        if (noa->noa_counter == 1)
        {
            return (0);
        }

        // Need to know how many absences have been missed since start_time (on peer side)
        missed_intv = ((tsf_peer - start_time) / noa->noa_intv_us) + 1;
    }
    else
    {
        missed_intv = 0;
    }

    TRACE_P2P(NOA_DET, "{VIF-%d} NoA startime=%ld, Peer's TSF=%ld => miss %d iteration",
              vif->index, TR_32(start_time), TR_32(tsf_peer), missed_intv);

    // Compute next absence time in Peer and local Time
    noa->peer_next_noa_time = start_time + (missed_intv * noa->noa_intv_us);
    absence_time = p2p_peer_2_local_time(vif, noa->peer_next_noa_time);

    if (noa->noa_counter != P2P_NOA_CONTINUOUS_COUNTER)
    {
        noa->noa_counter -= co_min(missed_intv, noa->noa_counter);
    }

    // Check that computed time is not in the past
    while (hal_machw_time_past(absence_time) && noa->noa_counter)
    {
        absence_time += noa->noa_intv_us;

        if (noa->noa_counter != P2P_NOA_CONTINUOUS_COUNTER)
        {
            // Decrease number of remaining absences
            noa->noa_counter--;
        }
    }

    if (!noa->noa_counter)
    {
        // No more absence are coming
        return (0);
    }

    TRACE_P2P(NOA_DET, "Next NOA iteration starts %t (TSF offset=%ld)",
              TR_32(absence_time), TR_32(vif->u.sta.last_tsf_offset));

    return (absence_time);
}

/**
 ****************************************************************************************
 * @brief Start NOA procedure on P2P client side.
 *
 * @param[in]  noa          NOA Entry
 * @param[in]  vif_index    VIF index
 * @param[in]  start_time   NOA Start Time value read in the received Beacon
 * @param[in]  tsf_peer     Peer TSF at Beacon Transmission
 *
 * @return True if NOA has been started, false otherwise.
 ****************************************************************************************
 */
static bool p2p_cli_noa_start(struct p2p_noa_info_tag *noa, uint8_t vif_index,
                              uint32_t start_time, uint32_t tsf_peer)
{
    uint32_t first_abs_time;
    TRACE_P2P(NOA, "{VIF-%d} start NOA-%d: count=%d intv=%ld us, dur=%ld us",
              vif_index, noa->noa_inst, noa->noa_counter,
              TR_32(noa->noa_intv_us), TR_32(noa->noa_dur_us));

    first_abs_time = p2p_cli_noa_get_abs_time(noa, vif_index, start_time, tsf_peer);
    if (first_abs_time)
    {
        // Program the noa timer with the computed time
        p2p_noa_timer_prog(noa, first_abs_time);
        // And Update the NOA status
        noa->noa_status = P2P_NOA_TIMER_WAIT_NEXT_ABS;
    }

    return ((bool)first_abs_time);
}

/**
 ****************************************************************************************
 * @brief Cancel NOA procedure on P2P client side.
 *
 * @param[in]  noa  NOA Entry
 ****************************************************************************************
 */
static void p2p_cli_noa_cancel(struct p2p_noa_info_tag *noa)
{
    TRACE_P2P(NOA, "{VIF-%d} Cancel NOA-%d",
              p2p_info_tab[noa->p2p_index].vif_index, noa->noa_inst);

    if (noa->noa_status != P2P_NOA_TIMER_NOT_STARTED)
    {
        // Stop NOA timer
        mm_timer_clear(&noa->noa_timer);

        // Reset NOA status
        noa->noa_status = P2P_NOA_TIMER_NOT_STARTED;

        PROF_P2P_NOA_ABS_CLR(noa->noa_inst);
    }
}

/**
 ****************************************************************************************
 * @brief Handle the received NOA attribute.
 *
 * @param[in]  p2p_index    Index of the P2P entry for which NOA attribute has been found
 * @param[in]  a_noa_att    Address of received NOA attribute
 * @param[in]  tsf_peer     Peer TSF at Beacon Transmission
 ****************************************************************************************
 */
static void p2p_cli_noa_handle_att(uint8_t p2p_index,
                                   uint32_t a_noa_att,
                                   uint32_t tsf_peer)
{
    struct p2p_info_tag *p2p = &p2p_info_tab[p2p_index];
    uint8_t noa_idx;

    TRACE_P2P(NOA_DET, "{VIF-%d} NOA attribute present=%d", p2p->vif_index, (a_noa_att != 0));

    if (a_noa_att)
    {
        uint8_t nb_noa_desc;
        uint8_t noa_index = co_read8p(a_noa_att + P2P_NOA_ATT_INDEX_OFT);
        uint16_t length;
        uint8_t ctw_oppps;

        TRACE_P2P(NOA_DET, "index=%d, last index=%d", noa_index, p2p->noa_index);

        // Check if a one of the NOA fields has been updated since last beacon
        if (p2p->is_noa_bcn && (p2p->noa_index == noa_index))
        {
            // Nothing changed, skip this one
            return;
        }

        // Keep in mind that NOA attribute was part of last received Beacon
        p2p->is_noa_bcn = true;
        p2p->noa_index = noa_index;

        // Check if a NOA procedure is already known and in progress
        if (p2p->noa_nb)
        {
            for (noa_idx = 0; noa_idx < P2P_NOA_NB_MAX; noa_idx++)
            {
                // Cancel NOA procedure
                p2p_cli_noa_cancel(&p2p->noa[noa_idx]);
                p2p->noa_nb = 0;
            }

            // Update Go PS state
            p2p_update_go_ps_state(p2p);
        }

        // Compute number of NoA Descriptor from length of NoA attribute
        length = co_read16p(a_noa_att + P2P_ATT_LEN_OFT);
        nb_noa_desc = (length - 2) / P2P_NOA_DESC_LENGTH;

        for (noa_idx = 0; noa_idx < nb_noa_desc; noa_idx++)
        {
            struct p2p_noa_info_tag *noa = &p2p->noa[noa_idx];
            // Get address of NoA descriptor in the memory
            uint32_t a_noa_desc = a_noa_att + P2P_NOA_ATT_NOA_DESC_OFT
                + (noa_idx * P2P_NOA_DESC_LENGTH);

            // Extract NOA Counter
            noa->noa_init_counter = co_read8p(a_noa_desc + P2P_NOA_DESC_COUNT_OFT);

            // Check if GO is using NOA
            if (noa->noa_init_counter != 0)
            {
                // Read absence interval
                uint32_t noa_intv = co_read32p(a_noa_desc + P2P_NOA_DESC_INTV_OFT);

                // Verify that interval is not 0 if count different than 1
                if ((noa->noa_init_counter > 1) && !noa_intv)
                {
                    continue;
                }

                // Extract NOA parameters
                noa->noa_dur_us  = co_read32p(a_noa_desc + P2P_NOA_DESC_DUR_OFT);
                noa->noa_intv_us = noa_intv;
                noa->noa_counter = noa->noa_init_counter;

                // Restart NOA procedure
                if (p2p_cli_noa_start(noa, p2p->vif_index,
                                      co_read32p(a_noa_desc + P2P_NOA_DESC_START_OFT),
                                      tsf_peer))
                {
                    p2p->noa_nb++;
                }
            }
        }

        ctw_oppps = co_read8p(a_noa_att + P2P_NOA_ATT_CTW_OPPPS_OFT);

        // Check if GO is using OppPS
        if ((ctw_oppps & P2P_OPPPS_MASK) == P2P_OPPPS_MASK)
        {
            // Extract CTWindow
            p2p->oppps_ctw = ctw_oppps & P2P_CTWINDOW_MASK;
            TRACE_P2P(PS, "{VIF-%d} P2P GO configured a CTW of %d TU",
                      p2p->vif_index, p2p->oppps_ctw)
        }
        else
        {
            // 0 value will indicate that the OppPS is not used
            p2p->oppps_ctw = 0;
        }
    }
    else
    {
        p2p->is_noa_bcn = false;

        // Check if a NOA procedure is already known and in progress
        if (p2p->noa_nb)
        {
            for (noa_idx = 0; noa_idx < P2P_NOA_NB_MAX; noa_idx++)
            {
                // Cancel NOA procedure
                p2p_cli_noa_cancel(&p2p->noa[noa_idx]);
                p2p->noa_nb = 0;
            }

            p2p_update_go_ps_state(p2p);
        }

        // Stop OppPS procedure
        p2p->oppps_ctw = 0;
    }
}

/*
 * PUBLIC FUNCTIONS
 ****************************************************************************************
 */

void p2p_init(void)
{
    uint8_t counter;

    TRACE_P2P(ITF, "P2P init");

    #if (NX_P2P_GO)
    memset(&p2p_env, 0, sizeof(struct p2p_env_tag));
    #endif //(NX_P2P_GO)

    // Initialize all vif_index to INVALID_VIF_IDX in order to indicate that the p2p
    // structure is unused
    for (counter = 0; counter < NX_P2P_VIF_MAX; counter++)
    {
        struct p2p_info_tag *p2p = &p2p_info_tab[counter];
        p2p->vif_index = INVALID_VIF_IDX;
    }
}

uint8_t p2p_create(uint8_t vif_index, uint8_t role)
{
    uint8_t p2p_index = P2P_INVALID_IDX;
    uint8_t p2p_idx;

    // Look for an available P2P Entry structure
    for (p2p_idx = 0; p2p_idx < NX_P2P_VIF_MAX; p2p_idx++)
    {
        struct p2p_info_tag *p2p = &p2p_info_tab[p2p_idx];

        if (p2p->vif_index == INVALID_VIF_IDX)
        {
            uint8_t noa_idx;

            // Initialize content
            memset(p2p, 0, sizeof(struct p2p_info_tag));

            // Return current index
            p2p_index = p2p_idx;

            // Store the provided vif index and the role
            p2p->vif_index = vif_index;
            p2p->role      = role;

            #if (NX_P2P_GO)
            if (p2p->role == P2P_ROLE_GO)
            {
                p2p_env.nb_p2p_go++;
            }
            #endif //(NX_P2P_GO)

            // Pre-configure NOA timers
            for (noa_idx = 0; noa_idx < P2P_NOA_NB_MAX; noa_idx++)
            {
                struct p2p_noa_info_tag *noa = &p2p->noa[noa_idx];

                noa->noa_timer.cb  = p2p_noa_timer_end;
                noa->noa_timer.env = noa;
                noa->p2p_index = p2p_idx;
                noa->noa_inst = noa_idx;
            }

            // Pre-configure OPPPS timer
            p2p->oppps_timer.cb  = p2p_oppps_timer_end;
            p2p->oppps_timer.env = p2p;

            // Initiate GO present status, will be present (no noa or oppps yet)
            p2p_update_go_ps_state(p2p);

            // Stop looping
            break;
        }
    }

    TRACE_P2P(ITF, "CREATE: {VIF-%d} role=%d, p2p_idx=%d", vif_index, role, p2p_index);

    return (p2p_index);
}

void p2p_cancel(uint8_t p2p_index, bool vif_del)
{
    struct p2p_info_tag *p2p = &p2p_info_tab[p2p_index];

    TRACE_P2P(ITF, "DELETE: {VIF-%d} p2p_idx=%d", p2p->vif_index, p2p_index);

    // Verify that the structure is used
    if (p2p->vif_index != INVALID_VIF_IDX)
    {
        // Cancel Beacon Timeout Timer if needed
        if (p2p->is_waiting_bcn)
        {
            PROF_P2P_WAIT_BCN_CLR();
        }

        #if (NX_P2P_GO)
        if (p2p->role == P2P_ROLE_GO)
        {
            p2p_env.nb_p2p_go--;
        }
        #endif //(NX_P2P_GO)

        if (vif_del)
        {
            // Reset VIF index
            p2p->vif_index = INVALID_VIF_IDX;
        }
    }
}

void p2p_set_vif_state(struct vif_info_tag *vif, bool active)
{
    struct p2p_info_tag *p2p;

    if (!vif->p2p)
    {
        return;
    }

    TRACE_P2P(ITF, "{VIF-%d} active=%d", vif->index, active);

    p2p = &p2p_info_tab[vif->p2p_index];

    if (vif->type == VIF_STA)
    {
        if (!active)
        {
            for (int noa_idx = 0; noa_idx < P2P_NOA_NB_MAX; noa_idx++)
            {
                p2p_cli_noa_cancel(&p2p->noa[noa_idx]);
            }

            p2p->oppps_ctw = 0;
            p2p->noa_index = 0;
            p2p->is_noa_bcn = false;

            // Cancel OppPS timer if needed
            if (p2p->oppps_status != P2P_OPPPS_TIMER_NOT_STARTED)
            {
                mm_timer_clear(&p2p->oppps_timer);
                p2p->oppps_status = P2P_OPPPS_TIMER_NOT_STARTED;
                PROF_P2P_CTW_CLR();
            }
            p2p_update_go_ps_state(p2p);
        }
    }
}

bool p2p_is_present(uint8_t p2p_index)
{
    struct p2p_info_tag *p2p = &p2p_info_tab[p2p_index];
    return (p2p->is_go_present);
}

bool p2p_go_get_next_NOA_date(struct vif_info_tag *vif,
                              uint32_t *next_p2p_present,
                              uint32_t *next_p2p_absent)
{
    struct p2p_info_tag *p2p;
    ASSERT_ERR(vif->p2p);

    p2p = &p2p_info_tab[vif->p2p_index];
    *next_p2p_present = p2p->next_present;
    *next_p2p_absent = p2p->next_absent;

    return hal_machw_time_cmp(*next_p2p_absent, *next_p2p_present);
}

uint32_t p2p_go_get_next_noa_end_date(struct vif_info_tag *vif,
                                      uint8_t noa_inst)
{
    struct p2p_info_tag *p2p;
    struct p2p_noa_info_tag *noa;
    ASSERT_ERR(vif->p2p && (noa_inst < P2P_NOA_NB_MAX));

    p2p = &p2p_info_tab[vif->p2p_index];
    noa = &p2p->noa[noa_inst];

    switch (noa->noa_status)
    {
        case P2P_NOA_TIMER_WAIT_END_ABS:
            return noa->noa_timer.time;
        case P2P_NOA_TIMER_WAIT_NEXT_ABS:
            return noa->noa_timer.time + noa->noa_dur_us;
        default:
            ASSERT_WARN(0);
    }

    return 0;
}

void p2p_tbtt_handle(struct vif_info_tag *vif, uint32_t tbtt_time)
{
    struct p2p_info_tag *p2p;

    if (!vif->p2p)
    {
        return;
    }
    TRACE_P2P(TBTT, "{VIF-%d} TBTT starts", vif->index);

    p2p = &p2p_info_tab[vif->p2p_index];

    PROF_P2P_WAIT_BCN_SET();

    // Keep in mind we are waiting for a beacon
    p2p->is_waiting_bcn = true;
    #if (NX_POWERSAVE)
    vif->prevent_sleep &= ~(PS_VIF_P2P_WAIT_TBTT);
    #endif //(NX_POWERSAVE)

    // Check if Opportunistic Power Save mode is used -> Start timer monitoring the CTWindow
    if (p2p->oppps_ctw)
    {
        // Compute end time of CT Window (oppps_ctw is in TUs)
        uint32_t ctw_end = tbtt_time + (p2p->oppps_ctw << 10);

        if (p2p->role == P2P_ROLE_CLIENT)
        {
            tbtt_time += 300; // For CLI take extra margin
            if (hal_machw_time_cmp(ke_time() + 50, tbtt_time))
            {
                // ctw really start at tbtt_time (which is more than 50us away) so delay update
                // of P2P GO presence update until then
                p2p->oppps_status = P2P_OPPPS_TIMER_WAIT_START_CTW;
                p2p->oppps_ctw_end = ctw_end;
                mm_timer_set(&p2p->oppps_timer, tbtt_time);
                return;
            }
        }

        p2p->oppps_status = P2P_OPPPS_TIMER_WAIT_END_CTW;
        mm_timer_set(&p2p->oppps_timer, ctw_end);
        PROF_P2P_CTW_SET();
    }

    // Update GO Power Save state
    p2p_update_go_ps_state(p2p);
}

void p2p_bcn_evt_handle(struct vif_info_tag *vif)
{
    struct p2p_info_tag *p2p = &p2p_info_tab[vif->p2p_index];

    TRACE_P2P(TBTT, "{VIF-%d} TBTT ends", vif->index);
    PROF_P2P_WAIT_BCN_CLR();

    // Update beacon reception status
    p2p->is_waiting_bcn = false;

    // Update GO Power Save state
    p2p_update_go_ps_state(p2p);
}

uint32_t p2p_cli_bcn_check_noa(struct vif_info_tag *vif,
                               struct rx_pbd *pyld_desc, struct rx_dmadesc *dma_hdrdesc)
{
    uint32_t noa_pos = 0;
    uint32_t rd_addr;
    uint32_t end_addr;
    struct bcn_frame *bcn;

    // Check vif is P2P client
    if ((!vif->p2p) || (vif->type != VIF_STA))
    {
        return noa_pos;
    }

    ASSERT_ERR(vif->p2p_index < NX_P2P_VIF_MAX);
    TRACE_P2P(NOA_DET, "{VIF-%d} Check for NOA attribute in beacon", vif->index);

    rd_addr  = pyld_desc->datastartptr + MAC_BEACON_VARIABLE_PART_OFT;
    end_addr = rd_addr + dma_hdrdesc->hd.frmlen - MAC_BEACON_VARIABLE_PART_OFT;

    // Look for P2P IE(s)
    while (rd_addr < end_addr)
    {
        // Keep the current address before computing the next one
        uint32_t addr = rd_addr;

        rd_addr += co_read8p(addr + MAC_INFOELT_LEN_OFT) + MAC_INFOELT_INFO_OFT;

        // Check if the current IE is the one we are looking for
        if (co_read8p(addr) != P2P_ELMT_ID)
        {
            // Move on to the next IE
            continue;
        }

        // Check if OUI type is P2P
        if (co_read8p(addr + P2P_IE_OUI_TYPE_OFT) != P2P_OUI_TYPE_P2P)
        {
            // Move on to the next IE
            continue;
        }

        // Look for NOA attribute
        noa_pos = p2p_att_find(addr + P2P_IE_ATT_OFT,
                               co_read8p(addr + P2P_IE_LEN_OFT) - 4,
                               P2P_ATT_ID_NOTICE_OF_ABSENCE);

        if (!noa_pos)
        {
            // NOA has not been found, continue. Another P2P element could contain NOA
            continue;
        }

        break;
    }

    // Get beacon data
    bcn = (struct bcn_frame *)HW2CPU(
        ((struct rx_pbd *)HW2CPU(dma_hdrdesc->hd.first_pbd_ptr))->datastartptr);

    p2p_cli_noa_handle_att(vif->p2p_index, noa_pos, bcn->tsf);

    return (noa_pos);
}

void p2p_cli_handle_action(struct vif_info_tag *vif, uint32_t a_frame, uint16_t length,
                           uint32_t rx_tsf)
{
    uint32_t addr = a_frame + MAC_SHORT_MAC_HDR_LEN;
    uint16_t len = length - MAC_SHORT_MAC_HDR_LEN;

    TRACE_P2P(NOA_DET, "{VIF-%d} check for NOA attribute in P2P action frame", vif->index);

    // Check Category Code, should be Vendor Specific (127)
    if (co_read8p(addr + MAC_ACTION_CATEGORY_OFT) != MAC_VENDOR_ACTION_CATEGORY)
    {
        return;
    }

    // Read P2P Action Subtype and react accordingly
    if (co_read8p(addr + MAC_ACTION_P2P_ACTION_OFT) == MAC_P2P_ACTION_NOA_SUBTYPE)
    {
        // Move to the tagged parameters
        addr += MAC_ACTION_P2P_TAGGED_OFT;
        len -= MAC_ACTION_P2P_TAGGED_OFT;

        // Check if the current IE is the one we are looking for
        if (co_read8p(addr + P2P_IE_ID_OFT) != P2P_ELMT_ID)
        {
            return;
        }

        // Check if OUI type is P2P
        if (co_read8p(addr + P2P_IE_OUI_TYPE_OFT) != P2P_OUI_TYPE_P2P)
        {
            return;
        }

        if (co_read8p(addr + P2P_IE_ATT_OFT) != P2P_ATT_ID_NOTICE_OF_ABSENCE)
        {
            return;
        }

        // Handle the NoA Attribute
        p2p_cli_noa_handle_att(vif->p2p_index, addr + P2P_IE_ATT_OFT,
                               p2p_local_2_peer_time(vif, rx_tsf));
    }
}

/*
 * --------------------------------------------------------------------------
 * |           FUNCTION TO BE USED FOR P2P GO BEACON MANAGEMENT             |
 * --------------------------------------------------------------------------
 */

#if (NX_P2P_GO)
#if (NX_POWERSAVE)
bool p2p_go_check_ps_mode(void)
{
    // Check if P2P module authorizes the system to enter in deep sleep when GO.
    // Note that if we are here, it is considered that ps_env.ps_on value is false
    // meaning that Legacy Power Save mode is disabled.
    // If ps_env.ps_on is true, P2P_GO_PRESENT bit in vif_mgmt_tag prevent sleep
    // value is enough.
    struct vif_info_tag *vif;

    // Check if we have a GO VIF
    if (!p2p_env.nb_p2p_go)
    {
        return false;
    }

    // Check if we have at least one STA VIFs or several VIF AP
    #if (NX_MULTI_ROLE)
    if (vif_mgmt_env.vif_sta_cnt || (vif_mgmt_env.vif_ap_cnt > 1))
    {
        return false;
    }
    #endif //(NX_MULTI_ROLE)

    vif = (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

    while (vif)
    {
        if ((vif->type == VIF_AP) && vif->p2p)
        {
            struct p2p_info_tag *p2p = &p2p_info_tab[vif->p2p_index];

            // Can go to deep sleep if GO is not present
            return ! p2p->is_go_present;
        }

        vif = (struct vif_info_tag *)vif_mgmt_next(vif);
    }

    return false;
}
#endif //(NX_POWERSAVE)

void p2p_go_td_evt(uint8_t vif_index, uint8_t new_status)
{
    struct vif_info_tag *vif = &vif_info_tab[vif_index];

    if (vif->p2p && (vif->type == VIF_AP))
    {
        struct p2p_info_tag *p2p = &p2p_info_tab[vif->p2p_index];

        if (new_status)
        {
            PROF_P2P_PS_PAUSED_SET();
            p2p->noa_paused = true;
        }
        else
        {
            PROF_P2P_PS_PAUSED_CLR();
            p2p->noa_paused = false;
        }

        if (p2p->is_noa_bcn)
        {
            // Update the NoA IE in the Beacon
            mm_bcn_update_p2p_noa(p2p->vif_index, P2P_BCN_UPD_OP_NOA_UPD);
        }

        p2p_update_go_ps_state(p2p);
    }
}

void p2p_go_oppps_start(struct vif_info_tag *vif, uint8_t ctw)
{
    struct p2p_info_tag *p2p = &p2p_info_tab[vif->p2p_index];

    TRACE_P2P(PS, "{VIF-%d} start oppPS CTW=%d TU", vif->index, ctw);

    // Initialize OPPPS parameters
    p2p->oppps_ctw = ctw;
    // Consider the CTW as opened before next TBTT
    p2p->oppps_status = P2P_OPPPS_TIMER_WAIT_END_CTW;

    // Update the NOA attribute, will be added in the beacon (see p2p_go_bcn_op_done)
    mm_bcn_update_p2p_noa(p2p->vif_index, P2P_BCN_UPD_OP_NOA_UPD);

    // Update GO PS state
    p2p_update_go_ps_state(p2p);
}

void p2p_go_oppps_stop(struct vif_info_tag *vif)
{
    struct p2p_info_tag *p2p = &p2p_info_tab[vif->p2p_index];

    TRACE_P2P(PS, "{VIF-%d} stop oppPS", vif->index);

    // Reset OPPPS parameters and stop time if needed
    p2p->oppps_ctw = 0;
    if (p2p->oppps_status != P2P_OPPPS_TIMER_NOT_STARTED)
    {
        mm_timer_clear(&p2p->oppps_timer);
        p2p->oppps_status = P2P_OPPPS_TIMER_NOT_STARTED;
        PROF_P2P_CTW_CLR();
    }
    p2p_update_go_ps_state(p2p);

    if (!p2p->noa_nb)
    {
        // If NOA is not used, remove NOA part of the Beacon
        mm_bcn_update_p2p_noa(p2p->vif_index, P2P_BCN_UPD_OP_NOA_RMV);
    }
    else
    {
        // Update NOA attribute content
        mm_bcn_update_p2p_noa(p2p->vif_index, P2P_BCN_UPD_OP_NOA_UPD);
    }
}

void p2p_go_ps_state_update(struct vif_info_tag *vif)
{
    if (!vif->p2p)
    {
        return;
    }

    TRACE_P2P(PS, "{VIF-%d} update GO PS state", vif->index);
    p2p_update_go_ps_state(&p2p_info_tab[vif->p2p_index]);
}

uint8_t p2p_go_noa_start(struct vif_info_tag *vif, bool concurrent, bool dyn_noa,
                         uint8_t counter, uint32_t intv_us, uint32_t dur_us,
                         uint32_t start_time)
{
    struct p2p_info_tag *p2p = &p2p_info_tab[vif->p2p_index];
    struct p2p_noa_info_tag *noa;
    uint8_t noa_inst = P2P_NOA_NB_MAX;

    if (p2p->role != P2P_ROLE_GO)
        return P2P_INVALID_IDX;

    noa_inst = p2p_go_get_noa_inst(p2p);

    TRACE_P2P(NOA, "{VIF-%d} start NOA-%d: count=%d, intv=%ld us, dur=%ld us, start %t",
              vif->index, noa_inst, counter, TR_32(intv_us), TR_32(dur_us),
              TR_32(start_time));

    if (noa_inst >= P2P_NOA_NB_MAX)
        return P2P_INVALID_IDX;

    noa = &p2p->noa[noa_inst];

    // Initialize NOA parameters
    noa->noa_counter      = counter;
    noa->noa_init_counter = counter;
    noa->noa_intv_us      = intv_us;
    noa->noa_dur_us       = dur_us;
    noa->noa_start_time   = start_time;
    noa->dyn_noa          = dyn_noa;
    noa->noa_type         = (concurrent) ? P2P_NOA_TYPE_CONCURRENT : P2P_NOA_TYPE_NORMAL;

    p2p->noa_nb++;

    if (concurrent)
    {
        p2p->noa_cm_nb++;
    }

    // Compute number of absence periods before start time update
    p2p_go_noa_get_intv_cnt(noa);

    // Program the noa timer with the computed start time
    p2p_noa_timer_prog(noa, noa->noa_start_time);
    // And Update the NOA status
    noa->noa_status = P2P_NOA_TIMER_WAIT_NEXT_ABS;

    // Update the NOA attribute, will be added in the beacon (see p2p_go_bcn_op_done)
    mm_bcn_update_p2p_noa(p2p->vif_index, P2P_BCN_UPD_OP_NOA_UPD);

    // Inform the host about the newly created NoA
    p2p_go_send_noa_upd_ind(p2p, noa_inst);

    p2p_update_noa_presence_date(p2p);

    // Update Go PS state
    p2p_update_go_ps_state(p2p);

    return noa_inst;
}

uint8_t p2p_go_noa_stop(struct vif_info_tag *vif, uint8_t noa_inst, bool host_req)
{
    struct p2p_info_tag *p2p = &p2p_info_tab[vif->p2p_index];
    struct p2p_noa_info_tag *noa;

    if ((p2p->role != P2P_ROLE_GO) || (noa_inst >= P2P_NOA_NB_MAX))
    {
        return CO_FAIL;
    }

    TRACE_P2P(NOA, "{VIF-%d} stop NOA-%d (host_req=%d)", vif->index, noa_inst, host_req);

    noa = &p2p->noa[noa_inst];
    if ((noa->noa_status == P2P_NOA_TIMER_NOT_STARTED) ||
        (host_req && (noa->noa_type == P2P_NOA_TYPE_CONCURRENT)))
    {
        return CO_FAIL;
    }

    // Stop NOA
    PROF_P2P_NOA_ABS_CLR(noa->noa_inst);
    mm_timer_clear(&noa->noa_timer);
    noa->noa_status = P2P_NOA_TIMER_NOT_STARTED;
    p2p_update_noa_presence_date(p2p);
    p2p_update_go_ps_state(p2p);
    p2p->noa_nb--;
    if (noa->noa_type == P2P_NOA_TYPE_NORMAL)
    {
        p2p->noa_cm_nb--;
    }

    if ((p2p->oppps_ctw == 0) && (p2p->noa_nb == 0))
    {
        // If OppPS is not used, remove NOA part of the Beacon
        mm_bcn_update_p2p_noa(p2p->vif_index, P2P_BCN_UPD_OP_NOA_RMV);
    }
    else
    {
        // otherwise update NOA attribute content
        mm_bcn_update_p2p_noa(p2p->vif_index, P2P_BCN_UPD_OP_NOA_UPD);
    }

    // Inform the host that a NoA has been stopped
    p2p_go_send_noa_upd_ind(p2p, noa_inst);

    return CO_OK;
}

#if (NX_CHNL_CTXT)
void p2p_go_noa_concurrent_move(struct vif_info_tag *vif, int32_t offset)
{
    struct p2p_info_tag *p2p = &p2p_info_tab[vif->p2p_index];
    bool update_date = false;
    uint8_t noa_idx;

    for (noa_idx = 0; noa_idx < P2P_NOA_NB_MAX; noa_idx++)
    {
        struct p2p_noa_info_tag *noa = &p2p->noa[noa_idx];
        if ((noa->noa_type == P2P_NOA_TYPE_CONCURRENT) &&
            (noa->noa_status != P2P_NOA_TIMER_NOT_STARTED))
        {
            p2p_noa_timer_prog(noa, noa->noa_timer.time + offset);
            update_date = true;
        }
    }

    if (update_date)
        p2p_update_noa_presence_date(p2p);
}
#endif //(NX_CHNL_CTXT)

uint8_t p2p_go_bcn_get_noa_len(uint8_t p2p_index)
{
    struct p2p_info_tag *p2p = &p2p_info_tab[p2p_index];
    uint8_t noa_ie_len = 0;

    if ((p2p->vif_index != INVALID_VIF_IDX) && p2p->is_noa_bcn)
    {
        noa_ie_len = P2P_IE_NOA_NO_NOA_DESC_LENGTH
                       + p2p->noa_in_bcn_nb * P2P_NOA_DESC_LENGTH;
    }

    return (noa_ie_len);
}

void p2p_go_bcn_init_noa_pyld(uint32_t a_noa_ie_elt)
{
    co_write8p(a_noa_ie_elt + P2P_IE_ID_OFT, P2P_ELMT_ID);
    co_write8p(a_noa_ie_elt + P2P_IE_OUI_OFT, P2P_OUI_WIFI_ALL_BYTE0);
    co_write8p(a_noa_ie_elt + P2P_IE_OUI_OFT + 1, P2P_OUI_WIFI_ALL_BYTE1);
    co_write8p(a_noa_ie_elt + P2P_IE_OUI_OFT + 2, P2P_OUI_WIFI_ALL_BYTE2);
    co_write8p(a_noa_ie_elt + P2P_IE_OUI_TYPE_OFT, P2P_OUI_TYPE_P2P);
    co_write8p(a_noa_ie_elt + P2P_IE_ATT_OFT, P2P_ATT_ID_NOTICE_OF_ABSENCE);
}

void p2p_go_bcn_upd_noa_pyld(uint8_t p2p_index, uint32_t a_noa_ie_elt)
{
    struct p2p_info_tag *p2p = &p2p_info_tab[p2p_index];
    uint8_t noa_idx, noa_proc_nb = 0;
    uint16_t att_length;

    TRACE_P2P(NOA_DET, "{VIF-%d} update NOA payload in beacon", p2p->vif_index);

    co_write8p(a_noa_ie_elt + P2P_IE_ATT_OFT + P2P_NOA_ATT_INDEX_OFT, p2p->noa_index);

    // Increment NOA index
    p2p->noa_index++;

    if (p2p->oppps_ctw)
    {
        // If CTWindow is not zero, opportunistic PS is used
        co_write8p(a_noa_ie_elt + P2P_IE_ATT_OFT + P2P_NOA_ATT_CTW_OPPPS_OFT,
                   p2p->oppps_ctw | P2P_OPPPS_MASK);
    }
    else
    {
        co_write8p(a_noa_ie_elt + P2P_IE_ATT_OFT + P2P_NOA_ATT_CTW_OPPPS_OFT, 0);
    }

    for (noa_idx = 0; noa_idx < P2P_NOA_NB_MAX; noa_idx++)
    {
        struct p2p_noa_info_tag *noa = &p2p->noa[noa_idx];

        // Check if the NoA instance is active
        // Do not insert non concurrent mode noa if paused due to traffic.
        if ((noa->noa_status != P2P_NOA_TIMER_NOT_STARTED) &&
            !(p2p->noa_paused && noa->dyn_noa && (noa->noa_type == P2P_NOA_TYPE_NORMAL)))
        {
            uint32_t a_noa_desc = a_noa_ie_elt + P2P_IE_ATT_OFT + P2P_NOA_ATT_NOA_DESC_OFT
                                               + (P2P_NOA_DESC_LENGTH * noa_idx);

            noa_proc_nb++;

            co_write8p(a_noa_desc + P2P_NOA_DESC_COUNT_OFT, noa->noa_init_counter);
            co_write32p(a_noa_desc + P2P_NOA_DESC_DUR_OFT, noa->noa_dur_us);
            co_write32p(a_noa_desc + P2P_NOA_DESC_INTV_OFT, noa->noa_intv_us);


            // Stored start time value is based on Monoatomic counter, value sent within
            // the beacon is a TSF value.
            co_write32p(a_noa_desc + P2P_NOA_DESC_START_OFT,
                        noa->noa_start_time - (ke_time() - nxmac_tsf_lo_get()));
        }
    }

    // Keep number of NoA in the beacon
    p2p->noa_in_bcn_nb = noa_proc_nb;

    // Update the length fields
    att_length = 2 + noa_proc_nb * P2P_NOA_DESC_LENGTH;

    co_write16p(a_noa_ie_elt + P2P_IE_ATT_OFT + P2P_ATT_LEN_OFT, att_length);
    co_write8p(a_noa_ie_elt + P2P_IE_LEN_OFT, 7 + (uint8_t)att_length);
}

void p2p_go_bcn_op_done(uint8_t p2p_index, uint8_t operation)
{
    struct p2p_info_tag *p2p = &p2p_info_tab[p2p_index];

    TRACE_P2P(NOA_DET, "{VIF-%d} Beacon updated operation=%d",
              p2p->vif_index, operation);

    switch (operation)
    {
        case (P2P_BCN_UPD_OP_NOA_ADD):
        {
            p2p->is_noa_bcn = true;
        } break;

        case (P2P_BCN_UPD_OP_NOA_RMV):
        {
            p2p->is_noa_bcn = false;
        } break;

        case (P2P_BCN_UPD_OP_NOA_UPD):
        {
            // If the NOA attribute is not part of the beacon, insert it
            if (!p2p->is_noa_bcn)
            {
                mm_bcn_update_p2p_noa(p2p->vif_index, P2P_BCN_UPD_OP_NOA_ADD);
            }
            // else nothing to do
        } break;

        default:
        {
            ASSERT_ERR(0);
        }
    }
}

#if (NX_POWERSAVE)
void p2p_go_pre_tbtt(struct vif_info_tag *vif)
{
    // Check if VIF is still active
    if (vif->p2p && vif->active)
    {
        // Set bit in order to be sure to be awoken for AP_TBTT interrupt
        // Don't use PS_VIF_P2P_GO_PRESENT has is some corner case P2P GO may become
        // absent between here and the actual AP_TBTT interrupt
        vif->prevent_sleep |= PS_VIF_P2P_WAIT_TBTT;
    }
}
#endif //(NX_POWERSAVE)
#endif //(NX_P2P_GO)

#endif //(NX_P2P)

/// @} end of group
