/**
 ****************************************************************************************
 *
 * @file ps.c
 *
 * @brief Power-Save mode implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup PS
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "ps.h"
#include "mac_ie.h"
#include "sta_mgmt.h"
#include "vif_mgmt.h"
#include "mm.h"
#include "co_utils.h"
#include "co_status.h"
#include "dbg.h"
#include "txl_frame.h"
#include "phy.h"
#include "tpc.h"

#if (NX_DPSM)
#include "td.h"
#endif //(NX_DPSM)

#if NX_POWERSAVE

/*
 * DEFINES
 ****************************************************************************************
 */

/// Maximum number of TX attempts for a NULL frame indicating PS mode change
#define PS_TX_ERROR_MAX      (3)

/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
/// PS context used to store information related to PS module
struct ps_env_tag ps_env;

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Sends PS Poll frame to peer AP
 *
 * Sends a ps poll to the peer AP to retrieve buffered traffic.
 *
 * @param[in] vif       VIF that need to send the PS Poll frame
 *
 * @return CO_OK if frame has been pushed, CO_FAIL otherwise
 ****************************************************************************************
 */
static uint8_t ps_send_pspoll(struct vif_info_tag *vif)
{
    struct txl_frame_desc_tag *frame;
    struct mac_hdr_ctrl *buf;
    struct tx_hd *thd;
    struct sta_info_tag *sta = &sta_info_tab[vif->u.sta.ap_id];
    int txtype;

    // Allocate a frame descriptor from the TX path
    txtype = vif_mgmt_get_txtype(vif);
    frame = txl_frame_get(txtype, MAC_PSPOLL_FRAME_SIZE);
    if (frame == NULL)
        return CO_FAIL;

    // update Tx power in policy table
    tpc_update_frame_tx_power(vif, frame);

    // Get the buffer pointer
    buf = txl_buffer_payload_get(&frame->txdesc);

    // Fill-in the frame
    buf->fctl = MAC_FCTRL_PSPOLL;
    buf->durid = (0x3 << 14) | sta->aid;
    buf->addr1 = sta->mac_addr;
    buf->addr2 = vif->mac_addr;

    // HW shall not modify the DUR/AID field for a PS-poll
    thd = &frame->txdesc.lmac.hw_desc->thd;
    thd->macctrlinfo2 |= DONT_TOUCH_DUR | TS_VALID_BIT | (MAC_FCTRL_PSPOLL >> 1);

    #if (NX_CHNL_CTXT || NX_P2P)
    // Set VIF and STA indexes
    frame->txdesc.host.vif_idx = sta->inst_nbr;
    frame->txdesc.host.staid   = sta->staid;
    #endif //(NX_CHNL_CTXT || NX_P2P)

    // Push the frame for TX
    txl_frame_push(frame, AC_VO);

    return CO_OK;
}

/**
 ****************************************************************************************
 * @brief Checks if NULL frame indicating a PS mode change has been acknowledged by the
 *        peer AP.
 * Up to 3 (@ref PS_TX_ERROR_MAX) retries are allowed. If this limit is reached, we
 * consider the connection with the AP has been lost and we continue processing the
 * confirmation properly so that we can continue the mode update.
 *
 * @param[in] vif           VIF on which the NULL frame has been sent
 * @param[in] status        Transmission status
 * @param[in] cfm           Confirmation callback if we need to resend a NULL frame.
 *
 * @return False if a null frame has been resent and confirmation must be awaited. True
 * if the frame has been sucessfully received, or retry limit has been reached.
 ****************************************************************************************
 */
static bool ps_check_tx_status(struct vif_info_tag *vif, uint32_t status,
                               cfm_func_ptr cfm)
{
    // Indicate if confirmation can be handled
    bool handle_cfm = true;

    // Check if an error has occured during the NULL frame transmission
    if (!(status & FRAME_SUCCESSFUL_TX_BIT))
    {
        // Increase number of errors
        vif->u.sta.ps_retry++;

        // Check if maximum number of retries has been reached
        if (vif->u.sta.ps_retry == PS_TX_ERROR_MAX)
        {
            #if (NX_CONNECTION_MONITOR)
            // Inform the host that the link has been lost
            mm_send_connection_loss_ind(vif);
            #endif //(NX_CONNECTION_MONITOR)
            //#else continue
        }
        else
        {
            // Send a new NULL frame
            txl_frame_send_null_frame(vif->u.sta.ap_id, cfm, vif);

            // Wait for next confirmation
            handle_cfm = false;
        }
    }

    return (handle_cfm);
}

/**
 ****************************************************************************************
 * @brief Callback associated to frame sent to inform that interface will enter PS.
 *
 * This function is called when a frame to indicate the peer that interface will enter PS
 * has been transmitted.
 * If frame has been sucessfully received by the peer and not more confirmation is pending,
 * this function will notify MM task that Ps is enabled.
 *
 * @param[in] env     VIF that sent the frame
 * @param[in] status  Transmission status
 ****************************************************************************************
 */
static void ps_enable_cfm(void *env, uint32_t status)
{
    // Check if an error has occurred during the NULL frame transmission
    if (!ps_check_tx_status((struct vif_info_tag *)env, status, ps_enable_cfm))
    {
        return;
    }

    if (ps_env.cfm_cnt)
    {
        // Decrease the number of expected confirmations
        ps_env.cfm_cnt--;
    }

    // Check if all confirmations have been received
    if (ps_env.cfm_cnt == 0)
    {
        #if NX_UAPSD
        // Check if UAPSD is in use
        if (ps_env.uapsd_on)
        {
            struct vif_info_tag *vif =
                (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

            mm_timer_set(&ps_env.uapsd_timer, ke_time() + ps_env.uapsd_timeout);
            ps_env.uapsd_tmr_on = true;
            while (vif != NULL)
            {
                if ((vif->type == VIF_STA) && vif->active  && vif->u.sta.uapsd_queues)
                    vif->prevent_sleep &= ~PS_VIF_WAITING_EOSP;

                // Go to next entry
                vif = (struct vif_info_tag *)co_list_next(&vif->list_hdr);
            }
        }
        #endif

        #if (NX_DPSM)
        if (PS_DPSM_STATE_GET(ON) && PS_DPSM_STATE_GET(RESUMING))
        {
            // Resume RESUMING bit
            PS_DPSM_STATE_CLEAR(RESUMING);
            // Clear PAUSE bit
            PS_DPSM_STATE_CLEAR(PAUSE);
            TRACE_STA(PS, "{VIF-%d} PS mode resumed", ((struct vif_info_tag *)env)->index);
        }
        else
        #endif //(NX_DPSM)
        {
            // Set the PS environment variable
            ps_env.ps_on = true;
            TRACE_STA(PS, "{VIF-%d} PS mode enabled", ((struct vif_info_tag *)env)->index);

            // Send the confirmation
            ke_msg_send_basic(MM_SET_PS_MODE_CFM, ps_env.taskid, TASK_MM);
        }

        #if (NX_DPSM)
        if (PS_DPSM_STATE_GET(SET_MODE_REQ))
        {
            // Reset SET_MODE_REQ bit
            PS_DPSM_STATE_CLEAR(SET_MODE_REQ);
            // Set PS Mode as required by the application
            ps_set_mode(ps_env.next_mode, ps_env.taskid);
        }
        #endif //(NX_DPSM)
    }
}

/**
 ****************************************************************************************
 * @brief Callback associated to frame sent to inform that interface left PS.
 *
 * This function is called when a frame to indicate the peer that interface left PS has
 * been transmitted.
 * If frame has been sucessfully received by the peer and not more confirmation is pending,
 * this function will notify MM task that Ps is disabled.
 *
 * @param[in] env     VIF that sent the frame
 * @param[in] status  Transmission status
 ****************************************************************************************
 */
static void ps_disable_cfm(void *env, uint32_t status)
{
    // Check if an error has occurred during the NULL frame transmission
    if (!ps_check_tx_status((struct vif_info_tag *)env, status, ps_disable_cfm))
    {
        return;
    }

    if (ps_env.cfm_cnt)
    {
        // Decrease the number of expected confirmations
        ps_env.cfm_cnt--;
    }

    // Check if all confirmations have been received
    if (ps_env.cfm_cnt == 0)
    {
        struct vif_info_tag *vif =
            (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

        #if NX_UAPSD
        // Clear the UAPSD timer
        mm_timer_clear(&ps_env.uapsd_timer);
        ps_env.uapsd_tmr_on = false;
        #endif

        while (vif != NULL)
        {
            if ((vif->type == VIF_STA) && vif->active)
            {
                vif->prevent_sleep &= ~(PS_VIF_WAITING_BCMC | PS_VIF_WAITING_UC | PS_VIF_WAITING_EOSP);
            }

            // Go to next entry
            vif = (struct vif_info_tag *)co_list_next(&vif->list_hdr);
        }

        #if (NX_DPSM)
        // Check if DPSM is ON and if Pause was requested
        if (PS_DPSM_STATE_GET(ON) && PS_DPSM_STATE_GET(PAUSING))
        {
            // Resume PAUSING bit
            PS_DPSM_STATE_CLEAR(PAUSING);
            // Set PAUSE bit
            PS_DPSM_STATE_SET(PAUSE);
            TRACE_STA(PS, "{VIF-%d} PS mode paused", ((struct vif_info_tag *)env)->index);
        }
        else
        #endif //(NX_DPSM)
        {
            // Set the PS environment variable
            ps_env.ps_on = false;
            TRACE_STA(PS, "{VIF-%d} PS mode disabled", ((struct vif_info_tag *)env)->index);

            // Send the confirmation
            ke_msg_send_basic(MM_SET_PS_MODE_CFM, ps_env.taskid, TASK_MM);
        }

        #if (NX_DPSM)
        if (PS_DPSM_STATE_GET(SET_MODE_REQ))
        {
            // Reset SET_MODE_REQ bit
            PS_DPSM_STATE_CLEAR(SET_MODE_REQ);
            // Set PS Mode as required by the application
            ps_set_mode(ps_env.next_mode, ps_env.taskid);
        }
        #endif //(NX_DPSM)
    }
}

#if (NX_DPSM)
/**
 ****************************************************************************************
 * @brief Updates Dynamic Power Save mode status
 *
 * Updates status and send null frame on each active STA vif to indicate that it will enter
 * or leave PS mode.
 *
 * @param[in] pause Whether Dynamic power save mode should be paused or not
 ****************************************************************************************
 */
static void ps_dpsm_update(bool pause)
{
    // VIF Entry variable used to parse list of created VIFs
    struct vif_info_tag *vif;
    // Function called when sent NULL frame is ACKed by AP
    cfm_func_ptr cfm ;

    PROF_PS_DPSM_UPDATE_SET();

    // Request to pause Power Save Mode
    if (pause)
    {
        cfm = ps_disable_cfm;

        // Keep in mind that pause has been requested
        PS_DPSM_STATE_SET(PAUSING);

        // Disallow channel switching
        ps_env.prevent_sleep |= PS_PSM_PAUSED;

        PROF_PS_PAUSE_SET();
    }
    // Request to resume Power Save Mode
    else
    {
        cfm = ps_enable_cfm;

        // Keep in mind that resume has been requested
        PS_DPSM_STATE_SET(RESUMING);

        // Allow channel switching
        ps_env.prevent_sleep &= ~PS_PSM_PAUSED;

        PROF_PS_PAUSE_CLR();
    }

    // Update macCntrl1Reg.pwrMgt bit
    nxmac_pwr_mgt_setf((uint8_t)(pause == false));

    // Reset the confirmation counter
    ps_env.cfm_cnt = 0;

    // Go through the list of STA VIF
    vif = (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

    while (vif)
    {
        if ((vif->type == VIF_STA) && vif->active
            #if (NX_CHNL_CTXT)
            && chan_is_tx_allowed(vif)
            #endif //(NX_CHNL_CTXT)
           )
        {
            // Reset number of retries for this transmission
            vif->u.sta.ps_retry = 0;
            // We expect one more confirmation
            ps_env.cfm_cnt++;

            // Send a NULL frame to indicate to the AP that we are sleeping
            txl_frame_send_null_frame(vif->u.sta.ap_id, cfm, vif);
        }

        // Go to next entry
        vif = (struct vif_info_tag *)co_list_next(&vif->list_hdr);
    }

    // If no NULL frame has been sent, directly call the cfm function
    if (ps_env.cfm_cnt == 0)
    {
        cfm(NULL, FRAME_SUCCESSFUL_TX_BIT);
    }

    PROF_PS_DPSM_UPDATE_CLR();
}
#endif //(NX_DPSM)

#if (NX_UAPSD)
/**
 ****************************************************************************************
 * @brief Callback function for UAPSD timer
 *
 * Sends a trigger frame to AP to retrieve the data that are buffered
 *
 * @param[in] env Pointer to env variable associated to the timer. (NULL in this case)
 ****************************************************************************************
 */
static void ps_uapsd_timer_handle(void *env)
{
    bool sta_active = false;
    // Go through the list of STA VIF
    struct vif_info_tag *vif = (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

    while (vif != NULL)
    {
        if ((vif->type == VIF_STA) && vif->active && vif->u.sta.uapsd_queues
            #if NX_CHNL_CTXT
            && chan_is_tx_allowed(vif)
            #endif
           )
        {
            // Send a trigger frame to all APs with which U-APSD is used
            sta_active = true;

            // Check if we need to send the frame
            if (ke_time_past(vif->u.sta.uapsd_last_rxtx + ps_env.uapsd_timeout/2))
            {
                #if (NX_P2P)
                if (vif->p2p && !p2p_is_present(vif->p2p_index))
                {
                    // If GO is absent no need to send trigger frame, assume that Service
                    // Period has been paused and it will be restarted when GO is present
                    // (cf ps_p2p_absence_update)
                    vif->u.sta.sp_paused = true;
                }
                else
                #endif //(NX_P2P)
                if (txl_frame_send_qosnull_frame(vif->u.sta.ap_id,
                                                 vif->u.sta.uapsd_tid << MAC_QOSCTRL_UP_OFT,
                                                 NULL, NULL) == CO_OK)
                {
                    TRACE_STA(PS, "{VIF-%d} start periodic UAPSD Service Period",
                              vif->index);
                    vif->prevent_sleep |= PS_VIF_WAITING_EOSP;
                    vif->u.sta.uapsd_last_rxtx = ke_time();
                }
            }
        }

        // Go to next entry
        vif = (struct vif_info_tag *)co_list_next(&vif->list_hdr);
    }

    // Check if at least one STA is active
    if (sta_active)
    {
        // Restart the UAPSD Timer
        mm_timer_set(&ps_env.uapsd_timer, ke_time() + ps_env.uapsd_timeout);
    }
    else
    {
        // UAPSD timer is not active anymore
        ps_env.uapsd_tmr_on = false;
    }
}
#endif //(NX_UAPSD)

void ps_init(void)
{
    // Fully reset the PS environment variable
    memset(&ps_env, 0, sizeof(ps_env));

    #if (NX_UAPSD)
    // Initialize mm_timer used for UAPSD
    ps_env.uapsd_timer.cb = ps_uapsd_timer_handle;
    #endif //(NX_UAPSD)
}

void ps_set_mode(uint8_t mode, ke_task_id_t taskid)
{
    struct vif_info_tag *vif;
    cfm_func_ptr cfm;

    // Save the ID of the requester
    ps_env.taskid = taskid;

    #if (NX_DPSM)
    // If update has been required internally, wait for end of procedure
    if (PS_DPSM_STATE_GET(ON) && (PS_DPSM_STATE_GET(PAUSING) || PS_DPSM_STATE_GET(RESUMING)))
    {
        // Keep mode in mind
        ps_env.next_mode = mode;
        // Set SET_MODE_REQ bit to 1
        PS_DPSM_STATE_SET(SET_MODE_REQ);

        return;
    }
    #endif //(NX_DPSM)

    if (mode == PS_MODE_OFF)
    {
        // Set the confirmation function pointer
        cfm = ps_disable_cfm;

        #if (NX_DPSM)
        // Clear DPSM state
        ps_env.dpsm_state = 0;
        #endif //(NX_DPSM)

        // PS mode is disabled, then the macCntrl1Reg.pwrMgt bit must be set to 0,
        // in order to reset the PM bit in Frame Control from now on.
        nxmac_pwr_mgt_setf(0);
    }
    else
    {
        // Set the confirmation function pointer
        cfm = ps_enable_cfm;

        #if (NX_DPSM)
        // Check if Dynamic Power Save mode is required
        if (mode == PS_MODE_ON_DYN)
        {
            // Set DPSM ON bit
            PS_DPSM_STATE_SET(ON);
        }
        #endif //(NX_DPSM)

        // PS mode is enabled, then the macCntrl1Reg.pwrMgt bit must be set to 1,
        // all the frames must have their PM bit set to 1 in Frame Control field.
        nxmac_pwr_mgt_setf(1);
    }

    // Reset the confirmation counter
    ps_env.cfm_cnt = 0;
    #if NX_UAPSD
    /// Set the UAPSD ON flag
    ps_env.uapsd_on = false;
    #endif

    // Go through the list of STA VIF
    vif = (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

    while (vif != NULL)
    {
        if ((vif->type == VIF_STA) && vif->active
            #if NX_CHNL_CTXT
            && chan_is_tx_allowed(vif)
            #endif
            )
        {
            // Reset number of retries for this transmission
            vif->u.sta.ps_retry = 0;
            // We expect one more confirmation
            ps_env.cfm_cnt++;

            #if NX_UAPSD
            // Check if UAPSD is enabled for this interface
            if (vif->u.sta.uapsd_queues)
                ps_env.uapsd_on = true;
            #endif

            // Send a NULL frame to indicate to the AP that we are sleeping
            txl_frame_send_null_frame(vif->u.sta.ap_id, cfm, vif);
        }

        // Go to next entry
        vif = (struct vif_info_tag *)co_list_next(&vif->list_hdr);
    }

    // Check if are waiting for some confirmations before updating our state
    if (ps_env.cfm_cnt == 0)
    {
        // Call confirmation function
        cfm(NULL, FRAME_SUCCESSFUL_TX_BIT);
    }
}

/**
 ****************************************************************************************
 * @brief Check if TIM Information Element received from either an AP or a peer Mesh STA if
 * buffered frame targeting us are announced.
 *
 * @param[in] a_tim     Address of the TIM IE in the memory
 * @param[in] aid       Association ID
 *
 * @return true if bit for the aid is set in the TIM IE, false otherwise.
 ****************************************************************************************
 */
#if (!RW_MESH_EN)
__INLINE
#endif //(!RW_MESH_EN)
bool ps_check_tim(uint32_t a_tim, uint16_t aid)
{
    // Value used to read the TIM element
    uint8_t mask;
    unsigned int n, n1, n2;

    // Now we check if some unicast traffic is buffered
    // Compute the byte index of this AID
    n = aid / 8;

    // Compute the bit corresponding to this AID inside the byte
    mask = CO_BIT(aid & 0x7);

    // Now we need to compute the start and end byte indexes present in the TIM
    n1 = co_read8p(a_tim + MAC_TIM_BMPC_OFT) & 0xFE;
    n2 = co_read8p(a_tim + MAC_TIM_LEN_OFT) + n1 - 4;

    // First check if the AID byte is present in the bitmap
    if ((n < n1) || (n > n2))
    {
        return false;
    }

    // Byte is present, now computes its offset in the bitmap array
    n -= n1;

    if (co_read8p(a_tim + MAC_TIM_BMP_OFT + n) & mask)
    {
        return true;
    }

    return false;
}

void ps_check_beacon(uint32_t tim, uint16_t len, struct vif_info_tag *vif)
{
    uint16_t aid = sta_info_tab[vif->u.sta.ap_id].aid;

    // Beacon was received
    vif->prevent_sleep &= ~PS_VIF_WAITING_BCN;

    // Check if we are in PS mode
    if (!ps_env.ps_on)
        return;

    #if (NX_DPSM)
    if (PS_DPSM_STATE_GET(PAUSE))
        return;
    #endif //(NX_DPSM)

    // Check if TIM is present
    if (tim == 0)
        return;

    PROF_PS_CHECK_BCN_SET();

    // First check if there is some BC/MC traffic buffered by the AP
    if (!vif->u.sta.dont_wait_bcmc)
    {
        int bcmc_octet = vif->u.sta.bssid_index / 8;
        int bcmc_bit = vif->u.sta.bssid_index % 8;
        if (co_read8p(tim + MAC_TIM_BMPC_OFT + bcmc_octet) & CO_BIT(bcmc_bit))
        {
            vif->prevent_sleep |= PS_VIF_WAITING_BCMC;
            TRACE_STA(PS, "{VIF-%d} AP has BroadCast/Multicast traffic buffered",
                      vif->index);
        }
        else
            vif->prevent_sleep &= ~PS_VIF_WAITING_BCMC;
    }

    // Check if unicast traffic is present
    if (ps_check_tim(tim, aid))
    {
        TRACE_STA(PS, "{VIF-%d} AP has traffic buffered for us", vif->index);

        #if NX_UAPSD
        if ((vif->u.sta.uapsd_queues & PS_ALL_UAPSD_ACS) == PS_ALL_UAPSD_ACS)
        {
            // All ACs are uapsd-enabled: send a QoS Null on a VO TID
            if (txl_frame_send_qosnull_frame(vif->u.sta.ap_id,
                                             TID_7 << MAC_QOSCTRL_UP_OFT,
                                             NULL, NULL) == CO_OK)
            {
                vif->prevent_sleep |= PS_VIF_WAITING_EOSP;
                TRACE_STA(PS, "Start UAPSD Service Period");
            }
        }
        else
        #endif
        {
            // Since some ACs are not uapsd-enabled the TIM only refers to the
            // ACs configured to use 802.11 Legacy power save mode
            if (ps_send_pspoll(vif) == CO_OK)
            {
                vif->prevent_sleep |= PS_VIF_WAITING_UC;
                TRACE_STA(PS, "Send PS-POLL");
            }
        }
    }
    else
    {
        #if NX_UAPSD
        if ((vif->u.sta.uapsd_queues & PS_ALL_UAPSD_ACS) == PS_ALL_UAPSD_ACS)
        {
            vif->prevent_sleep &= ~PS_VIF_WAITING_EOSP;
        }
        else
        #endif
            vif->prevent_sleep &= ~PS_VIF_WAITING_UC;
    }

    PROF_PS_CHECK_BCN_CLR();
}

void ps_check_frame(uint8_t *frame, uint32_t statinfo, struct vif_info_tag *vif)
{
    struct mac_hdr *machdr = (struct mac_hdr *)frame;
    uint16_t framectrl = machdr->fctl;

    // Check if we are in PS mode
    if (!ps_env.ps_on)
        return;

    // Check if the frame is individually or group addressed
    if (MAC_ADDR_GROUP(&machdr->addr1))
    {
        // Check if other group frames will follow this one
        if (!(framectrl & MAC_FCTRL_MOREDATA) || vif->u.sta.dont_wait_bcmc)
            // No more group frames, allow going back to sleep
            vif->prevent_sleep &= ~PS_VIF_WAITING_BCMC;
    }
    else
    {
        #if (NX_UAPSD)
        bool uapsd_frame = false;
        #endif

        // Unicast frame - Ensure that the frame is for us
        if (statinfo & RX_HD_ADDRMIS)
            return;

        PROF_PS_CHECK_RX_SET();

        #if (NX_UAPSD)
        if (ps_env.uapsd_on)
        {
            // Check if it is a QoS-data frame
            if ((framectrl & MAC_FCTRL_QOS_DATA) == MAC_FCTRL_QOS_DATA)
            {
                uint16_t qos;
                uint8_t tid;

                // Compute the QoS control field offset in the frame
                if ((framectrl & MAC_FCTRL_TODS_FROMDS) == MAC_FCTRL_TODS_FROMDS)
                {
                    struct mac_hdr_long_qos *hdr = (struct mac_hdr_long_qos *)machdr;
                    qos = hdr->qos;
                }
                else
                {
                    struct mac_hdr_qos *hdr = (struct mac_hdr_qos *)machdr;
                    qos = hdr->qos;
                }

                // Get the QoS control field in the frame and the TID
                tid = (qos & MAC_QOSCTRL_UP_MSK) >> MAC_QOSCTRL_UP_OFT;

                // Check if UAPSD is enabled for this TID
                if (vif->u.sta.uapsd_queues & CO_BIT(mac_tid2ac[tid]))
                {
                    // Frame is serviced in a UAPSD service period
                    uapsd_frame = true;

                    // Save the time of this reception
                    vif->u.sta.uapsd_last_rxtx = ke_time();

                    // UAPSD is enabled for this TID, check if EOSP is set
                    if (qos & MAC_QOSCTRL_EOSP)
                    {
                        // End of service period, we can go to sleep again
                        vif->prevent_sleep &= ~PS_VIF_WAITING_EOSP;
                        TRACE_STA(PS, "{VIF-%d} EOSP received", vif->index);
                    }
                }
            }
            else if ((framectrl & MAC_FCTRL_TYPE_MASK) == MAC_FCTRL_MGT_T)
            {
                // Management frames are supposed to be sent in AC_VO,
                // so if VO is UAPSD enabled, we consider them as
                // UAPSD frames
                if (vif->u.sta.uapsd_queues & CO_BIT(AC_VO))
                {
                    // Frame is serviced in a UAPSD service period
                    uapsd_frame = true;

                    // Save the time of this reception
                    vif->u.sta.uapsd_last_rxtx = ke_time();
                }
            }
        }

        // Check the more data bit (except if we received the frame in
        // a UAPSD service period, as in that case we rely on the EOSP
        // to wait or not for new frames)
        if (!uapsd_frame)
        #endif //(NX_UAPSD)
        {
            #if (NX_DPSM)
            // Update Traffic Detection statistics for PS
            td_pck_ps_ind(vif->index, true);
            #endif //(NX_DPSM)

            TRACE_STA(PS, "{VIF-%d} received PS frame. (more data = %d)",
                      vif->index, (framectrl & MAC_FCTRL_MOREDATA));

            if (framectrl & MAC_FCTRL_MOREDATA)
            {
                #if (NX_DPSM)
                if (!PS_DPSM_STATE_GET(PAUSE))
                #endif //(NX_DPSM)
                {
                    if (ps_send_pspoll(vif) != CO_OK)
                    {
                        // Allow going back to sleep
                        vif->prevent_sleep &= ~PS_VIF_WAITING_UC;
                    }
                }
            }
            else
            {
                // Allow going back to sleep
                vif->prevent_sleep &= ~PS_VIF_WAITING_UC;
            }
        }

        PROF_PS_CHECK_RX_CLR();
    }
}

#if (NX_UAPSD || NX_DPSM)
bool ps_check_tx_frame(uint8_t staid, uint8_t tid)
{
    struct sta_info_tag *sta;
    struct vif_info_tag *vif;
    bool is_trigger = false;
    #if (NX_DPSM)
    bool skip_stat = false;
    #endif

    // Check if we are in PS mode
    if (!ps_env.ps_on)
        return false;

    // Check if destination STA is known
    if (staid == 0xFF)
        return false;

    // Check if TID is valid (indicating that the frame is QoS frame)
    if (tid == 0xFF)
        return false;

    // Retrieve the associated VIF entry
    sta = &sta_info_tab[staid];
    vif = &vif_info_tab[sta->inst_nbr];

    // Check if the frame is individually or group addressed
    if ((vif->type != VIF_STA) || !vif->active)
        return false;

    PROF_PS_CHECK_TX_SET();

    #if (NX_UAPSD)
    // Check if UAPSD is enabled for this TID
    if (vif->u.sta.uapsd_queues & CO_BIT(mac_tid2ac[tid]))
    {
        // UAPSD is enabled for this TID, we will have to wait for the EOSP
        vif->prevent_sleep |= PS_VIF_WAITING_EOSP;
        is_trigger = true;

        // Save the time of this transmission
        vif->u.sta.uapsd_last_rxtx = ke_time();

        #if NX_DPSM
        // When all the queues are UAPSD enabled, we don't want to increase PS
        // traffic statistics in order not to active DPSM for the UAPSD traffic.
        if ((vif->u.sta.uapsd_queues & PS_ALL_UAPSD_ACS) != PS_ALL_UAPSD_ACS)
            skip_stat = true;
        #endif
    }
    #endif //(NX_UAPSD)

    #if (NX_DPSM)
    if (!skip_stat)
        // Update Traffic Detection statistics for PS
        td_pck_ps_ind(vif->index, false);
    #endif //(NX_DPSM)

    PROF_PS_CHECK_TX_CLR();
    return is_trigger;
}
#endif //(NX_UAPSD || NX_DPSM)

#if (NX_UAPSD)
void ps_uapsd_set(struct vif_info_tag *vif, uint8_t hw_queue, bool uapsd)
{
    if (uapsd)
    {
        // Set the corresponding bit in the VIF parameters
        vif->u.sta.uapsd_queues |= CO_BIT(hw_queue);

        // Check if U-APSD timer is already enabled
        if (ps_env.ps_on && !ps_env.uapsd_tmr_on)
        {
            // UAPSD is used
            ps_env.uapsd_on = true;

            mm_timer_set(&ps_env.uapsd_timer, ke_time() + ps_env.uapsd_timeout);
            ps_env.uapsd_tmr_on = true;
        }
    }
    else
    {
        vif->u.sta.uapsd_queues &= ~CO_BIT(hw_queue);
    }

    // Get the TID to be used for the trigger frame transmission
    if (vif->u.sta.uapsd_queues)
    {
        uint8_t ac2tid[AC_MAX] = {TID_2, TID_0, TID_5, TID_7};
        uint8_t ac_highest = 31 - co_clz(vif->u.sta.uapsd_queues);
        vif->u.sta.uapsd_tid = ac2tid[ac_highest];
    }
}
#endif //(NX_UAPSD)

#if (NX_DPSM)
void ps_traffic_status_update(uint8_t vif_index, uint8_t new_status)
{
    // VIF Information
    struct vif_info_tag *vif;
    // Indicate if some traffic has been detected on one VIF
    bool has_traffic = (new_status != 0);
    // Indicate if PS is currently paused or not
    bool is_ps_paused;

    // Verify that PS and DPSM module is ON
    if (!ps_env.ps_on || !PS_DPSM_STATE_GET(ON))
        return;

    // Verify that DPSM is not a transition state
    if (PS_DPSM_STATE_GET(PAUSING) || PS_DPSM_STATE_GET(RESUMING))
        return;

    if (!has_traffic)
    {
        // Go through the list of used VIFs in order to know if there is another VIF with traffic
        vif = (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

        while (vif)
        {
            // Skip provided VIF
            if (vif->index != vif_index)
            {
                // Consider only active STA VIFs
                if (vif->active && (vif->type == VIF_STA))
                {
                    if (td_get_ps_status(vif->index))
                    {
                        has_traffic = true;
                        break;
                    }
                }
            }

            // Get next VIF
            vif = (struct vif_info_tag *)co_list_next(&vif->list_hdr);
        };
    }

    // Check current DPSM state
    is_ps_paused = PS_DPSM_STATE_GET(PAUSE);

    // Based on traffic detection status and current DPSM state, pause or not PS mode
    if (has_traffic)
    {
        // Traffic has been detected, pause PS mode if not already done
        if (!is_ps_paused)
        {
            ps_dpsm_update(true);
        }
    }
    else
    {
        // No traffic has been detected, go back to PSM if not already done
        if (is_ps_paused)
        {
            ps_dpsm_update(false);
        }
    }
}
#endif //(NX_DPSM)

#if (NX_P2P && NX_UAPSD)
void ps_p2p_absence_update(struct vif_info_tag *vif, bool absent)
{
    if (!vif->active)
        return;

    if (vif->type == VIF_STA)
    {
        if (absent)
        {
            if (vif->prevent_sleep & PS_VIF_WAITING_EOSP)
            {
                // Pause UAPSD service period
                vif->u.sta.sp_paused = true;
                vif->prevent_sleep &= ~PS_VIF_WAITING_EOSP;
            }
        }
        else
        {
            // When a Service Period is interrupted by a NoA, the CLI should
            // retriggers the GO when the absence period is over.
            if (vif->u.sta.sp_paused)
            {
                // Restart UAPSD service period
                vif->u.sta.sp_paused = false;
                if (txl_frame_send_qosnull_frame(vif->u.sta.ap_id,
                                                 vif->u.sta.uapsd_tid << MAC_QOSCTRL_UP_OFT,
                                                 NULL, NULL) == CO_OK)
                {
                    vif->prevent_sleep |= PS_VIF_WAITING_EOSP;
                    vif->u.sta.uapsd_last_rxtx = ke_time();
                }
            }
        }
    }
    #if NX_UMAC_PRESENT && NX_P2P_GO
    else if ((vif->type == VIF_AP) && absent)
    {
        // On going service period must be stopped when NOA starts
        struct sta_info_tag *sta = (struct sta_info_tag *)co_list_pick(&vif->sta_list);

        while (sta != NULL)
        {
            if (sta->ps_service_period & UAPSD_SERVICE_PERIOD)
            {
                if (sta->ps_service_period & UAPSD_SERVICE_PERIOD_HOST)
                    mm_traffic_req_ind(sta->staid, PS_SP_INTERRUPTED, true);
                sta->ps_service_period = NO_SERVICE_PERIOD;
            }
            sta = (struct sta_info_tag *)sta->list_hdr.next;
        }
    }
    #endif // NX_P2P_GO
}
#endif //(NX_P2P && NX_UAPSD)

#if (NX_UAPSD && NX_UMAC_PRESENT)
void ps_check_tx_trigger_sent(struct hostdesc *hostdesc, uint32_t tx_status)
{
    struct vif_info_tag *vif;

    if (!(hostdesc->flags & TXU_CNTRL_UASPD_TRIGGER) ||
        !(tx_status & FRAME_SUCCESSFUL_TX_BIT))
        return;

    vif = &vif_info_tab[hostdesc->vif_idx];
    vif->prevent_sleep |= PS_VIF_WAITING_EOSP;
}

#endif //(NX_UAPSD && NX_UMAC_PRESENT)

#endif

/// @}
