/**
 ****************************************************************************************
 *
 * @file rwnx.c
 *
 * @brief This file contains the implementation of the nX-MAC platform APIs.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup MACSW
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "version.h"
#include "dbg.h"
#include "ke_event.h"
#include "mm.h"
#include "hal_machw.h"
#include "rxl_cntrl.h"
#include "txl_cntrl.h"
#include "ps.h"
#include "rwnx.h"
#include "macif.h"
#if (NX_TX_FRAME)
#include "vif_mgmt.h"
#endif //(NX_TX_FRAME)
#if NX_UMAC_PRESENT
#include "me.h"
#endif
#ifdef CFG_RTOS
#include "rtos.h"
#endif
#if NX_FULLY_HOSTED
#include "fhost.h"
#endif

/*
 * GLOBAL VARIABLES DECLARATIONS
 ****************************************************************************************
 */
#if NX_POWERSAVE
/// Define global rwnx status
struct rwnx_env_tag
{
    /// Previous state of the HW, prior to go to DOZE
    uint8_t prev_hw_state;
    /// Flag indicating if the HW is in Doze, when waking-up
    bool hw_in_doze;
};

/// RWNX context
static struct rwnx_env_tag rwnx_env;
#endif

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */

void rwnxl_init(void)
{
    // Initialize trace buffer
    trace_init(false, true);

    #if NX_POWERSAVE
    rwnx_env.hw_in_doze = false;
    #endif

    // Embedded modules initialization
    dbg_init();

    #if NX_UMAC_PRESENT
    // UMAC initialization
    me_init();
    #endif

    // LMAC initialization
    mm_init();

    // MAC kernel initialization
    ke_init();

    #if NX_SYS_STAT
    // Reset the system statistic logging
    dbg_sys_stat_reset();
    #endif

    // Initialize the MAC interface layer
    macif_init();
}

void rwnxl_start(void)
{
    // Print version info
    dbg(D_CRT "%s - build: %s\n", nx_version_str, nx_build_date);
    #if NX_FULLY_HOSTED
    dbg(D_CRT "%s", nx_build_fhost);
    #endif

    // Print profiling configuration information
    #if NX_PROFILING_ON
    if (DBG_PROF_MAX != 0)
    {
        dbg(D_CRT "SW profiling configuration:\n");
        for (int i = 0; i < DBG_PROF_MAX; i++)
        {
            dbg(D_CRT "  - %s: %d\n", dbg_prof_conf[i], i);
        }
    }
    #endif

    #ifndef CFG_RTOS
    GLOBAL_INT_START();

    for (;;)
    {
        // schedule all pending events
        ke_evt_schedule();

        #if NX_POWERSAVE
        // then go to sleep if this is possible
        GLOBAL_INT_DISABLE();
        // check if there aren't any pending events
        if (rwnxl_cpu_can_sleep())
        {
            // perform the required operations to go to sleep
            rwnxl_sleep();

            // wait for interrupt
            WFI();

            // perform the required operations to exit sleep
            rwnxl_wakeup();
        }

        // exit from sleep
        GLOBAL_INT_RESTORE();
        #endif
    }

    #else
    if (rtos_init())
    {
        ASSERT_ERR(0);
    }

    #if NX_FULLY_HOSTED
    if (fhost_init(false) || fhost_application_init())
    {
        ASSERT_ERR(0);
    }

    #else
    if (rtos_wifi_task_create())
    {
        ASSERT_ERR(0);
    }
    #endif // NX_FULLY_HOSTED

    // Start the scheduler
    rtos_start_scheduler();
    for( ;; );

    #endif // CFG_RTOS
}


void rwnxl_reset_evt(int dummy)
{
    // Do the full reset procedure with interrupt disabled
    GLOBAL_INT_DISABLE();

    // Clear the reset event
    ke_evt_clear(KE_EVT_RESET_BIT);

    #if NX_DEBUG_DUMP
    // Get HW diagnostic port state - need to do that before resetting the HW
    hal_machw_get_diag_state();
    #endif

    // Reset the MAC HW (this will reset the PHY too)
    hal_machw_reset();

    #if NX_DEBUG_DUMP
    // Signal the error to the host
    dbg_error_ind(dbg_env.error, DBG_ERROR_RECOVERABLE);
    #endif

    // Reset the RX path
    rxl_reset();

    // Reset the TX path
    txl_reset();

    // Reset the MM
    mm_reset();

    #if (NX_TX_FRAME)
    // Push postponned internal frames
    vif_mgmt_reset();
    #endif //(NX_TX_FRAME)

    // Restore the interrupts
    GLOBAL_INT_RESTORE();
}

bool rwnxl_cpu_can_sleep(void)
{
    return (ke_evt_get() == 0);
}

void rwnxl_sleep(void)
{
    #if NX_POWERSAVE
    // Check if PS allows sleeping
    if (!ps_sleep_check())
        return;

    // Check if TX path allows sleeping
    if (!txl_sleep_check())
        return;

    // Check the HW timers
    if (!hal_machw_sleep_check())
        return;

    // Store the current HW state to recover it at wake-up
    rwnx_env.prev_hw_state = nxmac_current_state_getf();

    // Ask HW to go to IDLE
    if (nxmac_current_state_getf() != HW_IDLE)
    {
        nxmac_enable_master_gen_int_en_setf(0);
        nxmac_next_state_setf(HW_IDLE);
        while (nxmac_status_idle_interrupt_getf() != 1);
        nxmac_gen_int_ack_clear(NXMAC_IDLE_INTERRUPT_BIT);
        nxmac_enable_master_gen_int_en_setf(1);
    }

    TRACE_LMAC(DOZE, "MACHW enters DOZE mode");

    // Ask HW to go to DOZE
    DBG_DOZE_START();
    rwnx_env.hw_in_doze = true;
    nxmac_next_state_setf(HW_DOZE);
    #endif // NX_POWERSAVE
}

void rwnxl_wakeup(void)
{
    #if NX_POWERSAVE
    if (rwnx_env.hw_in_doze)
    {
        nxmac_enable_master_gen_int_en_setf(0);
        // Start the Wake-Up from doze procedure
        nxmac_wake_up_from_doze_setf(1);
        rwnx_env.hw_in_doze = false;

        // Wait for idle interrupt
        while (nxmac_status_idle_interrupt_getf() != 1);
        nxmac_gen_int_ack_clear(NXMAC_IDLE_INTERRUPT_BIT);
        nxmac_enable_master_gen_int_en_setf(1);

        // Update the system statistics
        DBG_DOZE_END();

        // Move back to the previous state
        if (rwnx_env.prev_hw_state != HW_IDLE)
        {
            nxmac_next_state_setf(rwnx_env.prev_hw_state);
            while (nxmac_current_state_getf() != rwnx_env.prev_hw_state);
        }

        // Wake-Up from doze procedure is done
        nxmac_wake_up_from_doze_setf(0);

        TRACE_LMAC(DOZE, "MACHW exits DOZE mode");
    }
    #endif // NX_POWERSAVE
}


/// @}
