/**
 ****************************************************************************************
 * @file rxl_cntrl.c
 *
 * @brief Implementation of Rx path APIs.
 *
 * Copyright (C) RivieraWaves 2011-2019
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup RX_CNTRL
 * @ingroup RX
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "co_int.h"
// For bool
#include "co_bool.h"
// For NULL
#include <string.h>
// For INLINE
#include "compiler.h"
// RX control inclusions
#include "rxl_cntrl.h"
// For KE Event calls
#include "ke_event.h"
// For frame type
#include "mac_frame.h"
// For power-save processing
#include "ps.h"
#include "sta_mgmt.h"
#include "vif_mgmt.h"
// Communication with upper layers
#include "macif.h"
// IPC DMA descriptor allocation
#include "dma.h"
// IPC host buffer allocation
#include "hal_machw.h"
// STA Info table
#include "mm.h"
// Function prototypes and HW desc macros
#include "rxl_hwdesc.h"
// For debug functions
#include "dbg.h"
// For IE access functions
#include "mac_ie.h"

#if (NX_P2P)
// P2P Definitions and Functions
#include "p2p.h"
#endif //(NX_P2P)

#if (NX_TD)
/// Traffic Detection Module Definition
#include "td.h"
#endif //(NX_TD)

#if (RW_BFMER_EN)
/// TX Beamforming Module Definition
#include "bfr.h"
#endif //(RW_BFMER_EN)

#if (NX_REORD)
#include "mac_frame.h"
#endif //(NX_REORD)

#if (NX_UMAC_PRESENT)
#include "rxu_cntrl.h"
#include "apm.h"
#endif

/// TDLS module definition
#include "tdls.h"

/*
 * DEFINES
 ****************************************************************************************
 */
/// Reception timeout
#define RX_TIMEOUT       200  ///< 200us

/// Threshold of frame upload preparation before which we handle the DMA interrupts
#define RX_FRAME_PREP_THD     4

/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
struct rxl_cntrl_env_tag rxl_cntrl_env;

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Pick an upload control descriptor from the pending queue.
 *
 * @return Pointer to the picked upload control descriptor
 ****************************************************************************************
 */
__INLINE struct rx_upload_cntrl_tag *rxl_upload_cntrl_pick_pending(void)
{
    return ((struct rx_upload_cntrl_tag *)co_list_pick(&rxl_cntrl_env.upload_pending));
}

/**
 ****************************************************************************************
 * @brief Pop an upload control descriptor from the pending queue.
 *
 * @return Pointer to the popped upload control descriptor
 ****************************************************************************************
 */
__INLINE struct rx_upload_cntrl_tag *rxl_upload_cntrl_pop_pending(void)
{
    return ((struct rx_upload_cntrl_tag *)co_list_pop_front(&rxl_cntrl_env.upload_pending));
}

/**
 ****************************************************************************************
 * @brief   This function checks if the IPC DMA already processed the transfers up to the
 * target count.
 *
 * @param[in] next_lli_cnt  The target LLI count
 *
 * @return true if the target is reached or exceeded, false otherwise
 *
 ****************************************************************************************
 */
static bool rxl_lli_done(uint16_t next_lli_cnt)
{
    return (((uint16_t)(dma_lli_counter_get(IPC_DMA_LLI_DATA_RX0) - (next_lli_cnt)))
                                                                 < (((uint16_t)-1) / 2));
}

#if (NX_RX_FRAME_HANDLING)
#if (NX_UMAC_PRESENT || NX_P2P_GO)
/**
 ****************************************************************************************
 * @brief This function processes Power Management information in a received frames
 *
 * For AP interfaces, it checks if a STA enters of leaves Power Save mode and takes
 * corresponding action.
 *
 * @param[in] frame   Pointer to the MAC header of the frame
 * @param[in] sta_idx Index of the sending sta
 * @param[in] vif_idx Index of the VIF that received the frame
 ****************************************************************************************
 */
static void rxl_pm_check(uint8_t *frame, uint8_t sta_idx, uint8_t vif_idx)
{
    struct vif_info_tag *vif = &vif_info_tab[vif_idx];
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    uint16_t frame_ctrl;

    // Check the PM state only in AP mode
    if (vif->type != VIF_AP)
        return;

    #if (!NX_UMAC_PRESENT)
    if (!vif->p2p)
        return;
    #endif //(!NX_UMAC_PRESENT)

    // Get the frame control field from the packet
    frame_ctrl = co_read16(frame);

    // Check if the station is in PS or not
    if (sta->ps_state == PS_MODE_ON)
    {
        // Check if the peer device is waking up: PWRMGMT bit and MOREFRAG bit have to
        // be set to 0 in a Data or Management frame
        uint16_t test_val = frame_ctrl & (MAC_FCTRL_TYPE_MASK |
                                          MAC_FCTRL_PWRMGT | MAC_FCTRL_MOREFRAG);
        if ((test_val == MAC_FCTRL_DATA_T) || (test_val == MAC_FCTRL_MGT_T))
        {
            #if (NX_UMAC_PRESENT)
            PROF_PS_PEER_STATE_SET();
            PROF_PS_STATE_VAL_SET(PS_MODE_OFF);
            // Indicate the PS mode change to the upper layers
            TRACE_AP(PS, "{VIF-%d} {STA-%d} exit PS mode", vif_idx, sta_idx);
            mm_ps_change_ind(sta_idx, PS_MODE_OFF);
            #if NX_BEACONING
            apm_tx_int_ps_clear(vif, sta_idx);
            #endif // NX_BEACONING
            PROF_PS_PEER_STATE_CLR();
            #else
            sta->ps_state = PS_MODE_OFF;
            #endif //(NX_UMAC_PRESENT)

            // Update the number of PS stations
            vif->u.ap.ps_sta_cnt--;

            #if (NX_UMAC_PRESENT)
            if (!vif->u.ap.ps_sta_cnt)
            {
                PROF_PS_BCMC_STATE_SET();
                PROF_PS_STATE_VAL_SET(PS_MODE_OFF);
                TRACE_AP(PS, "{VIF-%d} disable PS on broadcast/multicast STA", vif_idx);
                mm_ps_change_ind(VIF_TO_BCMC_IDX(vif_idx), PS_MODE_OFF);
                #if NX_BEACONING
                apm_tx_int_ps_clear(vif, VIF_TO_BCMC_IDX(vif_idx));
                #endif // NX_BEACONING
                PROF_PS_BCMC_STATE_CLR();
            }
            #endif //(NX_UMAC_PRESENT)

            #if (NX_P2P_GO)
            p2p_go_ps_state_update(vif);
            #endif //(NX_P2P_GO)
            return;
        }

        #if (NX_UMAC_PRESENT)
        // Check if the received frame shall trigger some transmissions from us
        if ((frame_ctrl & MAC_FCTRL_TYPESUBTYPE_MASK) == MAC_FCTRL_PSPOLL)
        {
            PROF_PS_PSPOLL_RX_SET();
            TRACE_AP(PS, "{VIF-%d} received PS-Poll from STA-%d", vif->index, sta_idx);

            #if (NX_BEACONING)
            if (sta->traffic_avail & PS_TRAFFIC_INT)
            {
                // U-APSD service period may be already on going
                sta->ps_service_period |= PS_SERVICE_PERIOD;
                sta_mgmt_send_postponed_frame(vif, sta, 1);
                sta->ps_service_period &= ~PS_SERVICE_PERIOD;
            }
            else
            #endif // NX_BEACONING
                mm_traffic_req_ind(sta_idx, 1, false);

            PROF_PS_PSPOLL_RX_CLR();
        }

        // Check if the received frame shall trigger some transmissions from us
        if ((frame_ctrl & (MAC_FCTRL_TYPE_MASK | MAC_QOS_ST_BIT)) ==
            (MAC_FCTRL_DATA_T | MAC_QOS_ST_BIT))
        {
            uint8_t tid;

            // Get the TID
            if ((frame_ctrl & (MAC_FCTRL_TODS | MAC_FCTRL_FROMDS)) ==
                (MAC_FCTRL_TODS | MAC_FCTRL_FROMDS))
            {
                struct mac_hdr_long_qos *hdr = (struct mac_hdr_long_qos *)frame;
                // Get the QoS control field in the frame and the TID
                tid = (hdr->qos & MAC_QOSCTRL_UP_MSK) >> MAC_QOSCTRL_UP_OFT;
            }
            else
            {
                struct mac_hdr_qos *hdr = (struct mac_hdr_qos *)frame;
                // Get the QoS control field in the frame and the TID
                tid = (hdr->qos & MAC_QOSCTRL_UP_MSK) >> MAC_QOSCTRL_UP_OFT;
            }

            // Check if the received frame is a UAPSD trigger frame
            if (mac_ac2uapsd[mac_tid2ac[tid]] & sta->info.uapsd_queues)
            {
                TRACE_AP(PS, "{VIF-%d} received UAPSD trigger from STA-%d sn=%d",
                         vif->index, sta_idx,
                         ((struct mac_hdr *)frame)->seq >> MAC_SEQCTRL_NUM_OFT);

                // Check if UAPSD traffic is available and
                // no SP is already ongoing
                if ((sta->traffic_avail & UAPSD_TRAFFIC) &&
                    !(sta->ps_service_period & UAPSD_SERVICE_PERIOD))
                {
                    int max_sp = sta->info.max_sp_len;

                    #if (NX_BEACONING)
                    // First transmit internal frames if any
                    if (sta->traffic_avail & UAPSD_TRAFFIC_INT)
                    {
                        int nb;
                        sta->ps_service_period = UAPSD_SERVICE_PERIOD_INT;
                        nb = sta_mgmt_send_postponed_frame(vif, sta, max_sp);

                        if (max_sp)
                        {
                            max_sp -= nb;
                            if (!max_sp)
                                max_sp = -1;
                        }
                    }

                    if (max_sp < 0 || !(sta->traffic_avail & UAPSD_TRAFFIC_HOST))
                    {
                        // If no more space or no host traffic end service period.
                        uint16_t qos = MAC_QOSCTRL_EOSP | (tid << MAC_QOSCTRL_UP_OFT);
                        TRACE_AP(PS, "{VIF-%d} SP completed by fw: send EOSP to STA-%d",
                                 vif->index, sta_idx);
                        txl_frame_send_qosnull_frame(sta->staid, qos, NULL, NULL);
                        sta->ps_service_period = NO_SERVICE_PERIOD;
                    }
                    else if (sta->traffic_avail & UAPSD_TRAFFIC_HOST)
                    #endif // (NX_BEACONING)
                    {
                        // Ask host for packets
                        TRACE_AP(PS, "{VIF-%d} Ask host to complete SP for STA-%d",
                                 vif->index, sta_idx);
                        sta->ps_service_period = UAPSD_SERVICE_PERIOD_HOST;
                        mm_traffic_req_ind(sta_idx, max_sp, true);
                    }
                }
                else if (!(sta->ps_service_period & UAPSD_SERVICE_PERIOD))
                {
                    // No traffic available. Reply with a QoS NULL frame
                    // indicating the end of service period
                    uint16_t qos = MAC_QOSCTRL_EOSP | (tid << MAC_QOSCTRL_UP_OFT);
                    sta->ps_service_period = UAPSD_SERVICE_PERIOD_INT;
                    txl_frame_send_qosnull_frame(sta->staid, qos, NULL, NULL);
                    TRACE_AP(PS, "{VIF-%d} no data: send EOSP to STA-%d",
                             vif->index, sta_idx);
                    sta->ps_service_period = NO_SERVICE_PERIOD;
                }
            }
        }
        #endif //(NX_UMAC_PRESENT)
    }
    else
    {
        // Check if the peer device is going to sleep
        uint16_t test_val = frame_ctrl & (MAC_FCTRL_TYPE_MASK |
                                          MAC_FCTRL_PWRMGT | MAC_FCTRL_MOREFRAG);
        if ((test_val == (MAC_FCTRL_DATA_T | MAC_FCTRL_PWRMGT)) ||
            (test_val == (MAC_FCTRL_MGT_T | MAC_FCTRL_PWRMGT)))
        {
            #if (NX_UMAC_PRESENT)
            PROF_PS_PEER_STATE_SET();
            PROF_PS_STATE_VAL_SET(PS_MODE_ON);
            // Indicate the PS mode change to the upper layers
            TRACE_AP(PS, "{VIF-%d} {STA-%d} enter PS mode", vif_idx, sta_idx);
            mm_ps_change_ind(sta_idx, PS_MODE_ON);
            PROF_PS_PEER_STATE_CLR();

            // Update the number of PS stations
            if (!vif->u.ap.ps_sta_cnt)
            {
                PROF_PS_BCMC_STATE_SET();
                PROF_PS_STATE_VAL_SET(PS_MODE_ON);
                TRACE_AP(PS, "{VIF-%d} enable PS on broadcast/multicast STA", vif_idx);
                mm_ps_change_ind(VIF_TO_BCMC_IDX(vif_idx), PS_MODE_ON);
                PROF_PS_BCMC_STATE_CLR();
            }
            #else
            sta->ps_state = PS_MODE_ON;
            #endif //(NX_UMAC_PRESENT)

            // Update the number of PS stations
            vif->u.ap.ps_sta_cnt++;

            #if (NX_P2P_GO)
            p2p_go_ps_state_update(vif);
            #endif //(NX_P2P_GO)
            return;
        }
    }
}
#endif //(NX_UMAC_PRESENT || NX_P2P_GO)

/**
 ****************************************************************************************
 * @brief This function processes the received frames that could carry useful information
 * for some LMAC features (connection monitoring, power-save mode, etc.)
 *
 * @param[in]  rxdesc    SW header descriptor of the frame
 * @param[out] dont_free Whether this frame should be freed if not uploaded or not
 *
 * @return true if the frame shall be uploaded to UMAC, false otherwise
 ****************************************************************************************
 */
static uint8_t rxl_frame_handle(struct rxdesc* rxdesc, bool *dont_free)
{
    struct sta_info_tag *sta;
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    struct rx_hd *rhd = &dma_hdrdesc->hd;
    struct rx_pbd *pd;
    uint32_t statinfo;
    uint16_t key_idx_hw;
    uint8_t sta_idx;
    struct vif_info_tag *vif;
    uint8_t *frame;
    uint16_t framectrl;
    bool upload = true;

    do
    {
        // Ensure the packet has a length
        if (!rhd->frmlen)
            return false;

        // Sanity check: frames with length not NULL have at least 1 buffer descriptor
        ASSERT_REC_VAL(rhd->first_pbd_ptr != 0, false);

        // Get value of the buffer descriptor pointer
        pd = HW2CPU(rhd->first_pbd_ptr);

        // Get the status information from the RX header descriptor
        statinfo = rhd->statinfo;

        // Get the pointer to the frame data
        frame = HW2CPU(pd->datastartptr);

        // Get the frame control
        framectrl = co_read16(frame);

        // Check if the frame comes from a known device and is valid
        if ((statinfo & (RX_HD_KEYIDV | RX_HD_SUCCESS)) != (RX_HD_KEYIDV | RX_HD_SUCCESS))
        {
            TRACE_LMAC(RX_ALL, "from unknown STA: SN=%d %fc",
                       ((struct mac_hdr *)frame)->seq >> MAC_SEQCTRL_NUM_OFT, framectrl);
            break;
        }

        // Get the HW key index
        key_idx_hw = (uint16_t)RX_HD_KEYID_GET(statinfo);

        // Sanity check
        ASSERT_REC_VAL(key_idx_hw >= MM_SEC_DEFAULT_KEY_COUNT, false);

        // Retrieve the station index and instance number
        sta_idx = MM_KEY_TO_STA(key_idx_hw);

        // Check if the STA is registered
        if (!sta_mgmt_is_valid(sta_idx))
        {
            rhd->statinfo &= ~RX_HD_KEYIDV;
            TRACE_LMAC(RX_ALL, "from invalid STA-%d: SN=%d %fc", sta_idx,
                       ((struct mac_hdr *)frame)->seq >> MAC_SEQCTRL_NUM_OFT, framectrl);
            break;
        }

        // Get the pointer to the frame data
        frame = HW2CPU(pd->datastartptr);

        // Get the frame control
        framectrl = co_read16(frame);

        // Retrieve the associated STA entry
        sta = &sta_info_tab[sta_idx];

        // Check if we received this packet from the reference BSSID, while we are
        // connected to a nonTransmittedBSSID of the same Multiple BSSID set
        if (sta->aid == STA_REF_BSSID_AID)
        {
            // We received such a packet - we handle only the beacon case, because we
            // need to synchronize our TBTT and check the TIM element on this beacon
            if ((framectrl & MAC_FCTRL_TYPESUBTYPE_MASK) != MAC_FCTRL_BEACON)
            {
                rhd->statinfo &= ~RX_HD_KEYIDV;
                break;
            }

            // Replace the key index and station pointer by the one of the nonTransmitted
            // BSSID - This will allow handling the beacon as if it would have been sent
            // by our BSSID
            sta = sta->linked_sta;
            rhd->statinfo &= ~RX_HD_KEYID;
            rhd->statinfo |= MM_STA_TO_KEY(sta->staid) << RX_HD_KEYID_LSB;
        }

        // Retrieve the associated VIF entry
        vif = &vif_info_tab[sta->inst_nbr];

        TRACE_LMAC(RX, "{VIF-%d} from STA-%d: SN=%d %fc", sta->inst_nbr, sta_idx,
                   ((struct mac_hdr *)frame)->seq >> MAC_SEQCTRL_NUM_OFT, framectrl);

        #if (NX_UMAC_PRESENT || NX_P2P_GO)
        rxl_pm_check(frame, sta_idx, vif->index);
        #endif //(NX_UMAC_PRESENT || NX_P2P_GO)

        #if (RW_BFMER_EN)
        {
            uint8_t bfr_status = (bfr_is_enabled()) ? bfr_rx_frame_ind(sta_idx, rxdesc, frame)
                                                    : BFR_RX_STATUS_NOT_VALID;

            if (bfr_status != BFR_RX_STATUS_NOT_VALID)
            {
                // Frame is well a beamforming report, not needed to upload the whole frame.
                upload = false;

                if (bfr_status == BFR_RX_STATUS_VALID)
                {
                    *dont_free = true;
                }

                break;
            }
        }
        #endif //(RW_BFMER_EN)

        // Check if this STA interface is associated with an AP
        if (!vif->active)
            break;

        #if (NX_TD)
        if (((framectrl & MAC_FCTRL_TYPE_MASK) == MAC_FCTRL_DATA_T) ||
            ((framectrl & MAC_FCTRL_TYPE_MASK) == MAC_FCTRL_MGT_T))
        {
            td_pck_ind(vif->index, sta_idx, true);
        }
        #endif //(NX_TD)

        // Check if this interface is a STA or a MP interface
        if (vif->type == VIF_STA)
        {
            // Decode the frame control to know if we have to handle the frame internally
            if ((framectrl & MAC_FCTRL_TYPESUBTYPE_MASK) == MAC_FCTRL_BEACON)
            {
                #if NX_POWERSAVE || NX_CONNECTION_MONITOR || NX_MULTI_ROLE
                uint32_t tim = 0;
                #endif

                TRACE_STA(BCN, "{VIF-%d} Beacon received sn=%d", vif->index,
                          ((struct mac_hdr *)frame)->seq >> MAC_SEQCTRL_NUM_OFT);

                #if NX_CONNECTION_MONITOR || NX_MULTI_ROLE
                // Let the MM handle the connection monitoring procedures
                upload = mm_check_beacon(rhd, vif, sta, &tim);
                #elif NX_POWERSAVE
                tim = mac_ie_tim_find(pd->datastartptr + MAC_BEACON_VARIABLE_PART_OFT,
                                      rhd->frmlen - MAC_BEACON_VARIABLE_PART_OFT);
                #endif

                #if NX_POWERSAVE
                // Let the PS module check if it has something to do with this beacon
                ps_check_beacon(tim, rhd->frmlen, vif);
                #endif

                #if (NX_P2P || NX_CHNL_CTXT)
                vif_mgmt_bcn_recv(vif);
                #endif //(NX_P2P || NX_CHNL_CTXT)

                #if (NX_P2P)
                if (upload)
                {
                    // Look for NOA Attribute only if beacon has been modified
                    p2p_cli_bcn_check_noa(vif, pd, dma_hdrdesc);
                }
                #endif //(NX_P2P)
            }
            else if (((framectrl & MAC_FCTRL_TYPE_MASK) == MAC_FCTRL_DATA_T) ||
                     ((framectrl & MAC_FCTRL_TYPE_MASK) == MAC_FCTRL_MGT_T))
            {
                #if NX_POWERSAVE
                // Let the PS module check if it has something to do with this frame
                ps_check_frame(frame, statinfo, vif);
                #endif

                #if (NX_P2P)
                if (vif->p2p)
                {
                    if ((framectrl & MAC_FCTRL_TYPESUBTYPE_MASK) == MAC_FCTRL_ACTION)
                    {
                        p2p_cli_handle_action(vif, pd->datastartptr, rhd->frmlen, rhd->tsflo);
                    }
                }
                #endif //(NX_P2P)

                #if NX_UMAC_PRESENT && NX_TDLS
                // Check if the frame comes from a TDLS STA and if TDLS STA exists
                if (sta->is_tdls)
                {
                    upload = tdls_check_frame(frame, vif, sta_idx);
                }
                #endif // NX_UMAC_PRESENT && NX_TDLS
            }
        }
        #if (RW_UMESH_EN)
        else if (vif->type == VIF_MESH_POINT)
        {
            // Decode the frame control to know if we have to handle the frame internally
            if ((framectrl & MAC_FCTRL_TYPESUBTYPE_MASK) == MAC_FCTRL_BEACON)
            {
                uint32_t tim;

                // Extract the TIM
                tim = mac_ie_tim_find(pd->datastartptr + MAC_BEACON_VARIABLE_PART_OFT,
                                      rhd->frmlen - MAC_BEACON_VARIABLE_PART_OFT);
                mm_tbtt_compute((struct bcn_frame *)frame, rhd->frmlen, rhd, vif,
                                sta, tim);

                // Provide the beacon to the Mesh PS Module
                mesh_ps_rx_beacon_handle(vif->index, sta_idx,
                                         pd->datastartptr, rhd->frmlen);

                // Do not upload the beacon
                upload = false;
            }
        }
        #endif //(RW_UMESH_EN)
    } while (0);

    return (upload);
}
#endif //(NX_RX_FRAME_HANDLING)

/**
 ****************************************************************************************
 * @brief   This function initializes the Rx Context Data.
 ****************************************************************************************
 */
static void rxl_cntrl_init(void)
{
    // List for frames pending for upload.
    co_list_init(&rxl_cntrl_env.upload_pending);

    // Initialize the bridge DMA counts
    rxl_cntrl_env.bridgedmacnt = dma_lli_counter_get(IPC_DMA_LLI_DATA_RX0);
    #if !NX_FULLY_HOSTED
    rxl_cntrl_env.packet_thd = macif_rx_get_packet_threshold() - 1;
    rxl_cntrl_env.packet_cnt = rxl_cntrl_env.packet_thd;
    #endif
}

#if !NX_FULLY_HOSTED
/**
 ****************************************************************************************
 * @brief Start the host interrupt mitigation timer.
 *
 * The timer is started upon each frame upload confirmation in case the packet count is
 * different from its default value.
 ****************************************************************************************
 */
static void rxl_host_irq_mitigation_timeout_set(void)
{
    // Check if we need to trigger an interrupt to the host
    if (rxl_cntrl_env.packet_cnt != rxl_cntrl_env.packet_thd)
    {
        // Restart the RX IRQ mitigation timeout
        nxmac_abs_timer_set(HAL_RX_TIMER, hal_machw_time() + RX_TIMEOUT);
        nxmac_timers_int_event_clear(HAL_RX_TIMER_BIT);
        nxmac_timers_int_un_mask_set(nxmac_timers_int_un_mask_get() | HAL_RX_TIMER_BIT);
    }
}

void rxl_host_irq_mitigation_update(void *env)
{
    // Update the uploaded packet count
    rxl_cntrl_env.packet_cnt++;

    // Check if we need to trigger an interrupt to the host
    if (rxl_cntrl_env.packet_cnt > rxl_cntrl_env.packet_thd)
    {
        // Indicate the packets to the host
        macif_rx_data_ind();
        rxl_cntrl_env.packet_cnt = 0;
    }
}

void rxl_timeout_int_handler(void)
{
    // Current A-MPDU reception is considered as finished
    // Check if we have some packets to indicate to the host
    if (rxl_cntrl_env.packet_cnt > 0)
    {
        // Some packets are available, so indicate them
        macif_rx_data_ind();
    }

    // Reset the packet counter in order to force the host to be warned upon the next
    // A-MPDU reception start
    rxl_cntrl_env.packet_cnt = rxl_cntrl_env.packet_thd;

    // Disable the RX timeout interrupt
    nxmac_timers_int_un_mask_set(nxmac_timers_int_un_mask_get() & ~HAL_RX_TIMER_BIT);
}
#endif

void rxl_init(void)
{
    // initialize the header descriptors
    rxl_hwdesc_init();

    // initialize the RX environment
    rxl_cntrl_init();

    #if NX_UMAC_PRESENT
    // initialize the RXU environment
    rxu_cntrl_init();
    #endif
}

void rxl_cntrl_evt(int dummy)
{
    struct rxdesc *rxdesc;
    #if NX_RX_FRAME_HANDLING
    uint8_t upload;
    #endif
    int prep_cnt = 0;

    // for profiling
    RX_CNTRL_EVT_SET();

    while (1)
    {
        // Pick the first SW descriptor ready for processing
        rxdesc = rxl_rxdesc_get();

        #if NX_RX_RING
        // clear the interrupt
        nxmac_tx_rx_int_ack_clear(NXMAC_RX_BUFFER_1_TRIGGER_BIT);
        #endif

        // Clear the kernel event
        ke_evt_clear(KE_EVT_RXREADY_BIT);

        // Check if we still have ready descriptors
        if (rxdesc == NULL)
            break;

        // Check if we reached the threshold for DMA interrupt handling
        if (prep_cnt >= RX_FRAME_PREP_THD)
        {
            // Set the kernel event to handle next frames immediately after the DMA
            ke_evt_set(KE_EVT_RXREADY_BIT);

            // for profiling
            RX_CNTRL_EVT_CLR();
            return;
        }

        #if (NX_UMAC_PRESENT)
        // Check if a host buffer and a descriptor are available for DMA transfer
        if (!macif_rx_buf_check()
         || !rxu_cntrl_desc_check())
        #else
        // Check if a host buffer is available for DMA transfer
        if (!macif_rx_buf_check())
        #endif //(NX_UMAC_PRESENT)
        {
            // No host buffers available, so stop the handling of this packet.
            // The IPC will then call again the RX data handler once a buffer is available
            // for profiling
            RX_CNTRL_EVT_CLR();
            return;
        }

        // Upload is possible, indicate that the RX descriptor will be processed
        rxl_rxdesc_ready_for_processing(rxdesc);

        do
        {
            #if (NX_RX_FRAME_HANDLING)
            bool dont_free = false;

            // Check if the packet is of interest for the LMAC
            upload = rxl_frame_handle(rxdesc, &dont_free);

            if (!upload)
            {
                if (!dont_free)
                {
                    // No need to upload this packet, free it immediately
                    rxl_mpdu_free(rxdesc);
                }

                break;
            }
            #endif //(NX_RX_FRAME_HANDLING)

            #if NX_UMAC_PRESENT
            // Perform the UMAC related handling
            upload = rxu_cntrl_frame_handle(rxdesc);
            if (!upload)
            {
                // No need to upload this packet, free it immediately
                rxl_mpdu_free(rxdesc);
                break;
            }
            #else
            // Try to upload the frame
            rxl_mpdu_transfer(rxdesc);
            #endif
        } while (0);

        // Increase the counter
        prep_cnt++;
    }

    #if NX_RX_RING
    nxmac_enable_rx_buffer_1_trigger_setf(1);
    #endif

    // for profiling
    RX_CNTRL_EVT_CLR();
}

void rxl_dma_int_handler(void)
{
    // For profiling
    PROF_RX_DMA_IRQ_SET();

    // Disable DMA IRQ for the RX. It will be reenabled by the associated event
    dma_lli_disable(IPC_DMA_LLI_DATA_RX0 + DMA_LLI_IRQ_LSB);

    // Acknowledge the interrupt
    dma_int_ack_clear(CO_BIT(IPC_DMA_LLI_DATA_RX0 + DMA_LLI_IRQ_LSB));

    ke_evt_set(KE_EVT_RXUPLOADED_BIT);

    // For profiling
    PROF_RX_DMA_IRQ_CLR();
}

void rxl_dma_evt(int dummy)
{
    struct rx_upload_cntrl_tag *upload_cntrl = rxl_upload_cntrl_pick_pending();

    // For profiling
    PROF_RX_DMA_EVT_SET();

    ke_evt_clear(KE_EVT_RXUPLOADED_BIT);

    // Acknowledge the interrupt at least once in case no SW descriptor is pending
    dma_int_ack_clear(CO_BIT(IPC_DMA_LLI_DATA_RX0 + DMA_LLI_IRQ_LSB));

    while (upload_cntrl)
    {
        struct rxdesc *rxdesc;

        #if NX_RX_RING
        // When a RX ring buffer is used, the freeing of buffers must be done in order.
        // We might get here buffers that were not uploaded to upper layers and therefore
        // did not trigger any LLI counter increment, but that we need to free anyway.
        if ((upload_cntrl->flags & RX_NO_UPLOAD) == 0)
        #endif
        {
            // Compute the target LLI count of the current descriptor
            uint16_t next_lli_cnt = rxl_cntrl_env.bridgedmacnt + 1;

            // if there are no more descriptors to handle
            if (!rxl_lli_done(next_lli_cnt))
            {
                break;
            }

            // Acknowledge the interrupt each time we get an LLI done to avoid getting it
            // again later
            dma_int_ack_clear(CO_BIT(IPC_DMA_LLI_DATA_RX0 + DMA_LLI_IRQ_LSB));

            // increment the handled LLI counter
            rxl_cntrl_env.bridgedmacnt = next_lli_cnt;
        }

        // Pop the descriptor
        rxl_upload_cntrl_pop_pending();

        // Both HW descriptor release and access on packet_cnt need to be protected
        GLOBAL_INT_DISABLE();

        // Call the appropriate function
        if (upload_cntrl->cb)
            upload_cntrl->cb(upload_cntrl->env);

        rxdesc = upload_cntrl->rxdesc;
        if (rxdesc)
        {
            // Release the descriptors associated with this SW descriptor
            rxl_frame_release(rxdesc);
        }

        // Re-enable the interrupts
        GLOBAL_INT_RESTORE();

        // Handle next descriptor
        upload_cntrl = rxl_upload_cntrl_pick_pending();
    }

    #if NX_FULLY_HOSTED
    // Forward the RX descriptors that are ready for that
    macif_rx_desc_upload(&rxu_cntrl_env.rxdesc_ready);
    #else
    rxl_host_irq_mitigation_timeout_set();
    #endif

    // Enable DMA LLI RX IRQ to get an interrupt once payloads are transfered
    dma_lli_enable(IPC_DMA_LLI_DATA_RX0 + DMA_LLI_IRQ_LSB);

    // For profiling
    PROF_RX_DMA_EVT_CLR();
}

void rxl_reset(void)
{
    struct rx_upload_cntrl_tag *upload_cntrl = rxl_upload_cntrl_pick_pending();

    while (upload_cntrl)
    {
        // Compute the target LLI count of the current descriptor
        uint16_t next_lli_cnt = rxl_cntrl_env.bridgedmacnt + 1;

        // Wait for the DMA transfer to be finished
        while (!rxl_lli_done(next_lli_cnt));

        // Pop the descriptor
        rxl_upload_cntrl_pop_pending();

        // Call the appropriate function
        if (upload_cntrl->cb)
            upload_cntrl->cb(upload_cntrl->env);

        #if !NX_FULLY_HOSTED
        // call the LMAC-UMAC interface function to handle the frame
        macif_rx_data_ind();
        #endif

        // increment the handled LLI counter
        rxl_cntrl_env.bridgedmacnt = next_lli_cnt;

        // handle next descriptor
        upload_cntrl = rxl_upload_cntrl_pick_pending();
    }

    #if NX_FULLY_HOSTED
    // Forward the RX descriptors that are ready for that
    macif_rx_desc_upload(&rxu_cntrl_env.rxdesc_ready);
    #endif

    // Acknowledge the DMA interrupt
    dma_int_ack_clear(CO_BIT(IPC_DMA_LLI_DATA_RX0 + DMA_LLI_IRQ_LSB));

    // Re-initialize the HW descriptor lists
    rxl_hwdesc_init();
}


/// @}
