/**
 ****************************************************************************************
 * @file v10/rxl_hwdesc.c
 *
 * @brief Implementation of the API function used to initialize the pools.
 *
 * Copyright (C) RivieraWaves 2011-2019
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup RX_HWDESC
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "rxl_hwdesc.h"

#include "ke_event.h"
#include "rxl_cntrl.h"
#if NX_UMAC_PRESENT
#include "rxu_cntrl.h"
#endif
#include "co_utils.h"
#include "macif.h"
#if NX_AMPDU_TX
#include "txl_cfm.h"
#endif
#if NX_MAC_HE
#include "txl_he.h"
#endif
#include "co_endian.h"
#include "reg_mac_pl.h"
#include "reg_mac_core.h"

/*
 * TYPE DEFINITION
 ****************************************************************************************
 */

/// Buffer Management control structure
struct rxl_hwdesc_env_tag
{
    /// Frames waiting for upload
    struct co_list ready;
    /// First DMA descriptor the SW has to process.
    struct rx_dmadesc *first_dmadesc;
    /// The last DMA descriptor in the HW list.
    struct rx_dmadesc *last_dmadesc;
    /// The first free DMA descriptor in the HW list.
    struct rx_dmadesc *free_dmadesc;
    /// First RX payload buffer descriptor chained to HW
    struct rx_pbd *first;
    /// Last RX payload buffer descriptor chained to HW
    struct rx_pbd *last;
    /// Free (spare) RX payload buffer descriptor (not chained to HW)
    struct rx_pbd *free;
    /// Flag indicating whether monitor mode is enabled or not
    bool monitor_en;
};

/*
 * VARIABLES
 ****************************************************************************************
 */
/// Array of RX SW descriptors
static struct rxdesc rx_swdesc_tab[NX_RXDESC_CNT];

/// RXL HW descriptor environment variable
static struct rxl_hwdesc_env_tag rx_hwdesc_env;

/// Array of struct containing the RX header descriptors
struct rx_dmadesc       rx_dma_hdrdesc[NX_RXDESC_CNT] __SHAREDRAM;
/// Array of struct containing the RX buffer descriptors
struct rx_payloaddesc   rx_payload_desc[NX_RX_PAYLOAD_DESC_CNT] __SHAREDRAM;

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Appends a descriptor to the list visible to HW
 * Function to initialize the freed HW desc and tail it to the list of HW descriptors. The
 * element pointers are linked and the valid bit is set. The New Head/New Tail bit is also
 * set accordingly. It also updates the pointer pointing to the last desc tailed.
 *
 * @param[in]   desc      HW desc to be freed and appended
 ****************************************************************************************
 */
static void rxl_hd_append(struct rx_dmadesc *desc)
{
    struct rx_dmadesc *free_desc;

    // sanity check: function can not be called with NULL
    ASSERT_ERR(desc != NULL);

    // check if the HW is still pointing to the spare descriptor
    if (HW2CPU(nxmac_debug_rx_hdr_c_ptr_get()) == rx_hwdesc_env.free_dmadesc)
    {
        // The HW is still pointing to the spare descriptor, therefore we cannot release
        // it now. Instead we release the current descriptor.
        free_desc = desc;
    }
    else
    {
        // Otherwise we free the spare descriptor
        free_desc = rx_hwdesc_env.free_dmadesc;

        // Save the new free descriptor
        rx_hwdesc_env.free_dmadesc = desc;
    }

    // clear the MPDU status info
    free_desc->hd.statinfo = 0;

    // reset the next pointer
    free_desc->hd.next = 0;

    // reset the first payload descriptor pointer
    free_desc->hd.first_pbd_ptr = 0;

    // reset the frame length field
    free_desc->hd.frmlen = 0;

    // Link the RHD to the current list and trigger a newTail
    rx_hwdesc_env.last_dmadesc->hd.next = CPU2HW(&(free_desc->hd));
    nxmac_dma_cntrl_set(NXMAC_RX_HEADER_NEW_TAIL_BIT);

    // record the last pointer
    rx_hwdesc_env.last_dmadesc = free_desc;

    // check if we reached the end of the RX header descriptor list
    if (rx_hwdesc_env.first_dmadesc == NULL)
    {
        rx_hwdesc_env.first_dmadesc = free_desc;
    }
}

/**
 ****************************************************************************************
 * @brief Appends a list of RX payload buffer descriptors to the list visible to HW
 * Function to initialize the freed HW desc and tail it to the list of HW descriptors. The
 * element pointers are linked and the valid bit is set. The New Head/New Tail bit is also
 * set accordingly. It also updates the pointer pointing to the last desc tailed.
 *
 * @param[in]   first      First buffer descriptor of the list
 * @param[in]   last       Last buffer descriptor of the list
 * @param[in]   spare      Spare descriptor that won't be chained immediately to HW
 ****************************************************************************************
 */
static void rxl_pd_append(struct rx_pbd *first, struct rx_pbd* last, struct rx_pbd* spare)
{
    // sanity check: the function can not be called with NULL
    ASSERT_ERR(spare != NULL);

    // check if the HW is still pointing to the spare descriptor
    if (HW2CPU(nxmac_debug_rx_pay_c_ptr_get()) == rx_hwdesc_env.free)
    {
        // The HW is still pointing to the spare descriptor, therefore we cannot release
        // it now. Instead we release the current descriptor.
        // Add the free element at the end of the list
        if (last == NULL)
        {
            // No element in the list
            first = spare;
        }
        last = spare;

        // Reset the stat info
        last->bufstatinfo = 0;
    }
    else
    {
        // Otherwise we free the spare descriptor
        struct rx_pbd *free = rx_hwdesc_env.free;

        // Save the new free descriptor
        rx_hwdesc_env.free = spare;

        // Add the free element at the end of the list
        if (last == NULL)
        {
            // No element in the list
            first = last = free;
        }
        else
        {
            free->next = CPU2HW(first);
            first = free;
        }

        // Reset the stat info
        first->bufstatinfo = 0;
    }

    // reset the next pointer
    last->next = 0;

    // Link the descriptors to the current list and trigger a newTail
    rx_hwdesc_env.last->next = CPU2HW(first);
    nxmac_dma_cntrl_set(NXMAC_RX_PAYLOAD_NEW_TAIL_BIT);

    // record the last pointer
    rx_hwdesc_env.last = last;

    // check if the pointer to the first descriptor is NULL (debug and error logs only)
    if (rx_hwdesc_env.first == NULL)
    {
        rx_hwdesc_env.first = first;
    }
}

#if NX_AMSDU_DEAGG
/**
 ****************************************************************************************
 * @brief Compute the A-MSDU subframe length to be uploaded from a given payload
 * descriptor.
 * The function also checks if the A-MSDU subframe can fit into the remaining MPDU length.
 *
 * @param[in] pd          Pointer to the payload descriptor in which the A-MSDU subframe
 *                        is supposed to start
 * @param[in] payl_offset Offset in the payload, where the A-MSDU subframe is supposed to
 *                        start
 * @param[in] mpdu_len    Remaining MPDU length from offset to the end of the MPDU
 *
 * @return    If the A-MSDU subframe is valid, its length, including the possible padding.
 *            Otherwise 0.
 ****************************************************************************************
 */
static uint16_t rxl_amsdu_subframe_len_get(struct rx_payloaddesc *pd,
                                           uint16_t payl_offset,
                                           uint16_t mpdu_len)
{
    uint16_t subfrm_len;
    uint32_t subfrm_len_addr;

    // Check if we still have at least the space for a A-MSDU subframe header in the MPDU
    if (sizeof_b(struct amsdu_hdr) > mpdu_len)
        return 0;

    // Check if the A-MSDU subframe length is present in the current payload buffer
    // or in the next one - Note that due to the A-MSDU subframe alignment in the MPDU,
    // we are sure that the length field is not split across two payload descriptors,
    // and we can do a 16-bit aligned read to get it
    if (payl_offset + sizeof_b(struct amsdu_hdr) <= NX_RX_PAYLOAD_LEN)
    {
        subfrm_len_addr = pd->pbd.datastartptr + payl_offset + 2 * MAC_ADDR_LEN;
    }
    else
    {
        uint16_t offset = payl_offset + 2 * MAC_ADDR_LEN - NX_RX_PAYLOAD_LEN;
        struct rx_payloaddesc *pd_tmp = (struct rx_payloaddesc *)HW2CPU(pd->pbd.next);

        subfrm_len_addr = pd_tmp->pbd.datastartptr + offset;
    }

    // Read the length from the A-MSDU subframe header
    subfrm_len = co_ntohs(co_read16(HW2CPU(subfrm_len_addr))) + sizeof_b(struct amsdu_hdr);

    // Check if the A-MSDU subframe is valid
    if ((subfrm_len > mpdu_len) || (subfrm_len > RX_MAX_AMSDU_SUBFRAME_LEN))
        return 0;

    // Check if some padding is present
    if ((mpdu_len - subfrm_len) < 3)
        // Last A-MSDU subframe, so we consider that the subframe length is the rest of
        // the MPDU
        subfrm_len = mpdu_len;
    else
        // Intermediate A-MSDU subframe, so correct the length by adding the padding
        subfrm_len = CO_ALIGN4_HI(subfrm_len);

    return subfrm_len;
}
#endif

/**
 ****************************************************************************************
 * @brief This function processes the received control frames.
 *
 * @param[in] rxdesc SW header descriptor of the frame
 *
 * @return true if the packet was handled internally, false otherwise
 ****************************************************************************************
 */
static bool rxl_rxcntrl_frame(struct rxdesc* rxdesc)
{
    bool handled = true;
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);

    // Check if we received a NDP frame
    if (dma_hdrdesc->hd.frmlen != 0)
    {
        uint16_t framectrl;
        #if NX_AMPDU_TX
        uint32_t statinfo;
        #endif
        struct rx_pbd *pd = HW2CPU(dma_hdrdesc->hd.first_pbd_ptr);

        // If monitor mode is enabled, we don't handle the frame and it will be uploaded
        // to the upper layers
        if (rx_hwdesc_env.monitor_en)
            return false;

        // Sanity check: frames with length not NULL have at least 1 buffer descriptor
        ASSERT_REC_VAL(pd != NULL, true);

        // Get the frame control
        framectrl = co_read16(HW2CPU(pd->datastartptr));

        #if NX_AMPDU_TX
        // Get the status information from the RX header descriptor
        statinfo = dma_hdrdesc->hd.statinfo;
        #endif

        // Decode the frame control to know if we have to handle the frame internally
        switch(framectrl & MAC_FCTRL_TYPESUBTYPE_MASK)
        {
            #if NX_AMPDU_TX
            case MAC_FCTRL_BA :
                // Check if this BA is a response to a sent A-MPDU
                if ((statinfo & (RX_HD_RSP_FRM | RX_HD_SUCCESS)) ==
                                                       (RX_HD_RSP_FRM | RX_HD_SUCCESS))
                {
                    //keep BA rxdesc in list to analyse in tx confirm event handler
                    txl_ba_push(rxdesc);
                }
                break;
            #endif
            #if NX_MAC_HE
            case MAC_FCTRL_HE_TRIGGER :
                // Check if this trigger is for us
                if (statinfo & RX_HD_SUCCESS)
                {
                    txl_he_trigger_push(rxdesc);
                }
                break;
            #endif
            default:
                // LMAC does not handle this control frame, so forward it to upper MAC
                handled = false;
                break;
        }
    }
    else
    {
        // Sanity check: NDP frames have no buffer descriptor attached
        ASSERT_REC_VAL(dma_hdrdesc->hd.first_pbd_ptr == 0, true);
    }

    // Check if the frame was handled
    if (handled)
    {
        // Control and NDP frames have only one or less RBD
        if (rxdesc->dma_hdrdesc->hd.first_pbd_ptr != 0)
        {
            rxdesc->spare_pbd = HW2CPU(rxdesc->dma_hdrdesc->hd.first_pbd_ptr);

            // Store the pointer to the next available RBD (for debug and error logs only)
            rx_hwdesc_env.first = HW2CPU(rxdesc->spare_pbd->next);
        }
        else
        {
            rxdesc->spare_pbd = NULL;
        }
        rxdesc->last_pbd = NULL;

        // Release the frame
        if (rxdesc->spare_pbd != NULL)
        {
            // release the payload descriptors associated with this SW descriptor
            rxl_pd_append(NULL, NULL, rxdesc->spare_pbd);
        }

        // release the HW DMA descriptors
        rxl_hd_append(rxdesc->dma_hdrdesc);
    }
    return (handled);
}

/**
 ****************************************************************************************
 * @brief Upload a payload to a higher layer buffer.
 *
 * This function takes as payload source address the datastartptr of the buffer descriptor
 * passed as parameter, corrected by the payl_offset offset. It then goes through the list
 * of buffer descriptor until upload_len has been copied.
 *
 * @param[in,out] curr_pd Pointer to the buffer descriptor on which the payload to be
 *                        uploaded starts. At the end of the upload, the last processed
 *                        buffer descriptor is returned in this variable.
 * @param[in] upload_len  Length to be uploaded
 * @param[in] hostbuf     Address in host memory where the upload has to be performed
 * @param[in,out] payl_offset Offset inside the first buffer descriptor where the data
 *                        to upload is located. At the end of the upload, this parameter
 *                        is set to the  offset inside the last handled buffer descriptor
 *                        where non uploaded data is located.
 * @param[in] dma_idx     Index of the DMA descriptor to be used for the upload.
 * @param[in] irq_en      Set to 1 to get a DMA interrupt at the end of the upload, and
 *                        to 0 otherwise
 *
 * @return The pointer to the buffer descriptor handled before handling the last buffer
 * descriptor returned in curr_pd
 ****************************************************************************************
 */
static struct rx_payloaddesc *rxl_payload_transfer(struct rx_payloaddesc **curr_pd,
                                                   uint16_t upload_len, uint32_t hostbuf,
                                                   uint16_t *payl_offset, int dma_idx,
                                                   int irq_en)
{
    uint16_t dma_len;
    struct dma_desc *dma_desc;
    struct dma_desc *first_dma_desc;
    struct rx_payloaddesc *pd = *curr_pd;
    struct rx_payloaddesc *prev_pd = NULL;
    uint16_t payl_off = *payl_offset;

    // Get the IPC DMA descriptor of the MAC header
    dma_desc = &pd->dma_desc[dma_idx];
    // Save the pointer to the first desc, as it will be passed to the DMA driver later
    first_dma_desc = dma_desc;

    // Loop as long as the MPDU still has data to copy
    while (1)
    {
        struct dma_desc *dma_desc_next;

        // Fill the destination address of the DMA descriptor
        dma_desc->dest = hostbuf;
        dma_desc->src = pd->pbd.datastartptr + payl_off;

        // Check if we have reached the last payload buffer containing the MPDU
        if ((upload_len + payl_off) < NX_RX_PAYLOAD_LEN)
        {
            // DMA only the remaining bytes of the payload
            dma_len = upload_len;

            #if NX_AMSDU_DEAGG
            // Update the offset
            payl_off += upload_len;
            #endif
        }
        else
        {
            // The complete payload buffer has to be DMA'ed
            dma_len = NX_RX_PAYLOAD_LEN - payl_off;

            // Reset the offset
            payl_off = 0;
        }

        // Fill the DMA length in the IPC DMA descriptor
        dma_desc->length = dma_len;

        // By default no DMA IRQ on this intermediate transfer
        dma_desc->ctrl = RX_LLICTRL(0);

        // Move the pointer in the host buffer
        hostbuf += dma_len;

        // Compute remaining length to be DMA'ed to host
        upload_len -= dma_len;

        // Check if we have finished to program the current payload transfer
        if (upload_len == 0)
        {
            break;
        }

        // Move to the next RBD
        prev_pd = pd;
        pd = (struct rx_payloaddesc *)HW2CPU(pd->pbd.next);
        dma_desc_next = &pd->dma_desc[0];

        // Sanity check - There shall be a payload descriptor available
        ASSERT_ERR(pd != NULL);

        // Link the new descriptor with the previous one
        dma_desc->next = CPU2HW(dma_desc_next);

        // Retrieve the new DMA descriptor from the payload descriptor
        dma_desc = dma_desc_next;
    }

    // Last descriptor of the MPDU, enable the interrupt on it if requested
    dma_desc->ctrl = RX_LLICTRL(irq_en);

    // Push the DMA descriptor in the DMA engine
    dma_push(first_dma_desc, dma_desc, RX_DATA_UPLOAD_CHAN);

    *curr_pd = pd;
    *payl_offset = payl_off;

    return prev_pd;
}

#if NX_AMSDU_DEAGG
/**
 ****************************************************************************************
 * @brief Go through a received A-MSDU and upload its constituent MSDUs to separate host
 * buffers.
 *
 * @param[out] dma_hdrdesc Pointer to the RX DMA descriptor that contains the A-MSDU
 * @param[in,out] curr_pd Pointer to the buffer descriptor on which the A-MSDU starts. At
 *                        the end of the upload, the last processed buffer descriptor is
 *                        returned in this variable.
 * @param[in] mpdu_len    Length of the A-MSDU
 * @param[in] hostbuf     Address in host memory where the upload of the first MSDU has to
 *                        be performed
 * @param[in] payl_offset Offset inside the first buffer descriptor where the data
 *                        to upload is located.
 *
 * @return The pointer to the buffer descriptor handled before handling the last buffer
 * descriptor returned in curr_pd
 ****************************************************************************************
 */
static struct rx_payloaddesc *rxl_amsdu_deagg(struct rx_dmadesc *dma_hdrdesc,
                                              struct rx_payloaddesc **curr_pd,
                                              uint16_t mpdu_len, uint32_t hostbuf,
                                              uint16_t payl_offset)
{
    int dma_idx = 0;
    int msdu_idx = 0;
    uint16_t upload_len;
    struct rx_payloaddesc *pd = *curr_pd;
    struct rx_payloaddesc *prev_pd = NULL;

    // Reset the A-MSDU hostids
    memset(dma_hdrdesc->amsdu_hostids, 0, sizeof(dma_hdrdesc->amsdu_hostids));

    // If the MPDU is a A-MSDU, we will upload the different A-MSDU subframes in
    // separate host buffers, so get the length of the first sub-frame
    upload_len = rxl_amsdu_subframe_len_get(pd, payl_offset, mpdu_len);
    // Loop as long as the MPDU still has data to copy
    while (1)
    {
        prev_pd = rxl_payload_transfer(&pd, upload_len, hostbuf, &payl_offset, dma_idx, 0);
        mpdu_len -= upload_len;

        // Check if we still have A-MSDU subframes to upload and one available buffer
        if ((mpdu_len != 0) && macif_rx_buf_check())
        {
            // Check if the next A-MSDU subframe starts in the same payload descriptor
            // or in the next one
            if (payl_offset)
            {
                // Move to next DMA descriptor inside the current payload descriptor
                dma_idx++;
                // Is there a DMA structure available?
                if (dma_idx >= NX_DMADESC_PER_RX_PDB_CNT)
                     break;
            }
            else
            {
                // Move to the next payload descriptor
                prev_pd = pd;
                pd = (struct rx_payloaddesc *)HW2CPU(pd->pbd.next);

                // Sanity check - There shall be a payload descriptor available
                ASSERT_ERR(pd != NULL);

                dma_idx = 0;
            }
            // Retrieve the length of the next A-MSDU sub-frame
            upload_len = rxl_amsdu_subframe_len_get(pd, payl_offset, mpdu_len);

            // Check if the next A-MSDU subframe is valid
            if (upload_len == 0)
                break;

            // Get a RX buffer
            hostbuf = macif_rx_buf_get(&dma_hdrdesc->amsdu_hostids[msdu_idx++]) +
                      RXL_PAYLOAD_OFFSET;
        }
        else
            break;
    }

    *curr_pd = pd;

    return prev_pd;
}
#endif // NX_AMSDU_DEAGG

/**
 ****************************************************************************************
 * @brief Loop on the payload descriptors until the @ref RX_PD_LASTBUF flag is found.
 *
 * The function then assigns the last and spare payload descriptor pointers to the SW
 * descriptor in order to allow the later freeing of the HW descriptors.
 *
 * @param[out] rxdesc  Pointer to the RX SW descriptor
 * @param[in]  pd      Pointer to the buffer descriptor to start with.
 * @param[in]  prev_pd Pointer to the buffer descriptor just preceding pd (might be NULL).
 ****************************************************************************************
 */
static void rxl_go_to_last_rbd(struct rxdesc *rxdesc, struct rx_payloaddesc *pd,
                             struct rx_payloaddesc *prev_pd)
{
    // Go through the RBD
    while (1)
    {
        uint32_t statinfo = pd->pbd.bufstatinfo;

        // If this is the last RBD, exit the loop
        if (statinfo & RX_PD_LASTBUF)
            break;

        // Otherwise move to the next one
        prev_pd = pd;
        pd = (struct rx_payloaddesc *)HW2CPU(pd->pbd.next);

        // Sanity check - There shall be a payload descriptor available
        ASSERT_REC(pd != NULL);
    };

    // Store the pointer to the last RBD consumed by the HW. It will be used when freeing the frame
    rxdesc->last_pbd = &prev_pd->pbd;
    rxdesc->spare_pbd = &pd->pbd;

    // Store the pointer to the next available RBD (for debug and error logs only)
    rx_hwdesc_env.first = HW2CPU(rxdesc->spare_pbd->next);
}

void rxl_hwdesc_init(void)
{
    uint32_t i;

    // Reset the environment
    memset(&rx_hwdesc_env, 0, sizeof(rx_hwdesc_env));

    // Initialize the DMA header descriptor
    for (i = 0; i < NX_RXDESC_CNT; i++)
    {
        // No data buffer directly attached to the RHD
        rx_dma_hdrdesc[i].hd.datastartptr = 0;
        rx_dma_hdrdesc[i].hd.dataendptr = 0;

        // Fill in the Upattern
        rx_dma_hdrdesc[i].hd.upatternrx = RX_HEADER_DESC_PATTERN;

        // clear the MPDU status info
        rx_dma_hdrdesc[i].hd.statinfo = 0;

        // clear the control field
        rx_dma_hdrdesc[i].hd.headerctrlinfo = 0;

        // Chain to the next RHD
        rx_dma_hdrdesc[i].hd.next = CPU2HW(&rx_dma_hdrdesc[i + 1].hd);

        // reset the first payload descriptor pointer
        rx_dma_hdrdesc[i].hd.first_pbd_ptr = 0;

        // Link the HW descriptor with the corresponding SW one
        rx_dma_hdrdesc[i].hd.rxdesc = &rx_swdesc_tab[i];

        // clear the frame length
        rx_dma_hdrdesc[i].hd.frmlen = 0;

        // Prepare DMA descriptor for the RX vectors DMA transfer
        rx_dma_hdrdesc[i].dma_desc.src = CPU2HW(&rx_dma_hdrdesc[i].hd.frmlen);
        rx_dma_hdrdesc[i].dma_desc.length = RXL_HEADER_INFO_LEN;

        // Link the HW descriptor with the corresponding SW one
        rx_swdesc_tab[i].dma_hdrdesc = &rx_dma_hdrdesc[i];
        rx_swdesc_tab[i].upload_cntrl.rxdesc = &rx_swdesc_tab[i];
    }

    // Reset the next pointer of the last RHD
    rx_dma_hdrdesc[NX_RXDESC_CNT - 1].hd.next = 0;

    // write the buffer descriptor into the receive header head pointer register
    nxmac_rx_header_head_ptr_set(CPU2HW(&rx_dma_hdrdesc[1].hd));

    // set new head bit in DMA control register
    nxmac_dma_cntrl_set(NXMAC_RX_HEADER_NEW_HEAD_BIT);

    // Initialize the buffer descriptors
    // sanity check
    ASSERT_ERR((CPU2HW(&rx_payload_desc[0].buffer) & WORD_ALIGN) == 0);

    for (i = 0 ; i < NX_RX_PAYLOAD_DESC_CNT ; i++)
    {
        struct rx_payloaddesc *desc = &rx_payload_desc[i];
        struct rx_pbd *pbd = &desc->pbd;

        pbd->next = CPU2HW(&rx_payload_desc[i+1].pbd);

        // Update the upattern
        pbd->upattern = RX_PAYLOAD_DESC_PATTERN;

        // Clear the bufstatinfo
        pbd->bufstatinfo = 0;

        //   (end pointer is inclusive, hence the -1)
        pbd->datastartptr = CPU2HW(&(desc->buffer[0]));
        pbd->dataendptr   = pbd->datastartptr + NX_RX_PAYLOAD_LEN - 1;
    }

    // Reset the next pointer on the last one
    rx_payload_desc[NX_RX_PAYLOAD_DESC_CNT - 1].pbd.next = 0;

    // program new buffer desc header in list for the MAC HW
    nxmac_rx_payload_head_ptr_set(CPU2HW(&rx_payload_desc[1].pbd));

    // set the new head bit in the DMA control register
    nxmac_dma_cntrl_set(NXMAC_RX_PAYLOAD_NEW_HEAD_BIT);

    // initialize the RX trigger timers (RPD count reached for trig 1/4 of available , packet timeout is 1*32us, absolute is 10*32us)
    nxmac_rx_trigger_timer_pack(NX_RX_PAYLOAD_DESC_CNT/4, 1, 10);

    // initialize the first element of the DMA header pool
    rx_hwdesc_env.free_dmadesc = &rx_dma_hdrdesc[0];
    rx_hwdesc_env.first_dmadesc = &rx_dma_hdrdesc[1];
    rx_hwdesc_env.last_dmadesc = &rx_dma_hdrdesc[NX_RXDESC_CNT - 1];

    // Record the last buffer desc
    rx_hwdesc_env.free = &rx_payload_desc[0].pbd;
    rx_hwdesc_env.first = &rx_payload_desc[1].pbd;
    rx_hwdesc_env.last = &rx_payload_desc[NX_RX_PAYLOAD_DESC_CNT - 1].pbd;

    co_list_init(&rx_hwdesc_env.ready);
}

void rxl_hwdesc_monitor(bool enable)
{
    // Set the monitor flag, so that BA and other control frames are forwarded to the
    // upper layers or handled internally
    rx_hwdesc_env.monitor_en = enable;
}

void rxl_frame_release(struct rxdesc* rxdesc)
{
    // HW descriptor release needs to be protected
    GLOBAL_INT_DISABLE();

    // release the payload descriptors associated with this SW descriptor
    rxl_pd_append(HW2CPU(rxdesc->dma_hdrdesc->hd.first_pbd_ptr), rxdesc->last_pbd, rxdesc->spare_pbd);

    // release the HW DMA descriptors
    rxl_hd_append(rxdesc->dma_hdrdesc);

    // Re-enable the interrupts
    GLOBAL_INT_RESTORE();
}

void rxl_mpdu_copy(struct rx_pbd *pbd, uint16_t length, uint16_t offset, uint32_t *dst)
{
    uint16_t dst_offset = 0;
    uint32_t *src;
    while (length)
    {
        uint16_t copy_len;
        if (length < (NX_RX_PAYLOAD_LEN - offset))
            copy_len = length;
        else
            copy_len = NX_RX_PAYLOAD_LEN - offset;

        src = HW2CPU(pbd->datastartptr + offset);
        co_copy32(dst + dst_offset/4, src, copy_len);

        length -= copy_len;
        offset = 0;
        dst_offset += copy_len;
        pbd = HW2CPU(pbd->next);
        ASSERT_ERR(pbd != NULL);
    }
}

void rxl_mpdu_partial_transfer(struct rxdesc *rxdesc, uint16_t upload_len,
                               uint32_t hostbuf, uint16_t payl_offset,
                               cb_rx_dma_func_ptr cb, void *env)
{
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    struct rx_payloaddesc *pd = HW2CPU(dma_hdrdesc->hd.first_pbd_ptr);
    struct rx_payloaddesc *prev_pd = NULL;

    // for profiling
    RX_MPDU_XFER_SET();

    // Program the payload transfer
    prev_pd = rxl_payload_transfer(&pd, upload_len, hostbuf, &payl_offset, 0, 1);

    // Now go through the additional RBD if any (e.g. for FCS, ICV)
    rxl_go_to_last_rbd(rxdesc, pd, prev_pd);

    // for profiling
    RX_MPDU_XFER_CLR();

    // Update upload control descriptor
    rxdesc->upload_cntrl.cb = cb;
    rxdesc->upload_cntrl.env = env;

    // Insert the element in the pending list
    rxl_upload_cntrl_push_pending(&rxdesc->upload_cntrl);
}

void rxl_mpdu_transfer(struct rxdesc *rxdesc)
{
    #if (NX_UMAC_PRESENT)
    struct rx_cntrl_rx_status *rx_status = &rxu_cntrl_env.rx_status;
    uint16_t payl_offset = rx_status->payl_offset;
    #else
    uint16_t payl_offset = 0;
    #endif
    uint32_t hostbuf, hostbuf_start;
    struct dma_desc *dma_desc;
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    uint16_t mpdu_len;
    struct rx_payloaddesc *pd;
    struct rx_payloaddesc *prev_pd = NULL;

    // for profiling
    RX_MPDU_XFER_SET();

    // Cache the pointers for faster access
    pd = (struct rx_payloaddesc *)HW2CPU(dma_hdrdesc->hd.first_pbd_ptr);

    // Get a host buffer address for DMA transfer (host buffer) if needed
    #if (NX_UMAC_PRESENT)
    hostbuf_start = macif_rx_buf_get(&rxu_cntrl_env.hostid_current);
    #else
    hostbuf_start = macif_rx_buf_get();
    #endif //(NX_UMAC_PRESENT)

    // Copy at the beginning of the host buffer + offset
    hostbuf = hostbuf_start + RXL_PAYLOAD_OFFSET;

    // Get the MPDU body length
    mpdu_len = dma_hdrdesc->hd.frmlen;

    // Sanity check - The frame length cannot be NULL...
    ASSERT_REC(mpdu_len != 0);
    // as well as not greater than the max allowed length
    ASSERT_REC(mpdu_len <= RWNX_MAX_AMSDU_RX);

    #if NX_AMSDU_DEAGG
    // Check if the frame is a A-MSDU
    if (dma_hdrdesc->flags & RX_FLAGS_IS_AMSDU_BIT)
    {
        // If the MPDU is a A-MSDU and the A-MSDU deaggregator is compiled, call the
        // A-MSDU de-aggregation function
        prev_pd = rxl_amsdu_deagg(dma_hdrdesc, &pd, mpdu_len, hostbuf, payl_offset);
    }
    else
    #endif
    {
        // If the MPDU is not a A-MSDU, or if the A-MSDU deaggregator is not compiled,
        // then we will upload the complete packet in a single host buffer
        prev_pd = rxl_payload_transfer(&pd, mpdu_len, hostbuf, &payl_offset, 0, 0);

        #if (NX_UMAC_PRESENT)
        // Save position of next fragment upload
        rx_status->host_buf_addr = hostbuf + mpdu_len;
        #endif
    }

    // Get the information on the current channel from the PHY driver
    phy_get_channel(&dma_hdrdesc->phy_info, PHY_PRIM);
    // Now elaborate the final DMA sending for the PHY Vectors and status
    // "Stamp" it as the final DMA of this MPDU transfer: insert a pattern for the upper
    // layer to be able to know that this hostbuf has been filled
    dma_hdrdesc->pattern = DMA_HD_RXPATTERN;
    // Get the IPC DMA descriptor of the PHYVECT header part
    dma_desc = &dma_hdrdesc->dma_desc;
    // Link the new DMA desc with the previous one
    dma_desc->next = CPU2HW(&dma_hdrdesc->dma_desc);
    // Fill the destination address of the DMA descriptor
    // (hostbuf address is already up-to-date at this point)
    dma_desc->dest = hostbuf_start;

    #if NX_UMAC_PRESENT && !NX_FULLY_HOSTED
    // Interruption only needed for the descriptor
    dma_desc->ctrl = RX_LLICTRL(0);
    #else
    // Last descriptor of the MPDU, so enable the interrupt on it
    dma_desc->ctrl = RX_LLICTRL(1);
    #endif //NX_UMAC_PRESENT && !NX_FULLY_HOSTED

    // Push the DMA descriptor in the DMA engine
    dma_push(dma_desc, dma_desc, RX_DATA_UPLOAD_CHAN);

    // Now go through the additional RBD if any (e.g. for FCS, ICV)
    rxl_go_to_last_rbd(rxdesc, pd, prev_pd);

    // for profiling
    RX_MPDU_XFER_CLR();

    #if NX_FULLY_HOSTED || !NX_UMAC_PRESENT
    // and push it in the pending list
    rxl_upload_cntrl_push_pending(&rxdesc->upload_cntrl);
    #endif
}

void rxl_mpdu_free(struct rxdesc *rxdesc)
{
    struct rx_payloaddesc *pd;
    struct rx_payloaddesc *prev_pd = NULL;
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);

    RX_MPDU_FREE_SET();

    // Cache the pointers for faster access
    pd = (struct rx_payloaddesc *)HW2CPU(dma_hdrdesc->hd.first_pbd_ptr);

    // Go through the RBD
    rxl_go_to_last_rbd(rxdesc, pd, prev_pd);

    // Release the frame
    rxl_frame_release(rxdesc);

    RX_MPDU_FREE_CLR();
}

struct rxdesc *rxl_rxdesc_get(void)
{
    // Pick a SW descriptor from the list
    return ((struct rxdesc *)co_list_pick(&rx_hwdesc_env.ready));
}

void rxl_rxdesc_ready_for_processing(struct rxdesc *rxdesc)
{
    // Nothing special to do on the descriptor, except popping it from the ready list
    GLOBAL_INT_DISABLE();
    co_list_pop_front(&rx_hwdesc_env.ready);
    GLOBAL_INT_RESTORE();
}

void rxl_mpdu_isr(void)
{
    struct rx_dmadesc *dma_hdrdesc;
    struct rxdesc *rxdesc;

    // for profiling
    PROF_RX_MAC_IRQ_SET();

    // clear the interrupt
    nxmac_tx_rx_int_ack_clear(NXMAC_TIMER_RX_TRIGGER_BIT | NXMAC_COUNTER_RX_TRIGGER_BIT);

    while (1)
    {
        // check if there are more elements to handle
        if ((rx_hwdesc_env.first_dmadesc == NULL) ||
            (!RX_HD_DONE_GET(rx_hwdesc_env.first_dmadesc->hd.statinfo)))
            break;

        // retrieve the pointer to the first element of the list
        dma_hdrdesc = rx_hwdesc_env.first_dmadesc;

        // prepare the SW descriptor for this frame
        rxdesc = dma_hdrdesc->hd.rxdesc;

        // get the next Header descriptor pointer now as it may be modified by
        // rxl_rxcntrl_frame() if the frame is released immediately
        rx_hwdesc_env.first_dmadesc = (struct rx_dmadesc*)HW2CPU(dma_hdrdesc->hd.next);

        // check if it is a control frame and handle it if necessary
        if (!rxl_rxcntrl_frame(rxdesc))
        {
            // Reset upload control structure
            #if NX_FULLY_HOSTED
            rxdesc->upload_cntrl.cb = NULL;
            #else
            rxdesc->upload_cntrl.cb = rxl_host_irq_mitigation_update;
            #endif
            rxdesc->upload_cntrl.env = NULL;
            // Push the frame to the ready list
            co_list_push_back(&rx_hwdesc_env.ready, &rxdesc->upload_cntrl.list_hdr);
        }
    }

    // Check if frames are ready
    if (!co_list_is_empty(&rx_hwdesc_env.ready))
    {
        if (macif_rx_buf_check())
        {
            ke_evt_set(KE_EVT_RXREADY_BIT);
        }
    }

    // for profiling
    PROF_RX_MAC_IRQ_CLR();
}

void rxl_immediate_frame_get(void)
{
    rxl_mpdu_isr();
}

void rxl_current_desc_get(struct rx_hd **rhd, struct rx_pbd **rbd)
{
    // First RX Header Descriptor chained to the HW
    *rhd = &(rx_hwdesc_env.free_dmadesc->hd);

    // First RX Buffer Descriptor chained to the HW
    *rbd = rx_hwdesc_env.free;
}

/// @} // end of group RXHWDESC
