/**
 ****************************************************************************************
 * @file v20/rxl_hwdesc.c
 *
 * @brief Implementation of the API function used to initialize the pools.
 *
 * Copyright (C) RivieraWaves 2011-2019
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup RX_HWDESC
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "rxl_hwdesc.h"

#include "ke_event.h"
#include "rxl_cntrl.h"
#if NX_UMAC_PRESENT
#include "rxu_cntrl.h"
#endif
#include "co_utils.h"
#include "macif.h"
#if NX_AMPDU_TX
#include "txl_cfm.h"
#endif
#if NX_MAC_HE
#include "txl_he.h"
#endif
#include "co_endian.h"
#include "reg_mac_pl.h"
#include "reg_mac_core.h"

/*
 * TYPE DEFINITION
 ****************************************************************************************
 */

/// Structure caching the RX buffers parameters
struct rxl_buffer_tag
{
    /// Start address of the buffer
    uint32_t start;
    /// End address of the buffer
    uint32_t end;
    /// Read address of the FW
    uint32_t read;
};

/// Buffer Management control structure
struct rxl_hwdesc_env_tag
{
    /// Buffer parameters
    struct rxl_buffer_tag buf[2];
};

/*
 * DEFINES
 ****************************************************************************************
 */
/// Bit indicating a buffer wrap in the read and write addresses
#define RXL_WRAP CO_BIT(31)
/// RX descriptor size
#define RXL_RXDESC_SIZE (sizeof_b(struct rxdesc))
/// Head room for RHD
#define RXL_RHD_HEADER (sizeof_b(struct rx_upload_cntrl_tag) / 4)
/// Tail room for RHD
#define RXL_RHD_FOOTER ((RXL_RXDESC_SIZE - sizeof_b(struct rx_hd)) / 4 - RXL_RHD_HEADER)
/// Head room for RBD
#define RXL_RBD_HEADER  0
/// Tail room for RBD
#define RXL_RBD_FOOTER ((sizeof_b(struct rx_payloaddesc) - sizeof_b(struct rx_pbd)) / 4)
/// Size of the RX buffer 1 (the size of 2 big A-MSDUs + their descriptors)
#define RXL_BUFFER1_SIZE ((RWNX_MAX_AMSDU_RX + RXL_RXDESC_SIZE) / 2)
/// Size of the RX buffer 2 (512 bytes + the size of a RX descriptor)
#define RXL_BUFFER2_SIZE ((512 + RXL_RXDESC_SIZE) / 4)

/// RX buffers identifiers
enum rx_buf_id
{
    /// Id of the buffer requiring deferred handling (i.e. for normal data/management)
    DH_BUF = 0,
    /// Id of the buffer requiring immediate handling (i.e. for BA/HE trigger)
    IH_BUF = 1
};

/*
 * VARIABLES
 ****************************************************************************************
 */
/// RXL HW descriptor environment variable
static struct rxl_hwdesc_env_tag rx_hwdesc_env;

/// RX buffer 1, i.e. buffer used by the MAC HW to store all received frames except those
/// requiring a time critical handling (i.e. BA and HE trigger frames)
static uint32_t rxl_hw_buffer1[RXL_BUFFER1_SIZE] __SHAREDRAM;

/// RX buffer 2, i.e. buffer used by the MAC HW to store frames requiring a time critical
/// handling (i.e. BA and HE trigger frames)
static uint32_t rxl_hw_buffer2[RXL_BUFFER2_SIZE] __SHAREDRAM;

/*
 * MACROS
 ****************************************************************************************
 */
/// Macro initializing the buffer pointers to the HW
#define RXL_HW_BUFFER_INIT(id)                                                           \
do {                                                                                     \
    struct rxl_buffer_tag *buf = &rx_hwdesc_env.buf[id - 1];                             \
    buf->start = CPU2HW(rxl_hw_buffer##id);                                              \
    buf->end = buf->start + CO_ALIGN4_LO(sizeof_b(rxl_hw_buffer##id));                   \
    buf->read = buf->start;                                                              \
    nxmac_rx_buf_##id##_start_ptr_set(buf->start);                                       \
    nxmac_rx_buf_##id##_end_ptr_set(buf->end - 4);                                       \
    nxmac_rx_buf_##id##_rd_ptr_set(buf->start);                                          \
    nxmac_rx_buf_##id##_wr_ptr_set(buf->start);                                          \
} while(0)

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Return the DH buffer status.
 *
 * @return true if data is available in the DH buffer, false otherwise
 ****************************************************************************************
 */
__INLINE bool rxl_dh_buf_has_data(void)
{
    return (rx_hwdesc_env.buf[DH_BUF].read != nxmac_rx_buf_1_wr_ptr_get());
}

/**
 ****************************************************************************************
 * @brief Return the IH buffer status.
 *
 * @return true if data is available in the IH buffer, false otherwise
 ****************************************************************************************
 */
__INLINE bool rxl_ih_buf_has_data(void)
{
    return (rx_hwdesc_env.buf[IH_BUF].read != nxmac_rx_buf_2_wr_ptr_get());
}

/**
 ****************************************************************************************
 * @brief Set correctly the wrap bit (bit 31) to the new read address, based on the
 * previous read address.
 *
 * The function checks if the new read address causes a wrap of the buffer, and toggles
 * the wrap bit if this is the case.
 *
 * @param[in] new_read  New read address (without the wrap bit)
 * @param[in] old_read  Old read address (with current value of wrap bit)
 *
 * @return The new read address, that now contains the right value of the wrap bit.
 ****************************************************************************************
 */
__INLINE uint32_t rxl_add_wrap_bit(uint32_t new_read, uint32_t old_read)
{
    uint32_t wrap_bit = old_read & RXL_WRAP;

    old_read &= ~RXL_WRAP;
    if (new_read < old_read)
        wrap_bit ^= RXL_WRAP;

    return (new_read | wrap_bit);
}

/**
 ****************************************************************************************
 * @brief Update the read address HW register.
 *
 * This address is used by the HW to know where it shall stop writing into the HW buffer.
 * To be able to distinguish between buffer full and buffer empty cases, the HW toggles
 * the bit 31 of the address each time it wraps around. We therefore have to update this
 * bit as well when updating the read address.
 *
 * @param[in] new_read Address inside the RX buffer where we expect the next frame buffer
 * @param[in] id       Index of the RX buffer (@ref rx_buf_id)
 ****************************************************************************************
 */
__INLINE void rxl_update_hw_buf_read(uint32_t new_read, int id)
{
    if (id == DH_BUF)
    {
        nxmac_rx_buf_1_rd_ptr_set(rxl_add_wrap_bit(new_read, nxmac_rx_buf_1_rd_ptr_get()));
    }
    else
    {
        nxmac_rx_buf_2_rd_ptr_set(rxl_add_wrap_bit(new_read, nxmac_rx_buf_2_rd_ptr_get()));
    }
}

/**
 ****************************************************************************************
 * @brief Update the read address of the FW.
 *
 * This address is used to know where the FW is expecting the next frame written by the
 * HW inside the RX buffer. In order to know if a frame is present in the RX buffer, this
 * address is compared with the HW write address. To be able to distinguish between buffer
 * full and buffer empty cases, the HW toggles the bit 31 of the address each time it
 * wraps around. We use this mechanism on the SW read address too.
 *
 * @param[in] new_read Address inside the RX buffer where we expect the next frame buffer
 * @param[in,out] buf  Pointer to the buffer information structure.
 ****************************************************************************************
 */
__INLINE void rxl_update_sw_buf_read(uint32_t new_read, struct rxl_buffer_tag *buf)
{
    buf->read = rxl_add_wrap_bit(new_read, buf->read);
}

/**
 ****************************************************************************************
 * @brief Check if there is enough space between the SW read address and the end of the
 * buffer. If not the SW read address is set to the beginning of the buffer, in order to
 * stay aligned with the location where the HW wrote the frame.
 *
 * This function shall be called only after comparing SW read and HW write address to
 * determine if some frame is available for the SW.
 *
 * @param[in] buf The buffer to be checked
 ****************************************************************************************
 */
__INLINE void rxl_adjust_sw_buf_read(struct rxl_buffer_tag *buf)
{
    if (buf->end - (buf->read & ~RXL_WRAP) < RXL_RXDESC_SIZE)
        rxl_update_sw_buf_read(buf->start, buf);
}

/**
 ****************************************************************************************
 * @brief Remove the MIC/ICV/FCS length from the frame length.
 *
 * The MAC HW is now including the length of these fields in the frame length reported in
 * the RHD, whereas it was not the case before. As all the RX SW is expecting those
 * lengths not to be included, let's remove them prior to any SW processing of the frame.
 *
 * @param[in,out] rhd  Pointer to the RHD to be updated.
 ****************************************************************************************
 */
__INLINE void rxl_adjust_frmlen(struct rx_hd *rhd)
{
    // In case frame length is null (e.g NDP), don't adjust the field
    if (!rhd->frmlen)
        return;

    switch (rhd->statinfo & RX_HD_DECRSTATUS)
    {
        case RX_HD_DECR_CCMP128_SUCCESS:
            rhd->frmlen -= MIC_LEN + MAC_FCS_LEN;
            break;
        case RX_HD_DECR_WEP_SUCCESS:
            rhd->frmlen -= ICV_LEN + MAC_FCS_LEN;
            break;
        case RX_HD_DECR_TKIP_SUCCESS:
            rhd->frmlen -= ICV_LEN + MAC_FCS_LEN;
            break;
        default:
            rhd->frmlen -= MAC_FCS_LEN;
            break;
    }
}

#if NX_AMSDU_DEAGG
/**
 ****************************************************************************************
 * @brief Compute the A-MSDU subframe length to be uploaded from a given payload
 * descriptor.
 * The function also checks if the A-MSDU subframe can fit into the remaining MPDU length.
 *
 * @param[in] pd          Pointer to the payload descriptor in which the A-MSDU subframe
 *                        is supposed to start
 * @param[in] payl_offset Offset in the payload, where the A-MSDU subframe is supposed to
 *                        start
 * @param[in] mpdu_len    Remaining MPDU length from offset to the end of the MPDU
 *
 * @return    If the A-MSDU subframe is valid, its length, including the possible padding.
 *            Otherwise 0.
 ****************************************************************************************
 */
static uint16_t rxl_amsdu_subframe_len_get(struct rx_payloaddesc *pd,
                                           uint16_t payl_offset,
                                           uint16_t mpdu_len)
{
    uint16_t subfrm_len;
    uint32_t subfrm_len_addr;
    uint16_t payl_len = pd->pbd.dataendptr - pd->pbd.datastartptr + 1;

    // Check if we still have at least the space for a A-MSDU subframe header in the MPDU
    if (sizeof_b(struct amsdu_hdr) > mpdu_len)
        return 0;

    // Check if the A-MSDU subframe length is present in the current payload buffer
    // or in the next one - Note that due to the A-MSDU subframe alignment in the MPDU,
    // we are sure that the length field is not split across two payload descriptors,
    // and we can do a 16-bit aligned read to get it
    if (payl_offset + sizeof_b(struct amsdu_hdr) <= payl_len)
    {
        subfrm_len_addr = pd->pbd.datastartptr + payl_offset + 2 * MAC_ADDR_LEN;
    }
    else
    {
        uint16_t offset = payl_offset + 2 * MAC_ADDR_LEN - payl_len;
        struct rx_payloaddesc *pd_tmp = (struct rx_payloaddesc *)HW2CPU(pd->pbd.next);

        subfrm_len_addr = pd_tmp->pbd.datastartptr + offset;
    }

    // Read the length from the A-MSDU subframe header
    subfrm_len = co_ntohs(co_read16(HW2CPU(subfrm_len_addr))) + sizeof_b(struct amsdu_hdr);

    // Check if the A-MSDU subframe is valid
    if ((subfrm_len > mpdu_len) || (subfrm_len > RX_MAX_AMSDU_SUBFRAME_LEN))
        return 0;

    // Check if some padding is present
    if ((mpdu_len - subfrm_len) < 3)
        // Last A-MSDU subframe, so we consider that the subframe length is the rest of
        // the MPDU
        subfrm_len = mpdu_len;
    else
        // Intermediate A-MSDU subframe, so correct the length by adding the padding
        subfrm_len = CO_ALIGN4_HI(subfrm_len);

    return subfrm_len;
}
#endif

/**
 ****************************************************************************************
 * @brief This function processes the received control frames.
 *
 * @param[in] rxdesc SW header descriptor of the frame
 *
 * @return true if the packet was handled internally, false otherwise
 ****************************************************************************************
 */
void rxl_rxcntrl_frame(struct rxdesc* rxdesc)
{
    uint32_t new_read;
    struct rxl_buffer_tag *buf = &rx_hwdesc_env.buf[IH_BUF];
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);

    // Check if we received a NDP frame
    if (dma_hdrdesc->hd.frmlen != 0)
    {
        uint16_t framectrl;
        #if NX_AMPDU_TX
        uint32_t statinfo;
        #endif
        struct rx_pbd *pd = HW2CPU(dma_hdrdesc->hd.first_pbd_ptr);

        // Sanity check: frames with length not NULL have at least 1 buffer descriptor
        ASSERT_REC(pd != NULL);

        // Get the frame control
        framectrl = co_read16(HW2CPU(pd->datastartptr));

        #if NX_AMPDU_TX
        // Get the status information from the RX header descriptor
        statinfo = dma_hdrdesc->hd.statinfo;
        #endif

        // Decode the frame control to know if we have to handle the frame internally
        switch(framectrl & MAC_FCTRL_TYPESUBTYPE_MASK)
        {
            #if NX_AMPDU_TX
            case MAC_FCTRL_BA :
                // Check if this BA is a response to a sent A-MPDU
                if ((statinfo & (RX_HD_RSP_FRM | RX_HD_SUCCESS)) ==
                                                       (RX_HD_RSP_FRM | RX_HD_SUCCESS))
                {
                    //keep BA rxdesc in list to analyse in tx confirm event handler
                    txl_ba_push(rxdesc);
                }
                break;
            #endif
            #if NX_MAC_HE
            case MAC_FCTRL_HE_TRIGGER :
                // Check if this trigger is for us
                if (statinfo & RX_HD_SUCCESS)
                {
                    txl_he_trigger_push(rxdesc);
                }
                break;
            #endif
            default:
                break;
        }

        new_read = CO_ALIGN4_LO(pd->dataendptr) + 4;
    }
    else
    {
        // Sanity check: NDP frames have no buffer descriptor attached
        ASSERT_REC(dma_hdrdesc->hd.first_pbd_ptr == 0);
        new_read = CPU2HW(rxdesc + 1);
    }

    if (new_read == buf->end)
        new_read = buf->start;

    rxdesc->new_read = new_read;

    // Update our read address
    rxl_update_sw_buf_read(rxdesc->new_read, buf);

    // Release the frame
    rxl_frame_release(rxdesc);
}

/**
 ****************************************************************************************
 * @brief Upload a payload to a higher layer buffer.
 *
 * This function takes as payload source address the datastartptr of the buffer descriptor
 * passed as parameter, corrected by the payl_offset offset. It then goes through the list
 * of buffer descriptor until upload_len has been copied.
 *
 * @param[in,out] curr_pd Pointer to the buffer descriptor on which the payload to be
 *                        uploaded starts. At the end of the upload, the last processed
 *                        buffer descriptor is returned in this variable.
 * @param[in] upload_len  Length to be uploaded
 * @param[in] hostbuf     Address in host memory where the upload has to be performed
 * @param[in,out] payl_offset Offset inside the first buffer descriptor where the data
 *                        to upload is located. At the end of the upload, this parameter
 *                        is set to the  offset inside the last handled buffer descriptor
 *                        where non uploaded data is located.
 * @param[in] dma_idx     Index of the DMA descriptor to be used for the upload.
 * @param[in] irq_en      Set to 1 to get a DMA interrupt at the end of the upload, and
 *                        to 0 otherwise
 *
 * @return The pointer to the buffer descriptor handled before handling the last buffer
 * descriptor returned in curr_pd
 ****************************************************************************************
 */
static struct rx_payloaddesc *rxl_payload_transfer(struct rx_payloaddesc **curr_pd,
                                                   uint16_t upload_len, uint32_t hostbuf,
                                                   uint16_t *payl_offset, int dma_idx,
                                                   int irq_en)
{
    uint16_t dma_len;
    struct dma_desc *dma_desc;
    struct dma_desc *first_dma_desc;
    struct rx_payloaddesc *pd = *curr_pd;
    struct rx_payloaddesc *prev_pd = NULL;
    uint16_t payl_off = *payl_offset;

    // Get the IPC DMA descriptor of the MAC header
    dma_desc = &pd->dma_desc[dma_idx];
    // Save the pointer to the first desc, as it will be passed to the DMA driver later
    first_dma_desc = dma_desc;

    // Loop as long as the MPDU still has data to copy
    while (1)
    {
        struct dma_desc *dma_desc_next;
        uint16_t payl_len = pd->pbd.dataendptr - pd->pbd.datastartptr + 1;

        // Fill the destination address of the DMA descriptor
        dma_desc->dest = hostbuf;
        dma_desc->src = pd->pbd.datastartptr + payl_off;

        // Check if we have reached the last payload buffer containing the MPDU
        if ((upload_len + payl_off) < payl_len)
        {
            // DMA only the remaining bytes of the payload
            dma_len = upload_len;

            #if NX_AMSDU_DEAGG
            // Update the offset
            payl_off += upload_len;
            #endif
        }
        else
        {
            // The complete payload buffer has to be DMA'ed
            dma_len = payl_len - payl_off;

            // Reset the offset
            payl_off = 0;
        }

        // Fill the DMA length in the IPC DMA descriptor
        dma_desc->length = dma_len;

        // By default no DMA IRQ on this intermediate transfer
        dma_desc->ctrl = RX_LLICTRL(0);

        // Move the pointer in the host buffer
        hostbuf += dma_len;

        // Compute remaining length to be DMA'ed to host
        upload_len -= dma_len;

        // Check if we have finished to program the current payload transfer
        if (upload_len == 0)
        {
            break;
        }

        // Move to the next RBD
        prev_pd = pd;
        pd = (struct rx_payloaddesc *)HW2CPU(pd->pbd.next);
        dma_desc_next = &pd->dma_desc[0];

        // Sanity check - There shall be a payload descriptor available
        ASSERT_ERR(pd != NULL);

        // Link the new descriptor with the previous one
        dma_desc->next = CPU2HW(dma_desc_next);

        // Retrieve the new DMA descriptor from the payload descriptor
        dma_desc = dma_desc_next;
    }

    // Last descriptor of the MPDU, enable the interrupt on it if requested
    dma_desc->ctrl = RX_LLICTRL(irq_en);

    // Push the DMA descriptor in the DMA engine
    dma_push(first_dma_desc, dma_desc, RX_DATA_UPLOAD_CHAN);

    *curr_pd = pd;
    *payl_offset = payl_off;

    return prev_pd;
}

#if NX_AMSDU_DEAGG
/**
 ****************************************************************************************
 * @brief Go through a received A-MSDU and upload its constituent MSDUs to separate host
 * buffers.
 *
 * @param[out] dma_hdrdesc Pointer to the RX DMA descriptor that contains the A-MSDU
 * @param[in,out] curr_pd Pointer to the buffer descriptor on which the A-MSDU starts. At
 *                        the end of the upload, the last processed buffer descriptor is
 *                        returned in this variable.
 * @param[in] mpdu_len    Length of the A-MSDU
 * @param[in] hostbuf     Address in host memory where the upload of the first MSDU has to
 *                        be performed
 * @param[in] payl_offset Offset inside the first buffer descriptor where the data
 *                        to upload is located.
 *
 * @return The pointer to the buffer descriptor handled before handling the last buffer
 * descriptor returned in curr_pd
 ****************************************************************************************
 */
static struct rx_payloaddesc *rxl_amsdu_deagg(struct rx_dmadesc *dma_hdrdesc,
                                              struct rx_payloaddesc **curr_pd,
                                              uint16_t mpdu_len, uint32_t hostbuf,
                                              uint16_t payl_offset)
{
    int dma_idx = 0;
    int msdu_idx = 0;
    uint16_t upload_len;
    struct rx_payloaddesc *pd = *curr_pd;
    struct rx_payloaddesc *prev_pd = NULL;

    // Reset the A-MSDU hostids
    memset(dma_hdrdesc->amsdu_hostids, 0, sizeof(dma_hdrdesc->amsdu_hostids));

    // If the MPDU is a A-MSDU, we will upload the different A-MSDU subframes in
    // separate host buffers, so get the length of the first sub-frame
    upload_len = rxl_amsdu_subframe_len_get(pd, payl_offset, mpdu_len);
    // Loop as long as the MPDU still has data to copy
    while (1)
    {
        prev_pd = rxl_payload_transfer(&pd, upload_len, hostbuf, &payl_offset, dma_idx, 0);
        mpdu_len -= upload_len;

        // Check if we still have A-MSDU subframes to upload and one available buffer
        if ((mpdu_len != 0) && macif_rx_buf_check())
        {
            // Check if the next A-MSDU subframe starts in the same payload descriptor
            // or in the next one
            if (payl_offset)
            {
                // Move to next DMA descriptor inside the current payload descriptor
                dma_idx++;
                // Is there a DMA structure available?
                if (dma_idx >= NX_DMADESC_PER_RX_PDB_CNT)
                     break;
            }
            else
            {
                // Move to the next payload descriptor
                prev_pd = pd;
                pd = (struct rx_payloaddesc *)HW2CPU(pd->pbd.next);

                // Sanity check - There shall be a payload descriptor available
                ASSERT_ERR(pd != NULL);

                dma_idx = 0;
            }
            // Retrieve the length of the next A-MSDU sub-frame
            upload_len = rxl_amsdu_subframe_len_get(pd, payl_offset, mpdu_len);

            // Check if the next A-MSDU subframe is valid
            if (upload_len == 0)
                break;

            // Get a RX buffer
            hostbuf = macif_rx_buf_get(&dma_hdrdesc->amsdu_hostids[msdu_idx++]) +
                      RXL_PAYLOAD_OFFSET;
        }
        else
            break;
    }

    *curr_pd = pd;

    return prev_pd;
}
#endif // NX_AMSDU_DEAGG

/**
 ****************************************************************************************
 * @brief Loop on the payload descriptors until the @ref RX_PD_LASTBUF flag is found.
 *
 * The function then assigns the last and spare payload descriptor pointers to the SW
 * descriptor in order to allow the later freeing of the HW descriptors.
 *
 * @param[out] rxdesc  Pointer to the RX SW descriptor
 * @param[in]  pd      Pointer to the buffer descriptor to start with.
 * @param[in]  prev_pd Pointer to the buffer descriptor just preceding pd (might be NULL).
 ****************************************************************************************
 */
static void rxl_go_to_last_rbd(struct rxdesc *rxdesc, struct rx_payloaddesc *pd,
                               struct rx_payloaddesc *prev_pd)
{
    uint32_t new_read;
    struct rxl_buffer_tag *buf = &rx_hwdesc_env.buf[DH_BUF];

    // In case frame length is null (e.g NDP), no payload descriptor is available and
    // the next read address is just after the RX descriptor
    if (pd == NULL)
    {
        new_read = CPU2HW(rxdesc + 1);
    }
    else
    {
        // Go through the RBD
        while (1)
        {
            uint32_t statinfo = pd->pbd.bufstatinfo;

            // If this is the last RBD, exit the loop
            if (statinfo & RX_PD_LASTBUF)
                break;

            // Otherwise move to the next one
            pd = (struct rx_payloaddesc *)HW2CPU(pd->pbd.next);

            // Sanity check - There shall be a payload descriptor available
            ASSERT_REC(pd != NULL);
        };

        // Compute and save the address just following the frame in the buffer. This address
        // will be used when freeing the frame.
        new_read = CO_ALIGN4_LO(pd->pbd.dataendptr) + 4;
    }

    if (new_read == buf->end)
        new_read = buf->start;

    rxdesc->new_read = new_read;

    // Update our read address
    rxl_update_sw_buf_read(rxdesc->new_read, buf);
}

void rxl_hwdesc_init(void)
{
    // Initialize the various buffer pointers to the HW
    RXL_HW_BUFFER_INIT(1);
    RXL_HW_BUFFER_INIT(2);

    // Initialize other control registers
    nxmac_rx_buf_config_pack(RXL_RBD_FOOTER, RXL_RBD_HEADER, RXL_RHD_FOOTER, RXL_RHD_HEADER);
    nxmac_rx_cntrl_2_pack(nxmac_disable_rx_buffer_2_getf(), 1, 1, 1, 128, 2, 1, 1);
}

void rxl_hwdesc_monitor(bool enable)
{
    // Allow/disallow reception of all HE trigger frames
    nxmac_accept_all_trigger_frames_setf(enable);

    // Enable/disable the RX buffer 2
    nxmac_disable_rx_buffer_2_setf(enable);

    // Re-initialize the buffers
    rxl_hwdesc_init();
}

void rxl_frame_release(struct rxdesc* rxdesc)
{
    // Update HW read register
    rxl_update_hw_buf_read(rxdesc->new_read, rxdesc->buf_id);
}

void rxl_mpdu_copy(struct rx_pbd *pbd, uint16_t length, uint16_t offset, uint32_t *dst)
{
    uint16_t dst_offset = 0;
    uint32_t *src;
    while (length)
    {
        uint16_t copy_len;
        uint16_t payl_len = pbd->dataendptr - pbd->datastartptr + 1;
        if (length < (payl_len - offset))
            copy_len = length;
        else
            copy_len = payl_len - offset;

        src = HW2CPU(pbd->datastartptr + offset);
        co_copy32(dst + dst_offset/4, src, copy_len);

        length -= copy_len;
        offset = 0;

        if (length == 0)
            break;

        dst_offset += copy_len;
        pbd = HW2CPU(pbd->next);
        ASSERT_ERR(pbd != NULL);
    }
}

void rxl_mpdu_partial_transfer(struct rxdesc *rxdesc, uint16_t upload_len,
                               uint32_t hostbuf, uint16_t payl_offset,
                               cb_rx_dma_func_ptr cb, void *env)
{
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    struct rx_payloaddesc *pd = HW2CPU(dma_hdrdesc->hd.first_pbd_ptr);
    struct rx_payloaddesc *prev_pd = NULL;

    // for profiling
    RX_MPDU_XFER_SET();

    // Program the payload transfer
    prev_pd = rxl_payload_transfer(&pd, upload_len, hostbuf, &payl_offset, 0, 1);

    // Now go through the additional RBD if any (e.g. for FCS, ICV)
    rxl_go_to_last_rbd(rxdesc, pd, prev_pd);

    // for profiling
    RX_MPDU_XFER_CLR();

    // Update upload control descriptor
    rxdesc->upload_cntrl.cb = cb;
    rxdesc->upload_cntrl.env = env;

    // Insert the element in the pending list
    rxl_upload_cntrl_push_pending(&rxdesc->upload_cntrl);
}

void rxl_mpdu_transfer(struct rxdesc *rxdesc)
{
    #if (NX_UMAC_PRESENT)
    struct rx_cntrl_rx_status *rx_status = &rxu_cntrl_env.rx_status;
    uint16_t payl_offset = rx_status->payl_offset;
    #else
    uint16_t payl_offset = 0;
    #endif
    uint32_t hostbuf, hostbuf_start;
    struct dma_desc *dma_desc;
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    uint16_t mpdu_len;
    struct rx_payloaddesc *pd;
    struct rx_payloaddesc *prev_pd = NULL;

    // for profiling
    RX_MPDU_XFER_SET();

    // Cache the pointers for faster access
    pd = (struct rx_payloaddesc *)HW2CPU(dma_hdrdesc->hd.first_pbd_ptr);

    // Get a host buffer address for DMA transfer (host buffer) if needed
    #if (NX_UMAC_PRESENT)
    hostbuf_start = macif_rx_buf_get(&rxu_cntrl_env.hostid_current);
    #else
    hostbuf_start = macif_rx_buf_get();
    #endif //(NX_UMAC_PRESENT)

    // Copy at the beginning of the host buffer + offset
    hostbuf = hostbuf_start + RXL_PAYLOAD_OFFSET;

    // Get the MPDU body length
    mpdu_len = dma_hdrdesc->hd.frmlen;

    // Sanity check - The frame length cannot be NULL...
    ASSERT_REC(mpdu_len != 0);
    // as well as not greater than the max allowed length
    ASSERT_REC(mpdu_len <= RWNX_MAX_AMSDU_RX);

    #if NX_AMSDU_DEAGG
    // Check if the frame is a A-MSDU
    if (dma_hdrdesc->flags & RX_FLAGS_IS_AMSDU_BIT)
    {
        // If the MPDU is a A-MSDU and the A-MSDU deaggregator is compiled, call the
        // A-MSDU de-aggregation function
        prev_pd = rxl_amsdu_deagg(dma_hdrdesc, &pd, mpdu_len, hostbuf, payl_offset);
    }
    else
    #endif
    {
        // If the MPDU is not a A-MSDU, or if the A-MSDU deaggregator is not compiled,
        // then we will upload the complete packet in a single host buffer
        prev_pd = rxl_payload_transfer(&pd, mpdu_len, hostbuf, &payl_offset, 0, 0);

        #if (NX_UMAC_PRESENT)
        // Save position of next fragment upload
        rx_status->host_buf_addr = hostbuf + mpdu_len;
        #endif
    }

    // Get the information on the current channel from the PHY driver
    phy_get_channel(&dma_hdrdesc->phy_info, PHY_PRIM);
    // Now elaborate the final DMA sending for the PHY Vectors and status
    // "Stamp" it as the final DMA of this MPDU transfer: insert a pattern for the upper
    // layer to be able to know that this hostbuf has been filled
    dma_hdrdesc->pattern = DMA_HD_RXPATTERN;
    // Get the IPC DMA descriptor of the PHYVECT header part
    dma_desc = &dma_hdrdesc->dma_desc;
    // Fill the destination address of the DMA descriptor
    // (hostbuf address is already up-to-date at this point)
    dma_desc->dest = hostbuf_start;
    dma_desc->src = CPU2HW(&dma_hdrdesc->hd.frmlen);
    dma_desc->length = RXL_HEADER_INFO_LEN;

    #if NX_UMAC_PRESENT && !NX_FULLY_HOSTED
    // Interruption only needed for the descriptor
    dma_desc->ctrl = RX_LLICTRL(0);
    #else
    // Last descriptor of the MPDU, so enable the interrupt on it
    dma_desc->ctrl = RX_LLICTRL(1);
    #endif //NX_UMAC_PRESENT && !NX_FULLY_HOSTED

    // Push the DMA descriptor in the DMA engine
    dma_push(dma_desc, dma_desc, RX_DATA_UPLOAD_CHAN);

    // Now go through the additional RBD if any (e.g. for FCS, ICV)
    rxl_go_to_last_rbd(rxdesc, pd, prev_pd);

    // for profiling
    RX_MPDU_XFER_CLR();

    #if NX_FULLY_HOSTED || !NX_UMAC_PRESENT
    // and push it in the pending list
    rxl_upload_cntrl_push_pending(&rxdesc->upload_cntrl);
    #endif
}

void rxl_mpdu_free(struct rxdesc *rxdesc)
{
    struct rx_payloaddesc *pd;
    struct rx_payloaddesc *prev_pd = NULL;
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);

    RX_MPDU_FREE_SET();

    // Cache the pointers for faster access
    pd = (struct rx_payloaddesc *)HW2CPU(dma_hdrdesc->hd.first_pbd_ptr);

    // Go through the RBD
    rxl_go_to_last_rbd(rxdesc, pd, prev_pd);

    if (rxl_upload_pending())
    {
        rxdesc->upload_cntrl.cb = NULL;
        rxdesc->upload_cntrl.flags |= RX_NO_UPLOAD;
        rxl_upload_cntrl_push_pending(&rxdesc->upload_cntrl);
    }
    else
    {
        // Release the frame
        rxl_frame_release(rxdesc);
    }

    RX_MPDU_FREE_CLR();
}

struct rxdesc *rxl_rxdesc_get(void)
{
    struct rxl_buffer_tag *buf = &rx_hwdesc_env.buf[DH_BUF];

    if (rxl_dh_buf_has_data())
    {
        // Ensure we are aligned with HW and adjust our read address if required
        rxl_adjust_sw_buf_read(buf);

        return (HW2CPU(buf->read & ~RXL_WRAP));
    }

    return NULL;
}

void rxl_rxdesc_ready_for_processing(struct rxdesc *rxdesc)
{
    struct rx_hd *rhd = &rxdesc->dma_hdrdesc.hd;

    // Sanity check - Ensure we are aligned with the HW
    ASSERT_ERR(rhd->upatternrx == RX_HEADER_DESC_PATTERN);

    // The HW now includes the MIC/ICV/FCS length in the total length, remove them as
    // all the RX path does not expect that.
    rxl_adjust_frmlen(rhd);

    // Initialize the RX descriptor
    #if NX_FULLY_HOSTED
    rxdesc->upload_cntrl.cb = NULL;
    #else
    rxdesc->upload_cntrl.cb = rxl_host_irq_mitigation_update;
    #endif
    rxdesc->upload_cntrl.rxdesc = rxdesc;
    rxdesc->upload_cntrl.flags = 0;
    rxdesc->buf_id = DH_BUF;
}

void rxl_mpdu_isr(void)
{
    // for profiling
    PROF_RX_MAC_IRQ_SET();

    // clear the interrupt
    nxmac_tx_rx_int_ack_clear(NXMAC_RX_BUFFER_1_TRIGGER_BIT);
    nxmac_enable_rx_buffer_1_trigger_setf(0);

    // Defer the handling in background
    ke_evt_set(KE_EVT_RXREADY_BIT);

    // for profiling
    PROF_RX_MAC_IRQ_CLR();
}

void rxl_immediate_frame_get(void)
{
    struct rxl_buffer_tag *buf = &rx_hwdesc_env.buf[IH_BUF];

    while (rxl_ih_buf_has_data())
    {
        struct rxdesc *rxdesc;

        // Ensure we are aligned with HW and adjust our read address if required
        rxl_adjust_sw_buf_read(buf);

        rxdesc = HW2CPU(buf->read & ~RXL_WRAP);

        rxl_rxdesc_ready_for_processing(rxdesc);
        rxdesc->buf_id = IH_BUF;

        rxl_rxcntrl_frame(rxdesc);
    }
}

void rxl_current_desc_get(struct rx_hd **rhd, struct rx_pbd **rbd)
{
    struct rxdesc *rxdesc = rxl_rxdesc_get();

    // First RX Header Descriptor chained to the HW
    if (rxdesc)
    {
        struct rx_hd *rx_hd = &rxdesc->dma_hdrdesc.hd;
        *rhd = rx_hd;
        *rbd = HW2CPU(rx_hd->first_pbd_ptr);
    }
    else
    {
        *rhd = NULL;
        *rbd = NULL;
    }
}

/// @} // end of group RXHWDESC
