/**
 ****************************************************************************************
 *
 * @file scan.c
 *
 * @brief Scanning module implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup SCAN
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
// for mode
#include "scan.h"
#include "scan_task.h"
#include "mac_frame.h"
#include "hal_dma.h"
#include "mac_ie.h"
#include "vif_mgmt.h"
#include "txl_frame.h"
#include "chan.h"
#include "tpc.h"

#if NX_HW_SCAN
/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */

/**
 * SCAN Context variable, used to store information related to scan module
 */
struct scan_env_tag scan_env;

/**
 ****************************************************************************************
 * @brief Searchs for DS IE in probe request frame
 *
 * If DS IE is present in the probe request frame provided by UMAC, then save pointer
 * so that it can updated with the channel number before sending the probe request
 ****************************************************************************************
 */
static void scan_search_ds(void)
{
    struct scan_start_req const *param = scan_env.param;
    struct scan_probe_req_ie_tag *ie_desc = &scan_probe_req_ie;

    // Search for the DS IE
    scan_env.ds_ie = mac_ie_ds_find(CPU2HW(ie_desc->buf), param->add_ie_len);
}

/**
 ****************************************************************************************
 * @brief Callback for DMA transfer of probe request frame from host memory
 *
 * @param[in] env       Pointer associated to DMA transfer (NULL is this case)
 * @param[in] dma_queue DMA queue on which transfer happened
 ****************************************************************************************
 */
static void dma_cb(void *env, int dma_queue)
{
    // Search the channel information element in the 2.4GHz IE buffer
    scan_search_ds();

    // Send the set channel request
    scan_set_channel_request();
}


void scan_init(void)
{
    struct scan_probe_req_ie_tag *ie_desc = &scan_probe_req_ie;
    struct tx_pbd *pbd = &ie_desc->pbd;
    struct hal_dma_desc_tag *gp_dma = &scan_env.dma_desc;

    // Reset the SCAN environment
    memset(&scan_env, 0, sizeof(scan_env));

    // Set the state of the SCAN task
    ke_state_set(TASK_SCAN, SCAN_IDLE);

    // Initialize the DMA descriptors

    // Initialization of the GP DMA descriptor
    gp_dma->cb = dma_cb;
    gp_dma->env = NULL;
    gp_dma->dma_desc = &ie_desc->dma_desc;

    // Initialization of the HW DMA descriptor
    ie_desc->dma_desc.dest = CPU2HW(ie_desc->buf);

    // Initialization of the PBD descriptor
    pbd->bufctrlinfo = 0;
    pbd->datastartptr = CPU2HW(ie_desc->buf);
    pbd->upatterntx = TX_PAYLOAD_DESC_PATTERN;
    pbd->next = 0;
}

void scan_ie_download(struct scan_start_req const *param)
{
    struct scan_probe_req_ie_tag *ie_desc = &scan_probe_req_ie;
    struct tx_pbd *pbd = &ie_desc->pbd;
    #if !NX_UMAC_PRESENT
    struct dma_desc *hw_dma = &ie_desc->dma_desc;
    struct hal_dma_desc_tag *gp_dma = &scan_env.dma_desc;
    #endif

    do
    {
        #if NX_UMAC_PRESENT
        // Search for the channel IE in the buffer
        scan_search_ds();

        // Launch the scan request
        scan_set_channel_request();
        #else
        // Check if we have an IE buffer for this band
        if (param->add_ies == 0)
        {
            // Send the set channel request
            scan_set_channel_request();

            // No IEs, so exit immediately
            break;
        }

        // Sanity check - Ensure we have enough space in our IE buffer
        ASSERT_ERR(param->add_ie_len <= SCAN_MAX_IE_LEN);

        // Initialization of the HW DMA descriptor
        hw_dma->src = param->add_ies;
        hw_dma->length = param->add_ie_len;

        // Push the DMA request
        hal_dma_push(gp_dma, DMA_DL);

        // Change the task state
        ke_state_set(TASK_SCAN, SCAN_WAIT_IE_DWNLD);
        #endif

        // Initialization of the PBD descriptor
        pbd->bufctrlinfo = 0;
        pbd->dataendptr = pbd->datastartptr + param->add_ie_len - 1;
    } while(0);
}

void scan_set_channel_request(void)
{
    struct mac_chan_def const *chan = &scan_env.param->chan[scan_env.chan_idx];

    // Request the scan channel to the MM channel manager
    chan_scan_req(chan->band, chan->freq,  chan->tx_power,
                  (chan->flags & (CHAN_NO_IR | CHAN_RADAR))?SCAN_PASSIVE_DURATION:SCAN_ACTIVE_DURATION,
                  chan->flags,
                  scan_env.param->vif_idx);

    // Fill-in the DS information element field
    if (scan_env.ds_ie != 0)
    {
        co_write8p(scan_env.ds_ie + MAC_DS_CHANNEL_OFT, phy_freq_to_channel(chan->band, chan->freq));
    }

    // Change the task state
    ke_state_set(TASK_SCAN, SCAN_WAIT_CHANNEL);
}

void scan_probe_req_tx(void)
{
    struct scan_start_req const *param = scan_env.param;
    struct mac_chan_def const *chan = &param->chan[scan_env.chan_idx];
    struct scan_probe_req_ie_tag *ie_desc = &scan_probe_req_ie;
    struct vif_info_tag *vif = &vif_info_tab[param->vif_idx];
    struct txl_frame_desc_tag *frame;
    struct preq_frame *buf;
    uint32_t ssid_addr;
    int txtype;
    int i;

    for (i = 0; i < param->ssid_cnt; i++)
    {
        struct tx_hd *thd;
        struct mac_ssid const *ssid = &param->ssid[i];
        int length;

        #if (NX_P2P)
        if (vif->p2p)
        {
            txtype = TX_DEFAULT_5G;
        }
        else
        #endif //(NX_P2P)
        {
            // Chose the right rate according to the band
            txtype = ((chan->band == PHY_BAND_2G4) && (!param->no_cck))?TX_DEFAULT_24G:TX_DEFAULT_5G;
        }

        // Compute the ProbeReq length
        length = MAC_SHORT_MAC_HDR_LEN + MAC_SSID_SSID_OFT + ssid->length +
                                                         param->add_ie_len;

        // Allocate a frame descriptor from the TX path
        frame = txl_frame_get(txtype, length);
        if (frame == NULL)
            break;

        // Get the buffer pointer
        buf = txl_buffer_payload_get(&frame->txdesc);
        thd = &frame->txdesc.lmac.hw_desc->thd;

        // Prepare the MAC Header
        buf->h.fctl = MAC_FCTRL_PROBEREQ;
        buf->h.durid = 0;
        buf->h.addr1 = mac_addr_bcst;
        buf->h.addr2 = vif->mac_addr;
        buf->h.addr3 = param->bssid;
        buf->h.seq = txl_get_seq_ctrl();

        // Copy the SSID in the packet
        ssid_addr = CPU2HW(&buf->payload);
        co_write8p(ssid_addr++, MAC_ELTID_SSID);
        co_write8p(ssid_addr++, ssid->length);
        co_pack8p(ssid_addr, ssid->array, ssid->length);

        // Attach the IE buffer to the ProbeReq THD
        thd->first_pbd_ptr = CPU2HW(&ie_desc->pbd);
        thd->dataendptr -= param->add_ie_len;

        // Fill-in the confirmation structure
        frame->cfm.cfm_func = NULL;
        frame->cfm.env = NULL;

        #if (NX_CHNL_CTXT || NX_P2P)
        // Set VIF and STA indexes
        frame->txdesc.host.vif_idx = param->vif_idx;
        frame->txdesc.host.staid   = 0xFF;
        #endif //(NX_CHNL_CTXT || NX_P2P)

        // Push the frame for TX
        txl_frame_push(frame, AC_VO);
    }
}

void scan_send_cancel_cfm(uint8_t status, ke_task_id_t dest_id)
{
    struct scan_cancel_cfm *cfm = KE_MSG_ALLOC(SCAN_CANCEL_CFM,
                                               dest_id, TASK_SCAN,
                                               scan_cancel_cfm);

    cfm->status = status;

    // Send the response
    ke_msg_send(cfm);
}

struct mac_chan_def const *scan_get_chan(void)
{
    return (&scan_env.param->chan[scan_env.chan_idx]);
}
#endif //(NX_HW_SCAN)

/// @} end of group
