/**
 ****************************************************************************************
 *
 * @file sta_mgmt.c
 *
 * @brief UMAC Station Management implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup STA_MGMT
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "sta_mgmt.h"
#include "mac_frame.h"
#include "co_utils.h"
#include "co_status.h"
#include "txl_buffer.h"
#include "txl_frame.h"
#include "vif_mgmt.h"
#if (NX_UMAC_PRESENT)
#include "bam.h"
#include "mm_task.h"
#include "apm.h"
#endif //(NX_UMAC_PRESENT)
#if (NX_P2P)
#include "p2p.h"
#endif //(NX_P2P)
#if (RW_BFMER_EN)
/// TX Beamforming Definition
#include "bfr.h"
#endif //(RW_BFMER_EN)
#if (RW_UMESH_EN)
#include "mm.h"
#include "mesh_ps.h"
#endif //(RW_UMESH_EN)
#if (NX_UMAC_PRESENT && NX_TDLS)
#include "tdls.h"
#endif // (NX_UMAC_PRESENT && NX_TDLS)

/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
struct sta_info_env_tag sta_info_env;
struct sta_info_tag sta_info_tab[STA_MAX];

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Initializes (resets) the content of a STA entry
 *
 * @param[in] sta       A pointer to the STA entry to reset
 ****************************************************************************************
 */
static void sta_mgmt_entry_init(struct sta_info_tag *sta)
{
    #if (NX_TX_FRAME)
    // Free all pending descriptor
    while (!co_list_is_empty(&sta->tx_desc_post))
    {
        // Get first TX desc in the list
        struct txdesc *txdesc = (struct txdesc *)co_list_pop_front(&sta->tx_desc_post);

        // Insert it back in the list of free TX descriptors
        txl_frame_release(txdesc, true);
    }
    #endif //(NX_TX_FRAME)

    // Reset table
    memset(sta, 0, sizeof(*sta));

    #if (NX_UMAC_PRESENT)
    for (int tid = 0; tid < TID_MAX; tid++)
    {
        sta->ba_info[tid].bam_idx_rx = BAM_INVALID_TASK_IDX;
        sta->ba_info[tid].bam_idx_tx = BAM_INVALID_TASK_IDX;
    }
    #endif //(NX_UMAC_PRESENT)

    // Set the instance number to 0xFF, indicate the sta is free
    sta->inst_nbr = 0xFF;
}

void sta_mgmt_init(void)
{
    int i;

    // Initialize STA search lists
    // init free list
    co_list_init(&sta_info_env.free_sta_list);

    // push all the entries to the free list
    for(i = 0; i < NX_REMOTE_STA_MAX ; i++)
    {
        struct sta_info_tag *sta = &sta_info_tab[i];
        // Init STA info table.
        sta_mgmt_entry_init(sta);
        // Push to free list.
        co_list_push_back(&sta_info_env.free_sta_list, (struct co_list_hdr*)sta);
    }

    #if NX_UMAC_PRESENT
    // Initialize the BC/MC pseudo stations
    for(i = 0; i < NX_VIRT_DEV_MAX ; i++)
    {
        uint8_t idx = VIF_TO_BCMC_IDX(i);
        struct sta_info_tag *sta = &sta_info_tab[idx];
        struct vif_info_tag *vif = &vif_info_tab[i];

        // Init STA info table.
        sta_mgmt_entry_init(sta);

        // Attach the station to its VIF
        sta->inst_nbr = i;
        sta->pol_tbl.buf_ctrl = &txl_buffer_control_desc_bcmc[i];
        sta->ctrl_port_state = PORT_CLOSED;
        sta->sta_sec_info.cur_key = &vif->default_key;
        sta->staid = idx;
        sta->aid = 0;
    }
    #endif

    #if (RW_MESH_EN && !NX_UMAC_PRESENT)
    // Initialize Mesh Information (for security keys management)
    for (i = 0; i < RW_MESH_LINK_NB; i++)
    {
        sta_info_env.mesh_link_idx[i] = INVALID_STA_IDX;
    }

    sta_info_env.nb_mesh_sta = 0;
    #endif //(RW_MESH_EN && !NX_UMAC_PRESENT)
}

uint8_t sta_mgmt_register(struct mm_sta_add_req const *param,
                          uint8_t *sta_idx)
{
    struct sta_info_tag *sta;
    struct sta_info_tag *ref_bssid_sta;
    #if (NX_UMAC_PRESENT || NX_P2P || NX_MULTI_ROLE || RW_MESH_EN || NX_TD_STA || NX_TDLS)
    struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];
    #endif //(NX_UMAC_PRESENT || NX_P2P || NX_MULTI_ROLE || RW_MESH_EN)
    #if NX_UMAC_PRESENT
    uint32_t time;
    #endif

    #if (RW_MESH_EN && !NX_UMAC_PRESENT)
    // If Mesh VIF, number of available connection is limited
    if ((vif->type == VIF_MESH_POINT) && (sta_info_env.nb_mesh_sta == RW_MESH_LINK_NB))
    {
        return CO_FAIL;
    }
    #endif //(RW_MESH_EN && !NX_UMAC_PRESENT)

    // get an entry from the free station list
    sta = (struct sta_info_tag*)co_list_pop_front(&sta_info_env.free_sta_list);

    // no entries available return immediately
    if (sta == NULL)
        // there are no free tables
        return CO_FAIL;

    // If required, get a second entry for reference BSSID from the free station list
    if (param->bssid_index)
    {
        ref_bssid_sta = (struct sta_info_tag*)co_list_pop_front(&sta_info_env.free_sta_list);
        if (ref_bssid_sta == NULL)
        {
            // Push back the station entry in the free list
            co_list_push_back(&sta_info_env.free_sta_list, (struct co_list_hdr*)sta);
            return CO_FAIL;
        }

        // Link the stations together
        sta->linked_sta = ref_bssid_sta;
        ref_bssid_sta->linked_sta = sta;

        // Configure the required fields the linked station entry
        mac_ref_bssid_get(param->bssid_index, param->max_bssid_ind, &param->mac_addr,
                          &ref_bssid_sta->mac_addr);
        ref_bssid_sta->aid = STA_REF_BSSID_AID;
        ref_bssid_sta->staid = CO_GET_INDEX(ref_bssid_sta, sta_info_tab);
    }

    // Initialize some fields from the parameters
    sta->mac_addr = param->mac_addr;
    sta->ampdu_spacing_min = co_max(param->ampdu_spacing_min, NX_TX_MPDU_SPACING);
    sta->ampdu_size_max_ht = param->ampdu_size_max_ht;
    sta->ampdu_size_max_vht = param->ampdu_size_max_vht;
    sta->ampdu_size_max_he = param->ampdu_size_max_he;
    sta->paid_gid = param->paid_gid;
    sta->inst_nbr = param->inst_nbr;

    // Get the station index
    *sta_idx = CO_GET_INDEX(sta, sta_info_tab);
    sta->staid = *sta_idx;
    #if (NX_TDLS)
    if (vif->type == VIF_STA)
    {
        sta->is_tdls = param->tdls_sta;
        #if (NX_UMAC_PRESENT)
        if (param->tdls_sta)
        {
            tdls_add_sta(sta, param->tdls_initiator, param->tdls_chsw_allowed);
        }
        else
        {
            vif->u.sta.tdls_chsw_sta_idx = INVALID_STA_IDX;
        }
        #endif //(NX_UMAC_PRESENT)
    }
    #endif

    #if (RW_MESH_EN && !NX_UMAC_PRESENT)
    // Get a Mesh Link Index
    if (vif->type == VIF_MESH_POINT)
    {
        // Get Mesh Link Index
        for (int i = 0; i < RW_MESH_LINK_NB; i++)
        {
            if (sta_info_env.mesh_link_idx[i] == INVALID_STA_IDX)
            {
                sta->mlink_idx = i;
                sta_info_env.mesh_link_idx[i] = *sta_idx;
                break;
            }
        }

        // Increase number of Mesh STA currently added
        sta_info_env.nb_mesh_sta++;
    }
    #endif //(RW_MESH_EN && !NX_UMAC_PRESENT)

    #if (!NX_UMAC_PRESENT && NX_MULTI_ROLE)
    if (vif->type == VIF_STA)
    {
        sta->bcn_int = vif->u.sta.ap_bcn_intv * TU_DURATION;
    }
    #endif //(!NX_UMAC_PRESENT && NX_MULTI_ROLE)
    {
        sta->bcn_int = 100 * TU_DURATION;
    }

    #if (NX_P2P)
    if (vif->p2p)
    {
        // Increase number of P2P STAs registered for the VIF
        vif->p2p_link_nb++;
    }
    #endif //(NX_P2P)

    #if NX_UMAC_PRESENT
    sta->rx_nqos_last_seqcntl  = 0xFFFF;

    for (int i = 0; i < TID_MAX; i++)
    {
        sta->rx_qos_last_seqcntl[i] = 0xFFFF;
    }
    sta->pol_tbl.buf_ctrl = &txl_buffer_control_desc[*sta_idx];
    sta->ctrl_port_state = PORT_CLOSED;
    if ((vif->flags & WPA_WPA2_IN_USE)
        #if (RW_MESH_EN)
            || (vif->type == VIF_MESH_POINT)
        #endif //(RW_MESH_EN)
        #if (NX_TDLS)
            || ((vif->type == VIF_STA) && (param->tdls_sta))
        #endif
            )
    {
        struct sta_mgmt_sec_info *sec = &sta->sta_sec_info;
        sec->cur_key = &sec->pairwise_key;
    }
    else
    {
        struct sta_mgmt_sec_info *sec = &sta->sta_sec_info;
        sec->cur_key = &vif->default_key;
    }
    time = hal_machw_time();
    for (int i = 0; i < TID_MAX; i++)
    {
        sta->ba_info[i].last_ba_add_time = time - BAM_ADDBA_REQ_INTERVAL;
    }

    #if NX_MFP
    if (vif->flags & MFP_IN_USE)
    {
        STA_CAPA_SET(sta, MFP);
    }
    #endif //(NX_MFP)
    #endif //(NX_UMAC_PRESENT)

    #if (NX_UMAC_PRESENT || NX_TD_STA)
    // Link the STA to its associated VIF
    co_list_push_back(&vif->sta_list, &sta->list_hdr);
    #endif //(NX_UMAC_PRESENT || NX_TD_STA)

    #if NX_UMAC_PRESENT && NX_MAC_HE
    // Put a default value in the BSR to ensure initial HE SU transmissions would have
    // a meaningful value to put in the +HTC HE Variant
    sta->bsr = MAC_HTC_HE(TX_HE_BSR_FULL);
    #endif

    // In order to keep the status of the STA and potential linked STA aligned in case
    // they are checked under interrupt, update them in a critical section
    GLOBAL_INT_DISABLE();
    // The station and potential linked station are now considered as active
    sta->valid = true;
    if (sta->linked_sta != NULL)
        sta->linked_sta->valid = true;
    GLOBAL_INT_RESTORE();

    // return the allocated station entry index
    return CO_OK;
}

void sta_mgmt_unregister(uint8_t sta_idx)
{
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    #if (NX_P2P || NX_UMAC_PRESENT || RW_MESH_EN || NX_TD_STA || NX_TDLS)
    struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];
    #endif

    #if (NX_P2P)
    if (vif->p2p)
    {
        // Decrease number of P2P STAs registered for the VIF
        vif->p2p_link_nb--;
    }
    #endif //(NX_P2P)

    #if (RW_BFMER_EN)
    if (bfr_is_enabled())
    {
        bfr_del_sta_ind(sta_idx);
    }
    #endif //(RW_BFMER_EN)

    #if (NX_UMAC_PRESENT || NX_TD_STA)
    co_list_extract(&vif->sta_list, &sta->list_hdr);
    #endif //(NX_UMAC_PRESENT || NX_TD_STA)

    #if (RW_MESH_EN && !NX_UMAC_PRESENT)
    // Get a Mesh Link Index
    if (vif->type == VIF_MESH_POINT)
    {
        // Sanity check
        ASSERT_ERR(sta_info_env.mesh_link_idx[sta->mlink_idx] == sta->staid);

        // Unlink STA index and Link index
        sta_info_env.mesh_link_idx[sta->mlink_idx] = INVALID_STA_IDX;

        // Decrease number of Mesh STA currently added
        sta_info_env.nb_mesh_sta--;
    }
    #endif //(RW_MESH_EN && !NX_UMAC_PRESENT)

    #if (NX_TDLS)
    if (sta->is_tdls)
    {
        #if (NX_UMAC_PRESENT)
        tdls_del_sta(sta);
        #else
        if (sta->tdls.chsw_active)
        {
            // Clear channel switch request timer
            mm_timer_clear(&sta->tdls.chsw_req_timer);
            sta->tdls.chsw_active = false;
        }
        if (vif->u.sta.tdls_chsw_sta_idx == sta_idx)
        {
            vif->u.sta.tdls_chsw_sta_idx = INVALID_STA_IDX;
        }
        #endif //(NX_UMAC_PRESENT)
    }
    #endif

    // In order to keep the status of the STA and potential linked STA aligned in case
    // they are checked under interrupt, update them in a critical section
    GLOBAL_INT_DISABLE();
    sta->valid = false;
    if (sta->linked_sta != NULL)
        sta->linked_sta->valid = false;
    GLOBAL_INT_RESTORE();

    if (sta->linked_sta != NULL)
    {
        // Fully reset the reference BSSID STA entry
        sta_mgmt_entry_init(sta->linked_sta);

        // Push it back in the free list
        co_list_push_back(&sta_info_env.free_sta_list, (struct co_list_hdr*)sta->linked_sta);
    }

    // Fully reset the entry
    sta_mgmt_entry_init(sta);

    // Push it back in the free list
    co_list_push_back(&sta_info_env.free_sta_list, (struct co_list_hdr*)sta);
}

#if (NX_UMAC_PRESENT)
void sta_mgmt_add_key(struct mm_key_add_req const *param, uint8_t hw_key_idx)
{
    struct sta_info_tag *sta = &sta_info_tab[param->sta_idx];
    struct sta_mgmt_sec_info *sec = &sta->sta_sec_info;
    struct key_info_tag *key;
    #if RW_WAPI_EN
    struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];
    #endif

    #if (RW_UMESH_EN)
    if (hw_key_idx >= MM_SEC_MAX_MFP_KEY_NBR)
    {
        key = &sec->key_mfp_mesh_info[MM_MESH_MFP_KEY_TO_KEYID(hw_key_idx)];
    }
    else
    #endif //(RW_UMESH_EN)
    {
        key = &sec->key_info;
    }

    // Store the key information
    key->hw_key_idx = hw_key_idx;
    key->cipher = param->cipher_suite;
    key->key_idx = param->key_idx;

    // Reset the replay counters
    memset(key->rx_pn, 0, TID_MAX * sizeof(uint64_t));

    // Check which encryption type has to be used
    switch(key->cipher)
    {
        case MAC_CIPHER_WEP40:
        case MAC_CIPHER_WEP104:
            key->tx_pn = co_rand_word() & 0xFFFFFF;
            break;
        case MAC_CIPHER_TKIP:
            key->tx_pn = 0;
            key->u.mic.tx_key[0] = param->key.array[4];
            key->u.mic.tx_key[1] = param->key.array[5];
            key->u.mic.rx_key[0] = param->key.array[6];
            key->u.mic.rx_key[1] = param->key.array[7];
            break;
        #if RW_WAPI_EN
        case MAC_CIPHER_WPI_SMS4:
            key->tx_pn = 0x5c365c365c365c36ULL;
            if (vif->type == VIF_AP)
                key->tx_pn++;
            break;
        #endif
        #if (RW_UMESH_EN)
        #if NX_MFP
        case MAC_CIPHER_BIP_CMAC_128:
            memcpy(key->u.mfp.key, param->key.array, sizeof(key->u.mfp.key));
            key->tx_pn = 0;
            break;
        #endif
        #endif //(RW_UMESH_EN)
        default:
            key->tx_pn = 0;
            break;
    }

    // Key is now valid
    key->valid = true;

    #if (RW_UMESH_EN)
    if (hw_key_idx < MM_SEC_MAX_MFP_KEY_NBR)
    #endif //(RW_UMESH_EN)
    {
        // Set the pairwise key and current key pointers
        sec->pairwise_key = key;
    }
}

void sta_mgmt_del_key(struct sta_info_tag *sta)
{
    struct sta_mgmt_sec_info *sec = &sta->sta_sec_info;
    struct key_info_tag *key = &sec->key_info;

    // Key is now invalid
    key->valid = false;
    sec->pairwise_key = NULL;

    // Port is now controlled
    sta->ctrl_port_state = PORT_CONTROLED;
}
#endif //(NX_UMAC_PRESENT)

#if (NX_TX_FRAME)
int sta_mgmt_send_postponed_frame(struct vif_info_tag *vif, struct sta_info_tag *sta,
                                  int limit)
{
    int nb = 0;

    // Go through list of packets pending for transmission
    while (!co_list_is_empty(&sta->tx_desc_post))
    {
        // Get first TX desc
        struct txdesc *txdesc = (struct txdesc *)co_list_pick(&sta->tx_desc_post);
        // Get frame descriptor
        struct txl_frame_desc_tag *frame_desc;
        #if (NX_UMAC_PRESENT && NX_BEACONING)
        int stop = 0;
        #endif // (NX_UMAC_PRESENT && NX_BEACONING)

        // Check that the packet can be transmitted
        if (!txl_cntrl_tx_check(vif)
        #if (NX_UMAC_PRESENT && NX_BEACONING)
            || !apm_tx_int_ps_check(txdesc)
        #endif // (NX_UMAC_PRESENT && NX_BEACONING)
        #if (RW_UMESH_EN)
            || !mesh_ps_check_peer_presence(vif, sta->staid)
        #endif //(RW_UMESH_EN)
            )
        {
            break;
        }

        #if (NX_UMAC_PRESENT && NX_BEACONING)
        txdesc = apm_tx_int_ps_get_postpone(vif, sta, &stop);
        if (stop)
            break;
        else if (!txdesc)
        #endif //(NX_UMAC_PRESENT && NX_BEACONING)
            txdesc = (struct txdesc *)co_list_pop_front(&sta->tx_desc_post);
        frame_desc = (struct txl_frame_desc_tag *)txdesc;

        #if (RW_UMESH_EN)
        mesh_ps_send_postponed_frame(vif, sta, frame_desc);
        #endif //(RW_UMESH_EN)

        // clear postponed flagm
        frame_desc->postponed = false;

        // Send the frame
        // In this case, Access Category has been stored in TID field
        txl_cntrl_push_int(txdesc, txdesc->host.tid);

        // Count the number of frame sent and limit if requested
        nb++;
        if ((limit) && (limit == nb))
            break;
    }

    return nb;
}
#endif //(NX_TX_FRAME)

/// @}
