/**
 ****************************************************************************************
 *
 * @file sta_mgmt.h
 *
 * @brief UMAC Station Management definitions.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

#ifndef _STA_MGMT_H_
#define _STA_MGMT_H_

/**
 ****************************************************************************************
 * @defgroup STA_MGMT STA_MGMT
 * @ingroup LMAC
 * @brief Station Management.
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "rwnx_config.h"

// for linked list definitions
#include "co_list.h"
// for mac_addr and other structure definitions
#include "mac.h"
// for struct mm_sta_add_req definition
#include "mm_task.h"

#if (NX_TDLS)
// for struct tdls_chan_switch_req definition
#include "tdls_task.h"
// for mm_timer_tag
#include "mm_timer.h"
#endif

#if (NX_UMAC_PRESENT)
#include "rxu_cntrl.h"
#include "bam.h"
#endif //(NX_UMAC_PRESENT)

/*
 * DEFINES
 ****************************************************************************************
 */

#if (NX_UMAC_PRESENT)
/**
 ***********************************************************************
 * Protection Configuration Management
 ***********************************************************************
 */
/// Mask of protection flags in policy table
#define STA_MGMT_PROT_HW_MASK           (PROT_FRM_EX_RCX_MASK | MCS_INDEX_PROT_TX_RCX_MASK | BW_PROT_TX_RCX_MASK |\
                                         FORMAT_MOD_PROT_TX_RCX_MASK)

#endif //(NX_UMAC_PRESENT)

/// Station indexes.
enum
{
    #if NX_UMAC_PRESENT
    /// BROADCAST/GROUP DATA TX STA Index for first virtual AP.
    BROADCAST_STA_IDX_MIN = NX_REMOTE_STA_MAX,
    /// BROADCAST/GROUP DATA TX STA Index for last virtual AP.
    BROADCAST_STA_IDX_MAX = NX_REMOTE_STA_MAX + NX_VIRT_DEV_MAX - 1,
    /// Maximum number of supported STA
    STA_MAX,
    #else
    STA_MAX = NX_REMOTE_STA_MAX,
    #endif
    /// Invalid STA Index used for error checking.
    INVALID_STA_IDX = 0xFF
};

/// Macro giving the BCMC station index for the VIF index passed as parameter
#define VIF_TO_BCMC_IDX(idx) (BROADCAST_STA_IDX_MIN + (idx))

/// Specific AID value indicating that STA is the reference BSSID, in case we are
/// connected to a nonTransmitted BSSID
#define STA_REF_BSSID_AID ((uint8_t)-1)

/*
 * ENUMERATIONS
 ****************************************************************************************
 */

#if (NX_UMAC_PRESENT)
/// Definitions of key types
enum sta_mgmt_key_type
{
    STA_MGMT_KEY_TYPE_PAIRWISE   = 0,
    STA_MGMT_KEY_TYPE_GROUP,

    STA_MGMT_KEY_TYPE_MAX
};

/// Logical port state
enum sta_mgmt_port_state
{
    /// No data traffic could be exchanged with this station
    STA_MGMT_PORT_CLOSED         = 0,
    /// Encryption key is not yet available, only EAP frames could be sent
    STA_MGMT_PORT_CONTROLED,
    /// Any data types could be sent
    STA_MGMT_PORT_OPEN
};

/// Policy Table update bit field indexes
enum sta_mgmt_pol_upd
{
    /// Update Rate
    STA_MGMT_POL_UPD_RATE,
    /// Update Protection Configuration
    STA_MGMT_POL_UPD_PROT,
    /// Update PPDU TX Configuration
    STA_MGMT_POL_UPD_PPDU_TX,
    /// Update operational bandwidth
    STA_MGMT_POL_UPD_BW,
    /// Update tx power
    STA_MGMT_POL_UPD_TX_POWER,
    /// Update Trial Rate
    STA_MGMT_POL_TRIAL_RATE,
    /// Update BSS color
    STA_MGMT_POL_UPD_BSS_COLOR,

    STA_MGMT_POL_UPD_MAX
};

/// PS traffic type
enum sta_ps_traffic
{
    /// Legacy PS traffic from Host
    PS_TRAFFIC_HOST = CO_BIT(0),
    /// Legacy PS traffic generated by firmware
    PS_TRAFFIC_INT = CO_BIT(1),
    /// Legacy PS traffic
    PS_TRAFFIC = (PS_TRAFFIC_HOST | PS_TRAFFIC_INT),

    /// UAPSD traffic from the HOST
    UAPSD_TRAFFIC_HOST = CO_BIT(2),
    /// UAPSD traffic generated by firmware
    UAPSD_TRAFFIC_INT = CO_BIT(3),
    /// UAPSD traffic
    UAPSD_TRAFFIC = (UAPSD_TRAFFIC_HOST | UAPSD_TRAFFIC_INT)
};

/// PS service period
enum sta_ps_sp
{
    /// No Service period on going
    NO_SERVICE_PERIOD = 0,
    /// Legacy PS service period on going (i.e. answer to PS-Poll)
    PS_SERVICE_PERIOD = CO_BIT(0),
    /// UAPSD service period on going (for internal frames)
    UAPSD_SERVICE_PERIOD_INT = CO_BIT(1),
    /// UAPSD service period on going (for frames from host)
    UAPSD_SERVICE_PERIOD_HOST = CO_BIT(2),
    /// UAPSD service period on going
    UAPSD_SERVICE_PERIOD = (UAPSD_SERVICE_PERIOD_INT | UAPSD_SERVICE_PERIOD_HOST),
    /// Any internal service period
    ANY_SERVICE_PERIOD_INT = (PS_SERVICE_PERIOD | UAPSD_SERVICE_PERIOD_INT),
    /// Beacon Service Period (i.e. multicast/broadcast traffic after DTIM beacon)
    BCN_SERVICE_PERIOD = CO_BIT(3),
};

#endif //(NX_UMAC_PRESENT)

/// logical port state
enum
{
    /// no data traffic could be exchanged with this station
    PORT_CLOSED = 0,
    /// encryption key is not yet available, only EAP frames could be sent
    PORT_CONTROLED,
    /// any data types could be sent
    PORT_OPEN
};

/*
 * TYPE DEFINITIONS
 ****************************************************************************************
 */

#if (NX_UMAC_PRESENT)
// Forward declarations
struct mm_key_add_req;

/// Station capability information
struct sta_capa_info
{
    /// Legacy rate set supported by the STA
    struct mac_rateset rate_set;
    /// HT capabilities
    struct mac_htcapability ht_cap;
    #if NX_VHT
    /// VHT capabilities
    struct mac_vhtcapability vht_cap;
    #endif
    #if NX_HE
    /// HE capabilities
    struct mac_hecapability he_cap;
    #endif
    /// Bitfield showing some capabilities of the STA (@ref STA_QOS_CAPA, @ref STA_HT_CAPA,
    /// @ref STA_VHT_CAPA, @ref STA_MFP_CAPA)
    uint32_t capa_flags;
    /// Maximum PHY channel bandwidth supported by the STA
    uint8_t phy_bw_max;
    /// Current channel bandwidth for the STA
    uint8_t bw_cur;
    /// Bit field indicating which queues have to be delivered upon U-APSD trigger
    uint8_t uapsd_queues;
    /// Maximum size, in frames, of a APSD service period
    uint8_t max_sp_len;
    /// Maximum number of spatial streams supported for STBC reception
    uint8_t stbc_nss;
};

/// Station Information related to Security
struct sta_mgmt_sec_info
{
    /// Information relative to the key
    struct key_info_tag key_info;
    #if (RW_MESH_EN)
    /// Mesh Group Management Keys used by peer Mesh STA
    struct key_info_tag key_mfp_mesh_info[MAC_DEFAULT_MFP_KEY_COUNT - MAC_DEFAULT_KEY_COUNT];
    #endif //(RW_MESH_EN)
    /// Pointer to the pairwise key
    struct key_info_tag *pairwise_key;
    /// Pointer to the currently used key pointer
    struct key_info_tag **cur_key;

};

/// Information related to the used policy table
struct sta_pol_tbl_cntl
{
    /// Pointer to the buffer control structure
    struct txl_buffer_control *buf_ctrl;
    /// Pointer to the rate control algorithm structure
    struct rc_sta_stats *sta_stats;
    /**
     * Protection Configuration
     * Mapped on Rate Control entry in policy table provided to the HW (see MAC HW USer Manual)
     *      Bit 28:26 - Format and Modulation of Protection frame for Transmission
     *      Bit 25:24 - Bandwidth of Protection Frame for Transmission
     *      Bit 23:17 - MCS Index of Protection Frame for Transmission
     *      Bit 16:14 - NAV Protection Frame Exchange
     * Pushed inside Policy Table if Update Protection bit set to 1 in upd_field
     */
    uint32_t prot_cfg;
    /**
     * PPDU Transmission Configuration
     * Mapped on Rate Control entry in policy table provided to the HW (see MAC HW USer Manual)
     *      Bit 10 - Preamble Type of PPDU for Transmission -> Bit 10
     * Pushed inside Policy Table if Update PPDU TX bit set to 1 in upd_field
     */
    uint16_t ppdu_tx_cfg;
    /// Bit field indicating update to be done in the Policy Table
    uint8_t upd_field;
};

/// Information about RX and TX Block Ack for one TID
struct sta_mgmt_ba_info
{
    /// Last time a ADD BA request has been attempted
    uint32_t last_ba_add_time;
    /// BAM task instance (RX Side)
    uint8_t bam_idx_rx;
    /// BAM task instance (TX Side)
    uint8_t bam_idx_tx;
    /// Offset to apply to the credit count of this STA/TID pair
    int8_t credit_oft;
    /// Number of frames currently in the TX path for this STA/TID pair
    uint8_t tx_pkt_cnt;
};
#endif //(NX_UMAC_PRESENT)

#if (NX_AMPDU_TX)
///BA agreement structure - one for every possible TID, valid if params non 0
struct ba_agmt
{
    ///Starting sequence number - the one allowed as lowest in an A-MPDU - will change as MPDUs get Acked
    uint16_t ssn;
    ///Buffer size - max number of MPDUs that can be held in recipient's buffer
    uint8_t bufsz;
};
#endif //(NX_AMPDU_TX)

#if (NX_TDLS)
/// Information related to an established TDLS link
struct tdls_info_tag
{
    /// Flag indicating if the TDLS peer is the TDLS link initiator
    bool initiator;
    /// Flag indicating if the TDLS channel switch is allowed
    bool chsw_allowed;
    /// Flag indicating if the TDLS peer is the TDLS channel switch initiator
    bool chsw_initiator;
    /// Dialog token
    uint8_t dialog_token;
    /// Traffic available for the TDLS STA
    bool traffic_available;
    /// Flag indicating if there is a TDLS Channel Switch request active for the STA
    bool chsw_active;
    /// Flag indicating if a keep-alive frame should be transmitted
    bool ka_tx_req;
    /// Number of keep-alive frame to be sent before informing the host
    uint8_t ka_retries;
    /// Flag indicating if a PTI Frame has already been sent to AP and PTI Timeout not
    /// yet expired
    bool pti_timer_running;
    /// TDLS Channel Switch channel
    struct mac_chan_op chsw;
    /// TDLS Channel Switch channel number
    uint8_t chsw_num;
    /// TDLS Channel Switch operating class
    uint8_t chsw_op_class;
    /// TDLS Channel Switch time
    uint16_t chsw_time;
    /// TDLS Channel Switch timeout
    uint16_t chsw_timeout;
    /// TDLS Peer Traffic Indication Response timeout timer
    struct mm_timer_tag pti_timeout_timer;
    /// TDLS peer keep-alive timer
    struct mm_timer_tag keep_alive_timer;
    /// TDLS Channel Switch time timer
    struct mm_timer_tag chsw_time_timer;
    /// TDLS Channel Switch timeout timer
    struct mm_timer_tag chsw_timeout_timer;
    /// TDLS Channel Switch request timer
    struct mm_timer_tag chsw_req_timer;
    /// TDLS Channel Switch end timer
    struct mm_timer_tag chsw_end_timer;
};
#endif

/// STA Info Table
struct sta_info_tag
{
    /// linked list header
    struct co_list_hdr list_hdr;
    /**
     * Beacon Interval used by peer AP or peer Mesh STA (in microseconds).
     * Used only for a VIF of type STA or MP
     */
    uint32_t bcn_int;
    /// maximum size of HE A-MPDUs supported by the STA
    uint32_t ampdu_size_max_he;
    /// maximum size of VHT A-MPDUs supported by the STA
    uint32_t ampdu_size_max_vht;
    /// maximum size of HT A-MPDUs supported by the STA
    uint16_t ampdu_size_max_ht;
    /// PartialAid/GroupId to be put in the THD when transmitting to this STA
    uint32_t paid_gid;
    /// Minimal spacing to be inserted when transmitting to this STA
    uint8_t ampdu_spacing_min;
    #if (NX_POWERSAVE)
    /**
     * Maximum drift on one beacon period
     * Used only for a VIF of type STA or MP
     */
    uint16_t drift;
    #endif //(NX_POWERSAVE)
    /// AID of the station
    uint16_t aid;
    /// Interface the station belongs to
    uint8_t inst_nbr;
    /// Station index
    uint8_t staid;
    /// Power save state of the associated STA
    uint8_t ps_state;
    /// Flag indicating if the STA entry is valid or not
    bool valid;
    /// MAC address of the STA
    struct mac_addr mac_addr;
    /// STA entry linked to this one (used for Multiple BSSID support in STA mode)
    struct sta_info_tag *linked_sta;
    #if (NX_UMAC_PRESENT)
    /// Logical port state
    uint8_t ctrl_port_state;
    /// PS/UAPSD traffic availability on host and/or internally
    enum sta_ps_traffic traffic_avail;
    /// PS/UAPSD service period status
    enum sta_ps_sp ps_service_period;
    /// control port protocol
    uint16_t ctrl_port_ethertype;
    /// STA Security Information
    struct sta_mgmt_sec_info sta_sec_info;
    /// Generic station information
    struct sta_capa_info info;
    /// Per-TID sequence numbers
    uint16_t seq_nbr[TID_MAX];
    /// TX policy table control
    struct sta_pol_tbl_cntl pol_tbl;
    #endif //(NX_UMAC_PRESENT)

    #if (NX_AMPDU_TX)
    /// For TX BA agreements per TID
    struct ba_agmt ba_agmts_tx[TID_MAX];
    #endif //(NX_AMPDU_TX && !NX_UMAC_PRESENT)

    #if (NX_REORD)
    /// For RX BA agreements per TID
    struct rxu_cntrl_reord *ba_agmts_rx[TID_MAX];
    #endif //(NX_REORD)

    #if (NX_UMAC_PRESENT)
    /// UMAC information related to the TX block ack agreements established
    struct sta_mgmt_ba_info ba_info[TID_MAX];
    /// Duplicate Detection Status (non-QoS data frames)
    uint16_t rx_nqos_last_seqcntl;
    /// Duplicate Detection Status (QoS data frames)
    uint16_t rx_qos_last_seqcntl[TID_MAX];
    #endif //(NX_UMAC_PRESENT)

    #if (NX_TX_FRAME)
    /// List of TX descriptor whose transmission has been postponed
    struct co_list tx_desc_post;
    #endif //(NX_TX_FRAME)

    #if (RW_MESH_EN)
    /// Mesh Link Index
    uint8_t mlink_idx;
    #endif //(RW_MESH_EN)

    #if (NX_TDLS)
    /// TDLS specific information
    struct tdls_info_tag tdls;
    /// Flag indicating if the station is a TDLS station
    bool is_tdls;
    #endif
    #if NX_UMAC_PRESENT && NX_MAC_HE
    /// Value of the BSR control that shall be put in the HE SU MPDUs
    uint32_t bsr;
    #endif
};

/// STA management module environment
struct sta_info_env_tag
{
    /// list of all the free stations entries
    struct co_list free_sta_list;

    #if (RW_MESH_EN && !NX_UMAC_PRESENT)
    /// Array allowing to connect a Mesh Link index with a STA index
    uint8_t mesh_link_idx[RW_MESH_LINK_NB];
    /// Number of Mesh STA currently added
    uint8_t nb_mesh_sta;
    #endif //(RW_MESH_EN && !NX_UMAC_PRESENT)
};

/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
/// STA management environment variable
extern struct sta_info_env_tag sta_info_env;

/// STA information table
extern struct sta_info_tag sta_info_tab[STA_MAX];

/**
 ****************************************************************************************
 * Test whether the specified capability is supported by the STA
 * @param[in] sta Pointer to the STA information structure
 * @param[in] type Capability type (QOS, MFP, HT, VHT or HE)
 * @return true if supported, false otherwise
 ****************************************************************************************
 */
#define STA_CAPA(sta, type) (((sta)->info.capa_flags & STA_##type##_CAPA) != 0)

/**
 ****************************************************************************************
 * Set the specified STA capability
 * @param[out] sta Pointer to the STA information structure
 * @param[in] type Capability type (QOS, HT, VHT or HE)
 ****************************************************************************************
 */
#define STA_CAPA_SET(sta, type) ((sta)->info.capa_flags |= STA_##type##_CAPA)

/**
 ****************************************************************************************
 * Clear the specified STA capability
 * @param[out] sta Pointer to the STA information structure
 * @param[in] type Capability type (QOS, HT, VHT or HE)
 ****************************************************************************************
 */
#define STA_CAPA_CLR(sta, type) ((sta)->info.capa_flags &= ~STA_##type##_CAPA)

/*
 * FUNCTION PROTOTYPES
 ****************************************************************************************
 */

#if (NX_UMAC_PRESENT)
/**
 ****************************************************************************************
 * @brief Retrieve the port state for the given STA index
 *
 * @param[in]  sta_idx    STA Index
 *
 * @return The control port state for the STA
 ****************************************************************************************
 */
__INLINE uint8_t sta_mgmt_get_port_state(uint8_t sta_idx)
{
    return (sta_info_tab[sta_idx].ctrl_port_state);
}

/**
 ****************************************************************************************
 * @brief Retrieve the control protocol used by the STA
 *
 * @param[in]  sta_idx    STA Index
 *
 * @return The ethertype used on the control port.
 ****************************************************************************************
 */
__INLINE uint16_t sta_mgmt_get_port_ethertype(uint8_t sta_idx)
{
    return (sta_info_tab[sta_idx].ctrl_port_ethertype);
}

#endif //(NX_UMAC_PRESENT)

/**
 ****************************************************************************************
 * @brief Initialize all the entries of the station table.
 ****************************************************************************************
 */
void sta_mgmt_init(void);

/**
 ****************************************************************************************
 * @brief Allocates new station entry, initializes it and fills it with information passed
 * as parameters.
 *
 * @param[in] param      MM_STA_ADD_REQ message parameters
 * @param[in] sta_idx    Pointer to index of the STA to be registered, which will be filled
 *
 * @return  The allocated station table index.
 ****************************************************************************************
 */
uint8_t sta_mgmt_register(struct mm_sta_add_req const *param,
                          uint8_t *sta_idx);

/**
 ****************************************************************************************
 * @brief Deletes the station table and free all the resources allocated for this station.
 *
 * @param[in]  sta_idx           Index of the station to be deleted.
 ****************************************************************************************
 */
void sta_mgmt_unregister(uint8_t sta_idx);

#if (NX_UMAC_PRESENT)
/**
 ****************************************************************************************
 * @brief Add a key for a STA
 *
 * @param[in]  param      Pointer to the key parameters.
 * @param[in]  hw_key_idx Index of the key in the HW key RAM
 ****************************************************************************************
 */
void sta_mgmt_add_key(struct mm_key_add_req const *param, uint8_t hw_key_idx);

/**
 ****************************************************************************************
 * @brief Delete a key for a STA
 *
 * @param[in]  sta  Pointer to the STA for which the key is deleted
 ****************************************************************************************
 */
void sta_mgmt_del_key(struct sta_info_tag *sta);

/**
 ****************************************************************************************
 * @brief Return if a TX Block Ack agreement exists for the provided [STAID, TID] tuple.
 *
 * @param[in] sta_idx   Station Index
 * @param[in] tid       TID
 *
 * @return BAM_INVALID_TASK_IDX if no agreement exists, else allocated BAM task instance
 ****************************************************************************************
 */
__INLINE uint8_t sta_mgmt_get_tx_bam_idx(uint8_t sta_idx, uint8_t tid)
{
    return (sta_info_tab[sta_idx].ba_info[tid].bam_idx_tx);
}

/**
 ****************************************************************************************
 * @brief Return if a RX Block Ack agreement exists for the provided [STAID, TID] tuple.
 *
 * @param[in] sta_idx   Station Index
 * @param[in] tid       TID
 *
 * @return BAM_INVALID_TASK_IDX if no agreement exists, else allocated BAM task instance
 ****************************************************************************************
 */
__INLINE uint8_t sta_mgmt_get_rx_bam_idx(uint8_t sta_idx, uint8_t tid)
{
    return (sta_info_tab[sta_idx].ba_info[tid].bam_idx_rx);
}

/**
 ****************************************************************************************
 * @brief Get the number of packets currently in the TX path for a given STA/TID pair.
 *
 * @param[in] sta_idx   Station Index
 * @param[in] tid       TID
 *
 * @return The number of packets currently in the TX path for the given STA/TID pair
 ****************************************************************************************
 */
__INLINE uint8_t sta_mgmt_ba_tx_cnt_get(uint8_t sta_idx, uint8_t tid)
{
    return sta_info_tab[sta_idx].ba_info[tid].tx_pkt_cnt;
}

/**
 ****************************************************************************************
 * @brief Increment the number of packets currently in the TX path for a given STA/TID
 * pair.
 *
 * @param[in] sta_idx   Station Index
 * @param[in] tid       TID
 ****************************************************************************************
 */
__INLINE void sta_mgmt_ba_tx_cnt_inc(uint8_t sta_idx, uint8_t tid)
{
    sta_info_tab[sta_idx].ba_info[tid].tx_pkt_cnt++;
}

/**
 ****************************************************************************************
 * @brief Decrement the number of packets currently in the TX path for a given STA/TID
 * pair.
 *
 * @param[in] sta_idx   Station Index
 * @param[in] tid       TID
 ****************************************************************************************
 */
__INLINE void sta_mgmt_ba_tx_cnt_dec(uint8_t sta_idx, uint8_t tid)
{
    sta_info_tab[sta_idx].ba_info[tid].tx_pkt_cnt--;
}

/**
 ****************************************************************************************
 * @brief Return the last time a BlockAck agreement was established for this STA/TID.
 *
 * @param[in] sta_idx   Station Index
 * @param[in] tid       TID
 *
 * @return The last BA time
 ****************************************************************************************
 */
__INLINE uint32_t sta_mgmt_get_add_ba_time(uint8_t sta_idx, uint8_t tid)
{
    return (sta_info_tab[sta_idx].ba_info[tid].last_ba_add_time);
}

/**
 ****************************************************************************************
 * @brief Set the last time a BlockAck agreement was established for this STA/TID.
 *
 * @param[in] sta_idx   Station Index
 * @param[in] tid       TID
 * @param[in] time      New BA Time
 ****************************************************************************************
 */
__INLINE void sta_mgmt_set_add_ba_time(uint8_t sta_idx, uint8_t tid, uint32_t time)
{
    sta_info_tab[sta_idx].ba_info[tid].last_ba_add_time = time;
}

/**
 ****************************************************************************************
 * @brief Return the next SN for this STA/TID pair and increase it for the next time
 *
 * @param[in]  sta_idx      STA Index
 * @param[in]  tid          TID
 *
 * @return The next Sequence Number to be used
 ****************************************************************************************
 */
__INLINE uint16_t sta_mgmt_get_tx_ssn_and_inc(uint8_t sta_idx, uint8_t tid)
{
    uint16_t sn = sta_info_tab[sta_idx].seq_nbr[tid]++;

    sta_info_tab[sta_idx].seq_nbr[tid] &= 0xFFF;

    return sn;
}

/**
 ****************************************************************************************
 * @brief Return the next SN to be used for this STA/TID pair
 *
 * @param[in]  sta_idx      STA Index
 * @param[in]  tid          TID
 *
 * @return The next Sequence Number to be used
 ****************************************************************************************
 */
__INLINE uint16_t sta_mgmt_get_tx_ssn(uint8_t sta_idx, uint8_t tid)
{
    return sta_info_tab[sta_idx].seq_nbr[tid];
}

/**
 ****************************************************************************************
 * @brief Return the PS state of a given station
 *
 * @param[in]  sta_idx      STA Index
 *
 * @return true if the station is in PS, false otherwise
 ****************************************************************************************
 */
__INLINE bool sta_mgmt_is_in_ps(uint8_t sta_idx)
{
    return (sta_info_tab[sta_idx].ps_state != 0);
}
#endif //(NX_UMAC_PRESENT)

/**
 ****************************************************************************************
 * @brief Return the validity state of a specific station index, i.e it indicates if the
 * given station is currently registered or not
 *
 * @param[in]  sta_idx      STA Index
 *
 * @return true if the station is registered, false otherwise
 ****************************************************************************************
 */
__INLINE bool sta_mgmt_is_valid(uint8_t sta_idx)
{
    return (sta_info_tab[sta_idx].valid);
}

#if (RW_BFMER_EN || NX_UMAC_PRESENT)
/**
 ****************************************************************************************
 * @brief Retrieve the peer STA MAC Address matching with the provided STA Index
 *
 * @param[in]  sta_idx      STA Index
 *
 * @return NULL if no MAC Address not found, else the associated MAC Address
 ****************************************************************************************
 */
__INLINE struct mac_addr *sta_mgmt_get_peer_addr(uint8_t sta_idx)
{
    return (&sta_info_tab[sta_idx].mac_addr);
}

/**
 ****************************************************************************************
 * @brief Return the VIF index whose the STA depends on
 *
 * @param[in]  sta_idx      STA Index
 *
 * @return The index of the VIF
 ****************************************************************************************
 */
__INLINE uint8_t sta_mgmt_get_vif_idx(uint8_t sta_idx)
{
    return (sta_info_tab[sta_idx].inst_nbr);
}
#endif //(RW_BFMER_EN || NX_UMAC_PRESENT)

#if (NX_AMPDU_TX || NX_UMAC_PRESENT)
/**
 ****************************************************************************************
 * @brief Set the Start Sequence Number of the BA agreement for this STA/TID pair
 *
 * @param[in]  sta_idx      STA Index
 * @param[in]  tid          TID
 * @param[in]  ssn          Start sequence number of the BA agreement
 ****************************************************************************************
 */
__INLINE void sta_mgmt_set_tx_ssn(uint8_t sta_idx, uint8_t tid, uint16_t ssn)
{
    #if (NX_UMAC_PRESENT)
    uint8_t bam_idx = sta_info_tab[sta_idx].ba_info[tid].bam_idx_tx;

    bam_env[bam_idx].ssn = ssn;
    #elif (NX_AMPDU_TX)
    sta_info_tab[sta_idx].ba_agmts_tx[tid].ssn = ssn;
    #endif //(NX_UMAC_PRESENT)
}

#if (NX_AMPDU_TX)
/**
 ****************************************************************************************
 * @brief Get the maximum number of MPDUs that can be put in an A-MPDU for this STA-TID pair
 *
 * @param[in]  sta_idx      STA Index
 * @param[in]  tid          TID
 *
 * @return The maximum number of MPDUs that can be put in an A-MPDU for this STA-TID pair
 ****************************************************************************************
 */
__INLINE uint8_t sta_mgmt_get_tx_buff_size(uint8_t sta_idx, uint8_t tid)
{
    return (sta_info_tab[sta_idx].ba_agmts_tx[tid].bufsz);
}

/**
 ****************************************************************************************
 * @brief Set the maximum number of MPDUs that can be put in an A-MPDU for this STA-TID pair
 *
 * @param[in]  sta_idx      STA Index
 * @param[in]  tid          TID
 * @param[in]  buff_size    Maximum number of MPDUs that can be put in an A-MPDU for this STA-TID pair
 ****************************************************************************************
 */
__INLINE void sta_mgmt_set_tx_buff_size(uint8_t sta_idx, uint8_t tid, uint8_t buff_size)
{
    sta_info_tab[sta_idx].ba_agmts_tx[tid].bufsz = buff_size;
}
#endif //(NX_AMPDU_TX)
#endif //(NX_AMPDU_TX || NX_UMAC_PRESENT)

#if (NX_TX_FRAME)
struct vif_info_tag;

/**
 ****************************************************************************************
 * @brief Go through list of postponed internal frames and try again to push these frames.
 * @param[in] vif           VIF Entry
 * @param[in] sta           STA Entry
 * @param[in] limit         Maximum number of frame to send. Unlimited if 0.
 *
 * @return the number of frames pushed.
 ****************************************************************************************
 */
int sta_mgmt_send_postponed_frame(struct vif_info_tag *vif, struct sta_info_tag *sta,
                                  int limit);
#endif //(NX_TX_FRAME)

/// @}

#endif // _STA_MGMT_H_
