/**
 ****************************************************************************************
 * @file td.c
 *
 * @brief Traffic Detection (TD) Module
 *
 * Copyright (C) RivieraWaves 2015-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup TD
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */

#include "td.h"

#if (NX_TD)

#include "ke_timer.h"

#if (NX_DPSM)
#include "ps.h"
#endif //(NX_DPSM)
#if (NX_CHNL_CTXT)
#include "chan.h"
#include "vif_mgmt.h"
#endif //(NX_CHNL_CTXT)
#if (NX_P2P_GO)
#include "p2p.h"
#endif //(NX_P2P_GO)
#if (NX_TD_STA)
#include "sta_mgmt.h"
#include "vif_mgmt.h"
#endif //(NX_TD_STA)
#if NX_MAC_HE
#include "txl_he.h"
#endif

/*
 * DEBUG (DEFINES, MACROS, ...)
 ****************************************************************************************
 */

/**
 * Debug Configuration for TD Module
 *      0 - Traces are disabled
 *      1.. - Level of verbosity
 */
#define TD_DEBUG_TRACES_EN    (0)

#if (TD_DEBUG_TRACES_EN)
/// Function used to print module information
#define TD_DEBUG_PRINT(lvl, format, ...)                            \
    do {                                                            \
        if (lvl <= TD_DEBUG_TRACES_EN)                              \
        {                                                           \
            dbg(format, ## __VA_ARGS__);                            \
        }                                                           \
    } while (0);
#else
#define TD_DEBUG_PRINT(lvl, format, ...)
#endif //(TD_DEBUG_TRACES_EN)

/*
 * MACROS
 ****************************************************************************************
 */

/// Return the VIF index of a Traffic Detection environment
#define TD_GET_VIF_INDEX(td_env)                                  \
    ((td_env - &td_env_tab[0]) / sizeof(struct td_env_tag))

/*
 * PRIVATE VARIABLES DECLARATION
 ****************************************************************************************
 */

struct td_env_tag td_env_tab[NX_VIRT_DEV_MAX];
#if (NX_TD_STA)
/// List of TD Environment Entries used for per-STA traffic detection
struct td_sta_env_tag td_sta_env[NX_REMOTE_STA_MAX];
#endif //(NX_TD_STA)

/*
 * PRIVATE FUNCTIONS DECLARATION
 ****************************************************************************************
 */

#if (NX_TD_STA)
/**
 ****************************************************************************************
 * @brief
 ****************************************************************************************
 */
static void td_update_sta_status(uint8_t sta_idx)
{
    // Get TD STA environment
    struct td_sta_env_tag *tds_env = &td_sta_env[sta_idx];
    // New status
    uint8_t new_status = 0;

    if (tds_env->pck_cnt_tx >= TD_DEFAULT_PCK_NB_THRES)
    {
        new_status |= CO_BIT(TD_STATUS_TX);
    }

    if (tds_env->pck_cnt_rx >= TD_DEFAULT_PCK_NB_THRES)
    {
        new_status |= CO_BIT(TD_STATUS_RX);
    }

    // Update the status
    tds_env->status = new_status;

    // Reset counters
    tds_env->pck_cnt_tx = 0;
    tds_env->pck_cnt_rx = 0;
}
#endif //(NX_TD_STA)

/**
 ****************************************************************************************
 * @brief
 ****************************************************************************************
 */
static void td_timer_end(void *env)
{
    // Get TD environment
    struct td_env_tag *td_env = (struct td_env_tag *)env;
    // Get current time
    uint32_t current_time = ke_time();
    // New status
    uint8_t new_status = 0;

    TD_DEBUG_PRINT(2, D_CRT "td_timer_end v=%d c=%d tx=%d rx=%d\n", td_env->vif_index,
            td_env->has_active_chan, td_env->pck_cnt_tx, td_env->pck_cnt_rx);

    // For profiling
    PROF_TD_TIMER_END_SET();

    #if NX_MAC_HE
    // Update the BSR of the STA linked to the VIF
    txl_he_bsr_compute(&vif_info_tab[td_env->vif_index]);
    #endif

    #if (NX_CHNL_CTXT)
    // Check if VIF's channel has been scheduled during last TD interval
    if (td_env->has_active_chan)
    #endif //(NX_CHNL_CTXT)
    {
        #if (NX_TD_STA)
        // Get VIF Entry
        struct vif_info_tag *vif = &vif_info_tab[td_env->vif_index];
        // STA Entry
        struct sta_info_tag *sta;
        #endif //(NX_TD_STA)

        // Check if traffic has been detected on TX path
        if (td_env->pck_cnt_tx >= TD_DEFAULT_PCK_NB_THRES)
        {
            new_status |= CO_BIT(TD_STATUS_TX);
        }

        // Check if traffic has been detected on RX path
        if (td_env->pck_cnt_rx >= TD_DEFAULT_PCK_NB_THRES)
        {
            new_status |= CO_BIT(TD_STATUS_RX);
        }

        #if (NX_DPSM)
        // Check if traffic has been detected on TX path after PS filtering
        if (td_env->pck_cnt_tx_ps >= TD_DEFAULT_PCK_NB_THRES)
        {
            new_status |= CO_BIT(TD_STATUS_TX_PS);
        }

        // Check if traffic has been detected on RX path after PS filtering
        if (td_env->pck_cnt_rx_ps >= TD_DEFAULT_PCK_NB_THRES)
        {
            new_status |= CO_BIT(TD_STATUS_RX_PS);
        }

        // If PS traffic status has been modified, notify PS module
        if ((td_env->status ^ new_status) & (CO_BIT(TD_STATUS_TX_PS) | CO_BIT(TD_STATUS_RX_PS)))
        {
            // Notify PS module
            ps_traffic_status_update(td_env->vif_index,
                                     (new_status & (CO_BIT(TD_STATUS_TX_PS) | CO_BIT(TD_STATUS_RX_PS))));
        }
        #endif //(NX_DPSM)

        #if (NX_P2P_GO)
        // If traffic status has been modified, notify P2P module
        if ((td_env->status ^ new_status) & (CO_BIT(TD_STATUS_TX) | CO_BIT(TD_STATUS_RX)))
        {
            // Notify P2P module
            p2p_go_td_evt(td_env->vif_index, new_status & (CO_BIT(TD_STATUS_TX) | CO_BIT(TD_STATUS_RX)));
        }
        #endif //(NX_P2P_GO)

        // Store status
        td_env->status = new_status;

        #if (NX_TD_STA)
        sta = (struct sta_info_tag *)co_list_pick(&vif->sta_list);

        while (sta)
        {
            // Update STA traffic status
            td_update_sta_status(sta->staid);

            // Get next STA entry
            sta = (struct sta_info_tag *)sta->list_hdr.next;
        }
        #endif //(NX_TD_STA)
    }

    // Reset packets counters
    td_env->pck_cnt_tx = 0;
    td_env->pck_cnt_rx = 0;
    #if (NX_DPSM)
    td_env->pck_cnt_tx_ps = 0;
    td_env->pck_cnt_rx_ps = 0;
    #endif //(NX_DPSM)

    #if (NX_CHNL_CTXT)
    if (chan_env.current_ctxt == vif_info_tab[td_env->vif_index].chan_ctxt)
    {
        td_env->has_active_chan = true;
    }
    else
    {
        td_env->has_active_chan = false;
    }
    #endif //(NX_CHNL_CTXT)

    // Reprogram timer
    mm_timer_set(&td_env->td_timer, current_time + TD_DEFAULT_INTV_US);

    // For profiling
    PROF_TD_TIMER_END_CLR();
}

/*
 * PUBLIC FUNCTIONS
 ****************************************************************************************
 */

void td_init(void)
{
    uint8_t counter;

    TD_DEBUG_PRINT(1, D_CRT "td_init\n");

    // Reset all TD environments
    for (counter = 0; counter < NX_VIRT_DEV_MAX; counter++)
    {
        td_reset(counter);
    }

    #if (NX_TD_STA)
    // Reset all TD environments used for per-STA Traffic Detection
    for (counter = 0; counter < NX_REMOTE_STA_MAX; counter++)
    {
        td_sta_reset(counter);
    }
    #endif //(NX_TD_STA)
}

void td_reset(uint8_t vif_index)
{
    // Get TD environment
    struct td_env_tag *td_env = &td_env_tab[vif_index];

    TD_DEBUG_PRINT(1, D_CRT "td_reset idx=%d\n", vif_index);

    if (td_env->is_on)
    {
        // Stop TD timer
        mm_timer_clear(&td_env->td_timer);
    }

    // Initialize memory
    memset(td_env, 0, sizeof(struct td_env_tag));

    // Set timer information
    td_env->td_timer.cb  = td_timer_end;
    td_env->td_timer.env = td_env;

    // Store VIF Index
    td_env->vif_index = vif_index;
}

#if (NX_TD_STA)
void td_sta_reset(uint8_t sta_index)
{
    // Get TD environment for provided STA
    struct td_sta_env_tag *tds_env = &td_sta_env[sta_index];

    TD_DEBUG_PRINT(1, D_CRT "td_sta_reset idx=%d\n", sta_index);

    // Initialize content of environment
    tds_env->pck_cnt_tx = 0;
    tds_env->pck_cnt_rx = 0;
    tds_env->status = 0;
}
#endif //(NX_TD_STA)

void td_start(uint8_t vif_index)
{
    // Get TD environment
    struct td_env_tag *td_env = &td_env_tab[vif_index];

    // Check if Traffic Detected is not already on
    if (!td_env->is_on)
    {
        // Get current time
        uint32_t current_time = ke_time();

        TD_DEBUG_PRINT(1, D_CRT "td_start idx=%d\n", vif_index);

        // Keep in mind that the TD timer is on
        td_env->is_on = true;

        // Program end of TD interval
        mm_timer_set(&td_env->td_timer, current_time + TD_DEFAULT_INTV_US);
    }
}

void td_pck_ind(uint8_t vif_index, uint8_t sta_index, bool rx)
{
    // Get TD environment
    struct td_env_tag *td_env = &td_env_tab[vif_index];
    #if (NX_TD_STA)
    // Get TD STA environment
    struct td_sta_env_tag *tds_env = (sta_index < NX_REMOTE_STA_MAX) ? &td_sta_env[sta_index] : NULL;
    #endif //(NX_TD_STA)

    TD_DEBUG_PRINT(3, D_CRT "td_pck_ind vif_idx=%d, sta_idx=%d, rx=%d\n", vif_index, sta_idx, rx);

    if (rx)
    {
        // For profiling
        PROF_TD_CHECK_RX_SET();

        // Increase number of RX packets
        td_env->pck_cnt_rx++;

        #if (NX_TD_STA)
        // Sanity check
        ASSERT_ERR(tds_env);

        if (tds_env->pck_cnt_rx < TD_DEFAULT_PCK_NB_THRES)
        {
            tds_env->pck_cnt_rx++;
        }
        #endif //(NX_TD_STA)

        // For profiling
        PROF_TD_CHECK_RX_CLR();
    }
    else
    {
        // For profiling
        PROF_TD_CHECK_TX_SET();

        // Increase number of TX packets
        td_env->pck_cnt_tx++;

        #if (NX_TD_STA)
        if (tds_env && (tds_env->pck_cnt_tx < TD_DEFAULT_PCK_NB_THRES))
        {
            tds_env->pck_cnt_tx++;
        }
        #endif //(NX_TD_STA)

        // For profiling
        PROF_TD_CHECK_TX_CLR();
    }
}

#if (NX_DPSM)
void td_pck_ps_ind(uint8_t vif_index, bool rx)
{
    TD_DEBUG_PRINT(3, D_CRT "td_pck_ps_ind idx=%d, rx=%d\n", vif_index, rx);

    if (rx)
    {
        // For profiling
        PROF_TD_CHECK_RX_PS_SET();

        // Increase number of RX packets
        td_env_tab[vif_index].pck_cnt_rx_ps++;

        // For profiling
        PROF_TD_CHECK_RX_PS_CLR();
    }
    else
    {
        // For profiling
        PROF_TD_CHECK_TX_PS_SET();

        // Increase number of RX packets
        td_env_tab[vif_index].pck_cnt_tx_ps++;

        // For profiling
        PROF_TD_CHECK_TX_PS_CLR();
    }
}
#endif //(NX_DPSM)

#endif //(NX_TD)

/// @} end of group
