/**
 ****************************************************************************************
 *
 * @file tdls_task.c
 *
 * @brief TDLS task.
 *
 * Copyright (C) RivieraWaves 2016-2019
 *
 ****************************************************************************************
 */

/**
 *****************************************************************************************
 * @addtogroup TDLS_TASK
 * @{
 *****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "co_endian.h"

#include "mac_frame.h"
#include "tdls.h"
#include "ps.h"
#if NX_UMAC_PRESENT
#include "me.h"
#include "rxu_cntrl.h"
#endif
#include "tdls_task.h"
#include "mm_bcn.h"
#include "phy.h"

#include "reg_mac_pl.h"
#include "reg_mac_core.h"
#include "hal_machw.h"
#include "dbg.h"

#include "rxl_cntrl.h"
#include "txl_cntrl.h"

#include "sta_mgmt.h"
#include "vif_mgmt.h"

#include "version.h"

#include "mm_timer.h"

#include "chan.h"
#if (RW_BFMER_EN)
#include "bfr.h"
#endif //(RW_BFMER_EN)

#include "tpc.h"
#include "mm.h"

#if (NX_TDLS)

/*
 * MESSAGE HANDLERS
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @brief TDLS channel switch request handler.
 * This function handles the @ref TDLS_CHAN_SWITCH_REQ message, which enables the channel
 * switch to the TDLS target channel.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static
int tdls_chan_switch_req_handler(ke_msg_id_t const msgid,
                                 struct tdls_chan_switch_req const *param,
                                 ke_task_id_t const dest_id,
                                 ke_task_id_t const src_id)
{
    uint8_t status = CO_FAIL;

    #if NX_UMAC_PRESENT
    do
    {
        struct vif_info_tag *vif;
        struct sta_info_tag *sta;

        if (param->vif_index == INVALID_VIF_IDX)
        {
            break;
        }
        vif = &vif_info_tab[param->vif_index];

        if ((param->sta_idx == INVALID_STA_IDX) ||
            (vif->u.sta.tdls_chsw_sta_idx != INVALID_STA_IDX))
        {
            break;
        }

        // Set TDLS STA index for channel switch request
        vif->u.sta.tdls_chsw_sta_idx = param->sta_idx;
        sta = &sta_info_tab[param->sta_idx];

        if (!sta->tdls.chsw_allowed)
        {
            TRACE_TDLS(ERR, "TDLS channel switch is not allowed.");
            break;
        }

        // Set the parameters
        if (sta->valid)
        {
            sta->tdls.chsw = param->chan;
            sta->tdls.chsw_op_class = param->op_class;
            sta->tdls.chsw_req_timer.cb = tdls_chsw_req_evt;
            sta->tdls.chsw_req_timer.env = vif;
            sta->tdls.chsw_active = true;
            sta->tdls.chsw_initiator = false;

            status = CO_OK;

            // Reset state
            ke_state_set(TASK_TDLS, TDLS_BASE_CHANNEL);

            PROF_TDLS_CHSW_REQ_TX_TIMER_SET();
            // Set next channel switch request
            mm_timer_set(&sta->tdls.chsw_req_timer,
                    vif->tbtt_timer.time + TDLS_CHSW_REQ_DELAY_US);
        }
    } while(0);
    #endif // NX_UMAC_PRESENT

    // Send back the confirmation
    struct tdls_chan_switch_cfm *cfm = KE_MSG_ALLOC(TDLS_CHAN_SWITCH_CFM,
            src_id, dest_id,
            tdls_chan_switch_cfm);
    cfm->status = status;

    ke_msg_send(cfm);

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief TDLS cancel channel switch request handler.
 * This function handles the @ref TDLS_CANCEL_CHAN_SWITCH_REQ message, which disables the
 * channel switch to the TDLS target channel.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static
int tdls_cancel_chan_switch_req_handler(ke_msg_id_t const msgid,
                                        struct tdls_cancel_chan_switch_req const *param,
                                        ke_task_id_t const dest_id,
                                        ke_task_id_t const src_id)
{
    uint8_t status = CO_FAIL;

    #if NX_UMAC_PRESENT
    do
    {
        struct vif_info_tag *vif = &vif_info_tab[param->vif_index];
        struct sta_info_tag *sta;

        if (param->sta_idx == INVALID_STA_IDX)
        {
            break;
        }

        if (param->sta_idx != vif->u.sta.tdls_chsw_sta_idx)
        {
            TRACE_TDLS(ERR, "TDLS Channel Switch is not active for STA %d.", param->sta_idx);
            break;
        }

        sta = &sta_info_tab[param->sta_idx];
        // Clear channel switch request timer
        PROF_TDLS_CHSW_REQ_TX_TIMER_CLR();
        mm_timer_clear(&sta->tdls.chsw_req_timer);
        sta->tdls.chsw_active = false;
        sta->tdls.chsw_initiator = false;
        if (ke_state_get(TASK_TDLS) == TDLS_BASE_CHANNEL)
        {
            // If already on base channel, reset TDLS ChSw station index
            vif->u.sta.tdls_chsw_sta_idx = INVALID_STA_IDX;
        }
        else
        {
            // If on off channel, switch back to base channel
            struct mm_remain_on_channel_req roc_param;

            PROF_TDLS_CHSW_TIME_TIMER_CLR();
            mm_timer_clear(&sta->tdls.chsw_time_timer);
            PROF_TDLS_CHSW_END_TIMER_CLR();
            mm_timer_clear(&sta->tdls.chsw_end_timer);
            roc_param.vif_index = vif->index;
            roc_param.op_code = MM_ROC_OP_CANCEL;
            PROF_TDLS_SWITCH_TO_BASECH_SET();
            chan_roc_req(&roc_param, TASK_TDLS);
            PROF_TDLS_SWITCH_TO_BASECH_CLR();

            ke_state_set(TASK_TDLS, TDLS_BASE_CHANNEL);
        }
        status = CO_OK;
    } while (0);

    #endif // NX_UMAC_PRESENT

    // Send back the confirmation
    struct tdls_cancel_chan_switch_cfm *cfm = KE_MSG_ALLOC(TDLS_CANCEL_CHAN_SWITCH_CFM,
            src_id, dest_id,
            tdls_cancel_chan_switch_cfm);
    cfm->status = status;

    ke_msg_send(cfm);

    return (KE_MSG_CONSUMED);
}

#if NX_UMAC_PRESENT
/**
 ****************************************************************************************
 * @brief TDLS Peer Traffic Indication request handler.
 * This function handles the @ref TDLS_PEER_TRAFFIC_IND_REQ message, which sends the TDLS Peer
 * Traffic Indication to the TDLS Peer, through the AP.
 *
 * @param[in] msgid Id of the message received.
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static
int tdls_peer_traffic_ind_req_handler(ke_msg_id_t const msgid,
                              struct tdls_peer_traffic_ind_req const *param,
                              ke_task_id_t const dest_id,
                              ke_task_id_t const src_id)
{
    uint8_t status = CO_FAIL;

    if (param->sta_idx == INVALID_STA_IDX)
    {
        return status;
    }

    // Send Peer Traffic Indication frame
    status = txl_frame_send_tdls_peer_traffic_ind_frame(param, &tdls_peer_traffic_ind_tx_cfm);

    if (status == CO_OK)
    {
        struct sta_info_tag *sta = &sta_info_tab[param->sta_idx];
        // set dialog token
        sta->tdls.dialog_token = param->dialog_token;
        // keep in mind that there is traffic bufferized at host level
        sta->tdls.traffic_available = true;
        // set TDLS STA status
        ke_state_set(TASK_TDLS, TDLS_TRAFFIC_IND_TX);
    }

    // Send back the confirmation
    struct tdls_peer_traffic_ind_cfm *cfm = KE_MSG_ALLOC(TDLS_PEER_TRAFFIC_IND_CFM,
                                                         src_id, dest_id,
                                                         tdls_peer_traffic_ind_cfm);
    cfm->status = status;

    ke_msg_send(cfm);

    return (KE_MSG_CONSUMED);

}
#endif // NX_UMAC_PRESENT

/*
 * TASK DESCRIPTOR DEFINITIONS
 ****************************************************************************************
 */
/// Message handlers in state DEFAULT.
const struct ke_msg_handler tdls_default_state[] =
{
    // From UMAC
    {TDLS_CHAN_SWITCH_REQ, (ke_msg_func_t)tdls_chan_switch_req_handler},
    // From UMAC
    {TDLS_CANCEL_CHAN_SWITCH_REQ, (ke_msg_func_t)tdls_cancel_chan_switch_req_handler},
#if (NX_UMAC_PRESENT)
    // From UMAC
    {TDLS_PEER_TRAFFIC_IND_REQ, (ke_msg_func_t)tdls_peer_traffic_ind_req_handler},
#endif
};

/// Specifies the message handlers that are common to all states.
const struct ke_state_handler tdls_default_handler =
    KE_STATE_HANDLER(tdls_default_state);

/// Defines the placeholder for the states of all the task instances.
ke_state_t tdls_state[TDLS_IDX_MAX];

#endif // NX_TDLS

/// @} end of group
