/**
 ******************************************************************************
 *
 * @file tpc.c
 *
 * @brief TPC (Transmit Power Control) module.
 *
 * Copyright (C) RivieraWaves 2016-2019
 *
 ******************************************************************************
 */

#include "tpc.h"

void tpc_update_tx_power(int8_t pwr)
{
    uint8_t idx = 0;

    phy_get_rf_gain_idx(&pwr, &idx);

    nxmac_dsss_max_pwr_level_setf(idx);
    nxmac_ofdm_max_pwr_level_setf(idx);
}

void tpc_get_vif_tx_power(struct vif_info_tag *vif, int8_t *pwr, uint8_t *idx)
{
    int8_t _pwr = vif->tx_power;
    uint8_t _idx;

    if ( _pwr == VIF_UNDEF_POWER)
        _idx = nxmac_ofdm_max_pwr_level_getf();
    else
        phy_get_rf_gain_idx(&_pwr, &_idx);

    if (pwr)
        *pwr = _pwr;
    if (idx)
        *idx = _idx;
}

void tpc_set_vif_tx_power(struct vif_info_tag *vif, int8_t req_pwr)
{
    int8_t prev_pwr = vif->tx_power;
    int8_t pwr = req_pwr;
    uint8_t idx;

    if (req_pwr == VIF_UNDEF_POWER)
        return;

    phy_get_rf_gain_idx(&pwr, &idx);
    vif->tx_power = pwr;

    if (prev_pwr != pwr)
    {
        #if NX_UMAC_PRESENT
        /* Mark all sta associated to the vif, in order to update policy table
           with new tx power on next transmit */
        struct co_list_hdr *sta_hdr;
        sta_hdr = co_list_pick(&vif->sta_list);
        while (sta_hdr != NULL)
        {
            struct sta_info_tag *sta;
            struct sta_pol_tbl_cntl *rc;

            sta = (struct sta_info_tag *)sta_hdr;
            rc = &sta->pol_tbl;
            rc->upd_field |= CO_BIT(STA_MGMT_POL_UPD_TX_POWER);
            sta_hdr = co_list_next(sta_hdr);
        }
        #endif

        #if NX_CHNL_CTXT
        if (vif->chan_ctxt)
            chan_update_tx_power(vif->chan_ctxt);
        #else
        /* Without channel context always update power */
        int i;

        for (i = 0; i < NX_VIRT_DEV_MAX; i++) {
            if (vif_info_tab[i].active && vif_info_tab[i].tx_power < pwr)
                pwr = vif_info_tab[i].tx_power;
        }

        tpc_update_tx_power(pwr);
        #endif // NX_CHNL_CTXT
    }
}

#if NX_TX_FRAME
void tpc_update_frame_tx_power(struct vif_info_tag *vif, struct txl_frame_desc_tag *frame)
{
    struct tx_policy_tbl *pol;
    uint8_t idx;

    tpc_get_vif_tx_power(vif, NULL, &idx);
    pol = HW2CPU(frame->txdesc.lmac.hw_desc->thd.policyentryaddr);
    pol->powercntrlinfo[0] = TX_PWR_LEVEL_SET(idx);
}
#endif // NX_TX_FRAME

#if NX_UMAC_PRESENT
void tpc_update_vif_tx_power(struct vif_info_tag *vif)
{
    int8_t pwr;

    pwr = vif->bss_info.chan.tx_power - vif->bss_info.power_constraint;
    if ((vif->user_tx_power != VIF_UNDEF_POWER) && (vif->user_tx_power < pwr))
        pwr = vif->user_tx_power;

    // Force reset
    vif->tx_power = VIF_UNDEF_POWER;

    tpc_set_vif_tx_power(vif, pwr);
    TRACE_LMAC(TPC, "{VIF-%d} set TX power to %ddBm (regulatory=%ddBm, local_constraint=%ddBm, user=%ddBm)",
               vif->index, vif->tx_power, vif->bss_info.chan.tx_power, vif->bss_info.power_constraint,
               vif->user_tx_power);
}
#endif // NX_UMAC_PRESENT
