/**
 ****************************************************************************************
 *
 * @file txl_agg.c
 *
 * @brief TX aggregation function implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup TX_AGG
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "co_int.h"
#include "co_bool.h"
#include <string.h>

#include "dbg_assert.h"
#include "mac.h"
#include "mac_frame.h"
#include "mm.h"
#include "macif.h"
#include "rxl_hwdesc.h"
#include "txl_buffer.h"
#if NX_TX_FRAME
#include "txl_frame.h"
#endif
#include "tx_swdesc.h"
#include "txl_cfm.h"
#include "txl_cntrl.h"
#include "rxl_cntrl.h"
#include "reg_mac_pl.h"
#include "reg_mac_core.h"
#include "dbg.h"
#include "ps.h"
#if (NX_P2P)
#include "p2p.h"
#endif //(NX_P2P)
#if (NX_TD)
#include "td.h"
#endif //(NX_TD)
#if NX_UMAC_PRESENT
#include "txu_cntrl.h"
#include "apm.h"
#endif
#if NX_MFP
#include "mfp.h"
#endif
#if (RW_BFMER_EN)
#include "bfr.h"
#endif //(RW_BFMER_EN)
#if NX_MAC_HE
#include "txl_he.h"
#endif
#include "txl_agg.h"

#if NX_AMPDU_TX

/*
 * DEFINES
 ****************************************************************************************
 */
#if RW_MUMIMO_TX_EN
/// Bits indicating activity on secondary users (merged trigger and buffer)
#define TX_SEC_IRQ_BITS_MERGED  ( NXMAC_SEC_U_3AC_3_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_3AC_2_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_3AC_1_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_3AC_0_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_2AC_3_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_2AC_2_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_2AC_1_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_2AC_0_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_1AC_3_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_1AC_2_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_1AC_1_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_1AC_0_TX_TRIGGER_BIT )

/// Operations performed upon the closure of a MU-MIMO A-MPDU during A-MPDU formation
#define MU_AMPDU_CLOSE()                                                                 \
{                                                                                        \
    txl_mumimo_ampdu_finish(ac, user_pos);                                               \
    /* Check if the other user positions are still open */                               \
    if (txlist->mumimo.open & txlist->mumimo.users)                                      \
    {                                                                                    \
        /* We could still receive packets on other user positions, */                    \
        /* so save the current packet */                                                 \
        ASSERT_ERR(txlist->mumimo.txdesc[user_pos] == NULL);                             \
        txlist->mumimo.txdesc[user_pos] = txdesc;                                        \
        status = MU_PAUSED;                                                              \
        break;                                                                           \
    }                                                                                    \
    else                                                                                 \
    {                                                                                    \
        /* Close the MU-MIMO PPDU under construction */                                  \
        txl_agg_mumimo_close(ac);                                                        \
                                                                                         \
        /* Start a new MU-MIMO PPDU */                                                   \
        continue;                                                                        \
    }                                                                                    \
}
#endif

/// Table linking HT or VHT MCS+GI+BW to minimum number of bytes for 1us min MPDU start spacing in AMPDU
/// Index of the table is (MCS_IDX << 3) | (BW << 1) | (GI_400)
///  where BW is 0 for 20 MHz, 1 for 40MHz and 2 for 80MHz
///        GI_400 is 1 if packet is being sent with 400ns GI, 0 if 800ns GI
const uint8_t TX_VHT_RATE_TO_MIN_SEP[10*4*2] =
{
    //BW20,GI800  BW20,GI400    BW40,GI800    BW40,GI400   MCS Index
    [  0] =  1,   [  1] =  1,   [  2] =  2,   [  3] =  2, //MCS 0
    [  8] =  2,   [  9] =  2,   [ 10] =  4,   [ 11] =  4, //MCS 1
    [ 16] =  3,   [ 17] =  3,   [ 18] =  6,   [ 19] =  6, //MCS 2
    [ 24] =  4,   [ 25] =  4,   [ 26] =  7,   [ 27] =  8, //MCS 3
    [ 32] =  5,   [ 33] =  6,   [ 34] = 11,   [ 35] = 12, //MCS 4
    [ 40] =  7,   [ 41] =  8,   [ 42] = 14,   [ 43] = 15, //MCS 5
    [ 48] =  8,   [ 49] =  9,   [ 50] = 16,   [ 51] = 17, //MCS 6
    [ 56] =  9,   [ 57] = 10,   [ 58] = 17,   [ 59] = 19, //MCS 7
    [ 64] = 10,   [ 65] = 11,   [ 66] = 21,   [ 67] = 23, //MCS 8
    [ 72] = 11,   [ 73] = 13,   [ 74] = 23,   [ 75] = 25, //MCS 9
    //BW80,GI800  BW80,GI400    BW160,GI800   BW160,GI400
    [  4] =  4,   [  5] =  5,   [  6] =  8,   [  7] =  9, //MCS 0
    [ 12] =  8,   [ 13] =  9,   [ 14] = 15,   [ 15] = 17, //MCS 1
    [ 20] = 11,   [ 21] = 13,   [ 22] = 22,   [ 23] = 25, //MCS 2
    [ 28] = 15,   [ 29] = 17,   [ 30] = 30,   [ 31] = 33, //MCS 3
    [ 36] = 22,   [ 37] = 25,   [ 38] = 44,   [ 39] = 49, //MCS 4
    [ 44] = 30,   [ 45] = 33,   [ 46] = 59,   [ 47] = 65, //MCS 5
    [ 52] = 33,   [ 53] = 37,   [ 54] = 66,   [ 55] = 74, //MCS 6
    [ 60] = 37,   [ 61] = 41,   [ 62] = 74,   [ 63] = 82, //MCS 7
    [ 68] = 44,   [ 69] = 49,   [ 70] = 88,   [ 71] = 98, //MCS 8
    [ 76] = 49,   [ 77] = 55,   [ 78] = 98,   [ 79] = 109, //MCS 9
};


/// Table linking HT or VHT MCS+GI+BW to the number of bytes transmitted in 32us
/// Index of the table is (MCS_IDX << 3) | (BW << 1) | (GI_400)
///  where BW is 0 for 20 MHz, 1 for 40MHz and 2 for 80MHz
///        GI_400 is 1 if packet is being sent with 400ns GI, 0 if 800ns GI
const uint16_t TX_VHT_RATE_TO_32US_LEN[10*4*2] =
{
    //BW20,GI800    BW20,GI400      BW40,GI800      BW40,GI400   MCS Index
    [  0] =   26,   [  1] =   28,   [  2] =   54,   [  3] =   60, //MCS 0
    [  8] =   52,   [  9] =   57,   [ 10] =  108,   [ 11] =  120, //MCS 1
    [ 16] =   78,   [ 17] =   86,   [ 18] =  162,   [ 19] =  180, //MCS 2
    [ 24] =  104,   [ 25] =  115,   [ 26] =  216,   [ 27] =  240, //MCS 3
    [ 32] =  156,   [ 33] =  173,   [ 34] =  324,   [ 35] =  360, //MCS 4
    [ 40] =  208,   [ 41] =  231,   [ 42] =  432,   [ 43] =  480, //MCS 5
    [ 48] =  234,   [ 49] =  260,   [ 50] =  486,   [ 51] =  540, //MCS 6
    [ 56] =  260,   [ 57] =  288,   [ 58] =  540,   [ 59] =  600, //MCS 7
    [ 64] =  312,   [ 65] =  346,   [ 66] =  648,   [ 67] =  720, //MCS 8
    [ 72] =  346,   [ 73] =  385,   [ 74] =  720,   [ 75] =  800, //MCS 9
    //BW80,GI800    BW80,GI400      BW160,GI800     BW160,GI400
    [  4] =  117,   [  5] =  130,   [  6] =  234,   [  7] =  260, //MCS 0
    [ 12] =  234,   [ 13] =  260,   [ 14] =  468,   [ 15] =  520, //MCS 1
    [ 20] =  351,   [ 21] =  390,   [ 22] =  702,   [ 23] =  780, //MCS 2
    [ 28] =  468,   [ 29] =  520,   [ 30] =  936,   [ 31] = 1040, //MCS 3
    [ 36] =  702,   [ 37] =  780,   [ 38] = 1404,   [ 39] = 1560, //MCS 4
    [ 44] =  936,   [ 45] = 1040,   [ 46] = 1872,   [ 47] = 2080, //MCS 5
    [ 52] = 1053,   [ 53] = 1170,   [ 54] = 2106,   [ 55] = 2340, //MCS 6
    [ 60] = 1170,   [ 61] = 1300,   [ 62] = 2340,   [ 63] = 2600, //MCS 7
    [ 68] = 1404,   [ 69] = 1560,   [ 70] = 2808,   [ 71] = 3120, //MCS 8
    [ 76] = 1560,   [ 77] = 1733,   [ 78] = 3120,   [ 79] = 3466, //MCS 9
};

#if RW_MUMIMO_TX_EN
/// Number of BW configurations in VHT
#define VHT_BW   4
/// Number of MCS in VHT
#define VHT_MCS  10

/// IEEE P802.11ac D3.0  Chptr 22.5 Parameters for VHT MCSs
/// Note that some BW, MCS, NSS combinations are not allowed (e.g 20MHz, MCS9, NSS 1,2)
/// The NDBPS value given for MCS9 20MHz is for NSS=3
const uint16_t VHT_NDBPS[VHT_BW][VHT_MCS] = {
    // MCS Index:  0     1      2       3      4      5      6      7      8      9
   [BW_20MHZ]  = {26,   52,    78,     104,   156,   208,   234,   260,   312,   1040},
   [BW_40MHZ]  = {54,   108,   162,    216,   324,   432,   486,   540,   648,   720 },
   [BW_80MHZ]  = {117,  234,   351,    468,   702,   936,   1053,  1170,  1404,  1560},
   [BW_160MHZ] = {234,  468,   702,    936,   1404,  1872,  2106,  2340,  2808,  3120},
};
#endif

/// Minimum of A-MPDU descriptors per queue
#define TX_MIN_AMPDU_NB_PER_AC  (3 * RW_USER_MAX)
/// Number of TX descriptors for 1 AGG descriptor
#define TX_AGG_DIVIDER          (8 / RW_USER_MAX)
/// Max number of AMPDU descriptors for the BK queue
#if (NX_TXDESC_CNT0 / TX_AGG_DIVIDER) >= TX_MIN_AMPDU_NB_PER_AC
#define TX_MAX_AMPDU_NB_PER_AC0  (NX_TXDESC_CNT0 / TX_AGG_DIVIDER)
#else
#define TX_MAX_AMPDU_NB_PER_AC0  TX_MIN_AMPDU_NB_PER_AC
#endif
/// Max number of AMPDU descriptors for the BE queue
#if (NX_TXDESC_CNT1 / TX_AGG_DIVIDER) >= TX_MIN_AMPDU_NB_PER_AC
#define TX_MAX_AMPDU_NB_PER_AC1  (NX_TXDESC_CNT1 / TX_AGG_DIVIDER)
#else
#define TX_MAX_AMPDU_NB_PER_AC1  TX_MIN_AMPDU_NB_PER_AC
#endif
/// Max number of AMPDU descriptors for the VI queue
#if (NX_TXDESC_CNT2 / TX_AGG_DIVIDER) >= TX_MIN_AMPDU_NB_PER_AC
#define TX_MAX_AMPDU_NB_PER_AC2  (NX_TXDESC_CNT2 / TX_AGG_DIVIDER)
#else
#define TX_MAX_AMPDU_NB_PER_AC2  TX_MIN_AMPDU_NB_PER_AC
#endif
/// Max number of AMPDU descriptors for the VO queue
#if (NX_TXDESC_CNT3 / TX_AGG_DIVIDER) >= TX_MIN_AMPDU_NB_PER_AC
#define TX_MAX_AMPDU_NB_PER_AC3  (NX_TXDESC_CNT3 / TX_AGG_DIVIDER)
#else
#define TX_MAX_AMPDU_NB_PER_AC3  TX_MIN_AMPDU_NB_PER_AC
#endif
#if NX_BEACONING
/// Max number of AMPDU descriptors for the BCN queue
#if (NX_TXDESC_CNT4 / TX_AGG_DIVIDER) > TX_MIN_AMPDU_NB_PER_AC
#define TX_MAX_AMPDU_NB_PER_AC4  (NX_TXDESC_CNT4 / TX_AGG_DIVIDER)
#else
#define TX_MAX_AMPDU_NB_PER_AC4  TX_MIN_AMPDU_NB_PER_AC
#endif
#endif

/// Number of A-MPDU descriptor queues
#define TX_AMPDU_DESC_QUEUE_CNT (NX_TXQ_CNT + NX_MAC_HE)

#if NX_MAC_HE
/// Number of A-MPDU descriptors for HE TB queue
#define TX_MAX_AMPDU_NB_FOR_HE_TB 4
/// Index of the HE TB A-MPDU descriptor queue
#define TX_HE_TB_AMPDU_QUEUE_IDX (TX_AMPDU_DESC_QUEUE_CNT - 1)
/// Approximate time needed to finish a A-MPDU (in us)
#define TX_AGG_FINISH_DUR 5
#endif

/*
 * GLOBAL VARIABLE DEFINITIONS
 ****************************************************************************************
 */
/// Array of aggregation descriptors for the BK queue
static struct tx_agg_desc tx_agg_desc_array0[TX_MAX_AMPDU_NB_PER_AC0] __SHAREDRAM;
/// Array of aggregation descriptors for the BE queue
static struct tx_agg_desc tx_agg_desc_array1[TX_MAX_AMPDU_NB_PER_AC1] __SHAREDRAM;
/// Array of aggregation descriptors for the VI queue
static struct tx_agg_desc tx_agg_desc_array2[TX_MAX_AMPDU_NB_PER_AC2] __SHAREDRAM;
/// Array of aggregation descriptors for the VO queue
static struct tx_agg_desc tx_agg_desc_array3[TX_MAX_AMPDU_NB_PER_AC3] __SHAREDRAM;
#if NX_BEACONING
/// Array of aggregation descriptors for the BCN queue
static struct tx_agg_desc tx_agg_desc_array4[TX_MAX_AMPDU_NB_PER_AC4] __SHAREDRAM;
#endif
#if NX_MAC_HE
/// Array of aggregation descriptors for the BCN queue
static struct tx_agg_desc tx_agg_desc_array5[TX_MAX_AMPDU_NB_FOR_HE_TB] __SHAREDRAM;
#endif

/// Table of pointers to the different aggregation descriptor arrays
static const struct tx_agg_desc * const tx_agg_desc_array[TX_AMPDU_DESC_QUEUE_CNT] =
{
    tx_agg_desc_array0,
    tx_agg_desc_array1,
    tx_agg_desc_array2,
    tx_agg_desc_array3,
    #if NX_BEACONING
    tx_agg_desc_array4,
    #endif
    #if NX_MAC_HE
    tx_agg_desc_array5,
    #endif
};

/// Number of aggregation descriptor per queue
static const int tx_agg_desc_cnt[TX_AMPDU_DESC_QUEUE_CNT] =
{
    TX_MAX_AMPDU_NB_PER_AC0,
    TX_MAX_AMPDU_NB_PER_AC1,
    TX_MAX_AMPDU_NB_PER_AC2,
    TX_MAX_AMPDU_NB_PER_AC3,
    #if NX_BEACONING
    TX_MAX_AMPDU_NB_PER_AC4,
    #endif
    #if NX_MAC_HE
    TX_MAX_AMPDU_NB_FOR_HE_TB,
    #endif
};

/// Per-ac pools of free aggregation descriptors
struct co_list tx_agg_desc_pool[TX_AMPDU_DESC_QUEUE_CNT];

/*
 * FUNCTION BODIES
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @brief Check if a txdesc can be aggregated in an AMPDU
 * @param[in] txdesc    Pending txdesc
 * @param[in] agg_desc  Already established params of ongoing aggregate
 *
 * @return true if MPDU can be aggregated, false otherwise
 ****************************************************************************************
 */
static bool txl_desc_is_agg(struct txdesc *txdesc, struct tx_agg_desc * agg_desc)
{
    if (is_mpdu_agg(txdesc) && !is_mpdu_first(txdesc))
    {
        //check if its (staid, tid) matches that of the started AMPDU
        if ( (agg_desc->sta_idx == txdesc->host.staid)  &&
             (agg_desc->tid     == txdesc->host.tid) )
        {
            return true;
        }
    }
    return false;
}

#if NX_UMAC_PRESENT
/**
 ****************************************************************************************
 * @brief Check if the phy_flags in TX descriptor indicate a HT or VHT rate, that would
 * allow aggregation. In case of a legacy rate this function returns false.
 *
 * @param[in] txdesc   Pointer to txdesc whose phy_flags is checked
 *
 * @return true if the rate is ht or vht, false otherwise
 ****************************************************************************************
 */
static bool txl_ampdu_has_ht_vht_rate(struct txdesc *txdesc)
{
    return (((txdesc->umac.phy_flags & FORMAT_MOD_TX_RCX_MASK)
                                   >> FORMAT_MOD_TX_RCX_OFT) > FORMATMOD_NON_HT_DUP_OFDM);
}
#endif

/**
 ****************************************************************************************
 * @brief Get the 32us length from the HT/VHT table for a given base index, bw pair
 *
 * @param[in] base_idx Base index in the table
 * @param[in] bw       Bandwidth of transmission
 *
 * @return The 32us byte length
 ****************************************************************************************
 */
static uint32_t txl_vht_idx_to_32us_len_get(int base_idx, uint8_t bw)
{
    int idx = base_idx | (bw << 1);
    return ((uint32_t)TX_VHT_RATE_TO_32US_LEN[idx]);
}

/**
 ****************************************************************************************
 * @brief Get the 1us length from the HT/VHT table for a given base index, bw pair
 *
 * @param[in] base_idx Base index in the table
 * @param[in] bw       Bandwidth of transmission
 *
 * @return The 1us byte length
 ****************************************************************************************
 */
static uint16_t txl_vht_idx_to_1us_len_get(int base_idx, uint8_t bw)
{
    int idx = base_idx | (bw << 1);
    return ((uint32_t)TX_VHT_RATE_TO_MIN_SEP[idx]);
}

/**
 ****************************************************************************************
 * @brief Compute the base rate index that should be used to compute the Maximum PHY
 * length of the HT/VHT A-MPDU and the minimum A-MPDU subframe length.
 *
 * @param[in]  txdesc      Pointer to txdesc whose length number of blank delimiters is calculated
 * @param[in]  format_mod  Modulation format (HT or VHT)
 * @param[out] max_len_sta Maximum A-MPDU length that the STA could receive
 * @param[out] nss         Number of spatial streams for the transmission
 *
 * @return The base rate index that should be used to get the maximum A-MPDU length that
 * could be transmitted on the requested PHY rate
 ****************************************************************************************
 */
static int txl_ht_vht_ampdu_param_get(struct txdesc *txdesc,
                                      uint8_t format_mod,
                                      uint32_t *max_len_sta,
                                      uint32_t *nss)
{
    uint32_t phy_flags = txdesc->umac.phy_flags;
    uint32_t mcs_idx = (phy_flags & MCS_INDEX_TX_RCX_MASK) >> MCS_INDEX_TX_RCX_OFT;
    uint32_t sgi = (phy_flags & SHORT_GI_TX_RCX_MASK) >> SHORT_GI_TX_RCX_OFT;
    struct sta_info_tag *sta = &sta_info_tab[txdesc->host.staid];

    // Check if modulation is HT or VHT
    if (format_mod == FORMATMOD_VHT)
    {
        // Extract NSS
        *nss = ((mcs_idx & VHT_NSS_MASK) >> VHT_NSS_OFT) + 1;

        // Extract MCS index
        mcs_idx = (mcs_idx & VHT_MCS_MASK) >> VHT_MCS_OFT;

        // Sanity checks: Maximum 4 SS supported, and MCS shall be between 0 and 9
        ASSERT_ERR((*nss > 0) && (*nss <= 4));
        ASSERT_ERR(mcs_idx <= 9);

        // Get the maximum VHT A-MPDU size supported by the receiver
        *max_len_sta = sta->ampdu_size_max_vht;
    }
    else
    {
        // Sanity check: The current implementation only supports MCS0-31
        ASSERT_ERR(mcs_idx <= 31);

        // Extract NSS
        *nss = ((mcs_idx & HT_NSS_MASK) >> HT_NSS_OFT) + 1;

        // Extract MCS index in 0-7 range
        mcs_idx = (mcs_idx & HT_MCS_MASK) >> HT_MCS_OFT;

        // Get the maximum HT A-MPDU size supported by the receiver
        *max_len_sta = sta->ampdu_size_max_ht;
    }

    // To compute the maximum length we take the minimal bandwidth, to ensure that the
    // packet won't cross the L-SIG duration limit in case the MAC HW would have to
    // reduce the transmission bandwidth
    return ((mcs_idx << 3) | sgi);
}

/**
 ****************************************************************************************
 * @brief Calculate number of blank delimiters to ensure MMSS after a txdesc
 * This number will be set in the THD of the NEXT txdesc in AMPDU
 * @param[in] txdesc   Pointer to txdesc whose length number of blank delimiters is calculated
 * @param[in] ac       Access Category
 * @param[in] agg      Pointer to the A-MPDU descriptor under builing process
 ****************************************************************************************
 */
static void txl_ampdu_constraints_get(struct txdesc *txdesc,
                                      uint8_t ac,
                                      struct txl_agg_build_tag *agg)
{
    int rate_idx;
    uint32_t max_len_sta;
    uint32_t max_len_phy;
    uint32_t nss;
    uint32_t bw = (txdesc->umac.phy_flags & BW_TX_RCX_MASK) >> BW_TX_RCX_OFT;
    struct sta_info_tag *sta = &sta_info_tab[txdesc->host.staid];
    #if NX_BW_LEN_ADAPT
    int i;
    #endif
    uint8_t format_mod;
    uint32_t (*idx_to_32us_len)(int, uint8_t);
    uint16_t (*idx_to_1us_len)(int, uint8_t);

    // Get modulation format
    format_mod = (txdesc->umac.phy_flags & FORMAT_MOD_TX_RCX_MASK) >> FORMAT_MOD_TX_RCX_OFT;

    // Sanity check: A-MPDUs cannot be sent at legacy rates
    ASSERT_ERR(format_mod > FORMATMOD_NON_HT_DUP_OFDM);

    #if NX_MAC_HE
    if (format_mod == FORMATMOD_HE_SU)
    {
        // Get A-MPDU parameters for HE A-MPDU
        rate_idx = txl_he_ampdu_param_get(txdesc, format_mod, &max_len_sta, &nss);
        idx_to_32us_len = txl_he_idx_to_32us_len_get;
        idx_to_1us_len = txl_he_idx_to_1us_len_get;
    }
    else
    #endif
    {
        // Get A-MPDU parameters for HT or VHT A-MPDU
        rate_idx = txl_ht_vht_ampdu_param_get(txdesc, format_mod, &max_len_sta, &nss);
        idx_to_32us_len = txl_vht_idx_to_32us_len_get;
        idx_to_1us_len = txl_vht_idx_to_1us_len_get;
    }

    // Compute the rate indexes
    agg->bw = bw;

    #if NX_BW_LEN_ADAPT
    agg->bw_idx = 0;

    // Compute the maximum length for each BW
    for (i = 0; i <= agg->bw; i++)
    {
        // Compute the maximum length of the A-MPDU depending on the PHY rate
        max_len_phy = idx_to_32us_len(rate_idx, i) * nss * mm_env.ampdu_max_dur[ac];

        // Get the constraints of the A-MPDU based on the PHY rate that will be used
        agg->max_len[i] = co_min(max_len_phy, max_len_sta);
    }

    #else
    // Compute the maximum length of the A-MPDU depending on the PHY rate
    max_len_phy = idx_to_32us_len(rate_idx, 0) * nss * mm_env.ampdu_max_dur[ac];

    // Get the constraints of the A-MPDU based on the PHY rate that will be used
    agg->max_len[0] = co_min(max_len_phy, max_len_sta);
    #endif

    // To compute the minimal A-MPDU subframe length we take the maximal bandwidth
    agg->mmss_bytes = idx_to_1us_len(rate_idx, bw) * nss *
                                                   (uint16_t)sta->ampdu_spacing_min;

    // Get the maximum number of MPDUs inside the A-MPDU from BA agreement
    agg->max_cnt = sta_mgmt_get_tx_buff_size(txdesc->host.staid, txdesc->host.tid);
}

/**
 ****************************************************************************************
 * @brief Fill BAR frame (THD and payload)
 * @param[in] agg_desc  Aggregate descriptor in which BAR THD and payload is found
 * @param[in] sn        Sequence number to be put in the BAR frame
 * @param[in] staid     Station ID for this AMPDU
 * @param[in] tid       TID for this AMPDU
 * @param[in] bw        Transmission bandwidth of this AMPDU
 ****************************************************************************************
 */
static void txl_fill_bar(struct tx_agg_desc *agg_desc, uint16_t sn,
                         uint8_t staid, uint8_t tid, uint8_t bw)
{
    struct tx_hd *bar_thd = &agg_desc->bar_thd;
    struct tx_policy_tbl *bar_pol = &agg_desc->bar_pol_tbl;
    struct bar_frame *bar_payl = &agg_desc->bar_payl;
    struct sta_info_tag *sta = &sta_info_tab[staid];
    struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];

    //------------------------------------------------------------------------------------
    //FILL PAYLOAD
    //------------------------------------------------------------------------------------
    //RA
    bar_payl->h.addr1 = sta->mac_addr;
    //TA - our own
    bar_payl->h.addr2 = vif->mac_addr;

    //bar_cntl - Normal ACK, Non-Multi tID, compressed BAR
    bar_payl->bar_cntrl = BAR_CNTL_COMPRESSED_BIT|(tid << BAR_CNTL_TID_OFT);
    //BAR Info - SSC
    //smallest in A-MPDU - that of 1st MPDU in AMPDU while there are no retries
    bar_payl->bar_information = sn << 4;

    //------------------------------------------------------------------------------------
    //Format THD
    //------------------------------------------------------------------------------------
    // Reset chaining between THD
    bar_thd->nextfrmexseq_ptr = 0;

    // Set status as 0
    bar_thd->statinfo = 0;

    //------------------------------------------------------------------------------------
    //Format policy table
    //------------------------------------------------------------------------------------
    // Set TX power
    bar_pol->powercntrlinfo[0] = TX_PWR_LEVEL_SET(nxmac_ofdm_max_pwr_level_getf());
    // Keep same bandwidth for BAR
    bar_pol->ratecntrlinfo[0] = (bw << BW_TX_RCX_OFT) |
                                (FORMATMOD_NON_HT_DUP_OFDM << FORMAT_MOD_TX_RCX_OFT) |
                                (HW_RATE_24MBPS << MCS_INDEX_TX_RCX_OFT);
}

/**
 ****************************************************************************************
 * @brief Initialize Hardware TX descriptors for A-MPDU
 ****************************************************************************************
 */
static void txl_agg_hwdesc_init(void)
{
    // Reset the TX Aggregation descriptor array elements
    memset(tx_agg_desc_pool, 0, sizeof(tx_agg_desc_pool));

    // Initialize the descriptor pools
    for (int i = 0; i < TX_AMPDU_DESC_QUEUE_CNT; i++)
    {
        struct co_list *list = &tx_agg_desc_pool[i];

        memset((void *)tx_agg_desc_array[i], 0, tx_agg_desc_cnt[i] * sizeof(struct tx_agg_desc));
        for (int j = 0; j < tx_agg_desc_cnt[i]; j++)
        {
            struct tx_agg_desc *agg_desc = (struct tx_agg_desc *)&tx_agg_desc_array[i][j];
            struct tx_hd *a_thd = &agg_desc->a_thd;
            struct tx_hd *bar_thd = &agg_desc->bar_thd;
            struct bar_frame *bar_payl = &agg_desc->bar_payl;
            struct tx_policy_tbl *bar_pol = &agg_desc->bar_pol_tbl;

            // Remember the free list in which the A-MPDU descriptor is pushed
            agg_desc->free_list = list;

            // Fill payload
            bar_payl->h.fctl = (MAC_FCTRL_CTRL_T | MAC_FCTRL_BAR_ST);
            bar_payl->h.durid = 0x0500;

            // Initialize default A-MPDU descriptor
            a_thd->upatterntx = TX_HEADER_DESC_PATTERN;
            a_thd->frmlifetime = 0;
            a_thd->macctrlinfo2 = WHICHDESC_AMPDU_EXTRA | INTERRUPT_EN_TX;
            a_thd->optlen[0] = 0;
            a_thd->optlen[1] = 0;
            a_thd->optlen[2] = 0;

            // Initialize BAR descriptor
            bar_thd->upatterntx = TX_HEADER_DESC_PATTERN;
            bar_thd->frmlifetime = 0;
            bar_thd->nextmpdudesc_ptr = 0;
            bar_thd->policyentryaddr = CPU2HW(&agg_desc->bar_pol_tbl);
            bar_thd->datastartptr = CPU2HW(bar_payl);
            bar_thd->dataendptr = (uint32_t)bar_thd->datastartptr + BAR_FRM_LEN_WITH_FCS - 1;
            bar_thd->first_pbd_ptr = 0;
            bar_thd->optlen[0] = 0;
            bar_thd->optlen[1] = 0;
            bar_thd->optlen[2] = 0;

            // Set frame length
            bar_thd->frmlen = BAR_FRM_LEN_WITH_FCS;
            // Set PHY control information
            bar_thd->phyctrlinfo = 0;
            // Set MAC control information 1
            bar_thd->macctrlinfo1 = EXPECTED_ACK_COMPRESSED_BLOCK_ACK;
            // Set WhichDescriptor field
            bar_thd->macctrlinfo2 = INTERRUPT_EN_TX | WHICHDESC_UNFRAGMENTED_MSDU |
                                            FRM_TYPE_CNTL | FRM_CNTL_SUBTYPE_BAR| TS_VALID_BIT;

            // Initialize BAR policy table
            bar_pol->upatterntx = POLICY_TABLE_PATTERN;
            bar_pol->phycntrlinfo1 = phy_get_ntx() << NX_TX_PT_OFT;
            bar_pol->phycntrlinfo2 = TX_NTX_2_ANTENNA_SET(phy_get_ntx());
            bar_pol->maccntrlinfo1 = 0;
            bar_pol->maccntrlinfo2 = 0xFFFF0704;

            // Push the descriptor into the list
            co_list_push_back(list, (struct co_list_hdr *) agg_desc);
        }
    }
}

/**
 ****************************************************************************************
 * @brief Reset all hardware TX descriptor for A-MPDU.
 ****************************************************************************************
 */
static void txl_agg_hwdesc_reset(void)
{
    // Initialize the descriptor pools
    for (int i = 0; i < TX_AMPDU_DESC_QUEUE_CNT; i++)
    {
        struct co_list *list = &tx_agg_desc_pool[i];

        // Initialize the list
        co_list_init(list);

        for (int j = 0; j < tx_agg_desc_cnt[i]; j++)
        {
            struct tx_agg_desc *agg_desc = (struct tx_agg_desc *)&tx_agg_desc_array[i][j];

            // Push the descriptor into the list
            co_list_push_back(list, (struct co_list_hdr *) agg_desc);
        }
    }
}

#if NX_MAC_HE
/**
 ****************************************************************************************
 * @brief Check if the frame passed as parameter is an internal frame or a management
 * frame transmitted by the host.
 *
 * @param[in] txdesc    Pending txdesc
 *
 * @return true if frame is an internal one or a management one sent from the host, false
 *         otherwise
 ****************************************************************************************
 */
static bool txl_agg_is_int_or_mgt(struct txdesc *txdesc)
{
    if (is_int_frame(txdesc))
        return true;

    #if NX_UMAC_PRESENT
    if (txdesc->host.flags & TXU_CNTRL_MGMT)
        return true;
    #endif

    return false;
}

/**
 ****************************************************************************************
 * @brief Set the UPH value in the +HTC field of each downloaded MPDU of a A-MPDU.
 *
 * The function starts with the MPDU pointed by the TX descriptor passed as parameter and
 * goes through the next MPDUs until reaching the last one of the A-MPDU, or a MPDU with
 * currently no allocated buffer.
 *
 * @param[in] txdesc  First or intermediate MPDU of the A-MPDU
 ****************************************************************************************
 */
static void txl_agg_set_uph(struct txdesc *txdesc)
{
    // Go through the MPDUs until we cannot put more MPDUs due to the HE TB PPDU
    // length, the timeout of the HE TB programming, or until we reach the end of the
    // A-MPDU
    while(1)
    {
        // Update the +HTC field with the UPH - if +HTC is not updated, it means that the
        // buffer is not allocated yet, and we can exit the loop
        if (!txl_buffer_update_htc(&txdesc->lmac.hw_desc->thd, txl_he_tb_uph_get()))
            break;

        // Maybe this MPDU is the last of the A-MPDU
        if (is_mpdu_last(txdesc))
            break;

        // Get next MPDU
        txdesc = tx_desc_next(txdesc);

        // Sanity check - The latest MPDU we handled was not the last of the A-MPDU, so
        // we should not reach the end of the list
        ASSERT_ERR(txdesc != NULL);
    }
}

/**
 ****************************************************************************************
 * @brief Split an A-MPDU so that it can fit into a HE TB PPDU.
 *
 * If the A-MPDU can fit as is, then it is not split.
 * If the A-MPDU does not fit into the HE TB PPDU, it can be split in two parts. One part
 * that will be programmed in the HE TB, and a second part that can be pushed back in the
 * AC queue. In the case the HE TB parameters don't allow to put any MPDU inside it, the
 * A-MPDU is kept as is.
 *
 * @param[in] txdesc  First MPDU of the A-MPDU to be checked
 * @param[out] txdesc_next Pointer to the TX descriptor following the last descriptor of
 * the HE TB PPDU.
 * @param[in] max_len Maximum length, in bytes, of the HE TB PPDU
 * @param[in] min_mpdu_len Minimum MPDU length inside the HE TB PPDU (for minimal MPDU
 * start spacing)
 * @param[in] ac Access category of the A-MPDU
 *
 * @return The pointer to the THD to be programmed in the HE TB PPDU (NULL if nothing can
 * be pushed).
 ****************************************************************************************
 */
static struct tx_hd *txl_agg_split(struct txdesc *txdesc, struct txdesc **txdesc_next,
                                   uint32_t max_len, uint16_t min_mpdu_len, uint8_t ac)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[ac];
    struct tx_agg_desc *agg_desc_old = txdesc->lmac.agg_desc;
    struct tx_agg_desc *agg_desc_new;
    struct tx_hd *thd_tb;
    struct tx_hd *a_thd_new, *a_thd_old;
    struct txdesc *txdesc_prev = NULL;
    #if NX_UMAC_PRESENT
    uint8_t current_cnt;
    #endif
    bool split = true;
    bool timeout = false;

    agg_desc_new = txl_agg_desc_alloc(TX_HE_TB_AMPDU_QUEUE_IDX);
    ASSERT_ERR(agg_desc_new != NULL);

    // Initialize some fields of the new A-MPDU descriptor from values of old one
    agg_desc_new->status = agg_desc_old->status;
    agg_desc_new->sta_idx = agg_desc_old->sta_idx;
    agg_desc_new->tid = agg_desc_old->tid;
    agg_desc_new->user_cnt = 1;

    // Initialize some fields of the new A-MPDU THD from values of old one
    a_thd_old = &agg_desc_old->a_thd;
    a_thd_new = &agg_desc_new->a_thd;
    a_thd_new->phyctrlinfo = a_thd_old->phyctrlinfo;
    a_thd_new->macctrlinfo1 = a_thd_old->macctrlinfo1;
    a_thd_new->macctrlinfo2 = a_thd_old->macctrlinfo2;
    a_thd_new->nextmpdudesc_ptr = a_thd_old->nextmpdudesc_ptr;
    a_thd_new->policyentryaddr = a_thd_old->policyentryaddr;
    a_thd_new->frmlen = 0;
    thd_tb = a_thd_new;

    // Prepare looping across old A-MPDU
    #if NX_UMAC_PRESENT
    current_cnt = 0;
    #endif

    // Go through the A-MPDU MPDUs until we cannot put more MPDUs due to the HE TB PPDU
    // length, the timeout of the HE TB programming, or until we reach the end of the
    // A-MPDU
    while(1)
    {
        struct tx_hd *thd = &txdesc->lmac.hw_desc->thd;
        uint16_t nb_delims_new, nb_delims_old;
        uint16_t subfrm_len_new, subfrm_len_old;

        // Check if the new frame is smaller than the minimum MPDU length
        nb_delims_old = (txdesc->umac.flags & NB_BLANK_DELIM_MSK) >> NB_BLANK_DELIM_OFT;
        subfrm_len_old = txl_mpdu_subframe_len(thd) + nb_delims_old * DELIMITER_LEN;
        if (subfrm_len_old < min_mpdu_len)
        {
            // Compute the new number of delimiters and sub-frame length
            nb_delims_new = txl_agg_mpdu_nb_delims(thd, min_mpdu_len);
            subfrm_len_new = txl_mpdu_subframe_len(thd) + nb_delims_new * DELIMITER_LEN;
        }
        else
        {
            // Sub-frame is already bigger than required, let's keep it like that
            subfrm_len_new = subfrm_len_old;
        }

        // Check if the MPDU fits into the HE TB PPDU length and ensure we will have
        // enough time to chain the PPDU
        if (((a_thd_new->frmlen + subfrm_len_new) > max_len) || timeout)
        {
            // Check if we can put at least one MPDU in the HE TB
            if (txdesc_prev == NULL)
            {
                // Request to disable the MU EDCA because the HE TB queue is blocked
                txl_he_mu_edca_blocked(a_thd_old, ac);

                // Free the allocated A-MPDU descriptor
                txl_agg_desc_free(agg_desc_new);

                return NULL;
            }
            // Check if the transmitted frame will be an A-MPDU or a singleton
            else if (is_mpdu_first(txdesc_prev))
            {
                // Transform the frame in singleton
                txl_agg_change_to_singleton(txdesc_prev, true);

                // For singleton sent as HE TB we need a A-MPDU descriptor attached so
                // put it back
                txdesc_prev->lmac.agg_desc = agg_desc_new;

                // The THD that needs to be chained in the HE TB is now the one attached
                // to the TX descriptor
                thd_tb = &txdesc_prev->lmac.hw_desc->thd;
                break;
            }
            else
            {
                struct tx_hd *thd_last = &txdesc_prev->lmac.hw_desc->thd;
                #if NX_UMAC_PRESENT
                struct tx_cfm_tag *cfm = txl_cfm_tag_get(txdesc_prev);
                cfm->ampdu_size = current_cnt;
                #endif

                // Remember the last TX descriptor in the A-MPDU descriptor
                agg_desc_new->txdesc_last = txdesc_prev;

                // Mark as last MPDU
                set_mpdu_pos(txdesc_prev, WHICHDESC_AMPDU_LAST);

                // And update the MAC Control Information 2 field to the correct value
                thd_last->macctrlinfo2 = txdesc_prev->umac.flags | INTERRUPT_EN_TX;
                thd_last->nextmpdudesc_ptr = 0;
                break;
            }
        }

        PROF_AGG_ADD_MPDU_SET();

        // Update the +HTC with the UPH
        txl_buffer_update_htc(thd, txl_he_tb_uph_get());

        // Attach the new A-MPDU descriptor to the TX descriptor
        txdesc->lmac.agg_desc = agg_desc_new;

        if (subfrm_len_old < min_mpdu_len)
        {
            // Update number of blank delimiters
            txdesc->umac.flags = thd->macctrlinfo2;
            txdesc->umac.flags &= ~NB_BLANK_DELIM_MSK;
            txdesc->umac.flags |= (nb_delims_new << NB_BLANK_DELIM_OFT);
            thd->macctrlinfo2 = txdesc->umac.flags | INTERRUPT_EN_TX;
        }
        else
        {
            subfrm_len_old = subfrm_len_new;
        }

        // Add MPDU to A-MPDU
        a_thd_new->frmlen += subfrm_len_new;
        // Now that we have added the MPDU to the new A-MPDU, let's "remove" it from the
        // old one
        a_thd_old->frmlen -= subfrm_len_old;

        PROF_AGG_ADD_MPDU_CLR();

        // Maybe this MPDU is the last of the previously programmed A-MPDU
        if (is_mpdu_last(txdesc))
        {
            agg_desc_new->txdesc_last = txdesc;
            split = false;
            break;
        }

        txdesc_prev = txdesc;
        #if NX_UMAC_PRESENT
        current_cnt++;
        #endif

        // If we reached the time limit, then request to exit the loop
        if (nxmac_tx_hetb_rem_dur_getf() < TX_HE_TB_PROG_TIME)
            timeout = true;

        txdesc = tx_desc_next(txdesc);

        // Sanity check - We should exit either because we reached the end of the A-MPDU
        // or we filled-up completely the HE TB PPDU, but we shall never reach the end of
        // the descriptor list
        ASSERT_ERR(txdesc != NULL);
    }

    // Sanity check. The pending A-MPDU descriptor shall be the first in the list of
    // descriptors. If it is not the case it means that we have probably been too late
    // to handle the previous A-MPDU BAR-THD/A-THD check interrupt.
    ASSERT_ERR(co_list_pick(&txlist->aggregates) == &agg_desc_old->list_hdr);

    // The A-MPDU might not be split, in case its original length was longer than the
    // allowed one, but new blank delimiter values have reduced it below this threshold.
    // Even if this case seems unlikely, let's handle it, it has a very limited cost.
    if (split)
    {
        // A-MPDU has been split, we'll have one more PPDU in the TX path
        txlist->ppdu_cnt++;

        // Maybe the first MPDU after the newly created A-MPDU was the last in the
        // split A-MPDU. In such case we need to pop its descriptor from the aggregates
        // list before pushing the new descriptor. The releasing of the old descriptor
        // will be done after the new A-MPDU is chained.
        if (is_mpdu_last(txdesc))
        {
            co_list_pop_front(&txlist->aggregates);
        }
        *txdesc_next = txdesc;
    }
    else
    {
        // A-MPDU was not split, so free the descriptor used previously
        co_list_pop_front(&txlist->aggregates);
        txl_agg_desc_free(agg_desc_old);
        *txdesc_next = tx_desc_next(txdesc);

        // Remove BAR THD from last MPDU
        txdesc->lmac.hw_desc->thd.nextmpdudesc_ptr = 0;
    }

    // Push the new A-MPDU descriptor in the list
    co_list_push_front(&txlist->aggregates, &agg_desc_new->list_hdr);

    return (thd_tb);
}

/**
 ****************************************************************************************
 * @brief Do the necessary checks and preparation prior to perform a concatenation
 * between a MPDU or A-MPDU, and a A-MPDU.
 *
 * The function first checks if the MPDUs parts of both PPDUs are compliant, i.e if they
 * are for the same STAID/TID. Then several checks are done to ensure the concatenation
 * is possible and won't result in PHY underflows that would be caused by some MPDU
 * payloads not yet downloaded. There are in particular some corner cases that are handled
 * here, that can be observed more specially with small packets.
 * The function may finally finish the second A-MPDU if all checks are OK if this A-MPDU
 * was still under formatting.
 *
 * @note The @ref txl_agg_finish function is not called here in order to avoid doing the
 * chaining of the finished A-MPDU in this specific case.
 *
 * @param[in] txdesc1  Pointer to one TX descriptor of the first PPDU
 * @param[in] txdesc2  Pointer to first TX descriptor of the A-MPDU to append to the first
 * PPDU.
 * @param[in] ac Access category of the PPDUs
 *
 * @return A boolean indicating if the concatenation can be done.
 ****************************************************************************************
 */
static bool txl_agg_he_tb_prep_for_cat(struct txdesc *txdesc1, struct txdesc *txdesc2,
                                       uint8_t ac)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[ac];
    struct txl_agg_build_tag *agg;
    struct txdesc *txdesc_first, *txdesc_last;
    struct tx_hd *thd;
    struct tx_pbd *tbd;
    struct txl_buffer_tag *buffer;
    #if NX_UMAC_PRESENT
    struct tx_cfm_tag *cfm;
    #endif
    struct tx_agg_desc *agg_desc = txdesc2->lmac.agg_desc;

    // Check if both PPDUs are compatible
    if ((txdesc1->host.staid != txdesc2->host.staid) ||
        (txdesc1->host.tid != txdesc2->host.tid))
        return false;

    // There is a gray zone when the first MPDU of a A-MPDU has been allocated, or enough
    // buffer allocation has been done for the A-MPDU to be appended, but we are still
    // waiting for the DMA interrupt following one of the previous events. The handling of
    // the awaited DMA interrupt could cause an erroneous chaining of the A-MPDU so don't
    // do the concatenation in that case
    if ((txl_buffer_get(txdesc2) && !(agg_desc->status & AGG_FIRST_DOWNLOADED)) ||
        ((agg_desc->status & (AGG_ALLOC | AGG_DOWNLOADED)) == AGG_ALLOC))
        return false;

    if (agg_desc->status & AGG_FORMATTED)
        return true;

    agg = &txlist->agg[0];

    // No concatenation if finished A-MPDU would result into a singleton MPDU
    if (agg->curr_cnt == 1)
        return false;

    txdesc_last = agg->txdesc_last;
    thd = &txdesc_last->lmac.hw_desc->thd;
    buffer = txl_buffer_get(txdesc_last);
    tbd = HW2CPU(thd->first_pbd_ptr);

    // If buffer is already allocated for this descriptor, check if it is
    // already downloaded or not
    if ((buffer != NULL) && (tbd->upatterntx != TX_PAYLOAD_DESC_PATTERN))
        return false;

    // Check if we still have enough time to do the finishing procedure
    if (nxmac_tx_hetb_rem_dur_getf() <= (TX_HE_TB_PROG_TIME + TX_AGG_FINISH_DUR))
        return false;

    // For profiling
    PROF_AGG_FINISH_AMPDU_SET();

    txdesc_first = txdesc2;
    #if NX_UMAC_PRESENT
    cfm = txl_cfm_tag_get(txdesc_last);
    #endif

    // Re-mark last MPDU as LAST not MIDDLE
    set_mpdu_pos(txdesc_last, WHICHDESC_AMPDU_LAST);

    agg_desc->txdesc_last = txdesc_last;
    txl_he_txop_dur_rtscts_thres_ampdu_check(agg->txdesc_first);

    #if NX_UMAC_PRESENT
    cfm->ampdu_size = agg->curr_cnt;
    #endif

    //BAR DESCRIPTOR
    txl_fill_bar(agg_desc, txdesc_last->umac.sn_win, agg_desc->sta_idx, agg_desc->tid, agg->bw);

    // Chain the BAR descriptor to the last MPDU
    thd->nextmpdudesc_ptr = CPU2HW(&agg_desc->bar_thd);

    // And update the MAC Control Information 2 field to the correct value
    thd->macctrlinfo2 = txdesc_last->umac.flags | INTERRUPT_EN_TX;

    // Current aggregate is now completed
    agg_desc->status |= AGG_FORMATTED;

    // If buffer is already allocated for this descriptor, check if it is
    // already downloaded or not
    if (buffer != NULL)
    {
        // Check if enough data is already allocated
        if (!(agg_desc->status & AGG_ALLOC))
        {
            // If this is not the case, check if the first MPDU is already downloaded
            if (!(agg_desc->status & AGG_FIRST_DOWNLOADED))
            {
                // The first MPDU is not downloaded, so we will wait for its DMA
                // interrupt and do the A-MPDU chaining at that moment
                struct txl_buffer_tag *buffer_first = txl_buffer_get(txdesc_first);
                buffer_first->flags |= BUF_ALLOC_OK;
            }
            else
                // The first MPDU is already downloaded, so we consider that we can
                // chain the A-MPDU immediately
                agg_desc->status |= AGG_DOWNLOADED;
        }
    }

    // Check if aggregate descriptor is ready to be programmed
    if (agg_desc->status & AGG_DOWNLOADED)
    {
        // Re-set the PHY flags in the policy table as they might have been changed
        // during the TXOP duration RTS/CTS threshold check
        struct tx_policy_tbl *pt = HW2CPU(agg_desc->a_thd.policyentryaddr);
        pt->ratecntrlinfo[0] = agg->txdesc_first->umac.phy_flags;
    }

    // Put aggregate in used list
    co_list_push_back(&txl_cntrl_env.txlist[ac].aggregates, &agg_desc->list_hdr);

    // We will now have some data programmed for transmission, so we can activate the A-MPDU
    // optimization
    txl_cntrl_env.txlist[ac].ppdu_cnt++;

    // Reinitialize the build state machine
    agg->desc = NULL;

    // For profiling
    PROF_AGG_FINISH_AMPDU_CLR();

    return true;
}

/**
 ****************************************************************************************
 * @brief Concatenate a A-MPDU with the MPDU just following it.
 *
 * The function first checks if with the concatenated MPDU the A-MPDU will still fit
 * into the HE TB. If not the concatenation is not performed.
 *
 * @param[in] txdesc_ampdu  First MPDU of the A-MPDU
 * @param[in, out] txdesc_next Pointer to the TX descriptor of the MPDU. If the
 * concatenation is done this pointer will be updated with the descriptor just following
 * the resulting A-MPDU.
 * @param[in] max_len Maximum length, in bytes, of the HE TB PPDU
 * @param[in] min_mpdu_len Minimum MPDU length inside the HE TB PPDU (for minimal MPDU
 * start spacing)
 * @param[in] ac Access category of the A-MPDU
 *
 * @return A boolean indicating if the concatenation was done. If false, then the caller
 * knows that no additional concatenation shall be attempted, and that the HE TB can be
 * programmed.
 ****************************************************************************************
 */
static bool txl_agg_he_tb_cat_ampdu_mpdu(struct txdesc *txdesc_ampdu, struct txdesc **txdesc_next,
                                         uint32_t max_len, uint16_t min_mpdu_len, uint8_t ac)
{
    struct tx_agg_desc *agg_desc = txdesc_ampdu->lmac.agg_desc;
    struct txdesc *txdesc_mpdu = *txdesc_next;
    struct txdesc *txdesc_last = agg_desc->txdesc_last;
    struct txl_list *txlist = &txl_cntrl_env.txlist[ac];
    struct tx_hd *thd = &txdesc_mpdu->lmac.hw_desc->thd;
    struct tx_hd *thd_last = &txdesc_last->lmac.hw_desc->thd;
    uint16_t nb_delims_new, subfrm_len_new;
    struct tx_hd *a_thd;

    // Check if MPDU is compatible with the A-MPDU
    if ((agg_desc->sta_idx != txdesc_mpdu->host.staid) ||
        (agg_desc->tid != txdesc_mpdu->host.tid))
        return false;

    a_thd = &agg_desc->a_thd;
    thd = &txdesc_mpdu->lmac.hw_desc->thd;

    // Re-add the +HTC length and order bit in the MPDU buffer if necessary
    txl_buffer_add_htc(thd);

    // Compute the new number of delimiters and the sub-frame length
    nb_delims_new = txl_agg_mpdu_nb_delims(thd, min_mpdu_len);
    subfrm_len_new = txl_mpdu_subframe_len(thd) + nb_delims_new * DELIMITER_LEN;

    if ((a_thd->frmlen + subfrm_len_new) > max_len)
        return false;

    // PPDUs will be concatenated, we'll have one less in the TX path
    txlist->ppdu_cnt--;

    set_mpdu_pos(txdesc_last, WHICHDESC_AMPDU_INT);
    thd_last->macctrlinfo2 = txdesc_last->umac.flags | INTERRUPT_EN_TX;
    thd_last->nextmpdudesc_ptr = CPU2HW(thd);

    set_mpdu_pos(txdesc_mpdu, WHICHDESC_AMPDU_LAST);
    txdesc_mpdu->umac.flags |= (nb_delims_new << NB_BLANK_DELIM_OFT) | UNDER_BA_SETUP_BIT;
    thd->macctrlinfo2 = txdesc_mpdu->umac.flags | INTERRUPT_EN_TX;
    thd->nextfrmexseq_ptr = 0;
    thd->policyentryaddr = 0;

    a_thd->frmlen += subfrm_len_new;

    txdesc_mpdu->lmac.agg_desc = agg_desc;
    agg_desc->txdesc_last = txdesc_mpdu;

    *txdesc_next = tx_desc_next(agg_desc->txdesc_last);

    // Update the +HTC with the UPH
    txl_buffer_update_htc(thd, txl_he_tb_uph_get());

    #if NX_UMAC_PRESENT
    txl_cfm_tag_get(txdesc_mpdu)->ampdu_size = txl_cfm_tag_get(txdesc_last)->ampdu_size + 1;
    #endif

    return true;
}

/**
 ****************************************************************************************
 * @brief Concatenate a MPDU with the A-MPDU just following it.
 *
 * After concatenation the function checks if the resulting A-MPDU can fit into the HE TB
 * PPDU. If not this A-MPDU is then split.
 *
 * @param[in] txdesc_mpdu Pointer to the MPDU descriptor
 * @param[in, out] txdesc_next Pointer to the TX descriptor of the first MPDU of the
 * A-MPDU. Once the concatenation is done this pointer will be updated with the descriptor
 * just following the resulting A-MPDU.
 * @param[in] max_len Maximum length, in bytes, of the HE TB PPDU
 * @param[in] min_mpdu_len Minimum MPDU length inside the HE TB PPDU (for minimal MPDU
 * start spacing)
 * @param[in] ac Access category of the A-MPDU
 *
 * @return A boolean indicating if the concatenation was done. If false, then the caller
 * knows that no additional concatenation shall be attempted, and that the HE TB can be
 * programmed.
 ****************************************************************************************
 */
static bool txl_agg_he_tb_cat_mpdu_ampdu(struct txdesc *txdesc_mpdu, struct txdesc **txdesc_next,
                                         uint32_t max_len, uint16_t min_mpdu_len, uint8_t ac)
{
    struct txdesc *txdesc_ampdu = *txdesc_next;
    struct tx_agg_desc *agg_desc = txdesc_ampdu->lmac.agg_desc;
    struct txl_list *txlist = &txl_cntrl_env.txlist[ac];
    struct tx_hd *thd = &txdesc_mpdu->lmac.hw_desc->thd;
    uint16_t nb_delims_new, subfrm_len_new;
    struct tx_hd *a_thd;

    // Ensure the A-MPDU can be concatenated to the first PPDU
    if (!txl_agg_he_tb_prep_for_cat(txdesc_mpdu, txdesc_ampdu, ac))
        return false;

    a_thd = &agg_desc->a_thd;
    thd = &txdesc_mpdu->lmac.hw_desc->thd;

    // Compute the new number of delimiters and the sub-frame length
    nb_delims_new = txl_agg_mpdu_nb_delims(thd, min_mpdu_len);
    subfrm_len_new = txl_mpdu_subframe_len(thd) + nb_delims_new * DELIMITER_LEN;

    // PPDUs will be concatenated, we'll have one less in the TX path
    txlist->ppdu_cnt--;

    thd->nextmpdudesc_ptr = a_thd->nextmpdudesc_ptr;
    a_thd->frmlen += subfrm_len_new;
    a_thd->nextmpdudesc_ptr = CPU2HW(thd);

    txdesc_mpdu->lmac.agg_desc = agg_desc;
    set_mpdu_pos(txdesc_mpdu, WHICHDESC_AMPDU_FIRST);
    txdesc_mpdu->umac.flags |= (nb_delims_new << NB_BLANK_DELIM_OFT) | UNDER_BA_SETUP_BIT;
    thd->macctrlinfo2 = txdesc_mpdu->umac.flags | INTERRUPT_EN_TX;
    thd->policyentryaddr = 0;

    set_mpdu_pos(txdesc_ampdu, WHICHDESC_AMPDU_INT);
    txdesc_ampdu->lmac.hw_desc->thd.macctrlinfo2 = txdesc_ampdu->umac.flags | INTERRUPT_EN_TX;

    #if NX_UMAC_PRESENT
    txl_cfm_tag_get(agg_desc->txdesc_last)->ampdu_size++;
    #endif

    if (a_thd->frmlen > max_len)
    {
        txl_agg_split(txdesc_mpdu, txdesc_next, max_len, min_mpdu_len, ac);
        return false;
    }

    // Update the UPH in all A-MPDU constituent MPDUs
    txl_agg_set_uph(txdesc_ampdu);

    *txdesc_next = tx_desc_next(agg_desc->txdesc_last);

    return true;
}

/**
 ****************************************************************************************
 * @brief Concatenate two A-MPDUs prior to a HE TB transmission.
 *
 * While concatenating the function checks if the resulting A-MPDU can fit into the HE TB
 * PPDU. If not the function will result in two A-MPDUs, one that can be transmitted
 * within the HE TB PPDU, and a subsequent one with the rest of the second A-MPDU.
 *
 * @param[in] txdesc  Pointer to the first MPDU of the first A-MPDU
 * @param[in, out] txdesc_next Pointer to the first MPDU of the second A-MPDU. Once the
 * concatenation is done this pointer will be updated with the descriptor just following
 * the resulting A-MPDU.
 * @param[in] max_len Maximum length, in bytes, of the HE TB PPDU
 * @param[in] min_mpdu_len Minimum MPDU length inside the HE TB PPDU (for minimal MPDU
 * start spacing)
 * @param[in] ac Access category of the A-MPDU
 *
 * @return A boolean indicating if the concatenation was complete, i.e. indicating that
 * both A-MPDUs can fit into the HE TB PPDU. If false, then the caller knows that no
 * additional concatenation shall be attempted, and that the HE TB can be
 * programmed.
 ****************************************************************************************
 */
static bool txl_agg_he_tb_cat_ampdu(struct txdesc *txdesc, struct txdesc **txdesc_next,
                                    uint32_t max_len, uint16_t min_mpdu_len, uint8_t ac)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[ac];
    struct txdesc *txdesc_n = *txdesc_next;
    struct tx_agg_desc *agg_desc_n = txdesc_n->lmac.agg_desc;
    struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;
    struct tx_hd *a_thd, *a_thd_n;
    struct txdesc *txdesc_last = agg_desc->txdesc_last;
    struct tx_hd *thd_n = &txdesc_n->lmac.hw_desc->thd;
    bool split = true;
    struct txdesc *txdesc_prev = NULL;
    #if NX_UMAC_PRESENT
    uint8_t current_cnt;
    #endif

    // Pop the first aggregate descriptor as we will have to pop also the next one
    co_list_pop_front(&txlist->aggregates);

    // Ensure the A-MPDU can be concatenated to the first PPDU
    if (!txl_agg_he_tb_prep_for_cat(txdesc, txdesc_n, ac))
    {
        co_list_push_front(&txlist->aggregates, &agg_desc->list_hdr);
        return false;
    }

    a_thd_n = &agg_desc_n->a_thd;
    a_thd = &agg_desc->a_thd;

    #if NX_UMAC_PRESENT
    current_cnt = txl_cfm_tag_get(txdesc_last)->ampdu_size;
    #endif

    // Chain the MPDU
    txdesc_last->lmac.hw_desc->thd.nextmpdudesc_ptr = CPU2HW(thd_n);
    set_mpdu_pos(txdesc_last, WHICHDESC_AMPDU_INT);
    txdesc_last->lmac.hw_desc->thd.macctrlinfo2 = txdesc_last->umac.flags | INTERRUPT_EN_TX;
    set_mpdu_pos(txdesc_n, WHICHDESC_AMPDU_INT);
    thd_n->macctrlinfo2 = (*txdesc_next)->umac.flags | INTERRUPT_EN_TX;

    // Go through the second A-MPDU MPDUs and update them to be part of the first A-MPDU,
    // until we cannot put more MPDUs due to the HE TB PPDU length, the timeout of the HE
    // TB programming, or until we reach the end of the second A-MPDU
    while(1)
    {
        struct tx_hd *thd = &txdesc_n->lmac.hw_desc->thd;
        uint16_t nb_delims_new, nb_delims_old;
        uint16_t subfrm_len_new, subfrm_len_old;

        // Check if the new frame is smaller than the minimum MPDU length
        nb_delims_old = (txdesc_n->umac.flags & NB_BLANK_DELIM_MSK) >> NB_BLANK_DELIM_OFT;
        subfrm_len_old = txl_mpdu_subframe_len(thd) + nb_delims_old * DELIMITER_LEN;
        if (subfrm_len_old < min_mpdu_len)
        {
            // Compute the new number of delimiters and the sub-frame length
            nb_delims_new = txl_agg_mpdu_nb_delims(thd, min_mpdu_len);
            subfrm_len_new = txl_mpdu_subframe_len(thd) + nb_delims_new * DELIMITER_LEN;
        }
        else
        {
            // Sub-frame is already bigger than required, let's keep it like that
            subfrm_len_new = subfrm_len_old;
        }

        // Check if the MPDU fits into the HE TB PPDU length and ensure we will have
        // enough time to chain the PPDU
        if (((a_thd->frmlen + subfrm_len_new) > max_len) ||
            (nxmac_tx_hetb_rem_dur_getf() < TX_HE_TB_PROG_TIME))
        {
            // Check if we can put at least one MPDU in the HE TB
            if (txdesc_prev == NULL)
            {
                // No MPDU of the second A-MPDU could be concatenated - Revert the
                // chaining previously done
                txdesc_last->lmac.hw_desc->thd.nextmpdudesc_ptr = 0;
                set_mpdu_pos(txdesc_last, WHICHDESC_AMPDU_LAST);
                txdesc_last->lmac.hw_desc->thd.macctrlinfo2 = txdesc_last->umac.flags | INTERRUPT_EN_TX;
                set_mpdu_pos(txdesc_n, WHICHDESC_AMPDU_FIRST);
                thd->macctrlinfo2 = txdesc_n->umac.flags | INTERRUPT_EN_TX;
                // Push back the A-MPDU descriptor in the list
                co_list_push_front(&txlist->aggregates, &agg_desc->list_hdr);
                *txdesc_next = txdesc_n;
                return false;
            }
            else
            {
                struct tx_hd *thd_last = &txdesc_prev->lmac.hw_desc->thd;
                #if NX_UMAC_PRESENT
                struct tx_cfm_tag *cfm = txl_cfm_tag_get(txdesc_prev);
                cfm->ampdu_size = current_cnt;
                #endif

                // Remember the last TX descriptor in the A-MPDU descriptor
                agg_desc->txdesc_last = txdesc_prev;

                // Mark as last MPDU
                set_mpdu_pos(txdesc_prev, WHICHDESC_AMPDU_LAST);

                // And update the MAC Control Information 2 field to the correct value
                thd_last->macctrlinfo2 = txdesc_prev->umac.flags | INTERRUPT_EN_TX;
                thd_last->nextmpdudesc_ptr = 0;
                break;
            }
        }

        PROF_AGG_ADD_MPDU_SET();

        // Update the +HTC with the UPH
        txl_buffer_update_htc(thd, txl_he_tb_uph_get());

        // Attach the new A-MPDU descriptor to the TX descriptor
        txdesc_n->lmac.agg_desc = agg_desc;

        if (subfrm_len_old < min_mpdu_len)
        {
            // Update number of blank delimiters
            txdesc_n->umac.flags = thd->macctrlinfo2;
            txdesc_n->umac.flags &= ~NB_BLANK_DELIM_MSK;
            txdesc_n->umac.flags |= (nb_delims_new << NB_BLANK_DELIM_OFT);
            thd->macctrlinfo2 = txdesc_n->umac.flags | INTERRUPT_EN_TX;
        }
        else
        {
            subfrm_len_old = subfrm_len_new;
        }

        // Add MPDU to A-MPDU
        a_thd->frmlen += subfrm_len_new;
        // Now that we have added the MPDU to the new A-MPDU, let's "remove" it from the
        // old one
        a_thd_n->frmlen -= subfrm_len_old;

        PROF_AGG_ADD_MPDU_CLR();

        // Maybe this MPDU is the last of the previously programmed A-MPDU
        if (is_mpdu_last(txdesc_n))
        {
            agg_desc->txdesc_last = txdesc_n;
            split = false;
            break;
        }

        txdesc_prev = txdesc_n;

        #if NX_UMAC_PRESENT
        current_cnt++;
        #endif

        txdesc_n = tx_desc_next(txdesc_n);

        // Sanity check - We should exit either because we reached the end of the A-MPDU
        // or we filled-up completely the HE TB PPDU, but we shall never reach the end of
        // the descriptor list
        ASSERT_ERR(txdesc_n != NULL);
    }

    // The A-MPDU might not be split, in case its original length was longer than the
    // allowed one, but new blank delimiter values have reduced it below this threshold.
    // Even if this case seems unlikely, let's handle it, it has a very limited cost.
    if (split)
    {
        // Maybe the first MPDU after the newly created A-MPDU was the last in the
        // split A-MPDU. In such case we need to pop its descriptor from the aggregates
        // list before pushing the new descriptor. The releasing of the old descriptor
        // will be done after the new A-MPDU is chained.
        if (is_mpdu_last(txdesc_n))
        {
            co_list_pop_front(&txlist->aggregates);
        }
        *txdesc_next = txdesc_n;
    }
    else
    {
        // A-MPDU has been fully concatenated, we'll have one less PPDU in the TX path
        txlist->ppdu_cnt--;

        // A-MPDU was not split, so free the descriptor used previously
        co_list_pop_front(&txlist->aggregates);
        txl_agg_desc_free(agg_desc_n);
        *txdesc_next = tx_desc_next(txdesc_n);

        // Remove BAR THD from last MPDU
        txdesc_n->lmac.hw_desc->thd.nextmpdudesc_ptr = 0;
    }

    // Push the new A-MPDU descriptor in the list
    co_list_push_front(&txlist->aggregates, &agg_desc->list_hdr);

    return !split;
}
#endif // NX_MAC_HE

#if RW_MUMIMO_TX_EN
/**
 ****************************************************************************************
 * @brief Convert a MU-MIMO A-MPDU or MPDU into a SU one
 *
 * @param[in] ac       Access Category
 * @param[in] user_pos User position
 ****************************************************************************************
 */
static void txl_mumimo_convert(uint8_t ac, uint8_t user_pos)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[ac];
    struct txl_mumimo_build_info_tag *mumimo = &txlist->mumimo;
    struct txdesc *txdesc = (struct txdesc *)co_list_pick(&mumimo->tx[user_pos]);
    struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;

    // We will now have some data programmed for transmission, so we can activate the A-MPDU
    // optimization
    txlist->ppdu_cnt++;

    // Chain the data THD to the primary A-MPDU THD
    if (is_mpdu_agg(txdesc))
    {
        struct txdesc *last = (struct txdesc *)co_list_pick_last(&mumimo->tx[user_pos]);
        struct tx_hd *thd = &last->lmac.hw_desc->thd;

        // Chain the BAR descriptor to the last MPDU
        thd->nextmpdudesc_ptr = CPU2HW(&agg_desc->bar_thd);

        // Put secondary aggregate in used list
        co_list_push_back(&txlist->aggregates, &agg_desc->list_hdr);
    }
    else
    {
        // Put back the aggregate in the free list
        txl_agg_desc_free(agg_desc);

        // Unchain the aggregate descriptor from the TX descriptor
        txdesc->lmac.agg_desc = NULL;
    }

    // Check if we can allocate TX buffers for the MPDUs of this user
    while (txlist->first_to_download[0] == NULL)
    {
        if (!txl_payload_alloc(txdesc, ac, 0))
            break;

        txdesc = tx_desc_next(txdesc);
        if (txdesc == NULL)
            break;
    }
    // Concatenate the MU-MIMO TX descriptor list with the transmitting list
    co_list_concat(&txlist->transmitting[0], &mumimo->tx[user_pos]);
}

/**
 ****************************************************************************************
 * @brief Compute the duration of the data part of a MU-MIMO A-MPDU.
 *
 * @param[in] length Total data length, in bytes, of the A-MPDU
 * @param[in] mcs MCS used for the transmission
 * @param[in] sgi Flag indicating whether Short GI is used or not
 * @param[in] bw The transmission bandwidth
 * @param[in] nss The number of Spatial Streams used for the transmission
 *
 * @return The duration of the A-MPDU in ns
 ****************************************************************************************
 */
static uint32_t txl_mumimo_duration_ns(uint32_t length, uint8_t mcs, bool sgi,
                                       uint8_t bw, uint8_t nss)
{
    // Total TXTIME
    uint32_t duration;

    // IEEE coresponding variables
    uint32_t ndbps;
    uint32_t nes;
    uint32_t bit_cnt;

    // Number of DATA BIT PER SYMBOL
    ndbps = ((uint16_t)nss) * VHT_NDBPS[bw][mcs];

    // nES = 6*NES
    // VHT IEEE P802.11ac D4.0
    // 20MHz and  40MHz Bandwidth
    if (bw < BW_80MHZ)
    {
        nes = (ndbps <= 2160) ? 6 : (ndbps <= 4320) ? 12 : 18;
    }
    // 80MHz Bandwidth
    else if (bw == BW_80MHZ)
    {
        if ((nss == 7) && (ndbps == 8190))
        {
            nes = 36;
        }
        else if((nss == 7) && (ndbps == 2457))
        {
            nes = 18;
        }
        else
        {
            nes = (ndbps <= 2106) ? 6  :
                  (ndbps <= 4212) ? 12 :
                  (ndbps <= 6318) ? 18 :
                  (ndbps <= 8424) ? 24 :
                                    36;
        }
    }
    // 160MHz Bandwidth
    else
    {
        if (((nss == 4) && (ndbps == 9360)) || ((nss == 7) && (ndbps == 9828)))
        {
            nes = 36;
        }
        else if ((ndbps == 14040) && ((nss == 5) || (nss == 6)))
        {
            nes = 48;
        }
        else if ((nss == 7) && (ndbps == 16380))
        {
            nes = 54;
        }
        else
        {
            nes = (ndbps <= 2106)  ? 6  :
                  (ndbps <= 4212)  ? 12 :
                  (ndbps <= 6318)  ? 18 :
                  (ndbps <= 8424)  ? 24 :
                  (ndbps <= 10530) ? 30 :
                  (ndbps <= 12636) ? 36 :
                  (ndbps <= 14742) ? 42 :
                  (ndbps <= 16848) ? 48 :
                  (ndbps <= 18720) ? 54 :
                                     72 ;
        }
    }

    // Data Time on Air for a MUMIMO A-MPDU
    // 1) The full data length in bit is length_i*8 + 16 bits of service field + 6*nES tail bits (IEEE Std 802.11 - 2012 Chapter 18.3.5.x)
    // 2) Convert into Nsymbols according to the MCS (round up the value)
    // 3) Convert Nsymbols into time duration according to Normal GI or Short GI (round up the value)

    // Total of Symbols = (length_i*8 + 16 + 6*NES) / NDBPS
    bit_cnt = (length << 3) + 16 + nes;
    duration =  bit_cnt / ndbps;
    duration += (duration * ndbps < bit_cnt) ? 1 : 0;

    // Compute actual duration
    duration *= sgi ? 3600 : 4000;

    // Return the value
    return (duration);
}

/**
 ****************************************************************************************
 * @brief Terminates the MU-MIMO A-MPDU under construction
 *
 * @param[in] ac  Access Category
 * @param[in] user_pos User Position in the group (only for MU-MIMO, 0 if not MU-MIMO)
 ****************************************************************************************
 */
static void txl_mumimo_ampdu_finish(uint8_t ac, uint8_t user_pos)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[ac];
    struct txl_agg_build_tag *agg = &txlist->agg[user_pos];
    struct tx_agg_desc *agg_desc = agg->desc;
    struct txdesc *txdesc = agg->txdesc_last;
    uint32_t length;

    // For profiling
    PROF_AGG_FINISH_AMPDU_SET();
    PROF_MU_USER_POS_SET(user_pos);

    // Check if the A-MPDU has only one MPDU
    if (agg->curr_cnt == 1)
    {
        struct tx_hd *thd = &txdesc->lmac.hw_desc->thd;

        // In such case, transform it to a singleton MPDU
        txdesc->umac.flags = WHICHDESC_UNFRAGMENTED_MSDU;

        // Clear the next_mpdu field (that was pointing to the BAR THD)
        thd->nextmpdudesc_ptr = 0;
        // Update the MAC Control Information 2 field to the correct value
        thd->macctrlinfo2 = WHICHDESC_UNFRAGMENTED_MSDU | INTERRUPT_EN_TX;

        // Get the length of the packet
        length = thd->frmlen;
    }
    else
    {
        struct tx_hd *thd = &txdesc->lmac.hw_desc->thd;
        struct tx_hd *a_thd = &agg_desc->a_thd;

        // Re-mark last MPDU as LAST not MIDDLE
        set_mpdu_pos(txdesc, WHICHDESC_AMPDU_LAST);
        #if NX_UMAC_PRESENT
        txdesc->lmac.hw_desc->cfm.ampdu_size = agg->curr_cnt;
        #endif

        // And update the MAC Control Information 2 field to the correct value
        thd->macctrlinfo2 = txdesc->umac.flags | INTERRUPT_EN_TX;

        // Current aggregate is now completed
        agg_desc->status |= AGG_FORMATTED;

        // Get the length of the A-MPDU
        length = a_thd->frmlen;
    }

    // BAR DESCRIPTOR
    txl_fill_bar(agg_desc, txdesc->umac.sn_win, agg_desc->sta_idx, agg_desc->tid, agg->bw);

    // Indicate that this user position is now closed for this MU-MIMO PPDU
    txlist->mumimo.open &= ~CO_BIT(user_pos);

    // Save the A-MPDU length (used later for duration computation)
    txlist->mumimo.length[user_pos] = length;

    // Reinitialize the build state machine
    agg->desc = NULL;

    // For profiling
    PROF_AGG_FINISH_AMPDU_CLR();
}

/**
 ****************************************************************************************
 * @brief This function checks if the newly pushed TX descriptor can be part of a MU-MIMO
 * PPDU under construction, or if it can start a new MU-MIMO PPDU.
 * Depending on the current status of the pending MU-MIMO PPDU, this function may decide
 * to close it and start the new one immediately, or keep the current one open and pause
 * the IPC queue of the new TX descriptor until the MU-MIMO PPDU is closed.
 *
 * @param[in] txdesc      TX descriptor of the new packet pushed
 * @param[in] ac          Access Category
 * @param[out] user_pos   Pointer to the user position that will be returned
 *
 * @return The status of the packet just pushed (@ref SU_PACKET, @ref MU_PACKET or
 * @ref MU_PAUSED if the descriptor has been saved and the IPC needs to be paused).
 ****************************************************************************************
 */
static int txl_mumimo_check(struct txdesc * txdesc, uint8_t ac, uint8_t *user_pos)
{
    int status = SU_PACKET;
    uint8_t group_id = get_group_id(txdesc);
    struct txl_list *txlist = &txl_cntrl_env.txlist[ac];

    // Check if the frame can be MU-MIMOed
    if (is_mpdu_agg(txdesc) && is_mumimo_group_id(group_id) &&
        bfr_is_calibrated(txdesc->host.staid))
    {
        // Frame can be MU-MIMOed
        status = MU_PACKET;
        *user_pos = get_user_pos(txdesc);

        // Check if some users are already part of an ongoing construction
        if (txlist->mumimo.users)
        {
            // Check if the pushed frame is part of the same group
            if (group_id != txlist->mumimo.group_id)
            {
                // Close this user position
                if (txlist->mumimo.users & CO_BIT(*user_pos))
                    txl_mumimo_ampdu_finish(ac, *user_pos);
                else
                    txlist->mumimo.open &= ~CO_BIT(*user_pos);

                // Check if the other user positions are still open
                if (txlist->mumimo.open & txlist->mumimo.users)
                {
                    // We could still receive packets on other user positions,
                    // so save the current packet
                    ASSERT_ERR(txlist->mumimo.txdesc[*user_pos] == NULL);
                    txlist->mumimo.txdesc[*user_pos] = txdesc;
                    status = MU_PAUSED;
                }
                else
                {
                    // Close the MU-MIMO PPDU under construction
                    txl_agg_mumimo_close(ac);
                    return (MU_RESTART_CHECK);
                }
            }
            else
            {
                txdesc->umac.phy_flags &= VHT_MCS_MASK;
                txdesc->umac.phy_flags |= txlist->mumimo.phy_flags;
            }
        }
        else
        {
            // Start a new MU-MIMO PPDU
            PROF_MU_PPDU_START_SET();
            txlist->mumimo.group_id = group_id;
            // Remember the BW and GI of the actual transmission
            txlist->mumimo.phy_flags = txdesc->umac.phy_flags & ~MCS_INDEX_TX_RCX_MASK;
            txdesc->umac.phy_flags &= ~VHT_NSS_MASK;
            PROF_MU_PPDU_START_CLR();
        }

        PROF_MU_USER_POS_SET(*user_pos);
    }
    else
    {
        // Check if a MU-MIMO PPDU is under construction
        if (txlist->mumimo.users)
        {
            *user_pos = get_user_pos(txdesc);
            // Close this user position
            if (txlist->mumimo.users & CO_BIT(*user_pos))
                txl_mumimo_ampdu_finish(ac, *user_pos);
            else
                txlist->mumimo.open &= ~CO_BIT(*user_pos);

            // Check if the other user positions are still open
            if (txlist->mumimo.open & txlist->mumimo.users)
            {
                // We could still receive packets on other user positions,
                // so save the current packet
                ASSERT_ERR(txlist->mumimo.txdesc[*user_pos] == NULL);
                txlist->mumimo.txdesc[*user_pos] = txdesc;
                status = MU_PAUSED;
            }
            else
            {
                // Close the MU-MIMO PPDU under construction
                txl_agg_mumimo_close(ac);
                return (MU_RESTART_CHECK);
            }
        }
        else
        {
            *user_pos = 0;
        }
    }

    return (status);
}

/**
 ****************************************************************************************
 * @brief Unchains the frame exchanges that have been completed
 *
 * This primitive goes through the list of completed frame exchanges, check their status,
 * and put them in the correct list (cfm or waiting block-ack)
 *
 * @param[in] access_category  Access category for the frame exchanges to unchain
 * @param[in] user_idx         User index (for MU-MIMO TX only, 0 otherwise)
 *
 ****************************************************************************************
 */
static void txl_mumimo_secondary_done(uint8_t access_category, uint8_t user_idx)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];

    // Sanity check - This handler is only for secondary users
    ASSERT_ERR(user_idx != 0);

    // Walk through the list of completed SW descriptors
    while (1)
    {
        struct txdesc *txdesc;
        struct tx_hd *txhd;
        uint32_t txstatus;
        struct tx_agg_desc *agg_desc;

        //pick first txdesc in transmitting list to check its doneTx status
        txdesc = (struct txdesc *)co_list_pick(&(txlist->transmitting[user_idx]));

        // a descriptor is waiting to be confirmed
        if (txdesc == NULL)
            break;

        txhd = &txdesc->lmac.hw_desc->thd;
        txstatus = txhd->statinfo;
        agg_desc = txdesc->lmac.agg_desc;

        //---------------
        //STATUS ANALYSIS
        //---------------
        // Check the doneTx status bit
        if (txstatus & DESC_DONE_TX_BIT)
        {
            // For profiling
            PROF_AGG_SMPDU_DONETX_SET();

            PROF_MU_USER_POS_IRQ_SET(get_user_pos(txdesc));

            #if !NX_FULLY_HOSTED
            // Free the MPDU buffers and allocate buffer(s) for the subsequent ones
            txl_free_done_mpdu(txdesc, access_category, user_idx);
            #endif

            // Put a first status to the descriptor here, indicating it has been
            // already handled by the HW
            txdesc->lmac.hw_desc->cfm.status = txstatus;

            // For profiling
            PROF_AGG_SMPDU_DONETX_CLR();
        }
        // THD not yet transmitted, stop here and come back later
        else
        {
            #if !NX_FULLY_HOSTED && NX_AMSDU_TX
            // Maybe some A-MSDU sub-frames (i.e. TBD) are done, check that
            txl_check_done_amsdu_subframe(txdesc, access_category, user_idx);
            #endif
            break;
        }

        // Pop picked txdesc definitively from list
        co_list_pop_front(&(txlist->transmitting[user_idx]));

        // Save the status and put the descriptor in the temporary confirmation queue
        txdesc->lmac.hw_desc->cfm.status = txstatus;
        co_list_push_back(&agg_desc->cfm, (struct co_list_hdr *)txdesc);
    }
}

void txl_agg_mumimo_close(uint8_t ac)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[ac];
    struct txl_mumimo_build_info_tag *mumimo = &txlist->mumimo;
    int i;
    uint32_t dur_max = 0;
    uint8_t prim_user_pos = 0;
    uint8_t nb_users = 0;
    bool mumimook = true;

    PROF_MU_PPDU_CLOSE_SET();

    // Go through the A-MPDU which are ready for this MU-MIMO PPDU
    for (i = 0; i < RW_USER_MAX; i++)
    {
        if (mumimo->users & CO_BIT(i))
        {
            struct txl_agg_build_tag *agg = &txlist->agg[i];
            struct txdesc *txdesc = (struct txdesc *)co_list_pick(&mumimo->tx[i]);
            uint32_t rate_info = mumimo->rateinfo[i];
            uint32_t duration;

            // Check if we have to close the A-MPDU under formation
            if (agg->desc != NULL)
            {
                txl_mumimo_ampdu_finish(ac, i);
            }

            // Now we need to ensure that the primary user is an A-MPDU
            if (!is_mpdu_agg(txdesc))
                mumimook = false;

            // Compute the duration of the A-MPDU
            duration = txl_mumimo_duration_ns(mumimo->length[i],
                                              (rate_info & VHT_MCS_MASK) >> VHT_MCS_OFT,
                                              (rate_info & SHORT_GI_TX_RCX_MASK) != 0,
                                              (rate_info & BW_TX_RCX_MASK) >> BW_TX_RCX_OFT,
                                              ((rate_info & VHT_NSS_MASK) >> VHT_NSS_OFT) + 1);

            // Check if the duration of the present A-MPDU is the greatest one
            if (duration > dur_max)
            {
                // The present A-MPDU duration is the greatest one, so save its duration
                // as well as the user position
                dur_max = duration;
                prim_user_pos = i;
            }

            // One more user in this MU-MIMO PPDU
            nb_users++;
        }
    }

    // Check if the MU-MIMO PPDU has several users
    if ((nb_users > 1) && (mumimook))
    {
        struct txdesc *txdesc = (struct txdesc *)co_list_pick(&mumimo->tx[prim_user_pos]);

        // Now we need to ensure that the primary user is an A-MPDU
        if (is_mpdu_agg(txdesc))
        {
            struct tx_agg_desc *prim_agg_desc = txdesc->lmac.agg_desc;
            struct tx_hd *a_thd = &prim_agg_desc->a_thd;
            struct tx_hd *bar_thd = &prim_agg_desc->bar_thd;
            struct txdesc *last = (struct txdesc *)co_list_pick_last(&mumimo->tx[prim_user_pos]);
            struct tx_hd *thd = &last->lmac.hw_desc->thd;
            int sec_idx = 1;
            uint32_t *sec_user_ptr = &a_thd->sec_user1_ptr;
            struct tx_agg_desc *agg_desc = prim_agg_desc;

            PROF_MU_USER_POS_SET(prim_user_pos);

            // Chain the BAR descriptor to the last MPDU
            thd->nextmpdudesc_ptr = CPU2HW(&prim_agg_desc->bar_thd);

            // This descriptor will be an intermediate one
            prim_agg_desc->status |= AGG_INT;
            prim_agg_desc->download = 0;
            prim_agg_desc->prim_agg_desc = prim_agg_desc;

            // Initialize the THD fields
            a_thd->phyctrlinfo = (mumimo->group_id << GID_TX_OFT) |
                                 (prim_user_pos << USER_POS_OFT) |
                                 USE_MUMIMO_TX_BIT;

            // Put primary aggregate in used list
            co_list_push_back(&txlist->aggregates, &prim_agg_desc->list_hdr);

            // We will now have some data programmed for transmission, so we can activate the A-MPDU
            // optimization
            txlist->ppdu_cnt++;

            // Perform the chaining to the primary A-MPDU and BAR THDs
            for (i = 0; i < RW_USER_MAX; i++)
            {
                if (mumimo->users & CO_BIT(i))
                {
                    int user_q_idx;
                    txdesc = (struct txdesc *)co_list_pick(&mumimo->tx[i]);

                    PROF_MU_USER_POS_SET(i);

                    if (i != prim_user_pos)
                    {
                        agg_desc = txdesc->lmac.agg_desc;

                        // Chain the data THD to the primary A-MPDU THD
                        if (is_mpdu_agg(txdesc))
                        {
                            thd = &agg_desc->a_thd;

                            // If the secondary PSDU is an A-MPDU chain the A-THD
                            *sec_user_ptr++ = CPU2HW(&agg_desc->a_thd);
                        }
                        else
                        {
                            thd = &txdesc->lmac.hw_desc->thd;

                            // Otherwise chain the Singleton THD
                            *sec_user_ptr++ = CPU2HW(&txdesc->lmac.hw_desc->thd);
                        }

                        // Chain the BAR THD to the previous BAR THD
                        bar_thd->nextfrmexseq_ptr = CPU2HW(&agg_desc->bar_thd);
                        bar_thd = &agg_desc->bar_thd;

                        // Initialize the THD fields
                        thd->phyctrlinfo = (i << USER_POS_OFT) | USE_MUMIMO_TX_BIT;

                        // The user queue index will map to the secondary index
                        user_q_idx = sec_idx++;

                        // Point the secondary AGG desc to the primary one
                        agg_desc->prim_agg_desc = prim_agg_desc;
                        agg_desc->status |= AGG_INT;

                        // Put secondary aggregate in used list
                        co_list_push_back(&txlist->aggregates, &agg_desc->list_hdr);
                    }
                    else
                    {
                        // The primary A-MPDU has the queue index 0
                        user_q_idx = 0;
                    }

                    // Indicate that some payload download are expected on this user index
                    prim_agg_desc->download |= CO_BIT(user_q_idx);

                    // Check if we can allocate TX buffers for the MPDUs of this user
                    while (txlist->first_to_download[user_q_idx] == NULL)
                    {
                        if (!txl_payload_alloc(txdesc, ac, user_q_idx))
                            break;

                        txdesc = tx_desc_next(txdesc);
                        if (txdesc == NULL)
                            break;
                    }
                    // Concatenate the MU-MIMO TX descriptor list with the transmitting list
                    co_list_concat(&txlist->transmitting[user_q_idx], &mumimo->tx[i]);
                }
            }
            // Reset the intermediate status of the last AGG descriptor
            agg_desc->status &= ~AGG_INT;
            // Save the last user BAR THD
            prim_agg_desc->last_bar_thd = bar_thd;
        }
        else
        {
            // The primary user is not an A-MPDU, so chain all the packets to the default
            // user queue (i.e 0)
            for (i = 0; i < RW_USER_MAX; i++)
            {
                if (mumimo->users & CO_BIT(i))
                {
                    // Convert the MU-MIMO A-MPDU or MPDU into a SU one
                    txl_mumimo_convert(ac, i);
                }
            }
        }
    }
    else
    {
        // The primary user is not an A-MPDU, so chain all the packets to the default
        // user queue (i.e 0)
        for (i = 0; i < RW_USER_MAX; i++)
        {
            if (mumimo->users & CO_BIT(i))
            {
                // Convert the MU-MIMO A-MPDU or MPDU into a SU one
                txl_mumimo_convert(ac, i);
            }
        }
    }

    // Clear the MU-MIMO preparation structure
    mumimo->nb_users = 0;
    mumimo->users = 0;
    mumimo->first_user_pos = (mumimo->first_user_pos + 1) % RW_USER_MAX;
    mumimo->open = MU_USER_MASK;

    // If IPC queues were paused and packets saved, handle them now
    for (i = 0; i < RW_USER_MAX; i++)
    {
        int pos = (mumimo->first_user_pos + i) % RW_USER_MAX;
        struct txdesc *txdesc = mumimo->txdesc[pos];

        if (txdesc != NULL)
        {
            int status;

            mumimo->txdesc[pos] = NULL;

            status = txl_agg_push_mpdu(txdesc, ac);

            if (status == SU_PACKET)
            {
                // Check if we can try to allocate a Tx buffer
                if (txlist->first_to_download[0] == NULL)
                {
                    txl_payload_alloc(txdesc, ac, 0);
                }
                co_list_push_back(&txlist->transmitting[0], &txdesc->list_hdr);
            }
        }
    }

    // Reenable APP queues
    macif_tx_enable_users(ac, mumimo->open);

    PROF_MU_PPDU_CLOSE_CLR();
}
#endif // RW_MUMIMO_TX_EN

/**
 ****************************************************************************************
 * @brief Calculate number of blank delimiters to ensure MMSS after a THD
 * This number will be set in the THD of the NEXT txdesc in AMPDU
 *
 * @param[in] thd   Pointer to THD whose length number of blank delimiters is calculated
 * @param[in] mmss_bytes Minimal number of bytes in the A-MPDU subframe
 *
 * @return Number of blank delimiters to ensure Min MPDU Start Spacing
 ****************************************************************************************
 */
uint16_t txl_agg_mpdu_nb_delims(struct tx_hd *thd, uint16_t mmss_bytes)
{
    uint16_t nb_delims = 0;
    uint16_t subfrm_len = txl_mpdu_subframe_len(thd);

    //calculate nb delimiters
    if (subfrm_len < mmss_bytes)
    {
        //keep only byte length which must be covered by blank delimiters
        mmss_bytes -= subfrm_len;

        //not a multiple of 4 bytes(blank delim length) -> add 1 after division to ensure space is reached
        nb_delims = (mmss_bytes + DELIMITER_LEN - 1) / DELIMITER_LEN;
    }
    return nb_delims;
}

void txl_agg_init(void)
{
    // Initialize the HW descriptors linked to aggregation
    txl_agg_hwdesc_init();
}

void txl_agg_reset(void)
{
    // Reset the HW descriptors linked to aggregation
    txl_agg_hwdesc_reset();
}

void txl_agg_finish(uint8_t ac)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[ac];
    struct txl_agg_build_tag *agg = &txlist->agg[0];
    struct tx_agg_desc *agg_desc = agg->desc;
    struct txdesc *txdesc = agg->txdesc_last;

    // For profiling
    PROF_AGG_FINISH_AMPDU_SET();

    // Check if the A-MPDU has only one MPDU
    if (agg->curr_cnt == 1)
    {
        struct tx_hd *thd = &txdesc->lmac.hw_desc->thd;
        struct txl_buffer_tag *buffer = txl_buffer_get(txdesc);

        // In such case, transform it to a singleton MPDU
        txdesc->umac.flags = WHICHDESC_UNFRAGMENTED_MSDU;

        // Clear the next_mpdu field (that was pointing to the BAR THD)
        thd->nextmpdudesc_ptr = 0;
        // Update the MAC Control Information 2 field to the correct value
        thd->macctrlinfo2 = WHICHDESC_UNFRAGMENTED_MSDU | INTERRUPT_EN_TX;

        // If buffer is already allocated for this descriptor, update the MAC_CONTROL_INFO2 field
        if (buffer != NULL)
        {
            struct txl_buffer_control *bufctrl = &buffer->buffer_control;
            struct tx_policy_tbl *pol = &bufctrl->policy_tbl;

            // Check if the MPDU is already downloaded
            if (agg_desc->status & AGG_FIRST_DOWNLOADED)
            {
                // Set phy control info
                thd->phyctrlinfo = bufctrl->phy_control_info;
                // Set policy table address
                thd->policyentryaddr = CPU2HW(pol);
                // Set MAC control info 1 field
                thd->macctrlinfo1 = bufctrl->mac_control_info;

                #if RW_BFMER_EN
                if (bfr_is_enabled() && !bfr_is_bfmed_sglt_allowed(txdesc))
                {
                    // The policy table attached to this packet may not be all compliant
                    // with beamforming (because of the additional rate steps), so we
                    // disable the beamforming for this frame
                    pol->phycntrlinfo2 &= ~(BMFED_BIT | SMM_INDEX_PT_MASK);

                    /*
                     * As frame is not beamformed, the SMM Index field indicates the
                     * spatial mapping in the modem:
                     *      0 = Stream 0 on antenna 0 and stream 1 on antenna 1
                     *      2 = Stream mixed on antenna 0 and antenna 1
                     */
                    pol->phycntrlinfo2 |= 2 << SMM_INDEX_PT_OFT;
                }
                #endif
                #if NX_MAC_HE
                // Remove the +HTC field if required
                if (!is_htc_sglt_allowed(txdesc))
                    txl_buffer_remove_htc(thd);
                #endif

                #if NX_AMSDU_TX
                // Check if this packet is split across multiple buffers
                if (txl_buffer_is_amsdu_multi_buf(txdesc))
                {
                    // Split packet - Disable HW retransmissions
                    pol->maccntrlinfo2 &= ~(LONG_RETRY_LIMIT_MASK | SHORT_RETRY_LIMIT_MASK);
                }
                #endif

                #if TRACE_COMPO(LMAC)
                {
                    struct mac_hdr *mac_hdr = (struct mac_hdr *)HW2CPU(txl_buffer_machdr_get(txdesc));
                    TRACE_LMAC(TX, "{VIF-%d} to STA-%d %fc SN=%d (A-MDPU to singleton)",
                               txdesc->host.vif_idx, txdesc->host.staid, mac_hdr->fctl,
                               (mac_hdr->seq >> MAC_SEQCTRL_NUM_OFT));
                }
                #endif // TRACE_COMPO(LMAC)

                #if NX_MAC_HE
                // Check if the singleton shall be protected with a RTS/CTS
                txl_he_txop_dur_rtscts_thres_mpdu_check(txdesc);
                #endif

                // Chain the THD to the MAC HW
                txl_frame_exchange_chain(thd, thd, ac);
            }
        }
        // Put back the aggregate in the free list
        txl_agg_desc_free(agg_desc);

        // Unchain the aggregate descriptor from the TX descriptor
        txdesc->lmac.agg_desc = NULL;
    }
    else
    {
        struct tx_hd *thd = &txdesc->lmac.hw_desc->thd;
        struct txl_buffer_tag *buffer = txl_buffer_get(txdesc);
        #if NX_UMAC_PRESENT
        struct tx_cfm_tag *cfm = txl_cfm_tag_get(txdesc);
        #endif

        // Re-mark last MPDU as LAST not MIDDLE
        set_mpdu_pos(txdesc, WHICHDESC_AMPDU_LAST);

        #if NX_MAC_HE
        agg_desc->txdesc_last = txdesc;
        txl_he_txop_dur_rtscts_thres_ampdu_check(agg->txdesc_first);
        #endif

        #if NX_UMAC_PRESENT
        cfm->ampdu_size = agg->curr_cnt;
        #endif

        //BAR DESCRIPTOR
        txl_fill_bar(agg_desc, txdesc->umac.sn_win, agg_desc->sta_idx, agg_desc->tid, agg->bw);

        // Chain the BAR descriptor to the last MPDU
        thd->nextmpdudesc_ptr = CPU2HW(&agg_desc->bar_thd);

        // And update the MAC Control Information 2 field to the correct value
        thd->macctrlinfo2 = txdesc->umac.flags | INTERRUPT_EN_TX;

        // Current aggregate is now completed
        agg_desc->status |= AGG_FORMATTED;

        // If buffer is already allocated for this descriptor, check if it is
        // already downloaded or not
        if (buffer != NULL)
        {
            // Check if enough data is already allocated
            if (!(agg_desc->status & AGG_ALLOC))
            {
                // If this is not the case, check if the first MPDU is already downloaded
                if (!(agg_desc->status & AGG_FIRST_DOWNLOADED))
                {
                    // The first MPDU is not downloaded, so we will wait for its DMA
                    // interrupt and do the A-MPDU chaining at that moment
                    struct txl_buffer_tag *buffer_first = txl_buffer_get(agg->txdesc_first);
                    buffer_first->flags |= BUF_ALLOC_OK;
                }
                else
                    // The first MPDU is already downloaded, so we consider that we can
                    // chain the A-MPDU immediately
                    agg_desc->status |= AGG_DOWNLOADED;
            }
        }

        // Check if aggregate descriptor is ready to be programmed
        if (agg_desc->status & AGG_DOWNLOADED)
        {
            struct tx_hd *next_prev_hd = &agg_desc->bar_thd;
            #if NX_MAC_HE
            // Re-set the PHY flags in the policy table as they might have been changed
            // during the TXOP duration RTS/CTS threshold check
            struct tx_policy_tbl *pt = HW2CPU(agg_desc->a_thd.policyentryaddr);
            pt->ratecntrlinfo[0] = agg->txdesc_first->umac.phy_flags;
            #endif

            #if TRACE_COMPO(LMAC)
            struct tx_hd *first_thb = (struct tx_hd *)HW2CPU(agg_desc->a_thd.nextmpdudesc_ptr);
            struct tx_pbd *first_pbd = (struct tx_pbd *)HW2CPU(first_thb->first_pbd_ptr);
            struct mac_hdr *mac_hdr = (struct mac_hdr *)HW2CPU(first_pbd->datastartptr);
            TRACE_LMAC(TX, "[AC %d] {VIF-%d} to STA-%d AMPDU (#%d, %lu bytes) SN=%d",
                       ac, txdesc->host.vif_idx, txdesc->host.staid, agg->curr_cnt,
                       TR_32(agg_desc->a_thd.frmlen), (mac_hdr->seq >> MAC_SEQCTRL_NUM_OFT));
            #endif // TRACE_COMPO(LMAC)

            // Chain the THD to the MAC HW
            txl_frame_exchange_chain(&agg_desc->a_thd, next_prev_hd, ac);
        }

        // Put aggregate in used list
        co_list_push_back(&txl_cntrl_env.txlist[ac].aggregates, &agg_desc->list_hdr);
    }

    // We will now have some data programmed for transmission, so we can activate the A-MPDU
    // optimization
    txl_cntrl_env.txlist[ac].ppdu_cnt++;

    // Reinitialize the build state machine
    agg->desc = NULL;

    // For profiling
    PROF_AGG_FINISH_AMPDU_CLR();
}

int txl_agg_push_mpdu(struct txdesc * txdesc, uint8_t ac)
{
    struct tx_hd *thd = &txdesc->lmac.hw_desc->thd;
    uint8_t user_pos = 0;
    int status = SU_PACKET;

    //make local copy of staid and tid of MPDU given by UMAC
    uint8_t  tid   = txdesc->host.tid;
    uint8_t  staid = txdesc->host.staid;

    do
    {
        struct txl_list *txlist = &txl_cntrl_env.txlist[ac];
        struct txl_agg_build_tag *agg;

        #if RW_MUMIMO_TX_EN
        status = txl_mumimo_check(txdesc, ac, &user_pos);

        // Check if the operation on this queue shall be paused
        if (status == MU_PAUSED)
            break;

        // Check if the MU checking has to be restarted
        if (status == MU_RESTART_CHECK)
            continue;
        #endif

        // Retrieve the A-MPDU construction structure for this user position
        agg = &txlist->agg[user_pos];

        // Check if we are currently building an aggregate
        if (agg->desc == NULL)
        {
            // No build ongoing, try to start a new one
            // Make sure the MPDU can be part of an aggregate
            if (!is_mpdu_agg(txdesc)
                #if NX_UMAC_PRESENT
                || !txl_ampdu_has_ht_vht_rate(txdesc)
                #endif
                )
            {
                // We will now have some data programmed for transmission, so we can
                // activate the A-MPDU optimization
                txl_cntrl_env.txlist[ac].ppdu_cnt++;
                break;
            }

            // Get a free aggregate descriptor
            agg->desc = txl_agg_desc_alloc(ac);

            // No more free aggregate descriptors, handle the MPDU as a normal one
            if (agg->desc == NULL)
            {
                #if RW_MUMIMO_TX_EN
                // Check if a MU-MIMO PPDU is under construction. If yes we need
                // to close it
                if (txlist->mumimo.users)
                {
                    txl_agg_mumimo_close(ac);
                }
                // This packet cannot be part of a MU-MIMO PPDU as no AGG descriptor
                // is available
                status = SU_PACKET;
                #endif
                //mark it as non aggregatable for the rest of the procedure
                txdesc->umac.flags = WHICHDESC_UNFRAGMENTED_MSDU;
                // We will now have some data programmed for transmission, so we can
                // activate the A-MPDU optimization
                txl_cntrl_env.txlist[ac].ppdu_cnt++;
                break;
            }

            // For profiling
            PROF_AGG_START_AMPDU_SET();

            // Initialize aggregate descriptor
            agg->desc->status = 0;
            agg->desc->available_len = 0;
            agg->desc->available_cnt = 0;
            agg->desc->sta_idx = staid;
            agg->desc->tid = tid;
            agg->desc->user_cnt = 1;
            #if RW_MUMIMO_TX_EN
            agg->desc->download = 0;
            agg->desc->prim_agg_desc = NULL;
            agg->desc->last_bar_thd = NULL;
            co_list_init(&agg->desc->cfm);
            #endif

            // Get the constraints of the A-MPDU based on the PHY rate that will be used
            // and the receiving STA
            txl_ampdu_constraints_get(txdesc, ac, agg);

            // For profiling
            PROF_AGG_START_AMPDU_CLR();

            // For profiling
            PROF_AGG_ADD_MPDU_SET();

            // Initialize the current A-MPDU length
            agg->desc->a_thd.frmlen = txl_mpdu_subframe_len(thd);
            //reset A-THD status
            agg->desc->a_thd.statinfo = 0;
            agg->desc->a_thd.nextmpdudesc_ptr = CPU2HW(thd);
            agg->desc->a_thd.first_pbd_ptr = 0;
            agg->desc->a_thd.optlen[0] = 0;
            agg->desc->a_thd.optlen[1] = 0;
            agg->desc->a_thd.optlen[2] = 0;

            // Initialize the current A-MPDU count
            agg->curr_cnt = 1;

            // Initialize the number of delimiters to be put in the next HW descriptor
            agg->nb_delims = txl_agg_mpdu_nb_delims(thd, agg->mmss_bytes);
            #if NX_MAC_HE
            //set number of delimiters
            txdesc->umac.flags |= (agg->nb_delims << NB_BLANK_DELIM_OFT);
            agg->desc->a_thd.frmlen += agg->nb_delims * DELIMITER_LEN;
            #endif

            // Save the pointer to the TX descriptor
            agg->txdesc_last = txdesc;
            agg->txdesc_first = txdesc;

            //mark the txdesc as first of AMPDU
            set_mpdu_pos(txdesc, WHICHDESC_AMPDU_FIRST);

            //link txdesc to agg_cntrl
            txdesc->lmac.agg_desc = agg->desc;
            txdesc->umac.flags |= UNDER_BA_SETUP_BIT;

            //set WhichDesc field in the THD
            thd->macctrlinfo2 = txdesc->umac.flags;

            #if RW_MUMIMO_TX_EN
            if (status == MU_PACKET)
            {
                PROF_MU_MPDU_ADD_SET();
                co_list_push_back(&txlist->mumimo.tx[user_pos], &txdesc->list_hdr);
                // Remember that this user position has an A-MPDU under creation on it
                txlist->mumimo.users |= CO_BIT(user_pos);
                txlist->mumimo.rateinfo[user_pos] = txdesc->umac.phy_flags;
                agg->desc->status |= AGG_MU;
                PROF_MU_MPDU_ADD_CLR();
            }
            #endif

            // For profiling
            PROF_AGG_ADD_MPDU_CLR();
            break;
        }
        else
        {
            struct tx_agg_desc *agg_desc = agg->desc;
            struct tx_hd *a_thd = &agg_desc->a_thd;
            struct tx_hd *prev_thd = &agg->txdesc_last->lmac.hw_desc->thd;
            uint32_t ampdu_subfrmlen;
            #if NX_BW_LEN_ADAPT
            uint8_t bw_idx = agg->bw_idx;
            #else
            uint8_t bw_idx = 0;
            #endif

            // txdesc must be compatible with current AMPDU
            if (!txl_desc_is_agg(txdesc, agg_desc))
            {
                #if RW_MUMIMO_TX_EN
                // Close this user position
                if (agg_desc->status & AGG_MU)
                {
                    MU_AMPDU_CLOSE();
                }
                else
                {
                #endif
                    // Close the A-MPDU under construction
                    txl_agg_finish(ac);

                    // And try to start a new one
                    continue;
                #if RW_MUMIMO_TX_EN
                }
                #endif
            }

            #if NX_MAC_HE
            agg->nb_delims = txl_agg_mpdu_nb_delims(thd, agg->mmss_bytes);
            #endif

            // Compute the MPDU subframe length (including delimiters)
            ampdu_subfrmlen = txl_mpdu_subframe_len(thd) + agg->nb_delims * DELIMITER_LEN;

            //check for AMPDU max len
            if ((ampdu_subfrmlen + a_thd->frmlen) > agg->max_len[bw_idx])
            {
                #if NX_BW_LEN_ADAPT
                if (bw_idx < agg->bw)
                {
                    // Check if we have at least 2 MPDUs inside the A-MPDU to ensure that
                    // in case of BW drop at lowest BW the transmitted packet will still
                    // be an A-MPDU. We also ensure that we can go to the next BW index,
                    // i.e that the new MPDU will still fit in the A-MPDU if transmitted
                    // at the next BW
                    if ((agg->curr_cnt == 1)
                     || ((ampdu_subfrmlen + a_thd->frmlen) > agg->max_len[bw_idx+1]))
                    {
                        #if RW_MUMIMO_TX_EN
                        // Close this user position
                        if (agg_desc->status & AGG_MU)
                        {
                            MU_AMPDU_CLOSE();
                        }
                        else
                        {
                        #endif
                            // Close the A-MPDU under construction
                            txl_agg_finish(ac);
                            // And try to start a new one
                            continue;
                        #if RW_MUMIMO_TX_EN
                        }
                        #endif
                    }

                    // For profiling
                    PROF_BW_DROP_STEP_SET();

                    // Save the last TX descriptor of this BW step
                    agg_desc->txdesc[bw_idx] = agg->txdesc_last;

                    // Update the THD with the optional length of this BW step
                    a_thd->optlen[bw_idx] = a_thd->frmlen;

                    // Increase the current BW index
                    agg->bw_idx++;

                    // For profiling
                    PROF_BW_DROP_STEP_CLR();
                }
                else
                #endif
                {
                    // Maximum length reached, finish current A-MPDU
                    #if RW_MUMIMO_TX_EN
                    // Close this user position
                    if (agg_desc->status & AGG_MU)
                    {
                        MU_AMPDU_CLOSE();
                    }
                    else
                    {
                    #endif
                        // Close the A-MPDU under construction
                        txl_agg_finish(ac);
                        // And try to start a new one
                        continue;
                    #if RW_MUMIMO_TX_EN
                    }
                    #endif
                }
            }

            // For profiling
            PROF_AGG_ADD_MPDU_SET();

            #if RW_MUMIMO_TX_EN
            // Check if the packet will be effectively part of a MU-PPDU or not
            if (!(agg_desc->status & AGG_MU))
            {
                status = SU_PACKET;
            }
            #endif

            //link txdesc to agg_cntrl
            txdesc->lmac.agg_desc = agg_desc;

            //set number of delims for previous txdesc into this one
            txdesc->umac.flags |= (agg->nb_delims << NB_BLANK_DELIM_OFT) | UNDER_BA_SETUP_BIT;

            //mark the txdesc as part of AMPDU
            set_mpdu_pos(txdesc, WHICHDESC_AMPDU_INT);

            //set WhichDesc and delimiter fields in the THD
            thd->macctrlinfo2 = txdesc->umac.flags;

            //chain the new THD to the previous one
            prev_thd->nextmpdudesc_ptr = CPU2HW(thd);

            //increase ampdu_desc length value
            a_thd->frmlen += ampdu_subfrmlen;

            #if !NX_MAC_HE
            // Compute the number of delimiters to be put in the next HW descriptor
            agg->nb_delims = txl_agg_mpdu_nb_delims(thd, agg->mmss_bytes);
            #endif

            // Save the pointer to the last handled TX descriptor
            agg->txdesc_last = txdesc;

            #if RW_MUMIMO_TX_EN
            if (status == MU_PACKET)
            {
                PROF_MU_MPDU_ADD_SET();
                co_list_push_back(&txlist->mumimo.tx[user_pos], &txdesc->list_hdr);
                PROF_MU_MPDU_ADD_CLR();
            }
            #endif

            // For profiling
            PROF_AGG_ADD_MPDU_CLR();

            //increase nb MPDUs in AMPDU counter
            agg->curr_cnt++;

            // Check if we have reached the maximum number of packets
            if (agg->curr_cnt >= agg->max_cnt)
            {
                // We have reached the maximum MPDU count, finish current A-MPDU
                #if RW_MUMIMO_TX_EN
                if (status == MU_PACKET)
                {
                    txl_mumimo_ampdu_finish(ac, user_pos);
                    // Check if other user positions are still open
                    if (txlist->mumimo.open & txlist->mumimo.users)
                    {
                        // We could still receive packets on other user positions,
                        // so pause this user queue
                        status = MU_PAUSED;
                    }
                    else
                    {
                        // Close the MU-MIMO PPDU under construction
                        txl_agg_mumimo_close(ac);
                    }
                }
                else
                #endif
                    txl_agg_finish(ac);
            }

            // Exit the loop
            break;
        }
    } while (1);

    return (status);
}


void txl_agg_release(struct tx_agg_desc *agg_desc)
{
    // Decrease the user count
    agg_desc->user_cnt--;

    // Free the descriptor
    if (agg_desc->user_cnt == 0)
    {
        txl_agg_desc_free(agg_desc);
    }
}
struct tx_hd *txl_agg_change_to_singleton(struct txdesc *txdesc, bool he_tb)
{
    struct tx_hd *thd = &txdesc->lmac.hw_desc->thd;
    struct txl_buffer_tag *buffer = txl_buffer_get(txdesc);

    // In such case, transform it to a singleton MPDU
    txdesc->umac.flags = WHICHDESC_UNFRAGMENTED_MSDU;

    // Unchain the aggregate descriptor from the TX descriptor
    txdesc->lmac.agg_desc = NULL;

    // Clear the next_mpdu field
    thd->nextmpdudesc_ptr = 0;

    // Update the MAC Control Information 2 field to the correct value
    thd->macctrlinfo2 = WHICHDESC_UNFRAGMENTED_MSDU | INTERRUPT_EN_TX;

    if (buffer)
    {
        struct txl_buffer_control *bufctrl = &buffer->buffer_control;

        #if NX_UMAC_PRESENT
        // Copy the buffer control
        txl_buffer_control_copy(txdesc, buffer);
        #if NX_MAC_HE
        // Remove the +HTC field if required
        if (!he_tb && !is_htc_sglt_allowed(txdesc))
            txl_buffer_remove_htc(thd);
        #endif
        #endif

        #if NX_MAC_HE
        // Mark the buffer as ready to be sent as HE TB singleton if required
        buffer->flags |= BUF_SINGLETON_READY;
        // Check if the singleton shall be protected with a RTS/CTS
        txl_he_txop_dur_rtscts_thres_mpdu_check(txdesc);
        #endif

        // Set MAC control info 1 field
        thd->macctrlinfo1 = bufctrl->mac_control_info;
        // Set phy control info
        thd->phyctrlinfo = bufctrl->phy_control_info;
        // Set policy table address
        thd->policyentryaddr = CPU2HW(&bufctrl->policy_tbl);
        #if NX_AMSDU_TX
        // Check if this packet is split across several buffers
        if (txl_buffer_is_amsdu_multi_buf(txdesc))
        {
            struct tx_policy_tbl *pol = &bufctrl->policy_tbl;

            // Split packet - Disable HW retransmissions
            pol->maccntrlinfo2 &= ~(LONG_RETRY_LIMIT_MASK | SHORT_RETRY_LIMIT_MASK);
        }
        #endif

        return (thd);
    }

    return NULL;
}

#if (NX_BW_LEN_ADAPT)
void txl_agg_bw_drop_handle(uint8_t access_category)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];
    struct txdesc * txdesc = (struct txdesc *)co_list_pick(&(txlist->transmitting[0]));
    struct tx_agg_desc *agg_desc;
    struct tx_hd *thd, *prev_thd, *next_thd;
    uint8_t bw;

    // Sanity check - we shall have a descriptor in the list
    ASSERT_ERR(txdesc != NULL);

    // Check if the HW triggered the mechanism for a singleton - maybe not correct
    while (!is_mpdu_agg(txdesc))
    {
        struct tx_hd *txhd = &txdesc->lmac.hw_desc->thd;
        uint32_t txstatus = txhd->statinfo;
        if (!(txstatus & DESC_DONE_TX_BIT))
            return;
        txdesc = tx_desc_next(txdesc);

        if (txdesc == NULL)
            return;
    }

    // Sanity check - the mechanism works only for A-MPDUs
    ASSERT_ERR(is_mpdu_first(txdesc));

    // Get the A-MPDU descriptor pointer
    agg_desc = txdesc->lmac.agg_desc;

    // Get the actual BW of transmission and the last TX desc that will be transmitted
    bw = nxmac_tx_bw_after_drop_getf();
    txdesc = agg_desc->txdesc[bw];
    thd = &txdesc->lmac.hw_desc->thd;

    ASSERT_ERR(!is_mpdu_last(txdesc));

    // Split the A-MPDU
    // Re-mark last MPDU as LAST not MIDDLE
    set_mpdu_pos(txdesc, WHICHDESC_AMPDU_LAST);

    // Chain the BAR descriptor to the last MPDU
    thd->nextmpdudesc_ptr = CPU2HW(&agg_desc->bar_thd);

    // And update the MAC Control Information 2 field to the correct value
    thd->macctrlinfo2 = txdesc->umac.flags | INTERRUPT_EN_TX;

    prev_thd = &agg_desc->bar_thd;
    next_thd = HW2CPU(agg_desc->bar_thd.nextfrmexseq_ptr);

    // Now reform an A-MPDU with the following descriptors
    txdesc = tx_desc_next(txdesc);
    ASSERT_ERR(txdesc != NULL);
    while (1)
    {
        bool last = false;

        if (is_mpdu_last(txdesc))
            last = true;

        // Increase the number of PPDU we have pushed into the data path
        txlist->ppdu_cnt++;

        // Change the A-MPDU member into a singleton frame
        thd = txl_agg_change_to_singleton(txdesc, false);

        if (thd)
        {
            prev_thd->nextfrmexseq_ptr = CPU2HW(thd);

            if (txl_cntrl_env.txlist[access_category].last_frame_exch == prev_thd)
                txl_cntrl_env.txlist[access_category].last_frame_exch = thd;

            prev_thd = thd;
        }

        if (last)
        {
            thd->nextfrmexseq_ptr = CPU2HW(next_thd);
            break;
        }

        txdesc = tx_desc_next(txdesc);
        ASSERT_ERR(txdesc != NULL);
    }
}
#endif

void txl_agg_check(uint8_t access_category)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];
    GLOBAL_INT_DISABLE();
    if (txlist->ppdu_cnt == 0)
    {
        #if RW_MUMIMO_TX_EN
        // Check if a MU-MIMO PPDU is under preparation
        if (txlist->mumimo.users)
        {
            txl_agg_mumimo_close(access_category);
        }
        else
        #endif
        if (txlist->agg[0].desc != NULL)
        {
            // Close the pending aggregate
            txl_agg_finish(access_category);
        }
    }
    GLOBAL_INT_RESTORE();
}


#if RW_MUMIMO_TX_EN
void txl_agg_sec_transmit_trigger(void)
{
    // For profiling
    PROF_MU_SEC_USER_IRQ_SET();

    while (nxmac_sec_users_tx_int_event_get() & TX_SEC_IRQ_BITS)
    {
        uint8_t access_category;
        uint8_t user_idx;
        uint8_t highest;
        uint32_t status = nxmac_sec_users_tx_int_event_get() & TX_SEC_IRQ_BITS;

        // sanity check: we cannot be handling a TX interrupt if there isn't one pending
        ASSERT_REC(status != 0);

        // don't care about tx header or buffer trigger, only access category and user index,
        // so merge trigger and buffer status
        status |= status >> (NXMAC_SEC_U_1AC_0_TX_BUF_TRIGGER_POS - NXMAC_SEC_U_1AC_0_TX_TRIGGER_POS);

        // Now proceed only the merged bits
        status &= TX_SEC_IRQ_BITS_MERGED;

        // Get the highest bit set
        highest =  31 - co_clz(status);

        // Compute the corresponding access category and secondary user index
        access_category = highest % 4;
        user_idx = highest / 8;

        // sanity checks
        ASSERT_ERR(access_category < NX_TXQ_CNT);
        ASSERT_ERR(user_idx < RW_USER_MAX);

        // clear the handled interrupts (buffer and TX)
        nxmac_sec_users_tx_int_event_clear(CO_BIT(user_idx*8 + access_category + NXMAC_SEC_U_1AC_0_TX_BUF_TRIGGER_POS) |
                                           CO_BIT(user_idx*8 + access_category + NXMAC_SEC_U_1AC_0_TX_TRIGGER_POS));

        // For profiling
        PROF_TX_AC_IRQ_SET(access_category);

        // Go through the list of already transmitted packets to release them
        txl_mumimo_secondary_done(access_category, user_idx + 1);
    }

    // For profiling
    PROF_MU_SEC_USER_IRQ_CLR();
}
#endif

void txl_agg_set_ampdu_protection(uint32_t *rc)
{
    uint8_t bw, rate, mcs;
    uint32_t rc_loc = *rc;

    // Check if the PPDU is already protected (e.g ERP protection) - In such case we rely
    // on this protection
    if ((rc_loc & PROT_FRM_EX_RCX_MASK) != PROT_NO_PROT)
        return;

    // Retrieve BW and MCS of the PPDU in order to compute the protection parameters
    bw = (rc_loc & BW_TX_RCX_MASK) >> BW_TX_RCX_OFT;
    mcs = ((rc_loc & FORMAT_MOD_TX_RCX_MASK) >> FORMAT_MOD_TX_RCX_OFT) >= FORMATMOD_VHT ?
                                             rc_loc & VHT_MCS_MASK : rc_loc & HT_MCS_MASK;
    rate = mcs > 2 ? HW_RATE_24MBPS : HW_RATE_6MBPS;
    rc_loc &= ~(PROT_FRM_EX_RCX_MASK | MCS_INDEX_PROT_TX_RCX_MASK | BW_PROT_TX_RCX_MASK |
                                                           FORMAT_MOD_PROT_TX_RCX_MASK);
    rc_loc |= (bw << BW_PROT_TX_RCX_OFT) | (rate << MCS_INDEX_PROT_TX_RCX_OFT) | PROT_SELF_CTS;
    rc_loc |= FORMATMOD_NON_HT_DUP_OFDM << FORMAT_MOD_PROT_TX_RCX_OFT;

    *rc = rc_loc;
}

void txl_agg_check_rtscts_retry_limit(struct tx_hd *a_thd, uint8_t access_category)
{
    uint32_t a_thd_status = a_thd->statinfo;

    // Check if A-MPDU descriptor is done
    if (a_thd_status & DESC_DONE_TX_BIT)
    {
        struct tx_policy_tbl *pol = HW2CPU(a_thd->policyentryaddr);

        // Sanity check - the only valid case coming here corresponds
        // to an RTS/CTS frame exchanged that failed
        ASSERT_REC(a_thd_status & RETRY_LIMIT_REACHED_BIT);

        // Reset the status of the A-MPDU descriptor
        a_thd->statinfo = 0;

        // Change the protection to CTS-to-self to ensure that the A-MPDU
        // will be transmitted on next attempt
        pol->ratecntrlinfo[0] &= ~PROT_FRM_EX_RCX_MASK;
        pol->ratecntrlinfo[0] |= PROT_SELF_CTS;

        // Reprogram the A-MPDU in TX (by setting the newHead bit)
        txl_cntrl_newhead(CPU2HW(a_thd), access_category);
    }
}

void txl_agg_check_saved_agg_desc(uint8_t access_category)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];
    struct tx_agg_desc *agg_desc_prev = txlist->agg_desc_prev;

    // Check if we have to free the previous A-MPDU descriptor
    if (agg_desc_prev != NULL)
    {
        // Decrease the user count
        agg_desc_prev->user_cnt--;

        // Free the descriptor
        if (agg_desc_prev->user_cnt == 0)
        {
            txl_agg_desc_free(agg_desc_prev);
        }

        // No more previous A-MPDU descriptor to be considered for now
        txlist->agg_desc_prev = NULL;
    }
}

#if NX_MAC_HE
struct tx_hd *txl_agg_set_new_ampdu_head(struct txdesc *txdesc, struct tx_hd **bar_thd)
{
    struct tx_hd *thd = &txdesc->lmac.hw_desc->thd;
    struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;
    struct txl_buffer_tag *buffer = txl_buffer_get(txdesc);

    thd = &agg_desc->a_thd;

    // A-MPDU was split
    thd->nextmpdudesc_ptr = CPU2HW(&txdesc->lmac.hw_desc->thd);
    thd->policyentryaddr = CPU2HW(&agg_desc->pol_tbl);
    set_mpdu_pos(txdesc, WHICHDESC_AMPDU_FIRST);
    txdesc->lmac.hw_desc->thd.macctrlinfo2 &= ~WHICHDESC_MSK;
    txdesc->lmac.hw_desc->thd.macctrlinfo2 |= WHICHDESC_AMPDU_FIRST;

    if (buffer == NULL)
    {
        agg_desc->status = AGG_FORMATTED;
        agg_desc->available_len = 0;
        return NULL;
    }

    *bar_thd = &agg_desc->bar_thd;
    return thd;
}

struct tx_hd *txl_agg_he_tb_prep(struct txdesc *txdesc, struct txdesc **txdesc_next,
                                 uint32_t max_len, uint16_t min_mpdu_len, uint8_t ac)
{
    struct tx_agg_desc *agg_desc;
    struct tx_hd *thd = &txdesc->lmac.hw_desc->thd;
    struct txl_list *txlist = &txl_cntrl_env.txlist[ac];

    // First check if the MPDU to transmit is already part of a A-MPDU
    if (is_mpdu_agg(txdesc))
    {
        // A-MPDU case
        // Ensure the A-MPDU is ready for TX
        agg_desc = txdesc->lmac.agg_desc;
        if (!is_mpdu_first(txdesc) ||
            ((agg_desc->status & (AGG_FORMATTED | AGG_DOWNLOADED)) !=
                         (AGG_FORMATTED | AGG_DOWNLOADED)))
            return NULL;

        // Check if it can fit entirely in the HE TB PPDU or not
        if (agg_desc->a_thd.frmlen > max_len)
            // Cannot fit - split it
            return txl_agg_split(txdesc, txdesc_next, max_len, min_mpdu_len, ac);

        // Set the UPH on all the MPDUs that have already a buffer
        txl_agg_set_uph(txdesc);

        *txdesc_next = tx_desc_next(agg_desc->txdesc_last);
    }
    else
    {
        // MPDU case
        // Ensure the MPDU is ready for TX
        struct txl_buffer_tag *buffer = txl_buffer_get(txdesc);
        if (!buffer || !(buffer->flags & BUF_SINGLETON_READY))
            return NULL;

        // If not already done set the +HTC length and the order bit in the MPDU buffer
        if (!txl_agg_is_int_or_mgt(txdesc))
            txl_buffer_add_htc(thd);

        // Check if it can fit entirely in the HE TB PPDU or not
        if (thd->frmlen > max_len)
        {
            // Request to disable the MU EDCA because the HE TB queue is blocked
            txl_he_mu_edca_blocked(thd, ac);
            return NULL;
        }

        // Update the HTC field with the UPH
        if (!txl_agg_is_int_or_mgt(txdesc))
            txl_buffer_update_htc(thd, txl_he_tb_uph_get());

        *txdesc_next = tx_desc_next(txdesc);
    }

    // We will be able to push at least one MPDU in the HE TB, let's try to concatenate
    // it with next MPDUs if any
    while (*txdesc_next)
    {
        struct txdesc *txdesc_n = *txdesc_next;

        // Don't aggregate frames sent internally, and check if we still have time to
        // concatenate new MPDUs
        if (txl_agg_is_int_or_mgt(txdesc) || txl_agg_is_int_or_mgt(txdesc_n) ||
                                     (nxmac_tx_hetb_rem_dur_getf() < TX_HE_TB_PROG_TIME))
            break;

        if (is_mpdu_agg(txdesc))
        {
            if (!is_mpdu_agg(txdesc_n))
            {
                if (!txl_agg_he_tb_cat_ampdu_mpdu(txdesc, txdesc_next, max_len, min_mpdu_len, ac))
                    break;
            }
            else if (!txl_agg_he_tb_cat_ampdu(txdesc, txdesc_next, max_len, min_mpdu_len, ac))
                break;
        }
        else if (!is_mpdu_agg(txdesc_n) ||
                 !txl_agg_he_tb_cat_mpdu_ampdu(txdesc, txdesc_next, max_len, min_mpdu_len, ac))
            break;
    };

    // Check if HE TB will be a A-MPDU
    if (is_mpdu_agg(txdesc))
    {
        thd = &txdesc->lmac.agg_desc->a_thd;
        // Remove BAR THD from last MPDU
        txdesc->lmac.agg_desc->txdesc_last->lmac.hw_desc->thd.nextmpdudesc_ptr = 0;
        return thd;
    }

    // The HE TB PPDU will contain only one MPDU
    // Check if there is already a AGG descriptor allocated for it - In such case no need
    // to allocate a new one.
    if (txdesc->lmac.agg_desc)
        return thd;

    // Allocate an aggregate descriptor for the HE TB singleton
    agg_desc = txl_agg_desc_alloc(TX_HE_TB_AMPDU_QUEUE_IDX);
    ASSERT_ERR(agg_desc != NULL);

    // Initialize it
    agg_desc->status = 0;
    agg_desc->user_cnt = 1;

    // Assign it to the TX descriptor
    txdesc->lmac.agg_desc = agg_desc;

    // Push it at the front of the aggregate list
    co_list_push_front(&txlist->aggregates, &agg_desc->list_hdr);

    return thd;
}
#endif // NX_MAC_HE


#endif


/// @}  // end of group TX_CNTRL
