/**
 ****************************************************************************************
 *
 * @file txl_buffer.c
 *
 * @brief Management of Tx payload buffers
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup TX_BUFFER
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "co_int.h"
#include "co_bool.h"
//#include "co_list.h"
#include "txl_buffer.h"
#include "txl_cntrl.h"
#include "txl_agg.h"
#include "macif.h"
#include "dbg_assert.h"
//for profiling
#include "dbg.h"
#include "co_utils.h"
#if NX_UMAC_PRESENT
#include "txu_cntrl.h"
#endif
#if (RW_BFMER_EN)
#include "bfr.h"
#endif //(RW_BFMER_EN)

#if !NX_FULLY_HOSTED
/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
/// Environment of the Tx buffer management module
struct txl_buffer_env_tag txl_buffer_env;

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
#if RW_BFMER_EN
/**
 ****************************************************************************************
 * @brief Add the DMA descriptor used for the report download if required
 *
 * @param[in]  txdesc        TX Descriptor for which we need to allocate the buffer
 * @param[in]  dma_desc_bfr  Pointer to the DMA descriptor reserved for beamforming
 * @param[in]  dma_desc      Last DMA descriptor currently in the chain
 *
 * @return The pointer to the new DMA descriptor at the end of the chain (maybe unchanged)
 ****************************************************************************************
 */
static struct dma_desc *txl_buffer_bfr_dma_desc_conf(struct txdesc *txdesc,
                                                     struct dma_desc *dma_desc_bfr,
                                                     struct dma_desc *dma_desc)
{
    if (bfr_is_enabled() && bfr_tx_frame_ind(txdesc, dma_desc_bfr))
    {
        // Chain the DMA descriptor for the BFR
        dma_desc->next = CPU2HW(dma_desc_bfr);
        dma_desc = dma_desc_bfr;
    }

    return (dma_desc);
}
#endif


/**
 ****************************************************************************************
 * @brief Get the different parameters (head_len, number of pool elements to allocate,
 * etc.) for the current allocation.
 *
 * @param[in]  txdesc      TX Descriptor for which we need to allocate the buffer
 * @param[out] size        Amount of data to transfer via DMA after allocation
 * @param[out] head_len    Amount of space reserved at the beginning of buffer
 * @param[in]  pkt_idx     Indicates which payload is transfered when several payloads are
 *                         present in the TX descriptors (i.e. in case of A-MSDU). 0xFF is
 *                         a specific value indicating that the txdesc describes a small
 *                         A-MSDU that shall be fully allocated in a single buffer.
 *
 * @return The number of pool elements that need to be allocated
 ****************************************************************************************
 */
static uint32_t txl_buffer_get_params(struct txdesc *txdesc, uint16_t *size, uint16_t *head_len
                                      #if NX_AMSDU_TX
                                      , uint8_t pkt_idx
                                      #endif
                                     )
{
    uint32_t needed;
    uint16_t add_len;

    #if NX_UMAC_PRESENT
    #if NX_AMSDU_TX
    // Check if a single buffer has to be allocated for the A-MSDU
    if (pkt_idx == 0xFF)
    {
        struct tx_hd *txhd = &txdesc->lmac.hw_desc->thd;
        pkt_idx = 0;
        add_len = txdesc->umac.head_len + txdesc->umac.tail_len;
        *size = txhd->frmlen - add_len - MAC_FCS_LEN;
        *head_len = txdesc->umac.head_len;
    }
    else
    {
        add_len = 0;
        *head_len = 0;
        *size = txdesc->host.packet_len[pkt_idx];
        if (pkt_idx == 0)
        {
            add_len = txdesc->umac.head_len;
            *head_len = txdesc->umac.head_len;
        }
        if ((pkt_idx + 1) == txdesc->host.packet_cnt)
        {
            add_len += txdesc->umac.tail_len;
        }
    }
    #else // NX_AMSDU_TX
    add_len = txdesc->umac.head_len + txdesc->umac.tail_len;
    *head_len = txdesc->umac.head_len;
    *size = txdesc->host.packet_len;
    #endif // NX_AMSDU_TX
    #if NX_MFP
    // For Protected Management frames make sure we allocate all the frame in a
    // contiguous space
    if (txdesc->host.flags & TXU_CNTRL_MGMT_ROBUST)
    {
        *head_len = *size + add_len;
    }
    #endif // NX_MFP
    #else // NX_UMAC_PRESENT
    add_len = 0;
    *head_len = 0;
    #if NX_AMSDU_TX
    *size = txdesc->host.packet_len[pkt_idx];
    #else
    *size = txdesc->host.packet_len;
    #endif // NX_AMSDU_TX
    #endif // NX_UMAC_PRESENT

    needed = CO_ALIGN4_HI(*size + add_len + sizeof_b(struct txl_buffer_tag) +
                          TX_BUFFER_PADDING_MAX) / 4;

    return (needed);
}


/**
 ****************************************************************************************
 * @brief Perform the allocation in the buffer space based on the parameters.
 *
 * @param[in]  idx       Pointer to the TX buffer pool in which the allocation is done
 * @param[in]  needed    Number of pool elements to allocate
 * @param[in]  head_len  Amount of space reserved at the beginning of the buffer
 * @param[out] remaining Number of pool elements between the allocated buffer position
 *                       and the end of the buffer space.
 *
 * @return The pointer to the allocated buffer structure
 ****************************************************************************************
 */
static struct txl_buffer_tag *txl_buffer_get_space(struct txl_buffer_idx_tag *idx,
                                                   uint32_t needed, uint16_t head_len,
                                                   uint32_t *remaining)
{
    struct txl_buffer_tag *buf = NULL;
    uint32_t free = idx->free;
    uint32_t freesz = idx->free_size;
    uint32_t *pool = idx->pool;

    // Sanity check - The pool pointer shall not be NULL
    ASSERT_ERR(pool != NULL);
    ASSERT_ERR(freesz <= TX_BUFFER_POOL_SIZE);
    ASSERT_ERR(free <= TX_BUFFER_POOL_SIZE);

    do
    {
        // By default we consider that the allocation will fail
        idx->next_needed = needed;

        // Check if have enough free space
        if (idx->free_size < needed)
            break;

        // Check if contiguous free space is large enough to store the TX buffer tag
        if (((TX_BUFFER_POOL_SIZE - free) * 4) < (sizeof_b(struct txl_buffer_tag) + head_len + TX_BUFFER_PADDING_MAX + 1))
        {
            uint32_t add_len = TX_BUFFER_POOL_SIZE - free;

            // Move the free pointer and length
            freesz -= add_len;
            free = 0;

            // Extend the length of the lastly allocated buffer
            if (idx->last != TX_BUFFER_NULL)
            {
                struct txl_buffer_tag *last_buf = (struct txl_buffer_tag *)&pool[idx->last];
                last_buf->length += add_len;
                idx->used_area += add_len;
            }

            // Check if we now have enough free space
            if (freesz < needed)
            {
                // Not enough space, update the free space value and exit
                idx->free_size = freesz;
                idx->free = free;
                break;
            }
        }

        // Allocate the buffer
        buf = (struct txl_buffer_tag *)&pool[free];
        buf->length = needed;
        buf->flags = 0;
        idx->last = free;
        idx->next_needed = -1;
        freesz -= needed;
        *remaining = TX_BUFFER_POOL_SIZE - free;
        if (*remaining >= needed)
            free += needed;
        else
            free = needed - *remaining;

        // Update the free index and used area
        idx->free = free;
        idx->free_size = freesz;
        idx->used_area += needed;
        idx->buf_size += needed;
        if (idx->buf_size >= TX_BUFFER_MIN_SIZE)
        {
            // One more super buffer available
            idx->count++;
            // We set this buffer as a frontier
            buf->flags |= BUF_FRONTIER;
            // We start the new one
            idx->buf_size = 0;
        }
        else
        {
            // This buffer is not a frontier between 2 super buffers
            buf->flags &= ~BUF_FRONTIER;
        }
    } while (0);

    return (buf);
}

/**
 ****************************************************************************************
 * @brief Create the platform DMA descriptor list that will be used to perform the
 * download of the payload.
 *
 * @param[in]  txdesc    TX Descriptor for which we need to allocate the buffer
 * @param[in]  access_category TX queue index for the TX
 * @param[in]  buf       Pointer to the allocated buffer structure
 * @param[in]  idx       Pointer to the TX buffer pool in which the allocation is done
 * @param[in]  size      Amount of bytes that will need to be transfered using DMA
 * @param[in]  remaining Number of pool elements between the allocated buffer position
 *                       and the end of the buffer space (used to detect buffer wrapping).
 * @param[in]  head_len  Amount of space reserved at the beginning of the buffer
 * @param[in]  pkt_idx   Indicates which payload is transfered when several payloads are
 *                       present in the TX descriptors (i.e. in case of A-MSDU). 0xFF is
 *                       a specific value indicating that the txdesc describes a small
 *                       A-MSDU that shall be fully allocated in a single buffer.
 ****************************************************************************************
 */
static void txl_buffer_transfer(struct txdesc *txdesc, uint8_t access_category,
                                struct txl_buffer_tag *buf, struct txl_buffer_idx_tag *idx,
                                uint16_t size, uint32_t remaining, uint16_t head_len
                                #if NX_AMSDU_TX
                                , uint8_t pkt_idx
                                #endif
                                )
{
    struct dma_desc *dma_desc, *dma_desc_pat;
    struct tx_pbd *tbd;
    uint16_t dma_len, dma_oft;
    uint32_t dma_dest;
    uint32_t *pool = idx->pool;
    uint32_t packet_addr;
    uint32_t dma_len_before_wrap;
    #if NX_UMAC_PRESENT && NX_AMSDU_TX
    int packet_cnt;
    #endif
    int i = 0;

    #if NX_UMAC_PRESENT
    dma_oft = head_len + offsetof_b(struct txl_buffer_tag, payload);
    #if NX_AMSDU_TX
    // Check if A-MSDU is downloaded into a single buffer or not
    if (pkt_idx == 0xFF)
    {
        pkt_idx = 0;
        packet_addr = txdesc->host.packet_addr[0];
        dma_len = txdesc->host.packet_len[0];
        packet_cnt = txdesc->host.packet_cnt;
    }
    else
    {
        packet_addr = txdesc->host.packet_addr[pkt_idx];
        dma_len = size;
        packet_cnt = 1;
    }
    #else
    packet_addr = txdesc->host.packet_addr;
    dma_len = size;
    #endif // NX_AMSDU_TX
    #else // NX_UMAC_PRESENT
    #if NX_AMSDU_TX
    packet_addr = txdesc->host.packet_addr[pkt_idx];
    #else
    packet_addr = txdesc->host.packet_addr;
    #endif
    #if NX_AMSDU_TX
    if (pkt_idx > 0)
    {
        // Download only the payload of subsequent A-MSDU sub-frames
        dma_len = size;
        dma_oft = offsetof_b(struct txl_buffer_tag, payload);
    }
    else
    #endif // NX_AMSDU_TX
    {
        // Download the payload and the buffer control
        dma_len = size + sizeof_b(struct txl_buffer_control) + TX_BUFFER_PADDING_MAX;
        dma_oft = offsetof_b(struct txl_buffer_tag, buffer_control);
    }
    #endif // NX_UMAC_PRESENT


    dma_len_before_wrap = remaining * 4 - dma_oft;
    dma_desc_pat = &buf->dma_desc_pat;
    dma_dest = CPU2HW(buf) + dma_oft;

    do
    {
        dma_desc = &buf->dma_desc[i];

        // Fill in the bridge descriptor
        dma_desc->src = packet_addr;
        dma_desc->dest = dma_dest;
        dma_desc->ctrl = 0;

        // Check if buffer is contiguous
        if (dma_len_before_wrap >= dma_len)
        {
            // Buffer is contiguous, only one DMA descriptor is needed
            dma_len_before_wrap -= dma_len;
            dma_desc->length = dma_len;

            #if NX_MFP
            if (txdesc->host.flags & TXU_CNTRL_MGMT_ROBUST)
            {
                // Protected Management frame: Correct the destination because the download
                // needs to be done at the start of the buffer
                dma_dest -= head_len;
                dma_desc->dest = dma_dest;

                // txdesc->umac.head_len > 0 means unicast protected frame. In this case
                // the "offset" must be added between MAC header and MDPU body (by
                // using 2 dma desc)
                if (txdesc->umac.head_len)
                {
                    dma_desc->length = MAC_SHORT_MAC_HDR_LEN;

                    // Second one will download MPDU body
                    dma_desc->next = CPU2HW(&buf->dma_desc[1]);
                    dma_desc = &buf->dma_desc[1];
                    dma_desc->src = packet_addr + MAC_SHORT_MAC_HDR_LEN;
                    dma_desc->dest = dma_dest + MAC_SHORT_MAC_HDR_LEN
                                      + txdesc->umac.head_len;
                    dma_desc->length = dma_len - MAC_SHORT_MAC_HDR_LEN;
                    dma_desc->ctrl = 0;
                }
            }
            #endif // NX_MFP
        }
        else
        {
            // First DMA descriptor will transfer up to the end of the pool
            dma_desc->length = dma_len_before_wrap;
            dma_desc->next = CPU2HW(&idx->desc->dma_desc);

            // Second one will transfer the remaining part
            dma_dest = CPU2HW(pool);
            dma_len = dma_len - dma_len_before_wrap;

            dma_desc = &idx->desc->dma_desc;
            dma_desc->dest = dma_dest;
            dma_desc->src = packet_addr + dma_len_before_wrap;
            dma_desc->length = dma_len;
            dma_desc->ctrl = 0;

            // Only one wrap is possible, so set remaining_size to the max value
            // to ensure no more wrapping procedure is invoked in next loops
            dma_len_before_wrap = 0xFFFFFFFF;
        }

        #if NX_UMAC_PRESENT && NX_AMSDU_TX
        i++;

        if (i == packet_cnt)
            break;

        // Link to the next DMA descriptor
        dma_desc->next = CPU2HW(&buf->dma_desc[i]);

        // Compute DMA length and destination for next loop
        if (dma_len_before_wrap == 0)
        {
            dma_len_before_wrap = 0xFFFFFFFF;
            dma_dest = CPU2HW(pool);
        }
        else
            dma_dest += dma_len;

        dma_len = txdesc->host.packet_len[i];
        dma_oft = 0;
        packet_addr = txdesc->host.packet_addr[i];
    }
    while (1);
    #else // NX_UMAC_PRESENT && NX_AMSDU_TX
    }
    while (0);
    #endif // NX_UMAC_PRESENT && NX_AMSDU_TX

    // Prepare the DMA descriptor for the pattern
    tbd = &buf->tbd;
    tbd->upatterntx = 0;
    dma_desc_pat->src = macif_tx_pattern_addr_get();
    dma_desc_pat->dest = CPU2HW(&tbd->upatterntx);
    dma_desc_pat->length = sizeof_b(tbd->upatterntx);
    dma_desc_pat->ctrl = 0;

    #if NX_AMSDU_TX
    if (pkt_idx > 0)
    {
        // Check if we need to perform some actions once this payload is transfered
        #if NX_AMPDU_TX
        if (is_mpdu_agg(txdesc))
        {
            struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;

            agg_desc->available_len += size;

            if (!(agg_desc->status & AGG_ALLOC) &&
                 (agg_desc->available_len > TX_BUFFER_MIN_AMPDU_DWNLD))
            {
                // Enough data is now considered as allocated for this
                // A-MPDU, so no need to get DMA interrupts anymore
                agg_desc->status |= AGG_ALLOC;
                buf->flags |= BUF_ALLOC_OK | BUF_INT_MSDU;

                // Enable the interrupt and LLI counter increment
                dma_desc_pat->ctrl = TX_LLICTRL(access_category, 1);

                txl_buffer_push(access_category, buf);
            }
        }
        #endif //NX_AMPDU_TX
    }
    else
    #endif // NX_AMSDU_TX
    {
        #if (NX_UMAC_PRESENT)
        if (!(txdesc->host.flags & TXU_CNTRL_MGMT))
        {
            // Call the UMAC for the header formatting
            txu_cntrl_frame_build(txdesc, CPU2HW(buf->payload) + txdesc->umac.head_len);
        }
        #endif //(NX_UMAC_PRESENT

        // Check if we need to perform some actions once this payload is transfered
        #if NX_AMPDU_TX
        if (is_mpdu_agg(txdesc))
        {
            struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;

            agg_desc->available_len += size;

            if (is_mpdu_first(txdesc))
            {
                // Enable the interrupt and LLI counter increment
                dma_desc_pat->ctrl = TX_LLICTRL(access_category, 1);

                #if NX_UMAC_PRESENT
                // Copy the policy table from the buffer control to the buffer
                txl_buffer_control_copy(txdesc, buf);
                #endif

                #if (RW_BFMER_EN)
                // Check if a beamforming report shall be downloaded
                dma_desc = txl_buffer_bfr_dma_desc_conf(txdesc, &buf->dma_desc_bfr,
                                                        dma_desc);
                #endif

                txl_buffer_push(access_category, buf);

                if (agg_desc->available_len > TX_BUFFER_MIN_AMPDU_DWNLD)
                {
                    // For profiling
                    PROF_AGG_FIRST_MPDU_DWNLD_SET();

                    // Enough data is now considered as allocated for this
                    // A-MPDU, so no need to get DMA interrupts anymore
                    agg_desc->status |= AGG_ALLOC;
                    buf->flags |= BUF_ALLOC_OK;

                    // For profiling
                    PROF_AGG_FIRST_MPDU_DWNLD_CLR();
                }
            }
            else if (!(agg_desc->status & AGG_ALLOC) &&
                    ((agg_desc->available_len > TX_BUFFER_MIN_AMPDU_DWNLD)
                                               || is_mpdu_last(txdesc)))
            {
                // For profiling
                PROF_AGG_FIRST_MPDU_DWNLD_SET();

                // Enough data is now considered as allocated for this
                // A-MPDU, so no need to get DMA interrupts anymore
                agg_desc->status |= AGG_ALLOC;
                buf->flags |= BUF_ALLOC_OK;

                // Enable the interrupt and LLI counter increment
                dma_desc_pat->ctrl = TX_LLICTRL(access_category, 1);

                txl_buffer_push(access_category, buf);

                // For profiling
                PROF_AGG_FIRST_MPDU_DWNLD_CLR();
            }
        }
        else
        {
            // For profiling
            PROF_AGG_FIRST_MPDU_DWNLD_SET();

            // Enable the interrupt and LLI counter increment on the last descriptor
            dma_desc_pat->ctrl = TX_LLICTRL(access_category, 1);

            #if NX_UMAC_PRESENT
            // Copy the policy table from the buffer control to the buffer
            txl_buffer_control_copy(txdesc, buf);
            #endif

            #if (RW_BFMER_EN)
            // Check if a beamforming report shall be downloaded
            dma_desc = txl_buffer_bfr_dma_desc_conf(txdesc, &buf->dma_desc_bfr,
                                                    dma_desc);
            #endif

            buf->flags |= BUF_ALLOC_OK;
            txl_buffer_push(access_category, buf);

            // For profiling
            PROF_AGG_FIRST_MPDU_DWNLD_CLR();
        }
        #else // ! NX_AMPDU_TX
        // Enable the interrupt and LLI counter increment on the last descriptor
        dma_desc_pat->ctrl = TX_LLICTRL(access_category, 1);

        #if NX_UMAC_PRESENT
        // Copy the policy table from the buffer control to the buffer
        txl_buffer_control_copy(txdesc, buf);
        #endif

        #if (RW_BFMER_EN)
        // Check if a beamforming report shall be downloaded
        dma_desc = txl_buffer_bfr_dma_desc_conf(txdesc, &buf->dma_desc_bfr,
                                                dma_desc);
        #endif

        txl_buffer_push(access_category, buf);
        #endif // NX_AMPDU_TX
    }

    // Chain and configure the DMA descriptor for the pattern
    dma_desc->next = CPU2HW(dma_desc_pat);

    // Start the transfer
    dma_push(&buf->dma_desc[0], dma_desc_pat, IPC_DMA_CHANNEL_DATA_TX);
}


void txl_buffer_reinit(void)
{
    int i, j;

    for (i=0; i<NX_TXQ_CNT; i++)
    {
        txl_buffer_env.list[i].first = NULL;
        txl_buffer_env.list[i].last = NULL;
        for (j = 0; j<RW_USER_MAX; j++)
        {
            struct txl_buffer_idx_tag *idx = &txl_buffer_env.buf_idx[i][j];
            idx->free = 0;
            idx->used_area = 0;
            idx->free_size = TX_BUFFER_POOL_SIZE;
            idx->last = TX_BUFFER_NULL;
            idx->count = 0;
            idx->buf_size = 0;
            if ((i != AC_BCN) || (j == 0))
            {
                struct txl_buffer_hw_desc_tag *hwdesc = &txl_buffer_hw_desc[i*RW_USER_MAX + j];

                hwdesc->pbd.upatterntx = TX_PAYLOAD_DESC_PATTERN;
                hwdesc->pbd.bufctrlinfo = 0;
                hwdesc->pbd.next = 0;

                idx->pool = &txl_buffer_pool[i*RW_USER_MAX + j][0];
                idx->desc = hwdesc;
            }
            else
            {
                idx->pool = NULL;
                idx->desc = NULL;
            }
            idx->next_needed = -1;
        }
    }
}

void txl_buffer_reset(uint8_t access_category)
{
    int i;

    txl_buffer_env.list[access_category].first = NULL;
    txl_buffer_env.list[access_category].last = NULL;
    for (i = 0; i < RW_USER_MAX; i++)
    {
        struct txl_buffer_idx_tag *idx = &txl_buffer_env.buf_idx[access_category][i];

        idx->free = 0;
        idx->used_area = 0;
        idx->free_size = TX_BUFFER_POOL_SIZE;
        idx->last = TX_BUFFER_NULL;
        idx->count = 0;
        idx->buf_size = 0;
        idx->next_needed = -1;
    }
}

#if NX_AMSDU_TX
struct txl_buffer_tag *txl_buffer_alloc(struct txdesc *txdesc, uint8_t access_category,
                                        uint8_t user_idx, uint8_t pkt_idx)
#else
struct txl_buffer_tag *txl_buffer_alloc(struct txdesc *txdesc, uint8_t access_category,
                                        uint8_t user_idx)
#endif
{
    struct txl_buffer_tag *buf;
    uint16_t size, head_len;
    uint32_t needed, remaining;
    struct txl_buffer_idx_tag *idx = &txl_buffer_env.buf_idx[access_category][user_idx];

    // Compute the different parameters of the allocation
    needed = txl_buffer_get_params(txdesc, &size, &head_len
                                   #if NX_AMSDU_TX
                                   , pkt_idx
                                   #endif
                                   );

    //for profiling
    PROF_TX_BUF_ALLOC_SET();

    do
    {
        // Get the space allocated for this payload
        buf = txl_buffer_get_space(idx, needed, head_len, &remaining);
        if (buf == NULL)
            break;

        buf->user_idx = user_idx;

        // Program the transfer of the payload
        txl_buffer_transfer(txdesc, access_category, buf, idx, size, remaining, head_len
                            #if NX_AMSDU_TX
                            , pkt_idx
                            #endif
                            );

    } while(0);

    //for profiling
    PROF_TX_BUF_ALLOC_CLR();

    return(buf);
}


bool txl_buffer_free(struct txl_buffer_tag *buf, uint8_t access_category)
{
    struct txl_buffer_idx_tag *idx = &txl_buffer_env.buf_idx[access_category][buf->user_idx];

    //for profiling
    PROF_TX_BUF_FREE_SET();

    // Update the used area
    idx->used_area -= buf->length;
    idx->free_size += buf->length;
    if (buf->flags & BUF_FRONTIER)
        idx->count--;

    // Sanity checks
    ASSERT_ERR(idx->free_size <= TX_BUFFER_POOL_SIZE);
    ASSERT_ERR(idx->used_area < TX_BUFFER_POOL_SIZE);

    // Check if we freed the last buffer
    if (idx->used_area == 0)
    {
        idx->free = 0;
        idx->last = TX_BUFFER_NULL;
        idx->count = 0;
        idx->buf_size = 0;
    }

    //for profiling
    PROF_TX_BUF_FREE_CLR();

    return ((buf->flags & BUF_FRONTIER) || (idx->free_size >= idx->next_needed));
}

void txl_buffer_free_all(struct txdesc *txdesc, uint8_t access_category)
{
    #if NX_AMSDU_TX
    int i;
    // Free all payloads associated to this MPDU
    for (i = 0; i < txdesc->host.packet_cnt; i++)
    {
        struct txl_buffer_tag *buf = txdesc->lmac.buffer[i];

        // Free the buffer associated with the descriptor
        if (buf != NULL)
        {
            // Free the buffer
            txl_buffer_free(buf, access_category);

            // Reset the buffer pointer for this index
            txdesc->lmac.buffer[i] = NULL;
        }
    };
    #else
    if (txdesc->lmac.buffer != NULL)
    {
        // Free the buffer associated with the descriptor
        txl_buffer_free(txdesc->lmac.buffer, access_category);
        txdesc->lmac.buffer = NULL;
    }
    #endif //NX_AMSDU_TX
}

#endif //!NX_FULLY_HOSTED

void txl_buffer_init(void)
{
    #if NX_UMAC_PRESENT
    int i;
    #endif

    #if !NX_FULLY_HOSTED
    txl_buffer_reinit();
    #endif //!NX_FULLY_HOSTED

    #if NX_UMAC_PRESENT
    for (i=0; i<NX_REMOTE_STA_MAX; i++)
    {
        struct tx_policy_tbl *pol = &txl_buffer_control_desc[i].policy_tbl;
        pol->upatterntx         = POLICY_TABLE_PATTERN;
        pol->phycntrlinfo1      = phy_get_ntx() << NX_TX_PT_OFT;
        pol->phycntrlinfo2      = TX_NTX_2_ANTENNA_SET(phy_get_ntx());
        pol->maccntrlinfo1      = 0;
        pol->maccntrlinfo2      = 0xFFFF0704;
        pol->ratecntrlinfo[0]   = HW_RATE_1MBPS << MCS_INDEX_TX_RCX_OFT;
        pol->ratecntrlinfo[1]   = HW_RATE_1MBPS << MCS_INDEX_TX_RCX_OFT;
        pol->ratecntrlinfo[2]   = HW_RATE_1MBPS << MCS_INDEX_TX_RCX_OFT;
        pol->ratecntrlinfo[3]   = HW_RATE_1MBPS << MCS_INDEX_TX_RCX_OFT;
        pol->powercntrlinfo[0]  = TX_PWR_LEVEL_SET(nxmac_ofdm_max_pwr_level_getf());
        pol->powercntrlinfo[1]  = TX_PWR_LEVEL_SET(nxmac_ofdm_max_pwr_level_getf());
        pol->powercntrlinfo[2]  = TX_PWR_LEVEL_SET(nxmac_ofdm_max_pwr_level_getf());
        pol->powercntrlinfo[3]  = TX_PWR_LEVEL_SET(nxmac_ofdm_max_pwr_level_getf());

        txl_buffer_control_desc[i].mac_control_info = EXPECTED_ACK_NORMAL_ACK | LOW_RATE_RETRY;
        txl_buffer_control_desc[i].phy_control_info = 63 << GID_TX_OFT;
    }
    for (i=0; i<NX_VIRT_DEV_MAX; i++)
    {
        struct tx_policy_tbl *pol = &txl_buffer_control_desc_bcmc[i].policy_tbl;
        pol->upatterntx         = POLICY_TABLE_PATTERN;
        pol->phycntrlinfo1      = phy_get_ntx() << NX_TX_PT_OFT;
        pol->phycntrlinfo2      = TX_NTX_2_ANTENNA_SET(phy_get_ntx());
        pol->maccntrlinfo1      = 0;
        pol->maccntrlinfo2      = 0xFFFF0704;
        pol->ratecntrlinfo[0]   = HW_RATE_1MBPS << MCS_INDEX_TX_RCX_OFT;
        pol->ratecntrlinfo[1]   = 0;
        pol->ratecntrlinfo[2]   = 0;
        pol->ratecntrlinfo[3]   = 0;
        pol->powercntrlinfo[0]  = TX_PWR_LEVEL_SET(nxmac_ofdm_max_pwr_level_getf());
        pol->powercntrlinfo[1]  = 0;
        pol->powercntrlinfo[2]  = 0;
        pol->powercntrlinfo[3]  = 0;

        txl_buffer_control_desc_bcmc[i].mac_control_info = EXPECTED_ACK_NO_ACK;
        txl_buffer_control_desc_bcmc[i].phy_control_info = 63 << GID_TX_OFT;
    }
    #endif
}


/// @}  // end of group BUFFER
