/**
 ****************************************************************************************
 *
 * @file txl_cfm.c
 *
 * @brief Tx confirmation module.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/*
 ****************************************************************************************
 * @addtogroup TX_CFM
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include <stddef.h>
#include "txl_cntrl.h"
#include "txl_cfm.h"
#include "ke_event.h"
#include "mac_frame.h"
#include "dbg.h"
#include "txl_agg.h"
#include "txl_buffer.h"
#include "txl_frame.h"
#include "co_utils.h"
#include "ps.h"

#if NX_AMPDU_TX
//for key count
#include "mm.h"
//for rx frame release
#include "rxl_hwdesc.h"
#endif
// For LMAC to UMAC communication interface
#include "macif.h"
#if NX_UMAC_PRESENT
#include "me_utils.h"
#include "txu_cntrl.h"
#endif
#if (RW_BFMER_EN)
#include "bfr.h"
#endif //(RW_BFMER_EN)

/*
 * GLOBAL VARIABLE DEFINITION
 ****************************************************************************************
 */
struct txl_cfm_env_tag txl_cfm_env;

const uint32_t txl_cfm_evt_bit[NX_TXQ_CNT] =
{
    #if NX_BEACONING
    [AC_BCN] = KE_EVT_TXCFM_BCN_BIT,
    #endif
    [AC_VO] = KE_EVT_TXCFM_AC3_BIT,
    [AC_VI] = KE_EVT_TXCFM_AC2_BIT,
    [AC_BE] = KE_EVT_TXCFM_AC1_BIT,
    [AC_BK] = KE_EVT_TXCFM_AC0_BIT,
};

#if NX_AMPDU_TX
/**
 ****************************************************************************************
 * @brief Recover the access category value corresponding to a received BA frame by
 *        extracting the TID from its payload.
 *
 * @param[in] rxdesc  Pointer to Rx descriptor for BA frame for which AC is determined
 *
 * @return One of the 4 access category values if any correspond, invalid value if not.
 ****************************************************************************************
 */
static uint8_t txl_ba_ac_get(struct rxdesc *rxdesc)
{
    #if NX_MAC_HE
    uint8_t ac;

    if (txl_he_tb_ongoing(&ac))
    {
        return ac;
    }
    else
    #endif
    {
        struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
        // Get the status information from the RX header descriptor
        uint32_t statinfo = dma_hdrdesc->hd.statinfo;

        // And extract the access category from it
        return (uint8_t)((statinfo & IMM_RSP_AC_MSK) >> IMM_RSP_AC_OFT);
    }
}

/**
 ****************************************************************************************
 * @brief This function checks if the received compressed BA content matches the A-MPDU it
 * acknowledges, i.e if TA and TID correspond to the respective parameters of the A-MPDU
 * we sent.
 *
 * @param[in] badesc   Pointer to Rx descriptor of BA frame
 * @param[in] agg_desc Pointer to the A-MPDU descriptor
 *
 * @return true if TA and TID of the BA frame match the respective parameters of the
 *         A-MPDU, false otherwise.
 ****************************************************************************************
 */
static bool txl_decode_compressed_ba(struct rxdesc *badesc, struct tx_agg_desc *agg_desc)
{
    uint8_t tid;
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(badesc);
    struct rx_hd *hdrdesc = &dma_hdrdesc->hd;
    struct rx_pbd *pd = HW2CPU(hdrdesc->first_pbd_ptr);
    struct ba_comp_frame *ba_payl = (struct ba_comp_frame *)HW2CPU(pd->datastartptr);

    // Compressed BA shall be unicast to our device
    if (hdrdesc->statinfo & RX_HD_ADDRMIS)
        return false;

    // Check if BA frame length is correct
    if (hdrdesc->frmlen != BA_COMPRESSED_LEN)
        // Length not correct
        return false;

    // Extract the TID
    // Byte 1 is MSB of BA Control field, its upper 4 bits = TID
    tid = (ba_payl->ba_cntrl & BA_PAYL_TID_BIT_MSK) >> BA_PAYL_TID_BIT_OFT;

    // Check if both TID match
    if (tid != agg_desc->tid)
        // No match, we consider BA as invalid
        return false;

    memcpy(&agg_desc->ssc_bitmap_tab, &ba_payl->ssc_bitmap, sizeof(ba_payl->ssc_bitmap));
    agg_desc->ssc_bitmap = (struct ba_ssc_bitmap *)(&agg_desc->ssc_bitmap_tab);

    // All checks passed, BA is valid
    return true;
}

/**
 ****************************************************************************************
 * @brief This function checks if the received BA contains acknowledgment information
 * for the A-MPDU we transmitted.
 *
 * @param[in] agg_desc A-MPDU descriptor (contains all required information)
 * @param[in] badesc BA frame descriptor
 *
 * @return true if the received BA frame has acknowledgment information for the A-MPDU,
 *         false otherwise.
 ****************************************************************************************
 */
static bool txl_is_ba_valid(struct tx_agg_desc *agg_desc, struct rxdesc *badesc)
{
    uint16_t key_idx_hw;
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(badesc);
    struct rx_hd *hdrdesc = &dma_hdrdesc->hd;
    uint32_t statinfo = hdrdesc->statinfo;
    #if NX_MAC_HE
    struct rx_pbd *pd = HW2CPU(hdrdesc->first_pbd_ptr);
    struct ba_base *ba_payl = (struct ba_base *)HW2CPU(pd->datastartptr);
    uint16_t ba_type;
    #endif

    // Check if the Key index is valid, i.e. if the TA is known
    if (!(statinfo & KEY_IDX_VALID_BIT))
        // TA not known, we consider the BA as invalid
        return false;

    // Get the key index
    key_idx_hw = (uint16_t)((statinfo & KEY_IDX_MSK) >> KEY_IDX_OFT);

    // Sanity check
    ASSERT_REC_VAL(key_idx_hw >= MM_SEC_DEFAULT_KEY_COUNT, false);

    // Check if both STA indexes match
    if (MM_KEY_TO_STA(key_idx_hw) != agg_desc->sta_idx)
        // No match, we consider BA as invalid
        return false;

    #if NX_MAC_HE
    // Get BA type
    ba_type = ba_payl->ba_cntrl & BA_PAYL_TYPE_MSK;

    if (ba_type == BA_MULTI_STA_TYPE)
        return txl_he_decode_m_ba(badesc, agg_desc);
    #endif

    // By default we support only compressed BlockAck
    return txl_decode_compressed_ba(badesc, agg_desc);
}

/**
 ****************************************************************************************
 * @brief Returns true if the first SN is the min of two sequence numbers
 *
 * @param[in] sn1     First sequence number to be compared
 * @param[in] sn2     Second sequence number to be compared
 *
 * @return True if sn1 <= sn2
 ****************************************************************************************
 */
static bool txl_comp_sn(uint16_t sn1, uint16_t sn2)
{
    if (((uint16_t)((sn1 - sn2) & 0x0FFF)) < 0x07FF)
        return false;
    else
        return true;
}

/**
 ****************************************************************************************
 * @brief Extract the ACK status of an MPDU from a BA frame bitmap
 * @param[in] txdesc   Pointer to Tx SW descriptor of the MPDU whose status will be extracted
 * @param[in] agg_desc  Pointer to the A-MPDU descriptor
 * @param[out] agg_ok  Variable to be incremented if the MPDU is confirmed
 * @return The status bits to be put in the descriptor status field
 ****************************************************************************************
 */
static uint32_t txl_ba_extract_ack(struct txdesc *txdesc, struct tx_agg_desc *agg_desc,
                                   int *agg_ok)
{
    struct ba_ssc_bitmap *ssc_bitmap = agg_desc->ssc_bitmap;
    uint16_t ssn;

    #if NX_MAC_HE
    // Case of "All Ack" feature - if no ssc_bitmap is present, it means everything was
    // received fine
    if (ssc_bitmap == NULL)
    {
        // Increment the number of MPDUs successfully transmitted
        (*agg_ok)++;
        return (BA_FRAME_RECEIVED_BIT | FRAME_SUCCESSFUL_TX_BIT);
    }
    #endif

    // byte 2 is LSB of Start Sequence control, at bit 4 the SSN starts
    ssn = ssc_bitmap->ssc >> SN_IN_SSC_BIT_OFT;

    //txdesc is below the start of the BA window - bad, means we moved the window! can never get an ACK for it
    if (txl_comp_sn(txdesc->host.sn, ssn))
    {
        return BA_FRAME_RECEIVED_BIT;
    }

    uint16_t bit_pos = (txdesc->host.sn - ssn) & 0xFFF;
    uint8_t word_idx = bit_pos/16;
    uint8_t bit_in_word_pos = bit_pos % 16;

    //make sure SN is in window - shouldn't be possible
    if (word_idx > 3)
    {
        return BA_FRAME_RECEIVED_BIT;
    }

    if (((ssc_bitmap->bitmap[word_idx] >> bit_in_word_pos) & 0x01) == 0x01)
    {
        // Increment the number of MPDUs successfully transmitted
        (*agg_ok)++;
        return (BA_FRAME_RECEIVED_BIT | FRAME_SUCCESSFUL_TX_BIT);
    }
    else
    {
        return BA_FRAME_RECEIVED_BIT;
    }
}
#endif


void txl_cfm_init(void)
{
    // Reset the environment
    memset(&txl_cfm_env, 0, sizeof(txl_cfm_env));

    for (int i = 0; i < NX_TXQ_CNT; i++)
    {
        co_list_init(&txl_cfm_env.cfmlist[i]);
    }
}

void txl_cfm_push(struct txdesc *txdesc, uint32_t status, uint8_t access_category)
{
    struct tx_cfm_tag *cfm = txl_cfm_tag_get(txdesc);
    cfm->status = status;

    // Push the descriptor in the confirmation queue
    co_list_push_back(&txl_cfm_env.cfmlist[access_category], (struct co_list_hdr *)txdesc);

    #if NX_AMPDU_TX
    //call the CFM event only if it is a singleton, for MPDUs in AMPDU CFM event is set when BAR is done
    if (!is_mpdu_agg(txdesc))
    #endif
        // Set an event to handle the confirmation in background
        ke_evt_set(txl_cfm_evt_bit[access_category]);
}

#if NX_AMPDU_TX
void txl_ba_push(struct rxdesc *rxdesc)
{
    uint8_t ac;
    struct tx_agg_desc *agg_desc;

    // For profiling
    PROF_AGG_BA_RXED_SET();

    // Get the access category value
    ac = txl_ba_ac_get(rxdesc);

    // Set BA in appropriate aggregate descriptor
    agg_desc = (struct tx_agg_desc *)co_list_pop_front(&(txl_cntrl_env.txlist[ac].aggregates));
    // Sanity check - There shall be an AGG descriptor for this BA, because unexpected
    // BAs are filtered earlier in the RX chain
    ASSERT_REC(agg_desc != NULL);

    // A BlockAck has been received for this PPDU
    agg_desc->status |= AGG_BA_RECEIVED;

    // Check if the received BA is valid
    if (txl_is_ba_valid(agg_desc, rxdesc))
        agg_desc->status |= AGG_BA_VALID;

    // For profiling
    PROF_AGG_BA_RXED_CLR();
}
#endif

void txl_cfm_evt(int access_category)
{
    struct txdesc *txdesc;
    uint32_t evt_bit = txl_cfm_evt_bit[access_category];
    struct co_list *cfm_list = &txl_cfm_env.cfmlist[access_category];
    #if NX_AMPDU_TX
    #if NX_UMAC_PRESENT
    struct txdesc *txdesc_first = NULL;
    #endif
    int agg_txed = 0;
    int agg_ok = 0;
    #endif

    // For profiling
    PROF_TX_CFM_EVT_SET();

    // Check that the event is correctly set
    ASSERT_ERR(ke_evt_get() & evt_bit);

    // Clear the event
    ke_evt_clear(evt_bit);

    // Start the TX confirmation upload procedure
    macif_tx_cfm_start(access_category);

    while(1)
    {
        // Pick the first desc from the cfm queue (either 1st of AMPDU or singleton)
        txdesc = (struct txdesc *)co_list_pick(cfm_list);

        if (txdesc != NULL)
        {
            #if NX_AMPDU_TX
            struct tx_cfm_tag *cfm = txl_cfm_tag_get(txdesc);
            struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;
            #endif

            // For profiling
            PROF_TX_AC_BG_SET(access_category);

            #if NX_UAPSD && NX_UMAC_PRESENT
            ps_check_tx_trigger_sent(&txdesc->host,
                                     txl_cfm_tag_get(txdesc)->status);
            #endif

            #if NX_AMPDU_TX
            // Check if the packet was part of an A-MPDU
            if (agg_desc)
            {
                uint8_t status = agg_desc->status;

                // Check if the A-MPDU status is ready
                if (!(status & AGG_DONE))
                    break;

                // One more aggregated MPDU was transmitted
                agg_txed++;

                //clear FRM successful bit
                cfm->status &= ~FRAME_SUCCESSFUL_TX_BIT;

                if (status & AGG_BA_VALID)
                {
                    //set ACK state extracted from BA bitmap in FRM successful bit position
                    cfm->status |= txl_ba_extract_ack(txdesc, agg_desc, &agg_ok);
                }

                #if (RW_BFMER_EN)
                if (bfr_is_enabled())
                {
                    // Check if the frame was beamformed
                    bfr_tx_cfm(txdesc);
                }
                #endif //(RW_BFMER_EN)

                #if NX_UMAC_PRESENT
                // Indicate the packet status to the upper MAC
                txu_cntrl_cfm(txdesc);

                if (is_mpdu_first(txdesc))
                    txdesc_first = txdesc;
                #endif

                // Protect the access to the lists from interrupt preemption
                GLOBAL_INT_DISABLE();

                // Extract the TX descriptor from the confirmation list
                co_list_pop_front(cfm_list);

                if (is_mpdu_last(txdesc))
                {
                    #if NX_UMAC_PRESENT
                    ASSERT_ERR(txdesc_first != NULL);
                    // Update the RC about the status
                    rc_cfm_ampdu(txdesc_first, agg_txed, agg_ok);
                    txdesc_first = NULL;
                    #endif

                    // Reset the packet counters
                    agg_txed = 0;
                    agg_ok = 0;

                    // Free all aggregation related frames and control (BA, BAR, AGG_CNTRL)
                    txl_agg_release(agg_desc);
                }

                // End of interrupt protection
                GLOBAL_INT_RESTORE();

                // Prepare the confirmation to upper MAC
                macif_tx_cfm_push(access_category, txdesc);
            }
            //singleton MPDU - confirm directly
            else
            {
            #endif
                // Protect the access to the list from interrupt preemption
                GLOBAL_INT_DISABLE();

                //only picked it, must pop it now
                co_list_pop_front(cfm_list);

                // End of interrupt protection
                GLOBAL_INT_RESTORE();

                #if (RW_BFMER_EN)
                if (bfr_is_enabled())
                {
                    // Check if the frame was beamformed
                    bfr_tx_cfm(txdesc);
                }
                #endif //(RW_BFMER_EN)

                #if NX_UMAC_PRESENT
                // Update the RC about the TX status
                // WARNING!!! Must be done before the call to txu_cntrl_cfm()!!!
                rc_cfm_singleton(txdesc);

                // Indicate the packet status to the upper MAC
                txu_cntrl_cfm(txdesc);
                #endif

                // Send the confirmation to upper MAC, already has status
                macif_tx_cfm_push(access_category, txdesc);

            #if NX_AMPDU_TX
            }
            #endif
        }
        else
        {
            // No more descriptors to confirm, so exit the loop
            break;
        }
    }

    // Stop the TX confirmation upload procedure
    macif_tx_cfm_done(access_category, false);

    // For profiling
    PROF_TX_CFM_EVT_CLR();
}

void txl_cfm_flush_desc(uint8_t access_category, struct txdesc *txdesc, uint32_t status)
{
    struct tx_cfm_tag *cfm = txl_cfm_tag_get(txdesc);

    // Start the TX confirmation upload procedure
    macif_tx_cfm_start(access_category);

    // Put the status in the descriptor, only if it was not already done by the HW
    if (txdesc->lmac.agg_desc != NULL)
        cfm->status = status | A_MPDU_LAST;
    else if (!(cfm->status & DESC_DONE_TX_BIT))
        cfm->status = status;

    #if NX_TX_FRAME
    // Check if the frame was generated internally
    if (is_int_frame(txdesc))
    {
        // Put the descriptor in the CFM queue
        txl_frame_cfm(txdesc);
    }
    else
    #endif
    {
        #if NX_UMAC_PRESENT
        // Indicate the packet status to the upper MAC
        txu_cntrl_cfm(txdesc);
        #endif

        #if (RW_BFMER_EN)
        if (bfr_is_enabled())
        {
            // Check if the frame was beamformed
            bfr_tx_cfm(txdesc);
        }
        #endif //(RW_BFMER_EN)

        // Prepare the confirmation to the host
        macif_tx_cfm_push(access_category, txdesc);

        #if !NX_FULLY_HOSTED
        txl_buffer_free_all(txdesc, access_category);
        #endif //!NX_FULLY_HOSTED
    }


    #if NX_TX_FRAME
    // Execute the internal frame confirmation event to complete the flush
    txl_frame_evt(0);
    #endif

    // Stop the TX confirmation upload procedure
    macif_tx_cfm_done(access_category, true);
}


void txl_cfm_flush(uint8_t access_category, struct co_list *list, uint32_t status)
{
    // Start the TX confirmation upload procedure
    macif_tx_cfm_start(access_category);

    // Go through the descriptors in the list
    while (1)
    {
        // Get the first descriptor from the list
        struct txdesc *txdesc = (struct txdesc *)co_list_pop_front(list);
        struct tx_cfm_tag *cfm;

        // Check if we reached the end of the list
        if (txdesc == NULL)
            break;

        // Put the status in the descriptor, only if it was not already done by the HW
        cfm = txl_cfm_tag_get(txdesc);
        if (txdesc->lmac.agg_desc != NULL)
            cfm->status = status | A_MPDU_LAST;
        else if (!(cfm->status & DESC_DONE_TX_BIT))
            cfm->status = status;

        #if NX_TX_FRAME
        // Check if the frame was generated internally
        if (is_int_frame(txdesc))
        {
            // Put the descriptor in the CFM queue
            txl_frame_cfm(txdesc);
        }
        else
        #endif
        {
            #if NX_UMAC_PRESENT
            // Indicate the packet status to the upper MAC
            txu_cntrl_cfm(txdesc);
            #endif

            #if (RW_BFMER_EN)
            if (bfr_is_enabled())
            {
                // Check if the frame was beamformed
                bfr_tx_cfm(txdesc);
            }
            #endif //(RW_BFMER_EN)

            // Prepare the confirmation to the host
            macif_tx_cfm_push(access_category, txdesc);

            #if !NX_FULLY_HOSTED
            txl_buffer_free_all(txdesc, access_category);
            #endif //!NX_FULLY_HOSTED
        }
    }

    #if NX_TX_FRAME
    // Execute the internal frame confirmation event to complete the flush
    txl_frame_evt(0);
    #endif

    // Stop the TX confirmation upload procedure
    macif_tx_cfm_done(access_category, true);
}


/// @}

