/**
 ****************************************************************************************
 *
 * @file txl_cntrl.c
 *
 * @brief LMAC TxPath implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup TX_CNTRL
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "co_int.h"
#include "co_bool.h"
#include <string.h>

#include "dbg_assert.h"
#include "mac.h"
#include "mac_frame.h"
#include "mm.h"
#include "macif.h"
#include "rxl_hwdesc.h"
#include "txl_buffer.h"
#if NX_TX_FRAME
#include "txl_frame.h"
#endif
#include "tx_swdesc.h"
#include "txl_cfm.h"
#include "txl_cntrl.h"
#include "rxl_cntrl.h"
#include "reg_mac_pl.h"
#include "reg_mac_core.h"
#include "dbg.h"
#include "ps.h"
#if (NX_P2P)
#include "p2p.h"
#endif //(NX_P2P)
#if (NX_TD)
#include "td.h"
#endif //(NX_TD)
#if NX_UMAC_PRESENT
#include "txu_cntrl.h"
#include "apm.h"
#endif
#if NX_MFP
#include "mfp.h"
#endif
#if (RW_BFMER_EN)
#include "bfr.h"
#endif //(RW_BFMER_EN)
#if NX_AMPDU_TX
#include "txl_agg.h"
#endif
#if NX_MAC_HE
#include "txl_he.h"
#endif

/*
 * DEFINES
 ****************************************************************************************
 */
/// TX IRQ bits enabled
#if NX_BEACONING
#define TX_IRQ_BITS  ( NXMAC_AC_0_TX_TRIGGER_BIT | NXMAC_AC_1_TX_TRIGGER_BIT |           \
                       NXMAC_AC_2_TX_TRIGGER_BIT | NXMAC_AC_3_TX_TRIGGER_BIT |           \
                       NXMAC_BCN_TX_TRIGGER_BIT )
#else
#define TX_IRQ_BITS  ( NXMAC_AC_0_TX_TRIGGER_BIT | NXMAC_AC_1_TX_TRIGGER_BIT |           \
                       NXMAC_AC_2_TX_TRIGGER_BIT | NXMAC_AC_3_TX_TRIGGER_BIT)
#endif

#if RW_MUMIMO_TX_EN
/// Bits indicating activity on secondary users (merged trigger and buffer)
#define TX_SEC_IRQ_BITS_MERGED  ( NXMAC_SEC_U_3AC_3_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_3AC_2_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_3AC_1_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_3AC_0_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_2AC_3_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_2AC_2_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_2AC_1_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_2AC_0_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_1AC_3_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_1AC_2_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_1AC_1_TX_TRIGGER_BIT     |                 \
                                  NXMAC_SEC_U_1AC_0_TX_TRIGGER_BIT )
#endif

#if NX_BW_LEN_ADAPT
/// Bits indicating that a BW drop has been triggered on an access category
#define TX_BW_DROP_IRQ  (NXMAC_AC_0BW_DROP_TRIGGER_BIT | NXMAC_AC_1BW_DROP_TRIGGER_BIT | \
                         NXMAC_AC_2BW_DROP_TRIGGER_BIT | NXMAC_AC_3BW_DROP_TRIGGER_BIT)
#endif

#if NX_MAC_HE
/// HE TB protocol trigger interrupt bit
#define TB_PROT_TRIGGER_BIT NXMAC_TB_PROT_TRIGGER_BIT
/// HE TB transmission trigger interrupt bit
#define TB_TX_TRIGGER_BIT (NXMAC_TB_TX_TRIGGER_BIT | NXMAC_TB_TX_BUF_TRIGGER_BIT | NXMAC_TB_TX_CANCELLED_BIT)
#else
/// HE TB protocol trigger interrupt bit
#define TB_PROT_TRIGGER_BIT 0
/// HE TB transmission trigger interrupt bit
#define TB_TX_TRIGGER_BIT 0
#endif

/// Secondary user trigger interrupt bit
#if RW_MUMIMO_TX_EN
#define SEC_USER_TX_TRIGGER_BIT NXMAC_SEC_USER_TX_TRIGGER_BIT
#else
#define SEC_USER_TX_TRIGGER_BIT 0
#endif

/// TX protocol trigger interrupt bits
#define TX_PROT_IRQ ( NXMAC_AC_0BW_DROP_TRIGGER_BIT     |                                \
                      NXMAC_AC_1BW_DROP_TRIGGER_BIT     |                                \
                      NXMAC_AC_2BW_DROP_TRIGGER_BIT     |                                \
                      NXMAC_AC_3BW_DROP_TRIGGER_BIT     |                                \
                      TB_PROT_TRIGGER_BIT               |                                \
                      NXMAC_AC_0_PROT_TRIGGER_BIT       |                                \
                      NXMAC_AC_1_PROT_TRIGGER_BIT       |                                \
                      NXMAC_AC_2_PROT_TRIGGER_BIT       |                                \
                      NXMAC_AC_3_PROT_TRIGGER_BIT )

/// TX transmit trigger interrupt bits
#define TX_TRANSMIT_IRQ ( SEC_USER_TX_TRIGGER_BIT           |                            \
                          TB_TX_TRIGGER_BIT                 |                            \
                          NXMAC_AC_0_TX_TRIGGER_BIT         |                            \
                          NXMAC_AC_1_TX_TRIGGER_BIT         |                            \
                          NXMAC_AC_2_TX_TRIGGER_BIT         |                            \
                          NXMAC_AC_3_TX_TRIGGER_BIT         |                            \
                          NXMAC_BCN_TX_TRIGGER_BIT          |                            \
                          NXMAC_AC_0_TX_BUF_TRIGGER_BIT     |                            \
                          NXMAC_AC_1_TX_BUF_TRIGGER_BIT     |                            \
                          NXMAC_AC_2_TX_BUF_TRIGGER_BIT     |                            \
                          NXMAC_AC_3_TX_BUF_TRIGGER_BIT     |                            \
                          NXMAC_BCN_TX_BUF_TRIGGER_BIT  )


/// Flags indicating that a frame has been correctly transmitted by the HW
#define FRAME_OK  (DESC_DONE_TX_BIT | FRAME_SUCCESSFUL_TX_BIT)

/// Table containing the TX timeout value per TX queue
const uint32_t TX_TIMEOUT[NX_TXQ_CNT] =
{
    TX_AC0_TIMEOUT,
    TX_AC1_TIMEOUT,
    TX_AC2_TIMEOUT,
    TX_AC3_TIMEOUT,
    #if (NX_BEACONING)
    TX_BCN_TIMEOUT
    #endif
};

/*
 * GLOBAL VARIABLE DEFINITION
 ****************************************************************************************
 */
struct txl_cntrl_env_tag txl_cntrl_env;

/*
 * FUNCTION BODIES
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Set the smoothing bit to the correct HW descriptor
 *
 * @param[in] thd  TX Header Descriptor attached to the PPDU
 * @param[in] pt   Policy table attached to the PPDU
 * @param[in] smoothing Smoothing bit to be set in the descriptor
 ****************************************************************************************
 */
__INLINE void txl_smoothing_set(struct tx_hd *thd, struct tx_policy_tbl *pt,
                                uint32_t smoothing)
{
    #if !NX_MAC_HE
    thd->phyctrlinfo |= smoothing;
    #else
    pt->phycntrlinfo1 |= smoothing;
    #endif
}

/**
 ****************************************************************************************
 * @brief Start a TX timer
 *
 * @param[in] ac  Access category for which the timer has to be started
 *
 ****************************************************************************************
 */
__INLINE void txl_timer_start(uint8_t ac)
{
    int timer_id = TX_AC2TIMER(ac);
    uint32_t timer_bit = CO_BIT(timer_id);
    uint32_t curr_time = hal_machw_time();

    // Set the timer
    nxmac_abs_timer_set(timer_id, curr_time + TX_TIMEOUT[ac]);

    // Enable the TX queue timeout
    nxmac_timers_int_event_clear(timer_bit);
    nxmac_timers_int_un_mask_set(nxmac_timers_int_un_mask_get() | timer_bit);
}

/**
 ****************************************************************************************
 * @brief Move the TX timer according to the access category
 *
 * @param[in] ac  Access category for which the timer has to be moved
 *
 ****************************************************************************************
 */
__INLINE void txl_timer_move(uint8_t ac)
{
    uint32_t curr_time = hal_machw_time();

    // Set pattern at end to not validate descriptor until it's ready
    nxmac_abs_timer_set(TX_AC2TIMER(ac), curr_time + TX_TIMEOUT[ac]);
}

/**
 ****************************************************************************************
 * @brief Clear a TX timer
 *
 * @param[in] ac  Access category for which the timer has to be cleared
 *
 ****************************************************************************************
 */
__INLINE void txl_timer_clear(uint8_t ac)
{
    uint32_t timer_msk = nxmac_timers_int_un_mask_get();
    uint32_t timer_bit = CO_BIT(TX_AC2TIMER(ac));

    nxmac_timers_int_un_mask_set(timer_msk & ~timer_bit);
    nxmac_timers_int_event_clear(timer_bit);
}

#if !NX_FULLY_HOSTED
#if NX_TX_FRAME
/**
 ****************************************************************************************
 * @brief Prepare the transfer of payload from host memory to emb memory
 *
 * This primitive first allocates a buffer of the required size, and then updates the
 * bridge descriptor according to the allocated buffer.
 *
 * @param[in] txdesc           Descriptor of the packet to be pushed into the Tx list
 * @param[in] access_category  Access category where to push the descriptor
 *
 * @return The status of the transfer (TRUE: programmed, FALSE: not programmed due to buffer full)
 *
 ****************************************************************************************
 */
static void txl_int_fake_transfer(struct txdesc *txdesc, uint8_t access_category)
{
    // Do a fake DMA transfer
    struct txl_buffer_tag *buf = txl_buffer_get(txdesc);
    struct tx_pbd *tbd = &buf->tbd;
    struct dma_desc *dma_desc_pat = &buf->dma_desc_pat;
    dma_desc_pat->src = macif_tx_pattern_addr_get();
    dma_desc_pat->dest = CPU2HW(&tbd->upatterntx);
    dma_desc_pat->length = sizeof_b(tbd->upatterntx);
    dma_desc_pat->ctrl = TX_LLICTRL(access_category, 1);
    buf->txdesc = txdesc;
    #if NX_MAC_HE
    buf->flags &= ~BUF_SINGLETON_READY;
    #endif
    txl_buffer_push(access_category, buf);
    dma_push(dma_desc_pat, dma_desc_pat, IPC_DMA_CHANNEL_DATA_TX);
}
#endif

/**
 ****************************************************************************************
 * @brief Prepare the transfer of payload from host memory to emb memory
 *
 * This primitive first allocates a buffer of the required size, and then updates the
 * bridge descriptor according to the allocated buffer.
 *
 * @param[in] txdesc           Descriptor of the packet to be pushed into the Tx list
 * @param[in] access_category  Access category where to push the descriptor
 * @param[in] user_idx         User index (for MU-MIMO TX only, 0 otherwise)
 *
 * @return The status of the transfer (TRUE: programmed, FALSE: not programmed due to buffer full)
 *
 ****************************************************************************************
 */
static bool txl_payload_transfer(struct txdesc *txdesc, uint8_t access_category,
                                 uint8_t user_idx)
{
    struct txl_buffer_tag *buffer = NULL;
    bool buffer_full = true;
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];
    #if NX_AMSDU_TX
    uint8_t pkt_idx;
    #endif

    #if NX_TX_FRAME
    if (is_int_frame(txdesc))
    {
        txl_int_fake_transfer(txdesc, access_category);
        return (false);
    }
    #endif

    // Allocate the payload buffer
    #if NX_AMSDU_TX
    #if NX_UMAC_PRESENT
    if (txl_buffer_is_amsdu_single_buf(txdesc))
    {
        ASSERT_ERR(txlist->dwnld_index[user_idx] == 0);
        pkt_idx = 0xFF;
    }
    else
    #endif
    {
        pkt_idx = txlist->dwnld_index[user_idx];
    }
    buffer = txl_buffer_alloc(txdesc, access_category, user_idx, pkt_idx);
    #else
    buffer = txl_buffer_alloc(txdesc, access_category, user_idx);
    #endif
    if (buffer != NULL)
    {
        // Update the Tx descriptor
        #if NX_AMSDU_TX
        txdesc->lmac.buffer[txlist->dwnld_index[user_idx]] = buffer;
        #else
        txdesc->lmac.buffer = buffer;
        #endif

        buffer->txdesc = txdesc;

        #if NX_AMSDU_TX
        if (txlist->dwnld_index[user_idx] > 0)
            // Set start and end pointers to the beginning and end of the frame
            txl_buffer_update_tbd(txdesc, access_category, txlist->dwnld_index[user_idx]);
        else
        #endif
            // Set start and end pointers to the beginning and end of the frame
            txl_buffer_update_thd(txdesc, access_category);

        #if NX_AMSDU_TX
        // Increase the packet index
        #if NX_UMAC_PRESENT
        if (pkt_idx == 0xFF)
            txlist->dwnld_index[user_idx] = txdesc->host.packet_cnt;
        else
        #endif
            txlist->dwnld_index[user_idx]++;
        #endif

        // Update the status
        buffer_full = false;
    }
    else
    {
        txlist->first_to_download[user_idx] = txdesc;
    }

    return (buffer_full);
}

/**
 ****************************************************************************************
 * @brief Go through the TX list and prepare the download of the payloads.
 * The process repeats until either no more payloads have to be downloaded, or enough
 * have been downloaded
 *
 * @param[in] access_category Access category on which preparation is invoked
 * @param[in] user_idx Index of the queue to prepare inside the access_category (if MU-MIMO
 * is not used, this parameter is supposed to be always 0)
 ****************************************************************************************
 */
static void txl_transmit_prep(int access_category, uint8_t user_idx)
{
    bool buffer_full = false;
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];

    // Pursue transfer of pending payloads in the Tx list
    while (1)
    {
        // Pick first descriptor from the pending list
        struct txdesc *txdesc = txlist->first_to_download[user_idx];

        // Check if we have enough data in advance
        if (txl_buffer_count(access_category, user_idx) >= TX_MIN_DOWNLOAD_CNT)
            break;

        if (txdesc == NULL)
            // Exit the loop as there are no more descriptors to handle
            break;

        // Ask for the transfer of the payload(s) corresponding to the current descriptor
        buffer_full = txl_payload_transfer(txdesc, access_category, user_idx);

        if (buffer_full)
            // Exit the loop as we cannot allocate any more buffer space
            break;

        #if NX_AMSDU_TX
        if (txlist->dwnld_index[user_idx] >= txdesc->host.packet_cnt)
        #endif
        {
            // Go to the next element
            txlist->first_to_download[user_idx] = tx_desc_next(txdesc);
            #if NX_AMSDU_TX
            // And reset the packet index
            txlist->dwnld_index[user_idx] = 0;
            #endif
        }
    }
}
#endif //!NX_FULLY_HOSTED


/**
 ****************************************************************************************
 * @brief Initialization of the HW descriptors prior to payload download
 *
 * This primitive configures the HW descriptors prior to the download of the payloads
 *
 * @param[in] txdesc          Descriptor of the packet to be pushed into the Tx queue
 * @param[in] access_category TX queue index
 *
 ****************************************************************************************
 */
static void txl_hwdesc_config_pre(struct txdesc *txdesc, int access_category)
{
    struct tx_hd *txhd = &txdesc->lmac.hw_desc->thd;
    #if NX_UMAC_PRESENT
    int add_len = txdesc->umac.head_len + txdesc->umac.tail_len;
    #else
    int add_len = 0;
    #endif

    // Set frame length
    #if NX_AMSDU_TX
    txhd->frmlen = add_len + MAC_FCS_LEN;
    for (int i = 0; i < txdesc->host.packet_cnt; i++)
    {
        txhd->frmlen += txdesc->host.packet_len[i];
    }
    #else
    txhd->frmlen = txdesc->host.packet_len + add_len + MAC_FCS_LEN;
    #endif

    // Reset the optional length fields
    txhd->optlen[0] = 0;
    txhd->optlen[1] = 0;
    txhd->optlen[2] = 0;

    // Reset the THD pattern
    txhd->upatterntx = TX_HEADER_DESC_PATTERN;

    // By default no next MPDU descriptor chained
    txhd->nextmpdudesc_ptr = 0;

    // No next Frame Exchange chained
    txhd->nextfrmexseq_ptr = 0;

    #if !NX_FULLY_HOSTED
    // No payload descriptor
    txhd->first_pbd_ptr = 0;

    // Reset data pointers
    txhd->datastartptr = 0;
    txhd->dataendptr = 0;
    #endif

    // Reset policy table pointer
    txhd->policyentryaddr = 0;

    // Set default WhichDesc value
    txhd->macctrlinfo2 = WHICHDESC_UNFRAGMENTED_MSDU;

    // Set frame lifetime
    txhd->frmlifetime = 0;

    // Set status as 0
    txhd->statinfo = 0;
}


/**
 ****************************************************************************************
 * @brief Initialization of the HW descriptors, after payload download
 *
 * This primitive configures the HW descriptors prior to transmission
 *
 * @param[in] txdesc          Descriptor of the packet to be pushed into the Tx queue
 * @param[in] access_category TX queue index
 *
 ****************************************************************************************
 */
static void txl_hwdesc_config_post(struct txdesc *txdesc, uint8_t access_category)
{
    struct txl_buffer_control *bufctrl = txl_buffer_control_get(txdesc);
    struct tx_hd *txhd = &txdesc->lmac.hw_desc->thd;
    struct tx_policy_tbl *pt = &bufctrl->policy_tbl;
    uint32_t smoothing = SMOOTHING_TX_BIT;

    #if NX_AMPDU_TX
    #if RW_MUMIMO_TX_EN
    // Check if the MPDU is inside a MU-MIMO PPDU
    if (is_in_mumimo_ppdu(txdesc))
    {
        // Handle differently the primary and secondary users
        if (is_primary_user(txdesc))
        {
            // only on the first descriptor
            if (is_mpdu_first(txdesc))
            {
                struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;
                struct tx_hd *a_thd = &agg_desc->a_thd;
                uint8_t smm_index;

                // Sanity check - The TX descriptor shall have a beamforming node attached
                ASSERT_ERR(txdesc->lmac.bfr_node);

                // Get the SMM index from the BFR node
                smm_index = txdesc->lmac.bfr_node->smm_index;

                // first MPDU of the A-MPDU has been downloaded
                agg_desc->status |= AGG_FIRST_DOWNLOADED;

                //set policy table pointer
                a_thd->policyentryaddr = CPU2HW(pt);

                // A-THD phy control information (don't care about GroupId, PAID and MU-MIMO
                // TX bit set by the host as these fields are handled internally in case
                // of MU-MIMO transmission)
                a_thd->phyctrlinfo |= bufctrl->phy_control_info &
                                      ~(PAID_TX_MASK | GID_TX_MASK | USE_MUMIMO_TX_BIT);

                // Set the rate to the correct value
                ASSERT_ERR((txdesc->umac.phy_flags & VHT_NSS_MASK) == 0);

                #if NX_MAC_HE
                if (txl_he_is_he_su(txdesc->umac.phy_flags))
                {
                    txl_he_ltf_type_set(txdesc->umac.phy_flags, &pt->powercntrlinfo[0]);
                }
                #endif

                // Set the Self-CTS protection
                txl_agg_set_ampdu_protection(&txdesc->umac.phy_flags);

                // Set the first rate control step with the actual value to be used - has
                // to be done after applying the A-MPDU protection
                pt->ratecntrlinfo[0] = txdesc->umac.phy_flags;

                // Disable the STBC in case of MU-MIMO transmission
                pt->phycntrlinfo1 &= ~STBC_PT_MASK;

                // Update the policy table for beamforming
                pt->phycntrlinfo2 &= ~SMM_INDEX_PT_MASK;
                pt->phycntrlinfo2 |= BMFED_BIT | (smm_index << SMM_INDEX_PT_OFT);

                #if NX_MAC_HE
                // For HE TB we have to remember the policy table to be used in case of split A-MPDU
                memcpy(&agg_desc->pol_tbl, pt, sizeof(agg_desc->pol_tbl));
                #endif
            }
        }
        else if (!is_mpdu_agg(txdesc) || is_mpdu_first(txdesc))
        {
            struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;
            struct tx_compressed_policy_tbl *cpt = &bufctrl->comp_pol_tbl;
            uint32_t hw_key_idx = (pt->maccntrlinfo1 & KEYSRAM_INDEX_MASK) >> KEYSRAM_INDEX_OFT;
            uint32_t fec_coding = (pt->phycntrlinfo1 & FEC_CODING_PT_BIT) >> FEC_CODING_PT_OFT;
            uint32_t mcs_idx = (txdesc->umac.phy_flags & MCS_INDEX_TX_RCX_MASK)
                                                                     >> MCS_INDEX_TX_RCX_OFT;
            uint32_t smm_index;

            // Sanity check - The TX descriptor shall have a beamforming node attached
            ASSERT_ERR(txdesc->lmac.bfr_node);

            // Get the SMM index from the BFR node
            smm_index = txdesc->lmac.bfr_node->smm_index;

            // Get TX header descriptor pointer that describe the transmission
            txhd = is_mpdu_agg(txdesc) ? &agg_desc->a_thd : txhd;

            // Set the compressed policy table address
            txhd->policyentryaddr = CPU2HW(cpt);

            // Update the compressed policy table
            cpt->upatterntx = POLICY_TABLE_PATTERN;
            cpt->sec_user_control = (mcs_idx << MCS_IDX_TX_CPT_OFT) |
                                    (fec_coding << FEC_CODING_CPT_OFT) |
                                    (smm_index << SMM_INDEX_CPT_OFT) |
                                    (hw_key_idx << KEYSRAM_INDEX_CPT_OFT);
        }
    }
    else
    #endif
    // Config HW descriptor of AMPDU and BAR too
    if (is_mpdu_agg(txdesc))
    {
        // only on the first descriptor
        if (is_mpdu_first(txdesc))
        {
            struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;
            struct tx_hd *a_thd = &agg_desc->a_thd;

            //set policy table pointer
            a_thd->policyentryaddr = CPU2HW(pt);

            #if NX_MAC_HE
            if (txl_he_is_he_su(txdesc->umac.phy_flags))
            {
                txl_he_ltf_type_set(txdesc->umac.phy_flags, &pt->powercntrlinfo[0]);
            }
            #endif

            // Set the Self-CTS protection
            txl_agg_set_ampdu_protection(&txdesc->umac.phy_flags);

            // Set the first rate control step with the actual value to be used - has
            // to be done after applying the A-MPDU protection
            pt->ratecntrlinfo[0] = txdesc->umac.phy_flags;

            #if RW_BFMER_EN
            // Check if the A-MPDU is supposed to be beamformed
            if (txdesc->lmac.bfr_node)
            {
                uint8_t smm_index = txdesc->lmac.bfr_node->smm_index;

                // Update the policy table for beamforming
                pt->phycntrlinfo2 &= ~SMM_INDEX_PT_MASK;
                pt->phycntrlinfo2 |= BMFED_BIT | (smm_index << SMM_INDEX_PT_OFT);

                // No smoothing when frame is beamformed
                smoothing = 0;
            }
            #endif

            // first MPDU of the A-MPDU has been downloaded
            agg_desc->status |= AGG_FIRST_DOWNLOADED;

            //A-THD phy control information
            a_thd->phyctrlinfo = bufctrl->phy_control_info;
            //A-THD macctrlinfo1 field
            a_thd->macctrlinfo1 = bufctrl->mac_control_info;
            a_thd->macctrlinfo1 &= (~EXPECTED_ACK_MSK);
            a_thd->macctrlinfo1 |= EXPECTED_ACK_COMPRESSED_BLOCK_ACK;

            // Set smoothing
            txl_smoothing_set(a_thd, pt, smoothing);

            #if NX_MAC_HE
            // Mark the buffer as ready to be sent as HE TB singleton if required
            txl_buffer_get(txdesc)->flags |= BUF_SINGLETON_READY;

            // For HE TB we have to remember the policy table to be used in case of split A-MPDU
            memcpy(&agg_desc->pol_tbl, pt, sizeof(agg_desc->pol_tbl));
            #endif
        }
    }
    else
    #endif
    {
        #if (NX_UMAC_PRESENT)
        if (txdesc->host.flags & TXU_CNTRL_MGMT)
        {
            uint32_t mac_hdr_addr = txl_buffer_machdr_get(txdesc);
            struct mac_hdr *hdr = (struct mac_hdr *)(HW2CPU(mac_hdr_addr));

            // Perform last settings
            #if NX_MFP
            if (txdesc->host.flags & TXU_CNTRL_MGMT_ROBUST)
            {
                if (txdesc->umac.head_len)
                {
                    // Unicast frame, need to add security header
                    txu_cntrl_protect_mgmt_frame(txdesc, hdr, MAC_SHORT_MAC_HDR_LEN);
                }
                else if (txdesc->umac.tail_len)
                {
                    #if NX_AMSDU_TX
                    int len = txdesc->host.packet_len[0];
                    #else
                    int len = txdesc->host.packet_len;
                    #endif
                    uint32_t mmic_addr = 0;

                    #if NX_FULLY_HOSTED
                    mmic_addr = txu_cntrl_mgmt_mic_pbd_append(txdesc);
                    #endif

                    // Broadcast frame, need to add MGMT MIC
                    mfp_add_mgmt_mic(txdesc, mac_hdr_addr, len, mmic_addr);
                }
            }
            #endif

            txhd->macctrlinfo2 &= ~(WHICHDESC_MSK | UNDER_BA_SETUP_BIT);
            if (MAC_ADDR_GROUP(&hdr->addr1))
                txhd->macctrlinfo1 = EXPECTED_ACK_NO_ACK;
            else
                txhd->macctrlinfo1 = EXPECTED_ACK_NORMAL_ACK;

            #if NX_FULLY_HOSTED
            // Enable interrupt
            txhd->macctrlinfo2 |= INTERRUPT_EN_TX;
            #endif

            txhd->statinfo = 0;

            // Set policy table address
            txhd->policyentryaddr = CPU2HW(&bufctrl->policy_tbl);
        }
        else
        #endif //(NX_UMAC_PRESENT)
        {
            #if NX_UMAC_PRESENT && !NX_FULLY_HOSTED
            // Check if we need to add a TKIP MIC
            txu_cntrl_tkip_mic_append(txdesc, access_category);
            #endif //NX_UMAC_PRESENT && !NX_FULLY_HOSTED

            // Set MAC control info 1 field
            txhd->macctrlinfo1 = bufctrl->mac_control_info;
            #if NX_AMSDU_TX
            // Check if this packet is split across several buffers
            if (txl_buffer_is_amsdu_multi_buf(txdesc))
            {
                // Split packet - Disable HW retransmissions
                pt->maccntrlinfo2 &= ~(LONG_RETRY_LIMIT_MASK | SHORT_RETRY_LIMIT_MASK);
            }
            #endif

            #if NX_FULLY_HOSTED
            // Enable interrupt
            txhd->macctrlinfo2 |= INTERRUPT_EN_TX;
            #endif

            #if RW_BFMER_EN
            // Check if the MPDU is supposed to be beamformed
            if (txdesc->lmac.bfr_node && bfr_is_bfmed_sglt_allowed(txdesc))
            {
                uint8_t smm_index = txdesc->lmac.bfr_node->smm_index;

                // Update the policy table for beamforming
                pt->phycntrlinfo2 &= ~SMM_INDEX_PT_MASK;
                pt->phycntrlinfo2 |= BMFED_BIT | (smm_index << SMM_INDEX_PT_OFT);

                // No smoothing when frame is beamformed
                smoothing = 0;
            }
            #endif
            #if NX_MAC_HE
            // Remove the +HTC field if required
            if (!is_htc_sglt_allowed(txdesc))
                txl_buffer_remove_htc(txhd);
            #endif

            // Set policy table address
            txhd->policyentryaddr = CPU2HW(pt);
        }

        // Set phy control info
        txhd->phyctrlinfo = bufctrl->phy_control_info;

        // Set smoothing
        txl_smoothing_set(txhd, pt, smoothing);

        #if NX_MAC_HE
        // Check if the singleton shall be protected with a RTS/CTS
        txl_he_txop_dur_rtscts_thres_mpdu_check(txdesc);
        // Mark the buffer as ready to be sent as HE TB singleton if required
        txl_buffer_get(txdesc)->flags |= BUF_SINGLETON_READY;
        #endif
    }
}


/**
 ****************************************************************************************
 * @brief Format the MAC header
 *
 * @param[in] machdrptr  HW Address of the MAC header in shared memory
 *
 ****************************************************************************************
 */
static void txl_machdr_format(uint32_t machdrptr)
{
    // Read the fragmentation number of the frame
    uint16_t seq_ctrl = co_read8p(machdrptr + MAC_HEAD_CTRL_OFT) & MAC_SEQCTRL_FRAG_MSK;

    // Non QoS data - Use common sequence number.
    // Check if we need to increment the sequence number (done for first fragment only)
    if (seq_ctrl == 0)
    {
        // Increment the sequence number
        txl_cntrl_env.seqnbr++;
    }
    seq_ctrl |= txl_cntrl_env.seqnbr << MAC_SEQCTRL_NUM_OFT;

    // Write back the sequence control field
    co_write16p(machdrptr + MAC_HEAD_CTRL_OFT, seq_ctrl);
}


/**
 ****************************************************************************************
 * @brief Set the new tail bit for the specified access category.
 *
 * @param[in] access_category   Access category
 *
 ****************************************************************************************
 */
static void txl_cntrl_newtail(uint8_t access_category)
{
    // Set the header pointer and the new head bit according to the access category
    switch (access_category)
    {
        #if NX_BEACONING
        case AC_BCN:
            nxmac_dma_cntrl_set(NXMAC_TX_BCN_NEW_TAIL_BIT);
            break;
        #endif
        case AC_VO:
            nxmac_dma_cntrl_set(NXMAC_TX_AC_3_NEW_TAIL_BIT);
            break;
        case AC_VI:
            nxmac_dma_cntrl_set(NXMAC_TX_AC_2_NEW_TAIL_BIT);
            break;
        case AC_BE:
            nxmac_dma_cntrl_set(NXMAC_TX_AC_1_NEW_TAIL_BIT);
            break;
        case AC_BK:
            nxmac_dma_cntrl_set(NXMAC_TX_AC_0_NEW_TAIL_BIT);
            break;
        default:
            ASSERT_ERR(0);
            break;
    }
}

/**
 ****************************************************************************************
 * @brief Unchains the frame exchanges that have been completed
 *
 * This primitive goes through the list of completed frame exchanges, check their status,
 * and put them in the correct list (cfm or waiting block-ack)
 *
 * @param[in] access_category  Access category for the frame exchanges to unchain
 *
 ****************************************************************************************
 */
static void txl_frame_exchange_done(uint8_t access_category)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];

    // Walk through the list of completed SW descriptors
    while (1)
    {
        struct txdesc * txdesc = NULL;

        //-----------------------
        //MPDU THD status check
        //-----------------------
        #if NX_AMPDU_TX
        if (txlist->chk_state == THD_CHK_STATE)
        {
        #endif
            //pick first txdesc in transmitting list to check its doneTx status
            txdesc = (struct txdesc *)co_list_pick(&(txlist->transmitting[0]));

            // a descriptor is waiting to be confirmed
            if (txdesc != NULL)
            {
                struct tx_hd *txhd = &txdesc->lmac.hw_desc->thd;
                uint32_t txstatus = txhd->statinfo;
                struct tx_cfm_tag *cfm = txl_cfm_tag_get(txdesc);

                //---------------
                //STATUS ANALYSIS
                //---------------
                // Check the doneTx status bit
                if (txstatus & DESC_DONE_TX_BIT)
                {
                    // For profiling
                    PROF_AGG_SMPDU_DONETX_SET();

                    #if (RW_MUMIMO_TX_EN)
                    PROF_MU_USER_POS_IRQ_SET(get_user_pos(txdesc));
                    #endif

                    #if !NX_FULLY_HOSTED
                    // Free the MPDU buffers and allocate buffer(s) for the subsequent ones
                    txl_free_done_mpdu(txdesc, access_category, 0);
                    #endif

                    // Put a first status to the descriptor here, indicating it has been
                    // already handled by the HW
                    cfm->status = txstatus;

                    #if NX_AMPDU_TX
                    txl_agg_check_saved_agg_desc(access_category);

                    if (!is_mpdu_agg(txdesc))
                    #endif
                    {
                        //get the next frame exchange pointer
                        struct tx_hd *nextdesc = HW2CPU(txhd->nextfrmexseq_ptr);

                        // Check if next descriptor is ongoing or not
                        if (nextdesc != NULL)
                        {
                            #if NX_AMSDU_TX
                            struct tx_pbd *next_pbd;
                            #endif
                            #if NX_AMPDU_TX
                            struct tx_hd *a_thd = NULL;
                            // If it's an AMPDU THD, get its first MPDU, only it will have
                            // its doneTx status updated by HW
                            if ((nextdesc->macctrlinfo2 & WHICHDESC_MSK) == WHICHDESC_AMPDU_EXTRA)
                            {
                                // Save the pointer to the A-MPDU header descriptor
                                a_thd = nextdesc;

                                nextdesc = HW2CPU(nextdesc->nextmpdudesc_ptr);

                                // Sanity check: The A-THD and first MPDU THD are linked together
                                // prior to A-MPDU chaining to HW, so nextdesc cannot be NULL
                                ASSERT_ERR(nextdesc != NULL);
                            }
                            #endif

                            // Next descriptor is ongoing, don't release current descriptor
                            #if NX_AMSDU_TX
                            next_pbd = HW2CPU(nextdesc->first_pbd_ptr);
                            if ((!next_pbd || !(next_pbd->bufctrlinfo & TBD_DONE_HW)) &&
                                !(nextdesc->statinfo & DESC_DONE_TX_BIT))
                            #else
                            if (!(nextdesc->statinfo & DESC_DONE_TX_BIT))
                            #endif
                            {
                                #if NX_AMPDU_TX
                                if (a_thd != NULL)
                                    // Check if the current A-MPDU has reached the RTS/CTS retry limit
                                    txl_agg_check_rtscts_retry_limit(a_thd, access_category);
                                #endif
                                // For profiling
                                PROF_AGG_SMPDU_DONETX_CLR();

                                // Move the TX activity timeout
                                txl_timer_move(access_category);
                                break;
                            }
                        }
                        else
                        {
                            txl_timer_clear(access_category);

                            txlist->last_frame_exch = NULL;

                            #if NX_AMPDU_TX
                            if (txlist->ppdu_cnt == 1)
                            {
                                #if RW_MUMIMO_TX_EN
                                // Data path will become empty, so finish building the pending A-MPDU
                                if (txlist->mumimo.users)
                                {
                                    txl_agg_mumimo_close(access_category);
                                }
                                else
                                #endif
                                if (txlist->agg[0].desc != NULL && !macif_tx_q_has_data(access_category))
                                {
                                    // Close the pending aggregate
                                    txl_agg_finish(access_category);
                                }
                            }
                            #endif
                        }
                    }

                    // For profiling
                    PROF_AGG_SMPDU_DONETX_CLR();
                }
                // THD not yet transmitted, stop here and come back later
                else
                {
                    #if !NX_FULLY_HOSTED && NX_AMSDU_TX
                    // Maybe some A-MSDU sub-frames (i.e. TBD) are done, check that
                    txl_check_done_amsdu_subframe(txdesc, access_category, 0);
                    #endif

                    #if NX_AMPDU_TX
                    // Check if the MPDU is the first of the A-MPDU
                    if (is_mpdu_first(txdesc))
                    {
                        // Check if the current A-MPDU has reached the RTS/CTS retry limit
                        txl_agg_check_rtscts_retry_limit(&txdesc->lmac.agg_desc->a_thd,
                                                     access_category);
                    }
                    #endif
                    break;
                }

                //------------
                //CONFIRMATION
                //------------
                #if NX_AMPDU_TX
                //at last MPDU in AMPDU, we stop pushing CFMs until BAR status done;keep last mpdu pointer
                if (is_mpdu_last(txdesc))
                {
                    txlist->agg_desc = txdesc->lmac.agg_desc;
                    txlist->chk_state = BAR_THD_CHK_STATE;
                }
                #endif

                //pop picked txdesc definitively from list
                co_list_pop_front(&(txlist->transmitting[0]));

                #if NX_AMPDU_TX
                if (!is_mpdu_agg(txdesc))
                {
                    // We have one PPDU less in the TX path
                    txlist->ppdu_cnt--;
                }
                #endif

                #if NX_TX_FRAME
                // Check if the frame was generated internally
                if (is_int_frame(txdesc))
                {
                    // Put the descriptor in the CFM queue
                    txl_frame_cfm(txdesc);
                }
                else
                #endif
                {
                    // Put the descriptor in the CFM queue
                    txl_cfm_push(txdesc, txstatus, access_category);
                }

                // Move the TX activity timeout
                txl_timer_move(access_category);
            //no more descriptors in transmitting
            }
            else
            {
                // Last frame exchange, we reset the last_frame_exch flag
                txlist->last_frame_exch = NULL;
                // And we disable the TX timeout interrupt
                txl_timer_clear(access_category);
                // No more descriptors, exit the loop
                break;
            }

        #if NX_AMPDU_TX
        //-----------------------
        //BAR THD status check
        //-----------------------
        }
        else
        {
            //get the last txdesc queued into cfm list for this ac, its agg_desc and then bar_desc
            struct tx_agg_desc *agg_desc = txlist->agg_desc;
            struct tx_hd *bar_thd = &agg_desc->bar_thd;
            uint32_t bar_thd_status = bar_thd->statinfo;

            #if RW_MUMIMO_TX_EN
            if (!(agg_desc->status & AGG_INT))
            #endif
            {
                // get the next frame exchange pointer
                struct tx_hd *nextdesc = HW2CPU(bar_thd->nextfrmexseq_ptr);

                // Check if the current A-MPDU is the last one
                if ((nextdesc == NULL) && (txlist->ppdu_cnt == 1))
                {
                    #if RW_MUMIMO_TX_EN
                    // Data path will become empty, so finish building the pending A-MPDU
                    if (txlist->mumimo.users)
                    {
                        txl_agg_mumimo_close(access_category);
                    }
                    else
                    #endif
                    if (txlist->agg[0].desc != NULL && !macif_tx_q_has_data(access_category))
                    {
                        // Close the pending aggregate
                        txl_agg_finish(access_category);
                    }
                }
            }

            //BAR status is updated received
            if (bar_thd_status & DESC_DONE_TX_BIT)
            {
                // For profiling
                PROF_AGG_BAR_DONETX_SET();

                // Move the TX activity timeout
                txl_timer_move(access_category);

                if (bar_thd_status & FRAME_SUCCESSFUL_TX_BIT)
                {
                    // Check if BA has already been received
                    if (!(agg_desc->status & AGG_BA_RECEIVED))
                    {
                        int i = 0;
                        // Make a call to the RX path to get the BA that is
                        // available, waiting for the RX timer to expire
                        do
                        {
                            rxl_immediate_frame_get();
                            i++;
                        } while ((i < 5) && (!(agg_desc->status & AGG_BA_RECEIVED)));
                    }

                    // Sanity check - The BA shall now be available
                    ASSERT_REC(agg_desc->status & AGG_BA_RECEIVED);
                }
                else
                {
                    // Pop the aggregate descriptor from the list
                    co_list_pop_front(&txlist->aggregates);
                }

                // Mark aggregate as ready for confirmation
                agg_desc->status |= AGG_DONE;

                // Set an event to handle the confirmation of the queued aggregated MPDU in background
                ke_evt_set(txl_cfm_evt_bit[access_category]);

                #if RW_MUMIMO_TX_EN
                // Check if the A-MPDU was transmitted as a secondary user in a MU-MIMO PPDU
                if (agg_desc->prim_agg_desc && (agg_desc->prim_agg_desc != agg_desc))
                {
                    ASSERT_ERR(!co_list_is_empty(&agg_desc->cfm));
                    // The TX descriptors of this A-MPDU are not in the CFM queue yet,
                    // we need to move them
                    co_list_concat(&txl_cfm_env.cfmlist[access_category], &agg_desc->cfm);
                }

                // Check if this AGG descriptor is an intermediate one
                if (agg_desc->status & AGG_INT)
                {
                    // This AGG descriptor is an intermediate one, i.e it means that we
                    // are waiting for the BA or at least one of the secondary users of
                    // this transmission. We stay in BAR_THD_CHK_STATE and we just move the
                    // pointer to the AGG descriptor for which the next BA is expected
                    txlist->agg_desc = (struct tx_agg_desc *)co_list_next(&agg_desc->list_hdr);

                    // Sanity check - We shall have a valid pointer
                    ASSERT_ERR(txlist->agg_desc);
                }
                else
                #endif
                {
                    #if RW_MUMIMO_TX_EN
                    // Check if the A-MPDU was transmitted in a MU-MIMO PPDU
                    if (agg_desc->prim_agg_desc)
                    {
                        // Decrease the number of MU PPDUs chained to the HW
                        ASSERT_ERR(txl_cntrl_env.mumimo_ppdu_cnt);
                        txl_cntrl_env.mumimo_ppdu_cnt--;

                        // Check if there are still MU PPDU chained to the HW
                        if (!txl_cntrl_env.mumimo_ppdu_cnt)
                        {
                            // No MU PPDUs chained anymore, re-enable the BW drop
                            nxmac_drop_to_lower_bw_setf(1);
                        }
                    }
                    #endif

                    // We have one PPDU less in the TX path
                    txlist->ppdu_cnt--;

                    //bar has been handled, BA reception status will be known at cfm,
                    //change state back to MPDUs check
                    txlist->chk_state = THD_CHK_STATE;
                    txlist->agg_desc = NULL;

                    // Check if we have another PPDU chained to this one
                    if (bar_thd->nextfrmexseq_ptr != 0)
                    {
                        // A PPDU is chained, therefore we have to wait for its transmission
                        // before freeing the current A-MPDU descriptor
                        txlist->agg_desc_prev = agg_desc;

                        // Increase the user count of the A-MPDU descriptor
                        agg_desc->user_cnt++;
                    }
                    else
                    {
                        // Last frame exchange, we reset the last_frame_exch flag
                        txlist->last_frame_exch = NULL;
                    }
                }

                // For profiling
                PROF_AGG_BAR_DONETX_CLR();
            }
            //BAR status not updated yet, exit and change nothing
            else
            {
                break;
            }
        }
        #endif
    }
}


/**
 ****************************************************************************************
 * @brief Manage atomic frame exchange formatting and chaining of previous one to it
 *
 * @param[in]       txdesc  First Txdesc of the next frame exchange
 * @param[in]       buffer  Pointer to the buffer that was just downloaded
 * @param[in]       access_category  Access category for the frame exchanges to format
 *
 ****************************************************************************************
 */
static void txl_frame_exchange_manage(struct txdesc *txdesc,
                                      struct txl_buffer_tag *buffer,
                                      uint8_t  access_category)
{
    struct tx_hd *new_hd = NULL;
    struct tx_hd *next_prev_hd = NULL;

    #if NX_AMPDU_TX
    #if RW_MUMIMO_TX_EN
    if (is_in_mumimo_ppdu(txdesc))
    {
        struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;
        struct tx_agg_desc *prim_agg_desc = agg_desc->prim_agg_desc;

            // Check if enough data is allocated
        if (!(buffer->flags & BUF_ALLOC_OK))
            return;

        // Clear the corresponding bit in the primary aggregate descriptor
        prim_agg_desc->download &= ~CO_BIT(buffer->user_idx);

        // Check if enough data has been allocated for all users in this MU-MIMO PPDU
        if (prim_agg_desc->download)
            return;

        // A new frame exchange starts
        new_hd = &prim_agg_desc->a_thd;

        next_prev_hd = prim_agg_desc->last_bar_thd;

        // Check if this will be the only MU PPDU chained to the HW
        if (!txl_cntrl_env.mumimo_ppdu_cnt)
        {
            // Disable the BW drop as long as MU PPDUs are chained
            nxmac_drop_to_lower_bw_setf(0);
        }

        // Increase the number of MU PPDUs chained to the HW
        txl_cntrl_env.mumimo_ppdu_cnt++;
    }
    else
    #endif
    // extra THDs chaining
    if (is_mpdu_agg(txdesc))
    {
        struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;

        // Check if enough data is allocated
        if (!(buffer->flags & BUF_ALLOC_OK))
            return;

        // A-MPDU has now enough data downloaded to be chained
        agg_desc->status |= AGG_DOWNLOADED;

        // Check if A-MPDU is already formatted
        if (!(agg_desc->status & AGG_FORMATTED))
            // A-MPDU is not yet formatted, therefore it cannot be chained
            // Once formatted, the chaining will be done in txl_agg_finish()
            return;

        // A new frame exchange starts
        new_hd = &agg_desc->a_thd;

        next_prev_hd = &agg_desc->bar_thd;
    }
    else
    #endif
    {
        // It will start an atomic frame exchange by itself
        new_hd = &txdesc->lmac.hw_desc->thd;
        next_prev_hd = new_hd;
    }

    #if TRACE_COMPO(LMAC)
    if (txdesc->lmac.hw_desc->thd.frmlen)
    {
        struct mac_hdr *mac_hdr = (struct mac_hdr *)HW2CPU(txl_buffer_machdr_get(txdesc));
        if (access_category < AC_MAX)
        {
            TRACE_LMAC(TX, "[AC %d]{VIF-%d} to STA-%d %fc SN=%d",
                       access_category, txdesc->host.vif_idx, txdesc->host.staid,
                       mac_hdr->fctl, (mac_hdr->seq >> MAC_SEQCTRL_NUM_OFT));
        }
        else
        {
            TRACE_AP(BCN, "{VIF-%d} SN=%d (tbtt in %dus)", txdesc->host.vif_idx,
                     (mac_hdr->seq >> MAC_SEQCTRL_NUM_OFT), nxmac_next_tbtt_get() << 5);
        }
    }
    else
    {
        TRACE_LMAC(TX, "[AC %d]{VIF-%d} to STA-%d Null Data Packet",
                   access_category, txdesc->host.vif_idx, txdesc->host.staid);
    }
    #endif // TRACE_COMPO(LMAC)

    // Chain the THD to the MAC HW
    txl_frame_exchange_chain(new_hd, next_prev_hd, access_category);
}

#if (NX_TX_FRAME)
#if (!NX_UMAC_PRESENT) && (NX_CHNL_CTXT || NX_P2P)
/**
 ****************************************************************************************
 * @brief Discard a packet ready for TX
 *
 * @param[in]       txdesc  First Txdesc of the next frame exchange
 * @param[in]       access_category  Access category for the frame exchanges to format
 *
 ****************************************************************************************
 */
static void txl_cntrl_discard(struct txdesc *txdesc, uint8_t access_category)
{
    PROF_CHAN_CTXT_TX_DISCARD_SET();

    #if NX_POWERSAVE
    // Increase the number of packets in the TX path - needed because decreased when confirmation is sent
    txl_cntrl_env.pck_cnt++;
    #endif

    txdesc->lmac.hw_desc->cfm.status = DESC_DONE_SW_TX_BIT;

    // Push the descriptor in the confirmation queue
    GLOBAL_INT_DISABLE();
    co_list_push_back(&txl_cfm_env.cfmlist[access_category], (struct co_list_hdr *)txdesc);
    GLOBAL_INT_RESTORE();

    // Set an event to handle the confirmation in background
    ke_evt_set(txl_cfm_evt_bit[access_category]);

    PROF_CHAN_CTXT_TX_DISCARD_CLR();
}
#endif //(!NX_UMAC_PRESENT)

/**
 ****************************************************************************************
 * @brief Postpone an internal packet ready for TX
 *
 * @param[in]       txdesc    First Txdesc of the next frame exchange
 * @param[in]       access_category  Access category for the frame exchanges to format
 *
 ****************************************************************************************
 */
static void txl_cntrl_postpone(struct txdesc *txdesc, uint8_t access_category)
{
    // Get STA Information entry
    struct sta_info_tag *sta = &sta_info_tab[txdesc->host.staid];
    // Get frame descriptor
    struct txl_frame_desc_tag *frame_desc = (struct txl_frame_desc_tag *)txdesc;

    // Keep in mind that the frame has been postponed
    frame_desc->postponed = true;
    // Store access category in TID field of the TX descriptor
    txdesc->host.tid = access_category;

    // Insert descriptor in list of postponed descriptors
    co_list_push_back(&sta->tx_desc_post, &txdesc->list_hdr);

    #if (NX_UMAC_PRESENT && NX_BEACONING)
    apm_tx_int_ps_postpone(txdesc, sta);
    #endif // (NX_UMAC_PRESENT && NX_BEACONING)

}
#endif //(NX_TX_FRAME)

#if (NX_UMAC_PRESENT)
/**
 ****************************************************************************************
 * @brief Check if the PM bit monitoring shall be started on the given MAC Header
 *
 * @param[in] mac_hdr    Pointer to the MAC Header of the transmitted frame
 *
 * @return true if the PM monitoring is started, false otherwise.
 ****************************************************************************************
 */
static bool txl_cntrl_start_pm_mon(struct mac_hdr *mac_hdr)
{
    do
    {
        uint16_t type_subtype = mac_hdr->fctl & MAC_FCTRL_TYPESUBTYPE_MASK;

        // Check if the transmitted frame is an association or reassociation response
        if ((type_subtype != MAC_FCTRL_ASSOCRSP) && (type_subtype != MAC_FCTRL_REASSOCRSP))
            break;

        // Check if its status is successful or not
        if (co_read16p(CPU2HW(mac_hdr) + MAC_SHORT_MAC_HDR_LEN + MAC_ASSO_RSP_STATUS_OFT) != 0)
            break;

        return (true);

    } while(0);

    return (false);
}
#endif //(NX_UMAC_PRESENT)

#if (NX_BCN_AUTONOMOUS_TX)
/**
 ****************************************************************************************
 * @brief Set and/or Save more_data flag status for frame sent on AC_BCN queue.
 *
 * AC_BCN queue is used for beacon and BC/MC traffic in PS mode.
 * In the latter case:
 * - In softmac driver cannot set MORE_DATA flag, so set it here based on the number
 *   of packet buffered for this vif in the AC_BCN queue when beacon has been sent.
 * - In both softmac and fullmac cases it is necessary to save whether MORE_DATA flag is
 *   set or not, so that bcmc traffic can be continued on next (potentially non dtim) beacon.
 *
 * @param[in] txdesc   Txdesc for the frame
 * @param[in] access_category AC used to send this frame
 * @param[in] machdrptr  Address of the MAC Header of the transmitted frame
 ****************************************************************************************
 */
static void txl_check_bcmc_status(struct txdesc *txdesc, uint8_t access_category,
                                  uint32_t machdrptr)
{
    struct vif_info_tag *vif;
    uint16_t frame_ctrl;

    if (access_category != AC_BCN)
        return;

    vif = &vif_info_tab[txdesc->host.vif_idx];

    // Read the frame control field from the MAC header
    frame_ctrl = co_read16p(machdrptr + MAC_HEAD_FCTRL_OFT);

    #if (!NX_UMAC_PRESENT)
    if (vif->u.ap.bc_mc_nb > 1) {
        frame_ctrl |= MAC_FCTRL_MOREDATA;
        vif->u.ap.bc_mc_nb--;

        // Write back the frame control field
        co_write16p(machdrptr + MAC_HEAD_FCTRL_OFT, frame_ctrl);
    }
    #endif

    if (frame_ctrl & MAC_FCTRL_MOREDATA)
        vif->u.ap.bc_mc_status |= VIF_AP_BCMC_MOREDATA;
    else
        vif->u.ap.bc_mc_status &= ~(VIF_AP_BCMC_MOREDATA);
}
#endif // NX_BCN_AUTONOMOUS_TX

/**
 ****************************************************************************************
 * @brief Perform operations on payloads that have been transfered from host memory
 *
 * This primitive is called by the TX DMA ISR in case of split partitioning and from the
 * background context or TX trigger interrupt context in case of full host partitioning.
 *
 * @param[in] txdesc          TX descriptor attached to the available payload
 * @param[in] buffer          Buffer structure attached to the available payload
 * @param[in] access_category AC used to send this frame
 *
 ****************************************************************************************
 */
static void txl_payload_handle(struct txdesc *txdesc,
                               struct txl_buffer_tag *buffer,
                               uint8_t access_category)
{
    #if NX_TX_FRAME
    if (!is_int_frame(txdesc))
    #endif
    {
        #if NX_AMSDU_TX
        if (!(buffer->flags & BUF_INT_MSDU))
        #endif
        {
            uint32_t mac_hdr_addr = txl_buffer_machdr_get(txdesc);

            #if (NX_BCN_AUTONOMOUS_TX)
            txl_check_bcmc_status(txdesc, access_category, mac_hdr_addr);
            #endif

            // Check if we need to format the MAC header
            if (!is_qos_data(txdesc))
            {
                #if (NX_UMAC_PRESENT)
                if (txdesc->host.flags & TXU_CNTRL_MGMT)
                {
                    struct mac_hdr *mac_hdr = (struct mac_hdr *)(HW2CPU(mac_hdr_addr));

                    // Check if we need to start a PM monitoring procedure
                    if (txl_cntrl_start_pm_mon(mac_hdr))
                    {
                        // Remember that the PM monitoring has been started for this frame
                        txdesc->host.flags |= TXU_CNTRL_MGMT_PM_MON;
                        // Start the monitoring
                        rxu_cntrl_monitor_pm(&mac_hdr->addr1);
                    }
                }
                #endif //(NX_UMAC_PRESENT)
                PROF_TX_PAYL_HDL_SET();
                txl_machdr_format(mac_hdr_addr);
                PROF_TX_PAYL_HDL_CLR();
            }

            // Format the TX DMA descriptor
            PROF_TX_PAYL_HDL_SET();
            txl_hwdesc_config_post(txdesc, access_category);
            PROF_TX_PAYL_HDL_CLR();
        }
    }
    #if NX_MAC_HE
    else
    {
        // Mark the buffer as ready to be sent as HE TB singleton if required
        txl_buffer_get(txdesc)->flags |= BUF_SINGLETON_READY;
    }
    #endif

    // Manage an atomic frame exchange formatting and chaining to previous one in HW
    PROF_TX_PAYL_HDL_SET();
    txl_frame_exchange_manage(txdesc, buffer, access_category);
    PROF_TX_PAYL_HDL_CLR();
}

#if !NX_FULLY_HOSTED
bool txl_payload_alloc(struct txdesc *txdesc, uint8_t access_category,
                              uint8_t user_idx)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];
    bool success = false;

    #if NX_AMSDU_TX
    while (1)
    {
        if ((txl_buffer_count(access_category, user_idx) >= TX_MIN_DOWNLOAD_CNT) &&
             (txlist->first_to_download[user_idx] == NULL))
        {
            txlist->first_to_download[user_idx] = txdesc;
            break;
        }

        success = !txl_payload_transfer(txdesc, access_category, user_idx);
        if (!success)
            break;

        if (txlist->dwnld_index[user_idx] == txdesc->host.packet_cnt)
        {
            txlist->dwnld_index[user_idx] = 0;
            break;
        }
    }
    #else
    if (txl_buffer_count(access_category, user_idx) < TX_MIN_DOWNLOAD_CNT)
        // Try to allocate a buffer and transfer it
        success = !txl_payload_transfer(txdesc, access_category, user_idx);
    else
        txlist->first_to_download[user_idx] = txdesc;
    #endif

    return (success);
}
#endif //!NX_FULLY_HOSTED

void txl_cntrl_newhead(uint32_t desc,
                       uint8_t access_category)
{
    // Set the header pointer and the new head bit according to the access category
    switch (access_category)
    {
        #if NX_BEACONING
        case AC_BCN:
            ASSERT_REC(nxmac_tx_bcn_state_getf() != 2);
            nxmac_tx_bcn_head_ptr_set(desc);
            nxmac_dma_cntrl_set(NXMAC_TX_BCN_NEW_HEAD_BIT);
            break;
        #endif
        case AC_VO:
            ASSERT_REC(nxmac_tx_ac_3_state_getf() != 2);
            nxmac_tx_ac_3_head_ptr_set(desc);
            nxmac_dma_cntrl_set(NXMAC_TX_AC_3_NEW_HEAD_BIT);
            break;
        case AC_VI:
            ASSERT_REC(nxmac_tx_ac_2_state_getf() != 2);
            nxmac_tx_ac_2_head_ptr_set(desc);
            nxmac_dma_cntrl_set(NXMAC_TX_AC_2_NEW_HEAD_BIT);
            break;
        case AC_BE:
            ASSERT_REC(nxmac_tx_ac_1_state_getf() != 2);
            nxmac_tx_ac_1_head_ptr_set(desc);
            nxmac_dma_cntrl_set(NXMAC_TX_AC_1_NEW_HEAD_BIT);
            break;
        case AC_BK:
            ASSERT_REC(nxmac_tx_ac_0_state_getf() != 2);
            nxmac_tx_ac_0_head_ptr_set(desc);
            nxmac_dma_cntrl_set(NXMAC_TX_AC_0_NEW_HEAD_BIT);
            break;
        default:
            ASSERT_ERR(0);
            break;
    }

    // Enable the TX queue timeout
    txl_timer_start(access_category);
}

void txl_frame_exchange_chain(struct tx_hd *first_thd,
                              struct tx_hd *last_thd,
                              uint8_t access_category)
{
    struct tx_hd *prev_hd = txl_cntrl_env.txlist[access_category].last_frame_exch;

    // Previous frame exchange, if existing, must be chained to the new frame exchange
    if (prev_hd != NULL)
    {
        // For profiling
        PROF_TX_NEW_TAIL_SET();

        // Chain the new header descriptor to the previous one
        prev_hd->nextfrmexseq_ptr = CPU2HW(first_thd);

        // Set the newTail bit
        #if NX_MAC_HE
        if (txl_he_tb_can_chain_edca(access_category))
        #endif
            txl_cntrl_newtail(access_category);

        // For profiling
        PROF_TX_NEW_TAIL_CLR();
    }
    else
    {
        // Write the header desc pointer to the HW register and set the newHead bit
        #if NX_MAC_HE
        if (txl_he_tb_can_chain_edca(access_category))
        #endif
            txl_cntrl_newhead(CPU2HW(first_thd), access_category);
    }

    // Remind the last THD of the newly established frame exchange sequence
    txl_cntrl_env.txlist[access_category].last_frame_exch = last_thd;
}

#if !NX_FULLY_HOSTED
void txl_free_done_mpdu(struct txdesc *txdesc, uint8_t access_category, uint8_t user_idx)
{
    #if NX_AMSDU_TX
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];
    #endif

    #if NX_TX_FRAME
    // Check if the frame was generated internally
    if (!is_int_frame(txdesc))
    #endif
    {
        #if NX_AMSDU_TX
        // Free all payloads associated to this MPDU
        while (txlist->tx_index[user_idx] < txdesc->host.packet_cnt)
        {
            struct txl_buffer_tag *buf = txdesc->lmac.buffer[txlist->tx_index[user_idx]];

            // Free the buffer associated with the descriptor
            if (buf != NULL)
            {
                if (txl_buffer_free(buf, access_category))
                    txl_transmit_prep(access_category, user_idx);

                // Reset the buffer pointer for this index
                txdesc->lmac.buffer[txlist->tx_index[user_idx]] = NULL;
            }
            txlist->tx_index[user_idx]++;
        };
        #else
        if (txdesc->lmac.buffer != NULL)
        {
            // Free the buffer associated with the descriptor
            if (txl_buffer_free(txdesc->lmac.buffer, access_category))
                txl_transmit_prep(access_category, user_idx);
            txdesc->lmac.buffer = NULL;
        }
        #endif
    }

    #if NX_AMSDU_TX
    // Reset the TX packet index
    txlist->tx_index[0] = 0;
    #endif // NX_AMSDU_TX
}

#if NX_AMSDU_TX
void txl_check_done_amsdu_subframe(struct txdesc *txdesc, uint8_t access_category,
                                   uint8_t user_idx)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];

    #if NX_UMAC_PRESENT
    if (txl_buffer_is_amsdu_multi_buf(txdesc))
    #else
    if (is_mpdu_split(txdesc))
    #endif
    {
        // Check if payload buffer descriptor has been handled by HW
        while (txlist->tx_index[user_idx] < txdesc->host.packet_cnt)
        {
            struct txl_buffer_tag *buf = txdesc->lmac.buffer[txlist->tx_index[user_idx]];
            struct tx_pbd *tbd;

            // Check if the buffer is available
            if (buf == NULL)
                break;

            // Get the transmit buffer descriptor linked to this payload
            tbd = &buf->tbd;
            if (buf->flags & BUF_SPLIT)
                tbd = HW2CPU(tbd->next);

            // Check if this TBD is done by the HW
            if (!(tbd->bufctrlinfo & TBD_DONE_HW))
                break;

            // Free the buffer associated with the descriptor
            if (txl_buffer_free(buf, access_category))
                txl_transmit_prep(access_category, user_idx);

            // Reset the buffer pointer for this index
            txdesc->lmac.buffer[txlist->tx_index[user_idx]] = NULL;

            txlist->tx_index[user_idx]++;
        }
    }
}
#endif // NX_AMSDU_TX
#endif // !NX_FULLY_HOSTED

void txl_cntrl_init(void)
{
    int i, j;

    #if NX_AMPDU_TX
    txl_agg_init();
    #endif
    tx_txdesc_init();
    txl_buffer_init();
    txl_cfm_init();
    #if NX_TX_FRAME
    txl_frame_init(false);
    #endif
    #if NX_MAC_HE
    txl_he_init();
    #endif

    memset(&txl_cntrl_env, 0, sizeof(txl_cntrl_env));

    // Initialize TX Lists
    for (i=0; i<NX_TXQ_CNT; i++)
    {
        for (j=0; j<RW_USER_MAX; j++)
        {
            co_list_init(&(txl_cntrl_env.txlist[i].transmitting[j]));
        }

        #if RW_MUMIMO_TX_EN
        txl_cntrl_env.txlist[i].mumimo.open = MU_USER_MASK;
        #endif
        txl_cntrl_env.txlist[i].last_frame_exch = NULL;
        txl_cntrl_env.txlist[i].bridgedmacnt = dma_lli_counter_get(TX_AC2LLI(i));
        txl_cntrl_env.txlist[i].chk_state = THD_CHK_STATE;

        #if NX_AMPDU_TX
        txl_cntrl_env.txlist[i].ppdu_cnt = 0;
        txl_cntrl_env.txlist[i].agg_desc = NULL;
        txl_cntrl_env.txlist[i].agg_desc_prev = NULL;
        co_list_init(&(txl_cntrl_env.txlist[i].aggregates));
        #endif
    }

    // Initialize non-QoS sequence number
    txl_cntrl_env.seqnbr = 0;
}

#if (NX_TX_FRAME)
bool txl_cntrl_tx_check(struct vif_info_tag *vif)
{
    // Do not accept frames for tx during reset procedure
    if (txl_cntrl_env.reset)
    {
        return (false);
    }

    #if (NX_CHNL_CTXT)
    if (!chan_is_tx_allowed(vif))
    {
        return (false);
    }
    #endif //(NX_CHNL_CTXT)

    #if (NX_P2P)
    // If P2P interface, check if GO is present
    if (vif->p2p && !p2p_is_present(vif->p2p_index))
    {
        return (false);
    }
    #endif //(NX_P2P)

    return (true);
}
#endif //(NX_TX_FRAME)

void txl_cntrl_halt_ac(uint8_t access_category)
{
    // Clear the TX timer
    txl_timer_clear(access_category);

    // Set the header pointer and the new head bit according to the access category
    switch (access_category)
    {
        #if NX_BEACONING
        case AC_BCN:
            nxmac_dma_cntrl_set(NXMAC_HALT_BCN_AFTER_TXOP_BIT);
            while(nxmac_tx_bcn_state_getf() != 0);
            nxmac_dma_cntrl_clear(NXMAC_HALT_BCN_AFTER_TXOP_BIT);
            break;
        #endif
        case AC_VO:
            nxmac_dma_cntrl_set(NXMAC_HALT_AC_3_AFTER_TXOP_BIT);
            while(nxmac_tx_ac_3_state_getf() != 0);
            nxmac_dma_cntrl_clear(NXMAC_HALT_AC_3_AFTER_TXOP_BIT);
            break;
        case AC_VI:
            nxmac_dma_cntrl_set(NXMAC_HALT_AC_2_AFTER_TXOP_BIT);
            while(nxmac_tx_ac_2_state_getf() != 0);
            nxmac_dma_cntrl_clear(NXMAC_HALT_AC_2_AFTER_TXOP_BIT);
            break;
        case AC_BE:
            nxmac_dma_cntrl_set(NXMAC_HALT_AC_1_AFTER_TXOP_BIT);
            while(nxmac_tx_ac_1_state_getf() != 0);
            nxmac_dma_cntrl_clear(NXMAC_HALT_AC_1_AFTER_TXOP_BIT);
            break;
        case AC_BK:
            nxmac_dma_cntrl_set(NXMAC_HALT_AC_0_AFTER_TXOP_BIT);
            while(nxmac_tx_ac_0_state_getf() != 0);
            nxmac_dma_cntrl_clear(NXMAC_HALT_AC_0_AFTER_TXOP_BIT);
            break;
        default:
            ASSERT_ERR(0);
            break;
    }
}

void txl_cntrl_flush_ac(uint8_t access_category, uint32_t status)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];
    int i;

    // Flush all the queues
    txl_cfm_flush(access_category, &txl_cfm_env.cfmlist[access_category], status);
    for (i=0; i < RW_USER_MAX; i++)
    {
        txl_cfm_flush(access_category, &txlist->transmitting[i], status);
    }

    // Reset the list flags
    txlist->last_frame_exch = NULL;
    for (i=0; i < RW_USER_MAX; i++)
    {
        txlist->first_to_download[i] = NULL;
    }

    #if !NX_FULLY_HOSTED
    // Reset the TX buffer
    txl_buffer_reset(access_category);
    #endif

    // Clear the TX timer
    txl_timer_clear(access_category);
}

bool txl_cntrl_push(struct txdesc *txdesc, uint8_t access_category)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];
    #if RW_MUMIMO_TX_EN
    int status;
    #endif

    // For profiling
    PROF_TX_AC_BG_SET(access_category);

    #if (!NX_UMAC_PRESENT)
    #if (NX_CHNL_CTXT || NX_P2P)
    struct vif_info_tag *vif = &vif_info_tab[txdesc->host.vif_idx];

    // Check if packet can be transmitted
    if (!txl_cntrl_tx_check(vif))
    {
        // Check if packet can be transmitted
        txl_cntrl_discard(txdesc, access_category);

        return (false);
    }
    #endif //(NX_CHNL_CTXT || NX_P2P)
    #endif //(!NX_UMAC_PRESENT)

    // Initialize some fields of the THD
    txl_hwdesc_config_pre(txdesc, access_category);

    GLOBAL_INT_DISABLE();
    #if NX_AMPDU_TX
    #if RW_MUMIMO_TX_EN
    status =
    #endif
    txl_agg_push_mpdu(txdesc, access_category); // Try to aggregate the descriptor
    #endif

    #if RW_MUMIMO_TX_EN
    if (status == SU_PACKET)
    #endif
    {
        #if !NX_FULLY_HOSTED
        // Check if we can try to allocate a Tx buffer
        if (txlist->first_to_download[0] == NULL)
        {
            txl_payload_alloc(txdesc, access_category, 0);
        }
        #endif
        co_list_push_back(&txlist->transmitting[0], &txdesc->list_hdr);
    }
    GLOBAL_INT_RESTORE();

    #if NX_POWERSAVE
    // Increase the number of packets in the TX path
    txl_cntrl_env.pck_cnt++;
    #endif

    #if (NX_TD)
    td_pck_ind(txdesc->host.vif_idx, txdesc->host.staid, false);
    #endif //(NX_TD)

    #if (NX_UAPSD || NX_DPSM)
    // Check if this frame will be a UAPSD trigger frame
    ps_check_tx_frame(txdesc->host.staid, txdesc->host.tid);
    #endif //(NX_UAPSD || NX_DPSM)

    #if NX_FULLY_HOSTED
    #if NX_AMPDU_TX
    if (!is_mpdu_agg(txdesc) || is_mpdu_first(txdesc))
    #endif
    {
        txl_buffer_control_copy(txdesc, txdesc->lmac.buffer);
        GLOBAL_INT_DISABLE();
        txl_payload_handle(txdesc, txdesc->lmac.buffer, access_category);
        GLOBAL_INT_RESTORE();
    }
    #endif

    #if RW_MUMIMO_TX_EN
    return (status == MU_PAUSED);
    #else
    return (false);
    #endif
}

#if (NX_UMAC_PRESENT)
void txl_cntrl_inc_pck_cnt(void)
{
    #if NX_POWERSAVE
    // Increase the number of packets in the TX path
    txl_cntrl_env.pck_cnt++;
    #endif
}
#endif //(NX_UMAC_PRESENT)

#if NX_TX_FRAME
bool txl_cntrl_push_int(struct txdesc *txdesc, uint8_t access_category)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];
    struct tx_hd *thd = &txdesc->lmac.hw_desc->thd;
    // Get VIF information entry
    struct vif_info_tag *vif = &vif_info_tab[txdesc->host.vif_idx];

    // For profiling
    PROF_TX_FRAME_PUSH_SET();

    // Check if packet can be transmitted
    if (!txl_cntrl_tx_check(vif)
        #if (NX_UMAC_PRESENT && NX_BEACONING)
        || !apm_tx_int_ps_check(txdesc)
        #endif // (NX_UMAC_PRESENT)
        #if (RW_UMESH_EN)
        || !mesh_ps_check_peer_presence(vif, txdesc->host.staid)
        #endif //(RW_UMESH_EN)
        )
    {
        if (txdesc->host.staid != INVALID_STA_IDX)
        {
            txl_cntrl_postpone(txdesc, access_category);

            return (true);
        }
        else
        {
            // Push back the TX Descriptor in the list of free descriptors
            txl_frame_release(txdesc, false);

            return (false);
        }
    }

    // Force a TX interrupt to be generated when the transmission is completed
    thd->macctrlinfo2 |= INTERRUPT_EN_TX;

    // Check if we can push the descriptor to the transmit queue immediately or if we
    // have to wait because an A-MPDU formatting may be ongoing
    GLOBAL_INT_DISABLE();
    #if NX_AMPDU_TX
    // We will now have some data programmed for transmission, so we can
    // activate the A-MPDU optimization
    txl_cntrl_env.txlist[access_category].ppdu_cnt++;
    // Close the pending aggregate
    #if RW_MUMIMO_TX_EN
    if (txlist->mumimo.users)
    {
        txl_agg_mumimo_close(access_category);
    }
    else
    #endif
    if (txlist->agg[0].desc != NULL)
    {
        txl_agg_finish(access_category);
    }
    #endif

    #if !NX_FULLY_HOSTED
    // If data path is ready, program the fake DMA transfer
    if (txlist->first_to_download[0] == NULL)
        txl_int_fake_transfer(txdesc, access_category);
    #endif

    // Push the descriptor to the TX list
    co_list_push_back(&txlist->transmitting[0], &txdesc->list_hdr);
    GLOBAL_INT_RESTORE();

    #if (NX_POWERSAVE)
    // Increase the number of packets in the TX path
    txl_cntrl_env.pck_cnt++;
    #endif

    #if NX_FULLY_HOSTED
    GLOBAL_INT_DISABLE();
    txl_payload_handle(txdesc, txl_buffer_get(txdesc), access_category);
    GLOBAL_INT_RESTORE();
    #endif

    // For profiling
    PROF_TX_FRAME_PUSH_CLR();

    return (true);
}
#endif

#if !NX_FULLY_HOSTED
void txl_cntrl_dma_isr(void)
{
    uint32_t irqstatus;

    // For profiling
    PROF_TX_DMA_IRQ_SET();

    // read the currently pending IPC interrupt and mask only the LLI IRQs for TX
    irqstatus = dma_int_status_get() & IPC_DMA_LLI_TX_MASK;

    // Walk through all access categories for which DMA IRQ is pending
    while (irqstatus)
    {
        uint8_t access_category;
        struct txdesc *txdesc;
        struct txl_list *txlist;
        uint8_t lliidx = (31 - DMA_LLI_IRQ_LSB) - co_clz(irqstatus);

        // Retrieve the access category from the DMA index
        access_category = TX_LLI2AC(lliidx);
        txlist = &txl_cntrl_env.txlist[access_category];

        // acknowledge the IPC interrupt BEFORE reading the LLI counter
        dma_int_ack_clear(CO_BIT(lliidx + DMA_LLI_IRQ_LSB));

        // For profiling
        PROF_TX_AC_IRQ_SET(access_category);

        // Walk through the Tx list to handle payloads that have been downloaded
        while (txlist->bridgedmacnt != dma_lli_counter_get(lliidx))
        {
            struct txl_buffer_tag *buffer = txl_buffer_pop(access_category);

            // Sanity check - There shall be a buffer in the list
            ASSERT_ERR(buffer != NULL);

            // Retrieve the descriptor from the buffer
            txdesc = buffer->txdesc;

            // Acknowledge the IPC interrupt BEFORE reading the LLI counter
            dma_int_ack_clear(CO_BIT(lliidx + DMA_LLI_IRQ_LSB));

            // Perform the operations required once payload is available
            txl_payload_handle(txdesc, buffer, access_category);

            // Increment SW bridge DMA counter to keep track with HW one
            txlist->bridgedmacnt++;
        }

        // re-read the currently pending IPC interrupt and mask only the LLI IRQs for TX
        irqstatus = dma_int_status_get() & IPC_DMA_LLI_TX_MASK;
    }

    // For profiling
    PROF_TX_DMA_IRQ_CLR();
}
#endif // !NX_FULLY_HOSTED

void txl_prot_trigger(void)
{
    uint32_t status;

    // retrieve the hardware status
    status = nxmac_tx_rx_int_status_get() & TX_PROT_IRQ;

    #if NX_MAC_HE
    if (status & NXMAC_TB_PROT_TRIGGER_BIT)
    {
        txl_he_tb_prot_trigger();
    }
    #endif

    #if NX_BW_LEN_ADAPT
    if (status & TX_BW_DROP_IRQ)
    {
        uint8_t access_category;
        uint32_t bw_drop_status = status & TX_BW_DROP_IRQ;

        // For profiling
        PROF_BW_DROP_IRQ_SET();

        // compute the access category
        access_category =  31 - co_clz(bw_drop_status) - NXMAC_AC_0BW_DROP_TRIGGER_POS;

        // For profiling
        PROF_TX_AC_IRQ_SET(access_category);

        // handle the interrupt
        txl_agg_bw_drop_handle(access_category);

        // For profiling
        PROF_BW_DROP_IRQ_CLR();
    }
    #endif

    // acknowledge the interrupt
    nxmac_tx_rx_int_ack_clear(status);
}

void txl_transmit_trigger(void)
{
    uint8_t access_category;
    uint32_t status;

    // retrieve the hardware status (process only the tx part)
    status = nxmac_tx_rx_int_status_get() & TX_TRANSMIT_IRQ;

    #if NX_MAC_HE
    // Check if the HE TB transmission has been cancelled by the HW
    if (status & NXMAC_TB_TX_CANCELLED_BIT)
    {
        // Cancel the HE TB PPDU transmission
        txl_he_tb_transmit_cancelled();
        nxmac_tx_rx_int_ack_clear(NXMAC_TB_TX_CANCELLED_BIT);
    }
    // Check if a trigger based interrupt is pending
    if (status & (NXMAC_TB_TX_TRIGGER_BIT | NXMAC_TB_TX_BUF_TRIGGER_BIT))
    {
        // Get the HE TB TX confirmations
        txl_he_tb_transmit_trigger();
        nxmac_tx_rx_int_ack_clear(NXMAC_TB_TX_TRIGGER_BIT | NXMAC_TB_TX_BUF_TRIGGER_BIT);
    }
    #endif

    #if RW_MUMIMO_TX_EN
    // Check if secondary user interrupts are pending
    if (status & NXMAC_SEC_USER_TX_TRIGGER_BIT)
        txl_agg_sec_transmit_trigger();
    #endif

    #if NX_AMSDU_TX
    // don't care about tx header or buffer trigger, only access category, so duplicate all in tx
    status |= status >> (NXMAC_AC_0_TX_BUF_TRIGGER_POS - NXMAC_AC_0_TX_TRIGGER_POS);
    #endif

    // process only the tx part (and duplicated buffer trigger)
    status &= TX_IRQ_BITS;

    // Check if a TX interrupt is pending
    if (status == 0)
        return;

    // For profiling
    PROF_TX_MAC_IRQ_SET();

    // compute the access category
    access_category =  31 - co_clz(status) - NXMAC_AC_0_TX_TRIGGER_POS;

    // sanity check
    ASSERT_ERR(access_category < NX_TXQ_CNT);

    // clear the handled interrupts (buffer and TX)
    #if NX_AMSDU_TX
    nxmac_tx_rx_int_ack_clear(CO_BIT(access_category + NXMAC_AC_0_TX_BUF_TRIGGER_POS) |
                              CO_BIT(access_category + NXMAC_AC_0_TX_TRIGGER_POS));
    #else
    nxmac_tx_rx_int_ack_clear(CO_BIT(access_category + NXMAC_AC_0_TX_TRIGGER_POS));
    #endif

    // For profiling
    PROF_TX_AC_IRQ_SET(access_category);

    // Go through the list of already transmitted packets to release them
    txl_frame_exchange_done(access_category);

    // For profiling
    PROF_TX_MAC_IRQ_CLR();
}

void txl_current_desc_get(int access_category, struct tx_hd **thd)
{
    #if NX_AMPDU_TX
    if (txl_cntrl_env.txlist[access_category].chk_state != THD_CHK_STATE)
    {
        // A-MPDU case, start from the A-THD
        struct tx_agg_desc *agg_desc = txl_cntrl_env.txlist[access_category].agg_desc;
        *thd = &agg_desc->a_thd;
    }
    else
    #endif
    {
        // Pick first txdesc in transmitting list
        struct txdesc *txdesc = (struct txdesc *)co_list_pick(&(txl_cntrl_env.txlist[access_category].transmitting[0]));

        // A descriptor is waiting to be confirmed
        if (txdesc != NULL)
        {
            #if NX_AMPDU_TX
            // Check if the packet is part of an A-MPDU
            if (is_mpdu_agg(txdesc))
            {
                // A-MPDU case, start from the A-THD
                struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;
                *thd = &agg_desc->a_thd;
            }
            else
            #endif
            {
                struct tx_hd *txhd = &txdesc->lmac.hw_desc->thd;

                // Singleton case, start with the first THD chained to the HW
                *thd = txhd;
            }
        }
        else
        {
            // No TX ongoing
            *thd = NULL;
        }
    }
}

void txl_reset(void)
{
    int i,j;
    uint16_t seq_num = txl_cntrl_env.seqnbr;

    // Clear the possibly pending TX kernel events
    ke_evt_clear(KE_EVT_TXCFM_MASK);

    // Wait for all DMA transfers to be finished
    while (dma_dma_status_oft_free_getf() != DMA_OFT_FREE_MASK);

    #if (NX_TX_FRAME)
    // Set reset mode
    txl_cntrl_env.reset = true;
    #endif //(NX_TX_FRAME)

    for (i = 0; i < NX_TXQ_CNT; i++)
    {
        uint8_t lliidx = TX_AC2LLI(i);
        struct txl_list *txlist = &txl_cntrl_env.txlist[i];

        // Acknowledge the DMA interrupt
        dma_int_ack_clear(CO_BIT(lliidx));

        // Flush all the queues
        txl_cfm_flush(i, &txl_cfm_env.cfmlist[i], DESC_DONE_SW_TX_BIT);
        for (j=0; j<RW_USER_MAX; j++)
        {
            txl_cfm_flush(i, &txlist->transmitting[j], DESC_DONE_SW_TX_BIT);
        }
        #if RW_MUMIMO_TX_EN
        for (j=0; j<RW_USER_MAX; j++)
        {
            if (txlist->mumimo.users & CO_BIT(j))
                txl_cfm_flush(i, &txlist->mumimo.tx[j], DESC_DONE_SW_TX_BIT);

            if (txlist->mumimo.txdesc[j] != NULL)
                txl_cfm_flush_desc(i, txlist->mumimo.txdesc[j], DESC_DONE_SW_TX_BIT);
        }
        while (1)
        {
            struct tx_agg_desc *agg_desc = (struct tx_agg_desc *)co_list_pop_front(&txlist->aggregates);

            if (agg_desc == NULL)
                break;

            txl_cfm_flush(i, &agg_desc->cfm, DESC_DONE_SW_TX_BIT);
        }
        #endif
    }

    // Reset the required data structures
    #if NX_AMPDU_TX
    txl_agg_reset();
    #endif
    #if !NX_FULLY_HOSTED
    txl_buffer_reinit();
    #endif
    txl_cfm_init();
    #if NX_MAC_HE
    txl_he_reset();
    #endif

    memset(&txl_cntrl_env, 0, sizeof(txl_cntrl_env));

    #if RW_MUMIMO_TX_EN
    // No MU PPDUs chained anymore, re-enable the BW drop
    nxmac_drop_to_lower_bw_setf(1);
    #endif

    // Put back the value of the sequence number
    txl_cntrl_env.seqnbr = seq_num;

    for (i = 0; i < NX_TXQ_CNT; i++)
    {
        for (j=0; j<RW_USER_MAX; j++)
        {
            co_list_init(&(txl_cntrl_env.txlist[i].transmitting[j]));
        }

        #if RW_MUMIMO_TX_EN
        txl_cntrl_env.txlist[i].mumimo.open = MU_USER_MASK;
        macif_tx_enable_users(i, MU_USER_MASK);
        #endif

        txl_cntrl_env.txlist[i].last_frame_exch = NULL;
        txl_cntrl_env.txlist[i].bridgedmacnt = dma_lli_counter_get(TX_AC2LLI(i));
        txl_cntrl_env.txlist[i].chk_state = THD_CHK_STATE;

        #if NX_AMPDU_TX
        txl_cntrl_env.txlist[i].ppdu_cnt = 0;
        txl_cntrl_env.txlist[i].agg_desc = NULL;
        txl_cntrl_env.txlist[i].agg_desc_prev = NULL;
        co_list_init(&(txl_cntrl_env.txlist[i].aggregates));
        #endif
    }
}



/// @}  // end of group TX_CNTRL
