/**
 ****************************************************************************************
 *
 * @file txl_frame.c
 *
 * @brief Management of internal frame generation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/*
 ****************************************************************************************
 * @addtogroup TX_FRM
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include <stddef.h>
#include "txl_cntrl.h"
#include "txl_frame.h"
#include "ke_event.h"
#include "mac_frame.h"
#include "dbg.h"
#include "txl_buffer.h"
#include "co_utils.h"
#include "reg_mac_core.h"
#include "vif_mgmt.h"
#include "phy.h"
#include "tpc.h"
#include "tdls.h"

#if NX_TX_FRAME
/*
 * STRUCTURE DEFINITION
 ****************************************************************************************
 */
/*
 * GLOBAL VARIABLE DEFINITION
 ****************************************************************************************
 */
/// Pool of TX frame descriptors
static struct txl_frame_desc_tag txl_frame_desc[NX_TXFRAME_CNT];

/// TX frame environment variables
struct txl_frame_env_tag txl_frame_env;

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */
void txl_frame_init_desc(struct txl_frame_desc_tag *frame,
                         struct txl_buffer_tag *buffer,
                         struct tx_hw_desc *hwdesc,
                         struct txl_buffer_control *bufctrl)
{
    struct tx_hd *thd = &hwdesc->thd;

    // Reset the frame descriptors
    memset(frame, 0, sizeof(*frame));

    // Initialize the descriptor
    #if NX_AMSDU_TX
    frame->txdesc.lmac.buffer[0] = buffer;
    #else
    frame->txdesc.lmac.buffer = buffer;
    #endif
    #if NX_UMAC_PRESENT
    frame->txdesc.umac.buf_control = bufctrl;
    #endif
    frame->txdesc.lmac.hw_desc = hwdesc;
    frame->type = TX_EXT;

    // Initialize the frame buffer
    thd->upatterntx = TX_HEADER_DESC_PATTERN;
    thd->datastartptr = CPU2HW(txl_buffer_payload_get(&frame->txdesc));
    thd->frmlifetime = 0;
    thd->optlen[0] = 0;
    thd->optlen[1] = 0;
    thd->optlen[2] = 0;
    bufctrl->policy_tbl.upatterntx = POLICY_TABLE_PATTERN;
}

void txl_frame_init(bool reset)
{
    int i;
    // Policy table
    struct tx_policy_tbl *pol_tbl;

    // Initialize the lists
    co_list_init(&txl_frame_env.desc_free);
    co_list_init(&txl_frame_env.desc_done);

    // Populate the free list and initialize the TX frame descriptors
    for (i=0; i<NX_TXFRAME_CNT; i++)
    {
        struct txl_frame_desc_tag *frame = &txl_frame_desc[i];
        struct txl_buffer_tag *buffer = (struct txl_buffer_tag *)&txl_frame_pool[i];
        struct tx_hw_desc *hwdesc = &txl_frame_hwdesc_pool[i];
        struct tx_hd *thd = &hwdesc->thd;
        #if NX_UMAC_PRESENT
        struct txl_buffer_control *bufctrl = &txl_frame_buf_ctrl[i];
        #else
        struct txl_buffer_control *bufctrl = &buffer->buffer_control;
        #endif

        // Reset the frame descriptor only during init or when frame has not been postponed
        if (!reset || !frame->postponed)
        {
            // Reset the frame descriptor
            memset(frame, 0, sizeof(struct txl_frame_desc_tag));

            // Initialize the descriptor
            #if NX_AMSDU_TX
            frame->txdesc.lmac.buffer[0] = buffer;
            #else
            frame->txdesc.lmac.buffer = buffer;
            #endif
            #if NX_UMAC_PRESENT
            frame->txdesc.umac.buf_control = bufctrl;
            #endif
            frame->txdesc.lmac.hw_desc = hwdesc;
            frame->type = TX_INT;

            // Initialize the frame buffer
            thd->upatterntx = TX_HEADER_DESC_PATTERN;
            thd->datastartptr = CPU2HW(txl_buffer_payload_get(&frame->txdesc));
            thd->frmlifetime = 0;
            thd->optlen[0] = 0;
            thd->optlen[1] = 0;
            thd->optlen[2] = 0;
            bufctrl->policy_tbl.upatterntx = POLICY_TABLE_PATTERN;

            // Push it in the free list
            co_list_push_back(&txl_frame_env.desc_free, &frame->txdesc.list_hdr);
        }
    }

    // Initialize default policy tables (2.4 GHz)
    #if (NX_UMAC_PRESENT)
    pol_tbl = &txl_buffer_control_24G.policy_tbl;

    txl_buffer_control_24G.mac_control_info = 0;
    txl_buffer_control_24G.phy_control_info = 0;
    #else
    pol_tbl = &txl_frame_pol_24G;
    #endif //(NX_UMAC_PRESENT)

    pol_tbl->upatterntx        = POLICY_TABLE_PATTERN;
    pol_tbl->phycntrlinfo1     = phy_get_ntx() << NX_TX_PT_OFT;
    pol_tbl->phycntrlinfo2     = TX_NTX_2_ANTENNA_SET(phy_get_ntx());
    pol_tbl->maccntrlinfo1     = 0;
    pol_tbl->maccntrlinfo2     = 0xFFFF0704;
    pol_tbl->ratecntrlinfo[0]  = (HW_RATE_1MBPS << MCS_INDEX_TX_RCX_OFT) | PRE_TYPE_TX_RCX_MASK;
    pol_tbl->ratecntrlinfo[1]  = 0;
    pol_tbl->ratecntrlinfo[2]  = 0;
    pol_tbl->ratecntrlinfo[3]  = 0;
    pol_tbl->powercntrlinfo[1] = 0;
    pol_tbl->powercntrlinfo[2] = 0;
    pol_tbl->powercntrlinfo[3] = 0;

    // Initialize default policy tables (5 GHz)
    #if (NX_UMAC_PRESENT)
    pol_tbl = &txl_buffer_control_5G.policy_tbl;

    txl_buffer_control_5G.mac_control_info = 0;
    txl_buffer_control_5G.phy_control_info = 0;
    #else
    pol_tbl = &txl_frame_pol_5G;
    #endif //(NX_UMAC_PRESENT)

    pol_tbl->upatterntx         = POLICY_TABLE_PATTERN;
    pol_tbl->phycntrlinfo1      = phy_get_ntx() << NX_TX_PT_OFT;
    pol_tbl->phycntrlinfo2      = TX_NTX_2_ANTENNA_SET(phy_get_ntx());
    pol_tbl->maccntrlinfo1      = 0;
    pol_tbl->maccntrlinfo2      = 0xFFFF0704;
    pol_tbl->ratecntrlinfo[0]   = HW_RATE_6MBPS << MCS_INDEX_TX_RCX_OFT;
    pol_tbl->ratecntrlinfo[1]   = 0;
    pol_tbl->ratecntrlinfo[2]   = 0;
    pol_tbl->ratecntrlinfo[3]   = 0;
    pol_tbl->powercntrlinfo[1]  = 0;
    pol_tbl->powercntrlinfo[2]  = 0;
    pol_tbl->powercntrlinfo[3]  = 0;

    #if (RW_BFMER_EN)
    #if (NX_UMAC_PRESENT)
    pol_tbl = &txl_buffer_control_ndpa_brp.policy_tbl;

    txl_buffer_control_ndpa_brp.mac_control_info = 0;
    txl_buffer_control_ndpa_brp.phy_control_info = 0;
    #else
    pol_tbl = &txl_frame_pol_ndpa_brp;
    #endif //(NX_UMAC_PRESENT)

    pol_tbl->upatterntx         = POLICY_TABLE_PATTERN;
    pol_tbl->phycntrlinfo1      = phy_get_ntx() << NX_TX_PT_OFT;
    pol_tbl->phycntrlinfo2      = TX_NTX_2_ANTENNA_SET(phy_get_ntx());
    pol_tbl->maccntrlinfo1      = 0;
    pol_tbl->maccntrlinfo2      = 0xFFFF0000;
    pol_tbl->ratecntrlinfo[0]   = BW_80MHZ << BW_TX_RCX_OFT;
    pol_tbl->ratecntrlinfo[0]   |= FORMATMOD_VHT << FORMAT_MOD_TX_RCX_OFT;
    pol_tbl->ratecntrlinfo[0]   |= (0x0 << 4) << MCS_INDEX_TX_RCX_OFT;
    pol_tbl->ratecntrlinfo[1]   = 0;
    pol_tbl->ratecntrlinfo[2]   = 0;
    pol_tbl->ratecntrlinfo[3]   = 0;
    pol_tbl->powercntrlinfo[1]  = 0;
    pol_tbl->powercntrlinfo[2]  = 0;
    pol_tbl->powercntrlinfo[3]  = 0;

    #if (NX_UMAC_PRESENT)
    pol_tbl = &txl_buffer_control_ndp.policy_tbl;

    txl_buffer_control_ndp.mac_control_info = 0;
    txl_buffer_control_ndp.phy_control_info = 0;
    #else
    pol_tbl = &txl_frame_pol_ndp;
    #endif //(NX_UMAC_PRESENT)

    pol_tbl->upatterntx         = POLICY_TABLE_PATTERN;
    pol_tbl->phycntrlinfo1      = phy_get_ntx() << NX_TX_PT_OFT;
    pol_tbl->phycntrlinfo2      = TX_NTX_2_ANTENNA_SET(phy_get_ntx());
    pol_tbl->maccntrlinfo1      = 0;
    pol_tbl->maccntrlinfo2      = 0xFFFF0000;
    pol_tbl->ratecntrlinfo[0]   = 0;
    pol_tbl->ratecntrlinfo[0]   |= FORMATMOD_VHT << FORMAT_MOD_TX_RCX_OFT;
    pol_tbl->ratecntrlinfo[0]   |= (phy_get_nss() << 4) << MCS_INDEX_TX_RCX_OFT;
    pol_tbl->ratecntrlinfo[1]   = 0;
    pol_tbl->ratecntrlinfo[2]   = 0;
    pol_tbl->ratecntrlinfo[3]   = 0;
    pol_tbl->powercntrlinfo[1]  = 0;
    pol_tbl->powercntrlinfo[2]  = 0;
    pol_tbl->powercntrlinfo[3]  = 0;
    #endif //(RW_BFMER_EN)
}

struct txl_frame_desc_tag *txl_frame_get(int type, int len)
{
    // Allocate a new frame
    struct txl_frame_desc_tag *frame;

    // Allocate a new frame
    frame = (struct txl_frame_desc_tag *)co_list_pop_front(&txl_frame_env.desc_free);

    // Check if allocation was successful
    if (frame != NULL)
    {
        txl_frame_prepare(frame, type, len);
    }

    return frame;
}

void txl_frame_prepare(struct txl_frame_desc_tag *frame, int type, int len)
{
    struct txl_buffer_control *bufctrl = txl_buffer_control_get(&frame->txdesc);
    struct tx_hd *thd = &frame->txdesc.lmac.hw_desc->thd;
    struct tx_policy_tbl *pol;

    // Fill-up the TX header descriptor
    #if (RW_BFMER_EN)
    thd->frmlen = len;

    if (len == 0)
    {
        thd->datastartptr = 0;
        thd->dataendptr = 0;
    }
    else
    {
        // Get buffer
        struct txl_buffer_tag *buffer = txl_buffer_get(&frame->txdesc);

        thd->frmlen += MAC_FCS_LEN;
        thd->datastartptr = CPU2HW(buffer->payload);
        thd->dataendptr = (uint32_t)thd->datastartptr + len - 1;
    }
    #else
    thd->dataendptr = (uint32_t)thd->datastartptr + len - 1;
    thd->frmlen = len + MAC_FCS_LEN;
    #endif //(RW_BFMER_EN)

    switch (type)
    {
        case TX_DEFAULT_24G:
            #if (NX_UMAC_PRESENT)
            pol = &txl_buffer_control_24G.policy_tbl;
            #else
            pol = &txl_frame_pol_24G;
            #endif //(NX_UMAC_PRESENT)
            break;
        case TX_DEFAULT_5G:
            #if (NX_UMAC_PRESENT)
            pol = &txl_buffer_control_5G.policy_tbl;
            #else
            pol = &txl_frame_pol_5G;
            #endif //(NX_UMAC_PRESENT)
            break;
            #if (RW_BFMER_EN)
        case TX_DEFAULT_NDPA_BRP:
            #if (NX_UMAC_PRESENT)
            pol = &txl_buffer_control_ndpa_brp.policy_tbl;
            #else
            pol = &txl_frame_pol_ndpa_brp;
            #endif //(NX_UMAC_PRESENT)
            break;
        case TX_DEFAULT_NDP:
            #if (NX_UMAC_PRESENT)
            pol = &txl_buffer_control_ndp.policy_tbl;
            #else
            pol = &txl_frame_pol_ndp;
            #endif //(NX_UMAC_PRESENT)
            break;
            #endif //(RW_BFMER_EN)
        default:
            pol = &bufctrl->policy_tbl;
            break;
    }

    // Make and use a copy of the policy table so that subsequent modifications inside
    // it (e.g for encryption key index) are not affecting the default policy table
    memcpy(&bufctrl->policy_tbl, pol, sizeof(struct tx_policy_tbl));
    pol = &bufctrl->policy_tbl;

    // Set TX power
    pol->powercntrlinfo[0] = TX_PWR_LEVEL_SET(nxmac_ofdm_max_pwr_level_getf());
    thd->policyentryaddr = CPU2HW(pol);
    thd->phyctrlinfo = 0;
    thd->macctrlinfo2 = 0;
    thd->first_pbd_ptr = 0;

    // Reset the confirmation structure
    frame->cfm.cfm_func = NULL;
    frame->cfm.env = NULL;
}

bool txl_frame_push(struct txl_frame_desc_tag *frame, uint8_t ac)
{
    struct tx_hd *thd = &frame->txdesc.lmac.hw_desc->thd;
    struct mac_hdr *hdr;

    // Sanity check - the pushed frame cannot be NULL
    ASSERT_ERR(frame != NULL);
    ASSERT_ERR((thd->datastartptr & 0x01) == 0);

    // Perform last settings
    hdr = HW2CPU(thd->datastartptr);
    thd->nextfrmexseq_ptr = 0;
    thd->nextmpdudesc_ptr = 0;
    thd->macctrlinfo2 &= ~(WHICHDESC_MSK | UNDER_BA_SETUP_BIT);
    if (MAC_ADDR_GROUP(&hdr->addr1))
        thd->macctrlinfo1 = EXPECTED_ACK_NO_ACK;
    else
        thd->macctrlinfo1 = EXPECTED_ACK_NORMAL_ACK;

    thd->statinfo = 0;

    // Push the descriptor to the TXL controller for transmission
    return (txl_cntrl_push_int((struct txdesc *)frame, ac));
}

void txl_frame_cfm(struct txdesc *txdesc)
{
    // Push the descriptor into the confirmation queue
    co_list_push_back(&txl_frame_env.desc_done, &txdesc->list_hdr);

    // Trigger the frame confirmation event
    ke_evt_set(KE_EVT_TXFRAME_CFM_BIT);
}

void txl_frame_release(struct txdesc *txdesc, bool postponed)
{
    struct txl_frame_desc_tag *frame = (struct txl_frame_desc_tag *)txdesc;

    // If the descriptor was not statically allocated, push it back in the list of free descriptors
    if (frame->type == TX_INT)
    {
        // Push the descriptor into the confirmation queue
        co_list_push_back(&txl_frame_env.desc_free, &txdesc->list_hdr);
    }

    // If frame is released after being postponed, call the associated callback
    if (postponed)
    {
        // Call the callback if any
        if (frame->cfm.cfm_func != NULL)
        {
            // Status bits show that the frame has not been sent
            frame->cfm.cfm_func(frame->cfm.env, 0);
        }
    }
}

void txl_frame_evt(int dummy)
{
    struct txl_frame_desc_tag *frame;

    // Trigger the frame confirmation event
    ke_evt_clear(KE_EVT_TXFRAME_CFM_BIT);

    // Go through the list of done descriptors
    while (1)
    {
        // Pop a descriptor from the list
        GLOBAL_INT_DISABLE();
        frame = (struct txl_frame_desc_tag *)co_list_pop_front(&txl_frame_env.desc_done);
        GLOBAL_INT_RESTORE();

        // Check if we reached the end of the list
        if (frame == NULL)
            break;

        // For profiling
        PROF_TX_FRAME_CFM_SET();

        #if NX_POWERSAVE
        // Decrease the number of packets in the TX path
        txl_cntrl_env.pck_cnt--;
        #endif

        // Call the confirmation callback if any
        if (frame->cfm.cfm_func != NULL)
        {
            frame->cfm.cfm_func(frame->cfm.env, frame->txdesc.lmac.hw_desc->thd.statinfo);

            #if (NX_UMAC_PRESENT)
            /*
             * Check if descriptor can be pushed back in list of free descriptor. If not it means that the
             * packet has been pushed back in the TX path for retransmission
             */
            if (frame->keep_desc)
            {
                // Reset the status
                frame->keep_desc = false;

                continue;
            }
            #endif //(NX_UMAC_PRESENT)
        }

        // For profiling
        PROF_TX_FRAME_CFM_CLR();

        // Check if we need to free the descriptor or not
        if (frame->type == TX_INT)
        {
            // Free the frame descriptor
            co_list_push_back(&txl_frame_env.desc_free, &frame->txdesc.list_hdr);
        }
    }
}

uint8_t txl_frame_send_null_frame(uint8_t sta_idx, cfm_func_ptr cfm, void *env)
{
    struct txl_frame_desc_tag *frame;
    struct mac_hdr *buf;
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];
    int txtype;

    // Allocate a frame descriptor from the TX path
    txtype = vif_mgmt_get_txtype(vif);
    frame = txl_frame_get(txtype, MAC_NULL_FRAME_SIZE);
    if (frame == NULL)
        return CO_FAIL;

    // update Tx power in policy table
    tpc_update_frame_tx_power(vif, frame);

    // Get the buffer pointer
    buf = txl_buffer_payload_get(&frame->txdesc);

    // Fill-in the frame
    buf->fctl = MAC_FCTRL_NULL_FUNCTION | MAC_FCTRL_TODS;
    buf->durid = 0;
    buf->addr1 = sta->mac_addr;
    buf->addr2 = vif->mac_addr;
    buf->addr3 = sta->mac_addr;
    buf->seq = txl_get_seq_ctrl();

    // Fill-in the confirmation structure
    frame->cfm.cfm_func = cfm;
    frame->cfm.env = env;

    #if (NX_CHNL_CTXT || NX_P2P)
    // Set VIF and STA indexes
    frame->txdesc.host.vif_idx = sta->inst_nbr;
    frame->txdesc.host.staid   = sta_idx;
    #endif //(NX_CHNL_CTXT || NX_P2P)

    // Push the frame for TX
    txl_frame_push(frame, AC_VO);

    return CO_OK;
}


uint8_t txl_frame_send_qosnull_frame(uint8_t sta_idx, uint16_t qos, cfm_func_ptr cfm, void *env)
{
    struct txl_frame_desc_tag *frame;
    struct mac_hdr_qos *buf;
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];
    int txtype;

    // Allocate a frame descriptor from the TX path
    txtype = vif_mgmt_get_txtype(vif);
    frame = txl_frame_get(txtype, MAC_QOS_NULL_FRAME_SIZE);
    if (frame == NULL)
        return  CO_FAIL;

    // update Tx power in policy table
    tpc_update_frame_tx_power(vif, frame);

    // Get the buffer pointer
    buf = txl_buffer_payload_get(&frame->txdesc);

    // Fill-in the frame
    buf->fctl = MAC_FCTRL_QOS_NULL;
    buf->durid = 0;
    buf->addr1 = sta->mac_addr;
    buf->addr2 = vif->mac_addr;
    if (vif->type == VIF_STA)
    {
        #if NX_TDLS
        if (sta->is_tdls)
        {
            buf->addr3 = vif->bssid;
        }
        else
            #endif
        {
            buf->fctl |= MAC_FCTRL_TODS;
            buf->addr3 = sta->mac_addr;
        }
    }
    else
    {
        buf->fctl |= MAC_FCTRL_FROMDS;
        buf->addr3 = vif->mac_addr;
    }
    buf->seq = 0;
    buf->qos = qos;

    // Fill-in the confirmation structure
    frame->cfm.cfm_func = cfm;
    frame->cfm.env = env;

    #if (NX_CHNL_CTXT || NX_P2P)
    // Set VIF and STA indexes
    frame->txdesc.host.vif_idx = sta->inst_nbr;
    frame->txdesc.host.staid   = sta_idx;
    #endif //(NX_CHNL_CTXT || NX_P2P)

    // Push the frame for TX
    txl_frame_push(frame, AC_VO);

    return CO_OK;
}

#if NX_TDLS && NX_UMAC_PRESENT
/**
 ****************************************************************************************
 * @brief Add security header in a TDLS frame
 *
 * @param[in] frame Pointer to the frame to update
 * @param[in] buf   Pointer to the MAC header in the frame
 * @param[in] key   Pointer to the key to use
 ****************************************************************************************
 */
static void txl_frame_tdls_add_security_header(struct txl_frame_desc_tag *frame,
                                               struct mac_hdr_qos *buf,
                                               struct key_info_tag *key)
{
    struct txl_buffer_control *bufctrl = txl_buffer_control_get(&frame->txdesc);
    struct tx_policy_tbl *pol = &bufctrl->policy_tbl;
    uint32_t mac_control_info1 = pol->maccntrlinfo1 & KEYSRAM_INDEX_RA_MASK;
    uint16_t *iv = (&buf->qos) + 1;

    key->tx_pn++;
    iv[0] = (uint16_t)(key->tx_pn & 0xFFFF);
    iv[1] = EIV_PRESENT | (key->key_idx << 14);
    iv[2] = (uint16_t)((key->tx_pn >> 16) & 0xFFFF);
    iv[3] = (uint16_t)((key->tx_pn >> 32) & 0xFFFF);;
    buf->fctl |= MAC_FCTRL_PROTECTEDFRAME;
    // Update the policy table with the right key index
    pol->maccntrlinfo1 = mac_control_info1 | key->hw_key_idx;
}


uint8_t txl_frame_send_tdls_channel_switch_req_frame(struct tdls_chan_switch_req const *param,
                                                     cfm_func_ptr cfm)
{
    struct txl_frame_desc_tag *frame;
    struct mac_hdr_qos *buf;
    struct vif_info_tag *vif = &vif_info_tab[param->vif_index];
    struct sta_info_tag *sta = &sta_info_tab[param->sta_idx];
    struct key_info_tag *key = *sta->sta_sec_info.cur_key;
    uint8_t tid = TID_5;
    int txtype;
    int head_len = 0;
    int tail_len = 0;
    uint32_t payload;
    uint8_t i;
    uint16_t ch_switch_time = TDLS_CHSW_SWITCH_TIME_US;
    uint16_t ch_switch_timeout = tdls_get_dt_us(vif->tbtt_timer.time, hal_machw_time()) -
        2*ch_switch_time -
        3*TDLS_CHSW_TX_FRAME_TIME_US;

    if (ch_switch_timeout > TDLS_MAX_CHSW_SWITCH_TIME_US)
    {
        ch_switch_timeout = TDLS_MAX_CHSW_SWITCH_TIME_US;
    }

    PROF_TDLS_CHSW_REQ_TX_SET();

    // Check if we have a valid key for this STA
    if (key)
    {
        ASSERT_ERR(key->cipher == MAC_CIPHER_CCMP);
        head_len = IV_LEN + EIV_LEN;
        tail_len = MIC_LEN;
    }

    // Allocate a frame descriptor from the TX path
    txtype = vif_mgmt_get_txtype(vif);
    frame = txl_frame_get(txtype, MAC_SHORT_QOS_MAC_HDR_LEN + head_len +
                          MAC_ENCAPSULATED_PAYLOAD_OFT + 38);
    if (frame == NULL)
    {
        PROF_TDLS_CHSW_REQ_TX_CLR();
        return CO_FAIL;
    }

    // update Tx power in policy table
    tpc_update_frame_tx_power(vif, frame);

    // Get the buffer pointer
    buf = txl_buffer_payload_get(&frame->txdesc);

    // Fill-in the action frame to be encapsulated into a data frame
    buf->fctl = MAC_FCTRL_QOS_DATA;
    buf->durid = 0;
    MAC_ADDR_CPY(&buf->addr1, &param->peer_mac_addr);
    MAC_ADDR_CPY(&buf->addr2, &vif->mac_addr);
    MAC_ADDR_CPY(&buf->addr3, &vif->bssid);
    buf->seq = sta_mgmt_get_tx_ssn_and_inc(param->sta_idx, tid) << MAC_SEQCTRL_NUM_OFT;
    buf->qos = tid << MAC_QOSCTRL_UP_OFT;

    // Build IV/EIV
    if (key)
    {
        txl_frame_tdls_add_security_header(frame, buf, key);
    }

    payload = CPU2HW(buf) + MAC_SHORT_QOS_MAC_HDR_LEN + head_len;

    // Build LLC, SNAP and Payload Type fields
    co_write32p(payload, FRAME_BODY_LLC_H);
    co_write32p(payload + MAC_ENCAPSULATED_LLC_L_OFT, FRAME_BODY_LLC_L);
    co_write8p(payload + MAC_ENCAPSULATED_PAYLOAD_TYPE_OFT, PAYLOAD_TYPE_TDLS);

    // Build the payload
    payload += MAC_ENCAPSULATED_PAYLOAD_OFT;
    // Category
    co_write8p(payload, MAC_TDLS_ACTION_CATEGORY);
    // Action
    co_write8p(payload + MAC_ACTION_ACTION_OFT, MAC_TDLS_ACTION_CHANSW_REQ);
    // Target Channel
    co_write8p(payload + TDLS_CHANSW_REQ_TARGET_CH_OFFSET,
               (uint8_t)phy_freq_to_channel(param->chan.band, param->chan.prim20_freq));
    // Operating Class
    co_write8p(payload + TDLS_CHANSW_REQ_OP_CLASS, param->op_class);
    // Information Elements
    payload += TDLS_CHANSW_REQ_IES_OFFSET;
    // Secondary Channel Offset IE
    if (param->chan.type == PHY_CHNL_BW_40)
    {
        co_write8p(payload, MAC_ELTID_SEC_CH_OFFSET);
        co_write8p(payload + MAC_INFOELT_LEN_OFT, MAC_INFOELT_SEC_CH_OFFSET_INFO_LEN);
        if (param->chan.center1_freq > param->chan.prim20_freq)
        {
            co_write8p(payload + MAC_INFOELT_SEC_CH_OFFSET_SEC_CH_OFT,
                       MAC_INFOELT_SEC_CH_OFFSET_SEC_ABOVE);
        }
        else
        {
            co_write8p(payload + MAC_INFOELT_SEC_CH_OFFSET_SEC_CH_OFT,
                       MAC_INFOELT_SEC_CH_OFFSET_SEC_BELOW);
        }
        payload += TDLS_CHANSW_REQ_IE_SEC_CH_OFT_LEN;
    }

    // Link Identifier IE
    co_write8p(payload, MAC_ELTID_LINK_IDENTIFIER);
    co_write8p(payload + MAC_INFOELT_LEN_OFT, MAC_INFOELT_LINK_ID_LEN);
    for (i = 0; i < MAC_ADDR_LEN/2; i++)
    {
        co_write16p(payload + MAC_INFOELT_LINK_ID_BSSID_OFT + 2*i, vif->bssid.array[i]);
        if (param->initiator)
        {
            co_write16p(payload + MAC_INFOELT_LINK_ID_INIT_STA_OFT + 2*i,
                        sta->mac_addr.array[i]);
            co_write16p(payload + MAC_INFOELT_LINK_ID_RESP_STA_OFT + 2*i,
                        vif->mac_addr.array[i]);
        }
        else
        {
            co_write16p(payload + MAC_INFOELT_LINK_ID_INIT_STA_OFT + 2*i,
                        vif->mac_addr.array[i]);
            co_write16p(payload + MAC_INFOELT_LINK_ID_RESP_STA_OFT + 2*i,
                        sta->mac_addr.array[i]);
        }
    }
    payload += TDLS_CHANSW_REQ_IE_LINK_ID_LEN;
    // Channel Switch Timing IE
    co_write8p(payload, MAC_ELTID_CHANNEL_SWITCH_TIMING);
    co_write8p(payload + MAC_INFOELT_LEN_OFT, MAC_INFOELT_CH_SWITCH_TIMING_LEN);
    co_write16p(payload + MAC_INFOELT_CH_SWITCH_TIMING_SWTIME_OFT, ch_switch_time);
    co_write16p(payload + MAC_INFOELT_CH_SWITCH_TIMING_SWTOUT_OFT, ch_switch_timeout);
    payload += TDLS_CHANSW_REQ_IE_CH_SWITCH_TIMING_LEN;
    if (param->chan.type > PHY_CHNL_BW_40)
    {
        // Wide Bandwidth Channel Switch IE
        co_write8p(payload, MAC_ELTID_WIDE_BANDWIDTH_CHAN_SWITCH);
        co_write8p(payload + MAC_INFOELT_LEN_OFT, MAC_INFOELT_WIDE_BW_CHAN_SWITCH_INFO_LEN);
        co_write8p(payload + MAC_INFOELT_WIDE_BW_CHAN_SWITCH_NEW_CW_OFT,
                   (param->chan.type == PHY_CHNL_BW_20) ? 0 : (param->chan.type - 1));
        co_write8p(payload + MAC_INFOELT_WIDE_BW_CHAN_SWITCH_NEW_CENTER1_OFT,
                   param->chan.center1_freq);
        co_write8p(payload + MAC_INFOELT_WIDE_BW_CHAN_SWITCH_NEW_CENTER2_OFT,
                   param->chan.center2_freq);
        payload += TDLS_CHANSW_REQ_IE_WIDE_BW_CHAN_SWITCH_LEN;

        // Country IE (optional)

        // VHT Transmit Power Envelope IE (optional)
    }

    // Update THD fields based on actual frame length
    frame->txdesc.lmac.hw_desc->thd.dataendptr = payload - 1;
    frame->txdesc.lmac.hw_desc->thd.frmlen = payload - CPU2HW(buf) + MAC_FCS_LEN + tail_len;

    // Fill-in the confirmation structure
    frame->cfm.cfm_func = cfm;
    frame->cfm.env = vif;

    // Set VIF and STA indexes
    frame->txdesc.host.vif_idx = vif->index;
    frame->txdesc.host.staid = vif->u.sta.ap_id;

    // Push the frame for TX
    txl_frame_push(frame, mac_tid2ac[tid]);

    PROF_TDLS_CHSW_REQ_TX_CLR();
    return CO_OK;
}

uint8_t txl_frame_send_tdls_channel_switch_rsp_frame(struct vif_info_tag *vif,
                                                     uint16_t status_code, cfm_func_ptr cfm)
{
    struct txl_frame_desc_tag *frame;
    struct mac_hdr_qos *buf;
    uint8_t status = CO_FAIL;
    int txtype;
    uint32_t payload;
    uint8_t i;
    struct sta_info_tag *sta = &sta_info_tab[vif->u.sta.tdls_chsw_sta_idx];
    struct key_info_tag *key = *sta->sta_sec_info.cur_key;
    uint8_t tid = TID_5;
    int head_len = 0;
    int tail_len = 0;

    PROF_TDLS_CHSW_RESP_TX_SET();

    // Check if we have a valid key for this STA
    if (key)
    {
        ASSERT_ERR(key->cipher == MAC_CIPHER_CCMP);
        head_len = IV_LEN + EIV_LEN;
        tail_len = MIC_LEN;
    }

    // Allocate a frame descriptor from the TX path
    txtype = vif_mgmt_get_txtype(vif);
    frame = txl_frame_get(txtype, MAC_SHORT_QOS_MAC_HDR_LEN + head_len +
                          MAC_ENCAPSULATED_PAYLOAD_OFT + 30);
    if (frame == NULL)
    {
        PROF_TDLS_CHSW_RESP_TX_CLR();
        return status;
    }

    // update Tx power in policy table
    tpc_update_frame_tx_power(vif, frame);

    // Get the buffer pointer
    buf = txl_buffer_payload_get(&frame->txdesc);

    // Fill-in the action frame to be incapsulated into a data frame
    buf->fctl = MAC_FCTRL_QOS_DATA;
    buf->durid = 0;
    MAC_ADDR_CPY(&buf->addr1, &sta->mac_addr);
    MAC_ADDR_CPY(&buf->addr2, &vif->mac_addr);
    MAC_ADDR_CPY(&buf->addr3, &vif->bssid);
    buf->seq = sta_mgmt_get_tx_ssn_and_inc(sta->staid, tid) << MAC_SEQCTRL_NUM_OFT;
    buf->qos = tid << MAC_QOSCTRL_UP_OFT;

    // Build IV/EIV
    if (key)
    {
        txl_frame_tdls_add_security_header(frame, buf, key);
    }

    payload = CPU2HW(buf) + MAC_SHORT_QOS_MAC_HDR_LEN + head_len;

    // Build LLC, SNAP and Payload Type fields
    co_write32p(payload, FRAME_BODY_LLC_H);
    co_write32p(payload + MAC_ENCAPSULATED_LLC_L_OFT, FRAME_BODY_LLC_L);
    co_write8p(payload + MAC_ENCAPSULATED_PAYLOAD_TYPE_OFT, PAYLOAD_TYPE_TDLS);

    // Build the payload
    payload += MAC_ENCAPSULATED_PAYLOAD_OFT;
    co_write8p(payload, MAC_TDLS_ACTION_CATEGORY);

    co_write8p(payload + MAC_ACTION_ACTION_OFT, MAC_TDLS_ACTION_CHANSW_RSP);
    co_write16p(payload + TDLS_CHANSW_RSP_STATUS_OFFSET, status_code);
    payload += TDLS_CHANSW_RSP_IES_OFFSET;
    // Link Identifier element
    co_write8p(payload, MAC_ELTID_LINK_IDENTIFIER);
    co_write8p(payload + MAC_INFOELT_LEN_OFT, MAC_INFOELT_LINK_ID_LEN);
    for (i = 0; i < MAC_ADDR_LEN/2; i++)
    {
        co_write16p(payload + MAC_INFOELT_LINK_ID_BSSID_OFT + 2*i, vif->bssid.array[i]);
        if (sta->tdls.initiator)
        {
            co_write16p(payload + MAC_INFOELT_LINK_ID_INIT_STA_OFT + 2*i,
                        sta->mac_addr.array[i]);
            co_write16p(payload + MAC_INFOELT_LINK_ID_RESP_STA_OFT + 2*i,
                        vif->mac_addr.array[i]);
        }
        else
        {
            co_write16p(payload + MAC_INFOELT_LINK_ID_INIT_STA_OFT + 2*i,
                        vif->mac_addr.array[i]);
            co_write16p(payload + MAC_INFOELT_LINK_ID_RESP_STA_OFT + 2*i,
                        sta->mac_addr.array[i]);
        }
    }
    payload += TDLS_CHANSW_REQ_IE_LINK_ID_LEN;
    co_write8p(payload, MAC_ELTID_CHANNEL_SWITCH_TIMING);
    co_write8p(payload + MAC_INFOELT_LEN_OFT, MAC_INFOELT_CH_SWITCH_TIMING_LEN);
    co_write16p(payload + MAC_INFOELT_CH_SWITCH_TIMING_SWTIME_OFT, sta->tdls.chsw_time);
    co_write16p(payload + MAC_INFOELT_CH_SWITCH_TIMING_SWTOUT_OFT, sta->tdls.chsw_timeout);
    payload += TDLS_CHANSW_REQ_IE_CH_SWITCH_TIMING_LEN;

    // Update THD fields based on actual frame length
    frame->txdesc.lmac.hw_desc->thd.dataendptr = payload - 1;
    frame->txdesc.lmac.hw_desc->thd.frmlen = payload - CPU2HW(buf) + MAC_FCS_LEN + tail_len;

    // Set VIF and STA indexes
    frame->txdesc.host.vif_idx = vif->index;
    frame->txdesc.host.staid = vif->u.sta.ap_id;

    if (status_code == 0)
    {
        // Fill-in the confirmation structure
        frame->cfm.cfm_func = cfm;
        frame->cfm.env = vif;
        status = CO_OK;
    }

    // Push the frame for TX
    txl_frame_push(frame, mac_tid2ac[tid]);

    PROF_TDLS_CHSW_RESP_TX_CLR();
    return (status);
}

uint8_t txl_frame_send_tdls_peer_traffic_ind_frame(struct tdls_peer_traffic_ind_req const *param,
                                                   cfm_func_ptr cfm)
{
    struct txl_frame_desc_tag *frame;
    struct mac_hdr_qos *buf;
    struct vif_info_tag *vif = &vif_info_tab[param->vif_index];
    struct sta_info_tag *sta = &sta_info_tab[param->sta_idx];
    struct sta_info_tag *ap = &sta_info_tab[vif->u.sta.ap_id];
    struct key_info_tag *key = *ap->sta_sec_info.cur_key;
    int txtype;
    // Payload length
    uint32_t payload;
    uint8_t i;
    uint8_t tid = TID_5;
    int head_len = 0;
    int tail_len = 0;

    // Check if we have a valid key for the AP
    if (key)
    {
        ASSERT_ERR(key->cipher == MAC_CIPHER_CCMP);
        head_len = IV_LEN + EIV_LEN;
        tail_len = MIC_LEN;
    }

    // Allocate a frame descriptor from the TX path
    txtype = vif_mgmt_get_txtype(vif);
    frame = txl_frame_get(txtype, MAC_SHORT_QOS_MAC_HDR_LEN + head_len +
                          MAC_ENCAPSULATED_PAYLOAD_OFT + 32);
    if (frame == NULL)
    {
        return CO_FAIL;
    }

    // update Tx power in policy table
    tpc_update_frame_tx_power(vif, frame);

    // Get the buffer pointer
    buf = txl_buffer_payload_get(&frame->txdesc);

    // Fill-in the action frame to be incapsulated into a data frame
    buf->fctl = MAC_FCTRL_QOS_DATA;
    buf->durid = 0;
    MAC_ADDR_CPY(&buf->addr1, &vif->bssid);
    MAC_ADDR_CPY(&buf->addr2, &vif->mac_addr);
    buf->fctl |= MAC_FCTRL_TODS;
    MAC_ADDR_CPY(&buf->addr3, &param->peer_mac_addr);
    buf->seq = sta_mgmt_get_tx_ssn_and_inc(sta->staid, tid) << MAC_SEQCTRL_NUM_OFT;
    buf->qos = tid << MAC_QOSCTRL_UP_OFT;

    // Build IV/EIV
    if (key)
    {
        txl_frame_tdls_add_security_header(frame, buf, key);
    }

    payload = CPU2HW(buf) + MAC_SHORT_QOS_MAC_HDR_LEN + head_len;

    // Build LLC, SNAP and Payload Type fields
    co_write32p(payload, FRAME_BODY_LLC_H);
    co_write32p(payload + MAC_ENCAPSULATED_LLC_L_OFT, FRAME_BODY_LLC_L);
    co_write8p(payload + MAC_ENCAPSULATED_PAYLOAD_TYPE_OFT, PAYLOAD_TYPE_TDLS);

    // Build the payload
    payload += MAC_ENCAPSULATED_PAYLOAD_OFT;
    // Category
    co_write8p(payload, MAC_TDLS_ACTION_CATEGORY);
    // Action
    co_write8p(payload + MAC_ACTION_ACTION_OFT, MAC_TDLS_ACTION_PEER_TRAFFIC_IND);
    // Dialog token
    co_write16p(payload + MAC_ACTION_TOKEN_OFT, param->dialog_token);

    // Information Elements
    payload += TDLS_PEER_TRAFFIC_IND_IES_OFFSET;
    // Link Identifier IE
    co_write8p(payload, MAC_ELTID_LINK_IDENTIFIER);
    co_write8p(payload + MAC_INFOELT_LEN_OFT, MAC_INFOELT_LINK_ID_LEN);
    for (i = 0; i < MAC_ADDR_LEN/2; i++)
    {
        co_write16p(payload + MAC_INFOELT_LINK_ID_BSSID_OFT + 2*i, vif->bssid.array[i]);
        if (sta->tdls.initiator)
        {
            co_write16p(payload + MAC_INFOELT_LINK_ID_INIT_STA_OFT + 2*i,
                        param->peer_mac_addr.array[i]);
            co_write16p(payload + MAC_INFOELT_LINK_ID_RESP_STA_OFT + 2*i,
                        vif->mac_addr.array[i]);
        }
        else
        {
            co_write16p(payload + MAC_INFOELT_LINK_ID_INIT_STA_OFT + 2*i,
                        vif->mac_addr.array[i]);
            co_write16p(payload + MAC_INFOELT_LINK_ID_RESP_STA_OFT + 2*i,
                        param->peer_mac_addr.array[i]);
        }
    }
    payload += TDLS_PEER_TRAFFIC_IND_IE_LINK_ID_LEN;
    // PTI Control IE (optional)
    co_write8p(payload, MAC_ELTID_PTI_CONTROL);
    co_write8p(payload + MAC_INFOELT_LEN_OFT, MAC_INFOELT_PTI_CONTROL_LEN);
    co_write8p(payload + MAC_INFOELT_PTI_CONTROL_TID_OFT, param->last_tid);
    co_write16p(payload + MAC_INFOELT_PTI_CONTROL_SEQ_CTRL_OFT, param->last_sn);
    payload += TDLS_PEER_TRAFFIC_IND_IE_PTI_CTRL_LEN;
    // TPU Buffer Status IE
    co_write8p(payload, MAC_ELTID_TPU_BUFFER_STATUS);
    co_write8p(payload + MAC_INFOELT_LEN_OFT, MAC_INFOELT_TPU_BUF_STATUS_LEN);
    co_write8p(payload + MAC_INFOELT_TPU_BUF_STATUS_AC_STATUS, vif->u.sta.uapsd_queues);
    payload += TDLS_PEER_TRAFFIC_IND_IE_TPU_BUF_STATUS_LEN;

    // Update THD fields based on actual frame length
    frame->txdesc.lmac.hw_desc->thd.dataendptr = payload - 1;
    frame->txdesc.lmac.hw_desc->thd.frmlen = payload - CPU2HW(buf) + MAC_FCS_LEN + tail_len;

    // Fill-in the confirmation structure
    frame->cfm.cfm_func = cfm;
    frame->cfm.env = sta;

    // Set VIF and STA indexes
    frame->txdesc.host.vif_idx = vif->index;
    frame->txdesc.host.staid = vif->u.sta.ap_id;

    // Push the frame for TX
    txl_frame_push(frame, AC_VI);

    return CO_OK;
}
#endif // NX_TDLS && NX_UMAC_PRESENT

#if NX_MAC_HE && NX_UMAC_PRESENT && NX_BEACONING
uint8_t txl_frame_send_he_trigger(uint8_t sta_idx, uint8_t type, uint16_t ul_length,
                                  uint8_t ru_alloc, uint8_t mcs, uint8_t pref_ac,
                                  cfm_func_ptr cfm, void *env)
{
    struct txl_frame_desc_tag *frame;
    struct he_trigger_base *buf;
    struct he_user_info_base *user_info;
    struct mac_hdr_ctrl *hdr;
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];
    struct vif_info_tag *vif = &vif_info_tab[sta->inst_nbr];
    int txtype;
    struct txl_buffer_control *bufctrl;
    struct tx_policy_tbl *pol;
    struct mac_hecapability *hecap = &sta->info.he_cap;
    struct mac_hecapability *hecaploc = &me_env.he_cap;
    uint32_t coding_bit = 0;
    uint8_t user_info_len = 0;

    if (type == HE_TRIG_TYPE_BASIC)
        user_info_len = sizeof_b(struct he_basic_trigger_user_info);
    else
        user_info_len = sizeof_b(struct he_user_info_base);

    // Allocate a frame descriptor from the TX path
    txtype = vif_mgmt_get_txtype(vif);
    frame = txl_frame_get(txtype, sizeof_b(struct he_trigger_base) + user_info_len);
    if (frame == NULL)
        return CO_FAIL;

    bufctrl = txl_buffer_control_get(&frame->txdesc);
    pol = &bufctrl->policy_tbl;

    // Update Tx power in policy table
    tpc_update_frame_tx_power(vif, frame);

    // Get the buffer pointer
    buf = txl_buffer_payload_get(&frame->txdesc);
    user_info = (struct he_user_info_base *)(buf + 1);
    hdr = &buf->h;

    // Disable HW retransmissions
    pol->maccntrlinfo2 &= ~(LONG_RETRY_LIMIT_MASK | SHORT_RETRY_LIMIT_MASK);

    // Fill-in the frame
    hdr->fctl = MAC_FCTRL_HE_TRIGGER;
    hdr->durid = 0;
    hdr->addr1 = mac_addr_bcst;
    hdr->addr2 = vif->mac_addr;

    buf->common[0] = type | (ul_length << HE_TRIG_UL_LENGTH_OFT);
    buf->common[1] = 0;

    if (HE_PHY_CAPA_BIT_IS_SET(hecaploc, LDPC_CODING_IN_PAYLOAD) &&
        HE_PHY_CAPA_BIT_IS_SET(hecap, LDPC_CODING_IN_PAYLOAD))
        coding_bit = HE_TRIG_FEC_CODING_BIT;

    user_info->user_info1 = (sta->aid << HE_TRIG_AID12_OFT)    |
                            (ru_alloc << HE_TRIG_RU_ALLOC_OFT) |
                            (mcs << HE_TRIG_UL_MCS_OFT)        |
                            coding_bit;

    user_info->ul_target_rssi = (uint8_t)-40;
    if (type == HE_TRIG_TYPE_BASIC)
    {
        struct he_basic_trigger_user_info *basic_user_info =
                (struct he_basic_trigger_user_info *)user_info;
        basic_user_info->user_info2 = mac_ac2aci[pref_ac] << HE_TRIG_PREF_AC_OFT;
    }

    // Fill-in the confirmation structure
    frame->cfm.cfm_func = cfm;
    frame->cfm.env = env;

    // Push the frame for TX
    if (!txl_frame_push(frame, pref_ac))
        return CO_FAIL;

    return CO_OK;
}
#endif // NX_MAC_HE && NX_UMAC_PRESENT && NX_BEACONING

#endif


/// @}

