/**
 ****************************************************************************************
 *
 * @file txl_he.c
 *
 * @brief LMAC HE TX implementation implementation.
 *
 * Copyright (C) RivieraWaves 2018-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup TX_HE
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */
#include "co_int.h"
#include "co_bool.h"
#include <string.h>

#include "dbg_assert.h"
#include "mac.h"
#include "mac_frame.h"
#include "mm.h"
#include "macif.h"
#include "rxl_hwdesc.h"
#include "txl_buffer.h"
#include "tx_swdesc.h"
#include "txl_he.h"
#include "txl_cfm.h"
#include "txl_agg.h"
#include "txl_cntrl.h"
#include "rxl_cntrl.h"
#include "reg_mac_pl.h"
#include "reg_mac_core.h"
#include "dbg.h"
#include "sta_mgmt.h"
#include "vif_mgmt.h"


#if NX_MAC_HE

/*
 * DEFINES
 ****************************************************************************************
 */
/// Table linking a HE MCS+GI+BW combination to the minimum number of bytes required
/// for 1us min MPDU start spacing in AMPDU
/// Index of the table is (MCS_IDX * 6) + ((BW >> 1) * 3) + (GI)
///  where BW is 0 for 20 MHz, 1 for 40MHz and 2 for 80MHz
///        GI is 0 for 0.8us, 1 for 1.6us and 2 for 3.2us
static const uint16_t TX_HE_RATE_TO_MIN_SEP[12*2*3] =
{
    //BW20/40,GI0.8us    BW20/40,GI1.6us      BW20/40,GI3.2us    MCS Index
    [  0] =  2,          [  1] =  2,          [  2] =  1,        // MCS 0
    [  6] =  3,          [  7] =  3,          [  8] =  2,        // MCS 1
    [ 12] =  4,          [ 13] =  4,          [ 14] =  3,        // MCS 2
    [ 18] =  5,          [ 19] =  5,          [ 20] =  4,        // MCS 3
    [ 24] =  7,          [ 25] =  7,          [ 26] =  6,        // MCS 4
    [ 30] =  9,          [ 31] =  9,          [ 32] =  8,        // MCS 5
    [ 36] = 10,          [ 37] = 10,          [ 38] =  9,        // MCS 6
    [ 42] = 11,          [ 43] = 11,          [ 44] = 10,        // MCS 7
    [ 48] = 13,          [ 49] = 13,          [ 50] = 11,        // MCS 8
    [ 54] = 15,          [ 55] = 14,          [ 56] = 13,        // MCS 9
    [ 60] = 17,          [ 61] = 16,          [ 62] = 14,        // MCS 10
    [ 66] = 18,          [ 67] = 17,          [ 68] = 16,        // MCS 11
    //BW80/160,GI0.8us   BW80/160,GI1.6us     BW80/160,GI3.2us   MCS Index
    [  3] =  5,          [  4] =  5,          [  5] =  4,        // MCS 0
    [  9] = 10,          [ 10] =  9,          [ 11] =  8,        // MCS 1
    [ 15] = 14,          [ 16] = 13,          [ 17] = 12,        // MCS 2
    [ 21] = 19,          [ 22] = 18,          [ 23] = 16,        // MCS 3
    [ 27] = 28,          [ 28] = 26,          [ 29] = 23,        // MCS 4
    [ 33] = 37,          [ 34] = 35,          [ 35] = 31,        // MCS 5
    [ 39] = 41,          [ 40] = 39,          [ 41] = 35,        // MCS 6
    [ 45] = 46,          [ 46] = 43,          [ 47] = 39,        // MCS 7
    [ 51] = 55,          [ 52] = 52,          [ 53] = 46,        // MCS 8
    [ 57] = 61,          [ 58] = 57,          [ 59] = 52,        // MCS 9
    [ 63] = 68,          [ 64] = 64,          [ 65] = 58,        // MCS 10
    [ 69] = 76,          [ 70] = 71,          [ 71] = 64,        // MCS 11
};

/// Table linking a HE MCS+GI+BW combination to the number of bytes transmitted in 32us
/// Index of the table is (MCS_IDX * 6) + ((BW >> 1) * 3) + (GI)
///  where BW is 0 for 20 MHz, 1 for 40MHz and 2 for 80MHz
///        GI is 0 for 0.8us, 1 for 1.6us and 2 for 3.2us
static const uint16_t TX_HE_RATE_TO_32US_LEN[12*2*3] =
{
    //BW20/40,GI0.8us    BW20/40,GI1.6us      BW20/40,GI3.2us    MCS Index
    [  0] =   34,        [  1] =   32,        [  2] =   29,      // MCS 0
    [  6] =   68,        [  7] =   65,        [  8] =   58,      // MCS 1
    [ 12] =  103,        [ 13] =   97,        [ 14] =   87,      // MCS 2
    [ 18] =  137,        [ 19] =  130,        [ 20] =  117,      // MCS 3
    [ 24] =  206,        [ 25] =  195,        [ 26] =  175,      // MCS 4
    [ 30] =  275,        [ 31] =  260,        [ 32] =  234,      // MCS 5
    [ 36] =  309,        [ 37] =  292,        [ 38] =  263,      // MCS 6
    [ 42] =  344,        [ 43] =  325,        [ 44] =  292,      // MCS 7
    [ 48] =  412,        [ 49] =  390,        [ 50] =  351,      // MCS 8
    [ 54] =  458,        [ 55] =  433,        [ 56] =  390,      // MCS 9
    [ 60] =  516,        [ 61] =  487,        [ 62] =  438,      // MCS 10
    [ 66] =  573,        [ 67] =  541,        [ 68] =  487,      // MCS 11
    //BW80/160,GI0.8us   BW80/160,GI1.6us     BW80/160,GI3.2us   MCS Index
    [  3] =  144,        [  4] =  136,        [  5] =  122,      // MCS 0
    [  9] =  288,        [ 10] =  272,        [ 11] =  245,      // MCS 1
    [ 15] =  432,        [ 16] =  408,        [ 17] =  367,      // MCS 2
    [ 21] =  576,        [ 22] =  544,        [ 23] =  490,      // MCS 3
    [ 27] =  864,        [ 28] =  816,        [ 29] =  735,      // MCS 4
    [ 33] = 1152,        [ 34] = 1088,        [ 35] =  980,      // MCS 5
    [ 39] = 1297,        [ 40] = 1225,        [ 41] = 1102,      // MCS 6
    [ 45] = 1441,        [ 46] = 1361,        [ 47] = 1225,      // MCS 7
    [ 51] = 1729,        [ 52] = 1633,        [ 53] = 1470,      // MCS 8
    [ 57] = 1921,        [ 58] = 1814,        [ 59] = 1633,      // MCS 9
    [ 63] = 2161,        [ 64] = 2041,        [ 65] = 1837,      // MCS 10
    [ 69] = 2401,        [ 70] = 2268,        [ 71] = 2041,      // MCS 11
};

/// Table linking a HE MCS+GI combination to the minimum number of bytes required
/// for 1us min MPDU start spacing in AMPDU when using a 26-tone RU
/// Index of the table is (MCS_IDX * 3) + (GI)
///        GI is 0 for 0.8us, 1 for 1.6us and 2 for 3.2us
static const uint16_t TX_HE_26_TONE_TO_MIN_SEP[12*3] =
{
    //GI0.8us            GI1.6us              GI3.2us            MCS Index
    [  0] =  1,          [  1] =  1,          [  2] =  1,        // MCS 0
    [  3] =  1,          [  4] =  1,          [  5] =  1,        // MCS 1
    [  6] =  1,          [  7] =  1,          [  8] =  1,        // MCS 2
    [  9] =  1,          [ 10] =  1,          [ 11] =  1,        // MCS 3
    [ 12] =  1,          [ 13] =  1,          [ 14] =  1,        // MCS 4
    [ 15] =  1,          [ 16] =  1,          [ 17] =  1,        // MCS 5
    [ 18] =  1,          [ 19] =  1,          [ 20] =  1,        // MCS 6
    [ 21] =  2,          [ 22] =  2,          [ 23] =  1,        // MCS 7
    [ 24] =  2,          [ 25] =  2,          [ 26] =  2,        // MCS 8
    [ 27] =  2,          [ 28] =  2,          [ 29] =  2,        // MCS 9
    [ 30] =  2,          [ 31] =  2,          [ 32] =  2,        // MCS 10
    [ 33] =  2,          [ 34] =  2,          [ 35] =  2,        // MCS 11
};

/// Table linking a HE MCS+GI combination to the minimum number of bytes required
/// for 1us min MPDU start spacing in AMPDU when using a 52-tone RU
/// Index of the table is (MCS_IDX * 3) + (GI)
///        GI is 0 for 0.8us, 1 for 1.6us and 2 for 3.2us
static const uint16_t TX_HE_52_TONE_TO_MIN_SEP[12*3] =
{
    //GI0.8us            GI1.6us              GI3.2us            MCS Index
    [  0] =  1,          [  1] =  1,          [  2] =  1,        // MCS 0
    [  3] =  1,          [  4] =  1,          [  5] =  1,        // MCS 1
    [  6] =  1,          [  7] =  1,          [  8] =  1,        // MCS 2
    [  9] =  1,          [ 10] =  1,          [ 11] =  1,        // MCS 3
    [ 12] =  2,          [ 13] =  2,          [ 14] =  2,        // MCS 4
    [ 15] =  2,          [ 16] =  2,          [ 17] =  2,        // MCS 5
    [ 18] =  2,          [ 19] =  2,          [ 20] =  2,        // MCS 6
    [ 21] =  3,          [ 22] =  3,          [ 23] =  2,        // MCS 7
    [ 24] =  3,          [ 25] =  3,          [ 26] =  3,        // MCS 8
    [ 27] =  3,          [ 28] =  3,          [ 29] =  3,        // MCS 9
    [ 30] =  4,          [ 31] =  4,          [ 32] =  3,        // MCS 10
    [ 33] =  4,          [ 34] =  4,          [ 35] =  4,        // MCS 11
};

/// Table linking a HE MCS+GI combination to the minimum number of bytes required
/// for 1us min MPDU start spacing in AMPDU when using a 106-tone RU
/// Index of the table is (MCS_IDX * 3) + (GI)
///        GI is 0 for 0.8us, 1 for 1.6us and 2 for 3.2us
static const uint16_t TX_HE_106_TONE_TO_MIN_SEP[12*3] =
{
    //GI0.8us            GI1.6us              GI3.2us            MCS Index
    [  0] =  1,          [  1] =  1,          [  2] =  1,        // MCS 0
    [  3] =  1,          [  4] =  1,          [  5] =  1,        // MCS 1
    [  6] =  2,          [  7] =  2,          [  8] =  2,        // MCS 2
    [  9] =  2,          [ 10] =  2,          [ 11] =  2,        // MCS 3
    [ 12] =  3,          [ 13] =  3,          [ 14] =  3,        // MCS 4
    [ 15] =  4,          [ 16] =  4,          [ 17] =  4,        // MCS 5
    [ 18] =  5,          [ 19] =  4,          [ 20] =  4,        // MCS 6
    [ 21] =  5,          [ 22] =  5,          [ 23] =  4,        // MCS 7
    [ 24] =  6,          [ 25] =  6,          [ 26] =  5,        // MCS 8
    [ 27] =  7,          [ 28] =  6,          [ 29] =  6,        // MCS 9
    [ 30] =  8,          [ 31] =  7,          [ 32] =  6,        // MCS 10
    [ 33] =  8,          [ 34] =  8,          [ 35] =  7,        // MCS 11
};

/// Conversion table between bits b2-b1 of the fragment number of a BA SSC field and the
/// BA bitmap length in bytes
static const uint8_t fragb2b1_to_bitmap_len[4] =
{
    [0] = 8,
    [1] = 16,
    [2] = 32,
    [3] = 4
};

/// Type of RUs
enum
{
    RU_26TONE = 0,
    RU_52TONE,
    RU_106TONE,
    RU_242TONE,
    RU_484TONE,
    RU_996TONE,
    RU_2x996TONE,
};

static const uint16_t *ru_1us_tables[3] =
{
    [RU_26TONE] = TX_HE_26_TONE_TO_MIN_SEP,
    [RU_52TONE] = TX_HE_52_TONE_TO_MIN_SEP,
    [RU_106TONE] = TX_HE_106_TONE_TO_MIN_SEP,
};

/// Minimum time needed to program a HE TB (in us)
#define TX_HE_TB_PROG_MIN_TIME  10

/// HE trigger-based transmission timeout
#define TX_HE_TB_TIMEOUT   7000

/// Period of transmission of the trigger frames
#define TX_HE_TRIG_PERIOD  5000

/// Period of the MU EDCA timer - 8 TUs
#define TX_MU_EDCA_TIMER_PERIOD    (8 * 1024)

/// Minimal PSDU length required for a QoS-NULL frame sent over HE TB (including +HTC)
#define TX_HE_TB_MIN_PSDU_LEN CO_ALIGN4_HI(DELIMITER_LEN + MAC_QOS_NULL_FRAME_SIZE +     \
                                           MAC_HTC_LEN + MAC_FCS_LEN)

/// Maximum scaling factor (in bytes) we can put in a +HTC HE BSR field
#define BSR_SF_MAX  32768

/// Maximum queue size we can put in a +HTC HE BSR field
#define BSR_QSIZE_MAX  254

/// Maximum buffered size that can be coded using Queue Size and Scaling Factor
#define BSR_BUFFERED_MAX (BSR_QSIZE_MAX * BSR_SF_MAX)

/// Default value to put in the +HTC HE UPH, in case we never got a value from HW
#define HTC_HE_UPH_DEF (MAC_HTC_HE_CTRL_ID_UPH | (0x07 << MAC_HTC_HE_CTRL_INFO_OFT))

/*
 * TYPE and STRUCT DEFINITIONS
 ****************************************************************************************
 */
/// HW descriptors for pre-programmed HE-TB BSRP frame
struct tx_he_tb_bsrp_qos_null_desc
{
    /// THD
    struct tx_hd thd;
    /// PBD
    struct tx_pbd pbd;
    /// QoS control field
    uint16_t qos_ctrl;
};

/// HW descriptors for pre-programmed HE-TB BSRP A-MPDU
struct tx_he_tb_bsrp_desc
{
    /// THD
    struct tx_hd a_thd;
    /// Mac header
    struct mac_hdr hdr;
    /// Descriptor for HE TB QOS NULL frame
    struct tx_he_tb_bsrp_qos_null_desc qos_null[TID_MAX];
};

/// HW descriptors for pre-programmed HE-TB QoS NULL frame
struct tx_he_tb_qos_null_desc
{
    /// THD
    struct tx_hd thd;
    /// Payload space (just a QoS + HTC MAC header, no FCS)
    struct mac_hdr_qos_htc payl;
};

/// HW descriptors for pre-programmed NDP (used in case not enough space is allocated by
/// the AP in the HE TB, even for a QoS-NULL)
struct tx_he_tb_ndp_desc
{
    /// THD
    struct tx_hd thd;
};

/// Structure containing the different HW descriptors used for HE TB transmission
struct tx_he_tb_desc
{
    /// Descriptor for HE TB BSRP frame
    struct tx_he_tb_bsrp_desc brsp;
    /// Descriptor for HE TB QOS NULL frame
    struct tx_he_tb_qos_null_desc qos_null;
    /// Descriptors for NDP
    struct tx_he_tb_ndp_desc ndp;
    /// Policy table for HE TB PPDU transmission
    struct tx_policy_tbl tb_pol;
};

/*
 * GLOBAL VARIABLE DEFINITION
 ****************************************************************************************
 */
struct txl_he_env_tag txl_he_env;

/// Various TX HW descriptors used for HE TB PPDU transmissions
static struct tx_he_tb_desc he_tb_desc __SHAREDRAM;

/*
 * FUNCTION BODIES
 ****************************************************************************************
 */
/**
 ****************************************************************************************
 * @brief Get the scaling factor, queue size all and queue size high sub-fields of a +HTC
 * HE BSR element.
 *
 * As per P802.11ax_D5.0, chapter 9.2.4.6a.4, the buffered traffic at a STA can be
 * approximated by the AP by multiplying the received Queue Size (High or All) subfield
 * by the Scaling Factor (SF) indicated in the Scaling Factor subfield. The SF can be 16,
 * 256, 2048 and 32768. A queue size subfield of 254 indicates that buffered traffic is
 * greater than 254 * SF.
 * The standard also states that the Queue Size subfields are the total sizes, rounded up
 * to the nearest multiple of SF octets, of all MSDUs and A-MSDUs buffered at the STA.
 * There is therefore a hole when (253 * 32768) < buffered size <= (254 * 32768), because
 * in such case the computed Queue Size subfield should be 254, which is however supposed
 * to represent a buffered size greater than 254 * 32768.
 * Our implementation will therefore use a QueueSize value of 254 if buffered_data
 * rounded up to the nearest multiple of 32768 octets is greater than *or equal to*
 * 254 * 32768, and up to 253 in all other cases.
 * The buffered data threshold in the function are computed considering this limit of 253,
 * multiplied by the possible Scaling Factors.
 *
 * @param[in] buffered_all Total traffic buffered at the STA in bytes
 * @param[in] buffered_high Buffered traffic at the STA in bytes, for the ACI having the
 * most buffered traffic.
 * @param[out] qsize_all QueueSizeAll subfield
 * @param[out] qsize_high QueueSizeHigh subfield
 *
 * @return The Scaling Factor subfield
 ****************************************************************************************
 */
__INLINE uint8_t txl_he_bsr_get_sf_and_qsize(uint32_t buffered_all, uint32_t buffered_high,
                                             uint8_t *qsize_all, uint8_t *qsize_high)
{
    uint8_t sf;

    if (buffered_all <= 4048)
    {
        sf = 0;
        *qsize_all = (CO_ALIGNx_HI(buffered_all, 16)) / 16;
        *qsize_high = (CO_ALIGNx_HI(buffered_high, 16)) / 16;
    }
    else if (buffered_all <= 64768)
    {
        sf = 1;
        *qsize_all = (CO_ALIGNx_HI(buffered_all, 256)) / 256;
        *qsize_high = (CO_ALIGNx_HI(buffered_high, 256)) / 256;
    }
    else if (buffered_all <= 518144)
    {
        sf = 2;
        *qsize_all = (CO_ALIGNx_HI(buffered_all, 2048)) / 2048;
        *qsize_high = (CO_ALIGNx_HI(buffered_high, 2048)) / 2048;
    }
    else
    {
        buffered_all = (buffered_all > BSR_BUFFERED_MAX) ? BSR_BUFFERED_MAX : buffered_all;
        buffered_high = (buffered_high > BSR_BUFFERED_MAX) ? BSR_BUFFERED_MAX : buffered_high;
        sf = 3;
        *qsize_all = (CO_ALIGNx_HI(buffered_all, 32768)) / 32768;
        *qsize_high = (CO_ALIGNx_HI(buffered_high, 32768)) / 32768;
    }

    return sf;
}

/**
 ****************************************************************************************
 * @brief Compute the number of bytes transmitted for a 32us transmission at the given
 * rate.
 *
 * @param[in] rc Rate control step for which the length is computed
 *
 * @return The number of bytes transmitted for a 32us transmission at the given rate
 ****************************************************************************************
 */
static uint32_t txl_he_dur_to_len(uint32_t rc)
{
    uint32_t mcs_idx = (rc & MCS_INDEX_TX_RCX_MASK) >> MCS_INDEX_TX_RCX_OFT;
    uint32_t gi = (rc & HE_GI_TYPE_TX_RCX_MASK) >> HE_GI_TYPE_TX_RCX_OFT;
    uint32_t bw = (rc & BW_TX_RCX_MASK) >> BW_TX_RCX_OFT;
    int idx = mcs_idx * 6 + 3 * (bw >> 1) + gi;
    int shift = bw & 0x01;
    return (((uint32_t)TX_HE_RATE_TO_32US_LEN[idx]) << shift);
}

/**
 ****************************************************************************************
 * @brief Apply the RTS/CTS duration threshold to the RC step passed as parameter,
 * depending on the frame length and the actual transmission rate.
 *
 * @param[in,out] rc Rate control step for the transmission to be checked
 * @param[in] frm_len Length of the MPDU or A-MPDU
 * @param[in] txop_dur_rts_thres Threshold advertised by the HE AP
 ****************************************************************************************
 */
static void txl_he_txop_dur_rtscts_thres_apply(uint32_t *rc, uint32_t frm_len,
                                               uint16_t txop_dur_rts_thres)
{
    uint32_t rc_loc = *rc;

    if (txl_he_is_he_su(rc_loc))
    {
        // As the TXOP duration RTS/CTS threshold cannot be 0, let's decrease it by one in
        // order to take into account an approximate value of the preamble duration
        uint32_t thres = txl_he_dur_to_len(rc_loc) * (txop_dur_rts_thres - 1);

        if (frm_len > thres)
        {
            // Check if a protection is already activated. If yes we keep the same rate
            // as it might be an ERP protection sent in DSSS.
            if ((rc_loc & PROT_FRM_EX_RCX_MASK) != PROT_NO_PROT)
            {
                // Just clear the type of protection to replace it by RTS/CTS
                rc_loc &= ~PROT_FRM_EX_RCX_MASK;
                rc_loc |= PROT_RTS_CTS;
            }
            else
            {
                uint8_t bw, rate, mcs;

                // Retrieve BW and MCS of the PPDU in order to compute the protection parameters
                bw = (rc_loc & BW_TX_RCX_MASK) >> BW_TX_RCX_OFT;
                mcs = rc_loc & VHT_MCS_MASK;
                rate = mcs > 2 ? HW_RATE_24MBPS : HW_RATE_6MBPS;
                rc_loc &= ~(PROT_FRM_EX_RCX_MASK | MCS_INDEX_PROT_TX_RCX_MASK | BW_PROT_TX_RCX_MASK |
                                                                       FORMAT_MOD_PROT_TX_RCX_MASK);
                rc_loc |= (bw << BW_PROT_TX_RCX_OFT) | (rate << MCS_INDEX_PROT_TX_RCX_OFT) | PROT_RTS_CTS;
                rc_loc |= FORMATMOD_NON_HT_DUP_OFDM << FORMAT_MOD_PROT_TX_RCX_OFT;
            }

            *rc = rc_loc;
        }
    }
}

/**
 ****************************************************************************************
 * @brief Check whether the given AC queue is currently halted (from MAC HW point of view).
 *
 * This function is used when disabling EDCA on a AC, to detect when the queue is
 * effectively halted.
 *
 * @param[in] ac  Access category to be checked
 *
 * @return true if the queue is halted, false otherwise
 ****************************************************************************************
 */
static bool txl_he_edca_queue_halted(uint8_t ac)
{
    // Set the header pointer and the new head bit according to the access category
    switch (ac)
    {
        #if NX_BEACONING
        case AC_BCN:
            return(nxmac_tx_bcn_state_getf() == 0);
        #endif
        case AC_VO:
            return(nxmac_tx_ac_3_state_getf() == 0);
        case AC_VI:
            return(nxmac_tx_ac_2_state_getf() == 0);
        case AC_BE:
            return(nxmac_tx_ac_1_state_getf() == 0);
        case AC_BK:
            return(nxmac_tx_ac_0_state_getf() == 0);
        default:
            ASSERT_ERR(0);
            break;
    }
    return false;
}

/**
 ****************************************************************************************
 * @brief Restart EDCA operation on the given access category.
 *
 * This function does not change the EDCA HW registers. It simply allows again the EDCA
 * scheduling if it was disabled, and re-schedules the packets pending for transmission.
 *
 * @param[in] ac  Access category on which the EDCA is restarted
 ****************************************************************************************
 */
static void txl_he_restart_edca(uint8_t ac)
{
    struct mu_edca_param_tag *mu_edca_param = &txl_he_env.mu_edca.params[ac];

    if (!mu_edca_param->edca_off)
        return;

    mu_edca_param->edca_off = false;

    if (mu_edca_param->first_frame_exch)
        txl_cntrl_newhead(CPU2HW(mu_edca_param->first_frame_exch), ac);

    mu_edca_param->first_frame_exch = NULL;
}

/**
 ****************************************************************************************
 * @brief Callback function of the HE MU EDCA timer.
 *
 * In this function we update the MU EDCA counters and the EDCA parameters if required.
 *
 * @param[in] env  Dummy parameter
 *
 ****************************************************************************************
 */
static void txl_he_mu_edca_timer(void *env)
{
    struct mu_edca_tag *mu_edca = &txl_he_env.mu_edca;
    uint32_t *edca_reg = HW2CPU(NXMAC_EDCA_AC_0_ADDR);
    uint8_t mu_edca_active = 0;

    // Perform all this handling with interrupt disabled to avoid corruption from HE TB
    // TX interrupt
    GLOBAL_INT_DISABLE();

    // Go through the different access categories
    for (int i = 0; i < AC_MAX; i++)
    {
        uint8_t ac_bit = CO_BIT(i);

        // Check if the EDCA queue is now stopped
        if (mu_edca->edca_stopping & ac_bit)
        {
            if (!txl_he_edca_queue_halted(i))
            {
                // Mark MU EDCA as active for this queue
                mu_edca_active |= ac_bit;
                continue;
            }

            mu_edca->edca_stopping &= ~ac_bit;
        }

        // Check if we had HE TB activity on this AC during the previous period
        if (mu_edca->he_tb_activity & ac_bit)
        {
            uint8_t aifsn = mu_edca->params[i].edca & 0x0F;
            mu_edca->he_tb_activity &= ~ac_bit;
            if (aifsn)
            {
                // Apply the latest MU EDCA value received for this AC
                edca_reg[i] = mu_edca->params[i].edca;

                // Restart EDCA if required
                txl_he_restart_edca(i);
            }
            else if (!mu_edca->params[i].edca_off)
            {
                // AIFSN=0, disable EDCA
                mu_edca->params[i].edca_off = true;
                mu_edca->edca_stopping |= ac_bit;
            }

            // Reset the timer to its default
            mu_edca->params[i].timer = mu_edca->params[i].timeout;
            // Mark MU EDCA as active for this queue
            mu_edca_active |= ac_bit;
        }
        // Check if have the MU EDCA timer is pending for this AC
        else if (mu_edca->params[i].timer > 0)
        {
            // Decrement the timer
            mu_edca->params[i].timer--;

            if (mu_edca->params[i].timer == 0)
            {
                // Put back the normal EDCA values
                edca_reg[i] = txl_he_env.tb_vif->txq_params[i];

                // Restart EDCA if required
                txl_he_restart_edca(i);
            }
            else
            {
                // Mark MU EDCA as active for this queue
                mu_edca_active |= ac_bit;
            }
        }
    }

    // If MU EDCA is still active, restart the timer
    if (mu_edca_active)
        mm_timer_set(&mu_edca->timer, hal_machw_time() + TX_MU_EDCA_TIMER_PERIOD);
    else
        mu_edca->timer.cb = NULL;

    // Restore interrupts
    GLOBAL_INT_RESTORE();
}

/**
 ****************************************************************************************
 * @brief Start (or re-start) MU EDCA procedure for the given access category
 *
 * This function is called each time a HE TB PPDU has been transmitted and acknowledged.
 *
 * @param[in] ac  Access category on which we had HE TB activity
 ****************************************************************************************
 */
static void txl_he_mu_edca_start(uint8_t ac)
{
    struct mu_edca_tag *mu_edca = &txl_he_env.mu_edca;

    // Check if we have valid MU EDCA parameters available
    if (!mu_edca->valid)
        return;

    // Indicate HE TB activity on this access category
    mu_edca->he_tb_activity |= CO_BIT(ac);

    // Start the MU EDCA timer if not already done
    if (!txl_he_env.mu_edca.timer.cb)
    {
        ASSERT_REC(txl_he_edca_queue_halted(ac));
        mu_edca->timer.cb = txl_he_mu_edca_timer;
        txl_he_mu_edca_timer(mu_edca);
    }

}

/**
 ****************************************************************************************
 * @brief Stop MU-EDCA operation.
 *
 * This function is called if HE TB operation is not possible anymore (e.g. because more
 * than one VIF is registered to the LMAC)
 ****************************************************************************************
 */
static void txl_he_mu_edca_stop(void)
{
    struct mu_edca_tag *mu_edca = &txl_he_env.mu_edca;
    uint32_t *edca_reg = HW2CPU(NXMAC_EDCA_AC_0_ADDR);

    // Go through the different access categories
    for (int i = 0; i < AC_MAX; i++)
    {
        // Put back the normal EDCA values
        edca_reg[i] = txl_he_env.tb_vif->txq_params[i];

        // Restart EDCA if required
        txl_he_restart_edca(i);
    }

    // Disable the MU EDCA timer
    mm_timer_clear(&mu_edca->timer);
    mu_edca->timer.cb = NULL;
}

#if NX_UMAC_PRESENT && NX_BEACONING
/**
 ****************************************************************************************
 * @brief Callback function of the HE trigger scheduler timer.
 *
 * In this function we program a HE trigger frame for transmission and we set the timer
 * for the next trigger.
 *
 * @param[in] env  Pointer to the STA structure
 *
 ****************************************************************************************
 */
static void txl_he_trig_time(void *env)
{
    struct sta_info_tag *sta = (struct sta_info_tag *)env;

    if (sta->valid)
    {
        mm_timer_set(&txl_he_env.trig_timer, hal_machw_time() + TX_HE_TRIG_PERIOD);
        txl_frame_send_he_trigger(sta->staid, HE_TRIG_TYPE_BASIC, 2500, 37, 5, AC_VI,
                                  NULL, NULL);
    }
    else
    {
        txl_he_env.trig_timer.cb = NULL;
        txl_he_env.trig_timer.env = NULL;
    }
}
#endif // NX_UMAC_PRESENT && NX_BEACONING

/**
 ****************************************************************************************
 * @brief Halt the HE TB HW queue
 ****************************************************************************************
 */
static void txl_he_tb_halt_queue(void)
{
    nxmac_dma_cntrl_set(NXMAC_HALT_TB_AFTER_TXOP_BIT);
    while(nxmac_tx_tb_state_getf() != 0);
    nxmac_dma_cntrl_clear(NXMAC_HALT_TB_AFTER_TXOP_BIT);
}

/**
 ****************************************************************************************
 * @brief Start the TX HE TB timer
 *
 ****************************************************************************************
 */
__INLINE void txl_he_tb_timer_start(void)
{
    nxmac_abs_timer_set(HAL_HE_TB_TIMER, hal_machw_time() + TX_HE_TB_TIMEOUT);
    nxmac_timers_int_event_clear(HAL_HE_TB_TIMER_BIT);
    nxmac_timers_int_un_mask_set(nxmac_timers_int_un_mask_get() | HAL_HE_TB_TIMER_BIT);

    txl_he_env.tb_ongoing = true;
}

/**
 ****************************************************************************************
 * @brief Clear the TX HE TB timer
 *
 ****************************************************************************************
 */
__INLINE void txl_he_tb_timer_clear(void)
{
    uint32_t timer_msk = nxmac_timers_int_un_mask_get();

    nxmac_timers_int_un_mask_set(timer_msk & ~HAL_HE_TB_TIMER_BIT);
    nxmac_timers_int_event_clear(HAL_HE_TB_TIMER_BIT);

    txl_he_env.tb_ongoing = false;
}

/**
 ****************************************************************************************
 * @brief Get HE TB PSDU maximum length
 *
 * This function reads the HE TB parameters advertised by the AP in the trigger frame,
 * stores the ones that might be reused later in the HE TB processing, and gets the
 * maximum length that the HE TB could carry from the dedicated HW register.
 *
 * @return The PSDU maximum length
 ****************************************************************************************
 */
static uint32_t txl_he_tb_psdulen_get(void)
{
    uint32_t common;
    uint32_t user;
    uint32_t l_length;
    uint32_t gi_ltf_type;
    uint32_t mcs;
    uint32_t rulen;
    uint32_t psdulen;
    struct he_trigger_tag *he_tb = &txl_he_env.trigger;

    // Get HE TB parameters from the trigger frame
    common = nxmac_rx_he_trig_common_info_get();
    user = nxmac_rx_he_trig_user_info_get();
    l_length = (common & NXMAC_UL_LENGTH_MASK) >> NXMAC_UL_LENGTH_LSB;
    gi_ltf_type = (common & NXMAC_UL_GILTF_TYPE_MASK) >> NXMAC_UL_GILTF_TYPE_LSB;
    mcs = (user & NXMAC_UL_MCS_MASK) >> NXMAC_UL_MCS_LSB;
    rulen = (user & NXMAC_UL_RU_SIZE_MASK) >> NXMAC_UL_RU_SIZE_LSB;
    he_tb->trig_type = (common & NXMAC_UL_TRIGGER_TYPE_MASK) >> NXMAC_UL_TRIGGER_TYPE_LSB;
    he_tb->ul_bw = (common & NXMAC_UL_BW_MASK) >> NXMAC_UL_BW_LSB;
    he_tb->ul_length = l_length;
    he_tb->gi_type = gi_ltf_type < 2 ? 1 : 2;
    he_tb->uora = (user & NXMAC_UL_RU_TYPE_BIT) != 0 ? true : false;
    he_tb->ul_mcs = mcs;
    he_tb->ul_nss = (user & NXMAC_UL_NSS_MASK) >> NXMAC_UL_NSS_LSB;
    he_tb->ru_size = rulen;
    if (he_tb->trig_type == HE_TRIG_TYPE_BASIC)
    {
        uint32_t td_user = (user & NXMAC_TD_USER_INFO_SUBFIELD_MASK)
                                                       >> NXMAC_TD_USER_INFO_SUBFIELD_LSB;
        he_tb->pref_ac = mac_aci2ac[(td_user & HE_TRIG_PREF_AC_MSK) >> HE_TRIG_PREF_AC_OFT];
        he_tb->spacing_factor = (td_user & HE_TRIG_SPC_FACTOR_MSK) >> HE_TRIG_SPC_FACTOR_OFT;
    }

    while (!nxmac_ul_psdu_len_valid_getf());
    psdulen = nxmac_ul_psdu_len_getf();

    return CO_ALIGN4_LO(psdulen);
}

/**
 ****************************************************************************************
 * @brief This function computes the maximum length and minimum start spacing, in bytes,
 * that the HE TB PPDU should comply with.
 ****************************************************************************************
 */
static void txl_he_tb_params_compute(void)
{
    struct he_trigger_tag *he_tb = &txl_he_env.trigger;
    uint16_t len_1us;
    uint32_t max_len_phy;
    struct sta_info_tag *sta = he_tb->sta;

    // Compute the maximum length we can put inside the HE TB
    max_len_phy = txl_he_tb_psdulen_get();

    // Now use the correct table to retrieve the parameters
    switch (he_tb->ru_size)
    {
        case RU_26TONE:
        case RU_52TONE:
        case RU_106TONE:
            len_1us =  ru_1us_tables[he_tb->ru_size][he_tb->ul_mcs * 3 + he_tb->gi_type];
            break;
        default:
            len_1us = txl_he_idx_to_1us_len_get(he_tb->ul_mcs * 6 + he_tb->gi_type,
                                                he_tb->ru_size - 3);
            break;
    }

    he_tb->max_len = co_min(max_len_phy, sta->ampdu_size_max_he);
    he_tb->mmss = len_1us * (uint16_t)sta->ampdu_spacing_min *
                  (uint16_t)(1 << he_tb->spacing_factor);
    // Get UPH from HW register and format the +HTC HE Variant field
    he_tb->uph = MAC_HTC_HE(MAC_HTC_HE_CTRL(UPH, nxmac_he_tbuph_control_getf()));
}

/**
 ****************************************************************************************
 * @brief This function computes the length of a AID/TID information structure inside a
 * Multi-STA BlockAck frame.
 *
 * It also checks, based on the remaining length in the buffer in which the AID/TID info
 * structure is located, if it fits in the buffer.
 *
 * @param[in] aid_tid_info  Pointer to the AID/TID information structure
 * @param[in] remaining_len Length from the start of the AID/TID information structure up
 *                          to the end of the buffer where it is located
 *
 * @return The length of the AID/TID information if all checks are OK, -1 otherwise.
 ****************************************************************************************
 */
static int txl_he_m_ba_aid_tid_len(struct ba_msta_aid_tid_info *aid_tid_info,
                                   int remaining_len)
{
    int len;
    uint16_t frag;
    uint16_t aid = (aid_tid_info->aid_tid & BA_MULTI_STA_AID_MSK) >> BA_MULTI_STA_AID_OFT;

    if (aid == 2045)
    {
        return 12;
    }
    else if (aid_tid_info->aid_tid & BA_MULTI_STA_ACK_BIT)
    {
        return 2;
    }
    else if (remaining_len < (int)sizeof_b(*aid_tid_info))
    {
        return -1;
    }

    // Get fragment number inside SSC
    frag = aid_tid_info->ssc_bitmap.ssc & MAC_SEQCTRL_FRAG_MSK;

    if (frag & CO_BIT(3))
        return -1;

    len = fragb2b1_to_bitmap_len[frag >> 1] + sizeof_b(*aid_tid_info);

    if (remaining_len < len)
        return -1;

    // All checks passed, return length
    return len;
}

/**
 ****************************************************************************************
 * @brief Copy the characteristics of the HE TB PPDU in the specific RC statistics element
 ****************************************************************************************
 */
static void txl_he_tb_set_rc_info(void)
{
    #if NX_UMAC_PRESENT
    struct he_trigger_tag *he_tb = &txl_he_env.trigger;
    struct rc_sta_stats *rc_ss = he_tb->sta->pol_tbl.sta_stats;
    struct rc_rate_stats *rc_rs = &rc_ss->rate_stats[RC_MAX_N_SAMPLE];

    rc_rs->rate_config = (FORMATMOD_HE_MU << FORMAT_MOD_TX_RCX_OFT) |
             (he_tb->ul_mcs << VHT_MCS_OFT) | (he_tb->gi_type << HE_GI_TYPE_TX_RCX_OFT) |
             (he_tb->ul_bw << BW_TX_RCX_OFT) |(he_tb->ul_nss << VHT_NSS_OFT);
    rc_rs->ru_and_length = he_tb->ru_size | (he_tb->ul_length << 3);
    #endif
}

/**
 ****************************************************************************************
 * @brief Chain a PPDU into the HE TB queue.
 *
 * @param[in] thd  TX Header Descriptor to be chained
 ****************************************************************************************
 */
static void txl_he_tb_ppdu_chain(struct tx_hd *thd)
{
    thd->statinfo &= NUM_MPDU_RETRIES_MSK;
    thd->nextfrmexseq_ptr = 0;
    thd->policyentryaddr = CPU2HW(&he_tb_desc.tb_pol);
    nxmac_tx_tb_head_ptr_set(CPU2HW(thd));
    nxmac_dma_cntrl_set(NXMAC_TX_TB_NEW_HEAD_BIT);

    txl_he_env.trigger.trig_valid = false;
}

/**
 ****************************************************************************************
 * @brief Push a NDP frame in the HE TB queue.
 *
 * This frame is the shortest one we can transmit. It is used when the resources
 * allocated by the AP in the trigger frame are not enough to put at least one QoS-NULL
 * frame in the HE TB PPDU.
 ****************************************************************************************
 */
static void txl_he_tb_send_ndp(void)
{
    struct tx_hd *thd = &he_tb_desc.ndp.thd;

    txl_he_tb_ppdu_chain(thd);
}

/**
 ****************************************************************************************
 * @brief Push a QoS-NULL frame in the HE TB queue.
 *
 * The content of the QoS-NULL frame is taken from the information decoded from the
 * trigger frame.
 ****************************************************************************************
 */
static void txl_he_tb_send_qos_null(void)
{
    struct he_trigger_tag *he_tb = &txl_he_env.trigger;
    struct tx_hd *thd = &he_tb_desc.qos_null.thd;
    struct mac_hdr_qos_htc *payl = &he_tb_desc.qos_null.payl;

    payl->addr1 = he_tb->ta;
    payl->addr2 = txl_he_env.tb_vif->mac_addr;
    payl->addr3 = he_tb->ta;
    payl->htc = he_tb->uph;

    txl_he_tb_ppdu_chain(thd);
}

/**
 ****************************************************************************************
 * @brief Send HE-TB BSRP A-MPDU.
 *
 * @param[in] he_tb  Pointer to Information structure about the received HE trigger frame
 ****************************************************************************************
 */
__INLINE void txl_he_tb_send_bsrp(struct he_trigger_tag *he_tb)
{
    struct tx_hd *a_thd = &he_tb_desc.brsp.a_thd;
    struct tx_hd *thd_prev = a_thd;
    struct mac_hdr *hdr = &he_tb_desc.brsp.hdr;
    uint8_t tid, queue_size;
    uint16_t nb_delims, subfrm_len;
    uint32_t buffered;
    uint32_t whichdesc = WHICHDESC_AMPDU_FIRST;

    a_thd->frmlen = 0;

    hdr->addr1 = he_tb->sta->mac_addr;
    hdr->addr2 = txl_he_env.tb_vif->mac_addr;
    hdr->addr3 = he_tb->sta->mac_addr;

    // Compute the new number of delimiters and the sub-frame length
    nb_delims = txl_agg_mpdu_nb_delims(&he_tb_desc.brsp.qos_null[0].thd, he_tb->mmss);
    subfrm_len = txl_mpdu_subframe_len(&he_tb_desc.brsp.qos_null[0].thd) + nb_delims * DELIMITER_LEN;

    for (tid = 0; tid < TID_MAX; tid++)
    {
        struct tx_hd *thd = &he_tb_desc.brsp.qos_null[tid].thd;
        uint16_t *qos = &he_tb_desc.brsp.qos_null[tid].qos_ctrl;

        buffered = macif_buffered_get(he_tb->sta->staid, tid);
        if (!buffered)
            continue;

        // Check if the MPDU fits into the remaining length
        if ((a_thd->frmlen + subfrm_len) > he_tb->max_len)
            break;

        queue_size = txl_he_get_queue_size(buffered);
        // set the 4th bit of the QoS Control field
        *qos = MAC_QOSCTRL_QUEUE_SIZE_PRESENT | MAC_QOSCTRL_ACK_NOACK;
        // Add the queue_size
        *qos |= queue_size << MAC_QOSCTRL_AMSDU_OFT;
        // Add the TID
        *qos |= tid << MAC_QOSCTRL_UP_OFT;

        thd->policyentryaddr = 0;
        thd->statinfo = 0;
        thd->macctrlinfo2 = whichdesc | (nb_delims << NB_BLANK_DELIM_OFT);
        whichdesc = WHICHDESC_AMPDU_INT;
        thd_prev->nextmpdudesc_ptr = CPU2HW(thd);

        a_thd->frmlen += subfrm_len;

        thd_prev = thd;
    }

    if (a_thd->frmlen)
    {
        thd_prev->nextmpdudesc_ptr = 0;
        if ((thd_prev->macctrlinfo2 & WHICHDESC_MSK) == WHICHDESC_AMPDU_FIRST)
        {
            // Send QoS NULL as singleton
            thd_prev->macctrlinfo2 = WHICHDESC_UNFRAGMENTED_MSDU;
            txl_he_tb_ppdu_chain(thd_prev);
        }
        else
        {
            thd_prev->macctrlinfo2 &= ~WHICHDESC_MSK;
            thd_prev->macctrlinfo2 |= WHICHDESC_AMPDU_LAST;
            txl_he_tb_ppdu_chain(a_thd);
        }
    }
    else
    {
        struct tx_hd *thd = &he_tb_desc.brsp.qos_null[0].thd;
        uint16_t *qos = &he_tb_desc.brsp.qos_null[0].qos_ctrl;

        // No traffic present on any TID, simply send a QoS-NULL with TID0 and queue size
        // equal to 0
        *qos = MAC_QOSCTRL_QUEUE_SIZE_PRESENT | MAC_QOSCTRL_ACK_NOACK;
        thd->macctrlinfo2 = WHICHDESC_UNFRAGMENTED_MSDU;
        thd->nextmpdudesc_ptr = 0;

        txl_he_tb_ppdu_chain(thd);
    }
}
/**
 ****************************************************************************************
 * @brief Push a data PPDU frame in the HE TB queue.
 *
 * @param[in] thd The pointer to the THD to be chained in the HE TB queue.
 ****************************************************************************************
 */
static void txl_he_tb_send_data(struct tx_hd *thd)
{
    #if NX_PROFILING_ON
    struct he_trigger_tag *he_tb = &txl_he_env.trigger;
    #endif

    PROF_TX_AC_IRQ_SET(he_tb->ac);

    // Start the TX activity timer
    txl_he_tb_timer_start();

    txl_he_tb_ppdu_chain(thd);
}

/**
 ****************************************************************************************
 * @brief Free a singleton MPDU detected as done when looking for the HE TB PPDU to
 * be transmitted.
 *
 * @param[in] txdesc The pointer to the TX descriptor done
 * @param[in] access_category The access category of the descriptor
 ****************************************************************************************
 */
static void txl_he_tb_free_done(struct txdesc *txdesc, uint8_t access_category)
{
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];
    struct tx_hd *txhd = &txdesc->lmac.hw_desc->thd;

    // Pop picked txdesc definitively from list
    co_list_pop_front(&(txlist->transmitting[0]));

    // We have one PPDU less in the TX path
    txlist->ppdu_cnt--;

    #if NX_TX_FRAME
    // Check if the frame was generated internally
    if (is_int_frame(txdesc))
    {
        // Put the descriptor in the CFM queue
        txl_frame_cfm(txdesc);
    }
    else
    #endif
    {
        // Put the descriptor in the CFM queue
        txl_cfm_push(txdesc, txhd->statinfo, access_category);
    }
}

/**
 ****************************************************************************************
 * @brief Get the PPDU to be programmed in the HE TB.
 *
 * This function goes through the TX queues until it finds a queue with data to be sent
 * in the HE TB.
 *
 * @param[out] thd A pointer to the TX descriptor associated with the MPDU just following
 * the MPDU(s) transmitted over the HE TB
 *
 * @return The pointer to the THD to be chained in the HE TB queue.
 ****************************************************************************************
 */
static struct tx_hd *txl_he_tb_ppdu_get(struct txdesc **txdesc_next)
{
    struct txdesc *txdesc = NULL;
    int access_category;
    struct he_trigger_tag *he_tb = &txl_he_env.trigger;

    *txdesc_next = NULL;

    // Check we have already an HE TB ongoing
    if (txl_he_env.tb_ongoing)
        return NULL;

    // For now only consider the AC specified in the trigger frame
    for (access_category = AC_VO; access_category >= he_tb->pref_ac; access_category--)
    {
        struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];

        he_tb->ac = access_category;

        // In case we were waiting for the BAR THD to be updated, it means that we did not
        // receive the BlockAck of the last A-MPDU transmission. Don't program any data
        // in the HE TB in that case
        if (txlist->chk_state == BAR_THD_CHK_STATE)
            return NULL;

        while (1)
        {
            struct tx_hd *thd;
            struct tx_agg_desc *agg_desc;

            // Pick first txdesc in transmitting list
            txdesc = (struct txdesc *)co_list_pick(&(txlist->transmitting[0]));

            // Check if a descriptor is waiting to be confirmed
            if (txdesc == NULL)
                break;

            // Check the doneTx status bit
            if (!is_mpdu_agg(txdesc) && (txdesc->lmac.hw_desc->thd.statinfo & DESC_DONE_TX_BIT))
            {
                txl_he_tb_free_done(txdesc, access_category);
                continue;
            }

            thd = txl_agg_he_tb_prep(txdesc, txdesc_next, he_tb->max_len, he_tb->mmss,
                                     he_tb->ac);

            if (!thd)
                break;

            agg_desc = txdesc->lmac.agg_desc;

            // All the HE TB PPDU transmissions are done with a AGG descriptor attached
            ASSERT_ERR(agg_desc != NULL);

            // Indicate that this AGG descriptor corresponds to a HE TB PPDU transmission
            // This will be used when deciding whether the +HTC value to put is the BSR
            // or the UPH
            agg_desc->status |= AGG_TB;
            return thd;
        }
    }

    return NULL;
}

/**
 ****************************************************************************************
 * @brief Halt and re-program the AC queue from which a HE TB frame was extracted and
 * chained.
 *
 * @param[in] txdesc TX descriptor just following the last MPDU transmitted in the HE TB
 * PPDU (NULL if no MPDU available).
 ****************************************************************************************
 */
static void txl_he_tb_ac_reconfig(struct txdesc *txdesc)
{
    struct he_trigger_tag *he_tb = &txl_he_env.trigger;
    struct mu_edca_param_tag *mu_edca_param = &txl_he_env.mu_edca.params[he_tb->ac];
    struct txl_list *txlist = &txl_cntrl_env.txlist[he_tb->ac];
    struct tx_hd *last_thd = NULL;
    struct tx_hd *next_thd = NULL;

    // First halt the queue
    txl_cntrl_halt_ac(he_tb->ac);

    // Check if a previous A-MPDU descriptor was saved and needs to be freed
    txl_agg_check_saved_agg_desc(he_tb->ac);

    do
    {
        // Check if a packet could be available for transmission after the HE TB PPDU
        // transmission
        if (!txdesc)
            break;

        if (is_mpdu_agg(txdesc))
        {
            struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;

            // Check if the next packet to transmit is a remainder of a possibly split
            // A-MPDU we programmed as HE TB, or the head of a fresh A-MPDU
            if (is_mpdu_last(txdesc))
            {
                // The programmed HE TB PPDU is a split A-MPDU, and only one MPDU
                // composes the remainder of this A-MPDU. Let's change it into a
                // singleton MPDU
                next_thd = last_thd = txl_agg_change_to_singleton(txdesc, false);

                if (next_thd)
                    next_thd->nextfrmexseq_ptr = agg_desc->bar_thd.nextfrmexseq_ptr;

                txl_agg_desc_free(agg_desc);
            }
            else if (is_mpdu_interm(txdesc))
            {
                // The programmed HE TB PPDU is a split A-MPDU, and several packets
                // compose the remainder of this A-MPDU. Let's set the present MPDU as
                // the first of the next A-MPDU to transmit
                next_thd = txl_agg_set_new_ampdu_head(txdesc, &last_thd);
            }
            // The next packet of the queue is the head of a fresh A-MPDU, let's verify
            // if it is ready to be programmed or not
            else if ((agg_desc->status & (AGG_FORMATTED | AGG_DOWNLOADED)) ==
                                                        (AGG_FORMATTED | AGG_DOWNLOADED))
            {
                next_thd = &agg_desc->a_thd;
                last_thd = &agg_desc->bar_thd;
            }
        }
        else
        {
            if (txl_buffer_get(txdesc) == NULL)
                break;

            next_thd = last_thd = &txdesc->lmac.hw_desc->thd;

            // Remove the +HTC field if required
            if (!is_htc_sglt_allowed(txdesc))
                txl_buffer_remove_htc(next_thd);
        }

        // Ensure that we really have something to program as next frame exchange
        if (next_thd == NULL)
            break;

        // Remind the THD of the next frame exchange sequence to be chained
        mu_edca_param->first_frame_exch = next_thd;
        if (last_thd->nextfrmexseq_ptr == 0)
            txlist->last_frame_exch = last_thd;

        // All checks are OK, so we can return
        return;
    } while(0);

    // Nothing ready to program as next frame exchange, clear all the queue parameters
    txlist->last_frame_exch = NULL;
    mu_edca_param->first_frame_exch = NULL;
}

/**
 ****************************************************************************************
 * @brief Prepare and program the transmission of a HE TB PPDU, upon the reception of a
 * trigger frame.
 ****************************************************************************************
 */
static void txl_he_tb_ppdu_send(void)
{
    struct txdesc *txdesc = NULL;
    struct tx_hd *thd = NULL;
    struct he_trigger_tag *he_tb = &txl_he_env.trigger;

    // Check validity of trigger frame
    if (!he_tb->trig_valid)
    {
        // If not valid, we simply send a NDP in response, in order to avoid getting
        // a PHY error
        txl_he_tb_send_ndp();
        return;
    }

    // BSRP Trigger frame
    if (he_tb->trig_type == HE_TRIG_TYPE_BSRP)
    {
        txl_he_tb_send_bsrp(he_tb);
        return;
    }

    // Get the information about the HE TB PPDU to transmit
    thd = txl_he_tb_ppdu_get(&txdesc);

    if (thd == NULL)
    {
        txl_he_tb_send_qos_null();
        return;
    }

    // Program the HE TB PPDU for transmission
    txl_he_tb_send_data(thd);

    // Re-configure the HW AC queue with the next packet to transmit
    txl_he_tb_ac_reconfig(txdesc);
}

/**
 ****************************************************************************************
 * @brief Initialize the HW descriptors used for HE TB PPDU transmissions
 ****************************************************************************************
 */
static void txl_he_tb_desc_init(void)
{
    struct tx_hd *thd;
    struct mac_hdr_qos *payl;
    struct mac_hdr *hdr;
    struct tx_policy_tbl *pol = &he_tb_desc.tb_pol;
    uint8_t tid;

    // Reset all fields in a all descriptors
    memset(&he_tb_desc, 0, sizeof(he_tb_desc));

    /*
     ************************************************************************************
     * HE TB policy table
     ************************************************************************************
     */
    pol->upatterntx = POLICY_TABLE_PATTERN;

    /*
     ************************************************************************************
     * Default QoS-NULL descriptors
     ************************************************************************************
     */
    payl = (struct mac_hdr_qos *)&he_tb_desc.qos_null.payl;
    thd = &he_tb_desc.qos_null.thd;

    // Fill payload
    payl->fctl = MAC_FCTRL_QOS_NULL | MAC_FCTRL_TODS;
    payl->qos = MAC_QOSCTRL_ACK_NOACK | MAC_QOSCTRL_HE_QUEUE_SIZE;

    // Initialize THD descriptor
    thd->upatterntx = TX_HEADER_DESC_PATTERN;
    thd->policyentryaddr = CPU2HW(pol);
    thd->frmlen = MAC_QOS_NULL_FRAME_SIZE + MAC_HTC_LEN + MAC_FCS_LEN;
    thd->datastartptr = CPU2HW(payl);
    thd->dataendptr = (uint32_t)thd->datastartptr + thd->frmlen - 1;
    thd->first_pbd_ptr = 0;
    thd->macctrlinfo1 = EXPECTED_ACK_NO_ACK;

    /*
     ************************************************************************************
     * Default NDP descriptors
     ************************************************************************************
     */
    thd = &he_tb_desc.ndp.thd;
    thd->upatterntx = TX_HEADER_DESC_PATTERN;
    thd->macctrlinfo1 = EXPECTED_ACK_NO_ACK;
    thd->policyentryaddr = CPU2HW(pol);

    /*
     ************************************************************************************
     * Default BSRP descriptors
     ************************************************************************************
     */
    thd = &he_tb_desc.brsp.a_thd;
    thd->upatterntx = TX_HEADER_DESC_PATTERN;
    thd->macctrlinfo2 = WHICHDESC_AMPDU_EXTRA;
    thd->policyentryaddr = CPU2HW(pol);
    hdr = &he_tb_desc.brsp.hdr;
    hdr->fctl = MAC_FCTRL_QOS_NULL | MAC_FCTRL_TODS;
    for (tid = 0; tid < TID_MAX; tid++)
    {
        struct tx_pbd *pbd = &he_tb_desc.brsp.qos_null[tid].pbd;
        thd = &he_tb_desc.brsp.qos_null[tid].thd;

        thd->upatterntx = TX_HEADER_DESC_PATTERN;
        thd->frmlen = MAC_QOS_NULL_FRAME_SIZE + MAC_FCS_LEN;
        thd->datastartptr = CPU2HW(hdr);
        thd->dataendptr = (uint32_t)thd->datastartptr + sizeof_b(*hdr) - 1;
        thd->first_pbd_ptr = CPU2HW(pbd);
        thd->macctrlinfo1 = EXPECTED_ACK_NO_ACK;
        pbd->upatterntx = TX_PAYLOAD_DESC_PATTERN;
        pbd->datastartptr = CPU2HW(&he_tb_desc.brsp.qos_null[tid].qos_ctrl);
        pbd->dataendptr = (uint32_t)pbd->datastartptr + 1;
    }
}

/**
 ****************************************************************************************
 * @brief Handle the HE TB TX trigger for a Singleton MPDU.
 *
 * This function checks if the MPDU has been correctly acknowledged and programs its
 * retransmission if required.
 *
 * @param[in] txdesc TX descriptor that is marked as done by the HW
 * @param[in] txstatus TX status written by the HW
 ****************************************************************************************
 */
static void txl_he_tb_mpdu_transmit_trigger(struct txdesc *txdesc,
                                            uint32_t txstatus)
{
    struct he_trigger_tag *he_tb = &txl_he_env.trigger;
    uint8_t access_category = he_tb->ac;
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];
    struct mu_edca_param_tag *mu_edca_param = &txl_he_env.mu_edca.params[access_category];
    struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;
    struct tx_hd *txhd = &txdesc->lmac.hw_desc->thd;

    // Check HW status - In case of HE TB a successful status does not automatically mean
    // that the MPDU was correctly acknowledged.
    // Indeed we can receive a multi-STA BA in response, which is enough for HW to
    // consider the MPDU transmission as successful, but this multi-STA BA may not
    // include acknowledgment information for our device. Additional checks are therefore
    // required.
    if (txstatus & FRAME_SUCCESSFUL_TX_BIT)
    {
        if (he_tb->uora)
            // HE TB transmission is considered successful in the selected RA-RU
            nxmac_e_ocw_setf(txl_he_env.uora.eocw_min);
        else
            // Restart the MU EDCA for this access category
            txl_he_mu_edca_start(he_tb->ac);

        // Get possibly received BA frame
        rxl_immediate_frame_get();
    }
    else
    {
        if (he_tb->uora)
        {
            // HE TB transmission in RA-RU is unsuccessful
            uint8_t new_eocw = nxmac_e_ocw_getf();
            new_eocw = new_eocw + 1;

            if (new_eocw > txl_he_env.uora.eocw_max)
                new_eocw = txl_he_env.uora.eocw_max;

            nxmac_e_ocw_setf(new_eocw);
        }
    }

    // If we did not get any BA, we need to pop the descriptor here
    if (!(agg_desc->status & AGG_BA_RECEIVED))
    {
        // Pop the aggregate descriptor from the list
        co_list_pop_front(&txlist->aggregates);
    }
    else if (!(agg_desc->status & AGG_BA_VALID))
    {
        // We received a multi-STA BA, but it does not contain any acknowledgment for us
        txhd->statinfo &= ~FRAME_SUCCESSFUL_TX_BIT;
        txstatus &= ~FRAME_SUCCESSFUL_TX_BIT;
    }

    // Release the A-MPDU descriptor immediately, because in case of singleton it is used
    // only for the BA reception handling, but not during the confirmation phase
    txl_agg_release(agg_desc);
    txdesc->lmac.agg_desc = NULL;

    // Check if we need to retry the MPDU
    if (!(txstatus & FRAME_SUCCESSFUL_TX_BIT))
    {
        uint32_t nb_retry = (txstatus & NUM_MPDU_RETRIES_MSK) >> NUM_MPDU_RETRIES_OFT;
        bool amsdu = false;

        #if NX_AMSDU_TX
        // Check if this packet is split across several buffers, because if this is the
        // case we cannot do the retry as some of the buffers have already been freed
        // from shared memory. The retries in such case are performed as part of the BA
        // agreement which is established (A-MSDUs are used only if a BA agreement is in
        // place)
        if (txl_buffer_is_amsdu_multi_buf(txdesc))
            amsdu = true;
        #endif

        if (!amsdu && (nb_retry < 5))
        {
            struct txl_buffer_control *bufctrl = txl_buffer_control_get(txdesc);

            // For the first retransmission, let's copy (maybe again) the policy table
            // in the buffer, as it might not be present if the MPDU was initially
            // supposed to be transmitted inside an A-MPDU.
            if (!nb_retry)
            {
                #if NX_UMAC_PRESENT
                // Copy the buffer control
                txl_buffer_control_copy(txdesc, txl_buffer_get(txdesc));
                #endif

                // Check if the singleton shall be protected with a RTS/CTS
                txl_he_txop_dur_rtscts_thres_mpdu_check(txdesc);
            }

            nb_retry++;
            txhd->macctrlinfo1 = bufctrl->mac_control_info;
            txhd->phyctrlinfo = bufctrl->phy_control_info;
            txhd->policyentryaddr = CPU2HW(&bufctrl->policy_tbl);
            txhd->statinfo = nb_retry << NUM_MPDU_RETRIES_OFT;

            // Chain the retransmission at the head of the list, because it shall be
            // the next packet transmitted in this access category, in order to ensure
            // correct ordering in case no BA agreement is setup
            if (mu_edca_param->first_frame_exch)
                txhd->nextfrmexseq_ptr = CPU2HW(mu_edca_param->first_frame_exch);
            else
                txlist->last_frame_exch = txhd;
            mu_edca_param->first_frame_exch = txhd;

            // Increase by one the PPDU count because the current packet is retried, and
            // this counter will be decreased later in the processing
            txlist->ppdu_cnt++;
            return;
        }
        else
        {
            // We reached the retry limit, let's indicate it to the upper layers
            txhd->statinfo |= RETRY_LIMIT_REACHED_BIT;
        }
    }

    #if !NX_FULLY_HOSTED
    // Free the MPDU buffers and allocate buffer(s) for the subsequent ones
    txl_free_done_mpdu(txdesc, access_category, 0);
    #endif

    // Pop picked txdesc definitively from list
    co_list_pop_front(&(txlist->transmitting[0]));

    #if NX_TX_FRAME
    // Check if the frame was generated internally
    if (is_int_frame(txdesc))
    {
        // Put the descriptor in the CFM queue
        txl_frame_cfm(txdesc);
    }
    else
    #endif
    {
        // Put the descriptor in the CFM queue
        txl_cfm_push(txdesc, txstatus, access_category);
    }
}

/**
 ****************************************************************************************
 * @brief Handle the HE TB TX trigger for a A-MPDU constituent.
 *
 * @param[in] txdesc TX descriptor that is marked as done by the HW
 * @param[in] txstatus TX status written by the HW
 ****************************************************************************************
 */
static void txl_he_tb_ampdu_mpdu_transmit_trigger(struct txdesc *txdesc,
                                                  uint32_t txstatus)
{
    struct he_trigger_tag *he_tb = &txl_he_env.trigger;
    uint8_t access_category = he_tb->ac;
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];

    #if !NX_FULLY_HOSTED
    // Free the MPDU buffers and allocate buffer(s) for the subsequent ones
    txl_free_done_mpdu(txdesc, access_category, 0);
    #endif

    // Pop picked txdesc definitively from list
    co_list_pop_front(&(txlist->transmitting[0]));

    // Put the descriptor in the CFM queue
    txl_cfm_push(txdesc, txstatus, access_category);

    // At last MPDU in AMPDU, we stop pushing CFMs until BAR status done;
    // Keep last mpdu pointer
    if (is_mpdu_last(txdesc))
    {
        txlist->agg_desc = txdesc->lmac.agg_desc;
        txlist->chk_state = ATHD_CHK_STATE;
    }
}

void txl_he_bsr_compute(struct vif_info_tag *vif)
{
    #if NX_UMAC_PRESENT
    uint32_t bsr, buffered, buffered_all = 0, buffered_aci_high = 0;
    uint8_t aci_bitmap = 0, delta_tid = 0, tids_nb = 0;
    uint8_t aci_bitmap_set = 0, aci_high = 0, tid;
    uint8_t qs_high = 0, qs_all, sf;
    uint32_t buffered_per_aci[AC_MAX];
    struct sta_info_tag *sta;

    // Check if VIF is of type STA and active
    if ((vif->type != VIF_STA) || !vif->active)
        return;

    // Get STA information
    sta = &sta_info_tab[vif->u.sta.ap_id];
    if (!sta->valid || !STA_CAPA(sta, HE))
        return;

    // Compute BSR
    bsr = MAC_HTC_TYPE_HE | (MAC_HTC_HE_CTRL_ID_BSR << MAC_HTC_HE_A_CONTROL_OFT);

    memset(buffered_per_aci, 0, sizeof(buffered_per_aci));
    for (tid = 0; tid < TID_MGT; tid++)
    {
        uint8_t aci = mac_ac2aci[mac_tid2ac[tid]];
        buffered = macif_buffered_get(sta->staid, tid);
        if (!buffered)
            continue;

        tids_nb++;

        // Check whether the amount of buffered data for this TID is higher than the max
        // we can code in the BSR - If this is the case, limit it to this size, as this
        // will ensure there is no overflow of the 32-bit variables used in the
        // computation
        if (buffered > BSR_BUFFERED_MAX)
            buffered = BSR_BUFFERED_MAX;

        buffered_all += buffered;
        buffered_per_aci[aci] += buffered;

        // Check whether the current ACI is the one having the more buffered data
        if (buffered_per_aci[aci] > buffered_aci_high)
        {
            buffered_aci_high = buffered_per_aci[aci];
            aci_high = aci;
        }

        // Check if the bit is already set
        if (CO_BIT(aci) & aci_bitmap)
            continue;

        // Set ACI in bitmap
        aci_bitmap |= CO_BIT(aci);
        aci_bitmap_set++;
    }

    if (tids_nb == TID_MGT)
    {
        aci_bitmap = 0;
        delta_tid = 3;
    }
    else
        delta_tid = tids_nb - aci_bitmap_set;

    sf = txl_he_bsr_get_sf_and_qsize(buffered_all, buffered_aci_high, &qs_all, &qs_high);

    bsr |= aci_bitmap << (MAC_HTC_HE_BSR_ACI_BMP_OFT + MAC_HTC_HE_A_CONTROL_OFT);
    bsr |= delta_tid << (MAC_HTC_HE_BSR_DELTA_TID_OFT + MAC_HTC_HE_A_CONTROL_OFT);
    bsr |= aci_high << (MAC_HTC_HE_BSR_ACI_HIGH_OFT + MAC_HTC_HE_A_CONTROL_OFT);
    bsr |= sf << (MAC_HTC_HE_BSR_SCALING_FAC_OFT + MAC_HTC_HE_A_CONTROL_OFT);
    bsr |= qs_high << (MAC_HTC_HE_BSR_QSIZE_HIGH_OFT + MAC_HTC_HE_A_CONTROL_OFT);
    bsr |= qs_all << (MAC_HTC_HE_BSR_QSIZE_ALL_OFT + MAC_HTC_HE_A_CONTROL_OFT);

    sta->bsr = bsr;
    #endif
}

void txl_he_tb_transmit_cancelled(void)
{
    uint8_t access_category;
    struct txl_list *txlist;
    struct txdesc *txdesc;
    struct tx_agg_desc *agg_desc;
    struct mu_edca_param_tag *mu_edca_param;

    // Halt the HE TB HW queue in order to ensure that the HW does not restart the same
    // HE TB PPDU transmission upon the next trigger frame reception
    txl_he_tb_halt_queue();

    // If only QoS-Null or BSRP response has been programmed, no need to free anything
    if (!txl_he_env.tb_ongoing)
        return;

    access_category = txl_he_env.trigger.ac;
    txlist = &txl_cntrl_env.txlist[access_category];
    mu_edca_param = &txl_he_env.mu_edca.params[access_category];

    PROF_TX_MAC_IRQ_SET();
    PROF_TX_AC_IRQ_SET(access_category);

    // Pick first TX descriptor in the transmitting list
    txdesc = (struct txdesc *)co_list_pick(&(txlist->transmitting[0]));

    // Here we check if the next TX descriptor for which the buffer has to be allocated
    // and downloaded is part of the A-MPDU sent in this HE TB PPDU. If this is the case,
    // and because this HE TB PPDU won't be transmitted, we move the first_to_download
    // pointer to the TX descriptor immediately following the last one sent in the HE TB
    // PPDU. This avoids spending time to allocate, format and free buffers that will
    // anyway not be transmitted - Do this only if we are in the THD check state, i.e. if
    // we programmed some data MPDUs that were not transmitted because the HE TB is
    // cancelled. If we programmed a BAR in the HE TB, we shall not enter into this code
    if (txlist->chk_state == THD_CHK_STATE)
    {
        agg_desc = txdesc->lmac.agg_desc;
        if (is_mpdu_agg(txdesc) && (txlist->first_to_download[0]->lmac.agg_desc == agg_desc))
        {
            txlist->first_to_download[0] = tx_desc_next(agg_desc->txdesc_last);
        }
    }

    // Walk through the list of SW descriptors programmed in the HE TB, and free them all
    while (1)
    {
        //-----------------------
        // MPDU THD status check
        //-----------------------
        if (txlist->chk_state == THD_CHK_STATE)
        {
            struct tx_hd *txhd;
            uint32_t txstatus;

            // Sanity check - When handling HE TB we should never reach a point where
            // we don't have descriptors available anymore
            ASSERT_ERR(txdesc != NULL);

            txhd = &txdesc->lmac.hw_desc->thd;

            // Set the done bit manually but not the success bit, as if the frame would
            // have been transmitted but not acknowledged
            txhd->statinfo |= HE_TB_TX_BIT | DESC_DONE_TX_BIT;
            txstatus = txhd->statinfo;

            // Manage singleton case
            if (!is_mpdu_agg(txdesc))
            {
                txl_he_tb_mpdu_transmit_trigger(txdesc, txstatus);
                break;
            }

            // Otherwise manage A-MPDU sub-frame case
            txl_he_tb_ampdu_mpdu_transmit_trigger(txdesc, txstatus);

            // Get next txdesc in transmitting list
            txdesc = (struct txdesc *)co_list_pick(&(txlist->transmitting[0]));

        }
        //-----------------------
        // A-THD status check
        //-----------------------
        else
        {
            //get the last txdesc queued into cfm list for this ac, its agg_desc and then bar_desc
            struct tx_agg_desc *agg_desc = txlist->agg_desc;

            // For profiling
            PROF_AGG_BAR_DONETX_SET();

            // Mark aggregate as ready for confirmation, without any BA received
            agg_desc->status |= AGG_DONE;

            // Pop the aggregate descriptor from the list
            co_list_pop_front(&txlist->aggregates);

            // Set an event to handle the confirmation of the queued aggregated MPDU in background
            ke_evt_set(txl_cfm_evt_bit[access_category]);

            // Change state back to MPDUs check
            txlist->chk_state = THD_CHK_STATE;
            txlist->agg_desc = NULL;

            // For profiling
            PROF_AGG_BAR_DONETX_CLR();
            break;
        }
    }

    // Re-configure the HW AC queue with the next packet to transmit
    if (!txl_he_is_edca_off(access_category) && (mu_edca_param->first_frame_exch != NULL))
    {
        txl_cntrl_newhead(CPU2HW(mu_edca_param->first_frame_exch), access_category);
    }

    // Clear the TX activity timeout
    txl_he_tb_timer_clear();

    // We have one PPDU less in the TX path
    txlist->ppdu_cnt--;

    // Check if we need to close a pending A-MPDU
    txl_agg_check(access_category);

    PROF_TX_MAC_IRQ_CLR();
}

void txl_he_tb_transmit_trigger(void)
{
    uint8_t access_category = txl_he_env.trigger.ac;
    struct txl_list *txlist = &txl_cntrl_env.txlist[access_category];
    struct he_trigger_tag *he_tb = &txl_he_env.trigger;
    struct mu_edca_param_tag *mu_edca_param = &txl_he_env.mu_edca.params[access_category];

    PROF_TX_MAC_IRQ_SET();
    PROF_TX_AC_IRQ_SET(access_category);

    // Walk through the list of completed SW descriptors
    while (1)
    {
        struct txdesc * txdesc = NULL;

        //-----------------------
        // MPDU THD status check
        //-----------------------
        if (txlist->chk_state == THD_CHK_STATE)
        {
            struct tx_hd *txhd;
            uint32_t txstatus;

            //pick first txdesc in transmitting list to check its doneTx status
            txdesc = (struct txdesc *)co_list_pick(&(txlist->transmitting[0]));

            // Sanity check - When handling HE TB we should never reach a point where
            // we don't have descriptors available anymore
            ASSERT_ERR(txdesc != NULL);

            txhd = &txdesc->lmac.hw_desc->thd;

            // Check if HW handled this MPDU
            if (!(txhd->statinfo & DESC_DONE_TX_BIT))
            {
                #if !NX_FULLY_HOSTED && NX_AMSDU_TX
                // Maybe some A-MSDU subframes (i.e. TBD) are done, check that
                txl_check_done_amsdu_subframe(txdesc, access_category, 0);
                #endif
                // MPDU not yet transmitted, stop here and come back later
                PROF_TX_MAC_IRQ_CLR();
                return;
            }

            // Frame was sent in a HE TB PPDU
            txhd->statinfo |= HE_TB_TX_BIT;
            txstatus = txhd->statinfo;

            // Manage singleton case
            if (!is_mpdu_agg(txdesc))
            {
                txl_he_tb_mpdu_transmit_trigger(txdesc, txstatus);
                break;
            }

            // Otherwise manage A-MPDU sub-frame case
            txl_he_tb_ampdu_mpdu_transmit_trigger(txdesc, txstatus);
        }
        //-----------------------
        // A-THD/BAR THD status check
        //-----------------------
        else
        {
            //get the last txdesc queued into cfm list for this ac, its agg_desc and then bar_desc
            struct tx_agg_desc *agg_desc = txlist->agg_desc;
            struct tx_hd *thd;
            uint32_t thd_status;

            if (txlist->chk_state == ATHD_CHK_STATE)
                // Get the A-THD descriptor
                thd = &agg_desc->a_thd;
            else
                // Get the BAR THD descriptor
                thd = &agg_desc->bar_thd;

            thd_status = thd->statinfo;
            //BAR status is updated received
            if (!(thd_status & DESC_DONE_TX_BIT))
            {
                PROF_TX_MAC_IRQ_CLR();
                return;
            }

            // For profiling
            PROF_AGG_BAR_DONETX_SET();

            if (thd_status & FRAME_SUCCESSFUL_TX_BIT)
            {
                // Check if BA has already been received
                if (!(agg_desc->status & AGG_BA_RECEIVED))
                {
                    int i = 0;
                    // Make a call to the RX path to get the BA that is
                    // available, waiting for the RX timer to expire
                    do
                    {
                        rxl_immediate_frame_get();
                        i++;
                    } while ((i < 5) && (!(agg_desc->status & AGG_BA_RECEIVED)));
                }

                // Sanity check - The BA shall now be available
                ASSERT_REC(agg_desc->status & AGG_BA_RECEIVED);

                if (he_tb->uora)
                    // HE TB PPDU transmission is considered successful in the selected RA-RU
                    nxmac_e_ocw_setf(txl_he_env.uora.eocw_min);
                else
                    // Restart the MU EDCA for this access category
                    txl_he_mu_edca_start(txl_he_env.trigger.ac);
            }
            else
            {
                // Pop the aggregate descriptor from the list
                co_list_pop_front(&txlist->aggregates);

                if (he_tb->uora)
                {
                    // HE TB PPDU transmission in RA-RU is unsuccessful
                    uint8_t new_eocw = nxmac_e_ocw_getf();
                    new_eocw = new_eocw + 1;

                    if (new_eocw > txl_he_env.uora.eocw_max)
                        new_eocw = txl_he_env.uora.eocw_max;

                    nxmac_e_ocw_setf(new_eocw);
                }
            }

            // Mark aggregate as ready for confirmation
            agg_desc->status |= AGG_DONE;

            // Set an event to handle the confirmation of the queued aggregated MPDU in background
            ke_evt_set(txl_cfm_evt_bit[access_category]);

            //bar has been handled, BA reception status will be known at cfm,
            //change state back to MPDUs check
            txlist->chk_state = THD_CHK_STATE;
            txlist->agg_desc = NULL;

            // For profiling
            PROF_AGG_BAR_DONETX_CLR();
            break;
        }
    }

    // Re-configure the HW AC queue with the next packet to transmit
    if (!txl_he_is_edca_off(access_category) && (mu_edca_param->first_frame_exch != NULL))
    {
        txl_cntrl_newhead(CPU2HW(mu_edca_param->first_frame_exch), access_category);
    }

    // Clear the TX activity timeout
    txl_he_tb_timer_clear();

    // We have one PPDU less in the TX path
    txlist->ppdu_cnt--;

    // Check if we need to close a pending A-MPDU
    txl_agg_check(access_category);

    PROF_TX_MAC_IRQ_CLR();
}

void txl_he_init(void)
{
    memset(&txl_he_env, 0, sizeof(txl_he_env));

    txl_he_tb_desc_init();

    // By default, use a fixed value for UPH, until we get it from HW register
    txl_he_env.trigger.uph = MAC_HTC_HE(HTC_HE_UPH_DEF);

    #if NX_MAC_VER > 20
    // Limit the MCS that we can transmit over HE TB (MCS11 if LDPC is supported, MCS9 otherwise)
    if (phy_ldpc_tx_supported())
        nxmac_max_mcs_in_hetb_setf(11);
    else
        nxmac_max_mcs_in_hetb_setf(9);
    #endif
}

void txl_he_reset(void)
{
    struct mu_edca_tag mu_edca;
    struct vif_info_tag *tb_vif;

    #if NX_UMAC_PRESENT && NX_BEACONING
    struct sta_info_tag *trigged_sta;
    bool trig_started = (txl_he_env.trig_timer.cb != NULL);

    // Check if the HE trigger timer is started
    if (trig_started)
    {
        // Save the trigger timer env variable
        trigged_sta = txl_he_env.trig_timer.env;
        // Extract it from the list of timers
        mm_timer_clear(&txl_he_env.trig_timer);
        trig_started = true;
    }
    #endif // NX_UMAC_PRESENT && NX_BEACONING

    // Save the HE TB VIF pointer
    tb_vif = txl_he_env.tb_vif;

    // Save the MU EDCA parameters
    if (tb_vif)
    {
        mu_edca = txl_he_env.mu_edca;
        // Check if the MU EDCA operation is started
        if (mu_edca.timer.cb != NULL)
            // Extract it from the list of timers
            mm_timer_clear(&txl_he_env.mu_edca.timer);
    }

    // Re-initialize the full HE environment
    txl_he_init();

    // Restore the HE TB VIF pointer
    txl_he_env.tb_vif = tb_vif;

    // Restore the MU EDCA parameters
    if (tb_vif)
    {
        // Restore MU EDCA parameters
        txl_he_env.mu_edca.valid = mu_edca.valid;
        for (int i = 0; i < AC_MAX; i++)
        {
            txl_he_env.mu_edca.params[i].edca = mu_edca.params[i].edca;
            txl_he_env.mu_edca.params[i].timeout = mu_edca.params[i].timeout;
        }
    }

    #if NX_UMAC_PRESENT && NX_BEACONING
    // Check if the HE trigger timer has to be re-started
    if (trig_started)
    {
        // Restart it
        txl_he_start_trigger_scheduler(trigged_sta);
    }
    #endif // NX_UMAC_PRESENT && NX_BEACONING
}

bool txl_he_decode_m_ba(struct rxdesc *badesc, struct tx_agg_desc *agg_desc)
{
    struct sta_info_tag *sta = &sta_info_tab[agg_desc->sta_idx];
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(badesc);
    struct rx_hd *hdrdesc = &dma_hdrdesc->hd;
    struct rx_pbd *pd = HW2CPU(hdrdesc->first_pbd_ptr);
    struct ba_base *ba_payl = (struct ba_base *)HW2CPU(pd->datastartptr);
    struct ba_msta_aid_tid_info *aid_tid_info;
    int remaining_len = hdrdesc->frmlen;
    bool addr1_match = (hdrdesc->statinfo & RX_HD_ADDRMIS) == 0;
    bool addr1_group = (hdrdesc->statinfo & RX_HD_GA_FRAME) != 0;

    // Multi-STA BA can be unicast to our device or broadcast
    if (!addr1_match && !addr1_group)
        return false;

    aid_tid_info = (struct ba_msta_aid_tid_info *)(ba_payl + 1);
    remaining_len -= sizeof_b(struct ba_base);

    // Go through the AID/TID tuples
    while (remaining_len >= 2)
    {
        uint16_t aid = (aid_tid_info->aid_tid & BA_MULTI_STA_AID_MSK) >> BA_MULTI_STA_AID_OFT;
        int aid_tid_len = txl_he_m_ba_aid_tid_len(aid_tid_info, remaining_len);

        if (aid_tid_len < 0)
            return false;

        // Check if the current AID matches ours
        if (aid == sta->aid)
        {
            uint8_t tid;

            if (aid_tid_info->aid_tid & BA_MULTI_STA_ACK_BIT)
            {
                agg_desc->ssc_bitmap = NULL;
                return true;
            }

            tid = (aid_tid_info->aid_tid & BA_MULTI_STA_TID_MSK) >> BA_MULTI_STA_TID_OFT;

            // Check if both TID match
            if (tid != agg_desc->tid)
                // No match, we consider BA as invalid
                return false;

            memcpy(&agg_desc->ssc_bitmap_tab, &aid_tid_info->ssc_bitmap,
                   aid_tid_len - sizeof_b(aid_tid_info->aid_tid));
            agg_desc->ssc_bitmap = (struct ba_ssc_bitmap *)(&agg_desc->ssc_bitmap_tab);
            return true;
        }

        // Continue looping through the AID/TID tuples
        aid_tid_info = HW2CPU(CPU2HW(aid_tid_info) + aid_tid_len);
        remaining_len -= aid_tid_len;
    };

    // No match, we consider BA as invalid
    return false;
}

void txl_he_trigger_push(struct rxdesc *rxdesc)
{
    uint8_t sta_idx;
    uint16_t key_idx_hw;
    struct rx_dmadesc *dma_hdrdesc = rxl_dmadesc_get(rxdesc);
    struct rx_hd *hdrdesc = &dma_hdrdesc->hd;
    struct rx_pbd *pd = HW2CPU(hdrdesc->first_pbd_ptr);
    uint32_t statinfo = hdrdesc->statinfo;
    struct he_trigger_tag *he_tb = &txl_he_env.trigger;
    struct sta_info_tag *sta;
    struct vif_info_tag *vif;

    // By default we consider the trigger frame as invalid
    he_tb->trig_valid = false;

    // Sanity checks - the HW is supposed to analyze the trigger frame, so we consider
    // invalid frames as errors
    ASSERT_REC(hdrdesc->frmlen >= HE_TRIG_FRM_MIN_LEN);

    // Check if the sending STA is registered
    if (!(statinfo & KEY_IDX_VALID_BIT))
    {
        MAC_ADDR_CPY(&he_tb->ta, HW2CPU(pd->datastartptr + MAC_HEAD_ADDR2_OFT));
        return;
    }

    // Get the key index
    key_idx_hw = (uint16_t)((statinfo & KEY_IDX_MSK) >> KEY_IDX_OFT);

    // Sanity check
    ASSERT_REC(key_idx_hw >= MM_SEC_DEFAULT_KEY_COUNT);

    // Check if the peer STA is still active
    sta_idx = (uint8_t)(key_idx_hw - MM_SEC_DEFAULT_KEY_COUNT);
    sta = &sta_info_tab[sta_idx];
    if (sta->valid)
    {
        he_tb->ta = sta->mac_addr;
    }
    else
    {
        MAC_ADDR_CPY(&he_tb->ta, HW2CPU(pd->datastartptr + MAC_HEAD_ADDR2_OFT));
        return;
    }

    // Ensure we are aligned with the HE TB VIF
    vif = &vif_info_tab[sta->inst_nbr];
    if (vif != txl_he_env.tb_vif)
        return;

    // Ensure we will have enough time to program something
    if (nxmac_tx_hetb_rem_dur_getf() < TX_HE_TB_PROG_MIN_TIME)
        return;

    // Compute the HE TB parameters
    txl_he_tb_params_compute();

    // Is trigger frame valid ? Today the we only check if we have enough resources in
    // the HE TB to send at least a QoS-NULL frame
    he_tb->trig_valid = (he_tb->max_len >= TX_HE_TB_MIN_PSDU_LEN)?true:false;

    // Fill HE TB information structure
    he_tb->sta = sta;

    // Update RC about received trigger frame
    txl_he_tb_set_rc_info();
}

void txl_he_tb_prot_trigger(void)
{
    // For profiling
    PROF_TX_HE_TRIG_IRQ_SET();

    // Call the RX path to retrieve the trigger frame
    rxl_immediate_frame_get();

    // Send the HE TB PPDU based on the received trigger frame
    txl_he_tb_ppdu_send();

    // For profiling
    PROF_TX_HE_TRIG_IRQ_CLR();
}

void txl_he_tb_config(void)
{
    struct vif_info_tag *vif = vif_mgmt_get_single_sta_vif();

    // Check if HE is supported by the MAC HW and PHY
    if (hal_machw_he_support() && (vif != NULL))
    {
        // Save TB VIF pointer
        txl_he_env.tb_vif = vif;

        // Enable the HE TB operation
        txl_he_tb_enable();
    }
    else
    {
        // Disable the HE TB operation
        txl_he_tb_disable();

        // Clear TB VIF pointer
        txl_he_env.tb_vif = NULL;
    }
}

void txl_he_tb_enable(void)
{
    struct tx_policy_tbl *pol;
    uint8_t hw_key_idx;

    if (txl_he_env.tb_vif == NULL || !txl_he_env.tb_vif->active)
        return;

    // Compute MAC control information fields of the HE TB policy table
    pol = &he_tb_desc.tb_pol;
    hw_key_idx = MM_STA_TO_KEY(txl_he_env.tb_vif->u.sta.ap_id);
    pol->maccntrlinfo1 = (hw_key_idx << KEYSRAM_INDEX_RA_OFT) | hw_key_idx;
    pol->maccntrlinfo2 = 0xFFFF0704;

    // Enable TB protocol interrupts
    nxmac_enable_tb_prot_trigger_setf(1);
    nxmac_enable_tb_tx_trigger_setf(1);
    nxmac_enable_tb_tx_cancelled_setf(1);
    #if NX_AMSDU_TX
    nxmac_enable_tb_tx_buf_trigger_setf(1);
    #endif
    nxmac_accept_trigger_sw_frames_setf(1);
    nxmac_ra_ru_enable_setf(1);
    nxmac_ra_ru_type_setf(0);

}

void txl_he_tb_disable(void)
{
    if (txl_he_env.tb_vif != NULL)
        // Disable the MU EDCA operation
        txl_he_mu_edca_stop();

    // Disable TB protocol interrupts
    nxmac_accept_trigger_sw_frames_setf(0);
    nxmac_enable_tb_prot_trigger_setf(0);
    nxmac_enable_tb_tx_trigger_setf(0);
    nxmac_enable_tb_tx_cancelled_setf(0);
    #if NX_AMSDU_TX
    nxmac_enable_tb_tx_buf_trigger_setf(0);
    #endif
    nxmac_ra_ru_enable_setf(0);
}

#if NX_UMAC_PRESENT
uint32_t txl_he_htc_get(struct txdesc *txdesc, struct sta_info_tag *sta)
{
    struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;

    // Check if we are sending the frame in a HE TB PPDU or not
    if (agg_desc && ((agg_desc->status & AGG_TB) != 0))
        // HE TB PPDU - We shall put the UPH control field
        return txl_he_env.trigger.uph;

    // HE SU case - We shall put the BSR control field
    return sta->bsr;
}
#endif

uint32_t txl_he_idx_to_32us_len_get(int base_idx, uint8_t bw)
{
    int idx = base_idx + 3 * (bw >> 1);
    int shift = bw & 0x01;
    return (((uint32_t)TX_HE_RATE_TO_32US_LEN[idx]) << shift);
}

uint16_t txl_he_idx_to_1us_len_get(int base_idx, uint8_t bw)
{
    int idx = base_idx + 3 * (bw >> 1);
    int shift = bw & 0x01;
    return (((uint32_t)TX_HE_RATE_TO_MIN_SEP[idx]) << shift);
}

int txl_he_ampdu_param_get(struct txdesc *txdesc,
                           uint8_t format_mod,
                           uint32_t *max_len_sta,
                           uint32_t *nss)
{
    uint32_t phy_flags = txdesc->umac.phy_flags;
    uint32_t mcs_idx = (phy_flags & MCS_INDEX_TX_RCX_MASK) >> MCS_INDEX_TX_RCX_OFT;
    uint32_t gi = (phy_flags & HE_GI_TYPE_TX_RCX_MASK) >> HE_GI_TYPE_TX_RCX_OFT;
    struct sta_info_tag *sta = &sta_info_tab[txdesc->host.staid];

    // Only HE SU for now
    ASSERT_ERR(format_mod == FORMATMOD_HE_SU);

    // Extract NSS
    *nss = ((mcs_idx & VHT_NSS_MASK) >> VHT_NSS_OFT) + 1;

    // Extract MCS index
    mcs_idx = (mcs_idx & VHT_MCS_MASK) >> VHT_MCS_OFT;

    // Sanity checks: Maximum 4 SS supported, and MCS shall be between 0 and 11
    ASSERT_ERR((*nss > 0) && (*nss <= 4));
    ASSERT_ERR(mcs_idx <= 11);

    // Get the maximum HE A-MPDU size supported by the receiver
    *max_len_sta = sta->ampdu_size_max_he;

    // To compute the maximum length we take the minimal bandwidth, to ensure that the
    // packet won't cross the L-SIG duration limit in case the MAC HW would have to
    // reduce the transmission bandwidth
    return (mcs_idx * 6 + gi);
}

void txl_he_mu_edca_param_set(struct mm_set_mu_edca_req const *param)
{
    struct mu_edca_tag *mu_edca = &txl_he_env.mu_edca;

    // Go through the different access categories
    for (int i = 0; i < AC_MAX; i++)
    {
        uint8_t aifsn = param->param[i] & 0x0000000F;
        uint8_t timeout = (param->param[i] & 0x00FF0000) >> 16;
        uint16_t cw = (param->param[i] & 0x0000FF00) >> 4;

        mu_edca->params[i].timeout = timeout;
        mu_edca->params[i].edca = cw | aifsn;
        // Check if a HE TB is enabled
        if (txl_he_env.tb_vif)
        {
            // Keep TXOP as advertised in WMM parameter element
            mu_edca->params[i].edca |= txl_he_env.tb_vif->txq_params[i] & 0xFFFFF000;
        }
    }

    // MU EDCA parameters are now valid
    mu_edca->valid = true;
}

void txl_he_uora_param_set(struct mm_set_uora_req const *param)
{
    struct uora_tag *uora = &txl_he_env.uora;

    uora->eocw_min = param->eocw_min;
    uora->eocw_max = param->eocw_max;
}

void txl_he_mu_edca_blocked(struct tx_hd *thd, uint8_t ac)
{
    struct mu_edca_tag *mu_edca = &txl_he_env.mu_edca;
    struct mu_edca_param_tag *mu_edca_param = &txl_he_env.mu_edca.params[ac];

    // Check if we have valid MU EDCA parameters available
    if (mu_edca->timer.cb)
    {
        // If EDCA is off remind the head of the EDCA queue
        if (txl_he_is_edca_off(ac))
            mu_edca_param->first_frame_exch = thd;

        // Decrease MU EDCA timer so that EDCA is re-started on next tick
        mu_edca_param->timer = 1;
    }
}

#if NX_UMAC_PRESENT && NX_BEACONING
void txl_he_start_trigger_scheduler(struct sta_info_tag *sta)
{
    txl_he_env.trig_timer.cb = txl_he_trig_time;
    txl_he_env.trig_timer.env = sta;
    mm_timer_set(&txl_he_env.trig_timer, hal_machw_time() + TX_HE_TRIG_PERIOD);
}
#endif // NX_UMAC_PRESENT && NX_BEACONING

void txl_he_txop_dur_rtscts_thres_ampdu_check(struct txdesc *txdesc)
{
    struct vif_info_tag *vif = &vif_info_tab[txdesc->host.vif_idx];
    struct tx_agg_desc *agg_desc = txdesc->lmac.agg_desc;

    if (vif->txop_dur_rts_thres != MAC_HE_OPER_TXOP_DUR_RTS_THRES_DISABLED)
    {
        txl_he_txop_dur_rtscts_thres_apply(&txdesc->umac.phy_flags, agg_desc->a_thd.frmlen,
                                           vif->txop_dur_rts_thres);
    }
}

void txl_he_txop_dur_rtscts_thres_mpdu_check(struct txdesc *txdesc)
{
    struct vif_info_tag *vif = &vif_info_tab[txdesc->host.vif_idx];
    struct txl_buffer_control *bufctrl = txl_buffer_control_get(txdesc);
    struct tx_hd *txhd = &txdesc->lmac.hw_desc->thd;
    struct tx_policy_tbl *pt = &bufctrl->policy_tbl;

    if (vif->txop_dur_rts_thres != MAC_HE_OPER_TXOP_DUR_RTS_THRES_DISABLED)
    {
        for (int i = 0; i < RATE_CONTROL_STEPS; i++)
        {
            txl_he_txop_dur_rtscts_thres_apply(&pt->ratecntrlinfo[i], txhd->frmlen,
                                               vif->txop_dur_rts_thres);
        }
    }
}

#endif // NX_MAC_HE

/// @}  // end of group TX_HE
