/**
 ****************************************************************************************
 *
 * @file vif_mgmt.c
 *
 * @brief Virtual Interface Management implementation.
 *
 * Copyright (C) RivieraWaves 2011-2019
 *
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @addtogroup VIF_MGMT
 * @{
 ****************************************************************************************
 */

/*
 * INCLUDE FILES
 ****************************************************************************************
 */

#include "vif_mgmt.h"
#include "mm.h"
#include "chan.h"
#include "mm_bcn.h"
#include "co_utils.h"
#include "co_status.h"
#include "dbg.h"

#if (NX_CHNL_CTXT || NX_P2P)
#include "txl_cntrl.h"
#if (NX_P2P)
// P2P Definitions and Functions
#include "p2p.h"
#endif //(NX_P2P)
#if (NX_POWERSAVE)
#include "ps.h"
#endif //(NX_POWERSAVE)
#endif //(NX_CHNL_CTXT || NX_P2P)

#if (NX_TD)
// Traffic Detection Functions
#include "td.h"
#endif //(NX_TD)

#if (RW_UMESH_EN)
#include "mesh.h"
#endif //(RW_UMESH_EN)

#if NX_UMAC_PRESENT
#include "me_utils.h"
#include "tpc.h"
#endif

/*
 * GLOBAL VARIABLES
 ****************************************************************************************
 */
struct vif_mgmt_env_tag vif_mgmt_env;
struct vif_info_tag vif_info_tab[NX_VIRT_DEV_MAX];

/*
 * FUNCTION DEFINITIONS
 ****************************************************************************************
 */

#if (NX_P2P || NX_CHNL_CTXT)
/**
 ****************************************************************************************
 * @brief Handles the action to be performed just before beacon reception
 *
 * @param[in] env   Pointer to the VIF environment variables
 ****************************************************************************************
 */
static void vif_mgmt_bcn_to_evt(void *env)
{
    // Retrieve the VIF
    struct vif_info_tag *vif = (struct vif_info_tag *)env;

    #if (NX_P2P)
    // Informed interested modules
    if (vif->p2p)
    {
        p2p_bcn_evt_handle(vif);
    }
    #endif //(NX_P2P)

    if (vif->chan_ctxt)
    {
        chan_bcn_to_evt(vif);
    }
}
#endif //(NX_P2P || NX_CHNL_CTXT)

/**
 ****************************************************************************************
 * @brief Initializes (resets) the content of a VIF entry
 *
 * @param[in] vif       A pointer to the VIF entry to reset
 ****************************************************************************************
 */
static void vif_mgmt_entry_init(struct vif_info_tag *vif)
{
    // Reset table
    memset(vif, 0, sizeof(*vif));

    vif->type = VIF_UNKNOWN;
    vif->tx_power = VIF_UNDEF_POWER;
    #if NX_UMAC_PRESENT
    vif->user_tx_power = VIF_UNDEF_POWER;
    #endif

    #if (NX_P2P || NX_CHNL_CTXT)
    // Initialize callback and environment for Beacon TimeOut timer
    vif->tmr_bcn_to.cb  = vif_mgmt_bcn_to_evt;
    vif->tmr_bcn_to.env = vif;
    #endif //(NX_P2P || NX_CHNL_CTXT)
}

void vif_mgmt_init(void)
{
    int i;

    // Reset VIF management environment
    memset(&vif_mgmt_env, 0, sizeof(vif_mgmt_env));

    // Initialize VIF lists
    co_list_init(&vif_mgmt_env.free_list);
    co_list_init(&vif_mgmt_env.used_list);

    // push all the entries to the free list
    for(i = 0; i < NX_VIRT_DEV_MAX ; i++)
    {
        struct vif_info_tag *vif = &vif_info_tab[i];
        // Init STA info table.
        vif_mgmt_entry_init(vif);
        // Push to free list.
        co_list_push_back(&vif_mgmt_env.free_list, (struct co_list_hdr*)vif);
    }

    // Initialize monitor VIF to invalid
    vif_mgmt_env.monitor_vif = INVALID_VIF_IDX;
}

#if (NX_TX_FRAME)
void vif_mgmt_reset(void)
{
    struct vif_info_tag *vif = (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

    while (vif)
    {
        // Try to push the frame marked as postponed
        vif_mgmt_send_postponed_frame(vif);

        vif = (struct vif_info_tag *)vif->list_hdr.next;
    }
}
#endif //(NX_TX_FRAME)

uint8_t vif_mgmt_register(struct mac_addr const *mac_addr, uint8_t vif_type, bool p2p,
                          uint8_t *vif_idx)
{
    struct vif_info_tag *vif;

    // Check if there are still some free VIFs
    if (co_list_is_empty(&vif_mgmt_env.free_list))
        // All possible VIFs are active. Reject the new VIF.
        return CO_FAIL;

    #if (NX_P2P)
    if (p2p && (vif_type == VIF_AP))
    {
        #if (NX_P2P_GO)
        if (p2p_env.nb_p2p_go)
            // Only one P2P GO VIF is supported
            return CO_FAIL;
        #else
        // Do not create a P2P GO VIF if not supported
        return CO_FAIL;
        #endif //(NX_P2P_GO)
    }
    #endif //(NX_P2P)

    // Check if an interface is already defined or not
    if (co_list_is_empty(&vif_mgmt_env.used_list))
    {
        #if NX_MULTI_ROLE
        // Configure the HW properly for the first interface added
        mm_hw_info_set(mac_addr);
        #else
        #if NX_BEACONING && (NX_POWERSAVE || NX_CONNECTION_MONITOR || NX_UMAC_PRESENT)
        // Depending on the interface type, check if the beaconing has to be enabled
        if ((vif_type == VIF_AP) || (vif_type == VIF_IBSS))
        {
            mm_env.beaconing = true;
        }
        else
        {
            mm_env.beaconing = false;
        }
        #endif
        // Configure the HW properly
        mm_hw_interface_info_set(vif_type, mac_addr);
        #endif
    }
    else
    {
        uint32_t vif_addr_low, vif_addr_high;

        #if (!NX_MULTI_ROLE)
        struct vif_info_tag *vif = (struct vif_info_tag *)
            co_list_pick(&vif_mgmt_env.used_list);

        // We can only have multiple APs. In all other cases, we reject the new VIF
        if ((vif->type != VIF_AP) || (vif_type != VIF_AP))
            return CO_FAIL;
        #endif //(!NX_MULTI_ROLE)

        #if (NX_P2P)
        if (p2p)
        {
            // Check if a new P2P VIF can be created
            if (vif_mgmt_env.nb_p2p_vifs == NX_P2P_VIF_MAX)
                return CO_FAIL;
        }
        #endif //(NX_P2P)

        // Check if VIF MAC address matches with the mask - i.e. only MSB might differ
        vif_addr_low = mac_addr->array[0] | (((uint32_t)mac_addr->array[1]) << 16);
        vif_addr_high = mac_addr->array[2];
        if ((vif_addr_low != nxmac_mac_addr_low_get()) ||
            (((vif_addr_high ^ nxmac_mac_addr_hi_get()) & ~nxmac_mac_addr_hi_mask_get()) != 0))
            return CO_FAIL;

        #if (NX_MULTI_ROLE)
        // More than one entity will now be available, so disable HW filtering on BSSID
        mm_rx_filter_lmac_enable_set(NXMAC_ACCEPT_OTHER_BSSID_BIT);
        #endif //(NX_MULTI_ROLE)
    }

    // get a free element from the free VIF list
    vif = (struct vif_info_tag*)co_list_pop_front(&vif_mgmt_env.free_list);

    // Initialize some fields from the parameters
    vif->type = vif_type;
    vif->mac_addr = *mac_addr;
    vif->index = CO_GET_INDEX(vif, vif_info_tab);
    vif->txq_params[AC_BK] = NXMAC_EDCA_AC_0_RESET;
    vif->txq_params[AC_BE] = NXMAC_EDCA_AC_1_RESET;
    vif->txq_params[AC_VI] = NXMAC_EDCA_AC_2_RESET;
    vif->txq_params[AC_VO] = NXMAC_EDCA_AC_3_RESET;
    vif->tx_power = VIF_UNDEF_POWER;
    #if NX_UMAC_PRESENT
    vif->user_tx_power = VIF_UNDEF_POWER;
    #endif
    #if (NX_CHNL_CTXT)
    vif->chan_ctxt = NULL;
    vif->tbtt_switch.vif_index = vif->index;
    #endif //(NX_CHNL_CTXT)
    #if NX_MAC_HE
    vif->txop_dur_rts_thres = MAC_HE_OPER_TXOP_DUR_RTS_THRES_DISABLED;
    #endif
    #if (NX_P2P)
    vif->p2p = p2p;
    vif->u.sta.sp_paused = false;
    #endif //(NX_P2P)

    #if (NX_MULTI_ROLE || NX_BCN_AUTONOMOUS_TX || NX_REORD)
    switch (vif_type)
    {
        case (VIF_STA):
        {
            #if NX_MULTI_ROLE
            // One more STA VIF
            vif_mgmt_env.vif_sta_cnt++;

            // Initialize TBTT timer
            vif->tbtt_timer.cb = mm_sta_tbtt;
            vif->tbtt_timer.env = vif;
            #if !NX_UMAC_PRESENT
            vif->u.sta.ap_bcn_intv = 100;
            #endif
            #endif

            vif->u.sta.ap_id = INVALID_STA_IDX;

            #if NX_UMAC_PRESENT
            vif->u.sta.csa_count = 0;
            vif->u.sta.csa_occured = false;
            #endif
        } break;

        #if (RW_MESH_EN)
        case (VIF_MESH_POINT):
        {
            #if (RW_UMESH_EN)
            vif->mvif_idx = MESH_INVALID_MESH_IDX;
            #endif //(RW_UMESH_EN)

            vif_mgmt_env.vif_mp_cnt++;

            // Disable filtering on other BSSID as 4 adresses are used for mesh QoS data
            mm_rx_filter_lmac_enable_set(NXMAC_ACCEPT_OTHER_BSSID_BIT);
        } // no break;
        #endif //(RW_MESH_EN)

        case (VIF_AP):
        {
            #if NX_MULTI_ROLE
            // Check if this is the first AP VIF created
            if (!vif_mgmt_env.vif_ap_cnt)
                mm_hw_ap_info_set();

            // One more AP VIF
            vif_mgmt_env.vif_ap_cnt++;
            #endif

            #if (!NX_UMAC_PRESENT && NX_BCN_AUTONOMOUS_TX)
            // Set a default beacon interval value
            vif->u.ap.bcn_int = 100;
            #endif //(!NX_UMAC_PRESENT)

            #if (NX_P2P_GO && NX_POWERSAVE)
            if (vif->p2p)
            {
                // Initialize TBTT timer
                vif->tbtt_timer.cb = mm_ap_pre_tbtt;
                vif->tbtt_timer.env = vif;
            }
            #endif //(NX_P2P_GO && NX_POWERSAVE)

            #if (NX_BCN_AUTONOMOUS_TX)
            mm_bcn_init_vif(vif);
            #endif
        } break;

        case (VIF_MONITOR):
        {
            if (vif_mgmt_env.monitor_vif != INVALID_VIF_IDX)
                return CO_FAIL;

            // Assign monitor vif
            vif_mgmt_env.monitor_vif = vif->index;

            #if (NX_UMAC_PRESENT)
            // Check if first interface
            if (co_list_is_empty(&vif_mgmt_env.used_list))
            {
                struct mac_chan_op chan;
                uint8_t chan_idx;

                // Use default channel config
                chan.band         = PHY_BAND_2G4;
                chan.type         = PHY_CHNL_BW_20;
                chan.prim20_freq  = 2437;
                chan.center1_freq = 2437;
                chan.center2_freq = 0;

                // Set MAC HW to monitor mode
                hal_machw_monitor_mode();

                // Add channel context
                if (chan_ctxt_add(&chan, &chan_idx) != CO_OK)
                {
                    return CO_FAIL;
                }
                #if NX_POWERSAVE
                // Disallow deep sleep while monitor mode is active
                vif->prevent_sleep |= PS_VIF_MONITOR;
                #endif
                // Link it to the monitor interface
                chan_ctxt_link_monitor(chan_idx);
            }
            else
                #if NX_MON_DATA
            {
                // Get first used interface
                struct vif_info_tag *vif = (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

                // Set RX filter
                mm_rx_filter_umac_set(MM_RX_FILTER_MONITOR);

                // Link monitor VIF with used interface context (if valid)
                if (vif->chan_ctxt)
                {
                    chan_ctxt_link_monitor(vif->chan_ctxt->idx);
                }
            }
            #else
            {
                // Set monitor vif to invalid
                vif_mgmt_env.monitor_vif = INVALID_STA_IDX;

                return CO_FAIL;
            }
            #endif
            #endif /* NX_UMAC_PRESENT */
        } break;

        default:
            break;
    }
    #endif //(NX_MULTI_ROLE || NX_BCN_AUTONOMOUS_TX || NX_REORD)

    #if (NX_P2P)
    if (p2p)
    {
        // Update number of registered P2P VIFs
        vif_mgmt_env.nb_p2p_vifs++;

        // Check if a new P2P interface can be created
        vif->p2p_index = p2p_create(vif->index,
                                    (vif_type == VIF_STA) ? P2P_ROLE_CLIENT : P2P_ROLE_GO);

        // Sanity Check
        ASSERT_ERR(vif->p2p_index != P2P_INVALID_IDX);
    }
    #endif //(NX_P2P)

    #if (NX_TD)
    if (vif_type != VIF_MONITOR)
        // Start Traffic Detection on this VIF
        td_start(vif->index);
    #endif //(NX_TD)

    // Get the station index
    *vif_idx = vif->index;

    // Push the VIF entry to the used list
    co_list_push_back(&vif_mgmt_env.used_list, &vif->list_hdr);

    #if NX_MAC_HE
    // Enable/disable the HE TB operation
    txl_he_tb_config();
    #endif

    // Set the BSSID mask according to the registered VIFs
    vif_mgmt_set_bssid_mask();

    return CO_OK;
}

void vif_mgmt_unregister(uint8_t vif_idx)
{
    struct vif_info_tag *vif = &vif_info_tab[vif_idx];

    // Extract the VIF entry from the used list
    co_list_extract(&vif_mgmt_env.used_list, &vif->list_hdr);

    #if NX_MULTI_ROLE
    // Check if it was a STA or AP VIF
    switch (vif->type)
    {
        case (VIF_STA):
        {
            // One less STA VIF
            vif_mgmt_env.vif_sta_cnt--;
        } break;

        #if (RW_MESH_EN)
        case (VIF_MESH_POINT):
        {
            vif_mgmt_env.vif_mp_cnt--;
        } //no break;
        #endif //(RW_MESH_EN)
        case (VIF_AP):
        {
            // One less AP VIF
            vif_mgmt_env.vif_ap_cnt--;

            // Check if there are still some APs enabled
            if (!vif_mgmt_env.vif_ap_cnt)
                mm_hw_ap_info_reset();
        } break;
        case (VIF_MONITOR):
        {
            #if NX_UF_EN
            //Configure unsupported frame handling
            if (phy_uf_supported())
            {
                //Disable Unsupported HT Frame logging
                phy_uf_enable(false);
            }
            #endif /* NX_UF_EN */

            #if (NX_UMAC_PRESENT)
            {
                struct vif_info_tag *vif = &vif_info_tab[vif_mgmt_env.monitor_vif];

                #if NX_POWERSAVE
                // Allow again deep sleep now that monitor mode is inactive
                vif->prevent_sleep &= ~PS_VIF_MONITOR;
                #endif
                //Check if monitor channel is already linked
                if (vif->chan_ctxt)
                {
                    // Unlink monitor channel
                    chan_ctxt_unlink(vif_mgmt_env.monitor_vif);
                }

                if (vif_mgmt_env.vif_ap_cnt)
                {
                    mm_hw_ap_info_set();
                }
                else
                {
                    mm_rx_filter_umac_set(MM_RX_FILTER_ACTIVE);
                }
            }
            #endif /* NX_UMAC_PRESENT */

            // Reset monitor vif
            vif_mgmt_env.monitor_vif = INVALID_STA_IDX;
        } break;

        default:
        {

        } break;
    }

    // Check if there is only one entity again
    if ((vif_mgmt_used_cnt() == 1)
        #if (RW_MESH_EN)
            && (!vif_mgmt_env.vif_mp_cnt)
        #endif //(RW_MESH_EN)
            )
    {
        struct vif_info_tag *vif = (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

        // More than one entity will now be available, so reenable HW filtering on BSSID
        mm_rx_filter_lmac_enable_clear(NXMAC_ACCEPT_OTHER_BSSID_BIT);
        // Only one VIF used, so put the BSSID in the HW
        // write lower 4 bytes of BSSID
        nxmac_bss_id_low_setf(vif->bssid.array[0] | (((uint32_t)vif->bssid.array[1]) << 16));

        // write higher 2 bytes of BSSID
        nxmac_bss_id_high_setf(vif->bssid.array[2]);
    }
    #endif

    #if (NX_MULTI_ROLE || NX_CHNL_CTXT || NX_P2P_GO)
    // Clear TBTT Timer
    mm_timer_clear(&vif->tbtt_timer);
    #endif //(NX_MULTI_ROLE || NX_CHNL_CTXT || NX_P2P_GO)

    #if (NX_CHNL_CTXT || NX_P2P)
    // Clear Beacon Timeout Timer
    mm_timer_clear(&vif->tmr_bcn_to);
    #endif //(NX_CHNL_CTXT || NX_P2P)

    #if (NX_P2P)
    if (vif->p2p)
    {
        p2p_cancel(vif->p2p_index, true);

        #if (NX_CHNL_CTXT)
        // Update number of registered P2P VIFs
        vif_mgmt_env.nb_p2p_vifs--;
        #endif //(NX_CHNL_CTXT)
    }
    #endif //(NX_P2P)

    #if (NX_TD)
    // Stop Traffic Detection on this VIF
    td_reset(vif->index);
    #endif //(NX_TD)

    #if NX_MAC_HE
    // Enable/disable the HE TB operation
    txl_he_tb_config();
    #endif

    // Set the BSSID mask according to the registered VIFs
    vif_mgmt_set_bssid_mask();

    // Fully reset the entry
    vif_mgmt_entry_init(vif);

    // Push back the station entry in the free list
    co_list_push_back(&vif_mgmt_env.free_list, (struct co_list_hdr*)vif);
}

#if (NX_UMAC_PRESENT)
void vif_mgmt_add_key(struct mm_key_add_req const *param, uint8_t hw_key_idx)
{
    struct vif_info_tag *vif = &vif_info_tab[param->inst_nbr];
    struct key_info_tag *key = &vif->key_info[param->key_idx];

    // Store the key information
    key->hw_key_idx = hw_key_idx;
    key->cipher = param->cipher_suite;
    key->key_idx = param->key_idx;

    // Reset the replay counters
    memset(key->rx_pn, 0, TID_MAX * sizeof(uint64_t));

    // Check which encryption type has to be used
    switch(key->cipher)
    {
        case MAC_CIPHER_WEP40:
        case MAC_CIPHER_WEP104:
            key->tx_pn = co_rand_word() & 0xFFFFFF;
            break;
        case MAC_CIPHER_TKIP:
            key->tx_pn = 0;
            key->u.mic.tx_key[0] = param->key.array[4];
            key->u.mic.tx_key[1] = param->key.array[5];
            key->u.mic.rx_key[0] = param->key.array[6];
            key->u.mic.rx_key[1] = param->key.array[7];
            break;
        #if RW_WAPI_EN
        case MAC_CIPHER_WPI_SMS4:
            key->tx_pn = 0x5c365c365c365c36ULL;
            break;
        #endif
        #if NX_MFP
        case MAC_CIPHER_BIP_CMAC_128:
            memcpy(key->u.mfp.key, param->key.array, sizeof(key->u.mfp.key));
            key->tx_pn = 0;
            break;
        #endif
        default:
            key->tx_pn = 0;
            break;
    }

    // Key is now valid
    key->valid = true;

    #if NX_MFP
    if (key->cipher == MAC_CIPHER_BIP_CMAC_128)
        vif->default_mgmt_key = key;
    else
    #endif
    // This key is now the one used by default
    vif->default_key = key;
}

void vif_mgmt_del_key(struct vif_info_tag *vif, uint8_t keyid)
{
    int i;
    struct key_info_tag *key = &vif->key_info[keyid];

    // Key is now invalid
    key->valid = false;

    // Check if we still have a valid default key
    if (vif->default_key == key)
    {
        vif->default_key = NULL;
        for (i = 0; i < MAC_DEFAULT_KEY_COUNT; i++)
        {
            key = &vif->key_info[i];
            if (key->valid)
            {
                vif->default_key = key;
                return;
            }
        }
    }
    #if NX_MFP
    if (vif->default_mgmt_key == key)
    {
        vif->default_mgmt_key = NULL;
        for (i = MAC_DEFAULT_KEY_COUNT; i < MAC_DEFAULT_MFP_KEY_COUNT; i++)
        {
            key = &vif->key_info[i];
            if (key->valid)
            {
                vif->default_mgmt_key = key;
                return;
            }
        }
    }
    #endif
}

uint8_t vif_mgmt_get_staid(const struct vif_info_tag *vif, const struct mac_addr *sta_addr)
{
    struct co_list_hdr *list_hdr = co_list_pick(&vif->sta_list);

    // TODO: Using MACHW to retrieve KeyRam Idx from MAC addr may be faster
    while (list_hdr != NULL)
    {
        struct sta_info_tag *sta = (struct sta_info_tag *)list_hdr;
        if (MAC_ADDR_CMP(&sta->mac_addr, sta_addr))
            return sta->staid;
        list_hdr = co_list_next(list_hdr);
    }
    return INVALID_STA_IDX;
}
#endif

#if (NX_TX_FRAME)
void vif_mgmt_send_postponed_frame(struct vif_info_tag *vif)
{
    // STA Information entry
    struct sta_info_tag *sta;

    #if (NX_UMAC_PRESENT || NX_TD_STA)
    // Get first STA associated with peer
    sta = (struct sta_info_tag *)co_list_pick(&vif->sta_list);

    while (sta != NULL)
    {
        sta_mgmt_send_postponed_frame(vif, sta, 0);

        // Get next STA entry
        sta = (struct sta_info_tag *)sta->list_hdr.next;
    }
    #else
    // Counter
    uint8_t cnt;

    // Go through list of STA
    for (cnt = 0; cnt < STA_MAX; cnt++)
    {
        sta = &sta_info_tab[cnt];

        // Check if STA in linked with provided VIF
        if (sta->inst_nbr != vif->index)
        {
            // Go to next STA
            continue;
        }

        sta_mgmt_send_postponed_frame(vif, sta, 0);
    }
    #endif //(NX_UMAC_PRESENT || NX_TD_STA)
}
#endif //(NX_TX_FRAME)

#if (NX_CHNL_CTXT || NX_P2P)
void vif_mgmt_bcn_to_prog(struct vif_info_tag *vif)
{
    mm_timer_set(&vif->tmr_bcn_to, ke_time() + VIF_MGMT_BCN_TO_DUR);
}

void vif_mgmt_bcn_recv(struct vif_info_tag *vif)
{
    do
    {
        #if (NX_POWERSAVE)
        // If PS not used, wait for timeout
        if (!ps_env.ps_on)
        {
            break;
        }

        // Check PS is paused due to traffic, wait for timeout
        if (ps_env.prevent_sleep & PS_PSM_PAUSED)
        {
            break;
        }

        // Check if we are waiting for data
        if (vif->prevent_sleep)
        {
            break;
        }
        #endif //(NX_POWERSAVE)

        // Stop Timeout Timer
        mm_timer_clear(&vif->tmr_bcn_to);

        // End Timeout period
        vif_mgmt_bcn_to_evt((void *)vif);
    } while (0);
}
#endif //(NX_CHNL_CTXT || NX_P2P)

void vif_mgmt_set_ap_bcn_int(struct vif_info_tag *vif, uint16_t bcn_int)
{
    // Beacon Interval to be used for the VIF
    uint16_t set_bcn_int = bcn_int;

    #if NX_BCN_AUTONOMOUS_TX
    // Store the Beacon Interval
    vif->u.ap.bcn_int = bcn_int;
    #endif

    #if (NX_UMAC_PRESENT && NX_BEACONING)
    // Protect from AP TBTT interrupt
    GLOBAL_INT_DISABLE();

    // Check if first AP/MP interface
    if (vif_mgmt_env.vif_ap_cnt > 1)
    {
        struct vif_info_tag *rd_vif;
        // Cache lowest Beacon Interval currently in use
        uint16_t lowest_bcn_int = vif_info_tab[vif_mgmt_env.low_bcn_int_idx].u.ap.bcn_int;

        if (bcn_int < lowest_bcn_int)
        {
            // Provided VIF is now the one with the lowest beacon interval
            vif_mgmt_env.low_bcn_int_idx = vif->index;
        }
        else
        {
            set_bcn_int = lowest_bcn_int;
        }

        // Loop over the AP/MP VIFs
        rd_vif = (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

        while (rd_vif)
        {
            // Get number of time we see a TBTT between two beacon transmissions
            uint8_t ratio = (uint8_t)(rd_vif->u.ap.bcn_int / set_bcn_int);

            // Store the ratio
            rd_vif->u.ap.bcn_tbtt_ratio = ratio;
            // Initialize counter to our next TBTT
            rd_vif->u.ap.bcn_tbtt_cnt = 1;

            // Get next VIF
            rd_vif = (struct vif_info_tag *)(rd_vif->list_hdr.next);
        }
    }
    else
    {
        // Store the VIF index
        vif_mgmt_env.low_bcn_int_idx = vif->index;
        // And initialize beacon transmission values
        vif->u.ap.bcn_tbtt_ratio = 1;
        vif->u.ap.bcn_tbtt_cnt = 1;
    }
    #endif //(NX_UMAC_PRESENT)

    // If we are here update the beacon interval in the HW
    nxmac_beacon_int_setf(set_bcn_int);

    #if (NX_UMAC_PRESENT && NX_BEACONING)
    // Restore interrupts
    GLOBAL_INT_RESTORE();
    #endif //(NX_UMAC_PRESENT)
}

#if NX_UMAC_PRESENT
void vif_mgmt_switch_channel(struct vif_info_tag *vif)
{
    struct mm_csa_finish_ind *ind = KE_MSG_ALLOC(MM_CSA_FINISH_IND, TASK_API,
                                                 TASK_MM, mm_csa_finish_ind);
    struct me_bss_info *bss = &vif->bss_info;
    struct mac_chan_op *dest = &vif->csa_channel;
    uint8_t chan_idx = 0xFF;
    uint8_t res;

    // Unlink the VIF from its current channel context
    chan_ctxt_unlink(vif->index);

    // Update bss info regarding channel and TX power
    bss->chan = *dest;
    bss->power_constraint = 0;
    tpc_update_vif_tx_power(vif);

    // Create a channel context for new channel
    res = chan_ctxt_add(dest, &chan_idx);
    ind->status = res;
    ind->chan_idx = chan_idx;

    if (vif->type == VIF_STA)
    {
        vif->u.sta.csa_count = 0;

        if (res == CO_OK)
        {
            struct sta_info_tag *sta = &sta_info_tab[vif->u.sta.ap_id];

            chan_ctxt_link(vif->index, chan_idx);

            // reset beacon related timer and counter. Traffic will be re-enabled
            // once a beacon is detected on the new channel.
            mm_timer_clear(&vif->tmr_bcn_to);
            mm_timer_set(&vif->tbtt_timer, ke_time() + sta->bcn_int);
            vif->u.sta.beacon_loss_cnt = 0;
            vif->u.sta.csa_occured = true;
        }
        else
        {
            mm_send_connection_loss_ind(vif);
        }
    }
    #if NX_BCN_AUTONOMOUS_TX // if not defined this function is not called for AP itf
    else if (vif->type == VIF_AP)
    {
        vif->u.ap.csa_count = 0;

        if (res == CO_OK)
        {
            chan_ctxt_link(vif->index, chan_idx);

            // prevent beacon transmission until it has been updated
            mm_bcn_env.update_ongoing = true;
        }
    }
    #endif

    ke_msg_send(ind);
}
#endif

struct vif_info_tag *vif_mgmt_get_single_sta_vif(void)
{
    struct vif_info_tag *vif;

    #if NX_MULTI_ROLE
    if (vif_mgmt_used_cnt() != 1)
        return NULL;
    #endif

    vif = (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

    if (vif->type != VIF_STA)
        return NULL;

    return vif;
}

void vif_mgmt_set_bssid_mask(void)
{
    struct vif_info_tag *vif;

    // By default don't set any mask, meaning that all the received BSSID bits are
    // compared with the register
    nxmac_bss_id_hi_mask_set(0);

    #if NX_MULTI_ROLE
    if (vif_mgmt_used_cnt() > 1)
        return;
    #endif

    vif = (struct vif_info_tag *)co_list_pick(&vif_mgmt_env.used_list);

    // Check if the VIF is a STA one, connected to a nonTransmitted BSSID
    if ((vif->type == VIF_STA) && vif->active && vif->u.sta.bssid_index &&
        (vif->u.sta.max_bssid_ind <= 8))
        nxmac_bss_id_hi_mask_set((CO_BIT(vif->u.sta.max_bssid_ind) - 1) << 8);
}

/// @}
