/**
****************************************************************************************
*
* @file apm.c
*
* Copyright (C) RivieraWaves 2011-2019
*
* @brief Definition of the APM module environment.
*
****************************************************************************************
*/

#include "apm.h"
#include "apm_task.h"
#include "mac_frame.h"
#include "vif_mgmt.h"
#include "me_utils.h"
#include "me_task.h"
#include "ps.h"
#include "txu_cntrl.h"

#if NX_BEACONING

/** @addtogroup APM
* @{
*/

/// Definition of the global environment.
struct apm apm_env;

/*
 * FUNCTION IMPLEMENTATIONS
 ****************************************************************************************
 */

/**
 ****************************************************************************************
 * @brief Initialize the BSS configuration list.
 ****************************************************************************************
 */
static void apm_bss_config_init(void)
{
    // Sanity check - No BSS config should be ongoing
    ASSERT_ERR(co_list_is_empty(&apm_env.bss_config));

    // Initialize the BSS configuration list
    co_list_init(&apm_env.bss_config);
}

/**
 ****************************************************************************************
 * @brief Push a BSS configuration message to the list.
 *
 * @param[in] param  Pointer to the message parameters
 ****************************************************************************************
 */
static void apm_bss_config_push(void *param)
{
    struct ke_msg *msg = ke_param2msg(param);

    co_list_push_back(&apm_env.bss_config, &msg->hdr);
}

void apm_init(void)
{
    // Reset the environment
    memset(&apm_env, 0, sizeof(apm_env));
    apm_env.aging_sta_idx = 0;

    // Set state as Idle
    ke_state_set(TASK_APM, APM_IDLE);
}

void apm_start_cfm(uint8_t status)
{
    struct apm_start_req const *param = apm_env.param;
    struct apm_start_cfm *cfm = KE_MSG_ALLOC(APM_START_CFM, TASK_API, TASK_APM, apm_start_cfm);

    // Check if the status is OK
    if (status == CO_OK)
    {
        struct vif_info_tag *vif = &vif_info_tab[param->vif_idx];
        struct sta_info_tag *sta = &sta_info_tab[VIF_TO_BCMC_IDX(param->vif_idx)];
        struct mm_set_vif_state_req *req = KE_MSG_ALLOC(MM_SET_VIF_STATE_REQ, TASK_MM,
                                                        TASK_APM, mm_set_vif_state_req);

        // Fill the message parameters
        req->active = true;
        req->inst_nbr = vif->index;

        // Send the message to the task
        ke_msg_send(req);

        vif->flags = param->flags;
        vif->u.ap.ctrl_port_ethertype = param->ctrl_port_ethertype;
        vif->u.ap.ps_sta_cnt = 0;
        cfm->ch_idx = vif->chan_ctxt->idx;
        cfm->bcmc_idx = VIF_TO_BCMC_IDX(param->vif_idx);

        // Copy the rate set to the station entry
        sta->info.rate_set = param->basic_rates;
        // Open the control port for this station
        sta->ctrl_port_state = PORT_OPEN;

        // Initialize the policy table
        me_init_bcmc_rate(sta);

        // Initialize tx power on first transmit
        sta->pol_tbl.upd_field |= CO_BIT(STA_MGMT_POL_UPD_TX_POWER);

        // Initialize MAC address with muticast bit set
        sta->mac_addr.array[0] = 0x1;
        sta->staid = VIF_TO_BCMC_IDX(param->vif_idx);
    }

    // Fill-in the message parameters
    cfm->status = status;
    cfm->vif_idx = param->vif_idx;

    // Send the message
    ke_msg_send(cfm);

    // Free the parameters
    ke_msg_free(ke_param2msg(param));
    apm_env.param = NULL;

    // We are now waiting for the channel addition confirmation
    ke_state_set(TASK_APM, APM_IDLE);
}


void apm_set_bss_param(void)
{
    struct apm_start_req const *param = apm_env.param;
    struct vif_info_tag *vif = &vif_info_tab[param->vif_idx];
    struct me_set_ps_disable_req *ps = KE_MSG_ALLOC(ME_SET_PS_DISABLE_REQ, TASK_ME, TASK_APM,
                                                    me_set_ps_disable_req);
    struct mm_set_bssid_req *bssid = KE_MSG_ALLOC(MM_SET_BSSID_REQ, TASK_MM, TASK_APM,
                                                  mm_set_bssid_req);
    struct mm_set_basic_rates_req *brates = KE_MSG_ALLOC(MM_SET_BASIC_RATES_REQ, TASK_MM,
                                                         TASK_APM, mm_set_basic_rates_req);
    struct mm_set_beacon_int_req *bint = KE_MSG_ALLOC(MM_SET_BEACON_INT_REQ, TASK_MM,
                                                      TASK_APM, mm_set_beacon_int_req);
    struct me_set_active_req *active =  KE_MSG_ALLOC(ME_SET_ACTIVE_REQ, TASK_ME,
                                                   TASK_APM, me_set_active_req);

    // Initialize the BSS configuration list
    apm_bss_config_init();

    // Disable PS mode when an AP is started
    #if (NX_P2P_GO)
    if ( vif->p2p )
    {
        // P2P GO interface may enter sleep mode once NOA and/or Opportunistic PS
        // are enabled but not until then.
        nxmac_pwr_mgt_setf(0);
        #if NX_POWERSAVE
        vif->prevent_sleep |= PS_VIF_P2P_GO_PRESENT;
        #endif
    }
    else
    #endif //(NX_P2P_GO)
    {
        ps->ps_disable = true;
        ps->vif_idx = vif->index;
        apm_bss_config_push(ps);
    }

    // BSSID
    bssid->bssid = vif->mac_addr;
    bssid->inst_nbr = param->vif_idx;
    apm_bss_config_push(bssid);

    // Basic rates
    brates->band = param->chan.band;
    brates->rates = me_legacy_rate_bitfield_build(&param->basic_rates, true);
    brates->inst_nbr = param->vif_idx;
    apm_bss_config_push(brates);

    // Beacon interval
    bint->beacon_int = param->bcn_int;
    bint->inst_nbr = param->vif_idx;
    apm_bss_config_push(bint);

    // Go back to ACTIVE after setting the BSS parameters
    active->active = true;
    active->vif_idx = param->vif_idx;
    apm_bss_config_push(active);

    // Send the first BSS configuration message
    apm_bss_config_send();

    // We are now waiting for the channel addition confirmation
    ke_state_set(TASK_APM, APM_BSS_PARAM_SETTING);
}

void apm_bss_config_send(void)
{
    struct ke_msg *msg = (struct ke_msg *)co_list_pop_front(&apm_env.bss_config);

    // Sanity check - We shall have a message available
    ASSERT_ERR(msg != NULL);

    // Send the message
    ke_msg_send(ke_msg2param(msg));
}

void apm_bcn_set(void)
{
    struct apm_start_req const *param = apm_env.param;
    struct mm_bcn_change_req *bcn =  KE_MSG_ALLOC(MM_BCN_CHANGE_REQ, TASK_MM,
                                                  TASK_APM, mm_bcn_change_req);
    // Fill-in the parameters
    bcn->bcn_ptr = param->bcn_addr;
    bcn->bcn_len = param->bcn_len;
    bcn->tim_oft = param->tim_oft;
    bcn->tim_len = param->tim_len;
    bcn->inst_nbr = param->vif_idx;

    // Send the beacon information to the LMAC
    ke_msg_send(bcn);

    ke_state_set(TASK_APM, APM_BCN_SETTING);
}

void apm_stop(struct vif_info_tag *vif)
{
    struct me_set_ps_disable_req *ps = KE_MSG_ALLOC(ME_SET_PS_DISABLE_REQ, TASK_ME, TASK_APM,
                                                    me_set_ps_disable_req);
    struct me_set_active_req *idle =  KE_MSG_ALLOC(ME_SET_ACTIVE_REQ, TASK_ME,
                                                    TASK_APM, me_set_active_req);

    // Initialize the BSS configuration list
    apm_bss_config_init();

    // Re-allow PS mode in case it was disallowed
    ps->ps_disable = false;
    ps->vif_idx = vif->index;
    apm_bss_config_push(ps);

    // VIF state
    if (vif->active)
    {
        struct mm_set_vif_state_req *state = KE_MSG_ALLOC(MM_SET_VIF_STATE_REQ, TASK_MM,
                                                          TASK_APM, mm_set_vif_state_req);
        state->active = false;
        state->inst_nbr = vif->index;
        apm_bss_config_push(state);
    }

    // Unlink the VIF from the channel context
    if (vif->chan_ctxt != NULL)
    {
        struct mm_chan_ctxt_unlink_req *unlk = KE_MSG_ALLOC(MM_CHAN_CTXT_UNLINK_REQ,TASK_MM,
                                                            TASK_APM, mm_chan_ctxt_unlink_req);
        unlk->vif_index = vif->index;
        apm_bss_config_push(unlk);
    }

    idle->active = false;
    idle->vif_idx = vif->index;
    apm_bss_config_push(idle);

    // Send the first BSS configuration message
    apm_bss_config_send();

    ke_state_set(TASK_APM, APM_STOPPING);
}

bool apm_tx_int_ps_check(struct txdesc *txdesc)
{
    struct vif_info_tag *vif = &vif_info_tab[txdesc->host.vif_idx];

    /* Don't push frame to connected STA in PS */
    if ((vif->type == VIF_AP) &&
        (txdesc->host.staid < STA_MAX) &&
        ((sta_info_tab[txdesc->host.staid].ps_state == PS_MODE_ON) &&
         !(sta_info_tab[txdesc->host.staid].ps_service_period & ANY_SERVICE_PERIOD_INT)))
    {
        /* For now all possible internally generated frame are bufferable */
        txdesc->host.flags |= TXU_CNTRL_POSTPONE_PS;
        return false;
    }

    return true;
}

void apm_tx_int_ps_postpone(struct txdesc *txdesc, struct sta_info_tag *sta)
{
    if (txdesc->host.flags & TXU_CNTRL_POSTPONE_PS)
    {
        bool new_traffic = true;

        if (sta->info.uapsd_queues & mac_ac2uapsd[txdesc->host.tid])
        {
            if (sta->traffic_avail & UAPSD_TRAFFIC_INT)
                new_traffic = false;
            else
                sta->traffic_avail |= UAPSD_TRAFFIC_INT;
        }
        else
        {
            if (sta->traffic_avail & PS_TRAFFIC_INT)
                new_traffic = false;
            else
                sta->traffic_avail |= PS_TRAFFIC_INT;
        }

        if (new_traffic &&
            (!(sta->info.uapsd_queues & mac_ac2uapsd[txdesc->host.tid]) ||
             (sta->info.uapsd_queues == MAC_QOS_INFO_STA_UAPSD_ENABLED_ALL)))
        {
            struct mm_tim_update_req *tim = KE_MSG_ALLOC(MM_TIM_UPDATE_REQ, TASK_MM,
                                                         TASK_ME, mm_tim_update_req);

            tim->aid = sta->aid;
            tim->inst_nbr = sta->inst_nbr;
            tim->tx_avail = true;

            ke_msg_send(tim);
        }
    }
}

struct txdesc *apm_tx_int_ps_get_postpone(struct vif_info_tag *vif,
                                          struct sta_info_tag *sta,
                                          int *stop)
{
    struct txdesc *txdesc, *txdesc_prev = NULL;
    enum sta_ps_traffic traffic_int, traffic_host;
    bool more = false;

    if ((vif->type != VIF_AP) ||
        (sta->ps_service_period == NO_SERVICE_PERIOD))
    {
        *stop = 0;
        return NULL;
    }

    /* Check that there is still requested traffic in queue */
    if (sta->ps_service_period & PS_SERVICE_PERIOD)
    {
        traffic_int = PS_TRAFFIC_INT;
        traffic_host = PS_TRAFFIC_HOST;
    }
    else
    {
        traffic_int = UAPSD_TRAFFIC_INT;
        traffic_host = UAPSD_TRAFFIC_HOST;
    }

    if (!(sta->traffic_avail & traffic_int))
    {
        *stop = 1;
        return NULL;
    }

    /* Get first txdesc that match the service period (i.e. legacy or uapsd) */
    txdesc = (struct txdesc *)co_list_pick(&sta->tx_desc_post);
    while (txdesc)
    {
        if (!!(sta->info.uapsd_queues & mac_ac2uapsd[txdesc->host.tid]) ==
            (sta->ps_service_period == UAPSD_SERVICE_PERIOD_INT))
        {
            break;
        }

        txdesc_prev = txdesc;
        txdesc = tx_desc_next(txdesc);
    }

    if (!txdesc)
    {
        ASSERT_WARN(0);
        *stop = 1;
        return NULL;
    }

    co_list_remove(&sta->tx_desc_post, &txdesc_prev->list_hdr, &txdesc->list_hdr);

    if (sta->ps_service_period & BCN_SERVICE_PERIOD)
    {
        txdesc->host.tid = AC_BCN;
    }
    else
    {
        /* To avoid disorder with EOSP, alwasy sent in AC_VO */
        /* In practice most of internally generated frames are already on AC_VO */
        txdesc->host.tid = AC_VO;
    }

    /* Check if more txdesc is present for the same kind of service period */
    if (txdesc_prev)
        txdesc_prev = tx_desc_next(txdesc_prev);
    else
        txdesc_prev = (struct txdesc *)co_list_pick(&sta->tx_desc_post);
    while (txdesc_prev)
    {
        if (!!(sta->info.uapsd_queues & mac_ac2uapsd[txdesc_prev->host.tid]) ==
            (sta->ps_service_period == UAPSD_SERVICE_PERIOD_INT))
        {
            more = true;
            break;
        }

        txdesc_prev = tx_desc_next(txdesc_prev);
    }

    /* If no more internal traffic, check host status */
    if (!more)
    {
        sta->traffic_avail &= ~traffic_int;
        more = (sta->traffic_avail & traffic_host);
    }

    if (more)
    {
        struct mac_hdr *mac_hdr = txl_buffer_payload_get(txdesc);
        mac_hdr->fctl |= MAC_FCTRL_MOREDATA;
    }
    else
    {
        struct mm_tim_update_req *tim = KE_MSG_ALLOC(MM_TIM_UPDATE_REQ, TASK_MM,
                                                     TASK_ME, mm_tim_update_req);
        tim->aid = sta->aid;
        tim->inst_nbr = sta->inst_nbr;
        tim->tx_avail = false;
        ke_msg_send(tim);
    }

    return txdesc;
}

void apm_tx_int_ps_clear(struct vif_info_tag *vif, uint8_t sta_idx)
{
    struct sta_info_tag *sta = &sta_info_tab[sta_idx];

    if (vif->type != VIF_AP)
        return;

    /* push all postponed frame */
    sta_mgmt_send_postponed_frame(vif, sta, 0);

    if (sta->traffic_avail & (UAPSD_TRAFFIC_INT | PS_TRAFFIC_INT))
    {
        sta->traffic_avail &= ~(UAPSD_TRAFFIC_INT | PS_TRAFFIC_INT);

        /* If there is still HOST traffic wait for host to update TIM */
        if (!sta->traffic_avail)
        {
            struct mm_tim_update_req *tim = KE_MSG_ALLOC(MM_TIM_UPDATE_REQ, TASK_MM,
                                                         TASK_ME, mm_tim_update_req);
            tim->aid = sta->aid;
            tim->inst_nbr = sta->inst_nbr;
            tim->tx_avail = false;
            ke_msg_send(tim);
        }
    }
}

/// @}

#endif /* NX_BEACONING */

