/**
****************************************************************************************
*
* @file apm_task.c
*
* @brief The UMAC's AP Manager (APM) module implementation.
*
* Copyright (C) RivieraWaves 2011-2019
*
****************************************************************************************
*/

#include "apm.h"
#include "apm_task.h"
#include "me_utils.h"
#include "bam.h"
#include "ke_timer.h"
#include "vif_mgmt.h"
#include "chan.h"
#include "me_task.h"
#include "me.h"
#include "ps.h"
#include "tpc.h"

#if NX_BEACONING
/**
 * @addtogroup TASK_APM
 * @{
 */

/**
****************************************************************************************
* @brief APM module message handler.
*
* This function handles the message APM_AP_START_REQ from host.
* This message requests the UMAC to start an AP in a BSS network.
*
* @param[in] msgid Id of the message received (probably unused)
* @param[in] param Pointer to the parameters of the message.
* @param[in] dest_id TaskId of the receiving task.
* @param[in] src_id TaskId of the sending task.
* @return Whether the message was consumed or not.
****************************************************************************************
*/
static int
apm_start_req_handler(ke_msg_id_t const msgid,
                         struct apm_start_req *param,
                         ke_task_id_t const dest_id,
                         ke_task_id_t const src_id)
{

    struct apm_start_cfm *cfm;
    uint8_t status;
    // Channel context index
    uint8_t chan_idx;

    do
    {
        struct vif_info_tag *vif = &vif_info_tab[param->vif_idx];

        // Check if the VIF is configured as AP
        if (vif->type != VIF_AP)
        {
            status = CO_BAD_PARAM;
            break;
        }

        // Check if we are busy or not
        if (ke_state_get(TASK_APM) != APM_IDLE)
        {
            status = CO_BUSY;
            break;
        }

        // Check if the AP is not already started
        if (vif->active)
        {
            status = CO_OP_IN_PROGRESS;
            break;
        }

        // Sanity check - We should not have a channel context already registered
        ASSERT_ERR(vif->chan_ctxt == NULL);

        // Save the parameters
        apm_env.param = param;

        if (chan_ctxt_add(&param->chan, &chan_idx) == CO_OK)
        {
            struct me_bss_info *bss = &vif->bss_info;

            // Save the BW and channel information in the VIF
            bss->chan = param->chan;

            // Initialize TX power
            bss->power_constraint = 0;
            tpc_update_vif_tx_power(vif);

            // Link the VIF to the channel context
            chan_ctxt_link(param->vif_idx, chan_idx);

            // If the AP is started on 2.4GHz band, we need to get the highest 11b rate
            // that might be used for the non-ERP protection
            if (param->chan.band == PHY_BAND_2G4)
            {
                uint32_t basic_11b_rates;
                basic_11b_rates = me_legacy_rate_bitfield_build(&(bss->rate_set), true) & 0x0F;
                // Get highest allowed 11b rate
                if (basic_11b_rates)
                    // Get highest allowed 11b rate
                    bss->high_11b_rate = 31 - co_clz(basic_11b_rates);
                else
                    // If no 11b basic rates, set the highest mandatory one
                    bss->high_11b_rate = HW_RATE_2MBPS;
            }

            // Set the BSS parameters to LMAC
            apm_set_bss_param();
        }
        else
        {
            status = CO_FAIL;
            break;
        }

        // We will now proceed to the AP starting
        return (KE_MSG_NO_FREE);
    } while(0);

    cfm = KE_MSG_ALLOC(APM_START_CFM, src_id, dest_id, apm_start_cfm);
    cfm->status = status;
    cfm->vif_idx = param->vif_idx;

    // Send the message
    ke_msg_send(cfm);

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief APM module message handler.
 *
 * This function handles the message ME_SET_PS_DISABLE_CFM.
 * It is sent by ME Task once request Power Save mode has been configured. Power Save mode
 * is disabled when an AP is started and re-enabled when it is stopped.
 *
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_set_ps_disable_cfm_handler(ke_msg_id_t const msgid,
                              void const *param,
                              ke_task_id_t const dest_id,
                              ke_task_id_t const src_id)
{
    // Sanity check
    ASSERT_ERR((ke_state_get(TASK_APM) == APM_BSS_PARAM_SETTING) ||
               (ke_state_get(TASK_APM) == APM_IDLE) ||
               (ke_state_get(TASK_APM) == APM_STOPPING));

    // Check the state
    if ((ke_state_get(TASK_APM) == APM_BSS_PARAM_SETTING) ||
        (ke_state_get(TASK_APM) == APM_STOPPING))
    {
        // Send the next BSS configuration message
        apm_bss_config_send();
    }

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief APM module message handler.
 *
 * This function handles the messages MM_SET_BSSID_CFM and MM_SET_BEACON_INT_CFM.
 * They are sent by MM task once requested configuration has been completed. Configuration
 * is done only when the AP is started.
 *
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_bss_param_setting_handler(ke_msg_id_t const msgid,
                             void const *param,
                             ke_task_id_t const dest_id,
                             ke_task_id_t const src_id)
{
    // Sanity check - This message can be received only in the CHAN CTXT ADDING state
    ASSERT_ERR(ke_state_get(TASK_APM) == APM_BSS_PARAM_SETTING);

    // Send the next BSS parameter configuration message
    apm_bss_config_send();

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief APM module message handler.
 *
 * This function handles the message ME_SET_ACTIVE_CFM.
 * It is sent by ME Task once requested state is reached. Vif is activated when the AP is
 * started and deactivated when the AP is stopped.
 *
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
me_set_active_cfm_handler(ke_msg_id_t const msgid,
                          void const *param,
                          ke_task_id_t const dest_id,
                          ke_task_id_t const src_id)
{
    // Sanity check - This message can be received only in the STA ADDING state
    ASSERT_ERR((ke_state_get(TASK_APM) == APM_BSS_PARAM_SETTING) ||
               (ke_state_get(TASK_APM) == APM_IDLE) ||
               (ke_state_get(TASK_APM) == APM_STOPPING));

    // Check the state
    if (ke_state_get(TASK_APM) == APM_BSS_PARAM_SETTING)
    {
        // Sanity check - All the BSS configuration parameters shall be set
        ASSERT_ERR(co_list_is_empty(&apm_env.bss_config));

        // Set the beacon information to the LMAC
        apm_bcn_set();
    }
    else if (ke_state_get(TASK_APM) == APM_STOPPING)
    {
        // Send the confirmation
        ke_msg_send_basic(APM_STOP_CFM, TASK_API, TASK_APM);
        ke_state_set(TASK_APM, APM_IDLE);
    }

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief APM module message handler.
 *
 * This function handles the message MM_BCN_CHANGE_CFM.
 * It is sent by MM layer once the beacon has been configured. Beacon is only configured
 * once when the AP is started. It is the last step in the start of the AP.
 * @note If Beacon need to be modified while AP is running, the modification is directly
 * sent to MM task.
 *
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_bcn_change_cfm_handler(ke_msg_id_t const msgid,
                          void const *param,
                          ke_task_id_t const dest_id,
                          ke_task_id_t const src_id)
{
    // Sanity check - This message can be received only in the APM_SCHEDULING_CHAN_CTX state
    ASSERT_ERR(ke_state_get(TASK_APM) == APM_BCN_SETTING);

    // Set the beacon information to the LMAC
    apm_start_cfm(CO_OK);

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief @ref MM_CHAN_CTXT_UNLINK_CFM message handler.
 * This function is called once the channel context associated to a VIF has been unlinked.
 *
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_chan_ctxt_unlink_cfm_handler(ke_msg_id_t const msgid,
                                void const *param,
                                ke_task_id_t const dest_id,
                                ke_task_id_t const src_id)
{
    // Sanity check - This message can be received during the stopping procedure only
    ASSERT_ERR(ke_state_get(TASK_APM) == APM_STOPPING);

    // Send the next BSS stopping message
    apm_bss_config_send();

    return (KE_MSG_CONSUMED);
}

/**
 ****************************************************************************************
 * @brief @ref MM_SET_VIF_STATE_CFM message handler.
 * This function is called once the VIF has been put to active state at the end of the
 * AP creation, or to inactive state upon AP deletion.
 * @param[in] msgid Id of the message received (probably unused)
 * @param[in] param Pointer to the parameters of the message.
 * @param[in] dest_id TaskId of the receiving task.
 * @param[in] src_id TaskId of the sending task.
 * @return Whether the message was consumed or not.
 ****************************************************************************************
 */
static int
mm_set_vif_state_cfm_handler(ke_msg_id_t const msgid,
                             void const *param,
                             ke_task_id_t const dest_id,
                             ke_task_id_t const src_id)
{
    // Sanity check - This message can be received only in the stopping and idle states
    ASSERT_ERR((ke_state_get(TASK_APM) == APM_STOPPING) ||
               (ke_state_get(TASK_APM) == APM_IDLE));

    // Check if we are in the connection process
    if (ke_state_get(TASK_APM) == APM_STOPPING)
    {
        // Send the next BSS stopping message
        apm_bss_config_send();
    }

    return (KE_MSG_CONSUMED);
}

/**
****************************************************************************************
* @brief APM module message handler.
*
* This function handles the message APM_AP_STOP_REQ from host.
* This message requests the UMAC to stop the AP.
*
* @param[in] msgid Id of the message received (probably unused)
* @param[in] param Pointer to the parameters of the message.
* @param[in] dest_id TaskId of the receiving task.
* @param[in] src_id TaskId of the sending task.
* @return Whether the message was consumed or not.
****************************************************************************************
*/
static int
apm_stop_req_handler(ke_msg_id_t const msgid,
                     struct apm_stop_req const *param,
                     ke_task_id_t const dest_id,
                     ke_task_id_t const src_id)
{
    struct vif_info_tag *vif = &vif_info_tab[param->vif_idx];

    if ((vif->type != VIF_AP) || (!vif->active))
    {
        // Not an AP (or not started) respond immediately
        ke_msg_send_basic(APM_STOP_CFM, src_id, dest_id);
        return KE_MSG_CONSUMED;
    }
    else if (ke_state_get(TASK_APM) != APM_IDLE)
    {
        // Task is currently busy, postpone this message
        return KE_MSG_SAVED;
    }

    // Stop the AP (confirmation will be sent once stopped)
    apm_stop(vif);
    return KE_MSG_CONSUMED;
}

/**
****************************************************************************************
* @brief APM module message handler.
* This function handles the message APM_AP_START_CAC_REQ from SME.
* This message requests the UMAC to start listening on the specified channel
* until APM_AP_STOP_CAC_REQ is received.
* CAC (Channel Availability Check) is done to check for radar before starting
* an AP on a DFS channel
*
* @param[in] msgid Id of the message received (probably unused)
* @param[in] param Pointer to the parameters of the message.
* @param[in] dest_id TaskId of the receiving task.
* @param[in] src_id TaskId of the sending task.
* @return Whether the message was consumed or not.
****************************************************************************************
*/
static int
apm_start_cac_req_handler(ke_msg_id_t const msgid,
                          struct apm_start_cac_req const *param,
                          ke_task_id_t const dest_id,
                          ke_task_id_t const src_id)
{
    struct apm_start_cac_cfm *cfm;
    uint8_t status = CO_OK;
    uint8_t chan_idx = 0;

    do
    {
        struct vif_info_tag *vif = &vif_info_tab[param->vif_idx];

        // Check if the VIF is configured as AP
        if (vif->type != VIF_AP) {
            status = CO_BAD_PARAM;
            break;
        }

        // Check if the AP is not already started
        if (vif->active) {
            status = CO_BUSY;
            break;
        }

        // Check if we are busy or not
        if (ke_state_get(TASK_APM) != APM_IDLE) {
            status = CO_BUSY;
            break;
        }

        // create new channel context
        if (chan_ctxt_add(&param->chan, &chan_idx) == CO_OK) {
            // and Link the VIF to the channel context
            chan_ctxt_link(param->vif_idx, chan_idx);
        } else {
            status = CO_FAIL;
            break;
        }

        #if (NX_POWERSAVE)
        // Disable PS during the whole CAC period
        GLOBAL_INT_DISABLE();
        ps_env.prevent_sleep |= PS_CAC_STARTED;
        GLOBAL_INT_RESTORE();
        #endif


    } while(0);

    // Send the confirmation
    cfm = KE_MSG_ALLOC(APM_START_CAC_CFM, src_id, dest_id, apm_start_cac_cfm);
    cfm->status = status;
    cfm->ch_idx = chan_idx;

    // Send the message
    ke_msg_send(cfm);

    return (KE_MSG_CONSUMED);
}

/**
****************************************************************************************
* @brief APM module message handler.
* This function handles the message APM_AP_STOP_CAC_REQ from SME.
* This message requests the UMAC to stop listening of the specified channel.
* CAC (Channel Availability Check) ends if a radar is detected or after a defined period
*
* @param[in] msgid Id of the message received (probably unused)
* @param[in] param Pointer to the parameters of the message.
* @param[in] dest_id TaskId of the receiving task.
* @param[in] src_id TaskId of the sending task.
* @return Whether the message was consumed or not.
****************************************************************************************
*/
static int
apm_stop_cac_req_handler(ke_msg_id_t const msgid,
                         struct apm_stop_cac_req const *param,
                         ke_task_id_t const dest_id,
                         ke_task_id_t const src_id)
{
    do
    {
        struct vif_info_tag *vif = &vif_info_tab[param->vif_idx];

        // Check if the VIF is configured as AP
        if (vif->type != VIF_AP) {
            break;
        }

        // Check if the AP is not already started
        if (vif->active) {
            break;
        }

        // Check if we are busy or not
        if (ke_state_get(TASK_APM) != APM_IDLE) {
            break;
        }

        chan_ctxt_unlink(param->vif_idx);

        #if (NX_POWERSAVE)
        //  Power Save can be used
        GLOBAL_INT_DISABLE();
        ps_env.prevent_sleep &= ~PS_CAC_STARTED;
        GLOBAL_INT_RESTORE();
        #endif

    } while(0);

    // Send the confirmation
    ke_msg_send_basic(APM_STOP_CAC_CFM, src_id, dest_id);

    return (KE_MSG_CONSUMED);
}


/// DEFAULT handler definition.
const struct ke_msg_handler apm_default_state[] =
{
    {APM_START_REQ, (ke_msg_func_t)apm_start_req_handler},
    {APM_STOP_REQ, (ke_msg_func_t)apm_stop_req_handler},
    {ME_SET_ACTIVE_CFM, (ke_msg_func_t)me_set_active_cfm_handler},
    {MM_BCN_CHANGE_CFM, (ke_msg_func_t)mm_bcn_change_cfm_handler},
    {MM_SET_BSSID_CFM, mm_bss_param_setting_handler},
    {MM_SET_BASIC_RATES_CFM, mm_bss_param_setting_handler},
    {MM_SET_BEACON_INT_CFM, mm_bss_param_setting_handler},
    {ME_SET_PS_DISABLE_CFM, (ke_msg_func_t)me_set_ps_disable_cfm_handler},
    {MM_SET_VIF_STATE_CFM, (ke_msg_func_t)mm_set_vif_state_cfm_handler},
    {MM_CHAN_CTXT_UNLINK_CFM, (ke_msg_func_t)mm_chan_ctxt_unlink_cfm_handler},
    {APM_START_CAC_REQ, (ke_msg_func_t)apm_start_cac_req_handler},
    {APM_STOP_CAC_REQ, (ke_msg_func_t)apm_stop_cac_req_handler},
};

/// Specifies the message handlers that are common to all states.
const struct ke_state_handler apm_default_handler =
    KE_STATE_HANDLER(apm_default_state);

/// Defines the placeholder for the states of all the task instances.
ke_state_t apm_state[APM_IDX_MAX];

/// @} end of addtogroup

#endif

